/******************************************/
/* Wmufo - Mini SETI@Home monitor         */
/******************************************/

/******************************************/
/* This program is free software; you can redistribute it and/or
/* modify it under the terms of the GNU General Public License
/* as published by the Free Software Foundation; either version 2
/* of the License, or (at your option) any later version.
/* 
/* This program is distributed in the hope that it will be useful,
/* but WITHOUT ANY WARRANTY; without even the implied warranty of
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/* GNU General Public License for more details.
/* 
/* You should have received a copy of the GNU General Public License
/* along with this program; if not, write to the Free Software
/* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
/******************************************/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <signal.h>
#include <pwd.h>
#include <dirent.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <math.h>

#include <gdk/gdk.h>
#include <gdk/gdkx.h>

#include "seti.h"
#include "session.h"
#include "rgba.h"
#include "cmap.h"

#include "master.xpm"
#include "icon.xpm"

/******************************************/
/* Defines                                */
/******************************************/

#define XMAX 56
#define YMAX 56
#define RGBSIZE (XMAX * YMAX * 3)
#define NRBALL 4
#define NRSTAR 15

#define UPDATE 30000

#define SETI_DIRMAX	10

#define STAT_TITLE	1
#define STAT_PROG	2
#define STAT_CPU	3
#define STAT_TIMELEFT	4
#define STAT_NWUS	5
#define STAT_NRESULTS	6
#define STAT_PID	7
#define STAT_TOTALCPU	8
#define STAT_BAR	9

/******************************************/
/* Structures                             */
/******************************************/

typedef struct {
	int x;			/* Current x position */
	int y;			/* Current y position */
	int speed;		/* Distance to move */
	int travel;		/* Distance travelled beyond screen */
	int rev;		/* Horizontal direction */
	int frame;		/* Current animation frame */
	int delay;		/* Frame delay */
	sprite_rgba sp;		/* Sprite data */
} sp_rgba;

typedef struct {
	int x;			/* Current x position */
	int y;			/* Current y position */
	int speed;		/* Ditance to move */
	int travel;		/* Distance travelled beyond screen */
	int rev;		/* Horizontal direction */
	int frame;		/* Current animation frame */
	int delay;		/* Frame delay */
	sprite_cmap sp;		/* Sprite data */
} sp_cmap;

typedef struct {
	Display *display;	/* X11 display */
	GdkWindow *win;		/* Main window */
	GdkWindow *iconwin;	/* Icon window */
	GdkGC *gc;		/* Drawing GC */
	GdkPixmap *pixmap;	/* Main pixmap */
	GdkBitmap *mask;	/* Dockapp mask */

	int x;			/* Window X position */
	int y;			/* Window Y position */

	sp_cmap ufo;		/* Flying saucer */
	sp_rgba ball[NRBALL];	/* Glowing ball */
	sp_rgba stars[NRSTAR];	/* Stars */
	sp_cmap text;		/* Characters */

	unsigned char rgb[RGBSIZE];
	unsigned char rgb_back[RGBSIZE];
} wmufo_data;

typedef struct {
	char *program;
	char *dirseti[SETI_DIRMAX];
	int dirtotal;
	int dircurrent;
	int line[10];
	int timeout;
	int update;
} stat_data;

/******************************************/
/* Functions                              */
/******************************************/

void init_sprites();
static void star_update();
static void ufo_update();
static void ball_update();
static void proximity_update();
void random_update();

void setup_background();

GdkCursor *setup_cursor();
void make_wmufo_dockapp();
static void draw_sprite_cmap(int, int, int, sprite_cmap);
static void draw_sprite_rgba(int, int, int, sprite_rgba);
static void draw_string(int, int, char *, int, sprite_cmap);
static void draw_bar(int, int, int, int, int);
static void info_update(int);

int insert_setidir(const char *);
char *expand_dir(const char *);
char *secs_2_time(const long, char *);
char *secs_2_epoch(const long, char *);

void read_config(int, char **);
void do_help();

/******************************************/
/* Globals                                */
/******************************************/

static wmufo_data bm;
static stat_data stats;
static seti_data seti;

char *session_id = NULL;

int proximity = 0;
int broken_wm = 0;
int debug = 0;

/******************************************/
/* Main                                   */
/******************************************/

int
main(int argc, char **argv)
{
	GdkEvent *event;
	GdkCursor *cursor;
	int i;

	int animation = 0;
	int dirseti = 0;
	int pauseseti = 0;
	int runseti = 0;
	int count = 0;

	srand(time(NULL));

	/* Initialize GDK */
	if (!gdk_init_check(&argc, &argv)) {
		fprintf(stderr, "GDK init failed. Check \"DISPLAY\" variable.\n");
		exit(-1);
	}

	/* Initialise invisible cursor */
	cursor = setup_cursor();

	/* Zero main data structures */
	memset(&bm, 0, sizeof (bm));
	memset(&seti, 0, sizeof (seti));

	/* Parse command line and configuration file */
	read_config(argc, argv);
#ifdef SESSION
        smc_connect(argc, argv, session_id);
#endif

	/* Create dockapp window. creates windows, allocates memory, etc */
	make_wmufo_dockapp();

	/* Initialise sprite graphics and motion */
	init_sprites();

	/* Initialise statistics */
	if (stats.dirtotal)
		read_stats(stats.dirseti[stats.dircurrent], &seti);

	while (1) {
		while (gdk_events_pending()) {
			event = gdk_event_get();
			if (event) {
				switch (event->type) {
				case GDK_DESTROY:
				case GDK_DELETE:
					gdk_cursor_destroy(cursor);
					exit(0);
					break;
				case GDK_BUTTON_PRESS:
				case GDK_SCROLL:
					if (stats.dirtotal) {
						if (event->button.button == 1) {
							if (!animation || stats.dirtotal == 1
							    || (event->button.x < 32 && event->button.y < 32))
								animation = !animation;
							else
								dirseti = 1;
						} else if (event->button.button == 2)
							runseti = 1;
						else if (event->button.button == 3)
							pauseseti = 1;
						else if (event->scroll.direction == GDK_SCROLL_UP)
							dirseti = 1;
						else if (event->scroll.direction == GDK_SCROLL_DOWN)
							dirseti= -1;
						if (animation) {
							count = stats.update;
							gdk_window_set_cursor(bm.win, NULL);
							animation = 1;
						} else
							gdk_window_set_cursor(bm.win, cursor);
					}
					break;
				case GDK_ENTER_NOTIFY:
					/* Make ball ufo run and hide */
					proximity = 1;
					if (!animation)
						gdk_window_set_cursor(bm.win, cursor);
					for (i = 0; i < NRBALL; i++)
						bm.ball[i].speed += (rand() % 2) + 1;
					break;
				case GDK_LEAVE_NOTIFY:
					proximity = 0;
					gdk_window_set_cursor(bm.win, NULL);
					/* Make ball ufo cautiously return */
					for (i = 0; i < NRBALL; i++)
						bm.ball[i].speed = 1;
					break;
				default:
					break;
				}
				gdk_event_free(event);
			}
		}

		if (stats.dirtotal) {
			/* Pause or continue current setiathome client */
			if (pauseseti) {
				read_stats(stats.dirseti[stats.dircurrent], &seti);

				if (seti.status != 'U') {
					if (seti.status == 'R')
						kill(seti.pid, SIGSTOP);
					else
						kill(seti.pid, SIGCONT);
				}
				pauseseti = 0;
			}

			/* Run or kill current setiathome client */
			if (runseti) {
				read_stats(stats.dirseti[stats.dircurrent], &seti);

				if (seti.status == 'U') {
					chdir(stats.dirseti[stats.dircurrent]);
					/* Glib2 reaps dead setiathome clients */
					g_spawn_command_line_async(stats.program, NULL);
				} else {
					/* Paused programs don't die until continued */
					if (seti.status != 'R')
						kill(seti.pid, SIGCONT);
					kill(seti.pid, SIGTERM);
				}
				runseti = 0;
			}

			/* Change setiathome directory */
			if (dirseti) {
				if (dirseti > 0) {
					stats.dircurrent++;
					if (stats.dircurrent >= stats.dirtotal)
						stats.dircurrent = 0;
				} else {
					stats.dircurrent--;
					if (stats.dircurrent < 0)
						stats.dircurrent = stats.dirtotal - 1;
				}

				read_stats(stats.dirseti[stats.dircurrent], &seti);

				dirseti = 0;
			}
		}

		usleep(UPDATE);

		if (!animation) {
			/* Draw background */
			memcpy(&bm.rgb, &bm.rgb_back, RGBSIZE);

			if (stats.dirtotal && (count += UPDATE) >= stats.update) {
				read_stats(stats.dirseti[stats.dircurrent], &seti);
				count = 0;
			}

			/* Update and draw sprites */
			star_update();
			ufo_update();
			ball_update();
			proximity_update();
			random_update();
		} else {
			if ((count += UPDATE) >= stats.update) {
				if (animation++ > stats.timeout && stats.timeout != 0) {
					gdk_window_set_cursor(bm.win, cursor);
					animation = 0;
				} else {
					/* Update setiathome statistics */
					read_stats(stats.dirseti[stats.dircurrent], &seti);

					/* Reset grey background */
					memset(&bm.rgb, 32, RGBSIZE);

					/* Draw seti statistics */
					info_update(animation % 2);
				}

				count = 0;
			}
		}

		/* Draw the rgb buffer to screen */
		if (!broken_wm)
			gdk_draw_rgb_image(bm.iconwin, bm.gc, 4, 4, XMAX, YMAX, GDK_RGB_DITHER_NONE, bm.rgb, XMAX * 3);
		else
			gdk_draw_rgb_image(bm.win, bm.gc, 4, 4, XMAX, YMAX, GDK_RGB_DITHER_NONE, bm.rgb, XMAX * 3);

	}

	return 0;
}

/******************************************/
/* Initialise sprite attributes           */
/******************************************/

void
init_sprites()
{
	int i;

	for (i = 0; i < NRSTAR; i++) {
		bm.stars[i].sp = star;
		bm.stars[i].x = rand() % (XMAX - bm.stars[i].sp.w);
		bm.stars[i].y = rand() % (YMAX - bm.stars[i].sp.h);
	}

	bm.ball[0].sp = haloball;
	bm.ball[1].sp = redball;
	bm.ball[2].sp = greenball;
	bm.ball[3].sp = blueball;

	for (i = 0; i < NRBALL; i++) {
		bm.ball[i].x = -(bm.ball[i].sp.w);
		bm.ball[i].y = YMAX / 2;
		bm.ball[i].speed = (rand() % 2) + 1;
		bm.ball[i].rev = (i % 2) ? 1 : 0;
	}

	bm.ufo.sp.w = saucer_width;
	bm.ufo.sp.h = saucer_height;
	bm.ufo.sp.f = saucer_frames;
	bm.ufo.sp.cmap = *saucer_cmap;
	bm.ufo.sp.data = saucer_data;

	bm.ufo.x = -(bm.ufo.sp.h);
	bm.ufo.y = YMAX / 2;
	bm.ufo.speed = 1;
	bm.ufo.rev = 0;

	bm.text.sp.w = char_width;
	bm.text.sp.h = char_height;
	bm.text.sp.f = char_frames;
	bm.text.sp.cmap = *char_cmap;
	bm.text.sp.data = char_data;

	setup_background();
}

/******************************************/
/* Animate and draw stars                 */
/******************************************/

static void
star_update()
{
	int i, j;

	/* Moving starfield */
	for (i = 0; i < NRSTAR; i++) {
		bm.stars[i].x--;
		if (bm.stars[i].x < -((int) bm.stars[i].sp.w)) {
			bm.stars[i].x = rand() % XMAX + XMAX;
			bm.stars[i].y = rand() % (YMAX - bm.stars[i].sp.h);
		}

		/* Random vertical movement */
		j = rand() % 64;
		if (j == 16)
			bm.stars[i].y--;
		else if (j == 32)
			bm.stars[i].y++;

		draw_sprite_rgba(bm.stars[i].x, bm.stars[i].y, 0, bm.stars[i].sp);
	}
}

/******************************************/
/* Animate and draw saucer ufo            */
/******************************************/

static void
ufo_update()
{
#ifdef UFO_FIFTIES
	static int pause=0;
#endif

	if (seti.state == SETI_STATE_NORMAL || !stats.dirtotal) {
		/* Move UFO horizontally */
		if (!bm.ufo.rev) {
			bm.ufo.x -= bm.ufo.speed;
			if (bm.ufo.x < -((int) bm.ufo.sp.w) - bm.ufo.travel) {
				/* Off screen : change direction, position & speed */
				bm.ufo.travel = rand() % (XMAX * 2);
				bm.ufo.x = -((int) bm.ufo.sp.w) - bm.ufo.travel;
				bm.ufo.y = rand() % (YMAX - (int) bm.ufo.sp.h / (int) bm.ufo.sp.f);
				bm.ufo.rev = 1;
				bm.ufo.speed = (rand() % 2) + 1;
			}
		} else {
			bm.ufo.x += bm.ufo.speed;
			if (bm.ufo.x > XMAX + bm.ufo.travel) {
				/* Off screen : change direction, position & speed */
				bm.ufo.travel = rand() % (XMAX * 2);
				bm.ufo.x = XMAX + bm.ufo.travel;
				bm.ufo.y = rand() % (YMAX - (int) bm.ufo.sp.h / (int) bm.ufo.sp.f);
				bm.ufo.rev = 0;
				bm.ufo.speed = (rand() % 2) + 1;
			}
		}
	} else {
		/* Move UFO vertically */
		if (!bm.ufo.rev) {
			bm.ufo.y -= bm.ufo.speed;
			if (bm.ufo.y < -((int) bm.ufo.sp.h / (int) bm.ufo.sp.f) - bm.ufo.travel) {
				/* Off screen : change direction, position & speed */
				bm.ufo.travel = rand() % (YMAX * 2);
				bm.ufo.y = -((int) bm.ufo.sp.h / (int)bm.ufo.sp.f) - bm.ufo.travel;
				bm.ufo.x = -(int) bm.ufo.sp.w / 2 + rand() % XMAX;
				bm.ufo.rev = 1;
				bm.ufo.speed = (rand() % 2) + 1;
			}
		} else {
			bm.ufo.y += bm.ufo.speed;
			if (bm.ufo.y > YMAX + bm.ufo.travel) {
				/* Off screen : change direction, position & speed */
				bm.ufo.travel = rand() % (YMAX * 2);
				bm.ufo.y = YMAX + bm.ufo.travel;
				bm.ufo.x = -(int) bm.ufo.sp.w / 2 + rand() % XMAX;
				bm.ufo.rev = 0;
				bm.ufo.speed = (rand() % 2) + 1;
			}
		}
	}

#ifdef UFO_FIFTIES
	if (pause=!pause)
#endif
		bm.ufo.frame++;

	if (bm.ufo.frame >= bm.ufo.sp.f)
		bm.ufo.frame = 0;

	draw_sprite_cmap(bm.ufo.x, bm.ufo.y, bm.ufo.frame, bm.ufo.sp);
}

/******************************************/
/* Animate and draw ball ufos             */
/******************************************/

static void
ball_update()
{
	int i, j;

	for (i = 0; i < NRBALL; i++) {

		/* Frozen ball (off screen) */
		if (bm.ball[i].speed == 0)
			continue;


		if (seti.state == SETI_STATE_NORMAL || !stats.dirtotal) {
			/* Apply horizontal movement */
			if (!bm.ball[i].rev) {
				bm.ball[i].x -= bm.ball[i].speed;
				if (bm.ball[i].x < -((int) bm.ball[i].sp.w) - bm.ball[i].travel) {
					/* Off screen : change direction, position & speed */
					bm.ball[i].travel = rand() % XMAX;
					bm.ball[i].x = -((int) bm.ball[i].sp.w) - bm.ball[i].travel;
					bm.ball[i].y = rand() % (YMAX - (int) bm.ball[i].sp.h);
					bm.ball[i].rev = 1;
					bm.ball[i].speed = proximity ? 0 : (rand() % 2) + 1;
				}
			} else {
				bm.ball[i].x += bm.ball[i].speed;
				if (bm.ball[i].x > XMAX + bm.ball[i].travel) {
					/* Off screen : change direction, position & speed */
					bm.ball[i].travel = rand() % XMAX;
					bm.ball[i].x = XMAX + bm.ball[i].travel;
					bm.ball[i].y = rand() % (YMAX - (int) bm.ball[i].sp.h);
					bm.ball[i].rev = 0;
					bm.ball[i].speed = proximity ? 0 : (rand() % 2) + 1;
				}
			}

			/* Random vertical movement */
			j = rand() % 16;
			if (j == 8)
				bm.ball[i].y--;
			else if (j == 12)
				bm.ball[i].y++;
		} else {
			if (!bm.ball[i].rev) {
				bm.ball[i].y -= bm.ball[i].speed;
				if (bm.ball[i].y < -((int) bm.ball[i].sp.h / (int) bm.ball[i].sp.f) - bm.ball[i].travel) {
					/* Off screen : change direction, position & speed */
					bm.ball[i].travel = rand() % YMAX;
					bm.ball[i].y = -((int) bm.ball[i].sp.h / (int) bm.ball[i].sp.f) - bm.ball[i].travel;
					bm.ball[i].x = rand() % (XMAX - (int) bm.ball[i].sp.w);
					bm.ball[i].rev = 1;
					bm.ball[i].speed = proximity ? 0 : (rand() % 2) + 1;
				}
			} else {
				bm.ball[i].y += bm.ball[i].speed;
				if (bm.ball[i].y > YMAX + bm.ball[i].travel) {
					/* Off screen : change direction, position & speed */
					bm.ball[i].travel = rand() % YMAX;
					bm.ball[i].y = YMAX + bm.ball[i].travel;
					bm.ball[i].x = rand() % (XMAX - (int) bm.ball[i].sp.w);
					bm.ball[i].rev = 0;
					bm.ball[i].speed = proximity ? 0 : (rand() % 2) + 1;
				}
			}

			/* Random horizontal movement */
			j = rand() % 16;
			if (j == 8)
				bm.ball[i].x--;
			else if (j == 12)
				bm.ball[i].x++;
		}

		draw_sprite_rgba(bm.ball[i].x, bm.ball[i].y, 0, bm.ball[i].sp);
	}
}

/******************************************/
/* Animate and draw proximity sprites     */
/******************************************/

static void
proximity_update()
{
	int x, y, i;
	int onoff;
	static int onscreen = 0;
	static int dx[15] = { -26, -13, XMAX, -26, -13, XMAX, -26, -13, XMAX, -26, -13, XMAX, -26, -13, XMAX };

	/* Match colour to client status */
	if (seti.status == 'R')
		i = 3;
	else if (seti.status == 'U')
		i = 1;
	else
		i = 2;

	/* Draw ball at mouse position */
	if (proximity) {
		/* Update setiathome statitics three times per second */
		if (!(proximity++ % 10) && stats.dirtotal)
			read_stats(stats.dirseti[stats.dircurrent], &seti);

		gdk_window_get_pointer(bm.win, &x, &y, NULL);

		draw_sprite_rgba(x - bm.ball[i].sp.w, y - bm.ball[i].sp.h, 0, bm.ball[i].sp);
	}

	/* Draw directory after two seconds */
	if (stats.dirtotal > 1 && (proximity > 70 || onscreen)) {
		switch (stats.dircurrent) {
		case 0:
			onoff = 9362;
			break;
		case 1:
			onoff = 29671;
			break;
		case 2:
			onoff = 31207;
			break;
		case 3:
			onoff = 18925;
			break;
		case 4:
			onoff = 31183;
			break;
		case 5:
			onoff = 31695;
			break;
		case 6:
			onoff = 18727;
			break;
		case 7:
			onoff = 31727;
			break;
		case 8:
			onoff = 18927;
			break;
		case 9:
			onoff = 31599;
			break;
		default:
			onoff = 23213;
			break;
		}

		if (!proximity)
			onoff = 0;

		onscreen = 0;

		/* X & Y equations for number digits determined */
		/* by trial and error for specific sprite size  */
		for (y = 0; y < 15; y++) {
			x = 9 + 13 * (y % 3);

			/* If "on" fly to desired position  */
			/* If "off" fly to storage position */
			if ((onoff >> y) % 2) {
				if (dx[y] < x - 2)
					dx[y] = dx[y] + 3;
				else if (dx[y] > x + 2)
					dx[y] = dx[y] - 3;
				else
					dx[y] = x;
			} else {
				if (dx[y] <= (x - rand() % 2) && dx[y] > -((int) bm.ball[i].sp.w) - x)
					dx[y] = dx[y] - 3;
				else if (dx[y] < XMAX + x)
					dx[y] = dx[y] + 3;
			}

			if (dx[y] > -((int) bm.ball[i].sp.w) && dx[y] < XMAX) {
				draw_sprite_rgba(dx[y], 5 + 8 * (y / 3), 0, bm.ball[i].sp);
				onscreen++;
			}
		}
	}
}

/******************************************/
/* Animate and draw random sprites        */
/******************************************/

void
random_update()
{
	static int active = 0;
	static int rotate = 0;
	static int inout = 0;
	static int d = 50;
	int i, x ,y;
	double psi;

	if (active || !(random() % 4096)) {
		active = 1;

		if (d == 51) {
			active = 0;
			rotate = 0;
			inout = 0;
			d = 50;
		}
	
		if ((rotate%5) == 1) {
			if (d == 15)
				inout++;

			if (inout == 10)
				d++;
			else if (inout == 0)
				d--;
		}

		for (i = (rotate/3) % 3; i < 30; i+=3) {
			psi = i * 3.14 / 15.0;
			x = floor(sin(psi) * d) + 22;
			y = floor(-cos(psi) * d) + 22;
			draw_sprite_rgba(x, y, 0, bm.ball[1].sp);
		}

		rotate++;
	}
}

/******************************************/
/* Draw gradient                          */
/******************************************/

void
setup_background()
{
	int i;
	int j = 0;
	int b = 0;
	unsigned char g = 0;

	/* Black to dark blue gradient */
	for (i = 1; i < YMAX; i++) {
		g = g + 2;
		b = i * XMAX * 3 + 2;
		for (j = 0; j < XMAX; j++) {
			bm.rgb_back[b] = g;
			b = b + 3;
		}
	}
}

/******************************************/
/* Setup invisible cursor                 */
/******************************************/

GdkCursor *
setup_cursor()
{
	GdkPixmap *source, *mask;
	GdkColor col = { 0, 0, 0, 0 };
	GdkCursor *cursor;
	unsigned char hide[] = { 0x00 };

	/* No obviously invisible cursor available though */
	/* X/GDK, so using a custom 1x1 bitmap instead    */

	source = gdk_bitmap_create_from_data(NULL, hide, 1, 1);
	mask = gdk_bitmap_create_from_data(NULL, hide, 1, 1);

	cursor = gdk_cursor_new_from_pixmap(source, mask, &col, &col, 0, 0);

	gdk_pixmap_unref(source);
	gdk_pixmap_unref(mask);

	return cursor;
}

/******************************************/
/* Create dock app window                 */
/******************************************/

void
make_wmufo_dockapp()
{
#define MASK GDK_BUTTON_PRESS_MASK | GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK | GDK_POINTER_MOTION_HINT_MASK

	GdkWindowAttr attr;
	GdkWindowAttr attri;
	Window win;
	Window iconwin;

	GdkPixmap *icon;

	XSizeHints sizehints;
	XWMHints wmhints;

	memset(&attr, 0, sizeof (GdkWindowAttr));

	attr.width = 64;
	attr.height = 64;
	attr.title = "wmufo";
	attr.event_mask = MASK;
	attr.wclass = GDK_INPUT_OUTPUT;
	attr.visual = gdk_visual_get_system();
	attr.colormap = gdk_colormap_get_system();
	attr.wmclass_name = "wmufo";
	attr.wmclass_class = "wmufo";
	attr.window_type = GDK_WINDOW_TOPLEVEL;

	/* Make a copy for the iconwin - parameters are the same */
	memcpy(&attri, &attr, sizeof (GdkWindowAttr));
	attri.window_type = GDK_WINDOW_CHILD;

	sizehints.flags = USSize;
	sizehints.width = 64;
	sizehints.height = 64;

	bm.win = gdk_window_new(NULL, &attr, GDK_WA_TITLE | GDK_WA_WMCLASS | GDK_WA_VISUAL | GDK_WA_COLORMAP);
	if (!bm.win) {
		fprintf(stderr, "FATAL: Cannot make toplevel window\n");
		exit(1);
	}

	bm.iconwin = gdk_window_new(bm.win, &attri, GDK_WA_TITLE | GDK_WA_WMCLASS);
	if (!bm.iconwin) {
		fprintf(stderr, "FATAL: Cannot make icon window\n");
		exit(1);
	}

	win = GDK_WINDOW_XWINDOW(bm.win);
	iconwin = GDK_WINDOW_XWINDOW(bm.iconwin);
	XSetWMNormalHints(GDK_WINDOW_XDISPLAY(bm.win), win, &sizehints);

	wmhints.initial_state = WithdrawnState;
	wmhints.icon_window = iconwin;
	wmhints.icon_x = 0;
	wmhints.icon_y = 0;
	wmhints.window_group = win;
	wmhints.flags = StateHint | IconWindowHint | IconPositionHint | WindowGroupHint;

	bm.gc = gdk_gc_new(bm.win);

	bm.pixmap = gdk_pixmap_create_from_xpm_d(bm.win, &(bm.mask), NULL, master_xpm);
	gdk_window_shape_combine_mask(bm.win, bm.mask, 0, 0);
	gdk_window_shape_combine_mask(bm.iconwin, bm.mask, 0, 0);

	gdk_window_set_back_pixmap(bm.win, bm.pixmap, False);
	gdk_window_set_back_pixmap(bm.iconwin, bm.pixmap, False);

#if 0
	gdk_window_set_type_hint(bm.win, GDK_WINDOW_TYPE_HINT_DOCK);
#else
	gdk_window_set_decorations(bm.win, 0);
	gdk_window_set_skip_taskbar_hint(bm.win, 1);
#endif

	icon = gdk_pixmap_create_from_xpm_d(bm.win, NULL, NULL, icon_xpm);
	gdk_window_set_icon(bm.win, bm.iconwin, icon, NULL);

	gdk_window_show(bm.win);

	XSetWMNormalHints(GDK_WINDOW_XDISPLAY(bm.win), win, &sizehints);

	if (bm.x > 0 || bm.y > 0)
		gdk_window_move(bm.win, bm.x, bm.y);
#undef MASK
}

/******************************************/
/* Draw cmap based sprites                */
/******************************************/

static void
draw_sprite_cmap(int x, int y, int f, sprite_cmap sp)
{
	/* Sprite height */
	int sh = sp.h / sp.f;
	/* Bounding box of the clipped sprite */
	int dw, di, dh, ds;
	/* Loop counters */
	int w, h;
	/* Position in sprite data */
	int ps_sp_init;
	/* Postion in rgb buffer */
	int ps_bf_curr, ps_bf_row;
	/* Colour map reference */
	unsigned int c;

	/* Off screen */
	if ((y < -(sh)) || (y > YMAX) || (x > XMAX) || (x < -((int) sp.w)))
		return;

	/* Invalid frame */
	if (f >= sp.f)
		return;

	/* Clip top */
	ds = 0;
	if (y < 0)
		ds = -(y);

	/* Clip bottom */
	dh = sh;
	if ((y + sh) > YMAX)
		dh = YMAX - y;

	/* Clip right */
	dw = (int) sp.w;
	if (x > (XMAX - (int) sp.w))
		dw = XMAX - x;

	/* Clip left */
	di = 0;
	if (x < 0)
		di = -(x);

	/* Offset to current sprite frame */
	ps_sp_init = f * sh * (int) sp.w;

	for (h = ds; h < dh; h++) {
		/* Offset to beginning of current row */
		ps_bf_row = (h + y) * XMAX;
		for (w = di; w < dw; w++) {
			/* Background is transparent - skip */
			if ((c = 3 * sp.data[ps_sp_init + h * (int) sp.w + w]) != 0) {
				ps_bf_curr = (ps_bf_row + x + w) * 3;

				memcpy(&bm.rgb[ps_bf_curr], &sp.cmap[c], 3);

				/* This is equivalent to the orignal method */
				/* below, but faster....well, maybe :-)     */
				/* bm.rgb[ps_bf_curr] = sp.cmap[c];         */
				/* bm.rgb[++ps_bf_curr] = sp.cmap[++c];     */
				/* bm.rgb[++ps_bf_curr] = sp.cmap[++c];     */
			}
		}
	}
}

/******************************************/
/* Draw rgba based sprites                */
/******************************************/

static void
draw_sprite_rgba(int x, int y, int f, sprite_rgba sp)
{
	/* Sprite height */
	int sh = sp.h / sp.f;
	/* Bounding box of clipped sprite */
	int dw, di, dh, ds;
	/* Loop counters */
	int w, h;
	/* Postion in sprite data */
	int ps_sp_init, ps_sp_curr;
	/* Positon in rgb buffer */
	int ps_bf_row, ps_bf_curr;
	/* Alpha value */
	unsigned char a;

	/* Off screen */
	if ((y < -(sh)) || (y > YMAX) || (x > XMAX) || (x < -((int) sp.w)))
		return;

	/* Invalid frame */
	if (f >= sp.f)
		return;

	/* Clip top */
	ds = 0;
	if (y < 0)
		ds = -(y);

	/* Clip bottom */
	dh = sh;
	if ((y + sh) > YMAX)
		dh = YMAX - y;

	/* Clip right */
	dw = (int) sp.w;
	if (x > (XMAX - (int) sp.w))
		dw = XMAX - x;

	/* Clip left */
	di = 0;
	if (x < 0)
		di = -(x);

	/* Offset to current sprite frame */
	ps_sp_init = f * sh * (int) sp.w * (int) sp.b;

	for (h = ds; h < dh; h++) {
		/* Offset to beginning of current row */
		ps_bf_row = (h + y) * XMAX;
		for (w = di; w < dw; w++) {
			ps_sp_curr = ps_sp_init + (h * (int) sp.w + w) * (int) sp.b;
			/* Alpha is fully transparent - skip */
			if ((a = sp.data[ps_sp_curr + 3]) != 0) {
				ps_bf_curr = (ps_bf_row + x + w) * 3;

				bm.rgb[ps_bf_curr] = (a * sp.data[ps_sp_curr] + (256 - a) * bm.rgb[ps_bf_curr]) >> 8;
				bm.rgb[++ps_bf_curr] = (a * sp.data[++ps_sp_curr] + (256 - a) * bm.rgb[ps_bf_curr]) >> 8;
				bm.rgb[++ps_bf_curr] = (a * sp.data[++ps_sp_curr] + (256 - a) * bm.rgb[ps_bf_curr]) >> 8;
			}
		}
	}
}

/******************************************/
/* Draw string                            */
/******************************************/

static void
draw_string(int x, int y, char *string, int kern, sprite_cmap sp)
{
	/* Source is specific to wmufo character map */
	char source[] = " 0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ:./%";
	int letter;
	int i;

	for (i = 0; string[i]; i++) {
		letter = toupper(string[i]);

		/* Check if valid character - else use default */
		if (strchr(source, letter) != 0)
			letter = strchr(source, letter) - source;
		else
			letter = 0;

		draw_sprite_cmap(x, y, letter, sp);
		x = x + sp.w + kern;
	}
}

/******************************************/
/* Draw bar                               */
/******************************************/

static void
draw_bar(int x, int y, int w, int h, int percent)
{
	int i, j, p;
	unsigned char c;

	/* Green to red status bar */
	for (i = x; i < (x + (w * percent / 100)); i++) {
		c = 256 * (i - x) / w;
		for (j = y; j < (y + h); j++) {
			p = 3 * (XMAX * j + i);
			bm.rgb[p] = c;
			bm.rgb[++p] = 255 - c;
			bm.rgb[++p] = 0;
		}
	}
}

/******************************************/
/* Draw statistics                        */
/******************************************/

static void
info_update(int which)
{
	int i, j;
	int ycoord;
	char left[10], right[10];

	for (i = 0; i < 5; i++) {
		ycoord = 6 + 11 * i;

		j = stats.line[i + 5 * which];

		strcpy(right, "\0");

		if (j == STAT_BAR)
			draw_bar(1, 2 + i * 11, 53, bm.text.sp.h / bm.text.sp.f, 100 * seti.progress);
		else {
			switch (j) {
			case STAT_TITLE:
				strcpy(left, "seti");
				if (stats.dirtotal > 1)
					sprintf(right, "%c", 65 + stats.dircurrent);
				break;
			case STAT_PROG:
				sprintf(left, "%.2f%%", seti.progress * 100);
				sprintf(right, "%c", seti.status);
				break;
			case STAT_CPU:
				secs_2_time(seti.cpuunit, left);
				break;
			case STAT_TIMELEFT:
				if (seti.state == SETI_STATE_UPLOAD)
					strcpy(left, "UPLOAD");
				else if (seti.state == SETI_STATE_CONNECT)
					strcpy(left, "CONNECT");
				else if (seti.state == SETI_STATE_DOWNLOAD)
					strcpy(left, "DOWNLOAD");
				else if (seti.progress)
					secs_2_time(seti.cpuunit / seti.progress - seti.cpuunit, left);
				else
					secs_2_time(0, left);
				break;
			case STAT_NWUS:
				strcpy(left, "Rx:");
				sprintf(right, "%i", seti.nwus);
				break;
			case STAT_NRESULTS:
				strcpy(left, "Tx:");
				sprintf(right, "%i", seti.nresults);
				break;
			case STAT_PID:
				strcpy(left, "PID:");
				sprintf(right, "%i", seti.pid);
				break;
			case STAT_TOTALCPU:
				strcpy(left, "Pt:");
				secs_2_epoch(seti.cputotal, right);
				break;
			default:
				strcpy(left, "ERROR");
				break;
			}

			strncat(left, "         ", 9 - strlen(left) - strlen(right));
			strcat(left, right);

			draw_string(1, 2 + i * 11, left, 1, bm.text.sp);
		}
	}
}

/******************************************/
/* Check and add seti@home directory      */
/******************************************/

int
insert_setidir(const char *dir)
{
	int i;
	char *buf;
	DIR *dp;
	FILE *fp;

	if (stats.dirtotal >= SETI_DIRMAX) {
		if (debug)
			fprintf(stderr, "DIR: Too many directories\n");
		return 0;
	}

	if (!(buf = expand_dir(dir))) {
		if (debug)
			fprintf(stderr, "DIR: Failed to expand directory - %s\n", dir);
		return 0;
	}

	if (dp = opendir(buf))
		closedir(dp);
	else {
		if (debug)
			fprintf(stderr, "DIR: Failed to access directory - %s\n", buf);
		free(buf);
		return 0;
	}

	if (stats.dirtotal) {
		for (i = 0; i < stats.dirtotal; i++) {
			if (!strcmp(buf, stats.dirseti[i])) {
				if (debug)
					fprintf(stderr, "DIR: Duplicate directory - %s\n", buf);
				free(buf);
				return 0;
			}
		}
	}

	stats.dirseti[stats.dirtotal] = strdup(buf);
	stats.dirtotal++;
	free(buf);

	return 1;
}

/******************************************/
/* Expand directory                       */
/******************************************/

char *
expand_dir(const char *dir)
{
	char rootdir[128], subdir[128];
	struct passwd pswd;
	struct passwd *pw = &pswd;
	int i = 0;

	/* Split into relevant components */
	while (*(dir + i) != '/' && *(dir + i) != '\0')
		i++;
	strcpy(subdir, dir + i);

	/* Expand "~/" to home directory */
	/* Expand "~other/" to other users home directory */
	if (*dir == '~' && *(dir + 1) == '/') {
		if (!strcpy(rootdir, getenv("HOME"))) {
			if (pw = getpwuid(getuid()))
				strcpy(rootdir, pw->pw_dir);
			else
				return;
		}
	} else if (*dir == '~' && *(dir + 1) != '/') {
		i = 1;
		while (*(dir + i) != '/') {
			rootdir[i - 1] = *(dir + i);
			i++;
		}
		rootdir[i - 1] = '\0';

		if (pw = getpwnam(rootdir))
			strcpy(rootdir, pw->pw_dir);
		else
			return;
	} else
		return strdup(dir);

	strcat(rootdir, subdir);

	return strdup(rootdir);
}

/******************************************/
/* Convert seconds to hh:mm:ss            */
/******************************************/

char *
secs_2_time(const long seconds, char *time)
{
	int h = 0, m = 0, s = 0;

	h = seconds / 3600;
	m = seconds % 3600 / 60;
	s = seconds % 60;

	sprintf(time, "%02d:%02d:%02d", h, m, s);

	return time;
}

/******************************************/
/* Convert seconds to yy:ddd or ddd:hh    */
/******************************************/

char *
secs_2_epoch(const long seconds, char *epoch)
{
	int y = 0, d = 0, h = 0;

	strcpy(epoch, "0/000");

	y = seconds / 31536000;
	d = seconds % 31536000 / 86400;
	h = seconds % 86400 / 3600;

	if (y)
		sprintf(epoch, "%d/%03d", y, d);
	else
		sprintf(epoch, "%d/%02d", d, h);

	return epoch;
}

/******************************************/
/* Read config file                       */
/******************************************/

void
read_config(int argc, char **argv)
{
	int i, j, cmddir = 0;
	char buf[256], left[128], right[128];
	char *file, *c;
	FILE *fp;

	stats.dirtotal = 0;
	stats.dircurrent = 0;
	stats.timeout = 4;
	stats.update = 2500000;

	stats.line[0] = STAT_TITLE;
	stats.line[1] = STAT_PROG;
	stats.line[2] = STAT_CPU;
	stats.line[3] = STAT_TIMELEFT;
	stats.line[4] = STAT_NRESULTS;
	stats.line[5] = STAT_TITLE;
	stats.line[6] = STAT_PROG;
	stats.line[7] = STAT_CPU;
	stats.line[8] = STAT_BAR;
	stats.line[9] = STAT_TOTALCPU;

	while ((i = getopt(argc, argv, "hvbs:d:p:t:g:S:")) != -1) {
		switch (i) {
		case 'v':
			debug = 1;
			break;
		case 'h':
			do_help();
			exit(1);
		}
	}
	optind = 0;

	file = expand_dir("~/.wmsetirc");

	if (fp = fopen(file, "r")) {
		while (fgets(buf, sizeof (buf), fp)) {
			if (sscanf(buf, "%s %s", &left, &right) == 2) {
				if (atoi(left) > 0 && atoi(left) <= 10) {
					i = atoi(left) - 1;
					if (!strcasecmp(right, "title"))
						stats.line[i] = STAT_TITLE;
					else if (!strcasecmp(right, "prog"))
						stats.line[i] = STAT_PROG;
					else if (!strcasecmp(right, "cpu"))
						stats.line[i] = STAT_CPU;
					else if (!strcasecmp(right, "timeleft"))
						stats.line[i] = STAT_TIMELEFT;
					else if (!strcasecmp(right, "nwus"))
						stats.line[i] = STAT_NWUS;
					else if (!strcasecmp(right, "nresults"))
						stats.line[i] = STAT_NRESULTS;
					else if (!strcasecmp(right, "pid"))
						stats.line[i] = STAT_PID;
					else if (!strcasecmp(right, "totalcpu"))
						stats.line[i] = STAT_TOTALCPU;
					else if (!strcasecmp(right, "bar"))
						stats.line[i] = STAT_BAR;
				} else if (!strcasecmp(left, "setidir")) {
					if (c = strchr(right, '\n'))
						*c = 0;
					insert_setidir(right);
				} else if (!strcasecmp(left, "setiprg")) {
					if (c = strchr(buf, '\n'))
						*c = 0;
					stats.program = expand_dir(strstr(buf, right));
				} else if (!strcasecmp(left, "update"))
					stats.update = atoi(right) * 1000;
				else if (!strcasecmp(left, "timeout"))
					stats.timeout = atoi(right) * 1000;
			}
		}
		fclose(fp);
	}

	free(file);

	while ((i = getopt(argc, argv, "hvbs:d:p:t:g:S:")) != -1) {
		switch (i) {
		case 'S':
			if (optarg)
				session_id = strdup(optarg);
			break;
		case 'g':
			if (optarg) {
				j = XParseGeometry(optarg, &bm.x, &bm.y, &j, &j);

				if (j & XNegative)
					bm.x = gdk_screen_width() - 64 + bm.x;
				if (j & YNegative)
					bm.y = gdk_screen_height() - 64 + bm.y;
			}
			break;
		case 's':
			if (optarg)
				stats.update = atoi(optarg) * 1000;
			break;
		case 'd':
			if (optarg) {
				if (cmddir == 0) {
					for (j = 0; j < stats.dirtotal; j++)
						free(stats.dirseti[j]);
					stats.dirtotal = 0;
					cmddir = 1;
				}
				insert_setidir(optarg);
			}
			break;
		case 'p':
			if (optarg) {
				if (stats.program)
					free(stats.program);
				stats.program = expand_dir(optarg);
			}
			break;
		case 't':
			if (optarg)
				stats.timeout = atoi(optarg);
			break;
		case 'v':
			break;
		case 'b':
			broken_wm = 1;
			break;
		case 'h':
		default:
			do_help();
			exit(1);
		}
	}

	if (!stats.program)
		stats.program = strdup("setiathome");

	if (!stats.dirtotal) {
		if (!insert_setidir("~/setiathome") && debug)
			fprintf(stderr, "DIR: Default directory failed, no setiathome\n");
	}

	if (debug) {
		fprintf(stderr, "P: %s\n", stats.program);
		for (i = 0; i < stats.dirtotal; i++)
			fprintf(stderr, "D%d: %s\n", i, stats.dirseti[i]);
		for (i = 0; i < 10; i++)
			fprintf(stderr, "S%d: %d\n", i, stats.line[i]);
	}
}

/******************************************/
/* Help                                   */
/******************************************/

void
do_help()
{
	fprintf(stderr, "\nWmufo - SETI@Home Monitor Dock V %s\n\n", VERSION);
	fprintf(stderr, "Usage: wmufo [ options... ]\n\n");
	fprintf(stderr, "\t-g [{+-}X{+-}Y]\t\tinital window position\n");
	fprintf(stderr, "\t-p [...]\t\tspecify command line for running setiathome\n");
	fprintf(stderr, "\t-d [...]\t\tspecify a SETI@home directory (overrides config)\n");
	fprintf(stderr, "\t\t\t\tcan be applied multiple times: -d [...] -d [...]\n");
	fprintf(stderr, "\t-s [...]\t\tupdate seti text statistcs  (default:%dms)\n", stats.update / 1000);
	fprintf(stderr, "\t-t [...]\t\ttimeout from text cycle [disable:0]  (default:%d)\n", stats.timeout);
	fprintf(stderr, "\t-v\t\t\tverbose debugging messages\n");
	fprintf(stderr, "\t-b\t\t\tactivate broken window manager fix\n");
	fprintf(stderr, "\t-h\t\t\tprints this help\n");
}
