#include "driver.h"
#include "common.h"

INLINE UINT8 NIBBLE(UINT32 n, UINT8 a, UINT8 b, UINT8 c, UINT8 d)
{
	UINT32 a1 = (n >> a) & 0x1;
	UINT32 b1 = (n >> b) & 0x1;
	UINT32 c1 = (n >> c) & 0x1;
	UINT32 d1 = (n >> d) & 0x1;
	return ((a1 << 3) | (b1 << 2) | (c1 << 1) | d1);
}

INLINE UINT8 BITS5(UINT32 n, UINT8 a, UINT8 b, UINT8 c, UINT8 d, UINT8 e)
{
	UINT32 a1 = (n >> a) & 0x1;
	UINT32 b1 = (n >> b) & 0x1;
	UINT32 c1 = (n >> c) & 0x1;
	UINT32 d1 = (n >> d) & 0x1;
	UINT32 e1 = (n >> e) & 0x1;
	return ((a1 << 4) | (b1 << 3) | (c1 << 2) | (d1 << 1) | e1);
}


static unsigned char crypt_table1[][16] =
{
	/*  0,   1,   2,   3,   4,   5,   6,   7,   8,   9,   A,   B,   C,   D,   E,   F */
	{ 0x2, 0xe, 0x6, 0x9, 0x3, 0xf, 0x7, 0x8, 0x1, 0xd, 0x5, 0xb, 0x0, 0xc, 0x4, 0xa }
};

static unsigned char crypt_table2[][32] =
{
	/*    0,   1,   2,   3,   4,   5,   6,   7,   8,   9,   A,   B,   C,   D,   E,   F */
	{
		0x9, 0x8, 0xf, 0xe, 0xb, 0xa, 0xc, 0xd, 0x1, 0x0, 0x7, 0x6, 0x3, 0x2, 0x4, 0x5,
		0x9, 0x8, 0xf, 0xe, 0xb, 0xa, 0xc, 0xd, 0x1, 0x0, 0x7, 0x6, 0x3, 0x2, 0x4, 0x5,
	},
	{
		0x9, 0x8, 0xf, 0xe, 0xb, 0xa, 0xc, 0xd, 0x1, 0x0, 0x7, 0x2, 0x3, 0x6, 0x4, 0x5,
		0x9, 0x8, 0xf, 0x6, 0xb, 0x2, 0xc, 0xd, 0x1, 0xe, 0x7, 0xa, 0x3, 0x0, 0x4, 0x5,
	},
};

static unsigned char crypt_table3[][32] =
{
	/*    0,   1,   2,   3,   4,   5,   6,   7,   8,   9,   A,   B,   C,   D,   E,   F */
	{
		0x2, 0x6, 0xa, 0xe, 0xa, 0xe, 0x2, 0x6, 0xb, 0xf, 0x3, 0x7, 0xb, 0xf, 0x3, 0x7,
		0x0, 0x4, 0x8, 0xc, 0x8, 0xc, 0x0, 0x4, 0x9, 0xd, 0x1, 0x5, 0x9, 0xd, 0x1, 0x5,
	},
	{
		0xc, 0x8, 0x4, 0x0, 0xc, 0x8, 0x4, 0x0, 0xd, 0x9, 0x5, 0x1, 0x5, 0x1, 0xd, 0x9,
		0xf, 0xb, 0x7, 0x3, 0x7, 0x3, 0xf, 0xb, 0xe, 0xa, 0x6, 0x2, 0xe, 0xa, 0x6, 0x2,
	},
};

static unsigned char crypt_table4[][16] =
{
	/*  0,   1,   2,   3,   4,   5,   6,   7,   8,   9,   A,   B,   C,   D,   E,   F */
	{ 0x0, 0x8, 0x4, 0xc, 0x2, 0xa, 0x6, 0xe, 0x1, 0x9, 0x5, 0xd, 0x3, 0xb, 0x7, 0xf },
	{ 0x8, 0x4, 0xc, 0x2, 0xa, 0x6, 0xe, 0x1, 0x9, 0x5, 0xd, 0x3, 0xb, 0x7, 0xf, 0x0 },
	{ 0x4, 0xc, 0x2, 0xa, 0x6, 0xe, 0x1, 0x9, 0x5, 0xd, 0x3, 0xb, 0x7, 0xf, 0x0, 0x8 },
	{ 0xc, 0x2, 0xa, 0x6, 0xe, 0x1, 0x9, 0x5, 0xd, 0x3, 0xb, 0x7, 0xf, 0x0, 0x8, 0x4 },

	{ 0x2, 0xa, 0x6, 0xe, 0x1, 0x9, 0x5, 0xd, 0x3, 0xb, 0x7, 0xf, 0x0, 0x8, 0x4, 0xc },
	{ 0xa, 0x6, 0xe, 0x1, 0x9, 0x5, 0xd, 0x3, 0xb, 0x7, 0xf, 0x0, 0x8, 0x4, 0xc, 0x2 },
	{ 0x6, 0xe, 0x1, 0x9, 0x5, 0xd, 0x3, 0xb, 0x7, 0xf, 0x0, 0x8, 0x4, 0xc, 0x2, 0xa },
	{ 0xe, 0x1, 0x9, 0x5, 0xd, 0x3, 0xb, 0x7, 0xf, 0x0, 0x8, 0x4, 0xc, 0x2, 0xa, 0x6 },

	{ 0x1, 0x9, 0x5, 0xd, 0x3, 0xb, 0x7, 0xf, 0x0, 0x8, 0x4, 0xc, 0x2, 0xa, 0x6, 0xe },
	{ 0x9, 0x5, 0xd, 0x3, 0xb, 0x7, 0xf, 0x0, 0x8, 0x4, 0xc, 0x2, 0xa, 0x6, 0xe, 0x1 },
	{ 0x5, 0xd, 0x3, 0xb, 0x7, 0xf, 0x0, 0x8, 0x4, 0xc, 0x2, 0xa, 0x6, 0xe, 0x1, 0x9 },
	{ 0xd, 0x3, 0xb, 0x7, 0xf, 0x0, 0x8, 0x4, 0xc, 0x2, 0xa, 0x6, 0xe, 0x1, 0x9, 0x5 },

	{ 0x3, 0xb, 0x7, 0xf, 0x0, 0x8, 0x4, 0xc, 0x2, 0xa, 0x6, 0xe, 0x1, 0x9, 0x5, 0xd },
	{ 0xb, 0x7, 0xf, 0x0, 0x8, 0x4, 0xc, 0x2, 0xa, 0x6, 0xe, 0x1, 0x9, 0x5, 0xd, 0x3 },
	{ 0x7, 0xf, 0x0, 0x8, 0x4, 0xc, 0x2, 0xa, 0x6, 0xe, 0x1, 0x9, 0x5, 0xd, 0x3, 0xb },
	{ 0xf, 0x0, 0x8, 0x4, 0xc, 0x2, 0xa, 0x6, 0xe, 0x1, 0x9, 0x5, 0xd, 0x3, 0xb, 0x7 },
};

static unsigned char crypt_table5[][16] =
{
	/*  0,   1,   2,   3,   4,   5,   6,   7,   8,   9,   A,   B,   C,   D,   E,   F */
	{ 0x9, 0x1, 0xd, 0x5, 0xa, 0x2, 0xe, 0x6, 0x8, 0x0, 0xc, 0x4, 0xb, 0x3, 0xf, 0x7 },
	{ 0x1, 0xd, 0x5, 0x9, 0xa, 0x2, 0xe, 0x6, 0x8, 0x0, 0xc, 0x4, 0x3, 0xf, 0x7, 0xb },
	{ 0x1, 0xd, 0x5, 0x9, 0x2, 0xe, 0x6, 0xa, 0x0, 0xc, 0x4, 0x8, 0x3, 0xf, 0x7, 0xb },
	{ 0xd, 0x5, 0x9, 0x1, 0x2, 0xe, 0x6, 0xa, 0x0, 0xc, 0x4, 0x8, 0xf, 0x7, 0xb, 0x3 },
	{ 0xd, 0x5, 0x9, 0x1, 0xe, 0x6, 0xa, 0x2, 0xc, 0x4, 0x8, 0x0, 0xf, 0x7, 0xb, 0x3 },
	{ 0x5, 0x9, 0x1, 0xd, 0xe, 0x6, 0xa, 0x2, 0xc, 0x4, 0x8, 0x0, 0x7, 0xb, 0x3, 0xf },
	{ 0x5, 0x9, 0x1, 0xd, 0x6, 0xa, 0x2, 0xe, 0x4, 0x8, 0x0, 0xc, 0x7, 0xb, 0x3, 0xf },
	{ 0x9, 0x1, 0xd, 0x5, 0x6, 0xa, 0x2, 0xe, 0x4, 0x8, 0x0, 0xc, 0xb, 0x3, 0xf, 0x7 },

	{ 0xa, 0x2, 0xe, 0x6, 0x8, 0x0, 0xc, 0x4, 0xb, 0x3, 0xf, 0x7, 0x9, 0x1, 0xd, 0x5 },
	{ 0xa, 0x2, 0xe, 0x6, 0x8, 0x0, 0xc, 0x4, 0x3, 0xf, 0x7, 0xb, 0x1, 0xd, 0x5, 0x9 },
	{ 0x2, 0xe, 0x6, 0xa, 0x0, 0xc, 0x4, 0x8, 0x3, 0xf, 0x7, 0xb, 0x1, 0xd, 0x5, 0x9 },
	{ 0x2, 0xe, 0x6, 0xa, 0x0, 0xc, 0x4, 0x8, 0xf, 0x7, 0xb, 0x3, 0xd, 0x5, 0x9, 0x1 },
	{ 0xe, 0x6, 0xa, 0x2, 0xc, 0x4, 0x8, 0x0, 0xf, 0x7, 0xb, 0x3, 0xd, 0x5, 0x9, 0x1 },
	{ 0xe, 0x6, 0xa, 0x2, 0xc, 0x4, 0x8, 0x0, 0x7, 0xb, 0x3, 0xf, 0x5, 0x9, 0x1, 0xd },
	{ 0x6, 0xa, 0x2, 0xe, 0x4, 0x8, 0x0, 0xc, 0x7, 0xb, 0x3, 0xf, 0x5, 0x9, 0x1, 0xd },
	{ 0x6, 0xa, 0x2, 0xe, 0x4, 0x8, 0x0, 0xc, 0xb, 0x3, 0xf, 0x7, 0x9, 0x1, 0xd, 0x5 },

	{ 0x8, 0x0, 0xc, 0x4, 0xb, 0x3, 0xf, 0x7, 0x9, 0x1, 0xd, 0x5, 0xa, 0x2, 0xe, 0x6 },
	{ 0x8, 0x0, 0xc, 0x4, 0x3, 0xf, 0x7, 0xb, 0x1, 0xd, 0x5, 0x9, 0xa, 0x2, 0xe, 0x6 },
	{ 0x0, 0xc, 0x4, 0x8, 0x3, 0xf, 0x7, 0xb, 0x1, 0xd, 0x5, 0x9, 0x2, 0xe, 0x6, 0xa },
	{ 0x0, 0xc, 0x4, 0x8, 0xf, 0x7, 0xb, 0x3, 0xd, 0x5, 0x9, 0x1, 0x2, 0xe, 0x6, 0xa },
	{ 0xc, 0x4, 0x8, 0x0, 0xf, 0x7, 0xb, 0x3, 0xd, 0x5, 0x9, 0x1, 0xe, 0x6, 0xa, 0x2 },
	{ 0xc, 0x4, 0x8, 0x0, 0x7, 0xb, 0x3, 0xf, 0x5, 0x9, 0x1, 0xd, 0xe, 0x6, 0xa, 0x2 },
	{ 0x4, 0x8, 0x0, 0xc, 0x7, 0xb, 0x3, 0xf, 0x5, 0x9, 0x1, 0xd, 0x6, 0xa, 0x2, 0xe },
	{ 0x4, 0x8, 0x0, 0xc, 0xb, 0x3, 0xf, 0x7, 0x9, 0x1, 0xd, 0x5, 0x6, 0xa, 0x2, 0xe },

	{ 0xb, 0x3, 0xf, 0x7, 0x9, 0x1, 0xd, 0x5, 0xa, 0x2, 0xe, 0x6, 0x8, 0x0, 0xc, 0x4 },
	{ 0x3, 0xf, 0x7, 0xb, 0x1, 0xd, 0x5, 0x9, 0xa, 0x2, 0xe, 0x6, 0x8, 0x0, 0xc, 0x4 },
	{ 0x3, 0xf, 0x7, 0xb, 0x1, 0xd, 0x5, 0x9, 0x2, 0xe, 0x6, 0xa, 0x0, 0xc, 0x4, 0x8 },
	{ 0xf, 0x7, 0xb, 0x3, 0xd, 0x5, 0x9, 0x1, 0x2, 0xe, 0x6, 0xa, 0x0, 0xc, 0x4, 0x8 },
	{ 0xf, 0x7, 0xb, 0x3, 0xd, 0x5, 0x9, 0x1, 0xe, 0x6, 0xa, 0x2, 0xc, 0x4, 0x8, 0x0 },
	{ 0x7, 0xb, 0x3, 0xf, 0x5, 0x9, 0x1, 0xd, 0xe, 0x6, 0xa, 0x2, 0xc, 0x4, 0x8, 0x0 },
	{ 0x7, 0xb, 0x3, 0xf, 0x5, 0x9, 0x1, 0xd, 0x6, 0xa, 0x2, 0xe, 0x4, 0x8, 0x0, 0xc },
	{ 0xb, 0x3, 0xf, 0x7, 0x9, 0x1, 0xd, 0x5, 0x6, 0xa, 0x2, 0xe, 0x4, 0x8, 0x0, 0xc },
};

static unsigned char crypt_table6[][16] =
{
	/*  0,   1,   2,   3,   4,   5,   6,   7,   8,   9,   A,   B,   C,   D,   E,   F */
	{ 0x5, 0x9, 0x1, 0xf, 0x7, 0xb, 0x3, 0xd, 0x4, 0x8, 0x0, 0xe, 0x6, 0xa, 0x2, 0xc },
	{ 0x9, 0x1, 0xf, 0x7, 0xb, 0x3, 0xd, 0x5, 0x8, 0x0, 0xe, 0x6, 0xa, 0x2, 0xc, 0x4 },
	{ 0x1, 0xf, 0x7, 0xb, 0x3, 0xd, 0x5, 0x9, 0x0, 0xe, 0x6, 0xa, 0x2, 0xc, 0x4, 0x8 },
	{ 0xe, 0x6, 0xa, 0x2, 0xc, 0x4, 0x8, 0x0, 0xf, 0x7, 0xb, 0x3, 0xd, 0x5, 0x9, 0x1 },

	{ 0x6, 0xa, 0x2, 0xc, 0x4, 0x8, 0x0, 0xe, 0x7, 0xb, 0x3, 0xd, 0x5, 0x9, 0x1, 0xf },
	{ 0xa, 0x2, 0xc, 0x4, 0x8, 0x0, 0xe, 0x6, 0xb, 0x3, 0xd, 0x5, 0x9, 0x1, 0xf, 0x7 },
	{ 0x2, 0xc, 0x4, 0x8, 0x0, 0xe, 0x6, 0xa, 0x3, 0xd, 0x5, 0x9, 0x1, 0xf, 0x7, 0xb },
	{ 0xc, 0x4, 0x8, 0x0, 0xe, 0x6, 0xa, 0x2, 0xd, 0x5, 0x9, 0x1, 0xf, 0x7, 0xb, 0x3 },

	{ 0x4, 0x8, 0x0, 0xe, 0x6, 0xa, 0x2, 0xc, 0x5, 0x9, 0x1, 0xf, 0x7, 0xb, 0x3, 0xd },
	{ 0x8, 0x0, 0xe, 0x6, 0xa, 0x2, 0xc, 0x4, 0x9, 0x1, 0xf, 0x7, 0xb, 0x3, 0xd, 0x5 },
	{ 0x0, 0xe, 0x6, 0xa, 0x2, 0xc, 0x4, 0x8, 0x1, 0xf, 0x7, 0xb, 0x3, 0xd, 0x5, 0x9 },
	{ 0xf, 0x7, 0xb, 0x3, 0xd, 0x5, 0x9, 0x1, 0xe, 0x6, 0xa, 0x2, 0xc, 0x4, 0x8, 0x0 },

	{ 0x7, 0xb, 0x3, 0xd, 0x5, 0x9, 0x1, 0xf, 0x6, 0xa, 0x2, 0xc, 0x4, 0x8, 0x0, 0xe },
	{ 0xb, 0x3, 0xd, 0x5, 0x9, 0x1, 0xf, 0x7, 0xa, 0x2, 0xc, 0x4, 0x8, 0x0, 0xe, 0x6 },
	{ 0x3, 0xd, 0x5, 0x9, 0x1, 0xf, 0x7, 0xb, 0x2, 0xc, 0x4, 0x8, 0x0, 0xe, 0x6, 0xa },
	{ 0xd, 0x5, 0x9, 0x1, 0xf, 0x7, 0xb, 0x3, 0xc, 0x4, 0x8, 0x0, 0xe, 0x6, 0xa, 0x2 },
};

static UINT8 dc1(UINT32 b, int adr)
{
	int t;
	UINT8 db, nb;
	nb = NIBBLE(b, 19,18,9,5);

	t = 0;
	db = crypt_table1[t][nb];
	return db;
}

static UINT8 dc2(UINT32 b, int adr)
{
	int t;
	UINT8 db, nb;
	nb = BITS5(b, 21,20,17,16,10);

	t = 0;
	if (adr > 0x8000-0x450)
		t = 1;
	db = crypt_table2[t][nb];
	return db;
}

static UINT8 dc3(UINT32 b, int adr)
{
	int t;
	UINT8 db, nb;
	nb = BITS5(b, 22,21,15,11,6);

	t = 0;
	if (adr > 0x8000-0x450)
		t = 1;
	db = crypt_table3[t][nb];
	return db;
}

static UINT8 dc4(UINT32 b, int adr)
{
	int t;
	UINT8 nb = NIBBLE(b, 15,14,4,23);
	UINT8 db;

	t = 0;
	t |= ((adr + 0x450) >> 12) & 0xf;

	db = crypt_table4[t][nb];
	return db;
}

static UINT8 dc5(UINT32 b, int adr)
{
	int t;
	UINT8 nb = NIBBLE(b, 0,1,7,8);
	UINT8 db;

	t = 0;
	t |= ((adr + 0x50) >> 7) & 0x1f;
	db = crypt_table5[t][nb];

	return db;
}

static UINT8 dc6(UINT32 b, int adr)
{
	int t;
	UINT8 nb = NIBBLE(b, 13,12,3,2);
	UINT8 db;

	t = (adr >> 4) & 0xf;

	db = crypt_table6[t][nb];
	return db;
}

void seibuspi_text_decrypt(unsigned char *rom)
{
	int i;
	for(i=0; i<0x10000; i++) {
		UINT8 r1,r2,r3,r4,r5,r6;
		UINT32 w;

		w = (rom[(i*3) + 0] << 16) | (rom[(i*3) + 1] << 8) | (rom[(i*3) +2]);

		r1 = dc1(w, i);
		r2 = dc2(w, i);
		r3 = dc3(w, i);
		r4 = dc4(w, i);
		r5 = dc5(w, i);
		r6 = dc6(w, i);

		rom[(i*3) + 0] = (r1 << 4) | r2;
		rom[(i*3) + 1] = (r3 << 4) | r4;
		rom[(i*3) + 2] = (r5 << 4) | r6;
	}
}

void seibuspi_bg_decrypt(unsigned char *rom, int size)
{
	int i,j;
	for(j=0; j<size; j+=0xc0000) {
		for(i=0; i<0x40000; i++) {
			UINT8 r1,r2,r3,r4,r5,r6;
			UINT32 w;

			w = (rom[j + (i*3) + 0] << 16) | (rom[j + (i*3) + 1] << 8) | (rom[j + (i*3) + 2]);

			r1 = dc1(w, i/4);
			r2 = dc2(w, i/4);
			r3 = dc3(w, i/4);
			r4 = dc4(w, i/4);
			r5 = dc5(w, i/4);
			r6 = dc6(w, i/4);

			rom[j + (i*3) + 0] = (r1 << 4) | r2;
			rom[j + (i*3) + 1] = (r3 << 4) | r4;
			rom[j + (i*3) + 2] = (r5 << 4) | r6;
		}
	}
}



/* RISE10 (Raiden Fighters 2) */

static unsigned char rise10_crypt_table1[][16] =
{
	{ 0x9, 0x1, 0x8, 0x0, 0xb, 0x3, 0xa, 0x2, 0xd, 0x5, 0xc, 0x4, 0xf, 0x7, 0xe, 0x6 }
};

static unsigned char rise10_crypt_table2[][16] =
{
	{ 0x6, 0x2, 0xc, 0x8, 0xa, 0xe, 0x4, 0x0, 0xf, 0xb, 0xd, 0x9, 0x7, 0x3, 0x5, 0x1 }
};

static unsigned char rise10_crypt_table3[][16] =
{
	{ 0x2, 0x4, 0x3, 0x5, 0x0, 0x7, 0x1, 0x6, 0xa, 0xc, 0xb, 0xd, 0x8, 0xf, 0x9, 0xe },
	{ 0xc, 0x8, 0xd, 0x9, 0xf, 0xb, 0xe, 0xa, 0x4, 0x0, 0x5, 0x1, 0x7, 0x3, 0x6, 0x2 }
};

static unsigned char rise10_crypt_table4[][16] =
{
	/*  0,   1,   2,   3,   4,   5,   6,   7,   8,   9,   A,   B,   C,   D,   E,   F */
	{ 0xd, 0x9, 0x5, 0x1, 0xf, 0xb, 0x7, 0x3, 0xc, 0x8, 0x4, 0x0, 0xe, 0xa, 0x6, 0x2 },
	{ 0x5, 0x1, 0xd, 0x9, 0x7, 0x3, 0xf, 0xb, 0xc, 0x8, 0x4, 0x0, 0xe, 0xa, 0x6, 0x2 },
	{ 0x5, 0x1, 0xd, 0x9, 0x7, 0x3, 0xf, 0xb, 0x4, 0x0, 0xc, 0x8, 0x6, 0x2, 0xe, 0xa },
	{ 0xd, 0x9, 0x5, 0x1, 0xf, 0xb, 0x7, 0x3, 0x4, 0x0, 0xc, 0x8, 0x6, 0x2, 0xe, 0xa },
	{ 0x9, 0xf, 0x1, 0x7, 0xb, 0xd, 0x3, 0x5, 0x8, 0xe, 0x0, 0x6, 0xa, 0xc, 0x2, 0x4 },
	{ 0x1, 0x7, 0x9, 0xf, 0x3, 0x5, 0xb, 0xd, 0x8, 0xe, 0x0, 0x6, 0xa, 0xc, 0x2, 0x4 },
	{ 0x1, 0x7, 0x9, 0xf, 0x3, 0x5, 0xb, 0xd, 0x0, 0x6, 0x8, 0xe, 0x2, 0x4, 0xa, 0xc },
	{ 0x9, 0xf, 0x1, 0x7, 0xb, 0xd, 0x3, 0x5, 0x0, 0x6, 0x8, 0xe, 0x2, 0x4, 0xa, 0xc },
	{ 0xf, 0xb, 0x7, 0x3, 0xd, 0x9, 0x5, 0x1, 0xe, 0xa, 0x6, 0x8, 0xc, 0x2, 0x4, 0x0 },
	{ 0x7, 0x3, 0xf, 0xb, 0x5, 0x1, 0xd, 0x9, 0xe, 0xa, 0x6, 0x8, 0xc, 0x2, 0x4, 0x0 },
	{ 0x7, 0x3, 0xf, 0xb, 0x5, 0x1, 0xd, 0x9, 0x6, 0x8, 0xe, 0xa, 0x4, 0x0, 0xc, 0x2 },
	{ 0xf, 0xb, 0x7, 0x3, 0xd, 0x9, 0x5, 0x1, 0x6, 0x8, 0xe, 0xa, 0x4, 0x0, 0xc, 0x2 },

	{ 0xf, 0xb, 0x7, 0x3, 0xd, 0x9, 0x5, 0x1, 0xe, 0xa, 0x6, 0x8, 0xc, 0x2, 0x4, 0x0 },
	{ 0x7, 0x3, 0xf, 0xb, 0x5, 0x1, 0xd, 0x9, 0xe, 0xa, 0x6, 0x8, 0xc, 0x2, 0x4, 0x0 },
	{ 0x7, 0x3, 0xf, 0xb, 0x5, 0x1, 0xd, 0x9, 0x6, 0x8, 0xe, 0xa, 0x4, 0x0, 0xc, 0x2 },
	{ 0xf, 0xb, 0x7, 0x3, 0xd, 0x9, 0x5, 0x1, 0x6, 0x8, 0xe, 0xa, 0x4, 0x0, 0xc, 0x2 },
	{ 0xb, 0xd, 0x3, 0x5, 0x9, 0xf, 0x1, 0x7, 0xa, 0xc, 0x4, 0x2, 0x8, 0xe, 0x0, 0x6 },
	{ 0x3, 0x5, 0xb, 0xd, 0x1, 0x7, 0x9, 0xf, 0xa, 0xc, 0x4, 0x2, 0x8, 0xe, 0x0, 0x6 },
	{ 0x3, 0x5, 0xb, 0xd, 0x1, 0x7, 0x9, 0xf, 0x4, 0x2, 0xa, 0xc, 0x0, 0x6, 0x8, 0xe },
	{ 0xb, 0xd, 0x3, 0x5, 0x9, 0xf, 0x1, 0x7, 0x4, 0x2, 0xa, 0xc, 0x0, 0x6, 0x8, 0xe },
	{ 0xc, 0x0, 0x4, 0x8, 0xe, 0xa, 0x6, 0x2, 0xd, 0x9, 0x5, 0x1, 0xf, 0xb, 0x7, 0x3 },
	{ 0xc, 0x0, 0x4, 0x8, 0xe, 0xa, 0x6, 0x2, 0x5, 0x1, 0xd, 0x9, 0x7, 0x3, 0xf, 0xb },
	{ 0x4, 0x8, 0xc, 0x0, 0x6, 0x2, 0xe, 0xa, 0x5, 0x1, 0xd, 0x9, 0x7, 0x3, 0xf, 0xb },
	{ 0x4, 0x8, 0xc, 0x0, 0x6, 0x2, 0xe, 0xa, 0xd, 0x9, 0x5, 0x1, 0xf, 0xb, 0x7, 0x3 },


	{ 0xc, 0x0, 0x4, 0x8, 0xe, 0xa, 0x6, 0x2, 0xd, 0x9, 0x5, 0x1, 0xf, 0xb, 0x7, 0x3 },
	{ 0xc, 0x0, 0x4, 0x8, 0xe, 0xa, 0x6, 0x2, 0x5, 0x1, 0xd, 0x9, 0x7, 0x3, 0xf, 0xb },
	{ 0x4, 0x8, 0xc, 0x0, 0x6, 0x2, 0xe, 0xa, 0x5, 0x1, 0xd, 0x9, 0x7, 0x3, 0xf, 0xb },
	{ 0x4, 0x8, 0xc, 0x0, 0x6, 0x2, 0xe, 0xa, 0xd, 0x9, 0x5, 0x1, 0xf, 0xb, 0x7, 0x3 },
	{ 0xa, 0xe, 0x2, 0x6, 0x8, 0xc, 0x0, 0x4, 0xb, 0xf, 0x3, 0x7, 0x9, 0xd, 0x1, 0x5 },
	{ 0xa, 0xe, 0x2, 0x6, 0x8, 0xc, 0x0, 0x4, 0x3, 0x7, 0xb, 0xf, 0x1, 0x5, 0x9, 0xd },
	{ 0x2, 0x6, 0xa, 0xe, 0x0, 0x4, 0x8, 0xc, 0x3, 0x7, 0xb, 0xf, 0x1, 0x5, 0x9, 0xd },
	{ 0x2, 0x6, 0xa, 0xe, 0x0, 0x4, 0x8, 0xc, 0xb, 0xf, 0x3, 0x7, 0x9, 0xd, 0x1, 0x5 },
	{ 0xe, 0xa, 0x6, 0x2, 0xc, 0x8, 0x4, 0x0, 0xf, 0xb, 0x7, 0x3, 0xd, 0x9, 0x5, 0x1 },
	{ 0xe, 0xa, 0x6, 0x2, 0xc, 0x8, 0x4, 0x0, 0x7, 0x3, 0xf, 0xb, 0x5, 0x1, 0xd, 0x9 },
	{ 0x6, 0x2, 0xe, 0xa, 0x4, 0x0, 0xc, 0x8, 0x7, 0x3, 0xf, 0xb, 0x5, 0x1, 0xd, 0x9 },
	{ 0x6, 0x2, 0xe, 0xa, 0x4, 0x0, 0xc, 0x8, 0xf, 0xb, 0x7, 0x3, 0xd, 0x9, 0x5, 0x1 },

	{ 0xe, 0xa, 0x6, 0x2, 0xc, 0x8, 0x4, 0x0, 0xf, 0xb, 0x7, 0x3, 0xd, 0x9, 0x5, 0x1 },
	{ 0xe, 0xa, 0x6, 0x2, 0xc, 0x8, 0x4, 0x0, 0x7, 0x3, 0xf, 0xb, 0x5, 0x1, 0xd, 0x9 },
	{ 0x6, 0x2, 0xe, 0xa, 0x4, 0x0, 0xc, 0x8, 0x7, 0x3, 0xf, 0xb, 0x5, 0x1, 0xd, 0x9 },
	{ 0x6, 0x2, 0xe, 0xa, 0x4, 0x0, 0xc, 0x8, 0xf, 0xb, 0x7, 0x3, 0xd, 0x9, 0x5, 0x1 },
	{ 0xa, 0xc, 0x2, 0x4, 0x8, 0xe, 0x0, 0x6, 0xb, 0xd, 0x3, 0x5, 0x9, 0xf, 0x1, 0x7 },
	{ 0xa, 0xc, 0x2, 0x4, 0x8, 0xe, 0x0, 0x6, 0x3, 0x5, 0xb, 0xd, 0x1, 0x7, 0x9, 0xf },
	{ 0x2, 0x4, 0xa, 0xc, 0x0, 0x6, 0x8, 0xe, 0x3, 0x5, 0xb, 0xd, 0x1, 0x7, 0x9, 0xf },
	{ 0x2, 0x4, 0xa, 0xc, 0x0, 0x6, 0x8, 0xe, 0xb, 0xd, 0x3, 0x5, 0x9, 0xf, 0x1, 0x7 },
	{ 0xd, 0x9, 0x5, 0x1, 0xf, 0xb, 0x7, 0x3, 0xc, 0x8, 0x4, 0x0, 0xe, 0xa, 0x6, 0x2 },
	{ 0x5, 0x1, 0xd, 0x9, 0x7, 0x3, 0xf, 0xb, 0xc, 0x8, 0x4, 0x0, 0xe, 0xa, 0x6, 0x2 },
	{ 0x5, 0x1, 0xd, 0x9, 0x7, 0x3, 0xf, 0xb, 0x4, 0x0, 0xc, 0x8, 0x6, 0x2, 0xe, 0xa },
	{ 0xd, 0x9, 0x5, 0x1, 0xf, 0xb, 0x7, 0x3, 0x4, 0x0, 0xc, 0x8, 0x6, 0x2, 0xe, 0xa },
};

static unsigned char rise10_crypt_table5[][16] =
{
	/*  0,   1,   2,   3,   4,   5,   6,   7,   8,   9,   A,   B,   C,   D,   E,   F */
	{ 0x9, 0x8, 0xa, 0xb, 0xd, 0xc, 0xe, 0xf, 0x1, 0x0, 0x2, 0x3, 0x5, 0x4, 0x6, 0x7 },
	{ 0x1, 0x8, 0xa, 0x3, 0x5, 0xc, 0xe, 0x7, 0xd, 0x0, 0x2, 0xf, 0x4, 0xb, 0x6, 0x9 },
	{ 0x1, 0x0, 0x2, 0x3, 0x5, 0x4, 0x6, 0x7, 0xd, 0xc, 0xe, 0xf, 0xa, 0xb, 0x8, 0x9 },
	{ 0xd, 0x0, 0x2, 0xf, 0x4, 0xb, 0x6, 0x9, 0x5, 0xc, 0xe, 0x7, 0xa, 0x3, 0x8, 0x1 },
	{ 0xc, 0xd, 0xf, 0xe, 0xb, 0xa, 0x9, 0x8, 0x4, 0x5, 0x7, 0x6, 0x3, 0x2, 0x1, 0x0 },
	{ 0xc, 0x5, 0x7, 0xe, 0x3, 0xa, 0x1, 0x8, 0xb, 0x4, 0x9, 0x6, 0xf, 0x2, 0xd, 0x0 },
	{ 0x4, 0x5, 0x7, 0x6, 0x3, 0x2, 0x1, 0x0, 0xb, 0xa, 0x9, 0x2, 0xf, 0xe, 0xd, 0xc },
	{ 0xb, 0x4, 0x9, 0x6, 0xf, 0x2, 0xd, 0x0, 0x3, 0xa, 0x1, 0x8, 0x7, 0xe, 0x5, 0xc },

	{ 0xb, 0xa, 0x9, 0x8, 0xf, 0xe, 0xd, 0xc, 0x3, 0x2, 0x1, 0x0, 0x7, 0x6, 0x5, 0x4 },
	{ 0x3, 0xa, 0x1, 0x8, 0x7, 0xe, 0x5, 0xc, 0xf, 0x2, 0xd, 0x0, 0x9, 0x6, 0x4, 0xb },
	{ 0x3, 0x2, 0x1, 0x0, 0x7, 0x6, 0x5, 0x4, 0xf, 0xe, 0xd, 0xc, 0x9, 0x8, 0xa, 0xb },
	{ 0xf, 0x2, 0xd, 0x0, 0x9, 0x6, 0x4, 0xb, 0x7, 0xe, 0x5, 0xc, 0x1, 0x8, 0xa, 0x3 },
	{ 0xf, 0xe, 0xd, 0xc, 0x9, 0x8, 0xa, 0xb, 0x7, 0x6, 0x5, 0x4, 0x1, 0x0, 0x2, 0x3 },
	{ 0x7, 0xe, 0x5, 0xc, 0x1, 0x8, 0xa, 0x3, 0x9, 0x6, 0x4, 0xb, 0xd, 0x0, 0x2, 0xf },
	{ 0x7, 0x6, 0x5, 0x4, 0x1, 0x0, 0x2, 0x3, 0x9, 0x8, 0xa, 0xb, 0xd, 0xc, 0xe, 0xf },
	{ 0x9, 0x6, 0x4, 0xb, 0xd, 0x0, 0x2, 0xf, 0x1, 0x8, 0xa, 0x3, 0x5, 0xc, 0xe, 0x7 },

	{ 0x8, 0x9, 0xb, 0xa, 0xc, 0xd, 0xf, 0xe, 0x0, 0x1, 0x3, 0x2, 0x4, 0x5, 0x7, 0x6 },
	{ 0x8, 0x1, 0x3, 0xa, 0xc, 0x5, 0x7, 0xe, 0x0, 0xd, 0xf, 0x2, 0xb, 0x4, 0x9, 0x6 },
	{ 0x0, 0x1, 0x3, 0x2, 0x4, 0x5, 0x7, 0x6, 0xc, 0xd, 0xf, 0xe, 0xb, 0xa, 0x9, 0x8 },
	{ 0x0, 0xd, 0xf, 0x2, 0xb, 0x4, 0x9, 0x6, 0xc, 0x5, 0x7, 0xe, 0x3, 0xa, 0x1, 0x8 },
	{ 0xd, 0xc, 0xe, 0xf, 0xa, 0xb, 0x8, 0x9, 0x5, 0x4, 0x6, 0x7, 0x2, 0x3, 0x0, 0x1 },
	{ 0x5, 0xc, 0xe, 0x7, 0xa, 0x3, 0x8, 0x1, 0x4, 0xb, 0x6, 0x9, 0x2, 0xf, 0x0, 0xd },
	{ 0x5, 0x4, 0x6, 0x7, 0x2, 0x3, 0x0, 0x1, 0xa, 0xb, 0x8, 0x9, 0xe, 0xf, 0xc, 0xd },
	{ 0x4, 0xb, 0x6, 0x9, 0x2, 0xf, 0x0, 0xd, 0xa, 0x3, 0x8, 0x1, 0xe, 0x7, 0xc, 0x5 },

	{ 0xa, 0xb, 0x8, 0x9, 0xe, 0xf, 0xc, 0xd, 0x2, 0x3, 0x0, 0x1, 0x6, 0x7, 0x4, 0x5 },
	{ 0xa, 0x3, 0x8, 0x1, 0xe, 0x7, 0xc, 0x5, 0x2, 0xf, 0x0, 0xd, 0x6, 0x9, 0xb, 0x4 },
	{ 0x2, 0x3, 0x0, 0x1, 0x6, 0x7, 0x4, 0x5, 0xe, 0xf, 0xc, 0xd, 0x8, 0x9, 0xb, 0xa },
	{ 0x2, 0xf, 0x0, 0xd, 0x6, 0x9, 0xb, 0x4, 0xe, 0x7, 0xc, 0x5, 0x8, 0x1, 0x3, 0xa },
	{ 0xe, 0xf, 0xc, 0xd, 0x8, 0x9, 0xb, 0xa, 0x6, 0x7, 0x4, 0x5, 0x0, 0x1, 0x3, 0x2 },
	{ 0xe, 0x7, 0xc, 0x5, 0x8, 0x1, 0x3, 0xa, 0x6, 0x9, 0xb, 0x4, 0x0, 0xd, 0xf, 0x2 },
	{ 0x6, 0x7, 0x4, 0x5, 0x0, 0x1, 0x3, 0x2, 0x8, 0x9, 0xb, 0xa, 0xc, 0xd, 0xf, 0xe },
	{ 0x6, 0x9, 0xb, 0x4, 0x0, 0xd, 0xf, 0x2, 0x8, 0x1, 0x3, 0xa, 0xc, 0x5, 0x7, 0xe },
};

static unsigned char rise10_crypt_table6[][16] =
{
	/*  0,   1,   2,   3,   4,   5,   6,   7,   8,   9,   A,   B,   C,   D,   E,   F */
	{ 0x5, 0xd, 0x1, 0x9, 0x7, 0xf, 0x3, 0xb, 0x4, 0xc, 0x0, 0x8, 0x6, 0xe, 0x2, 0xa },
	{ 0xd, 0x5, 0x9, 0x1, 0xf, 0x7, 0xb, 0x3, 0xc, 0x4, 0x8, 0x0, 0xe, 0x6, 0xa, 0x2 },
	{ 0x2, 0xa, 0x6, 0xe, 0x0, 0x8, 0x4, 0xc, 0x3, 0xb, 0x7, 0xf, 0x1, 0x9, 0x5, 0xd },
	{ 0xa, 0x2, 0xe, 0x6, 0x8, 0x0, 0xc, 0x4, 0xb, 0x3, 0xf, 0x7, 0x9, 0x1, 0xd, 0x5 },

	{ 0x6, 0xe, 0x2, 0xa, 0x4, 0xc, 0x0, 0x8, 0x7, 0xf, 0x3, 0xb, 0x5, 0xd, 0x1, 0x9 },
	{ 0xe, 0x6, 0xa, 0x2, 0xc, 0x4, 0x8, 0x0, 0xf, 0x7, 0xb, 0x3, 0xd, 0x5, 0x9, 0x1 },
	{ 0x0, 0x8, 0x4, 0xc, 0x2, 0xa, 0x6, 0xe, 0x1, 0x9, 0x5, 0xd, 0x3, 0xb, 0x7, 0xf },
	{ 0x8, 0x0, 0xc, 0x4, 0xa, 0x2, 0xe, 0x6, 0x9, 0x1, 0xd, 0x5, 0xb, 0x3, 0xf, 0x7 },

	{ 0x4, 0xc, 0x0, 0x8, 0x6, 0xe, 0x2, 0xa, 0x5, 0xd, 0x1, 0x9, 0x7, 0xf, 0x3, 0xb },
	{ 0xc, 0x4, 0x8, 0x0, 0xe, 0x6, 0xa, 0x2, 0xd, 0x5, 0x9, 0x1, 0xf, 0x7, 0xb, 0x3 },
	{ 0x3, 0xb, 0x7, 0xf, 0x1, 0x9, 0x5, 0xd, 0x2, 0xa, 0x6, 0xe, 0x0, 0x8, 0x4, 0xc },
	{ 0xb, 0x3, 0xf, 0x7, 0x9, 0x1, 0xd, 0x5, 0xa, 0x2, 0xe, 0x6, 0x8, 0x0, 0xc, 0x4 },

	{ 0x7, 0xf, 0x3, 0xb, 0x5, 0xd, 0x1, 0x9, 0x6, 0xe, 0x2, 0xa, 0x4, 0xc, 0x0, 0x8 },
	{ 0xf, 0x7, 0xb, 0x3, 0xd, 0x5, 0x9, 0x1, 0xe, 0x6, 0xa, 0x2, 0xc, 0x4, 0x8, 0x0 },
	{ 0x1, 0x9, 0x5, 0xd, 0x3, 0xb, 0x7, 0xf, 0x0, 0x8, 0x4, 0xc, 0x2, 0xa, 0x6, 0xe },
	{ 0x9, 0x1, 0xd, 0x5, 0xb, 0x3, 0xf, 0x7, 0x8, 0x0, 0xc, 0x4, 0xa, 0x2, 0xe, 0x6 },
};

static UINT8 rise10_dc1(UINT32 b, int adr)
{
	int t;
	UINT8 db, nb;
	nb = NIBBLE(b, 17,11,10,5);

	t = 0;
	db = rise10_crypt_table1[t][nb];
	return db;
}

static UINT8 rise10_dc2(UINT32 b, int adr)
{
	int t;
	UINT8 db, nb;
	nb = NIBBLE(b, 18,12,9,8);

	t = 0;
	db = rise10_crypt_table2[t][nb];
	return db;
}

static UINT8 rise10_dc3(UINT32 b, int adr)
{
	int t;
	UINT8 db, nb;
	nb = NIBBLE(b, 19,14,13,6);

	t = 0;
	if( adr >= 0xeba0 )
		t = 1;
	db = rise10_crypt_table3[t][nb];
	return db;
}

static UINT8 rise10_dc4(UINT32 b, int adr)
{
	int t;
	UINT8 db, nb;
	nb = NIBBLE(b, 23,22,16,4);

	t = ((adr + 0x60) >> 7) & 0x3;
	if( (adr & 0x3fff) >= 0xba0 && (adr & 0x3fff) < 0x2ba0 )
		t += 4;
	if( (adr & 0x3fff) >= 0x2ba0 )
		t += 8;
	if( (adr & 0x7fff) >= 0x4000 )
		t += 12;
	if( adr > 0x8000 )
		t += 24;
	db = rise10_crypt_table4[t][nb];
	return db;
}

static UINT8 rise10_dc5(UINT32 b, int adr)
{
	int t;
	UINT8 db, nb;
	nb = NIBBLE(b, 16,7,1,0);

	t = ((adr + 0x60) >> 7) & 0xf;
	t |= (((adr + 0x1660) >> 11) & 0x1) << 4;
	db = rise10_crypt_table5[t][nb];
	return db;
}

static UINT8 rise10_dc6(UINT32 b, int adr)
{
	int t;
	UINT8 db, nb;
	nb = NIBBLE(b, 21,20,3,2);

	t = (adr >> 4) & 0xf;
	db = rise10_crypt_table6[t][nb];
	return db;
}

void seibuspi_rise10_text_decrypt(unsigned char *rom)
{
	int i;
	for(i=0; i<0x10000; i++) {
		UINT8 r1,r2,r3,r4,r5,r6;
		UINT32 w;

		w = (rom[(i*3) + 0] << 16) | (rom[(i*3) + 1] << 8) | (rom[(i*3) +2]);

		r1 = rise10_dc1(w, i);
		r2 = rise10_dc2(w, i);
		r3 = rise10_dc3(w, i);
		r4 = rise10_dc4(w, i);
		r5 = rise10_dc5(w, i);
		r6 = rise10_dc6(w, i);


		rom[(i*3) + 0] = (r1 << 4) | r2;
		rom[(i*3) + 1] = (r3 << 4) | r4;
		rom[(i*3) + 2] = (r5 << 4) | r6;
	}
}

void seibuspi_rise10_bg_decrypt(unsigned char *rom, int size)
{
	int i,j;
	for(j=0; j<size; j+=0xc0000) {
		for(i=0; i<0x40000; i++) {
			UINT8 r1,r2,r3,r4,r5,r6;
			UINT32 w;

			w = (rom[j + (i*3) + 0] << 16) | (rom[j + (i*3) + 1] << 8) | (rom[j + (i*3) + 2]);

			r1 = rise10_dc1(w, i/4);
			r2 = rise10_dc2(w, i/4);
			r3 = rise10_dc3(w, i/4);
			r4 = rise10_dc4(w, i/4);
			r5 = rise10_dc5(w, i/4);
			r6 = rise10_dc6(w, i/4);

			rom[j + (i*3) + 0] = (r1 << 4) | r2;
			rom[j + (i*3) + 1] = (r3 << 4) | r4;
			rom[j + (i*3) + 2] = (r5 << 4) | r6;
		}
	}
}

static UINT8 rise10s_tb1[256] =
{
	/*		0     1     2     3     4     5     6     7     8     9     A     B     C     D     E     F		*/
	/* 0 */ 0x06, 0x86, 0x26, 0xa6, 0x16, 0x96, 0x3e, 0xbe, 0x56, 0xc6, 0x66, 0xe6, 0x46, 0xd6, 0x7e, 0xfe,
	/* 1 */ 0x0e, 0x8e, 0x2e, 0xae, 0x1e, 0x9e, 0x36, 0xb6, 0x5e, 0xce, 0x6e, 0xee, 0x4e, 0xde, 0x76, 0xf6,
	/* 2 */ 0x00, 0x80, 0x20, 0xa0, 0x10, 0x90, 0x38, 0xb8, 0x50, 0xc0, 0x60, 0xe0, 0x40, 0xd0, 0x78, 0xf8,
	/* 3 */ 0x08, 0x88, 0x28, 0xa8, 0x18, 0x98, 0x30, 0xb0, 0x58, 0xc8, 0x68, 0xe8, 0x48, 0xd8, 0x70, 0xf0,
	/* 4 */ 0x02, 0x82, 0x22, 0xa2, 0x12, 0x92, 0x3a, 0xba, 0x52, 0xc2, 0x62, 0xe2, 0x42, 0xd2, 0x7a, 0xfa,
	/* 5 */ 0x0a, 0x8a, 0x2a, 0xaa, 0x1a, 0x9a, 0x32, 0xb2, 0x5a, 0xca, 0x6a, 0xea, 0x4a, 0xda, 0x72, 0xf2,
	/* 6 */ 0x04, 0x84, 0x24, 0xa4, 0x14, 0x94, 0x3c, 0xbc, 0x54, 0xc4, 0x64, 0xe4, 0x44, 0xd4, 0x7c, 0xfc,
	/* 7 */ 0x0c, 0x8c, 0x2c, 0xac, 0x1c, 0x9c, 0x34, 0xb4, 0x5c, 0xcc, 0x6c, 0xec, 0x4c, 0xdc, 0x74, 0xf4,
	/* 8 */ 0x07, 0x87, 0x27, 0xa7, 0x17, 0x97, 0x3f, 0xbf, 0x57, 0xc7, 0x67, 0xe7, 0x47, 0xd7, 0x7f, 0xff,
	/* 9 */ 0x0f, 0x8f, 0x2f, 0xaf, 0x1f, 0x9f, 0x37, 0xb7, 0x5f, 0xcf, 0x6f, 0xef, 0x4f, 0xdf, 0x77, 0xf7,
	/* A */ 0x01, 0x81, 0x21, 0xa1, 0x11, 0x91, 0x39, 0xb9, 0x51, 0xc1, 0x61, 0xe1, 0x41, 0xd1, 0x79, 0xf9,
	/* B */ 0x09, 0x89, 0x29, 0xa9, 0x19, 0x99, 0x31, 0xb1, 0x59, 0xc9, 0x69, 0xe9, 0x49, 0xd9, 0x71, 0xf1,
	/* C */ 0x03, 0x83, 0x23, 0xa3, 0x13, 0x93, 0x3b, 0xbb, 0x53, 0xc3, 0x63, 0xe3, 0x43, 0xd3, 0x7b, 0xfb,
	/* D */ 0x0b, 0x8b, 0x2b, 0xab, 0x1b, 0x9b, 0x33, 0xb3, 0x5b, 0xcb, 0x6b, 0xeb, 0x4b, 0xdb, 0x73, 0xf3,
	/* E */ 0x05, 0x85, 0x25, 0xa5, 0x15, 0x95, 0x3d, 0xbd, 0x55, 0xc5, 0x65, 0xe5, 0x45, 0xd5, 0x7d, 0xfd,
	/* F */ 0x0d, 0x8d, 0x2d, 0xad, 0x1d, 0x9d, 0x35, 0xb5, 0x5d, 0xcd, 0x6d, 0xed, 0x4d, 0xdd, 0x75, 0xf5,
};

static UINT8 rise10s_tb2[256] =
{	/*		0     1     2     3     4     5     6     7     8     9     A     B     C     D     E     F		*/
	/* 0 */ 0x07, 0xd7, 0x27, 0xef, 0x57, 0x8f, 0x6f, 0xaf, 0x0c, 0xdc, 0x2c, 0xe4, 0x54, 0x8c, 0x6c, 0xac,
	/* 1 */ 0x1f, 0xcf, 0x3f, 0xf7, 0x4f, 0x97, 0x77, 0xb7, 0x14, 0xc4, 0x34, 0xfc, 0x4c, 0x94, 0x74, 0xb4,
	/* 2 */ 0x03, 0xd3, 0x23, 0xeb, 0x53, 0x8b, 0x6b, 0xab, 0x08, 0xd8, 0x28, 0xe0, 0x50, 0x88, 0x68, 0xa8,
	/* 3 */ 0x1b, 0xcb, 0x3b, 0xf3, 0x4b, 0x93, 0x73, 0xb3, 0x10, 0xc0, 0x30, 0xf8, 0x48, 0x90, 0x70, 0xb0,
	/* 4 */ 0x0f, 0xdf, 0x2f, 0xe7, 0x5f, 0x87, 0x67, 0xa7, 0x04, 0xd4, 0x24, 0xec, 0x5c, 0x84, 0x64, 0xa4,
	/* 5 */ 0x17, 0xc7, 0x37, 0xff, 0x47, 0x9f, 0x7f, 0xbf, 0x1c, 0xcc, 0x3c, 0xf4, 0x44, 0x9c, 0x7c, 0xbc,
	/* 6 */ 0x0b, 0xdb, 0x2b, 0xe3, 0x5b, 0x83, 0x63, 0xa3, 0x00, 0xd0, 0x20, 0xe8, 0x58, 0x80, 0x60, 0xa0,
	/* 7 */ 0x13, 0xc3, 0x33, 0xfb, 0x43, 0x9b, 0x7b, 0xbb, 0x18, 0xc8, 0x38, 0xf0, 0x40, 0x98, 0x78, 0xb8,
	/* 8 */ 0x06, 0xd6, 0x26, 0xee, 0x56, 0x8e, 0x6e, 0xae, 0x0d, 0xdd, 0x2d, 0xe5, 0x55, 0x8d, 0x6d, 0xad,
	/* 9 */ 0x1e, 0xce, 0x3e, 0xf6, 0x4e, 0x96, 0x76, 0xb6, 0x15, 0xc5, 0x35, 0xfd, 0x4d, 0x95, 0x75, 0xb5,
	/* A */ 0x02, 0xd2, 0x22, 0xea, 0x52, 0x8a, 0x6a, 0xaa, 0x09, 0xd9, 0x29, 0xe1, 0x51, 0x89, 0x69, 0xa9,
	/* B */ 0x1a, 0xca, 0x3a, 0xf2, 0x4a, 0x92, 0x72, 0xb2, 0x11, 0xc1, 0x31, 0xf9, 0x49, 0x91, 0x71, 0xb1,
	/* C */ 0x0e, 0xde, 0x2e, 0xe6, 0x5e, 0x86, 0x66, 0xa6, 0x05, 0xd5, 0x25, 0xed, 0x5d, 0x85, 0x65, 0xa5,
	/* D */ 0x16, 0xc6, 0x36, 0xfe, 0x46, 0x9e, 0x7e, 0xbe, 0x1d, 0xcd, 0x3d, 0xf5, 0x45, 0x9d, 0x7d, 0xbd,
	/* E */ 0x0a, 0xda, 0x2a, 0xe2, 0x5a, 0x82, 0x62, 0xa2, 0x01, 0xd1, 0x21, 0xe9, 0x59, 0x81, 0x61, 0xa1,
	/* F */ 0x12, 0xc2, 0x32, 0xfa, 0x42, 0x9a, 0x7a, 0xba, 0x19, 0xc9, 0x39, 0xf1, 0x41, 0x99, 0x79, 0xb9,
};

static UINT8 rise10s_tb3[256] =
{	/*		0     1     2     3     4     5     6     7     8     9     A     B     C     D     E     F		*/
	/* 0 */ 0xD4, 0xD0, 0xE4, 0xE0, 0x64, 0x60, 0x54, 0x50, 0xD6, 0xD2, 0xE6, 0xE2, 0x66, 0x62, 0x56, 0x52,
	/* 1 */ 0xC4, 0xC0, 0xF4, 0xF0, 0x74, 0x70, 0x44, 0x40, 0xC6, 0xC2, 0xF6, 0xF2, 0x76, 0x72, 0x46, 0x42,
	/* 2 */ 0xDC, 0xD8, 0xEC, 0xE8, 0x6C, 0x68, 0x5C, 0x58, 0xDE, 0xDA, 0xEE, 0xEA, 0x6E, 0x6A, 0x5E, 0x5A,
	/* 3 */ 0xCC, 0xC8, 0xFC, 0xF8, 0x7C, 0x78, 0x4C, 0x48, 0xCE, 0xCA, 0xFE, 0xFA, 0x7E, 0x7A, 0x4E, 0x4A,
	/* 4 */ 0x94, 0x90, 0xA4, 0xA0, 0x14, 0x10, 0x24, 0x20, 0x96, 0x92, 0xA6, 0xA2, 0x16, 0x12, 0x26, 0x22,
	/* 5 */ 0x84, 0x80, 0xB4, 0xB0, 0x04, 0x00, 0x34, 0x30, 0x86, 0x82, 0xB6, 0xB2, 0x06, 0x02, 0x36, 0x32,
	/* 6 */ 0x9C, 0x98, 0xAC, 0xA8, 0x1C, 0x18, 0x2C, 0x28, 0x9E, 0x9A, 0xAE, 0xAA, 0x1E, 0x1A, 0x2E, 0x2A,
	/* 7 */ 0x8C, 0x88, 0xBC, 0xB8, 0x0C, 0x08, 0x3C, 0x38, 0x8E, 0x8A, 0xBE, 0xBA, 0x0E, 0x0A, 0x3E, 0x3A,
	/* 8 */ 0xD5, 0xD1, 0xE5, 0xE1, 0x65, 0x61, 0x55, 0x51, 0xD7, 0xD3, 0xE7, 0xE3, 0x67, 0x63, 0x57, 0x53,
	/* 9 */ 0xC5, 0xC1, 0xF5, 0xF1, 0x75, 0x71, 0x45, 0x41, 0xC7, 0xC3, 0xF7, 0xF3, 0x77, 0x73, 0x47, 0x43,
	/* A */ 0xDD, 0xD9, 0xED, 0xE9, 0x6D, 0x69, 0x5D, 0x59, 0xDF, 0xDB, 0xEF, 0xEB, 0x6F, 0x6B, 0x5F, 0x5B,
	/* B */ 0xCD, 0xC9, 0xFD, 0xF9, 0x7D, 0x79, 0x4D, 0x49, 0xCF, 0xCB, 0xFF, 0xFB, 0x7F, 0x7B, 0x4F, 0x4B,
	/* C */ 0x95, 0x91, 0xA5, 0xA1, 0x15, 0x11, 0x25, 0x21, 0x97, 0x93, 0xA7, 0xA3, 0x17, 0x13, 0x27, 0x23,
	/* D */ 0x85, 0x81, 0xB5, 0xB1, 0x05, 0x01, 0x35, 0x31, 0x87, 0x83, 0xB7, 0xB3, 0x07, 0x03, 0x37, 0x33,
	/* E */ 0x9D, 0x99, 0xAD, 0xA9, 0x1D, 0x19, 0x2D, 0x29, 0x9F, 0x9B, 0xAF, 0xAB, 0x1F, 0x1B, 0x2F, 0x2B,
	/* F */ 0x8D, 0x89, 0xBD, 0xB9, 0x0D, 0x09, 0x3D, 0x39, 0x8F, 0x8B, 0xBF, 0xBB, 0x0F, 0x0B, 0x3F, 0x3B,
};

static UINT8 rise10s_tb4[256] =
{	/*		0     1     2     3     4     5     6     7     8     9     A     B     C     D     E     F		*/
	/* 0 */ 0xAC, 0xA8, 0xEC, 0xE8, 0xAD, 0xA9, 0xED, 0xE9, 0xAE, 0xAA, 0xEE, 0xEA, 0xAF, 0xAB, 0xEF, 0xEB,
	/* 1 */ 0x8C, 0x88, 0xCC, 0xC8, 0x8D, 0x89, 0xCD, 0xC9, 0x8E, 0x8A, 0xCE, 0xCA, 0x8F, 0x8B, 0xCF, 0xCB,
	/* 2 */ 0x9C, 0x98, 0xDC, 0xD8, 0x9D, 0x99, 0xDD, 0xD9, 0x9E, 0x9A, 0xDE, 0xDA, 0x9F, 0x9B, 0xDF, 0xDB,
	/* 3 */ 0xBC, 0xB8, 0xFC, 0xF8, 0xBD, 0xB9, 0xFD, 0xF9, 0xBE, 0xBA, 0xFE, 0xFA, 0xBF, 0xBB, 0xFF, 0xFB,
	/* 4 */ 0x6C, 0x68, 0x2C, 0x28, 0x6D, 0x69, 0x2D, 0x29, 0x6E, 0x6A, 0x2E, 0x2A, 0x6F, 0x6B, 0x2F, 0x2B,
	/* 5 */ 0x4C, 0x48, 0x0C, 0x08, 0x4D, 0x49, 0x0D, 0x09, 0x4E, 0x4A, 0x0E, 0x0A, 0x4F, 0x4B, 0x0F, 0x0B,
	/* 6 */ 0x5C, 0x58, 0x1C, 0x18, 0x5D, 0x59, 0x1D, 0x19, 0x5E, 0x5A, 0x1E, 0x1A, 0x5F, 0x5B, 0x1F, 0x1B,
	/* 7 */ 0x7C, 0x78, 0x3C, 0x38, 0x7D, 0x79, 0x3D, 0x39, 0x7E, 0x7A, 0x3E, 0x3A, 0x7F, 0x7B, 0x3F, 0x3B,
	/* 8 */ 0xA4, 0xA0, 0xE4, 0xE0, 0xA5, 0xA1, 0xE5, 0xE1, 0xA6, 0xA2, 0xE6, 0xE2, 0xA7, 0xA3, 0xE7, 0xE3,
	/* 9 */ 0x84, 0x80, 0xC4, 0xC0, 0x85, 0x81, 0xC5, 0xC1, 0x86, 0x82, 0xC6, 0xC2, 0x87, 0x83, 0xC7, 0xC3,
	/* A */ 0x94, 0x90, 0xD4, 0xD0, 0x95, 0x91, 0xD5, 0xD1, 0x96, 0x92, 0xD6, 0xD2, 0x97, 0x93, 0xD7, 0xD3,
	/* B */ 0xB0, 0xB4, 0xF0, 0xF4, 0xB1, 0xB5, 0xF1, 0xF5, 0xB2, 0xB6, 0xF2, 0xF6, 0xB3, 0xB7, 0xF3, 0xF7,
	/* C */ 0x64, 0x60, 0x24, 0x20, 0x65, 0x61, 0x25, 0x21, 0x66, 0x62, 0x26, 0x22, 0x67, 0x63, 0x27, 0x23,
	/* D */ 0x44, 0x40, 0x04, 0x00, 0x45, 0x41, 0x05, 0x01, 0x46, 0x42, 0x06, 0x02, 0x47, 0x43, 0x07, 0x03,
	/* E */ 0x54, 0x50, 0x14, 0x10, 0x55, 0x51, 0x15, 0x11, 0x56, 0x52, 0x16, 0x12, 0x57, 0x53, 0x17, 0x13,
	/* F */ 0x70, 0x74, 0x30, 0x34, 0x71, 0x75, 0x31, 0x35, 0x72, 0x76, 0x32, 0x36, 0x73, 0x77, 0x33, 0x37,
};

static UINT8 rise10s_tb5[256] =
{	/*		0     1     2     3     4     5     6     7     8     9     A     B     C     D     E     F		*/
	/* 0 */ 0x4a, 0xca, 0x2a, 0xaa, 0x6a, 0xea, 0x0a, 0x8a, 0x52, 0xd2, 0x32, 0xb2, 0x72, 0xf2, 0x12, 0x92,
	/* 1 */ 0x42, 0xc2, 0x22, 0xa2, 0x62, 0xe2, 0x02, 0x82, 0x5a, 0xda, 0x3a, 0xba, 0x7a, 0xfa, 0x1a, 0x9a,
	/* 2 */ 0x4e, 0xce, 0x2e, 0xae, 0x6e, 0xee, 0x0e, 0x8e, 0x56, 0xd6, 0x36, 0xb6, 0x76, 0xf6, 0x16, 0x96,
	/* 3 */ 0x46, 0xc6, 0x26, 0xa6, 0x66, 0xe6, 0x06, 0x86, 0x5e, 0xde, 0x3e, 0xbe, 0x7e, 0xfe, 0x1e, 0x9e,
	/* 4 */ 0x48, 0xc8, 0x28, 0xa8, 0x68, 0xe8, 0x08, 0x88, 0x50, 0xd0, 0x30, 0xb0, 0x70, 0xf0, 0x10, 0x90,
	/* 5 */ 0x40, 0xc0, 0x20, 0xa0, 0x60, 0xe0, 0x00, 0x80, 0x58, 0xd8, 0x38, 0xb8, 0x78, 0xf8, 0x18, 0x98,
	/* 6 */ 0x4c, 0xcc, 0x2c, 0xac, 0x6c, 0xec, 0x0c, 0x8c, 0x54, 0xd4, 0x34, 0xb4, 0x74, 0xf4, 0x14, 0x94,
	/* 7 */ 0x44, 0xc4, 0x24, 0xa4, 0x64, 0xe4, 0x04, 0x84, 0x5c, 0xdc, 0x3c, 0xbc, 0x7c, 0xfc, 0x1c, 0x9c,
	/* 8 */ 0x4b, 0xcb, 0x2b, 0xab, 0x6b, 0xeb, 0x0b, 0x8b, 0x53, 0xd3, 0x33, 0xb3, 0x73, 0xf3, 0x13, 0x93,
	/* 9 */ 0x43, 0xc3, 0x23, 0xa3, 0x63, 0xe3, 0x03, 0x83, 0x5b, 0xdb, 0x3b, 0xbb, 0x7b, 0xfb, 0x1b, 0x9b,
	/* A */ 0x4f, 0xcf, 0x2f, 0xaf, 0x6f, 0xef, 0x0f, 0x8f, 0x57, 0xd7, 0x37, 0xb7, 0x77, 0xf7, 0x17, 0x97,
	/* B */ 0x47, 0xc7, 0x27, 0xa7, 0x67, 0xe7, 0x07, 0x87, 0x5f, 0xdf, 0x3f, 0xbf, 0x7f, 0xff, 0x1f, 0x9f,
	/* C */ 0x49, 0xc9, 0x29, 0xa9, 0x69, 0xe9, 0x09, 0x89, 0x51, 0xd1, 0x31, 0xb1, 0x71, 0xf1, 0x11, 0x91,
	/* D */ 0x41, 0xc1, 0x21, 0xa1, 0x61, 0xe1, 0x01, 0x81, 0x59, 0xd9, 0x39, 0xb9, 0x79, 0xf9, 0x19, 0x99,
	/* E */ 0x4d, 0xcd, 0x2d, 0xad, 0x6d, 0xed, 0x0d, 0x8d, 0x55, 0xd5, 0x35, 0xb5, 0x75, 0xf5, 0x15, 0x95,
	/* F */ 0x45, 0xc5, 0x25, 0xa5, 0x65, 0xe5, 0x05, 0x85, 0x5d, 0xdd, 0x3d, 0xbd, 0x7d, 0xfd, 0x1d, 0x9d,
};

static UINT8 rise10s_tb6[256] =
{	/*		0     1     2     3     4     5     6     7     8     9     A     B     C     D     E     F		*/
/* 0 */ 0x73, 0xf3, 0x33, 0xb3, 0x53, 0xd3, 0x13, 0x93, 0x63, 0xe3, 0x23, 0xa3, 0x43, 0xc3, 0x03, 0x83,
/* 1 */ 0x7b, 0xfb, 0x3b, 0xbb, 0x5b, 0xdb, 0x1b, 0x9b, 0x6b, 0xeb, 0x2b, 0xab, 0x4b, 0xcb, 0x0b, 0x8b,
/* 2 */ 0x77, 0xf7, 0x37, 0xb7, 0x57, 0xd7, 0x17, 0x97, 0x67, 0xe7, 0x27, 0xa7, 0x47, 0xc7, 0x07, 0x87,
/* 3 */ 0x7f, 0xff, 0x3f, 0xbf, 0x5f, 0xdf, 0x1f, 0x9f, 0x6f, 0xef, 0x2f, 0xaf, 0x4f, 0xcf, 0x0f, 0x8f,
/* 4 */ 0x71, 0xf1, 0x31, 0xb1, 0x51, 0xd1, 0x11, 0x91, 0x61, 0xe1, 0x21, 0xa1, 0x41, 0xc1, 0x01, 0x81,
/* 5 */ 0x79, 0xf9, 0x39, 0xb9, 0x59, 0xd9, 0x19, 0x99, 0x69, 0xe9, 0x29, 0xa9, 0x49, 0xc9, 0x09, 0x89,
/* 6 */ 0x75, 0xf5, 0x35, 0xb5, 0x55, 0xd5, 0x15, 0x95, 0x65, 0xe5, 0x25, 0xa5, 0x45, 0xc5, 0x05, 0x85,
/* 7 */ 0x7d, 0xfd, 0x3d, 0xbd, 0x5d, 0xdd, 0x1d, 0x9d, 0x6d, 0xed, 0x2d, 0xad, 0x4d, 0xcd, 0x0d, 0x8d,
/* 8 */ 0x72, 0xf2, 0x32, 0xb2, 0x52, 0xd2, 0x12, 0x92, 0x62, 0xe2, 0x22, 0xa2, 0x42, 0xc2, 0x02, 0x82,
/* 9 */ 0x7a, 0xfa, 0x3a, 0xba, 0x5a, 0xda, 0x1a, 0x9a, 0x6a, 0xea, 0x2a, 0xaa, 0x4a, 0xca, 0x0a, 0x8a,
/* A */ 0x76, 0xf6, 0x36, 0xb6, 0x56, 0xd6, 0x16, 0x96, 0x66, 0xe6, 0x26, 0xa6, 0x46, 0xc6, 0x06, 0x86,
/* B */ 0x7e, 0xfe, 0x3e, 0xbe, 0x5e, 0xde, 0x1e, 0x9e, 0x6e, 0xee, 0x2e, 0xae, 0x4e, 0xce, 0x0e, 0x8e,
/* C */ 0x70, 0xf0, 0x30, 0xb0, 0x50, 0xd0, 0x10, 0x90, 0x60, 0xe0, 0x20, 0xa0, 0x40, 0xc0, 0x00, 0x80,
/* D */ 0x78, 0xf8, 0x38, 0xb8, 0x58, 0xd8, 0x18, 0x98, 0x68, 0xe8, 0x28, 0xa8, 0x48, 0xc8, 0x08, 0x88,
/* E */ 0x74, 0xf4, 0x34, 0xb4, 0x54, 0xd4, 0x14, 0x94, 0x64, 0xe4, 0x24, 0xa4, 0x44, 0xc4, 0x04, 0x84,
/* F */ 0x7c, 0xfc, 0x3c, 0xbc, 0x5c, 0xdc, 0x1c, 0x9c, 0x6c, 0xec, 0x2c, 0xac, 0x4c, 0xcc, 0x0c, 0x8c,
};

static void sprite_reorder(data16_t *buffer)
{
	int j;
	data16_t temp[32];
	for( j=0; j < 16; j++ ) {
		temp[(j*2)+0] = buffer[j+0];
		temp[(j*2)+1] = buffer[j+16];
	}
	memcpy(buffer, temp, sizeof(data16_t)*32);
}

void seibuspi_rise10_sprite_decrypt(data16_t *rom, int size)
{
	int i,j;
	int s = size/2;

	for( i=0; i < s; i+=32 ) {
		sprite_reorder(&rom[i]);
		sprite_reorder(&rom[s+i]);
		sprite_reorder(&rom[(s*2)+i]);
		for( j=0; j < 32; j++ ) {
			UINT16 b1,b2,b3;
			UINT8 d1,d2,d3,d4,d5,d6;

			b1 = BITSWAP16(rom[i+j], 7,6,5,4,2,1,12,11,3,0,15,14,13,10,9,8);
			b2 = BITSWAP16(rom[s+i+j], 7,3,1,0,15,14,12,10,5,4,2,0,15,11,9,8);
			b3 = BITSWAP16(rom[(s*2)+i+j], 15,14,13,12,11,10,1,0,7,6,5,4,3,2,1,0);

			d1 = rise10s_tb1[(b1 >> 8) & 0xff];
			d2 = rise10s_tb2[(b1 >> 0) & 0xff];
			d3 = rise10s_tb3[(b2 >> 8) & 0xff];
			d4 = rise10s_tb4[(b2 >> 0) & 0xff];
			d5 = rise10s_tb5[(b3 >> 0) & 0xff];
			d6 = rise10s_tb6[(b3 >> 8) & 0xff];

			rom[i+j] = (d5 << 8) | d6;
			rom[s+i+j] = (d3 << 8) | d4;
			rom[(s*2)+i+j] = (d1 << 8) | d2;
		}
	}
}



/* RISE11 (Raiden Fighters Jet) */

static unsigned char rise11_crypt_table1[][16] =
{
	{ 0xc, 0x3, 0xf, 0x1, 0x4, 0xb, 0x7, 0x9, 0xd, 0x2, 0xe, 0x0, 0x5, 0xa, 0x6, 0x8 }
};

static unsigned char rise11_crypt_table3[][16] =
{
	{ 0x8, 0xc, 0x9, 0xd, 0xa, 0xe, 0xb, 0xf, 0x0, 0x4, 0x1, 0x5, 0x2, 0x6, 0x3, 0x7 },
	{ 0x0, 0x4, 0x1, 0x5, 0x2, 0x6, 0x3, 0x7, 0x8, 0xc, 0x9, 0xd, 0xa, 0xe, 0xb, 0xf }
};

static UINT8 rise11_dc1(UINT32 b, int adr)
{
	UINT8 db, nb;
	nb = NIBBLE(b, 18,17,9,8);

	db = rise11_crypt_table1[0][nb];
	return db;
}

static UINT8 rise11_dc2(UINT32 b, int adr)
{
	return 0xf;
}

static UINT8 rise11_dc3(UINT32 b, int adr)
{
	int t;
	UINT8 db, nb;
	nb = NIBBLE(b, 19,14,13,6);

	t = 0;
	if( adr >= 0x8ac0 )
		t = 1;
	db = rise11_crypt_table3[t][nb];
	return db;
}

static UINT8 rise11_dc4(UINT32 b, int adr)
{
	return 0xf;
}

static UINT8 rise11_dc5(UINT32 b, int adr)
{
	return 0xf;
}

static UINT8 rise11_dc6(UINT32 b, int adr)
{
	return 0xf;
}

void seibuspi_rise11_text_decrypt(unsigned char *rom)
{
	int i;
	for(i=0; i<0x10000; i++) {
		UINT8 r1,r2,r3,r4,r5,r6;
		UINT32 w;

		w = (rom[(i*3) + 0] << 16) | (rom[(i*3) + 1] << 8) | (rom[(i*3) +2]);

		r1 = rise11_dc1(w, i);
		r2 = rise11_dc2(w, i);
		r3 = rise11_dc3(w, i);
		r4 = rise11_dc4(w, i);
		r5 = rise11_dc5(w, i);
		r6 = rise11_dc6(w, i);

		rom[(i*3) + 0] = (r1 << 4) | r2;
		rom[(i*3) + 1] = (r3 << 4) | r4;
		rom[(i*3) + 2] = (r5 << 4) | r6;
	}
}

void seibuspi_rise11_bg_decrypt(unsigned char *rom, int size)
{
	int i,j;
	for(j=0; j<size; j+=0xc0000) {
		for(i=0; i<0x40000; i++) {
			UINT8 r1,r2,r3,r4,r5,r6;
			UINT32 w;

			w = (rom[j + (i*3) + 0] << 16) | (rom[j + (i*3) + 1] << 8) | (rom[j + (i*3) + 2]);

			r1 = rise11_dc1(w, i/4);
			r2 = rise11_dc2(w, i/4);
			r3 = rise11_dc3(w, i/4);
			r4 = rise11_dc4(w, i/4);
			r5 = rise11_dc5(w, i/4);
			r6 = rise11_dc6(w, i/4);

			rom[j + (i*3) + 0] = (r1 << 4) | r2;
			rom[j + (i*3) + 1] = (r3 << 4) | r4;
			rom[j + (i*3) + 2] = (r5 << 4) | r6;
		}
	}
}

void seibuspi_rise11_sprite_decrypt(data16_t *rom, int size)
{
	int i;
	int s = size/2;

	for( i=0; i < s; i+=32 ) {
		sprite_reorder(&rom[i]);
		sprite_reorder(&rom[s+i]);
		sprite_reorder(&rom[(s*2)+i]);
	}
}
