/*
 *  XNap
 *
 *  A pure java file sharing client.
 *
 *  See AUTHORS for copyright information.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package xnap;

import xnap.util.FileHelper;
import xnap.util.JarClassLoader;
import xnap.util.VersionParser;

import java.io.*;
import java.lang.reflect.*;
import java.net.*;
import java.util.*;

/**
 * Checks if a version of XNap is available in the users preferences
 * directory and launches the newest version found.
 */
public class XNapLoader {

    //--- Constant(s) ---

    //--- Data field(s) ---

    /**
     * Save arguments for restart. In case XNap is not started through the
     * loader null pointer exceptions are avoided by initialization with
     * an empty array.
     */
    private static String[] arguments = new String[0];

    //--- Method(s) ---

    /**
     *
     */
    public static void main(String[] argv) 
    {
	arguments = argv;
	getXNapJar();
	File xnapJar = checkForXNapJar();
	if (xnapJar != null) {
	    // load version from home directory
	    try {
		run(xnapJar);
		System.exit(0);
	    }
	    catch(IOException e) {
		e.printStackTrace(System.err);
	    }
	}

	XNap.main(argv);
    }

    public static File checkForXNapJar() 
    {
	String dir = FileHelper.getHomeDir() + "jars" + File.separatorChar;
	File xnapFile = new File(dir, "xnap.jar");
	File propsFile = new File(xnapFile.getAbsolutePath() + ".properties");

	if (!(propsFile.canRead() && xnapFile.canRead())) {
	    return null;
	}

	try {
	    // read properties
	    FileInputStream in = new FileInputStream(propsFile);
	    Properties p = new Properties();
	    p.load(in);

	    try {
		// compare version
		long jarReleaseNr = Long.parseLong(p.getProperty("releaseNr"));
		if (jarReleaseNr > XNap.RELEASE_NR) {
		    System.out.println
			("newer version found: " + p.getProperty("version"));
		    return xnapFile;
		}
	    }
	    catch (NumberFormatException e) {
	    }
	}
	catch (IOException e) {
	}

	return null;
    }

    public static File getXNapJar()
    {
	URL url = FileHelper.getResource("xnap/XNap.class");
	String path = url.getFile();
	int index = path.lastIndexOf("!/");
	if (index > 0) {
	    path = "//" + path.substring(5, index);
	    File f = new File(path);
	    System.out.println("found xnap.jar: " + f.getAbsolutePath());
	    return (f.exists() && f.canWrite()) ? f : null;
	}

	return null;
    }

    public static void loadXNap(String[] argv)
    {
	JarClassLoader jcl = JarClassLoader.getInstance();
	jcl.init();

	try {
	    Class c = jcl.loadClass("xnap.XNap");
	    Method m = c.getMethod("main", new Class[] { argv.getClass() });
	    m.setAccessible(true);
	    int mods = m.getModifiers();
	    if (m.getReturnType() != void.class || !Modifier.isStatic(mods) ||
		!Modifier.isPublic(mods)) {
		throw new NoSuchMethodException("main");
	    }

	    m.invoke(null, new Object[] { argv });
	}
	catch (Throwable e) {
	    e.printStackTrace();
	}
    }

    public static void run(String javaCmd, File jarFile) throws IOException
    {
	String[] cmds = new String[] {
	    javaCmd, "-jar", jarFile.getAbsolutePath()
	};

	String[] args = new String[cmds.length + arguments.length];
	System.arraycopy(cmds, 0, args, 0, cmds.length);
	System.arraycopy(arguments, 0, args, cmds.length, arguments.length);

	System.out.print("running: ");
	for (int i = 0; i < args.length; i++) {
	    System.out.print(args[i] + " ");
	}
	System.out.println();

	Process p = Runtime.getRuntime().exec(args);
    }

    public static void run(File jarFile) throws IOException    
    {
	try {
	    run("java", jarFile);
	}
	catch (IOException e) {
	    run("/usr/bin/java", jarFile);
	}
    }

}
