/*
 *  Java Napster version x.yz (for current version number as well as for
 *  additional information see version.txt)
 *
 *  Previous versions of this program were written by Florian Student
 *  and Michael Ransburg available at www.weblicity.de/jnapster and
 *  http://www.tux.org/~daneel/content/projects/10.shtml respectively.
 *
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package xnap.gui.tree;

import xnap.util.*;

import javax.swing.JTree;
import javax.swing.tree.*;
import javax.swing.event.*;
import java.io.File;
import java.util.*;

public class FileTreeModel extends AbstractTreeModel implements Comparator {

    // --- Data Field(s)

    private Vector subRoots;
    private Hashtable subChildren = new Hashtable();
    private File cachedDir;
    private boolean cacheSorted;
    private File[] cache;
    private boolean filterDotFiles = true;
    private boolean showFiles = false;

    // --- Constructor(s) ---

    public FileTreeModel(String root, File[] roots)
    {
	super(root);

	if (roots != null) {
	    subRoots = new Vector(roots.length);
	    for (int i = 0; i < roots.length; i++) {
		addSubRoot(roots[i]);
	    }
	}
	else {
	    subRoots = new Vector();
	}
    }

    public FileTreeModel(String root)
    {
        this(root, null);
    }

    // --- Method(s)

    public int compare(Object o1, Object o2)
    {
	return 
	    o1.toString().toLowerCase().compareTo(o2.toString().toLowerCase());
    }

    public boolean isLeaf(Object node)
    {
        return false;
    }

    public int getChildCount(Object node)
    {
	//Debug.log("getChildCount: " + node);

	if (node instanceof File && ((File)node).canRead()) {
	    return getSubDirs((File)node, false).length;
	} 
	else if (node instanceof String) {
	    if (node.equals(root)) {
		return subRoots.size();
	    }
	    return ((Vector)subChildren.get(node)).size();
	}
	else {
	    return 0;
	}
    }
    
    public Object getChild(Object parent, int index)
    {
	//Debug.log("getChild: " + parent + ":" + index);

	if (parent instanceof File) {
	    File[] children = getSubDirs((File)parent, true);

	    if ((children == null) || (index >= children.length)) 
		return null;
	    return new FileNode(children[index]);
	} 
	else if (parent instanceof String) {
	    if (parent.equals(root) && index < subRoots.size())
		return subRoots.get(index);
	    
	    Vector v = (Vector)subChildren.get(parent);
	    if (index < v.size())
		return v.elementAt(index);
	    
	    return null;
	}
	else {
	    return null;
	}
    }
    
    public int getIndexOfChild(Object parent, Object child)
    {
	//Debug.log("getIndexOfChild: " + parent + " " + child);

	if (parent instanceof File) {
	    File[] children = getSubDirs((File)parent, true);

	    if (children == null) 
		return -1;

	    for (int i = 0; i < children.length; i++) {
		if (children[i] == child)
		    return i;
	    }
	}
	else if (parent instanceof String) {
	    if (parent.equals(root)) {
		return subRoots.indexOf(child);
	    }
	    return ((Vector)subChildren.get(parent)).indexOf(child);
	}

	return -1; 
    }

    private File[] getSubDirs(File f, boolean doSort)
    {
	if (f == cachedDir && cacheSorted == doSort)
	    return cache;

	//System.out.println("getSubDirs: " + f);

	File[] children = ((File)f).listFiles(); 
	int count = 0;
	if (children != null) {
	    for (int i = 0; i < children.length; i++) {
		if (children[i].isDirectory()) {
		    if (filterDotFiles 
			&& children[i].getName().startsWith("."))
			continue;

		    children[count] = children[i];
		    count++;
		}
	    }
	}

	cache = new File[count];
	System.arraycopy(children, 0, cache, 0, count);

	if (doSort)
	    Arrays.sort(cache, this);

	cachedDir = f;
	cacheSorted = doSort;

	return cache;
    }

    public void addSubRoot(String s)
    {
	if (subRoots.contains(s))
	    return;
	
	subRoots.add(s);
	
	/* add vector for children of this node */
	subChildren.put(s, new Vector());

	Object[] path = { root };
	int[] indices = { subRoots.size() - 1 };
	Object[] children = { s };
	fireTreeNodesInserted(new TreeModelEvent(this, path, indices,
						 children));
	
    }

    public void addSubRoot(File f)
    {
	if (subRoots.contains(f))
 	    return;
	
 	subRoots.add(f);
	
 	Object[] path = { root };
 	int[] indices = { subRoots.size() - 1 };
 	Object[] children = { f };
 	fireTreeNodesInserted(new TreeModelEvent(this, path, indices, 
 						 children));
    }

    public void removeSubRoots()
    {
	/* remove respective vectors in hash tree */
	for (int i = 0; i < subRoots.size(); i++)
	    subChildren.remove(subRoots.get(i));
	
	subRoots.clear();
	
	Object[] path = { root };
	fireTreeStructureChanged(new TreeModelEvent(this, path));
    }
    
    public void removeChildrenOfSubRoot(String s)
    {
	if (!subRoots.contains(s))
	    return;

	Vector v = (Vector)subChildren.get(s);

	v.removeAllElements();
		
	Object[] path = { root, s };
	fireTreeStructureChanged(new TreeModelEvent(this, path));
    }

    public void addChildOfSubRoot(File f, String s)
    {
	addChildOfSubRoot(f, s, null);
    }

    public void addChildOfSubRoot(File f, String s, String label)
    {
	addSubRoot(s);
	
	Vector kids = (Vector) subChildren.get(s);
	
	FileNode node = new FileNode(f, true, label);
	
	if (kids.contains(node))
  	    return;
	
	kids.add(node);
	Object[] path = { root, s };
	int[] indices = { kids.size() - 1 };
	Object[] children = { f };
	fireTreeNodesInserted(new TreeModelEvent(this, path, indices,
						 children ));
    }

    public void removeChildOfSubRoot(File f, String s)
    {
	if (!subRoots.contains(s))
	    return;
	    
	Vector kids = (Vector) subChildren.get(s);
	
	FileNode node = new FileNode(f, true);

	int index = kids.indexOf(node);

	if (index == -1)
	    return;
	
	kids.remove(index);

	Object[] path = { root, s };
	int[] indices = { index };
	Object[] children = { node };
	fireTreeNodesRemoved(new TreeModelEvent(this, path, indices,
						children));
    }



}

