/*
 *  XNap
 *
 *  A pure java file sharing client.
 *
 *  See AUTHORS for copyright information.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

/**
 * If you hack around, make sure all pathes are returned with a trailing
 * file separator and all values are initialized to something other than
 * null.
 *
 * Most important, keep all preferences sorted alphabetically everywhere.
 */
package xnap.plugin.nap.util;

import xnap.XNap;
import xnap.plugin.nap.net.GlobalUser;
import xnap.user.UserManager;
import xnap.util.PluginPreferencesSupport;
import xnap.util.QuotedStringTokenizer;
import xnap.util.StringHelper;
import xnap.util.FileHelper;
import xnap.util.prefs.IntValidator;
import xnap.util.prefs.PortRangeValidator;

import java.util.*;

public class NapPreferences extends PluginPreferencesSupport {

    //--- Constant(s) ---

    public static final int VERSION = 5;

    //--- Data field(s) ---

    private static NapPreferences singleton = null;

    //--- Constructor(s) ---

    private NapPreferences()
    {
	super("plugin.nap", VERSION);

	setDefault("autoFetchNaptigator", "true");
	setDefault("autoLoadNaptigator", "true");
	setDefault("autoSaveNaptigator", "true");
	setDefault("clientInfo", "XNap " + XNap.VERSION);
	setDefault("customizeClientInfo", "false");
	setDefault("fakeNetworks", "n/a");
	setDefault("filterNetworks", "Napster");
	setDefault("hotlistUsers", "");
	setDefault("limitSharesPerServer", "false");
	setDefault("localPort", "6600-6700", new PortRangeValidator());
	setDefault("maxAutoconnectServers", "10");
	setDefault("maxLoginsPerNetwork", "1");
	setDefault("maxPacketsPerTick", "50");
	setDefault("maxSearchResultsPerServer", "200",
		   new IntValidator(0, Integer.MAX_VALUE));
	setDefault("maxSearchServers", "5");
	setDefault("maxSharesPerServer", "500");
	setDefault("minimumShares", "0");
	setDefault("minimumSharesMessage", "Sorry, you are not allowed to"
		   + " download since you do not share enough.");
	setDefault("napigatorFile", FileHelper.getHomeDir() 
		   + "napigator_hosts");
	setDefault("napigatorURL", "http://www.napigator.com/servers.php"
		   + "?version=112&client=napigator");
	setDefault("removeFailedServers", "false");
	setDefault("sendMinimumSharesMessage", "false");
	setDefault("sendWholeRepository", "false");
	setDefault("serverFile", FileHelper.getHomeDir() + "hosts");
	setDefault("serverTableColumns", "0;1;4;5;6;7");
	setDefault("useAutoconnector", "true");
	setDefault("useMinimumShares", "false");
	setDefault("tickLength", "100");
	setDefault("whoisQueryOnTransfer", "false");
	setDefault("whoisTableColumns", "0;2;3;4;5;6;7;8");

	if (!getCustomizeClientInfo()) {
	    setClientInfo("XNap " + XNap.VERSION);
	}
    }

    //--- Method(s) ---

    public static synchronized NapPreferences getInstance() {
	if (singleton == null) {
	    singleton = new NapPreferences();
	}

	return singleton;
    }

    /**
     * Sub classes should overwrite this.
     */
    public void convert(int oldVersion)
    {
	if (oldVersion <= 0) {
	    setTableColumns("user", "0;2;3;4;5;6;7;8");
	    setTableColumns("server", "0;1;4;5;6;7");
	}
	
	if (oldVersion <= 1) {
	    removeProperty("xnap.plugin.nap.userTableColumns");
	    renameProperty("xnap.serverTableColumns", 
			   "xnap.plugin.nap.serverTableColumns");
	    renameProperty("xnap.serverTableColumnWidth", 
			   "xnap.plugin.nap.serverTableColumnWidth");
	    renameProperty("xnap.serverTableMaintainSortOrder", 
			   "xnap.plugin.nap.serverTableMaintainSortOrder");
	    renameProperty("xnap.serverTableSortedColumn", 
			   "xnap.plugin.nap.serverTableSortedColumn");
	}

	if (oldVersion <= 2) {
	    // deprecated, makes no sense
	    // just make sure nobody unchecked this by accident
	    //setUseSinglePort(true);
	}
	
	if (oldVersion <= 3) {
	    removeProperty("hotlistDividerLocation");
	    removeProperty("hotlistTableColumns");

	    LinkedList l 
		= StringHelper.toList(get("bannedUsers"), ARRAY_SEPARATOR);
	    for (Iterator i = l.iterator(); i.hasNext();) {
		GlobalUser u = new GlobalUser(i.next().toString(), false);
		u.setMaxUploads(0);
		u.setCategory(XNap.tr("Banned"));
		UserManager.getInstance().add(u);
	    }
	    removeProperty("bannedUsers");

	    QuotedStringTokenizer t 
		= new QuotedStringTokenizer(get("hotlistUsers"));
	    while (t.hasMoreTokens()) {
		GlobalUser u = new GlobalUser(t.nextToken(), false);
		UserManager.getInstance().add(u);
	    }
	    UserManager.getInstance().write();
	    removeProperty("hotlistUsers");
	}

	if (oldVersion <= 4) {
	    removeProperty("useSinglePort");
	}
    }

    public void destroy() {
	singleton = null;
    }

    public boolean getAutoFetchNapigator() {
	return getBoolean("autoFetchNaptigator");
    }
    
    public void setAutoFetchNapigator(boolean newValue) {
	set("autoFetchNaptigator", newValue);
    }

    public boolean getAutoLoadNapigator() {
	return getBoolean("autoLoadNaptigator");
    }
    
    public void setAutoLoadNapigator(boolean newValue) {
	set("autoLoadNaptigator", newValue);
    }

    public boolean getAutoSaveNapigator() {
	return getBoolean("autoSaveNaptigator");
    }
    
    public void setAutoSaveNapigator(boolean newValue) {
	set("autoSaveNaptigator", newValue);
    }
    
    public String getClientInfo() {
        return get("clientInfo");
    }

    public void setClientInfo(String newValue) {
	if (newValue.equals("")) {
	    return;
	}
	set("clientInfo", newValue);
    }

    public boolean getCustomizeClientInfo() {
	return getBoolean("customizeClientInfo");
    }
    
    public void setCustomizeClientInfo(boolean newValue) {
	set("customizeClientInfo", newValue);
    }

    public String getFakeNetworks() {
	return get("fakeNetworks");
    }

    public String getFilterNetworks() {
	return get("filterNetworks");
    }

    public boolean getLimitSharesPerServer() {
	return getBoolean("limitSharesPerServer");
    }
    
    public void setLimitSharesPerServer(boolean newValue) {
	set("limitSharesPerServer", newValue);
    }

    public String getLocalPortRange() {
        return get("localPort");
    }

    public void setLocalPortRange(String newValue) {
	set("localPort", newValue);
    }

    public int getMaxAutoconnectServers() {
	return getInt("maxAutoconnectServers");
    }

    public void setMaxAutoconnectServers(int newValue) {
	if (newValue < 0) {
	    return;
	}
	set("maxAutoconnectServers", newValue);
    }

    public int getMaxLoginsPerNetwork() {
        return getInt("maxLoginsPerNetwork");
    }

    public int getMaxPacketsPerTick() {
	return getInt("maxPacketsPerTick");
    }

    public int getMaxSearchResultsPerServer() {
        return getInt("maxSearchResultsPerServer");
    }

    public void setMaxSearchResultsPerServer(int newValue) {
	set("maxSearchResultsPerServer", newValue);
    }

    public int getMaxSearchServers() {
        return getInt("maxSearchServers");
    }

    public void setMaxSearchServers(int newValue) {
	if (newValue < 1) {
	    return;
	}
	set("maxSearchServers", newValue);
    }

    public int getMaxSharesPerServer() {
        return getInt("maxSharesPerServer");
    }

    public void setMaxSharesPerServer(int newValue) {
	set("maxSharesPerServer", newValue);
    }

    public int getMinimumShares() {
        return getInt("minimumShares");
    }

    public void setMinimumShares(int newValue) {
	set("minimumShares", newValue);
    }

    public String getMinimumSharesMessage() {
	return get("minimumSharesMessage");
    }
    
    public void setMinimumSharesMessage(String newValue) {
	set("minimumSharesMessage", newValue);
    }

    public String getNapigatorFile() {
	return get("napigatorFile");
    }
    
    public String getNapigatorURL() {
	return get("napigatorURL");
    }

    /**
     * Hidden.
     */
    public boolean getPreferSlavanap() {
	if (get("preferSlavanap").length() > 0) {
	    return getBoolean("preferSlavanap");
	}
	return false;
    }

    public boolean getRemoveFailedServers() {
	return getBoolean("removeFailedServers");
    }
    
    public void setRemoveFailedServers(boolean newValue) {
	set("removeFailedServers", newValue);
    }

    public boolean getSendMinimumSharesMessage() {
	return getBoolean("sendMinimumSharesMessage");
    }

    public void setSendMinimumSharesMessage(boolean newValue) {
	set("sendMinimumSharesMessage", newValue);
    }

    public boolean getSendWholeRepository() {
	return getBoolean("sendWholeRepository");
    }

    public void setSendWholeRepository(boolean newValue) {
	set("sendWholeRepository", newValue);
    }

    public String getServerFile() {
	return get("serverFile");
    }

    public String getServerTableColumns() {
        return get("serverTableColumns");
    }

    public String[] getServerTableColumnsArray() 
    {
	return StringHelper.toArray(getServerTableColumns(), ARRAY_SEPARATOR);
    }

    public void setServerTableColumns(String newValue) {
	set("serverTableColumns", newValue);
    }

    public void setServerTableColumns(Collection c) {
	setServerTableColumns(StringHelper.toString(c, ARRAY_SEPARATOR));
    }

    public boolean getUseAutoconnector() {
        return getBoolean("useAutoconnector");
    }

    public void setUseAutoconnector(boolean newValue) {
	set("useAutoconnector", newValue);
    }

    public boolean getUseMinimumShares() {
        return getBoolean("useMinimumShares");
    }

    public void setUseMinimumShares(boolean newValue) {
	set("useMinimumShares", newValue);
    }

    public long getTickLength() {
	return getLong("tickLength");
    }

    public boolean getWhoisQueryOnTransfer() {
	return getBoolean("whoisQueryOnTransfer");
    }

    public void setWhoisQueryOnTransfer(boolean newValue) {
	set("whoisQueryOnTransfer", newValue);
    }
}
