/*
 *  Java Napster version x.yz (for current version number as well as for
 *  additional information see version.txt)
 *
 *  Previous versions of this program were written by Florian Student
 *  and Michael Ransburg available at www.weblicity.de/jnapster and
 *  http://www.tux.org/~daneel/content/projects/10.shtml respectively.
 *
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package xnap.cmdl;

import xnap.*;
import xnap.net.*;
import xnap.plugin.*;
import xnap.util.*;
import xnap.util.event.*;

import java.io.*;
import java.net.*;
import java.util.*;

/**
 * This class is thread safe.
 */
public class Executer implements CommandLineSupport {

    //--- Data Field(s) ---
    
    private static Executer singleton = new Executer();
    private static Console console = Console.getInstance();

    static {
	installHandler(singleton);
    }

    private AbstractCommand[] commands = {
	new HelpCmd(), new LicenseCmd(), new ReadmeCmd()
    };
    private Hashtable cmdHandlers = new Hashtable();
    private Completer fallbackCompleter = new Completer();

    //--- Constructor(s) ---
    
    private Executer() 
    {
    }
    
    //--- Method(s) ---


    public AbstractCommand[] getCommands()
    {
	return commands;
    }

    /**
     * Always returns a valid completer.
     */
    public static Completer getCompleter()
    {
	Completer c = console.getCompleter();
	return (c != null) ? c : singleton.fallbackCompleter;
    }

    /**
     * Adds command. Only the first command is added to the readline completer.
     */
    public static synchronized void addCommand(AbstractCommand command)
    {
	String[] cmds = command.getCommands();
	for (int i = 0; i < cmds.length; i++) {
	    singleton.cmdHandlers.put(cmds[i], command);
	    if (i == 0) {
		getCompleter().add(cmds[i]);
	    }
	}
    }

    public static synchronized void removeCommand(AbstractCommand command)
    {
	String[] cmds = command.getCommands();
	for (int i = 0; i < cmds.length; i++) {
	    if (singleton.cmdHandlers.get(cmds[i]) == command) {
		singleton.cmdHandlers.remove(cmds[i]);
		if (i == 0) {
		    getCompleter().remove(cmds[i]);
		}
	    }
	}
    }

    public static void installHandler(CommandLineSupport cls)
    {
	AbstractCommand[] cmds = cls.getCommands();
	for (int i = 0; i < cmds.length; i++) {
	    addCommand(cmds[i]);
	}
    }

    public static void removeHandler(CommandLineSupport cls)
    {
	AbstractCommand[] cmds = cls.getCommands();
	for (int i = 0; i < cmds.length; i++) {
	    removeCommand(cmds[i]);
	}
    }
    public static synchronized boolean parseCommand(String args)
    {
	if (args == null) {
	    return false;
	}
	args = args.trim();
  	if (args.startsWith("/")) {
	    args = args.substring(1);
	}
	if (args.length() == 0) {
	    return true;
	}

	if (!console.isEchoing()) {
	    console.println("XNap> " + args);
	}

	QuotedStringTokenizer t = new QuotedStringTokenizer(args);
	String[] argv = t.getTokens();

	AbstractCommand handler = singleton.getHandler(argv[0]);
	if (handler != null) {
	    try {
		if (!handler.execute(argv)) {
		    throw new SyntaxException();
		}
	    }
	    catch (SyntaxException e) {
		console.println("Usage: " + argv[0] + " "
				+ handler.getParameter());
	    }
	    catch (ParameterException e) {
		console.println(e.getMessage());
	    }
	}
	else {
	    console.println("unknown command (h for help)");
	}

	return true;
    }

    protected AbstractCommand getHandler(String command)
    {
	return (AbstractCommand)cmdHandlers.get(command);
    }

    protected class HelpCmd extends AbstractCommand
    {
	public HelpCmd()
	{
	    putValue(CMD, new String[] {"help", "h", "?"});
	    putValue(PARAMETER, "[command]");
	    putValue(SHORT_HELP, "Shows help about command.");
	}
	
	public boolean execute(String[] argv)
	{
	    if (argv.length == 1) {
		String[] cmds = getCompleter().getCompletions();
		for (int i = 0; i < cmds.length; i++) {
		    AbstractCommand handler = singleton.getHandler(cmds[i]);
		    if (handler != null) {
			console.println(cmds[i] + " - " 
					+ handler.getShortHelp());
		    }
		}
	    }
	    else if (argv.length == 2) {
		AbstractCommand handler = singleton.getHandler(argv[1]);
		if (handler != null) {
		    String[] cmds = handler.getCommands();
		    console.println("Command: " + cmds[0] + " "
				    + handler.getParameter());
		    StringBuffer s = new StringBuffer();
		    for (int i = 1; i < cmds.length; i++) {
			s.append(" ");
			s.append(cmds[i]);
		    }
		    console.println("Aliases:" + s.toString());
		    console.println("Help: " + handler.getLongHelp());
		}
		else {
		    console.println("Command not found.");
		}
	    }
	    else {
		return false;
	    }

	    return true;
	}
    }

    protected class LicenseCmd extends AbstractCommand
    {
	public LicenseCmd()
	{
	    putValue(CMD, new String[] {"license"});
	    putValue(SHORT_HELP, "Print license.");
	}
	
	public boolean execute(String[] parms)
	{
	    console.printFile("COPYING");
	    return true;
	}
    }

    protected class ReadmeCmd extends AbstractCommand
    {
	public ReadmeCmd()
	{
	    putValue(CMD, new String[] {"readme"});
	    putValue(SHORT_HELP, "Print readme.");
	}
	
	public boolean execute(String[] parms)
	{
	    console.printFile("README");
	    return true;
	}
    }
    
}
