/*
 *  XNap
 *
 *  A pure java file sharing client.
 *
 *  See AUTHORS for copyright information.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package xnap.gui;

import xnap.XNap;
import xnap.util.FileHelper;
import xnap.util.Updater;

import java.awt.*;
import java.awt.event.*;
import java.io.File;
import javax.swing.*;

public class DirectoryPanel extends JPanel {
    
    //--- Data field(s) ---

    private JTextField jtfFilename;

    //--- Constructor(s) ---

    public DirectoryPanel(String text, int size) 
    {
	setLayout(new BoxLayout(this, BoxLayout.X_AXIS));

	jtfFilename = new JTextField(text, size);
	add(jtfFilename);

	add(Box.createHorizontalStrut(1));

	// This works around a bug in Mac OS X.
	// When a modal dialog (this dialog) is shown on top of another modal 
	// dialog (like the wizard), the second dialog does not respond to
	// gui events and gets stuck in the background.
	Action action;
	if (Updater.isMacOSX()) {
	    action = new SwingDirectoryChooserAction();
	}
	else {
	    action = new DirectoryChooserAction();
	}
	JButton jbChooser = new JButton(action);
	jbChooser.setMargin(new Insets(1, 1, 1, 1));
	add(jbChooser);
    }

    // --- Method(s) ---

    /**
     * Sub classes can overwrite this.
     */
    protected void directorySelected(String dir)
    {
    }

    public String getDirectory()
    {
	return FileHelper.directory(jtfFilename.getText());
    }

    public void setDirectory(String dir)
    {
	jtfFilename.setText(dir);
    }

    public JTextField getTextField()
    {
	return jtfFilename;
    }

    // --- Inner Class(es) ---

    private class DirectoryChooserAction extends AbstractAction {

	private DirectoryChooser chooser;
	
        public DirectoryChooserAction() 
	{
	    chooser = new DirectoryChooser();

	    putValue(Action.SMALL_ICON,
		     XNapFrame.getSmallIcon("fileopen.png"));
            putValue(Action.SHORT_DESCRIPTION, XNap.tr("Choose a directory"));
	}

        public void actionPerformed(ActionEvent event) 
	{
	    chooser.setSelectedDirectory(new File(getDirectory()));

	    if (chooser.showChooseDialog(DirectoryPanel.this) 
		== DirectoryChooser.APPROVE_OPTION) {
		setDirectory(chooser.getSelectedDirectory().getAbsolutePath());
		directorySelected(getDirectory());
	    }
	}
    } 

    private class SwingDirectoryChooserAction extends AbstractAction {

	private JFileChooser chooser;
	
        public SwingDirectoryChooserAction() 
	{
	    chooser = new JFileChooser();
	    chooser.setApproveButtonText(XNap.tr("OK"));
	    chooser.setDialogTitle(XNap.tr("Choose Directory"));
	    chooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);

	    putValue(Action.SMALL_ICON,
		     XNapFrame.getSmallIcon("fileopen.png"));
            putValue(Action.SHORT_DESCRIPTION, XNap.tr("Choose a directory"));
	}

        public void actionPerformed(ActionEvent event) 
	{
	    chooser.setSelectedFile(new File(getDirectory()));
	    
	    if (chooser.showOpenDialog(DirectoryPanel.this)
		== JFileChooser.APPROVE_OPTION) {
		setDirectory(chooser.getSelectedFile().getAbsolutePath());
		directorySelected(getDirectory());
	    }
	}
    } 

}
