/*
 *  XNap
 *
 *  A pure java file sharing client.
 *
 *  See AUTHORS for copyright information.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package xnap.gui.table;

import xnap.*;
import xnap.gui.table.*;
import xnap.net.*;
import xnap.plugin.INetworkPlugin;
import xnap.util.*;
import xnap.util.event.*;

import java.util.*;
import javax.swing.event.*;
import javax.swing.table.*;

public class UserTableModel extends AbstractDynamicTableModel
{

    //--- Constant(s) ---

    public static final int NAME = 0;
    public static final int FILE_COUNT = 1;
    public static final int LINK_SPEED = 2;
    public static final int STATUS = 3;
    public static final int DOWNLOADS = 4;
    public static final int UPLOADS = 5;
    public static final int CLIENT = 6;
    public static final int COMMENT = 7;

    //--- Data field(s) ---

    protected Column columns[] = new Column[] {
	new Column(XNap.tr("User"), String.class),
	new Column(XNap.tr("File Count"), Integer.class, 
		   new NumberCellRenderer()),
	new Column(XNap.tr("Link Speed"), String.class, 
		   new LinkSpeedCellRenderer()),
	new Column(XNap.tr("Status"), String.class),
	new Column(XNap.tr("Downloads"), Integer.class,
		   new NumberCellRenderer()),
	new Column(XNap.tr("Uploads"), Integer.class,
		   new NumberCellRenderer()),
	new Column(XNap.tr("Client"), String.class),
	new Column(XNap.tr("Comment"), String.class),
    };

    protected ArrayList rows = new ArrayList();

    //--- Constructor(s) ---

    public UserTableModel(String table) 
    {
	super(table, Preferences.getInstance());
	setColumns(columns);
    }

    //--- Method(s) ---

    public void add(IUser u, boolean filterDuplicates)
    {
	if (!filterDuplicates || !contains(u)) {
	    rows.add(u);
	    fireTableRowsInserted(rows.size() - 1, rows.size() - 1);
	}
    }

    public void add(IUser u)
    {
	add(u, true);
    }

    public void clear()
    {
	rows.clear();
	fireTableDataChanged();
    }

    public boolean contains(IUser u)
    {
	return rows.contains(u);
    }

    public IUser get(int i)
    {
	return (IUser)rows.get(mapToIndex(i));
    }

    public int getRowCount() 
    {
        return rows.size();
    }
    
    public Object get(int i, int j) 
    {
        if (i >= rows.size()) 
	    return null;

        IUser u = (IUser)rows.get(i);

        if (u == null) 
	    return null;

        switch (j) {
	case NAME:
	    return u.getName();
	case FILE_COUNT:
	    return new Integer(u.getFileCount());
	case LINK_SPEED:
	    return (new LinkSpeed(u.getLinkSpeed())).toString();
	case STATUS:
	    return u.getStatusText();
	case DOWNLOADS:
	    return new Integer(u.getLocalDownloadCount());
	case UPLOADS:
	    return new Integer(u.getLocalUploadCount());
	case CLIENT:
	    return u.getClientInfo();
	case COMMENT:
	    return u.getComment();
	default:
	    return "";
        }
    }

    public String getTableName()
    {
	return XNap.tr("User Table");
    }

    public int indexOf(IUser u)
    {
	int i = rows.indexOf(u);
	return i != -1 ? mapToIndex(i) : -1;
    }

    public void remove(IUser u)
    {
	int i = rows.indexOf(u);
	if (i != -1) {
	    rows.remove(u);
	    fireTableRowsDeleted(i, i);
	}
    }

}


