/*
 *  XNap
 *
 *  A pure java file sharing client.
 *
 *  See AUTHORS for copyright information.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package xnap.plugin.nap;

import xnap.gui.*;
import xnap.gui.UserActionFactory;
import xnap.gui.action.*;
import xnap.net.IUser;
import xnap.user.UserManager;
import xnap.plugin.*;
import xnap.plugin.nap.gui.*;
import xnap.plugin.nap.net.DirectBrowse;
import xnap.plugin.nap.net.GlobalUser;
import xnap.plugin.nap.net.User;
import xnap.plugin.nap.net.msg.MessageHandler;
import xnap.plugin.nap.net.msg.client.WhoisRequestMessage;
import xnap.plugin.nap.util.*;
import xnap.util.*;
import xnap.util.event.*;

import java.awt.*;
import java.awt.event.*;
import java.beans.*;
import java.util.*;
import javax.swing.*;
import javax.swing.border.*;

import java.util.*;

public class GUIPlugin extends Plugin implements IGUIPlugin, StatusListener
{
    
    //--- Data Field(s) ---

    private JMenu menu;
    private JTabbedPane napPane;
    private ConsolePanel consolePanel;
    private ServerPanel serverPanel;
    private WhoisPanel whoisPanel;    
    private StatusPanel statusPanel;

    private Preferences prefs = Preferences.getInstance();

    //--- Method(s) ---

    public void start()
    {
	napPane = new JTabbedPane();
	napPane.setName("opennap");

	serverPanel = new ServerPanel();
	napPane.add(Plugin.tr("Server"), serverPanel);

      	whoisPanel = new WhoisPanel();
	whoisPanel.setName("whois");
    	napPane.add(Plugin.tr("Whois"), whoisPanel);

	consolePanel = new ConsolePanel();
	napPane.add(Plugin.tr("Console"), consolePanel);
	
	// instantiate, if not already
	menu = getMenu();

	JMenu jmStartup = new JMenu(Plugin.tr("Startup"));
	menu.add(jmStartup);

	jmStartup.add((new UseAutoConnectorAction()).getCheckBoxMenuItem());

	menu.addSeparator();

	menu.add(new JMenuItem(new AddUserToHotlistAction()));

	menu.addSeparator();

        JMenuItem jmiNapigator 
	    = new JMenuItem(serverPanel.getAskNapigatorAction());
	// FIX: looks like JDK 1.3.1 ignores the accel key of the action
	jmiNapigator.setAccelerator
	    (KeyStroke.getKeyStroke(KeyEvent.VK_N, Event.CTRL_MASK));
        menu.add(jmiNapigator);

	// stats updates
	statusPanel = new StatusPanel(this);
	Connector.getInstance().addStatsListener(this);

	UserActionManager.getInstance().install
	    (GlobalUser.class, new GlobalActionFactory());
	UserActionManager.getInstance().install
	    (User.class, new ActionFactory());

	super.start();
    }

    public void stop()
    {
	menu.removeAll();

	UserActionManager.getInstance().uninstall(GlobalUser.class);
	UserActionManager.getInstance().uninstall(User.class);
	Connector.getInstance().removeStatsListener(this);

	super.stop();

	consolePanel.die();
    }

    public Icon getIcon()
    {
	return XNapFrame.getBigIcon("network_local.png");
    }

    public JMenu getMenu()
    {
	if (menu == null) {
	    menu = new JMenu("OpenNap");
	}
	return menu;
    }

    public JComponent getPanel()
    {
	return napPane;
    }

    public Icon getPreferencesIcon()
    {
	return XNapFrame.getSmallIcon("network_local.png");
    }

    public AbstractPreferencesPanel[] getPrefsPanels()
    {
	return new AbstractPreferencesPanel[]
	    { new NapPreferencesPanel(), new NapAdvancedPreferencesPanel() };
    }

    public AbstractPreferencesPanel getWizardPanel()
    {
	return new WizardPanel();
    }

    public JComponent getStatusPanel()
    {
	return statusPanel;
    }

    public void setStatus(String newValue)
    {
	statusPanel.update();
    }

    public void setStatusListener(StatusListener newValue) 
    {
	serverPanel.setStatusListener(newValue);
	whoisPanel.setStatusListener(newValue);
	consolePanel.setStatusListener(newValue);
    }

    public class ActionFactory implements UserActionFactory
    {
	public AbstractAction[] create(IUser[] users)
	{
	    User[] napUsers = new User[users.length];
	    System.arraycopy(users, 0, napUsers, 0, napUsers.length);

	    return new AbstractAction[] {
		new DirectBrowseAction(napUsers),
		new BrowseAction(users), 
		new ChatAction(users),
		new WhoisAction(napUsers), 
		new LookupAction(users),
		new AddToHotlistAction(users),
		new EditUserAction(users),
	    };
	}
    }

    public class GlobalActionFactory implements UserActionFactory
    {
	public AbstractAction[] create(IUser[] users)
	{
	    return new AbstractAction[] {
		new EditUserAction(users),
		new LookupAction(users),
		new BrowseAction(users),
		new ChatAction(users),
	    };
	}
    }

    public class AddUserToHotlistAction extends AbstractAction
    {

	public AddUserToHotlistAction()
	{
            putValue(Action.NAME, Plugin.tr("Add User To Hotlist") + "...");
            putValue(Action.SHORT_DESCRIPTION, 
		     Plugin.tr("Add any user to the hotlist."));
	    putValue(Action.SMALL_ICON, XNapFrame.getIcon("edit_add.png"));
        }

        public void actionPerformed(ActionEvent event) 
	{
	    String nick = JOptionPane.showInputDialog
		(napPane, Plugin.tr("Nick"), 
		 Plugin.tr("Add User To Hotlist"), 
		 JOptionPane.QUESTION_MESSAGE);
		
	    if (nick != null && nick.trim().length() > 0) {
		IUser u = User.getParentByName(nick.trim());
		u.setCategory(Plugin.tr("Hotlist"));
		u.setTemporary(false);
		UserManager.getInstance().write();
	    }
	}
    }

    public class DirectBrowseAction extends AbstractAction
    {
	private User[] users;

	public DirectBrowseAction(User[] users)
	{
	    this.users = users;

            putValue(Action.NAME, Plugin.tr("Direct Browse"));
            putValue(Action.SHORT_DESCRIPTION, 
		     Plugin.tr("Directly browse selected user"));
	    putValue(Action.SMALL_ICON, XNapFrame.getIcon("run.png"));
        }

        public void actionPerformed(ActionEvent event) 
	{
	    boolean focus = (event.getModifiers() & ActionEvent.CTRL_MASK) > 0;
	    for (int i = 0; i < users.length; i++) {
		if (focus) {
		    XNapFrame.setNextFocusPolicy(!prefs.getFocusOnAllEvents());
		}
		SearchManager.getInstance().browse(new DirectBrowse(users[i]));
	    }
	}
    }

    private class LookupAction extends AbstractAction 
    {
	private IUser[] users;

        public LookupAction(IUser[] users)
	{
	    this.users = users;

            putValue(Action.NAME, Plugin.tr("Lookup"));
            putValue(Action.SHORT_DESCRIPTION, 
		    Plugin.tr("Do a whois query on all connected servers"));
	    putValue(Action.SMALL_ICON, XNapFrame.getIcon("viewmag_plus.png"));
        }

        public void actionPerformed(ActionEvent event) 
	{
	    for (int i = 0; i < users.length; i++) {
		WhoisRequestMessage msg 
		    = new WhoisRequestMessage(users[i].getName());
		MessageHandler.send(msg);
	    }
	    if ((event.getModifiers() & event.CTRL_MASK) > 0) {
		XNapFrame.setNextFocusPolicy(!prefs.getFocusOnAllEvents());
	    }
	    XNapFrame.setFocusTo("opennap.whois");
        }

    }

    public class WhoisAction extends AbstractAction
    {
	private User[] users;

	public WhoisAction(User[] users)
	{
	    this.users = users;

            putValue(Action.NAME, Plugin.tr("Whois"));
            putValue(Action.SHORT_DESCRIPTION, 
		     Plugin.tr("Get information about selected user"));
	    putValue(Action.SMALL_ICON, XNapFrame.getIcon("viewmag.png"));
        }

        public void actionPerformed(ActionEvent event) 
	{
	    for (int i = 0; i < users.length; i++) {
		whoisPanel.addUser(users[i]);
	    }
	    
	    if ((event.getModifiers() & event.CTRL_MASK) > 0) {
		XNapFrame.setNextFocusPolicy(!prefs.getFocusOnAllEvents());
	    }
	    XNapFrame.setFocusTo("opennap.whois");
	}
    }

    public class UseAutoConnectorAction extends AbstractAction
	implements PropertyChangeListener
    {
	JCheckBoxMenuItem jmi;

	public UseAutoConnectorAction()
	{
            putValue(Action.NAME, Plugin.tr("Connect automatically"));

	    jmi = new JCheckBoxMenuItem(this);
	    jmi.setSelected(napPrefs.getUseAutoconnector());

	    napPrefs.addPropertyChangeListener("useAutoconnector", this);
        }

        public void actionPerformed(ActionEvent event) 
	{
	    napPrefs.setUseAutoconnector(jmi.isSelected());
	}

	public JCheckBoxMenuItem getCheckBoxMenuItem()
	{
	    return jmi;
	}

	public void propertyChange(PropertyChangeEvent e)
	{
	    jmi.setSelected(napPrefs.getUseAutoconnector());
	}

    }

}
