/*
 *  XNap
 *
 *  A pure java file sharing client.
 *
 *  See AUTHORS for copyright information.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package xnap.plugin.nap.net.msg;

import xnap.plugin.nap.Plugin;
import xnap.plugin.nap.net.Server;
import xnap.plugin.nap.util.Connector;
import xnap.plugin.nap.net.msg.client.ClientMessage;

import java.io.*;
import java.util.*;
import org.apache.log4j.Logger;

public class MessageSender {

    //--- Constant(s) ---

    /**
     * Start with this many send worker threads. More are spawned as needed.
     */
    public static final int SEND_WORKER_COUNT = 3;

    //--- Data field(s) ---

    protected static Logger logger = Logger.getLogger(MessageSender.class);
    protected static MessageSender singleton;

    protected boolean died = true;
    /**
     * Queues messages.
     */
    protected LinkedList queue = new LinkedList();
    protected HashSet lockedServers = new HashSet();
    protected LinkedList sendWorkers = new LinkedList();

    //--- Constructor(s) ---

    protected MessageSender()
    {
    }
    
    //--- Method(s) ---

    public static synchronized MessageSender getInstance()
    {
	if (singleton == null) {
	    singleton = new MessageSender();
	}

	return singleton;
    }

    /**
     * This should be called every once in a while. Spawns a new SendWorker 
     * if needed. We always want to have at least one SendWorker ready to 
     * rumble.
     */
    public static void ensureLiveness()
    {
	if (getInstance().lockedServers.size() 
	    >= getInstance().sendWorkers.size()) {
	    getInstance().startNewSendWorker();
	}
    }

    /**
     * Asynchronously sends <code>msg</code>.
     */
    public static void send(Server server, ClientMessage msg, 
			    boolean lowPriority)
    {
	getInstance().add(server, msg, lowPriority, true);
    }

    /**
     * Sends <code>msg</code> to all connected servers.
     */
    public static void send(ClientMessage msg)
    {
	Object[] servers 
	    = Connector.getInstance().getConnectedServers().toArray();
	for (int i = 0; i < servers.length; i++) {
	    getInstance().add((Server)servers[i], msg, true, true);
	}
    }

    public static void sendLater(Server server, ClientMessage msg)
    {
	getInstance().add(server, msg, true, false);
    }

    public static void sendPending(Server server)
    {
	getInstance().notifyServer(server);
    }

    public synchronized void add(Server server, ClientMessage msg, 
				 boolean lowPriority, boolean notify)
    {
	if (lowPriority) {
	    queue.addLast(new SendMessage(server, msg));
	}
	else {
	    queue.addFirst(new SendMessage(server, msg));
	}
	if (notify) {
	    notifyServer(server);
	}
    }

    /**
     * Worker could not send message, because server has died. 
     * Remove all pending messages for server.
     */
    public synchronized void failed(Server server)
    {
	logger.debug("removing all messages for: " + server);
	for (Iterator i = queue.iterator(); i.hasNext();) {
	    SendMessage message = (SendMessage)i.next();
	    if (message.server == server) {
		i.remove();
		if (message.msg.listener != null) {
		    message.msg.listener.exceptionThrown
			(new IOException(Plugin.tr("Server disconnected")));
		}
	    }
	}
    }

    public Stats getStats()
    {
	return new Stats();
    }

    public boolean hasDied()
    {
	return died;
    }

    public synchronized SendMessage next()
    {
	logger.debug("fetching next: send queue size: " + queue.size()
		     + ", " + lockedServers.size() + " locked servers");

	for (Iterator i = queue.iterator(); i.hasNext();) {
	    SendMessage message = (SendMessage)i.next();
	    if (!lockedServers.contains(message.server)) {
		i.remove();
		lockedServers.add(message.server);
		// FIX: we should not call it that often
		ensureLiveness();
		return message;
	    }
	}

	return null;
    }

    public synchronized void notifyServer(Server server)
    {
	if (!lockedServers.contains(server)) {
	    notify();
	}
    }

    public synchronized void sent(SendMessage message)
    {
	lockedServers.remove(message.server);
    }

    public void start()
    {
	died = false;

	for (int i = 0; i < SEND_WORKER_COUNT; i++) {
	    startNewSendWorker();
	}
    }

    public void startNewSendWorker()
    {
	String threadName = "SendWorker " + (sendWorkers.size() + 1);
	SendWorker worker = new SendWorker(threadName, this);
	sendWorkers.add(worker);

	logger.info(sendWorkers.size() + " SendWorkers running");
    }

    public synchronized void stop()
    {
	died = true;
	notifyAll();
	singleton = null;
    }

    public class Stats {

	public int sendQueueSize;
	public int busySendWorkerCount;
	public int sendWorkerCount;

	public Stats() 
	{
	    sendQueueSize = queue.size();
	    busySendWorkerCount = lockedServers.size();
	    sendWorkerCount = sendWorkers.size();
	}

    }

}
