/*
 *  XNap
 *
 *  A pure java file sharing client.
 *
 *  See AUTHORS for copyright information.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package xnap.util.launcher;

import xnap.XNap;
import xnap.cmdl.Console;
import xnap.util.Preferences;
import xnap.util.QuotedStringTokenizer;
import xnap.util.Updater;

import java.io.*;
import java.net.*;
import java.util.*;
import org.apache.log4j.Logger;
import ziga.util.WindowsFileLauncher;

public class LauncherSupport
{

    //--- Data field(s) ---

    private static Preferences prefs = Preferences.getInstance();
    private static Logger logger = Logger.getLogger(LauncherSupport.class);

    public static String[][] LAUNCHERS = {
	{ "ziga.dll", "Windows (requires ziga.dll)", "" },
	{ "kfmclient", "KDE", "kfmclient" },
	{ "open", "Mac OS X", "open" },
	{ "other", XNap.tr("Other"), ""},
    };
    
    //--- Constructor(s) ---

    private LauncherSupport() 
    {
    }

    //--- Method(s) ---

    public static boolean isEnabled()
    {
	String type = prefs.getFileLauncherType();
	if (type.equals("ziga.dll")) {
	    return Updater.isZigaDllLoaded();
	}
	else {
	    return prefs.getFileLauncherCmd().length() > 0;
	}
    }

    public static void open(File f) throws IOException
    {
	logger.debug("launching " + f.getAbsolutePath());

	String type = prefs.getFileLauncherType();
	if (type.equals("kfmclient")) {
	    String args[] = { 
		prefs.getFileLauncherCmd(), "exec", f.getAbsolutePath()
	    };
	    exec(args);
	}
	else if (type.equals("ziga.dll")) {
	    try {
		WindowsFileLauncher.open(f.getAbsolutePath());
	    }
	    catch (UnsatisfiedLinkError e) {
		logger.error("ziga.dll", e);
		throw new IOException("ziga.dll not loaded");
	    }
	}
	else if (type.equals("open")) {
	    String args[] = {
		prefs.getFileLauncherCmd(), f.getAbsolutePath()
	    };
	    exec(args);
	}
	else {
	    exec(prefs.getFileLauncherCmd(), new File[] { f });
	}
    }

    private static Process exec(String command) throws IOException
    {
	return exec(command, null);
    }

    public static Process exec(String command, File[] files) 
	throws IOException
    {
	boolean inserted = false;
	QuotedStringTokenizer t = new QuotedStringTokenizer(command);
	ArrayList args = new ArrayList(t.countTokens() + files.length);
	while (t.hasMoreTokens()) {
	    String s = t.nextToken();
	    if (s.equals("{}")) {
		insertFiles(args, files);
		inserted = true;
	    }
	    else {
		args.add(s);
	    }
	}
	
	if (!inserted) {
	    insertFiles(args, files);
	}

	String[] array = new String[args.size()];
	System.arraycopy(args.toArray(), 0, array, 0, array.length);
	return exec(array);
    }

    private static void insertFiles(ArrayList args, File[] files)
    {
	if (files != null) {
	    for (int i = 0; i < files.length; i++) {
		if (files[i] != null || files[i].length() != 0) {
		    args.add(files[i].getAbsolutePath());
		}
	    }
	}
    }

    private static Process exec(String[] args) throws IOException
    {
	Process opener = Runtime.getRuntime().exec(args);
	if (Preferences.getInstance().getCaptureLauncherOutput()) {
	    Console.getInstance().add(opener);
	}

	return opener;
    }

}


