/*
 * options.c
 *
 * Copyright (C) 1997,98 Rasca, Berlin
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *
 * This file contains the Xt GUI's options dialog
 *
 */

#include "../config.h"	/* autoconf output */

#include <stdio.h>
#include <stdlib.h>
#include <limits.h>
#include "job.h"
#include "xutil.h"
#include "app_data.h"

#define OPS_FILE	".xvidcap.scf"

extern tCodecs;
extern const char * const tCodecNames[];

/*
 * save the values
 */
Boolean XVC_WriteOptionsFile (Job *job)
{
	char *home;
	char file[PATH_MAX+1];
	FILE *fp;

	/* save it to $HOME/.xvidcap.scf */
	home = getenv ("HOME");
	sprintf (file, "%s/%s", home, OPS_FILE);
	fp = fopen (file, "w");
	if (!fp) {
		perror (file);
		return FALSE;
	}
	fprintf (fp, "#SCFF\n# xvidcap configuration file\n\n");
	fprintf (fp, "# frames per second\nfps: %f\n\n", job_fps());
	fprintf (fp, "# file pattern\nfile: %s\n\n", job_file());
	fprintf (fp, "# max frames \nmax_frames: %d\n\n", job->max_frames);
	fprintf (fp, "# max time \nmax_time: %f\n\n", job->max_time);
	fprintf (fp, "# quality (JPEG)\nquality: %d\n\n", job_quality());
	fprintf (fp, "# compression (PNG, GZIP)\ncompression: %d\n\n", job_compression());
	fprintf (fp, "# use shared memory\nshm: %d\n\n", job_shm());
	fprintf (fp, "# video codec used by ffmpeg\ncodec: %s\n\n", tCodecNames[(job->targetCodec)]);
	fprintf (fp, "# what kind of mouse pointer should be recorded? 0 = none, 1 = white, 2 = black\n");
	fprintf (fp, "mouse_wanted: %i\n\n", job->mouseWanted);
#ifdef HAVE_FFMPEG_AUDIO        
        fprintf (fp, "# toggle audio capture (0/1)\naudio: %i\n\n", ((job->flags & FLG_AUDIO_WANTED) ? 1 : 0 ) );
        fprintf (fp, "# device to grab audio from\naudio_in: %s\n\n", ((strcmp(job->snd_device, "pipe:")==0)?"-":job->snd_device));
        fprintf (fp, "# sample rate for audio capture\naudio_rate: %i\n\n", job->snd_rate );
        fprintf (fp, "# bit rate for audio capture\naudio_bits: %i\n\n", job->snd_smplsize);
        fprintf (fp, "# number of channels to use in audio capture\n");
        fprintf (fp, "audio_channels: %i\n\n", job->snd_channels);
#endif // HAVE_FFMPEG_AUDIO        
        fprintf (fp, "# toggle autocontinue (0/1)\ncontinue: %i\n\n", ((job->flags & FLG_AUTO_CONTINUE) ? 1 : 0 ) );

	fclose (fp);
        
        return TRUE;
}


/*
 * read options file
 */
Boolean XVC_ReadOptionsFile (AppData *app)
{
#include "codecs.h"

    char line[260];
    char *home;
    char file[PATH_MAX+1];
    FILE *fp;

    home = getenv ("HOME");
    sprintf (file, "%s/%s", home, OPS_FILE);
    fp = fopen (file, "r");
    if (fp) {
	    while ( fgets( line, 255, fp) ) {
		    char *token, *value;
		    char *n_hash;
		    char *c_hash = strchr(line, '#');

		    /* ignore comments */
		    if (c_hash) 
		    	sprintf(c_hash, "\0");
		    /* get rid of newlines */
		    n_hash = strchr(line, '\n');
		    if (n_hash) 
		    	sprintf(n_hash, "\0");
		    
		    /* if smth. is left, parse line */
		    if (strlen(line) > 0) {
		    	token = strtok(line, " :=\"");
			/* this has found the first token */
			value = strtok(NULL, " :=\"");

			if (strcasecmp(token,"fps")==0) {
				app->fps = atof(value);
			} else if (strcasecmp(token,"file")==0) {
				app->file = strdup(value);
			} else if (strcasecmp(token,"max_frames")==0) {
				app->frames = atoi(value);
				if (atoi(value) > 0) app->time = 0;
			} else if (strcasecmp(token,"max_time")==0) {
				app->time = atof(value);
				if (atof(value) > 0) app->frames = 0;
			} else if (strcasecmp(token,"quality")==0) {
				app->quality = atoi(value);
				if ((app->quality < 0) || (app->quality > 100))
				     app->quality = 75; /* reset to default */
			} else if (strcasecmp(token,"compression")==0) {
				app->compress = atoi(value);
			} else if (strcasecmp(token,"shm")==0) {
				if (atoi(value) == FLG_USE_SHM ) app->source = "shm";
			} else if (strcasecmp(token,"codec")==0) {
				int n, a = -1;

				for (n = 0; n < NUMCODECS; n++ ) {
					if (strcasecmp(value,tCodecNames[n])==0) a = n;
				}
				if (a < CODEC_MPEG1) {
					a = CODEC_MPEG1;
					fprintf(stderr, "reading unsupported target codec from options file\nresetting to MPEG1\n");
				}
				app->targetCodec = a;
			} else if (strcasecmp(token,"mouse_wanted")==0) {
				app->mouseWanted = atoi(value);
				if ((app->mouseWanted > 2) || (app->mouseWanted < 0)) { 
                                    fprintf(stderr, "Invalid parameter 'mouse_wanted: %s' in preferences file\n", value);
                                    fprintf(stderr, "Disabling mouse capture\n");
                                    app->mouseWanted = 0;
                                }
			} 
#ifdef HAVE_FFMPEG_AUDIO                        
                        else if (strcasecmp(token,"audio")==0) {
                                if (strcasecmp(value,"0")==0) {
                                    app->flags &= ~FLG_AUDIO_WANTED;
                                } else if (strcasecmp(value,"1")==0) {
                                    app->flags |= FLG_AUDIO_WANTED;
                                } else {
                                    fprintf(stderr, "Invalid parameter 'audio: %s' in preferences file\n", value);
                                    fprintf(stderr, "Disabling audio capture unless overridden by command line\n");
                                    app->flags &= ~FLG_AUDIO_WANTED;
                                }
                        } else if (strcasecmp(token,"audio_in")==0) {
                            app->snddev = strdup(value);
                        } else if (strcasecmp(token,"audio_rate")==0) {
                            if (value) app->sndrate = atoi(value);
                        } else if (strcasecmp(token,"audio_bits")==0) {
                            if (value) app->sndsize = atoi(value);
                        } else if (strcasecmp(token,"audio_channels")==0) {
                            if (value) app->sndchannels = atoi(value);
                        } 
#endif // HAVE_FFMPEG_AUDIO                       
                        else if (strcasecmp(token,"continue")==0) {
                            if (value) app->flags |= FLG_AUTO_CONTINUE;
                        } 
		    }
	    }
    } else { // !fp
        return FALSE;
    }
    return TRUE;
}
