/*
 * xtoffmpeg.c
 *
 * Copyright (C) 2003 Karl H. Beckers, Frankfurt
 * EMail: khb@jarre-de-the.net
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
#include "../config.h" // autoconf defines ...

#include <stdio.h>
#include <stdlib.h>
#include <limits.h>
#include <sys/types.h>
//#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>

#include <X11/X.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>

// xvidcap specific
#include "app_data.h"
#include "job.h"
#include "util.h"
#include "colors.h"
#include "frame.h"
#include "xtopnm.h"

// ffmpeg stuff
#include <ffmpeg/avcodec.h>
#include <ffmpeg/avformat.h>
#define PIX_FMT_ARGB32 PIX_FMT_RGBA32 // this is just my personal convenience

#ifdef HAVE_FFMPEG_AUDIO
#define MAX_AUDIO_PACKET_SIZE (128 * 1024)

typedef struct AVOutputStream {
    int file_index;          /* file index */
    int index;               /* stream index in the output file */
    int source_index;        /* AVInputStream index */
    AVStream *st;            /* stream in the output file */
    int encoding_needed;     /* true if encoding needed for this stream */
    int frame_number;
    /* input pts and corresponding output pts
     for A/V sync */
    double sync_ipts;
    double sync_ipts_offset;
    int64_t sync_opts;
    /* video only */
    int video_resample;      /* video_resample and video_crop are mutually exclusive */
    AVPicture pict_tmp;      /* temporary image for resampling */
    ImgReSampleContext *img_resample_ctx; /* for image resampling */
    
    int video_crop;          /* video_resample and video_crop are mutually exclusive */
    int topBand;             /* cropping area sizes */
    int leftBand;
    
    /* audio only */
    int audio_resample;
    ReSampleContext *resample; /* for audio resampling */
    FifoBuffer fifo;     /* for compression: one audio fifo per codec */
    FILE *logfile;
} AVOutputStream;

typedef struct AVInputStream {
    int file_index;
    int index;
    AVStream *st;
    int discard;             /* true if stream data should be discarded */
    int decoding_needed;     /* true if the packets must be decoded in 'raw_
     ' */
    int64_t sample_index;      /* current sample */
    int frame_decoded;       /* true if a video or audio frame has been decoded */
    
    int64_t       start;     /* time when read started */
    unsigned long frame;     /* current frame */
    AVFrac        next_pts;  /* synthetic pts for cases where pkt.pts == 0 */
    int64_t       pts;       /* current pts */
} AVInputStream;
#endif // HAVE_FFMPEG_AUDIO

/*
 * globals
 *
 * first libavcodec related stuff
 */
static AVCodec *codec;                 /* params for the codecs video */
static AVCodecContext *c=NULL;         /* codec context video */
static AVPicture inpic, *p_inpic; 	/* a AVPicture as wrapper around the original image data */
static AVPicture outpic, *p_outpic;	/* and one for the image converted to yuv420p */
static AVFrame *frame;			/* the encoder takes a frame for its input, so convert outpic to AVFrame */
static uint8_t *pic_out, *outbuf;	/* output buffer for encoded frame --- post 0.4.6 speak */
static AVFormatContext *output_file;	/* output file via avformat */
static AVOutputFormat *file_oformat;	/* ... plus related data */
static AVStream *out_st=NULL;		/* ... */

static int image_size, input_pixfmt;	/* size of yuv image, pix_fmt of original image */
static int out_size;			/* size of the encoded frame to write to file */
static double audio_pts, video_pts;

// other globals
static Boolean XVC_DEBUG;		/* want debug output yes|no */

static ColorInfo c_info;
static uint8_t *scratchbuf8bit;



#ifdef HAVE_FFMPEG_AUDIO

#include <pthread.h>
#include <signal.h>

// FIXME: check if this all needs to be global
static AVCodec *au_codec;
static AVCodecContext *au_c;
static AVFormatContext *ic;
static char *audio_grab_format = "audio_device";
static Boolean grab_audio = TRUE;
static AVInputFormat *grab_iformat;
static AVOutputStream *au_out_st;
static AVInputStream *au_in_st;

static pthread_attr_t tattr;
static pthread_mutex_t mp = PTHREAD_MUTEX_INITIALIZER;
static pthread_t tid = 0;
static int tret;


// functions ...

static void do_audio_out (AVFormatContext *s,
AVOutputStream *ost,
AVInputStream *ist,
unsigned char *buf, int size) {
    uint8_t *buftmp;
    static uint8_t *audio_buf = NULL;
    static uint8_t *audio_out = NULL;
    const int audio_out_size= 4*MAX_AUDIO_PACKET_SIZE;
    int size_out, frame_bytes, ret;
    AVCodecContext *enc;
    
    /* SC: dynamic allocation of buffers */
    if (!audio_buf)
        audio_buf = av_malloc (2*MAX_AUDIO_PACKET_SIZE);
    if (!audio_out)
        audio_out = av_malloc (audio_out_size);
    if (!audio_buf || !audio_out)
        return;               /* Should signal an error ! */
    
    enc = &ost->st->codec;
    
    // resampling is only used for pipe input here
    if (ost->audio_resample) {
        buftmp = audio_buf;
        size_out = audio_resample (ost->resample,
        (short *)buftmp, (short *)buf,
        size / (ist->st->codec.channels * 2));
        size_out = size_out * enc->channels * 2;
    } else {
        buftmp = buf;
        size_out = size;
    }
    
    /* now encode as many frames as possible */
    if (enc->frame_size > 1) {
        /* output resampled raw samples */
        fifo_write (&ost->fifo, buftmp, size_out,
        &ost->fifo.wptr);
        
        frame_bytes = enc->frame_size * 2 * enc->channels;
        
        while (fifo_read (&ost->fifo, audio_buf, frame_bytes,
        &ost->fifo.rptr) == 0) {
            ret = avcodec_encode_audio (enc, audio_out, audio_out_size,
            (short *)audio_buf);

//            audio_pts = (double)au_out_st->st->pts.val * output_file->pts_num / output_file->pts_den;
//            video_pts = (double)out_st->pts.val * output_file->pts_num / output_file->pts_den;

//            if ( audio_pts < video_pts && pthread_mutex_trylock (&mp) == 0 ) {
            if (  pthread_mutex_trylock (&mp) == 0 ) {
                /* write the compressed frame in the media file */
                if (av_write_frame (s, ost->st->index, audio_out, ret) != 0) {
                    fprintf (stderr, "Error while writing audio frame\n");
                    exit (1);
                }
                if ( pthread_mutex_unlock (&mp) > 0 ) {
                    fprintf (stderr, "Couldn't unlock mutex lock for writing audio frame\n");
                }
            } /* else {
             fprintf (stderr, "Can't lock mutex for writing audio frame\n");
             } */
        }
    } /* else {
     // output a pcm frame ... we shouldn't need this,
     // output in avi is always either mp2 or ac3
     // XXX: change encoding codec API to avoid this ?
     switch(enc->codec->id) {
     case CODEC_ID_PCM_S16LE:
     case CODEC_ID_PCM_S16BE:
     case CODEC_ID_PCM_U16LE:
     case CODEC_ID_PCM_U16BE:
     break;
     default:
     size_out = size_out >> 1;
     break;
     }
     ret = avcodec_encode_audio (enc, audio_out, size_out,
     (short *)buftmp);
      
     if ( pthread_mutex_trylock (&mp) == 0 ) {
     // write the compressed frame in the media file
     if (av_write_frame (s, ost->st->index, audio_out, ret) != 0) {
     fprintf (stderr, "Error while writing audio frame\n");
     exit (1);
     }
     if ( pthread_mutex_unlock (&mp) > 0 ) {
     fprintf (stderr, "Couldn't unlock mutex lock for writing audio frame\n");
     }
     }
     } */
}


void cleanup_thread_when_stopped () {
    int retval = 0;
    
    av_close_input_file (ic);
    pthread_exit (&retval);
}



void capture_audio_thread (Job *job) {
    unsigned long start, stop, start_s, stop_s;
    struct timeval thr_curr_time;
    long sleep;
    int retval, len, data_size;
    uint8_t *ptr, *data_buf;
    short samples[AVCODEC_MAX_AUDIO_FRAME_SIZE / 2];
    AVPacket pkt;
    
    
    signal (SIGUSR1, cleanup_thread_when_stopped);
    
    while (TRUE) {
        //get start time
        gettimeofday (&thr_curr_time, NULL);
        start_s = thr_curr_time.tv_sec;
        start = thr_curr_time.tv_usec;
        
        //FIXME: this needs to get more sophisticated for pausing efficiently ...
        if (job->state == VC_REC ) {

            audio_pts = (double)au_out_st->st->pts.val * output_file->pts_num / output_file->pts_den;
            video_pts = (double)out_st->pts.val * output_file->pts_num / output_file->pts_den;

            if ( audio_pts < video_pts ) {


            /* read a packet from it and output it in the fifo */
            if (av_read_packet (ic, &pkt) < 0) {
                fprintf (stderr, "error reading audio packet\n");
            } else {
                //                printf ("input stream #%d, size=%d:\n", pkt.stream_index, pkt.size);
                //                av_hex_dump(pkt.data, pkt.size);
            }
            
            len = pkt.size;
            ptr = pkt.data;
            while (len > 0) {
                /* decode the packet if needed */
                data_buf = NULL; /* fail safe */
                data_size = 0;
                
                if (au_in_st->decoding_needed) {
                    //FIXME: It looks like I should be doing more here about pts stuff
                    //FIXME: I just don't understand that stuff ;S ...
                    //
                    /* NOTE1: we only take into account the PTS if a new
                     frame has begun (MPEG semantics) */
                    /* NOTE2: even if the fraction is not initialized,
                     av_frac_set can be used to set the integer part */
                    if (au_in_st->frame_decoded) {
                        /* If pts is unavailable -- we have to use synthetic one */
                        if( pkt.pts != AV_NOPTS_VALUE ) {
                            au_in_st->pts = au_in_st->next_pts.val = pkt.pts;
                        }
                        else {
                            au_in_st->pts = au_in_st->next_pts.val;
                        }
                        au_in_st->frame_decoded = 0;
                    }
                    
                    /* XXX: could avoid copy if PCM 16 bits with same
                     endianness as CPU */
                    retval = avcodec_decode_audio (&au_in_st->st->codec, samples, &data_size,
                    ptr, len);
                    if (retval < 0) {
                        fprintf (stderr, "couldn't decode captured audio packet\n");
                        break;
                    }
                    /* Some bug in mpeg audio decoder gives */
                    /* data_size < 0, it seems they are overflows */
                    if (data_size <= 0) {
                        /* no audio frame */
                        // fprintf (stderr, "no audio frame\n");
                        ptr += retval;
                        len -= retval;
                        continue;
                    }
                    data_buf = (uint8_t *)samples;
                    //FIXME: this seems to be important for pts audio/video sync stuff
                    //FIXME: atm this segfaults, prolly because I haven't set up pts stuff
                    //FIXME: correctly.
                    //                av_frac_add(&au_in_st->next_pts,
                    //                ic->pts_den * data_size / (2 * au_in_st->st->codec.channels));
                } else {
                    data_buf = ptr;
                    data_size = len;
                    retval = len;
                }
                ptr += retval;
                len -= retval;
                au_in_st->frame_decoded = 1;
                
                do_audio_out (output_file, au_out_st, au_in_st, data_buf, data_size);

            }
            // discard packet
            av_free_packet (&pkt);

	    } // end if pts ...
        } // end if VC_REC
        // get end time
        gettimeofday (&thr_curr_time, NULL);
        stop_s = thr_curr_time.tv_sec;
        stop = thr_curr_time.tv_usec;
        stop += ( ( stop_s - start_s ) * 1000000 );
        sleep = ( 1000000 / job->snd_rate ) - ( stop - start );
        //FIXME: perhaps this whole stuff is irrelevant. Haven't really seen a situation
        //FIXME: where the encoding was faster than the time needed for a decent frame-rate
        //FIXME: need to look into more details about audio/video sync in libavcodec/-format
        if ( sleep < 0 ) sleep = 0;
        
        //printf ("starts %li start %li - stops %li stop %li\n", start_s, start, stop_s, stop);
        //printf ("sleep %i microsecs .. frame took %i microsecs\n", sleep, (stop - start));
        usleep (sleep);
    }
    retval = 1;
    pthread_exit (&retval);
}

#endif // HAVE_FFMPEG_AUDIO


#ifdef DEBUG
/*
 * dump info about XImage - for debugging purposes
 */
void x2ffmpeg_dump_ximage_info (XImage *img, FILE *fp) {
    fprintf (fp, " width %d\n", img->width);
    fprintf (fp, " height %d\n", img->height);
    fprintf (fp, " xoffset %d\n", img->xoffset);
    fprintf (fp, " format %d\n", img->format);
    fprintf (fp, " data addr 0x%X\n", (int)img->data);
    fprintf (fp, " first four bytes of data 0x%X 0x%X 0x%X 0x%X\n",
    (unsigned char)(img->data[0]), (unsigned char)(img->data[1]), (unsigned char)img->data[2], (unsigned
    char)img->data[3]);
    fprintf (fp, " byte_order %s\n", img->byte_order ? "MSBFirst":"LSBFirst");
    fprintf (fp, " bitmap_unit %d\n", img->bitmap_unit);
    fprintf (fp, " bitmap_bit_order %s\n", img->bitmap_bit_order ? "MSBFirst" : "LSBFirst");
    fprintf (fp, " bitmap_pad %d\n", img->bitmap_pad);
    fprintf (fp, " depth %d\n", img->depth);
    fprintf (fp, " bytes_per_line %d\n", img->bytes_per_line);
    fprintf (fp, " bits_per_pixel %d\n", img->bits_per_pixel);
    fprintf (fp, " red_mask 0x%.8X\n", img->red_mask);
    fprintf (fp, " green_mask 0x%.8X\n", img->green_mask);
    fprintf (fp, " blue_mask 0x%.8X\n", img->blue_mask);
}


/*
 * dump 32bit image to pnm
 * this is just used for debugging purposes
 */
void dump32bit (XImage *input) {
    int row, col;
    static char head[256];
    
    static FILE *fp2 = NULL;
    uint8_t *ptr2, *output;
    long size;
    
    register unsigned int
    rm = input->red_mask,
    gm = input->green_mask,
    bm = input->blue_mask,
    rs = c_info.red_shift,
    gs = c_info.green_shift,
    bs = c_info.blue_shift,
    *p32 = (unsigned int *) input->data;
    
    if (XVC_DEBUG) fprintf (stderr, "xtoffmpeg.dump32bit(): entered\n");
    
    sprintf (head, "P6\n%d %d\n%d\n", input->width, input->height, 255);
    size = ((input->bytes_per_line * input->height) / 4) * 3;
    output = malloc (size);
    ptr2 = output;
    
    for (row = 0; row < input->height; row++) {
        for (col = 0; col < input->width; col++) {
            *output++ = ((*p32 & rm) >> rs);
            *output++ = ((*p32 & gm) >> gs);
            *output++ = ((*p32 & bm) >> bs);
            p32++; /* ignore alpha values */
        }
        /* eat paded bytes, for better speed we use shifting,
         * (bytes_per_line - bits_per_pixel / 8 * width ) / 4
         */
        p32 += (input->bytes_per_line
        - (input->bits_per_pixel >> 3 )
        * input->width ) >> 2;
    }
    
    fp2 = fopen ("/tmp/pic.rgb.pnm", "w");
    fwrite (head, strlen (head), 1, fp2);
    /*	x2ffmpeg_dump_ximage_info (input, fp2); */
    fwrite (ptr2, size, 1, fp2);
    fclose (fp2);
    free (ptr2);
    if (XVC_DEBUG) fprintf (stderr, "xtoffmpeg.dump32bit(): leaving\n");
}


/*
 * dump 8 bit image to ppm
 * this is just used for debugging purposes
 */
void dump8bit (XImage *image, u_int32_t *ct) {
    static char head[256];
    static unsigned int image_size;
    register unsigned char *line_ptr, *col_ptr;
    unsigned char *pnm_image = NULL;
    int row, col;
    
    static FILE *fp2 = NULL;
    
    if (XVC_DEBUG) fprintf (stderr, "xtoffmpeg.dump8bit(): entered\n");
    
    sprintf (head, "P6\n%d %d\n%d\n", image->width, image->height, 255);
    image_size = image->width * 3 * image->height; /* RGB */
    pnm_image = (unsigned char *) malloc (image_size);
    
    fp2 = fopen ("/tmp/pic.rgb.pnm", "w");
    fwrite (head, strlen (head), 1, fp2);
    
    line_ptr = pnm_image;
    for (row = 0; row < image->height; row++) {
        col_ptr = image->data + (row * image->bytes_per_line);
        for (col = 0; col < image->width; col++) {
            *line_ptr++ = ((ct[*col_ptr] & 0x00FF0000) >> 16);
            *line_ptr++ = ((ct[*col_ptr] & 0x0000FF00) >> 8);
            *line_ptr++ = (ct[*col_ptr] & 0x000000FF);
            col_ptr++;
        }
    }
    fwrite (pnm_image, image_size, 1, fp2);
    
    /*	x2ffmpeg_dump_ximage_info (input, fp2); */
    fclose (fp2);
    free (pnm_image);
    if (XVC_DEBUG) fprintf (stderr, "xtoffmpeg.dump8bit(): leaving\n");
}
#endif /* DEBUG */


/*
 * convert bgra32 to rgba32
 * needed on Solaris/SPARC because the ffmpeg version used doesn't know PIX_FMT_ABGR32
 */
void myABGR32toARGB32 (XImage *image) {
    char *pdata, *counter;
    
    #ifdef DEBUG
    if (XVC_DEBUG) fprintf (stderr, "xtoffmpeg.myBGRA32toRGBA32(): entered\n");
    #endif /* DEBUG */
    
    pdata = image->data;
    
    for (counter = pdata; counter < (pdata + (image->width * image->height * 4)); counter += 4 ) {
        char swap, swap1, swap2, swap3, swap4;
        
        if (image->byte_order) { //MSBFirst has order argb -> abgr = rgba32
            swap = *(counter + 1);
            *(counter + 1) = *(counter + 3);
            *(counter + 3) = swap;
        } else { //LSBFirst has order bgra -> rgba
            swap = *counter;
            *counter = *(counter + 2);
            *(counter + 2) = swap;
        }
    }
    
    #ifdef DEBUG
    if (XVC_DEBUG) fprintf (stderr, "xtoffmpeg.myBGRA32toRGBA32(): leaving\n");
    #endif  /* DEBUG */
}


/*
 * prepare the color table for pseudo color input to libavcodec's imgconvert
 * I'm downsampling the 16 bit color entries to 8 bit as it expects
 * 32 bit (=4 * 8)
 * from looking at imgconvert_template.c I'd say libavcodec expects argb
 * logically, not byte-wise
 */
void *
FFMPEGcolorTable (XColor *colors, int ncolors) {
    u_int32_t *color_table, *pixel;
    int i, n;
    
    color_table = malloc (256 * 4);
    if (!color_table)
        return (NULL);
    
    pixel = color_table;
    for (i = 0 ; i < ncolors; i++) {
        u_int32_t color_table_entry, swap;
        
        color_table_entry = 0;
        // alpha alway zero
        
        swap = colors[i].red;
        swap &= 0x0000FF00; // color is 16 bits, delete ls 8 bits
        swap <<= 8; // then shift ms 8 bits into position
        color_table_entry = (color_table_entry | swap);
        
        swap = colors[i].green;
        swap &= 0x0000FF00; // color is 16 bits, ms 8 bits already in position, delete ls 8 bits
        color_table_entry = (color_table_entry | swap);
        
        swap = colors[i].blue;
        swap >>= 8;
        color_table_entry = (color_table_entry | swap);
        
        *pixel = color_table_entry;
        pixel++;
    }
    #ifdef DEBUG
    printf ("inside: color_table pointer: %p\n", color_table);
    printf ("inside: color_table third entry: 0x%.8X\n", *(color_table + 2)); // first and second black & white
    #endif /* DEBUG */
    return (color_table);
}


/*
 * write ximage as mpeg file to 'fp'
 */
void
XImageToFFMPEG (FILE *fp, XImage *image, Job *job) {
    #include "codecs.h"
    
    AVFormatParameters fParams, *p_fParams = &fParams;  // video stream params
    AVFormatParameters params, *ap = &params;           // audio stream params
    AVImageFormat *image_format;
    int ret, err, i;
    
    #ifdef DEBUG
    if (job->flags & FLG_RUN_VERBOSE) {
        XVC_DEBUG = TRUE;
        fprintf (stderr, "xtoffmpeg.XImageToFFMPEG(): entered\n");
    }
    #endif  /* DEBUG */
    
    /* encoder needs to be prepared only once ..
     */
    if (job->state & VC_START /* it's the first call */) {
        int transCodec; //desired codec in ffmpeg speak
        
        #ifdef DEBUG
        if (XVC_DEBUG) fprintf (stderr, "xtoffmpeg.XImageToFFMPEG(): doing x2ffmpeg init\n");
        #endif  /* DEBUG */
        
        /* find the video encoder */
        #ifdef DEBUG
        if (XVC_DEBUG) fprintf (stderr, "xtoffmpeg.XImageToFFMPEG(): targetCodec: %i\n", job->targetCodec);
        #endif /* DEBUG */
        
        switch (job->targetCodec) {
            case CODEC_FLV:
                transCodec = CODEC_ID_FLV1;
                flvenc_init();
                file_oformat = guess_format ("flv", "bla.flv", "video/x-flashvideo");
                break;
            case CODEC_MSDIV2:
                transCodec = CODEC_ID_MSMPEG4V2;
                avienc_init ();
                file_oformat = guess_stream_format ("avi", "bla.avi", "video/x-msvideo");
                break;
            case CODEC_MSDIV3:
                transCodec = CODEC_ID_MSMPEG4V3;
                asf_init ();
                file_oformat = guess_format ("asf", "bla.asf", "video/x-ms-asf");
                break;
            case CODEC_MJPEG:
                transCodec = CODEC_ID_MJPEG;
                avienc_init ();
                file_oformat = guess_stream_format ("avi", "bla.avi", "video/x-msvideo");
                break;
            case CODEC_MPEG4:
                transCodec = CODEC_ID_MPEG4;
                avienc_init ();
                file_oformat = guess_stream_format ("avi", "bla.avi", "video/x-msvideo");
                break;
            case CODEC_MPEG1:
            default:
                if (job->targetCodec != CODEC_MPEG1)
                    printf ("Unknown Video Codec ... falling back to MPEG1VIDEO!\n");
                transCodec = CODEC_ID_MPEG1VIDEO;
                avienc_init ();
                file_oformat = guess_stream_format ("avi", "bla.avi", "video/x-msvideo");
        }

        /* register various libavcodec/-format functions ...
           needs to go here because putting this before asf_init() makes audio capture
           for ASFs fail. */
        av_register_all ();
        
        /*
         * prepare format context
         */
        output_file = av_mallocz (sizeof(AVFormatContext));
        output_file->oformat = file_oformat;
        output_file->priv_data = av_mallocz (output_file->oformat->priv_data_size);
        if (!output_file->priv_data) {
            fprintf (stderr, "xtoffmpeg.XImageToFFMPEG(): Error allocating private data for format context ... aborting\n");
            exit (1);
        }
        /* prepare output file for format context */
        {
            char first;
            char tmp_buf[PATH_MAX+1];
            
            first = job->file[0];
            sprintf(tmp_buf, job->file, job->movie_no);
            /* if the filename's first char is a / we have an absolute path
             * and we want one for the file URL. If we don't have one, we
             * construct one */
            if (first != '/') {
                sprintf (output_file->filename, "file://%s/%s", getenv ("PWD"), tmp_buf);
            } else {
                sprintf (output_file->filename, "file://%s", tmp_buf);
            }
        }
        register_protocol (&file_protocol);
        
        /* add the video stream
         * and initialize the codecs */
        /* prepare stream */
        out_st = av_new_stream (output_file, 0);
        if (!out_st) {
            fprintf (stderr, "xtoffmpeg.XImageToFFMPEG(): Could not alloc stream ... aborting\n");
            exit (1);
        }
        
        c = &out_st->codec;
        c->codec_id = transCodec;
        c->codec_type = CODEC_TYPE_VIDEO;
        
        /* resolution must be a multiple of two ... this is taken care of elsewhere */
        c->width = image->width;
        c->height = image->height;
        c->bit_rate = ( c->width * c->height * (((((c->height + c->width)/ 100)- 5)>> 1)+ 10) * job->quality ) / 100;
        if (c->bit_rate < 300000) c->bit_rate = 300000;
        #ifdef DEBUG
        if (XVC_DEBUG) fprintf (stderr, "xtoffmpeg.XImageToFFMPEG(): bitrate = %i\n", c->bit_rate);
        #endif  /* DEBUG */
        
        /* frames per second */
        c->frame_rate = job->fps;
        c->frame_rate_base = 1;
        c->gop_size = 250; /* emit one intra frame every 10 frames */
        //        c->me_method = ME_EPZS;
        //	    c->debug = 0x00000FFF;
        
        /*
         * set params
         */
        memset (p_fParams, 0, sizeof(*p_fParams));
        p_fParams->image_format = image_format;
        p_fParams->frame_rate = out_st->codec.frame_rate;
        p_fParams->frame_rate_base = out_st->codec.frame_rate_base;
        p_fParams->width = out_st->codec.width;
        p_fParams->height = out_st->codec.height;
        if (av_set_parameters (output_file, p_fParams) < 0) {
            //        if (av_set_parameters(output_file, NULL) < 0) {
            fprintf (stderr, "xtoffmpeg.XImageToFFMPEG(): Invalid encoding parameters ... aborting\n");
            exit (1);
        }
        
        /*
         * open codec
         */
        c = &out_st->codec;
        
        /* find the video encoder */
        codec = avcodec_find_encoder (c->codec_id);
        if (!codec) {
            fprintf (stderr, "codec not found\n");
            exit (1);
        }
        
        /* open the codec */
        if (avcodec_open (c, codec) < 0) {
            fprintf (stderr, "could not open codec\n");
            exit (1);
        }
        
#ifdef HAVE_FFMPEG_AUDIO
        if (job->flags & FLG_AUDIO_WANTED) {
            if (!strcmp (job->snd_device, "-")) {
                job->snd_device = "pipe:";
                grab_audio = FALSE;
            } else {
                grab_audio = TRUE;
            }
            
            // prepare input stream
            // ----------------------------------------------------------------------------
            memset (ap, 0, sizeof(*ap));
            ap->device = job->snd_device;
            
            if (grab_audio) {
                ap->sample_rate = job->snd_rate;
                ap->channels = job->snd_channels;
                
                grab_iformat = av_find_input_format ("audio_device");
                //            printf ("pointer to grab_iformat %p\n", grab_iformat);
                //            if (grab_iformat) printf ("input format %s , %s\n", grab_iformat->name, grab_iformat->long_name );
                if (av_open_input_file (&ic, "", grab_iformat, 0, ap) < 0) {
                    fprintf (stderr, "Could not find audio grab device %s\n", job->snd_device);
                    exit (1);
                }
            } else {
                /*            ap->sample_rate = job->snd_rate;
                 ap->channels = job->snd_channels;
                 ap->frame_rate = 25;
                 ap->frame_rate_base = 1;
                 */
                /* open the input file with generic libav function */
                //            err = av_open_input_file (&ic, job->snd_device, grab_iformat, 0, ap);
                register_protocol (&pipe_protocol);
                
                grab_iformat = av_find_input_format ("mp3");
//                printf ("pointer to grab_iformat %p\n", grab_iformat);
//                if (grab_iformat) printf ("input format %s , %s\n", grab_iformat->name, grab_iformat->long_name );
                
                
                err = av_open_input_file (&ic, job->snd_device, grab_iformat, 0, ap);
                if (err < 0) {
                    fprintf (stderr, "error opening input file %s: %i\n", job->snd_device, err);
                    exit (1);
                }
            }
            au_in_st = av_mallocz (sizeof(AVInputStream));
            if (!au_in_st) {
                fprintf (stderr, "audio.start_audio.recording(): Could not alloc input stream ... aborting\n");
                exit (1);
            }
            au_in_st->st = ic->streams[0];
            
            /* If not enough info to get the stream parameters, we decode the
             first frames to get it. (used in mpeg case for example) */
            ret = av_find_stream_info (ic);
            if (ret < 0) {
                fprintf (stderr, "%s: could not find codec parameters\n", job->snd_device);
                exit (1);
            }
            
            /* dump the file content */
            if (XVC_DEBUG) dump_format (ic, 0, job->snd_device, 0);
            
            // OUTPUT
            // setup output codec ------------------------------------------------------------
            /* prepare codec */
            au_c = avcodec_alloc_context ();
            /* put sample parameters */
            au_c->codec_id = CODEC_ID_MP2;
            au_c->codec_type = CODEC_TYPE_AUDIO;
            au_c->bit_rate = job->snd_smplsize;
            au_c->sample_rate = job->snd_rate;
            au_c->channels = job->snd_channels;
            //        au_c->debug = 0x00000FFF;
            
            /* prepare output stream */
            au_out_st = av_mallocz (sizeof(AVOutputStream));
            if (!au_out_st) {
                fprintf (stderr, "audio.start_audio.recording(): Could not alloc stream ... aborting\n");
                exit (1);
            }
            au_out_st->st = av_new_stream (output_file, 1);
            if (!au_out_st->st) {
                fprintf (stderr, "Could not alloc stream\n");
                exit (1);
            }
            au_out_st->st->codec = *au_c;
            
            if (fifo_init (&au_out_st->fifo, 2 * MAX_AUDIO_PACKET_SIZE)) {
                fprintf (stderr, "Can't initialize fifo for audio recording\n");
                exit (1);
            }
            
            // This bit is important for inputs other than self-sampled.
            // The sample rates and no of channels a user asks for
            // are the ones he/she wants in the encoded mpeg. For self-
            // sampled audio, these are also the values used for sampling.
            // Hence there is no resampling necessary (case 1).
            // Once we get support for dubbing from a pipe or a different
            // file, we might have different sample rates or no of channels
            // in the input file.....
            if (au_c->channels == au_in_st->st->codec.channels &&
            au_c->sample_rate == au_in_st->st->codec.sample_rate) {
                au_out_st->audio_resample = 0;
            } else {
                if (au_c->channels != au_in_st->st->codec.channels &&
                au_in_st->st->codec.codec_id == CODEC_ID_AC3) {
                    /* Special case for 5:1 AC3 input */
                    /* and mono or stereo output      */
                    /* Request specific number of channels */
                    au_in_st->st->codec.channels = au_c->channels;
                    if (au_c->sample_rate == au_in_st->st->codec.sample_rate)
                        au_out_st->audio_resample = 0;
                    else {
                        au_out_st->audio_resample = 1;
                        au_out_st->resample = audio_resample_init (au_c->channels, au_in_st->st->codec.channels,
                        au_c->sample_rate, au_in_st->st->codec.sample_rate);
                        if(!au_out_st->resample) {
                            printf ("Can't resample.  Aborting.\n");
                            exit (1);
                            // av_abort ();
                        }
                    }
                    /* Request specific number of channels */
                    au_in_st->st->codec.channels = au_c->channels;
                } else {
                    au_out_st->audio_resample = 1;
                    au_out_st->resample = audio_resample_init (au_c->channels, au_in_st->st->codec.channels,
                    au_c->sample_rate, au_in_st->st->codec.sample_rate);
                    if(!au_out_st->resample) {
                        printf ("Can't resample.  Aborting.\n");
                        exit (1);
                        //av_abort ();
                    }
                }
            }
            au_in_st->decoding_needed = 1;
            au_out_st->encoding_needed = 1;
            
            // open encoder
            au_codec = avcodec_find_encoder (au_out_st->st->codec.codec_id);
            if (avcodec_open (&au_out_st->st->codec, au_codec) < 0) {
                fprintf (stderr, "Error while opening codec for output stream\n");
                exit (1);
            }
            
            // open decoder
            au_codec = avcodec_find_decoder (ic->streams[0]->codec.codec_id);
            if (!au_codec) {
                fprintf (stderr, "Unsupported codec (id=%d) for input stream\n");
                exit (1);
            }
            if (avcodec_open (&ic->streams[0]->codec, au_codec) < 0) {
                fprintf (stderr, "Error while opening codec for input stream\n");
                exit (1);
            }
            
            if (XVC_DEBUG) dump_format (output_file, 0, output_file->filename, 1);
            
            /* initialize a mutex lock to its default value */
            ret = pthread_mutex_init (&mp, NULL);
            
            /* create and start capture thread */
            /* initialized with default attributes */
            tret = pthread_attr_init (&tattr);
            
            /* call an appropriate functions to alter a default value */
            //tret = pthread_attr_setdetachstate(&tattr, PTHREAD_CREATE_DETACHED);
            
            /* create the thread */
            tret = pthread_create (&tid, &tattr, (void*) capture_audio_thread, job);
        } // end if flags & FLG_AUDIO_WANTED
        
#endif // HAVE_FFMPEG_AUDIO



        /* open the file */
        if (url_fopen (&output_file->pb, output_file->filename, URL_WRONLY) < 0) {
            fprintf (stderr, "xtoffmpeg.XImageToFFMPEG(): Could not open '%s' ... aborting\n", output_file->filename);
            exit (1);
        }
        
        if (av_write_header (output_file) < 0) {
            dump_format (output_file, 0, output_file->filename, 1);
            fprintf (stderr, "xtoffmpeg.XImageToFFMPEG(): Could not write header for output file (incorrect codec paramters ?) ... aborting\n");
            exit (1);
        }
        
        /* color info only needs to be retrieved once for true color X ... dunno about pseudo color */
        GetColorInfo (image, &c_info);
        #ifdef DEBUG
        if (XVC_DEBUG) {
            FILE *errout;
            
            fprintf (stderr, "xtoffmpeg.XImageToFFMPEG(): got color info\n");
            errout = fdopen (2, "w");
            x2ffmpeg_dump_ximage_info (image, errout);
            fprintf (stderr, "xtoffmpeg.XImageToFFMPEG(): alpha_mask: 0x%.8X\n", c_info.alpha_mask);
            fprintf (stderr, "xtoffmpeg.XImageToFFMPEG(): alpha_shift: %i\n", c_info.alpha_shift);
            fprintf (stderr, "xtoffmpeg.XImageToFFMPEG(): red_shift: %i\n", c_info.red_shift);
            fprintf (stderr, "xtoffmpeg.XImageToFFMPEG(): green_shift: %i\n", c_info.green_shift);
            fprintf (stderr, "xtoffmpeg.XImageToFFMPEG(): blue_shift: %i\n", c_info.blue_shift);
        }
        #endif /* DEBUG */
        
        /* determine input picture format */
        // FIXME: test for exotic formats
        // endianness is treated by avcodec
        #ifdef DEBUG
        if (XVC_DEBUG) fprintf (stderr, "xtoffmpeg.XImageToFFMPEG(): image->byte_order: %i, msb=%i, lsb=%i\n", image->byte_order, MSBFirst, LSBFirst);
        #endif  /* DEBUG */
        switch (image->bits_per_pixel) {
            case 8:
                #ifdef DEBUG
                if (XVC_DEBUG) fprintf (stderr, "xtoffmpeg.XImageToFFMPEG(): 8 bit pallete\n");
                #endif  /* DEBUG */
                input_pixfmt = PIX_FMT_PAL8;
                break;
            case 16:
                if ( image->red_mask == 0xF800 && image->green_mask == 0x07E0
                && image->blue_mask == 0x1F ) {
                    #ifdef DEBUG
                    if (XVC_DEBUG) fprintf (stderr, "xtoffmpeg.XImageToFFMPEG(): 16 bit RGB565\n");
                    #endif /* DEBUG */
                    input_pixfmt = PIX_FMT_RGB565;
                } else if ( image->red_mask == 0x7C00 && image->green_mask == 0x03E0
                && image->blue_mask == 0x1F ) {
                    #ifdef DEBUG
                    if (XVC_DEBUG) fprintf (stderr, "xtoffmpeg.XImageToFFMPEG(): 16 bit RGB555\n");
                    #endif /* DEBUG */
                    input_pixfmt = PIX_FMT_RGB555;
                    
                } else {
                    fprintf (stderr, "xtoffmpeg.XImageToFFMPEG(): rgb ordering at image depth %i not supported ... aborting\n", image->bits_per_pixel);
                    fprintf (stderr, "xtoffmpeg.XImageToFFMPEG(): color masks: r 0x%.6X g 0x%.6X b 0x%.6X\n", image->red_mask, image->green_mask, image->blue_mask);
                }
                break;
            case 24:
                if ( image->red_mask == 0xFF0000 && image->green_mask == 0xFF00
                && image->blue_mask == 0xFF ) {
                    input_pixfmt = PIX_FMT_BGR24;
                } else if ( image->red_mask == 0xFF && image->green_mask == 0xFF00
                && image->blue_mask == 0xFF0000 ) {
                    input_pixfmt = PIX_FMT_RGB24;
                } else {
                    fprintf (stderr, "xtoffmpeg.XImageToFFMPEG(): rgb ordering at image depth %i not supported ... aborting\n", image->bits_per_pixel);
                    fprintf (stderr, "xtoffmpeg.XImageToFFMPEG(): color masks: r 0x%.6X g 0x%.6X b 0x%.6X\n", image->red_mask, image->green_mask, image->blue_mask);
                    exit (1);
                }
                break;
            case 32:
                if ( c_info.alpha_mask == 0xFF000000 && image->green_mask == 0xFF00 ) {
                    // byte order is relevant here, not endianness
                    // endianness is handled by avcodec, but atm no such thing
                    // as having ABGR, instead of ARGB in a word. Since we
                    // need this for Solaris/SPARC, but need to do the conversion
                    // for every frame we do it outside of this loop, cf. below
                    // this matches both ARGB32 and ABGR32
                    input_pixfmt = PIX_FMT_ARGB32;
                } else {
                    fprintf (stderr, "xtoffmpeg.XImageToFFMPEG(): image depth %i not supported ... aborting\n", image->bits_per_pixel);
                    exit (1);
                }
                break;
            default:
                fprintf (stderr, "xtoffmpeg.XImageToFFMPEG(): image depth %i not supported ... aborting\n", image->bits_per_pixel);
                exit (1);
        }
        
        /* prepare pictures */
        p_inpic = &inpic;
        p_outpic = &outpic;
        
        image_size = avpicture_get_size (PIX_FMT_YUV420P, c->width, c->height);
        pic_out = av_malloc (image_size);
        if (!pic_out) {
            fprintf (stderr, "xtoffmpeg.XImageToFFMPEG(): Could not allocate buffer for YUV pic (pic_out)! ... aborting\n");
            exit (1);
        }
        if (input_pixfmt == PIX_FMT_PAL8) {
            scratchbuf8bit = malloc ((c->width * c->height));
            
            avpicture_fill (p_inpic, scratchbuf8bit, input_pixfmt, c->width, c->height);
            p_inpic->data[1] = job->color_table;
            #ifdef DEBUG
            if (XVC_DEBUG) fprintf (stderr, "xtoffmpeg.XImageToFFMPEG(): first 4 colors a r g b each: 0x%.8X 0x%.8X 0x%.8X 0x%.8X\n", *(u_int32_t*)(job->color_table), *((u_int32_t*)(job->color_table) + 1), *((u_int32_t*)(job->color_table) + 2), *((u_int32_t*)(job->color_table) + 3) );
            #endif /* DEBUG */
        } else {
            avpicture_fill (p_inpic, image->data, input_pixfmt, c->width, c->height);
        }
        avpicture_fill (p_outpic, pic_out, PIX_FMT_YUV420P, c->width, c->height);
        
        frame = avcodec_alloc_frame ();
        frame->type = FF_BUFFER_TYPE_SHARED;
        
        frame->data[0] = pic_out;
        frame->data[1] = frame->data[0] + ( c->width * c->height);
        frame->data[2] = frame->data[1] + ( c->width * c->height) / 4;
        frame->data[3] = NULL;
        frame->linesize[0] = c->width;
        frame->linesize[1] = c->width / 2;
        frame->linesize[2] = c->width / 2;
        frame->linesize[3] = 0;
        
        /* prepare output buffer for encoded frames */
        outbuf = malloc (image_size);
        if (!outbuf) {
            fprintf (stderr, "xtoffmpeg.XImageToFFMPEG(): Could not allocate buffer for encoded frame (outbuf)! ... aborting\n");
            exit (1);
        }
        
        #ifdef DEBUG
        if (XVC_DEBUG) fprintf (stderr, "xtoffmpeg.XImageToFFMPEG(): leaving xffmpeg init\n");
        #endif /* DEBUG */
    }
    
    /* do preprocessing for ffmpeg's imgconvert */
    if (input_pixfmt == PIX_FMT_ARGB32 && c_info.alpha_mask == 0xFF000000 &&
    image->red_mask == 0xFF && image->green_mask == 0xFF00 &&
    image->blue_mask == 0xFF0000 ) {
        myABGR32toARGB32 (image);
    } else if (input_pixfmt == PIX_FMT_PAL8) {
        /* 8bit pseudo-color images may have lines padded by excess bytes
         * these need to be removed before conversion
         * FIXME: other formats might also have this problem */
        if (image->bytes_per_line > c->width) {
            int y, x;
            uint8_t *in, *out;
            
            in = image->data;
            out = scratchbuf8bit;
            
            for (y = 0; y < c->height; y++) {
                if (! memcpy (out, in, c->width)) {
                    fprintf (stderr, "xtoffmpeg.XImageToFFMPEG(): error preprocessing 8bit pseudo color input ... aborting\n");
                    exit (1);
                }
                in += image->bytes_per_line;
                out += c->width;
            }
        } else {
            scratchbuf8bit = memcpy (scratchbuf8bit, image->data, (c->width * c->height));
        }
    }
    
    /* convert input pic to yuv420p */
    #ifdef DEBUG
    if (XVC_DEBUG && (input_pixfmt == PIX_FMT_ARGB32)) dump32bit (image);
    if (XVC_DEBUG && (input_pixfmt == PIX_FMT_PAL8)) dump8bit (image, (u_int32_t*)job->color_table);
    #endif /* DEBUG */
    if (img_convert (p_outpic, PIX_FMT_YUV420P,
    p_inpic, input_pixfmt, c->width, c->height) < 0) {
        fprintf (stderr, "xtoffmpeg.XImageToFFMPEG(): pixel format conversion not handled ... aborting\n");
        exit (1);
    }
    #ifdef DEBUG
    if (XVC_DEBUG) {
        FILE *lfp;
        
        lfp = fopen ("/tmp/pic.yuv", "w");
        fwrite (pic_out, image_size, 1, lfp);
        fclose (lfp);
    }
    #endif /* DEBUG */
    
    /* encode the image */
    out_size = avcodec_encode_video (c, outbuf, image_size, frame);
    
#ifdef HAVE_FFMPEG_AUDIO
    if (job->flags & FLG_AUDIO_WANTED) {
        if ( pthread_mutex_lock (&mp) > 0 ) {
            fprintf (stderr, "mutex lock for writing video frame failed ... aborting\n");
            exit (1);
        }
        // the following bit is for synching audio and video streams ...
        audio_pts = (double)au_out_st->st->pts.val * output_file->pts_num / output_file->pts_den;
        video_pts = (double)out_st->pts.val * output_file->pts_num / output_file->pts_den;
    }
#endif // HAVE_FFMPEG_AUDIO
    /* write frame to file */
    av_write_frame (output_file, out_st->index, outbuf, out_size);
#ifdef HAVE_FFMPEG_AUDIO
    /* release the mutex */
    if (job->flags & FLG_AUDIO_WANTED) {
        if ( pthread_mutex_unlock (&mp) > 0 ) {
            fprintf (stderr, "couldn't release the mutex for writing video frame ... aborting\n");
        }
    }
#endif // HAVE_FFMPEG_AUDIO

}

/*
 * KHB: FFMPEG cleanup
 */
FFMPEGClean (Job *job) {
    int j;
    
    #ifdef DEBUG
    if (XVC_DEBUG) 
        fprintf (stderr, "xtoffmpeg.FFMPEGClean(): entered\n");
    #endif  /* DEBUG */
    
    #ifdef HAVE_FFMPEG_AUDIO
    //
    if (job->flags & FLG_AUDIO_WANTED && tid > 0) {
        {
            int tret;
            
            tret = pthread_kill (tid, SIGUSR1);
            pthread_join (tid, NULL);
            tid = 0;
        }
    }
    #endif // HAVE_FFMPEG_AUDIO
    
    if (out_st) {
        avcodec_close (&out_st->codec);
        out_st = NULL;
    }

    if (output_file) {
        /* write trailer */
        av_write_trailer (output_file);
        /* free stream(s) ... probably always only one */
        for(j=0;j<output_file->nb_streams;j++)
            av_free (output_file->streams[j]);
    
        /* close file */
        url_fclose (&output_file->pb);
        /* free format context */
        av_free (output_file);
        output_file = NULL;
    }
    
    if (pic_out) { 
        av_free (pic_out);
        pic_out = NULL;
    }

    //free(outbuf); 	/* avcodec seems to do that job */
    outbuf = NULL;
    if (frame) {
        free (frame);
        frame = NULL;
    }
    
    // avcodec_close(c);
    /* free(c);		/* according to ffmpeg's apiexample.c */
    // don't seem to need to close anything if I'm freeing the stream
    // prior to this ...
    c = NULL;
    p_inpic = NULL;
    p_outpic = NULL;
    
    if (input_pixfmt == PIX_FMT_PAL8 && scratchbuf8bit) {
        free (scratchbuf8bit);
        scratchbuf8bit = NULL;
    }
    
    codec = NULL;
    
    av_free_static ();
    #ifdef DEBUG
    if (XVC_DEBUG) fprintf (stderr, "xtoffmpeg.FFMPEGClean(): leaving cleanup\n");
    #endif  /* DEBUG */
    
}


