/*
 * xtopng.c,
 * XImage to PNG module
 *
 * Copyright (C) 1997,98 Rasca, Berlin
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "../config.h"  /* autoconf output */

#ifdef HAVE_LIBPNG
#include <stdio.h>
#include <stdlib.h>
#include <limits.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/XWDFile.h>
#ifndef HAVE_LIBZ
#error "You need zlib for PNG support"
#endif
#include <zlib.h>
#include <png.h>
#include "job.h"
#include "xtopng.h"
#include "colors.h"

/* image size for ZPixmap
 */
#define ZImageSize(i) (i->bytes_per_line * i->height)

/* globals */
static png_structp png_ptr = NULL;
static png_infop info_ptr = NULL;
static unsigned char *line = NULL;

/*
 * build a color table for indexed PNG graphics
 */
void *
PNGcolorTable (XColor *colors, int ncolors)
{
	png_color *color_tab;
	int i;

	color_tab = (png_color *) malloc (ncolors * sizeof (png_color));
	if (!color_tab)
		return (NULL);
	for (i =0; i < ncolors; i++) {
		color_tab[i].red = colors[i].red;
		color_tab[i].green = colors[i].green;
		color_tab[i].blue = colors[i].blue;
	}
	return (color_tab);
}

/*
 * for TrueColor and DirectColor
 * write a PNG out to the named file pointer
 */
void
XImageToPNGC (FILE *fp, XImage *image, Job *job)
{
	int row, col, line_size, pad;
	unsigned char *col_ptr, *row_ptr, *p8;
	unsigned short *p16;
	static png_color_8 sig_bit;
	static ColorInfo c_info;

	if ( job->state & VC_START ) {
		/* it's the first call, prepare some stuff
		 */
#ifdef DEBUG2
		dump_ximage_info (image);
#endif
		GetColorInfo (image, &c_info);
		if (image->bits_per_pixel != 24) {
			/* in 24bpp mode we don't need the buffer
			 */
			line_size = image->width * 3; /* RGB */
			line = (unsigned char *) malloc (line_size);

			/* on my 16bpp server these values are 5,6,5
			 */
			sig_bit.red   = c_info.red_bit_depth;
			sig_bit.green = c_info.green_bit_depth;
			sig_bit.blue  = c_info.blue_bit_depth;
			sig_bit.alpha = c_info.alpha_bit_depth;
		}
	}

	png_ptr = png_create_write_struct (PNG_LIBPNG_VER_STRING,
				(void *)NULL, NULL, NULL);
	if (!png_ptr)
		return;
	info_ptr = png_create_info_struct (png_ptr);
	if (!info_ptr) {
		png_destroy_write_struct (&png_ptr, (png_infopp)NULL);
		return;
	}
	if (setjmp (png_ptr->jmpbuf)) {
		png_destroy_write_struct (&png_ptr, &info_ptr);
		(*job->close) (fp);
		return;
	}

	png_init_io (png_ptr, fp);
	png_set_compression_level (png_ptr, job->compress);

	info_ptr->width = image->width;
	info_ptr->height= image->height;
	info_ptr->bit_depth = 8;
	info_ptr->color_type = PNG_COLOR_TYPE_RGB;
	if ((image->bits_per_pixel == 16) ||
		(image->bits_per_pixel == 8)) {
		info_ptr->color_type |= PNG_COLOR_MASK_COLOR;
		/* may be we need this? seems to be so
		 */
		png_set_sBIT (png_ptr, info_ptr, &sig_bit);
		png_set_shift (png_ptr, &sig_bit);
	}
	png_write_info (png_ptr, info_ptr);

	/* for ZPixmap bits_per_pixel could be 1,4,8,16,24,32
	 * but for Direct- and TrueColor it could only be 8,16,24,32 (?)
	 */
	switch (image->bits_per_pixel) {
		case 8:
			/* for 8bpp x server
			 */
			p8 = (unsigned char *) image->data;
			pad = image->bytes_per_line - image->width;
			for (row = 0; row < image->height; row++) {
				row_ptr = line;
				for (col = 0; col < image->width; col++) {
					*row_ptr++ = (*p8 & image->red_mask)   >>c_info.red_shift;
					*row_ptr++ = (*p8 & image->green_mask) >>c_info.green_shift;
					*row_ptr++ = (*p8 & image->blue_mask)  >>c_info.blue_shift;
                    p8++;
				}
				/* write out the row
				 */
				png_write_row (png_ptr, (png_bytep)line);
				/* eat paded bytes .. */
				p8 += pad;
			}
			break;
		case 16:
			/* for 16bpp and 15bpp x server
			 */
			p16 = (unsigned short *) image->data;
			pad = (image->bytes_per_line - (image->width << 1)) >> 1;
			for (row = 0; row < image->height; row++) {
				row_ptr = line;
				for (col = 0; col < image->width; col++) {
					*row_ptr++ = (*p16 & image->red_mask)  >>c_info.red_shift;
					*row_ptr++ = (*p16 & image->green_mask)>>c_info.green_shift;
					*row_ptr++ = (*p16 & image->blue_mask) >>c_info.blue_shift;
                    p16++;
				}
				/* write out the row
				 */
				png_write_row (png_ptr, (png_bytep)line);
				/* we have to eat paded bytes .. */
				p16 += pad;
			}
			break;
		case 24:
			for (row = 0; row < image->height; row++) {
				row_ptr = image->data+(row * image->bytes_per_line);
				/* write out the row
				 */
				if (image->byte_order == LSBFirst) {
					/* we have to swap all RGBs :( */
					unsigned char t;
					col_ptr = row_ptr;
					for (col = 0; col < image->width; col++) {
						t = col_ptr[0];
						col_ptr[0] = col_ptr[2];
						col_ptr[2] = t;
						col_ptr += 3;
					}
				}
				png_write_row (png_ptr, (png_bytep)row_ptr);
			}
			break;
		case 32: /* alpha + RGB */
			{
			register unsigned int
				rmask = image->red_mask,
				gmask = image->green_mask,
				bmask = image->blue_mask,
				rshift= c_info.red_shift,
				gshift= c_info.green_shift,
				bshift= c_info.blue_shift,
				*p32 = (unsigned int *) image->data;
			pad = (image->bytes_per_line - (image->width << 2)) >> 2;
			for (row = 0; row < image->height; row++) {
				row_ptr = line;
				for (col = 0; col < image->width; col++) {
					*row_ptr++ = (*p32 & rmask) >> rshift;
					*row_ptr++ = (*p32 & gmask) >> gshift;
					*row_ptr++ = (*p32 & bmask) >> bshift;
					p32++;
				}
				png_write_row (png_ptr, (png_bytep)line);
				/* we have to eat paded bytes */
				p32 += pad;
			}
			} break;
		default:
			printf ("bits_per_pixel not supported: %d\n",image->bits_per_pixel);
			break;
	}
	png_write_end (png_ptr, info_ptr);
	png_destroy_write_struct (&png_ptr, &info_ptr);
}

/*
 * for indexed color and gray scale PNG files
 */
void
XImageToPNG8 (FILE *fp, XImage *image, Job *job)
{
	int row;
	unsigned char *row_ptr;

	if ( job->state & VC_START ) {
		/* it's the first call, prepare some stuff
		 */
#ifdef DEBUG2
		dump_ximage_info (image);
#endif
	}

	png_ptr = png_create_write_struct (PNG_LIBPNG_VER_STRING,
				(void *)NULL, NULL, NULL);
	if (!png_ptr)
		return;
	info_ptr = png_create_info_struct (png_ptr);
	if (!info_ptr) {
		png_destroy_read_struct (&png_ptr, (png_infopp)NULL, (png_infopp)NULL);
		return;
	}
	if (setjmp (png_ptr->jmpbuf)) {
		png_destroy_write_struct (&png_ptr, &info_ptr);
		(*job->close) (fp);
		return;
	}

	png_init_io (png_ptr, fp);
	png_set_compression_level (png_ptr, job->compress);

	info_ptr->width = image->width;
	info_ptr->height= image->height;
	info_ptr->bit_depth = 8;
	if (job->win_attr.visual->class == StaticGray) {
		info_ptr->color_type = PNG_COLOR_TYPE_GRAY;
	} else {
		info_ptr->color_type = PNG_COLOR_TYPE_PALETTE;
		png_set_PLTE (png_ptr, info_ptr, job->color_table, job->ncolors);
	}

	png_write_info (png_ptr, info_ptr);

	/* for ZPixmap bits_per_pixel could be 1,4,8,16,24,32
	 * but for Direct- and TrueColor it could only be 16,24,32
	 */
	switch (image->bits_per_pixel) {
		case 8:
			/* for 8bpp server
			 */
			for (row = 0; row < image->height; row++) {
				row_ptr = image->data+(row * image->bytes_per_line);
				/* write out the row
				 */
				png_write_row (png_ptr, (png_bytep)row_ptr);
				/* do we have to eat paded bytes ?? */
			}
			break;
		default:
			printf ("bits_per_pixel not supported: %d\n",image->bits_per_pixel);
			break;
	}
	png_write_end (png_ptr, info_ptr);
	png_destroy_write_struct (&png_ptr, &info_ptr);
}

/*
 * clean up some allocated stuff
 */
void
PngClean (Job *job)
{
	if (line) {
		free (line);
		line = NULL;
	}
}
#endif /* HAVE_LIBPNG */

