# -*- coding: utf-8 -*-


import os
import subprocess
from glob import glob

from RestrictedManager.core import FWHandler

from gettext import gettext as _

class Bcm43xx(FWHandler):
    is_handler = True

    name = "bcm43xx"
    rationale = _("While the bcm43xx driver is free, it relies on proprietary firmware. \
Without that you will not be able to use your wireless card.")
    driver_package = "bcm43xx-fwcutter"

    def __init__(self, module):
        FWHandler.__init__(self, module)

    def description(self):
        return _("Firmware for Broadcom 43xx chipset family")

    def enable(self):
        """Enable the firmware."""

        # When installing bcm43xx-fwcutter, the package, trough debconf
        # tries to cut the fw. We prefer to avoid it since:
        # - fw gets cutted to wrong dir, moreover
        # - debconf asks the user.. this could be very misleading
        # - we want r-m to do the job
        #
        # since bmc43xx postinst is like this
        #
        # db_get bcm43xx-fwcutter/cut_firmware
        # if [ "$RET" = "true" -a ! -e /lib/firmware/bcm43xx_microcode11.fw ]; then
        #     /usr/share/bcm43xx-fwcutter/install_bcm43xx_firmware.sh
        # fi
        #
        # we write a fake bcm43xx_microcode11.fw file in order to skip
        # install_bcm43xx_firmware.sh script
        fw_fake_file = open("/lib/firmware/bcm43xx_microcode11.fw", "w")
        fw_fake_file.write("this is just a fake file to prevent postinst scripts")
        fw_fake_file.close()

        # Synaptic runs in non-interactive mode so that the debconf
        # question is skipped
        if not self.package_installed(self.driver_package):

            # we need to pass a driver handler instance to this method
            # because it actually belongs to the frontend
            # so the 'self' passed by it is not the driver instance
            # this behaviour is a design flaw and will be corrected soon

            handler_instance = self
            self.install_package(handler_instance, self.driver_package)
        if not self.package_installed(self.driver_package):
            # could not be installed
            return

        res, fw_location = self.FWSelectDialog(UrlText='http://xeve.de/down/wl_apsta.o',
                                               FWCheckCommand=["/usr/bin/bcm43xx-fwcutter", "-i"]
                                               ).run()
        if res != True:
            return

        # create path for the fw
        fw_path = os.path.join("/lib/firmware/", os.uname()[2])
        if not(os.path.exists(fw_path)):
            os.mkdir(fw_path)
        # cut the fw file
        proc = subprocess.Popen(["/usr/bin/bcm43xx-fwcutter", "-w", fw_path, fw_location])
        proc.wait()

        FWHandler.enable(self)

    def is_enabled(self):
        """Returns True if the fw is installed."""

        return len(glob("/lib/firmware/%s/bcm43xx_*" % os.uname()[2])) > 0

    def disable(self):
        """Uninstalls bcm43xx package and removes fw files."""

        if self.package_installed(self.driver_package):

            # we need to pass a driver handler instance to this method
            # because it actually belongs to the frontend
            # so the 'self' passed by it is not the driver instance
            # this behaviour is a design flaw and will be corrected soon

            handler_instance = self
            self.remove_package(handler_instance, self.driver_package)

        for f in glob('/lib/firmware/%s/bcm43xx_*' % os.uname()[2]):
            os.unlink(f)

        fw_fake_file = "/lib/firmware/bcm43xx_microcode11.fw"
        if os.path.exists(fw_fake_file):
            os.unlink(fw_fake_file)

        FWHandler.disable(self)

    def is_loaded(self):
        """Returns True if the module is loaded."""

        return self.is_enabled()
