# -*- coding: utf-8 -*-

import os, os.path, subprocess

from RestrictedManager.handlers.lrm_driver import LRMDriverHandler, MANAGER_ROOT, notify_reboot_required

import xorgconfig

from gettext import gettext as _

class XorgDriverHandler(LRMDriverHandler):
    """Abstract class for a restricted X.org graphics driver."""

    def __init__(self, module, driver_package, xorg_driver, extra_conf_options,
        free_driver, add_modules, remove_modules):
        """Build graphics driver handler for a particular concrete kernel
        module."""

        LRMDriverHandler.__init__(self, module)
        self.driver_package = driver_package
        self.xorg_driver = xorg_driver
        self.extra_xorg_driver_options = extra_conf_options
        self.free_driver = free_driver
        self.add_modules = add_modules
        self.remove_modules = remove_modules

        try:
            self.xorg_conf = xorgconfig.readConfig("/etc/X11/xorg.conf")
        except (IOError, xorgconfig.ParseException, AttributeError):
            self.xorg_conf = None

    def can_change(self):
        if self.xorg_conf:
            return None
        else:
            return _("Reconfiguring X.org video drivers is not possible: /etc/X11/xorg.conf is invalid or does not exist.")

    def is_enabled(self):
        """Returns True if the module is enabled."""

        if self.xorg_conf:
            driver = self.xorg_conf.getSections("device")[0].driver
            if driver == self.xorg_driver:
                return LRMDriverHandler.is_enabled(self)
            else:
                return False

        else:
            return LRMDriverHandler.is_enabled(self)

    def _mod_enabled(self, module):
        """Check if current xorg configuration enables a module."""

        if len(self.xorg_conf.getSections("module")) < 1:
            return False
        for m in self.xorg_conf.getSections("module")[0]._contents:
            if len(m._row) >= 2 and m._row[1] == module:
                return True
        return False

    def enable_config_hook(self):
        """Can be overridden by subclasses for custom self.xorg_config changes
        after driver, modules, and extra driver options have been changed."""

        pass

    def disable_config_hook(self):
        """Can be overridden by subclasses for custom self.xorg_config changes
        after driver, modules, and extra driver options have been changed."""

        pass

    def enable(self):
        """Enable the module."""

        if not self.package_installed(self.driver_package):

            # we need to pass a driver handler instance to this method
            # because it actually belongs to the frontend
            # so the 'self' passed by it is not the driver instance
            # this behaviour is a design flaw and will be corrected soon

            handler_instance = self
            self.install_package(handler_instance, self.driver_package)

        # do not mangle xorg.conf if package installation has been aborted
        if not self.package_installed(self.driver_package):
            return

        LRMDriverHandler.enable(self)

        # backup the current xorg.conf
        open(os.path.join(MANAGER_ROOT, self.name + ".oldconf"), "w").write(
            open("/etc/X11/xorg.conf").read())

        device = self.xorg_conf.getSections("device")[0]
        device.driver = self.xorg_driver

        have_modules = len(self.xorg_conf.getSections("module")) > 0
        if have_modules:
            for m in self.remove_modules:
                self.xorg_conf.getSections("module")[0].removeModule(m)
        if self.add_modules:
            if not have_modules:
                self.xorg_conf.append(self.xorg_conf.makeSection(None, ["Section",
                    "Module"]))
            for m in self.add_modules:
                if not self._mod_enabled(m):
                    self.xorg_conf.getSections("module")[0].addModule(m)

        for k, v in self.extra_xorg_driver_options.iteritems():
            device.option.removeOptionByName(k)
            device.option.appendOptionRow([k, v])

        self.enable_config_hook()

        self.xorg_conf.writeConfig("/etc/X11/xorg.conf")

        notify_reboot_required()

    def disable(self):
        """Disable the module."""

        if self.package_installed(self.driver_package):

            # we need to pass a driver handler instance to this method
            # because it actually belongs to the frontend
            # so the 'self' passed by it is not the driver instance
            # this behaviour is a design flaw and will be corrected soon

            handler_instance = self
            self.remove_package(handler_instance, self.driver_package)

        # do not mangle xorg.conf if package uninstallation has been aborted
        if self.package_installed(self.driver_package):
            return

        LRMDriverHandler.disable(self)

        # if we have the previous xorg.conf, restore that
        oldconf = os.path.join(MANAGER_ROOT, self.name + ".oldconf")
        if os.path.exists(oldconf):
            open("/etc/X11/xorg.conf", "w").write(open(oldconf).read())
            self.xorg_conf = xorgconfig.readConfig("/etc/X11/xorg.conf")
        else: # no backup, so mangle current config manually
            # try to get the previous driver
            try:
                p = os.path.join(MANAGER_ROOT, self.name + ".olddriver")
                free_driver = open(p).read().strip()
                os.unlink(p)
            except IOError:
                free_driver = self.free_driver

            device = self.xorg_conf.getSections("device")[0]
            device.driver = free_driver

            have_modules = len(self.xorg_conf.getSections("module")) > 0
            if have_modules:
                for m in self.add_modules:
                    self.xorg_conf.getSections("module")[0].removeModule(m)
            if self.remove_modules:
                if not have_modules:
                    self.xorg_conf.append(self.xorg_conf.makeSection(None, ["Section",
                        "Module"]))
                for m in self.remove_modules:
                    if not self._mod_enabled(m):
                        self.xorg_conf.getSections("module")[0].addModule(m)

            for k in self.extra_xorg_driver_options:
                device.option.removeOptionByName(k)

            self.disable_config_hook()

            self.xorg_conf.writeConfig("/etc/X11/xorg.conf")

        notify_reboot_required()
