; Miscellaneous emacs utilities written by Matt Kaufmann to help with
; markup of ACL2 documentation strings.

; Can we have tilde directives in verbatim enviroments, as in deflabel xargs
; and the :doc for LD?

;; When bringing up a file:

;; (find-bogus-lines)

;; Repeat (find-bad-section-break).

;; control-t [

;; control-t v

;; Look for :more and :doc, (setq case-fold-search nil) .

;; Finish proof-checker-b.lisp

;; When I'm ALL done, I should search for "l[doc-string]" and refer
;; perhaps to the markup-table issue.  But mainly, I should update
;; the documentation for doc-string.  Be sure to mention that ~ev
;; should be on a new line...

;; Also when I'm done, look for broken lines in code:
; (tags-search "~c\\[[^]]*\n")
; Or search forward:
; (re-search-forward "~c\\[[^]]*\n")

; Need to stay in fundamental mode.

; May need to supply the current directory below.
(load "fill-tildes.el")

(defvar tilde-loaded-p)

(defun tilde-extend-auto-mode-alist ()
  (cond
   ((not (boundp 'tilde-loaded-p))
    (setq auto-mode-alist
	  (cons '("\\.lisp" . fundamental-mode)
		auto-mode-alist))
    (setq tilde-loaded-p t))))

; (tilde-extend-auto-mode-alist)

(or (boundp 'ctl-t-keymap)
    (progn (defvar ctl-t-keymap)
	   (setq ctl-t-keymap (make-sparse-keymap))
	   (define-key (current-global-map) "\C-T" ctl-t-keymap)))

(defun in-doc-section-p ()
  (save-excursion
    (let ((saved-point (point))
          (case-fold-search t))
      (cond
       ((search-backward "\":doc-section" nil t)
        (let ((start-point (point)))
          (forward-sexp 1)
          (end-of-line) ;we want to fill the final paragraph too
          (let ((end-point (point)))
            (and (< start-point saved-point)
                 (<= saved-point end-point)))))))))

(defun move-to-doc-section ()
  ;; Returns t iff already in a doc section.  Also, moves past the line
  ;; starting with :doc-section.
  (cond
   ((in-doc-section-p)
    (cond
     ((save-excursion
        (beginning-of-line)
        (looking-at "[ ]+\":Doc-Section"))
      (beginning-of-line)
      (forward-line 2)))
    t)
   ((let ((case-fold-search t))
      (search-forward "\":doc-section" nil t))
    (forward-line 2)
    nil)
   (t (error "No more :Doc-Sections in this buffer."))))

(defun my-query (msg)
  (progn
    (message (concat msg " (y, n, q)"))
    (let ((ans (read-char)))
      (cond
       ((memq ans '(89 121))
        (message "")
        t)
       ((memq ans '(78 110))
        (message "")
        nil)
       (t (error "Aborting."))))))

(defun my-query3 (msg)
  (progn
    (message (concat msg " (y, n, s, q)"))
    (let ((ans (read-char)))
      (cond
       ((memq ans '(89 121))
        (message "")
        'y)
       ((memq ans '(78 110))
        (message "")
        'n)
       ((memq ans '(83 115))
        's)
       (t (error "Aborting."))))))

(defun replace-see-doc (str endpoint)
  (kill-region (point) endpoint)
  (delete-char 1)
  (insert str)
  (forward-sexp 1)
  (insert "]")
  (message "Done."))

(defun replace-doc-1 ()
  (let ((case-fold-search nil))
    (search-forward ":DOC")
    (while (not (move-to-doc-section))	;so, we have to move
      (search-forward ":DOC"))
    (save-excursion
      (let ((saved-point (point)))
	(backward-char 4)
	(backward-sexp 1)
	(cond
	 ((looking-at "See")
	  (if (my-query "Replace with ~l?")
	      (replace-see-doc "~l[" saved-point)
	    nil))
	 ((looking-at "see")
	  (if (my-query "Replace with ~pl?")
	      (replace-see-doc "~pl[" saved-point)
	    nil))
	 (t
	  (goto-char saved-point)
	  (error "No suggestions.  Control is returned to you.")))))))

(defun replace-doc ()
  (interactive)
  (let ((flg t))
    (while flg
      (replace-doc-1)
      (setq flg (my-query "Continue?"))))
  (message "Done."))

(defun end-verbatim ()
  ;; Returns 'forced if the end was made without query, t if the end was made
  ;; with query.
  (let (flg)
    (while (not flg)
      (search-forward "

")
      (cond
       ((let ((temp (buffer-substring (- (point) 2) (- (point) 4))))
          (or (equal temp "~/")
              (equal temp "\")")))
        (setq flg t)
        (beep)
        (setq flg 'forced)
        (backward-char 4)
        (insert "\n  ~ev[]")
        (backward-char 4))
       ((setq flg (my-query "End verbatim here?"))
        (re-search-forward "[^ ]")
        (backward-char 1)
        (insert "~ev[]"))))
    flg))

(defun verbatim-1 ()
  ;; Returns 'forced if the last end-verbatim was done without query,
  ;; t if it was done with query, else nil.
  (progn
    ;;(re-search-forward "[^\"][^\\]\n\n")
    (re-search-forward "\n\n")
    (backward-char 2)
    (cond
     ((move-to-doc-section)
      (let ((ans (my-query3 "Start verbatim?")))
        (cond
         ((eq ans 'y)
	  (if (not (equal (buffer-substring (point) (- (point) 7))
                          "~ev[]~/"))
              (fill-wrt-tildes))
          (insert "~bv[]")
          (forward-char 2)
          (end-verbatim))
         ((eq ans 'n)
          (if (not (equal (buffer-substring (point) (- (point) 7))
                          "~ev[]~/"))
              (fill-wrt-tildes))
          nil)
         (t ;(eq ans 's)
          nil)))))))

(defun verbatim ()
  (interactive)
  (let (flg)
    (while (not flg)
      (let ((temp (verbatim-1)))
        (if temp
            (setq flg (not (my-query
                            (if (eq temp 'forced)
                                "Previous verbatim ended.  Continue?"
                              "Continue?"))))
          (forward-char 3))))))

(defun tilde-fill-paragraphs ()
  (interactive)
  (let ((flg t))
    (while flg
      (move-to-doc-section)
      (fill-wrt-tildes)
      (and (setq flg (my-query "Continue")) 
           (search-forward "

"
                           )))))

(defun make-link (&optional arg)
  (interactive "P")
  (let ((saved-point (point))
        (case-fold-search nil))
    (search-forward ":DOC")
    (forward-sexp 1)
    (backward-sexp 1)
    (delete-region saved-point (point))
    (if arg
        (insert "~l[")
      (insert "~pl["))
    (forward-sexp 1)
    (insert "]")))

(defun emphasize-section-header ()
  (interactive)
  (beginning-of-line)
  (re-search-forward "[^ ]")
  (backward-char 1)
  (insert "~em[")
  (end-of-line)
  (insert "]"))

(define-key ctl-t-keymap "[" 'replace-doc)
(define-key ctl-t-keymap "m" 'make-link)
(define-key ctl-t-keymap "v" 'verbatim)
(define-key ctl-t-keymap "p" 'tilde-fill-paragraphs)
(define-key ctl-t-keymap "]" 'emphasize-section-header)
(define-key ctl-t-keymap "f" 'fill-wrt-tildes)

(defun kill-except-doc-sections ()
  (interactive)
  (beginning-of-buffer)
  (let (done)
    (while (not done)
      (let ((saved-point (point)))
        (cond
         ((search-forward ":doc-section" nil t)
          (beginning-of-defun)
          (delete-region saved-point (point))
          (open-line 2)
          (search-forward ":doc-section")
          (beginning-of-line)
          (let ((beg (point))
                (end (progn (forward-sexp 1) (point))))
            (let ((str (buffer-substring beg end)))
              (delete-region beg end)
              (beginning-of-defun)
              (end-of-line)
              (newline 2)
              (insert str)
              (let ((saved-point1 (point)))
                (delete-region saved-point1
                               (progn (end-of-defun) (point)))
                (insert ")")))))
         (t (delete-region saved-point (point-max))
            (newline)
            (setq done t)))))))

(defun find-bad-section-break ()
  (interactive)
  (re-search-forward "~/[\n][ ]+[^\n]"))

(defun find-bogus-lines ()
  (interactive)
  (beginning-of-buffer)
  (while (re-search-forward "\n[ ]+\n" nil t)
    (previous-line 1)
    (end-of-line)
    (cond
     ((my-query "replace?")
      (let ((saved-point (point)))
        (beginning-of-line)
        (delete-region (point) saved-point))))
    (cond
     ((not (my-query "continue?"))
      (error "Aborting.")))))

(defun find-unbalanced-bv ()
  (beginning-of-buffer)
  (while (re-search-forward "[^~]~bv\\[" nil t)
    (let ((beg (point)))
      (if (re-search-forward "[^~]~ev\\[" nil t)
          (let ((end (point)))
            (if (re-search-backward "[^~]~bv" beg t)
                (error "Extra ~bv here!")))
        (error "No closing ~ev for here!")))) 
  (message "All bv/ev pairs seem balanced"))

(defun find-unbalanced-ev ()
  (beginning-of-buffer)
  (while (re-search-forward "[^~]~ev\\[" nil t)
    (let ((beg (point)))
      (or (re-search-forward "[^~]~bv\\[" nil t)
          (end-of-buffer))
      (let ((end (point)))
        (if (re-search-backward "[^~]~ev" beg t)
            (error "Extra ~ev here!"))))) 
  (message "All bv/ev pairs seem balanced"))

(defun fub ()
  (interactive)
  (find-unbalanced-bv)
  (find-unbalanced-ev))

(defun spaces-prefix ()
  (save-excursion
    (beginning-of-line)
    (let ((saved-point (point)))
      (re-search-forward "[^ ]")
      (buffer-substring saved-point (1- (point))))))

(defun find-bad-ev ()
  (interactive)
  (re-search-forward "[^ ]~ev"))

(defun fix-verbatim ()
  (interactive)
  (let ((continue-flg t))
    (while (and continue-flg
                (or (search-forward "~bv" nil t)
                    (progn (beep) (message "All done.") nil)))
      (let ((spaces-prefix (spaces-prefix)))
        (cond
         ((my-query "replace?")
          (delete-region (- (point) 3) (+ (point) 2))
          (cond
           ((looking-at "\n")
            (forward-line 1)
            (insert spaces-prefix)
            (insert "~bv[]")
            (search-forward "~ev")
            (cond
             ((looking-at "\\[\\]~/~bv\\[\\]")
              (beep)
              (cond
               ((my-query "replace with ~/ ?")
                (beginning-of-line)
                (let ((kill-whole-line nil))
                  (kill-line))
                (delete-char -1)
                (insert "~/")
                (search-forward "~ev"))
               (t (error "aborting")))))
            (cond
             ((not (equal (buffer-substring (- (point) (+ (length spaces-prefix) 5))
                                            (- (point) (+ (length spaces-prefix) 3)))
                          "\n\n"))
              (error "Not looking at newline!"))
             ((my-query "replace?")
              (delete-region (- (point) 3) (+ (point) 2))
              (forward-line -1)
              (beginning-of-line)
              (cond
               ((looking-at "\n")
                (insert spaces-prefix)
                (insert "~ev[]"))
               (t (error "Not looking at newline!"))))
             (t (setq continue-flg nil))))
           (t (error "Not looking at newline!"))))
         (t (setq continue-flg nil)))))))

(define-key ctl-t-keymap "b" 'fix-verbatim)

(defun codify-string (string)
  (interactive "sString to codify: ")
  (let ((pat (concat "[\n ]" string "[\n )'>.,;:]"))
        (continue-flg t))
    (while continue-flg
      (cond
       ((re-search-forward pat nil t)
        (let ((beg (+ (match-beginning 0) 1))
              (end (- (match-end 0) 1)))
          (cond
           ((not (move-to-doc-section)))
           ((my-query "Mark with ~c?")
            (backward-char 1)
            (insert "]")
            (goto-char beg)
            (insert "~c[")
            (goto-char end)
            (cond
             ((my-query "Continue?"))
             (t (setq continue-flg nil)))))))
       (t (beep)
          (setq continue-flg nil)
          (message "Done."))))))

(defvar *acl2-files*
  '("acl2.lisp" "acl2-fns.lisp" "init.lsp" "acl2-init.lisp"
    "akcl-acl2-trace.lisp" "axioms.lisp" "basis.lisp" "translate.lisp"
    "type-set-a.lisp" "type-set-b.lisp" "rewrite.lisp" "simplify.lisp"
    "other-processes.lisp" "induct.lisp" "prove.lisp"
    "history-management.lisp" "defuns.lisp" "proof-checker-a.lisp"
    "defthm.lisp" "other-events.lisp" "ld.lisp"
    "proof-checker-pkg.lisp" "proof-checker-b.lisp" "tutorial.lisp"
    "interface-raw.lisp" "defpkgs.lisp"))

; May need to edit the following.
(defvar *acl2-directory*
  "/slocal/src/acl2/v1-8/")

(defun find-files-read-only (files)
  (save-window-excursion
    (let ((rest files))
      (while rest
        (let ((next-file (concat *acl2-directory* (car rest))))
          (message next-file)
          (find-file-read-only next-file)
          (setq rest (cdr rest)))))
    (message "Done.")))

(defun acl2-find-files-read-only ()
  (interactive)
  (find-files-read-only *acl2-files*))
