; ACL2 Version 2.8 -- A Computational Logic for Applicative Common Lisp
; Copyright (C) 2004  University of Texas at Austin

; This version of ACL2 is a descendent of ACL2 Version 1.9, Copyright
; (C) 1997 Computational Logic, Inc.  See the documentation topic NOTE-2-0.

; This program is free software; you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation; either version 2 of the License, or
; (at your option) any later version.

; This program is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.

; You should have received a copy of the GNU General Public License
; along with this program; if not, write to the Free Software
; Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

; Written by:  Matt Kaufmann               and J Strother Moore
; email:       Kaufmann@cs.utexas.edu      and Moore@cs.utexas.edu
; Department of Computer Sciences
; University of Texas at Austin
; Austin, TX 78712-1188 U.S.A.

(in-package "ACL2")

;  We permit macros under the following constraints on the args.

;  1.  No destructuring.  (Maybe some day.)
;  2.  No &aux.           (LET* is better.)
;  3.  Initforms must be quotes.  (Too hard for us to do evaluation right.)
;  4.  No &environment.   (Just not clearly enough specified in CLTL.)
;  5.  No nonstandard lambda-keywords.  (Of course.)
;  6.  No multiple uses of :allow-other-keys.  (Implementations differ.)

;  There are three nests of functions that have the same view of
;  the subset of macro args that we support:  macro-vars...,
;  chk-macro-arglist..., and bind-macro-args...  Of course, it is
;  necessary to keep them all with the same view of the subset.

(defun macro-vars-key (args)

;  We have passed &key.

  (cond ((null args) nil)
        ((eq (car args) '&allow-other-keys)
         (cond ((null (cdr args))
                nil)
               (t (er hard nil "macro-vars-key"))))
        ((atom (car args))
         (cons (car args) (macro-vars-key (cdr args))))
        (t (let ((formal (cond
                          ((atom (car (car args)))
                           (car (car args)))
                          (t (cadr (car (car args)))))))
             (cond ((int= (length (car args)) 3)
                    (cons formal
                          (cons (caddr (car args))
                                (macro-vars-key (cdr args)))))
                   (t (cons formal (macro-vars-key (cdr args)))))))))

(defun macro-vars-after-rest (args)

;  We have just passed &rest or &body.

  (cond ((null args) nil)
        ((eq (car args) '&key)
         (macro-vars-key (cdr args)))
        (t (er hard nil "macro-vars-after-rest"))))

(defun macro-vars-optional (args)

;  We have passed &optional but not &key or &rest or &body.

  (cond ((null args) nil)
        ((eq (car args) '&key)
         (macro-vars-key (cdr args)))
        ((member (car args) '(&rest &body))
         (cons (cadr args) (macro-vars-after-rest (cddr args))))
        ((symbolp (car args))
         (cons (car args) (macro-vars-optional (cdr args))))
        ((int= (length (car args)) 3)
         (cons (caar args)
               (cons (caddr (car args))
                     (macro-vars-optional (cdr args)))))
        (t (cons (caar args)
                 (macro-vars-optional (cdr args))))))

(defun macro-vars (args)
  (cond ((null args)
         nil)
        ((eq (car args) '&whole)
         (cons (cadr args) (macro-vars (cddr args))))
        ((member (car args) '(&rest &body))
         (cons (cadr args) (macro-vars-after-rest (cddr args))))
        ((eq (car args) '&optional)
         (macro-vars-optional (cdr args)))
        ((eq (car args) '&key)
         (macro-vars-key (cdr args)))
        ((or (not (symbolp (car args)))
             (lambda-keywordp (car args)))
         (er hard nil "macro-vars"))
        (t (cons (car args) (macro-vars (cdr args))))))

(defun chk-legal-defconst-name (name state)
  (cond ((legal-constantp name) (value nil))
        ((legal-variable-or-constant-namep name)
         (er soft (cons 'defconst name)
             "The symbol ~x0 may not be declared as a constant because ~
              it does not begin and end with the character *."
             name))
        (t (er soft (cons 'defconst name)
               "Constant symbols must ~*0.  Thus, ~x1 may not be ~
                declared as a constant.  See :DOC name and :DOC ~
                defconst."
               (tilde-@-illegal-variable-or-constant-name-phrase name)
               name))))

(defun defconst-fn1 (name val doc doc-pair w state)
  (let ((w (update-doc-data-base
            name doc doc-pair
            (putprop name 'const (kwote val) w))))
    (value w)))

(defun defconst-fn (name form state doc event-form)

; Important Note:  Don't change the formals of this function without
; reading the *initial-event-defmacros* discussion in axioms.lisp.

  (with-ctx-summarized
   (if (output-in-infixp state) event-form (cons 'defconst name))
   (let ((wrld1 (w state))
         (event-form (or event-form (list* 'defconst name form
                                           (if doc (list doc) nil)))))
     (er-progn
      (chk-all-but-new-name name ctx 'const wrld1 state)
      (chk-legal-defconst-name name state)
      (let ((const-prop (getprop name 'const nil 'current-acl2-world wrld1)))
        (cond
         ((and const-prop
               (equal event-form (get-event name wrld1)))

; We stop the redundant event even before evaluating the form.  We believe
; that this is merely an optimization, even if the form calls compress1 or
; compress2 (which will not update the 'acl2-array property when supplied the
; same input as the last time the compress function was called).

          (stop-redundant-event state))
         (t
          (er-let*
           ((pair (simple-translate-and-eval form nil
                                             nil
                                             "The second argument of defconst"
                                             ctx wrld1 state))
            (val (value (cdr pair))))
           (cond
            ((and (consp const-prop)
                  (equal (cadr const-prop) val))

; When we store the 'const property, we kwote it so that it is a term.
; Thus, if there is no 'const property, we will getprop the nil and
; the consp will fail.

             (stop-redundant-event state))
            (t
             (er-let*
              ((wrld2 (chk-just-new-name name 'const nil ctx wrld1 state))
               (doc-pair (translate-doc name doc ctx state))
               (wrld3 (defconst-fn1 name val doc doc-pair wrld2 state)))
              (install-event name
                             event-form
                             'defconst
                             name
                             nil
                             (if (eq (small-p wrld1) t)
                                 nil
                               (list 'defconst name form val))
                             wrld3 state))))))))))))

(defun chk-legal-init (x ctx state)

; See the note in chk-macro-arglist before changing this fn to
; translate the init value.

  (cond ((and (consp x)
              (true-listp x)
              (int= 2 (length x))
              (eq (car x) 'quote))
         (value nil))
        (t (er soft ctx
               "Illegal initial value.  In ACL2 we require that ~
                initial values be quoted forms and you used ~x0.~
                ~#1~[  You should just write '~x0 instead.  Warren ~
                Teitelman once remarked that it was really dumb of a ~
                Fortran compiler to say ``missing comma!''  ``If it ~
                knows a comma is missing, why not just put one in?''  ~
                Indeed.~/~]  See :DOC macro-args."
               x
               (if (or (eq x nil)
                       (eq x t)
                       (acl2-numberp x)
                       (stringp x)
                       (characterp x))
                   0
                   1)))))

(defun chk-macro-arglist-keys (args keys-passed ctx state)
  (cond ((null args) (value nil))
        ((eq (car args) '&allow-other-keys)
         (cond ((null (cdr args)) (value nil))
               (t (er soft ctx
                      "&ALLOW-OTHER-KEYS may only occur as the last ~
                       member of an arglist so it is illegal to ~
                       follow it with ~x0.  See :DOC macro-args."
                      (cadr args)))))
        ((atom (car args))
         (cond ((symbolp (car args))
                (let ((new (intern (symbol-name (car args)) "KEYWORD")))
                  (cond ((member new keys-passed)
                         (er soft ctx
                             "The symbol-name of each keyword ~
                              parameter specifier must be distinct.  ~
                              But you have used the symbol-name ~s0 ~
                              twice.  See :DOC macro-args."
                             (symbol-name (car args))))
                        (t (chk-macro-arglist-keys
                            (cdr args)
                            (cons new
                                  keys-passed)
                            ctx state)))))
               (t (er soft ctx
                      "Each keyword parameter specifier must be ~
                       either a symbol or a list.  Thus, ~x0 is ~
                       illegal.  See :DOC macro-args."
                      (car args)))))
        ((or (not (true-listp (car args)))
             (> (length (car args)) 3))
         (er soft ctx
             "Each keyword parameter specifier must be either a ~
              symbol or a truelist of length 1, 2, or 3.  Thus, ~x0 ~
              is illegal.  See :DOC macro-args."
             (car args)))
        (t (er-progn
            (cond ((symbolp (caar args)) (value nil))
                  (t (cond ((or (not (true-listp (caar args)))
                                (not (equal (length (caar args))
                                            2))
                                (not (keywordp (car (caar args))))
                                (not (symbolp (cadr (caar args)))))
                            (er soft ctx
                                "Keyword parameter specifiers in ~
                                 which the keyword is specified ~
                                 explicitly, e.g., specifiers of the ~
                                 form ((:key var) init svar), must ~
                                 begin with a truelist of length 2 ~
                                 whose first element is a keyword and ~
                                 whose second element is a symbol.  ~
                                 Thus, ~x0 is illegal.  See :DOC ~
                                 macro-args."
                                (car args)))
                           (t (value nil)))))
            (let ((new (cond ((symbolp (caar args))
                              (intern (symbol-name (caar args))
                                      "KEYWORD"))
                             (t (car (caar args))))))
              (er-progn
               (cond ((member new keys-passed)
                      (er soft ctx
                          "The symbol-name of each keyword parameter ~
                           specifier must be distinct.  But you have ~
                           used the symbol-name ~s0 twice.  See :DOC ~
                           macro-args."
                          (symbol-name new)))
                     (t (value nil)))
               (cond ((> (length (car args)) 1)
                      (chk-legal-init (cadr (car args)) ctx state))
                     (t (value nil)))
               (cond ((> (length (car args)) 2)
                      (cond ((symbolp (caddr (car args)))
                             (value nil))
                            (t (er soft ctx
                                   "~x0 is an illegal keyword ~
                                    parameter specifier because the ~
                                    ``svar'' specified, ~x1, is not a ~
                                    symbol.  See :DOC macro-args."
                                   (car args)
                                   (caddr (car args))))))
                     (t (value nil)))
               (chk-macro-arglist-keys (cdr args) (cons new keys-passed)
                                       ctx state)))))))

(defun chk-macro-arglist-after-rest (args ctx state)
  (cond ((null args) (value nil))
        ((eq (car args) '&key)
         (chk-macro-arglist-keys (cdr args) nil ctx state))
        (t (er soft ctx
               "Only keyword specs may follow &REST or &BODY.  See ~
                :DOC macro-args."))))

(defun chk-macro-arglist-optional (args ctx state)
  (cond ((null args) (value nil))
        ((member (car args) '(&rest &body))
         (cond ((and (cdr args)
                     (symbolp (cadr args))
                     (not (lambda-keywordp (cadr args))))
                (chk-macro-arglist-after-rest (cddr args) ctx state))
               (t (er soft ctx
                      "~x0 must be followed by a variable symbol.  ~
                       See :DOC macro-args."
                      (car args)))))
        ((eq (car args) '&key)
         (chk-macro-arglist-keys (cdr args) nil ctx state))
        ((symbolp (car args))
         (chk-macro-arglist-optional (cdr args) ctx state))
        ((or (atom (car args))
             (not (true-listp (car args)))
             (not (< (length (car args)) 4)))
         (er soft ctx
             "Each optional parameter specifier must be either a ~
              symbol or a true list of length 1, 2, or 3.  ~x0 is ~
              thus illegal.  See :DOC macro-args."
             (car args)))
        (t (er-progn
            (cond ((symbolp (car (car args))) (value nil))
                  (t (er soft ctx
                         "~x0 is an illegal optional parameter ~
                          specifier because the ``variable symbol'' ~
                          used is not a symbol.  See :DOC macro-args."
                         (car args))))
            (cond ((> (length (car args)) 1)
                   (chk-legal-init (cadr (car args)) ctx state))
                  (t (value nil)))
            (cond ((int= (length (car args)) 3)
                   (cond ((symbolp (caddr (car args)))
                          (value nil))
                         (t (er soft ctx
                                "~x0 is an illegal optional parameter ~
                                 specifier because the ``svar'' ~
                                 specified, ~x1, is not a symbol.  ~
                                 See :DOC macro-args."
                                (car args)
                                (caddr (car args))))))
                  (t (value nil)))
            (chk-macro-arglist-optional (cdr args) ctx state)))))

(defun chk-macro-arglist1 (args ctx state)
  (cond ((null args) (value nil))
        ((not (symbolp (car args)))
         (er soft ctx
             "~x0 is illegal as the name of a required formal ~
              paramter.  See :DOC macro-args."
             (car args)))
        ((member (car args) '(&rest &body))
         (cond ((and (cdr args)
                     (symbolp (cadr args))
                     (not (lambda-keywordp (cadr args))))
                (chk-macro-arglist-after-rest (cddr args)
                                              ctx state))
               (t (er soft ctx
                      "~x0 must be followed by a variable symbol.  ~
                       See :DOC macro-args."
                      (car args)))))
        ((eq (car args) '&optional)
         (chk-macro-arglist-optional (cdr args) ctx state))
        ((eq (car args) '&key)
         (chk-macro-arglist-keys (cdr args) nil ctx state))
        (t (chk-macro-arglist1 (cdr args) ctx state))))

(defun subsequencep (lst1 lst2)

; We return t iff lst1 is a subsequence of lst2, in the sense that
; '(a c e) is a subsequence of '(a b c d e f) but '(a c b) is not.

  (cond ((null lst1) t)
        (t (let ((tl (member (car lst1) lst2)))
             (cond ((null tl) nil)
                   (t (subsequencep (cdr lst1) (cdr tl))))))))

(defun collect-lambda-keywordps (lst)
  (cond ((null lst) nil)
        ((lambda-keywordp (car lst))
         (cons (car lst) (collect-lambda-keywordps (cdr lst))))
        (t (collect-lambda-keywordps (cdr lst)))))

(defun chk-macro-arglist (args ctx state)

; Any modification to this function and its subordinates must cause
; one to reflect on the two function nests bind-macro-args...  and
; macro-vars... because they assume the presence of the structure that
; this function checks for.  See the comment before macro-vars for the
; restrictions we impose on macros.

; The subordinates of this function do not check that symbols that
; occur in binding spots are non-keywords and non-constants and
; without duplicates.  That check is performed here, with chk-arglist,
; as a final pass.

; Important Note:  If ever we change this function so that instead of
; just checking the args it "translates" the args, so that it returns
; the translated form of a proper arglist, then we must visit a similar
; change on the function primordial-event-macro-and-fn, which currently
; assumes that if a defmacro will be processed without error then
; the macro-args are exactly as presented in the defmacro.

; The idea of translating macro args is not ludicrous.  For example,
; the init-forms in keyword parameters must be quoted right now.  We might
; want to allow naked numbers or strings or t or nil.  But then we'd
; better go look at primordial-event-macro-and-fn.

; It is very suspicious to think about allowing the init forms to be
; anything but quoted constants because Common Lisp is very vague about
; when you get the bindings for free variables in such expressions
; or when such forms are evaluated.

  (er-progn
   (cond ((not (true-listp args))
          (er soft ctx
              "The arglist ~x0 is not a true list.  See :DOC ~
               macro-args."
              args))
         (t (value nil)))
   (let ((lambda-keywords (collect-lambda-keywordps args)))
     (cond
      ((or (subsequencep lambda-keywords
                         '(&whole &optional &rest &key &allow-other-keys))
           (subsequencep lambda-keywords
                         '(&whole &optional &body &key &allow-other-keys)))
       (cond (args
              (cond ((eq (car args) '&whole)
;  &whole can only appear at the very beginning.
                     (cond ((and (consp (cdr args))
                                 (symbolp (cadr args))
                                 (not (lambda-keywordp (cadr args))))
                            (chk-macro-arglist1 (cddr args) ctx state))
                           (t (er soft ctx
                                  "When the &whole lambda-list ~
                                   keyword is used it must be the ~
                                   first element of the lambda-list ~
                                   and it must be followed by a variable ~
                                   symbol.  This is not the case in ~
                                   ~x0.  See :DOC macro-args."
                                  args))))
                    (t (chk-macro-arglist1 args ctx state))))
             (t (value nil))))
      (t (er soft ctx
             "The lambda-list keywords allowed by ACL2 are &WHOLE, ~
              &OPTIONAL, &REST, &BODY, &KEY, and &ALLOW-OTHER-KEYS.  ~
              These must occur (if at all) in that order, with no ~
              duplicate occurrences and at most one of &REST and ~
              &BODY.  The argument list ~x0 is thus illegal."
             args))))
   (chk-arglist (macro-vars args) t ctx (w state) state)))

(defun defmacro-fn1 (name args doc doc-pair guard body w state)
  (let ((w (update-doc-data-base
            name doc doc-pair
            (putprop
             name 'macro-args args
             (putprop
              name 'macro-body body

; Below we store the guard. We currently store it in unnormalized form.
; If we ever store it in normalized form -- or in any form other than
; the translated user input -- then reconsider redundant-defmacrop
; below.

              (putprop-unless name 'guard guard *t* w))))))
    (value w)))

(defun chk-defmacro-width (rst ctx state)
  (cond ((or (not (true-listp rst))
             (not (> (length rst) 2)))
         (er soft ctx
             "Defmacro requires at least 3 arguments.  ~x0 is ~
              ill-formed.  See :DOC defmacro."
             (cons 'defmacro rst)))
        (t
         (let ((name (car rst))
               (args (cadr rst))
               (value (car (last rst)))
               (dcls-and-docs (butlast (cddr rst) 1)))
           (value (list name args dcls-and-docs value))))))

(defun redundant-defmacrop (name args guard body w)

; We determine whether there is already a defmacro of name with the
; given args, guard, and body.  We know that body is a term.  Hence,
; it is not nil.  Hence, if name is not a macro and there is no 
; 'macro-body, the first equal below will fail.

  (and (getprop name 'absolute-event-number nil 'current-acl2-world w)

; You might think the above test is redundant, given that we look for
; properties like 'macro-body below and find them.  But you would be wrong.
; Certain defmacros, in particular, those in *initial-event-defmacros* have
; 'macro-body and other properties but haven't really been defined yet!

       (equal (getprop name 'macro-body nil 'current-acl2-world w) body)
       (equal (macro-args name w) args)
       (equal (guard name nil w) guard)))

(defun defmacro-fn (mdef state event-form)

; Important Note:  Don't change the formals of this function without
; reading the *initial-event-defmacros* discussion in axioms.lisp.

  (with-ctx-summarized
   (if (output-in-infixp state) event-form (cons 'defmacro (car mdef)))
   (let ((wrld1 (w state))
         (event-form (or event-form (cons 'defmacro mdef))))
     (er-let*
      ((four (chk-defmacro-width mdef ctx state)))
      (let ((name (car four))
            (args (cadr four))
            (dcls (caddr four))
            (body (cadddr four)))
        (er-progn
         (chk-all-but-new-name name ctx 'macro wrld1 state)

; Important Note:  In chk-macro-arglist there is a comment warning us about
; the idea of "translating" the args to a macro to obtain the "internal"
; form of acceptable args.  See that comment before implementing any such
; change.

         (chk-macro-arglist args ctx state)
         (er-let*
          ((edcls (collect-declarations
                   dcls (macro-vars args)
                   'defmacro state ctx)))
          (let ((doc (if (stringp (car edcls)) (car edcls) nil))
                (edcls (if (stringp (car edcls)) (cdr edcls) edcls)))
            (er-let*
             ((tguard (translate
                       (conjoin (get-guards1 edcls wrld1))
                       '(nil) nil t ctx wrld1 state))

; known-stobjs = t, above and below.  But it doesn't matter because we
; know, from chk-macro-arglist above, that no stobjs occur in the
; formals of the macro and we check below, in
; chk-free-and-ignored-vars, that tguard and tbody use only those
; vars.

              (tbody (translate body '(nil) nil t ctx wrld1 state)))
             (cond
              ((redundant-defmacrop name args tguard tbody wrld1)
               (stop-redundant-event state))
              (t
               (er-let*
                ((wrld2 (chk-just-new-name name 'macro nil ctx wrld1 state))
                 (ignored (value (get-declared-ignored-variables edcls)))
                 (doc-pair (translate-doc name doc ctx state)))
                (er-progn
                 (chk-xargs-keywords1 edcls '(:guard) ctx state)
                 (chk-free-and-ignored-vars name (macro-vars args) tguard *0*
                                            ignored tbody ctx state)
                 (er-let*
                  ((wrld3 (defmacro-fn1 name args doc doc-pair
                            tguard tbody wrld2 state)))
                  (install-event name
                                 event-form
                                 'defmacro
                                 name
                                 nil
                                 (if (eq (small-p wrld1) t)
                                     nil
                                   (cons 'defmacro mdef))
                                 wrld3 state)))))))))))))))

; The following functions support boot-strapping.  Consider what
; happens when we begin to boot-strap.  The first form is read.
; Suppose it is (defconst nil 'nil).  It is translated wrt the
; initial world.  Unless 'defconst has a macro definition in that
; initial world, we won't get off the ground.  The same remark holds
; for the other primitive event functions encountered in axioms.lisp.
; Therefore, before we first call translate we have got to construct a
; world with certain properties already set.

; We compute those properties with the functions below, from the
; following constant.  This constant must be the quoted form of the
; event defmacros found in axioms.lisp!  It was obtained by
; going to the axioms.lisp buffer, grabbing all of the text in the
; "The *initial-event-defmacros* Discussion", moving it over here,
; embedding it in "(defconst *initial-event-defmacros* '(&))" and
; then deleting the #+acl2-loop-only commands, comments, and documentation
; strings.

(defconst *initial-event-defmacros*
  '((defmacro in-package (str)
      (list 'in-package-fn
            (list 'quote str)
            'state))
    (defmacro defpkg (&whole event-form name form &optional doc book-path)
      (list 'defpkg-fn
            (list 'quote name)
            (list 'quote form)
            'state
            (list 'quote doc)
            (list 'quote book-path)
            (list 'quote event-form)))
    (defmacro defchoose (&whole event-form &rest def)
      (list 'defchoose-fn
            (list 'quote def)
            'state
            (list 'quote event-form)))
    (defmacro defun (&whole event-form &rest def)
      (list 'defun-fn
            (list 'quote def)
            'state
            (list 'quote event-form)
            #+:non-standard-analysis ; std-p
            nil))
    (defmacro defuns (&whole event-form &rest def-lst)
      (list 'defuns-fn
            (list 'quote def-lst)
            'state
            (list 'quote event-form)
            #+:non-standard-analysis ; std-p
            nil))
    (defmacro verify-termination (&whole event-form &rest lst)
      (list 'verify-termination-fn
            (list 'quote lst)
            'state
            (list 'quote event-form)
            #+:non-standard-analysis ; std-p
            nil))
    (defmacro verify-guards (&whole event-form name
                                    &key hints otf-flg doc)
      (list 'verify-guards-fn
            (list 'quote name)
            'state
            (list 'quote hints)
            (list 'quote otf-flg)
            (list 'quote doc)
            (list 'quote event-form)))
    (defmacro defmacro (&whole event-form &rest mdef)
      (list 'defmacro-fn
            (list 'quote mdef)
            'state
            (list 'quote event-form)))
    (defmacro defconst (&whole event-form name form &optional doc)
      (list 'defconst-fn
            (list 'quote name)
            (list 'quote form)
            'state
            (list 'quote doc)
            (list 'quote event-form)))
    (defmacro defstobj (&whole event-form name &rest args)
      (list 'defstobj-fn
            (list 'quote name)
            (list 'quote args)
            'state
            (list 'quote event-form)))
    (defmacro defthm (&whole event-form
                      name term
                      &key (rule-classes '(:REWRITE))
                      instructions
                      hints
                      otf-flg
                      doc)
      (list 'defthm-fn
            (list 'quote name)
            (list 'quote term)
            'state
            (list 'quote rule-classes)
            (list 'quote instructions)
            (list 'quote hints)
            (list 'quote otf-flg)
            (list 'quote doc)
            (list 'quote event-form)
            #+:non-standard-analysis ; std-p
            nil))
    (defmacro defaxiom (&whole event-form
                        name term
                             &key (rule-classes '(:REWRITE))
                             doc)
      (list 'defaxiom-fn
            (list 'quote name)
            (list 'quote term)
            'state
            (list 'quote rule-classes)
            (list 'quote doc)
            (list 'quote event-form)))
    (defmacro deflabel (&whole event-form name &key doc)
      (list 'deflabel-fn
            (list 'quote name)
            'state
            (list 'quote doc)
            (list 'quote event-form)))
    (defmacro defdoc (&whole event-form name doc)
      (list 'defdoc-fn
            (list 'quote name)
            'state
            (list 'quote doc)
            (list 'quote event-form)))
    (defmacro deftheory (&whole event-form name expr &key doc)
      (list 'deftheory-fn
            (list 'quote name)
            (list 'quote expr)
            'state
            (list 'quote doc)
            (list 'quote event-form)))
    (defmacro in-theory (&whole event-form expr &key doc)
      (list 'in-theory-fn
            (list 'quote expr)
            'state
            (list 'quote doc)
            (list 'quote event-form)))
    (defmacro in-arithmetic-theory (&whole event-form expr &key doc)
      (list 'in-arithmetic-theory-fn
	    (list 'quote expr)
	    'state
	    (list 'quote doc)
	    (list 'quote event-form)))
    (defmacro push-untouchable (&whole event-form name &key doc)
      (list 'push-untouchable-fn
            (list 'quote name)
            'state
            (list 'quote doc)
            (list 'quote event-form)))
    (defmacro table (&whole event-form name &rest args)
      (list 'table-fn
            (list 'quote name)
            (list 'quote args)
            'state
            (list 'quote event-form)))
    (defmacro encapsulate (&whole event-form signatures &rest cmd-lst)
      (list 'encapsulate-fn
            (list 'quote signatures)
            (list 'quote cmd-lst)
            'state
            (list 'quote event-form)))
    (defmacro include-book (&whole event-form user-book-name
                                   &key
                                   (load-compiled-file ':warn)
                                   (uncertified-okp 't)
                                   (defaxioms-okp 't)
                                   (skip-proofs-okp 't)
                                   dir
                                   doc)
      (list 'include-book-fn
            (list 'quote user-book-name)
            'state
            (list 'quote load-compiled-file)
            (list 'quote uncertified-okp)
            (list 'quote defaxioms-okp)
            (list 'quote skip-proofs-okp)
            (list 'quote doc)
            (list 'quote dir)
            (list 'quote event-form)))
    (defmacro local (x)
      (list 'if
            '(equal (ld-skip-proofsp state) 'include-book)
            '(mv nil nil state)
            (list 'if 
                  '(equal (ld-skip-proofsp state) 'initialize-acl2)
                  '(mv nil nil state)
                  (list 'state-global-let*
                        '((in-local-flg t))
                        (list 'when-logic "LOCAL" x)))))
))

; Because of the Important Boot-Strapping Invariant noted in axioms.lisp,
; we can compute from this list the following things for each event:

; the macro name
; the macro args
; the macro body
; the -fn name corresponding to the macro
; the formals of the -fn

; The macro name and args are easy.  The macro body must be obtained
; from the list above by translating the given bodies, but we can't use
; translate yet because the world is empty and so, for example, 'list
; is not defined as a macro in it.  So we use the following boot-strap
; version of translate that is capable (just) of mapping the bodies above
; into their translations under a properly initialized world.

(defun boot-translate (x)
  (cond ((atom x)
         (cond ((eq x nil) *nil*)
               ((eq x t) *t*)
               ((keywordp x) (kwote x))
               ((symbolp x) x)
               (t (kwote x))))
        ((eq (car x) 'quote) x)
        ((eq (car x) 'if)
         (list 'if
               (boot-translate (cadr x))
               (boot-translate (caddr x))
               (boot-translate (cadddr x))))
        ((eq (car x) 'equal)
         (list 'equal
               (boot-translate (cadr x))
               (boot-translate (caddr x))))
        ((eq (car x) 'ld-skip-proofsp)
         (list 'ld-skip-proofsp
               (boot-translate (cadr x))))
        ((or (eq (car x) 'list)
             (eq (car x) 'mv))
         (cond ((null (cdr x)) *nil*)
               (t (list 'cons
                        (boot-translate (cadr x))
                        (boot-translate (cons 'list (cddr x)))))))
        ((eq (car x) 'when-logic)
         (list 'if
               '(eq (default-defun-mode-from-state state) ':program)
               (list 'skip-when-logic (list 'quote (cadr x)) 'state)
               (boot-translate (caddr x))))
        (t (er hard 'boot-translate
               "Boot-translate was called on ~x0, which is ~
                unrecognized.  If you want to use such a form in one ~
                of the *initial-event-defmacros* then you must modify ~
                boot-translate so that it can translate the form."
               x))))

; The -fn name corresponding to the macro is easy.  Finally to get the
; formals of the -fn we have to walk through the actuals of the call of
; the -fn in the macro body and unquote all the names but 'STATE.  That
; is done by:

(defun primordial-event-macro-and-fn1 (actuals)
  (cond ((null actuals) nil)
        ((equal (car actuals) '(quote state))
         (cons 'state (primordial-event-macro-and-fn1 (cdr actuals))))
        #+:non-standard-analysis
        ((or (equal (car actuals) nil)
             (equal (car actuals) t))

; Since nil and t are not valid names for formals, we need to transform (car
; actuals) to something else.  Up until the non-standard extension this never
; happened.  We henceforth assume that values of nil and t correspond to the
; formal std-p.

         (cons 'std-p (primordial-event-macro-and-fn1 (cdr actuals))))
        ((and (consp (car actuals))
              (eq (car (car actuals)) 'list)
              (equal (cadr (car actuals)) '(quote quote)))
         (cons (caddr (car actuals))
               (primordial-event-macro-and-fn1 (cdr actuals))))
        (t (er hard 'primordial-event-macro-and-fn1
               "We encountered an unrecognized form of actual, ~x0, ~
                in trying to extract the formals from the actuals in ~
                some member of *initial-event-defmacros*.  If you ~
                want to use such a form in one of the initial event ~
                defmacros, you must modify ~
                primordial-event-macro-and-fn1 so that it can recover ~
                the corresponding formal name from the actual form."
               (car actuals)))))

(defun primordial-event-macro-and-fn (form wrld)

; Given a member of *initial-event-defmacros* above, form, we check that
; it is of the desired shape, extract the fields we need as described,
; and putprop them into wrld.

  (case-match form
              (('defmacro 'local macro-args macro-body)
               (putprop
                'local 'macro-args macro-args
                (putprop
                 'local 'macro-body (boot-translate macro-body)
                 (putprop
                  'ld-skip-proofsp 'symbol-class :common-lisp-compliant
                  (putprop
                   'ld-skip-proofsp 'formals '(state)
                   (putprop
                    'ld-skip-proofsp 'stobjs-in '(state)
                    (putprop
                     'ld-skip-proofsp 'stobjs-out '(nil)

; See the fakery comment below for an explanation of this infinite
; recursion!  This specious body is only in effect during the
; processing of the first part of axioms.lisp during boot-strap.  It
; is overwritten by the accepted defun of ld-skip-proofsp.  Similarly
; for default-defun-mode-from-state and skip-when-logic.

                     (putprop
                      'ld-skip-proofsp 'body
                      '(ld-skip-proofsp state)
                      (putprop
                       'default-defun-mode-from-state 'symbol-class
                       :common-lisp-compliant
                       (putprop
                        'default-defun-mode-from-state 'formals '(state)
                        (putprop
                         'default-defun-mode-from-state 'stobjs-in '(state)
                         (putprop
                          'default-defun-mode-from-state 'stobjs-out '(nil)
                          (putprop
                           'default-defun-mode-from-state 'body
                           '(default-defun-mode-from-state state)
                           (putprop
                            'skip-when-logic 'symbol-class
                            :common-lisp-compliant
                            (putprop
                             'skip-when-logic 'formals '(str state)
                             (putprop
                              'skip-when-logic 'stobjs-in '(nil state)
                              (putprop
                               'skip-when-logic 'stobjs-out '(nil nil state)
                               (putprop
                                'skip-when-logic 'body
                                '(skip-when-logic str state)
                                wrld))))))))))))))))))
              (('defmacro name macro-args
                 ('list ('quote name-fn) . actuals))
               (let* ((formals (primordial-event-macro-and-fn1 actuals))
                      (stobjs-in (compute-stobj-flags formals t wrld))

; known-stobjs = t but, in this case it could just as well be
; known-stobjs = '(state) because we are constructing the primordial world
; and state is the only stobj.

                      (macro-body (boot-translate (list* 'list
                                                         (kwote name-fn)
                                                         actuals))))

; We could do a (putprop-unless name 'guard *t* *t* &) and a
; (putprop-unless name-fn 'guard *t* *t* &) here, but it would be silly.

                 (putprop
                  name 'macro-args macro-args
                  (putprop
                   name 'macro-body macro-body
                   (putprop
                    name-fn 'symbol-class :common-lisp-compliant
                    (putprop
                     name-fn 'formals formals
                     (putprop
                      name-fn 'stobjs-in stobjs-in
                      (putprop
                       name-fn 'stobjs-out '(nil nil state)

; The above may make sense, but the following act of fakery deserves
; some comment.  In order to get, e.g. defconst-fn, to work before
; it is defined in a boot-strap, we give it a body, which makes
; ev-fncall think it is ok to take a short cut and use the Common Lisp
; definition.  Of course, we are asking for trouble by laying down
; this recursive call!  But it never happens.

                       (putprop
                        name-fn 'body (cons name-fn formals)
                        wrld)))))))))
              (& (er hard 'primordial-event-macro-and-fn
                     "The supplied form ~x0 was not of the required ~
                      shape.  Every element of ~
                      *initial-event-defmacros* must be of the form ~
                      expected by this function.  Either change the ~
                      event defmacro or modify this function."
                     form))))

(defun primordial-event-macros-and-fns (lst wrld)

; This function is given *initial-event-defmacros* and just sweeps down it,
; putting the properties for each event macro and its corresponding -fn.

  (cond
   ((null lst) wrld)
   (t (primordial-event-macros-and-fns
       (cdr lst)
       (primordial-event-macro-and-fn (car lst) wrld)))))

; We need to declare the 'type-prescriptions for those fns that are
; referenced before they are defined in the boot-strapping process.
; Actually, apply is such a function, but it has an unrestricted type
; so we leave its 'type-prescriptions nil.

(defconst *initial-type-prescriptions*
  (list (list 'o-p
              (make type-prescription
                    :rune *fake-rune-for-anonymous-enabled-rule*
                    :nume nil
                    :term '(o-p x)
                    :hyps nil
                    :basic-ts *ts-boolean*
                    :vars nil
                    :corollary '(booleanp (o-p x))))
        (list 'o<
              (make type-prescription
                    :rune *fake-rune-for-anonymous-enabled-rule*
                    :nume nil
                    :term '(o< x y)
                    :hyps nil
                    :basic-ts *ts-boolean*
                    :vars nil
                    :corollary '(booleanp (o< x y))))))

(defun strip-caddrs (x)
  (declare (xargs :guard (all->=-len x 3)))
  (cond ((null x) nil)
        (t (cons (caddar x) (strip-caddrs (cdr x))))))

(defun collect-world-globals (wrld ans)
  (cond ((null wrld) ans)
        ((eq (cadar wrld) 'global-value)
         (collect-world-globals (cdr wrld)
                                (add-to-set-eq (caar wrld) ans)))
        (t (collect-world-globals (cdr wrld) ans))))

(defun primordial-world-globals (small-p distributed-books-dir
                                         operating-system)

; This function is the standard place to initialize a world global.
; Among the effects of this function is to set the global variable
; 'world-globals to the list of all variables initialized.  Thus,
; it is very helpful to follow the discipline of initializing all
; globals here, whether their initial values are important or not.

; Historical Note: Once upon a time, before we kept a stack of
; properties on the property lists representing installed worlds, it
; was necessary, when retracting from a world, to scan the newly
; exposed world to find the new current value of any property removed.
; This included the values of world globals and it often sent us all
; the way back to the beginning of the primordial world.  We then
; patched things up by using this collection of names at the end of
; system initialization to "float" to the then-top of the world the
; values of all world globals.  That was the true motivation of
; collecting the initialization of all globals into one function: so
; we could get 'world-globals so we knew who to float.

  (let ((wrld
         (global-set-lst
          (list*
           (list 'event-landmark (make-event-tuple -1 0 nil nil 0 nil nil))
           (list 'command-landmark (make-command-tuple -1 :logic nil nil))
           (list 'known-package-alist *initial-known-package-alist*)
           (list 'well-founded-relation-alist
                 (list (cons 'o<
                             (cons 'o-p
                                   *fake-rune-for-anonymous-enabled-rule*))))
           (list 'recognizer-alist *initial-recognizer-alist*)
           (list 'built-in-clauses
                 (classify-and-store-built-in-clause-rules
                  *initial-built-in-clauses*
                  nil
; The value of wrld supplied below, nil, just means that all function symbols
; of initial-built-in-clauses will seem to have level-no 0.
                  nil))
           (list 'half-length-built-in-clauses
                 (floor (length *initial-built-in-clauses*) 2))
           (list 'type-set-inverter-rules *initial-type-set-inverter-rules*)
           (list 'global-enabled-structure
                 (initial-global-enabled-structure "ENABLED-ARRAY-"))
           (list 'global-arithmetic-enabled-structure
                 (initial-global-enabled-structure
                  "ARITHMETIC-ENABLED-ARRAY-"))
           (let ((globals
                  '((event-index nil)
                    (command-index nil)
                    (event-number-baseline 0)
                    (command-number-baseline 0)
                    (embedded-event-lst nil)
                    (cltl-command nil)
                    (include-book-alist nil)
                    (include-book-path nil)
                    (certification-tuple nil)
                    (documentation-alist nil)
                    (proved-functional-instances-alist nil)
                    (nonconstructive-axiom-names nil)
                    (reversed-standard-theories (nil nil nil nil))
                    (current-theory nil)
                    (generalize-rules nil)
                    (boot-strap-flg t)
                    (boot-strap-pass-2 nil)
                    (skip-proofs-seen nil)
                    (free-var-runes-all nil)
                    (free-var-runes-once nil)
                    (chk-new-name-lst
                     (if iff implies not
                         in-package
                         defpkg defun defuns mutual-recursion defmacro defconst
                         defstobj defthm defaxiom encapsulate include-book 
                         deflabel defdoc deftheory
                         in-theory in-arithmetic-theory
                         push-untouchable table verify-guards
                         verify-termination
                         local defchoose ld-skip-proofsp
                         in-package-fn defpkg-fn defun-fn defuns-fn
                         mutual-recursion-fn defmacro-fn defconst-fn
                         defstobj-fn
                         defthm-fn defaxiom-fn encapsulate-fn include-book-fn
                         deflabel-fn defdoc-fn
                         deftheory-fn in-theory-fn in-arithmetic-theory-fn
                         push-untouchable-fn table-fn
                         verify-guards-fn verify-termination-fn defchoose-fn
                         apply o-p o< default-defun-mode-from-state
                         skip-when-logic

; The following names are here simply so we can deflabel them for
; documentation purposes:

                         state
                         declare apropos
                         enter-boot-strap-mode exit-boot-strap-mode
                         lp acl2-defaults-table let let*
                         complex complex-rationalp

                         ))
                    (untouchables nil))))
             (list* `(small-p ,small-p)
                    `(distributed-books-dir ,distributed-books-dir)
                    `(operating-system ,operating-system)
                    globals)))
          nil)))
    (global-set 'world-globals
                (collect-world-globals wrld '(world-globals))
                wrld)))

;; RAG - I added the treatment of *non-standard-primitives*

(defun primordial-world (small-p distributed-books-dir operating-system)
  (let ((names (strip-cars *primitive-formals-and-guards*))
        (arglists (strip-cadrs *primitive-formals-and-guards*))
        (guards (strip-caddrs *primitive-formals-and-guards*))
        (ns-names #+:non-standard-analysis *non-standard-primitives*
                  #-:non-standard-analysis nil))

    (add-command-landmark
     :logic
     (list 'enter-boot-strap-mode
           small-p distributed-books-dir operating-system)
     (add-event-landmark
      (list 'enter-boot-strap-mode
            small-p distributed-books-dir operating-system)
      'enter-boot-strap-mode
      (append (strip-cars *primitive-formals-and-guards*)
              (strip-non-hidden-package-names *initial-known-package-alist*))
      small-p
      (putprop
       'equal
       'coarsenings
       '(equal)
       (putprop-x-lst1
        names 'absolute-event-number 0
        (putprop-defun-runic-mapping-pairs
         names nil
         (putprop-x-lst1
          ns-names
          'classicalp nil
          (putprop-x-lst1
           ns-names
           'constrainedp t
           (putprop-x-lst1
            names
            'symbol-class :common-lisp-compliant
            (putprop-x-lst2-unless
             names 'guard guards *t*
             (putprop-x-lst2
              names 'formals arglists
              (putprop-x-lst2
               (strip-cars *initial-type-prescriptions*)
               'type-prescriptions
               (strip-cdrs *initial-type-prescriptions*)
               (putprop-x-lst1
                names 'coarsenings nil
                (putprop-x-lst1
                 names 'congruences nil
                 (primordial-event-macros-and-fns
                  *initial-event-defmacros*

; This putprop must be here, into the world seen by
; primordial-event-macros-and-fns!

                  (putprop
                   'state 'stobj '(*the-live-state*)
                   (primordial-world-globals
                    small-p
                    distributed-books-dir
                    operating-system))))))))))))))))))

(defun same-name-twice (l)
  (cond ((null l) nil)
        ((null (cdr l)) nil)
        ((equal (symbol-name (car l))
                (symbol-name (cadr l)))
         (list (car l) (cadr l)))
        (t (same-name-twice (cdr l)))))

(defun conflicting-imports (l)
  (same-name-twice (merge-sort-symbol-< l)))

(defun chk-new-stringp-name (ev-type name ctx w state)
  (cond
   ((not (stringp name))
    (er soft ctx
        "The first argument to ~s0 must be a string.  You provided ~
         the object ~x1.  See :DOC ~s."
        (cond
         ((eq ev-type 'defpkg) "defpkg")
         (t "include-book"))
        name))
   (t (let ((entry
             (find-package-entry name (global-val 'known-package-alist w))))
        (cond
         ((and entry
               (not (and (eq ev-type 'defpkg)
                         (package-entry-hidden-p entry))))
          (er soft ctx
              "The name ~x0 is in use as a package name.  We do not permit ~
               package names~s1 to participate in redefinition.  If you must ~
               redefine this name, use :ubt to undo the existing definition."
              name
              (if (package-entry-hidden-p entry)
                  " (even those that are hidden; see :DOC hidden-death-package"
                "")))
         ((assoc-equal name (global-val 'include-book-alist w))

; Name is thus a full-book-name.

          (cond
           ((eq ev-type 'include-book)
            (value name))
           (t (er soft ctx
                  "The name ~x0 is in use as a book name.  You are trying to ~
                   redefine it as a package.  We do not permit package names ~
                   to participate in redefinition.  If you must redefine this ~
                   name, use :ubt to undo the existing definition."
                  name))))
         (t (value nil)))))))

(deflabel package-reincarnation-import-restrictions
  :doc
  ":Doc-Section Miscellaneous

   re-defining undone ~ilc[defpkg]s~/

   Suppose ~c[(defpkg \"pkg\" imports)] is the most recently executed
   successful definition of ~c[\"pkg\"] in this ACL2 session and that it
   has since been undone, as by ~c[:]~ilc[ubt].  Any future attempt in this
   session to define ~c[\"pkg\"] as a package must specify an identical
   imports list.~/

   The restriction stems from the need to implement the reinstallation
   of saved logical ~il[world]s as in error recovery and the ~c[:]~ilc[oops] ~il[command].
   Suppose that the new ~ilc[defpkg] attempts to import some symbol, ~c[a::sym],
   not imported by the previous definition of ~c[\"pkg\"].  Because it was
   not imported in the original package, the symbol ~c[pkg::sym], different
   from ~c[a::sym], may well have been created and may well be used in some
   saved ~il[world]s.  Those saved ~il[world]s are Common Lisp objects being held
   for you ``behind the scenes.''  In order to import ~c[a::sym] into
   ~c[\"pkg\"] now we would have to unintern ~c[pkg::sym], rendering those
   saved ~il[world]s ill-formed.  It is because of saved ~il[world]s that we do
   not actually clear out a package when it is undone.

   At one point we thought it was sound to allow the new ~ilc[defpkg] to
   import a subset of the old.  But that is incorrect.  Suppose the old
   definition of ~c[\"pkg\"] imported ~c[a::sym] but the new one does not.
   Suppose we allowed that and implemented it simply by setting the
   imports of ~c[\"pkg\"] to the new subset.  Then consider the conjecture
   ~c[(eq a::sym pkg::sym)].  This ought not be a theorem because we did
   not import ~c[a::sym] into ~c[\"pkg\"].  But in fact in AKCL it is a theorem
   because ~c[pkg::sym] is read as ~c[a::sym] because of the old imports."

#| Once upon a time the documentation included the following additional
   text.  We deleted it on the grounds that we shouldn't tell the user
   how to go behind our backs.  But we might want to recall this hack
   in the future for our own use.

   If you really must change the imports list of a previously defined
   but now undone package, we recommend that you either invent a new
   package name for subsequent use in this session or that you save
   your state and reconstruct it in a new ACL2 session.  If you wish to
   try behind the scenes surgery to allow the new ~ilc[defpkg] to succeed ~-[]
   at the expense of ACL2's soundness in the rest of the session ~-[]
   exit ~ilc[lp] and type the following to raw Lisp:
   ~bv[]
   (let ((name \"pkg\")           ; fill in pkg name
         (new-imports '(...))     ; fill in new imports
         (p (find-package name)))
     (do-symbols (sym p) (unintern sym p))
     (import new-imports p)
     (setq *ever-known-package-alist*
           (cons (make-package-entry :name name :imports new-imports)
                 (remove-package-entry name *ever-known-package-alist*)))
     name)
   ~ev[]
   This will render ill-formed any saved ~il[world]s involving symbols in
   ~c[\"pkg\"] and it may be impossible to recover from certain errors.  In
   addition, because ACL2 is probably unsound after this hack we
   recommend that you treat the rest of the session as merely
   exploratory.
|#
)

(defun chk-package-reincarnation-import-restrictions (name proposed-imports)

; Logically, this function always returns t, but it may cause a hard
; error because we cannot create a package with the given name and imports.
; See :DOC package-reincarnation-import-restrictions.

  #+acl2-loop-only
  (declare (ignore name proposed-imports))
  #-acl2-loop-only
  (chk-package-reincarnation-import-restrictions2 name proposed-imports)
  t)

(defun convert-book-name-to-cert-name (x)

; X is assumed to satisfy chk-book-name.  We generate the
; corresponding certification file name.

; The cddddr below chops of the "lisp" but leaves the dot.

  (coerce (append (reverse (cddddr (reverse (coerce x 'list))))
                  '(#\c #\e #\r #\t))
          'string))

(defun tilde-@-defpkg-error-phrase (name package-entry new-not-old old-not-new
                                         book-path defpkg-book-path w)
  (list
   "The proposed defpkg conflicts with a previously-executed defpkg for name ~
    ~x0~@1.  ~#a~[For example, symbol ~s2::~s3 is in the list of imported ~
    symbols for the ~s4 definition but not for the other.~/The two have the ~
    same lists of imported symbols, but not in the same order.~]  The previous ~
    defpkg is ~#5~[at the top level.~/in the .cert file ~x6 for the book ~x7, ~
    which is included at the top level~/in the .cert file ~x6 for the book ~
    ~x7, which is included via the following path, from top-most book down to ~
    the above file.~|  ~F8~]~@9~@b"
   (cons #\0 name)
   (cons #\1 (if (package-entry-hidden-p package-entry)
                 " that no longer exists in the current ACL2 logical world ~
                  (see :DOC hidden-death-package)"
               ""))
   (cons #\a (if (or new-not-old old-not-new) 0 1))
   (cons #\2 (symbol-package-name (if new-not-old
                                      (car new-not-old)
                                    (car old-not-new))))
   (cons #\3 (symbol-name (if new-not-old
                              (car new-not-old)
                            (car old-not-new))))
   (cons #\4 (if new-not-old "current" "previous"))
   (cons #\5 (zero-one-or-more book-path))
   (cons #\6 (convert-book-name-to-cert-name (car book-path)))
   (cons #\7 (car book-path))
   (cons #\8 (reverse book-path))
   (cons #\9 (if defpkg-book-path
                 "~|This previous defpkg event appears to have been created ~
                  because of a defpkg that was hidden by a local include-book; ~
                  see :DOC hidden-death-package."
               ""))
   (cons #\b (let ((include-book-path
                    (global-val 'include-book-path w)))
               (if (or include-book-path
                       defpkg-book-path)
                   (msg "~|The new proposed defpkg event may be found by ~
                         following the sequence of include-books below, from ~
                         top-most book down to the book whose portcullis ~
                         contains the following defpkg form.~|  ~F0"
                        (reverse (append defpkg-book-path include-book-path)))
                 "")))))

(defun chk-acceptable-defpkg (name form defpkg-book-path ctx w state)

; We return an error triple.  The non-error value is either 'redundant or a
; triple (tform value . package-entry), where tform and value are a translated
; form and its value, and either package-entry is nil in the case that no
; package with name name has been seen, or else is an existing entry for name
; in known-package-alist with field hidden-p=t (see the Essay on Hidden
; Packages).

  (mv-let
    (all-names state)
    (list-all-package-names state)
    (cond
     ((not (stringp name))
      (er soft ctx
          "Package names must be string constants and ~x0 is not.  See :DOC ~
          defpkg."
          name))
     ((equal name "")

; In Allegro CL, "" is prohibited because it is already a nickname for the
; KEYWORD package.  But in GCL we could prove nil up through v2-7 by certifying
; the following book with the indicated portcullis:

; (in-package "ACL2")
;
; Portcullis:
; (defpkg "" nil)
;
; (defthm bug
;   nil
;   :hints (("Goal" :use ((:instance intern-in-package-of-symbol-symbol-name
;                                    (x '::abc) (y 17)))))
;   :rule-classes nil)

      (er soft ctx
          "The empty string is not a legal package name for defpkg."
          name))
     ((and (member-equal name all-names)
           (not (global-val 'boot-strap-flg w))
           (not (member-equal name

; It seems very likely that the following value contains the same names as
; (strip-cars (known-package-alist state)), except for built-in packages, now
; that we keep hidden packages around.  But it seems harmless enough to keep
; this state global around.

                              (f-get-global 'packages-created-by-defpkg
                                            state))))
      (er soft ctx
          "The package named ~x0 already exists and was not created by ACL2's ~
           defpkg.  We cannot (re)define an existing package.  See :DOC defpkg."
          name))
     ((not (true-listp defpkg-book-path))
      (er soft ctx
          "The book-path argument to defpkg, if supplied, must be a ~
           true-listp.  It is not recommended to supply this argument, since ~
           the system makes use of it for producing useful error messages.  ~
           The defpkg of ~x0 is thus illegal."
          name))
     (t (er-let*
         ((pair (simple-translate-and-eval form nil nil
                                           "The second argument to defpkg"
                                           ctx w state)))
         (let ((tform (car pair))
               (imports (cdr pair)))
           (cond
            ((not (symbol-listp imports))
             (er soft ctx
                 "The second argument of defpkg must eval to a list of ~
                  symbols.  See :DOC defpkg."))
            ((member-symbol-name *pkg-witness-name* imports)
             (er soft ctx
                 "It is illegal to import symbol ~x0 because its name has been ~
                  reserved to refer to a symbol whose package-name in the new ~
                  package."
                 (car (member-symbol-name *pkg-witness-name* imports))))
            (t (let ((conflict (conflicting-imports imports))
                     (base-symbol (packn (cons name '("-PACKAGE")))))

; Base-symbol is the the base symbol of the rune for the rule added by
; defpkg describing the properties of symbol-package-name on interns
; with the new package.

                 (cond
                  (conflict
                   (er soft ctx
                       "The value of the second (imports) argument of defpkg ~
                        may not contain two symbols with the same symbol name, ~
                        e.g. ~&0.  See :DOC defpkg."
                       conflict))
                  (t (let ((package-entry
                            (and (not (global-val 'boot-strap-flg w))
                                 (find-package-entry
                                  name
                                  (global-val 'known-package-alist w)))))
                       (cond
                        ((and package-entry
                              (not (equal imports
                                          (package-entry-imports
                                           package-entry))))
                         (let* ((new-not-old
                                 (set-difference-eq
                                  imports
                                  (package-entry-imports package-entry)))
                                (old-not-new
                                 (and (null new-not-old) ; optimization
                                      (set-difference-eq
                                       (package-entry-imports package-entry)
                                       imports)))
                                (book-path
                                 (package-entry-book-path package-entry)))
                           (er soft ctx
                               "~@0"
                               (tilde-@-defpkg-error-phrase
                                name package-entry new-not-old old-not-new
                                book-path defpkg-book-path w))))
                        ((and package-entry
                              (not (package-entry-hidden-p package-entry)))
                         (prog2$ (chk-package-reincarnation-import-restrictions
                                  name imports)
                                 (value 'redundant)))
                        (t (er-progn
                            (chk-new-stringp-name 'defpkg name ctx w state)
                            (chk-all-but-new-name base-symbol ctx nil w state)

; Note:  Chk-just-new-name below returns a world which we ignore because
; we know redefinition of 'package base-symbols is disallowed, so the
; world returned is w when an error isn't caused.

; Warning: In maybe-push-undo-stack and maybe-pop-undo-stack we rely
; on the fact that the symbol name-PACKAGE is new!

                            (chk-just-new-name base-symbol
                                               'package nil ctx w state)
                            (prog2$
                             (chk-package-reincarnation-import-restrictions
                              name imports)
                             (value (list* tform
                                           imports
                                           package-entry ; hidden-p is true
                                           ))))))))))))))))))

(defun defpkg-fn (name form state doc book-path event-form)

; Important Note:  Don't change the formals of this function without
; reading the *initial-event-defmacros* discussion in axioms.lisp.

; Like defconst, defpkg evals its second argument.

; We forbid interning into a package before its imports are set once and for
; all.  In the case of the main Lisp package, we assume that we have no control
; over it and simply refuse requests to intern into it.

  (with-ctx-summarized
   (if (output-in-infixp state) event-form (cons 'defpkg name))
   (let ((w (w state))
         (event-form (or event-form
                         (list* 'defpkg name form
                                (if (or doc book-path) (list doc) nil)
                                (if book-path (list book-path) nil)))))
     (er-let* ((doc-pair (translate-doc name doc ctx state))
               (tform-imports-entry
                (chk-acceptable-defpkg name form book-path ctx w state)))
              (cond
               ((eq tform-imports-entry 'redundant)
                (stop-redundant-event state))
               (t
                (let* ((imports (cadr tform-imports-entry))
                       (w1 (global-set
                            'known-package-alist
                            (cons (make-package-entry
                                   :name name
                                   :imports imports
                                   :hidden-p nil
                                   :book-path
                                   (append book-path
                                           (global-val
                                            'include-book-path
                                            w))
                                   :defpkg-event-info
                                   (cons event-form (car tform-imports-entry)))
                                  (if (cddr tform-imports-entry)
                                      (remove-package-entry
                                       name
                                       (known-package-alist state))
                                    (global-val 'known-package-alist w)))
                            w))

; Defpkg adds an axiom, labelled ax below.  We make a :REWRITE rule out of ax.
; Warning: If the axiom added by defpkg changes, be sure to consider the
; initial packages that are not defined with defpkg, e.g., "ACL2".  In
; particular, for each primitive package in *initial-known-package-alist*
; (except for the "COMMON-LISP" package) there is a defaxiom in axioms.lisp
; exactly analogous to the add-rule below.  So if you change this code, change
; that code.

                       (ax
                        `(implies
                          ,(if (null imports)
                               `(if (stringp x)
                                    (if (symbolp y)
                                        (equal (symbol-package-name y)
                                               (quote ,name))
                                      ,*nil*)
                                  ,*nil*)
                             `(if (stringp x)
                                  (if (not (member-symbol-name
                                            x (quote ,imports)))
                                      (if (symbolp y)
                                          (equal (symbol-package-name y)
                                                 (quote ,name))
                                        ,*nil*)
                                    ,*nil*)
                                ,*nil*))
                          (equal (symbol-package-name
                                  (intern-in-package-of-symbol x y))
                                 (quote ,name))))
                       (w2
                        (add-rules
                         (packn (cons name '("-PACKAGE")))
                         `((:REWRITE :COROLLARY ,ax))
                         ax ax w1 state))
                       (w3 (update-doc-data-base name doc doc-pair w2)))
                  (pprogn
                   (f-put-global 'packages-created-by-defpkg
                                 (cons name
                                       (f-get-global 'packages-created-by-defpkg
                                                     state))
                                 state)
                   (install-event name
                                  event-form
                                  'defpkg
                                  name
                                  nil
                                  (list 'defpkg name form)
                                  w3 state)))))))))

; We now start the development of deftheory and theory expressions.

; First, please read the Essay on Enabling, Enabled Structures, and
; Theories for a refresher course on such things as runes, common
; theories, and runic theories.  Roughly speaking, theory expressions
; are terms that produce common theories as their results.  Recall
; that a common theory is a truelist of rule name designators.  A rule
; name designator is an object standing for a set of runes; examples
; include APP, which might stand for {(:DEFINITION app)}, (APP), which
; might stand for {(:EXECUTABLE-COUNTERPART app)}, and LEMMA1, which
; might stand for the set of runes {(REWRITE lemma1 . 1) (REWRITE
; lemma1 . 2) (ELIM lemma1)}.  Of course, a rune is a rule name designator
; and stands for the obvious: the singleton set containing that rune.

; To every common theory there corresponds a runic theory, obtained
; from the common theory by unioning together the designated sets of
; runes and then ordering the result by nume.  Runic theories are
; easier to manipulate (e.g., union together) because they are
; ordered.

; To define deftheory we need not define any any "theory manipulation
; functions" (e.g., union-theories, or universal-theory) because
; deftheory just does a full-blown eval of whatever expression the
; user provides.  We could therefore define deftheory now.  But there
; are a lot of useful theory manipulation functions and they are
; generally used only in deftheory and in-theory, so we define them
; now.

; Calls of these functions will be typed by the user in theory
; expressions.  Those expressions will be executed to obtain new
; theories.  Furthermore, the user may well define his own theory
; producing functions which will be mixed in with ours in his
; expressions.  How do we know a "theory expression" will produce a
; theory?  We don't.  We just evaluate it and check the result.  But
; this raises a more serious question: how do we know our theory
; manipulation functions are given theories as their arguments?
; Indeed, they may not be given theories because of misspellings, bugs
; in the user's functions, etc.  Because of the presence of
; user-defined functions in theory expressions we can't syntactically
; check that an expression is ok.  And at the moment we don't see that
; it is worth the trouble of making the user prove "theory theorems"
; such as (THEORYP A W) -> (THEORYP (MY-FN A) W) that would let us so
; analyze his expressions.

; So we have decided to put run-time checks into our theory functions.
; We have two methods available to us: we could put guards on them or
; we could put checks into them.  The latter course does not permit us
; to abort on undesired arguments -- because we don't want theory
; functions to take STATE and be multi-valued.  Thus, once past the
; guards all we can do is coerce unwanted args into acceptable ones.

; There are several sources of tension.  It was such tensions that
; led to the idea of "common" v. "runic" theories and, one level deeper,
; "rule name designators" v. runes.

; (1) When our theory functions are getting input directly from the
;     user we wish they did a throrough job of checking it and were
;     forgiving about such things as order, e.g., sorted otherwise ok
;     lists, so that the user didn't need to worry about order.

; (2) When our theory functions are getting input produced by one of
;     our functions, we wish they didn't check anything so they could
;     just fly.

; (3) These functions have to be admissible under the definitional principle
;     and not cause errors when called on the utter garbage that the user
;     might type.

; (4) Checking the well-formedness of a theory value requires access to
;     wrld.

; We have therefore chosen the following strategy.

; First, all theory manipulation functions take wrld as an argument.
; Some need it, e.g., the function that returns all the available rule
; names.  Others wouldn't need it if we made certain choices on the
; handling of run-time checks.  We've chosen to be uniform: all have
; it.  This uniformity saves the user from having to remember which
; functions do and which don't.

; Second, all theory functions have guards that check that their
; "theory" arguments "common theories."  This means that if a theory
; function is called on utter garbage the user will get an error
; message.  But it means we'll pay the price of scanning each theory
; value on each function entry in his expression to check
; rule-name-designatorp.

; To compute on theories we will convert common theories to runic ones
; (actually, all the way to augmented runic theories) and we will
; always return runic theories because they can be verified faster.
; This causes a second scan every time but in general will not go into
; sorting because our intermediate results will always be ordered.
; This gives us "user-friendliness" for top-level calls of the theory
; functions without (too much?)  overhead.

; Now we define union, intersect, and set-difference for lists of rule
; names.

(defun intersection-augmented-theories-fn1 (lst1 lst2 ans)

; Let lst1 and lst2 be augmented theories: ascendingly ordered lists
; of pairs mapping numes to runes.  We return the intersection of the
; two theories -- as a runic theory, not as an augmented runic theory.
; That is, we strip off the numes as we go.  This is unesthetic: it
; would be more symmetric to produce an augmented theory since we take
; in augmented theories.  But this is more efficient because we don't
; have to copy the result later to strip off the numes.

  (cond
   ((null lst1) (revappend ans nil))
   ((null lst2) (revappend ans nil))
   ((= (car (car lst1)) (car (car lst2)))
    (intersection-augmented-theories-fn1 (cdr lst1) (cdr lst2)
                                         (cons (cdr (car lst1)) ans)))
   ((< (car (car lst1)) (car (car lst2)))
    (intersection-augmented-theories-fn1 (cdr lst1) lst2 ans))
   (t (intersection-augmented-theories-fn1 lst1 (cdr lst2) ans))))

(defun intersection-theories-fn (lst1 lst2 wrld)
  (declare (xargs :guard (and (theoryp! lst1 wrld)
                              (theoryp! lst2 wrld))))
  (intersection-augmented-theories-fn1 (augment-theory lst1 wrld)
                                       (augment-theory lst2 wrld)
                                       nil))

(defmacro intersection-theories (lst1 lst2)

  #-small-acl2-image
  ":Doc-Section Theories

  intersect two ~il[theories]~/
  ~bv[]
  Example:
  (intersection-theories (current-theory :here)
                         (theory 'arith-patch))~/

  General Form:
  (intersection-theories th1 th2)
  ~ev[]
  where ~c[th1] and ~c[th2] are theories (~pl[theories]).  To each of
  the arguments there corresponds a runic theory.  This function
  returns the intersection of those two runic ~il[theories], represented as
  a list and ordered chronologically.

  This ``function'' is actually a macro that expands to a term
  mentioning the single free variable ~ilc[world].  When theory expressions
  are evaluated by ~ilc[in-theory] or the ~c[:]~ilc[in-theory] hint, ~ilc[world] is bound to
  the current ACL2 ~il[world].~/

  :cited-by theory-functions"

  (list 'intersection-theories-fn
        lst1
        lst2
        'world))

(defun union-augmented-theories-fn1 (lst1 lst2 ans)

; Let lst1 and lst2 be augmented theories: ascendingly ordered lists
; of pairs mapping numes to runes.  We return their union as an
; unagumented runic theory.  See intersection-augmented-theories-fn1.

  (cond ((null lst1) (revappend ans (strip-cdrs lst2)))
        ((null lst2) (revappend ans (strip-cdrs lst1)))
        ((= (car (car lst1)) (car (car lst2)))
         (union-augmented-theories-fn1 (cdr lst1) (cdr lst2) (cons (cdr (car lst1)) ans)))
        ((< (car (car lst1)) (car (car lst2)))
         (union-augmented-theories-fn1 (cdr lst1) lst2 (cons (cdr (car lst1)) ans)))
        (t (union-augmented-theories-fn1 lst1 (cdr lst2) (cons (cdr (car lst2)) ans)))))

(defun union-theories-fn (lst1 lst2 wrld)
  (declare (xargs :guard (and (theoryp! lst1 wrld)
                              (theoryp! lst2 wrld))))
  (union-augmented-theories-fn1 (augment-theory lst1 wrld)
                                (augment-theory lst2 wrld)
                                nil))

(defmacro union-theories (lst1 lst2)

  #-small-acl2-image
  ":Doc-Section Theories

  union two ~il[theories]~/
  ~bv[]
  Example:
  (union-theories (current-theory 'lemma3)
                  (theory 'arith-patch))~/

  General Form:
  (union-theories th1 th2)
  ~ev[]
  where ~c[th1] and ~c[th2] are theories (~pl[theories]).  To each of
  the arguments there corresponds a runic theory.  This function
  returns the union of those two runic ~il[theories], represented as a list
  and ordered chronologically.

  This ``function'' is actually a macro that expands to a term
  mentioning the single free variable ~ilc[world].  When theory expressions
  are evaluated by ~ilc[in-theory] or the ~c[:]~ilc[in-theory] hint, ~ilc[world] is bound to
  the current ACL2 ~il[world].~/

  :cited-by theory-functions"

  (list 'union-theories-fn
        lst1
        lst2
        'world))

(defun rev-strip-cdrs (x y)
  (if (null x)
      y
    (rev-strip-cdrs (cdr x) (cons (cdar x) y))))

(defun set-difference-augmented-theories-fn1 (lst1 lst2 ans)

; Let lst1 and lst2 be augmented theories: ascendingly ordered lists
; of pairs mapping numes to runes.  We return their set-difference as
; an unagumented runic theory.  See intersection-augmented-theories-fn1.

  (cond ((null lst1) (revappend ans nil))
        ((null lst2) (reverse (rev-strip-cdrs lst1 ans)))
        ((= (car (car lst1)) (car (car lst2)))
         (set-difference-augmented-theories-fn1 (cdr lst1) (cdr lst2) ans))
        ((< (car (car lst1)) (car (car lst2)))
         (set-difference-augmented-theories-fn1 (cdr lst1) lst2 (cons (cdr (car lst1)) ans)))
        (t (set-difference-augmented-theories-fn1 lst1 (cdr lst2) ans))))

(defun set-difference-theories-fn (lst1 lst2 wrld)
  (declare (xargs :guard (let ((x (theoryp! lst1 wrld))
                               (y (theoryp! lst2 wrld)))
                           (and x y))))
  (set-difference-augmented-theories-fn1 (augment-theory lst1 wrld)
                                         (augment-theory lst2 wrld)
                                         nil))

(defmacro set-difference-theories (lst1 lst2)

  #-small-acl2-image
  ":Doc-Section Theories

  difference of two ~il[theories]~/
  ~bv[]
  Example:
  (set-difference-theories (current-theory :here)
                           '(fact (fact)))~/

  General Form:
  (set-difference-theories th1 th2)
  ~ev[]
  where ~c[th1] and ~c[th2] are ~il[theories] (~pl[theories]).  To each of
  the arguments there corresponds a runic theory.  This function
  returns the set-difference of those two runic ~il[theories], represented
  as a list and ordered chronologically.  That is, a ~il[rune] is in the
  result iff it is in the first runic theory but not in the second.

  The standard way to ``disable'' a theory, ~c[lst], is:
  ~c[(in-theory (set-difference-theories (current-theory :here) lst))].

  This ``function'' is actually a macro that expands to a term
  mentioning the single free variable ~ilc[world].  When theory expressions
  are evaluated by ~ilc[in-theory] or the ~c[:]~ilc[in-theory] hint, ~ilc[world] is bound to
  the current ACL2 ~il[world].~/

  :cited-by theory-functions"

  (list 'set-difference-theories-fn
        lst1
        lst2
        'world))

; Now we define a few useful theories.

(defun append-strip-cdrs (x y)

; This is (append (strip-cdrs x) y).

  (cond ((null x) y)
        (t (cons (cdr (car x)) (append-strip-cdrs (cdr x) y)))))

; In computing the useful theories we will make use of previously stored values
; of those theories.  However, those stored values might contain "runes" that
; are no longer runes because of redefinition.  The following function is used
; to delete from those non-runes, based on the redefined base symbols.

(defun revappend-delete-runes-based-on-symbols (runes symbols ans)

; We delete from runes all those with base-symbols listed in symbols
; and accumulate them in reverse order onto ans.

  (cond ((null runes) ans)
        ((member-eq (base-symbol (car runes)) symbols)
         (revappend-delete-runes-based-on-symbols (cdr runes) symbols ans))
        (t (revappend-delete-runes-based-on-symbols (cdr runes)
                                                    symbols
                                                    (cons (car runes) ans)))))

(defun universal-theory-fn1 (lst ans redefined)

; Lst is a cdr of the current world.  We scan down lst accumulating
; onto ans every rune in every 'runic-mapping-pairs property.  Our
; final ans is ascendingly ordered.  This works because the world is
; ordered reverse-chronologically, so the runes in the first
; 'runic-mapping-pairs we see will have the highest numes.

; If at any point we encounter the 'global-value for the variable
; 'reversed-standard-theories then we assume the value is of the form
; (r-unv r-fn1 r-fn2 r-fn3), where r-unv is the reversed universal
; theory as of that world, r-fn1 is the reversed function symbol
; theory, r-fn2 is the reversed executable counterpart theory, and
; r-fn3 is the reversed function theory.  If we find such a binding we
; stop and revappend r-unv to our answer and quit.  By this hack we
; permit the precomputation of a big theory and save having to scan
; down world -- which really means save having to swap world into
; memory.

; At the end of the bootstrap we will save the standard theories just
; to prevent the swapping in of prehistoric conses.

; Note: :REDEF complicates matters.  If a name is redefined the runes based on
; its old definition are invalid.  We can tell that sym has been redefined when
; we encounter on lst a triple of the form (sym RUNIC-MAPPING-PAIRS
; . :ACL2-PROPERTY-UNBOUND).  This means that all runes based on sym
; encountered subsequently must be ignored or deleted (ignored when encountered
; as RUNIC-MAPPING-PAIRS and deleted when seen in the stored standard theories.
; The list redefined contains all such syms encountered.

  (cond ((null lst)
         #+acl2-metering (meter-maid 'universal-theory-fn1 500)
         ans)
        ((eq (cadr (car lst)) 'runic-mapping-pairs)
         #+acl2-metering (setq meter-maid-cnt (1+ meter-maid-cnt))
         (cond
          ((eq (cddr (car lst)) *acl2-property-unbound*)
           (universal-theory-fn1 (cdr lst) ans
                                 (add-to-set-equal (car (car lst)) redefined)))
          ((member-eq (car (car lst)) redefined)
           (universal-theory-fn1 (cdr lst) ans redefined))
          (t (universal-theory-fn1 (cdr lst)
                                   (append-strip-cdrs (cddr (car lst)) ans)
                                   redefined))))
        ((and (eq (car (car lst)) 'reversed-standard-theories)
              (eq (cadr (car lst)) 'global-value))
         #+acl2-metering (meter-maid 'universal-theory-fn1 500)
         (revappend-delete-runes-based-on-symbols (car (cddr (car lst)))
                                                  redefined
                                                  ans))
        (t
         #+acl2-metering (setq meter-maid-cnt (1+ meter-maid-cnt))
         (universal-theory-fn1 (cdr lst) ans redefined))))

(defun universal-theory-fn (logical-name wrld)

; Return the theory containing all of the rule names in the world created
; by the event that introduced logical-name.

  (declare (xargs :guard (logical-namep logical-name wrld)))

; It is possible that wrld starts with a triple of the form (name REDEFINED
; . mode) in which case that triple is followed by an arbitrary number of
; triples "renewing" various properties of name.  Among those properties is,
; necessarily, RUNIC-MAPPING-PAIRS.  This situation only arises if we are
; evaluating a theory expression as part of an event that is in fact redefining
; name.  These "mid-event" worlds are odd precisely because they do not start
; on event boundaries (with appropriate interpretation given to the occasional
; saving of worlds and theories).

; Now we are asked to get a theory as of logical-name and hence must decode
; logical name wrt wrld, obtaining some tail of wrld, wrld1.  If we are in the
; act of redefining name then we add to wrld1 the triple unbinding
; RUNIC-MAPPING-PAIRS of name.  Why not add all the renewing triples?  The
; reason is that this is the only renewed property that is relevant to
; universal-theory1, the workhorse here.


  (let* ((wrld1 (decode-logical-name logical-name wrld))
         (wrld2 (if (eq (cadr (car wrld)) 'redefined)
                    (cons (list* (car (car wrld))
                                 'runic-mapping-pairs
                                 *acl2-property-unbound*)
                          wrld1)
                  wrld1)))
    (universal-theory-fn1 wrld2 nil nil)))

(defmacro universal-theory (logical-name)

  #-small-acl2-image
  ":Doc-Section Theories

  all rules as of logical name~/
  ~bv[]
  Examples:
  (universal-theory :here)
  (universal-theory 'lemma3)
  ~ev[]
  ~l[logical-name].~/
  ~bv[]
  General Form:
  (universal-theory logical-name)
  ~ev[]
  Returns the theory consisting of all the ~il[rune]s that existed
  immediately after ~ilc[logical-name] was introduced.  ~l[theories]
  and ~pl[logical-name].  The theory includes ~ilc[logical-name] itself
  (if there is a rule by that name).  (Note that since some ~il[events] do
  not introduce rules (e.g., ~ilc[defmacro], ~ilc[defconst] or ~ilc[defthm] with
  ~c[:]~ilc[rule-classes] ~c[nil]), the universal-theory does not necessarily
  include a ~il[rune] for every event name.)  The universal-theory is very
  long and you will probably regret printing it.

  You may experience a fencepost problem in deciding which
  ~il[logical-name] to use.  ~ilc[Deflabel] can always be used to mark
  unambiguously for future reference a particular point in the
  development of your theory.  This is convenient because ~ilc[deflabel]
  does not introduce any rules and hence it doesn't matter if you
  count it as being in the interval or not.  The order of ~il[events] in
  the vicinity of an ~ilc[encapsulate] is confusing.  ~l[encapsulate].

  This ``function'' is actually a macro that expands to a term
  mentioning the single free variable ~ilc[world].  When theory expressions
  are evaluated by ~ilc[in-theory] or the ~c[:]~ilc[in-theory] hint, ~ilc[world] is bound to
  the current ACL2 ~il[world].

  Also ~pl[current-theory].  ~c[Current-theory] is much more commonly used than
  ~c[universal-theory].  The former includes only the ~il[enable]d ~il[rune]s
  as of the given ~ilc[logical-name], which is probably what you want, while
  the latter includes ~il[disable]d ones as well.~/

  :cited-by theory-functions"

  (list 'universal-theory-fn
        logical-name
        'world))

(defun function-theory-fn1 (token lst ans redefined)

; Token is either :DEFINITION, :EXECUTABLE-COUNTERPART or something
; else.  Lst is a cdr of the current world.  We scan down lst and
; accumulate onto ans all of the runes of the indicated type (or both
; if token is neither of the above).

; As in universal-theory-fn1, we also look out for the 'global-value of
; 'reversed-standard-theories and for *acl2-property-unbound*.  See the comment
; there.

  (cond ((null lst)
         #+acl2-metering (meter-maid 'function-theory-fn1 500)
         ans)
        ((eq (cadr (car lst)) 'runic-mapping-pairs)
         #+acl2-metering (setq meter-maid-cnt (1+ meter-maid-cnt))
         (cond
          ((eq (cddr (car lst)) *acl2-property-unbound*)
           (function-theory-fn1 token (cdr lst) ans
                                (add-to-set-equal (car (car lst)) redefined)))
          ((member-eq (car (car lst)) redefined)
           (function-theory-fn1 token (cdr lst) ans redefined))
          ((eq (car (cdr (car (cddr (car lst))))) :DEFINITION)

; The test above extracts the token of the first rune in the mapping pairs and
; this is a function symbol iff it is :DEFINITION.

           (function-theory-fn1 token (cdr lst)
                                (case token
                                      (:DEFINITION
                                       (cons (cdr (car (cddr (car lst)))) ans))
                                      (:EXECUTABLE-COUNTERPART

; Note that we might be looking at the result of storing a :definition rule, in
; which case there will be no :executable-counterpart rune.  So, we check that
; we have something before accumulating it.

                                       (let ((x (cdr (cadr (cddr (car lst))))))
                                         (if (null x)
                                             ans
                                           (cons x ans))))
                                      (otherwise
                                       (cons (cdr (car (cddr (car lst))))
                                             (cons (cdr (cadr (cddr (car lst))))
                                                   ans))))
                                redefined))
          (t (function-theory-fn1 token (cdr lst) ans redefined))))
        ((and (eq (car (car lst)) 'reversed-standard-theories)
              (eq (cadr (car lst)) 'global-value))
         #+acl2-metering (meter-maid 'function-theory-fn1 500)
         (revappend-delete-runes-based-on-symbols
          (case token
                (:DEFINITION (cadr (cddr (car lst))))
                (:EXECUTABLE-COUNTERPART (caddr (cddr (car lst))))
                (otherwise (cadddr (cddr (car lst)))))
          redefined
          ans))
        (t
         #+acl2-metering (setq meter-maid-cnt (1+ meter-maid-cnt))
         (function-theory-fn1 token (cdr lst) ans redefined))))

(defun function-theory-fn (logical-name wrld)

; Return the theory containing all of the function names in the world
; created by the user event that introduced logical-name.

  (declare (xargs :guard (logical-namep logical-name wrld)))

; See universal-theory-fn for an explanation of the production of wrld2.

  (let* ((wrld1 (decode-logical-name logical-name wrld))
         (wrld2 (if (eq (cadr (car wrld)) 'redefined)
                    (cons (list* (car (car wrld))
                                 'runic-mapping-pairs
                                 *acl2-property-unbound*)
                          wrld1)
                  wrld1)))
    (function-theory-fn1 :DEFINITION wrld2 nil nil)))

(defmacro function-theory (logical-name)

  #-small-acl2-image
  ":Doc-Section Theories

  function symbol rules as of logical name~/
  ~bv[]
  Examples:
  (function-theory :here)
  (function-theory 'lemma3)
  ~ev[]
  ~l[logical-name].~/
  ~bv[]
  General Form:
  (function-theory logical-name)
  ~ev[]
  Returns the theory containing all the ~c[:]~ilc[definition] ~il[rune]s, whether
  ~il[enable]d or not, that existed immediately after ~ilc[logical-name] was
  introduced.  See the documentation for ~il[theories],
  ~il[logical-name] and ~ilc[executable-counterpart-theory].

  You may experience a fencepost problem in deciding which logical
  name to use.  ~ilc[Deflabel] can always be used to mark unambiguously for
  future reference a particular point in the development of your
  theory.  The order of ~il[events] in the vicinity of an ~ilc[encapsulate] is
  confusing.  ~l[encapsulate].

  This ``function'' is actually a macro that expands to a term
  mentioning the single free variable ~ilc[world].  When theory expressions
  are evaluated by ~ilc[in-theory] or the ~c[:]~ilc[in-theory] hint, ~ilc[world] is bound to
  the current ACL2 ~il[world].~/

  :cited-by theory-functions"

  (list 'function-theory-fn
        logical-name
        'world))

(defun executable-counterpart-theory-fn (logical-name wrld)

; Return the theory containing all of the executable-counterpart names
; in the world created by the event that introduced logical-name.

  (declare (xargs :guard (logical-namep logical-name wrld)))

; See universal-theory-fn for an explanation of the production of wrld2.

  (let* ((wrld1 (decode-logical-name logical-name wrld))
         (wrld2 (if (eq (cadr (car wrld)) 'redefined)
                    (cons (list* (car (car wrld))
                                 'runic-mapping-pairs
                                 *acl2-property-unbound*)
                          wrld1)
                  wrld1)))
    (function-theory-fn1 :executable-counterpart wrld2 nil nil)))

(defmacro executable-counterpart-theory (logical-name)

  #-small-acl2-image
  ":Doc-Section Theories

  executable counterpart rules as of logical name~/
  ~bv[]
  Examples:
  (executable-counterpart-theory :here)
  (executable-counterpart-theory 'lemma3)
  ~ev[]
  ~l[logical-name].~/
  ~bv[]
  General Form:
  (executable-counterpart-theory logical-name)
  ~ev[]
  Returns the theory containing all the ~c[:]~ilc[executable-counterpart]
  ~il[rune]s, whether ~il[enable]d or not, that existed immediately after
  ~ilc[logical-name] was introduced.  See the documentation for
  ~il[theories], ~il[logical-name], ~il[executable-counterpart] and
  ~ilc[function-theory].

  You may experience a fencepost problem in deciding which logical
  name to use.  ~ilc[Deflabel] can always be used to mark unambiguously for
  future reference a particular point in the development of your
  theory.  The order of ~il[events] in the vicinity of an ~ilc[encapsulate] is
  confusing.  ~l[encapsulate].

  This ``function'' is actually a macro that expands to a term
  mentioning the single free variable ~ilc[world].  When theory expressions
  are evaluated by ~ilc[in-theory] or the ~c[:]~ilc[in-theory] hint, ~ilc[world] is bound to
  the current ACL2 ~il[world].~/

  :cited-by theory-functions"

  (list 'executable-counterpart-theory-fn
        logical-name
        'world))

; Having defined the functions for computing the standard theories,
; we'll now define the function for precomputing them.

(defun reversed-standard-theories (wrld)
  (list (reverse (universal-theory-fn1 wrld nil nil))
        (reverse (function-theory-fn1 :definition wrld nil nil))
        (reverse (function-theory-fn1 :executable-counterpart wrld nil nil))
        (reverse (function-theory-fn1 :both wrld nil nil))))

(defun current-theory1 (lst ans redefined)

; Lst is a cdr of wrld.  We wish to return the enabled theory as of the time
; lst was wrld.  When in-theory is executed it stores (the REVERSE of) the
; newly enabled theory under the 'global-value of the variable 'current-theory.
; When new rule names are introduced, they are automatically considered
; enabled.  Thus, the enabled theory at any point is the union of the current
; value of 'current-theory and the names introduced since that value was set.
; However, :REDEF complicates matters.  See universal-theory-fn1.

  (cond ((null lst)
         #+acl2-metering (meter-maid 'current-theory1 500)
         ans)
        ((eq (cadr (car lst)) 'runic-mapping-pairs)
         #+acl2-metering (setq meter-maid-cnt (1+ meter-maid-cnt))
         (cond
          ((eq (cddr (car lst)) *acl2-property-unbound*)
           (current-theory1 (cdr lst) ans
                            (add-to-set-equal (car (car lst)) redefined)))
          ((member-eq (car (car lst)) redefined)
           (current-theory1 (cdr lst) ans redefined))
          (t 
           (current-theory1 (cdr lst)
                            (append-strip-cdrs (cddr (car lst)) ans)
                            redefined))))
        ((and (eq (car (car lst)) 'current-theory)
              (eq (cadr (car lst)) 'global-value))

; Recall that the value of 'current-theory is the reverse of what its name
; suggests.  So we reverse it as we append it to ans, thereby putting the rune
; with lowest nume (e.g., 0) first.  In this scan we also delete all the
; redefined runes.

         #+acl2-metering (meter-maid 'current-theory1 500)
         (revappend-delete-runes-based-on-symbols (cddr (car lst))
                                                  redefined ans))
        (t
         #+acl2-metering (setq meter-maid-cnt (1+ meter-maid-cnt))
         (current-theory1 (cdr lst) ans redefined))))

(defun current-theory-fn (logical-name wrld)

; We return the theory that was enabled in the world created by the
; event that introduced logical-name.

  (declare (xargs :guard (logical-namep logical-name wrld)))

; See universal-theory-fn for an explanation of the production of wrld2.

  (let* ((wrld1 (decode-logical-name logical-name wrld))
         (wrld2 (if (eq (cadr (car wrld)) 'redefined)
                    (cons (list* (car (car wrld))
                                 'runic-mapping-pairs
                                 *acl2-property-unbound*)
                          wrld1)
                  wrld1)))
    (current-theory1 wrld2 nil nil)))

(defmacro current-theory (logical-name)

  #-small-acl2-image
  ":Doc-Section Theories

  currently ~il[enable]d rules as of logical name~/
  ~bv[]
  Examples:
  (current-theory :here)
  (current-theory 'lemma3)
  ~ev[]
  ~l[logical-name].~/
  ~bv[]
  General Form:
  (current-theory logical-name)
  ~ev[]
  Returns the current theory as it existed immediately after the
  introduction of ~ilc[logical-name] provided it is evaluated in
  an environment in which the variable symbol WORLD is bound to the
  current ACL2 logical world, ~c[(w state)].  Thus,
  ~bv[]
  ACL2 !>(current-theory :here)
  ~ev[]
  will cause an (unbound variable) error while
  ~bv[]
  ACL2 !>(let ((world (w state))) (current-theory :here))
  ~ev[]
  will return the current theory in world.

  ~l[theories] and ~pl[logical-name] for a discussion of
  theories in general and why the commonly used ``theory functions''
  such as ~c[current-theory] are really macros that expand into terms
  involving the variable ~c[world].  

  The theory returned by ~c[current-theory] is in fact the theory selected by
  the ~ilc[in-theory] event most recently preceding logical name, extended by
  the rules introduced up through ~ilc[logical-name].

  You may experience a fencepost problem in deciding which logical
  name to use.  ~ilc[Deflabel] can always be used to mark unambiguously for
  future reference a particular point in the development of your
  theory.  The order of ~il[events] in the vicinity of an ~ilc[encapsulate] is
  confusing.  ~l[encapsulate].

  This ``function'' is actually a macro that expands to a term
  mentioning the single free variable ~ilc[world].  When theory expressions
  are evaluated by ~ilc[in-theory] or the ~c[:]~ilc[in-theory] hint, ~ilc[world] is bound to
  the current ACL2 ~il[world].~/

  :cited-by theory-functions"

  (list 'current-theory-fn logical-name
        'world))

; Essay on Theory Manipulation Performance

; Below we show some statistics on our theory manipulation functions.
; These are recorded in case we someday change these functions and
; wish to compare the old and new implementations.  The expressions
; shown should be executed in raw lisp, not LP, because they involve
; the time function.  These expressions were executed in a newly
; initialized ACL2.  The times are on a Sparc 2 (Rana).

; The following expression is intended as a "typical" heavy duty
; theory expression.  For the record, the universal theory at the time
; of these tests contained 1307 runes.

#|(let ((world (w *the-live-state*)))
  (time
   (length
    (union-theories
     (intersection-theories (current-theory :here)
                            (executable-counterpart-theory :here))
     (set-difference-theories (universal-theory :here)
                              (function-theory :here))))))|#

; Repeated runs were done.  Typical results were:
;   real time : 0.350 secs
;   run time  : 0.233 secs
;   993

; The use of :here above meant that all the theory functions involved
; just looked up their answers in the 'reversed-standard-theories at
; the front of the initialized world.  The following expression forces
; the exploration of the whole world.  In the test, "ACL2-USER" was
; the event printed by :pc -1, i.e., the last event before ending the
; boot.

#|(let ((world (w *the-live-state*)))
  (time
   (length
    (union-theories
     (intersection-theories (current-theory "ACL2-USER")
                            (executable-counterpart-theory "ACL2-USER"))
     (set-difference-theories (universal-theory "ACL2-USER")
                              (function-theory "ACL2-USER"))))))|#

; Repeated tests produced the following typical results.
;   real time : 0.483 secs
;   run time  : 0.383 secs
;   993
; The first run, however, had a real time of almost 10 seconds because
; wrld had to be paged in.

; The final test stresses sorting.  We return to the :here usage to
; get our theories, but we reverse the output every chance we get so
; as force the next theory function to sort.  In addition, we
; strip-cadrs all the input runic theories to force the reconstruction
; of runic theories from the wrld.

#|(let ((world (w *the-live-state*)))
  (time
   (length
    (union-theories
     (reverse
      (intersection-theories
        (reverse (strip-base-symbols (current-theory :here)))
        (reverse (strip-base-symbols (executable-counterpart-theory :here)))))
     (reverse
      (set-difference-theories
        (reverse (strip-base-symbols (universal-theory :here)))
        (reverse (strip-base-symbols (function-theory :here)))))))))|#

; Typical times were
;   real time : 1.383 secs
;   run time  : 0.667 secs
;   411
; The size of the result is smaller because the strip-cadrs identifies
; several runes, e.g., (:DEFINITION fn) and (:EXECUTABLE-COUNTERPART
; fn) both become fn which is then understood as (:DEFINITION fn).

; End of performance data.

(defun end-prehistoric-world (wrld)
  (let ((wrld1 (global-set-lst
                (list (list 'untouchables
                            (append *initial-untouchables*
                                    (global-val 'untouchables wrld)))
                      (list 'reversed-standard-theories
                            (reversed-standard-theories wrld))
                      (list 'current-theory
                            (reverse (current-theory1 wrld nil nil)))
                      (list 'boot-strap-flg nil)
                      (list 'boot-strap-pass-2 nil)
                      (list 'small-p nil)
                      (list 'command-number-baseline
                            (next-absolute-command-number wrld))
                      (list 'event-number-baseline
                            (next-absolute-event-number wrld))
                      (list 'skip-proofs-seen nil))
                wrld)))
    (add-command-landmark
     :logic
     '(exit-boot-strap-mode)
     (add-event-landmark
      '(exit-boot-strap-mode)
      'exit-boot-strap-mode
      0
      nil
      wrld1))))

(defun theory-namep (name wrld)

; We return t or nil according to whether name is the name of a theory,
; i.e., a name introduced by deftheory.

  (and (symbolp name)
       (not (eq (getprop name 'theory t 'current-acl2-world wrld)
                t))))

(defun theory-fn (name wrld)

; We deliver the value of the defined theory named name.

  (declare (xargs :guard (theory-namep name wrld)))
  (getprop name 'theory nil 'current-acl2-world wrld))

(defmacro theory (name)

  #-small-acl2-image
  ":Doc-Section Theories

  retrieve named theory~/
  ~bv[]
  Example:
  (theory 'ground-zero)
  ~ev[]
  In the example above, the theory returned is the one in force when ACL2 is
  started up (~pl[ground-zero]).~/

  ~bv[]
  General Form:
  (theory name)
  ~ev[]
  where ~c[name] is the name of a previously executed ~ilc[deftheory] event.
  Returns the named theory.  ~l[theories].

  This ``function'' is actually a macro that expands to a term
  mentioning the single free variable ~ilc[world].  When theory expressions
  are evaluated by ~ilc[in-theory] or the ~c[:]~ilc[in-theory] hint, ~ilc[world] is bound to
  the current ACL2 ~il[world].~/

  :cited-by theory-functions"

  (list 'theory-fn name 'world))

(defun deftheory-fn (name expr state doc event-form)

; Historical Note:  Once upon a time deftheory-fn did not exist even
; though deftheory did.  We defined deftheory as a macro which expanded
; into a defconstant-fn expression.  In particular,

; (deftheory *a* (union *b* (universe w)))

; was mapped to

; (er-let* ((lst (translate-in-theory-hint
;                   '(union *b* (universe w))
;                   nil
;                   '(deftheory . *a*)
;                   (w state)
;                   state)))
;          (defconstant-fn '*a*
;            (list 'quote lst)
;            state
;            nil))

; Thus, the "semantics" of a successful execution of deftheory was that of
; defconstant.  This suffered from letting theories creep into formulas.  For
; example, one could later write in a proposed theorem (member 'foo *a*) and
; the truth of that proposition depended upon the particular theory computed
; for *a*.  This made it impossible to permit either the use of state in
; "theory expressions" (since different theories could be computed for
; identical worlds, depending on ld-skip-proofsp) or the use of deftheory in
; encapsulate (see below).  The state prohibition forced upon us the ugliness
; of permitting the user to reference the current ACL2 world via the free
; variable W in theory expressions, which we bound appropriately before evaling
; the expressions.

; We abandoned the use of defconstant (now defconst) for these reasons.

; Here is a comment that once illustrated why we did not allow deftheory
; to be used in encapsulate:

; We do not allow deftheory expressions in encapsulate.  This may be a
; severe restriction but it is necessary for soundness given the current
; implementation of deftheory.  Consider the following:

; (encapsulate nil
;   (local (defun foo () 1))
;   (deftheory *u* (all-names w))
;   (defthm foo-thm (member 'foo *u*)))

; where all-names is a user defined function that computes the set of
; all names in a given world.  [Note: Intuitively, (all-names w) is
; (universal-theory nil w).  Depending on how event descriptors are
; handled, that may or may not be correct.  In a recent version of
; ACL2, (universal-theory nil w), if used in an encapsulate, had the
; effect of computing all the names in the theory as of the last
; world-chaning form executed by the top-level loop.  But because
; encapsulate did not so mark each term as it executed them,
; universal-theory backed up to the point in w just before the
; encapsulate.  Thus, universal-theory could not be used to get the
; effect intended here.  However, (all-names w) could be defined by
; the user to get what is intended here.]

; When the above sequence is processed in pass 1 of encapsulate *u*
; includes 'foo and hence the defthm succeeds.  But when it is processed
; in pass 2 *u* does not include 'foo and so the assumption of the
; defthm is unsound!  In essence, permitting deftheory in encapsulate is
; equivalent to permitting (w state) in defconst forms.  That is
; disallowed too (as is the use of any variable in an defconst form).
; If you can set a constant as a function of the world, then you can use
; the constant to determine which encapsulate pass you are in.

  (when-logic
   "DEFTHEORY"
   (with-ctx-summarized
    (if (output-in-infixp state) event-form (cons 'deftheory name))
    (let ((wrld (w state))
          (event-form (or event-form
                          (list* 'deftheory name expr
                                 (if doc
                                     (list :doc doc)
                                   nil)))))
      (er-progn
       (chk-all-but-new-name name ctx nil wrld state)
       (er-let*
        ((wrld1 (chk-just-new-name name 'theory nil ctx wrld state))
         (doc-pair (translate-doc name doc ctx state))
         (theory (translate-in-theory-hint expr nil ctx wrld1 state)))
        (let ((wrld2 (update-doc-data-base
                      name doc doc-pair
                      (putprop name 'theory theory wrld1))))

; Note:  We do not permit DEFTHEORY to be made redundant.  If this
; is changed, change the text of the :doc for redundant-events.

          (install-event (length theory)
                         event-form
                         'deftheory
                         name
                         nil nil wrld2 state))))))))

; And now we move on to the in-theory event, in which we process a theory
; expression into a theory and then load it into the global enabled
; structure.

(defun in-theory-fn (expr state doc event-form)
  (when-logic
   "IN-THEORY"
   (with-ctx-summarized
    (if (output-in-infixp state)
        event-form
      (cond ((atom expr)
             (cond ((null doc)
                    (msg "(IN-THEORY ~x0)" expr))
                   (t (cons 'in-theory expr))))
            ((symbolp (car expr))
             (cond ((null doc)
                    (msg "(IN-THEORY (~x0 ...))"
                         (car expr)))
                   (t (msg "(IN-THEORY (~x0 ...) ...)"
                           (car expr)))))
            ((null doc) "(IN-THEORY (...))")
            (t "(IN-THEORY (...) ...)")))
    (let ((wrld (w state))
          (event-form (or event-form
                          (list* 'in-theory expr
                                 (if doc
                                     (list :doc doc)
                                   nil)))))
      (er-let*
       ((doc-pair (translate-doc nil doc ctx state))
        (theory (translate-in-theory-hint expr t ctx wrld state)))
       (let* ((ens1 (global-val 'global-enabled-structure wrld))
              (force-xnume-en1 (enabled-numep *force-xnume* ens1))
              (imm-xnume-en1 (enabled-numep *immediate-force-modep-xnume* ens1))
              (ens2 (load-theory-into-enabled-structure theory ens1 nil wrld))
              (wrld1
               (global-set 'current-theory
                           (reverse theory)
                           (global-set 'global-enabled-structure ens2 wrld))))

; Note:  We do not permit IN-THEORY to be made redundant.  If this
; is changed, change the text of the :doc for redundant-events.

         (pprogn
          (if (member-equal expr
                            '((enable (:EXECUTABLE-COUNTERPART force))
                              (disable (:EXECUTABLE-COUNTERPART force))
                              (enable (:EXECUTABLE-COUNTERPART
                                       immediate-force-modep))
                              (disable (:EXECUTABLE-COUNTERPART
                                        immediate-force-modep))))
              state
            (maybe-warn-about-theory ens1 force-xnume-en1 imm-xnume-en1 ens2
                                     ctx wrld state))
          (install-event (length theory)
                         event-form
                         'in-theory
                         0
                         nil
                         nil
                         wrld1 state))))))))

; JSM:  Visit all occurrences of with-ctx-summarized and
; make sure the strings being passed have a space after the open paren.
; The strings below don't.

(defun in-arithmetic-theory-fn (expr state doc event-form)
  (when-logic
   "IN-ARITHMETIC-THEORY"
   (with-ctx-summarized
    (if (output-in-infixp state)
        event-form
      (cond ((atom expr)
             (cond ((null doc)
                    (msg "(IN-ARITHMETIC-THEORY ~x0)" expr))
                   (t (cons 'in-arithmetic-theory expr))))
            ((symbolp (car expr))
             (cond ((null doc)
                    (msg "(IN-ARITHMETIC-THEORY (~x0 ...))"
                         (car expr)))
                   (t (msg "(IN-ARITHMETIC-THEORY (~x0 ...) ...)"
                           (car expr)))))
            ((null doc) "(IN-ARITHMETIC-THEORY (...))")
            (t "(IN-ARITHMETIC-THEORY (...) ...)")))
    (let ((wrld (w state))
          (event-form (or event-form
                          (list* 'in-arithmetic-theory expr
                                 (if doc
                                     (list :doc doc)
                                   nil)))))
      (cond
       ((not (quotep expr))
        (er soft ctx
            "Arithmetic theory expressions must be quoted constants.  ~
             See :DOC in-arithmetic-theory."))
       (t
        (er-let*
          ((doc-pair (translate-doc nil doc ctx state))
           (theory (translate-in-theory-hint expr t ctx wrld state)))
          (let* ((ens1 (global-val 'global-arithmetic-enabled-structure wrld))
                 (ens2 (load-theory-into-enabled-structure theory ens1 nil wrld))
                 (wrld1 (global-set 'global-arithmetic-enabled-structure ens2 wrld)))

; Note:  We do not permit IN-THEORY to be made redundant.  If this
; is changed, change the text of the :doc for redundant-events.

            (install-event (length theory)
                           event-form
                           'in-arithmetic-theory
                           0
                           nil
                           nil
                           wrld1 state)))))))))

(defmacro disable (&rest rst)

  #-small-acl2-image
  ":Doc-Section Theories

  deletes names from current theory~/
  ~bv[]
  Example:
  (disable fact (fact) associativity-of-app)~/

  General Form:
  (disable name1 name2 ... namek)
  ~ev[]
  where each ~c[namei] is a runic designator; ~pl[theories].  The
  result is the theory that contains all the names in the current
  theory except those listed.  Note that this is merely a function
  that returns a theory.  The result is generally a very long list of
  ~il[rune]s and you will probably regret printing it.

  The standard way to ``disable'' a fixed set of names, is:
  ~bv[]
  (in-theory (disable name1 name2 ... namek)) ; globally
  :in-theory (disable name1 name2 ... namek)  ; locally
  ~ev[]
  Note that all the names are implicitly quoted.  If you wish to
  disable a computed list of names, ~c[lst], use the theory expression
  ~c[(set-difference-theories (current-theory :here) lst)].~/

  :cited-by theory-functions"

  `(set-difference-theories (current-theory :here) ',rst))

(defmacro enable (&rest rst)

  #-small-acl2-image
  ":Doc-Section Theories

  adds names to current theory~/
  ~bv[]
  Example:
  (enable fact (fact) associativity-of-app)~/

  General Form:
  (enable name1 name2 ... namek)
  ~ev[]
  where each ~c[namei] is a runic designator; ~pl[theories].  The
  result is the theory that contains all the names in the current
  theory plus those listed.  Note that this is merely a function that
  returns a theory.  The result is generally a very long list of ~il[rune]s
  and you will probably regret printing it.

  The standard way to ``enable'' a fixed set of names, is
  ~bv[]
  (in-theory (enable name1 name2 ... namek)) ; globally
  :in-theory (enable name1 name2 ... namek)  ; locally
  ~ev[]
  Note that all the names are implicitly quoted.  If you wish to
  enable a computed list of names, ~c[lst], use the theory expression
  ~c[(union-theories (current-theory :here) lst)].~/

  :cited-by theory-functions"

  `(union-theories (current-theory :here) ',rst))

; Next we handle the table event.

(defun chk-table-nil-args (op bad-arg bad-argn ctx state)

; See table-fn1 for representative calls of this weird little function.

  (cond (bad-arg
         (er soft ctx
             "Table operation ~x0 requires that the ~n1 argument to ~
              TABLE be nil.  Hence, ~x2 is an illegal ~n1 argument.  ~
              See :DOC table."
             op bad-argn bad-arg))
        (t (value nil))))

(defun chk-table-guard (name key val ctx wrld state)
  (let ((term (getprop name 'table-guard
                       *t* 'current-acl2-world wrld)))
    (mv-let
     (erp okp latches)
     (ev term
         (list (cons 'key key)
               (cons 'val val)
               (cons 'world wrld))
         state nil nil)
     (declare (ignore latches))
     (cond
      (erp (pprogn
            (error-fms nil ctx (car okp) (cdr okp) state)
            (er soft ctx
                "The TABLE :guard for ~x0 on the key ~x1 and value ~x2 could ~
                 not be evaluated."
                name key val)))
      (okp (value nil))
      (t (er soft ctx
             "The TABLE :guard for ~x0 disallows the combination of ~
              key ~x1 and value ~x2.  The :guard is ~X34.  See :DOC ~
              table."
             name key val (untranslate term t wrld) nil))))))

(defun chk-table-guards (name alist ctx wrld state)
  (if alist
      (er-progn (chk-table-guard name (caar alist) (cdar alist) ctx wrld state)
                (chk-table-guards name (cdr alist) ctx wrld state))
    (value nil)))

(defun put-assoc-equal-fast (name val alist)

; If there is a large number of table events for a given table all with
; different keys, the use of assoc-equal to update the table (in table-fn1)
; causes a quadratic amount of cons garbage.  The following is thus used
; instead.

  (declare (xargs :guard (alistp alist)))
  (if (assoc-equal name alist)
      (put-assoc-equal name val alist)
    (acons name val alist)))

(defun table-fn1 (name key val op term ctx wrld state event-form)

; This is just the rational version of table-fn, with key, val, op and
; term all handled as normal (evaluated) arguments.  The chart in
; table-fn explains the legal ops and arguments.

  (case op
        (:alist 
         (er-progn
          (chk-table-nil-args :alist
                              (or key val term)
                              (cond (key '(2)) (val '(3)) (t '(5)))
                              ctx state)
          (value (table-alist name wrld))))
        (:get
         (er-progn
          (chk-table-nil-args :get
                              (or val term)
                              (cond (val '(3)) (t '(5)))
                              ctx state)
          (value
           (cdr (assoc-equal key
                             (getprop name 'table-alist nil
                                      'current-acl2-world wrld))))))
        (:put
         (with-ctx-summarized
          (if (output-in-infixp state) event-form ctx)
          (er-progn
           (chk-table-nil-args :put term '(5) ctx state)
           (chk-table-guard name key val ctx wrld state)
           (install-event name
                          event-form
                          'table
                          0
                          nil
                          nil
                          (putprop name 'table-alist
                                   (put-assoc-equal-fast
                                    key val
                                    (getprop name 'table-alist nil
                                             'current-acl2-world wrld))
                                   wrld)
                          state))))
        (:clear
         (with-ctx-summarized
          (if (output-in-infixp state) event-form ctx)
          (er-progn
           (chk-table-nil-args :clear
                               (or key term)
                               (cond (key '(2)) (t '(5)))
                               ctx state)
           (if (alistp val)
               (value nil)
             (er soft 'table ":CLEAR requires an alist, but ~x0 is not." val))
           (let ((val (if (duplicate-keysp val)
                          (reverse (clean-up-alist val nil))
                        val)))
             (er-progn
              (if (subsetp-equal val (table-alist name (w state)))
                  ;; this is an optimization that avoids the table-guard checking
                  (value nil)
                (chk-table-guards name val ctx wrld state))
              (install-event name event-form 'table 0 nil nil
                             (putprop name 'table-alist val wrld)
                             state))))))
        (:guard
         (cond
          ((eq term nil)
           (er-progn
            (chk-table-nil-args op
                                (or key val)
                                (cond (key '(2)) (t '(3)))
                                ctx state)
            (value (getprop name 'table-guard *t* 'current-acl2-world wrld))))
          (t
           (with-ctx-summarized
            (if (output-in-infixp state) event-form ctx)
            (er-progn
             (chk-table-nil-args op
                                 (or key val)
                                 (cond (key '(2)) (t '(3)))
                                 ctx state)
             (er-let* ((tterm (translate term '(nil) nil nil ctx wrld state)))

; known-stobjs = nil.  No variable is treated as a stobj in tterm.
; But below we check that the only vars mentioned are KEY, VAL and
; WORLD.  These could, in principle, be declared stobjs by the user.
; But when we ev tterm in the future, we will always bind them to
; non-stobjs.

                      (let ((old-guard
                             (getprop name 'table-guard nil
                                      'current-acl2-world wrld)))
                        (cond
                         ((equal old-guard tterm)
                          (stop-redundant-event state))
                         (old-guard
                          (er soft ctx
                              "It is illegal to change the :guard on a table ~
                               after it has been given an explicit :guard.  ~
                               The :guard of ~x0 is ~X12 and this can be ~
                               changed only by undoing the event that set it.  ~
                               See :DOC table."
                              name
                              (untranslate (getprop name 'table-guard nil
                                                    'current-acl2-world wrld)
                                           t wrld)
                              nil))
                         ((getprop name 'table-alist nil
                                   'current-acl2-world wrld)

; At one time Matt wanted the option of setting the :val-guard of a
; non-empty table, but he doesn't recall why.  Perhaps we'll add such
; an option in the future if others express such a desire.

                          (er soft ctx
                              "It is illegal to set the :guard of the ~
                               non-empty table ~x0.  See :DOC table."
                              name))
                         (t
                          (let ((legal-vars '(key val world))
                                (vars (all-vars tterm)))
                            (cond ((not (subsetp-eq vars legal-vars))
                                   (er soft ctx
                                       "The only variables permitted in the ~
                                        :guard of a table are ~&0, but your ~
                                        guard uses ~&1.  See :DOC table."
                                       legal-vars vars))
                                  (t (install-event name
                                                    event-form
                                                    'table
                                                    0
                                                    nil
                                                    nil
                                                    (putprop name
                                                             'table-guard
                                                             tterm
                                                             wrld)
                                                    state)))))))))))))
        (otherwise (er soft ctx
                       "Unrecognized table operation, ~x0.  See :DOC table."
                       op))))

(defun table-fn (name args state event-form)

; This is an unusual "event" because it sometimes has no effect on
; STATE and thus is not an event!  In general this function applies
; an operation, op, to some arguments (and to the table named name).
; Ideally, args is of length four and of the form (key val op term).
; But when args is shorter it is interpreted as follows.

; args              same as args
; ()                (nil nil :alist nil)
; (key)             (key nil :get   nil)
; (key val)         (key val :put   nil)
; (key val op)      (key val op     nil)

; Key and val are both treated as forms and evaluated to produce
; single results (which we call key and val below).  Op and term are
; not evaluated.  A rational version of this function that takes key,
; val, op and term all as normal arguments is table-fn1.  The odd
; design of this function with its positional interpretation of op and
; odd treatment of evaluation is due to the fact that it represents
; the macroexpansion of a form designed primarily to be typed by the
; user.

; Op may be any of :alist, :get, :put, :clear, or :guard.  Each op
; enforces certain restrictions on the other three arguments.

; op         restrictions and meaning
; :alist     Key val and term must be nil.  Return the table as an
;            alist pairing keys to their non-nil vals.
; :get       Val and term must be nil.Return the val associated with
;            key.
; :put       Key and val satisfy :guard and term must be nil.  Store
;            val with key.
; :clear     Key and term must be nil.  Clear the table, setting it
;            to val if val is supplied (else to nil).  Note that val
;            must be an alist, and as with :put, the keys and entries
;            must satisfy the :guard.
; :guard     Key and val must be nil, term must be a term mentioning
;            only the variables KEY, VAL, and WORLD, and returning one
;            result.  The table must be empty.  Store term as the
;            table's :guard.

; Should table events be permitted to have documentation strings?  No.
; The reason is that we do not protect other names from being used as
; tables.  For example, the user might set up a table with the name
; defthm.  If we permitted a doc-string for that table, :DOC defthm
; would be overwritten.

  (let* ((ctx (cons 'table name))
         (wrld (w state))
         (event-form (or event-form
                         `(table ,name ,@args)))
         (n (length args))
         (key-form (car args))
         (val-form (cadr args))
         (op (cond ((= n 2) :put)
                   ((= n 1) :get)
                   ((= n 0) :alist)
                   (t (caddr args))))
         (term (cadddr args)))
    (er-progn
     (cond ((not (symbolp name))
            (er soft ctx
                "The first argument to table must be a symbol, but ~
                 ~x0 is not.  See :DOC table."
                name))
           ((< 4 (length args))
            (er soft ctx
                "Table may be given no more than five arguments.  In ~
                 ~x0 it is given ~n1.  See :DOC table."
                event-form
                (1+ (length args))))
           (t (value nil)))
     (er-let* ((key-pair
                (simple-translate-and-eval
                 key-form
                 (if (eq name 'acl2-defaults-table)
                     nil
                     (list (cons 'world wrld)))
                 nil
                 (if (eq name 'acl2-defaults-table)
                     "In (TABLE ACL2-DEFAULTS-TABLE key ...), key"
                     "The second argument of TABLE")
                 ctx wrld state))
               (val-pair
                (simple-translate-and-eval
                 val-form
                 (if (eq name 'acl2-defaults-table)
                     nil
                     (list (cons 'world wrld)))
                 nil
                 (if (eq name 'acl2-defaults-table)
                     "In (TABLE ACL2-DEFAULTS-TABLE key val ...), val"
                     "The third argument of TABLE")
                 ctx wrld state)))
              (table-fn1 name (cdr key-pair) (cdr val-pair) op term
                         ctx wrld state event-form)))))

; A use of tables is to allow the user to specify "theory invariants."
; We set that up now.

; The theory-invariant-table maps arbitrary keys to translated terms
; involving only the variables THEORY and STATE:

(table theory-invariant-table nil nil
       :guard (and (consp val)
                   (termp (car val) world)
                   (booleanp (cdr val))
                   (subsetp-eq (all-vars (car val)) '(theory state))))

(defmacro theory-invariant (&whole event-form term &key key (error 't))

  #-small-acl2-image
  ":Doc-Section Events

  user-specified invariants on ~il[theories]~/
  ~bv[]
  Example:
  (theory-invariant (not (and (member-equal '(:rewrite left-to-right)
                                             theory)
                              (member-equal '(:rewrite right-to-left)
                                            theory)))
                    :key my-invariant
                    :error nil)~/

  General Form:
  (theory-invariant term &key key error)
  ~ev[]
  where:~bq[]

  o ~c[term] is a term that uses no variables other than ~ilc[theory] and
  ~ilc[world];

  o ~c[key] is an arbitrary ``name'' for this invariant (if omitted, an integer
  is generated and used); and

  o ~c[:error] specifies the action to be taken when an invariant is violated
  ~-[] either ~c[nil] if a warning is to be printed, else ~c[t] (the default)
  if an error is to be caused.

  ~eq[]~c[Theory-invariant] is an event that adds to or modifies the ~il[table]
  of user-supplied theory invariants that are checked each time a theory
  expression is evaluated.

  The theory invariant mechanism is provided via a table
  (~pl[table]) named ~c[theory-invariant-table].  In fact, the
  ~c[theory-invariant] ``event'' is just a macro that expands into a use
  of the ~ilc[table] event.  More general access to the ~c[theory-invariant]
  ~il[table] is provided by ~ilc[table] itself.  For example, the ~il[table] can be
  inspected or cleared (setting the invariant to ~c[t]) with ~ilc[table].

  ~c[Theory-invariant-table] maps arbitrary keys to terms mentioning, at
  most, the variables ~ilc[theory] and ~ilc[world].  Every time an alleged theory
  expression is evaluated, e.g., in the ~ilc[in-theory] event or ~c[:]~ilc[in-theory]
  hint, each of the terms in ~c[theory-invariant-table] is evaluated with
  ~ilc[theory] bound to the runic theory (~pl[theories]) obtained from
  the theory expression and ~ilc[world] bound to the current ACL2 ~il[world]
  (~pl[world]).  If the result is ~c[nil], a message is printed and an error
  occurs (except, only a warning occurs if ~c[:error nil] is specified).  Thus,
  the ~il[table] can be thought of as a list of conjuncts.  Each ~c[term] in
  the ~il[table] has a ``name,'' which is just the key under which the term is
  stored.  When a theory violates the restrictions specified by some term, both
  the name and the term are printed.  By calling ~c[theory-invariant] with a
  new term but the same name, you can overwrite that conjunct of the theory
  invariant.

  Theory invariants are particularly useful in the context of large
  rule sets intended for re-use.  Such sets often contain conflicting
  rules, e.g., rules that are to be ~il[enable]d when certain function
  symbols are ~il[disable]d, rules that rewrite in opposite directions and
  thus loop if simultaneously ~il[enable]d, groups of rules which should be
  ~il[enable]d in concert, etc.  The developer of such rule sets
  understands these restrictions and probably documents them.  The
  theory invariant mechanism allows the developer to codify his
  restrictions so that the user is alerted when they are violated.

  Since theory invariants are arbitrary terms, macros may be used to
  express commonly used restrictions.  Because theory invariants are a
  new idea in ACL2, we have only defined one such macro for
  illustrative purposes.  Executing the event
  ~bv[]
  (theory-invariant (incompatible (:rewrite left-to-right)
                                  (:rewrite right-to-left)))
  ~ev[]
  would subsequently cause an error any time the current theory contained both
  of the two ~il[rune]s shown.  Of course, ~il[incompatible] is just defined as
  a macro.  Its definition may be inspected with ~c[:pe incompatible].

  In order for a ~c[theory-invariant] event to be accepted, the proposed theory
  invariant must be satisfied by the current theory (~pl[current-theory]).  The
  value returned upon successful execution of the event is the key (whether
  user-supplied or generated).

  Note: If the ~il[table] event is used directly to ~c[:put] a term into the
  theory invariant ~il[table], be aware that the term must be in translated
  form.  This is enforced by the ~c[value] invariant for
  ~c[theory-invariant-table].  But the upshot of this is that you will be
  unable to use macros in theory invariants stored directly with the
  ~c[:put] ~il[table] event.~/"

; Note: This macro "really" expands to a TABLE event (after computing
; the right args for it!) and hence it should inherit the TABLE event's
; semantics under compilation, which is to say, is a noop.  This
; requirement wasn't noticed until somebody put a THEORY-INVARIANT
; event into a book and then the compiled book compiled the logical
; code below and thus loading the .o file essentially tried to
; reexecute the table event after it had already been executed by the
; .lisp code in the book.  A hard error was caused.

  #-acl2-loop-only
  (declare (ignore event-form term key))
  #-acl2-loop-only

; The clisp compiler (version 2.27) complains whether or not error is declared
; ignored above or not, if the next form is simply nil, perhaps because it is
; assigned a value above as an (&key) arg.  So we go ahead and "use" it here.

  (and (not (equal error error)) nil)

  #+acl2-loop-only
  `(when-logic
    "THEORY-INVARIANT"
    (er-let* ((tterm
               (translate ',term '(nil) nil '(state)
                          'theory-invariant (w state) state)))

; known-stobjs ='(state).  All other variables in term are treated as
; non- stobjs.  This is ok because the :guard on the
; theory-invariant-table will check that the only variables involved
; in tterm are THEORY and STATE and when we ev the term THEORY will be
; bound to a non-stobj (and STATE to state, of course).

             (let ((key ,(if key
                             `(quote ,key)
                           '(1+
                             (length (table-alist 'theory-invariant-table
                                                  (w state)))))))
               (er-progn
                (table-fn1 'theory-invariant-table
                           key
                           (cons tterm ',error)
                           :put
                           nil
                           'theory-invariant
                           (w state)
                           state
                           ',event-form)
                (mv-let (erp val state)
                        (in-theory (current-theory :here))
                        (declare (ignore val))
                        (cond
                         (erp
                          (er soft 'theory-invariant
                              "The specified theory invariant fails for the ~
                             current ACL2 world, and hence is rejected.  This ~
                             failure can probably be overcome by supplying an ~
                             appropriate in-theory event first."))
                         (t (value key)))))))))

(defmacro incompatible (rune1 rune2)
  #-small-acl2-image
  ":Doc-Section Theories

  declaring that two rules should not both be ~il[enable]d~/
  ~bv[]
  Example:
  (theory-invariant (incompatible (:rewrite left-to-right)
                                  (:rewrite right-to-left)))~/

  General Form:
  (incompatible rune1 rune2)
  ~ev[]
  where ~c[rune1] and ~c[rune2] are two specific ~il[rune]s.  The arguments are
  not evaluated.  ~c[Invariant] is just a macro that expands into a term
  that checks that ~ilc[theory] does not contain both ~il[rune]s.
  ~l[theory-invariant].~/"

  `(not (and (member-equal ',rune1 theory)
             (member-equal ',rune2 theory))))

; We now begin the development of the encapsulate event.  Often in this
; development we refer to the Encapsulate Essay.  See the comment in
; the function encapsulate-fn, below.

(deflabel signature
  :doc
  ":Doc-Section Miscellaneous

  how to specify the arity of a constrained function~/
  ~bv[]
  Examples:
  ((hd *) => *)
  ((printer * state) => (mv * * state))
  ((mach * mach-state * state) => (mv * mach-state)

  General Form:
  ((fn ...) => *)
  ((fn ...) => stobj)
  or
  ((fn ...) => (mv ...))
  ~ev[]

  where ~c[fn] is the constrained function symbol, ~c[...] is a list
  of asterisks and/or the names of single-threaded objects and
  ~c[stobj] is a single-threaded object name.  ACL2 also supports an
  older style of signature described below after we describe the
  preferred style.~/

  Signatures specify three syntactic aspects of a function symbol: (1)
  the ``arity'' or how many arguments the function takes, (2) the
  ``multiplicity'' or how many results it returns via ~c[MV], and (3)
  which of those arguments and results are single-threaded objects and
  which objects they are.

  For a discussion of single-threaded objects, ~pl[stobj].  For
  the current purposes it is sufficient to know that every single-
  threaded object has a unique symbolic name and that ~ilc[state] is
  the name of the only built-in single-threaded object.  All other
  stobjs are introduced by the user via ~ilc[defstobj].  An object that
  is not a single-threaded object is said to be ``ordinary.''

  The general form of a signature is ~c[((fn x1 ... xn) => val)].  So
  a signature has two parts, separated by the symbol ``=>''.  The
  first part, ~c[(fn x1 ... xn)], is suggestive of a call of the
  constrained function.  The number of ``arguments,'' ~c[n], indicates
  the arity of ~c[fn].  Each ~c[xi] must be a symbol.  If a given
  ~c[xi] is the symbol ``*'' then the corresponding argument must be
  ordinary.  If a given ~c[xi] is any other symbol, that symbol must
  be the name of a single-threaded object and the corresponding
  argument must be that object.  No stobj name may occur twice among the
  ~c[xi].

  The second part, ~c[val], of a signature is suggestive of a term and
  indicates the ``shape'' of the output of ~c[fn].  If ~c[val] is a
  symbol then it must be either the symbol ``*'' or the name of a
  single-threaded object.  In either case, the multiplicity of ~c[fn]
  is 1 and ~c[val] indicates whether the result is ordinary or a
  stobj.  Otherwise, ~c[val] is of the form ~c[(mv y1 ... yk)], where
  ~c[k] > 1.  Each ~c[yi] must be either the symbol ``*'' or the name
  of a stobj.  Such a ~c[val] indicates that ~c[fn] has multiplicity
  ~c[k] and the ~c[yi] indicate which results are ordinary and which
  are stobjs.  No stobj name may occur twice among the ~c[yi].

  Finally, a stobj name may appear in ~c[val] only if appears among the
  ~c[xi].

  Before ACL2 supported user-declared single-threaded objects there
  was only one single-threaded object: ACL2's built-in notion of
  ~ilc[state].  The notion of signature supported then gave a special
  role to the symbol ~c[state] and all other symbols were considered
  to denote ordinary objects.  ACL2 still supports the old form of
  signature, but it is limited to functions that operate on ordinary
  objects or ordinary objects and ~c[state].

  ~bv[]
  Old-Style General Form:
  (fn formals result)
  ~ev[]

  where ~c[fn] is the constrained function symbol, ~c[formals] is a
  suitable list of formal parameters for it, and ~c[result] is either
  a symbol denoting that the function returns one result or else
  ~c[result] is an ~ilc[mv] expression, ~c[(mv s1 ... sn)], where
  ~c[n>1], each ~c[si] is a symbol, indicating that the function
  returns ~c[n] results.  At most one of the formals may be the symbol
  ~c[STATE], indicating that corresponding argument must be ACL2's
  built-in ~ilc[state].  If ~c[state] appears in ~c[formals] then
  ~c[state] may appear once in ~c[result].  All ``variable symbols''
  other than ~c[state] in old style signatures denote ordinary
  objects, regardless of whether the symbol has been defined to be a
  single-threaded object name!

  We also support a variation on old style signatures allowing the user
  to declare which symbols (besides ~c[state]) are to be considered
  single-threaded object names.  This form is
  ~bv[]
  (fn formals result :stobjs names)
  ~ev[]
  where ~c[names] is either the name of a single-threaded object or else
  is a list of such names.  Every name in ~c[names] must have been
  previously defined as a stobj via ~c[defstobj].~/")

(defun gen-formals-from-pretty-flags1 (pretty-flags i avoid)
  (cond ((endp pretty-flags) nil)
        ((eq (car pretty-flags) '*)
         (let ((xi (pack2 'x i)))
           (cond ((member-eq xi avoid)
                  (let ((new-var (genvar 'genvar ;;; ACL2 package
                                         "GENSYM"
                                         1
                                         avoid)))
                    (cons new-var 
                          (gen-formals-from-pretty-flags1
                           (cdr pretty-flags)
                           (+ i 1)
                           (cons new-var avoid)))))
                 (t (cons xi
                          (gen-formals-from-pretty-flags1
                           (cdr pretty-flags)
                           (+ i 1)
                           avoid))))))
        (t (cons (car pretty-flags)
                 (gen-formals-from-pretty-flags1
                  (cdr pretty-flags)
                  (+ i 1)
                  avoid)))))

(defun gen-formals-from-pretty-flags (pretty-flags)

; Given a list of prettyified stobj flags, e.g., '(* * $S * STATE) we
; generate a proposed list of formals, e.g., '(X1 X2 $S X4 STATE).  We
; guarantee that the result is a list of symbols as long as
; pretty-flags.  Furthermore, a non-* in pretty-flags is preserved in
; the same slot in the output.  Furthermore, the symbol generated for
; each * in pretty-flags is unique and not among the symbols in
; pretty-flags.  Finally, STATE is not among the symbols we generate.

  (gen-formals-from-pretty-flags1 pretty-flags 1 pretty-flags))

(defconst *generic-bad-signature-string*
  "The object ~x0 is not a legal signature.  It should be of one of the ~
   following three forms:  ((fn sym1 ... symn) => val) or (fn (var1 ... varn) ~
   val) or (fn (var1 ... varn) val :stobjs names).  But it is of none of these ~
   forms!  See :DOC signature.")

(defun chk-signature (x ctx wrld state)

; Warning: If you change the acceptable form of signatures, change the
; raw lisp code for encapsulate in axioms.lisp and change
; signature-fns.

; X is supposed to be the external form of a signature of a function,
; fn.  This function either causes an error (if x is ill-formed) or
; returns a pair of the form (insig .  wrld1), where insig is the
; internal form of the signature of fn.

; The preferred external form of a signature is of the form:

; ((fn . pretty-flags) => pretty-flag)
; ((fn . pretty-flags) => (mv . pretty-flags))

; where fn is a new or redefinable name, pretty-flag is an asterisk or
; stobj name, and pretty-flags is a true list of pretty flags.  Note
; that this ``preferred form'' is deficient because it does not allow
; the specification of the formals of fn.

; We thus support the old style:

; (fn formals val)

; which is deficient because it does not allow the inclusion of a
; stobj other than STATE.  So we also support a generalization of the
; old style:

; (fn formals val :stobjs names)

; that is the most general of the forms supported and allows a
; specification of the formals of fn while also allowing, in a
; context-free sense, the naming of whatever stobjs are required.

; If we do not cause an error, we return (insig . wrld1), where wrld1
; is the world in which we are to perform the constraint of fn and
; insig is of the form:

; (fn formals' stobjs-in stobjs-out)

; where formals' is an appropriately generated arglist.

  (mv-let
   (msg fn formals val stobjs)
   (case-match
    x
    (((fn . pretty-flags1) arrow val)
     (cond
      ((not (and (symbolp arrow) (equal (symbol-name arrow) "=>")))
       (mv (msg *generic-bad-signature-string* x) nil nil nil nil))
      ((not (and (symbol-listp pretty-flags1)
                 (no-duplicatesp-equal
                  (collect-non-x '* pretty-flags1))))
       (mv (msg
            "The object ~x0 is not a legal signature because ~x1 is ~
             not applied to a true-list of distinct symbols but to ~
             ~x2 instead."
            x fn pretty-flags1)
           nil nil nil nil))
      ((not (or (symbolp val)
                (and (consp val)
                     (eq (car val) 'mv)
                     (symbol-listp (cdr val))
                     (no-duplicatesp-equal
                      (collect-non-x '* (cdr val))))))
       (mv (msg
            "The object ~x0 is not a legal signature because the ~
             result, ... => ~x1, is not a symbol or an MV form ~
             containing distinct symbols."
            x val)
           nil nil nil nil))
      ((or (member-eq t pretty-flags1)
           (member-eq nil pretty-flags1)
           (eq val t)
           (eq val nil)
           (and (consp val)
                (or (member-eq t (cdr val))
                    (member-eq nil (cdr val)))))
       (mv (msg
            "The object ~x0 is not a legal signature because it mentions ~
             T or NIL in places that must be filled by asterisks (*) ~
             or single-threaded object names."
            x)
           nil nil nil nil))
      ((not (subsetp-eq (collect-non-x '* (if (consp val)
                                              (cdr val)
                                            (list val)))
                        pretty-flags1))
       (mv (msg
            "The object ~x0 is not a legal signature because the ~
             result, ~x1, refers to one or more single-threaded ~
             objects, ~&2, not displayed among the inputs in ~x3."
            x
            val
            (set-difference-eq (if (consp val)
                                   (cdr val)
                                 (list val))
                               (cons '* pretty-flags1))
            (cons fn pretty-flags1))
           nil nil nil nil))
      (t
       (let* ((formals (gen-formals-from-pretty-flags pretty-flags1))

; Note:  Stobjs will contain duplicates iff formals does.  Stobjs will
; contain STATE iff formals does.

              (stobjs (collect-non-x '* pretty-flags1)))
         (mv nil fn formals val stobjs)))))
    ((fn formals val)
     (cond ((true-listp formals)
            (let ((stobjs (if (member-eq 'state formals) '(state) nil)))
              (mv nil fn formals val stobjs)))
           (t (mv (msg
                   "The object ~x0 is not a legal signature.  It ~
                    appears to be in the form (fn (var1 ... varn) ~
                    val) but is actually of the form (fn (var1 ... . ~
                    varn) val)!"
                   x)
                  nil nil nil nil))))
    ((fn formals val ':stobjs stobjs)
     (cond ((and (true-listp formals)
                 (or (symbolp stobjs)
                     (true-listp stobjs)))
            (let ((stobjs (if (and (member-eq 'state formals)
                                   (not (member-eq 'state
                                                   (if (symbolp stobjs)
                                                       (list stobjs)
                                                     stobjs))))
                              (cons 'state
                                    (if (symbolp stobjs)
                                        (list stobjs)
                                      stobjs))
                            (if (symbolp stobjs)
                                (list stobjs)
                              stobjs))))
              (mv nil fn formals val stobjs)))
           (t (mv (msg
                   "The object ~x0 is not a legal signature, either ~
                    because the proffered formals are not a true-list ~
                    or because the proffered stobj names are ~
                    ill-formed.  The stobj names are expected to be ~
                    either a single symbol or a true list of symbols."
                   x)
                  nil nil nil nil))))
    (& (mv (msg *generic-bad-signature-string* x) nil nil nil nil)))
   (cond
    (msg (er soft ctx "~@0" msg))
    (t
     (er-progn
      (chk-all-but-new-name fn ctx 'constrained-function wrld state)
      (chk-arglist formals t ctx wrld state)
      (chk-all-stobj-names stobjs
                           (msg "~x0" x)
                           ctx wrld state)
      (er-let*
        ((wrld1 (chk-just-new-name fn 'function nil ctx wrld state)))
        (er-progn
         (cond ((not (or (symbolp val)
                         (and (consp val)
                              (eq (car val) 'mv)
                              (symbol-listp (cdr val))
                              (> (length val) 2))))
                (er soft ctx
                    "The purported signature ~x0 is not a legal ~
                      signature because ~x1 is not a legal output ~
                      description.  Such a description should either ~
                      be a symbol or of the form (mv sym1 ... symn), ~
                      where n>=2."
                    x val))
               (t (value nil)))
         (let* ((syms (cond ((symbolp val) (list val))
                            (t (cdr val))))
                (stobjs-in (compute-stobj-flags formals
                                                stobjs
                                                wrld))
                (stobjs-out (compute-stobj-flags syms
                                                 stobjs
                                                 wrld)))
           (cond
            ((not (subsetp (collect-non-x nil stobjs-out)
                           (collect-non-x nil stobjs-in)))
             (er soft ctx
                 "It is impossible to return single-threaded objects ~
                 (such as ~&0) that are not among the formals!  Thus, ~
                 the input signature ~x1 and the output signature ~x2 ~
                 are incompatible."
                 (set-difference-eq (collect-non-x nil stobjs-out)
                                    (collect-non-x nil stobjs-in))
                 formals
                 val))
            ((not (no-duplicatesp (collect-non-x nil stobjs-out)))
             (er soft ctx
                 "It is illegal to return the same single-threaded ~
                 object in more than one position of the output ~
                 signature.  Thus, ~x0 is illegal because ~&1 ~
                 ~#1~[is~/are~] duplicated."
                 val
                 (duplicates (collect-non-x nil stobjs-out))))
            (t (value (cons (list fn
                                  formals
                                  stobjs-in
                                  stobjs-out)
                            wrld1))))))))))))

(defun chk-signatures (signatures ctx wrld state)

; We return a pair containing the list of internal signatures and the
; final wrld in which we are to do the introduction of these fns, or
; else cause an error.

  (cond ((atom signatures)
         (cond ((null signatures) (value (cons nil wrld)))
               (t (er soft ctx
                      "The list of the signatures of the functions ~
                       constrained by an encapsulation is supposed to ~
                       be a true list, but yours ends in ~x0.  See ~
                       :DOC encapsulate."
                      signatures))))
        ((and (consp (cdr signatures))
              (symbolp (cadr signatures))
              (equal (symbol-name (cadr signatures)) "=>"))

; This clause is meant as an optimization helpful to the user.  It is
; an optimization because if we didn't have it here we would proceed
; to apply chk-signature first the (car signatures) -- which will
; probably fail -- and then to '=> -- which would certainly fail.
; These error messages are less understandable than the one we
; generate here.

         (er soft ctx
             "The signatures argument of ENCAPSULATE is supposed to ~
              be a list of signatures.  But you have provided ~x0, ~
              which might be a single signature.  Try writing ~x1."
             signatures
             (list signatures)))


        (t (er-let* ((pair1 (chk-signature (car signatures)
                                           ctx wrld state))
                     (pair2 (chk-signatures (cdr signatures)
                                            ctx (cdr pair1) state)))
                    (let ((insig (car pair1))
                          (insigs (car pair2))
                          (wrld1 (cdr pair2)))
                      (cond ((assoc-eq (car insig) insigs)
                             (er soft ctx
                                 "The name ~x0 is mentioned twice in the ~
                                  signatures of this encapsulation. See :DOC ~
                                  encapsulate."
                                 (car insig)))
                            (t (value (cons (cons insig insigs) wrld1)))))))))

(defun chk-acceptable-encapsulate1 (signatures form-lst ctx wrld state)

; This function checks that form-lst is a plausible list of forms to evaluate
; and that signatures parses into a list of function signatures for new
; function symbols.  We return the internal signatures and the world in which
; they are to be introduced, as a pair (insigs . wrld1).  This function is
; executed before the first pass of encapsulate.

  (er-progn
   (cond ((not (and (true-listp form-lst)
                    (consp form-lst)
                    (consp (car form-lst))))

; Observe that if the car is not a consp then it couldn't possibly be an
; event.  We check this particular case because we fear the user might get
; confused and write an explicit (progn expr1 ...  exprn) or some other
; single expression and this will catch all but the open lambda case.

          (er soft ctx
              "The arguments to encapsulate, after the first, are ~
               each supposed to be embedded event forms.  There must ~
               be at least one form.  See :DOC encapsulate and :DOC ~
               embedded-event-form."))
         (t (value nil)))
   (chk-signatures signatures ctx wrld state)))

; The following is a complete list of the macros that are considered
; "primitive event macros".  This list includes every macro that calls
; install-event except for defpkg, which is omitted as
; explained below.  In addition, the list includes defun (which is
; just a special call of defuns).  Every name on this list has the
; property that while it takes state as an argument and possibly
; changes it, the world it produces is a function only of the world in
; the incoming state and the other arguments.  The function does not
; change the world as a function of, say, some global variable in the
; state.

; The claim above, about changing the world, is inaccurate for include-book!
; It changes the world as a function of the contents of some arbitrarily
; named input object file.  How this can be explained, I'm not sure.

; All event functions have the property that they install into state
; the world they produce, when they return non-erroneously.  More
; subtly they have the property that when the cause an error, they do
; not change the installed world.  For simple events, such as DEFUN
; and DEFTHM, this is ensured by not installing any world until the
; final STOP-EVENT.  But for compound events, such as ENCAPSULATE and
; INCLUDE-BOOK, it is ensured by the more expensive use of
; REVERT-WORLD-ON-ERROR.

(defconst *primitive-event-macros*

; Warning:  See the warnings below!

  '(defun
     #+:non-standard-analysis
     defun-std
     mutual-recursion
     defuns
     defthm
     #+:non-standard-analysis
     defthm-std
     defaxiom
     defconst
     defstobj
;    defpkg                   ; We prohibit defpkgs except in very
                              ; special places.  See below.
     deflabel
     defdoc
     deftheory
     defchoose
     verify-guards
     defmacro
     in-theory
     in-arithmetic-theory
     push-untouchable
     table
     encapsulate
     include-book
     theory-invariant
     verify-termination
     logic program
     add-match-free-override
     add-include-book-dir
     set-match-free-default
     set-verify-guards-eagerness
     set-non-linearp
     set-compile-fns set-measure-function set-well-founded-relation
     set-invisible-fns-table
     set-bogus-mutual-recursion-ok
     set-irrelevant-formals-ok
     set-ignore-ok
     set-inhibit-warnings set-state-ok
     set-let*-abstractionp
     set-nu-rewriter-mode
     set-case-split-limitations
     set-default-hints
     set-rewrite-stack-limit))

; Warning: If a symbol is on this list then it is allowed into books.
; If it is allowed into books, it will be compiled.  Thus, if you add a
; symbol to this list you must consider how compile will behave on it
; and what will happen when the .o file is loaded.  Most of the symbols
; on this list have #-acl2-loop-only definitions that make them
; no-ops.  At least one, defstub, expands into a perfectly suitable
; form involving the others and hence inherits its expansion's
; semantics for the compiler.

; Warning: If this list is changed, inspect the following definitions,
; down through CHK-EMBEDDED-EVENT-FORM.  Also consider modifying the
; list *fmt-ctx-spacers* as well.

; We define later the notion of an embedded event.  Only such events
; can be included in the body of an ENCAPSULATE or a file named by
; INCLUDE-BOOK.

; We do not allow defpkg as an embedded event.  In fact, we do not allow
; defpkg anywhere in a blessed set of files except in files that contain
; nothing but top-level defpkg forms (and those files must not be compiled).
; The reason is explained in deflabel embedded-event-form below.

; Once upon a time we allowed in-package expressions inside of
; encapsulates, in a "second class" way.  That is, they were not
; allowed to be hidden in LOCAL forms.  But the whole idea of putting
; in-package expressions in encapsulated event lists is silly:
; In-package is meant to change the package into which subsequent
; forms are read.  But no reading is being done by encapsulate and the
; entire encapsulate event list is read into whatever was the current
; package when the encapsulate was read.

; Here is an example of why in-package should never be hidden (i.e.,
; in LOCAL), even in a top-level list of events in a file.

; Consider the following list of events:

; (DEFPKG ACL2-MY-PACKAGE '(DEFTHM SYMBOL-PACKAGE-NAME EQUAL))

; (LOCAL (IN-PACKAGE "ACL2-MY-PACKAGE"))

; (DEFTHM GOTCHA (EQUAL (SYMBOL-PACKAGE-NAME 'IF) "ACL2-MY-PACKAGE"))

; When processed in pass 1, the IN-PACKAGE is executed and thus
; the subsequent form (and hence the symbol 'IF) is read into package
; ACL2-MY-PACKAGE.  Thus, the equality evaluates to T and GOTCHA is a
; theorem.  But when processed in pass 2, the IN-PACKAGE is not
; executed and the subsequent form is read into the "ACL2" package.  The
; equality evaluates to NIL and GOTCHA is not a theorem.

(deflabel embedded-event-form
  :doc
  ":Doc-Section Miscellaneous

  forms that may be embedded in other ~il[events]~/
  ~bv[]
  Examples:
  (defun hd (x) (if (consp x) (car x) 0))
  (local (defthm lemma23 ...))
  (progn (defun fn1 ...)
         (local (defun fn2 ...))
         ...)~/

  General Form:
  An embedded event form is a term, x, such that

    x is a call of an event function other than DEFPKG (see the
    documentation for ~il[events] for a listing of the event functions);

    x is of the form (LOCAL x1) where x1 is an embedded event form;

    x is of the form (WITH-OUTPUT ... x1) where x1 is an embedded event form;

    x is of the form (PROGN x1 ... xn), where each xi is an embedded
    event form;

    x is of the form (VALUE-TRIPLE &), where & is any term;

    x macroexpands to one of the forms above.
  ~ev[]
  An exception: an embedded event form may not set the
  ~ilc[acl2-defaults-table] when in the context of ~ilc[local].  Thus for example,
  the form
  ~bv[]
  (local (table acl2-defaults-table :defun-mode :program))
  ~ev[]
  is not an embedded event form, nor is the form ~c[(local (program))],
  since the latter sets the ~ilc[acl2-defaults-table] implicitly.  An
  example at the end of the discussion below illustrates why there is
  this restriction.

  When an embedded event is executed while ~ilc[ld-skip-proofsp] is
  ~c[']~ilc[include-book], those parts of it inside ~ilc[local] forms are ignored.
  Thus,
  ~bv[]
     (progn (defun f1 () 1)
            (local (defun f2 () 2))
            (defun f3 () 3))
  ~ev[]
  will define ~c[f1], ~c[f2], and ~c[f3] when ~ilc[ld-skip-proofsp] is ~c[nil] but will
  define only ~c[f1] and ~c[f3] when ~ilc[ld-skip-proofsp] is ~c[']~ilc[include-book].

  ~em[Discussion:]

  ~ilc[Encapsulate] and ~ilc[include-book] place restrictions on the kinds of
  forms that may be processed.  These restrictions ensure that the
  non-local ~il[events] (which will ultimately be processed with
  ~c[ld-skip-proofs] ~c[t]) are indeed admissible provided that the sequence
  of ~il[local] and non-local ~il[events] is admissible when ~c[ld-skip-proofs] is
  ~c[nil].

  ~ilc[Local] permits the hiding of an event or group of ~il[events] in the sense
  that ~il[local] ~il[events] are processed when we are trying to establish the
  admissibility of a sequence of embedded ~il[events] but are ignored when
  we are constructing the ~il[world] produced by assuming that sequence.
  Thus, for example, a particularly ugly and inefficient ~c[:]~ilc[rewrite] rule
  might be made ~il[local] to an ~il[encapsulate] that ``exports'' a desirable
  theorem whose proof requires the ugly lemma.

  To see why we can't allow just anything in as an embedded event,
  consider allowing the form
  ~bv[]
  (if (ld-skip-proofsp state)
      (defun foo () 2)
      (defun foo () 1))
  ~ev[]
  followed by
  ~bv[]
  (defthm foo-is-1 (equal (foo) 1)).
  ~ev[]
  When we process the ~il[events] with ~ilc[ld-skip-proofsp], ~c[nil] the second
  ~ilc[defun] is executed and the ~ilc[defthm] succeeds.  But when we process the
  ~il[events] with ~ilc[ld-skip-proofsp] ~c[']~ilc[include-book], the second ~ilc[defun] is
  executed, so that ~c[foo] no longer has the same definition it did when
  we proved ~c[foo-is-1].  Thus, an invalid formula is assumed when we
  process the ~ilc[defthm] while skipping proofs.  Thus, the first form
  above is not a legal embedded event form.

  ~ilc[Defpkg] is not allowed because it affects how things are read after
  it is executed.  But all the forms embedded in an event are read
  before any are executed.  That is,
  ~bv[]
  (encapsulate nil
               (defpkg \"MY-PKG\" nil)
               (defun foo () 'my-pkg::bar))
  ~ev[]
  makes no sense since ~c[my-pkg::bar] must have been read before the
  ~ilc[defpkg] for ~c[\"MY-PKG\"] was executed.

  Finally, let us elaborate on the restriction mentioned earlier
  related to the ~ilc[acl2-defaults-table].  Consider the following form.
  ~bv[]
  (encapsulate
   ()
   (local (program))
   (defun foo (x)
     (if (equal 0 x)
         0
       (1+ (foo (- x))))))
  ~ev[]
  ~l[local-incompatibility] for a discussion of how ~ilc[encapsulate]
  processes event forms.  Briefly, on the first pass through the
  ~il[events] the definition of ~c[foo] will be accepted in ~ilc[defun] mode
  ~c[:]~ilc[program], and hence accepted.  But on the second pass the form
  ~c[(local (program))] is skipped because it is marked as ~il[local], and
  hence ~c[foo] is accepted in ~ilc[defun] mode ~c[:]~ilc[logic].  Yet, no proof has been
  performed in order to admit ~c[foo], and in fact, it is not hard to
  prove a contradiction from this definition!~/")

; One can imagine adding new event forms.  The requirement is that
; either they not take state as an argument or else they not be
; sensitive to any part of state except the current ACL2 world.

(defun name-introduced (trip functionp)

; Trip is a triple from a world alist.  We seek to determine whether
; this triple introduces a new name, and if so, which name.  We return
; the name or nil.  If functionp is T we only return function names.
; That is, we return nil if the name introduced is not the name of a
; function, e.g., is a theorem or constant.  Otherwise, we return any
; logical name introduced.  The event functions are listed below.
; Beside each is listed the triple that we take as the unique
; indication that that event introduced name.  Only those having
; FORMALS are considered to be function names.

; event function            identifying triple

; defun-fn                   (name FORMALS . &)
; defuns-fn                  (name FORMALS . &)
; defthm-fn                  (name THEOREM . &)
; defaxiom-fn                (name THEOREM . &)
; defconst-fn                (name CONST . &)
; defstobj-fn                (name STOBJ . names)
;                                [Name is a single-threaded
;                                 object, e.g., $st, and has the
;                                 associated recognizers, accessors
;                                 and updaters.  But those names are
;                                 considered introduced by their
;                                 associated FORMALS triples.]
; deflabel-fn                (name LABEL . T)
; defdoc-fn                  ---
; deftheory-fn               (name THEORY . &)
; defchoose-fn               (name FORMALS . &)
; verify-guards-fn           ---
; defmacro-fn                (name MACRO-BODY . &)
; in-theory-fn               ---
; in-arithmetic-theory-fn    ---
; push-untouchable-fn        ---
; table-fn                   ---
; encapsulate-fn             --- [However, the signature functions
;                                 are introduced with (name FORMALS . &)
;                                 and those names, along with any others
;                                 introduced by the embedded events, are
;                                 returned.]
; include-book-fn            (CERTIFICATION-TUPLE GLOBAL-VALUE 
;                              ("name" "user name" "short name" . chk-sum))

; Those marked "---" introduce no names.

; If redefinition has occurred we have to avoid being fooled by trips such
; as (name FORMALS . *acl2-property-unbound*) and
; (name THEOREM . *acl2-property-unbound*).

  (cond ((eq (cddr trip) *acl2-property-unbound*)
         nil)
        ((eq (cadr trip) 'formals)
         (car trip))
        (functionp nil)
        ((member-eq (cadr trip) '(theorem const macro-body label theory stobj))
         (car trip))
        ((and (eq (car trip) 'certification-tuple)
              (eq (cadr trip) 'global-value)
              (cddr trip))

; The initial value of 'certification-tuple is nil (see initialize-
; world-globals) so we filter it out.  Observe that name is a string
; here.  This name is not the name that occurs in the include-book
; event -- that name is called "user name" in the identifying triple
; column above -- but is in fact the full name of the book, complete
; with the current-book-directory.

         (car (cddr trip)))
        (t nil)))

(defconst *chk-embedded-event-form-portcullisp-string*
  "The command ~x0, used in the construction of the current world, cannot be ~
   included in the portcullis of a certified book~@1.  See :DOC portcullis.")

(defconst *chk-embedded-event-form-normal-string*
  "The form ~x0 is not an embedded event form.  See :DOC embedded-event-form.")

(defconst *chk-embedded-event-form-in-local-string*
  "The form ~x0 is not an embedded event form in the context of LOCAL~@1.  See ~
   :DOC embedded-event-form.")

(defconst *chk-embedded-event-form-in-encapsulate-string*
  "The form ~x0 is not an embedded event form in the context of ~
   ENCAPSULATE~@1.  See :DOC embedded-event-form.")

(mutual-recursion

(defun chk-embedded-event-form
  (form wrld ctx state names portcullisp in-local-flg in-encapsulatep)

; Note:  For a test of this function, see the reference to foo.lisp below.

; This function checks that form is a tree whose tips are calls of
; the symbols listed in names, and whose interior nodes are each of one of
; the forms:

; (local &)
; (progn & ... &)
; (value-triple #)
; (skip-proofs &)
; (with-output ... &)

; where each & is checked.  The # forms above are unrestricted.

; In addition, if portcullisp is t we are checking that the forms are
; acceptable as the portcullis of some book and we enforce the additional
; restriction noted below.

;   (local &) is illegal because such a command would be skipped
;   when executing the portcullis during the subsequent include-book.

; Formerly we also checked here that include-book is only applied to
; absolute pathnames.  That was important for insuring that the book
; that has been read into the certification world is not dependent
; upon :cbd.  Remember that (include-book "file") will find its way
; into the portcullis of the book we are certifying and there is no
; way of knowing in the portcullis which directory that book comes
; from if it doesn't explicitly say.  However, we now use
; fix-portcullis-cmds to modify include-book forms that use relative
; pathnames so that they use absolute pathnames instead, or cause an
; error trying.

; We allow defaxioms and skip-proofs in the portcullis, but we mark the
; book's certificate appropriately.

; If in-local-flg is t, we enforce the restriction that (table
; acl2-defaults-table ...) is illegal, even if table is among names,
; because we do not permit acl2-defaults-table to be changed locally.
; Similarly, defun-mode events and set-compile-fns events are illegal.  (We
; used to make these restrictions when portcullisp is t, because we
; restored the initial acl2-defaults-table before certification, and
; hence it was misguided for the user to be setting the defun-mode or the
; compile flag in the certification world since they were irrelevant
; to the world in which the certification is done.)

; Moreover, we do not allow local defaxiom events.  Imagine locally including a
; book that has nil as a defaxiom.  You can prove anything you want in your
; book, and then when you later include the book, there will be no trace of the
; defaxiom in your logical world!

; We do not check that the tips are well-formed calls of the named functions
; (though we do ensure that they are all true lists).

; If names is *primitive-event-macros* and form can be translated and
; evaluated without error, then it is in fact an embedded event form
; as described in :DOC embedded-event-form.

; We sometimes call this function with names extended by the addition
; of 'DEFPKG.

; If form is rejected, the error message is that printed by str, with
; #\0 bound to the subform (of form) that was rejected.

  (let* ((str (if portcullisp
                  *chk-embedded-event-form-portcullisp-string*
                  *chk-embedded-event-form-normal-string*))
         (local-str *chk-embedded-event-form-in-local-string*))
    (cond ((or (atom form)
               (not (symbolp (car form)))
               (not (true-listp (cdr form))))
           (er soft ctx str form ""))
          ((and in-local-flg (eq (car form) 'defaxiom))
           (er soft ctx local-str form
               " because it adds an axiom whose traces will disappear!"))
          ((and in-encapsulatep (eq (car form) 'defaxiom))
           (er soft ctx *chk-embedded-event-form-in-encapsulate-string* form
               " because we do not permit defaxiom events in the scope of an ~
                encapsulate."))
          ((and in-encapsulatep
                (not in-local-flg)
                (eq (car form) 'include-book))
           (er soft ctx *chk-embedded-event-form-in-encapsulate-string* form
               " because we do not permit non-local include-book forms in the ~
                scope of an encapsulate.  We fear that such forms will ~
                generate unduly large constraints that will impede the ~
                successful use of :functionally-instantiate lemma instances.  ~
                Consider moving your include-book form outside the ~
                encapsulates, or else making it local."))
          ((and portcullisp
                (eq (car form) 'local))
           (er soft ctx str form
               " because such LOCAL commands are not executed ~
                by include-book."))
          ((member-eq (car form) names)

; Names is often *primitive-event-macros* or an extension, and hence
; contains encapsulate and include-book.  This is quite reasonable,
; since they do their own checking.  And because they restore the
; acl2-defaults-table when they complete, we don't have to worry that
; they are sneaking in a ``local defun-mode.''

           (value nil))
          ((eq (car form) 'progn)
           (chk-embedded-event-form-lst (cdr form) wrld ctx state names
                                        portcullisp in-local-flg
                                        in-encapsulatep))
          ((and (eq (car form) 'local)
                (consp (cdr form))
                (null (cddr form)))
           (cond
            ((and (consp (cadr form))
                  (eq (car (cadr form)) 'table)
                  (consp (cdr (cadr form)))
                  (eq (cadr (cadr form)) 'acl2-defaults-table))
             (er soft ctx local-str (cadr form)
                 " because it sets the acl2-defaults-table.  Note ~
                  that LOCAL is probably not useful here anyhow, ~
                  since the acl2-defaults-table is restored upon ~
                  completion of encapsulate, include-book, and ~
                  certify-book forms; that is, no changes to the ~
                  acl2-defaults-table are exported"))
            ((and (consp (cadr form))
                  (member-eq (car (cadr form))
                             '(logic program
                                     add-match-free-override
                                     add-include-book-dir
                                     set-match-free-default
                                     set-non-linearp
                                     set-verify-guards-eagerness
                                     set-compile-fns
                                     set-measure-function
                                     set-well-founded-relation
                                     set-bogus-mutual-recursion-ok
                                     set-irrelevant-formals-ok
                                     set-ignore-ok
                                     set-inhibit-warnings
                                     set-state-ok
                                     set-backchain-limit
                                     set-default-backchain-limit
                                     set-rewrite-stack-limit
                                     set-let*-abstractionp
                                     set-nu-rewriter-mode
                                     set-case-split-limitations
                                     set-default-hints)))
             (er soft ctx local-str (cadr form)
                 " because it implicitly sets the ~
                  acl2-defaults-table.  Note that LOCAL is probably ~
                  not useful here anyhow, since the ~
                  acl2-defaults-table is restored upon completion of ~
                  encapsulate, include-book, and certify-book forms; ~
                  that is, no changes to the acl2-defaults-table are ~
                  exported"))
            (t
             (chk-embedded-event-form (cadr form) wrld ctx state names
                                      portcullisp t in-encapsulatep))))
          ((and (eq (car form) 'skip-proofs)
                (consp (cdr form))
                (null (cddr form)))
           (pprogn
            (warning$ ctx "Skip-proofs"
                      "ACL2 has encountered a SKIP-PROOFS form, ~x0, in the ~
                       context of a book or an encapsulate event.  Therefore, ~
                       no logical claims may be soundly made in this context.  ~
                       See :DOC SKIP-PROOFS."
                      form)
            (chk-embedded-event-form (cadr form) wrld ctx state names
                                     portcullisp in-local-flg
                                     in-encapsulatep)))
          ((and (eq (car form) 'with-output)
                (consp (cdr form)))

; We let the with-output macro check the details of the form structure.

           (chk-embedded-event-form (car (last form)) wrld ctx state names
                                    portcullisp in-local-flg
                                    in-encapsulatep))
          ((eq (car form) 'value-triple) (value t))
          ((getprop (car form) 'macro-body nil 'current-acl2-world wrld)
           (er-let*
            ((expansion (macroexpand1 form ctx state)))
            (chk-embedded-event-form expansion wrld ctx state names
                                     portcullisp in-local-flg
                                     in-encapsulatep)))
          (t (er soft ctx str form
                 " because it is not an embedded event form")))))

(defun chk-embedded-event-form-lst
  (forms wrld ctx state names portcullisp in-local-flg in-encapsulatep)
  (cond
   ((null forms) (value nil))
   (t (er-progn
       (chk-embedded-event-form (car forms) wrld ctx state names
                                portcullisp in-local-flg in-encapsulatep)
       (chk-embedded-event-form-lst (cdr forms) wrld ctx state names
                                    portcullisp in-local-flg
                                    in-encapsulatep)))))

)

; We have had a great deal of trouble correctly detecting embedded defaxioms!
; Here is a book that tests the cases we've thought of!

; (1) Create the book named bar.lisp containing:

#|
 ; bar.lisp
 (in-package "ACL2")
 (defaxiom bad-ax nil :rule-classes nil)
|#

; (2) (certify-book "bar")
; (3) (u)

; (4) Create the book named baruser.lisp containing:

#|
 ; baruser.lisp
 (in-package "ACL2")
 (include-book "bar")
|#

; (5) (certify-book "baruser")
; (6) (u)

; (7) Create the book named foo.lisp containing:

#|

 (in-package "ACL2")

 (defmacro my-local (x) `(local ,x))

; Here are some alternatives.

;(a1) - this should fail because we check directly
;(local (defaxiom bad-ax nil :rule-classes nil))

;(a21) - this should fail because we check directly
;(my-local (defaxiom bad-ax nil :rule-classes nil))

;(b1) - this fails because "bar" contains an defaxiom and when we do
;      the chk-embedded-event-form on the forms in "bar" we know
;      in-local-flg.
;(local (include-book "bar"))

;(b2)
;(my-local (include-book "bar"))

;(b3)
;(local (include-book "baruser"))

;(c1) - this fails because we don't permit local include-books that
;      defaxiom, just as above
;(encapsulate () (local (include-book "bar")) ;...

;(c2)
;(encapsulate () (my-local (include-book "bar")) ;...

;(d) - this fails because we don't permit local defaxioms
;(encapsulate () (local (defaxiom bad-ax nil :rule-classes nil))

;(e) - this fails because don't permit defaxioms in encapsulates
;(encapsulate () (defaxiom bad-ax nil :rule-classes nil)

;(f) - this fails because we don't permit non-LOCAL include-books.
;(encapsulate () (include-book "bar")

 (defthm bad
   nil
   :rule-classes nil
   :hints (("Goal" :use bad-ax)))

;)                                  ; The encapsulate examples require uncommenting this!

|#

; (5) Modify foo.lisp so as to uncomment, one at a time, each of the forms marked
; (a), (b), (c), etc and then to (certify "foo").  Note that when uncommenting the last
; ones you must also uncomment the final close paren.

; The certification of all of these books should fail!

(defun eval-event-lst (ev-lst in-encapsulatep ctx state)

; This function takes a true list of forms, ev-lst, and successively
; evals each one, cascading state through successive elements.
; However, it insists that each form is an embedded-event-form.  We
; return an error/value/state triple in which the value component is
; irrelevant.

  (let ((channel (proofs-co state)))
    (cond
     ((null ev-lst) (value nil))
     (t (pprogn
         (cond
          ((or (ld-skip-proofsp state)

; If state global inhibit-output-lst includes both 'prove and 'event, then do
; not print the event or the prompt.  Why?

; First, it's worth pointing out for starters that with one trivial exception,
; eval-event-lst is called only by process-embedded-events, which is executed
; on behalf of include-book, encapsulate, defstobj, and certify-book.
; Include-book works fine already, as does the second pass of encapsulate,
; because ld-skip-proofsp inhibits event printing.

; The first pass of encapsulate and defstobj use process-embedded-events to
; process subsidiary events.  Certify-book uses process-embedded-events to
; handle all events in the file.  Now I'll "prove" that in these cases it's
; reasonable to inhibit event (with prompt) printing if and only if both 'prove
; and 'event are inhibited.

; If 'prove is NOT inhibited, then it makes sense to see events subsidiary to
; encapsulate (pass 1) and defstobj, in order to make sense of the proof, and
; it is reasonable to let certify-book continue to print events as it does now.
; If 'event is NOT inhibited, then it makes sense to print all events to be
; proved, even when subsidiary or executed on behalf of certify-book.

; Conversely, if both 'prove and 'event are inhibitied, then it's reasonable
; for us not to see subsidiary events or events printed by certify-book.

               (and
                (member-eq 'event (f-get-global 'inhibit-output-lst state))
                (member-eq 'prove (f-get-global 'inhibit-output-lst state))))
           state)
          (t
           (pprogn (newline channel state)
                   (fms "~@0~sr ~@1~*2~#3~[~q4~/~]~|"
                        (list
                         (cons #\0 (f-get-global 'current-package state))
                         (cons #\1 (defun-mode-prompt-string state))
                         (cons #\2 (list "" ">" ">" ">"
                                         (make-list-ac
                                          (1+ (f-get-global 'ld-level state))
                                          nil nil)))
                         (cons #\3 (if (eq (ld-pre-eval-print state) :never)
                                       1
                                     0))
                         (cons #\4 (car ev-lst))
                         (cons #\r
                               #+:non-standard-analysis "(r)"
                               #-:non-standard-analysis ""))
                        channel state nil))))
         (er-progn
          (chk-embedded-event-form (car ev-lst)
                                   (w state)
                                   ctx state
                                   *primitive-event-macros*
                                   nil
                                   (f-get-global 'in-local-flg state)
                                   in-encapsulatep)
          (mv-let
           (erp trans-ans state)
           (trans-eval (car ev-lst) ctx state)

; If erp is nil, trans-ans is 
; ((nil nil state) . (erp' val' replaced-state))
; because ev-lst contains nothing but embedded event forms.

           (let ((erp-prime (car (cdr trans-ans)))
                 (val-prime (cadr (cdr trans-ans))))
             (cond
              ((or erp erp-prime)

; These two sources of errors might be distinguished as those that
; arise from an unsuccessful attempt to translate and evaluate a form
; (if erp is non-nil) and those from the successful translation and
; evaluation of a form that led to a call of error (if erp' is
; non-nil).

               (mv t nil state))
              (t (pprogn
                  (cond ((ld-skip-proofsp state) state)
                        (t
                         (pprogn (ppr val-prime 0 channel state nil)
                                 (newline channel state))))
                  (eval-event-lst
                   (cdr ev-lst) in-encapsulatep ctx state))))))))))))

; After we have evaluated the event list and obtained wrld2, we
; will scrutinize the signatures and exports to make sure they are
; appropriate.  We will try to give the user as much help as we can in
; detecting bad signatures and exports, since it may take him a while
; to recreate wrld2 after fixing an error.  Indeed, he has already
; paid a high price to get to wrld2 and it is a real pity that we'll
; blow him out of the water now.  The guilt!  It's enough to make us
; think about implementing some sort of interactive version of
; encapsulate, when we don't have anything else to do.

(defun equal-insig (insig1 insig2)

; Suppose insig1 and insig2 are both internal form signatures, (fn
; formals stobjs-in stobjs-out).  We return t if they are ``equal.''
; But by equal we mean only that the fn, stobjs-in and stobjs-out are
; the same.  If the user has declared that fn has formals (x y z) and
; then witnessed fn with a function with formals (u v w), we don't
; care -- as long as the stobjs among the two lists are the same in
; corresponding positions.  But that information is captured in the
; stobjs-in.

  (and (equal (car insig1) (car insig2))
       (equal (caddr insig1) (caddr insig2))
       (equal (cadddr insig1) (cadddr insig2))))

;; RAG - I changed this so that non-classical witness functions are
;; not allowed.  The functions introduced by encapsulate are
;; implicitly taken to be classical, so a non-classical witness
;; function presents a (non-obvious) signature violation.

(defun bad-signature-alist (insigs udf-fns wrld)
  (cond ((null insigs) nil)
        ((member-eq (caar insigs) udf-fns)
         (bad-signature-alist (cdr insigs) udf-fns wrld))
        (t (let* ((declared-insig (car insigs))
                  (fn (car declared-insig))
                  (actual-insig (list fn
                                      (formals fn wrld)
                                      (stobjs-in fn wrld)
                                      (stobjs-out fn wrld))))
             (cond
              ((and (equal-insig declared-insig actual-insig)
                    #+:non-standard-analysis
                    (classical-fn-list-p (list fn) wrld))
               (bad-signature-alist (cdr insigs) udf-fns wrld))
              (t (cons (list fn declared-insig actual-insig)
                       (bad-signature-alist (cdr insigs) udf-fns wrld))))))))

(defmacro if-ns (test tbr fbr ctx)

; This is just (list 'if test tbr fbr), except that we expect test always to be
; false in the standard case.

  #+:non-standard-analysis
  (declare (ignore ctx))
  #-:non-standard-analysis
  (declare (ignore tbr))
  (list 'if
        test
        #+:non-standard-analysis
        tbr
        #-:non-standard-analysis
        `(er hard ,ctx
             "Unexpected intrusion of non-standard analysis into standard ~
              ACL2!  Please contact the implementors.")
        fbr))

(defun tilde-*-bad-insigs-phrase1 (alist)
  (cond ((null alist) nil)
        (t (let* ((fn (caar alist))
                  (dcl-insig (cadar alist))
                  (act-insig (caddar alist)))
             (cons
              (if-ns (equal-insig dcl-insig act-insig)
                     (msg
                      "The signature you declared for ~x0 is ~x1, but ~
                       your local witness for the function is not classical."
                      fn
                      (unparse-signature dcl-insig))
                     (msg
                      "The signature you declared for ~x0 is ~x1, but ~
                       the signature of your local witness for it is ~
                       ~x2."
                      fn
                      (unparse-signature dcl-insig)
                      (unparse-signature act-insig))
                     'tilde-*-bad-insigs-phrase1)
              (tilde-*-bad-insigs-phrase1 (cdr alist)))))))

(defun tilde-*-bad-insigs-phrase (alist)

; Each element of alist is of the form (fn insig1 insig2), where
; insig1 is the internal form of the signature presented by the user
; in his encapsulate and insig2 is the internal form signature of the
; witness.  For each element we print a sentence of the form "The
; signature for your local definition of fn is insig2, but the
; signature you declared for fn was insig1."

  (list "" "~@*" "~@*" "~@*"
        (tilde-*-bad-insigs-phrase1 alist)))

(defun union-eq-cars (alist)
  (cond ((null alist) nil)
        (t (union-eq (caar alist) (union-eq-cars (cdr alist))))))

(defun chk-acceptable-encapsulate2 (insigs wrld ctx state)

; Wrld is a world alist created by the execution of an event list.
; Insigs is a list of internal form function signatures.  We verify
; that they are defined as functions in wrld and have the signatures
; listed.

; This is an odd little function because it may generate more than one
; error message.  The trouble is that this wrld took some time to
; create and yet will have to be thrown away as soon as we find one of
; these errors.  So, as a favor to the user, we find all the errors we
; can.

  (let ((udf-fns (collect-non-function-symbols insigs wrld)))
    (mv-let
     (erp1 val state)
     (cond
      (udf-fns
       (er soft ctx
           "You provided signatures for ~&0, but ~#0~[that function ~
           was~/those functions were~] not defined by the ~
           encapsulated event list.  See :DOC encapsulate."
           (merge-sort-symbol-< udf-fns)))
      (t (value nil)))
     (declare (ignore val))
     (mv-let
      (erp2 val state)
      (let ((bad-sig-alist (bad-signature-alist insigs udf-fns wrld)))
        (cond
         (bad-sig-alist
          (er soft ctx
              "The signature~#0~[~/s~] provided for the ~
               function~#0~[~/s~] ~&0 ~#0~[is~/are~] incorrect.  See ~
               :DOC encapsulate.  ~*1"
              (strip-cars bad-sig-alist)
              (tilde-*-bad-insigs-phrase bad-sig-alist)))
         (t (value nil))))
      (declare (ignore val))
      (mv (or erp1 erp2) nil state)))))

(defun conjoin-into-alist (fn thm alist)

; Alist is an alist that maps function symbols to terms.  Fn is a function
; symbol and thm is a term.  If fn is not bound in alist we add (fn . thm)
; to it.  Otherwise, we change the binding (fn . term) in alist to
; (fn . (if thm term *nil*)).

  (cond ((null alist)
         (list (cons fn thm)))
        ((eq fn (caar alist))
         (cons (cons fn (conjoin2 thm (cdar alist)))
               (cdr alist)))
        (t (cons (car alist) (conjoin-into-alist fn thm (cdr alist))))))

(defun classes-theorems (classes)

; Classes is the 'classes property of some symbol.  We return the list of all
; corollary theorems from these classes.

  (cond
   ((null classes) nil)
   (t (let ((term (cadr (assoc-keyword :corollary (cdr (car classes))))))
        (if term
            (cons term (classes-theorems (cdr classes)))
          (classes-theorems (cdr classes)))))))

(defun constraints-introduced1 (thms fns ans)
  (cond
   ((endp thms) ans)
   ((ffnnamesp fns (car thms))

; We use add-to-set-equal below because an inner encapsulate may have both a
; 'body and 'constraint-lst property, and if 'body has already been put into
; ans then we don't want to include that constraint when we see it here.

    (constraints-introduced1 (cdr thms) fns (add-to-set-equal (car thms) ans)))
   (t (constraints-introduced1 (cdr thms) fns ans))))

(defun new-trips (wrld3 proto-wrld3 seen acc)

; Important:  This function returns those triples in wrld3 that are after
; proto-wrld3, in the same order they have in wrld3. See the comment labeled
; "Important" in the definition of constrained-functions.

; As with the function actual-props, we are only interested in triples
; that aren't superseded by *acl2-property-unbound*.  We therefore do
; not copy to our answer any *acl2-property-unbound* triple or any
; chronologically earlier bindings of the relevant symbol and key!
; That is, the list of triples returned by this function contains no
; *acl2-property-unbound* values and makes it appear as though the
; property list was really erased when that value was stored.

; Note therefore that the list of triples returned by this function
; will not indicate when a property bound in proto-wrld3 becomes
; unbound in wrld3.  However, if a property was stored during the
; production of wrld3 and the subsequently in the production of wrld3
; that property was set to *acl2-property-unbound*, then the property
; is gone from the new-trips returned here.

; Warning: The value of this function is sometimes used as though it
; were the 'current-acl2-world!  It is a legal property list world.
; If it gets into a getprop on 'current-acl2-world the answer is
; correct but slow.  Among other things, we use new-trips to compute
; the ancestors of a definition defined within an encapsulate --
; knowing that functions used in those definitions but defined outside
; of the encapsulate (and hence, outside of new-trips) will be treated
; as primitive.  That way we do not explore all the way back to ground
; zero when we are really just looking for the subfunctions defined
; within the encapsulate.

; Note on this recursion: The recursion below is potentially
; disastrously slow.  Imagine that proto-wrld3 is a list of 10,000
; repetitions of the element e.  Imagine that wrld3 is the extension
; produced by adding 1000 more copies of e.  Then the equal below will
; fail the first 1000 times, but it will only fail after confirming
; that the first 10,000 e's in wrld3 are the same as the corresponding
; ones in proto-wrld3, i.e., the equal will do a root-and-branch walk
; through proto-wrld3 1000 times.  When finally the equal succeeds it
; potentially does another root-and-branch exploration of proto-wrld3.
; However, this worst-case scenario is not likely.  More likely, if
; wrld3 is an extension of proto-wrld3 then the first element of wrld3
; differs from that of proto-wrld3 -- because either wrld3 begins with
; a putprop of a new name or a new list of lemmas or some other
; property.  Therefore, most of the time the equal below will fail
; immediately when the two worlds are not equal.  When the two worlds
; are in fact equal, they will be eq, because wrld3 was actually
; constructed by adding triples to proto-wrld3.  So the equal will
; succeed on its initial eq test and avoid a root-and-branch
; exploration.  This analysis is crucial to the practicality of this
; recursive scheme.  Our worlds are so large we simply cannot afford
; root-and-branch explorations.

  (cond ((equal wrld3 proto-wrld3)
         (reverse acc))
        ((assoc-eq-eq (caar wrld3) (cadar wrld3) seen)
         (new-trips (cdr wrld3) proto-wrld3 seen acc))
        ((eq (cddr (car wrld3)) *acl2-property-unbound*)
         (new-trips (cdr wrld3) proto-wrld3 (cons (car wrld3) seen) acc))
        (t
         (new-trips (cdr wrld3) proto-wrld3 (cons (car wrld3) seen)
                    (cons (car wrld3) acc)))))

(defun constraints-introduced (new-trips fns wrld defs-flg ans)

; Warning:  Keep this function in sync with definitional-constraints.

; New-trips is a list of triples from a property list world, none of them with
; cddr *acl2-property-unbound*.  We return the list of all formulas represented
; in new-trips that mention any function symbol in the list fns, either
; restricted to or excluding definitional (defuns, defchoose) axioms according
; to defs-flg.  We may skip properties such as 'congruences and 'lemmas that
; can only be there if some other property has introduced a formula for which
; the given property's implicit formula is a consequence.  However, we must
; include 'type-prescriptions that were computed under the assumption that a
; given axiom was a total recursive definition.  We can omit 'induction-machine
; because we have checked for subversive inductions.  Actually, a good way to
; look at this is that the only events that can introduce axioms are defuns,
; defthm, encapsulate, defaxiom, and include-book, and we have ruled out the
; last two.  Encapsulate is covered by 'constraint-lst.

; We could probably optimize by avoiding definitions of symbols whose
; 'constraint-lst property has already been seen, since those definitional
; axioms are already included in the 'constraint-lst properties.  However, we
; prefer not to rely this way on the order of properties.

  (cond
   ((endp new-trips) ans)
   (t (constraints-introduced
       (cdr new-trips)
       fns
       wrld
       defs-flg
       (let ((trip (car new-trips)))
         (case (cadr trip)
           (constraint-lst
            (cond
             (defs-flg ans)
             ((symbolp (cddr trip))

; Then the constraint list for (car trip) is held in the 'constraint-lst
; property of (cddr trip).  We know that this kind of "pointing" is within the
; current encapsulate, so it is safe to ignore this property, secure in the
; knowledge that we see the real constraint list at some point.

              ans)
             (t (constraints-introduced1 (cddr trip) fns ans))))
           (theorem
            (cond
             (defs-flg ans)
             ((ffnnamesp fns (cddr trip))
              (add-to-set-equal (cddr trip) ans))
             (t ans)))
           (defchoose-axiom
             (cond
              ((not defs-flg) ans)
              ((member-eq (car trip) fns)
               (add-to-set-equal (cddr trip) ans))
              (t ans)))
           (classes
            (cond
             (defs-flg ans)
             (t (constraints-introduced1
                 (classes-theorems (cddr trip)) fns ans))))
           (body
            (cond
             ((not defs-flg) ans)
             ((member-eq (car trip) fns)
              (add-to-set-equal
               (fcons-term* 'equal
                            (cons-term (car trip) (formals (car trip) wrld))
                            (cddr trip))
               ans))
             (t ans)))
           (type-prescriptions
            (cond
             ((and defs-flg
                   (member-eq (car trip) fns))

; We are only interested in type prescriptions put by the system at defun time.
; (The others are theorems or corollaries of theorems, which we handle
; elsewhere.)  Those type prescriptions have the property that the only
; non-primitive function symbol in their corresponding formulas is the one
; being defined.  Hence, we can catch all of those type prescriptions if we
; simply look for type prescriptions hung on function symbols that belong to
; fns.

              (let* ((tp (find-runed-type-prescription
                          (list :type-prescription (car trip))
                          (cddr trip))))
                (cond
                 ((null tp)
                  ans)
                 (t (add-to-set-equal (access type-prescription tp :corollary)
                                      ans)))))
             (t ans)))
           (otherwise ans)))))))

(defun putprop-constraints (fn constrained-fns constraint-lst wrld3)

; Wrld3 is almost wrld3 of the encapsulation essay.  We have added all the
; exports, but we have not yet stored the 'constraint-lst properties of the
; functions in the signature of the encapsulate.  Fn is the first function
; mentioned in the signature, while constrained-fns includes the others as well
; as all functions that have any function in the signature as an ancestor.  We
; have determined that the common constraint for all these functions is
; constraint-lst, which has presumably been obtained from all the new theorems
; introduced by the encapsulate that mention any functions in (fn
; . constrained-fns).

; We actually store the symbol fn as the value of the 'constraint-lst property
; for every function in constrained-fns.  For fn, we store a 'constraint-lst
; property of constraint-lst.  It is crucial that we store the 'constraint-lst
; property for fn before we store any other 'constraint-lst properties; see the
; comment in constrained-functions-save-one.

; Note that we store a 'constrain-lst property for every function in (fn
; . constrained-fns).  The function constraint-info will find this property
; rather than looking for a 'body or 'defchoose-axiom.

  (putprop-x-lst1 constrained-fns 'constraint-lst fn
                  (putprop fn 'constraint-lst constraint-lst wrld3)))

(deflabel local-incompatibility
  :doc
  ":Doc-Section Miscellaneous

  when non-local ~il[events] won't replay in isolation~/

  Sometimes a ``~il[local] incompatibility'' is reported while attempting
  to embed some ~il[events], as in an ~ilc[encapsulate] or ~ilc[include-book].  This is
  generally due to the use of a locally defined name in a non-local
  event or the failure to make a witnessing definition ~il[local].~/

  ~il[Local] incompatibilities may be detected while trying to execute the
  strictly non-local ~il[events] of an embedding.  For example, ~ilc[encapsulate]
  operates by first executing all the ~il[events] (~il[local] and non-local)
  with ~ilc[ld-skip-proofsp] ~c[nil], to confirm that they are all admissible.
  Then it attempts merely to assume the non-local ones to create the
  desired theory, by executing the ~il[events] with ~ilc[ld-skip-proofsp] set to
  ~c[']~ilc[include-book].  Similarly, ~ilc[include-book] assumes the non-local ones,
  with the understanding that a previously successful ~ilc[certify-book] has
  performed the admissiblity check.

  How can a sequence of ~il[events] admitted with ~ilc[ld-skip-proofsp] ~c[nil] fail
  when ~ilc[ld-skip-proofsp] is ~c[']~ilc[include-book]?  The key observation is that
  in the latter case only the non-local ~il[events] are processed.  The
  ~il[local] ones are skipped and so the non-local ones must not depend
  upon them.

  Two typical mistakes are suggested by the detection of a ~il[local]
  incompatibility: (1) a locally defined function or macro was used in
  a non-~ilc[local] event (and, in the case of ~ilc[encapsulate], was not included
  among the ~il[signature]s) and (2) the witnessing definition of a
  function that was included among the ~il[signature]s of an ~ilc[encapsulate]
  was not made ~ilc[local].

  An example of mistake (1) would be to include among your
  ~il[encapsulate]d ~il[events] both ~c[(local (defun fn ...))] and
  ~c[(defthm lemma (implies (fn ...) ...))].  Observe that ~c[fn] is
  defined locally but a formula involving ~c[fn] is defined
  non-locally.  In this case, either the ~ilc[defthm] should be made
  ~il[local] or the ~ilc[defun] should be made non-local.

  An example of mistake (2) would be to include ~c[(fn (x) t)] among your
  ~il[signature]s and then to write ~c[(defun fn (x) ...)] in your ~il[events],
  instead of ~c[(local (defun fn ...))].

  One subtle aspect of ~ilc[encapsulate] is that if you constrain any member
  of a mutually recursive clique you must define the entire clique
  locally and then you must constrain those members of it you want
  axiomatized non-locally.

  Errors due to ~il[local] incompatibility should never occur in the
  assumption of a fully certified book.  Certification ensures against
  it.  Therefore, if ~ilc[include-book] reports an incompatibility, we
  assert that earlier in the processing of the ~ilc[include-book] a warning
  was printed advising you that some book was uncertified.  If this is
  not the case ~-[] if ~ilc[include-book] reports an incompatibility and there
  has been no prior warning about lack of certification ~-[] please
  report it to us.

  When a ~il[local] incompatibility is detected, we roll-back to the ~il[world]
  in which we started the ~ilc[encapsulate] or ~ilc[include-book].  That is, we
  discard the intermediate ~il[world] created by trying to process the
  ~il[events] skipping proofs.  This is clean, but we realize it is very
  frustrating because the entire sequence of ~il[events] must be processed
  from scratch.  Assuming that the embedded ~il[events] were, once upon a
  time, processed as top-level ~il[command]s (after all, at some point you
  managed to create this sequence of ~il[command]s so that the ~il[local] and
  non-local ones together could survive a pass in which proofs were
  done), it stands to reason that we could define a predicate that
  would determine then, before you attempted to embed them, if ~il[local]
  incompatibilities exist.  We hope to do that, eventually.

  We conclude with a subtle example of ~il[local] incompatibility.  The problem
  is that in order for ~c[foo-type-prescription] to be admitted using the
  specified ~c[:typed-term] ~c[(foo x)], the conclusion ~c[(my-natp (foo x))]
  depends on ~c[my-natp] being a ~il[compound-recognizer].  This is fine on the
  first pass of the ~ilc[encapsulate], during which lemma ~c[my-natp-cr] is
  admitted.  But ~c[my-natp-cr] is skipped on the second pass because it is
  marked ~ilc[local], and this causes ~c[foo-type-prescription] to fail on the
  second pass.
  ~bv[]
  (defun my-natp (x)
    (declare (xargs :guard t))
    (and (integerp x)
         (<= 0 x)))

  (defun foo (x)
    (nfix x))

  (encapsulate
   ()
   (local (defthm my-natp-cr
            (equal (my-natp x)
                   (and (integerp x)
                        (<= 0 x)))
            :rule-classes :compound-recognizer))
   (defthm foo-type-prescription
     (my-natp (foo x))
     :hints ((\"Goal\" :in-theory (enable foo)))
     :rule-classes ((:type-prescription :typed-term (foo x)))))
  ~ev[]")

(defun maybe-install-acl2-defaults-table (acl2-defaults-table ctx state)
  (cond
   ((equal acl2-defaults-table
           (table-alist 'acl2-defaults-table (w state)))
    (value nil))
   (t
    (eval-event-lst
     `((table acl2-defaults-table nil
              ',acl2-defaults-table :clear))
     t ; may as well use a "severe" flag here; it shouldn't matter.
     ctx state))))

(defun in-encapsulatep (embedded-event-lst non-trivp)

; This function determines if we are in the scope of an encapsulate.
; If non-trivp is t, we restrict the interpretation to mean ``in the
; scope of a non-trivial encapsulate'', i.e., in an encapsulate that
; introduces a constrained function symbol.

  (cond
   ((endp embedded-event-lst) nil)
   ((and (eq (car (car embedded-event-lst)) 'encapsulate)
         (if non-trivp
             (cadr (car embedded-event-lst))
           t))
    t)
   (t (in-encapsulatep (cdr embedded-event-lst) non-trivp))))

(defun process-embedded-events
  (caller acl2-defaults-table skip-proofsp pkg ee-entry ev-lst ctx state)

; Warning: This function uses set-w and hence may only be called
; within a revert-world-on-error.  See the statement of policy in
; set-w.

; This function is the heart of the second pass of encapsulate,
; include-book, and certify-book.  Caller is in fact one of the five
; symbols 'encapsulate-pass-1, 'encapsulate-pass-2, 'include-book,
; 'certify-book, or 'defstobj.  Note: There is no function
; encapsulate-pass-1, but it is still a ``caller.''

; Acl2-defaults-table is a legal defaults alist.  That alist is
; installed as the acl2-defaults-table (if it is not already there)
; after executing the events in ev-lst.

; The name ee-entry stands for ``embedded-event-lst'' entry.  It is
; consed onto the embedded-event-lst for the duration of the processing
; of ev-lst.  The length of that list indicates how deep these evs are.
; For example, if the embedded-event-lst were:
;   ((defstobj ...)
;    (encapsulate nil) 
;    (include-book ...)
;    (encapsulate ((p (x y) (nil nil) (nil)) ...)))
; Then the ev-lst is the ``body'' of a defstobj, which occurs in the body of
; an encapsulate, which is in an include-book, which is in an encapsulate.

; The shape of an ee-entry is entirely up to the callers and the customers
; of the embedded-event-lst, with three exceptions:
; (a) the ee-entry must always be a consp;
; (b) if the car of the ee-entry is 'encapsulate then the cadr
;     is the internal form signatures of the functions being constrained; and
; (c) if the car of the ee-entry is 'include-book then the cadr is the
;     full-book-name.
; We refer to the signatures in (b) as insigs below and think of insigs as nil
; for all ee-entries other than encapsulates.

; Ev-lst is the list of alleged events.  Pkg is the value we should
; use for current-package while we are processing the events.  This
; affects how forms are prettyprinted.  It also affects how the prompt
; looks.

; We first extend the current world of state by insigs (if caller is
; 'encapsulate-pass-2) and extend the embedded event list by ee-entry.
; We then extend further by doing each of events in ev-lst while
; ld-skip-proofsp is set to skip-proofsp, checking that they are
; indeed embedded-event-forms.  If that succeeds, we restore
; embedded-event-lst, install the world, and return.

; Since the final world is in STATE, we use the value component of the
; non-erroneous return triple to return the world extended by the
; signatures (and the incremented depth).  That world, called
; proto-wrld3 in the encapsulate essay and below, is useful only for
; computing (via difference) the names introduced by the embedded
; events.

; If an error is caused by the attempt to embed the events, we print a
; warning message explaining and pass the error up.

; The world names used here are consistent with the encapsulate essay.

  (let* ((wrld1 (w state))
         (insigs (if (eq (car ee-entry) 'encapsulate)
                     (cadr ee-entry)
                   nil))
         (old-embedded-event-lst
          (global-val 'embedded-event-lst wrld1))
         (new-embedded-event-lst
          (cons ee-entry old-embedded-event-lst))

; We now declare the signatures of the hidden functions (when we're in
; pass 2 of encapsulate), producing what we here call proto-wrld3.  We
; also extend the embedded event list by ee-entry.  After installing that
; world in state we'll execute the embedded events on it to produce
; the wrld3 of the encapsulation essay.

         (proto-wrld3
          (global-set 'embedded-event-lst new-embedded-event-lst
                      (cond ((eq caller 'encapsulate-pass-2)
                             (intro-udf-lst insigs wrld1))
                            (t wrld1)))))
    (let ((state (set-w 'extension proto-wrld3 state)))
      (er-progn
       (cond ((not (find-non-hidden-package-entry pkg
                                                  (known-package-alist state)))
              (er soft 'in-package
                  "The argument to IN-PACKAGE must be a known package ~
                   name, but ~x0 is not.  The known packages are~*1"
                  pkg
                  (tilde-*-&v-strings
                   '&
                   (strip-non-hidden-package-names (known-package-alist state))
                   #\.)))
             (t (value nil)))

; If we really executed an (in-package-fn pkg state) it would do the
; check above and cause an error if pkg was unknown.  But we just bind
; current-package to pkg (with "unwind protection") and so we have to
; make the check ourselves.

       (mv-let (erp val state)
               (state-global-let*
                ((current-package pkg)
                 (ld-skip-proofsp skip-proofsp))
                (er-progn

; Once upon a time, under the same conditions on caller as shown
; below, we added '(logic) to the front of ev-lst before doing the
; eval-event-lst below.  But if the caller is an include-book inside a
; LOCAL, then the (LOGIC) event at the front is rejected by
; chk-embedded-event-form.  One might wonder whether an erroneous
; ev-lst would have left us in a different state than here.  The
; answer is no.  If ev-lst causes an error, eval-event-lst returns
; whatever the state was at the time of the error and does not do any
; cleanup.  The error is passed up to the revert-world-on-error we
; know is above us, which will undo the (logic) as well as anything
; else we changed.

                 (if (or (eq caller 'include-book)
                         (eq caller 'defstobj))

; The following is equivalent to (logic), without the PROGN (value
; :invisible).  The PROGN is illegal in Common Lisp code because its
; ACL2 semantics differs from its CLTL semantics.  Furthermore, we
; can't write (TABLE acl2-defaults-table :defun-mode :logic) because,
; like PROGN, its CLTL semantics is different.

                     (state-global-let*
                      ((inhibit-output-lst (cons 'summary (@ inhibit-output-lst))))
                      (table-fn 'acl2-defaults-table
                                '(:defun-mode :logic)
                                state
                                '(table acl2-defaults-table :defun-mode :logic)))
 
                   (value nil))
                 (eval-event-lst ev-lst
                                 (in-encapsulatep new-embedded-event-lst nil)
                                 ctx state)
                 (maybe-install-acl2-defaults-table
                  acl2-defaults-table ctx state)))
               (declare (ignore val))
               (cond
                (erp

; The evaluation of the embedded events caused an error.  If
; skip-proofsp is t, then we have a local incompatibility (because we
; know the events were successfully processed while not skipping proofs
; earlier).  If skip-proofsp is nil, we simply have an inappropriate
; ev-lst.

                 (cond
                  ((eq caller 'defstobj)
                   (value (er hard ctx
                              "An error has occurred while DEFSTOBJ ~
                               was defining the supporting functions. ~
                                This is supposed to be impossible!  ~
                               Please report this error to ~
                               moore@cs.utexas.edu.")))
                  (t
                   (pprogn
                    (warning$ ctx nil
                              (cond
                               ((or (eq skip-proofsp nil)
                                    (eq skip-proofsp t))
                                "The attempted ~x0 has failed while ~
                                 trying to establish the ~
                                 admissibility of one of the (local ~
                                 or non-local) forms in ~#1~[the body ~
                                 of the ENCAPSULATE~/the book to be ~
                                 certified~].")
                               ((eq caller 'encapsulate-pass-2)
                                "The error reported above is the ~
                                 manifestation of a local ~
                                 incompatibility.  See :DOC ~
                                 local-incompatibility.  The ~
                                 attempted ~x0 has failed.")
                               (t "The error reported above indicates ~
                                   that this book is incompatible ~
                                   with the current logical world.  ~
                                   The attempted ~x0 has failed."))
                              (if (or (eq caller 'encapsulate-pass-1)
                                      (eq caller 'encapsulate-pass-2))
                                  'encapsulate
                                caller)
                              (if (eq caller 'encapsulate-pass-1) 0 1))
                    (mv t nil state)))))
                (t 

; The evaluation caused no error.  The world inside state is the current one
; (because nothing but events were evaluated and they each install the world).
; Pop the embedded event list and install that world.  We let our caller extend
; it with constraints if that is necessary.  We return proto-wrld3 so the
; caller can compute the difference attributable to the embedded events.  This
; is how the constraints are determined.

                 (let ((state
                        (set-w 'extension
                               (global-set 'embedded-event-lst
                                           old-embedded-event-lst
                                           (w state))
                               state)))
                   (value proto-wrld3)))))))))

(defun constrained-functions (exported-fns sig-fns new-trips)

; New-trips is the list of triples introduced into wrld3 from proto-wrld3,
; where wrld3 is the world created from proto-wrld3 by the second pass of an
; encapsulate, the one in which local events have been skipped.  (See the
; encapsulate essay.)  We return all the functions in exported-fns that,
; according to the world segment represented by new-trips, have a member of
; sig-fns among their ancestors.  We include sig-fns in the result as well.

; Important:  The new-trips needs to be in the same order as in wrld3, because
; of the call of instantiable-ancestors below.

  (cond
   ((endp exported-fns) sig-fns)
   (t (let ((ancestors
             (instantiable-ancestors (list (car exported-fns)) new-trips nil)))
        (cond
         ((intersectp-eq sig-fns ancestors)
          (cons (car exported-fns)
                (constrained-functions (cdr exported-fns) sig-fns new-trips)))
         (t (constrained-functions (cdr exported-fns) sig-fns new-trips)))))))

(defun collect-logicals (names wrld)

; Names is a list of function symbols.  Collect the :logic ones.

  (cond ((null names) nil)
        ((logicalp (car names) wrld)
         (cons (car names) (collect-logicals (cdr names) wrld)))
        (t (collect-logicals (cdr names) wrld))))

(defun exported-function-names (new-trips)
  (cond ((endp new-trips)
         nil)
        (t (let ((new-name (name-introduced (car new-trips) t)))

; Because of the second argument of t, above, new-name is known to be
; a function name.

             (cond (new-name
                    (cons new-name (exported-function-names (cdr new-trips))))
                   (t (exported-function-names (cdr new-trips))))))))

(defun get-unnormalized-bodies (names wrld)
  (cond ((endp names) nil)
        (t (cons (getprop (car names) 'unnormalized-body nil
                          'current-acl2-world wrld)
                 (get-unnormalized-bodies (cdr names) wrld)))))

(defun collect-t-machines (fns wrld seen)

; Given a list of functions, this function partitions it into an
; alist, each pair in which is of the form

; ((fn1 ... fnk) . (t-machine1 ... t-machinek))

; with the proviso that in the case of a non-recursive function fn, the
; pair looks like

; ((fn) . nil)

  (cond ((endp fns) nil)
        ((member-eq (car fns) seen)
         (collect-t-machines (cdr fns) wrld seen))
        (t (let* ((recp (getprop (car fns) 'recursivep nil
                                 'current-acl2-world wrld))
                  (names (if recp
                             recp
                           (list (car fns)))))
             (cond
              (recp
               (cons (cons names
                           (termination-machines names
                                                 (get-unnormalized-bodies
                                                  names
                                                  wrld)))

; We put the entire clique of names into seen, to skip the other members.
; This is actually unnecessary if names is a singleton.

                     (collect-t-machines (cdr fns) wrld (append names seen))))
              (t (cons (cons names nil)
                       (collect-t-machines (cdr fns) wrld seen))))))))

(defun collect-instantiables (fns wrld)
  (cond ((endp fns) nil)
        ((instantiablep (car fns) wrld)
         (cons (car fns) (collect-instantiables (cdr fns) wrld)))
        (t (collect-instantiables (cdr fns) wrld))))

(defun subversivep (fns t-machine)

; See subversive-cliquep for conditions (1) and (2).

  (cond ((endp t-machine) nil)
        (t (or (intersectp-eq fns ; Condition (1)
                              (all-fnnames-lst (access tests-and-call
                                                       (car t-machine)
                                                       :tests)))
               (intersectp-eq fns ; Condition (2)
                              (all-fnnames-lst
                               (fargs (access tests-and-call
                                              (car t-machine)
                                              :call))))

               (subversivep fns (cdr t-machine))))))

(defun subversive-cliquep (fns t-machines)

; Here, fns is a list of functions introduced in an encapsulate.  If
; we are using [Front] to move some functions forward, then fns is the
; list of ones that are NOT moved: they all use the signature
; functions somehow.  T-machines is a list of termination machines for
; some clique of functions defined within the encapsulate.  The clique
; is subversive if some function defined in the clique is has a
; subversive t-machine.

; Intuitively, a t-machine is subversive if its admission depended on
; properties of the witnesses for signature functions.  That is, the
; definition uses signature functions in a way that affects the
; termination argument.

; Technically a t-machine is subversive if some tests-and-call record
; in it has either of the following properties:

; (1) a test mentions a function in fns

; (2) an argument of a call mentions a function in fns.

; Observe that if a clique is not subversive then every test and
; argument to every recursive call uses functions defined outside the
; encapsulate.  If we are in a top-level encapsulate, then a
; non-subversive clique is a ``tight'' clique wrt the functions in the
; initial world of the encapsulate.

  (cond ((endp t-machines) nil)
        (t (or (subversivep fns (car t-machines))
               (subversive-cliquep fns (cdr t-machines))))))

(defun contains-non-trivial-encapsulatep (new-trips)

; We return t if new-trips contains an encapsulate event with
; non-empty signature.  This function is used when we decided whether
; to ``rearrange'' the events inside an encapsulate, i.e., to use the
; theorems [Front] and [Back] of the encapsulate paper to move some
; events outside of the encapsulate.  New-trips is known not to
; contain *acl2-property-unbound* values.

  (cond ((endp new-trips) nil)
        (t (or (let ((trip (car new-trips)))
                 (and (eq (car trip) 'event-landmark)
                      (eq (cadr trip) 'global-value)
                      (eq (access-event-tuple-type (cddr trip))
                          'encapsulate)
                      (cadr (access-event-tuple-form (cddr trip)))))
               (contains-non-trivial-encapsulatep (cdr new-trips))))))

(defun definitional-constraints (fn wrld)

; Warning: Keep this function in sync with constraints-introduced.  We
; return a list of formulas consisting of the defchoose-axiom or the
; definitional equation and type-prescription.

  (let ((formula (formula fn nil wrld))
        (tp (find-runed-type-prescription
             (list :type-prescription fn)
             (getprop fn 'type-prescriptions nil 'current-acl2-world wrld))))

; Formula is either the definining equation of a definition, the
; defchoose-axiom of a defchoose, or nil for a primitive function like
; car or an constrained function (even one with a non-trivial
; constraints-lst).

; Tp is either a type-prescription or nil.  If the former, it is a
; type-prescription computed by the system for a defined function --
; because it has the name of the fn, not some user-supplied name like
; consp-fn.
        
    (cond (formula
           (cond (tp
                  (list formula
                        (access type-prescription tp :corollary)))
                 (t (list formula))))
          (tp (list (access type-prescription tp :corollary)))
          (t nil))))

(defun definitional-constraints-list (fns wrld)
  (cond ((endp fns) nil)
        (t (append (definitional-constraints (car fns) wrld)
                   (definitional-constraints-list (cdr fns) wrld)))))

(defun iteratively-grow-constraint1
  (fns                               ;;; the list of all fns not moved forward
   t-machine-alist                   ;;; a tail of collect-t-machines output
   formula-lst                       ;;; the list of constraints so far
   fns-in-formula-lst                ;;; all fns ``involved'' in formula-lst
                                     ;;; (including ancestors of involved fns)
   subversive-fns                    ;;; subversive members of fns, so far
   new-trips                         ;;; the new trips added by this encap
                                     ;;;  used to compute ancestors of defs
   wrld                              ;;; the world, starting with new-trips
   no-action-lst                     ;;; pairs of t-machine-alist already
                                     ;;; processed and thought to be irrelevant
   subversive-enlargementp           ;;; t if we have enlarged the formula with
                                     ;;; a subversive function
   everythingp                       ;;; t if we are to sweep everything into
                                     ;;; the constraint; nil if we can use
                                     ;;; [Back].
   infectious-fns)                   ;;; list of (non-subversive) fns infecting
                                     ;;; the constraint.

  (cond
   ((endp t-machine-alist)
    (cond
     (subversive-enlargementp

; We added to the formula some subversive function and suitably extended
; the fns-in-formula-lst.  But what about the elements of t-machine-alist
; that we had previously processed and found to be irrelevant?  We must
; go around again.

      (iteratively-grow-constraint1
       fns
       no-action-lst
       formula-lst
       fns-in-formula-lst
       subversive-fns
       new-trips
       wrld
       nil
       nil
       everythingp
       infectious-fns))
     (t
      (mv formula-lst                  ;;; final list of constraint formulas
          (intersection-eq             ;;; all fns constrained -- the inter-
           fns-in-formula-lst fns)     ;;;  section eliminates fns from before
                                       ;;;  the encapsulate started
          subversive-fns               ;;; all subversive fns
          infectious-fns               ;;; other fns infecting constraint.
          ))))
   (t (let ((names (car (car t-machine-alist)))
            (t-machines (cdr (car t-machine-alist))))
           
        (cond
         ((null t-machines)

; Names is a singleton and contains a non-recursive function.  It
; cannot be subversive.  The only question is whether this function is
; involved in the constraint.  If so, we want to enlarge the
; constraint.

          (cond
           ((or everythingp
                (member-eq (car names) fns-in-formula-lst))
            (let ((additional-constraints
                   (definitional-constraints (car names) wrld)))
              (iteratively-grow-constraint1
               fns
               (cdr t-machine-alist)
               (union-equal additional-constraints formula-lst)
               (if (member-eq (car names) fns-in-formula-lst)
                   fns-in-formula-lst
                 (instantiable-ffn-symbs-lst additional-constraints
                                             new-trips
                                             fns-in-formula-lst      
                                             nil))
               subversive-fns
               new-trips
               wrld
               no-action-lst
               subversive-enlargementp
               everythingp
               (if additional-constraints
                   (cons (car names) infectious-fns)
                 infectious-fns))))
           (t (iteratively-grow-constraint1
               fns
               (cdr t-machine-alist)
               formula-lst
               fns-in-formula-lst
               subversive-fns
               new-trips
               wrld
               (cons (car t-machine-alist) no-action-lst)
               subversive-enlargementp
               everythingp
               infectious-fns))))
         ((subversive-cliquep fns t-machines)
          (let* ((additional-constraints
                  (definitional-constraints-list names wrld)))
            (iteratively-grow-constraint1
             fns
             (cdr t-machine-alist)
             (union-equal additional-constraints formula-lst)
             (instantiable-ffn-symbs-lst additional-constraints
                                         new-trips
                                         fns-in-formula-lst      
                                         nil)
             (append names subversive-fns)
             new-trips
             wrld
             no-action-lst
             t
             everythingp
             infectious-fns)))
         ((or everythingp
              (intersectp-eq names fns-in-formula-lst))
          (let ((additional-constraints
                 (definitional-constraints-list names wrld)))
            (iteratively-grow-constraint1
             fns
             (cdr t-machine-alist)
             (union-equal additional-constraints formula-lst)
             (if (intersectp-eq names fns-in-formula-lst)
                 fns-in-formula-lst
               (instantiable-ffn-symbs-lst additional-constraints
                                           new-trips
                                           fns-in-formula-lst      
                                           nil))
             subversive-fns
             new-trips
             wrld
             no-action-lst
             subversive-enlargementp
             everythingp
             (if additional-constraints
                 (append names infectious-fns)
               infectious-fns))))
         (t (iteratively-grow-constraint1
               fns
               (cdr t-machine-alist)
               formula-lst
               fns-in-formula-lst
               subversive-fns
               new-trips
               wrld
               (cons (car t-machine-alist) no-action-lst)
               subversive-enlargementp
               everythingp
               infectious-fns)))))))

(defun iteratively-grow-constraint (sig-fns exported-names new-trips wrld)

; Sig-fns is the list of functions appearing in the signature of an
; encapsulate.  Exported-names is the list of all functions introduced
; (non-locally) in the body of the encapsulate (it doesn't include
; sig-fns).  New-trips is the list of property list triples added to
; the initial world to form wrld.  Wrld is the result of processing
; the non-local events in body.

; We return (mv constraints constrained-fns subversive-fns
; infectious-fns), where constraints is a list of the formulas that
; constrain all of the functions listed in constrained-fns.
; Subversive-fns is a list of exported functions which are not tight
; wrt the initial world.  Infectious-fns is the list of fns (other
; than subversives) whose defuns are in the constraint.  This could
; happen either because we were not allowed to rearrange but
; encountered a defun, or because some non-subversive definition is
; ancestral in the constraint.

; We do not actually rearrange anything.   Instead, we compute the constraint
; formula generated by this encapsulate as though we had pulled certain
; defuns and defchooses out before generating it.


  (let* ((rearrange-eventsp
          (and (not (in-encapsulatep
                     (global-val 'embedded-event-lst wrld)
                     t))

; Note: 'embedded-event-lst of this wrld does not include the current
; sig-fns.  That entry was chopped off at the very end of
; process-embedded-events.  The check above ensures that we are not in
; the scope of a non-trivial encapsulate.

               sig-fns
               (not
                (contains-non-trivial-encapsulatep new-trips))))
         (fns 
          (if rearrange-eventsp

; Imagine moving all definitions (defuns and defchooses) that we can,
; so that they are in front of the encapsulate, as described in :DOC
; constraint.  What's left is the list we define here: the function
; symbols introduced by the encapsulate for which the signature
; functions are ancestral.  Fns includes the signature functions.  It
; is empty if sig-fns is empty, i.e., all defuns and defchooses got
; moved up front.

              (constrained-functions
               (collect-logicals exported-names wrld)
               sig-fns
               new-trips)
            (append (collect-logicals exported-names wrld)
                    sig-fns)))
         (formula-lst (constraints-introduced new-trips fns wrld nil nil)))
    (iteratively-grow-constraint1
     fns
     (collect-t-machines fns wrld nil)
     formula-lst
     (instantiable-ffn-symbs-lst formula-lst new-trips sig-fns nil)
     nil
     new-trips
     wrld
     nil
     nil

; The last argument to iterative-grow-constraint1 is everythingp:  It should
; be t if we are not allowed to rearrange things, and should be nil if we
; are allowed to rearrange things.

     (not rearrange-eventsp)
     nil)))

(defun erase-induction-properties (subversive-fns wrld)
  
; We remove the 'induction-machine property of each fn in
; subversive-fns.  In addition, we remove the 'quick-block-info.  This
; makes these functions look just like mutually-recursive functions,
; i.e., of the properties put by put-induction-info, they have only
; 'recursivep, 'justification, and 'symbol-class.

  (putprop-x-lst1 subversive-fns
                  'induction-machine
                  *acl2-property-unbound*
                  (putprop-x-lst1 subversive-fns
                                  'quick-block-info
                                  *acl2-property-unbound*
                                  wrld)))

(defun encapsulate-pass-2 (insigs ev-lst saved-acl2-defaults-table ctx state)

; Warning: This function uses set-w and hence may only be called
; within a revert-world-on-error.  See the statement of policy in
; set-w.

; This is the second pass of the encapsulate event.  We assume that
; the installed world in state is wrld1 of the encapsulate essay.  We
; assume that chk-acceptable-encapsulate1 has approved of wrld1 and
; chk-acceptable-encapsulate2 has approved of the wrld2 generated in
; with ld-skip-proofsp nil.  Insigs is the internal form
; signatures list.  We either cause an error and return a state in
; which wrld1 is current or else we return normally and return a state
; in which wrld3 of the essay is current.  In the case of normal
; return, the value is a list containing

; * constrained-fns - the functions for which a new constraint-lst will
;   be stored

; * constraints - the corresponding list of constraints

; * exported-names - the exported names

; * subversive-fns - the subversive (non-tight) functions encountered

; * infectious-fns - list of (non-subversive) fns whose defun equations were
;   moved into the constraint (possibly because we were not allowed to
;   rearrange).

; This information is used by the output routines.

; Note:  The function could be declared to return five values, but we would
; rather use the standard state and error primitives and so it returns three
; and lists together the three "real" answers.

  (let ((wrld1 (w state)))
    (er-let* ((proto-wrld3

; The following process-embedded-events, which requires world reversion
; on errors, is protected by virtue of being in encapsulate-pass-2, which
; also requires such reversion.

; Note: The proto-wrld3 returned below is wrld1 above extended by the
; signatures.  The installed world after this process-embedded-events
; has the non-local events of ev-lst in it.

               (process-embedded-events 'encapsulate-pass-2
                                        saved-acl2-defaults-table
                                        'include-book
                                        (current-package state)
                                        (list 'encapsulate insigs)
                                        ev-lst ctx state)))
             (let* ((wrld (w state))
                    (new-trips (new-trips wrld proto-wrld3 nil nil))
                    (exported-names (exported-function-names new-trips)))
               (cond
                ((null insigs)
                 (value (list nil nil exported-names)))
                (t

; We are about to collect the constraint generated by this encapsulate
; on the signature functions.  We ``optimize'' one common case: if
; this is a top-level encapsulation that has a non-empty signature (so
; it introduces some constrained functions) and no encapsulate in its
; body introduces any constrained functions, then we may use the
; theorems [Front] and [Back] of the encapsulate paper to
; ``rearrange'' the events within this encapsulate.  If, on the other
; hand, this encapsulate is contained in another or it contains an
; encapsulate with a non-empty signature, we do not rearrange things.
; Of course, the whole point is moot if this encapsulate has an empty
; signature -- there will be no constraints anyway.

                 (let* ((new-trips (new-trips wrld wrld1 nil nil))
                        (sig-fns (strip-cars insigs)))
                   (mv-let
                    (constraints constrained-fns
                                 subversive-fns infectious-fns)
                    (iteratively-grow-constraint sig-fns exported-names
                                                 new-trips wrld)
                    (let ((state
                           (set-w 'extension
                                  (putprop-constraints
                                   (car sig-fns)
                                   (remove1-eq (car sig-fns) constrained-fns)
                                   constraints
                                   (erase-induction-properties subversive-fns
                                                               wrld))
                                  state)))
                      (value (list constrained-fns
                                   constraints
                                   exported-names
                                   subversive-fns
                                   infectious-fns)))))))))))

#|

; Here I have collected a sequence of encapsulates to test the implementation.
; After each is an undo.  They are not meant to co-exist.  Just eval each
; of the forms in this comment.  You should never get an error.

(set-state-ok t)

(defun test (val)
  (declare (xargs :mode :program))
  (if val
      'ok
    (er hard 'test "This example failed!")))
                                            
; I start with a collection of simple encapsulates, primarily to test the
; handling of signatures in their three forms.  I need a stobj.  

(defstobj $s x y)

; Here is a simple, typical encapsulate.
(encapsulate ((p (x) t))
  (local (defun p (x) (declare (ignore x)) t))
  (defthm booleanp-p (booleanp (p x))))

(test
 (equal
  (getprop 'p 'constraint-lst nil 'current-acl2-world (w state))
  '((booleanp (P X)))))

(u)

; The next set just look for errors that should never happen.

(encapsulate (((p *) => *))
             (local (defun p (x) x)))


#|
The following all cause errors.

(encapsulate (((p x) => x))
             (local (defun p (x) x)))

(encapsulate ((p x) => x)
             (local (defun p (x) x)))

(encapsulate (((p x $s) => (mv x $s)))
             (local (defun p (x $s) (declare (xargs :stobjs ($s))) (mv x $s))))

(encapsulate (((p * state $s) => state))
             (local (defun p (x state $s)
                      (declare (xargs :stobjs nil) (ignore x $s))
                      state)))

(encapsulate (((p * state *) => $s))
             (local (defun p (x state $s)
                      (declare (xargs :stobjs $s) (ignore x state))
                      $s)))

; Here are some of the "same" errors provoked in the old notation.


(encapsulate ((p (x $s) (mv * $s) :stobjs *))
             (local (defun p (x $s) (declare (xargs :stobjs ($s))) (mv x $s))))

(encapsulate ((p (* state $s) state))
             (local (defun p (x state $s)
                      (declare (xargs :stobjs nil) (ignore x $s))
                      state)))

(encapsulate ((p (y state $s) $s))
             (local (defun p (x state $s)
                      (declare (xargs :stobjs $s) (ignore x state))
                      $s)))

(encapsulate ((p (x state y) $s))
             (local (defun p (x state $s)
                      (declare (xargs :stobjs $s) (ignore x state))
                      $s)))

(encapsulate ((p (x state $s) $s :stobjs $s))
             (local (defun p (x state $s)
                      (declare (xargs :stobjs $s) (ignore x state))
                      $s)))
|#

; The rest of my tests are concerned with the constraints produced.

; Here is one that contains a function that can be moved forward out
; of encapsulate, even though it is used in the constraint.  Note that
; not every theorem proved becomes a constraint.  The theorem evp-+ is
; moved forward too.

(encapsulate ((p (x) t))
  (local (defun p (x) (declare (ignore x)) 2))
  (defun evp (n) (if (zp n) t (if (zp (- n 1)) nil (evp (- n 2)))))
  (defthm evp-+ (implies (and (integerp i)
                              (<= 0 i)
                              (evp i)
                              (integerp j)
                              (<= 0 j)
                              (evp j))
                         (evp (+ i j))))
  (defthm evp-p (evp (p x))))

(test
 (equal
  (getprop 'p 'constraint-lst nil 'current-acl2-world (w state))
  '((EVP (P X)))))

(u)

; This illustrates a function which uses the signature function p but
; which can be moved back out of the encapsulate.  The only constraint
; on p is (EVP (P X)).

; But if the function involves the constrained function, it cannot
; be moved forward.  It may be moved back, or it may become part of the
; constraint, depending on several things.

; Case 1.  The function uses p in a benign way and nothing is proved
; about the function.

(encapsulate ((p (x) t))
  (local (defun p (x) (ifix x)))
  (defun mapp (x)
    (if (consp x)
        (cons (p (car x)) (mapp (cdr x)))
      nil))
  (defthm integerp-p (integerp (p x))))

(test
 (and (equal (getprop 'p 'constraint-lst nil 'current-acl2-world (w state))
             '((integerp (p x))))
      (equal (getprop 'mapp 'constraint-lst nil 'current-acl2-world (w state))
             nil)))

(u)

; The constraint, above, on p is (INTEGERP (P X)).

; Case 2.  The function is subversive, i.e., uses p in a way critical to
; its termination.

(encapsulate ((p (x) t))
  (local (defun p (x) (cdr x)))
  (defthm len-p (implies (consp x) (< (len (p x)) (len x))))
  (defun bad (x)
    (if (consp x)
        (not (bad (p x)))
      t)))

(test
 (and (equal (getprop 'p 'constraint-lst nil 'current-acl2-world (w state))
             '((EQUAL (BAD X)
                      (IF (CONSP X)
                          (IF (BAD (P X)) 'NIL 'T)
                          'T))
               (IF (EQUAL (BAD X) 'T)
                   'T
                   (EQUAL (BAD X) 'NIL))
               (IMPLIES (CONSP X)
                        (< (LEN (P X)) (LEN X)))))
      (equal (getprop 'bad 'constraint-lst nil 'current-acl2-world (w state))
             'p)))

(u)

; The constraint, above, is 
; (AND (EQUAL (BAD X)
;            (OR (NOT (CONSP X))
;                (AND (NOT (BAD (P X))) T)))
;     (OR (EQUAL (BAD X) T)
;         (EQUAL (BAD X) NIL))
;     (IMPLIES (CONSP X)
;              (< (LEN (P X)) (LEN X))))
; 
; and it is associated both with p and bad.  That is, if you functionally
; instantiate p, the new function must satisfy the axiom for bad too,
; which means you must instantiate bad.  Similarly, if you instantiate
; bad, you must instantiate p.

; It would be better if you did this:

(encapsulate ((p (x) t))
  (local (defun p (x) (cdr x)))
  (defthm len-p (implies (consp x) (< (len (p x)) (len x)))))

(test
 (equal (getprop 'p 'constraint-lst nil 'current-acl2-world (w state))
        '((IMPLIES (CONSP X)
                   (< (LEN (P X)) (LEN X))))))

; The only constraint on p is 
; (IMPLIES (CONSP X) (< (LEN (P X)) (LEN X))).
; Now you can define bad outside:

(defun bad (x)
  (declare (xargs :measure (len x)))
  (if (consp x)
      (not (bad (p x)))
    t))

(u)
(u)

; Case 3.  The function uses p in a benign way but something is proved
; about the function, thus constraining p.

(encapsulate ((p (x) t))
  (local (defun p (x) (ifix x)))
  (defun mapp (x)
    (if (consp x)
        (cons (p (car x)) (mapp (cdr x)))
      nil))
  (defthm mapp-is-a-list-of-ints
    (integer-listp (mapp x))))

(test
 (and (equal (getprop 'p 'constraint-lst nil 'current-acl2-world (w state))
             '((EQUAL (MAPP X)
                      (IF (CONSP X)
                          (CONS (P (CAR X)) (MAPP (CDR X)))
                          'NIL))
               (TRUE-LISTP (MAPP X))
               (INTEGER-LISTP (MAPP X))))
      (equal (getprop 'mapp 'constraint-lst nil 'current-acl2-world (w state))
             'p)))

(u)

; The constraint above, on both p and mapp, is
; (AND (EQUAL (MAPP X)
;             (AND (CONSP X)
;                  (CONS (P (CAR X)) (MAPP (CDR X)))))
;      (TRUE-LISTP (MAPP X))
;      (INTEGER-LISTP (MAPP X)))

; Here is another case of a subversive definition, illustrating that
; we do not just check whether the function uses p but whether it uses
; p ancestrally.

(encapsulate ((p (x) t))
  (local (defun p (x) (cdr x)))
  (defun bad1 (x) (p x))
  (defun bad2 (x)
    (if (consp x)
        (not (bad2 (bad1 x)))
      t)))

(test
 (and (equal (getprop 'p 'constraint-lst nil 'current-acl2-world (w state))
             '((EQUAL (BAD1 X) (P X))
               (EQUAL (BAD2 X)
                      (IF (CONSP X)
                          (IF (BAD2 (BAD1 X)) 'NIL 'T)
                          'T))
               (IF (EQUAL (BAD2 X) 'T)
                   'T
                   (EQUAL (BAD2 X) 'NIL))))
      (equal (getprop 'bad1 'constraint-lst nil 'current-acl2-world (w state))
             'p)
      (equal (getprop 'bad2 'constraint-lst nil 'current-acl2-world (w state))
             'p)
      (equal (getprop 'bad2 'induction-machine nil
                      'current-acl2-world (w state))
             nil)))


(u)

(encapsulate ((p (x) t))
  (local (defun p (x) (cdr x)))
  (defun bad1 (x)
    (if (consp x) (bad1 (cdr x)) (p x)))
  (defun bad2 (x)
    (if (consp x)
        (not (bad2 (bad1 x)))
      t)))

(test
 (and (equal (getprop 'p 'constraint-lst nil 'current-acl2-world (w state))
             '((EQUAL (BAD1 X)
                      (IF (CONSP X)
                          (BAD1 (CDR X))
                          (P X)))
               (EQUAL (BAD2 X)
                      (IF (CONSP X)
                          (IF (BAD2 (BAD1 X)) 'NIL 'T)
                          'T))
               (IF (EQUAL (BAD2 X) 'T)
                   'T
                   (EQUAL (BAD2 X) 'NIL))))
      (equal (getprop 'bad1 'constraint-lst nil 'current-acl2-world (w state))
             'p)
      (equal (getprop 'bad2 'constraint-lst nil 'current-acl2-world (w state))
             'p)
      (not (equal (getprop 'bad1 'induction-machine nil
                           'current-acl2-world (w state))
                  nil))
      (equal (getprop 'bad2 'induction-machine nil
                      'current-acl2-world (w state))
             nil)))

(u)

; Once up a time we had a bug in encapsulate, because subversiveness was
; based on the induction machine rather than the termination machine
; and no induction machine is constructed for mutually recursive definitions.
; Here is an example that once led to unsoundness:

(encapsulate
 ((fn1 (x) t))
 (local (defun fn1 (x)
          (cdr x)))
 (mutual-recursion
  (defun fn2 (x)
    (if (consp x)
        (not (fn3 (fn1 x)))
      t))
  (defun fn3 (x)
    (if (consp x)
        (not (fn3 (fn1 x)))
      t))))

(test
 (and (equal (getprop 'fn1 'constraint-lst nil 'current-acl2-world (w state))
             '((EQUAL (FN2 X)
                      (IF (CONSP X)
                          (IF (FN3 (FN1 X)) 'NIL 'T)
                          'T))
               (IF (EQUAL (FN2 X) 'T)
                   'T
                   (EQUAL (FN2 X) 'NIL))
               (EQUAL (FN3 X)
                      (IF (CONSP X)
                          (IF (FN3 (FN1 X)) 'NIL 'T)
                          'T))
               (IF (EQUAL (FN3 X) 'T)
                   'T
                   (EQUAL (FN3 X) 'NIL))))
      (equal (getprop 'fn2 'constraint-lst nil 'current-acl2-world (w state))
             'fn1)
      (equal (getprop 'fn3 'constraint-lst nil 'current-acl2-world (w state))
             'fn1)
      (equal (getprop 'fn2 'induction-machine nil
                      'current-acl2-world (w state))
             nil)
      (equal (getprop 'fn3 'induction-machine nil
                      'current-acl2-world (w state))
             nil)))

; Now, fn1, fn2, and fn3 share both definitional constraints.

; It is possible to prove the following lemma

(defthm lemma
  (not (equal (fn1 '(a)) '(a)))
  :rule-classes nil
  :hints (("Goal" :use (:instance fn3 (x '(a))))))

; But in the unsound version it was then possible to functionally
; instantiate it, choosing the identity function for fn1, to derive
; a contradiction.  Here is the old killer:

; (defthm bad
;   nil
;   :rule-classes nil
;   :hints (("Goal" :use (:functional-instance lemma (fn1 identity)))))

(u)
(u)

; Now when you do that you have to prove an impossible theorem about
; fn3, namely

; (equal (fn3 x) (if (consp x) (not (fn3 x)) t))

; The only way to prove this is to show that nothing is a cons.

; This examples shows that a function can call a subversive one and
; not be subversive.

(encapsulate ((p (x) t))
  (local (defun p (x) (cdr x)))
  (defun bad1 (x) (p x))            ; tight: non-recursive

  (defun bad2 (x)                   ; not tight: recursive call involves
    (if (consp x)                   ; a fn (bad1) defined inside the encap
        (not (bad2 (bad1 x)))
      t))
  (defun bad3 (x)
    (if (consp x)
        (bad2 (bad3 (cdr x)))
      nil)))                        ; tight: even though it calls bad2

; Bad2 is swept into the constraint because it is not tight (subversive).  Bad1
; is swept into it because it introduces a function (bad1) used in the enlarged
; constraint.  Bad3 is not swept in.  Indeed, bad3 is moved [Back].

(test
 (and (equal (getprop 'p 'constraint-lst nil 'current-acl2-world (w state))
             '((EQUAL (BAD1 X) (P X))
               (EQUAL (BAD2 X)
                      (IF (CONSP X)
                          (IF (BAD2 (BAD1 X)) 'NIL 'T)
                          'T))
               (IF (EQUAL (BAD2 X) 'T)
                   'T
                   (EQUAL (BAD2 X) 'NIL))))
      (equal (getprop 'bad1 'constraint-lst nil 'current-acl2-world (w state))
             'p)
      (equal (getprop 'bad2 'constraint-lst nil 'current-acl2-world (w state))
             'p)
      (equal (getprop 'bad3 'constraint-lst nil 'current-acl2-world (w state))
             nil)
      (equal (getprop 'bad2 'induction-machine nil
                      'current-acl2-world (w state))
             nil)
      (not (equal (getprop 'bad3 'induction-machine nil
                           'current-acl2-world (w state))
                  nil))))

(u)

; Now what about nested encapsulates?

; Let us first consider the two simplest cases:

(encapsulate ((p (x) t))
  (local (defun p (x) (declare (ignore x)) 23))
  (encapsulate nil
     (defthm lemma1 (equal x x) :rule-classes nil)
     (defthm main (equal x x) :rule-classes nil))
  (defthm integerp-p (integerp (p x))))

; We are permitted to rearrange this, because the inner encap has a nil
; signature.  So we get what we expect:

(test
 (equal
  (getprop 'p 'constraint-lst nil 'current-acl2-world (w state))
  '((integerp (P X)))))

(u)

; The other simple case is

(encapsulate nil
   (defthm lemma1 (equal x x) :rule-classes nil)
   (defthm main (equal x x) :rule-classes nil)
   (encapsulate ((p (x) t))
                (local (defun p (x) (declare (ignore x)) 23))
                (defun benign (x)
                  (if (consp x) (benign (cdr x)) x))
                (defthm integerp-p (integerp (p x)))))

; Note that benign doesn't constrain p, because the containing encap
; contains no sig fns.

(test
 (equal
  (getprop 'p 'constraint-lst nil 'current-acl2-world (w state))
  '((integerp (P X)))))

(u)

; But if we have a pair of encaps, each of which introduces a sig fn,
; we lose the ability to rearrange things:

(encapsulate ((p1 (x) t))
             (local (defun p1 (x) x))             
             (defun benign1 (x)
               (if (consp x) (benign1 (cdr x)) t))
             (defthm p1-constraint (benign1 (p1 x)))
             (encapsulate  ((p2 (x) t))
                           (local (defun p2 (x) x))             
                           (defun benign2 (x)
                             (if (consp x) (benign2 (cdr x)) t))
                           (defthm p2-constraint (benign2 (p2 x)))))

(test
 (and (equal (getprop 'p1 'constraint-lst nil 'current-acl2-world (w state))
             '((EQUAL (BENIGN1 X)
                      (IF (CONSP X) (BENIGN1 (CDR X)) 'T))
               (BENIGN1 X)
               (BENIGN1 (P1 X))
               (BENIGN2 (P2 X))
               (BENIGN2 X)
               (EQUAL (BENIGN2 X)
                      (IF (CONSP X) (BENIGN2 (CDR X)) 'T))))
      (equal
       (getprop 'p2 'constraint-lst nil 'current-acl2-world (w state))
       'p1)
      (equal
       (getprop 'benign2 'constraint-lst nil 'current-acl2-world (w state))
       'p1)
      (equal
       (getprop 'benign1 'constraint-lst nil 'current-acl2-world (w state))
       'p1)))

(u)

(encapsulate ((f1 (x) t))
             (local (defun f1 (x) (declare (ignore x)) 0))
             (defun bad (x)
               (if (consp x)
                   (if (and (integerp (bad (cdr x)))
                            (<= 0 (bad (cdr x)))
                            (< (bad (cdr x)) (acl2-count x)))
                       (bad (bad (cdr x)))
                     (f1 x))
                 0)))

(test
 (and (equal (getprop 'f1 'constraint-lst nil 'current-acl2-world (w state))
             '((EQUAL (BAD X)
                      (IF (CONSP X)
                          (IF (INTEGERP (BAD (CDR X)))
                              (IF (< (BAD (CDR X)) '0)
                                  (F1 X)
                                  (IF (< (BAD (CDR X)) (ACL2-COUNT X))
                                      (BAD (BAD (CDR X)))
                                      (F1 X)))
                              (F1 X))
                          '0))))
      (equal
       (getprop 'bad 'constraint-lst nil 'current-acl2-world (w state))
       'f1)
      (equal
       (getprop 'bad 'induction-machine nil 'current-acl2-world (w state))
       nil)))

(u)



; Here is a sample involving defchoose.  In this example, the signature
; function is ancestral in the defchoose axiom.

(encapsulate ((p (y x) t))
             (local (defun p (y x) (member-equal y x)))
             (defchoose witless x (y) (p y x))
             (defthm consp-witless
               (consp (witless y))
               :rule-classes :type-prescription
               :hints (("Goal" :use (:instance witless (x (cons y nil)))))))

(test
 (and (equal (getprop 'p 'constraint-lst nil 'current-acl2-world (w state))
             '((IMPLIES (P Y X)
                        ((LAMBDA (X Y) (P Y X)) (WITLESS Y) Y))
               (CONSP (WITLESS Y))))
      (equal
       (getprop 'witless 'constraint-lst nil 'current-acl2-world (w state))
       'p)
      (equal
       (getprop 'witless 'defchoose-axiom nil 'current-acl2-world (w state))
       '(IMPLIES (P Y X)
                 ((LAMBDA (X Y) (P Y X)) (WITLESS Y) Y)))))

(u)

; and in this one it is not, indeed, the defchoose function can be
; moved to the [Front] even though it is used in the constraint of p.

(encapsulate ((p (y x) t))
             (local (defun p (y x) (member-equal y x)))
             (defchoose witless x (y) (member-equal y x))
             (defthm p-constraint (p y (witless y))
               :hints (("Goal" :use (:instance witless (x (cons y nil)))))))

(test
 (and (equal (getprop 'p 'constraint-lst nil 'current-acl2-world (w state))
             '((p y (witless y))))
      (equal
       (getprop 'witless 'constraint-lst nil 'current-acl2-world (w state))
       nil)
      (equal
       (getprop 'witless 'defchoose-axiom nil 'current-acl2-world (w state))
       '(IMPLIES (member-equal Y X)
                 ((LAMBDA (X Y) (member-equal Y X)) (WITLESS Y) Y)))))

(u)

(quote (the end of my encapsulate tests -- there follow two undo commands))
(u)
(u)

|#


(defun tilde-@-abbreviate-object-phrase (x)

; This function produces a tilde-@ phrase that describes the
; object x, especially if it is a list.  This is just a hack
; used in error reporting.

  (cond ((atom x) (msg "~x0" x))
        ((symbol-listp x)
         (cond ((< (length x) 3)
                (msg "~x0" x))
               (t
                (msg "(~x0 ... ~x1)"
                     (car x)
                     (car (last x))))))
        ((atom (car x))
         (cond ((and (consp (cdr x))
                     (atom (cadr x)))
                (msg "(~x0 ~x1 ...)"
                     (car x)
                     (cadr x)))
               (t
                (msg "(~x0 ...)"
                     (car x)))))
        ((atom (caar x))
         (cond ((and (consp (cdar x))
                     (atom (cadar x)))
                (msg "((~x0 ~x1 ...) ...)"
                     (caar x)
                     (cadar x)))
               (t
                (msg "((~x0 ...) ...)"
                     (caar x)))))
        (t "(((...) ...) ...)")))


(defun encapsulate-ctx (signatures form-lst)

; This function invents a suitable error context, ctx, for an
; encapsulate with the given signatures and form-lst.  The args have
; not been translated or checked.  Thus, this function is rough.
; However, we have to have some way to describe to the user which
; encapsulation is causing the problem, since we envision them often
; being nested.  Our guess is that the signatures, if non-nil, will be
; the most recognizable aspect of the encapsulate.  Otherwise, we'll
; abbreviate the form-lst.

  (cond
   (signatures
    (cond ((and (consp signatures)
                (consp (car signatures))
                (consp (caar signatures)))
           (msg "( ENCAPSULATE (~@0 ...) ...)"
                (tilde-@-abbreviate-object-phrase (car signatures))))
          (t
           (msg "( ENCAPSULATE ~@0 ...)"
                (tilde-@-abbreviate-object-phrase signatures)))))
   (form-lst
    (msg "( ENCAPSULATE NIL ~@0 ...)"
         (tilde-@-abbreviate-object-phrase (car form-lst))))
   (t "( ENCAPSULATE NIL)")))

(defun print-encapsulate-msg1 (insigs form-lst state)
  (declare (ignore insigs))
  (cond
   ((eq (ld-skip-proofsp state) t) state)
   (t
    (io? event nil state
         (fms "To verify that the ~#0~[~/~n1 ~]encapsulated event~#0~[~/s~] ~
               correctly extend~#0~[s~/~] the current theory we will evaluate ~
               ~#0~[it~/them~].  The theory thus constructed is only ~
               ephemeral.~|~#2~[~%Encapsulated Event~#0~[~/s~]:~%~/~]"
              (list (cons #\0 form-lst)
                    (cons #\1 (length form-lst))
                    (cons #\2 (if (eq (ld-pre-eval-print state) :never) 1 0)))
              (proofs-co state)
              state nil)))))

(defun print-encapsulate-msg2 (insigs form-lst state)
  (declare (ignore insigs))
  (cond
   ((eq (ld-skip-proofsp state) t) state)
   (t
    (io? event nil state
         (fms "End of Encapsulated Event~#0~[~/s~].~%"
              (list (cons #\0 form-lst))
              (proofs-co state)
              state nil)))))

(defun print-encapsulate-msg3/exported-names (insigs lst)

; This returns a list of tilde-@ phrases.  The list always has either
; 0 or 1 things in it.  The single element describes the exports of
; an encapsulation (if any).  Insigs is the list of internal form
; signatures of the constrained fns.

  (cond ((null lst)

; Say nothing if there are no additional names.

         nil)
        (insigs
         (list (msg "In addition to ~&0, we export ~&1.~|~%"
                    (strip-cars insigs)
                    lst)))
        (t (list (msg "We export ~&0.~|~%"
                      lst)))))

(defun print-encapsulate-msg3/constraints (constrained-fns constraints wrld)
  (cond
   ((null constraints)

; It's tempting in this case to say something like, "No new constraints are
; associated with any function symbols."  However, one could argue with that
; statement, since DEFUN introduces constraints in some sense, for example.
; This problem does not come up if there are constrained functions, since in
; that case (below), we are honestly reporting all of the constraints on the
; indicated functions.  So, we simply print nothing in the present case.

    nil)
   ((null constrained-fns)
    (er hard 'print-encapsulate-msg3/constraints
        "We had thought that the only way that there can be constraints is if ~
         there are constrained functions.  See ~
         print-encapsulate-msg3/constraints."))
   (t (list
       (msg "The following constraint is associated with ~#0~[the ~
             function~/both of the functions~/every one of the functions~] ~
             ~&1:~|~%~p2~|"
            (let ((n (length constrained-fns)))
              (case n
                    (1 0)
                    (2 1)
                    (otherwise 2)))
            constrained-fns
            (untranslate (conjoin constraints) t wrld))))))

(defun print-encapsulate-msg3 (ctx insigs form-lst exported-names
                                   constrained-fns constraints-introduced
                                   subversive-fns infectious-fns
                                   wrld state)

; This function prints a sequence of paragraphs, one devoted to each
; constrained function (its arities and constraint) and one devoted to
; a summary of the other names created by the encapsulation.

  (cond
   ((eq (ld-skip-proofsp state) t) state)
   (t
    (io? event nil state
         (pprogn
          (fms "Having verified that the encapsulated event~#0~[ ~
                validates~/s validate~] the signatures of the ~
                ENCAPSULATE event, we discard the ephemeral theory ~
                and extend the original theory as directed by the ~
                signatures and the non-LOCAL events.~|~%~*1"
               (list
                (cons #\0 form-lst)
                (cons #\1
                      (list "" "~@*" "~@*" "~@*"
                            (append
                             (print-encapsulate-msg3/exported-names
                              insigs exported-names)
                             (print-encapsulate-msg3/constraints
                              constrained-fns constraints-introduced wrld)
                             ))))
               (proofs-co state)
               state nil)
          (print-defun-msg/signatures (strip-cars insigs) wrld state)
          (if subversive-fns
              (warning$ ctx "Infected"
                        "Note that ~&0 ~#0~[is~/are~] ``subversive.'' ~
                         See :DOC subversive-recursions.  Thus, ~
                         ~#0~[its definitional equation ~
                         infects~/their definitional equations ~
                         infect~] the constraint of this ~
                         en~-cap~-su~-la~-tion.  Furthermore, ~#0~[this ~
                         function~/these functions~] will not suggest ~
                         any induction schemes to the theorem prover. ~
                          If possible, you should remove ~#0~[this ~
                         definition~/these definitions~] from the ~
                         encapsulate and introduce ~#0~[it~/them~] ~
                         afterwards.  A constraint containing a ~
                         definitional equation is often hard to use in ~
                         subsequent functional instantiations."
                        subversive-fns)
            state)
          (if infectious-fns
              (warning$ ctx "Infected"
                        "Note that the definitional ~
                         equation~#0~[~/s~] for ~&0 infect~#0~[s~/~] ~
                         the constraint of this ~
                         en~-cap~-su~-la~-tion.  That can be caused ~
                         either because we are not allowed to move a ~
                         defun out of nested non-trivial encapsulates ~
                         or because a function ancestrally involves ~
                         the constrained functions of an encapsulate ~
                         and is ancestrally involved in the ~
                         constraining theorems of those functions. In ~
                         any case, if at all possible, you should ~
                         move ~#0~[this definition~/these ~
                         definitions~] out of the encapsulation.  A ~
                         constraint containing a definitional ~
                         equation is often hard to use in subsequent ~
                         functional instantiations.  See ~
                         :DOC subversive-recursions for a discussion of ~
                         related issues."
                        infectious-fns)
            state))))))

(defun redundant-event-tuplep (event-form mode wrld)

; We return t iff the non-prehistoric (if that's where we start) part of wrld
; contains an event-tuple whose form is equal to event-form.

  (cond ((or (null wrld)
             (and (eq (caar wrld) 'command-landmark)
                  (eq (cadar wrld) 'global-value)
                  (equal (access-command-tuple-form (cddar wrld))
                         '(exit-boot-strap-mode))))
         nil)
        ((and (eq (caar wrld) 'event-landmark)
              (eq (cadar wrld) 'global-value)
              (equal (access-event-tuple-form (cddar wrld))
                     event-form))
         (eq (default-defun-mode wrld) mode))
        (t (redundant-event-tuplep event-form mode (cdr wrld)))))

(mutual-recursion

(defun find-first-non-local-name (x)

; X is allegedly an embedded event form.  It may be a call of some
; user macro and thus completely unrecognizable to us.  But it could
; be a call of one of our primitive fns.  We are interested in the
; question "If x is successfully executed, what is a logical name it
; will introduce?"  Since no user event will introduce nil, we use nil
; to indicate that we don't know about x (or, equivalently, that it is
; some user form we don't recognizer, or that it introduces no names,
; or that it is ill-formed and will blow up).  Otherwise, we return a
; logical name that x will create.

  (case-match x
              (('local . &) nil)
              (('defun name . &) name)
              (('defuns (name . &) . &) name)
              (('defthm name . &) name)
              (('defaxiom name . &) name)
              (('skip-proofs ev) (find-first-non-local-name ev))
              (('defconst name . &) name)
              (('deflabel name . &) name)
              (('deftheory name . &) name)
              (('defstobj name . &) name)
              (('defmacro name . &) name)
              (('mutual-recursion ('defun name . &) . &) name)
              (('encapsulate (((name . &) arrow &) . &) . &)
               (and (symbolp arrow)
                    (equal (symbol-name arrow) "=>")
                    name))
              (('encapsulate ((name . &) . &) . &) name)
              (('encapsulate nil . ev-lst)
               (find-first-non-local-name-lst ev-lst))
              (('include-book name . &) name)
              (& nil)))               

(defun find-first-non-local-name-lst (lst)

; Challenge: If lst is a true list of embedded event forms that is
; successfully processed with ld-skip-proofsp nil, name one name that
; would be created.  Now lst might not be a list of embedded event
; forms.  Or the forms might be doomed to cause errors or might be
; unrecognizable user macro calls.  So we return nil if we can't spot a
; suitable name.  Otherwise we return a name.  The only claim made is
; this: if we return non-nil and lst were successfully processed, then
; that name is a logical name that would be created.  Consequently, if
; that name is new in a world, we know that this lst has not been
; processed before.

  (cond ((atom lst) nil)
        (t (or (find-first-non-local-name (car lst))
               (find-first-non-local-name-lst (cdr lst))))))

)

(defun redundant-encapsulatep (signatures ev-lst event-form wrld)

; We wish to know whether (redundant-event-tuplep event-form wrld) is
; t or nil.  That is, is there an event-tuple in wrld that has
; event-form as its form?  We know, however, that event-form is really
; an encapsulate with the given two arguments.  We don't know if event-form
; will execute without error -- i.e., the args may be screwed up.
; But suppose we could find a name among signatures and ev-lst that is
; guaranteed to be created if event-form were successful.  Then if that
; name is new, we know we won't find event-form in wrld and needn't bother
; looking.  If we can't find a name -- e.g., because signatures is
; nil and all the events in ev-lst are user macros -- or if we do find
; a name but it is not new, then we suffer the search through world.
; How bad is this?  We expect most encapsulates to have a readily
; recognized name among their new args and most encapsulates are not
; redundant, so we think most of the time, we'll find a name and it
; will be new.

  (let ((name
         (find-first-non-local-name
          (list* 'encapsulate signatures ev-lst))))
    (cond ((and name
                (stringp name)
                (not (find-non-hidden-package-entry
                      name
                      (global-val 'known-package-alist wrld)))
                (not (assoc-equal name (global-val 'include-book-alist wrld))))

; If the name we find is a string then it can only be a full-book-name, e.g.,
; the first non-local event in the encapsulate was an include-book.  However,
; just to remind us that stringp names can be package names we look there too,
; even though a defpkg couldn't occur in an encapsulate.  Note that if we do
; not find the name in the 'include-book-alist or the 'known-package-alist
; (non-hidden; see the Essay on Hidden Packages) then this encapsulate could
; not have been executed so it is not redundant.

           nil)
          ((and name
                (symbolp name)
                (new-namep name wrld))
           nil)
          (t (redundant-event-tuplep event-form
                                     (default-defun-mode wrld)
                                     wrld)))))

(defun mark-missing-as-hidden-p (a1 a2)

; A1 and a2 are known-package-alists.  Return the result of marking each
; package-entry in a1 that is missing in a2 with hidden-p equal to t.

  (cond ((endp a1) nil)
        ((or (find-package-entry (package-entry-name (car a1)) a2)
             (package-entry-hidden-p (car a1)))
         (cons (car a1)
               (mark-missing-as-hidden-p (cdr a1) a2)))
        (t (cons (change-package-entry-hidden-p (car a1) t)
                 (mark-missing-as-hidden-p (cdr a1) a2)))))

(defun known-package-alist-included-p (a1 a2)

; Return true if every package-entry in a1 is present in a2, and moveover, is
; present non-hidden in a2 if present non-hidden in a1.

  (cond ((endp a1) t)
        (t (and (let ((a2-entry (find-package-entry
                                 (package-entry-name (car a1)) a2)))
                  (and a2-entry
                       (or (package-entry-hidden-p (car a1))
                           (not (package-entry-hidden-p a2-entry)))))
                (known-package-alist-included-p (cdr a1) a2)))))

(defun encapsulate-fix-known-package-alist (pass1-k-p-alist wrld)

; Pass1-k-p-alist is the known-package-alist from the end of the first pass of
; an encapsulate, and we are now at the end of the second pass in the given
; world, wrld.  The known-package-alist of wrld may be missing some
; package-entries from pass1-k-p-alist because of defpkg events that were only
; executed under locally included books in the first pass.  We return the
; result of setting the known-package-alist of the given world by marking each
; package-entry in pass1-k-p-alist that is missing in the current world's
; known-package-alist with hidden-p equal to t.

; The call of known-package-alist-included-p below checks that the second pass
; does not introduce any packages beyond those introduced in the first pass,
; nor does the second pass "promote" any package to non-hidden that was hidden
; in the first pass.  We rely on this fact in order to use the
; known-package-alist from the first pass as a basis for the alist returned, so
; that any package-entry present in the second pass's alist is present in the
; result alist, and moveover is non-hidden in the result if non-hidden in the
; second pass's alist.

; In fact we believe that the known-package-alist at the end of the second pass
; of an encapsulate is the same as at the beginning of the encapsulate, since
; local events are all skipped and include-books are all local.  However, we do
; not rely on this belief.

  (let ((pass2-k-p-alist (global-val 'known-package-alist wrld)))
    (cond ((equal pass1-k-p-alist pass2-k-p-alist) ; optimize for a common case
           wrld)
          (t (assert$
              (known-package-alist-included-p pass2-k-p-alist pass1-k-p-alist)
              (global-set 'known-package-alist
                          (mark-missing-as-hidden-p pass1-k-p-alist
                                                    pass2-k-p-alist)
                          wrld))))))

(defun encapsulate-fn (signatures ev-lst state event-form)

; Important Note:  Don't change the formals of this function without reading
; the *initial-event-defmacros* discussion in axioms.lisp.

; The Encapsulate Essay

; The motivation behind this event is to permit one to extend the theory by
; introducing function symbols, and theorems that describe their properties,
; without completely tying down the functions or including all of the lemmas
; and other hacks necessary to lead the system to the proofs.  Thus, this
; mechanism replaces the CONSTRAIN event of Nqthm.  It also offers one way of
; getting some name control, comparable to scopes.  However, it is better than
; just name control because the "hidden" rules are not just apparently hidden,
; they simply don't exist.

; Encapsulate takes two main arguments.  The first is a list of
; "signatures" that describe the function symbols to be hidden.  By
; signature we mean the formals, stobjs-in and stobjs-out of the
; function symbol.  The second is a list of events to execute.  Some
; of these events are tagged as "local" events and the others are not.
; Technically, each element of ev-lst is either an "event form" or
; else an s-expression of the form (LOCAL ev), where ev is an "event
; form."  The events of the second form are the local events.
; Informally, the local events are present only so that we can justify
; (i.e., successfully prove) the non-local events.  The local events
; are not visible in the final world constructed by an encapsulation.

; Suppose we execute an encapsulation starting with ld-skip-proofsp nil in
; wrld1.  We will actually make two passes through the list of events.  The
; first pass will execute each event, proving things, whether it is local or
; not.  This will produce wrld2.  In wrld2, we check that every function symbol
; in signatures is defined and has the signature alleged.  Then we back up to
; wrld1, declare the hidden functions with the appropriate signatures
; (producing what we call proto-wrld3) and replay only the non-local events.
; (Note: if redefinitions are allowed and are being handled by query, the user
; will be presented with two queries for each redefining non-local event.
; There is no assurance that he answers the same way both times and different
; worlds may result.  C'est la vie avec redefinitions.)  During this replay we
; skip proofs.  Having constructed that world we then collect all of the
; theorems that mention any of the newly-introduced functions and consider the
; resulting list as the constraint for all those functions.  (This is a
; departure from an earlier, unsound implementation, in which we only collected
; theorems mentioning the functions declared in the signature.)  However, we
; "optimize" by constructing this list of theorems using only those
; newly-introduced functions that have as an ancestor at least one function
; declared in the signature.  In particular, we do not introduce any
; constraints if the signature is empty, which is reasonable since in that
; case, we may view the encapsulate event the same as we view a book.  At any
; rate, the world we obtain by noting this constraint on the appropriate
; functions is called wrld3, and it is the world produced by a successful
; encapsulation.  By putting enough checks on the kinds of events executed we
; can guarantee that the formulas assumed to create wrld3 from wrld1 are
; theorems that were proved about defined functions in wrld2.

; This is a non-trivial claim and will be the focus of much of our discussion
; below.  This discussion could be eliminated if the second pass consisted of
; merely adding to wrld1 the formulas of the exported names, obtained from
; wrld2.  We do not do that because we want to be able to execute an
; encapsulation quickly if we process one while skipping proofs.  That is,
; suppose the user has produced a script of some session, including some
; encapsulations, and the whole thing has been processed with ld-skip-proofsp
; nil, once upon a time.  Now the user wants to assume that script and and
; continue -- i.e., he is loading a "book".

; Suppose we hit the encapsulation when skipping proofs.  Suppose we are
; again in wrld1 (i.e., processing the previous events of this script
; while skipping proofs has inductively left us in exactly the same
; state as when we did them with proofs).  We are given the event list
; and the signatures.  We want to do here exactly what we did in the
; second pass of the original proving execution of this encapsulate.
; Perhaps more informatively put, we want to do in the second pass of
; the proving execution exactly what we do here -- i.e., the relative
; paucity of information available here (we only have wrld1 and not
; wrld2) dictates how we must handle pass two back there.  Remember, our
; goal is to ensure that the final world we create, wrld3, is absolutely
; identical to that created above.

; Our main problem is that the event list is in untranslated form.
; Two questions arise.

; (1) If we skip an event because it is tagged LOCAL, how will we know
; we can execute (or even translate) the subsequent events without
; error?  For example, suppose one of the events skipped is the
; defmacro of deflemma, and then we see a (deflemma &).  We will have
; to make sure this doesn't happen.  The key here is that we know that
; the second pass of the proving execution of this encapsulate did
; whatever we're doing and it didn't cause an error.  But this is an
; important point about the proving execution of an encapsulate: even
; though we make a lot of checks before the first pass, it is possible
; for the second pass to fail.  When that happens, we'll revert back
; to wrld1 for sanity.  This is unfortunate because it means the user
; will have to suffer through the re-execution of his event list
; before seeing if he has fixed the last error.  We should eventually
; provide some sort of trial encapsulation mechanism so the user can
; see if he's got his signatures and exports correctly configured.

; (2) How do we know that the formulas generated during the second
; pass are exactly the same as those generated during the first pass?
; For example, one of the events might be:

; (if (ld-skip-proofsp state)
;     (defun foo () 3)
;     (defun foo () 2))

; In this case, (foo) would be 2 in wrld2 but 3 in wrld3.

; The key to the entire story is that we insist that the event list
; consist of certain kinds of events.  For lack of a better name, we
; call these "embedded event forms".  Not everything the user might
; want to type in an interactive ACL2 session is an embedded event
; form!  Roughly speaking, an event form translates to a PROGN of
; "primitive events", where the primitive events are appropriate calls
; of such user-level functions as defun and defthm.  By "appropriate"
; we mean STATE only appears where specified by the stobjs-in for each
; event.  The other arguments, e.g., the name of a defthm, must be
; occupied by state free terms -- well, almost.  We allow uses of w so
; that the user can compute things like gensyms wrt the world.  In a
; rough analogy with Lisp, the events are those kinds of commands that
; are treated specially when they are seen at the top-level of a file
; to be compiled.

; Events have the property that while they take state as an argument
; and change it, their changes to the world are a function only of the
; world (and their other arguments).  Because of this property, we
; know that if s1 and s1' are states containing the same world, and s2
; and s2' are the states obtained by executing an event on the two
; initial states, respectively, then the worlds of s2 and s2' are
; equal.

; Thus ends the encapsulate essay.

  (let ((ctx (encapsulate-ctx signatures ev-lst)))
    (with-ctx-summarized
     (if (output-in-infixp state) event-form ctx)
     (let* ((wrld1 (w state))
            (saved-acl2-defaults-table
             (table-alist 'acl2-defaults-table wrld1))
            (event-form (or event-form
                            (list* 'encapsulate signatures ev-lst))))
       (revert-world-on-error
        (cond
         ((redundant-encapsulatep signatures ev-lst event-form wrld1)
          (stop-redundant-event state))
         ((and (not (eq (ld-skip-proofsp state) 'include-book))
               (not (eq (ld-skip-proofsp state) 'include-book-with-locals))
               (not (eq (ld-skip-proofsp state) 'initialize-acl2)))

; Ld-skip-proofsp is either t or nil.  But whatever it is, we will be
; processing the LOCAL events.

          (er-let*
           ((pair (chk-acceptable-encapsulate1 signatures ev-lst
                                               ctx wrld1 state)))
           (let ((insigs (car pair))
                 (wrld1 (cdr pair)))
             (pprogn
              (set-w 'extension wrld1 state)
              (print-encapsulate-msg1 insigs ev-lst state)
              (er-progn
               (process-embedded-events 'encapsulate-pass-1
                                        saved-acl2-defaults-table
                                        (ld-skip-proofsp state)
                                        (current-package state)
                                        (list 'encapsulate insigs)
                                        ev-lst ctx state)
               (let ((wrld2 (w state)))
                 (pprogn
                  (print-encapsulate-msg2 insigs ev-lst state)
                  (er-progn
                   (chk-acceptable-encapsulate2 insigs wrld2 ctx state)
                   (let* ((pass1-known-package-alist
                           (global-val 'known-package-alist wrld2))
                          (state (set-w 'retraction wrld1 state)))
                     (er-let*
                      ((temp

; The following encapsulate-pass-2 is protected by the revert-world-on
; error above.
                        (encapsulate-pass-2
                         insigs ev-lst saved-acl2-defaults-table ctx state)))
                      (let ((wrld3 (w state))
                            (constrained-fns (nth 0 temp))
                            (constraints-introduced (nth 1 temp))
                            (exports (nth 2 temp))
                            (subversive-fns (nth 3 temp))
                            (infectious-fns (nth 4 temp)))
                        (pprogn
                         (print-encapsulate-msg3
                          ctx insigs ev-lst exports
                          constrained-fns constraints-introduced
                          subversive-fns infectious-fns wrld3 state)
                         (install-event t
                                        event-form
                                        'encapsulate
                                        (strip-cars insigs)
                                        nil nil
                                        (encapsulate-fix-known-package-alist
                                         pass1-known-package-alist
                                         wrld3)
                                        state)))))))))))))

         (t ; (ld-skip-proofsp state) = 'include-book
          ;;;                         'include-book-with-locals or
          ;;;                         'initialize-acl2

; We quietly execute our second pass.

          (er-let*
           ((pair (chk-signatures signatures ctx wrld1 state)))
           (let ((insigs (car pair))
                 (wrld1 (cdr pair)))
             (pprogn
              (set-w 'extension wrld1 state)
              (er-let*

; The following encapsulate-pass-2 is protected by the revert-world-on
; error above.

               ((temp
                 (encapsulate-pass-2
                  insigs ev-lst saved-acl2-defaults-table ctx state)))
               (let ((wrld3 (w state)))
                 (install-event t
                                event-form
                                'encapsulate
                                (signature-fns signatures)
                                nil nil

; We have considered calling encapsulate-fix-known-package-alist on wrld3, just
; as we do in the first case (when not doing this on behalf of include-book).
; But we do not see a need to do so, both because all include-books are local
; and hence skipped (hence the known-package-alist has not changed from before
; the encapsulate), and because we do not rely on tracking packages during
; include-book, :puff (where ld-skip-proofsp is include-book-with-locals), or
; initialization.

                                wrld3
                                state)))))))))))))

; Now we develop the book mechanism, which shares a lot with what
; we've just done.  In the discussion that follows, Unix is a
; trademark of Bell Laboratories.

; First, a broad question:  how much security are we trying to provide?
; After all, one could always fake a .cert file, say by calling checksum
; onesself.  Our claim is simply that we only fully "bless" certification runs,
; from scratch, of entire collections of books, without intervention.  Thus,
; there is no soundness problem with using (include-book "hd:ab.lisp") in a
; book certified in a Unix file system and having it mean something completely
; different on the Macintosh.  Presumably the attempt to certify this
; collection on the Macintosh would simply fail.

; How portable do we intend book names to be?  Suppose that one has a
; collection of books, some of which include-book some of the others, where all
; of these include-books use relative path names.  Can we set things up so that
; if one copies all of these .lisp and .cert files to another file system,
; preserving the hierarchical directory relationship, then we can guarantee
; that this collection of books is certifiable (modulo resource limitations)?
; The answer is yes: We use Unix-style pathnames within ACL2.  See :doc
; pathname, and see the Essay on Pathnames in interface-raw.lisp.  (Before
; Version 2.5 we also supported a notion of structured pathnames, similar to
; the "structured directories" concept in CLtL2.  However, the CLtL2 notion was
; just for directories, not file names, and although we continue to support
; structure pathnames in Version 2.5 for backwards compatibility, we are
; "deprecating" them by removing their mention from the documentation.)

; Note.  It is important that regardless of what initial information we store
; in the state that is based on the surrounding operating system, this
; information not be observable in the logical theory.  For example, it would
; really be unfortunate if we did something like:

;  (defconst *directory-separator*
;    #+apple #\:
;    #-apple #\/)

; because then we could certify a book in one ACL2 that contains a theorem
; (equal *directory-separator* #\/), and include this book in another world
; where that theorem fails, thus deriving a contradiction.  In fact, we make
; the operating-system part of the state (as a world global), and figure
; everything else out about book names using that information.

(deflabel books
  :doc
  ":Doc-Section  Books

  files of ACL2 event forms~/

  This documentation topic is about ACL2 input files.  However, there are two
  traditional (paper) books published about ACL2:  a textbook and a case
  studies book.  Further information is available by following links from the
  ACL2 home page, ~c[http://www.cs.utexas.edu/users/moore/acl2/].  Now, on to
  the real content of this topic:

  A ``book'' is a file of ACL2 ~il[events] that have been certified as
  admissible.  Using ~ilc[include-book] you can construct a new logical
  ~il[world] by assuming the ~il[events] in any number of mutually compatible
  books.  Relevant documented topics are listed below.  Following this list
  is a ``guided tour'' through the topics.
  ~terminal[You may start guided tour by typing :more.]~/

  ~em[Introduction.]

  A ``book'' is a file of ACL2 forms.  Books are prepared entirely by
  the user of the system, i.e., they are ``source'' files not
  ``object'' files.  Some of the forms in a book are marked ~ilc[local]
  and the others are considered ``non-local.''

  ~ilc[Include-book] lets you load a book into any ACL2 ~il[world].  If
  completed without error, the inclusion of a book extends the logic
  of the host ~il[world] by the addition of just the non-local ~il[events] in
  the book.  You may extend the ~il[world] by successively including a
  variety of books to obtain the desired collection of definitions and
  rules.  Unless name conflicts occur (which are detected and
  signalled) inclusion of a book is consistency preserving provided
  the book itself is consistent as discussed later.  However,
  ~ilc[include-book] merely assumes the validity of the ~il[events] in a book;
  if you include a book that contains an inconsistency (e.g., an
  inadmissible definition) then the resulting theory is inconsistent.

  It is possible to ``certify'' a book, with ~ilc[certify-book],
  guaranteeing that the error-free inclusion of the certified forms
  will produce a consistent extension of a consistent logic.
  Certification processes both the ~ilc[local] and non-local forms, so
  you can mark as ~ilc[local] those ~il[events] you need for certification
  that you want to hide from users of the book (e.g., hacks, crocks,
  and kludges on the way to a good set of ~c[:]~ilc[rewrite] rules).
  Certification can also ``compile'' a book, thereby speeding up the
  execution of the functions defined within it.  The desire to compile
  books is largely responsible for the restrictions we put on the
  forms allowed in books.

  Extensive ~il[documentation] is available on the various aspects of
  books.  We recommend that you read it all before using books.  It
  has been written so as to make sense when read in a certain linear
  sequence, called the ``guided tour'', though in general you may
  browse through it randomly.  If you are on the guided tour, you
  should next read the ~il[documentation] on book-example
  (~pl[book-example]~terminal[ and use :more to read through it]).~/

  :cite include-book")

(defun directory-list-p (lst back-ok-p)
  (declare (xargs :guard (true-listp lst)))
  (cond
   ((null lst)
    t)
   (t
    (let ((str (car lst)))
      (if (not (stringp str))
          (and back-ok-p
               (eq str :back)
               (directory-list-p (cdr lst) back-ok-p))
        (let ((lng (length str)))
          (and (< 0 lng)
               (not (member-char-stringp *directory-separator* str (1- lng)))
               (directory-list-p (cdr lst) nil))))))))

(defun structured-directory-p (x)
  (declare (xargs :guard (true-listp x)))
  (case (car x)
        (:absolute (directory-list-p (cdr x) nil))
        (:relative 

; The Common Lisp standard does not allow (:relative) as a structured
; representation of a directory.  Should we?  Given that (because of the use of
; :back) the function relative-structured-directory-to-string can be called on
; the empty list anyhow, there seems to be no reason to make such a restriction
; here.

         (directory-list-p (cdr x) t))
        (otherwise nil)))

(defun structured-path-p (x)
  (and (true-listp x)
       (< 1 (length x))
       (structured-directory-p (butlast x 1))
       (let ((filename (car (last x))))
         (and (stringp filename)
              (let ((len (length filename)))
                (and (not (= len 0))
                     (not (member-char-stringp
                           *directory-separator* filename (1- len)))))))))

(defun concatenate-strings-with-separator-string (string-lst)
  (cond
   ((null (cdr string-lst))
    (if (null string-lst)
        (er hard 'concatenate-strings-with-separator-string
            "Attempted to call ~
             concatenate-strings-with-separator-string with empty ~
             list.")
      (car string-lst)))
   (t (concatenate 'string (car string-lst) *directory-separator-string*
                   (concatenate-strings-with-separator-string
                    (cdr string-lst))))))

(defun absolute-structured-directory-to-string (lst)

; Lst is the cdr of an absolute structured directory.

  (if (null lst)
      *directory-separator-string*
    (concatenate
     'string
     *directory-separator-string*
     (concatenate-strings-with-separator-string lst)
     *directory-separator-string*)))

(defun cancel-backs1 (lst1 lst2)
  (cond
   ((or (null lst2)
        (not (eq (car lst2) :back)))
    (mv lst1 lst2))
   (t
    (cancel-backs1 (member *directory-separator* (cdr lst1)) (cdr lst2)))))

(defun cancel-backs (init-dir lst)

; Lst is a list of zero or more occurrences of the keyword :BACK followed by
; zero or more strings.  Init-dir is a string representing an absolute
; directory pathname.  This function returns a pair (mv new-init-dir new-lst);
; new-init-dir is a string representing the result of cancelling one level of
; directory structure in init-dir for each :back in lst, and new-lst is the
; result of removing each :BACK from lst.  However, if there are too many
; :BACKs, then new-init-dir is nil and new-lst is irrelevant.

; The algorithm:  init-dir is some string, which might or might not begin with
; the directory separator but definitely ends with one.  We coerce it to a
; list and reverse that, making sure that the result starts and ends with a
; directory separator.  Each time :back is removed, we delete everything back
; to the last directory separator.

  (cond
   ((or (null lst)
        (not (eq (car lst) :back)))
    (mv init-dir lst))
   ((not (eql (char init-dir (1- (length init-dir)))
              *directory-separator*))
    (mv init-dir
        (er hard 'cancel-backs
            "CANCEL-BACKS is supposed to be supplied a string ending with the ~
             directory separator, ~p0, but ~x1 is not such a string.  Please ~
             contact the implementors of ACL2."
            *directory-separator* init-dir)))
   (t
    (let* ((init-dir-chars0 (coerce init-dir 'list))
           (starts-with-separator-p (eql (car init-dir-chars0)
                                         *directory-separator*))
           (init-dir-chars-reversed
            (reverse
             (if starts-with-separator-p
                 init-dir-chars0
               (cons *directory-separator* init-dir-chars0)))))
      (mv-let (init-dir-remainder new-lst)
              (cancel-backs1 init-dir-chars-reversed lst)
              (mv (and init-dir-remainder
                       (coerce (if starts-with-separator-p
                                   (reverse init-dir-remainder)
                                 (cdr (reverse init-dir-remainder)))
                               'string))
                  new-lst))))))

(defun relative-structured-directory-to-string (lst)

; Lst is the cdr of a relative structured directory.

  (if (null lst)
      ""
    (concatenate
     'string
     (concatenate-strings-with-separator-string lst)
     *directory-separator-string*)))

(defun structured-directory-to-string (init-dir lst)

; Here init-dir is a string that is known to represent an absolute pathname,
; and lst is a structured directory (relative or absolute).  This function
; returns the string corresponding to the directory represented by lst if lst
; is absolute, and otherwise the string corresponding to the absolute pathname
; obtained by moving from init-dir using lst.

  (cond
   ((eq (car lst) :absolute)
    (absolute-structured-directory-to-string (cdr lst)))
   (t
    (mv-let (new-init-dir new-lst)
            (cancel-backs init-dir (cdr lst))
            (if new-init-dir
                (concatenate 'string
                             new-init-dir
                             (relative-structured-directory-to-string new-lst))
              nil)))))

(defun chk-book-name (book-name full-book-name ctx state)

; Book-name is something submitted by the user as a book name.
; Full-book-name is the first result of calling parse-book-name on
; book-name and state.  We check that full-book-name is a string
; ending in ".lisp" or cause an error.  But the error reports
; book-name as the offender.

; This check is important because to form the certification extension we strip
; off the "lisp" and replace it by "cert".  So if this is changed, change
; convert-book-name-to-cert-name and also the raw lisp function
; convert-book-name-to-compiled-name.

; Note: Because it is our own code, namely parse-book-name, that tacks on the
; ".lisp" extension, this check is now redundant.  Once upon a time, the user
; was expected to supply the .lisp extension, but that made the execution of
; (include-book "arith.lisp") in raw lisp load the .lisp file rather than the
; .o file.  We've left the redundant check in because we are not sure that
; parse-book-name will be kept in its current form; it has changed a lot
; lately.

  (cond
   ((and (stringp full-book-name)
         (let ((n (length full-book-name)))
           (and (> n 5)
                (eql (char full-book-name (- n 5)) #\.) 
                (eql (char full-book-name (- n 4)) #\l) 
                (eql (char full-book-name (- n 3)) #\i) 
                (eql (char full-book-name (- n 2)) #\s) 
                (eql (char full-book-name (- n 1)) #\p))))
    (pprogn
     (cond
      ((consp book-name)
       (warning$ ctx "Structured pathname"
        "Using pathname ~x0 for structured book name ~f1.  Structured book ~
         names might not be supported after ACL2 Version  2.5.  ACL2 now ~
         supports pathnames in the style of Unix (trademark of AT&T); ~
         see :DOC pathname.~%"
        full-book-name book-name))
      (t state))
     (value nil)))
   ((null full-book-name)
    (er soft ctx
        "~x0 is not a legal book name.  See :DOC book-name."
        book-name))
   (t (er soft ctx
          "~x0 is not a legal book name because it does not specify the ~
           ``.lisp'' extension.  See :DOC book-name."
          book-name))))

; The portcullis of a book consists of two things, a sequence of
; commands which must be executed with ld-skip-proofs nil without error
; and an include-book-alist-like structure which must be a subset of
; include-book-alist afterwards.  We describe the structure of an
; include-book-alist below.

(defun include-book-alist-subsetp (alist1 alist2)

; The include-book-alist contains elements of the
; general form         example value

; (full-book-name     ; "/usr/home/moore/project/arith.lisp"
;  user-book-name     ; "project/arith.lisp"
;  familiar-name      ; "arith"
;  cert-annotations   ; ((:SKIPPED-PROOFSP . sp) (:AXIOMSP . axp))
;  . ev-lst-chk-sum)  ; 12345678

; The include-book-alist becomes part of the certificate for a book,
; playing a role in both the pre-alist and the post-alist.  In the
; latter role some elements may be marked (LOCAL &).  When we refer to
; parts of the include-book-alist entries we have tried to use the
; tedious names above, to help us figure out what is used where.
; Please try to preserve this convention.

; Cert-annotations is an alist.  As of this writing (ACL2 Version_2.5)
; the alist has two possible keys, :SKIPPED-PROOFSP and :AXIOMSP.  The
; possible values of both are t, nil, or ?, indicating the presence, absence,
; or possible presence of skip-proof forms or defaxioms, respectively.  The
; forms in question may be either LOCAL or non-LOCAL and are in the book
; itself (not just in some subbook).  Even though the cert-annotations is an
; alist, we compare include-book-alists with equality on that component, not
; ``alist equality.''  So we are NOT free to drop or rearrange keys in these
; annotations.

; If the book is uncertified, the chk-sum entry is nil.

; Suppose the two alist arguments are each include-book-alists from
; different times.  We check that the first is a subset of the second,
; in the sense that the (familiar-name cert-annotations . chk-sum)
; parts of the first are all among those of the second.  We ignore the
; full names and the user names because they may change as the book or
; connected book directory moves around.

  (subsetp-equal (strip-cddrs alist1)
                 (strip-cddrs alist2)))

(defun remove-after-last-directory-separator (p)
  (let* ((p-rev (reverse p))
         (posn (position *directory-separator* p-rev)))
    (if posn
        (subseq p 0 (1- (- (length p) posn)))
      (er hard 'remove-after-last-directory-separator
          "Implementation error!  Unable to handle a directory string."))))

(defun merge-using-dot-dot (p s)

; P represents a directory pathname without a final "/", and s represents any
; pathname.  S may start with any number of sequences "../" and "./" where the
; last "/" is optional if it is at the end of s.  We want to "cancel" the
; leading "../"s in s against directories at the end of p, and elminate leading
; "./"s from s.  This code is intended to be simple, not necessarily efficient.

  (cond
   ((equal p "") s)
   ((equal s "..")
    (remove-after-last-directory-separator p))
   ((or (equal s ".") (equal s "./"))
    p)
   ((and (>= (length s) 3)
         (eql (char s 0) #\.)
         (eql (char s 1) #\.)
         (eql (char s 2) #\/))
    (merge-using-dot-dot (remove-after-last-directory-separator p)
                         (subseq s 3 (length s))))
   ((and (>= (length s) 2)
         (eql (char s 0) #\.)
         (eql (char s 1) #\/))
    (merge-using-dot-dot p (subseq s 2 (length s))))
   (t
    (concatenate 'string p *directory-separator-string* s))))

(defun our-merge-pathnames (p s)

; This is something like the Common Lisp function merge-pathnames.  P and s are
; (Unix-style) pathname strings, where s is the relative pathname of a file.
; (If s may be an absolute pathname, use extend-pathname instead.)  We allow p
; to be nil, which is a case that arises when p is (f-get-global
; 'connected-book-directory state) during boot-strapping; otherwise p should be
; an absolute directory pathname (though we allow "" as well).

  (cond
   ((and (not (equal s ""))
         (eql (char s 0) *directory-separator*))
    (er hard 'our-merge-pathnames
        "Attempt to merge with an absolute filename, ~p0.  Please contact the ~
         ACL2 implementors."
        s))
   ((or (null p) (equal p ""))
    s)
   ((stringp p) ; checked because before Version_2.5 we allowed pathnames
    (merge-using-dot-dot
     (if (eql (char p (1- (length p)))
              *directory-separator*)
         (subseq p 0 (1- (length p)))
       p)
     s))
   (t
    (er hard 'our-merge-pathnames
        "The first argument of our-merge-pathnames must be a string, ~
         but the following is not:  ~p0."
        p))))

(defun get-portcullis-cmds (wrld ans wrld-segs wrld-list)

; When certify-book is called, we scan down wrld to collect all the user
; commands into ans.  This answer is part of the portcullis of the certificate,
; once it has been cleaned up by fix-portcullis-cmds and new-defpkg-list.
; Second, we also accumulate corresponding world segments into wrld-segs.  Each
; command in ans will be "fixed" using its corresponding world segment.
; Finally, we also collect the entire world corresponding to each command,
; except the result has an extra world on the front of that list corresponding
; to the final world observed (probably the boot-strap world).

  (cond
   ((null wrld) (mv ans wrld-segs (cons wrld wrld-list)))
   ((and (eq (caar wrld) 'command-landmark)
         (eq (cadar wrld) 'global-value))
    (cond ((equal (access-command-tuple-form (cddar wrld))
                  '(exit-boot-strap-mode))
           (mv ans wrld-segs (cons wrld wrld-list)))
          (t (get-portcullis-cmds
              (cdr wrld)
              (cons (access-command-tuple-form (cddar wrld))
                    ans)
              (cons (world-to-next-command (cdr wrld) nil)
                    wrld-segs)
              (cons wrld wrld-list)))))
   (t (get-portcullis-cmds (cdr wrld) ans wrld-segs wrld-list))))

; We now develop code to "fix" the commands in the certification world before
; placing them in the portcullis of the certificate, in order to eliminate
; relative pathnames in include-book forms.  See the comment in
; fix-portcullis-cmds.

(mutual-recursion

(defun make-include-books-absolute (form book-name-alist names os ctx state)

; Form is a command from the current ACL2 world that is known to be an embedded
; event form with respect to names.  Keep this function in sync with
; chk-embedded-event-form in chk-acceptable-certify-book1).  We pattern the
; code after find-a-skip-proofs.  Like that function, this one returns an error
; triple because we use macroexpand1, but never causes an error.

  (cond
   ((atom form) (value form)) ; This should never happen.
   ((eq (car form) 'skip-proofs)
    (er-let* ((x (make-include-books-absolute (cadr form) book-name-alist names
                                              os ctx state)))
             (value (list (car form) x))))
   ((eq (car form) 'local)

; Include-books in the context of LOCAL will not generate 'certification-tuple
; entries in the world, hence will not contribute to book-name-alist.  If we
; were to explore (cadr form), we could then find an include-book of a relative
; pathname that is not found in book-name-alist.

    (value form))
   ((eq (car form) 'encapsulate)
    (er-let* ((x (make-include-books-absolute-lst (cddr form) book-name-alist
                                                  names os ctx state)))
             (value (list* (car form)
                           (cadr form)
                           x))))
   ((eq (car form) 'progn)
    (er-let* ((rest (make-include-books-absolute-lst (cdr form) book-name-alist
                                                     names os ctx state)))
             (value (cons (car form)
                          rest))))
   ((eq (car form) 'value)
    (value form))
   ((and (eq (car form) 'include-book)
         (stringp (cadr form))
         (not (absolute-pathname-string-p
               (cadr form)
               nil ; no directory check necessary here
               os)))
    (let ((full-book-names
           (or (cdr (assoc-equal (cadr form) book-name-alist))
               (er hard ctx
                   "Implementation error (did not find book name, function ~
                    make-include-books-absolute).  Please contact the ACL2 ~
                    implementors."))))
      (if (null (cdr full-book-names))
          (value (list* 'include-book
                        (car full-book-names)
                        (cddr form)))
        (er soft ctx
            "The certification world has include-book command ~x0 that ~
             correspond to more than one possible full pathname, namely:  ~
             ~&1. ACL2 cannot currently certify a book in such a world.  To ~
             work around this problem, use an absolute pathname in this ~
             include-book command (see :DOC pathname)."
            form
            full-book-names))))
   ((member-eq (car form) names)
    (value form))
   ((getprop (car form) 'macro-body nil 'current-acl2-world (w state))
    (er-let*
     ((form1 (macroexpand1 form ctx state)))
     (make-include-books-absolute form1 book-name-alist names os ctx state)))
   (t (value (er hard ctx
                 "Implementation error in make-include-books-absolute:  ~
                  unrecognized event type, ~x0.  Make-include-books-absolute ~
                  needs to be kept in sync with chk-embedded-event-form.  ~
                  Please send this error message to the implementors."
                 (car form))))))

(defun make-include-books-absolute-lst (forms book-name-alist names os ctx state)
  (if (endp forms)
      (value nil)
    (er-let* ((first (make-include-books-absolute (car forms) book-name-alist
                                                  names os ctx state))
              (rest (make-include-books-absolute-lst (cdr forms)
                                                     book-name-alist names os
                                                     ctx state)))
             (value (cons first rest)))))

)

(defun relative-book-name-alist (wrld-segment acc os ctx state)

; We return an alist that associates, for each relative pathname p for which
; (include-book p) was executed in the given world segment, the corresponding
; full book names that are (also) without the .lisp extension.

  (cond
   ((null wrld-segment)
    (value acc))
   ((and (eq (caar wrld-segment) 'certification-tuple)
         (eq (cadar wrld-segment) 'global-value))
    (let* ((cert-tuple (cddar wrld-segment))

; Cert-tuple is of the following form.

; (list full-book-name user-book-name familiar-name cert-annotations
;       ev-lst-chk-sum) 

           (full-book-name-raw
            (if (eq cert-tuple *acl2-property-unbound*)

; We do not expect to find *acl2-property-unbound* here.  If we do find it,
; then we cause an error.

                (er hard ctx
                    "Implementation error!  Unexpected find of unbound ~
                     certification tuple value in command-cbd!  Please ~
                     contact the ACL2 implementors and send this message.")
              (car cert-tuple)))
           (full-book-name
            (subseq full-book-name-raw 0 (- (length full-book-name-raw) 5)))
           (user-book-name (cadr cert-tuple)))
      (cond

; Since this code is being introduced in Version_2.6, we do not need to
; support structured pathnames.  We check for them just in case.

       ((not (stringp user-book-name))
        (er soft ctx
            "Structured book names, such as ~x0, are no longer supported by ~
             ACL2.  Please modify your include-book events accordingly."
            user-book-name))
       ((absolute-pathname-string-p user-book-name nil os)
        (relative-book-name-alist (cdr wrld-segment) acc os ctx state))
       (t
        (let ((pair (assoc-equal user-book-name acc)))
          (cond
           ((null pair)
            (relative-book-name-alist (cdr wrld-segment)
                                      (cons (list user-book-name
                                                  full-book-name)
                                            acc)
                                      os ctx state))
           ((member-equal full-book-name (cdr pair)) ; Can this happen?
            (relative-book-name-alist (cdr wrld-segment) acc os ctx state))
           (t
            (relative-book-name-alist (cdr wrld-segment)
                                      (put-assoc-equal
                                       (car pair)
                                       (cons full-book-name (cdr pair))
                                       acc)
                                      os ctx state))))))))
   (t
    (relative-book-name-alist (cdr wrld-segment) acc os ctx state))))

(defun first-known-package-alist (wrld-segment)
  (cond
   ((null wrld-segment)
    nil)
   ((and (eq (caar wrld-segment) 'known-package-alist)
         (eq (cadar wrld-segment) 'global-value))
    (let* ((kpa  (cddar wrld-segment)))
      (if (eq kpa *acl2-property-unbound*)

; We do not expect to find *acl2-property-unbound* here.  If we do find it,
; then we cause an error.

          (er hard 'first-known-package-alist
              "Implementation error!  Unexpected find of unbound ~
               known-package-alist value!  Please contact the ACL2 ~
               implementors and send this message.")
        kpa)))
   (t
    (first-known-package-alist (cdr wrld-segment)))))

(defun defpkg-items-rec (new-kpa old-kpa ctx w state acc)

; For background on the discussion below, see the Essay on Hidden Packages.

; We are given a world w (for example, the certification world of a
; certify-book command).  Old-kpa is the known-package-alist of w.  New-kpa is
; another known-package-alist, which may include entries not in old-kpa (for
; example, the known-package-alist after executing each event in the
; admissibility pass of certify-book).  We return a list of "defpkg items" for
; names of new-kpa, where each item is of the form (list name imports body doc
; book-path).  The intention is that the item can be used to form a defpkg
; event with indicated name, body, doc and book-path, where body may have been
; modified from a corresponding defpkg event so that it is suitable for
; evaluation in w.  Here, book-path is the book-path to be used if such an
; event is to be added to the end of the portcullis commands in the certificate
; of a book being certified.

; As a minor optimization we completely omit names of new-kpa from the result
; when they correspond to packages in old-kpa.

; It is helpful for efficiency if w is the current-acl2-world or a reasonably
; short extension of it, since we call termp and untranslate on that world.

; Finally, note that the order of packages in new-kpa is in the reverse of the
; order from the resulting list of defpkg events.  Thus, the newest packages
; are at the end of the returned list.

  (cond
   ((endp new-kpa) (value acc))
   (t (let* ((e (car new-kpa))
             (n (package-entry-name e)))
        (cond
         ((find-package-entry n old-kpa)
          (defpkg-items-rec (cdr new-kpa) old-kpa ctx w state acc))
         (t
          (let* ((imports (package-entry-imports e))
                 (pair (package-entry-defpkg-event-info e))
                 (event (car pair))
                 (tterm (cdr pair))
                 (name (cadr event))
                 (body (caddr event))
                 (doc (cadddr event))
                 (book-path (package-entry-book-path e)))
            (mv-let (erp pair state)
              (simple-translate-and-eval body nil nil
                                         "The second argument to defpkg"
                                         ctx w state)
              (defpkg-items-rec
                (cdr new-kpa) old-kpa ctx w state
                (cons (list name
                            imports
                            (assert$
                             event
                             (assert$
                              (equal n name)
                              (cond ((and (not erp)
                                          (equal (cdr pair) imports) ; always?
                                          (equal tterm (car pair)))
                                     body)
                                    ((termp tterm w)
                                     tterm)
                                    (t
                                     (kwote imports)))))
                            doc
                            book-path)
                      acc))))))))))

(defun defpkg-items (new-kpa ctx w state)

; This is just a wrapper for defpkg-items-rec, with error output turned off
; (because of calls of translate).  See the comment for defpkg-items-rec.

  (state-global-let*
   ((inhibit-output-lst (cons 'error
                              (f-get-global 'inhibit-output-lst state))))
   (defpkg-items-rec new-kpa (global-val 'known-package-alist w) ctx w state
     nil)))

(defun new-defpkg-list2 (imports discarded-items flg seen)
  (cond
   ((endp imports)
    (mv flg discarded-items))
   (t
    (let ((p (symbol-package-name (car imports))))
      (cond
       ((member-equal p seen)
        (new-defpkg-list2 (cdr imports) discarded-items
                          flg seen))
       (t (let ((item (assoc-equal p discarded-items)))
            (cond (item (new-defpkg-list2
                         (cdr imports)
                         (delete-assoc-equal p discarded-items)
                         t
                         (cons p seen)))
                  (t (new-defpkg-list2
                      (cdr imports)
                      discarded-items
                      flg
                      (cons p seen)))))))))))

(defun make-defpkg (name imports/doc/book-path)
  (let ((imports (car imports/doc/book-path))
        (doc (cadr imports/doc/book-path))
        (book-path (caddr imports/doc/book-path)))
    (cond
     (book-path `(defpkg ,name ,imports ,doc ,book-path))
     (doc       `(defpkg ,name ,imports ,doc))
     (t         `(defpkg ,name ,imports)))))

(defun new-defpkg-list1 (defpkg-items base-kpa added-defpkgs discarded-items)

; See the comment in new-defpkg-list.  Here, we maintain an accumulator,
; added-defpkgs, that contains the defpkg that need to be added based on what
; we have already processed in defpkg-items, in reverse order.  We also
; maintain a list discarded-items, that contains members of defpkg-items not
; added to added-defpkgs.

  (cond
   ((endp defpkg-items)
    (mv added-defpkgs discarded-items))
   (t
    (mv-let (added-defpkgs discarded-items)
      (new-defpkg-list1 (cdr defpkg-items) base-kpa added-defpkgs
                        discarded-items)
      (let* ((item (car defpkg-items))
             (name (car item)))
        (cond
         ((find-package-entry name base-kpa)
          (mv added-defpkgs (cons item discarded-items)))
         (t ; we want to add event, so may need to add some already "discarded"
          (let ((imports (cadr item)))
            (mv-let (flg remaining-discarded-items)
              (new-defpkg-list2 imports discarded-items nil nil)
              (mv-let (added-defpkgs discarded-items)
                (cond
                 (flg ; (not (equal discarded-items remaining-discarded-items))
                  (new-defpkg-list1
                   (set-difference-equal discarded-items
                                         remaining-discarded-items)
                   base-kpa added-defpkgs remaining-discarded-items))
                 (t (mv added-defpkgs discarded-items)))
                (mv (cons (make-defpkg name (cddr item))
                          added-defpkgs)
                    discarded-items)))))))))))
                
(defun new-defpkg-list (defpkg-items base-kpa)

; For background on the discussion below, see the Essay on Hidden Packages.

; Defpkg-items is a list of "defpkg items" each of the form (list name imports
; body doc book-path) representing a list of package definitions.  We return a
; list of defpkg events that can be executed in a world whose
; known-package-alist is base-kpa.  Defpkg-items corresponds to defpkg events
; in reverse order of their introduction, i.e., the newest ones are at the end
; of the list; and the returned list has a corresponding order.  The primary
; reason a defpkg is in the returned list is that its package is not in
; base-kpa (not even hidden).  The second reason is that we need to define a
; package P1 if we add another package P2 whose import list contains a symbol
; in package P1; we close under this process.

; This function is called at the end of the include-book phase of certify-book.
; In that case, base-kpa is the known-package-alist at that point, and
; defpkg-items contains an item for each name of a package in the
; known-package-alist at the end of the earlier, admissibility pass of
; certify-book that was not defined in the certification world.  To illustrate
; the "second reason" above, let us suppose that the book being certified
; contains forms (include-book "book1") and (local (include-book "book2")),
; where book1 defines (defpkg "PKG1" ...) and book2 defines (defpkg "PKG2"
; '(PKG1::SYM)).  Then we want to add the definition of "PKG2" to the
; portcullis, but in order to do so, we need to add the definition of "PKG1" as
; well, even though it will eventually be included by way of book1.  And, we
; need to be sure to add the defpkg of "PKG1" before that of "PKG2".

; This function is also called on behalf of puff-fn1, where defpkg-items
; corresponds to the packages in known-package-alist in the world at completion
; of the command about to be puffed, and base-kpa corresponds to the
; known-package-alist just before that command.  In that case there is no ned
; for the "second reason" above, but for simplicity we call this same function.

  (mv-let (added-defpkgs discarded-items)
    (new-defpkg-list1 defpkg-items base-kpa nil nil)
    (declare (ignore discarded-items))
    added-defpkgs))

(defun fix-portcullis-cmds (cmds wrld-segs wrld-list old-kpa ans names os ctx
                                 state)

; This function modifies cmds by making relative pathnames absolute, and also
; by adding defpkg events as explained in the Essay on Hidden Packages.  We
; explain these two aspects in turn.

; Certify-book needs to insist that any include-books in the portcullis refer
; to absolute pathnames for the operating system in which the book was
; certified.  This ensures that the actual file read is not dependent upon cbd.
; Suppose this restriction were dropped, and consider:

; :set-cbd "/usr/home/moore/"
; (include-book "prelude")
; :set-cbd "/usr/local/src/library/"
; (certify-book "user")

; A naive implementation would provide a portcullis for "user" that contains
; (include-book "prelude").  But there is no clue as to the directory on which
; "prelude" resides.  Note that "prelude" does not represent an absolute
; pathname.  If it did represent an absolute pathname, then it would have to be
; the full book name because parse-book-name returns x when x represents an
; absolute pathname.

; We deal with the issue above by allowing relative pathnames for include-book
; commands in the certification world, but modifying them to be appropriate
; absolute pathnames.  This function takes the original cmds, a list of
; embedded event forms, together with wrld-segs, a corresponding list of world
; segments.  Each member of wrld-segs contains the tuples laid down as a result
; of its corresponding command in cmds (up to but not including the command
; landmark).  We return a list of commands that is guaranteed to be free of
; include-books of relative pathnames, that nevertheless is equivalent to the
; original cmds from the standpoint of subsequent embedded events.  (Or, we
; return an error, but in fact we believe that that will not happen.)

; Our algorithm uses the 'certification-tuple tuples laid down in each element
; of wrld-segs to make necessary adjustments.  We will thus miss local
; include-book events.  But we do not need to make those absolute, since we
; trust the portcullis and hence do not need to pay attention to local events.

; As mentioned at the outset above, this function also adds defpkg events.  Our
; approach is heavy-handed (it seemed easiest to code this way), in that it
; adds a defpkg event before each command for every defpkg admitted under that
; command.  if the command is a defkpg, then we add the identical defpkg in
; front of that defpkg!  But we take care of that and any other duplication in
; the call of remove-duplicates-equal-from-end in certify-book-fn.  If for some
; reason we fail to remove the duplicate for trivial reasons (say we fail to
; match the doc string or the book-path argument of a defpkg when adding an
; allegedly identical new on), the worst that can happen is that we get a
; surprising error; barring that, the worst is that we have duplicate defpkg
; events in .cert files.  Neither is a soundness issue.

; Call this function using the same names parameter as that used when verifying
; that cmds is a list of embedded event forms.

  (cond
   ((null cmds) (value (reverse ans)))
   (t (er-let* ((book-name-alist
                 (relative-book-name-alist (car wrld-segs) nil os ctx state))
                (cmd
                 (if (null book-name-alist) ; optimization:  nothing to fix
                     (value (car cmds))
                   (make-include-books-absolute (car cmds) book-name-alist
                                                names os ctx state)))
                (new-kpa (value (first-known-package-alist (car wrld-segs))))
                (defpkg-items (if new-kpa
                                  (defpkg-items
                                    new-kpa
                                    ctx (car wrld-list) state)
                                (value nil))))
               (fix-portcullis-cmds (cdr cmds)
                                    (cdr wrld-segs)
                                    (cdr wrld-list)
                                    new-kpa
                                    (cons cmd (revappend (new-defpkg-list
                                                          defpkg-items
                                                          old-kpa)
                                                         ans))
                                    names os ctx state)))))

(defun collect-uncertified-books (alist)

; Alist is an include-book-alist and thus contains elements of the
; form described in include-book-alist-subsetp.  A typical element is
; (full-book-name user-book-name familiar-name cert-annotations
; . ev-lst-chk-sum) and ev-lst-chk-sum is nil if the book has not been
; certified.

  (cond ((null alist) nil)
        ((null (cddddr (car alist)))  ; ev-lst-chk-sum
         (cons (caar alist)           ; full-book-name
               (collect-uncertified-books (cdr alist))))
        (t (collect-uncertified-books (cdr alist)))))

(defun certificate-filep (file1 state)
  (let ((file2 (convert-book-name-to-cert-name file1)))
    (mv-let
     (ch state)
     (open-input-channel file2 :object state)
     (cond
      ((null ch) (value nil))
      (t (pprogn (close-input-channel ch state)
                 (value t)))))))

(defun chk-in-package (channel file ctx state)

; Channel must be an open input object channel.  We assume (for error
; reporting purposes) that it is associated with the file named file.
; We read the first form in it and cause an error unless that form is
; an in-package.  If it is an in-package, we return the package name.

  (state-global-let*
   ((current-package "ACL2"))
   (mv-let (eofp val state)
           (read-object channel state)
           (cond
            (eofp (er soft ctx
                      "The file ~x0 is empty.  An IN-PACKAGE form, at ~
                       the very least, was expected."
                      file))
            ((and (true-listp val)
                  (= (length val) 2)
                  (eq (car val) 'in-package)
                  (stringp (cadr val)))
             (cond
              ((find-non-hidden-package-entry (cadr val)
                                              (known-package-alist state))
               (value (cadr val)))
              (t (er soft ctx
                     "The argument to IN-PACKAGE must be a known ~
                      package name, but ~x0, used in the first form ~
                      in ~x1, is not.  The known packages are~*2"
                     (cadr val)
                     file
                     (tilde-*-&v-strings
                      '&
                      (strip-non-hidden-package-names
                       (known-package-alist state))
                      #\.)))))
            (t (er soft ctx
                   "The first form in ~x0 was expected to be ~
                    (IN-PACKAGE \"pkg\") where \"pkg\" is a known ~
                    ACL2 package name.  See :DOC book-contents.  The first ~
                    form was, in fact, ~x1."
                   file val))))))

(defconst *ill-formed-certificate-msg*
  "The certificate for the book ~x0 is ill-formed.  Delete or rename ~
   the file ~x1 and recertify ~x0.  Remember that the certification ~
   world for ~x0 is described in the portcullis of ~x1 (see :DOC ~
   portcullis) so you might want to look at ~x1 to remind yourself of ~
   ~x0's certification world.") 

(defun include-book-er (file1 file2 str
                              keyword
                              suspect-book-action-alist
                              ctx state)

; Depending on various conditions we either do nothing, print a
; warning, or cause an error.  File1 and file2 are the full book name
; and its .cert file, respectively.  (Well, sometimes file2 is nil --
; we never use it ourselves but str might and supplies it when
; needed.)  Str is an arbitrary fmt string used as the error message
; and used in the warning message.  suspect-book-action-alist is the
; alist manufactured by include-book, specifying the values of its
; keyword arguments.  Among these are arguments that control our
; behavior on these errors.  Keyword specifies the kind of error this
; is, using the convention that it is either t, meaning cause an
; error, or the keyword used by include-book to specify the behavior.
; I.e., if this error reports the lack of a certificate, then keyword
; is :uncertified-okp.

  (let* ((keyword-string
          (case keyword
            (:uncertified-okp "Uncertified")
            (:skip-proofs-okp "Skip-proofs")
            (:defaxioms-okp "Defaxioms")
            (t (if (eq keyword t)
                   nil
                 (er hard 'include-book-er
                     "Include-book-er does not know the include-book ~
                        keyword argument ~x0."
                     keyword)))))
         (warning-summary
          (cond
           ((eq keyword t) nil)
           ((assoc-eq keyword suspect-book-action-alist)
            (cond
             ((cdr (assoc-eq keyword suspect-book-action-alist))
              (cond
               ((if (eq keyword :skip-proofs-okp)
                    (not (f-get-global 'skip-proofs-okp-cert state))
                  (and (eq keyword :defaxioms-okp)
                       (not (f-get-global 'defaxioms-okp-cert state))))

; Although suspect-book-action-alist allows this (implicit) include-book, we
; are attempting this include-book underneath a certify-book that disallows
; this keyword.  We signify this case by overloading warning-summary to be this
; keyword.

                keyword)
               (t keyword-string)))
             (t nil)))
           (t (er hard 'include-book-er
                  "There is a discrepancy between the keywords in the ~
                   suspect-book-action-alist, ~x0, and the keyword, ~x1, ~
                   supplied to include-book-er."
                  suspect-book-action-alist
                  keyword)))))

; If warning-summary is nil, we cause an error.  Otherwise, it is summary
; of the desired warning.

    (cond
     ((null warning-summary)
      (er soft ctx "~@2" file1 file2 str))
     ((symbolp warning-summary) ; keyword
      (er soft ctx "~@0  This is illegal because we are currently attempting a ~
                    book certification with ~x1 set to NIL.  You can avoid ~
                    this error by calling certify-book with a value of T for ~
                    ~x1; see :DOC certify-book."
          (list "~@2" (cons #\0 file1) (cons #\1 file2) (cons #\2 str))
          keyword))
     (t
      (pprogn
       (warning$ ctx warning-summary "~@2" file1 file2 str)
       (value nil))))))

(defun tilde-*-book-check-sums-phrase1 (flg reqd-alist actual-alist)

; The two alists are strip-cddrs of include-book-alists.  Thus, each
; entry in each is of the form (familiar-name cert-annotations
; . ev-lst-chk-sum).  For each entry in reqd-alist we either find an
; identical entry in actual-alist or else we print a message.

  (cond
   ((null reqd-alist) nil)
   (t (let* ((reqd-entry (car reqd-alist))
             (familiar-name (car reqd-entry))
             (actual-entry (assoc-equal familiar-name actual-alist)))

; We know there is an entry for familiar-name because otherwise we would have
; caused an error.  The question is only whether we found a cert file
; for it, etc.

        (cond
         ((equal reqd-entry actual-entry)
          (tilde-*-book-check-sums-phrase1 flg (cdr reqd-alist) actual-alist))
         (flg
          (cons
           (msg "the book \"~s0\" with certificate annotations ~x1 ~
                 and check sum ~x2"
                familiar-name
                (cadr reqd-entry)  ;;; cert-annotations
                (cddr reqd-entry)) ;;; ev-lst-chk-sum
           (tilde-*-book-check-sums-phrase1 flg
                                            (cdr reqd-alist)
                                            actual-alist)))
         (t
          (cons
           (cond
            ((null (cddr actual-entry))
             (msg "an uncertified version of ~x0 with certificate ~
                   annotations ~x1"
                  familiar-name
                  (cadr actual-entry) ; cert-annotations
                  ))
            (t (msg "a version of ~x0 with certificate annotations ~
                     ~x1 and check sum ~x2"
                    familiar-name
                    (cadr actual-entry)  ; cert-annotations
                    (cddr actual-entry))))
           (tilde-*-book-check-sums-phrase1 flg
                                            (cdr reqd-alist)
                                            actual-alist))))))))

(defun tilde-*-book-check-sums-phrase (flg reqd-alist actual-alist)

; Flg is t or nil and the two alists each contain pairs of the form
; (full-book-name user-book-name familiar-name cert-annotations
; . ev-lst-chk-sum).  Reqd-Alist shows what is required and
; actual-alist shows that is actual.  We know reqd-alist ought to be a
; `include-book alist subset' of actual-alist but it is not.  If flg
; is t we complete the phrase "this book requires ..."  and if flg is
; nil we complete "but we have ...".

  (list "" "~@*" "~@* and " "~@*, "
        (tilde-*-book-check-sums-phrase1 flg
                                         (strip-cddrs reqd-alist)
                                         (strip-cddrs actual-alist))))

(defun get-cmds-from-portcullis (file1 file2 ch ctx state ans)

; Keep this in sync with chk-raise-portcullis2.

; We read successive forms from ch, stopping when we get to
; :END-PORTCULLIS-CMDS and returning the list of forms read, which we
; accumulate onto ans as we go.  Ans should be nil initially.

  (mv-let (eofp form state)
          (state-global-let*
           ((infixp nil))
           (read-object ch state))
          (cond
           (eofp (er soft ctx *ill-formed-certificate-msg* file1 file2))
           ((eq form :END-PORTCULLIS-CMDS)
            (value (reverse ans)))
           (t (get-cmds-from-portcullis file1 file2 ch ctx state
                                        (cons form ans))))))

(defun chk-raise-portcullis2 (file1 file2 ch ctx state ans)

; Keep this in sync with get-cmds-from-portcullis.

; We read successive forms from ch and trans-eval them.  We stop when
; we get to :END-PORTCULLIS-CMDS.  We may cause an error.  It is
; assumed that each form evaluated is a DEFPKG or an event form and is
; responsible for installing its world in state.  This assumption is
; checked by chk-acceptable-certify-book, before a .cert file is
; written.  We return the list of forms read, which we accumulate onto
; ans as we go.  Ans should be nil initially.

  (mv-let (eofp form state)
          (state-global-let*
           ((infixp nil))
           (read-object ch state))
          (cond
           (eofp (er soft ctx *ill-formed-certificate-msg* file1 file2))
           ((eq form :END-PORTCULLIS-CMDS)
            (value (reverse ans)))
           (t (mv-let
               (error-flg trans-ans state)
               (trans-eval form
                           (msg "the portcullis for ~x0"
                                file1)
                           state)

; If error-flg is nil, trans-ans is of the form
; ((nil nil state) . (erp' val' replaced-state))
; because form is a DEFPKG or event form.

               (let ((erp-prime (car (cdr trans-ans))))
                 (cond
                  ((or error-flg erp-prime) ;erp'
                   (pprogn
                    (warning$ ctx "Portcullis"
                              "The error reported above was caused ~
                               while trying to raise the portcullis ~
                               for the book ~x0.  In particular, we ~
                               were trying to execute ~x1 when the ~
                               error occurred.  Because we cannot ~
                               raise the portcullis, we cannot ~
                               include this book in this world.  ~
                               There are two standard responses to ~
                               this situation.  Either change the ~
                               current logical world so that this ~
                               error does not occur, e.g., redefine ~
                               one of your functions, or recertify ~
                               the book in a different environment."
                              file1 form)
                    (mv t nil state)))
                  (t (chk-raise-portcullis2 file1 file2 ch ctx state
                                            (cons form ans))))))))))

(defun chk-raise-portcullis1 (file1 file2 ch ctx state)

; We read each of the forms in ch until we get to :END-PORTCULLIS-CMDS
; and eval them.  However, we temporarily skip proofs (in an error
; protected way).  We return the list of command forms in the
; portcullis.

  (state-global-let*
   ((ld-skip-proofsp 'include-book))
   (er-progn
    (maybe-install-acl2-defaults-table

; The point here is to re-create the environment in which the book to be
; included was originally certified.  If we do not install the original
; acl2-defaults-table then we can, for example, certify a book definining (foo
; x) = (car x), then in a new session include this book after
; (set-verify-guards-eagerness 2), and then get a hard error with (foo 3).

     `((:DEFUN-MODE . :LOGIC)
       (:INCLUDE-BOOK-DIR-ALIST . ((:SYSTEM . ,(global-val
                                                'distributed-books-dir
                                                (w state))))))
     'chk-raise-portcullis1 state)
    (chk-raise-portcullis2 file1 file2 ch ctx state nil))))

(defun mark-local-included-books (post-alist1 post-alist2)

; See make-certificate-file for an explanation of this function.  Roughly
; speaking, we copy post-alist1 (which is the include-book-alist after the
; events in the main book were successfully proved) and every time we find a
; non-local book in it that is not in post-alist2 (which is the
; include-book-alist after the main book was included by certify-book's second
; pass), we mark that element LOCAL.  We know that post-alist2 is a subset of
; post-alist1.  Thus, if we then throw out all the elements marked LOCAL we get
; post-alist2.

; One might ask why we mark post-alist1 this way rather than just put
; post-alist2 into the certificate object in the first place.  One reason
; is to allow a hand inspection of the certificate to see exactly what
; versions of the local subbooks participated in the certification.  But a more
; critical reason is to note the use of skip-proofs in locally included
; subbooks; see the Essay on Skip-proofs.

; Recall that each element of an include-book-alist is (full-book-name
; user-book-name familiar-name cert-annotations . ev-lst-chk-sum).  We
; only look at the full-book-name components below.

  (cond ((null post-alist1) nil)
        ((eq (caar post-alist1) 'local)
         (cons (car post-alist1)
               (mark-local-included-books (cdr post-alist1) post-alist2)))
        ((assoc-equal (caar post-alist1) post-alist2)
         (cons (car post-alist1)
               (mark-local-included-books (cdr post-alist1) post-alist2)))
        (t (cons (list 'local (car post-alist1))
                 (mark-local-included-books (cdr post-alist1) post-alist2)))))

(defun unmark-and-delete-local-included-books (post-alist3)

; See make-certificate-file for an explanation of this function.  Roughly
; speaking, this function undoes what mark-local-included-books does.  If
; post-alist3 is the result of marking post-alist1 and post-alist2, then this
; function produces post-alist2 from post-alist3.  Given our use of it it
; produces the include-book-alist you should have after any successful
; inclusion of the main book.

  (cond ((null post-alist3) nil)
        ((eq (caar post-alist3) 'LOCAL)
         (unmark-and-delete-local-included-books (cdr post-alist3)))
        (t (cons (car post-alist3)
                 (unmark-and-delete-local-included-books (cdr post-alist3))))))

(defun earlier-acl2-versionp (version1 version2)

; This amazingly complicated function allows us to compare ACL2
; Versions and to include version strings in our source code that name
; absolute version numbers without being normal version strings, e.g.,
; "ACL2 Version_2.5" is compared exactly as though it were written
; with a space instead of an underscore.  That is, you can think of
; there being two ways to write a version string, with a space or an
; underscore at that particular position.  They are interchangeable --
; for purposes of this inequality comparison only (not certainly not
; for equality comparisons!).  But by convention we never write them
; with the underscore in our source code except when we are referring
; to absolute, fixed version numbers that do not get bumped.  If
; either version1 or version2 is not a legal ACL2 Version string,
; i.e., "ACL2 Version ...", it is defaulted to the earliest possible
; version.

  (let* ((root "ACL2 Version")
         (v1 (if (and (stringp version1)
                      (<= 13 (length version1))
                      (equal (subseq version1 0 12) root)
                      (or (eql (char version1 12) #\Space)
                          (eql (char version1 12) #\_)))
                 (subseq version1 13 (length version1))
               ""))
         (v2 (if (and (stringp version2)
                      (<= 13 (length version2))
                      (equal (subseq version2 0 12) root)
                      (or (eql (char version2 12) #\Space)
                          (eql (char version2 12) #\_)))
                 (subseq version2 13 (length version2))
               "")))
    (alpha-< v1 v2)))

(defun acl2-version-r-p (version)
  (let ((p (position #\( version)))
    (and p
         (< (+ p 2) (length version))
         (equal (subseq version p (+ p 3)) "(r)"))))

; We need to check whether an include-book-alist found in a .cert file
; is legal.  It will be checked with respect to the version number
; stored in the file, since we sometimes change the shape of the
; alist.  We also need to convert old-style alists into new-style
; ones.  We do the same thing for every entry in the alist, depending
; on the version.  Rather than repeatedly asking whether the stored
; version is before or after some fixed landmark version number, we
; ask that question once and for all and convert the answer to an integer
; indicating which ``era'' the version was in.  Then we just case split
; on the era.

(defun acl2-version-era (version)
  (cond
   ((earlier-acl2-versionp version "ACL2 Version_2.5")
    0)
   (t 1)))

(defun cert-annotationsp (x)
  (case-match x
              (((':SKIPPED-PROOFSP . sp)
                (':AXIOMSP . ap))
               (and (member sp '(t nil ?))
                    (member ap '(t nil ?))
                    t))
              (& nil)))

(defun include-book-alist-entryp (era entry)
  (case era
    (0 (and (consp entry)
            (stringp (car entry))         ;;; full-book-name
            (consp (cdr entry))
            (or (stringp (cadr entry))    ;;; user-book-name
                (structured-path-p (cadr entry)))
            (consp (cddr entry))
            (stringp (caddr entry))       ;;; familiar-name
            (or (integerp (cdddr entry))  ;;; ev-lst-chk-sum
                (eq (cdddr entry) nil))))
    (t (and (consp entry)
            (stringp (car entry))         ;;; full-book-name
            (consp (cdr entry))
            (or (stringp (cadr entry))    ;;; user-book-name
                (structured-path-p (cadr entry)))
            (consp (cddr entry))
            (stringp (caddr entry))       ;;; familiar-name
            (consp (cdddr entry))
            (cert-annotationsp (cadddr entry)) ;;; cert-annotations
            (or (integerp (cddddr entry))      ;;; ev-lst-chk-sum
                (eq (cddddr entry) nil))))))

(defun include-book-alistp1 (era x local-markers-allowedp)
  (cond
   ((atom x) (equal x nil))
   ((and local-markers-allowedp
         (consp (car x))
         (eq (car (car x)) 'local)
         (consp (cdr (car x)))
         (equal (cddr (car x)) nil))
    (and (include-book-alist-entryp era (cadr (car x)))
         (include-book-alistp1 era
                               (cdr x)
                               local-markers-allowedp)))
   (t (and (include-book-alist-entryp era (car x))
           (include-book-alistp1 era
                                 (cdr x)
                                 local-markers-allowedp)))))

(defun include-book-alistp (version x local-markers-allowedp)

; We check whether x is a legal include-book-alist in the given
; version.  If local-markers-allowedp we consider entries of the form
; (LOCAL e) to be legal if e is legal; otherwise, LOCAL is given no
; special meaning.

  (or (include-book-alistp1 (acl2-version-era version)
                            x local-markers-allowedp)

; If it fails to be an alist of the proper shape, but the version is
; 2.5, it could be an old-style one.  We use the strange code below to
; recognize "ACL2 Version 2.5" simply to avoid having that string in
; our code during version bump.  Because we sometimes pass underscore
; versions (as we do with "ACL2 Version_1.8" in chk-certificate-file1)
; we might someday pass "ACL2 Version_2.5" into this code, even though
; it never occurs in books.  So we look for it too.

      (if (or (equal version "ACL2 Version_2.5")
              (and (stringp version)
                   (equal (length version) 16)
                   (equal (subseq version 0 13) "ACL2 Version ")
                   (equal (subseq version 13 16) "2.5")))
          (include-book-alistp1 0 x local-markers-allowedp)
        nil)))

; Now we repeat that for the function that modernizes an alist
; into the current era.

(defun modernize-include-book-alist-entry (era entry)
  (case era
    (0 (list* (car entry)               ;;; full-book-name
              (cadr entry)              ;;; user-book-name
              (caddr entry)             ;;; familiar-name
              '((:SKIPPED-PROOFSP . ?)  ;;; cert-annotations
                (:AXIOMSP . ?))
              (cdddr entry)))           ;;; ev-lst-chk-sum
    (t entry)))

(defun modernize-include-book-alist1 (era alist)
  (cond
   ((endp alist) nil)
   (t (cons (if (eq (car (car alist)) 'local)
                (list 'local
                      (modernize-include-book-alist-entry
                       era
                       (cadr (car alist))))
              (modernize-include-book-alist-entry era (car alist)))
            (modernize-include-book-alist1 era (cdr alist))))))

(defun modernize-include-book-alist (version alist)

; We modernize the alist to the current ACL2 version.  We assume it has passed
; include-book-alistp with this same version and whatever the appropriate
; LOCAL-allowed flag.  It may or may not contain LOCAL markers.  If so, we
; modernize them.

  (let ((era (acl2-version-era version)))
    (if (include-book-alistp1 era alist t)
        (modernize-include-book-alist1 era alist)
      (modernize-include-book-alist1 0 alist))))

(defun chk-raise-portcullis (version file1 file2 ch skip-pre-alist-chkp
                                     ctx state
                                     suspect-book-action-alist evalp)

; File1 is a book and file2 is its certificate file.  The version string
; recorded with the file is version.  Ch is an open object input channel to the
; certificate.  We have already read past the initial (in-package "ACL2"),
; *acl2-version* and the :BEGIN-PORTCULLIS-CMDS in ch.  We now read successive
; commands and, if evalp is true, evaluate them in state.  Ld-skip-proofsp is
; 'include-book for this operation because these commands have all been
; successfully carried out in a boot strap world.  If this doesn't cause an
; error, then we read the pre- and post- check sum alists and the final check
; sum.  If these objects are not present or are of the wrong type, or there is
; additional text in the file, or the final check sum is inaccurate, we cause
; an error.

; Unless we are told to ignore the pre-alist, we check that it is a subset of
; the current include-book-alist.  Failure of this check may lead either to an
; error or to the assumption that the book is uncertified, according to the
; suspect-book-action-alist.  If we don't cause an error we return either the
; certificate object, which is ((cmds . pre-alist) . post-alist), or else we
; return nil, indicating that the book is presumed uncertified.

  (er-let*
    ((portcullis-cmds (if evalp
                          (chk-raise-portcullis1 file1 file2 ch ctx state)
                        (get-cmds-from-portcullis file1 file2 ch ctx state nil))))
    (mv-let
     (eofp pre-alist state)
     (state-global-let*
      ((infixp nil))
      (read-object ch state))
     (er-let*
       ((modern-pre-alist
         (cond ((include-book-alistp version pre-alist nil)
                (value (modernize-include-book-alist version pre-alist)))
               (t (er soft ctx *ill-formed-certificate-msg* file1 file2)))))
       (let ((actual-alist (global-val 'include-book-alist (w state))))
         (cond
          (eofp
           (er soft ctx *ill-formed-certificate-msg* file1 file2))
          (t
           (mv-let
            (eofp post-alist3 state)
            (state-global-let*
             ((infixp nil))
             (read-object ch state))
            (er-let*
              ((modern-post-alist3
                (cond
                 ((include-book-alistp version post-alist3 t)
                  (value
                   (modernize-include-book-alist version post-alist3)))
                 (t (er soft ctx *ill-formed-certificate-msg* file1 file2)))))
              (cond
               (eofp
                (er soft ctx *ill-formed-certificate-msg* file1 file2))
               (t
                (mv-let
                 (eofp chk-sum1 state)
                 (state-global-let*
                  ((infixp nil))
                  (read-object ch state))
                 (cond
                  ((or eofp (not (integerp chk-sum1)))
                   (er soft ctx *ill-formed-certificate-msg* file1 file2))
                  (t
                   (mv-let
                    (eofp temp state)
                    (state-global-let*
                     ((infixp nil))
                     (read-object ch state))
                    (declare (ignore temp))
                    (cond
                     ((not eofp)
                      (er soft ctx *ill-formed-certificate-msg* file1 file2))
                     (t
                      (let ((certificate-object
                             (cons (cons portcullis-cmds
                                         pre-alist)
                                   post-alist3))
                            (modern-certificate-object
                             (cons (cons portcullis-cmds
                                         modern-pre-alist)
                                   modern-post-alist3)))
                        (mv-let
                         (chk-sum2 state)
                         (check-sum-obj certificate-object state)
                         (cond
                          ((or (not (integerp chk-sum2))
                               (not (int= chk-sum1 chk-sum2)))
                           (er soft ctx *ill-formed-certificate-msg*
                               file1 file2))
                          ((and (not skip-pre-alist-chkp)
                                (not (include-book-alist-subsetp
                                      modern-pre-alist
                                      actual-alist)))

; Note: Sometimes I have wondered how the expression above deals with
; LOCAL entries in the alists in question, because
; include-book-alist-subsetp does not handle them.  The answer is:
; there are no LOCAL entries in a pre-alist because we prohibit local
; events in the portcullis commands.

                           (include-book-er
                            file1 file2
                            (cons
                             "The portcullis for ~x0 requires ~*3 ~
                                but we have ~*4."
                             (list
                              (cons #\3 (tilde-*-book-check-sums-phrase
                                         t
                                         modern-pre-alist
                                         actual-alist))
                              (cons #\4 (tilde-*-book-check-sums-phrase
                                         nil
                                         modern-pre-alist
                                         actual-alist))))
                            :uncertified-okp
                            suspect-book-action-alist
                            ctx state))
                          (t (value modern-certificate-object)))))))))
                  )))))))))))))

(defun chk-certificate-file1 (file1 file2 ch skip-pre-alist-chkp
                                    ctx state suspect-book-action-alist
                                    evalp)

; File1 is a book name and file2 is its associated certificate file name.  Ch
; is a channel to file2.  We assume we have read the initial (in-package
; "ACL2") and temporarily slipped into that package.  Our caller will restore
; it.  We now read the rest of file2 and either open the portcullis (skipping
; evaluation if evalp is nil) and return the certificate object ((cmds .
; pre-alist) . post-alist) or nil if we are assuming the book, or we cause an
; error.

; The code below is tedious and we here document it.  The first thing we look
; for is the ACL2 Version number printed immediately after the in-package.
; This function is made more complicated by four facts.  First, until Version
; 1.9, certificates did not include the version but we will treat such early
; books as though they were certified with Version 1.8.  Second, we do not
; know for sure that the certificate file is well-formed in any version.
; Third, we do not know whether include-book-er causes an error or just prints
; a warning (because that is determined by suspect-book-action-alist and the
; values of the state globals defaxioms-okp-cert and skip-proofs-okp-cert).
; Suppose we read a purported version string, val and it is not
; *acl2-version*.  Then we cause an include-book-er which may or may not
; signal an error.  If it does not then we are to assume the uncertified book
; so we must proceed with the certificate check as though the version were ok.
; Basically this means we want to call chk-raise-portcullis, but we must first
; make sure we've read to the beginning of the portcullis.  If val is
; :BEGIN-PORTCULLIS-CMDS, then this file is probably a well-formed Version 1.8
; file and we are properly positioned.  If val looks like an ACL2 Version
; string, then this file is probably a well-formed Version 1.9+ file and we
; must read the :BEGIN-PORTCULLIS-CMDS before proceeding.  Otherwise, this
; isn't well-formed and we cause an error.

; The fourth complication is that until late in the development of
; Version 2.5, include-book-alist entries were of the form
; (full-book-name user-book-name familiar-name . ev-lst-chk-sum)
; instead of the current (full-book-name user-book-name familiar-name
; cert-annotations . ev-lst-chk-sum).  Thus, when we read an alist, we
; must modernize it.

  (mv-let
   (eofp version state)
   (state-global-let* ((infixp nil)) (read-object ch state))
   (cond
    (eofp (er soft ctx *ill-formed-certificate-msg* file1 file2))
    ((equal version *acl2-version*)
     (mv-let
      (eofp key state)
      (state-global-let* ((infixp nil)) (read-object ch state))
      (cond
       ((or eofp (not (eq key :BEGIN-PORTCULLIS-CMDS)))
        (er soft ctx *ill-formed-certificate-msg* file1 file2))
       (t (chk-raise-portcullis version file1 file2 ch skip-pre-alist-chkp
                                ctx state
                                suspect-book-action-alist
                                evalp)))))
    ((not (equal (acl2-version-r-p *acl2-version*)
                 (acl2-version-r-p version)))
     (er soft ctx
         "We do not permit ACL2 books to be processed by ACL2(r) or ~
          vice versa.  The book ~x0 was last certified with ~s1 but this is ~
          ~s2."
         file1
         version
         *acl2-version*))
    (t
     (mv-let
      (erp val state)
      (include-book-er
       file1 file2
       (cons "~x0 was apparently certified with ~sa.  The inclusion of this ~
              book in the current ACL2 may render this ACL2 sesion unsound!  ~
              We recommend you recertify the book with the current version, ~
              ~sb.  See :DOC version.  No compiled file will be loaded with ~
              this book."
             (list (cons #\a (if (eq version :BEGIN-PORTCULLIS-CMDS)
                                 "ACL2 Version 1.8"
                               version))
                   (cons #\b *acl2-version*)))
       :uncertified-okp
       suspect-book-action-alist
       ctx state)

; Because the book was certified under a different version of ACL2, we
; consider it uncertified and, at best, return nil rather than a
; certificate object below.  Of course, we might yet cause an error.

      (cond
       (erp (mv erp val state))
       ((eq version :BEGIN-PORTCULLIS-CMDS)
        (er-progn
         (chk-raise-portcullis "ACL2 Version_1.8"
                               file1 file2 ch skip-pre-alist-chkp
                               ctx state
                               suspect-book-action-alist
                               t)
         (value nil)))
       ((and (stringp version)
             (<= 13 (length version))
             (equal (subseq version 0 13) "ACL2 Version "))
        (mv-let
         (eofp key state)
         (state-global-let* ((infixp nil)) (read-object ch state))
         (cond
          ((or eofp (not (eq key :BEGIN-PORTCULLIS-CMDS)))
           (er soft ctx *ill-formed-certificate-msg* file1 file2))
          (t (er-progn
              (chk-raise-portcullis version file1 file2 ch skip-pre-alist-chkp
                                    ctx state suspect-book-action-alist t)
              (value nil))))))
       (t (er soft ctx *ill-formed-certificate-msg* file1 file2))))))))

(defun chk-certificate-file (file1 skip-pre-alist-chkp ctx state
                                   suspect-book-action-alist evalp)

; File1 is a full book name.  We see whether there is a certificate on file for
; it.  If so, and we can get past the portcullis (evaluating it if evalp is
; true), we return the certificate object ((cmds . pre-alist) . post-alist) or
; nil if we presume the book is uncertified.

; This function may actually execute some events or even some DEFPKGs as part
; of the raising of the portcullis in the case that evalp is true.  If
; skip-pre-alist-chkp is t, we do not enforce the requirement that the books
; included by the portcullis commands have the specified check sums.  This
; feature is used when we use this function to recover from an old certificate
; the portcullis commands to recertify the file.

; We make the convention that if a file has no certificate or has an invalid
; certificate, we will either assume it anyway or cause an error depending on
; suspect-book-action-alist.  In the case that we pronouce this book
; uncertified, we return nil.

  (let ((file2 (convert-book-name-to-cert-name file1)))
    (mv-let
     (ch state)
     (open-input-channel file2 :object state)
     (cond
      ((null ch)
       (include-book-er file1 file2
                        "There is no certificate on file for ~x0."
                        :uncertified-okp
                        suspect-book-action-alist
                        ctx state))
      (t (er-let*
          ((pkg 
            (state-global-let*
             ((infixp nil))
             (chk-in-package ch file2 ctx state))))
          (cond
           ((not (equal pkg "ACL2"))
            (er soft ctx *ill-formed-certificate-msg* file1 file2))
           (t
            (state-global-let*
             ((current-package "ACL2"))
             (mv-let (error-flg val state)
                     (chk-certificate-file1 file1 file2 ch
                                            skip-pre-alist-chkp
                                            ctx state
                                            suspect-book-action-alist evalp)
                     (pprogn (close-input-channel ch state)
                             (mv error-flg val state))))))))))))

; All of the above is used during an include-book to verify that a
; certificate is well-formed and to raise the portcullis of the book.
; It happens that the code is also used by certify-book to recover the
; portcullis of a book from an old certificate.  We now continue with
; certify-book's checking, which next moves on to the question of
; whether the environment in which certify-book was called is actually
; suitable for a certification.

(defun chk-acceptable-certify-book1 (file k cmds wrld-segs wrld-list
                                          wrld ctx state)

; This function is checking the appropriateness of the environment in
; which certify-book is called.

; This subroutine is called after we have the k proposed portcullis
; commands and wrld.  It must be the case that cmds is returned by
; (get-portcullis-cmds wrld nil nil nil).  We supply cmds simply because
; the caller already has it.

; Unless we cause an error, we return the portcullis of the file,
; namely (cmds . pre-alist).

  (let* ((pre-alist (global-val 'include-book-alist wrld))
         (uncert-books (collect-uncertified-books pre-alist)))
    (cond
     ((not (eq (default-defun-mode wrld) :logic))
      (er soft ctx
          "Books must be certified in :LOGIC mode.  The current mode is ~x0."
          (default-defun-mode wrld)))
     ((and (not (integerp k))
           (not (eq k '?)))
      (er soft ctx
          "The second argument to certify-book must be either ~x0, ~x1, or an ~
           integer.  You supplied ~x2.  See :DOC certify-book."
          t '? k))
     ((and (not (equal k (length cmds)))
           (not (eq k '?)))
      (er soft ctx
          "You indicated that the portcullis for ~x0 would be of ~
           length ~x1 but it is actually of length ~x2.  Perhaps you ~
           had better inspect the world and call certify-book again."
          file k (length cmds)))
     ((assoc-equal file pre-alist)

; Why do we do this?  By insuring that file is not in the include-book-alist
; initially, we ensure that it gets into the alist only at the end when we
; include-book the book.  This lets us cdr it off.  If it happened to be
; the alist initially, then the include-book would not add it and the cdr
; wouldn't remove it.  See the end of the code for certify-book.

      (er soft ctx
          "We cannot certify ~x0 in a world in which it has already ~
           been included."
          file))
     (t (let ((names
               
; Warning: If you change the list of names below, be sure to change it
; in the call of note-certification-world in certify-book-fn.

               (cons 'defpkg *primitive-event-macros*)))
          (er-progn
           (chk-embedded-event-form-lst cmds wrld ctx state names t nil nil)
           (cond
            (uncert-books
             (er soft ctx
                 "It is impossible to certify any book in the current world ~
                  because it is built upon ~*0 which ~#1~[is~/are~] ~
                  uncertified."
                 (tilde-*-&v-strings '& uncert-books #\,)
                 uncert-books))
            ((eq wrld-segs :omitted)
             (value (cons cmds pre-alist)))

; Now that we know we have a list of embedded event forms, we are ready to
; replace relative pathnames by absolute pathnames.  See fix-portcullis-cmds.

            (t (er-let* ((fixed-cmds
                          (fix-portcullis-cmds cmds wrld-segs wrld-list
                                               (global-val 'known-package-alist
                                                           (car wrld-list))
                                               nil names
                                               (os wrld) ctx state)))
                        (value (cons fixed-cmds pre-alist)))))))))))

(defun chk-acceptable-certify-book (book-name full-book-name k ctx state
                                              suspect-book-action-alist)

; This function determines that it is ok to run certify-book on
; full-book-name and k.  Unless an error is caused we return a pair
; (cmds . pre-alist) that contains the two parts of the portcullis.
; If k is t it means that the existing certificate file specifies the
; intended portcullis.  It also means that there must be such a file
; and that we are in the ground zero state.  If all those things check
; out, we will actually carry out the portcullis to get into the right
; state by the time we return.

  (er-progn
   (cond ((ld-skip-proofsp state)
          (er soft ctx
              "Certify-book must be called while ld-skip-proofsp set to nil."))
         ((f-get-global 'in-local-flg state)
          (er soft ctx
              "Certify-book may not be called inside a LOCAL command."))
         ((global-val 'skip-proofs-seen (w state))
          (er soft ctx
              "At least one command in the current ACL2 world was executed ~
               while the value of state global variable 'LD-SKIP-PROOFSP was ~
               not nil:~|~%  ~x0~|~%(If you did not explicitly use ~
               set-ld-skip-proofsp or call ld with :ld-skip-proofsp not nil, ~
               then some other function did so, for example, rebuild.)  ~
               Certification is therefore not allowed in this world.  If the ~
               intention was for proofs to be skipped for one or more events ~
               in the certification world, consider wrapping those events ~
               explicitly in skip-proofs forms.  See :DOC skip-proofs."
              (global-val 'skip-proofs-seen (w state))))
         (t (value nil)))
   (chk-book-name book-name full-book-name ctx state)
   (er-let*
    ((certp (certificate-filep full-book-name state)))
    (let ((wrld (w state)))
      (mv-let
       (cmds wrld-segs wrld-list)
       (get-portcullis-cmds wrld nil nil nil)
       (cond
        ((eq k t)
         (cond
          (cmds
           (er soft ctx
               "When you tell certify-book to recover the certification world ~
                from the old certificate, you must call certify-book in the ~
                initial ACL2 logical world -- so we don't have to worry about ~
                the certification world  clashing with the existing logical ~
                world.  But you are not in the initial logical world.  Use ~
                :pbt 1 to see the world."))
          ((not certp)
           (er soft ctx
               "There is no certificate on file for ~x0.  But you told ~
                certify-book to recover the certi~-fication world from the ~
                old certificate.  You will have to construct the ~
                certi~-fication world by hand (by executing the desired ~
                commands in the current logical world) and then call ~
                certify-book again."
               full-book-name))
          (t

; So k is t, we are in the initial world, and there is a certificate file
; from which we can recover the portcullis.  Do it.

           (er-let*
            ((cert-obj
              (chk-certificate-file full-book-name t ctx state
                                    (cons '(:uncertified-okp . nil)
                                          suspect-book-action-alist)
                                    t)))

; Cert-obj is of the form ((cmds . pre-alist) . post-alist).

            (chk-acceptable-certify-book1 full-book-name
                                          (length (caar cert-obj)) ;; k
                                          (caar cert-obj)          ;; cmds
                                          :omitted                 ;; wrld-segs
                                          wrld-list
                                          (w state)
                                          ctx state)))))
        (t (chk-acceptable-certify-book1 full-book-name k cmds wrld-segs
                                         wrld-list wrld ctx state))))))))

(defun print-objects (lst ch state)
  (cond ((null lst) state)
        (t (pprogn (print-object$ (car lst) ch state)
                   (print-objects (cdr lst) ch state)))))

(defun make-certificate-file (file portcullis post-alist1 post-alist2 state)

; We assume file satisfies chk-book-name.  The portcullis is a pair (cmds
; . pre-alist), where cmds is the list of portcullis commands that created the
; world in which the certification was done, and pre-alist is the
; include-book-alist just before certification was done.  Post-alist1 is the
; include-book-alist after proving the events in file and post-alist2 is the
; include-book-alist after just including the events in file.  If they are
; different it is because the book included some subbooks within LOCAL forms
; and those subbooks did not get loaded for post-alist2.

; To verify that a subsequent inclusion is ok, we really only need
; post-alist2.  That is, if the book included some LOCAL subbook then it is
; not necessary that that subbook even exist when we include the main book.
; On the other hand, it might be useful to know what version of the subbook we
; used during certification, although the code at the moment makes no use of
; that.  So we massage post-alist1 so that any subbook in it that is not in
; post-alist2 is marked LOCAL.  Thus, post-alist3, below, will be of the form

; ((full1 user1 familiar1 cert-annotations1 . chk-sum1)
;  ...
;  (LOCAL (fulli useri familiari cert-annotationsi . chk-sumi))
;  ...
;  (fullk userk familiark cert-annotationsk . chk-sumk))

; and thus is not really an include-book-alist.  By deleting the LOCAL
; elements from it we obtain post-alist2.

; We write a certificate file for file.  The certificate file has the
; following form:

; (in-package "ACL2")
; "ACL2 Version x.y"
; :BEGIN-PORTCULLIS-CMDS  ; this is here just to let us check that the file
; cmd1                    ; is not a normal list of events.
; ...
; cmdk
; :END-PORTCULLIS-CMDS
; pre-alist
; post-alist3
; chk-sum

; where chk-sum is the check sum of ((cmds . pre-alist) . post-alist3).

; The reason the portcullis commands are written this way, rather than
; as a single object, is that we can't read them all at once since
; they may contain DEFPKGs.  We have to read and eval the cmdi
; individually.

  (let ((certification-file (convert-book-name-to-cert-name file))
        (post-alist3 (mark-local-included-books post-alist1 post-alist2)))
    (mv-let
     (chk-sum state)
     (check-sum-obj (cons portcullis post-alist3) state)
     (cond
      ((not (integerp chk-sum))
       (value (er hard 'make-certificate-file
                  "Check-sum-obj returned a non-integerp value on the ~
                   portcullis and post-alist3!")))
      (t
       (mv-let
        (ch state)
        (open-output-channel certification-file :object state)
        (cond
         ((null ch)
          (er soft 'certify
              "We cannot open a certificate file for ~x0.  The file ~
               we tried to open for output was ~x1."
              file
              certification-file))
         (t (state-global-let*
             ((current-package "ACL2"))
             (pprogn
              (print-object$ '(in-package "ACL2") ch state)
              (print-object$ *acl2-version* ch state)
              (print-object$ :BEGIN-PORTCULLIS-CMDS ch state)
              (print-objects (car portcullis) ch state)
              (print-object$ :END-PORTCULLIS-CMDS ch state)
              (print-object$ (cdr portcullis) ch state)
              (print-object$ post-alist3 ch state)
              (print-object$ chk-sum ch state)
              (close-output-channel ch state)
              (value certification-file)))))))))))

; We now develop a general-purpose read-object-file, which expects
; the given file to start with an IN-PACKAGE and then reads into that
; package all of the remaining forms of the file, returning the list
; of all forms read.

(defun open-input-object-file (file ctx state)

; If this function returns without error, then a channel is returned.
; In our use of this function in INCLUDE-BOOK we know file is a string.
; Indeed, it is a book name.  But we write this function slightly more
; ruggedly so that read-object-file, below, can be used on an
; arbitrary alleged file name.

  (cond ((stringp file)
         (mv-let (ch state)
                 (open-input-channel file :object state)
                 (cond ((null ch)
                        (er soft ctx
                            "There is no file named ~x0 that can be ~
                             opened for input."
                            file))
                       (t (value ch)))))
        (t (er soft ctx
               "File names in ACL2 must be strings, so ~x0 is not a ~
                legal file name."
               file))))

(defun read-object-file1 (channel state ans)

; Channel is an open input object channel.  We have verified that the
; first form in the file is an in-package and we are now in that
; package.  We read all the remaining objects in the file and return
; the list of them.

  (mv-let (eofp val state)
          (read-object channel state)
          (cond (eofp (value (reverse ans)))
                (t (read-object-file1 channel state (cons val ans))))))

(defun read-object-file (file ctx state)

; We open file for object input (causing an error if file is
; inappropriate).  We then get into the package specified by the
; (in-package ...) at the top of file, read all the objects in file,
; return to the old current package, close the file and exit,
; returning the list of all forms read (including the IN-PACKAGE).

  (er-let* ((ch (open-input-object-file file ctx state))
            (new-current-package (chk-in-package ch file ctx state)))
           (state-global-let*
            ((current-package new-current-package))
            (er-let* ((lst (read-object-file1 ch state nil)))
                     (let ((state (close-input-channel ch state)))
                       (value (cons (list 'in-package new-current-package)
                                    lst)))))))

(defun collect-ideal-user-defuns1 (tl wrld ans)
  (cond
   ((or (null tl)
        (and (eq (caar tl) 'command-landmark)
             (eq (cadar tl) 'global-value)
             (equal (access-command-tuple-form (cddar tl))
                    '(exit-boot-strap-mode))))
    ans)
   ((and (eq (caar tl) 'cltl-command)
         (eq (cadar tl) 'global-value)
         (equal (caddar tl) 'defuns))
    (collect-ideal-user-defuns1
     (cdr tl)
     wrld
     (cond
      ((null (cadr (cddar tl)))

 ; Defun-mode-flg = nil means encapsulate or :non-executable.  In this case we
 ; do not pick up the function, but that's OK because we don't care if it is
 ; executed efficiently.

       ans)
      ((eq (symbol-class (caar (cdddr (cddar tl))) wrld) :ideal)
       (append (strip-cars (cdddr (cddar tl))) ans))
      (t ans))))
   (t (collect-ideal-user-defuns1 (cdr tl) wrld ans))))

(defun collect-ideal-user-defuns (wrld)

; We scan wrld down to command 0 (but not into prehistory), collecting those
; fns which were (a) introduced with defun or defuns and (b) are :ideal.

  (collect-ideal-user-defuns1 wrld wrld nil))

(defun cbd-fn (state)
  (or (f-get-global 'connected-book-directory state)
      (er hard 'cbd
          "The connected book directory has apparently not yet been set.  ~
           This could be a sign that the top-level ACL2 loop, generally ~
           entered using (LP), has not yet been entered.")))

(defmacro cbd nil
  #-small-acl2-image
  ":Doc-Section Books

  connected book directory string~/
  ~bv[]
  Example:
  ACL2 !>:cbd
  \"/usr/home/smith/\"
  ~ev[]
  The connected book directory is a nonempty string that specifies a
  directory as an absolute pathname.  (~l[pathname] for a
  discussion of file naming conventions.)  When ~ilc[include-book] is given
  a relative book name it elaborates it into a full book name,
  essentially by appending the connected book directory string to the
  left and ~c[\".lisp\"] to the right.  (For details,
  ~pl[book-name] and also ~pl[full-book-name].)  Furthermore,
  ~ilc[include-book] temporarily sets the connected book directory to the
  directory string of the resulting full book name so that references
  to inferior ~il[books] in the same directory may omit the directory.
  ~l[set-cbd] for how to set the connected book directory string.~/
  ~bv[]
  General Form:
  (cbd)
  ~ev[]
  This is a macro that expands into a term involving the single free
  variable ~ilc[state].  It returns the connected book directory string.

  The connected book directory (henceforth called the ``~c[cbd]'') is
  used by ~ilc[include-book] to elaborate the supplied book name into a
  full book name (~pl[full-book-name]).  For example, if the ~c[cbd]
  is ~c[\"/usr/home/smith/\"] then the elaboration of the ~il[book-name]
  ~c[\"project/task-1/arith\"] (to the ~c[\".lisp\"] extension) is
  ~c[\"/usr/home/smith/project/task-1/arith.lisp\"].  That
  ~il[full-book-name] is what ~il[include-book] opens to read the
  source text for the book.

  The ~c[cbd] may be changed using ~ilc[set-cbd] (~pl[set-cbd]).
  Furthermore, during the processing of the ~il[events] in a book,
  ~ilc[include-book] sets the ~c[cbd] to be the directory string of the
  ~il[full-book-name] of the book.  Thus, if the ~c[cbd] is
  ~c[\"/usr/home/smith/\"] then during the processing of ~il[events] by
  ~bv[]
  (include-book \"project/task-1/arith\")
  ~ev[]
  the ~c[cbd] will be set to ~c[\"/usr/home/smith/project/task-1/\"].
  Note that if ~c[\"arith\"] recursively includes a subbook, say
  ~c[\"naturals\"], that resides on the same directory, the
  ~ilc[include-book] event for it may omit the specification of that
  directory.  For example, ~c[\"arith\"] might contain the event
  ~bv[]
    (include-book \"naturals\").
  ~ev[]
  In general, suppose we have a superior book and several inferior
  ~il[books] which are included by ~il[events] in the superior book.  Any
  inferior book residing on the same directory as the superior book
  may be referenced in the superior without specification of the
  directory.

  We call this a ``relative'' as opposed to ``absolute'' naming.  The
  use of relative naming is preferred because it permits ~il[books]
  (and their accompanying inferiors) to be moved between directories
  while maintaining their ~il[certificate]s and utility.  Certified
  ~il[books] that reference inferiors by absolute file names are unusable
  (and rendered uncertified) if the inferiors are moved to new
  directories.

  ~em[Technical Note and a Challenge to Users:]

  After elaborating the book name to a full book name, ~ilc[include-book]
  opens a channel to the file to process the ~il[events] in it.  In some
  host Common Lisps, the actual file opened depends upon a notion of
  ``connected directory'' similar to our connected book directory.
  Our intention in always elaborating book names into absolute
  filename strings (~pl[pathname] for terminology) is to
  circumvent the sensitivity to the connected directory.  But we may
  have insufficient control over this since the ultimate file naming
  conventions are determined by the host operating system rather than
  Common Lisp (though, we do check that the operating system
  ``appears'' to be one that we ``know'' about).  Here is a question,
  which we'll pose assuming that we have an operating system that
  calls itself ``Unix.''  Suppose we have a file name, filename, that
  begins with a slash, e.g., ~c[\"/usr/home/smith/...\"].  Consider two
  successive invocations of CLTL's
  ~bv[]
  (open filename :direction :input)
  ~ev[]
  separated only by a change to the operating system's notion of
  connected directory.  Must these two invocations produce streams to
  the same file?  A candidate string might be something like
  ~c[\"/usr/home/smith/*/usr/local/src/foo.lisp\"] which includes some
  operating system-specific special character to mean ``here insert
  the connected directory'' or, more generally, ``here make the name
  dependent on some non-ACL2 aspect of the host's state.''  If such
  ``tricky'' name strings beginning with a slash exist, then we have
  failed to isolate ACL2 adequately from the operating system's file
  naming conventions.  Once upon a time, ACL2 did not insist that the
  ~c[cbd] begin with a slash and that allowed the string
  ~c[\"foo.lisp\"] to be tricky because if one were connected to
  ~c[\"/usr/home/smith/\"] then with the empty ~c[cbd] ~c[\"foo.lisp\"]
  is a full book name that names the same file as
  ~c[\"/usr/home/smith/foo.lisp\"].  If the actual file one reads is
  determined by the operating system's state then it is possible for
  ACL2 to have two distinct ``full book names'' for the same file, the
  ``real'' name and the ``tricky'' name.  This can cause ACL2 to
  include the same book twice, not recognizing the second one as
  redundant."

  `(cbd-fn state))

(defun maybe-add-separator (str)
  (if (and (not (equal str ""))
           (eql (char str (1- (length str))) *directory-separator*))
      str
    (string-append str *directory-separator-string*)))

(defun set-cbd-fn (str state)
  (cond ((and (true-listp str)
              (structured-directory-p str))
         (set-cbd-fn (structured-directory-to-string (cbd) str)
                     state))
        ((not (stringp str))
         (er soft (cons 'set-cbd str)
             "~x0 does not have the syntax of an ~
                ACL2 directory name.  See :DOC cbd."
             str))
        ((absolute-pathname-string-p str nil (os (w state)))
         (assign connected-book-directory (maybe-add-separator str)))
        ((not (absolute-pathname-string-p
               (f-get-global 'connected-book-directory state)
               nil
               (os (w state))))
         (er soft (cons 'set-cbd str)
             "An attempt was made to set the connected book directory (cbd) ~
              using relative pathname ~p0, but surprisingly, the existing cbd ~
              is ~p1, which is not an absolute pathname.  See :DOC pathname."
             str
             (f-get-global 'connected-book-directory state)))
        (t
         (assign connected-book-directory
                 (maybe-add-separator
                  (our-merge-pathnames
                   (f-get-global 'connected-book-directory state)
                   str))))))

(defmacro set-cbd (str)

  #-small-acl2-image
  ":Doc-Section books

  to set the connected book directory~/
  ~bv[]
  Example Forms:
  ACL2 !>:set-cbd \"/usr/home/smith/\"
  ACL2 !>:set-cbd \"my-acl2/books\"
  ~ev[]
  ~l[cbd] for a description of the connected book directory.~/
  ~bv[]
  General Form:
  (set-cbd str)
  ~ev[]

  where ~c[str] is a nonempty string that represents the desired
  directory (~pl[pathname]).  This command sets the connected book
  directory (~pl[cbd]) to the string representing the indicated
  directory.  Thus, this command may determine which files are
  processed by ~ilc[include-book] and ~ilc[certify-book] ~il[command]s typed at the
  top-level.  However, the ~ilc[cbd] is also temporarily set by those two
  book processing ~il[command]s.

  ~sc[Important]:  Pathnames in ACL2 are in the Unix (trademark of AT&T)
  style.  That is, the character ``~c[/]'' separates directory components
  of a pathname, and pathnames are absolute when they start with this
  character, and relative otherwise.  ~l[pathname]."

  `(set-cbd-fn ,str state))

(defun extend-pathname (dir file-name os)

; Dir is a string representing an absolute directory name, and file-name is a
; string representing a file or directory name.  We want to extend dir by
; file-name if subdir is relative, and otherwise return file-name.

  (cond
   ((absolute-pathname-string-p file-name nil os)
    file-name)
   (t
    (our-merge-pathnames dir file-name))))

(defun parse-book-name (dir x extension os)

; This function takes a directory name, dir, and a user supplied book name, x,
; which may be a string or may be "structured", and returns (mv full dir
; familiar), where full is the full book name string, dir is the directory
; name, and familiar is the familiar name string.  Extension is either nil or
; a string such as ".lisp" and the full book name is given the extension if it
; is non-nil.

; Given dir                and x with extension=".lisp"
; "/usr/home/moore/"           "nasa-t3/arith"       ; user name
; this function produces
; (mv "/usr/home/moore/nasa-t3/arith.lisp"           ; full name
;     "/usr/home/moore/nasa-t3/"                     ; directory name
;     "arith")                                       ; familiar name

; On the other hand, if x is "/usr/home/kaufmann/arith" then the result is
; (mv "/usr/home/kaufmann/arith.lisp"
;     "/usr/home/kaufmann/"
;     "arith")

; We work with Unix-style pathnames.

; Note that this function merely engages in string processing.  It does not
; actually guarantee that the named file exists or that the various names are
; in any sense well-formed (except for some minimal checks in the case of
; structured pathnames).  It does not change the connected book directory.  If
; x is not a string and not well-formed as a structured pathname, the result is
; (mv nil nil x).  Thus, if the full name returned is nil, we know something is
; wrong and the short name returned is whatever junk the user supplied.

  (cond
   ((structured-path-p x)
    (let ((dir (structured-directory-to-string dir (butlast x 1)))
          (familiar (car (last x))))
      (mv (if extension
              (concatenate 'string dir familiar extension)
            (concatenate 'string dir familiar))
          dir
          familiar)))
   ((stringp x)
    (let* ((lst (coerce x 'list))
           (rlst (reverse lst))
           (temp (member *directory-separator* rlst)))

; If x is "project/task3/arith.lisp" then temp is "project/task3/" except is a
; list of chars and is in reverse order (!).

      (let ((familiar (coerce (reverse (first-n-ac
                                        (- (length x) (length temp))
                                        rlst nil))
                              'string))
            (dir1 (extend-pathname dir
                                   (coerce (reverse temp) 'string)
                                   os)))
        (mv (if extension
                (concatenate 'string dir1 familiar extension)
              (concatenate 'string dir1 familiar))
            dir1
            familiar))))
   (t (mv nil nil x))))

(defun chk-cert-annotations
  (cert-annotations portcullis-cmds full-book-name suspect-book-action-alist
                    ctx state)

; Warning: Chk-cert-annotations and chk-cert-annotations-post-alist are nearly
; duplicates of one another.  If you change one, e.g., to add a new kind of
; annotation and its checker, change the other.

  (er-progn
   (cond
    ((eq (cdr (assoc :skipped-proofsp cert-annotations)) nil)
     (value nil))
    ((eq (cdr (assoc :skipped-proofsp cert-annotations)) t)
     (include-book-er full-book-name nil
                      (if portcullis-cmds
                          "The book ~x0 (including events from its portcullis) ~
                           contains one or more SKIP-PROOFS events."
                        "The book ~x0 contains one or more SKIP-PROOFS events.")
                      :skip-proofs-okp
                      suspect-book-action-alist ctx state))
    (t (include-book-er full-book-name nil
                        (if portcullis-cmds
                            "The book ~x0 (including events from its ~
                             portcullis) may contain SKIP-PROOFS events."
                          "The book ~x0 may contain SKIP-PROOFS events.")
                        :skip-proofs-okp
                        suspect-book-action-alist ctx state)))
   (cond
    ((eq (cdr (assoc :axiomsp cert-annotations)) nil)
     (value nil))
    ((eq (cdr (assoc :axiomsp cert-annotations)) t)
     (include-book-er full-book-name nil
                      (if portcullis-cmds
                          "The book ~x0 (including events from its portcullis) ~
                           contains one or more DEFAXIOM events."
                        "The book ~x0 contains one or more DEFAXIOM events.")
                      :defaxioms-okp
                      suspect-book-action-alist ctx state))
    (t (include-book-er full-book-name nil
                        (if portcullis-cmds
                            "The book ~x0 (including events from its ~
                             portcullis) may contain DEFAXIOM events."
                          "The book ~x0 may contain DEFAXIOM events.")
                        :defaxioms-okp
                        suspect-book-action-alist ctx state)))))

(defun chk-cert-annotations-post-alist
  (post-alist portcullis-cmds full-book-name suspect-book-action-alist ctx
              state)
  
; Warning: Chk-cert-annotations and chk-cert-annotations-post-alist are nearly
; duplicates of one another.  If you change one, e.g., to add a new kind of
; annotation and its checker, change the other.

; We are in the process of including the book full-book-name.  Post-alist is
; its locally-marked include-book alist as found in the .cert file.  We look
; at every entry (LOCAL or not) and check that its cert annotations are
; consistent with the suspect-book-action-list.

  (cond
   ((endp post-alist) (value nil))
   (t 

; An entry in the post-alist is (full user familiar cert-annotations . chk).
; It may optionally be embedded in a (LOCAL &) form.

      (let* ((localp (eq (car (car post-alist)) 'local))
             (full-subbook (if localp
                               (car (cadr (car post-alist)))
                             (car (car post-alist))))
             (cert-annotations (if localp
                                   (cadddr (cadr (car post-alist)))
                                 (cadddr (car post-alist)))))
        (er-progn
         (cond
          ((eq (cdr (assoc :skipped-proofsp cert-annotations)) nil)
           (value nil))
          ((eq (cdr (assoc :skipped-proofsp cert-annotations)) t)
           (include-book-er
            full-book-name nil
            (cons "The book ~x0~sp~#a~[~/ locally~] includes ~xb, which ~
                   contains one or more SKIP-PROOFS events."
                  (list (cons #\a (if localp 1 0))
                        (cons #\b full-subbook)
                        (cons #\p (if portcullis-cmds
                                      " (including events from its portcullis)"
                                    ""))))
            :skip-proofs-okp
            suspect-book-action-alist ctx state))
          (t (include-book-er
              full-book-name nil
              (cons "The book ~x0~sp~#a~[~/ locally~] includes ~xb, which ~
                     may contain SKIP-PROOFS events."
                    (list (cons #\a (if localp 1 0))
                          (cons #\b full-subbook)
                          (cons #\p (if portcullis-cmds
                                        " (including events from its portcullis)"
                                      ""))))
              :skip-proofs-okp
              suspect-book-action-alist ctx state)))
         (cond
          ((eq (cdr (assoc :axiomsp cert-annotations)) nil)
           (value nil))
          ((eq (cdr (assoc :axiomsp cert-annotations)) t)
           (include-book-er
            full-book-name nil
            (cons "The book ~x0~sp~#a~[~/ locally~] includes ~xb, which ~
                   contains one or more DEFAXIOM events."
                  (list (cons #\a (if localp 1 0))
                        (cons #\b full-subbook)
                        (cons #\p (if portcullis-cmds
                                      " (including events from its portcullis)"
                                    ""))))
            :defaxioms-okp
            suspect-book-action-alist ctx state))
          (t (include-book-er
              full-book-name nil
              (cons "The book ~x0~sp~#a~[~/ locally~] includes ~xb, which ~
                     may contain DEFAXIOM events."
                    (list (cons #\a (if localp 1 0))
                          (cons #\b full-subbook)
                          (cons #\p (if portcullis-cmds
                                        " (including events from its ~
                                         portcullis)"
                                      ""))))
              :defaxioms-okp
              suspect-book-action-alist ctx state)))
         (chk-cert-annotations-post-alist (cdr post-alist)
                                          portcullis-cmds
                                          full-book-name
                                          suspect-book-action-alist
                                          ctx state))))))

(defun spontaneous-decertificationp1 (ibalist alist files)

; Ibalist is an include-book alist, while alist is the strip-cddrs of
; an include-book alist.  Thus, an entry in ibalist is of the form
; (full-book-name user-book-name familiar-name cert-annotations
; . ev-lst-chk-sum), while an entry in alist is (familiar-name
; cert-annotations . ev-lst-chk-sum).  We know, from context, that
; (subsetp-equal (strip-cddrs ibalist) alist) fails.  Thus, there are
; entries in ibalist that are not ``in'' alist, where ``in'' compares
; (familiar-name cert-annotations . ev-lst-chk-sum) tuples.  We
; determine whether each such entry fails only because the chk-sum in
; the ibalist is nil while that in a corresponding entry in the alist
; is non-nil.  If so, then the most likely explanation is that a
; concurrent process is recertifying certain books and deleted their
; .cert files.  We return the list of all files which have been
; decertified.

  (cond ((endp ibalist) files)
        (t (let* ((familiar-name1 (caddr (car ibalist)))
                  (cert-annotations1 (cadddr (car ibalist)))
                  (ev-lst-chk-sum1 (cddddr (car ibalist)))
                  (temp (assoc-equal familiar-name1 alist))
                  (cert-annotations2 (cadr temp))
                  (ev-lst-chk-sum2 (cddr temp)))
             (cond
              (temp
               (cond
                ((equal (cddr (car ibalist)) temp)

; This entry is identical to its mate in alist.  So we keep
; looking.
                 (spontaneous-decertificationp1 (cdr ibalist) alist files))
                ((and (or (null cert-annotations1)
                          (equal cert-annotations1 cert-annotations2))
                      (equal ev-lst-chk-sum1 nil)
                      ev-lst-chk-sum2)

; The full-book-name (car (car ibalist)) spontaneously decertified.
; So we collect it and keep looking.

                 (spontaneous-decertificationp1 (cdr ibalist) alist
                                                (cons (car (car ibalist))
                                                      files)))
                (t nil)))
              (t nil))))))

(defun spontaneous-decertificationp (alist1 alist2)

; We know that alist1 is not an include-book-alist-subset of alist2.
; We check whether this is precisely because some files which were
; certified in alist2 are not certified in alist1.  If so, we return
; the list of all such files.  But if we find any other kind of
; discrepancy, we return nil.

  (spontaneous-decertificationp1 alist1 (strip-cddrs alist2) nil))

(defun raw-mode-ever-entered-p (state)
  (and (f-boundp-global 'raw-mode-restore-lst state)
       (consp (f-get-global 'raw-mode-restore-lst
                            state))))

(defun raw-mode-error (function-name further-info)
  (er hard function-name
      "It is illegal to call ~s0 if raw mode has ever been ~
       entered during an ACL2 session.~s1"
      function-name (or further-info "")))

; The following code is used to determine whether the portcullis
; contains a skip-proofs and to note include-books.

(mutual-recursion

(defun note-certification-world (form wrld ctx state names
                                      suspect-book-action-alist)

; We know that form has passed the chk-embedded-event-form in
; chk-acceptable-certify-book1.  This function returns an error triple with a
; state in which state globals include-book-alist-state and skipped-proofsp
; have been updated to reflect form.  An error only occurs if we detect an
; uncertified included book.  The value component of the returned error triple
; is irrelevant.  Before Version_2.6 there was an analogous but inadequate
; function, find-a-skip-proofs; see the Essay on Skip-proofs.

  (cond ((atom form) (value nil)) ; This should never happen.
        ((eq (car form) 'skip-proofs)
         (pprogn (set-skipped-proofsp state)
                 (value nil)))
        ((eq (car form) 'encapsulate)
         (note-certification-world-lst (cddr form) wrld ctx state names
                                       suspect-book-action-alist))
        ((eq (car form) 'progn)
         (note-certification-world-lst (cdr form) wrld ctx state names
                                       suspect-book-action-alist))
        ((eq (car form) 'local) ; This should never happen in portcullis.
         (note-certification-world (cadr form) wrld ctx state names
                                   suspect-book-action-alist))
        ((eq (car form) 'value) (value nil))
        ((eq (car form) 'include-book)

; Why do we need to deal with include-book?  After all, if this include-book
; was executed, then can't we expect the include-book-alist entries for that
; book and all its subbooks, even locally-included subbooks, to be represented
; in the world global include-book-alist?  The problem is that the include-book
; form at hand may have been locally included inside an encapsulate.  In that
; case the information from its certificate will not show up in the world.  In
; fact we may not even catch a locally included uncertified book with our check
; in chk-acceptable-certify-book1.  But, we will catch that here.

         (mv-let
          (full-book-name directory-name familiar-name)
          (parse-book-name (cbd) (cadr form) ".lisp" (os wrld))
          (declare (ignore directory-name familiar-name))
          (mv-let (erp cert-obj state)
                  (chk-certificate-file full-book-name t ctx state
                                        suspect-book-action-alist
                                        nil)
                  (if erp
                      (er soft ctx
                          "Note:  The error reported just above is due to the ~
                          form ~x0 in the certification world.  ~
                          See :DOC certify-book."
                          form)
                    (pprogn (f-put-global 'include-book-alist-state
                                          (union-equal
                                           (cdr cert-obj) ; post-alist
                                           (f-get-global
                                            'include-book-alist-state
                                            state))
                                          state)
                            (value nil))))))
        ((member-eq (car form) names) (value nil))
        ((getprop (car form) 'macro-body nil 'current-acl2-world wrld)
         (er-let* ((expansion (macroexpand1 form ctx state)))
           (note-certification-world expansion wrld ctx state names
                                     suspect-book-action-alist)))
        (t (er soft ctx

; Since form has passed the chk-embedded-event-form in
; chk-acceptable-certify-book1, we should not be here.

               "Unexpected form in certification world:~%  ~x0"
               form))))

(defun note-certification-world-lst (forms wrld ctx state names
                                           suspect-book-action-alist)
  (cond
   ((null forms) (value nil))
   (t (er-progn
       (note-certification-world (car forms) wrld ctx state names
                                 suspect-book-action-alist)
       (note-certification-world-lst (cdr forms) wrld ctx state names
                                     suspect-book-action-alist)))))

)

(defun remove-duplicates-equal-from-end (lst acc)
  (cond ((endp lst) (reverse acc))
        ((member-equal (car lst) acc)
         (remove-duplicates-equal-from-end (cdr lst) acc))
        (t (remove-duplicates-equal-from-end (cdr lst) (cons (car lst) acc)))))

(defun certify-book-fn (user-book-name k compile-flg
                                       defaxioms-okp
                                       skip-proofs-okp
                                       state)

; We assume that all the referenced books in user-book-name have already been
; certified (we will check this assumption).

; Note that we want to inhibit proof-tree output in the summary, which is why
; we bind inhibit-output-lst so early.

 (if (raw-mode-ever-entered-p state)
     (value (raw-mode-error 'certify-book nil))
   (state-global-let*
    ((inhibit-output-lst 
      (add-to-set-eq 'proof-tree
                     (f-get-global 'inhibit-output-lst state)))
     (match-free-error nil)
     (defaxioms-okp-cert defaxioms-okp)
     (skip-proofs-okp-cert skip-proofs-okp)

; By binding guard-checking-on to nil, we allow defconst and value-triple forms
; to be evaluated in the logic.

     (guard-checking-on nil))
    (with-ctx-summarized
     (if (output-in-infixp state)
         (list* 'certify-book user-book-name
                (if (and (equal k 0) (eq compile-flg t))
                    nil
                  '(irrelevant)))
       (cons 'certify-book user-book-name))
     (let ((saved-acl2-defaults-table
            (table-alist 'acl2-defaults-table (w state)))

; If you add more keywords to this list, make sure you do the same to
; the list constructed by include-book-fn.  Also, you might wish to
; handle the new warning summary in warning1.

           (suspect-book-action-alist
            (list '(:uncertified-okp . nil)
                  (cons :defaxioms-okp defaxioms-okp)
                  (cons :skip-proofs-okp skip-proofs-okp)))
           #-acl2-loop-only
           (*inside-include-book-fn* t)
           #+(and clisp (not acl2-loop-only))
           (custom::*suppress-check-redefinition* t)
           #+(and allegro (not acl2-loop-only))
           (excl:*redefinition-warnings* nil))
       (mv-let
        (full-book-name directory-name familiar-name)
        (parse-book-name (cbd) user-book-name ".lisp" (os (w state)))
        (er-let*
         ((portcullis ; (cmds . pre-alist).
           (chk-acceptable-certify-book user-book-name full-book-name k
                                        ctx state
                                        suspect-book-action-alist)))
         (let* ((wrld1 (w state)))
           (pprogn
            (io? event nil state
                 (fms "CERTIFICATION ATTEMPT FOR ~x0~%~s1~%~%"
                      (list (cons #\0 full-book-name)
                            (cons #\1 *acl2-version*))
                      (proofs-co state) state nil))
            (io? event nil state
                 (fms "* Step 1:  Read ~x0 and compute its check sum.~%"
                      (list (cons #\0 full-book-name))
                      (proofs-co state)
                      state nil))
            (er-let*
             ((ev-lst (read-object-file full-book-name ctx state)))
             (mv-let
              (chk-sum state)
              (check-sum-obj ev-lst state)
              (cond
               ((not (integerp chk-sum))
                (er soft ctx
                    "The file ~x0 is not a legal list of embedded event forms ~
                     because it contains an object, ~x1, which check sum was ~
                     unable to handle."
                    full-book-name chk-sum))
               (t (pprogn
                   (io? event nil state
                        (fms "* Step 2:  There ~#0~[were no forms in the file. ~
                              Why are you making such a silly book?~/was one ~
                              form in the file.~/were ~n1 forms in the file.~] ~
                              The check sum is ~x2.  We now attempt to ~
                              establish that each form, whether local or ~
                              non-local, is indeed an admissible embedded ~
                              event form in the context of the previously ~
                              admitted ones.  Note that proof-tree output is ~
                              inhibited during this check; see :DOC ~
                              proof-tree.~%"
                             (list (cons #\0 (zero-one-or-more ev-lst))
                                   (cons #\1 (length ev-lst))
                                   (cons #\2 chk-sum))
                             (proofs-co state) state nil))
                   (er-let*
                    ((triple
                      (state-global-let*
                       (

; We ``accumulate'' into the flag skipped-proofsp whether there are any
; skip-proofs in sight.  See the Essay on Skip-proofs.

                        (skipped-proofsp nil)
                        (include-book-alist-state nil)

; We will accumulate into the flag axiomsp whether any axioms have been added,
; starting with those in the portcullis.  We can identify axioms in the
; portcullis by asking if the current nonconstructive axioms are different from
; those at the end of boot-strap.

                        (axiomsp (not
                                  (equal
                                   (global-val   ;;; axioms as of boot-strap
                                    'nonconstructive-axiom-names
                                    (scan-to-landmark-number
                                     'event-landmark
                                     (global-val 'event-number-baseline
                                                 wrld1)
                                     wrld1))
                                   (global-val   ;;; axioms now.
                                    'nonconstructive-axiom-names
                                    wrld1))))

                        (ld-redefinition-action nil)
                        (connected-book-directory directory-name))
                       (revert-world-on-error
                        (er-progn
                         (note-certification-world-lst
                          (car portcullis)
                          (w state)
                          ctx
                          state

; This list of names must be the same as in chk-acceptable-certify-book1.

                          (cons 'defpkg
                                *primitive-event-macros*)
                          suspect-book-action-alist)

; The fact that we are under 'certify-book means that all calls of
; include-book will insist that the :uncertified-okp action is nil, meaning
; errors will be caused if uncertified books are read.

                         (process-embedded-events
                          'certify-book
                          saved-acl2-defaults-table
                          nil (cadr (car ev-lst))
                          (list 'certify-book full-book-name)
                          (cdr ev-lst)
                          'certify-book state)
                         (value (list (f-get-global 'skipped-proofsp state)
                                      (f-get-global 'axiomsp state)
                                      (f-get-global 'include-book-alist-state state))))))))
                    (let* ((pass1-known-package-alist
                            (global-val 'known-package-alist (w state)))
                           (skipped-proofsp (car triple))
                           (axiomsp (cadr triple))
                           (new-include-book-alist-state (caddr triple))
                           (cert-annotations
                            (list 
                                
; We set :skipped-proofsp in the certification annotations to t or nil
; according to whether there were any skipped proofs in either the
; portcullis or the body of this book.

                             (cons :skipped-proofsp
                                   skipped-proofsp)

; We similarly set :axiomsp to t or nil.  Note that axioms in subbooks 
; are not counted as axioms in this one.

                             (cons :axiomsp
                                   axiomsp)))
                           (post-alist1
                            (cons (list* full-book-name
                                         user-book-name
                                         familiar-name
                                         cert-annotations
                                         chk-sum)
                                  new-include-book-alist-state)))
                      (er-progn
                       (chk-cert-annotations cert-annotations
                                             (car portcullis)
                                             full-book-name
                                             suspect-book-action-alist
                                             ctx state)
                       (pprogn
                        (io? event nil state
                             (fms "* Step 3:  That completes the admissibility ~
                                   check.  Each form read was an embedded ~
                                   event form and was admissible. We now ~
                                   retract back to the initial world and try ~
                                   to include the book.  This may expose local ~
                                   incompatibilities.~%"
                                  nil
                                  (proofs-co state) state nil))
                        (set-w 'retraction wrld1 state)
                        (er-let*
                         ((defpkg-items
                            (defpkg-items
                              pass1-known-package-alist
                              ctx wrld1 state))
                          (new-chk-sum
                           (state-global-let*
                            ((certify-book-file full-book-name)
                             (ld-redefinition-action nil))

; Note that we do not bind connected-book-directory before calling
; include-book-fn, because it will bind it for us.  We leave the directory set
; as it was when we parsed user-book-name to get full-book-name, so that
; include-book-fn will parse user-book-name the same way again.

                            (include-book-fn user-book-name state :try
                                             nil
                                             defaxioms-okp skip-proofs-okp
                                             nil nil nil))))
                         (let* (#-acl2-loop-only
                                (os-full-book-name
                                 (pathname-unix-to-os full-book-name state))
                                (wrld2 (w state))
                                (new-defpkg-list
                                 (new-defpkg-list
                                  defpkg-items
                                  (global-val 'known-package-alist wrld2)))
                                (bad-fns
                                 (collect-ideal-user-defuns
                                  wrld2))
                                (post-alist2
                                 (cons (list* full-book-name
                                              user-book-name
                                              familiar-name

; We use the cert-annotations from the first pass.  They are the ones that
; include the LOCAL events too.

                                              cert-annotations
                                              new-chk-sum)
                                       (cdr (global-val 'include-book-alist
                                                        wrld2)))))

; The cdr above removes the certification tuple stored by the
; include-book-fn itself.  That pair is guaranteed to be last one
; because it is the most recently added one (with add-to-set-equal)
; and we know it was not already a member of the list because
; chk-acceptable-certify-book1 checked that.  Could a file include
; itself?  It could try.  But if (include-book file) is one of the
; events in file, then the attempt to (include-book file) will cause
; infinite recursion -- because we don't put file on the list of files
; we've included (and hence recognize as redundant) until after we've
; completed the processing.

                           (pprogn
                            (cond ((and compile-flg bad-fns)
                                   (warning$
                                    ctx "Guards"
                                    "You have told certify-book to produce a ~
                                     compiled file for ~x0. However, the ~
                                     function~#1~[ ~&1 has not had its~/s ~&1 ~
                                     have not had their~] guards verified.  A ~
                                     compiled file will be produced anyway, ~
                                     but all bets are off if you load it into ~
                                     raw Common Lisp and try to run the ~
                                     functions in it.  See :DOC verify-guards."
                                    full-book-name
                                    bad-fns))
                                  (t state))
                            (cond
                             ((not (include-book-alist-subsetp post-alist2
                                                               post-alist1))
                              (let ((files (spontaneous-decertificationp
                                            post-alist2
                                            post-alist1)))
                                (cond
                                 (files
                                  (er soft ctx
                                      "During Step 3, we loaded the ~
                                       uncertified ~#0~[book ~&0.  This book ~
                                       was certified when we looked at ~
                                       it~/books ~&0. These books were ~
                                       certified when we looked at them~] in ~
                                       Step 2!  The most likely explanation is ~
                                       that some concurrent job, possibly by ~
                                       another user of your file system, is ~
                                       currently recertifying ~#0~[this ~
                                       book~/these books~] (or subbooks of ~
                                       ~#0~[it~/them~]).  That hypothetical ~
                                       job might have deleted the certificate ~
                                       files of the books in question, ~
                                       rendering ~#0~[this one~/these~] ~
                                       uncertified.  If this explanation seems ~
                                       likely, we recommend that you identify ~
                                       the other job and wait until it has ~
                                       successfully completed."
                                      files))
                                 (t
                                  (er soft ctx
                                      "During Step 3, we loaded different ~
                                       books than were loaded by Step 2!  ~
                                       Perhaps some other user of your file ~
                                       system was editing the books during our ~
                                       Step 3?  You might think that some ~
                                       other job is recertifying the books (or ~
                                       subbooks) and has deleted the ~
                                       certificate files, rendering ~
                                       uncertified some of the books needed ~
                                       here.  But more has happened!  Some ~
                                       file has changed!~%~%Here is the ~
                                       include-book-alist as of the end of ~
                                       Step 2:~%~X02.~|~%And here is the alist ~
                                       as of the end of Step ~
                                       3:~%~X12.~|~%Frequently, the former has ~
                                       more entries than the latter because ~
                                       the former includes LOCAL books. So ~
                                       compare corresponding entries, focusing ~
                                       on those in the latter.  Each entry is ~
                                       of the form (name1 name2 name3 alist . ~
                                       chk-sum).  Name1 is the full name, ~
                                       name2 is the name as written in an ~
                                       include-book event, and name3 is the ~
                                       ``familiar'' name of the file. The ~
                                       alist indicates the presence or absence ~
                                       of problematic forms in the file, such ~
                                       as DEFAXIOM events.  For example, ~
                                       (:AXIOMSP . T) means there were ~
                                       defaxiom events; (:AXIOMSP . NIL) -- ~
                                       which actually prints as (:AXIOMSP) -- ~
                                       means there were no defaxiom events. ~
                                       Finally, chk-sum is either an integer ~
                                       check sum on the contents of the file ~
                                       at the time it was certified or else ~
                                       chk-sum is nil indicating that the file ~
                                       is not certified.  Note that if the ~
                                       chk-sum is nil, the entry prints as ~
                                       (name1 name2 name3 alist).  Go figure."
                                      post-alist1
                                      post-alist2
                                      nil)))))
                             (t (pprogn
                                 (io? event nil state
                                      (fms "* Step 4:  Write the certificate ~
                                            for ~x0 in ~x1.  The final check ~
                                            sum alist is ~x2.~%"
                                           (list
                                            (cons #\0 full-book-name)
                                            (cons
                                             #\1
                                             (convert-book-name-to-cert-name
                                              full-book-name))
                                            (cons #\2 post-alist1))
                                           (proofs-co state) state nil))
                                 (er-progn
                                  (make-certificate-file
                                   full-book-name
                                   (cons
                                    (remove-duplicates-equal-from-end
                                     (append (car portcullis) new-defpkg-list)
                                     nil)
                                    (cdr portcullis))
                                   post-alist1
                                   post-alist2
                                   state)
                                  (pprogn
                                   (cond
                                    (compile-flg
                                     (pprogn
                                      (io? event nil state
                                           (fms "* Step 5:  Compile the ~
                                                 functions defined in ~
                                                 ~x0.~#1~[~/  As noted above, ~
                                                 ~n2 function~#3~[ has not had ~
                                                 its guard~/s have not had ~
                                                 their guards~] verified.  A ~
                                                 compiled file will be ~
                                                 produced but you are advised ~
                                                 not to use it in raw Common ~
                                                 Lisp.  See :DOC guard for a ~
                                                 general discussion of the ~
                                                 issues.~]~%"
                                                (list (cons #\0 full-book-name)
                                                      (cons #\1
                                                            (if bad-fns 1 0))
                                                      (cons #\2 (length bad-fns))
                                                      (cons #\3 bad-fns))
                                                (proofs-co state) state nil))
                                      #-acl2-loop-only
                                      (progn
                                        (acl2-compile-file full-book-name
                                                           os-full-book-name)
                                        (let ((*connected-book-directory*
                                               directory-name))
                                          (load-compiled
                                           (convert-book-name-to-compiled-name
                                            os-full-book-name)))
                                        state)
                                      state))
                                    (t
                                     (pprogn
                                      #-acl2-loop-only
                                      (progn
                                        (delete-compiled-file os-full-book-name)
                                        state)
                                      state)))
                                   (value
                                    full-book-name)))))))))))))))))))))))))))

#+acl2-loop-only
(defmacro certify-book (user-book-name &optional
                                       (k '0)
                                       (compile-flg 't)
                                       &key
                                       (defaxioms-okp 'nil)
                                       (skip-proofs-okp 'nil))

  #-small-acl2-image
  ":Doc-Section Books

  how to produce a ~il[certificate] for a book~/
  ~bv[]
  Examples:
  (certify-book \"my-arith\" 3)   ;certify in a world with 3 commands
  (certify-book \"my-arith\")     ;certify in a world with 0 commands
  (certify-book \"my-arith\" 0 nil) ;as above, but do not compile
  (certify-book \"my-arith\" t)   ;certify from world of existing certificate~/

  General Form:
  (certify-book book-name k compile-flg
                :defaxioms-okp t/nil      ; [default nil]
                :skip-proofs-okp t/nil    ; [default nil]
                )
  ~ev[]
  where ~c[book-name] is a book name (~pl[book-name]), ~c[k] is
  either ~c[t] or an integer used to indicate your approval of the
  ``certification ~il[world],'' and ~c[compile-flg] indicates whether you
  wish to compile the (functions in the) book.  ~c[Compile-flg]
  defaults to ~c[t], meaning to compile.  The second argument ~c[k] is
  actually optional as well; it defaults to ~c[0].

  The two keyword arguments, ~c[:defaxioms-okp] and ~c[:skip-proofs-okp],
  determine how the system handles the inclusion of ~ilc[defaxiom] events
  and ~ilc[skip-proofs] events, respectively, in the book.  The value
  ~c[t] allows such events, but prints a warning message.  The value ~c[nil]
  is the default, and causes an error if such an event is found.

  For a general discussion of books, ~pl[books].  ~c[Certify-book]
  is akin to what we have historically called a ``proveall'': all the
  forms in the book are ``proved'' to guarantee their admissibility.
  More precisely, ~c[certify-book] (1) reads the forms in the book,
  confirming that the appropriate packages are defined in the
  certification ~il[world]; (2) does the full admissibility checks on
  each form (proving termination of recursive functions, proving
  theorems, etc.), checking as it goes that each form is an embedded
  event form (~pl[embedded-event-form]); (3) rolls the ~il[world]
  back to the initial certification ~il[world] and does an
  ~ilc[include-book] of the book to check for ~il[local] incompatibilities
  (~pl[local-incompatibility]); (4) writes a ~il[certificate]
  recording not only that the book was certified but also recording
  the ~il[command]s necessary to recreate the certification ~il[world] (so
  the appropriate packages can be defined when the book is included in
  other ~il[world]s) and the check sums of all the ~il[books] involved
  (~pl[certificate]); (5) compiles the book if so directed (and
  then loads the object file in that case).  The result of executing a
  ~c[certify-book] ~il[command] is the creation of a single new event, which
  is actually an ~ilc[include-book] event.  If you don't want its
  included ~il[events] in your present ~il[world], simply execute ~c[:]~ilc[ubt]
  ~c[:here] afterwards.

  ~c[Certify-book] requires that the default ~il[defun-mode]
  (~pl[default-defun-mode]) be ~c[:]~ilc[logic] when certification is
  attempted.  If the mode is not ~c[:]~ilc[logic], an error is signalled.

  An error will occur if ~c[certify-book] has to deal with any
  uncertified book other than the one on which it was called.  For
  example, if the book being certified includes another book, that
  subbook must already have been certified.

  Certification occurs in some logical ~il[world], called the
  ``certification ~il[world].'' That ~il[world] must contain the ~ilc[defpkg]s
  needed to read and execute the forms in the book.  The ~il[command]s
  necessary to recreate that ~il[world] from the ACL2 initial
  ~il[world] will be copied into the ~il[certificate] created for the
  book.  Those ~il[command]s will be re-executed whenever the book is
  included, to ensure that the appropriate packages (and all other
  names used in the certification ~il[world]) are correctly defined.  The
  certified book will be more often usable if the certification
  ~il[world] is kept to a minimal extension of the ACL2 initial
  ~il[world].  Thus, before you call ~c[certify-book] for the first
  time on a book, you should get into the initial ACL2 ~il[world]
  (e.g., with ~c[:ubt 1] or just starting a new version of ACL2),
  ~ilc[defpkg] the desired packages, and then invoke ~c[certify-book].

  The ~c[k] argument to ~c[certify-book] must be either a nonnegative integer
  or else one of the symbols ~c[t] or ~c[?] in the ~c[ACL2] package.  If ~c[k]
  is an integer, then it must be the number of ~il[command]s that have been
  executed to create the ~il[world] in which ~c[certify-book] was called.  One
  way to obtain this number is by doing ~c[:pbt 1] to see all the ~il[command]s
  back to the first one.  The last ~il[command] number printed in the
  ~c[:]~ilc[pbt] display is the appropriate ~c[k].  This number is just the
  maximum ~il[command] number, ~c[:]~ilc[max] ~-[] ~pl[command-descriptor] ~-[]
  but unless ~c[:]~ilc[max] is ~c[0,] ~c[certify-book] requires that you
  actually input the number as a way of reminding you to inspect the ~il[world]
  before calling ~c[certify-book].

  If ~c[k] is ~c[t] it means that ~c[certify-book] should use the same
  ~il[world] used in the last certification of this book.  ~c[K] may be
  ~c[t] only if you call ~c[certify-book] in the initial ACL2 ~il[world]
  (~c[:max = 0]) and there is a ~il[certificate] on file for the book being
  certified.  (Of course, the ~il[certificate] is probably invalid.)  In
  this case, ~c[certify-book] reads the old ~il[certificate] to obtain
  the ~il[portcullis] ~il[command]s and executes them to recreate the
  certification ~il[world].

  Finally, ~c[k] may be ~c[?], in which case there is no check made on the
  cerfification world.  That is, if ~c[k] is ~c[?] then no action related to
  the preceding two paragraphs is performed, which can be a nice convenience
  but at the cost of eliminating a potentially valuable check that the
  certification ~il[world] may be as expected.

  If you have a certified book that has remained unchanged for some
  time you are unlikely even to remember the appropriate ~ilc[defpkg]s
  for it.  If you begin to change the book, don't throw away its
  ~il[certificate] file just because it has become invalid!  It is an
  important historical document until the book is re-certified.

  When ~c[certify-book] is directed to produce a compiled file, it
  calls the Common Lisp function ~c[compile-file] on the original source
  file.  This creates a compiled file with an extension known to ACL2,
  e.g., if the book is named ~c[\"my-book\"] then the source file is
  ~c[\"my-book.lisp\"] and the compiled file under AKCL will be
  ~c[\"my-book.o\"] while under Lucid it will be ~c[\"my-book.lbin\"] or
  ~c[\"my-book.sbin\".]  The compiled file is then loaded.  When
  ~ilc[include-book] is used later on ~c[\"my-book\"] it will
  automatically load the compiled file, provided the compiled file has
  a later write date than the source file.  The only effect of such
  ~il[compilation] and loading is that the functions defined in the
  book execute faster.  ~l[guard] for a discussion of the issues.

  When ~c[certify-book] is directed not to produce a compiled file, it
  will delete any existing compiled file for the book, so as not to
  mislead ~ilc[include-book] into loading the now outdated compiled file.

  After execution of a ~c[certify-book] form, the value of
  ~ilc[acl2-defaults-table] is restored to what it was immediately before
  that ~c[certify-book] form was executed.
  ~l[acl2-defaults-table].

  This completes the tour through the ~il[documentation] of ~il[books].~/

  :cited-by other
  :cited-by Programming"

  (list 'certify-book-fn
        (list 'quote user-book-name)
        (list 'quote k)
        (list 'quote compile-flg)
        (list 'quote defaxioms-okp)
        (list 'quote skip-proofs-okp)
        'state))

(defmacro certify-book! (user-book-name &optional (k '0) (compile-flg 't))
  (declare (xargs :guard (and (integerp k) (<= 0 k))))
  
  #-small-acl2-image
  ":Doc-Section Other

  a variant of ~ilc[certify-book]~/
  ~bv[]
  Examples:
  (certify-book! \"my-arith\" 3)     ;Certify in a world with 3
                                     ; commands, starting in a world
                                     ; with at least 3 commands.
  (certify-book! \"my-arith\")       ;Certify in the initial world.
  (certify-book! \"my-arith\" 0 nil) ;As above, but do not compile.~/

  General Form:
  (certify-book! book-name k compile-flg)
  ~ev[]
  where ~c[book-name] is a book name (~pl[book-name]), ~c[k] is a
  nonnegative integer used to indicate the ``certification ~il[world],''
  and ~c[compile-flg] indicates whether you wish to compile the
  (functions in the) book.

  This ~il[command] is identical to ~ilc[certify-book], except that the second
  argument ~c[k] may not be ~c[t] in ~c[certify-book!] and if ~c[k]
  exceeds the current ~il[command] number, then an appropriate ~ilc[ubt!] will
  be executed first.  ~l[certify-book] and ~pl[ubt!].~/"

  `(progn (ubt! ,(1+ k))
          (certify-book ,user-book-name ,k ,compile-flg)))

(defun chk-input-object-file (file ctx state)

; This checks that an object file named file can be opened for input.
; It either causes an error or returns t.  It changes the state --
; because it opens and closes a channel to the file -- and it may well
; be that the file does not exist in the state returned!  C'est la
; guerre.  The purpose of this function is courtesy to the user.  It
; is nice to rather quickly determine, in include-book for example,
; whether an alleged file exists.

  (er-let* ((ch (open-input-object-file file ctx state)))
           (let ((state (close-input-channel ch state)))
             (value t))))

(defun include-book-dir (dir acl2-defaults-table)
  (declare (xargs :guard (and (symbol-alistp acl2-defaults-table)
                              (symbol-alistp
                               (assoc-eq dir
                                         (assoc-eq
                                          :include-book-dir-alist
                                          acl2-defaults-table))))))
  (cdr (assoc-eq dir
                 (cdr (assoc-eq :include-book-dir-alist acl2-defaults-table)))))

(defun include-book-fn (user-book-name state
                                       load-compiled-file
                                       uncertified-okp
                                       defaxioms-okp
                                       skip-proofs-okp
                                       doc
                                       dir
                                       event-form)
  (with-ctx-summarized
   (if (output-in-infixp state) event-form (cons 'include-book user-book-name))
   (let* ((wrld0 (w state))
          #-acl2-loop-only (*inside-include-book-fn* t)
          (old-include-book-path
           (global-val 'include-book-path wrld0))
          (saved-acl2-defaults-table
           (table-alist 'acl2-defaults-table wrld0))
          (cddr-event-form
           (if event-form
               (cddr event-form)
             (append 
              (if (not (eq load-compiled-file :warn))
                  (list :load-compiled-file
                        load-compiled-file)
                nil)
              (if (not (eq uncertified-okp t))
                  (list :uncertified-okp
                        uncertified-okp)
                nil)
              (if (not (eq defaxioms-okp t))
                  (list :defaxioms-okp
                        defaxioms-okp)
                nil)
              (if (not (eq skip-proofs-okp t))
                  (list :skip-proofs-okp
                        skip-proofs-okp)
                nil)
              (if doc
                  (list :doc doc)
                nil))))
          #+(and clisp (not acl2-loop-only))
          (custom::*suppress-check-redefinition* t)
          #+(and allegro (not acl2-loop-only))
          (excl:*redefinition-warnings* nil)
          )
     (er-let*
      ((dir-value
        (cond (dir (let ((dir-value
                          (include-book-dir dir saved-acl2-defaults-table)))
                     (if dir-value
                         (value dir-value)
                       (er soft ctx
                           "The legal values for the :dir argument of ~
                            include-book are keywords that include :SYSTEM as ~
                            well as those added by a call of ~
                            add-include-book-dir.  However, that argument is ~
                            ~x0, which is not among the list of those legal ~
                            values, ~x1."
                           dir
                           (strip-cars
                            (cdr (assoc-eq :include-book-dir-alist
                                           saved-acl2-defaults-table)))))))
              (t (value (cbd))))))
      (mv-let
        (full-book-name directory-name familiar-name)
        (parse-book-name dir-value user-book-name ".lisp" (os (w state)))

; If you add more keywords to the suspect-book-action-alist, make sure you do
; the same to the list constructed by certify-book-fn.  You might wish to
; handle the new warning summary in warning1.

        (let ((suspect-book-action-alist
               (list (cons :uncertified-okp
                           (if (assoc 'certify-book
                                      (global-val 'embedded-event-lst
                                                  wrld0))
                               nil
                             uncertified-okp))
                     (cons :defaxioms-okp defaxioms-okp)
                     (cons :skip-proofs-okp skip-proofs-okp)))
              (behalf-of-certify-flg
               (equal full-book-name (f-get-global 'certify-book-file state))))
          (er-progn
           (chk-book-name user-book-name full-book-name ctx state)
           (chk-input-object-file full-book-name ctx state)
           (revert-world-on-error
            (cond
             ((and (not (global-val 'boot-strap-flg wrld0))
                   full-book-name
                   (assoc-equal full-book-name
                                (global-val 'include-book-alist wrld0)))
              (stop-redundant-event state))
             (t
              (let ((wrld1 (global-set
                            'include-book-path
                            (cons full-book-name old-include-book-path)
                            wrld0)))
                (pprogn
                 (set-w 'extension wrld1 state)
                 (er-let*
                  ((redef (chk-new-stringp-name 'include-book full-book-name
                                                ctx wrld1 state))
                   (doc-pair (translate-doc full-book-name doc ctx state))
                   (cert-obj (if behalf-of-certify-flg
                                 (value nil)
                               (chk-certificate-file full-book-name nil ctx state
                                                     suspect-book-action-alist
                                                     t)))
                   (cert-full-book-name (value (car (car (cdr cert-obj))))))
                  (cond

; We try the redundancy check again, because it will be cert-full-book-name
; that is stored on the world's include-book-alist, not full-book-name (if the
; two book names differ).

                   ((and (not (equal full-book-name cert-full-book-name))
                         (not (global-val 'boot-strap-flg wrld1))
                         cert-full-book-name 
                         (assoc-equal cert-full-book-name
                                      (global-val 'include-book-alist wrld1)))

; Chk-certificate-file calls chk-certificate-file1, which calls
; chk-raise-portcullis, which calls chk-raise-portcullis1, which evaluates, for
; example, maybe-install-acl2-defaults-table.  So we need to revert the world
; here.

                    (pprogn (set-w! wrld1 state)
                            (stop-redundant-event state)))
                   (t
                    (er-let*
                     ((load-compiled-file (value (if (or behalf-of-certify-flg
                                                         cert-obj)
                                                     load-compiled-file
                                                   nil)))
                      (ev-lst (read-object-file full-book-name ctx state)))

; Cert-obj above is either nil, indicating that the file is
; uncertified or is ((cmds . pre-alist) . post-alist), where the car
; is the now raised portcullis, and the cdr is the check sum alist of
; the files that should be brought in by this inclusion.  The first
; element of post-alist is the one for this book.  It should look like
; this: (full-book-name' user-book-name' familiar-name cert-annotations
; . ev-lst-chk-sum), where the first two names are irrelevant here
; because they reflect where the book was when it was certified rather
; than where the book resides now.  However, the familiar-name,
; cert-annotations and the ev-lst-chk-sum ought to be those for the
; current book.

                     (mv-let
                       (ev-lst-chk-sum state)
                       (check-sum-obj ev-lst state)
                       (cond
                        ((not (integerp ev-lst-chk-sum))

; This error should never arise because check-sum-obj is only called
; on something produced by read-object, which checks that the object
; is ACL2 compatible.

                         (er soft ctx
                             "The file ~x0 is not a legal list of embedded event ~
                           forms because it contains an object, ~x1, which ~
                           check sum was unable to handle."
                             full-book-name ev-lst-chk-sum))
                        (t (er-progn

; Notice that we are reaching inside the certificate object to
; retrieve information about the book from the post-alist.  (car (cdr
; cert-obj)) is in fact of the form (full-book-name user-book-name
; familiar-name cert-annotations . ev-lst-chk-sum).


                            (cond
                             ((and cert-obj
                                   (not (equal (caddr (car (cdr cert-obj)))
                                               familiar-name)))
                              (include-book-er
                               full-book-name nil
                               (cons
                                "The cer~-ti~-fi~-cate on file for ~x0 lists the ~
                              book under the name ~x3 whereas we were ~
                              expecting it to give the name ~x4.  While we ~
                              allow a certified book to be moved from one ~
                              directory to another after ~
                              cer~-ti~-fi~-ca~-tion, we insist that it keep ~
                              the same familiar name.  This allows the ~
                              cer~-ti~-fi~-cate file to contain the familiar ~
                              name, making it easier to identify which ~
                              cer~-ti~-fi~-cates go with which files and ~
                              inspiring a little more confidence that the ~
                              cer~-ti~-fi~-cate really does describe the ~
                              alleged file.  In the present case, it looks as ~
                              though the familiar book name was changed after ~
                              cer~-ti~-fi~-ca~-tion.  For what it is worth, ~
                              the check sum of the file at ~
                              cer~-ti~-fi~-ca~-tion was ~x5.  Its check sum ~
                              now is ~x6."
                                (list (cons #\3 (caddr (car (cdr cert-obj))))
                                      (cons #\4 familiar-name)
                                      (cons #\5 (cddddr (car (cdr cert-obj))))
                                      (cons #\6 ev-lst-chk-sum)))
                               :uncertified-okp
                               suspect-book-action-alist
                               ctx state))
                             (t (value nil)))

                            (cond
                             ((and cert-obj
                                   (not (equal (cddddr (car (cdr cert-obj)))
                                               ev-lst-chk-sum)))
                              (include-book-er
                               full-book-name nil
                               (cons
                                "The certificate on file for ~x0 lists the check ~
                              sum of the certified book as ~x3.  But the check ~
                              sum of the events now in the file is ~x4. This ~
                              generally indicates that the file has been ~
                              modified since it was last certified."
                                (list (cons #\3 (cddddr (car (cdr cert-obj))))
                                      (cons #\4 ev-lst-chk-sum)))
                               :uncertified-okp
                               suspect-book-action-alist
                               ctx state))
                             (t (value nil)))

                            (let ((cert-annotations (cadddr (car (cdr cert-obj)))))

; It is possible for cert-annotations to be nil now.  That is because cert-obj was
; nil.  But we never use it if cert-obj is nil.

                              (cond
                               ((and cert-obj
                                     (or (cdr (assoc :skipped-proofsp
                                                     cert-annotations))
                                         (cdr (assoc :axiomsp
                                                     cert-annotations))))

                                (chk-cert-annotations cert-annotations
                                                      (caar cert-obj)
                                                      full-book-name
                                                      suspect-book-action-alist
                                                      ctx state))
                               (t (value nil))))

; The following process-embedded-events is protected by the revert-world-
; on-error above.

                            (state-global-let*
                             ((skipped-proofsp nil)
                              (axiomsp nil)
                              (connected-book-directory directory-name)
                              (match-free-error nil)

; By binding guard-checking-on to nil, we allow defconst and value-triple forms
; to be evaluated in the logic.

                              (guard-checking-on nil))
                             (process-embedded-events
                              'include-book
                              saved-acl2-defaults-table
                              (if (or cert-obj behalf-of-certify-flg)
                                  'include-book

; If we are including an uncertified book, then we want to do some of the checks.

                                'initialize-acl2)
                              (cadr (car ev-lst))
                              (list 'include-book full-book-name)
                              (cdr ev-lst) ctx state))

; This function returns what might be called proto-wrld3, which is
; equivalent to the current world of state before the
; process-embedded-events (since the insigs argument is nil), but it has
; an incremented embedded-event-depth.  We don't care about this
; world.  The interesting world is the one current in the state
; returned by by process-embedded-events.  It has all the embedded
; events in it and we are done except for certification issues.

                            (let* ((wrld2 (w state))
                                   (actual-alist (global-val 'include-book-alist
                                                             wrld2)))
                              (er-progn
                               (cond
                                ((and cert-obj
                                      (not (include-book-alist-subsetp
                                            (unmark-and-delete-local-included-books
                                             (cdr (cdr cert-obj)))
                                            actual-alist)))
                                 (include-book-er
                                  full-book-name nil
                                  (cons "The certified book ~x0 requires ~*3 but ~
                                      we have ~*4."
                                        (list
                                         (cons #\3
                                               (tilde-*-book-check-sums-phrase
                                                t
                                                (unmark-and-delete-local-included-books
                                                 (cdr (cdr cert-obj)))
                                                actual-alist))
                                         (cons #\4
                                               (tilde-*-book-check-sums-phrase
                                                nil
                                                (unmark-and-delete-local-included-books
                                                 (cdr (cdr cert-obj)))
                                                actual-alist))))
                                  :uncertified-okp
                                  suspect-book-action-alist
                                  ctx state))
                                (t (value nil)))

; Now we check that all the subbooks of this one are also compatible with the
; current settings of suspect-book-action-alist.  (cdr cert-obj) is the
; post-alist of the cert-object.  But the car of that is the part that deals
; with full-book-name itself.  So we deal below with the cdr, which lists the
; subbooks.  The cert-obj may be nil, which makes the test below a no-op.

                               (chk-cert-annotations-post-alist
                                (cdr (cdr cert-obj))
                                (caar cert-obj)
                                full-book-name
                                suspect-book-action-alist
                                ctx state)

                               (let* ((cert-annotations
                                       (cadddr (car (cdr cert-obj))))

; If cert-obj is nil, then cert-annotations is nil.  If cert-obj is
; non-nil, then cert-annotations is non-nil.  Cert-annotations came
; from a .cert file, and they are always non-nil.  But in the
; following, cert-annotations may be nil.

                                      (certification-tuple
                                       (cond
                                        ((and cert-obj
                                              (equal (caddr (car (cdr cert-obj)))
                                                     familiar-name)
                                              (equal (cddddr (car (cdr cert-obj)))
                                                     ev-lst-chk-sum)
                                              (include-book-alist-subsetp
                                               (unmark-and-delete-local-included-books
                                                (cdr (cdr cert-obj)))
                                               actual-alist))

; Below we use the full book name from the certificate, cert-full-book-name,
; rather than full-book-name (from the parse of the user-book-name), in
; certification-tuple, Intuitively, cert-full-book-name is unique
; representative of the class of all legal full book names (including those
; that involve soft links).  Before Version_2.7 we used full-book-name rather
; than cert-full-book-name, and this led to problems as shown in the example
; below.

                                         #|
    % ls temp*/*.lisp
    temp1/a.lisp  temp2/b.lisp  temp2/c.lisp
    % cat temp1/a.lisp
    (in-package "ACL2")
    (defun foo (x) x)
    % cat temp2/b.lisp
    (in-package "ACL2")
    (defun goo (x) x)
    % cat temp2/c.lisp
    (in-package "ACL2")
    (defun hoo (x) x)
    % 

  Below, two absolute pathnames are abbreviated as <path1> and <path2>.

  In temp2/ we LD a file with the following forms.

    (certify-book "<path1>/a")
    :u
    (include-book "../temp1/a")
    (certify-book "b" 1)
    :ubt! 1
    (include-book "b")
    (certify-book "c" 1)

  We then see the following error.  The problem is that <path1> involved symbolic
  links, and hence did not match up with the entry in the world's
  include-book-alist made by (include-book "../temp1/a") which expanded to an
  absolute pathname that did not involve symbolic links.

    ACL2 Error in (CERTIFY-BOOK "c" ...):  During Step 3, we loaded different
    books than were loaded by Step 2!  Perhaps some other user of your
    file system was editing the books during our Step 3?  You might think
    that some other job is recertifying the books (or subbooks) and has
    deleted the certificate files, rendering uncertified some of the books
    needed here.  But more has happened!  Some file has changed!

    Here is the include-book-alist as of the end of Step 2:
    (("<path2>/temp2/c.lisp"
          "c" "c" ((:SKIPPED-PROOFSP) (:AXIOMSP))
          . 48180423)
     ("<path2>/temp2/b.lisp"
          "b" "b" ((:SKIPPED-PROOFSP) (:AXIOMSP))
          . 46083312)
     (LOCAL ("<path1>/a.lisp"
                 "<path1>/a"
                 "a" ((:SKIPPED-PROOFSP) (:AXIOMSP))
                 . 43986201))).

    And here is the alist as of the end of Step 3:
    (("<path2>/temp2/c.lisp"
          "c" "c" ((:SKIPPED-PROOFSP) (:AXIOMSP))
          . 48180423)
     ("<path2>/temp2/b.lisp"
          "b" "b" ((:SKIPPED-PROOFSP) (:AXIOMSP))
          . 46083312)
     ("<path2>/temp1/a.lisp"
          "<path2>/temp1/a"
          "a" ((:SKIPPED-PROOFSP) (:AXIOMSP))
          . 43986201)).

    Frequently, the former has more entries than the latter because the
    former includes LOCAL books. So compare corresponding entries, focusing
    on those in the latter.  Each entry is of the form (name1 name2 name3
    alist . chk-sum).  Name1 is the full name, name2 is the name as written
    in an include-book event, and name3 is the ``familiar'' name of the
    file. The alist indicates the presence or absence of problematic forms
    in the file, such as DEFAXIOM events.  For example, (:AXIOMSP . T)
    means there were defaxiom events; (:AXIOMSP . NIL) -- which actually
    prints as (:AXIOMSP) -- means there were no defaxiom events. Finally,
    chk-sum is either an integer check sum on the contents of the file
    at the time it was certified or else chk-sum is nil indicating that
    the file is not certified.  Note that if the chk-sum is nil, the entry
    prints as (name1 name2 name3 alist).  Go figure.


    Summary
    Form:  (CERTIFY-BOOK "c" ...)
    Rules: NIL
    Warnings:  Guards
    Time:  0.01 seconds (prove: 0.00, print: 0.00, other: 0.01)

    ******** FAILED ********  See :DOC failure  ******** FAILED ********
     :ERROR
    ACL2 !>

|#

                                         (list* cert-full-book-name
                                                user-book-name
                                                familiar-name
                                                cert-annotations
                                                ev-lst-chk-sum))
                                        (t 

; The certification tuple below is marked as uncertified because the
; ev-lst-chk-sum is nil.  What about cert-annotations?  It may or may
; not correctly characterize the file, it may even be nil.  Is that
; bad?  No, the check sum will always save us.

                                         (list* full-book-name
                                                user-book-name
                                                familiar-name
                                                cert-annotations
                                                nil))))
                                      (wrld3
                                       (global-set
                                        'include-book-path
                                        old-include-book-path
                                        (update-doc-data-base
                                         full-book-name doc doc-pair
                                         (global-set
                                          'certification-tuple
                                          certification-tuple
                                          (global-set 'include-book-alist
                                                      (add-to-set-equal
                                                       certification-tuple
                                                       (global-val
                                                        'include-book-alist
                                                        wrld2))
                                                      wrld2))))))
                                 (pprogn
                                  #-acl2-loop-only
                                  (let ((*connected-book-directory* directory-name)
                                        (os-full-book-name
                                         (pathname-unix-to-os
                                          full-book-name
                                          *the-live-state*)))
                                    (progn

; We proclaim here because this book may be included by another book during the
; latter's certification, and we want all proclamations to be in effect.  Note
; that we proclaim before calling load-compiled-file-if-more-recent, because
; the latter can invoke the compiler if load-compiled-file is :comp.

                                      (proclaim-file os-full-book-name)
                                      (load-compiled-file-if-more-recent
                                       ctx load-compiled-file
                                       full-book-name os-full-book-name)
                                      state))
                                  (redefined-warning redef ctx state)
                                  (f-put-global 'include-book-alist-state
                                                (add-to-set-equal
                                                 certification-tuple
                                                 (union-equal
                                                  (cdr (cdr cert-obj))
                                                  (f-get-global
                                                   'include-book-alist-state
                                                   state)))
                                                state)
                                  (install-event
                                   (if behalf-of-certify-flg
                                       ev-lst-chk-sum
                                     (or cert-full-book-name
                                         full-book-name))
                                   (list* 'include-book

; We use the the unique representative of the full book name provided by the
; one in the .cert file, when the certificate is valid before execution of this
; event), namely, cert-full-book-name; otherwise, we use the full-book-name
; parsed from what the user supplied.  Either way, we have an absolute path
; name, which is useful for the :puff and :puff* commands.  These could fail
; before Version_2.7 because the relative path name stored in the event was not
; sufficient to find the book at :puff/:puff* time.

                                          (or cert-full-book-name
                                              full-book-name)
                                          cddr-event-form)
                                   'include-book
                                   full-book-name
                                   nil nil wrld3 state)))))))))))))))))))))))))

(deflabel pathname
  :doc
  ":Doc-Section acl2::Books

  introduction to filename conventions in ACL2~/

  The notion of pathname objects from Common Lisp is not supported in
  ACL2, nor is the function ~c[pathname].  However, ACL2 supports file
  operations, using conventions for naming files based on those of the
  Unix (trademark of AT&T) operating system, so that character ~c[/] is
  used to terminate directory names.  Some file names are ``absolute''
  (complete) descriptions of a file or directory; others are
  ``relative'' to the current working directory or to the connected
  book directory (~pl[cbd]).  We emphasize that even for users of
  Windows-based systems or Macintosh computers, ACL2 file names are in
  the Unix style.  We will call these ~em[ACL2 pathnames], often
  omitting the ``ACL2.''~/

  Pathnames starting with the directory separator (~c[/]) are absolute
  pathnames.  All other pathnames are relative pathnames.  An
  exception is in the Microsoft Windows operating system, where the
  drive may be included, e.g., ~c[\"c:/home/smith/acl2/book-1.lisp\"].
  In fact, the drive ~em[must] be included in the portcullis of a book;
  ~pl[portcullis].

  Consider the following examples.  The filename string
  ~bv[]
  \"/home/smith/acl2/book-1.lisp\"
  ~ev[]
  is an absolute pathname, with top-level directory ~c[\"home\"],
  under that the directory ~c[\"smith\"] and then the directory
  ~c[\"acl2\"], and finally, within that directory the file
  ~c[\"book-1.lisp\"].  If the the connected book directory is
  ~c[\"/home/smith/\"] (~pl[cbd]), then the filename string above
  also corresponds to the relative filename string \"acl2/book1.lisp\".~/")

(deflabel book-example
  :Doc
  ":Doc-Section Books

  how to create, certify, and use a simple book~/

  Suppose you have developed a sequence of admissible ~il[events] which you
  want to turn into a book.  We call this ``publishing'' the book.
  This note explains how to do that.~/

  A key idea of ~il[books] is that they are ``incremental'' in the
  sense that when you include a book in a host logical ~il[world], the
  ~il[world] is incrementally extended by the results established in that
  book.  This is allowed only if every name defined by the incoming
  book is either new or is already identically defined.
  ~l[redundant-events].  This is exactly the same problem faced
  by a programmer who wishes to provide a utility to other people: how
  can he make sure he doesn't create name conflicts?  The solution, in
  Common Lisp, is also the same: use packages.  While ~il[books] and
  packages have a very tenuous formal connection (every book must
  start with an ~ilc[in-package]), the creation of a book is intimately
  concerned with the package issue.  Having motivated what would
  otherwise appear as an unnecessary fascination with packages below,
  we now proceed with a description of how to publish a book.

  Just to be concrete, let's suppose you have already gotten ACL2 to
  accept the following sequence of ~il[command]s, starting in the ACL2
  initial ~il[state].
  ~bv[]
     (defpkg \"ACL2-MY-BOOK\"
             (union-eq *common-lisp-symbols-from-main-lisp-package*
                       *acl2-exports*))
     (in-package \"ACL2-MY-BOOK\")
     (defun app (x y)
       (if (consp x) (cons (car x) (app (cdr x) y)) y))
     (defun rev (x)
       (if (consp x) (app (rev (cdr x)) (list (car x))) nil))
     (defthm rev-app-hack
       (equal (rev (app a (list x))) (cons x (rev a))))
     (defthm rev-rev 
       (implies (acl2::true-listp x) (equal (rev (rev x)) x)))
  ~ev[]
  Observe that the first form above defines a package (which imports
  the symbols defined in CLTL such as ~ilc[if] and ~ilc[cons] and the
  symbols used to ~il[command] ACL2 such as ~ilc[defun] and ~ilc[defthm]).  The
  second form selects that package as the current one.  All subsequent
  forms are read into that package.  The remaining forms are just
  event forms: ~ilc[defun]s and ~ilc[defthm]s in this case.

  Typically you would have created a file with Emacs containing these
  forms and you will have submitted each of them interactively to ACL2
  to confirm that they are all admissible.  That interactive
  verification should start in ACL2's initial ~il[world] ~-[] although
  you might, of course, start your sequence of ~il[events] with some
  ~ilc[include-book]s to build a more elaborate ~il[world].

  The first step towards publishing a book containing the results
  above is to create a file that starts with the ~ilc[in-package] and
  then contains the rest of the forms.  Let's call that file
  ~c[\"my-book.lisp\"].  The name is unimportant, except it must end
  with ~c[\".lisp\"].  If there are ~il[events] that you do not wish to be
  available to the user of the book ~-[] e.g., lemmas you proved on your
  way toward proving the main ones ~-[] you may so mark them by
  enclosing them in ~ilc[local] forms.  ~l[local].  Let us suppose
  you wish to hide ~c[rev-app-hack] above.  You may also add standard Lisp
  comments to the file.  The final content of ~c[\"my-book.lisp\"]
  might be:
  ~bv[]
   ; This book contains my app and rev functions and the theorem
   ; that rev is its own inverse.

     (in-package \"ACL2-MY-BOOK\")
     (defun app (x y)
       (if (consp x) (cons (car x) (app (cdr x) y)) y))
     (defun rev (x)
       (if (consp x) (app (rev (cdr x)) (list (car x))) nil))

   ; The following hack is not exported.
     (local (defthm rev-app-hack
       (equal (rev (app a (list x))) (cons x (rev a)))))

     (defthm rev-rev 
       (implies (acl2::true-listp x) (equal (rev (rev x)) x)))
  ~ev[]
  The file shown above ~st[is] the book.  By the time this note is
  done you will have seen how to certify that the book is correct, how
  to compile it, and how to use it in other host ~il[world]s.  Observe that
  the ~ilc[defpkg] is not in the book.  It cannot be: Common Lisp
  compilers disagree on how to treat new package definitions appearing
  in files to be compiled.

  Since a book is just a source file typed by the user, ACL2 provides
  a mechanism for checking that the ~il[events] are all admissible and then
  marking the file as checked.  This is called certification.  To
  certify ~c[\"my-book.lisp\"] you should first get into ACL2 with an
  initial ~il[world].  Then, define the package needed by the book, by
  typing the following ~ilc[defpkg] to the ACL2 ~il[prompt]:
  ~bv[]
  ACL2 !>(defpkg \"ACL2-MY-BOOK\"
                 (union-eq *common-lisp-symbols-from-main-lisp-package*
                           *acl2-exports*))
  ~ev[]
  Then execute the ~il[command]:
  ~bv[]
  ACL2 !>(certify-book \"my-book\" 1 t) ; the `t' is in fact the default
  ~ev[]
  Observe that you do not type the ~c[\".lisp\"] part of the file
  name.  For purposes of ~il[books], the book's name is ~c[\"my-book\"] and
  by the time all is said and done, there will be several extensions
  in addition to the ~c[\".lisp\"] extension associated with it.

  The ~c[1] tells ~ilc[certify-book] that you acknowledge that there is
  one command in this ``certification ~il[world]'' (namely the ~ilc[defpkg]).
  To use the book, any prospective host ~il[world] must be extended by
  the addition of whatever ~il[command]s occurred before certification.  It
  would be a pity to certify a book in a ~il[world] containing junk because
  that junk will become the ``~il[portcullis]'' guarding entrance to
  the book.  The ~c[t] above tells ~ilc[certify-book] that you wish to
  compile ~c[\"my-book.lisp\"] also.  ~ilc[Certify-book] makes many checks
  but by far the most important and time-consuming one is that it
  ``proves'' every event in the file.

  When ~ilc[certify-book] is done it will have created two new files.
  The first will be called ~c[\"my-book.cert\"] and contains the
  ``~il[certificate]'' attesting to the admissibility of the ~il[events] in
  ~c[\"my-book.lisp\"].  The ~il[certificate] contains the ~ilc[defpkg] and any
  other forms necessary to construct the certification ~il[world].  It also
  contains various check sums used to help you keep track of which
  version of ~c[\"my-book.lisp\"] was certified.

  The second file created by ~ilc[certify-book] is the compiled version
  of ~c[\"my-book.lisp\"] and will have a name that is assigned by the
  host compiler (e.g., ~c[\"my-book.o\"] in AKCL, ~c[\"my-book.lbin\"]
  or ~c[\"my-book.sbin\"] in Lucid).  ~ilc[Certify-book] will also load
  this object file.  When ~ilc[certify-book] is done, you may throw away
  the logical ~il[world] it created, for example by executing the
  ~il[command] ~c[:u].

  To use the book later in any ACL2 session, just execute the event
  ~c[(include-book \"my-book\")].  This will do the necessary
  ~ilc[defpkg], load the non-~ilc[local] ~il[events] in ~c[\"my-book.lisp\"] and
  then load the compiled code for the non-local functions defined in
  that file.  Checks are made to ensure that the ~il[certificate] file
  exists and describes the version of ~c[\"my-book.lisp\"] that is
  read.  The compiled code is loaded if and only if it exists and has
  a later write date than the source file.

  Since ~ilc[include-book] is itself an event, you may put such forms
  into other ~il[books].  Thus it is possible for the inclusion of a single
  book to lead to the inclusion of many others.  The check sum
  information maintained in ~il[certificate]s helps deal with the
  version control problem of the referenced ~il[books].  I.e., if this
  version of ~c[\"my-book\"] is used during the certification of
  ~c[\"your-book\"], then the ~il[certificate] for ~c[\"your-book\"] includes
  the check sum of this version of ~c[\"my-book\"].  If a later
  ~c[(include-book \"your-book\")] finds a version of ~c[\"my-book\"]
  with a different check sum, an error is signalled.  But check sums
  are not perfect and the insecurity of the host file system prevents
  ACL2 from guaranteeing the logical soundness of an ~ilc[include-book]
  event, even for a book that appears to have a valid ~il[certificate]
  (they can be forged, after all).  (~l[certificate] for further
  discussion.)

  This concludes the example of how to create, certify and use a book.
  If you wish, you could now review the ~il[documentation] for book-related
  topics (~pl[books]) and browse through them.  They'll probably
  make sense in this context.  Alternatively, you could continue the
  ``guided tour'' through the rest of the ~il[documentation] of ~il[books].
  ~l[book-name], following the pointer given at the conclusion.")

(deflabel full-book-name
  :doc
  ":Doc-Section Books

  book naming conventions assumed by ACL2~/

  For this discussion we assume that the resident operating system is
  Unix (trademark of AT&T), but analogous remarks apply to other
  operating systems supported by ACL2, in particular, the Macintosh
  operating system where `~c[:]' plays roughly the role of `~c[/]' in
  Unix; ~pl[pathname].

  ACL2 defines a ``full book name'' to be an ``absolute filename
  string,'' that may be divided into contiguous sections:  a
  ``directory string'', a ``familiar name'' and an ``extension''.
  ~l[pathname] for the definitions of ``absolute,'' ``filename
  string,'' and other notions pertaining to naming files.  Below we
  exhibit the three sections of one such string:
  ~bv[]
  \"/usr/home/smith/project/arith.lisp\"

  \"/usr/home/smith/project/\"           ; directory string
                          \"arith\"      ; familiar name
                               \".lisp\" ; extension~/
  ~ev[]
  The sections are marked by the rightmost slash and rightmost dot,
  as shown below.
  ~bv[]
  \"/usr/home/smith/project/arith.lisp\"
                          |     |
                          slash dot
                          |     |
  \"/usr/home/smith/project/\"           ; directory string
                          \"arith\"      ; familiar name
                               \".lisp\" ; extension
  ~ev[]
  The directory string includes (and terminates with) the rightmost
  slash.  The extension includes (and starts with) the rightmost dot.
  The dot must be strictly to the right of the slash so that the
  familiar name is well-defined and nonempty.

  If you are using ACL2 on a system in which file names do not have
  this form, please contact the authors and we'll see what we can do
  about generalizing ACL2's conventions.")

(deflabel book-name
  :doc
  ":Doc-Section  Books

  conventions associated with book names~/
  ~bv[]
  Examples:
  \"list-processing\"
  \"/usr/home/smith/my-arith\"
  ~ev[]
  Book names are strings and lists that can be elaborated into file
  names.  We elaborate book names by concatenating the ``connected
  book directory'' (~pl[cbd]) string on the left and some
  ``extension,'' such as ~c[\".lisp\"], on the right.  However, the
  connected book directory is not added if the book name itself
  already represents an absolute file name.  Furthermore,
  ~ilc[include-book] and ~ilc[certify-book] temporarily reset the connected
  book directory to be the directory of the book being processed.
  This allows ~ilc[include-book] forms to use file names without explicit
  mention of the enclosing book's directory.  This in turn allows
  ~il[books] (together with those that they include, using
  ~ilc[include-book]) to be moved between directories while maintaining
  their certification and utility.

  You may wish to read elsewhere for details of ACL2 file name
  conventions (~pl[pathname]), for a discussion of the filename
  that is the result of the elaboration described here
  (~pl[full-book-name]), and for details of the concept of the
  connected book directory (~pl[cbd]).  For details of how
  ~ilc[include-book] (~pl[include-book]) and ~ilc[certify-book]
  (~pl[certify-book]) use these concepts, see below.~/

  Often a book name is simply the familiar name of the file.
  (~l[full-book-name] for discussion of the notions of
  ``directory string,'' ``familiar name,'' and ``extension''.  These
  concepts are not on the guided tour through ~il[books] and you
  should read them separately.)  However, it is permitted for book
  names to include a directory or part of a directory name.  Book
  names never include the extension, since ACL2 must routinely tack
  several different extensions onto the name during ~ilc[include-book].
  For example, ~ilc[include-book] uses the ~c[\".lisp\"], ~c[\".cert\"] and
  possibly the ~c[\".o\"] or ~c[\".lbin\"] extensions of the book name.

  Book names are elaborated into full file names by ~ilc[include-book]
  and ~ilc[certify-book].  This elaboration is sensitive to the
  ``connected book directory.'' The connected book directory is an
  absolute filename string (~pl[pathname]) that is part of the
  ACL2 ~ilc[state].  (You may wish to ~pl[cbd] and to
  ~pl[set-cbd] ~-[] note that these are not on the guided tour).
  If a book name is an absolute filename string, ACL2 elaborates it
  simply by appending the desired extension to the right.
  If a book name is a relative filename string, ACL2 appends the
  connected book directory on the left and the desired extension on
  the right.

  Note that it is possible that the book name includes some partial
  specification of the directory.  For example, if the connected book
  directory is ~c[\"/usr/home/smith/\"] then the book name
  ~c[\"project/task-1/arith\"] is a book name that will be elaborated
  to
  ~bv[]
  \"/usr/home/smith/project/task-1/arith.lisp\".
  ~ev[]

  Observe that while the ~il[events] in this ~c[\"arith\"] book are being
  processed the connected book directory will temporarily be set to
  ~bv[]
  \"/usr/home/smith/project/task-1/\".
  ~ev[]
  Thus, if the book requires other ~il[books], e.g.,
  ~bv[]
  (include-book \"naturals\")
  ~ev[]
  then it is not necessary to specify the directory on which they
  reside provided that directory is the same as the superior book.

  This inheritance of the connected book directory and its use to
  elaborate the names of inferior ~il[books] makes it possible to move
  ~il[books] and their inferiors to new directories, provided they maintain
  the same relative relationship.  It is even possible to move with
  ease whole collections of ~il[books] to different filesystems that use
  a different operating system than the one under which the original
  certification was performed.

  The ~c[\".cert\"] extension of a book, if it exists, is presumed to
  contain the most recent ~il[certificate] for the book.
  ~l[certificate] (or, if you are on the guided tour, wait until
  the tour gets there).

  ~l[book-contents] to continue the guided tour.")

(deflabel book-contents
  :doc
  ":Doc-Section  Books

  restrictions on the forms inside ~il[books]~/
  ~bv[]
  Example Book:

  ; This book defines my app function and the theorem that it is
  ; associative.  One irrelevant help lemma is proved first but
  ; it is local and so not seen by include-book.  I depend on the
  ; inferior book \"weird-list-primitives\" from which I get
  ; definitions of hd and tl.

  (in-package \"MY-PKG\")

  (include-book \"weird-list-primitives\")

  (defun app (x y) (if (consp x) (cons (hd x) (app (tl x) y)) y))

  (local
   (defthm help-lemma
     (implies (true-listp x) (equal (app x nil) x))))

  (defthm app-is-associative
    (equal (app (app a b) c) (app a (app b c))))~/

  ~ev[]
  The first form in a book must be ~c[(in-package \"pkg\")] where
  ~c[\"pkg\"] is some package name known to ACL2 whenever the book is
  certified.  The rest of the forms in a book are embedded event
  forms, i.e., ~ilc[defun]s, ~ilc[defthm]s, etc., some of which may be
  marked ~ilc[local].  ~l[embedded-event-form].  The usual Common
  Lisp commenting conventions are provided.  Note that since a book
  consists of embedded event forms, we can talk about the
  ``~il[local]'' and ``non-local'' ~il[events] of a book.

  Because ~ilc[in-package] is not an embedded event form, the only
  ~ilc[in-package] in a book is the initial one.  Because ~ilc[defpkg] is
  not an embedded event form, a book can never contain a ~ilc[defpkg]
  form.  Because ~ilc[include-book] is an embedded event form, ~il[books] may
  contain references to other ~il[books].  This makes ~il[books] structured
  objects.

  When the forms in a book are read from the file, they are read with
  ~ilc[current-package] set to the package named in the ~ilc[in-package]
  form at the top of the file.  The effect of this is that all symbols
  are ~il[intern]ed in that package, except those whose packages are given
  explicitly with the ``::'' notation.  For example, if a book begins
  with ~c[(in-package \"ACL2-X\")] and then contains the form
  ~bv[]
    (defun fn (x)
      (acl2::list 'car x))
  ~ev[]
  then ~ilc[defun], ~c[fn], ~c[x], and ~ilc[car] are all ~il[intern]ed in the
  ~c[\"ACL2-X\"] package.  I.e., it is as though the following form
  were read instead:
  ~bv[]
    (acl2-x::defun acl2-x::fn (acl2-x::x)
        (acl2::list 'acl2-x::car acl2-x::x)).
  ~ev[]
  Of course, ~c[acl2-x::defun] would be the same symbol as
  ~c[acl2::defun] if the ~c[\"ACL2-X\"] package imported ~c[acl2::defun].

  If each book has its own unique package name and all the names
  defined within the book are in that package, then name clashes
  between ~il[books] are completely avoided.  This permits the construction
  of useful logical ~il[world]s by the successive inclusion of many ~il[books].
  Although it is often too much trouble to manage multiple packages,
  their judicious use is a way to minimize name clashes.  Often, a
  better way is to use ~c[local]; ~pl[local].

  How does ~ilc[include-book] know the definitions of the packages used in a
  book, since ~ilc[defpkg]s cannot be among the forms?  More generally,
  how do we know that the forms in a book will be admissible in the
  host logical ~il[world] of an ~ilc[include-book]?  ~l[certificate] for
  answers to these questions.")

(deflabel certificate
  :doc
  ":Doc-Section Books

  how a book is known to be admissible and where its ~ilc[defpkg]s reside~/

  A book, say ~c[\"arith\"], is said to have a ``certificate'' if there
  is a file named ~c[\"arith.cert\"].  Certificates are created by the
  function ~ilc[certify-book] and inspected by ~ilc[include-book].  Check
  sums are used to help ensure that certificates are legitimate and
  that the corresponding book has not been modified since
  certification.  But because the file system is insecure and check
  sums are not perfect it is possible for the inclusion of a book to
  cause inconsistency even though the book carries an impeccable
  certificate.

  The certificate includes the version number of the certifying ACL2.
  A book is considered uncertified if it is included in an ACL2
  with a different ~il[version] number.~/

  The presence of a ``valid'' certificate file for a book attests to
  two things: all of the ~il[events] of the book are admissible in a
  certain extension of the initial ACL2 logic, and the non-~ilc[local]
  ~il[events] of the book are independent of the ~ilc[local] ones
  (~pl[local-incompatibility]).  In addition, the certificate
  contains the ~il[command]s used to construct the ~il[world] in which
  certification occurred.  Among those ~il[command]s, of course, are the
  ~ilc[defpkg]s defining the packages used in the book.  When a book is
  included into a host ~il[world], that ~il[world] is first extended
  by the ~il[command]s listed in the certificate for the book.  Unless that
  causes an error due to name conflicts, the extension ensures that
  all the packages used by the book are identically defined in the
  host ~il[world].

  ~em[Security:]

  Because the host file system is insecure, there is no way ACL2 can
  guarantee that the contents of a book remain the same as when its
  certificate was written.  That is, between the time a book is
  certified and the time it is used, it may be modified.  Furthermore,
  certificates can be counterfeited.  Check sums (~pl[check-sum])
  are used to help detect such problems.  But check sums provide
  imperfect security: two different files can have the same check sum.

  Therefore, from the strictly logical point of view, one must
  consider even the inclusion of certified ~il[books] as placing a burden
  on the user:~bq[]

  The non-erroneous inclusion of a certified book is consistency
  preserving provided (a) the objects read by ~ilc[include-book] from the
  certificate were the objects written there by a ~ilc[certify-book] and
  (b) the forms read by ~ilc[include-book] from the book itself are the
  forms read by the corresponding ~ilc[certify-book].

  ~eq[]We say that a given execution of ~ilc[include-book] is ``certified''
  if a certificate file for the book is present and well-formed and
  the check sum information contained within it supports the
  conclusion that the ~il[events] read by the ~ilc[include-book] are the ones
  checked by ~ilc[certify-book].  When an uncertified ~ilc[include-book]
  occurs, warnings are printed or errors are caused.  But even if no
  warning is printed, you must accept burdens (a) and (b) if you use
  ~il[books].  These burdens are easier to live with if you protect your
  ~il[books] so that other users cannot write to them, you abstain from
  running concurrent ACL2 jobs, and you abstain from counterfeiting
  certificates.  But even on a single user uniprocessor, you can shoot
  yourself in the foot by using the ACL2 ~il[io] primitives to fabricate an
  inconsistent book and the corresponding certificate.

  Note that part (a) of the burden described above implies, in
  particular, that there are no guarantees when a certificate is
  copied.  When ~il[books] are renamed (as by copying them), it is
  recommended that their certificates be removed and the ~il[books] be
  recertified.  The expectation is that recertification will go
  through without a hitch if relative ~il[pathname]s are used.
  ~l[pathname], which is not on the guided tour.

  Certificate essentially contain two parts, a ~il[portcullis] and a
  ~il[keep].  ~l[portcullis] to continue the guided tour through
  ~il[books].")

(deflabel portcullis

; This documentation string formerly concluded (just before "~l[keep] to
; continue...") with the following discussion, until Version  2.6.  Now that we
; change include-book forms in the portcuillis to use absolute pathnames, we do
; not need this.

#|
  Recall that we disallow ~ilc[include-book] ~il[events] from the portcullis
  unless the included book's name is an absolute filename
  (~l[pathname]).  Thus, for example, under the Unix operating
  system it is impossible to certify a book if the certification
  ~il[world] was created with
  ~bv[]
  ACL2 !>(~il[include-book] \"arith\")
  ~ev[]
  The problem here is that the file actually read on behalf of such
  an ~ilc[include-book] depends upon the then current setting of the
  connected book directory (~pl[cbd]).  That setting could be
  changed before the certification occurs.  If we were to copy
  ~c[(include-book \"arith\")] into the portcullis of the book being
  certified, there is no assurance that the ~c[\"arith\"] book included
  would come from the correct directory.  However, by requiring that
  the ~ilc[include-book]s in the certification ~il[world] give book names
  that begin with slash we effectively require you to specify the full
  file name of each book involved in creating your certification
  ~il[world].  Observe that the execution of
  ~bv[]
  (~il[include-book] \"/usr/local/src/acl2/library/arith\")
  ~ev[]
  does not depend on the current book directory.  On the other hand,
  this requirement ~-[] effectively that absolute file names be used in
  the certification ~il[world] ~-[] means that a book that requires
  another book in its certification ~il[world] will be rendered
  uncertified if the required book is removed to another directory.
  If possible, any ~ilc[include-book] ~il[command] required for a book ought
  to be placed in the book itself and not in the certification
  ~il[world].  The only time this cannot be done is if the required
  book is necessary to some ~ilc[defpkg] required by your book.  Of
  course, this is just the same advice we have been giving: keep the
  certification ~il[world] as elementary as possible.
|#

  :doc
  ":Doc-Section Books

  the gate guarding the entrance to a certified book~/

  The certificate (~pl[certificate] for general information) of a
  certified file is divided into two parts, a portcullis and a
  ~il[keep].  These names come from castle lore.  The portcullis of a
  castle is an iron grate that slides up through the ceiling of the
  tunnel-like entrance.  The portcullis of a book ensures that
  ~ilc[include-book] does not start to read the book until the
  appropriate context has been created.~/

  Technically, the portcullis consists of the ~il[version] number of
  the certifying ACL2, a list of ~il[command]s used to create the
  ``certification ~il[world]'' and an alist specifying the check sums
  of all the ~il[books] included in that ~il[world].  The portcullis
  is constructed automatically by ~ilc[certify-book] from the ~il[world]
  in which ~ilc[certify-book] is called, but that ~il[world] must have
  certain properties described below.  After listing the properties we
  discuss the issues more leisurely.

  Each ~il[command] in the portcullis must be either a ~ilc[defpkg] form or an
  embedded event form (~pl[embedded-event-form]).

  Consider a book to be certified.  The book is a file containing
  event forms.  Suppose the file contains references to such symbols
  as ~c[my-pkg::fn] and ~c[acl2-arith::cancel], but that the book itself
  does not create the packages.  Then a hard Lisp error would be
  caused merely by the attempt to read the expressions in the book.
  The corresponding ~ilc[defpkg]s cannot be written into the book itself
  because the book must be compilable and Common Lisp compilers differ
  on the rules concerning the inline definition of new packages.  The
  only safe course is to make all ~ilc[defpkg]s occur outside of compiled
  files.

  More generally, when a book is certified it is certified within some
  logical ~il[world].  That ``certification ~il[world]'' contains not only
  the necessary ~ilc[defpkg]s but also, perhaps, function and constant
  definitions and maybe even references to other ~il[books].  When
  ~ilc[certify-book] creates the ~il[certificate] for a file it recovers
  from the certification ~il[world] the ~il[command]s used to create that
  ~il[world] from the initial ACL2 ~il[world].  Those ~il[command]s become
  part of the portcullis for the certified book.  In addition,
  ~ilc[certify-book] records in the portcullis the check sums
  (~pl[check-sum]) of all the ~il[books] included in the certification
  ~il[world].

  ~ilc[Include-book] presumes that it is impossible even to read the
  contents of a certified book unless the portcullis can be
  ``raised.'' To raise the portcullis we must be able to execute
  (possibly redundantly, but certainly without error), all of the
  ~il[command]s in the portcullis and then verify that the ~il[books] thus
  included were identical to those used to build the certification
  ~il[world] (up to check sum).  This raising of the portcullis must
  be done delicately since ~ilc[defpkg]s are present: we cannot even read
  a ~il[command] in the portcullis until we have successfully executed the
  previous ones, since packages are being defined.

  Clearly, a book is most useful if it is certified in the most
  elementary extension possible of the initial logic.  If, for
  example, your certification ~il[world] happens to contain a
  ~ilc[defpkg] for ~c[\"MY-PKG\"] and the function ~c[foo], then those
  definitions become part of the portcullis for the book.  Every time
  the book is included, those names will be defined and will have to
  be either new or redundant (~pl[redundant-events]).  But if
  those names were not necessary to the certification of the book,
  their presence would unnecessarily restrict the utility of the book.

  ~l[keep] to continue the guided tour of ~il[books].")

(deflabel version
  :doc
  ":Doc-Section Miscellaneous

  ACL2 Version Number~/

  To determine the version number of your copy of ACL2, evaluate the ACL2
  constant ~c[*acl2-version*].  The value will be a string.  For example,
  ~bv[]
  ACL2 !>*acl2-version*
  \"ACL2 Version 1.9\"
  ~ev[]
  ~/

  ~il[Books] are considered certified only in the same version of ACL2
  in which the certification was done.  The ~il[certificate] file
  records the version number of the certifying ACL2 and
  ~il[include-book] considers the book uncertified if that does not
  match the current version number.  Thus, each time we release a new
  version of ACL2, previously certified books should be recertified.

  One reason for this is immediately obvious from the fact that the
  version number is a logical constant in the ACL2 theory:  changing
  the version number changes the axioms!  For example, in version
  1.8 you can prove 
  ~bv[]
  (defthm version-8
    (equal *acl2-version* \"ACL2 Version 1.8\"))
  ~ev[]
  while in version 1.9 you can prove
  ~bv[]
  (defthm version-9
    (equal *acl2-version* \"ACL2 Version 1.9\"))
  ~ev[]
  Thus, if a book containing the former were included into version
  1.9, one could prove a contradiction.

  We could eliminate this particular threat of unsoundness by not
  making the version number part of the axioms.  But there are over
  150 constants in the system, most having to do with the fact that
  ACL2 is coded in ACL2, and ``version number inconsistency'' is just
  the tip of the iceberg.  Other likely-to-change constants include
  ~c[*common-lisp-specials-and-constants*], which may change when we
  port ACL2 to some new implementation of Common Lisp, and
  ~c[*acl2-exports*], which lists commonly used ACL2 command names
  users are likely to import into new packages.  Furthermore, it is
  possible that from one version of the system to another we might
  change, say, the default values on some system function or otherwise
  make ``intentional'' changes to the axioms.  It is even possible one
  version of the system is discovered to be unsound and we release a
  new version to correct our error.

  Therefore we adopted the draconian policy that books are certified
  by a given version of ACL2 and ``must'' be recertified to be used
  in other versions.  We put ``must'' in quotes because in fact, ACL2
  allows a book that was certified in one ACL2 version to be included
  in a later version, using ~ilc[include-book].  But ACL2 does not allow
  ~ilc[certify-book] to succeed when such an ~ilc[include-book] is executed on its
  behalf.  Also, you may experience undesirable behavior if you avoid
  recertification when moving to a different version.  (We try to
  prevent some undesirable behavior by refusing to load the compiled
  code for an uncertified book, but this does not guarantee good
  behavior.)  Hence we recommend that you stick to the draconion
  policy of recertifying books when updating to a new ACL2 version.

  The string ~c[*acl2-version*] can contain implementation-specific
  information in addition to the version number.  For example, in
  Macintosh Common Lisp (MCL) ~c[(char-code #\Newline)] is 13, while as
  far as we know, it is 10 in every other Common Lisp.  Our concern is
  that one could certify a book in an MCL-based ACL2 with the theorem
  ~bv[]
  (equal (char-code #\Newline) 13)
  ~ev[]
  and then include this book in another Lisp and thereby prove ~c[nil].
  So, when a book is certified in an MCL-based ACL2, the book's
  ~il[certificate] mentions ``MCL'' in its version string.  Moreover,
  ~c[*acl2-version*] similarly mentions ``MCL'' when the ACL2 image has
  been built on top of MCL.  Thus, an attempt to include a book in an
  MCL-based ACL2 that was certified in a non-MCL-based ACL2, or
  vice-versa, will be treated like an attempt to include an
  uncertified book.~/")

(deflabel keep
  :doc
  ":Doc-Section Books

  how we know if ~ilc[include-book] read the correct files~/

  The certificate (~pl[certificate] for general information) of a
  certified file is divided into two parts, a ~il[portcullis] and a
  keep.  These names come from castle lore.  The keep is the strongest
  and usually tallest tower of a castle from which the entire
  courtyard can be surveyed by the defenders.  The keep of a book is a
  list of file names and check sums used after the book has been
  included, to determine if the files read were (up to check sum)
  those certified.~/

  Once the ~il[portcullis] is open, ~ilc[include-book] can enter the book
  and read the event forms therein.  The non-~ilc[local] event forms are
  in fact executed, extending the host theory.  That may read in other
  ~il[books].  When that has been finished, the keep of the
  ~il[certificate] is inspected.  The keep is a list of the book names
  which are included (hereditarily through all subbooks) in the
  certified book (including the certified book itself) together with
  the check sums of the objects in those ~il[books] at the time of
  certification.  We compare the check sums of the ~il[books] just included
  to the check sums of the ~il[books] stored in the keep.  If differences
  are found then we know that the book or one of its subbooks has been
  changed since certification.

  ~l[include-book] to continue the guided tour through ~il[books].")

; The documentation for include-book is in axioms.lisp, where the
; include-book event is defined.

(deflabel uncertified-books
  :doc
  ":Doc-Section Books

  invalid ~il[certificate]s and uncertified ~il[books]~/

  ~ilc[Include-book] has a special provision for dealing with uncertified
  ~il[books]: If the file has no ~il[certificate] or an invalid
  ~il[certificate] (i.e., one whose check sums describe files other
  than the ones actually read), a warning is printed and the book is
  otherwise processed as though it were certified and had an open
  ~il[portcullis].  (For details ~pl[books], ~pl[certificate],
  and ~pl[portcullis].)

  This can be handy, but it can have disastrous consequences.~/

  The provision allowing uncertified ~il[books] to be included can
  have disastrous consequences, ranging from hard lisp errors, to
  damaged memory, to quiet logical inconsistency.

  It is possible for the inclusion of an uncertified book to render
  the logic inconsistent.  For example, one of its non-~ilc[local] ~il[events]
  might be ~c[(defthm t-is-nil (equal t nil))].  It is also possible
  for the inclusion of an uncertified book to cause hard errors or
  ~il[breaks] into raw Common Lisp.  For example, if the file has been
  edited since it was certified, it may contain too many open
  parentheses, causing Lisp to read past ``end of file.'' Similarly,
  it might contain non-ACL2 objects such as ~c[3.1415] or ill-formed
  event forms that cause ACL2 code to break.

  Even if a book is perfectly well formed and could be certified (in a
  suitable extension of ACL2's initial ~il[world]), its uncertified
  inclusion might cause Lisp errors or inconsistencies!  For example,
  it might mention packages that do not exist in the host ~il[world].
  The ~il[portcullis] of a certified book ensures that the correct
  ~ilc[defpkg]s have been admitted, but if a book is read without
  actually raising its ~il[portcullis], symbols in the file, e.g.,
  ~c[acl2-arithmetic::fn], could cause ``unknown package'' errors in
  Common Lisp.  Perhaps the most subtle disaster occurs if the host
  ~il[world] does have a ~ilc[defpkg] for each package used in the book
  but the host ~ilc[defpkg] imports different symbols than those required
  by the ~il[portcullis].  In this case, it is possible that formulas
  which were theorems in the certified book are non-theorems in the
  host ~il[world], but those formulas can be read without error and
  will then be quietly assumed.

  In short, if you include an uncertified book, ~st[all bets are off]
  regarding the validity of the future behavior of ACL2.

  That said, it should be noted that ACL2 is pretty tough and if
  errors don't occur, the chances are that deductions after the
  inclusion of an uncertified book are probably justified in the
  (possibly inconsistent) logical extension obtained by assuming the
  admissibility and validity of the definitions and conjectures in the
  book.")

; We now use encapsulate to implement defstub.

(defun defstub-ignores (formals body)

; The test below is sufficient to ensure that the set-difference-equal
; used to compute the ignored vars will not cause an error.  We return
; a true list.  The formals and body will be checked thoroughly by the
; encapsulate, provided we generate it!  Provided they check out, the
; result returned are the ignored formals.

  (if (and (symbol-listp formals)
           (or (symbolp body)
               (and (consp body)
                    (symbol-listp (cdr body)))))
      (set-difference-equal
       formals
       (if (symbolp body)
           (list body)
         (cdr body)))
    nil))

(defun defstub-body (output)

; This strange little function is used to turn an output signature
; spec (in either the old or new style) into a term.  It never causes
; an error, even if output is ill-formed!  What it returns in that
; case is irrelevant.  If output is well-formed, i.e., is one of:

;       output               result
; *                           nil
; x                           x
; state                       state
; (mv * state *)              (mv nil state nil)
; (mv x state y)              (mv x state y)

; it replaces the *'s by nil and otherwise doesn't do anything.

  (cond ((atom output)
         (cond ((equal output '*) nil)
               (t output)))
        ((equal (car output) '*)
         (cons nil (defstub-body (cdr output))))
        (t (cons (car output) (defstub-body (cdr output))))))

; The following function is used to implement a slighly generalized
; form of macro args, namely one in which we can provide an arbitrary
; number of ordinary arguments terminated by an arbitrary number of
; keyword argument pairs.

(defun partition-rest-and-keyword-args1 (x)
  (cond ((endp x) (mv nil nil))
        ((keywordp (car x))
         (mv nil x))
        (t (mv-let (rest keypart)
                   (partition-rest-and-keyword-args1 (cdr x))
                   (mv (cons (car x) rest)
                       keypart)))))

(defun partition-rest-and-keyword-args2 (keypart keys alist)

; We return t if keypart is ill-formed as noted below.  Otherwise, we
; return ((:keyn . vn) ... (:key1 . v1)).

  (cond ((endp keypart) alist)
        ((and (keywordp (car keypart))
              (consp (cdr keypart))
              (not (assoc-eq (car keypart) alist))
              (member (car keypart) keys))
         (partition-rest-and-keyword-args2 (cddr keypart)
                                           keys
                                           (cons (cons (car keypart)
                                                       (cadr keypart))
                                                 alist)))
        (t t)))

(defun partition-rest-and-keyword-args (x keys)

; X is assumed to be a list of the form (a1 ... an :key1 v1 ... :keyk
; vk), where no ai is a keyword.  We return (mv erp rest alist), where
; erp is t iff the keyword section of x is ill-formed.  When erp is
; nil, rest is '(a1 ... an) and alist is '((:key1 . v1) ... (:keyk
; . vk)).

; The keyword section is ill-formed if it contains a non-keyword in an
; even numbered element, if it binds the same keyword more than once,
; or if it binds a keyword other than those listed in keys.

  (mv-let (rest keypart)
          (partition-rest-and-keyword-args1 x)
          (let ((alist (partition-rest-and-keyword-args2 keypart keys nil)))
            (cond
             ((eq alist t) (mv t nil nil))
             (t (mv nil rest alist))))))

(defmacro defstub (name &rest rst)

  #-small-acl2-image
  ":Doc-Section Events

  stub-out a function symbol~/
  ~bv[]
  Examples:
  ACL2 !>(defstub subr1 (* * state) => (mv * state))
  ACL2 !>(defstub add-hash (* * hash-table) => hash-table)~/

  General Forms:
  (defstub name args-sig => output-sig)
  (defstub name args-sig => output-sig :doc doc-string)
  ~ev[]

  ~c[Name] is a new function symbol and ~c[(name . args-sig) => output-sig)]
  is a ~il[signature].  If the optional ~ilc[doc-string] is supplied
  it should be a documentation string.  See also the ``Old Style''
  heading below.

  ~c[Defstub] macro expands into an ~ilc[encapsulate] event
  (~pl[encapsulate]).  Thus, no axioms are available about ~c[name]
  but it may be used wherever a function of the given signature is
  permitted.

  Old Style:
  ~bv[]
  Old Style General Form:
  (defstub name formals output)
  (defstub name formals output :doc doc-string)
  ~ev[] 
  where ~c[name] is a new function symbol, ~c[formals] is its list of
  formal parameters, and ~c[output] is either a symbol (indicating
  that the function returns one result) or a term of the form
  ~c[(mv s1 ... sn)], where each ~c[si] is a symbol (indicating that the
  function returns ~c[n] results).  Whether and where the symbol
  ~ilc[state] occurs in ~c[formals] and ~c[output] indicates how the
  function handles ~il[state].  It should be the case that 
  ~c[(name formals output)] is in fact a signature (~pl[signature]).

  Note that with the old style notation it is impossible to stub-out
  a function that uses any single-threaded object other than state.
  The old style is preserved for compatibility with earlier versions of
  ACL2."

  (mv-let (erp args key-alist)
          (partition-rest-and-keyword-args rst '(:doc))
          (cond
           ((or erp
                (not (or (equal (length args) 2)
                         (and (equal (length args) 3)
                              (symbol-listp (car args))
                              (symbolp (cadr args))
                              (equal (symbol-name (cadr args)) "=>")))))
            `(er soft 'defstub
                 "Defstub must be of the form (defstub name formals ~
                  body) or (defstub name args-sig => body-sig), where ~
                  args-sig is a true-list of symbols.  Both ~
                  forms permit an optional, final :DOC doc-string ~
                  argument.  See :DOC defstub."))
           (t
            (let ((doc (cdr (assoc-eq :doc key-alist))))
              (cond
               ((equal (length args) 2)

; Old style
                (let* ((formals (car args))
                       (body (cadr args))
                       (ignores (defstub-ignores formals body)))
                  `(encapsulate
                    ((,name ,formals ,body))
                    (logic)
                    (local
                     (defun ,name ,formals
                       (declare (ignore ,@ignores))
                       ,body))
                    ,@(if doc `((defdoc ,name ,doc)) nil))))
               (t (let* ((args-sig (car args))
                         (body-sig (caddr args))
                         (formals (gen-formals-from-pretty-flags args-sig))
                         (body (defstub-body body-sig))
                         (ignores (defstub-ignores formals body))
                         (stobjs (collect-non-x '* args-sig)))
                    `(encapsulate
                      (((,name ,@args-sig) => ,body-sig))
                      (logic)
                      (local
                       (defun ,name ,formals
                         (declare (ignore ,@ignores)
                                  (xargs :stobjs ,stobjs))
                         ,body))
                      ,@(if doc `((defdoc ,name ,doc)) nil))))))))))

; Next we implement defchoose and defun-sk.

(defun redundant-defchoosep (name event-form len wrld)

  (let* ((old-ev (get-event name wrld)))
    (and old-ev
         (eq (car old-ev) 'defchoose)
         (eq (nth 1 event-form) (nth 1 old-ev))
         (equal (nth 2 event-form) (nth 2 old-ev))
         (equal (nth 3 event-form) (nth 3 old-ev))
         (equal (if (= len 5)
                    (nth 5 event-form)
                  (nth 4 event-form))
                (if (= (length (cdr old-ev)) 5)
                    (nth 5 old-ev)
                  (nth 4 old-ev))))))

(defun chk-arglist-for-defchoose (args bound-vars-flg ctx state)
  (cond ((arglistp args) (value nil))
        ((not (true-listp args))
         (er soft ctx
             "The ~#0~[bound~/free~] variables of a DEFCHOOSE event must be a ~
              true list but ~x1 is not."
             (if bound-vars-flg 0 1)
             args))
        (t (mv-let (culprit explan)
                   (find-first-bad-arg args)
                   (er soft ctx
                       "The ~#0~[bound~/free~] variables of a DEFCHOOSE event ~
                        must be a true list of distinct, legal variable names.  ~
                        ~x1 is not such a list.  The element ~x2 violates the ~
                        rules because it ~@3."
                       (if bound-vars-flg 0 1)
                       args culprit explan)))))

(defun defchoose-constraint (fn bound-vars formals tbody ctx wrld state)

; It seems a pity to translate tbody, since it's already translated, but that
; seems much simpler than the alternatives.

  (cond
   ((null (cdr bound-vars))
    (er-let*
     ((consequent (translate
                   `(let ((,(car bound-vars) ,(cons fn formals)))
                      ,tbody)
                   t t t ctx wrld state)))
     (value (fcons-term*
             'implies
             tbody
             consequent))))
   (t
    (er-let*
     ((consequent (translate
                   `(mv-let ,bound-vars
                            ,(cons fn formals)
                            ,tbody)
                   t t t ctx wrld state)))
     (value (fcons-term*
             'if

; We need this true-listp axiom in order to prove guard conjectures generated
; by mv-nth in defun-sk.

             (fcons-term*
              'true-listp
              (cons-term fn formals))
             (fcons-term*
              'implies
              tbody
              consequent)
             *nil*))))))

(defun defchoose-fn (def state event-form)
  (when-logic
   "DEFCHOOSE"
   (with-ctx-summarized
    (if (output-in-infixp state) event-form (cons 'defchoose (car def)))
    (let* ((wrld (w state))
           (event-form (or event-form (cons 'defchoose def)))
           (len (length def)))
      (er-progn
       (chk-all-but-new-name (car def) ctx 'constrained-function wrld state)
       (cond
        ((not (and (true-listp def)
                   (member len '(4 5))))
         (er soft ctx
             "Defchoose forms must have the form (defchoose fn bound-vars ~
              formals body), optionally with a doc-string before the body.  ~
              However, ~x0 does not have this form."
             event-form))
        ((and (= len 5)
              (not (doc-stringp (nth 3 def))))
         (er soft ctx
             "When a DEFCHOOSE form has five arguments, the fourth must be a ~
              doc string.  The form ~x0 is thus illegal.  See :DOC doc-string."
             event-form))
        ((redundant-defchoosep (car def) event-form len wrld)
         (stop-redundant-event state))
        ((null (cadr def))
         (er soft ctx
             "The bound variables of a DEFCHOOSE form must be non-empty.  The ~
              form ~x0 is therefore illegal."
             event-form))
        (t
         (let ((fn (car def))
               (bound-vars (if (atom (cadr def))
                               (list (cadr def))
                             (cadr def)))
               (formals (caddr def))
               (doc (and (= len 5) (cadddr def)))
               (body (if (= len 5) (cadddr (cdr def)) (cadddr def))))
           (er-progn
            (chk-arglist-for-defchoose bound-vars t ctx state)
            (chk-arglist-for-defchoose formals nil ctx state)
            (er-let*
              ((tbody (translate body t t t ctx wrld state))
               (wrld (chk-just-new-name fn 'function nil ctx wrld state))
               (doc-pair (translate-doc fn doc ctx state)))
              (cond
               ((intersectp-eq bound-vars formals)
                (er soft ctx
                    "The bound and free variables of a DEFCHOOSE form ~
                     must not intersect, but their intersection for ~
                     the form ~x0 is ~x1."
                    event-form
                    (intersection-eq bound-vars formals)))
               (t
                (let* ((body-vars (all-vars tbody))
                       (bound-and-free-vars (append bound-vars formals))
                       (diff (set-difference-eq bound-and-free-vars
                                                body-vars))
                       (ignore-ok (cdr (assoc-eq
                                        :ignore-ok
                                        (table-alist 'acl2-defaults-table
                                                     wrld)))))
                  (cond
                   ((not (subsetp-eq body-vars bound-and-free-vars))
                    (er soft ctx
                        "All variables in the body of a DEFCHOOSE ~
                         form must appear among the bound or free ~
                         variables supplied in that form.  However, ~
                         the ~#0~[variable ~x0 does~/variables ~&0 ~
                         do~] not appear in the bound or free ~
                         variables of the form ~x1, even though ~
                         ~#0~[it appears~/they appear~] in its body."
                        (set-difference-eq body-vars
                                           (append bound-vars formals))
                        event-form))
                   ((and diff
                         (null ignore-ok))
                    (er soft ctx
                        "The variable~#0~[ ~&0~ occurs~/s ~&0 occur~] ~
                         in the body of the form ~x1.  However, ~
                         ~#0~[this variable does~/these variables ~
                         do~] not appear either in the bound ~
                         variables or the formals of that form.  In ~
                         order to avoid this error, see :DOC ~
                         set-ignore-ok."
                        diff
                        event-form))
                   (t
                    (pprogn
                     (cond
                      ((eq ignore-ok :warn)
                       (warning$ ctx "Ignored-variables"
                                 "The variable~#0~[ ~&0 occurs~/s ~&0 ~
                                  occur~] in the body of the ~
                                  following DEFCHOOSE ~
                                  form:~|~x1~|However, ~#0~[this ~
                                  variable does~/these variables do~] ~
                                  not appear either in the bound ~
                                  variables or the formals of that ~
                                  form.  In order to avoid this ~
                                  warning, see :DOC set-ignore-ok."
                                 diff
                                 event-form))
                      (t state))
                     (let* ((stobjs-in
                             (compute-stobj-flags formals nil wrld))
                            (stobjs-out
                             (compute-stobj-flags bound-vars nil wrld))
                            (wrld
                             (putprop
                              fn 'constrainedp t
                              (putprop
                               fn 'symbol-class
                               :common-lisp-compliant
                               (putprop-unless
                                fn 'stobjs-out stobjs-out nil
                                (putprop-unless
                                 fn 'stobjs-in stobjs-in nil
                                 (putprop
                                  fn 'formals formals
                                  (update-doc-data-base
                                   fn doc doc-pair wrld))))))))
                       (er-let*
                         ((constraint
                           (defchoose-constraint
                             fn bound-vars formals tbody
                             ctx wrld state)))
                         (install-event fn
                                        event-form
                                        'defchoose
                                        fn
                                        nil
                                        `(defuns nil nil

; Keep the following in sync with intro-udf-lst2.

                                           (,fn
                                            ,formals
                                            (declare (ignore ,@formals))
                                            (throw-raw-ev-fncall
                                             '(ev-fncall-null-body-er ,fn))))
                                        (putprop
                                         fn 'defchoose-axiom constraint wrld)
                                        state)))))))))))))))))))

(defun non-acceptable-defun-sk-p (name args body doc quant-ok)

; Since this is just a macro, we only do a little bit of vanilla checking,
; leaving it to the real events to implement the most rigorous checks.

  (declare (ignore doc))
  (let ((bound-vars (and (true-listp body) ;this is to guard cadr
                         (cadr body)
                         (if (atom (cadr body))
                             (list (cadr body))
                           (cadr body)))))
    (cond
     ((not (true-listp args))
      (msg "The second argument of DEFUN-SK must be a true list of legal ~
            variable names, but ~x0 is not a true-listp."
           args))
     ((not (arglistp args))
      (mv-let
       (culprit explan)
       (find-first-bad-arg args)
       (msg "The formal parameters (second argument) of a DEFUN-SK form must ~
             be a true list of distinct, legal variable names.  ~x0 is not ~
             such a list.  The element ~x1 violates the rules because it ~@2."
            args culprit explan)))
     ((not (and (true-listp body)
                (equal (length body) 3)
                (symbolp (car body))
                (member-equal (symbol-name (car body))
                              '("FORALL" "EXISTS"))
                (true-listp bound-vars)
                (null (collect-non-legal-variableps bound-vars))))
      (msg "The body (last argument) of a DEFUN-SK form must be a true list of ~
            the form (Q vars term), where Q is FORALL or EXISTS and vars is a ~
            variable or a true list of variables.  The body ~x0 is therefore ~
            illegal."
           body))
     ((member-eq 'state bound-vars)
      (msg "The body (last argument) of a DEFUN-SK form must be a true list of ~
            the form (Q vars term), where vars represents the bound ~
            variables.  The bound variables must not include STATE.  The body ~
            ~x0 is therefore illegal."
           body))
     ((null (cadr body))
      (msg "The variables of the body of a DEFUN-SK, following the quantifier ~
            EXISTS or FORALL, must be a non-empty list.  However, in DEFUN-SK ~
            of ~x0, they are empty."
           name))
     ((intersectp-eq bound-vars args)
      (msg "The formal parameters of a DEFN-SK form must be disjoint from the ~
            variables bound by its body.  However, the ~#0~[variable ~x0 ~
            belongs~/variables ~&0 belong~] to both the formal parameters, ~
            ~x1, and the bound variables, ~x2."
           (intersection-eq bound-vars args)
           args bound-vars))
     ((and (not quant-ok)
           (or (symbol-name-tree-occur 'forall (caddr body))
               (symbol-name-tree-occur 'exists (caddr body))))
      (msg "The symbol ~x0 occurs in the term you have supplied to DEFUN-SK, ~
            namely, ~x1.  By default, this is not allowed.  Perhaps you ~
            believe that DEFUN-SK can appropriately handle quantifiers other ~
            than one outermost quantifier; sadly, this is not yet the case ~
            (though you are welcome to contact the implementors and request ~
            this capability).  If however you really intend this DEFUN-SK form ~
            to be executed (because, for example, ~x0 is in the scope of a ~
            macro that expands it away), simply give a non-nil :quant-ok ~
            argument.  See :DOC defun-sk."
           (if (symbol-name-tree-occur 'forall (caddr body))
               'forall
             'exists)
           body))
     (t nil))))

(defmacro defun-sk (name args body &key doc quant-ok skolem-name thm-name)

  #-small-acl2-image
  ":Doc-Section Events

  define a function whose body has an outermost quantifier~/
  ~bv[]
  Examples:
  (defun-sk exists-x-p0-and-q0 (y z)
    (exists x
            (and (p0 x y z)
                 (q0 x y z))))

  (defun-sk exists-x-p0-and-q0 (y z) ; equivalent to the above
    (exists (x)
            (and (p0 x y z)
                 (q0 x y z))))

  (defun-sk forall-x-y-p0-and-q0 (z)
    (forall (x y)
            (and (p0 x y z)
                 (q0 x y z))))~/

  General Form:
  (defun-sk fn (var1 ... varn) body
    &key doc quant-ok skolem-name thm-name)
  ~ev[]
  where ~c[fn] is the symbol you wish to define and is a new symbolic
  name (~pl[name]), ~c[(var1 ... varn)] is its list of formal
  parameters (~pl[name]), and ~c[body] is its body, which must be
  quantified as described below.  The ~c[&key] argument ~ilc[doc] is an optional
  ~il[documentation] string to be associated with ~c[fn]; for a description
  of its form, ~pl[doc-string].  In the case that ~c[n] is 1, the list
  ~c[(var1)] may be replaced by simply ~c[var1].  The other arguments are
  explained below.  For a more elaborate example than those above,
  ~pl[Tutorial4-Defun-Sk-Example].

  ~l[quantifiers] for an example illustrating how the use of
  recursion, rather than explicit quantification with ~c[defun-sk], may be
  preferable.

  Below we describe the ~c[defun-sk] event precisely.  First, let us
  consider the examples above.  The first example, again, is:
  ~bv[]
  (defun-sk exists-x-p0-and-q0 (y z)
    (exists x
            (and (p0 x y z)
                 (q0 x y z))))
  ~ev[]
  It is intended to represent the predicate with formal parameters ~c[y]
  and ~c[z] that holds when for some ~c[x], ~c[(and (p0 x y z) (q0 x y z))]
  holds.  In fact ~c[defun-sk] is a macro that adds the following two
  ~ilc[events], as shown just below.  The first axiom guarantees that if
  this new predicate holds of ~c[y] and ~c[z], then the term in question,
  ~c[(exists-x-p0-and-q0-witness y z)], is an example of the ~c[x] that is
  therefore supposed to exist.  (Intuitively, we are axiomatizing
  ~c[exists-x-p0-and-q0-witness] to pick a witness if there is one.)
  Conversely, the second event below guarantees that if there is any
  ~c[x] for which the term in question holds, then the new predicate does
  indeed holds of ~c[y] and ~c[z]. 
  ~bv[]
  (defun exists-x-p0-and-q0 (y z)
    (let ((x (exists-x-p0-and-q0-witness y z)))
      (and (p0 x y z) (q0 x y z))))
  (defthm exists-x-p0-and-q0-suff
    (implies (and (p0 x y z) (q0 x y z))
             (exists-x-p0-and-q0 y z)))
  ~ev[]
  Now let us look at the third example from the introduction above:
  ~bv[]
  (defun-sk forall-x-y-p0-and-q0 (z)
    (forall (x y)
            (and (p0 x y z)
                 (q0 x y z))))
  ~ev[]
  The intention is to introduce a new predicate
  ~c[(forall-x-y-p0-and-q0 z)] which states that the indicated conjunction
  holds of all ~c[x] and all ~c[y] together with the given ~c[z].  This time, the
  axioms introduced are as shown below.  The first event guarantees
  that if the application of function ~c[forall-x-y-p0-and-q0-witness] to
  ~c[z] picks out values ~c[x] and ~c[y] for which the given term
  ~c[(and (p0 x y z) (q0 x y z))] holds, then the new predicate
  ~c[forall-x-y-p0-and-q0] holds of ~c[z].  Conversely, the (contrapositive
  of) the second axiom guarantees that if the new predicate holds of
  ~c[z], then the given term holds for all choices of ~c[x] and ~c[y] (and that
  same ~c[z]). 
  ~bv[]
  (defun forall-x-y-p0-and-q0 (z)
    (mv-let (x y)
            (forall-x-y-p0-and-q0-witness z)
            (and (p0 x y z) (q0 x y z))))
  (defthm forall-x-y-p0-and-q0-necc
    (implies (not (and (p0 x y z) (q0 x y z)))
             (not (forall-x-y-p0-and-q0 z))))
  ~ev[]
  The examples above suggest the critical property of ~c[defun-sk]:  it
  indeed does introduce the quantified notions that it claims to
  introduce.

  We now turn to a detailed description ~c[defun-sk], starting with a
  discussion of its arguments as shown in the \"General Form\" above.

  The third argument, ~c[body], must be of the form
  ~bv[]
  (Q bound-vars term)
  ~ev[]
  where:  ~c[Q] is the symbol ~ilc[forall] or ~ilc[exists] (in the \"ACL2\"
  package), ~c[bound-vars] is a variable or true list of variables
  disjoint from ~c[(var1 ... varn)] and not including ~ilc[state], and
  ~c[term] is a term.  The case that ~c[bound-vars] is a single variable
  ~c[v] is treated exactly the same as the case that ~c[bound-vars] is
  ~c[(v)].

  The result of this event is to introduce a ``Skolem function,'' whose name is
  the keyword argument ~c[skolem-name] if that is supplied, and otherwise is
  the result of modifying ~c[fn] by suffixing \"-WITNESS\" to its name.  The
  following definition and one of the following two theorems (as indicated) are
  introduced for ~c[skolem-name] and ~c[fn] in the case that ~c[bound-vars]
  (see above) is a single variable ~c[v].  The name of the ~ilc[defthm] event
  may be supplied as the value of the keyword argument ~c[:thm-name]; if it is
  not supplied, then it is the result of modifying ~c[fn] by suffixing
  \"-SUFF\" to its name in the case that the quantifier is ~ilc[exists], and
  \"-NECC\" in the case that the quantifier is ~ilc[forall].
  ~bv[]
  (defun fn (var1 ... varn)
    (let ((v (skolem-name var1 ... varn)))
      term))

  (defthm fn-suff ;in case the quantifier is EXISTS
    (implies term
             (fn var1 ... varn)))

  (defthm fn-necc ;in case the quantifier is FORALL
    (implies (not term)
             (not (fn var1 ... varn))))
  ~ev[]
  In the case that ~c[bound-vars] is a list of at least two variables, say
  ~c[(bv1 ... bvk)], the definition above is the following instead, but
  the theorem remains unchanged.
  ~bv[]
  (defun fn (var1 ... varn)
    (mv-let (bv1 ... bvk)
            (skolem-name var1 ... varn)
            term))
  ~ev[]

  In order to emphasize that the last element of the list ~c[body] is a
  term, ~c[defun-sk] checks that the symbols ~ilc[forall] and ~ilc[exists] do
  not appear anywhere in it.  However, on rare occasions one might
  deliberately choose to violate this convention, presumably because
  ~ilc[forall] or ~ilc[exists] is being used as a variable or because a
  macro call will be eliminating ``calls of'' ~ilc[forall] and ~ilc[exists].
  In these cases, the keyword argument ~c[quant-ok] may be supplied a
  non-~c[nil] value.  Then ~c[defun-sk] will permit ~ilc[forall] and
  ~ilc[exists] in the body, but it will still cause an error if there is
  a real attempt to use these symbols as quantifiers.

  ~c[Defun-sk] is a macro implemented using ~ilc[defchoose], and hence should
  only be executed in ~il[defun-mode] ~c[:]~ilc[logic]; ~pl[defun-mode] and
  ~pl[defchoose].

  If you find that the rewrite rules introduced with a particular use of
  ~c[defun-sk] are not ideal, then at least two reasonable courses of action
  are available for you.  Perhaps the best option is to prove the ~ilc[rewrite]
  rules you want.  If you see a pattern for creating rewrite rules from your
  ~c[defun-sk] events, you might want to write a macro that executes a
  ~c[defun-sk] followed by one or more ~ilc[defthm] events.  Another option is
  to write your own variant of the ~c[defun-sk] macro, say, ~c[my-defun-sk],
  for example by modifying a copy of the definition of ~c[defun-sk] from the
  ACL2 sources.

  If you want to represent nested quantifiers, you can use multiple
  ~c[defun-sk] events.  For example, in order to represent
  ~bv[]
  (forall x (exists y (p x y z)))
  ~ev[]
  you can use ~c[defun-sk] twice, for example as follows.
  ~bv[]
  (defun-sk exists-y-p (x z)
    (exists y (p x y z)))

  (defun-sk forall-x-exists-y-p (z)
    (forall x (exists-y-p x z)))
  ~ev[]

  Some distracting and unimportant warnings are inhibited during
  ~c[defun-sk].

  Note that this way of implementing quantifiers is not a new idea.
  Hilbert was certainly aware of it 60 years ago!  A paper by ACL2
  authors Kaufmann and Moore, entitled ``Structured Theory Development
  for a Mechanized Logic'' (Journal of Automated Reasoning 26, no. 2 (2001),
  pp. 161-203) explains why our use of ~ilc[defchoose] is appropriate, even in
  the presence of ~c[epsilon-0] induction.~/"

  (let* ((exists-p (and (true-listp body)
                        (symbolp (car body))
                        (equal (symbol-name (car body)) "EXISTS")))
         (bound-vars (and (true-listp body)
                          (or (symbolp (cadr body))
                              (true-listp (cadr body)))
                          (cond ((atom (cadr body))
                                 (list (cadr body)))
                                (t (cadr body)))))
         (body-guts (and (true-listp body) (caddr body)))
         (skolem-name
          (or skolem-name
              (intern-in-package-of-symbol
               (concatenate 'string (symbol-name name) "-WITNESS")
               name)))
         (thm-name
          (or thm-name
              (intern-in-package-of-symbol
               (concatenate 'string (symbol-name name)
                            (if exists-p "-SUFF" "-NECC"))
               name)))
         (msg (non-acceptable-defun-sk-p name args body doc quant-ok)))
    (if msg
        `(er soft '(defun-sk . ,name)
             "~@0"
             ',msg)
      `(encapsulate
        ()
        (set-match-free-default :all)
        (set-inhibit-warnings "Theory" "Use" "Free" "Non-rec" "Infected")
        (encapsulate
         ((,skolem-name ,args
                         ,(if (= (length bound-vars) 1)
                              (car bound-vars)
                            (cons 'mv bound-vars))))
         (local (in-theory '(implies)))
         (local
          (defchoose ,skolem-name ,bound-vars ,args
            ,(if exists-p
                 body-guts
               `(not ,body-guts))))

         ; A :type-prescription lemma is needed in the case of more than one bound
         ; variable, in case we want to do guard proofs.

         ,@(cond
            ((null (cdr bound-vars)) nil)
            (t
             `((local (defthm ,(intern-in-package-of-symbol
                                (concatenate 'string
                                             (symbol-name skolem-name)
                                             "-TYPE-PRESCRIPTION")
                                skolem-name)
                        (true-listp ,(cons skolem-name args))
                        :rule-classes :type-prescription
                        :hints (("Goal" :by ,skolem-name)))))))
         (defun ,name ,args
           ,(if (= (length bound-vars) 1)
                `(let ((,(car bound-vars) (,skolem-name ,@args)))
                   ,body-guts)
              `(mv-let (,@bound-vars)
                       (,skolem-name ,@args)
                       ,body-guts)))
         (in-theory (disable (,name)))
         (defthm ,thm-name
           ,(if exists-p
                `(implies ,body-guts
                           (,name ,@args))
              `(implies (not ,body-guts)
                        (not (,name ,@args))))
           :hints (("Goal"
                     :use ,skolem-name)))
         ,@(if doc
               `((defdoc ,name ,doc))
             nil))))))

(deflabel forall
  :doc
  ":Doc-Section Defun-sk

  universal quantifier~/~/

  The symbol ~c[forall] (in the ACL2 package) represents universal
  quantification in the context of a ~ilc[defun-sk] form.
  ~l[defun-sk] and ~pl[exists].

  ~l[quantifiers] for an example illustrating how the use of
  recursion, rather than explicit quantification with ~ilc[defun-sk], may be
  preferable.")

(deflabel exists
  :doc
  ":Doc-Section Defun-sk

  existential quantifier~/~/

  The symbol ~c[exists] (in the ACL2 package) represents existential
  quantification in the context of a ~ilc[defun-sk] form.
  ~l[defun-sk] and ~pl[forall].

  ~l[quantifiers] for an example illustrating how the use of
  recursion, rather than explicit quantification with ~ilc[defun-sk], may be
  preferable.")

(deflabel quantifiers
  :doc
  ":Doc-Section Defun-sk

  issues about quantification in ACL2~/

  ACL2 supports first-order quantifiers ~ilc[exists] and ~ilc[forall] by way of
  the ~ilc[defun-sk] event.  However, proof support for quantification is
  quite limited.  Therefore, we recommend using recursion in place of
  ~c[defun-sk] when possible (following common ACL2 practice).~/

  For example, the notion ``every member of ~c[x] has property ~c[p]'' can be
  defined either with recursion or explicit quantification, but proofs
  may be simpler when recursion is used.  We illustrate this point
  with two proofs of the same informal claim, one of which uses
  recursion which the other uses explicit quantification.  Notice that
  with recursion, the proof goes through fully automatically; but this
  is far from true with explicit quantification (especially notable is
  the ugly hint).

  The informal claim for our examples is:  If every member ~c[a] of each
  of two lists satisfies the predicate ~c[(p a)], then this holds of their
  ~ilc[append]; and, conversely.

  ~l[quantifiers-using-recursion] for a solution to this example
  using recursion.

  ~l[quantifiers-using-defun-sk] for a solution to this example
  using ~ilc[defun-sk].  Also ~l[quantifiers-using-defun-sk-extended]
  for an elaboration on that solution.")

(deflabel quantifiers-using-recursion
  :doc
  ":Doc-Section Quantifiers

  recursion for implementing quantification~/

  The following example illustrates the use of recursion as a means of
  avoiding proof difficulties that can arise from the use of explicit
  quantification (via ~ilc[defun-sk]).  ~l[quantifiers] for more about
  the context of this example.~/
  ~bv[]
  (in-package \"ACL2\")

  ; We prove that if every member A of each of two lists satisfies the
  ; predicate (P A), then this holds of their append; and, conversely.

  ; Here is a solution using recursively-defined functions.

  (defstub p (x) t)

  (defun all-p (x)
    (if (atom x)
        t
      (and (p (car x))
           (all-p (cdr x)))))

  (defthm all-p-append
    (equal (all-p (append x1 x2))
           (and (all-p x1) (all-p x2))))
  ~ev[]")

(deflabel quantifiers-using-defun-sk
  :doc
  ":Doc-Section Quantifiers

  quantification example~/

  ~l[quantifiers] for the context of this example.  It should be
  compared to a corresponding example in which a simpler proof is
  attained by using recursion in place of explicit quantification;
  ~pl[quantifiers-using-recursion].~/
  ~bv[]
  (in-package \"ACL2\")

  ; We prove that if every member A of each of two lists satisfies the
  ; predicate (P A), then this holds of their append; and, conversely.

  ; Here is a solution using explicit quantification.

  (defstub p (x) t)

  (defun-sk forall-p (x)
    (forall a (implies (member a x)
                       (p a))))

  (defthm member-append
    (iff (member a (append x1 x2))
         (or (member a x1) (member a x2))))

  (defthm forall-p-append
    (equal (forall-p (append x1 x2))
           (and (forall-p x1) (forall-p x2)))
    :hints ((\"Goal\" ; ``should'' disable forall-p-necc, but no need
             :use
             ((:instance forall-p-necc
                         (x (append x1 x2))
                         (a (forall-p-witness x1)))
              (:instance forall-p-necc
                         (x (append x1 x2))
                         (a (forall-p-witness x2)))
              (:instance forall-p-necc
                         (x x1)
                         (a (forall-p-witness (append x1 x2))))
              (:instance forall-p-necc
                         (x x2)
                         (a (forall-p-witness (append x1 x2))))))))
  ~ev[]

  Also ~pl[quantifiers-using-defun-sk-extended] for an
  elaboration on this example.") 

(deflabel quantifiers-using-defun-sk-extended
  :doc
  ":Doc-Section Quantifiers

  quantification example with details~/

  ~l[quantifiers-using-defun-sk] for the context of this example.~/
  ~bv[]
  (in-package \"ACL2\")

  ; We prove that if every member A of each of two lists satisfies the
  ; predicate (P A), then this holds of their append; and, conversely.

  ; Here is a solution using explicit quantification.

  (defstub p (x) t)

  (defun-sk forall-p (x)
    (forall a (implies (member a x)
                       (p a))))

  ; The defun-sk above introduces the following axioms.  The idea is that
  ; (FORALL-P-WITNESS X) picks a counterexample to (forall-p x) if there is one.

  #|
  (DEFUN FORALL-P (X)
    (LET ((A (FORALL-P-WITNESS X)))
         (IMPLIES (MEMBER A X) (P A))))

  (DEFTHM FORALL-P-NECC
    (IMPLIES (NOT (IMPLIES (MEMBER A X) (P A)))
             (NOT (FORALL-P X)))
    :HINTS ((\"Goal\" :USE FORALL-P-WITNESS)))
  |#

  ; The following lemma seems critical.

  (defthm member-append
    (iff (member a (append x1 x2))
         (or (member a x1) (member a x2))))

  ; The proof of forall-p-append seems to go out to lunch, so we break into
  ; directions as shown below.

  (defthm forall-p-append-forward
    (implies (forall-p (append x1 x2))
             (and (forall-p x1) (forall-p x2)))
    :hints ((\"Goal\" ; ``should'' disable forall-p-necc, but no need
             :use
             ((:instance forall-p-necc
                         (x (append x1 x2))
                         (a (forall-p-witness x1)))
              (:instance forall-p-necc
                         (x (append x1 x2))
                         (a (forall-p-witness x2)))))))

  (defthm forall-p-append-reverse
    (implies (and (forall-p x1) (forall-p x2))
             (forall-p (append x1 x2)))
    :hints ((\"Goal\"
             :use
             ((:instance forall-p-necc
                         (x x1)
                         (a (forall-p-witness (append x1 x2))))
              (:instance forall-p-necc
                         (x x2)
                         (a (forall-p-witness (append x1 x2))))))))

  (defthm forall-p-append
    (equal (forall-p (append x1 x2))
           (and (forall-p x1) (forall-p x2)))
    :hints ((\"Goal\" :use (forall-p-append-forward
                          forall-p-append-reverse))))

  ~ev[]")

; Here is the defstobj event.

; We start with the problem of finding the arguments to the defstobj event.
; The form looks likes 

; (defstobj name ... field-descri ... 
;           :renaming alist
;           :doc string)
;           :inline flag)

; where the :renaming, :doc, and :inline keyword arguments are
; optional.  This syntax is not supported by macros because you can't
; have an &REST arg and a &KEYS arg without all the arguments being in
; the keyword style.  So we use &REST and implement the new style of
; argument recovery.

; Once we have partitioned the args for defstobj, we'll have recovered
; the field-descriptors, a renaming alist, and a doc string.  Our next
; step is to check that the renaming alist is of the correct form.

(defun doublet-style-symbol-to-symbol-alistp (x)
  (cond ((atom x) (equal x nil))
        (t (and (consp (car x))
                (symbolp (caar x))
                (consp (cdar x))
                (symbolp (cadar x))
                (null (cddar x))
                (doublet-style-symbol-to-symbol-alistp (cdr x))))))

; Then, we can use the function defstobj-fnname to map the default
; symbols in the defstobj to the function names the user wants us to
; use.  (It is defined elsewhere because it is needed by translate.)

(defun chk-legal-defstobj-name (name state)
  (cond ((eq name 'state)
         (er soft (cons 'defstobj name)
             "STATE is an illegal name for a user-declared ~
              single-threaded object."))
        ((legal-variablep name)
         (value nil))
        (t
         (er soft (cons 'defstobj name)
             "The symbol ~x0 may not be declared as a single-threaded object ~
              name because it is not a legal variable name."
             name))))

(defun chk-unrestricted-guards-for-user-fns (names wrld ctx state)
  (cond
   ((null names) (value nil))
   ((or (acl2-system-namep (car names) wrld)
        (equal (guard (car names) nil wrld) *t*))
    (chk-unrestricted-guards-for-user-fns (cdr names) wrld ctx state))
   (t (er soft ctx
          "The guard for ~x0 is ~p1.  But in order to use ~x0 in the ~
           type-specification of a single-threaded object it must ~
           have a guard of T."
          (car names)
          (untranslate (guard (car names) nil wrld) t wrld)))))

(defconst *expt2-28* (expt 2 28))

(defun chk-stobj-field-descriptor (name field-descriptor ctx wrld state)

; See the comment just before chk-acceptable-defstobj1 for an
; explanation of our handling of Common Lisp compliance.

   (cond
    ((symbolp field-descriptor) (value nil))
    (t
     (er-progn
      (if (and (consp field-descriptor)
               (symbolp (car field-descriptor))
               (keyword-value-listp (cdr field-descriptor))
               (member-equal (length field-descriptor) '(1 3 5 7))
               (let ((keys (odds field-descriptor)))
                 (and (no-duplicatesp keys)
                      (subsetp-eq keys '(:type :initially :resizable)))))
          (value nil)
          (er soft ctx
              "The field descriptors of a single-threaded object ~
               definition must be a symbolic field-name or a list of ~
               the form (field-name :type type :initially val), where ~
               field-name is a symbol.  The :type and :initially ~
               keyword assignments are optional and their order is ~
               irrelevant.  The purported descriptor ~x0 for a field ~
               in ~x1 is not of this form."
              field-descriptor
              name))
      (let ((field (car field-descriptor))
            (type (if (assoc-keyword :type (cdr field-descriptor))
                      (cadr (assoc-keyword :type (cdr field-descriptor)))
                    t))
            (init (if (assoc-keyword :initially (cdr field-descriptor))
                      (cadr (assoc-keyword :initially (cdr field-descriptor)))
                    nil))
            (resizable (if (assoc-keyword :resizable (cdr field-descriptor))
                           (cadr (assoc-keyword :resizable (cdr field-descriptor)))
                         nil)))
        (cond
         ((and resizable (not (eq resizable t)))
          (er soft ctx
              "The :resizable value in the ~x0 field of ~x1 is ~
               illegal:  ~x2.  The legal values are t and nil."
              field name resizable))
         ((and (consp type)
               (eq (car type) 'array))
          (cond
           ((not (and (true-listp type)
                      (equal (length type) 3)
                      (true-listp (caddr type))
                      (equal (length (caddr type)) 1)))
            (er soft ctx
                "When a field descriptor specifies an ARRAY :type, ~
                 the type must be of the form (ARRAY etype (n)).  ~
                 Note that we only support single-dimensional arrays. ~
                  The purported ARRAY :type ~x0 for the ~x1 field of ~
                 ~x2 is not of this form."
                type field name))
          (t (let* ((etype (cadr type))
                    (etype-term (translate-declaration-to-guard etype 'x))
                    (n (car (caddr type))))
               (cond
                ((null etype-term)
                 (er soft ctx
                     "The element type specified for the ~x0 field of ~
                      ~x1, namely ~x0, is not recognized by ACL2 as a ~
                      type-spec.  See :DOC type-spec."
                     field name type))
                ((not (and (integerp n)
                           (<= 0 n)))
                 (er soft ctx
                     "Array dimensions must be non-negative integers.  ~
                      The :type ~x0 for the ~x1 field of ~x2 is thus ~
                      illegal."
                     type field name))
                (t
                 (er-let*
                   ((pair (simple-translate-and-eval etype-term
                                                     (list (cons 'x init))
                                                     nil
                                                     (msg
                                                      "The type ~x0"
                                                      etype-term)
                                                     ctx
                                                     wrld
                                                     state)))

; pair is (tterm . val), where tterm is a term and val is its value
; under x<-init.

                   (er-progn
                    (chk-common-lisp-compliant-subfunctions
                     nil (list field) (list (car pair))
                     wrld "auxiliary function" ctx state)
                    (chk-unrestricted-guards-for-user-fns
                     (all-fnnames (car pair))
                     wrld ctx state)
                    (cond
                     ((not (cdr pair))
                      (er soft ctx
                          "The value specified by the :initially ~
                           keyword, namely ~x0, fails to satisfy the ~
                           declared type ~x1 in the array ~
                           specification for the ~x2 field of ~x3."
                          init etype field name))
                     (t (value nil)))))))))))
         ((assoc-keyword :resizable (cdr field-descriptor))
          (er soft ctx
              "The :resizable keyword is only legal for array types, hence is ~
               illegal for the ~x0 field of ~x1."
              field name))
         (t (let ((type-term (translate-declaration-to-guard type 'x)))
              (cond
               ((null type-term)
                (er soft ctx
                    "The :type specified for the ~x0 field of ~x1, ~
                     namely ~x2, is not recognized by ACL2 as a ~
                     type-spec.  See :DOC type-spec."
                    field name type))
               (t
                (er-let*
                  ((pair (simple-translate-and-eval type-term
                                                    (list (cons 'x init))
                                                    nil
                                                    (msg
                                                     "The type ~x0"
                                                     type-term)
                                                    ctx
                                                    wrld
                                                    state)))

; pair is (tterm . val), where tterm is a term and val is its value
; under x<-init.

                  (er-progn
                   (chk-common-lisp-compliant-subfunctions
                    nil (list field) (list (car pair))
                    wrld "body" ctx state)
                   (chk-unrestricted-guards-for-user-fns
                     (all-fnnames (car pair))
                     wrld ctx state)
                   (cond
                    ((not (cdr pair))
                     (er soft ctx
                         "The value specified by the :initially ~
                          keyword, namely ~x0, fails to satisfy the ~
                          declared :type ~x1 for the ~x2 field of ~x3."
                         init type field name))
                    (t (value nil)))))))))))))))

(defun chk-acceptable-defstobj-renaming
  (name field-descriptors renaming ctx state default-names)

; We collect up all the default names and then check that the domain
; of renaming contains no duplicates and is a subset of the default
; names.  We already know that field-descriptors is well-formed and
; that renaming is a doublet-style symbol-to-symbol alist.

  (cond
   ((endp field-descriptors)
    (let ((default-names (list* name
                                (defstobj-fnname name :recognizer :top nil)
                                (defstobj-fnname name :creator :top nil)
                                (reverse default-names)))
          (domain (strip-cars renaming)))
      (cond
       ((null renaming)

; In this case, the default-names are the names the user intends us to use.

        (cond
         ((not (no-duplicatesp default-names))
          (er soft ctx
              "The field descriptors are illegal because they require ~
               the use of the same name for two different functions.  ~
               The duplicated name~#0~[ is~/s are~] ~&0.  You must ~
               change the component names so that no conflict occurs. ~
                You may then wish to use the :RENAMING option to ~
               introduce your own names for these functions.  See ~
               :DOC defstobj."
              (duplicates default-names)))
         (t (value nil))))
       ((not (no-duplicatesp default-names))
        (er soft ctx
            "The field descriptors are illegal because they require ~
             the use of the same default name for two different ~
             functions.  The duplicated default name~#0~[ is~/s are~] ~
             ~&0.  You must change the component names so that no ~
             conflict occurs.  Only then may you use the :RENAMING ~
             option to rename the default names."
            (duplicates default-names)))
       ((not (no-duplicatesp domain))
        (er soft ctx
            "No two entries in the :RENAMING alist may mention the ~
             same target symbol.  Your alist, ~x0, contains ~
             duplications in its domain."
            renaming))
       ((not (subsetp domain default-names))
        (er soft ctx
            "Your :RENAMING alist, ~x0, mentions ~#1~[a function ~
             symbol~/function symbols~] in its domain which ~
             ~#1~[is~/are~] not among the default symbols to be ~
             renamed.  The offending symbol~#1~[ is~/s are~] ~&1.  ~
             The default defstobj names for this event are ~&2."
            renaming
            (set-difference-equal domain default-names)
            default-names))
       (t (value nil)))))
   (t (let* ((field (if (atom (car field-descriptors))
                        (car field-descriptors)
                      (car (car field-descriptors))))
             (type (if (consp (car field-descriptors))
                       (or (cadr (assoc-keyword :type
                                                (cdr (car field-descriptors))))
                           t)
                     t))
             (key2 (if (and (consp type)
                            (eq (car type) 'array))
                       :array
                     :non-array)))
        (chk-acceptable-defstobj-renaming
         name (cdr field-descriptors) renaming ctx state
         (list* (defstobj-fnname field :updater key2 nil)
                (defstobj-fnname field :accessor key2 nil)
                (defstobj-fnname field :recognizer key2 nil)
                (cond ((eq key2 :array)
                       (list* (defstobj-fnname field :length key2 nil)
                              (defstobj-fnname field :resize key2 nil)
                              default-names))
                      (t default-names))))))))

; The functions introduced by defstobj are all defined with
; :VERIFY-GUARDS T.  This means we must ensure that their guards and
; bodies are compliant.  Most of this stuff is mechanically generated
; by us and is guaranteed to be compliant.  But there is a way that a
; user defined function can sneak in.  The user might use a type-spec
; such as (satisfies foo), where foo is a user defined function.

; To discuss the guard issue, we name the functions introduced by
; defstobj, following the convention used in the comment in
; defstobj-template.  The recognizer for the stobj itself will be
; called namep, and the creator will be called create-name.  For each
; field, the following names are introduced: recog-name - recognizer
; for the field value; accessor-name - accessor for the field;
; updater-name - updater for the field; length-name - length of array
; field; resize-name - resizing function for array field.

; We are interested in determining the conditions we must check to
; ensure that each of these functions is Common Lisp compliant.  Both
; the guard and the body of each function must be compliant.
; Inspection of defstobj-axiomatic-defs reveals the following.

; Namep is defined in terms of primitives and the recog-names.  The
; guard for namep is T.  The body of namep is always compliant, if the
; recog-names are compliant and have guards of T.

; Create-name is a constant with a guard of T.  Its body is always
; compliant.

; Recog-name has a guard of T.  The body of recog-name is interesting
; from the guard verification perspective, because it may contain
; translated type-spec such as (satisfies foo) and so we must check
; that foo is compliant.  We must also check that the guard of foo is
; T, because the guard of recog-name is T and we might call foo on
; anything.

; Accessor-name is not interesting:  its guard is namep and its body is
; primitive.  We will have checked that namep is compliant.

; Updater-name is not interesting:  its guard may involve translated
; type-specs and will involve namep, but we will have checked their
; compliance already.

; Length-name and resize-name have guards that are calls of namep, and
; their bodies are known to satisfy their guards.

; So it all boils down to checking the compliance of the body of
; recog-name, for each component.  Note that we must check both that
; the type-spec only involves compliant functions and that every
; non-system function used has a guard of T.

(defun defconst-name (name)
  (intern-in-package-of-symbol
   (concatenate 'string "*" (symbol-name name) "*")
   name))

(defun chk-acceptable-defstobj1
  (name field-descriptors ftemps renaming ctx wrld state names const-names)

; We check whether it is legal to define name as a single-threaded
; object with the description given in field-descriptors.  We know
; name is a legal (and new) stobj name and we know that renaming is an
; symbol to symbol doublet-style alist.  But we know nothing else.  We
; either signal an error or return the world in which the event is to
; be processed (thus implementing redefinitions).  Names is, in
; general, the actual set of names that the defstobj event will
; introduce.  That is, it contains the images of the default names
; under the renaming alist.  We accumulate the actual names into it as
; we go and check that it contains no duplicates at the termination of
; this function.  All of the names in names are to be defined as
; functions with :VERIFY-GUARDS T.  See the comment above about
; Common Lisp compliance.

  (cond
   ((endp ftemps)
    (let* ((recog-name (defstobj-fnname name :recognizer :top renaming))
           (creator-name (defstobj-fnname name :creator :top renaming))
           (names (list* recog-name creator-name names)))
      (er-progn
       (chk-all-but-new-name recog-name ctx 'function wrld state)
       (chk-all-but-new-name creator-name ctx 'function wrld state)
       (chk-acceptable-defstobj-renaming name field-descriptors renaming
                                         ctx state nil)

; Note: We insist that all the names be new.  In addition to the
; obvious necessity for something like this, we note that this does
; not permit us to have redundantly defined any of these names.  For
; example, the user might have already defined a field recognizer,
; PCP, that is identically defined to what we will lay down.  But we
; do not allow that.  We basically insist that we have control over
; every one of these names.

       (chk-just-new-names names 'function nil ctx wrld state)
       (chk-just-new-names const-names 'const nil ctx wrld state))))
   (t

; An element of field-descriptors (i.e., of ftemps) is either a
; symbolic field name, field, or else of the form (field :type type
; :initially val), where either or both of the keyword fields can be
; omitted.  Val must be an evg, i.e., an unquoted constant like t,
; nil, 0 or undef (the latter meaning the symbol 'undef).  :Type
; defaults to the unrestricted type t and :initially defaults to nil.
; Type is either a primitive type, as recognized by
; translate-declaration-to-guard, or else is of the form (array ptype
; (n)) where ptype is a primitive type and n is an positive integer
; constant.

    (er-progn
     (chk-stobj-field-descriptor name (car ftemps) ctx wrld state)
     (let* ((field (if (atom (car ftemps))
                       (car ftemps)
                     (car (car ftemps))))
            (type (if (consp (car ftemps))
                      (or (cadr (assoc-keyword :type
                                               (cdr (car ftemps))))
                          t)
                    t))
            (key2 (if (and (consp type)
                           (eq (car type) 'array))
                      :array
                    :non-array))
            (fieldp-name (defstobj-fnname field :recognizer key2 renaming))
            (accessor-name (defstobj-fnname field :accessor key2 renaming))
            (accessor-const-name (defconst-name accessor-name))
            (updater-name (defstobj-fnname field :updater key2 renaming))
            (length-name (defstobj-fnname field :length key2 renaming))
            (resize-name (defstobj-fnname field :resize key2 renaming)))
       (er-progn
        (chk-all-but-new-name fieldp-name ctx 'function wrld state)
        (chk-all-but-new-name accessor-name ctx 'function wrld state)
        (chk-all-but-new-name updater-name ctx 'function wrld state)
        (chk-all-but-new-name accessor-const-name ctx 'const wrld state)
        (if (eq key2 :array)
            (er-progn (chk-all-but-new-name length-name ctx 'function wrld state)
                      (chk-all-but-new-name resize-name ctx 'function wrld state))
          (value nil))
        (chk-acceptable-defstobj1 name field-descriptors (cdr ftemps)
                                  renaming ctx wrld state
                                  (list* fieldp-name
                                         accessor-name
                                         updater-name
                                         (if (eq key2 :array)
                                             (list* length-name
                                                    resize-name
                                                    names)
                                           names))
                                  (cons accessor-const-name
                                        const-names))))))))

(defun the-live-var (name)

; If the user declares a single-threaded object named $S then we will
; use *the-live-$s* as the Lisp parameter holding the live object
; itself.  One might wonder why we don't choose to name this object
; $s?  Perhaps we could, since starting with Version  2.6 we no longer
; get the symbol-value of *the-live-$s* except at the top level,
; because of local stobjs.  Below we explain our earlier thinking.

; Historical Plaque for Why the Live Var for $S Is Not $S

; [Otherwise] Consider how hard it would then be to define the raw defs
; (below).  $S is the formal parameter, and naturally so since we want
; translate to enforce the rules on single-threadedness.  The raw code
; has to check whether the actual is the live object.  We could hardly
; write (eq $S $S).

  (packn-pos (list "*THE-LIVE-" name "*") name))

(defconst *defstobj-keywords*
  '(:renaming :doc :inline))

(defun defstobj-redundancy-bundle (name args storep)

; See redundant-defstobjp to see how this is used.

; The treatment of erp below is justified as follows.  When this function is
; called to create the redundancy bundle for an admitted defstobj, erp is
; guaranteed to be nil (storep = t).  If this function is used to compute a
; redundancy bundle for a new purported but ill-formed defstobj, the bundle
; will contain the symbol 'error in the field-descriptors slot, which will
; cause it not to match any correct redundancy bundle.  Thus, the purported
; defstobj will not be considered redundant and the error will be detected by
; the admissions process.

  (mv-let
   (erp field-descriptors key-alist)
   (partition-rest-and-keyword-args args *defstobj-keywords*)
   (cons (if erp
             (if storep
                 (er hard 'defstobj-redundancy-bundle
                     "Implementation error: ~x0 returned an error when ~
                      storing the redundancy-bundle for defstobj.  Please ~
                      contact the implementors."
                     `(defstobj-redundancy-bundle ,name ,args ,storep))
               'error)
           field-descriptors)
         (cdr (assoc-eq :renaming key-alist)))))

(defun redundant-defstobjp (name args wrld)

; Note: At one time we stored the defstobj template on the property
; list of a defstobj name and we computed the new template from args
; and compared the two templates to identify redundancy.  To make this
; possible without causing runtime errors we had to check, here, that
; the arguments -- which have not yet been checked for well-formedness
; -- were at least of the right basic shape, e.g., that the renaming
; is a doublet-style-symbol-to-symbol-alistp and that each
; field-descriptor is either a symbol or a true-list of length 1, 3,
; or 5 with :type and :initially fields.  But this idea suffered the
; unfortunate feature that an illegal defstobj event could be
; considered redundant.  For example, if the illegal event had a
; renaming that included an unnecessary function symbol in its domain,
; that error was not caught.  The bad renaming produced a good
; template and if a correct version of that defstobj had previously
; been executed, the bad one was recognized as redundant.
; Unfortunately, if one were to execute the bad one first, an error
; would result.
       
; So we have changed this function to be extremely simple.  

  (and (getprop name 'stobj nil 'current-acl2-world wrld)
       (equal (getprop name 'redundancy-bundle nil
                       'current-acl2-world wrld)
              (defstobj-redundancy-bundle name args nil))))

(defun chk-acceptable-defstobj (name args ctx wrld state)

; We check that (defstobj name . args) is well-formed and either
; signal an error or return nil.

  (mv-let
   (erp field-descriptors key-alist)
   (partition-rest-and-keyword-args args *defstobj-keywords*)
   (cond
    (erp
     (er soft ctx
         "The keyword arguments to the DEFSTOBJ event must appear ~
          after all field descriptors.  The allowed keyword ~
          arguments are ~&0, and these may not be duplicated, and ~
          must be followed by the corresponding value of the keyword ~
          argument.  Thus, ~x1 is ill-formed."
         *defstobj-keywords*
         (list* 'defstobj name args)))
    (t
     (let ((renaming (cdr (assoc-eq :renaming key-alist)))
           (doc (cdr (assoc-eq :doc key-alist)))
           (inline (cdr (assoc-eq :inline key-alist))))
       (cond
        ((redundant-defstobjp name args wrld)
         (value 'redundant))
        ((not (booleanp inline))
         (er soft ctx
             "DEFSTOBJ requires the :INLINE keyword argument to have a Boolean ~
              value.  See :DOC stobj."
             (list* 'defstobj name args)))
        (t
         (er-progn

; The defstobj name itself is not subject to renaming.  So we check it
; before we even bother to check the well-formedness of the renaming alist.

          (chk-all-but-new-name name ctx 'stobj wrld state)
          (cond ((or (eq name 'I)
                     (eq name 'V))
                 (er soft ctx
                     "DEFSTOBJ does not allow single-threaded objects with ~
                      the names I or V because those symbols are used as ~
                      formals, along with the new stobj name itself, in ~
                      ``primitive'' stobj functions that will be ~
                      defined."))
                (t (value nil)))
          (chk-legal-defstobj-name name state)
          (cond ((not (doublet-style-symbol-to-symbol-alistp renaming))
                 (er soft ctx
                     "The :RENAMING argument to DEFSTOBJ must be an ~
                      alist containing elements of the form (sym ~
                      sym), where each element of such a doublet is a ~
                      symbol. Your argument, ~x0, is thus illegal."
                     renaming))
                (t (value nil)))

; We use translate-doc here just to check the string.  We throw away
; the section-symbol and citations returned.  We'll repeat this later.

          (translate-doc name doc ctx state)
          (er-let*
            ((wrld1 (chk-just-new-name name 'stobj nil ctx wrld state))
             (wrld2 (chk-just-new-name (the-live-var name) 'stobj-live-var
                                       nil ctx wrld1 state)))
            (chk-acceptable-defstobj1 name field-descriptors
                                      field-descriptors renaming
                                      ctx wrld2 state nil nil))))))))))

; Essay on Defstobj Definitions

; Consider the following defstobj:

  #|
  (defstobj $st
    (flag :type t :initially run)
    (pc   :type (integer 0 255) :initially 128)
    (mem  :type (array (integer 0 255) (256)) :initially 0)
    :renaming ((pc pcn)))
  |#

; If you call (defstobj-template '$st '((flag ...) ...)) you will get
; back a ``template'' which is sort of a normalized version of the
; event with the renaming applied and all the optional slots filled
; appropriately.  (See the definition of defstobj-template for details.)
; Let template be that template.

; To see the logical definitions generated by this defstobj event, invoke
;   (defstobj-axiomatic-defs '$st template)

; To see the raw lisp definitions generated, invoke
;   (defstobj-raw-defs '$st template)

; The *1* functions for the functions are all generated by oneifying
; the axiomatic defs.

; To see the deconsts generated, invoke
;   (defstobj-defconsts (strip-accessor-names (caddr template)) 0)

; It is important the guard conjectures for these functions be
; provable!  They are assumed by the admission process!  To prove
; the guards for the defstobj above, it helped to insert the following
; lemma after the defun of memp but before the definition of memi.

  #|
  (defthm memp-implies-true-listp
    (implies (memp x)
             (true-listp x)))
  |#

; Even without this lemma, the proof succeeded, though it took much
; longer and involved quite a few generalizations and inductions.

; If you change any of the functions, I recommend generating the axiomatic
; defs for a particular defstobj such as that above and proving the guards.

; Up through v2-7 we also believed that we ensured that the guards in the
; axiomatic defs are sufficient for the raw defs.  However, starting with v2-8,
; this became moot because of the following claim: the raw Lisp functions are
; only called on live stobjs (this change, and others involving :inline, were
; contributed by Rob Sumners).  We believe this claim because of the following
; argument.
;
;   a) The *1* function now has an additional requirement that not only does
;      guard checking pass, but also, all of the stobjs arguments passed in
;      must be the live stobjs in order to execute raw Common Lisp.
;   b) Due to the syntactic restrictions that ACL2 enforces, we know that the
;      direct correspondence between live stobjs and stobj arguments in the
;      raw Common Lisp functions will persist throughout evaluation.
;      -- This can be proven by induction over the sequence of function calls
;         in any evaluation.
;      -- The base case is covered by the binding of stobj parameters to
;         the global live stobj in the acl2-loop, or by the restrictions
;         placed upon with-local-stobj.
;      -- The induction step is proven by the signature requirements of
;         functions that access and/or update stobjs.

; A reasonable question is: Should the guard for resize-name be
; strengthened so as to disallow sizes of at least (1- (expt 2 28))?
; Probably there is no need for this.  Logically, there is no such
; restriction; it is OK for the implementation to insist on such a
; bound when actually executing.

; Now we introduce the idea of the "template" of a defstobj, which
; includes a normalized version of the field descriptors under the
; renaming.

(defun defstobj-fields-template (field-descriptors renaming)
  (cond
   ((endp field-descriptors) nil)
   (t
    (let* ((field (if (atom (car field-descriptors))
                      (car field-descriptors)
                    (car (car field-descriptors))))
           (type (if (consp (car field-descriptors))
                     (or (cadr (assoc-keyword :type
                                              (cdr (car field-descriptors))))
                         t)
                   t))
           (init (if (consp (car field-descriptors))
                     (cadr (assoc-keyword :initially
                                          (cdr (car field-descriptors))))
                   nil))
           (resizable (if (consp (car field-descriptors))
                          (cadr (assoc-keyword :resizable
                                               (cdr (car field-descriptors))))
                        nil))
           (key2 (if (and (consp type)
                          (eq (car type) 'array))
                     :array
                   :non-array))
           (fieldp-name (defstobj-fnname field :recognizer key2 renaming))
           (accessor-name (defstobj-fnname field :accessor key2 renaming))
           (updater-name (defstobj-fnname field :updater key2 renaming))
           (resize-name (defstobj-fnname field :resize key2 renaming))
           (length-name (defstobj-fnname field :length key2 renaming)))
      (cons (list fieldp-name
                  type
                  init
                  accessor-name
                  updater-name
                  length-name
                  resize-name
                  resizable)
            (defstobj-fields-template (cdr field-descriptors) renaming))))))

(defun defstobj-doc (args)

; We retrieve the doc string, if any, from (defstobj name . args).

  (mv-let (erp field-descriptors key-alist)
          (partition-rest-and-keyword-args args *defstobj-keywords*)
          (declare (ignore field-descriptors))
          (assert$ (not erp)
                   (cdr (assoc-eq :doc key-alist)))))

(defun defstobj-template (name args)

; We unpack the args to get the renamed field descriptors.  We return
; a list of the form (namep create-name fields doc inline), where:
; namep is the name of the recognizer for the single-threaded object;
; create-name is the name of the constructor for the stobj; fields is
; a list corresponding to the field descriptors, but normalized with
; respect to the renaming, types, etc.; doc is the doc string, or
; nil if no doc string is supplied; and inline is t if :inline t was
; specified in the defstobj event, else nil.  A field in fields is of
; the form (recog-name type init accessor-name updater-name
; length-name resize-name resizable).  The last three fields are nil
; unless type has the form (ARRAY ptype (n)), in which case ptype is a
; primitive type and n is a positive integer.  Init is the evg of a
; constant term, i.e., should be quoted to be a treated as a term.
; Doc is the value of the :doc keyword arg in args.

  (mv-let
   (erp field-descriptors key-alist)
   (partition-rest-and-keyword-args args *defstobj-keywords*)
   (cond
    (erp

; If the defstobj has been admitted, this won't happen.

     (er hard 'defstobj
         "The keyword arguments to the DEFSTOBJ event must appear ~
          after all field descriptors.  The allowed keyword ~
          arguments are ~&0, and these may not be duplicated.  Thus, ~
          ~x1 is ill-formed."
         *defstobj-keywords*
         (list* 'defstobj name args)))
    (t
     (let ((renaming (cdr (assoc-eq :renaming key-alist)))
           (doc (cdr (assoc-eq :doc key-alist)))
           (inline (cdr (assoc-eq :inline key-alist))))
       (list (defstobj-fnname name :recognizer :top renaming)
             (defstobj-fnname name :creator :top renaming)
             (defstobj-fields-template field-descriptors renaming)
             doc
             inline))))))

(defun defstobj-component-recognizer-calls (ftemps n var ans)

; Warning:  See the guard remarks in the Essay on Defstobj Definitions.

; Given a list of field templates, e.g., ((regp ...) (pcp ...) ...),
; where n is one less than the number of fields and var is some
; symbol, v, we return ((regp (nth 0 v)) (pcp (nth 1 v)) ...).  Except,
; if field represents a non-resizable array then we also include a
; corresponding length statement in the list.

  (cond ((endp ftemps)
         (reverse ans))
        (t (defstobj-component-recognizer-calls
             (cdr ftemps)
             (+ n 1)
             var
             (let* ((type (cadr (car ftemps)))
                    (nonresizable-ar (and (consp type)
                                          (eq (car type) 'array)
                                          (not (nth 7 (car ftemps)))))
                    (pred-stmt `(,(car (car ftemps)) (nth ,n ,var))))
               (if nonresizable-ar
                   (list* `(equal (len (nth ,n ,var)) ,(car (caddr type)))
                          pred-stmt
                          ans)
                 (cons pred-stmt ans)))))))

(defun defstobj-component-recognizer-axiomatic-defs (name template ftemps)

; Warning:  See the guard remarks in the Essay on Defstobj Definitions.

; We return a list of defs (see defstobj-axiomatic-defs) for all the
; recognizers for the single-threaded resource named name with the
; given template.  The answer contains the top-level recognizer and
; creator for the object, as well as the definitions of all component
; recognizers.  The answer contains defs for auxiliary functions used
; in array component recognizers.  The defs are listed in an order
; suitable for processing (components first, then top-level).

  (cond
   ((endp ftemps)
    (let* ((recog-name (car template))
           (field-templates (caddr template))
           (n (length field-templates)))

; Rockwell Addition: See comment below.

; Note: The recognizer for a stobj must be Boolean!  That is why we
; conclude the AND below with a final T.  The individual field
; recognizers need not be Boolean and sometimes are not!  For example,
; a field with :TYPE (MEMBER e1 ... ek) won't be Boolean, nor with
; certain :TYPE (OR ...) involving MEMBER.  The reason we want the
; stobj recognizer to be Boolean is so that we can replace it by T in
; guard conjectures for functions that have been translated with the
; stobj syntactic restrictions.  See optimize-stobj-recognizers.

      (list `(,recog-name (,name)
                          (declare (xargs :guard t
                                          :verify-guards t))
                          (and (true-listp ,name)
                               (= (length ,name) ,n)
                               ,@(defstobj-component-recognizer-calls
                                   field-templates 0 name nil)
                               t)))))
   (t
    (let ((recog-name (nth 0 (car ftemps)))
          (type (nth 1 (car ftemps))))

; Below we simply append the def or defs for this field to those for
; the rest.  We get two defs for each array field and one def for each
; of the others.

      (cons (cond
             ((and (consp type)
                   (eq (car type) 'array))
              (let ((etype (cadr type)))
                `(,recog-name (x)
                              (declare (xargs :guard t
                                              :verify-guards t))
                              (if (atom x)
                                  (equal x nil)
                                  (and ,(translate-declaration-to-guard
                                         etype '(car x))
                                       (,recog-name (cdr x)))))))
             (t (let ((type-term (translate-declaration-to-guard type 'x)))
                  
; We may not use x in the type-term and so have to declare it ignored.

                  (cond
                   ((member-eq 'x (all-vars type-term))
                    `(,recog-name (x)
                                  (declare (xargs :guard t
                                                  :verify-guards t))
                                  ,type-term))
                   (t 
                    `(,recog-name (x)
                                  (declare (xargs :guard t
                                                  :verify-guards t)
                                           (ignore x))
                                  ,type-term))))))
            (defstobj-component-recognizer-axiomatic-defs 
              name template (cdr ftemps)))))))

(defun defstobj-field-fns-axiomatic-defs (top-recog var n ftemps)

; Warning:  See the guard remarks in the Essay on Defstobj Definitions.

; We return a list of defs (see defstobj-axiomatic-defs) for all the accessors,
; updaters, and optionally, array resizing and length, of a single-threaded
; resource.

  (cond
   ((endp ftemps)
    nil)
   (t (let* ((field-template (car ftemps))
             (type (nth 1 field-template))
             (init (nth 2 field-template))
             (arrayp (and (consp type) (eq (car type) 'array)))
             (type-term (and (not arrayp)
                             (translate-declaration-to-guard type 'v)))
             (array-etype (and arrayp (cadr type)))
             (array-etype-term
              (and arrayp
                   (translate-declaration-to-guard array-etype 'v)))
             (array-length (and arrayp (car (caddr type))))
             (accessor-name (nth 3 field-template))
             (updater-name (nth 4 field-template))
             (length-name (nth 5 field-template))
             (resize-name (nth 6 field-template))
             (resizable (nth 7 field-template)))
        (cond
         (arrayp
          (append 
           `((,length-name (,var)
                           (declare (xargs :guard (,top-recog ,var)
                                           :verify-guards t)
                                    ,@(and (not resizable)
                                           `((ignore ,var))))
                           ,(if resizable
                                `(len (nth ,n ,var))
                              `,array-length))
             (,resize-name
              (i ,var)
              (declare (xargs :guard (,top-recog ,var)
                              :verify-guards t)
                       ,@(and (not resizable)
                              '((ignore i))))
              ,(if resizable
                   `(update-nth ,n
                                (resize-list (nth ,n ,var) i ,init)
                                ,var)
                 `(prog2$ (hard-error
                           ',resize-name
                           "The array field corresponding to accessor ~x0 of ~
                             stobj ~x1 was not declared :resizable t.  ~
                             Therefore, it is illegal to resize this array."
                           (list (cons #\0 ',accessor-name)
                                 (cons #\1 ',var)))
                          ,var)))
              (,accessor-name (i ,var)
                              (declare (xargs :guard
                                              (and (,top-recog ,var)
                                                   (integerp i)
                                                   (<= 0 i)
                                                   (< i (,length-name ,var)))
                                              :verify-guards t))
                              (nth i (nth ,n ,var)))
              (,updater-name (i v ,var)
                             (declare (xargs :guard
                                             (and (,top-recog ,var)
                                                  (integerp i)
                                                  (<= 0 i)
                                                  (< i (,length-name ,var))
                                                  ,@(if (equal array-etype-term
                                                               t)
                                                        nil
                                                      (list array-etype-term)))
                                             :verify-guards t))
                             (update-nth-array ,n i v ,var)))
           (defstobj-field-fns-axiomatic-defs
             top-recog var (+ n 1) (cdr ftemps))))
         (t
          (append 
           `((,accessor-name (,var)
                             (declare (xargs :guard (,top-recog ,var)
                                             :verify-guards t))
                             (nth ,n ,var))
             (,updater-name (v ,var)
                            (declare (xargs :guard
                                            ,(if (equal type-term t)
                                                 `(,top-recog ,var)
                                               `(and ,type-term
                                                     (,top-recog ,var)))
                                            :verify-guards t))
                            (update-nth ,n v ,var)))
           (defstobj-field-fns-axiomatic-defs
             top-recog var (+ n 1) (cdr ftemps)))))))))

(defun defstobj-axiomatic-init-fields (ftemps)

; Keep this in sync with defstobj-raw-init-fields.

  (cond
   ((endp ftemps) nil)
   (t (let* ((field-template (car ftemps))
             (type (nth 1 field-template))
             (arrayp (and (consp type) (eq (car type) 'array)))
             (array-size (and arrayp (car (caddr type))))
             (init (nth 2 field-template)))
        (cond
         (arrayp
          (cons `(make-list ,array-size :initial-element ',init)
                (defstobj-axiomatic-init-fields (cdr ftemps))))
         (t ; whether the type is given or not is irrelevant
          (cons (kwote init)
                (defstobj-axiomatic-init-fields (cdr ftemps)))))))))

(defun defstobj-creator-fn (creator-name field-templates)

; This function generates the logic initialization code for the given stobj
; name.

  `(,creator-name
    ()
    (declare (xargs :guard t :verify-guards t))
    (list ,@(defstobj-axiomatic-init-fields field-templates))))

(defun defstobj-axiomatic-defs (name template)

; Warning:  See the guard remarks in the Essay on Defstobj Definitions.

; Template is the defstobj-template for name and args and thus
; corresponds to some (defstobj name . args) event.  We generate the
; #+acl2-loop-only defs for that event and return a list of defs.  For
; each def it is the case that (defun . def) is a legal defun, and
; they can executed in the order returned.

; These defs are processed to axiomatize the recognizer, accessor and
; updater functions for the single-threaded resource.  They are also
; oneified when we process the defstobj CLTL-COMMAND to define the *1*
; versions of the functions.  Finally, parts of them are re-used in
; raw lisp code when the code is applied to an object other than the
; live one.

; WARNING: If you change the formals of these generated axiomatic
; defs, be sure to change the formals of the corresponding raw defs.

; See the Essay on Defstobj Definitions

  (let ((field-templates (caddr template)))
    (append
     (defstobj-component-recognizer-axiomatic-defs name template
       field-templates)
     (cons
      (defstobj-creator-fn (cadr template) field-templates)
      (defstobj-field-fns-axiomatic-defs (car template) name 0
        field-templates)))))

(defun simple-array-type (array-etype dimensions)
  (declare (ignore dimensions))
  (cond
   ((member-eq array-etype '(* t))
    `(simple-vector *))
   (t `(simple-array ,array-etype (*)))))

#-acl2-loop-only
(defun stobj-copy-array-aref (a1 a2 i n)
  (declare (type (unsigned-byte 28) i n))

; Copy the first n elements of array a1 into array a2, starting with index i,
; and then return a2.  See also copy-array-svref and stobj-copy-array-fix-aref.

  (cond
   ((>= i n) a2)
   (t (setf (aref a2 i)
            (aref a1 i))
      (stobj-copy-array-aref a1 a2
                             (the (unsigned-byte 28) (1+ i))
                             (the (unsigned-byte 28) n)))))

#-acl2-loop-only
(defun stobj-copy-array-svref (a1 a2 i n)
  (declare (type (unsigned-byte 28) i n)
           (type simple-vector a1 a2))

; This is a variant of copy-array-aref for simple vectors a1 and a2.

  (cond
   ((>= i n) a2)
   (t (setf (svref a2 i)
            (svref a1 i))
      (stobj-copy-array-svref a1 a2
                              (the (unsigned-byte 28) (1+ i))
                              (the (unsigned-byte 28) n)))))

#-acl2-loop-only
(defun stobj-copy-array-fix-aref (a1 a2 i n)
  (declare (type (unsigned-byte 28) i n)
           (type (simple-array (signed-byte 29) (*)) a1 a2))

; This is a variant of copy-array-aref for arrays of fixnums a1 and a2.  We
; need this special version to avoid fixnum boxing in GCL during resizing.

  (cond
   ((>= i n) a2)
   (t (setf (aref a2 i)
            (aref a1 i))
      (stobj-copy-array-fix-aref a1 a2
                                 (the (unsigned-byte 28) (1+ i))
                                 (the (unsigned-byte 28) n)))))

(defmacro the-live-stobjp (name)

; With the introduction of local stobjs, we still rely on the
; symbol-value of (the-live-var name) to get the global value of a
; stobj, but we must also consider let-bound values.  Because of
; translate and oneify, we know that we are binding the live var as we
; go.  We could use `(arrayp ,name) below, but we stick to the eq test
; for now.

  `(eq ,name ,(the-live-var name)))

(defun array-etype-is-fixnum-type (array-etype)
  (declare (xargs :guard 
                  (implies (consp array-etype)
                           (true-listp array-etype))))
  (and (consp array-etype)
       (case (car array-etype)
             (integer
              (let* ((e1 (cadr array-etype))
                     (int1 (if (integerp e1)
                               e1
                             (and (consp e1)
                                  (integerp (car e1))
                                  (1- (car e1)))))
                     (e2 (caddr array-etype))
                     (int2 (if (integerp e2)
                               e2
                             (and (consp e2)
                                  (integerp (car e2))
                                  (1- (car e2))))))
                (and int1
                     int2
                     (>= int1 (- *expt2-28*))
                     (< int2 *expt2-28*))))
             (mod
              (and (integerp (cadr array-etype))
                   (< (cadr array-etype) 
                      *expt2-28*)))
             (unsigned-byte
              (and (integerp (cadr array-etype))
                   (<= (cadr array-etype) 
                       28)))
             (signed-byte
              (and (integerp (cadr array-etype))
                   (<= (cadr array-etype) 
                       29))))))

(defun defstobj-field-fns-raw-defs (var inline n ftemps)

; Warning:  See the guard remarks in the Essay on Defstobj Definitions.

  (cond
   ((endp ftemps) nil)
   (t
    (append
     (let* ((field-template (car ftemps))
            (type (nth 1 field-template))
            (init (nth 2 field-template))
            (arrayp (and (consp type) (eq (car type) 'array)))
            (array-etype (and arrayp (cadr type)))
            (simple-type (and arrayp
                              (simple-array-type array-etype (caddr type))))
            (array-length (and arrayp (car (caddr type))))
            (vref (and arrayp
                       (if (eq (car simple-type) 'simple-vector)
                           'svref
                         'aref)))
            (fix-vref (and arrayp
                           (if (array-etype-is-fixnum-type array-etype)
                               'fix-aref
                             vref)))
            (max-index (and arrayp (1- *expt2-28*)))
            (accessor-name (nth 3 field-template))
            (updater-name (nth 4 field-template))
            (length-name (nth 5 field-template))
            (resize-name (nth 6 field-template))
            (resizable (nth 7 field-template)))
       (cond
        (arrayp
         `((,length-name
            (,var)
            ,@(and inline (list *stobj-inline-declare*))
            ,@(if (not resizable)
                  `((declare (ignore ,var))
                    ,array-length)
                `((the (integer 0 ,max-index)
                       (length (svref ,var ,n))))))
           (,resize-name
            (k ,var)
            ,@(if (not resizable)
                  `((declare (ignore k))
                    (prog2$
                      (er hard ',resize-name
                          "The array field corresponding to accessor ~x0 of ~
                           stobj ~x1 was not declared :resizable t.  ~
                           Therefore, it is illegal to resize this array."
                          ',accessor-name
                          ',var)
                      ,var))
                `((if (not (and (integerp k)
                                (>= k 0)
                                (< k ,max-index)))
                      (hard-error
                       ',resize-name
                       "Attempted array resize failed because the requested ~
                        size ~x0 was not an integer between 0 and (1- (expt ~
                        2 28)).  These bounds on array sizes are fixed by ~
                        ACL2."
                       (list (cons #\0 k)))
                    (let* ((old (svref ,var ,n))
                           (min-index (if (< k (length old))
                                          k
                                        (length old)))
                           (new (make-array$ k

; The :initial-element below is probably not necessary in the case
; that we are downsizing the array.  At least, CLtL2 does not make any
; requirements about specifying an :initial-element, even when an
; :element-type is supplied.  However, it seems harmless enough to go
; ahead and specify :initial-element even for downsizing: resizing is
; not expected to be fast, we save a case split here (at the expense
; of this comment!), and besides, we are protecting against the
; possibility that some Common Lisp will fail to respect the spec and
; will cause an error by trying to initialize a fixnum array (say)
; with NILs.

                                             :initial-element
                                             ',init
                                             :element-type
                                             ',array-etype)))
                      (setf (svref ,var ,n)
                            (,(pack2 'stobj-copy-array- fix-vref)
                             old new 0 min-index))
                      ,var)))))
           (,accessor-name
            (i ,var)
            (declare (type (integer 0 ,max-index) i))
            ,@(and inline (list *stobj-inline-declare*))
            (the ,array-etype
              (,vref (the ,simple-type (svref ,var ,n))
                     (the (integer 0 ,max-index) i))))
           (,updater-name
            (i v ,var)
            (declare (type (integer 0 ,max-index) i)
                     (type ,array-etype v))
            ,@(and inline (list *stobj-inline-declare*))
            (progn 
              (setf (,vref (the ,simple-type (svref ,var ,n))
                           (the (integer 0 ,max-index) i))
                    (the ,array-etype v))
              ,var))))
        ((equal type t)
         `((,accessor-name (,var)
                           ,@(and inline (list *stobj-inline-declare*))
                           (svref ,var ,n))
           (,updater-name (v ,var)
                          ,@(and inline (list *stobj-inline-declare*))
                          (progn (setf (svref ,var ,n) v) ,var))))
        (t
         `((,accessor-name (,var)
                           ,@(and inline (list *stobj-inline-declare*))
                           (the ,type
                                (aref (the (simple-array ,type (1))
                                           (svref ,var ,n))
                                      0)))
           (,updater-name (v ,var)
                          (declare (type ,type v))
                          ,@(and inline (list *stobj-inline-declare*))
                          (progn
                            (setf (aref (the (simple-array ,type (1))
                                             (svref ,var ,n))
                                        0)
                                  (the ,type v))
                            ,var))))))
     (defstobj-field-fns-raw-defs var inline (1+ n) (cdr ftemps))))))

(defun defstobj-raw-init-fields (ftemps)

; Keep this in sync with defstobj-axiomatic-init-fields.

  (cond
   ((endp ftemps) nil)
   (t (let* ((field-template (car ftemps))
             (type (nth 1 field-template))
             (arrayp (and (consp type) (eq (car type) 'array)))
             (array-etype (and arrayp (cadr type)))
             (array-size (and arrayp (car (caddr type))))
             (init (nth 2 field-template)))
        (cond
         (arrayp
          (cons `(make-array$ ,array-size
                              :element-type ',array-etype
                              :initial-element ',init)
                (defstobj-raw-init-fields (cdr ftemps))))
         ((equal type t)
          (cons (kwote init) (defstobj-raw-init-fields (cdr ftemps))))
         (t (cons `(make-array$ 1
                                :element-type ',type
                                :initial-element ',init)
                  (defstobj-raw-init-fields (cdr ftemps)))))))))

(defun defstobj-raw-init (template)

; This function generates the initialization code for the live object
; representing the stobj name.

  (let ((field-templates (caddr template)))
    `(vector ,@(defstobj-raw-init-fields field-templates))))

(defun defstobj-raw-defs (name template)

; Warning:  See the guard remarks in the Essay on Defstobj Definitions.

; This function generates a list of defs.  Each def is such that
; (defun . def) is a well-formed raw Lisp definition.  The defuns can
; be executed in raw lisp to define the versions of the recognizers,
; accessors, and updaters (and for array fields, length and resize
; functions) that are run when we know the guards are satisfied.  Many
; of these functions anticipate application to the live object itself.

; WARNING: If you change the formals of these generated raw defs be
; sure to change the formals of the corresponding axiomatic defs.

  (let* ((recog (first template))
         (creator (second template))
         (field-templates (third template))
         (inline (fifth template)))
    (append
     (all-but-last
      (defstobj-component-recognizer-axiomatic-defs name template
        field-templates))
     (list* `(,recog (,name)
                     (cond
                      ((the-live-stobjp ,name)
                       t)
                      (t (and (true-listp ,name)
                              (= (length ,name) ,(length field-templates))
                              ,@(defstobj-component-recognizer-calls
                                  field-templates 0 name nil)))))
            `(,creator ()
                       ,(defstobj-raw-init template))
            (defstobj-field-fns-raw-defs name inline 0 field-templates)))))

(defun put-stobjs-in-and-outs1 (name ftemps wrld)

; See put-stobjs-in-and-outs for a table that explains what we're doing.

  (cond
   ((endp ftemps) wrld)
   (t (let ((type (nth 1 (car ftemps)))
            (acc-fn (nth 3 (car ftemps)))
            (upd-fn (nth 4 (car ftemps)))
            (length-fn (nth 5 (car ftemps)))
            (resize-fn (nth 6 (car ftemps))))
        (put-stobjs-in-and-outs1
         name
         (cdr ftemps)
         (cond
          ((and (consp type)
                (eq (car type) 'array))
           (putprop
            length-fn 'stobjs-in (list name) 
            (putprop
             resize-fn 'stobjs-in (list nil name)
             (putprop
              resize-fn 'stobjs-out (list name)
              (putprop
               acc-fn 'stobjs-in (list nil name)
               (putprop
                upd-fn 'stobjs-in (list nil nil name)
                (putprop
                 upd-fn 'stobjs-out (list name) wrld)))))))
          (t
           (putprop
            acc-fn 'stobjs-in (list name)
            (putprop
             upd-fn 'stobjs-in (list nil name)
             (putprop
              upd-fn 'stobjs-out (list name) wrld))))))))))
          
(defun put-stobjs-in-and-outs (name template wrld)

; We are processing a (defstobj name . args) event for which template
; is the template.  Wrld is a world containing the definitions of the
; accessors, updaters and recognizers of the stobj -- all of which
; were processed before we declared that name is a stobj.  Wrld now
; also contains the belated declaration that name is a stobj.  We now
; put the STOBJS-IN and STOBJS-OUT properties for the appropriate
; names.

; Relevant functions and their settings:

;      fn                  stobjs-in         stobjs-out
; topmost recognizer       (name)            (nil)
; creator                  ()                (name)
; field recogs             (nil ...)         (nil)  
; simple accessor          (name)            (nil)
; array accessor           (nil name)        (nil)
; simple updater           (nil name)        (name)
; array updater            (nil nil name)    (name)

; The entries above not involving name were correctly computed before
; we knew that name was a stobj and hence are correct in wrld now.

; It is important to realize, in the case of the topmost recognizer
; and the accessors -- which do not return stobjs, that the appearance
; of name in the stobjs-in setting can be interpreted to mean ``the
; stobj name MAY be supplied here'' as opposed to ``MUST be supplied
; here.''

  (let ((recog-name (car template))
        (creator-name (cadr template))
        (field-templates (caddr template)))

; Each element of field templates is of the form:
;       0      1                     2    3    4        5
; (field-recog field-recog-helper-fn type init accessor updater)
; or, for arrays,
; (field-recog field-recog-helper-fn type init accessor updater length-name
;  resize-name)
; and we know if the field is simple or an array according to whether
; (car type) is ARRAY.

    (put-stobjs-in-and-outs1 name
                             field-templates
                             (putprop creator-name
                                      'STOBJS-OUT
                                      (list name)
                                      (putprop recog-name
                                               'STOBJS-IN
                                               (list name)
                                               wrld)))))

(defun defconst-name-alist (lst n)
  (if (endp lst)
      nil
    (cons (cons n (defconst-name (car lst)))
          (defconst-name-alist (cdr lst) (1+ n)))))

(defun accessor-array (name field-names)
  (let ((len (length field-names)))
    (compress1 name
               (cons `(:HEADER :DIMENSIONS (,len)
                               :MAXIMUM-LENGTH ,(+ 1 len)
                               :DEFAULT nil ; should be ignored
                               :NAME ,name)
                     (defconst-name-alist field-names 0)))))

(defun strip-accessor-names (x)

; This could just as well be called strip-cadddrs.  X is the caddr of a
; defstobj template; see defstobj-template.

  (if (endp x)
      nil
    (cons (cadddr (car x))
          (strip-accessor-names (cdr x)))))

(defun defstobj-defconsts (names index)
  (if (endp names)
      nil
    (cons `(defconst ,(defconst-name (car names)) ,index)
          (defstobj-defconsts (cdr names) (1+ index)))))

(defun defstobj-fn (name args state event-form)
  (with-ctx-summarized
   (if (output-in-infixp state)
       event-form
     (msg "( DEFSTOBJ ~x0 ...)" name))
   (let ((event-form (or event-form (list* 'defstobj name args))))
     (er-let*
       ((wrld1 (chk-acceptable-defstobj name args ctx (w state) state)))
       (cond
        ((eq wrld1 'redundant)
         (stop-redundant-event state))
        (t
         (let* ((template (defstobj-template name args))
                (field-names (strip-accessor-names (caddr template)))
                (defconsts (defstobj-defconsts field-names 0))
                (field-const-names (strip-cadrs defconsts))
                (ax-def-lst (defstobj-axiomatic-defs name template))
                (raw-def-lst (defstobj-raw-defs name template))
                (recog-name (car template))
                (creator-name (cadr template))
                (names (strip-cars ax-def-lst))
                (the-live-var (the-live-var name))
                (doc (defstobj-doc args)))
           (er-progn
            (cond ((set-equalp-equal names
                                     (strip-cars raw-def-lst))
                   (value nil))
                  (t (value
                      (er hard ctx
                          "Defstobj-axiomatic-defs and ~
                           defstobj-raw-defs are out of sync!  They ~
                           should each define the same set of names.  ~
                           Here are the functions with axiomatic defs ~
                           that have no raw defs:  ~x0.  And here are ~
                           the with raw defs but no axiomatic ones:  ~
                           ~x1."
                          (set-difference-equal
                           names
                           (strip-cars raw-def-lst))
                          (set-difference-equal
                           (strip-cars raw-def-lst)
                           names)))))
            (revert-world-on-error
             (pprogn
              (set-w 'extension wrld1 state)
              (er-progn
               (process-embedded-events 'defstobj 
                                        (table-alist 'acl2-defaults-table wrld1)
                                        t                     ;;; skip-proofsp
                                        (current-package state)
                                        (list 'defstobj name names)
                                        (append

; We only need to lay down these set-*-ok calls in the case that we do not
; allow array resizing, for the resizing and length field functions.  But for
; simplicity, we always lay them down.  Note that since we are really modifying
; the acl2-defaults-table here, their effect is local to the defstobj.

                                         '((set-ignore-ok t)
                                           (set-irrelevant-formals-ok t))
                                         (pairlis-x1 'defun ax-def-lst)
                                         defconsts

; It is important to disable the executable counterpart of the creator
; function, so as not to expose the live stobj during proofs.  We ensure in
; function chk-theory-expr-value1 that the :executable-counterpart rune below
; will never be enabled.

                                         `((in-theory
                                            (disable
                                             (:executable-counterpart
                                              ,creator-name)))))
                                        ctx
                                        state)


; The processing above will define the functions in the logic, using
; defun, and that, in turn, will define their *1* counterparts in
; Lisp.  But because of code in defuns-fn, the processing above will
; not define the raw Lisp versions of the functions themselves
; (normally that would be derived from the axiomatic defs just
; processed).  Instead, we will store a CLTL-COMMAND below that
; handles the raw Lisp defs only.

; What follows is hard to follow and rather arcane.  Why do we include
; name in the ee-entry computed above, (defstobj name names)?  That
; entry will be added to the embedded-event-lst by
; process-embedded-events and be inspected by the individual defuns
; done.  Those defuns will recognize their fn name, fn, among names,
; to detect that they are being done as part of a defstobj.  The defun
; will pick up the stobj name, name, from the ee-entry and build it
; into the ignorep entry of the defun CLTL-COMMAND, to be processed by
; add-trip.  In add-trip, the stobj name, name, will find its way into
; the oneify-cltl-code that generates the *1* body for fn.  That body
; contains a throw upon detection of a guard error.  The object thrown
; contains the stobjs-in of the offensive expression, so we will know
; how to print it.  But the stobjs-in of fn is incorrectly set in the
; world right now -- more accurately, will be incorrectly set in the
; world in which the defun is done and the throw form is constructed
; -- because we have not yet declared name to be a stobj.  Indeed, we
; cannot declare it to be a stobj yet since we are defining functions
; that treat it as an ordinary list.  This is the stobj version of the
; super-defun-wart problem.

               (er-let*
                 ((doc-pair (translate-doc name doc ctx state)))
                 (let* ((wrld2 (w state))
                        (wrld3 (update-doc-data-base
                                name doc doc-pair

; Here I declare that name is Common Lisp compliant.  Below I
; similarly declare the-live-var.  All elements of the namex list of
; an event must have the same symbol-class.

                                (putprop
                                 name 'symbol-class :common-lisp-compliant
                                 (put-stobjs-in-and-outs
                                  name template

; Rockwell Addition: It is convenient for the recognizer to be in a
; fixed position in this list, so I can find out its name.

                                  (putprop
                                   name 'stobj
                                   (cons the-live-var
                                         (cons recog-name
                                               (append (delete1-eq recog-name
                                                                   names)
                                                       field-const-names)))
                                   (putprop
                                    name 'redundancy-bundle
                                    (defstobj-redundancy-bundle name args t)
                                    (putprop-x-lst1
                                     names 'stobj-function name
                                     (putprop-x-lst1
                                      field-const-names 'stobj-constant name
                                      (putprop
                                       the-live-var 'stobj-live-var name
                                       (putprop
                                        the-live-var 'symbol-class
                                        :common-lisp-compliant
                                        (putprop
                                         name
                                         'accessor-names
                                         (accessor-array name field-names)
                                         wrld2))))))))))))

; The property 'stobj marks a single-threaded object name.  Its value
; is a non-nil list containing all the names associated with this
; object.  The car of the list is always the live variable name for
; the object.  The cadr of the list (for all stobjs but our STATE) is
; the stobj recognizer for the stobj.  The remaining elements are all
; the functions used in the definition of the recognizer, the
; accessors and the updaters.  We don't list any of the STATE
; functions, just the live name, so the property is non-nil.

; Every supporting function is marked with the property
; 'stobj-function, whose value is the object name.  The live var name
; is marked with 'stobj-live-var, whose value is the object name.

; CHEAT:  I ought, at this point, 
;                 (pprogn
;                  (update-user-stobj-alist
;                   (cons (cons name (create-stobj name template))
;                         (user-stobj-alist state))
;                   state)

; That is, I should add to the user-stobj-alist in state an entry for
; this new stobj, binding its name to its initial value.  But I don't
; want to create the logical counterpart of its initial value -- the
; function create-stobj cannot be used this way (only uses
; resulting from with-local-stobj will pass translate), and we do
; not want to hack our way through the admission of this function
; which is apparently consing a stobj into an alist.  Instead, I rely
; on the live object representing the stobj.  This live object is
; created when the CLTL-COMMAND below is processed by add-trip.
; Add-trip evals the init form in raw lisp to create the live object
; and assign it to global variables.  It also creates array-based
; accessors and updaters.  It then stores this live object in the
; user-stobj-alist of the state just as suggested above, provided this
; is not a redefinition.  (For a redefinition of the stobj, it does a
; put-assoc-eq rather than a cons.)

; The down-side to this cheat is that this only works while
; defstobj-fn is a :program mode function called on the live state,
; where the raw code operates.  If I admitted this function to the
; logic and then called it on the live state, I would get an effect on
; the live state not explained by the code.  Furthermore, if I called
; it on a fake state, I would get a new fake state in which the new
; stobj was not on the user-stobj-alist.

; It will be a while before these discrepancies bother me enough to
; fix.  As long as this is a :program mode function, we won't be able
; to prove that its effect on state is contrary to its semantics as
; expressed here.

                   (install-event name
                                  event-form
                                  'defstobj

; Note: The namex generated below has consists of the single-threaded
; object name, the live variable name, and then the names of all the
; functions introduced.  Big-d-little-d-event knows it can cdr past
; the first two elements of the namex of a defstobj to find the list
; of functions involved.

                                  (list* name the-live-var names)
                                  nil
                                  `(defstobj ,name
                                     ,the-live-var
                                     ,(defstobj-raw-init template)
                                     ,raw-def-lst
                                     ,template)
                                  wrld3
                                  state))))))))))))))

(deflabel stobj
  :doc
  ":Doc-Section stobj

  single-threaded objects or ``von Neumann bottlenecks''~/

  In ACL2, a ``single-threaded object'' is a data structure whose use
  is so syntactically restricted that only one instance of the object
  need ever exist and its fields can be updated by destructive
  assignments.

  The documentation in this section is laid out in the form of a tour
  that visits the documented topics in a reasonable order.  We
  recommend that you follow the tour the first time you read about
  stobjs.  The list of all stobj topics is shown below.  The tour
  starts immediately afterwards.~/

  As noted, a ``single-threaded object'' is a data structure whose use
  is so syntactically restricted that only one instance of the object
  need ever exist.  Updates to the object must be sequentialized.
  This allows us to update its fields with destructive assignments
  without wrecking the axiomatic semantics of update-by-copy.  For
  this reason, single-threaded objects are sometimes called ``von
  Neumann bottlenecks.''

  From the logical perspective, a single-threaded object is an
  ordinary ACL2 object, e.g., composed of integers and conses.
  Logically speaking, ordinary ACL2 functions are defined to allow the
  user to ``access'' and ``update'' its fields.  Logically speaking,
  when fields in the object, ~em[obj], are ``updated'' with new values, a
  new object, ~em[obj'], is constructed.

  But suppose that by syntactic means we could ensure that there were
  no more references to the ``old'' object, ~em[obj].  Then we could
  create ~em[obj'] by destructively modifying the memory locations
  involved in the representation of ~em[obj].  The syntactic means is
  pretty simple but draconian: the only reference to ~em[obj] is in
  the variable named ~c[OBJ].

  The consequences of this simple rule are far-reaching and require
  some getting used to.  For example, if ~c[OBJ] has been declared as a
  single-threaded object name, then:

  * ~c[OBJ] is a top-level global variable that contains the current
    object, ~em[obj].

  * If a function uses the formal parameter ~c[OBJ], the only 
    ``actual expression'' that can be passed into that slot is ~c[OBJ];
    thus, such functions can only operate on the current object.

  * The accessors and updaters have a formal parameter named ~c[OBJ],
    thus, those functions can only be applied to the current object.

  * The ACL2 primitives, such as ~c[CONS], ~c[CAR] and ~c[CDR], may not
    be applied to the variable ~c[OBJ].  Thus, for example, ~em[obj] may not
    be consed into a list (which would create another pointer to it) or
    accessed or copied via ``unapproved'' means.

  * The updaters return a ``new ~c[OBJ] object'', i.e., ~em[obj']; thus, when
    an updater is called, the only variable which can hold its result is
    ~c[OBJ].

  * If a function calls an ~c[OBJ] updater, it must return ~c[OBJ].

  * When a top-level expression involving ~c[OBJ] returns an ~c[OBJ]
    object, that object becomes the new current value of ~c[OBJ].

  What makes ACL2 different from other functional languages supporting
  such operations (e.g., Haskell's ``monads'' and Clean's ``uniqueness
  type system'') is that ACL2 also gives single-threaded objects an
  explicit axiomatic semantics so that theorems can be proved about
  them.  In particular, the syntactic restrictions noted above are
  enforced only when single-threaded objects are used in function
  definitions (which might be executed outside of the ACL2
  read-eval-print loop in Common Lisp).  The accessor and update
  functions for single-threaded objects may be used without
  restriction in formulas to be proved.  Since function evaluation is
  sometimes necessary during proofs, ACL2 must be able to evaluate
  these functions on logical constants representing the object, even
  when the constant is not ``the current object.''  Thus, ACL2
  supports both the efficient von Neumann semantics and the clean
  applicative semantics, and uses the first in contexts where
  execution speed is paramount and the second during proofs.

  To start the stobj tour, ~pl[stobj-example-1].~/")

(deflabel stobj-example-1
  :doc
  ":Doc-Section stobj

  an example of the use of single-threaded objects~/

  Suppose we want to sweep a tree and (1) count the number of interior
  nodes, (2) count the number of tips and (3) keep a record of every
  tip we encounter that is an integer.  We could use a single-threaded
  object as our ``accumulator''.  Such an object would have three
  fields, one holding the number of nodes seen so far, one holding the
  number of tips, and one holding all the integer tips seen.~/

  The following event declares ~c[counters] to be a single-threaded object.
  ~bv[]
  (defstobj counters
    (NodeCnt     :type integer :initially 0)
    (TipCnt      :type integer :initially 0)
    (IntTipsSeen :type t       :initially nil))
  ~ev[]
  It has three fields, ~c[NodeCnt], ~c[TipCnt], and ~c[IntTipsSeen].
  (As always in ACL2, capitalization is irrelevant in simple symbol
  names, so the first name could be written ~c[nodecnt] or
  ~c[NODECNT], etc.) Those are the name of the accessor functions for
  the object.  The corresponding update functions are named
  ~c[update-NodeCnt], ~c[update-TipCnt] and ~c[update-IntTipsSeen].

  If you do not like the default function names chosen above, there is
  a feature in the ~ilc[defstobj] event that allows you to specify other
  names.

  If you want to see the ACL2 definitions of all the functions defined
  by this event, look at ~il[stobj-example-1-defuns].

  If, after this event, we evaluate the top-level ``global variable''
  ~c[counters] in the ACL2 read-eval-print loop we get:
  ~bv[]
  ACL2 !>counters
  <counters>
  ~ev[]
  Note that the value printed is ``~c[<counters>]''.  Actually, the
  value of ~c[counters] in the logic is ~c[(0 0 NIL)].  But ACL2 always prints
  single-threaded objects in this non-informative way because they are
  usually so big that to do otherwise would be unpleasant.

  Had you tried to evaluate the ``global variable'' ~c[counters] before
  declaring it a single-threaded object, ACL2 would have complained that
  it does not support global variables.  So a lesson here is that
  once you have declared a new single-threaded object your top-level
  forms can reference it.  In versions of ACL2 prior to Version  2.4
  the only variable enjoying this status was ~c[STATE].  single-threaded
  objects are a straightforward generalization of the long-implemented
  von Neumann ~ilc[state] feature of ACL2.

  We can access the fields of ~c[counters] as with:
  ~bv[]
  ACL2 !>(NodeCnt counters)
  0
  ACL2 !>(IntTipsSeen counters)  
  NIL
  ~ev[]
  and we can set the fields of ~c[counters] as with:
  ~bv[]
  ACL2 !>(update-NodeCnt 3 counters)
  <counters>
  ACL2 !>(NodeCnt counters)
  3  
  ~ev[]
  Observe that when we evaluate an expression that returns a
  counter object, that object becomes the ``current value'' of
  ~c[counters].  

  Here is a function that ``converts'' the ~c[counters] object to its
  ``ordinary'' representation:
  ~bv[]
  (defun show-counters (counters)
    (declare (xargs :stobjs (counters)))
    (list (NodeCnt counters)
          (TipCnt counters)
          (IntTipsSeen counters)))
  ~ev[]
  Observe that we ~em[must] declare, at the top of the ~c[defun], that
  we mean to use the formal parameter ~c[counters] as a single-threaded
  object!  If we did not make this declaration, the body of
  ~c[show-counters] would be processed as though ~c[counters] were an
  ordinary object.  An error would be caused because the accessors
  used above cannot be applied to anything but the single-threaded
  object ~c[counters].  If you want to know why we insist on this
  declaration, ~pl[declare-stobjs].

  When ~c[show-counters] is admitted, the following message is printed:
  ~bv[]
  Since SHOW-COUNTERS is non-recursive, its admission is trivial.  We
  observe that the type of SHOW-COUNTERS is described by the theorem
  (AND (CONSP (SHOW-COUNTERS COUNTERS))
       (TRUE-LISTP (SHOW-COUNTERS COUNTERS))).
  We used primitive type reasoning.

  (SHOW-COUNTERS COUNTERS) => *.

  The guard conjecture for SHOW-COUNTERS is trivial to prove.  
  SHOW-COUNTERS is compliant with Common Lisp.
  ~ev[]
  The line above containing the ``=>'' is called the ``signature'' of
  ~c[show-counters]; it conveys the information that the first argument
  is the single-threaded object ~c[counters] and the only result is an
  ordinary object.  Here is an example of another signature:
  ~bv[]
  (PROCESSOR * * COUNTERS) => (MV * COUNTERS)
  ~ev[]
  which indicates that the function ~c[PROCESSOR] (which we haven't
  shown you) takes three arguments, the third of which is the 
  ~c[COUNTERS] stobj, and returns two results, the second of which
  is the modified ~c[COUNTERS].

  Returning to the admission of ~c[show-counters] above, the last
  sentence printed indicates that the ~ilc[guard] conjectures for the
  function were proved.  When some argument of a function is declared
  to be a single-threaded object via the ~c[xargs] ~c[:stobj], we
  automatically add (conjoin) to the guard the condition that the
  argument satisfy the recognizer for that single-threaded object.  In
  the case of ~c[show-counters] the guard is ~c[(countersp counters)].

  Here is an example of ~c[show-counters] being called:
  ~bv[]
  ACL2 !>(show-counters counters)
  (3 0 NIL)
  ~ev[]
  This is what we would see had we set the ~c[NodeCnt] field of the
  initial value of ~c[counters] to ~c[3], as we did earlier in this
  example.

  We next wish to define a function to reset the ~c[counters] object.
  We could define it this way:
  ~bv[]
  (defun reset-counters (counters)
    (declare (xargs :stobjs (counters)))
    (let ((counters (update-NodeCnt 0 counters)))
      (let ((counters (update-TipCnt 0 counters)))
        (update-IntTipsSeen nil counters))))
  ~ev[]
  which ``successively'' sets the ~c[NodeCnt] field to ~c[0], then the
  ~c[TipCnt] field to ~c[0], and then the ~c[IntTipsSeen] field to ~c[nil] and
  returns the resulting object.

  However, the nest of ~c[let] expressions is tedious and we use this
  definition instead.  This definition exploits a macro, here named
  ``~c[seq]'' (for ``sequentially'') which evaluates each of the forms
  given, binding their results successively to the stobj name given.  
  ~bv[]
  (defun reset-counters (counters)
    (declare (xargs :stobjs (counters)))
    (seq counters
         (update-NodeCnt 0 counters)
         (update-TipCnt 0 counters)
         (update-IntTipsSeen nil counters)))
  ~ev[]
  This definition is syntactically identical to the one above, after macro
  expansion.  Our definition of ~c[seq] is shown below and is not part of
  native ACL2.
  ~bv[]
  (defmacro seq (stobj &rest rst)
    (cond ((endp rst) stobj)
          ((endp (cdr rst)) (car rst))
          (t `(let ((,stobj ,(car rst)))
               (seq ,stobj ,@(cdr rst))))))
  ~ev[]

  The signature printed for ~c[reset-counters] is
  ~bv[]
  (RESET-COUNTERS COUNTERS) => COUNTERS.
  ~ev[]

  Here is an example.
  ~bv[]
  ACL2 !>(show-counters counters)
  (3 0 NIL)
  ACL2 !>(reset-counters counters)
  <counters>
  ACL2 !>(show-counters counters)
  (0 0 NIL) 
  ~ev[]

  Here finally is a function that uses ~c[counters] as a single-threaded
  accumulator to collect the desired information about the tree ~c[x].
  ~bv[]
  (defun sweep-tree (x counters)
    (declare (xargs :stobjs (counters)))
    (cond ((atom x)
           (seq counters
                (update-TipCnt (+ 1 (TipCnt counters)) counters)
                (if (integerp x)
                    (update-IntTipsSeen (cons x (IntTipsSeen counters))
                                    counters)
                  counters)))
          (t (seq counters
                  (update-NodeCnt (+ 1 (NodeCnt counters)) counters)
                  (sweep-tree (car x) counters)
                  (sweep-tree (cdr x) counters)))))
  ~ev[]
  We can paraphrase this definition as follows.  If ~c[x] is an atom,
  then increment the ~c[TipCnt] field of ~c[counters] and ~em[then],
  if ~c[x] is an integer, add ~c[x] to the ~c[IntTipsSeen] field, and
  return ~c[counters].  On the other hand, if ~c[x] is not
  an atom, then increment the ~c[NodeCnt] field of ~c[counters], and
  ~em[then] sweep the ~c[car] of ~c[x] and ~em[then] sweep the ~c[cdr]
  of ~c[x] and return the result.

  Here is an example of its execution.  We have displayed the input tree
  in full dot notation so that the number of interior nodes is just the
  number of dots.
  ~bv[]
  ACL2 !>(sweep-tree '((((a . 1) . (2 . b)) . 3)
                       . (4 . (5 . d)))
                     counters)
  <counters>
  ACL2 !>(show-counters counters)
  (7 8 (5 4 3 2 1))
  ACL2 !>(reset-counters counters)
  <counters>
  ACL2 !>(show-counters counters)
  (0 0 NIL)
  ~ev[]

  The ~c[counters] object has two integer fields and a field whose
  type is unrestricted.  single-threaded objects support other types of
  fields, such as arrays.  We deal with that in the ~il[stobj-example-2].
  But we recommend that you first consider the implementation issues for
  the ~c[counters] example (in ~il[stobj-example-1-implementation]) and
  then consider the proof issues (in ~il[stobj-example-1-proofs]).

  To continue the stobj tour, ~pl[stobj-example-2].~/")

(deflabel declare-stobjs
  :doc
  ":Doc-Section stobj

  declaring a formal parameter name to be a single-threaded object~/

  When a ~ilc[defun] uses one of its formals as a single-threaded object
  (~il[stobj]), the ~c[defun] ~em[must] include a declaration that the
  formal it to be so used.  An exception is the formal ``~ilc[state],'' which
  if not declared as explained below, may still be used provided an
  appropriate global ``declaration'' is issued:
  ~pl[set-state-ok].~/

  If the formal in question is ~c[counters] then an appropriate declaration
  is
  ~bv[]
  (declare (xargs :stobjs counters))
  ~ev[]
  or, more generally,
  ~bv[]
  (declare (xargs :stobjs (... counters ...)))
  ~ev[]
  where all the single-threaded formals are listed.

  For such a declaration to be legal it must be the case that all the names
  have previously been defined as single-threaded objects with ~ilc[defstobj].

  When an argument is declared to be single-threaded the guard of the
  function is augmented by conjoining to it the condition that the
  argument satisfy the recognizer for the single-threaded object.
  Furthermore, the syntactic checks done to enforce the legal use of
  single-threaded objects are also sufficient to allow these guard
  conjuncts to be automatically proved.

  The obvious question arises:  Why does ACL2 insist that you declare
  stobj names before using them in ~c[defun]s if you can only declare names
  that have already been defined with ~c[defstobj]?  What would go wrong if
  a formal were treated as a single-threaded object if and only if it had
  already been so defined?

  Suppose that one user, say Jones, creates a book in which ~c[counters]
  is defined as a single-threaded object.  Suppose another user, Smith,
  creates a book in which ~c[counters] is used as an ordinary formal
  parameter.  Finally, suppose a third user, Brown, wishes to use both
  books.  If Brown includes Jones' book first and then Smith's, then
  Smith's function treats ~c[counters] as single-threaded.  But if Brown
  includes Smith's book first, the argument is treated as ordinary.

  ACL2 insists on the declaration to ensure that the definition is
  processed the same way no matter what the context.~/")

(deflabel stobj-example-1-defuns
  :doc
  ":Doc-Section stobj

  the defuns created by the ~c[counters] stobj~/
  
  Consider the event shown in ~il[stobj-example-1]:
  ~bv[]
  (defstobj counters
    (NodeCnt     :type integer :initially 0)
    (TipCnt      :type integer :initially 0)
    (IntTipsSeen :type t       :initially nil))
  ~ev[]

  Here is a complete list of the defuns added by the event.~/

  The careful reader will note that the ~c[counters] argument below is
  ~em[not] declared with the ~c[:stobjs] ~c[xarg] even though we
  insist that the argument be a stobj in calls of these functions.
  This ``mystery'' is explained below.

  ~bv[]
  (defun NodeCntp (x)                 ;;; Recognizer for 1st field
    (declare (xargs :guard t :verify-guards t))
    (integerp x))

  (defun TipCntp (x)                  ;;; Recognizer for 2nd field
    (declare (xargs :guard t :verify-guards t))
    (integerp x))

  (defun IntTipsSeenp (x)             ;;; Recognizer for 3rd field
    (declare (xargs :guard t :verify-guards t) (ignore x))
    t)

  (defun countersp (counters)         ;;; Recognizer for object
    (declare (xargs :guard t :verify-guards t))
    (and (true-listp counters)
         (= (length counters) 3)
         (NodeCntp (nth 0 counters))
         (TipCntp (nth 1 counters))
         (IntTipsSeenp (nth 2 counters))
         t))

  (defun create-counters ()           ;;; Creator for object
    (declare (xargs :guard t :verify-guards t))
    (list '0 '0 'nil))

  (defun NodeCnt (counters)           ;;; Accessor for 1st field
    (declare (xargs :guard (countersp counters) :verify-guards t))
    (nth 0 counters))

  (defun update-NodeCnt (v counters)  ;;; Updater for 1st field
    (declare (xargs :guard
                    (and (integerp v)
                         (countersp counters))
                    :verify-guards t))
    (update-nth 0 v counters))

  (defun TipCnt (counters)            ;;; Accessor for 2nd field
    (declare (xargs :guard (countersp counters) :verify-guards t))
    (nth 1 counters))

  (defun update-TipCnt (v counters)   ;;; Updater for 2nd field
    (declare (xargs :guard
                    (and (integerp v)
                         (countersp counters))
                    :verify-guards t))
    (update-nth 1 v counters))

  (defun IntTipsSeen (counters)       ;;; Accessor for 3rd field
    (declare (xargs :guard (countersp counters) :verify-guards t))
    (nth 2 counters))

  (defun update-IntTipsSeen (v counters) ;;; Updater for 3rd field
    (declare (xargs :guard (countersp counters) :verify-guards t))
    (update-nth 2 v counters))
  ~ev[]

  Observe that there is a recognizer for each of the three fields and
  then a recognizer for the ~c[counters] object itself.  Then, for each
  field, there is an accessor and an updater.

  Observe also that the functions are guarded so that they expect a
  ~c[countersp] for their ~c[counters] argument and an appropriate value
  for the new field values.

  You can see all of the ~c[defuns] added by a ~c[defstobj] event by
  executing the event and then using the ~c[:pcb!] command on the stobj
  name.  E.g.,
  ~bv[]
  ACL2 !>:pcb! counters
  ~ev[]
  will print the defuns above.

  We now clear up the ``mystery'' mentioned above.  Note, for example
  in ~c[TipCnt], that the formal ~c[counters] is used.  From the
  discussion in ~il[stobj-example-1] it has been made clear that
  ~c[TipCnt] can only be called on the ~c[counters] object.  And yet,
  in that same discussion it was said that an argument is so treated
  only if it it declared among the ~c[:stobjs] in the definition of
  the function.  So why doesn't ~c[TipCnt] include something like
  ~c[(declare (xargs :stobjs (counters)))]?

  The explanation of this mystery is as follows.  At the time
  ~c[TipCnt] was defined, during the introduction of the ~c[counters]
  stobj, the name ``~c[counters]'' was not yet a single-threaded
  object.  The introduction of a new single-threaded object occurs in
  three steps: (1) The new primitive recognizers, accessors, and
  updaters are introduced as ``ordinary functions,'' producing their
  logical axiomatizations.  (2) The executable counterparts are
  defined in raw Lisp to support destructive updating.  (3) The new
  name is declared a single-threaded object to ensure that all future
  use of these primitives respects the single-threadedness of the
  object.  The functions defined as part of the introduction of a new
  single-threaded object are the only functions in the system that
  have undeclared stobj formals other than ~c[state].

  You may return to ~il[stobj-example-1] here.~/")

(deflabel stobj-example-1-implementation
  :doc
  ":doc-section stobj

  the implementation of the ~c[counters] stobj~/

  the event  
  ~bv[]
  (defstobj counters
    (NodeCnt     :type integer :initially 0)
    (TipCnt      :type integer :initially 0)
    (IntTipsSeen :type t       :initially nil))
  ~ev[]
  discussed in ~il[stobj-example-1], creates a Common Lisp object to
  represent the current value of ~c[counters].  That object is created
  by evaluating either of the following ``raw'' (non-ACL2) Common Lisp
  forms:
  ~bv[]
  (create-counters)

  (vector (make-array 1 :element-type 'integer
                        :initial-element '0)
          (make-array 1 :element-type 'integer
                        :initial-element '0)
          'nil)
  ~ev[]
  and the value is stored in the Common Lisp global variable named
  ~c[*the-live-counters*].
  ~/

  Thus, the ~c[counters] object is an array of length three.  The first
  two elements are arrays of size 1 and are used to hold the
  ~c[NodeCnt] and ~c[TipCnt] fields.  The third element is the
  ~c[IntTipsSeen] field.  The first two fields are represented by
  arrays so that we can implement the ~c[integer] type specification
  efficiently.  Generally, integers are ``boxed'' in some Common Lisp
  implementations, for example, GCL.  Creating a new integer requires
  creating a new box to put it in.  But in some lisps, including GCL,
  the integers inside arrays of integers are not boxed.

  The function ~c[NodeCnt] is defined in raw Lisp as:
  ~bv[]
  (defun NodeCnt (counters)
    (the integer
         (aref (the (simple-array integer (1))
                    (svref counters 0))
               0)))
  ~ev[]
  Observe that the form ~c[(svref counters 0)] is evaluated to get
  an array of size 1, which is followed by a call of ~c[aref] to
  access the 0th element of that array.

  The function ~c[update-NodeCnt] is defined in raw Lisp as:
  ~bv[]
  (defun update-NodeCnt (v counters)
    (declare (type integer v))
    (progn
     (setf (aref (the (simple-array integer (1))
                      (svref counters 0))
                 0)
           (the integer v))
     counters))
  ~ev[]  
  Note that when this function is called, it does not create a new
  vector of length three, but ``smashes'' the existing one.

  One way to see all the functions defined by a given ~c[defstobj] is to
  evaluate the ~c[defstobj] event and then evaluate, in the ACL2 loop,
  the expression ~c[(global-val 'cltl-command (w state))].  That will
  print a lisp object that you can probably figure out.

  We now recommend that you look at ~il[stobj-example-1-proofs].~/")

(deflabel stobj-example-1-proofs
  :doc
  ":Doc-Section stobj

  some proofs involving the ~c[counters] stobj~/

  Consider again the event  
  ~bv[]
  (defstobj counters
    (NodeCnt     :type integer :initially 0)
    (TipCnt      :type integer :initially 0)
    (IntTipsSeen :type t       :initially nil))
  ~ev[]
  discussed in ~il[stobj-example-1], followed by the definition
  ~bv[]
  (defun reset-counters (counters)
    (declare (xargs :stobjs (counters)))
    (seq counters
         (update-NodeCnt 0 counters)
         (update-TipCnt 0 counters)
         (update-IntTipsSeen nil counters)))
  ~ev[]
  which, because of the ~c[seq] macro in ~il[stobj-example-1], is just
  syntactic sugar for
  ~bv[]
  (defun reset-counters (counters)
    (declare (xargs :stobjs (counters)))
    (let ((counters (update-NodeCnt 0 counters)))
      (let ((counters (update-TipCnt 0 counters)))
        (update-IntTipsSeen nil counters)))).
  ~ev[]

  Here is a simple theorem about ~c[reset-counters].

  ~bv[]
  (defthm reset-counters-is-constant
    (implies (countersp x)
             (equal (reset-counters x)
                    '(0 0 nil))))
  ~ev[]
  ~/
  Before we talk about how to prove this theorem, note that the theorem
  is unusual in two respects.

  First, it calls ~c[reset-counters] on an argument other than the
  variable ~c[counters]!  That is allowed in theorems; logically
  speaking, the stobj functions are indistinguishable from ordinary
  functions.  Their use is syntactically restricted only in
  ~c[defun]s, which might be compiled and run in raw Lisp.  Those
  restrictions allow us to implement stobj modification destructively.
  But logically speaking, ~c[reset-counters] and other stobj
  ``modifying'' functions just create new objects, constructively.

  Second, the theorem above explicitly provides the hypothesis that
  ~c[reset-counters] is being applied to an object satisfying
  ~c[countersp].  Such a hypothesis is not always required:
  ~c[reset-counters] is total and will do something no matter what
  ~c[x] is.  But in this particular case, the result is not ~c['(0 0 nil)]
  unless ~c[x] is, at least, a true-list of length three.

  To make a long story short, to prove theorems about stobj functions you
  behave in exactly the way you would to prove the same theorems about the
  same functions defined without the stobj features.

  How can we prove the above theorem?  Unfolding the definition of
  ~c[reset-counters] shows that ~c[(reset-counters x)] is equal to
  ~bv[]
  (update-IntTipsSeen nil
    (update-TipCnt 0 
      (update-NodeCnt 0 x)))
  ~ev[]
  which in turn is
  ~bv[]
  (update-nth 2 nil
   (update-nth 1 0
    (update-nth 0 0 x))).
  ~ev[]
  Opening up the definition of ~c[update-nth] reduces this to
  ~bv[]
  (list* 0 0 nil (cdddr x)).
  ~ev[]
  This is clearly equal to ~c['(0 0 nil)], provided we know that ~c[(cdddr x)]
  is ~c[nil].

  Unfortunately, that last fact requires a lemma.  The most specific lemma we
  could provide is
  ~bv[]
  (defthm special-lemma-for-counters
    (implies (countersp x)
             (equal (cdddr x) nil)))
  ~ev[]
  but if you try to prove that lemma you will find that it requires some
  reasoning about ~c[len] and ~c[true-listp].  Furthermore, the special
  lemma above is of interest only for ~c[counters].

  The following lemma about ~c[len] is the one we prefer.
  ~bv[]
  (defthm equal-len-n
    (implies (syntaxp (quotep n))
             (equal (equal (len x) n)
                    (if (integerp n)
                        (if (< n 0)
                            nil
                          (if (equal n 0)
                              (atom x)
                            (and (consp x)
                                 (equal (len (cdr x)) (- n 1)))))
                      nil))))
  ~ev[]
  This lemma will simplify any equality in which a ~c[len] expression
  is equated to any explicitly given constant ~em[n], e.g.,
  ~c[3], reducing the equation to a conjunction of ~c[consp] terms
  about the first ~em[n] ~c[cdr]s.

  If the above lemma is available then ACL2 immediately proves
  ~bv[]
  (defthm reset-counters-is-constant
    (implies (countersp x)
             (equal (reset-counters x)
                    '(0 0 nil))))
  ~ev[]

  The point is presumably well made: proving theorems about
  single-threaded object accessors and updaters is no different than
  proving theorems about other recursively defined functions on lists.

  As we have seen, operations on ~il[stobj]s turn into definitions
  involving ~ilc[nth] and ~ilc[update-nth] in the logic.  Here are two lemmas
  that are useful for simplifying terms involving ~c[nth] and ~c[update-nth],
  which are therefore useful in reasoning about single-threaded objects.
  ~bv[]
  (defthm update-nth-update-nth-same
    (implies (equal (nfix i1) (nfix i2))
             (equal (update-nth i1 v1 (update-nth i2 v2 l))
                    (update-nth i1 v1 l))))

  (defthm update-nth-update-nth-diff
    (implies (not (equal (nfix i1) (nfix i2)))
             (equal (update-nth i1 v1 (update-nth i2 v2 l))
                    (update-nth i2 v2 (update-nth i1 v1 l))))
    :rule-classes ((:rewrite :loop-stopper ((i1 i2)))))
  ~ev[]
  These lemmas are due to Matt Wilding.  ~l[nu-rewriter] for a
  discussion of the efficient simplification of terms of the form
  ~c[(nth n (update-nth key val lst))], which can be critical in
  settings involving sequential bindings that commonly arise in
  operations involving stobjs.

  We now recommend that you ~pl[stobj-example-2].~/")

(deflabel stobj-example-2
  :doc
  ":doc-section stobj

  an example of the use of arrays in single-threaded objects~/
  
  The following event
  ~bv[]
  (defstobj ms
    (pcn  :type integer                  :initially 0)
    (mem  :type (array integer (100000)) :initially -1)
    (code :type t                        :initially nil))
  ~ev[]
  introduces a single-threaded object named ~c[ms] (which stands for
  ``machine state'').  The object has three fields, a ~c[pcn] or program
  counter, a ~c[mem] or memory, and a ~c[code] field.

  The ~c[mem] field is occupied by an object initially of type 
  ~c[(array integer (100000))].  Logically speaking, this is a list of
  length ~c[100000], each element of which is an integer.  But in the
  underlying implementation of the ~c[ms] object, this field is occupied
  by a raw Lisp array, initially of size 100000.~/

  You might expect the above ~c[defstobj] to define the accessor function
  ~c[mem] and the updater ~c[update-mem].  ~em[That does not happen!].

  The above event defines the accessor function ~c[memi] and the updater
  ~c[update-memi].  These functions do not access/update the ~c[mem] field of
  the ~c[ms] object; they access/update the individual elements of the
  array in that field.

  In particular, the logical definitions of the two functions are:
  ~bv[]
  (defun memi (i ms)
    (declare (xargs :guard
                    (and (msp ms)
                         (integerp i)
                         (<= 0 i)
                         (< i (mem-length ms)))))
    (nth i (nth 1 ms)))

  (defun update-memi (i v ms)
    (declare (xargs :guard
                    (and (msp ms)
                         (integerp i)
                         (<= 0 i)
                         (< i (mem-length ms))
                         (integerp v))))
    (update-nth-array 1 i v ms))
  ~ev[]

  For example, to access the 511th (0-based) memory location of the
  current ~c[ms] you could evaluate:
  ~bv[]
  ACL2 !>(memi 511 ms)
  -1
  ~ev[]
  The answer is ~c[-1] initially, because that is the above-specified
  initial value of the elements of the ~c[mem] array.

  To set that element you could do
  ~bv[]
  ACL2 !>(update-memi 511 777 ms)
  <ms>
  ACL2 !>(memi 511 ms)
  777
  ~ev[]

  The raw Lisp implementing these two functions is shown below.
  ~bv[]
  (defun memi (i ms)
    (declare (type (integer 0 268435455) i))
    (the integer
         (aref (the (simple-array integer (*))
                    (svref ms 1))
               (the (integer 0 268435455) i))))

  (defun update-memi (i v ms)
    (declare (type (integer 0 268435455) i)
             (type integer v))
    (progn
     (setf (aref (the (simple-array integer (*))
                      (svref ms 1))
                 (the (integer 0 268435455) i))
           (the integer v))
     ms))
  ~ev[]

  If you want to see the raw Lisp supporting a ~c[defstobj], execute the
  ~c[defstobj] and then evaluate the ACL2 form 
  ~c[(global-val 'cltl-command (w state))].  The s-expression printed
  will probably be self-explanatory given the examples here.

  To continue the stobj tour, ~pl[stobj-example-3].~/")

(deflabel stobj-example-3
  :doc
  ":Doc-Section stobj

  another example of a single-threaded object~/

  The event
  ~bv[]
  (defstobj $s
    (x :type integer :initially 0)
    (a :type (array (integer 0 9) (3)) :initially 9 :resizable t))
  ~ev[]
  introduces a stobj named ~c[$S].  The stobj has two fields, ~c[X] and
  ~c[A].  The ~c[A] field is an array.  The ~c[X] field contains an
  integer and is initially 0.  The ~c[A] field contains a list of
  integers, each between 0 and 9, inclusively.  (Under the hood, this
  ``list'' is actually implemented as an array.)  Initially, the ~c[A]
  field has three elements, each of which is 9.~/

  This event introduces the following sequence of function definitions:
  ~bv[]
  (DEFUN XP (X) ...)               ; recognizer for X field
  (DEFUN AP (X) ...)               ; recognizer of A field
  (DEFUN $SP ($S) ...)             ; top-level recognizer for stobj $S
  (DEFUN CREATE-$S NIL ...)        ; creator for stobj $S
  (DEFUN X ($S) ...)               ; accessor for X field
  (DEFUN UPDATE-X (V $S) ...)      ; updater for X field
  (DEFUN A-LENGTH ($S) ...)        ; length of A field
  (DEFUN RESIZE-A (K $S) ...)      ; resizer for A field
  (DEFUN AI (I $S) ...)            ; accessor for A field at index I
  (DEFUN UPDATE-AI (I V $S) ...)   ; updater for A field at index I
  ~ev[]

  Here is the definition of ~c[$SP]:
  ~bv[]
  (DEFUN $SP ($S)
    (DECLARE (XARGS :GUARD T :VERIFY-GUARDS T))
    (AND (TRUE-LISTP $S)
         (= (LENGTH $S) 2)
         (XP (NTH 0 $S))
         (AP (NTH 1 $S))
         T))
  ~ev[]
  This reveals that in order to satisfy ~c[$SP] an object must be
  a true list of length 2 whose first element satisfies ~c[XP] and whose
  second satisfies ~c[AP].  By printing the definition of ~c[AP] one
  learns that it requires its argument to be a true list, each element
  of which is an integer between 0 and 9.

  The initial value of stobj ~c[$S] is given by zero-ary ``creator''
  function ~c[CREATE-$S].  Creator functions may only be used in limited
  contexts.  ~l[with-local-stobj].

  Here is the definition of ~c[UPDATE-AI], the updater for the ~c[A] field
  at index ~c[I]:
  ~bv[]
  (DEFUN UPDATE-AI (I V $S)
    (DECLARE (XARGS :GUARD
                    (AND ($SP $S)
                         (INTEGERP I)
                         (<= 0 I)
                         (< I (A-LENGTH $S))
                         (AND (INTEGERP V) (<= 0 V) (<= V 9)))
                    :VERIFY-GUARDS T))
    (UPDATE-NTH-ARRAY 1 I V $S))
  ~ev[]
  By definition ~c[(UPDATE-NTH-ARRAY 1 I V $S)] is
  ~c[(UPDATE-NTH 1 (UPDATE-NTH I V (NTH 1 $S)) $S)].
  This may be a little surprising but should be perfectly clear.

  First, ignore the guard, since it is irrelevant in the logic.
  Reading from the inside out, ~c[(UPDATE-AI I V $S)] extracts ~c[(NTH 1 $S)],
  which is array ~c[a] of ~c[$S].  (Recall that ~ilc[NTH] is
  0-based.)  The next higher expression in the definition above,
  ~c[(UPDATE-NTH I V a)], ``modifies'' ~c[a] by setting its ~c[I]th
  element to ~c[V].  Call this ~c[a'].  The next higher expression,
  ~c[(UPDATE-NTH 1 a' $S)], ``modifies'' ~c[$S] by setting its 1st
  component to ~c[a'].  Call this result ~c[$s'].  Then ~c[$s'] is
  the result returned by ~c[UPDATE-AI].

  So the first useful observation is that from the perspective of the
  logic, the type ``restrictions'' on stobjs are irrelevant.  They
  are ``enforced'' by ACL2's guard mechanism, not by the definitions
  of the updater functions.

  As one might also imagine, the accessor functions do not really
  ``care,'' logically, whether they are applied to well-formed stobjs
  or not.  For example, ~c[(AI I $S)] is defined to be ~c[(NTH I (NTH 1 $S))].

  Thus, you will not be able to prove that (AI 2 $S) is an
  integer.  That is,
  ~bv[]
  (integerp (AI 2 $S))
  ~ev[]
  is not a theorem, because ~c[$S] may not be well-formed.

  Now ~c[(integerp (AI 2 $S))] will always evaluate to ~c[T] in the
  top-level ACL2 command loop, because we insist that the current value of
  the stobj ~c[$S] always satisfies ~c[$SP] by enforcing the guards on
  the updaters, independent of whether guard checking is on or off;
  ~pl[set-guard-checking].  But in a theorem ~c[$S] is just
  another variable, implicitly universally quantified.

  So ~c[(integerp (AI 2 $S))] is not a theorem because it is not true when
  the variable ~c[$S] is instantiated with, say,
  ~bv[]
  '(1 (0 1 TWO))
  ~ev[]
  because, logically speaking, ~c[(AI 2 '(1 (0 1 TWO)))] evaluates to
  the symbol ~c[TWO].  That is,
  ~bv[]
  (equal (AI 2 '(1 (0 1 TWO))) 'TWO)
  ~ev[]
  is true.

  However,
  ~bv[]
  (implies (and ($SP $S) (< 2 (A-LENGTH $S))) (integerp (AI 2 $S)))
  ~ev[]
  is a theorem.  To prove it, you will have to prove a lemma about
  ~c[AP].  The following will do:
  ~bv[]
  (defthm ap-nth
    (implies (and (AP x)
                  (integerp i)
                  (<= 0 i)
                  (< i (len x)))
             (integerp (nth i x)))).
  ~ev[]

  Similarly, 
  ~bv[]
  (implies (and (integerp i)
                (<= 0 i)
                (< i (A-LENGTH $S))
                (integerp v)
                (<= 0 v)
                (<= v 9))
           ($SP (UPDATE-AI i v $S)))
  ~ev[]
  is not a theorem until you add the additional hypothesis ~c[($SP $S)].
  To prove the resulting theorem, you will need a lemma such as the
  following.
  ~bv[]
  (defthm ap-update-nth
    (implies (and (AP a)
                  (integerp v)
                  (<= 0 v)
                  (<= v 9)
                  (integerp i)
                  (<= 0 i)
                  (< i (len a)))
             (AP (update-nth i v a))))
  ~ev[]

  The moral here is that from the logical perspective, you must
  provide the hypotheses that, as a programmer, you think are
  implicit on the structure of your stobjs, and you must prove their
  invariance.  This is a good area for further support, perhaps in
  the form of a library of macros.

  ~em[Resizing Array Fields]

  Recall the specification of the array field, ~c[A] for the stobj ~c[$S]
  introduced above:
  ~bv[]
  (a :type (array (integer 0 9) (3)) :initially 9 :resizable t)
  ~ev[]
  Logically, this field is a list, initially of length 3.  Under the
  hood, this field is implemented using a Common Lisp array with 3
  elements.  In some applications, one may wish to lengthen an array
  field, or even (to reclaim space) to shrink an array field.  The
  ~ilc[defstobj] event provides functions to access the current length
  of an array field and to change the array field, with default names
  obtained by suffixing the field name with ``~c[LENGTH-]'' or prefixing
  it with ``~c[RESIZE-],'' respectively.  The following log shows the uses
  of these fields in the above example.
  ~bv[]
  ACL2 !>(A-LENGTH $S)
  3
  ACL2 !>(RESIZE-A 10 $S) ; change length of A to 10
  <$s>
  ACL2 !>(A-LENGTH $S)
  10
  ACL2 !>(AI 7 $S)        ; new elements get value from :initially
  9
  ACL2 !>(RESIZE-A 2 $S)  ; truncate A down to first 2 elements
  <$s>
  ACL2 !>(A-LENGTH $S)
  2
  ACL2 !>(AI 7 $S)        ; error:  access past array bound


  ACL2 Error in TOP-LEVEL:  The guard for the function symbol AI, which
  is (AND ($SP $S) (INTEGERP I) (<= 0 I) (< I (A-LENGTH $S))), is violated
  by the arguments in the call (AI 7 $S).

  ACL2 !>
  ~ev[]
  Here are the definitions of the relevant functions for the above
  example; also ~pl[resize-list].
  ~bv[]
  (DEFUN A-LENGTH ($S)
    (DECLARE (XARGS :GUARD ($SP $S) :VERIFY-GUARDS T))
    (LEN (NTH 1 $S)))

  (DEFUN RESIZE-A (K $S)
    (DECLARE (XARGS :GUARD ($SP $S) :VERIFY-GUARDS T))
    (UPDATE-NTH 1
                (RESIZE-LIST (NTH 1 $S) K 9)
                $S))
  ~ev[]

  It is important to note that the implementation of array resizing in
  ACL2 involves copying the entire array into a newly allocated space
  and thus can be quite costly if performed often.  This approach was
  chosen in order to make array access and update as efficient as
  possible, with the suspicion that for most applications, array
  access and update are considerably more frequent than resizing
  (especially if the programmer is aware of the relative costs
  beforehand).

  It should also be noted that computations of lengths of stobj array
  fields should be fast (constant-time) in all or most Common Lisp
  implementations.

  Finally, if ~c[:resizable t] is not supplied as shown above, then
  an attempt to resize the array will result in an error.  If you do
  not intend to resize the array, it is better to omit the ~c[:resizable]
  option (or to supply ~c[:resizable nil]), since then the length
  function will be defined to return a constant, namely the initial
  length, which can simplify guard proofs (compare with the definition
  of ~c[A-LENGTH] above).

  This completes the tour through the documentation of ~il[stobj]s.
  However, you may now wish to read the documentation for the event
  that introduces a new single-threaded object; ~pl[defstobj].~/")

(defdoc resize-list
  ":Doc-Section Stobj

  list resizer in support of stobjs~/

  ~c[(Resize-list lst n default-value)] takes a list, ~c[lst], and a desired
  length, ~c[n], for the result list, as well as a ~c[default-value] to use
  for the extra elements if ~c[n] is greater than the length of ~c[lst].~/

  ~c[Resize-list] has a guard of ~c[t].  This function is called in the body
  of function, ~c[resize-<a>] where ~c[<a>] is an array field of a ~il[stobj].
  ~l[stobj] and ~pl[defstobj].~/")

#-acl2-loop-only
(defun mv-let-for-with-local-stobj (mv-let-form st creator w)

; If w is not nil, then it is the current ACL2 world and we are to oneify the
; appropriate subforms.

; It was tempting to have an acl2-loop-only version of the body below as well,
; which would omit the binding of the live var.  But if someone were to
; verify-termination of this function, we could presumably prove nil using the
; discrepancy between the two versions.  So we take the attitude that
; with-local-stobj is a special form, like let, that is not defined.

  (let ((producer (caddr mv-let-form))
        (rest (cdddr mv-let-form)))
    `(mv-let ,(cadr mv-let-form)
             (let ((,st (,creator)))

; We bind the live var so that user-stobj-alist-safe can catch misguided
; attempts to use functions like trans-eval in inappropriate contexts.

               (let ((,(the-live-var st) ,st))
                 ,(if w (oneify producer w) producer)))
             (declare (ignore ,st))
             ,@(if w
                   (if (cdr rest) ; rest is ((declare (ignore ...)) body)
                       (list (car rest) (oneify (cadr rest) w))
                     (list (oneify (car rest) w)))
                 rest))))

#-acl2-loop-only ; see the comment in mv-let-for-with-local-stobj
(defmacro with-local-stobj (&rest args)

; Below are some tests of local stobjs.
#|
 (defstobj foo bar xxx)

 (thm (equal (create-foo) '(nil nil))) ; succeeds

 (defun up1 (x foo)
   (declare (xargs :stobjs foo))
   (update-bar x foo))

 (bar foo) ; nil

 (up1 3 foo) ; <foo>

 (bar foo) ; 3

 (defun test (x) ; should fail; must use with-local-stobj explicitly
   (mv-let (a b foo)
           (let ((foo (create-foo)))
             (let ((foo (up1 (1+ x) foo)))
               (mv (bar foo) (xxx foo) foo)))
           (declare (ignore foo))
           (mv a b x)))

 (defun test (x)
   (declare (xargs :guard (acl2-numberp x) :verify-guards nil))
   (with-local-stobj
    foo
    (mv-let (a b foo)
            (let ((foo (up1 (1+ x) foo)))
              (mv (bar foo) (xxx foo) foo))
            (mv a b x))))

 (test 17) ; (18 NIL 17)

 (bar foo) ; 3

 (thm (equal (test x) (list (1+ x) nil x))) ; succeeds

 (thm (equal (test x) (list (1+ x) nil x)) ; succeeds
      :hints (("Goal"
               :in-theory
               (enable
                (:executable-counterpart create-foo)))))

 (thm (equal (test x) (list (1+ x) nil x)) ; fails, creating (NOT (NTH 1 (HIDE (CREATE-FOO))))
      :hints (("Goal"
               :in-theory
               (set-difference-theories
                (enable
                 (:executable-counterpart create-foo))
                '(create-foo)))))

 (verify-guards test)

 (test 17) ; (18 nil 17)

 (bar foo) ; 3

 (defun test2 (x)
   (with-local-stobj
    foo
    (mv-let (a foo)
            (let ((foo (up1 (1+ x) foo))) (mv (bar foo) foo))
            (mv a x))))

 (test2 12) ; (13 12)

 (bar foo) ; 3

 (thm (equal (test x) (mv-let (x y) (test2 x) (mv x nil y)))) ; succeeds

 (create-foo) ; should get graceful error

 (defun test3 (x) ; Should be OK.
   (with-local-stobj
    foo
    (mv-let (a foo)
            (let ((foo (up1 (1+ x) foo))) (mv (bar foo) foo))
            a)))

 (test3 11) ; 12

 (bar foo) ; 3

 (defun test4 (x foo) ; Should be OK.
   (declare (xargs :stobjs foo
                   :verify-guards nil))
   (let* ((x+1
          (with-local-stobj
           foo
           (mv-let (a foo)
                   (let ((foo (up1 (1+ x) foo))) (mv (bar foo) foo))
                   a)))
          (foo (up1 92 foo)))
     (mv x+1 foo)))

 (test4 19 foo) ; (20 <foo>)

 (bar foo) ; 92

 (defun test5 (x foo) ; Should be OK.
   (declare (xargs :stobjs foo
                   :verify-guards nil))
   (let* ((foo (up1 23 foo))
          (x+1
           (with-local-stobj
            foo
            (mv-let (a foo)
                    (let ((foo (up1 (1+ x) foo))) (mv (bar foo) foo))
                    a))))
     (mv x+1 foo)))

 (test5 35 foo) ; (36 <foo>)

 (bar foo) ; 23

 (with-local-stobj ; should get macroexpansion error or the equivalent
  foo
  (mv foo 3))

 (defun trans-eval-test (x foo state) ; this part is ok
   (declare (xargs :stobjs (foo state)
                   :mode :program))
   (mv-let (erp val state)
           (trans-eval '(update-bar (cons 3 (bar foo)) foo) 'top state)
           (declare (ignore erp val))
           (mv x foo state)))

 (with-local-stobj ; should fail; cannot use with-local-stobj in top level loop
  foo
  (mv-let (x foo state)
          (trans-eval-test 3 foo state)
          (mv x state)))

 (pprogn
  (with-local-stobj ; should fail with create-foo error
   foo
   (mv-let (x foo state)
           (trans-eval-test 3 foo state)
           (declare (ignore x))
           state))
  (mv 3 state))

 (defun test6 (a state)
   (declare (xargs :mode :program :stobjs state))
   (with-local-stobj
    foo
    (mv-let (x foo state)
            (trans-eval-test a foo state)
            (mv x state))))

 (test6 100 state) ; should get trans-eval error:  user-stobj-alist mismatch

 (bar foo) ; 23, still -- trans-eval did not affect global state

|#

; Below are some more tests, contributed by Rob Sumners.

#|

 (defstobj foo foo-fld)
 (defstobj bar bar-fld)

 (defun test-wls1 (x)
   (with-local-stobj
    foo
    (mv-let (result foo)
            (let ((foo (update-foo-fld 2 foo)))
              (mv (with-local-stobj
                   bar
                   (mv-let (result bar)
                           (let ((bar (update-bar-fld 3 bar)))
                             (mv x bar))
                           result))
                  foo))
            result)))

 (test-wls1 129) ; 129

 :comp t

 (test-wls1 '(adjka 202)) ; '(ADJKA 202)

 (thm (equal (test-wls1 x) x))

 (defun test-wls2 (x)
   (with-local-stobj
    foo
    (mv-let (result foo)
            (let ((foo (update-foo-fld 2 foo)))
              (mv (with-local-stobj
                   foo
                   (mv-let (result foo)
                           (let ((foo (update-foo-fld 3 foo)))
                             (mv x foo))
                           result))
                  foo))
            result)))

 (test-wls2 129) ; 129

 :comp t

 (test-wls2 '(adjka 202)) ; (ADJKA 202)

 (thm (equal (test-wls2 x) x))

 (defun test-wls3 (x)
   (if (atom x) x
     (with-local-stobj
      foo
      (mv-let (result foo)
              (mv (cons (car x)
                        (test-wls3 (cdr x)))
                  foo)
              (let ((x result))
                (if (atom x) x (cons (car x) (cdr x))))))))

 (test-wls3 129) ; 129

 :comp t

 (test-wls3 '(adjka 202)) ; (ADJKA 202)

 (thm (equal (test-wls3 x) x))

|#

  (mv-let (erp st mv-let-form creator)
          (parse-with-local-stobj args)
          (if (or erp
                  (not (and (true-listp mv-let-form)
                            (<= 3 (length mv-let-form)))))
              (er hard 'with-local-stobj
                  "Macroexpansion of a with-local-stobj call caused an error. ~
                   See :DOC with-local-stobj.")
            (mv-let-for-with-local-stobj mv-let-form st creator nil))))

(defdoc with-local-stobj
  ":Doc-Section Stobj

  locally bind a single-threaded object~/

  ~l[stobj] for an introduction to single-threaded objects.
  ~bv[]
  Example Form:
  (with-local-stobj
   st
   (mv-let (result st)
           (compute-with-st x st)
           result))
  ~ev[]
  ~c[With-local-stobj] can be thought of as a macro, where the example
  form above expands as follows.
  ~bv[]
  (mv-let (result st)
          (let ((st (create-st)))
            (compute-with-st x st))
          (declare (ignore st))
          result)
  ~ev[]
  However, ACL2 expects you to use ~c[with-local-stobj], not its
  expansion.  More precisely, stobj creator functions are not allowed
  except (implicitly) via ~c[with-local-stobj] and in logic-only
  situations (like theorems and hints).  Moreover, neither
  ~c[with-local-stobj] nor its expansions are legal when typed directly at
  the top-level loop.~/
  ~bv[]
  General Forms:
  (with-local-stobj stobj-name mv-let-form)
  (with-local-stobj stobj-name mv-let-form creator-name)
  ~ev[]
  where ~c[stobj-name] is the name of a ~il[stobj] other than ~ilc[state],
  ~c[mv-let-form] is a call of ~ilc[mv-let], and if ~c[creator-name] is supplied
  then it should be the  name of the creator function for ~c[stobj-name];
  ~pl[defstobj].  For the example form above, its expansion would
  use ~c[creator-name], if supplied, in place of ~c[create-st].

  ~c[With-local-stobj] can be useful when a stobj is used to memoize
  intermediate results during a computation, yet it is desired not to
  make the ~c[stobj] a formal parameter for the function and its
  callers.

  ACL2 can reason about these ``local stobjs,'' and in particular
  about stobj creator functions.  For technical reasons, ACL2 will not
  allow you to enable the ~c[:EXECUTABLE-COUNTERPART] ~il[rune] of a stobj
  creator function.~/")

(defun push-untouchable-fn (name state doc event-form)
  (when-logic
   "PUSH-UNTOUCHABLE"
   (with-ctx-summarized
    (if (output-in-infixp state)
        event-form
      (cond ((symbolp name)
             (cond ((null doc)
                    (msg "(PUSH-UNTOUCHABLE ~x0)" name))
                   (t (msg "(PUSH-UNTOUCHABLE ~x0 ...)" name))))
            ((null doc) "(PUSH-UNTOUCHABLE ...)")
            (t "(PUSH-UNTOUCHABLE ... ...)")))
    (let ((wrld (w state))
          (event-form (or event-form
                          (list* 'push-untouchable name
                                 (if doc
                                     (list :doc doc)
                                   nil))))
          (names (if (symbolp name) (list name) name)))
      (er-let*
        ((doc-pair (translate-doc nil doc ctx state)))

; With no name to hang it on, we don't permit a formatted doc string.
; So the above causes an error if the string is formatted.  Otherwise,
; we ignore doc-pair.

        (cond
         ((not (symbol-listp names))
          (er soft ctx
              "The argument to push-untouchable must be either a non-nil
              symbol or a non-empty true list of symbols and ~x0 is ~
              neither."
              name))
         ((subsetp-eq names (global-val 'untouchables wrld))
          (stop-redundant-event state))
         (t
          (let ((wrld
                 (global-set 'untouchables
                             (union-eq names (global-val 'untouchables wrld))
                             wrld)))
            (install-event name
                           event-form
                           'push-untouchable
                           0
                           nil
                           nil
                           wrld state)))))))))

; Section:  trace/untrace, with-error-trace (wet).

(defdoc trace
  ":Doc-Section Trace

  tracing functions in ACL2~/

  ACL2 provides utilities that rely on the underlying Lisp image to trace
  functions.  There are two interfaces to the underlying lisp trace:~bq[]

  o Macros ~ilc[trace$] and ~ilc[untrace$] call the underlying Lisp's ~c[trace]
  and ~c[untrace], respectively.  ~l[trace$] and ~pl[untrace$].

  o Macro ~ilc[with-error-trace], or ~ilc[wet] for short, provides a backtrace
  showing function calls that lead to an error.  ~l[wet].

  ~eq[]NOTES:

  1. ~ilc[wet] turns off all tracing (i.e., executes Lisp ~c[(untrace)]) other
  than temporarily doing some tracing under-the-hood in the evaluation of the
  form supplied to it.

  2. The underlying Lisp ~c[trace] and ~c[untrace] utilities have been modified
  for GCL and Allegro CL to trace the executable counterparts.  Other Lisps may
  give unsatisfying results.  For GCL and Allegro CL, you can invoke the
  original ~c[trace] and ~c[untrace] by exiting the ACL2 loop and invoking
  ~c[old-trace] and ~c[old-untrace], respectively..

  3. Trace output for ~ilc[trace$] and ~ilc[untrace$] can be redirected to a
  file.  ~l[open-trace-file] and ~pl[close-trace-file].  However, the backtrace
  printed by ~ilc[wet] always goes to ~ilc[standard-co].~/~/")

#-acl2-loop-only
(progn ; general tracing support

#-(or gcl clisp)
; Keep the above feature check in sync with the ones in trace-ppr and
; with-error-trace-fn.  Note that *trace-level* is already defined in gcl and
; clisp.
(defparameter *trace-level* 0)

(defparameter *trace-evisc-tuple*
  '(nil nil nil nil))  ;;; (evisc-tuple nil nil nil nil)

; Trace prints using the Lisp values of *print-level* and
; *print-length*.

(defun trace-evisc-tuple ()
  (if (and (eq (caaar *trace-evisc-tuple*)
               (w *the-live-state*))
           (eql *print-level* (cadr *trace-evisc-tuple*))
           (eql *print-length* (caddr *trace-evisc-tuple*)))
      *trace-evisc-tuple*
    (reset-trace-evisc-tuple)))

; Each time one traces a function, the *trace-evisc-tuple* stores the
; current *print-level* and *print-length* with the current world.

(defun reset-trace-evisc-tuple ()
  (setq *trace-evisc-tuple*
        (list (world-evisceration-alist *the-live-state* nil)
              *print-level*
              *print-length*
              (car (cddddr *trace-evisc-tuple*)))))

(defun trace-ppr (x)
  (let ((trace-evisc-tuple (trace-evisc-tuple)))
    (ppr (eviscerate x
                     (caddr trace-evisc-tuple)         ;;; print-level
                     (cadddr trace-evisc-tuple)        ;;; print-length
                     (car trace-evisc-tuple)           ;;; alist
                     (car (cddddr trace-evisc-tuple))) ;;; hiding-cars
         (min (+ 3
                 (* #-(or gcl clisp) *trace-level*
                    #+(or gcl clisp) system::*trace-level*
                    2))
              20)
         (f-get-global 'trace-co *the-live-state*)
         *the-live-state*
         t)))

)

(defun open-trace-file-fn (filename state)

; Logically, this function opens a channel to the given file.  But there is no
; logical accounting for subsequent writes to that channel on behalf of
; tracing.  We view those subsequent writes as being to the file, but not the
; channel, in analogy to how cw prints to the screen but not not modify the
; contents of *standard-co*.

  (mv-let (chan state)
          (open-output-channel filename :character state)
          (cond
           (chan #-acl2-loop-only
                 (setq *trace-output*
                       (get-output-stream-from-channel chan))
                 (pprogn
                  (if (equal (f-get-global 'trace-co state) *standard-co*)
                      state
                    (close-output-channel (f-get-global 'trace-co state)
                                          state))
                  (f-put-global 'trace-co chan state)))
           (t (prog2$
               (er hard 'open-trace-file
                   "Unable to open file ~s0 for trace output."
                   filename)
               state)))))

(defmacro open-trace-file (filename)

  #-small-acl2-image
  ":Doc-Section Trace

  redirect trace output to a file~/
  ~bv[]
  Example:
  (open-trace-file \"foo\") ; trace output will go to file foo~/

  General Form:
  (open-trace-file filename) ; trace output will go to file filename
  ~ev[]

  Output from ~ilc[trace$] normally goes to the screen, i.e.,
  ~ilc[standard-co].  But it can be redirected to a file as shown above.
  ~l[close-trace-file] for how to send trace output back to the screen.

  Note that the backtrace printed by ~ilc[wet] always goes to
  ~ilc[standard-co], even after the use of ~c[open-trace-file].~/"

  (declare (xargs :guard (stringp filename)))
  `(open-trace-file-fn ,filename state))

(defun close-trace-file-fn (state)
  #-acl2-loop-only
  (setq *trace-output* (get-output-stream-from-channel *standard-co*))
  (if (equal (f-get-global 'trace-co state) *standard-co*)
      (prog2$
       (er hard 'close-trace-file
           "No change: trace is already written to standard output.~%")
       state)
    (pprogn (close-output-channel (f-get-global 'trace-co state) state)
            (f-put-global 'trace-co *standard-co* state))))

(defmacro close-trace-file ()

  #-small-acl2-image
  ":Doc-Section Trace

  stop redirecting trace output to a file~/
  ~bv[]
  General Form:
  (close-trace-file) ; trace output is no longer redirected to a file~/
  ~ev[]

  Output from ~ilc[trace$] normally goes to the screen, or more precisely,
  ~ilc[standard-co].  It can be redirected to a file; ~pl[open-trace-file].
  Use ~c[close-trace-file] to redirect trace output to ~ilc[standard-co].~/"

  '(close-trace-file-fn state))

; Here we develop the code for wet (with-error-trace).

#-acl2-loop-only
(progn

(defun push-error-trace-stack (name arglist)

; It seems like a good idea to create this function.  Its compilation,
; including macroexpansion, may speed up its evaluation during a trace.  It
; needs to return nil even though it modifies state, so we go ahead and put it
; outside the ACL2 loop.

  (f-put-global
   'error-trace-stack
   (cons (cons name arglist)
         (f-get-global 'error-trace-stack *the-live-state*))
   *the-live-state*)
  nil)

(defun pop-error-trace-stack ()
  (f-put-global
   'error-trace-stack
   (cdr (f-get-global 'error-trace-stack *the-live-state*))
   *the-live-state*))

#+gcl
(defun with-error-trace-fns-forms-rec (names)
  (if (endp names)
      nil
    (let* ((name (car names))
           (*1*name (*1*-symbol name)))
      `((,name
         :entrycond
         (push-error-trace-stack ',name si::arglist)
         :exitcond
         (pop-error-trace-stack))
        (,*1*name
         :entrycond
         (push-error-trace-stack ',*1*name si::arglist)
         :exitcond
         (pop-error-trace-stack))
        ,@(with-error-trace-fns-forms-rec (cdr names))))))

#+clisp
(defun with-error-trace-fns-forms-rec (names)
  (if (endp names)
      nil
    (let* ((name (car names))
           (*1*name (*1*-symbol name)))
      `((,name
         :suppress-if t
         :pre
         (push-error-trace-stack ',name ext:*trace-args*)
         :post
         (pop-error-trace-stack))
        (,*1*name
         :suppress-if t
         :pre
         (push-error-trace-stack ',*1*name ext:*trace-args*)
         :post
         (pop-error-trace-stack))
        ,@(with-error-trace-fns-forms-rec (cdr names))))))

#+allegro
(defun with-error-trace-fns-forms (names)
  (if (endp names)
      nil
    (let* ((name (car names))
           (*1*name (*1*-symbol name)))
      `((excl:unadvise ,name)
        (excl:unadvise ,*1*name)
        (excl:advise ,name :before nil nil
                     (push-error-trace-stack ',name si::arglist))
        (excl:advise ,*1*name :before nil nil
                     (push-error-trace-stack ',*1*name si::arglist))
        (excl:advise ,name :after nil nil
                     (pop-error-trace-stack))
        (excl:advise ,*1*name :after nil nil
                     (pop-error-trace-stack))
        ,@(with-error-trace-fns-forms (cdr names))))))

#+(or gcl clisp)
(defun with-error-trace-fns-forms (names)
  `((trace ,@(with-error-trace-fns-forms-rec names))))

#-(or allegro gcl clisp)
(defun with-error-trace-fns-forms (names)
  (declare (ignore names))
  (er hard 'with-error-trace
      "It is illegal to use wet (with-error-trace) in this Common Lisp ~
       implementation.  If you want to see it supported for this Common Lisp, ~
       please contact the implementors."))

)

(defun executable-ancestors (flg fn wrld acc)

; Here flg is nil if fn is a single function, else fn is a list of functions.

  (cond
   (flg (if (null fn)
            acc
          (executable-ancestors
           flg (cdr fn) wrld
           (executable-ancestors nil (car fn) wrld acc))))
   ((member-eq fn acc) acc)
   ((equal (symbol-package-name fn) *main-lisp-package-name*)

; Should we exclude other functions that might get called a lot?

    acc)
   (t
    (mv-let (name x)
            (constraint-info fn wrld)
            (declare (ignore x))
            (cond
             (name acc)
             (t (let ((body (getprop fn 'unnormalized-body nil
                                     'current-acl2-world wrld)))
                  (cond
                   (body (executable-ancestors t (all-fnnames body) wrld
                                               (cons fn acc)))
                   (t acc)))))))))

#+(and allegro (not acl2-loop-only))
(defun trace-ppr-allegro (direction x)

; We need to provide all the output that one expects when using a trace
; facility.  Hence the cond clause and the first argument.

  (cond ((eq direction :in)

; Originally we incremented the trace level here.  But instead we wait until
; calling trace-ppr, in order to get the spacing to work out.

         (case *trace-level*
           (0 (princ "1> " *trace-output*))
           (1 (princ "  2> " *trace-output*))
           (2 (princ "    3> " *trace-output*))
           (3 (princ "      4> " *trace-output*))
           (4 (princ "        5> " *trace-output*))
           (5 (princ "          6> " *trace-output*))
           (6 (princ "            7> " *trace-output*))
           (7 (princ "              8> " *trace-output*))
           (8 (princ "                9> " *trace-output*))
           (t (princ (format nil "                  ~s >" *trace-level*)
                     *trace-output*))))
        (t
         (case *trace-level*
           (1 (princ "<1 " *trace-output*))
           (2 (princ "  <2 " *trace-output*))
           (3 (princ "    <3 " *trace-output*))
           (4 (princ "      <4 " *trace-output*))
           (5 (princ "        <5 " *trace-output*))
           (6 (princ "          <6 " *trace-output*))
           (7 (princ "            <7 " *trace-output*))
           (8 (princ "              <8 " *trace-output*))
           (9 (princ "                <9 " *trace-output*))
           (t (princ (format nil "                  <~s " *trace-level*)
                     *trace-output*)))
         (decf *trace-level*)))
  (trace-ppr x)
  (when (eq direction :in) (incf *trace-level*))
  (princ #\Newline *trace-output*)
  (finish-output *trace-output*))

(defun with-error-trace-fn (term fnlist omitfnlist state)
  (declare (xargs :stobjs state :mode :program)
           #+acl2-loop-only
           (ignore fnlist omitfnlist))
  #+(or acl2-loop-only no-hack)
  (er-let* ((val (trans-eval term 'with-error-trace state)))
    (value (cdr val)))
  #-(or acl2-loop-only no-hack)
  (let* ((fnlist0
          (or fnlist
              (mv-let (erp val bindings state)
                      (translate1 term :stobjs-out
                                  '((:stobjs-out . :stobjs-out)) t
                                  'top-level (w state) state)
                      (declare (ignore bindings))
                      (if erp
                          (er hard 'with-error-trace "With-error-trace failed.")
                        (executable-ancestors t (all-fnnames val) (w state)
                                              nil)))))
         (fnlist (if omitfnlist
                     (set-difference-eq fnlist0 omitfnlist)
                   fnlist0)))
    (prog1
        (acl2-unwind-protect
         "with-error-trace"
         (progn (eval '(untrace)) ; use current (not compile-time) untrace
                (eval (cons 'progn (with-error-trace-fns-forms fnlist)))
                (f-put-global 'error-trace-stack nil state)
                (er-let* ((val (trans-eval term 'with-error-trace state)))
                  (value (cdr val))))
         (progn #-(or gcl clisp) (setq *trace-level* 0)
                (let (#-allegro (level 0)
                                (*trace-output* (get-output-stream-from-channel
                                                 (standard-co state))))
                  (dolist (entry (reverse (f-get-global 'error-trace-stack state)))
                          (state-global-let*
                           ((trace-co (standard-co state)))
                           #+allegro
                           (progn
                             (trace-ppr-allegro
                              :in
                              (cons (car entry)
                                    (trace-hide-world-and-state
                                     (cdr entry))))

; Avoid compiler warning by Allegro:

                             state)
                           #-allegro
                           (progn (spaces (min 20 (* 2 level))
                                          0
                                          (f-get-global 'trace-co state)
                                          state)
                                  (setq level (1+ level))
                                  (fmt1 "~x0> "
                                        (list (cons #\0 level))
                                        0 (standard-co state) state nil)
                                  (trace-ppr (cons (car entry)
                                                   (trace-hide-world-and-state
                                                    (cdr entry))))
                                  (newline (standard-co state) state)))))
                (newline (f-get-global 'trace-co state) state))
         state)
      (progn
        (eval '(untrace)) ; use current (not compile-time) untrace
        #-(or gcl clisp) (setq *trace-level* 0)
        (f-put-global 'error-trace-stack nil state)))))

(defmacro with-error-trace (term &key fns omit)

  #-small-acl2-image
  ":Doc-Section Trace

  evaluate a form and print subsequent error trace~/

  ~c[With-error-trace] is the same as ~c[wet]; ~pl[wet].~/~/"

  (declare (xargs :guard (and (symbol-listp fns)
                              (symbol-listp omit))))
  `(with-error-trace-fn ',term ',fns ',omit state))

(defmacro wet (term &key fns omit)

  #-small-acl2-image
  ":Doc-Section Trace

  evaluate a form and print subsequent error trace~/
  NOTE:  This feature is onlyh available if you are using GCL, Allegro CL, or
  CLISP.
  ~bv[]
  Examples:
  (wet (bar 3))            ; evaluate (bar 3) and print backtrace upon error
  (wet (bar 3) :fns (f g)) ; as above but only include calls of f, g~/

  General Forms:
  (wet form)
  (wet form :fns  (fn1 fn2 ... fnk))
  (wet form :omit (fn1 fn2 ... fnk))
  ~ev[]
  where ~c[form] is an arbitrary ACL2 form and the ~c[fni] are function symbols
  whose calls are to appear in the backtrace if the evaluation of ~c[form]
  aborts.  If ~c[fns] are ~c[nil] or not supplied, then calls of all functions
  appear in the backtrace, with the exception of built-in functions that are
  either in the main Lisp package or are in ~c[:]~ilc[program] mode.  (In
  particular, all user-defined functions appear.)  The above description is
  modified if ~c[omit] is supplied, in which case calls of the specified
  function symbols are removed from the backtrace.

  The following example illustrates the use of ~c[wet], which stands for
  ``~ilc[with-error-trace]''.  We omit uninteresting output from this example.
  ~bv[]
  ACL2 !>(defun foo (x) (car x))
   ...
   FOO
  ACL2 !>(defun bar (x) (foo x))
   ...
   BAR
  ACL2 !>(bar 3)


  ACL2 Error in TOP-LEVEL:  The guard for the function symbol CAR, which
  is (OR (CONSP X) (EQUAL X NIL)), is violated by the arguments in the
  call (CAR 3).  To see a trace of calls leading up to this violation,
  execute (wet <form>) where <form> is the form you submitted to the
  ACL2 loop.  See :DOC wet for how to get an error backtrace.

  ACL2 !>(wet (bar 3))


  ACL2 Error in WITH-ERROR-TRACE:  The guard for the function symbol
  CAR, which is (OR (CONSP X) (EQUAL X NIL)), is violated by the arguments
  in the call (CAR 3).  (Backtrace is below.)

  1> (ACL2_*1*_ACL2::BAR 3)
    2> (ACL2_*1*_ACL2::FOO 3)

  ACL2 !>(wet (bar 3) :fns (foo))



  ACL2 Error in WITH-ERROR-TRACE:  The guard for the function symbol
  CAR, which is (OR (CONSP X) (EQUAL X NIL)), is violated by the arguments
  in the call (CAR 3).  (Backtrace is below.)

  1> (ACL2_*1*_ACL2::FOO 3)

  ACL2 !>
  ~ev[]
  Notice that because guards were not verified, the so-called
  ~il[executable-counterpart] functions are evaluated for ~c[foo] and
  ~c[bar].  These can be identified with package names beginning with the
  string \"ACL2_*1*_\".

  ~l[trace$] for a general tracing utility.

  NOTES:

  1. Recursive calls of ~il[executable-counterpart] functions will not
  generally be traced.

  2. In the (probably rare) event of a hard Lisp error, you will have to exit
  the Lisp break before seeing the backtrace.

  3. ~c[Wet] always untraces all functions before it installs the traces it
  needs, and it leaves all functions untraced when it completes.  If existing
  functions were traced then you will need to re-execute ~ilc[trace$] in order
  to re-install tracing on those functions after ~c[wet] is called on any form.

  4. ~c[Wet] returns an error triple ~c[(mv error-p value state)], where
  ~c[value] is a print representation of the value returned by the form given
  to ~c[wet].  Presumably ~c[value] is not particularly important anyhow, as
  the intended use of ~c[wet] is for the case that an error occurred in
  evaluation of a form.

  5. As mentioned above, functions in the main Lisp package (i.e., those built
  into Common Lisp) will not be traced be ~c[wet].~/"

  `(with-error-trace ,term :fns ,fns :omit ,omit))

(defun trace$-fn (fns)

; We declare :mode :program so that the Common Lisp function is called even if
; we ultimately put this in a file in :logic mode (in which case we could
; allow :mode :logic as long as we verify guards).

  (declare (xargs :mode :program)) ; so that Common Lisp function is called
  #+(or no-hack acl2-loop-only)
  (declare (ignore fns))
  #-(or no-hack acl2-loop-only)
  (eval (cons 'trace fns))
  nil)

(defmacro trace$ (&rest fns)

  #-small-acl2-image
  ":Doc-Section Trace

  trace the indicated functions~/
  ~bv[]
  Example:
  (trace$ foo bar)~/

  General Form:
  (trace$ fn1 fn2 ... fnk)
  ~ev[]
  where the ~c[fni] are defined or even constrained functions.

  ~pl[untrace$] for how to undo the effect of ~ilc[trace$].

  Basically, ~c[trace$] calls on the underlying Lisp to trace the specified
  functions as well as their ~ilc[executable-counterpart]s.  However, for GCL
  and Allegro CL the underlying Lisp trace routines are modified before an
  image is saved in order to hide the ACL2 world and other large data
  structures and provide slightly prettier output.

  Recursive calls of ~il[executable-counterpart] functions will not generally
  be traced.

  Output from ~ilc[trace$] normally goes to the screen, i.e.,
  ~ilc[standard-co].  But it can be redirected to a file;
  ~pl[open-trace-file].

  Also ~pl[wet] (``~ilc[with-error-trace]'') for a different way that ACL2
  takes advantage of the underlying Lisp, namely to provide a backtrace when
  there is an error.

  Note that from a logical perspective all trace printing is a fiction.  For a
  related fiction, ~pl[cw].  ~c[Trace$] returns ~c[nil]~/"

  `(trace$-fn ',fns))

(defun untrace$-fn (fns)

; We declare :mode :program so that the Common Lisp function is called even if
; we ultimately put this in a file in :logic mode (in which case we could
; allow :mode :logic as long as we verify guards).

  (declare (xargs :mode :program))
  #+(or no-hack acl2-loop-only)
  (declare (ignore fns))
  #-(or no-hack acl2-loop-only)
  (eval (cons 'untrace fns))
  nil)

(defmacro untrace$ (&rest fns)

  #-small-acl2-image
  ":Doc-Section Trace

  untrace functions~/
  ~bv[]
  Examples:
  (untrace$)         ; untrace all traced functions
  (untrace$ foo bar) ; untrace foo and bar~/

  General Forms:
  (untrace$)                 ; untrace all traced functions
  (untrace$ fn1 fn2 ... fnk) ; untrace the indicated functions
  ~ev[]
  where the ~c[fni] are defined or even constrained functions.

  ~c[Untrace$] undoes the effect of ~ilc[trace$].  ~l[trace$].
  ~c[Untrace] returns ~c[nil]~/"

  `(untrace$-fn ',fns))

(defmacro break-on-error (&optional (on 't))

  #-small-acl2-image
  ":Doc-Section Trace

  break when encountering a hard or soft error caused by ACL2.~/
  ~bv[]
  General forms:
  (break-on-error t)   ; installs a trace causing a continuable error (break)
                       ;   whenever a hard or soft error is invoked by ACL2.
  (break-on-error)     ; same as above
  (break-on-error nil) ; uninstall the above trace
  ~ev[]
  ~c[(Break-on-error)] is actually a macro that expands as follows.
  ~bv[]
  (trace$ (illegal :entry (break))
          (error1  :entry (break)))
  ~ev[]
  This trace should cause entry to the Lisp debugger (at least in most Lisps)
  whenever ACL2 calls its error routines.

  Also ~pl[trace$].~/~/"

  (if on
      '(trace$ (illegal :entry (break))
               (error1  :entry (break)))
    '(untrace$ illegal error1)))

(defun defexec-extract-key (x keyword result result-p)

; X is a keyword-value-listp from an xargs declaration, and result-p indicates
; whether we expect to see no further value of the indicated keyword (in which
; case we should return result and result-p unchanged if erp, below, is nil).
; We return (mv erp result result-p), where if erp is nil, result-p is nil
; coming in, and x contains (keyword result), then we return (mv nil result t).

  (declare (xargs :guard (and (keywordp keyword)
                              (keyword-value-listp x))))
  (cond ((endp x)
         (mv nil result result-p))
        (t (mv-let (erp result result-p)
             (defexec-extract-key (cddr x) keyword result result-p)
             (cond (erp (mv erp nil nil))
                   ((eq (car x) keyword)
                    (cond
                     (result-p (mv "more than one ~x0 has been specified"
                                  nil nil))
                     (t (mv nil (cadr x) t))))
                   (t (mv nil result result-p)))))))

(defun parse-defexec-dcls-1 (alist guard guard-p hints hints-p measure
                                   measure-p wfrel wfrel-p stobjs stobjs-p exec-xargs
                                   exec-test exec-default acc)

; We return (mv nil declare-form ...) as suggested in the first (endp) case
; below, where exec-xargs has been removed from alist in creating the declare
; form (the second returned value).

  (declare (xargs :guard (symbol-alistp alist)))
  (cond
   ((endp alist)
    (mv nil
        (cons 'declare (reverse acc))
        guard guard-p
        hints hints-p
        measure measure-p
        wfrel wfrel-p
        stobjs stobjs-p
        exec-xargs exec-test exec-default))
   (t (let* ((decl (car alist))
             (sym (car decl))
             (x (cdr decl)))
        (cond
         ((eq sym 'xargs)
          (cond
           ((keyword-value-listp x)
            (mv-let (erp guard guard-p)
              (defexec-extract-key x :GUARD guard guard-p)
              (cond
               (erp (mv erp nil nil nil nil nil nil nil nil nil nil nil nil nil
                        nil))
               (t (mv-let (erp hints hints-p)
                    (defexec-extract-key x :HINTS hints hints-p)
                    (cond
                     (erp (mv erp nil nil nil nil nil nil nil nil nil nil nil
                              nil nil nil))
                     (t (mv-let (erp measure measure-p)
                          (defexec-extract-key x :MEASURE measure measure-p)
                          (cond
                           (erp (mv erp nil nil nil nil nil nil nil nil nil nil
                                    nil nil nil nil))
                           (t (mv-let (erp wfrel wfrel-p)
                                (defexec-extract-key x :WELL-FOUNDED-RELATION
                                  wfrel wfrel-p)
                                (cond
                                 (erp (mv erp nil nil nil nil nil nil nil nil
                                          nil nil nil nil nil nil))
                                 (t (mv-let (erp stobjs stobjs-p)
                                      (defexec-extract-key x :STOBJS stobjs
                                        stobjs-p)
                                      (cond
                                       (erp (mv erp nil nil nil nil nil nil nil
                                                nil nil nil nil nil nil nil))
                                       (t (parse-defexec-dcls-1
                                           (cdr alist)
                                           guard guard-p
                                           hints hints-p
                                           measure measure-p
                                           wfrel wfrel-p
                                           stobjs stobjs-p
                                           exec-xargs exec-test exec-default
                                           (cons decl acc))))))))))))))))))
           (t (mv "we found (XARGS . x) where x is not a keyword-value-listp"
                  nil nil nil nil nil nil nil nil nil nil nil nil nil nil))))
         ((eq sym 'exec-xargs)
          (cond
           ((or exec-xargs exec-test exec-default)
            (mv "more than one EXEC-XARGS has been specified"
                nil nil nil nil nil nil nil nil nil nil nil nil nil nil))
           ((and (keyword-value-listp x) x)
            (let* ((exec-test (cadr (assoc-keyword :test x)))
                   (x (if exec-test (remove-keyword :test x) x))
                   (exec-default (cadr (assoc-keyword :default-value x)))
                   (x (if exec-default (remove-keyword :default-value x) x)))
              (parse-defexec-dcls-1 (cdr alist)
                                    guard guard-p
                                    hints hints-p
                                    measure measure-p
                                    wfrel wfrel-p
                                    stobjs stobjs-p
                                    x
                                    exec-test
                                    exec-default
                                    acc)))
           (t (mv "we found declaration (EXEC-XARGS . x) where x is not a ~
                   non-empty keyword-value-listp"
                  nil nil nil nil nil nil nil nil nil nil nil nil nil nil))))
         (t (parse-defexec-dcls-1 (cdr alist)
                                  guard guard-p
                                  hints hints-p
                                  measure measure-p
                                  wfrel wfrel-p
                                  stobjs stobjs-p
                                  x
                                  exec-test
                                  exec-default
                                  (cons (car alist) acc))))))))

(defun fix-exec-xargs (exec-xargs hints hints-p measure measure-p wfrel wfrel-p
                                  stobjs stobjs-p)
  (declare (xargs :guard (keyword-value-listp exec-xargs)))

; Update exec-xargs to incorporate the hints, measure, and stobjs extracted
; from the xargs (if any).

  (let* ((x (if (and hints-p (not (assoc-keyword :HINTS exec-xargs)))
                (list* :HINTS hints exec-xargs)
              exec-xargs))
         (x (if (and measure-p (not (assoc-keyword :MEASURE exec-xargs)))
                (list* :MEASURE measure x)
              x))
         (x (if (and wfrel-p (not (assoc-keyword :WELL-FOUNDED-RELATION
                                                 exec-xargs)))
                (list* :WELL-FOUNDED-RELATION wfrel x)
              x))
         (x (if (and stobjs-p (not (assoc-keyword :STOBJS exec-xargs)))
                (list* :STOBJS stobjs x)
              x)))
    x))

(defun parse-defexec-dcls (dcls-and-strings final guard guard-p hints hints-p
                                            measure measure-p wfrel wfrel-p
                                            stobjs stobjs-p exec-xargs
                                            exec-test exec-default)

; We return the following values.  Note that input guard-p is true if we have
; encountered a guard on an earlier call.

;  erp          - nil or a string that indicates an error
;  final        - what is left of dcls-and-strings after (exec-xargs ...) is
;                 removed
;  guard        - the guard from (xargs ... :guard ...)
;  exec-xargs   - the cdr of (exec-xargs ...) from input
;  exec-test    - from (exec-xargs ... :test ...) if present, else guard
;  exec-default - from (exec-xargs ... :default-value ...), else nil

  (cond
   ((endp dcls-and-strings)
    (cond
     ((null guard-p)
      (mv "no :GUARD has been specified in the XARGS" nil nil nil nil nil))
     (t
      (mv nil
          (reverse final)
          guard
          (fix-exec-xargs exec-xargs hints hints-p measure measure-p wfrel
                          wfrel-p stobjs stobjs-p)
          (or exec-test guard)
          exec-default))))
   (t (let ((x (car dcls-and-strings)))
        (cond
         ((stringp x)
          (parse-defexec-dcls (cdr dcls-and-strings) (cons x final) guard
                              guard-p hints hints-p measure measure-p wfrel
                              wfrel-p stobjs stobjs-p exec-xargs exec-test
                              exec-default))
         ((and (consp x)
               (eq (car x) 'declare)
               (symbol-alistp (cdr x)))
          (mv-let (erp decl guard guard-p hints hints-p measure measure-p wfrel
                       wfrel-p stobjs stobjs-p exec-xargs exec-test
                       exec-default)
            (parse-defexec-dcls-1 (cdr x) guard guard-p hints hints-p measure
                                  measure-p wfrel wfrel-p stobjs stobjs-p
                                  exec-xargs exec-test exec-default nil)
            (cond
             (erp (mv erp nil nil nil nil nil))
             (t (parse-defexec-dcls (cdr dcls-and-strings) (cons decl final)
                                    guard guard-p hints hints-p measure
                                    measure-p wfrel wfrel-p stobjs stobjs-p
                                    exec-xargs exec-test exec-default)))))
         (t
          (mv (msg "the form ~x0 is neither a string nor a form (declare . x) ~
                    where x is a symbol-alistp"
                   x)
              nil nil nil nil nil)))))))

(defmacro defexec (&whole whole fn formals &rest rest)

  #-small-acl2-image
  ":Doc-Section Events

  attach a terminating executable function to a definition~/
  
  Suppose you define a function ~c[(fn x)] with a ~il[guard] of
  ~c[(good-input-p x)], and you know that when the guard holds, the measure
  decreases on each recursive call.  Unfortunately, the definitional principle
  (~pl[defun]) ignores the guard.  For example, if the definition has the form
  ~bv[]
  (defun fn (x)
    (declare (xargs :guard (good-input-p x)))
    (if (not-done-yet x)
        (... (fn (destr x)) ...)
      ...))
  ~ev[]
  then in order to admit this definition, ACL2 must prove the appropriate
  formula asserting that ~c[(destr x)] is ``smaller than'' ~c[x] under the
  assumption ~c[(not-done-yet x)] but without the assumption
  ~c[(good-input-p x)], even if ~c[(not-done-yet x)] is true.  In essence, it
  may be necessary to submit instead the following definition. 
  ~bv[]
  (defun fn (x)
    (declare (xargs :guard (good-input-p x)))
    (if (good-input-p x)
        (if (not-done-yet x)
            (... (fn (destr x)) ...)
          ...)
      nil)
  ~ev[]
  But it is unfortunate that when calls of ~c[fn] are evaluated, for example
  when ~c[fn] is applied to an explicit constant during a proof, then a call of
  ~c[good-input-p] must now be evaluated on each recursive call.

  Fortunately, ~c[defexec] provides a way to keep the execution efficient.  For
  the example above we could use the following form.
  ~bv[]
  (defexec fn (x)
    (declare (xargs :guard (good-input-p x)))
    (mbe :logic (if (good-input-p x)
                    (if (not-done-yet x)
                        (... (fn (destr x)) ...)
                      ...)
                  nil)
         :exec  (if (not-done-yet x)
                    (... (fn (destr x)) ...)
                  ...)))
  ~ev[]
  Here ``~ilc[mbe]'' stands for ``must-be-equal'' and, roughly speaking, its
  call above is logically equal to the ~c[:logic] form but is evaluated using
  the ~c[:exec] form when the guard holds.  ~l[mbe].  The effect is thus to
  define ~c[fn] as shown in the ~ilc[defun] form above, but to cause execution
  of ~c[fn] using the ~c[:exec] body.  The use of ~c[defexec] instead of
  ~ilc[defun] in the example above causes a termination proof to be performed,
  in order to guarantee that evaluation always theoretically terminates, even
  when using the ~c[:exec] form for evaluation.
  ~bv[]
  Example:

  ; Some of the keyword arguments in the declarations below are irrelevant or
  ; unnecessary, but they serve to illustrate their use.

  (defexec f (x)
    (declare (xargs :measure (+ 15 (acl2-count x))
                    :hints ((\"Goal\" :in-theory (disable nth)))
                    :guard-hints ((\"Goal\" :in-theory (disable last)))
                    :guard (and (integerp x) (<= 0 x) (< x 25)))
             (exec-xargs
                    :test (and (integerp x) (<= 0 x))
                    :default-value 'undef ; defaults to nil
                    :measure (nfix x)
                    :well-founded-relation o<))
    (mbe :logic (if (zp x)
                    1
                  (* x (f (- x 1))))
         :exec  (if (= x 0)
                    1
                  (* x (f (- x 1))))))
  ~ev[]
  The above example macroexpands to the following.
  ~bv[]
  (ENCAPSULATE ()
   (LOCAL
    (ENCAPSULATE ()
     (SET-IGNORE-OK T)
     (SET-IRRELEVANT-FORMALS-OK T)
     (LOCAL (DEFUN F (X)
              (DECLARE
               (XARGS :VERIFY-GUARDS NIL
                      :HINTS ((\"Goal\" :IN-THEORY (DISABLE NTH)))
                      :MEASURE (NFIX X)
                      :WELL-FOUNDED-RELATION O<))
              (IF (AND (INTEGERP X) (<= 0 X))
                  (IF (= X 0) 1 (* X (F (- X 1))))
                  'UNDEF)))
     (LOCAL (DEFTHM F-GUARD-IMPLIES-TEST
              (IMPLIES (AND (INTEGERP X) (<= 0 X) (< X 25))
                       (AND (INTEGERP X) (<= 0 X)))
              :RULE-CLASSES NIL))))
   (DEFUN F (X)
     (DECLARE (XARGS :MEASURE (+ 15 (ACL2-COUNT X))
                     :HINTS ((\"Goal\" :IN-THEORY (DISABLE NTH)))
                     :GUARD-HINTS ((\"Goal\" :IN-THEORY (DISABLE LAST)))
                     :GUARD (AND (INTEGERP X) (<= 0 X) (< X 25))))
     (MBE :LOGIC
          (IF (ZP X) 1 (* X (F (- X 1))))
          :EXEC
          (IF (= X 0) 1 (* X (F (- X 1)))))))
  ~ev[]
  Notice that in the example above, the ~c[:]~ilc[hints] in the ~ilc[local]
  definition of ~c[F] are inherited from the ~c[:hints] in the ~ilc[xargs] of
  the ~c[defexec] form.  We discuss such inheritance below.
  ~bv[]
  General Form:
  (defexec fn (var1 ... varn) doc-string dcl ... dcl
    (mbe :LOGIC logic-body
         :EXEC  exec-body))
  ~ev[]
  where the syntax is identical to the syntax of ~ilc[defun] where the body is
  a call of ~c[mbe], with the exceptions described below.  Thus, ~c[fn] is the
  symbol you wish to define and is a new symbolic name and ~c[(var1 ... varn)]
  is its list of formal parameters (~pl[name]).  The first exception is that at
  least one ~c[dcl] (i.e., ~ilc[declare] form) must specify a ~c[:guard],
  ~c[guard].  The second exception is that one of the ~c[dcl]s is allowed to
  contain an element of the form ~c[(exec-xargs ...)].  The ~c[exec-xargs]
  form, if present, must specify a non-empty ~ilc[keyword-value-listp] each of
  whose keys is one of ~c[:test], ~c[:default-value], or one of the standard
  ~ilc[xargs] keys of ~c[:measure], ~c[:well-founded-relation], ~c[:hints], or
  ~c[:stobjs].  Any of these four standard ~c[xargs] keys that is present in an
  ~c[xargs] of some ~c[dcl] but is not specified in the (possibly nonexistent)
  ~c[exec-xargs] form is considered to be specified in the ~c[exec-xargs] form,
  as illustrated in the example above for ~c[:hints].  (So for example, if you
  want ~c[:hints] in the final, non-local definition but not in the local
  definition, then specify the ~c[:hints] in the ~c[xargs] but specify
  ~c[:hints nil] in the ~c[exec-xargs].)  If ~c[:test] is specified and not
  ~c[nil], let ~c[test] be its value; otherwise let ~c[test] default to
  ~c[guard].  If ~c[:default-value] is specified, let ~c[default-value] be its
  value; else ~c[default-value] is ~c[nil].  ~c[Default-value] should have the
  same ~il[signature] as ~c[exec-body]; otherwise the ~c[defexec] form will
  fail to be admitted.

  The above General Form's macroexpansion is of the form
  ~c[(PROGN encap final-def)], where ~c[encap] and ~c[final-def] are as
  follows.  ~c[Final-def] is simply the result of removing the ~c[exec-xargs]
  declaration (if any) from its ~ilc[declare] form, and is the result of
  evaluating the given ~c[defexec] form, since ~c[encap] is of the following
  form.
  ~bv[]
  ; encap
  (ENCAPSULATE ()
    (set-ignore-ok t)             ; harmless for proving termination
    (set-irrelevant-formals-ok t) ; harmless for proving termination
    (local local-def)
    (local local-thm))
  ~ev[]
  The purpose of ~c[encap] is to ensure the the executable version of ~c[name]
  terminates on all arguments.  Thus, ~c[local-def] and ~c[local-thm] are as
  follows, where the ~c[xargs] of the ~ilc[declare] form are the result of
  adding ~c[:VERIFY-GUARDS NIL] to the result of removing the ~c[:test] and
  (optional) ~c[:default-value] from the ~c[exec-xargs].
  ~bv[]
  ; local-def
  (DEFUN fn formals
    (DECLARE (XARGS :VERIFY-GUARDS NIL ...))
    (IF test
        exec-body
      default-value))

  ; local-thm
  (DEFTHM fn-EXEC-GUARD-HOLDS
    (IMPLIES guard test)
    :RULE-CLASSES NIL)
  ~ev[]
  We claim that if the above ~c[local-def] and ~c[local-thm] are admitted, then
  all evaluations of calls of ~c[fn] terminate.  The concern is that the use
  of ~ilc[mbe] in ~c[final-def] allows for the use of ~c[exec-body] for a call
  of ~c[fn], as well as for subsequent recursive calls, when ~c[guard] holds
  and assuming that the guards have been verified for ~c[final-def].  However,
  by ~c[local-thm] we can conclude in this case that ~c[test] holds, in which
  case the call of ~c[fn] may be viewed as a call of the version of ~c[fn]
  defined in ~c[local-def].  Moreover, since guards have been verified for
  ~c[final-def], then guards hold for subsequent evaluation of ~c[exec-body],
  and in particular for recursive calls of ~c[fn], which can thus continue to
  be viewed as calls using ~c[local=def].~/

  :cited-by mbe"
 
  (let ((dcls-and-strings (butlast rest 1))
        (body (car (last rest))))
    (mv-let (erp exec-body)
      (case-match body
        (('mbe ':logic & ':exec exec-body)
         (mv nil exec-body))
        (('mbe ':exec exec-body ':logic &)
         (mv nil exec-body))
        (('mbe . &)
         (mv 'mbe nil))
        (& (mv t nil)))
      (cond
       (erp `(er soft 'defexec
                 "A defexec form must have a body that is a valid call of mbe. ~
                  See :DOC ~s0."
                 ,(if (eq erp 'mbe) "mbe" "defexec")))
       ((not (symbolp fn))
        `(er soft 'defexec
             "The first argument of defexec must be a symbol, but ~x0 is not."
             ',fn))
       ((not (arglistp formals))
        `(er soft 'defexec
             "The second argument of defexec must be legal list of formals, ~
              but ~x0 is not."
             ',formals))
       (t (mv-let (erp final-dcls-and-strings guard exec-xargs exec-test
                       exec-default)
            (parse-defexec-dcls dcls-and-strings nil nil nil nil nil nil nil
                                nil nil nil nil nil nil nil)
            (cond
             (erp
              `(er soft 'defexec
                   "Macroexpansion of ~x0 has failed because ~@1."
                   ',whole
                   ',erp))
             (t `(encapsulate ()
                   (local
                    (encapsulate ()
                                 (set-ignore-ok t)
                                 (set-irrelevant-formals-ok t)
                                 (local (defun ,fn ,formals
                                          (declare (xargs :verify-guards nil
                                                          ,@exec-xargs))
                                          (if ,exec-test
                                              ,exec-body
                                            ,exec-default)))
                                 (local (defthm ,(packn
                                                  (list fn
                                                        '-GUARD-IMPLIES-TEST))
                                          (implies ,guard ,exec-test)
                                          :rule-classes nil))))
                   (defun ,fn ,formals
                     ,@final-dcls-and-strings
                     ,body))))))))))

; Start code for :pl and proof-checker show-rewrites command.

(defrec sar ; single-applicable-rewrite
  (lemma alist index)
  nil)

; Here's the idea.  Both showing and using of rewrites benefits from knowing
; which hypotheses are irrelevant.  But when rewriting in the proof-checker, we
; will try to do more, namely relieve all the hyps by instantiating free
; variables.  So we avoid doing any instantiation in forming the sar record.
; Actually, if we knew that rewriting were to be done with the empty
; substitution, then we'd go ahead and store the result of trying to relieve
; hypotheses at this point; but we don't.  Nevertheless, we should have a
; function that takes the fields of an sar record and returns an appropriate
; structure representing the result of trying to relieve the hyps (possibly
; starting with a unify-subst extending the one that was originally produced).

(defun applicable-rewrite-rules1 (term geneqv lemmas current-index
                                       target-name-or-rune target-index wrld)

; Call this initially with current-index equal to 1.

  (declare (xargs :guard (or (null target-index) (integerp target-index))))
  (if (consp lemmas)
      (let ((lemma (car lemmas)))
        ;; if the lemma needs to be considered, consder it
        (if (and (or (null target-name-or-rune)
                     (if (symbolp target-name-or-rune)
                         (equal target-name-or-rune
                                (cadr (access rewrite-rule lemma :rune)))
                       (equal target-name-or-rune
                              (access rewrite-rule lemma :rune))))
                 (member (access rewrite-rule lemma :subclass)
                         '(backchain abbreviation))
                 (geneqv-refinementp (access rewrite-rule lemma :equiv)
                                     geneqv
                                     wrld))
            (mv-let (flg alist)
                    (one-way-unify (access rewrite-rule lemma :lhs) term)
                    (if flg
                        (if target-index
                            (if (eql target-index current-index)
                                (list (make sar
                                            :index current-index
                                            :lemma lemma
                                            :alist alist))
                              (applicable-rewrite-rules1
                               term geneqv (cdr lemmas) (1+ current-index)
                               target-name-or-rune target-index wrld))
                          (cons (make sar
                                      :index (if target-name-or-rune
                                                 nil
                                               current-index)
                                      :lemma lemma
                                      :alist alist)
                                (applicable-rewrite-rules1
                                 term geneqv (cdr lemmas) (1+ current-index)
                                 target-name-or-rune target-index wrld)))
                      (applicable-rewrite-rules1
                       term geneqv (cdr lemmas) current-index
                       target-name-or-rune target-index wrld)))
          (applicable-rewrite-rules1
           term geneqv (cdr lemmas) current-index
           target-name-or-rune target-index wrld)))
    nil))

(defun pc-relieve-hyp (hyp unify-subst type-alist wrld state ens ttree)

; This function is adapted from ACL2 function relieve-hyp, but it prevents
; backchaining, instead returning the new hypotheses.  Notice that there are no
; arguments for obj, equiv, fnstack, ancestors, or simplify-clause-pot-lst.
; Also notice that rcnst has been replaced by ens (an enable structure).

; We return t or nil indicating whether we won, an extended unify-subst
; and a new ttree.  This function is a No-Change Loser.

  (cond ((f-big-clock-negative-p state)
         (mv nil unify-subst ttree))
        (t (let* ((forcep (and (nvariablep hyp)
                               (not (fquotep hyp))
                               (or (eq (ffn-symb hyp) 'force)
                                   (eq (ffn-symb hyp) 'case-split))))
                  (hyp (if forcep (fargn hyp 1) hyp)))
             (mv-let (lookup-hyp-ans unify-subst ttree)
                     (lookup-hyp hyp type-alist wrld
                                 unify-subst ttree)
                     (cond
                      (lookup-hyp-ans (mv t unify-subst ttree))
                      ((free-varsp hyp unify-subst)
                       (cond ((and (equalityp hyp)
                                   (variablep (fargn hyp 1))
                                   (not (assoc-eq (fargn hyp 1) unify-subst))
                                   (not (free-varsp (fargn hyp 2) unify-subst)))

; Unlike relieve-hyp, no rewriting is done.

                              (mv t
                                  (cons (cons (fargn hyp 1) (fargn hyp 2))
                                        unify-subst)
                                  ttree))
                             (t (search-ground-units
                                 hyp unify-subst
                                 type-alist
                                 ens
                                 (ok-to-force-ens ens)
                                 wrld ttree))))
                      (t
                       (let ((inst-hyp (sublis-var unify-subst hyp)))
                         (mv-let
                          (knownp nilp nilp-ttree)
                          (known-whether-nil inst-hyp type-alist ens
                                             (ok-to-force-ens ens) wrld ttree)
                          (cond
                           (knownp (mv (not nilp) unify-subst nilp-ttree))
                           (t
                            (mv-let
                             (not-flg atm)
                             (strip-not hyp)

; Again, we avoid rewriting in this proof-checker code.

                             (cond
                              (not-flg
                               (if (equal atm *nil*)
                                   (mv t unify-subst ttree)
                                 (mv nil unify-subst ttree)))
                              (t
                               (if (if-tautologyp atm)
                                   (mv t unify-subst ttree)
                                 (mv nil unify-subst ttree))))))))))))))))

(defun pc-relieve-hyps1 (hyps unify-subst unify-subst0 ttree0 type-alist wrld
                              state ens ttree)

; This function is adapted from ACL2 function relieve-hyp.  Notice that there
; are no arguments for obj, equiv, fnstack, ancestors, or
; simplify-clause-pot-lst.  Also notice that rcnst has been replaced by ens (an
; enable structure).

; In order to make relieve-hyps a No-Change Loser without requiring it have to
; test the answer to its own recursive calls, we have to pass down the original
; unify-subst and ttree so that when it fails it can return them instead of the
; accumulated versions.

  (cond ((f-big-clock-negative-p state)
         (mv nil unify-subst ttree))
        ((null hyps) (mv t unify-subst ttree))
        (t (mv-let (relieve-hyp-ans new-unify-subst ttree)

; We avoid rewriting in htis proof-checker code, so new-ttree = ttree.

             (pc-relieve-hyp (car hyps) unify-subst type-alist wrld state ens ttree)
             (cond
              (relieve-hyp-ans
               (pc-relieve-hyps1 (cdr hyps)
                                 new-unify-subst
                                 unify-subst0 ttree0
                                 type-alist wrld state ens ttree))
              (t (mv nil unify-subst0 ttree0)))))))

(defun pc-relieve-hyps (hyps unify-subst ; &extra formals

                          type-alist wrld state ens ttree)

; Adapted from ACL2 function relieve-hyp.  Notice that there are no arguments
; for obj, equiv, fnstack, ancestors, or simplify-clause-pot-lst.  Also notice
; that rcnst has been replaced by ens (an enable structure).

; We return t or nil indicating success, an extended unify-subst and
; a new ttree.  This function is a No-Change Loser.

  (cond ((f-big-clock-negative-p state)
         (mv nil unify-subst ttree))
        (t (pc-relieve-hyps1 hyps unify-subst unify-subst ttree type-alist
                             wrld state ens ttree))))

(defun remove-trivial-lits (lst type-alist alist wrld ens ttree)

; Removes trivially true lits from lst.  However, we don't touch elements of
; lst that contain free variables.  We apply the substitution at this point
; because we need to know whether a lit contains a free variable (one not bound
; by alist) that might get bound later, thus changing its truth value.

  (if (consp lst)
      (mv-let (rest-list ttree)
        (remove-trivial-lits (cdr lst) type-alist alist wrld ens ttree)
        (let ((new-lit (sublis-var alist (car lst))))
          (if (free-varsp (car lst) alist)
              (mv (cons new-lit rest-list) ttree)
            (mv-let (knownp nilp nilp-ttree)
              (known-whether-nil new-lit type-alist
                                 ens (ok-to-force-ens ens) wrld ttree)
              (if (and knownp (not nilp))
                  (mv rest-list nilp-ttree)
                (mv (cons new-lit rest-list) ttree))))))
    (mv nil ttree)))

(defun unrelieved-hyps (hyps unify-subst type-alist wrld state ens ttree)

; Returns unrelieved hyps (with the appropriate substitution applied), an
; extended substitution, and a new tag tree.  Note: the substitution really has
; been applied already to the returned hyps, even though we also return the
; extended substitution.

  (mv-let (success-flg new-unify-subst new-ttree)
          (pc-relieve-hyps hyps unify-subst type-alist wrld state ens ttree)
          (if success-flg
              (mv nil new-unify-subst new-ttree)
            (mv-let (lits ttree)
                    (remove-trivial-lits hyps type-alist unify-subst wrld ens ttree)
                    (mv lits unify-subst ttree)))))

(defun show-rewrite (index col rune nume subst-hyps subst-rhs free
                           abbreviations term-id-iff ens enabled-only-flg state)
  (let ((enabledp (enabled-numep nume ens)))
    (if (and enabled-only-flg
             (not enabledp))
        state
      (fms "~|~#a~[~c0. ~/  ~]~x1~#2~[~/ (disabled)~]~%~
            ~ ~ New term: ~x3~%~
            ~ ~ Hypotheses: ~#b~[<none>~/~y4~]~|~
            ~#5~[~/~ ~ Free variable:  ~&6.~/Free variables:  ~&6.~]~
            ~#7~[~/~ ~ WARNING:  One of the hypotheses is (equivalent to) NIL, ~
            and hence will apparently be impossible to relieve.~]~|"
           (list (cons #\a (if index 0 1))
                 (cons #\0 (cons index col))
                 (cons #\1
                       ;; Let's just print the name of the rune if it appears
                       ;; to be unique.
                       (if (cddr rune) rune (cadr rune)))
                 (cons #\2 (if enabledp 0 1))
                 (cons #\3 (untrans0 subst-rhs term-id-iff abbreviations))
                 (cons #\b (if subst-hyps 1 0))
                 (cons #\4 (untrans0-lst subst-hyps t abbreviations))
                 (cons #\5 (zero-one-or-more (length free)))
                 (cons #\6 free)
                 (cons #\7 (if (member-eq nil subst-hyps) 1 0)))
           (standard-co state) state nil))))

(defun set-difference-assoc-eq (lst alist)
  (declare (xargs :guard (or (symbol-listp lst)
                             (symbol-alistp alist))))
  (cond ((null lst) nil)
        ((assoc-eq (car lst) alist)
         (set-difference-assoc-eq (cdr lst) alist))
        (t (cons (car lst) (set-difference-assoc-eq (cdr lst) alist)))))

(defun show-rewrites (app-rewrite-rules col abbreviations term-id-iff ens
                                        type-alist enabled-only-flg w state)
  (if (null app-rewrite-rules)
      state
    (pprogn (let ((sar (car app-rewrite-rules)))
              (let ((lemma (access sar sar :lemma))
                    (alist (access sar sar :alist))
                    (index (access sar sar :index)))
                (let ((hyps (access rewrite-rule lemma :hyps))
                      (rhs (access rewrite-rule lemma :rhs)))
                  (mv-let (subst-hyps unify-subst ttree)
                    (unrelieved-hyps hyps alist type-alist w state ens nil)
                    (declare (ignore ttree))
                    (show-rewrite index col
                                  (access rewrite-rule lemma :rune)
                                  (access rewrite-rule lemma :nume)
                                  subst-hyps
                                  (sublis-var unify-subst rhs)
                                  (set-difference-assoc-eq
                                   (union-eq (all-vars rhs)
                                             (all-vars1-lst hyps nil))
                                   unify-subst)
                                  abbreviations term-id-iff ens
                                  enabled-only-flg state)))))
            (show-rewrites (cdr app-rewrite-rules) col abbreviations
                           term-id-iff ens type-alist enabled-only-flg w
                           state))))

(defun expand-assumptions-1 (term) 
  (case-match term
    (('if a b ''nil)
     (append (expand-assumptions-1 a) (expand-assumptions-1 b)))
    ((equality-p a b)
     (if (or (and (eq equality-p 'eq)
                  (or (and (consp a) (eq (car a) 'quote) (symbolp (cadr a)))
                      (and (consp b) (eq (car b) 'quote) (symbolp (cadr b)))))
             (and (eq equality-p 'eql)
                  (or (and (consp a) (eq (car a) 'quote) (eqlablep (cadr a)))
                      (and (consp b) (eq (car b) 'quote) (eqlablep (cadr b))))))
         (list term (fcons-term* 'equal a b))
       (list term)))
    (& (list term))))

(defun expand-assumptions (x)

; If x is (and a b) then we get (list a b), etc.

  (declare (xargs :guard (true-listp x)))
  (if x
      (append (expand-assumptions-1 (car x))
              (expand-assumptions (cdr x)))
    nil))

(defun hyps-type-alist (assumptions ens wrld state)

; Note that the force-flg arg to type-alist-clause is nil here, so we shouldn't
; wind up with any assumptions in the returned tag tree. Also note that we
; return (mv contradictionp type-alist fc-pair-lst), where actually fc-pair-lst
; is a ttree if contradictionp holds; normally we ignore fc-pair-lst otherwise.

  (forward-chain (dumb-negate-lit-lst (expand-assumptions assumptions))
                 nil
                 (ok-to-force-ens ens)
                 nil ; do-not-reconsiderp
                 wrld ens (match-free-override wrld) state))

(defun show-rewrites-fn (rule-id enabled-only-flg ens current-term
                                 abbreviations term-id-iff all-hyps geneqv state)
  (let ((name (and (symbolp rule-id) rule-id))
        (index (and (integerp rule-id) (< 0 rule-id) rule-id))
        (rune (and (consp rule-id)
                   (equal (car rule-id) :rewrite)
                   rule-id))
        (w (w state)))
    (cond
     ((and rule-id (not (or name index rune)))
      (fms "The rule-id argument to SHOW-REWRITES must be a name, a ~
                       positive integer, or a rewrite rule rune, but ~x0 is ~
                       none of these.~|"
           (list (cons #\0 rule-id)) (standard-co state) state nil))
     ((or (variablep current-term)
          (fquotep current-term)
          (flambdap (ffn-symb current-term)))
      (fms "It is only possible to apply rewrite rules to terms ~
                       that are not variables, (quoted) constants, or ~
                       applications of lambda expressions.  However, the ~
                       current term is:~%~ ~ ~y0.~|"
           (list (cons #\0 current-term)) (standard-co state) state nil))
     ((eq (ffn-symb current-term) 'if)
      (fms "It is only possible to apply rewrite rules to terms ~
                        that are applications of function symbols other than ~
                        IF.  However, the current term is~|~ ~ ~y0.~|"
           (list (cons #\0 current-term)) (standard-co state) state nil))
     (t
      (mv-let (flg hyps-type-alist ttree)
        (hyps-type-alist all-hyps ens w state)
        (declare (ignore ttree))
        (if flg
            (fms "*** Contradiction in the hypotheses! ***~%The S ~
                             command should complete this goal.~|"
                 nil (standard-co state) state nil)
          (let ((app-rewrite-rules
                 (applicable-rewrite-rules1
                  current-term
                  geneqv
                  (getprop (ffn-symb current-term) 'lemmas nil 'current-acl2-world w)
                  1 (or name rune) index w)))
            (if (null app-rewrite-rules)
                (if (and index (> index 1))
                    (fms "~|*** There are fewer than ~x0 applicable rewrite rules. ***~%"
                         (list (cons #\0 index)) (standard-co state) state nil)
                  (fms "~|*** There are no applicable rewrite rules. ***~%"
                       nil  (standard-co state) state nil))
              (show-rewrites app-rewrite-rules
                             (floor (length app-rewrite-rules) 10)
                             abbreviations term-id-iff
                             ens hyps-type-alist
                             enabled-only-flg w state)))))))))

(defun pl-fn (name state)
  (if (symbolp name)
      (let* ((wrld (w state))
             (name (deref-macro-name name (macro-aliases wrld))))
        (if (function-symbolp name wrld)
            (pprogn (print-lemmas
                     (getprop name 'lemmas nil 'current-acl2-world wrld)
                     t
                     (ens state)
                     wrld
                     state)
                    (value :invisible))
          (if (getprop name 'macro-body nil 'current-acl2-world wrld)
              (er soft 'pl
                  "The argument to PL must be a function symbol in ~
                   the current world, but ~x0 is a macro."
                  name)
            (er soft 'pl
                "The argument to PL must be a function symbol in the ~
                 current world."))))
    (er-let* ((term (translate name nil t nil 'pl (w state) state)))
      (pprogn (show-rewrites-fn nil nil (ens state) term
                                nil nil nil nil state)
              (value :invisible)))))

(defmacro pl (name)

  #-small-acl2-image
  ":Doc-Section History

  print the rules whose top function symbol is the given name~/
  ~bv[]
  Examples:
  :pl foo ; prints the rewrite rules that rewrite some call of foo
  ~ev[]~/

  ~c[Pl] takes one argument, a function symbol, and displays the lemmas
  that rewrite some term whose top function symbol is the given one.
  Note that names of macros are not relevant here; for example, for
  the rules about ~ilc[+] you should give the command ~c[:pl] ~ilc[binary-+].

  In fact the kinds of rules printed by ~c[:pl] are rewrite rules,
  definition rules, and ~il[meta] rules.~/"

  (list 'pl-fn name 'state))

