;;; History of this file:
;;; David Russinoff created the original version of this file.  In
;;; 9/99, Matt Kaufmann modified some of the lemmas with an eye toward
;;; increasing the automation provided by this book.  In the process,
;;; some previous stylistic conventions fell by the wayside, such as
;;; disabling :rewrite rules immediate after their statements.
;;; In 7/2001, Eric Smith moved many of the lemmas into basic.lisp
;;; and other books in the floating point library
;;; In 6/2002, Eric Smith made more changes to this book, incorporating some lemmas from merge4.lisp, etc.
(in-package "ACL2")

(include-book "add")

(local (in-theory (disable expt-monotone-linear)))
(local (in-theory (disable expt)))
(local (in-theory (disable logand))) ;test

;;;**********************************************************************
;;;                         REMAINDERS
;;;**********************************************************************

;move this stuff?

(local (defthm nk>=k-1
    (implies (and (integerp n)
		  (>= n 0)
		  (integerp k)
		  (> k 0)
		  (not (= (* n k) 0)))
	     (>= (* n k) k))
  :rule-classes ()
  :hints (("goal" :induct (induct-nat n)))))

(local (defthm nk>=k-2
    (implies (and (integerp n)
		  (>= n 0)
		  (integerp k)
		  (> k 0)
		  (not (= (* n k) 0)))
	     (>= (abs (* n k)) k))
  :rule-classes ()
  :hints (("goal" :use (nk>=k-1)))))

(local (defthm nk>=k-3
    (implies (and (integerp n)
		  (<= n 0)
		  (integerp k)
		  (> k 0)
		  (not (= (* n k) 0)))
	     (>= (abs (* n k)) k))
  :rule-classes ()
  :hints (("goal" :use ((:instance nk>=k-2 (n (- n))))))))

(defthm nk>=k
    (implies (and (integerp n)
		  (integerp k)
		  (> k 0)
		  (not (= (* n k) 0)))
	     (>= (abs (* n k)) k))
  :rule-classes ()
  :hints (("goal" :use (nk>=k-2 nk>=k-3))))

(defthm mod=mod<n
    (implies (and (integerp a) (>= a 0)
		  (integerp b) (>= b 0)
		  (integerp n) (> n 0)
		  (< (abs (- a b)) n)
		  (= (mod a n) (mod b n)))
	     (= a b))
  :rule-classes ()
  :hints (("goal" :use (mod=mod
			(:instance nk>=k (k n) (n (/ (- a b) n)))
			(:instance *cancell (x a) (y b) (z n))))))


(defthm mod-force-equal
    (implies (and (natp a)
		  (natp b)
		  (natp n)
		  (< (abs (- a b)) n)
		  (= (mod a n) (mod b n)))
	     (= a b))
  :rule-classes ()
  :hints (("Goal" :use (mod=mod<n))))

(defthm nk>=k-linear
    (implies (and (integerp n)
		  (integerp k)
		  (not (= n 0)))
	     (>= (abs (* n k)) k))
  :rule-classes :linear
  :hints (("Goal" :use nk>=k)))

(defthm mod-mod-2
    (implies (natp x)
	     (or (= (mod x 2) 0)
		 (= (mod x 2) 1)))
  :rule-classes ()
  :hints (("Goal" :use (mod012))))

(defthm mod-plus-mod-2
    (implies (and (natp x)
		  (natp y))
	     (iff (= (mod (+ x y) 2) (mod x 2))
		  (= (mod y 2) 0)))
  :rule-classes ()
  :hints (("Goal" :use (mod-x-y-x-2))))

(defthm mod-mod-2-not-equal
    (implies (natp x)
	     (not (= (mod x 2) (mod (1+ x) 2))))
  :rule-classes ()
  :hints (("Goal" :use (mod+1-2))))

(defthm x-or-x/2
    (implies (integerp x) 
	     (or (integerp (/ x 2)) (integerp (/ (1+ x) 2))))
  :rule-classes ())

(defthm mod-2*i-rewrite
    (implies (integerp i)
             ;; Rule A3 in fp.lisp suggests using (* 2 i) instead of
             ;; (+ i i).
	     (equal (mod (* 2 i) 2) 0))
    :hints (("Goal" :use mod-2*i)))

(defthm mod-2*i+1
    (implies (integerp i)
	     (not (equal (mod (1+ (* 2 i)) 2) 0)))
  :rule-classes ())

(defthm mod-2*i+1-rewrite
    (implies (natp i)
	     (equal (mod (1+ (* 2 i)) 2) 1))
  :hints (("Goal" :use (mod-2*i+1 (:instance mod-mod-2 (x i))))))

;;;**********************************************************************
;;;                             BITN (really mostly bvecp stuff)
;;;**********************************************************************

(defun bvecp (x k)
  (and (integerp x)
       (>= x 0)
       (< x (expt 2 k))))

(in-theory (disable bvecp)) ;move up?

(defthm bvecp-0
  (bvecp 0 k)
  :hints (("Goal" :in-theory (enable bvecp))))

(defthm bvecp-forward
  (implies (bvecp x k)
           (and (integerp x)
                (<= 0 x)
                (< x (expt 2 k))))
  :hints (("Goal" :in-theory (enable bvecp)))
  :rule-classes :forward-chaining)

(local
 (defthm eb-hack
    (implies (and (integerp n) (integerp k))
	     (= (expt 2 (+ -1 n (* -1 k) (* -1 (+ -1 n))))
		(expt 2 (- k))))
  :rule-classes ()))



(defthm exact-bits-a-b
  (implies (and (integerp x) (>= x 0)
                (integerp n) (>= n 0)
                (integerp k) (>= k 0)
                (= (expo x) (1- n))
                (< k n))
           (iff (integerp (/ x (expt 2 k)))
                (exactp x (- n k))))
  :rule-classes ()
  :hints (("goal" :in-theory (disable exactp2 exactp2-lemma)
           :use ((:instance eb-hack)
                 (:instance exactp2 (n (- n k)))))))

(defthm exact-bits-a-c
  (implies (and (integerp x) (>= x 0)
                (integerp n) (>= n 0)
                (integerp k) (>= k 0)
                (= (expo x) (1- n))
                (< k n))
           (iff (integerp (/ x (expt 2 k)))
                (= (bits x (1- n) k)
                   (/ x (expt 2 k)))))
  :rule-classes ()
  :hints (("goal" :in-theory (enable bits)
           :use ((:instance fl-int (x (/ x (expt 2 k))))
                 (:instance mod< (m x) (n (expt 2 n)))
                 (:instance expo-upper-bound)))))

(defthm exact-bits-a-d
    (implies (and (integerp x) (>= x 0)
		  (integerp n) (>= n 0)
		  (integerp k) (>= k 0)
		  (= (expo x) (1- n))
		  (< k n))
	     (iff (integerp (/ x (expt 2 k)))
		  (= (bits x (1- k) 0)
		     0)))
  :rule-classes ()
  :hints (("goal" :in-theory (enable bits)
           :use ((:instance mod-fl (m x) (n (expt 2 k)))))))

(defthm exact-bits-b-d
    (implies (and (integerp x) (>= x 0)
		  (integerp n) (>= n 0)
		  (integerp k) (>= k 0)
		  (= (expo x) (1- n))
		  (< k n))
	     (iff (exactp x (- n k))
		  (= (bits x (1- k) 0)
		     0)))
  :rule-classes ()
  :hints (("goal" :use (exact-bits-a-d exact-bits-a-b))))

(encapsulate
 ()
 (local (include-book "bvecp"))
 (local (include-book "expt"))

 (local (defthm bvecp-exactp-aux
          (implies (and (case-split (natp n))
                        (bvecp x n))
                   (exactp x n))
          :hints (("Goal" :in-theory (set-difference-theories
                                      (enable expt-split zip bvecp natp)
                                      '(a15))
                   :use ((:instance exact-bits-a-b (n (1+ (expo x))) (k 0))
                         (:instance expo<= (n (1- n)))
                         (:instance expo>= (n 0))
                         (:instance exactp-<= (m (1+ (expo x)))))))))

 (defthm bvecp-exactp
   (implies (bvecp x n)
            (exactp x n))

   ))




(defthm bitn-bvecp
  (implies (and (<= 1 k)
                (case-split (integerp k)))
           (bvecp (bitn x n) k)))

;kill have bitn-alt-0
(defthm bitn-rec-0
    (implies (natp x)
	     (equal (bitn x 0)
		    (mod x 2)))
  :hints (("Goal" :use (bitn-alt-0))))
(in-theory (disable bitn-rec-0))

(defthm bitn-rec-pos
    (implies (and (natp x)
		  (natp k)
		  (> k 0))
	     (equal (bitn x k)
		    (bitn (fl (/ x 2)) (1- k))))
  :hints (("Goal" :use (bitn-alt-pos))))
(in-theory (disable bitn-rec-pos))


(encapsulate
 ()
 (local (include-book "bits2"))
 (local (defthm bvecp-bits-0-aux
   (implies (and (case-split (natp i)) ;(natp i)
                 (case-split (<= j i))
                 (bvecp x j))
            (equal (bits x i j) 0))
   :hints (("Goal" :in-theory (set-difference-theories
                               (enable bits bvecp)
                               '(mod-equal))
            :use ((:instance mod-equal (m x) (n (expt 2 (1+ i))))
                  (:instance expt-monotone (n j) (m (1+ i))))))))

 (defthm bvecp-bits-0
   (implies (bvecp x j)
            (equal (bits x i j) 0))
   :hints (("Goal" :cases ((< i j))
            :in-theory (set-difference-theories
                        (enable natp)
                        '( bvecp )))))
 )
(in-theory (disable bvecp-bits-0))

;could combine these next two?

(defthm bitn-bvecp-0-thm
  (implies (bvecp x n)
           (equal (bitn x n) 0))
  :hints (("Goal" :in-theory (enable bitn bvecp-bits-0))))
(in-theory (disable bitn-bvecp-0-thm))

(defthm bitn-bvecp-0
    (implies (and (bvecp x n)
		  (natp m)
		  (natp n))
	     (equal (bitn x (+ m n)) 0))
  :hints (("Goal" :in-theory (union-theories (disable bitn-bvecp-0-thm
                                              ) '(bvecp natp))
		  :use ((:instance bitn-bvecp-0-thm (n (+ m n)))
			(:instance expt-monotone (m (+ m n)))))))

;could perhaps do without induction
(defthm bvecp-plus
    (implies (and (bvecp x m)
		  (bvecp y n)
		  (natp m)
		  (natp n))
	     (bvecp (* x y) (+ m n)))
  :rule-classes ()
  :hints (("Goal" :in-theory (set-difference-theories
                              (enable bvecp natp expt)
                              '()))))

(defthm bvecp-1-rewrite
    (iff (bvecp x 1)
	 (member x '(0 1)))
  :hints (("Goal" :in-theory (enable bvecp))))
(in-theory (disable bvecp-1-rewrite))

(defthm bitn-bvecp-1
    (implies (bvecp x 1)
	     (equal (bitn x 0) x))
    :hints (("Goal" :in-theory (enable bvecp-1-rewrite))))



(defthm bitn-force-1
    (implies (and (bvecp x (1+ n))
		  (<= (expt 2 n) x)
                  (natp n))
	     (equal (bitn x n) 1))
  :hints (("Goal" :use (bit-expo-b))))
(in-theory (disable bitn-force-1))

(defthm bit-basic-h-2
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp z) (>= z 0))
	     (= (logand (logior y z) x)
		(logior (logand y x) (logand z x))))
  :rule-classes ()
  :hints (("goal" :use ((:instance bit-basic-h)
			(:instance bit-basic-c (y (logior y z)))
			(:instance bit-basic-c)
			(:instance bit-basic-c (y z))))))


(defthm bitn-force-2
    (implies (and (bvecp x n) ; bind free var n here
                  (<= (- (expt 2 n) (expt 2 k)) x)
                  (< k n)
                  (natp n)
		  (natp k)
		  )
	     (equal (bitn x k) 1))
  :hints (("Goal" :use (bit-expo-c))))
(in-theory (disable bitn-force-2))

(defthm bitn-expt
    (implies (natp n) ;drop?
	     (equal (bitn (expt 2 n) n) 1))
  :hints (("goal" :in-theory (set-difference-theories
                              (enable expt-split)
                              '(a15))
           :use ((:instance bit-expo-b (x (expt 2 n)))
                        ))))
(in-theory (disable bitn-expt))
#|
;may cause case splits (maybe that's good?)
(defthm bitn-expt-2
  (implies (case-split (natp n)) ;drop?
           (equal (bitn (expt 2 n) n2)
                  (if (equal n n2)
                      1
                    0)))

  :hints (("Goal" :in-theory (enable bitn-expt)
           :cases ((< n n2))
  )))
|#
(defthm bit+expt
    (implies (and (natp x)
		  (natp n))
	     (not (equal (bitn (+ x (expt 2 n)) n)
			 (bitn x n))))
  :rule-classes ()
  :hints (("Goal" :use (bit+-a))))

(defthm bit+expt-2
    (implies (and (natp x)
		  (natp n)
		  (natp m)
		  (> m n))
	     (equal (bitn (+ x (expt 2 m)) n)
		    (bitn x n)))
  :hints (("Goal" :use (bit+-b))))
(in-theory (disable bit+expt-2))

(defthm bitn+mult
    (implies (and (natp x)
		  (natp k)
		  (natp n)
		  (natp m)
		  (> m n))
	     (equal (bitn (+ x (* k (expt 2 m))) n)
		    (bitn x n)))
  :hints (("Goal" :use (bit+*k))))
(in-theory (disable bitn+mult))

(defthm bitn-shift
    (implies (and (natp x)
		  (natp n)
		  (natp k))
	     (= (bitn (* x (expt 2 k)) (+ n k))
		(bitn x n)))
  :rule-classes ()
  :hints (("Goal" :use (bitn-n+k))))

(defthm mod+bitn
    (implies (and (natp a)
		  (natp n))
	     (= (mod a (expt 2 (1+ n)))
		(+ (* (bitn a n) (expt 2 n))
		   (mod a (expt 2 n)))))
  :rule-classes ()
  :hints (("Goal" :use (mod-n+1)
           :in-theory (enable bitn-rec-0))))

(defthm mod-bitn-0
    (implies (and (natp a)
		  (natp n))
	     (iff (= (mod a (expt 2 (1+ n))) 0)
		  (and (= (mod a (expt 2 n)) 0)
		       (= (bitn a n) 0))))
  :rule-classes ()
  :hints (("Goal" :use (mod-n-n+1)
           :in-theory (enable bitn-rec-0))))

(defthm bitn-shift-2
    (implies (and (natp x)
		  (natp k)
		  (natp r))
	     (equal (bitn (fl (/ x (expt 2 r))) k)
		    (bitn x (+ k r))))
  :hints (("Goal" :use (bitn-fl))))
(in-theory (disable bitn-shift-2))

(defthm bitn-shift-3
    (implies (and (natp n)
		  (natp m)
		  (natp k)
		  (bvecp x m)
		  (<= m n))
	     (equal (bitn (+ x (* k (expt 2 m))) n)
		    (bitn k (- n m))))
  :hints (("Goal" :use (bit+*k-2))))
(in-theory (disable bitn-shift-3))


;;;**********************************************************************
;;;                         BITS
;;;**********************************************************************

(defthm mod-bits-equal
    (implies (and (natp x)
		  (natp y)
		  (natp i)
		  (natp j)
		  (= (mod x (expt 2 (1+ i))) (mod y (expt 2 (1+ i)))))
	     (= (bits x i j) (bits y i j)))
  :rule-classes ()
  :hints (("Goal" :use (mod-bits))))





(local
(defthm and-bits-e-1
    (implies (and (integerp n) (>= n 0)
		  (integerp k) (>= k 0)
		  (integerp l) (>= l 0) (< l k)
		  (< k n))
	     (= (logand (- (1- (expt 2 n)) (expt 2 l)) (- (expt 2 n) (expt 2 k)))
		(* (expt 2 k) (bits (- (1- (expt 2 n)) (expt 2 l)) (1- n) k))))
  :rule-classes ()
  :hints (("goal" :use ((:instance and-bits-c (x (- (1- (expt 2 n)) (expt 2 l))))
			(:instance expt-strong-monotone (n l) (m n)))))))

(local
(defthm and-bits-e-2
    (implies (and (integerp n) (>= n 0)
		  (integerp k) (>= k 0)
		  (integerp l) (>= l 0) (< l k)
		  (< k n))
	     (= (logand (- (1- (expt 2 n)) (expt 2 l)) (- (expt 2 n) (expt 2 k)))
		(* (expt 2 k) (fl (/ (mod (- (1- (expt 2 n)) (expt 2 l)) (expt 2 n)) (expt 2 k))))))
  :rule-classes ()
  :hints (("goal" :in-theory (enable bits)
		  :use ((:instance and-bits-e-1))))))

(local
(defthm and-bits-e-3
    (implies (and (integerp n) (>= n 0)
		  (integerp k) (>= k 0)
		  (integerp l) (>= l 0) (< l k)
		  (< k n))
	     (= (mod (- (1- (expt 2 n)) (expt 2 l)) (expt 2 n))
		(- (1- (expt 2 n)) (expt 2 l))))
  :rule-classes ()
  :hints (("goal" :use ((:instance expt-strong-monotone (n l) (m n))
;			(:instance expt-pos (x l))
			(:instance mod< (m (- (1- (expt 2 n)) (expt 2 l))) (n (expt 2 n))))))))

(local
(defthm hack-m4
    (implies (= x y)
	     (= (fl (/ x (expt 2 k))) (fl (/ y (expt 2 k)))))
  :rule-classes ()))

(local
(defthm and-bits-e-4
    (implies (and (integerp n) (>= n 0)
		  (integerp k) (>= k 0)
		  (integerp l) (>= l 0) (< l k)
		  (< k n))
	     (= (fl (/ (mod (- (1- (expt 2 n)) (expt 2 l)) (expt 2 n))
		       (expt 2 k)))
		(fl (/ (- (1- (expt 2 n)) (expt 2 l))
		       (expt 2 k)))))		       
  :rule-classes ()
  :hints (("goal" :use (and-bits-e-3
			(:instance hack-m4 
				   (x (mod (- (1- (expt 2 n)) (expt 2 l)) (expt 2 n))) 
				   (y (- (1- (expt 2 n)) (expt 2 l)))))))))

(local
(defthm and-bits-e-5
    (implies (and (integerp n) (>= n 0)
		  (integerp k) (>= k 0)
		  (integerp l) (>= l 0) (< l k)
		  (< k n))
	     (= (logand (- (1- (expt 2 n)) (expt 2 l)) (- (expt 2 n) (expt 2 k)))
		(* (expt 2 k) (fl (/ (- (1- (expt 2 n)) (expt 2 l)) (expt 2 k))))))
  :rule-classes ()
  :hints (("goal" :hands-off (expt mod fl)
		  :use ((:instance and-bits-e-2)
			(:instance and-bits-e-4))))))

(local
(defthm and-bits-e-6
    (implies (and (integerp n) (>= n 0)
		  (integerp k) (>= k 0)
		  (integerp l) (>= l 0) (< l k)
		  (< k n))
	     (= (fl (/ (- (1- (expt 2 n)) (expt 2 l)) (expt 2 k)))
		(fl (- (expt 2 (- n k)) (/ (1+ (expt 2 l)) (expt 2 k))))))
  :rule-classes ()
  :hints (("goal" :use ((:instance expo+ (m (- n k)) (n k)))))))

(local (defthm hack-l2
    (implies (and (integerp x)
		  (integerp y)
		  (< 0 x)
		  (<= x y))
	     (and (> (/ x y) 0)
		  (<= (/ x y) 1)))
  :rule-classes ()))

(local
(defthm and-bits-e-7
    (implies (and (integerp k) (>= k 0)
		  (integerp l) (>= l 0) (< l k))
	     (and (> (/ (1+ (expt 2 l)) (expt 2 k)) 0)
		  (<= (/ (1+ (expt 2 l)) (expt 2 k)) 1)))
  :rule-classes ()
  :hints (("goal" :use (;(:instance expt-pos (x l))
			;(:instance expt-pos (x k))
			;(:instance integerp-expt-type (n l))
			;(:instance integerp-expt-type (n k))
			(:instance hack-l2 (x (1+ (expt 2 l))) (y (expt 2 k)))
			(:instance expt-strong-monotone (n l) (m k)))))))

(local
(defthm and-bits-e-8
    (implies (and (integerp n) (>= n 0)
		  (integerp k) (>= k 0)
		  (integerp l) (>= l 0) (< l k)
		  (< k n))
	     (= (fl (- (expt 2 (- n k)) (/ (1+ (expt 2 l)) (expt 2 k))))
		(- (expt 2 (- n k)) 1)))
  :rule-classes ()
  :hints (("goal" :in-theory (disable expt-2-integerp
                              )
		  :use (and-bits-e-6
			and-bits-e-7
			(:instance expt-2-integerp (i (- n k)))
			(:instance fl-unique 
				   (x (- (expt 2 (- n k)) (/ (1+ (expt 2 l)) (expt 2 k))))
				   (n (- (expt 2 (- n k)) 1))))))))

(local
(defthm and-bits-e-9
    (implies (and (integerp n) (>= n 0)
		  (integerp k) (>= k 0)
		  (integerp l) (>= l 0) (< l k)
		  (< k n))
	     (= (fl (/ (- (1- (expt 2 n)) (expt 2 l)) (expt 2 k)))
		(- (expt 2 (- n k)) 1)))
  :rule-classes ()
  :hints (("goal" :use (and-bits-e-8
			and-bits-e-6)))))

(local
(defthm and-bits-e-10
    (implies (and (integerp n) (>= n 0)
		  (integerp k) (>= k 0)
		  (integerp l) (>= l 0) (< l k)
		  (< k n))
	     (= (logand (- (1- (expt 2 n)) (expt 2 l)) (- (expt 2 n) (expt 2 k)))
		(* (expt 2 k) (- (expt 2 (- n k)) 1))))
  :rule-classes ()
  :hints (("goal" :hands-off (expt mod fl)
		  :use ((:instance and-bits-e-5)
			(:instance and-bits-e-9))))))

(defthm and-bits-e
    (implies (and (integerp n) (>= n 0)
		  (integerp k) (>= k 0)
		  (integerp l) (>= l 0) (< l k)
		  (< k n))
	     (= (logand (- (1- (expt 2 n)) (expt 2 l)) (- (expt 2 n) (expt 2 k)))
		(- (expt 2 n) (expt 2 k))))
  :rule-classes ()
  :hints (("goal" :hands-off (expt mod fl)
		  :use ((:instance and-bits-e-10)
			(:instance expo+ (m (- n k)) (n k))))))

(defthm bits-tail
    (implies (and (natp n)
		  (bvecp x (1+ n)))
	     (equal (bits x n 0)
		    x))
  :hints (("Goal" :in-theory (enable bvecp))))

;not needed.  just expand bits
(defthm bits-mod
  (implies (and (case-split (integerp x))
                (case-split (integerp i))
                (case-split (<= 0 i)))
           (equal (bits x i 0)
                  (mod x (expt 2 (1+ i)))))
  :hints (("Goal" :in-theory (enable bits))))
(in-theory (disable bits-mod))

(defthm bits-shift-1
    (implies (and (natp x)
		  (natp i)
		  (natp j)
		  (natp k))
	     (equal (bits (fl (/ x (expt 2 k)))
			  i
			  j)
		    (bits x (+ i k) (+ j k))))
  :hints (("Goal" :use ((:instance bit-bits-a (i (+ i k)) (j (+ j k)))))))
(in-theory (disable bits-shift-1))

(defthm bits-0-mod-0
    (implies (and (natp x)
		  (natp m)
		  (natp n))
	     (iff (= (mod x (expt 2 (+ m n 1))) 0)
		  (and (= (bits x (+ m n) n) 0)
		       (= (mod x (expt 2 n)) 0))))
  :rule-classes ()
  :hints (("Goal" :use (bits-mod-0))))

(defthm bits-0-bitn-0
    (implies (and (natp x)
		  (natp n)
		  (not (= n 0)))
	     (iff (= (bits x n 0) 0)
		  (and (= (bitn x n) 0)
		       (= (bits x (1- n) 0) 0))))
  :rule-classes ()
  :hints (("Goal" :use (bits-bitn))))

(defthm bits-plus-bits
    (implies (and (natp x)
		  (natp r)
		  (natp n)
		  (natp m)
		  (> n r)
		  (> m n))
	     (= (bits x (1- m) r)
		(+ (bits x (1- n) r)
		   (* (expt 2 (- n r)) (bits x (1- m) n)))))
  :rule-classes ()
  :hints (("Goal" :use (bits-bits-thm))))

(defthm bits-plus-bitn
    (implies (and (natp x)
		  (natp m)
		  (natp n)
		  (> n m))
	     (= (bits x n m)
		(+ (* (bitn x n) (expt 2 (- n m)))
		   (bits x (1- n) m))))
  :rule-classes ()
  :hints (("Goal" :use (bits+bitn))))

(defthm bits-n-n-rewrite
  (equal (bits x n n)
         (bitn x n))
  :hints (("Goal" :in-theory (enable bitn))))

(theory-invariant
 (not (and (member-equal '(:rewrite bits-n-n-rewrite)
                         theory)
           (member-equal '(:definition bitn)
                         theory)))
 :key :bitn-definition-invariant)


(defthm bitn-plus-bits
    (implies (and (natp x)
		  (natp n)
		  (natp m)
		  (> n m))
	     (= (bits x n m)
		(+ (bitn x m)
		   (* 2 (bits x n (1+ m))))))
  :rule-classes ()
  :hints (("Goal"
           :use (bitn+bits
                 ;; for m=0 case
                 (:instance bits-bits-thm
                            (m (1+ n))
                            (r 0)
                            (n 1))))))

(defthm bits-plus-mult
    (implies (and (natp m)
		  (natp n)
		  (>= n m)
		  (natp k)
		  (<= k m)
		  (natp y)
		  (bvecp x k))
	     (= (bits (+ x (* y (expt 2 k))) n m)
		(bits y (- n k) (- m k))))
  :rule-classes ()
  :hints (("Goal" :use (bits+2**k-2))))


;;;**********************************************************************
;;;                       LOGAND, LOGIOR, and LOGXOR
;;;**********************************************************************

(defthm logand-rewrite
    (implies (and (natp x)
		  (natp y))
	     (equal (logand x y)
		    (+ (* 2 (logand (fl (/ x 2)) (fl (/ y 2))))
		       (logand (mod x 2) (mod y 2)))))
  :rule-classes ((:definition :controller-alist ((binary-logand t t))))
  :hints (("Goal" :use (logand-def))))
(in-theory (disable logand-rewrite))

(defthm natp-logand
    (implies (and (natp i)
		  (natp j))
	     (natp (logand i j)))
  :rule-classes (:type-prescription :rewrite)
  :hints (("Goal" :use (logand-nat))))

(defthm logand-mod-2
    (implies (and (natp x)
		  (natp y))
	     (equal (mod (logand x y) 2)
		    (logand (mod x 2) (mod y 2))))
  :hints (("Goal" :use (logand-mod))))
(in-theory (disable logand-mod-2))

(defthm logand-fl-2
    (implies (and (natp x)
		  (natp y))
	     (equal (fl (/ (logand x y) 2))
		    (logand (fl (/ x 2)) (fl (/ y 2)))))
  :hints (("Goal" :use (logand-fl))))
(in-theory (disable logand-fl-2))

(defthm logior-rewrite
    (implies (and (natp i)
		  (natp j))
	     (equal (logior i j)
		    (+ (* 2 (logior (fl (/ i 2)) (fl (/ j 2))))
		       (logior (mod i 2) (mod j 2)))))
  :rule-classes ((:definition :controller-alist ((binary-logior t t))))
  :hints (("Goal" :use (logior-def))))
(in-theory (disable logior-rewrite))

(defthm natp-logior
    (implies (and (natp i)
		  (natp j))
	     (natp (logior i j)))
  :rule-classes (:type-prescription :rewrite)
  :hints (("Goal" :use (logior-nat))))

(defthm logior-mod-2
    (implies (and (natp i)
		  (natp j))
	     (equal (mod (logior i j) 2)
		    (logior (mod i 2) (mod j 2))))
  :hints (("Goal" :use (logior-mod))))
(in-theory (disable logior-mod-2))

(defthm logior-fl-2
    (implies (and (natp i)
		  (natp j))
	     (equal (fl (/ (logior i j) 2))
		    (logior (fl (/ i 2)) (fl (/ j 2)))))
  :hints (("Goal" :use (logior-fl))))
(in-theory (disable logior-fl-2))

(defthm logxor-def-rewrite
    (implies (and (natp x)
		  (natp y))
	     (equal (logxor x y)
		    (+ (* 2 (logxor (fl (/ x 2)) (fl (/ y 2))))
		       (logxor (mod x 2) (mod y 2)))))
  :rule-classes ((:definition :controller-alist ((binary-logxor t t))))
  :hints (("Goal" :use (logxor-def))))
(in-theory (disable logxor-def-rewrite))

(defthm natp-logxor
    (implies (and (natp i)
		  (natp j))
	     (natp (logxor i j)))
  :rule-classes (:type-prescription :rewrite)
  :hints (("Goal" :use (logxor-nat))))

(defthm logxor-mod-2
    (implies (and (natp i)
		  (natp j))
	     (equal (mod (logxor i j) 2)
		    (logxor (mod i 2) (mod j 2))))
  :hints (("Goal" :use (logxor-mod))))
(in-theory (disable logxor-mod-2))

(defthm logxor-fl-2
    (implies (and (natp i)
		  (natp j))
	     (equal (fl (/ (logxor i j) 2))
		    (logxor (fl (/ i 2)) (fl (/ j 2)))))
  :hints (("Goal" :use (logxor-fl))))
(in-theory (disable logxor-fl-2))

(defthm logxor-rewrite-2
    ;; !! Do we really want to get rid of logxor?
    (implies (and (bvecp x n)
		  (bvecp y n)
                  (natp n)
		  (not (= n 0)))
	     (equal (logxor x y)
		    (logior (logand x (comp1 y n))
			    (logand y (comp1 x n)))))
  :hints (("Goal" :use (logxor-rewrite))))
(in-theory (disable logxor-rewrite-2))

(defthm logxor-0-y
    (implies (integerp y)
	     (equal (logxor 0 y) y)))

(defthm logxor-x-0
    (implies (integerp x)
	     (equal (logxor x 0) x)))

(defthm logxor-self
    (implies (natp x)
	     (equal (logxor x x) 0)))

(defthm logand-ones
    (implies (and (natp n)
		  (bvecp x n))
	     (equal (logand x (1- (expt 2 n)))
		    x))
  :hints (("Goal" :use (:instance logand-2**n-1 (i x)))))
(in-theory (disable logand-ones))

(defthm logior-commutative
    (implies (and (integerp x)
		  (integerp y))
	     (equal (logior x y) (logior y x)))
  :hints (("Goal" :use (bit-basic-d))))

(defthm logxor-commutative
    (implies (and (integerp x)
		  (integerp y))
	     (equal (logxor x y) (logxor y x)))
  :hints (("Goal" :in-theory (enable lognot logxor)
		  :use ((:instance bit-basic-c 
				   (x (LOGIOR (+ -1 (* -1 Y)) X))
				   (y (LOGIOR (+ -1 (* -1 X)) Y)))))))

(defthm logior-associative
    (implies (and (natp x)
		  (natp y)
		  (natp z))
	     (equal (logior (logior x y) z)
		    (logior x (logior y z))))
  :hints (("Goal" :use (bit-basic-f))))

;(in-theory (disable logior-associative)) ;why??

(defthm logxor-associative
    (implies (and (natp x)
		  (natp y)
		  (natp z))
	     (equal (logxor (logxor x y) z)
		    (logxor x (logxor y z))))
  :hints (("Goal" :use (logxor-assoc))))

;(in-theory (disable logxor-associative)) ;why?

(defthm logior-logand
    (implies (and (natp x)
		  (natp y)
		  (natp z))
	     (equal (logior x (logand y z))
		    (logand (logior x y) (logior x z))))
  :hints (("Goal" :use (bit-basic-g))))
(in-theory (disable logior-logand))

(defthm logand-logior
    (implies (and (natp x)
		  (natp y)
		  (natp z))
	     (equal (logand x (logior y z))
		    (logior (logand x y) (logand x z))))
  :hints (("Goal" :use (bit-basic-h))))
(in-theory (disable logand-logior))

(defthm logior-logand-2
    (implies (and (natp x)
		  (natp y)
		  (natp z))
	     (equal (logand  (logior y z) x)
		    (logior (logand y x) (logand z x))))
  :hints (("Goal" :use (bit-basic-h-2))))
(in-theory (disable logior-logand-2))

(DEFTHM LOGAND-SELF
  (IMPLIES (CASE-SPLIT (INTEGERP I))
           (EQUAL (LOGAND I I) I)))

(defthm logior-bvecp
    (implies (and (natp n)
		  (bvecp x n)
		  (bvecp y n))
	     (bvecp (logior x y) n))
  :hints (("Goal" :in-theory (enable bvecp)
           :use (or-dist-a natp-logior))))

(defthm logior-expt
    (implies (and (natp n)
		  (natp x)
		  (bvecp y n))
	     (= (logior (* (expt 2 n) x) y)
		(+ (* (expt 2 n) x) y)))
  :rule-classes ()
  :hints (("Goal" :use (or-dist-b))))

(defthm logior-expt-2
    (implies (and (natp x)
		  (natp y)
		  (natp n))
	     (= (logior (* (expt 2 n) x)
			(* (expt 2 n) y))
		(* (expt 2 n) (logior x y))))
  :rule-classes ()
  :hints (("Goal" :use (or-dist-c))))

(defthm mod-logior
    (implies (and (natp x)
		  (natp y)
		  (natp n))
	     (equal (mod (logior x y) (expt 2 n))
		    (logior (mod x (expt 2 n)) (mod y (expt 2 n)))))
  :hints (("Goal" :use (or-dist-d))))
(in-theory (disable mod-logior))

(defthm logand-bnd
    (implies (and (natp x)
		  (natp y))
	     (<= (logand x y) x))
  :rule-classes :linear
  :hints (("Goal" :use (and-dist-a))))

(defthm logand-bvecp
    (implies (and (natp n)
		  (bvecp x n)
		  (natp y))
	     (bvecp (logand x y) n))
  :hints (("Goal" :in-theory (enable bvecp)
           :use (natp-logand logand-bnd))))

(defthm logand-expt
    (implies (and (natp x)
		  (natp y)
		  (natp n))
	     (= (logand (* (expt 2 n) x) y)
		(* (expt 2 n) (logand x (fl (/ y (expt 2 n)))))))
  :rule-classes ()
  :hints (("Goal" :use (and-dist-b))))

(defthm mod-logand-expt
    (implies (and (natp x)
		  (natp y)
		  (natp n))
	     (= (mod (logand x y) (expt 2 n))
		(logand (mod x (expt 2 n)) y)))
  :rule-classes ()
  :hints (("Goal" :use (and-dist-c))))

(defthm mod-logand-rewrite
    (implies (and (natp x)
		  (natp y)
		  (natp n))
	     (equal (mod (logand x y) (expt 2 n))
		    (logand (mod x (expt 2 n)) (mod y (expt 2 n)))))
  :hints (("Goal" :use (mod-logand))))
(in-theory (disable mod-logand-rewrite))

(local
 (defthm mod-logxor-0
   (implies (and (natp x)
                 (natp y)
                 (equal n 0))
            (equal (mod (logxor x y) (expt 2 n))
                   (logxor (mod x (expt 2 n))
                           (mod y (expt 2 n)))))
   :rule-classes nil
   :hints (("Goal" :use (mod-logxor)))))

(defthm mod-logxor-rewrite
    (implies (and (natp x)
		  (natp y)
		  (natp n))
	     (equal (mod (logxor x y) (expt 2 n))
		    (logxor (mod x (expt 2 n))
			    (mod y (expt 2 n)))))
  :hints (("Goal" :use (mod-logxor
                        mod-logxor-0
                        (:theorem
                         (implies (and (natp n) (< n 1))
                                  (equal n 0)))))))
(in-theory (disable mod-logxor-rewrite))

(defthm logand-mod-expt
    (implies (and (natp x)
		  (natp y)
		  (natp n)
		  (< x (expt 2 n)))
	     (= (logand x y)
		(logand x (mod y (expt 2 n)))))
  :rule-classes ()
  :hints (("Goal" :use (and-dist-d))))

(defthm logxor-bvecp
    (implies (and (natp n)
		  (bvecp x n)
		  (bvecp y n))
	     (bvecp (logxor x y) n))
  :hints (("Goal" :in-theory (enable bvecp)
           :use (logxor<2**n
                 (:instance logxor-nat (i x) (j y))))))

(defthm bitn-logand
    (implies (and (natp x)
		  (natp y)
		  (natp n))
	     (equal (bitn (logand x y) n)
		    (logand (bitn x n) (bitn y n))))
  :hints (("Goal" :use (bit-dist-a))))
(in-theory (disable bitn-logand)) ;why?

(defthm bitn-logior
    (implies (and (natp x)
		  (natp y)
		  (natp n))
	     (equal (bitn (logior x y) n)
		    (logior (bitn x n) (bitn y n))))
  :hints (("Goal" :use (bit-dist-b))))
(in-theory (disable bitn-logior)) ;why?

(defthm logand-expt-2
    (implies (and (natp x)
		  (natp k))
	     (= (logand x (expt 2 k))
		(* (expt 2 k) (bitn x k))))
  :rule-classes ()
  :hints (("Goal" :use (and-bits-a))))

(defthm logior-expt-3
    (implies (and (natp x)
		  (natp k))
	     (= (logior x (expt 2 k))
		(+ x
		   (* (expt 2 k) 
		      (- 1 (bitn x k))))))
  :rule-classes ()
  :hints (("Goal" :use (and-bits-b))))

;logand-expt-3 is in log.lisp

(defthm logand-expt-4
    (implies (and (natp n)
		  (natp k)
		  (natp l)
		  (< l k)
		  (<= k n))
	     (= (logand (- (1- (expt 2 n)) (expt 2 l)) (- (expt 2 n) (expt 2 k)))
		(- (expt 2 n) (expt 2 k))))
  :rule-classes ()
  :hints (("Goal" :use (and-bits-e))))

(defthm bitn-logxor-0
    (implies (and (natp a)
		  (natp b))
	     (= (bitn (+ a b) 0)
		(bitn (logxor a b) 0)))
  :rule-classes ()
  :hints (("Goal" :use (bitn-0-logxor-+))))


;;;**********************************************************************
;;;                            COMP1
;;;**********************************************************************

(defthm comp1-fl
    (implies (and (natp n)
		  (natp k)
		  (<= k n)
		  (bvecp x n))
	     (= (fl (/ (comp1 x n) (expt 2 k)))
		(comp1 (fl (/ x (expt 2 k))) (- n k))))
  :rule-classes ()
  :hints (("Goal" :use (fl-comp1))))

(defthm mod-comp1-2
    (implies (and (natp n)
		  (not (= n 0))
		  (bvecp x n))
	     (not (= (mod (comp1 x n) 2)
		     (mod x 2))))
  :rule-classes ()
  :hints (("Goal" :use (mod-comp1))))

(local (include-book "comp1"))

;proved in comp1.lisp
(defthm comp1-with-n-0
  (equal (comp1 x 0)
         0)
  :hints (("Goal" :in-theory (enable comp1))))

;proved in comp1.lisp
(defthm comp1-bvecp
  (implies (and (<= n k)
                (case-split (integerp k)))
           (bvecp (comp1 x n) k)))

;basically the same as bitn-comp1-thm
(defthm bitn-comp1-not-equal
    (implies (and (< k n)
                  (natp n)
		  (bvecp x n)
		  (natp k)
                  )
	     (not (= (bitn (comp1 x n) k)
		     (bitn x k))))
  :rule-classes ()
  :hints (("Goal" :use (bitn-comp1-thm))))

(defthm mod-comp1-rewrite
    (implies (and (natp n)
		  (natp m)
		  (bvecp x m)
		  (not (= n 0))
		  (>= m n))
	     (equal (mod (comp1 x m) (expt 2 n))
		    (comp1 (mod x (expt 2 n)) n)))
  :hints (("Goal" :use (comp1-mod))))

(in-theory (disable mod-comp1-rewrite))


;;**********************************************************************
;;                        NEW STUFF
;;**********************************************************************

(defthm logior-not-0
    (implies (and (natp x)
		  (natp y)
		  (= (logior x y) 0))
	     (and (= x 0) (= y 0)))
  :rule-classes ()
  :hints (("Goal"
           :expand ((natp x) (natp y))
           :use (expo-upper-bound
                 expo-lower-bound
                 bit-basic-d
                 (:instance bit-basic-b (x y))
                 (:instance expo>= (n 0))
                 (:instance bitn-0-1 (n (expo x)) (x y))
                 (:instance bit-dist-b (n (expo x)))
                 (:instance bit-expo-b (n (expo x)))))))

(defun ls-induct (k x)
  (if (zp k)
      x
    (ls-induct (1- k) (fl (/ x 2)))))

(local-defthm lshiftamt-low-3-1
    (implies (and (integerp k) (> k 0))
	     (= (fl (/ (1- (expt 2 k)) 2))
		(1- (expt 2 (1- k)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (set-difference-theories
                              (enable expt)
                              '(a15))
           :use ((:instance fl-unique
                                   (x (/ (1- (expt 2 k)) 2))
                                   (n (1- (expt 2 (1- k)))))))))

(local-defthm lshiftamt-low-3-2
    (implies (and (integerp k) (> k 0))
	     (= (mod (1- (expt 2 k)) 2) 1))
  :rule-classes ()
  :hints (("Goal" :in-theory (set-difference-theories
                              (enable expt)
                              '(a15))
           :use ((:instance mod012 (x (1- (expt 2 k))))
			(:instance mod+1-2 (x (1- (expt 2 k))))
			(:instance mod-2*i (i (expt 2 (1- k))))))))

(local-defthm lshiftamt-low-3
    (implies (and (integerp k) (>= k 0)
		  (integerp x) (>= x 0) (< x (expt 2 k)))
	     (= (logior (1- (expt 2 k)) x)
		(1- (expt 2 k))))
  :rule-classes ()
  :hints (("Goal" :in-theory (enable expt)
           :induct (ls-induct k x))
	  ("Subgoal *1/2" :use (lshiftamt-low-3-1
				lshiftamt-low-3-2
				mod012
				(:instance mod-fl (m x) (n 2))
				(:instance mod-fl (m (logior (1- (expt 2 k)) x)) (n 2))
				(:instance logior-nat (i (1- (expt 2 k))) (j x))
				(:instance fl-def-linear (x (/ x 2)))
				(:instance logior-fl (i (1- (expt 2 k))) (j x))
				(:instance logior-mod (i (1- (expt 2 k))) (j x))))))

(local-defthm lshiftamt-low-4
    (implies (and (integerp k) (>= k 0)
		  (integerp x) (> x 0)
		  (= (expo x) k))
	     (= (logior x (1- (expt 2 k)))
		(1- (expt 2 (1+ k)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (enable expt)
           :use (expo-upper-bound
			expo-lower-bound
;			(:instance expt-pos (x k))
;			(:instance integerp-expt-type (n k))
			(:instance bit-basic-d (x (- x (expt 2 k))) (y (1- (expt 2 k))))
			(:instance or-dist-b (n k) (x 1) (y (- x (expt 2 k))))
			(:instance bit-basic-f (x (expt 2 k)) (y (- x (expt 2 k))) (z (1- (expt 2 k))))
			(:instance lshiftamt-low-3 (x (- x (expt 2 k))))
			(:instance or-dist-b (n k) (x 1) (y (1- (expt 2 k))))))))

(defthm logior-x-0
    (implies (natp x)
	     (equal (logior x 0) x))
  :hints (("Goal" :use (bit-basic-b))))

(local-defthm logior-bits-1
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp i) (>= i j)
		  (integerp j) (> j 0))
	     (= (bits (logior x y) i 0)
		(logior (+ (* (expt 2 j) (bits x i j))
			   (bits x (1- j) 0))
			(+ (* (expt 2 j) (bits y i j))
			   (bits y (1- j) 0)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (enable bits
                                     )
		  :use ((:instance or-dist-d (n (1+ i)))
			(:instance expo+ (m 1) (n i))
			(:instance bits-bits-thm (m (1+ i)) (n j) (r 0))
			(:instance bits-bits-thm (x y) (m (1+ i)) (n j) (r 0))))))

(local-defthm logior-bits-2
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp i) (>= i j)
		  (integerp j) (> j 0))
	     (= (bits (logior x y) i 0)
		(logior (logior (* (expt 2 j) (bits x i j))
				(bits x (1- j) 0))
			(logior (* (expt 2 j) (bits y i j))
				(bits y (1- j) 0)))))
  :rule-classes ()
  :hints (("Goal"
           :use (logior-bits-1
                 (:instance bits< (i (1- j)) (j 0))
                 (:instance bits< (x y) (i (1- j)) (j 0))
                 (:instance or-dist-b (x (bits x i j)) (n j) (y (bits x (1- j) 0)))
                 (:instance or-dist-b (x (bits y i j)) (n j) (y (bits y (1- j) 0)))))))

(local-defthm logior-bits-3
    (implies (and (integerp a) (>= a 0)
		  (integerp b) (>= b 0)
		  (integerp c) (>= c 0)
		  (integerp d) (>= d 0))
	     (= (logior (logior a b) (logior c d))
		(logior (logior a c) (logior b d))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance bit-basic-f (x a) (y b) (z (logior c d)))
			(:instance bit-basic-f (x b) (y c) (z d))
			(:instance bit-basic-d (x b) (y c))
			(:instance bit-basic-f (x a) (y (logior c b)) (z d))
			(:instance bit-basic-f (x a) (y c) (z b))
			(:instance bit-basic-f (x (logior a c)) (y b) (z d))
			(:instance logior-nat (i c) (j d))
			(:instance logior-nat (i c) (j b))
			(:instance logior-nat (i a) (j c))))))

(local-defthm logior-bits-4
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp i) (>= i j)
		  (integerp j) (> j 0))
	     (= (bits (logior x y) i 0)
		(logior (logior (* (expt 2 j) (bits x i j))
				(* (expt 2 j) (bits y i j)))
			(logior (bits x (1- j) 0)
				(bits y (1- j) 0)))))
  :rule-classes ()
  :hints (("Goal"
           :use (logior-bits-2
;                 (:instance expt-pos (x j))
;                 (:instance integerp-expt-type (n j))
                 (:instance logior-bits-3
                            (a (* (expt 2 j) (bits x i j)))
                            (b (bits x (1- j) 0))
                            (c (* (expt 2 j) (bits y i j)))
                            (d (bits y (1- j) 0)))))))

;(in-theory (disable logior-mod-rewrite))

(local (include-book "fl2"))
(local (include-book "mod"))

(local-defthm logior-bits-5
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp i) (>= i j)
		  (integerp j) (> j 0))
	     (= (bits (logior x y) i 0)
		(logior (* (expt 2 j) (logior (bits x i j) (bits y i j)))
			(bits (logior x y) (1- j) 0))))
    :otf-flg t
  :rule-classes ()
  :hints (("Goal" :in-theory (set-difference-theories
                              (enable bits bits-mod
                                      )
                              '(logior-associative))
           :use (logior-bits-4
                 (:instance or-dist-c (n j) (x (bits x i j)) (y (bits y i j)))
                 (:instance or-dist-d (n j))))))

(local-defthm logior-bits-6
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp i) (>= i j)
		  (integerp j) (> j 0))
	     (= (bits (logior x y) i 0)
		(+ (* (expt 2 j) (logior (bits x i j) (bits y i j)))
		   (bits (logior x y) (1- j) 0))))
  :rule-classes ()
  :hints (("Goal"
           :use (logior-bits-5
                 (:instance logior-nat (i x) (j y))
                 (:instance logior-nat (i (bits x i j)) (j (bits y i j)))
                 (:instance bits< (x (logior x y)) (i (1- j)) (j 0))
                 (:instance or-dist-b
                            (x (logior (bits x i j) (bits y i j)))
                            (y (bits (logior x y) (1- j) 0))
                            (n j))))))

(local-defthm logior-bits-7
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp i) (>= i j)
		  (integerp j) (> j 0))
	     (= (bits (logior x y) i 0)
		(+ (* (expt 2 j) (bits (logior x y) i j))
		   (bits (logior x y) (1- j) 0))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance logior-nat (i x) (j y))
			(:instance bits-bits-thm (x (logior x y)) (m (1+ i)) (n j) (r 0))))))

(defthm bits-logior
    (implies (and (natp x)
		  (natp y)
		  (natp i)
		  (natp j)
		  (>= i j))
	     (equal (bits (logior x y) i j)
		    (logior (bits x i j) (bits y i j))))
  :hints (("Goal" :in-theory (enable bits-mod
                                     )
		  :use (logior-bits-6
			logior-bits-7
			(:instance cancel-equal-*
				   (a (expt 2 j))
				   (r (logior (bits x i j) (bits y i j)))
				   (s (bits (logior x y) i j)))
			(:instance logior-nat (i x) (j y))
			(:instance or-dist-d (n (1+ i)))
;			(:instance expt-pos (x j))
                        ))))

(in-theory (disable bits-logior))

(local-defthm logand-bits-1
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp i) (>= i j)
		  (integerp j) (> j 0))
	     (= (bits (logand x y) i 0)
		(logand (+ (* (expt 2 j) (bits x i j))
			   (bits x (1- j) 0))
			(+ (* (expt 2 j) (bits y i j))
			   (bits y (1- j) 0)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (enable bits-mod
                                     )
		  :use ((:instance mod-logand (n (1+ i)))
			(:instance expo+ (m 1) (n i))
			(:instance bits-bits-thm (m (1+ i)) (n j) (r 0))
			(:instance bits-bits-thm (x y) (m (1+ i)) (n j) (r 0))))))

(local-defthm logand-bits-2
 (implies (and (integerp x) (>= x 0)
               (integerp y) (>= y 0)
               (integerp i) (>= i j)
               (integerp j) (> j 0))
          (= (bits (logand x y) i 0)
             (logand (logior (* (expt 2 j) (bits x i j))
                             (bits x (1- j) 0))
                     (logior (* (expt 2 j) (bits y i j))
                             (bits y (1- j) 0)))))
 :rule-classes ()
 :hints (("Goal" :use (logand-bits-1
                       (:instance bits< (i (1- j)) (j 0))
                       (:instance bits< (x y) (i (1- j)) (j 0))
                       (:instance or-dist-b (x (bits x i j)) (n j) (y (bits x (1- j) 0)))
                       (:instance or-dist-b (x (bits y i j)) (n j) (y (bits y (1- j) 0)))))))

(local-defthm logand-bits-3
    (implies (and (integerp a) (>= a 0)
		  (integerp b) (>= b 0)
		  (integerp c) (>= c 0)
		  (integerp d) (>= d 0))
	     (= (logand (logior a b) (logior c d))
		(logior (logior (logand a c) (logand c b))
			(logior (logand b d) (logand a d)))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance bit-basic-h (x (logior a b)) (y c) (z d))
			(:instance bit-basic-h-2 (y a) (z b) (x c))
			(:instance bit-basic-h-2 (y a) (z b) (x d))
			(:instance bit-basic-c (x b) (y c))
			(:instance bit-basic-d (x (logand a d)) (y (logand b d)))
			(:instance logand-nat (i a) (j d))
			(:instance logand-nat (i b) (j d))
			(:instance logior-nat (i a) (j b))))))

(local-defthm logand-bits-4
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp i) (>= i j)
		  (integerp j) (> j 0))
	     (= (bits (logand x y) i 0)
		(logior (logior (logand (* (expt 2 j) (bits x i j))
					(* (expt 2 j) (bits y i j)))
				(logand (* (expt 2 j) (bits y i j))
					(bits x (1- j) 0)))					
			(logior (logand (bits x (1- j) 0)
					(bits y (1- j) 0))
				(logand (* (expt 2 j) (bits x i j))
					(bits y (1- j) 0))))))
				
  :rule-classes ()
  :hints (("Goal" :use (logand-bits-2
;                 (:instance expt-pos (x j))
;                 (:instance integerp-expt-type (n j))
                 (:instance logand-bits-3
                            (a (* (expt 2 j) (bits x i j)))
                            (b (bits x (1- j) 0))
                            (c (* (expt 2 j) (bits y i j)))
                            (d (bits y (1- j) 0)))))))

(local-defthm logand-bits-5
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp i) (>= i j)
		  (integerp j) (> j 0))
	     (= (logand (* (expt 2 j) (bits x i j))
			(* (expt 2 j) (bits y i j)))
		(* (expt 2 j) (logand (bits x i j) (bits y i j)))))				
  :rule-classes ()
  :hints (("Goal" :use (
;                 (:instance expt-pos (x j))
;                 (:instance integerp-expt-type (n j))
                 (:instance and-dist-b (n j) (x (bits x i j)) (y (* (expt 2 j) (bits y i j))))))))

(local-defthm hack-1
    (implies (and (rationalp x)
		   (rationalp y)
		   (> y 0)
		   (< x y))
	     (< (/ x y) 1))
  :rule-classes ())

(local-defthm logand-bits-6
    (implies (and (integerp y) (>= y 0)
		  (integerp j) (> j 0))
	     (< (/ (bits y (1- j) 0) (expt 2 j))
		1))
  :rule-classes ()
  :hints (("Goal" :use (;(:instance expt-pos (x j))
;			(:instance integerp-expt-type (n j))
			(:instance hack-1 (x (bits y (1- j) 0)) (y (expt 2 j)))
			(:instance bits< (x y) (i (1- j)) (j 0))))))

(local-defthm logand-bits-7
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp i) (>= i j)
		  (integerp j) (> j 0))
	     (= (logand (* (expt 2 j) (bits x i j))
			(bits y (1- j) 0))
		0))
  :rule-classes ()
  :hints (("Goal" :in-theory (enable logand)
           :use (
                 logand-bits-6
                 (:instance fl-unique (x (/ (bits y (1- j) 0) (expt 2 j))) (n 0))
;                 (:instance expt-pos (x j))
 ;                (:instance expt-pos (x (- 1 j)))
;                 (:instance integerp-expt-type (n j))
                 (:instance bits< (x y) (i (1- j)) (j 0))
                 (:instance bit-basic-a (x (bits x i j)))
                 (:instance and-dist-b (n j) (x (bits x i j)) (y (bits y (1- j) 0)))))))

(local-defthm logand-bits-8
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp i) (>= i j)
		  (integerp j) (> j 0))
	     (= (bits (logand x y) i 0)
		(logior (logior (* (expt 2 j) (logand (bits x i j) (bits y i j)))
				0)
			(logior (logand (bits x (1- j) 0)
					(bits y (1- j) 0))
				0))))				
  :rule-classes ()
  :hints (("Goal" :use (logand-bits-4
			logand-bits-5
			logand-bits-7
			(:instance logand-bits-7 (x y) (y x))))))

(local-defthm logand-bits-9
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp i) (>= i j)
		  (integerp j) (> j 0))
	     (= (bits (logand x y) i 0)
		(logior (* (expt 2 j) (logand (bits x i j) (bits y i j)))
			(logand (bits x (1- j) 0)
				(bits y (1- j) 0)))))
  :rule-classes ()
  :hints (("Goal" :use (logand-bits-8
;			(:instance expt-pos (x j))
;			(:instance integerp-expt-type (n j))
			(:instance logand-nat (i (bits x (1- j) 0)) (j (bits y (1- j) 0)))
			(:instance logand-nat (i (bits x i j)) (j (bits y i j)))
			(:instance bit-basic-b (x (* (expt 2 j) (logand (bits x i j) (bits y i j)))))
			(:instance bit-basic-b (x (logand (bits x (1- j) 0) (bits y (1- j) 0))))))))

(local-defthm logand-bits-10
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp i) (>= i j)
		  (integerp j) (> j 0))
	     (= (bits (logand x y) i 0)
		(logior (* (expt 2 j) (logand (bits x i j) (bits y i j)))
			(bits (logand x y) (1- j) 0))))
  :rule-classes ()
  :hints (("Goal" :in-theory (enable 
                              bits-mod
                              )
		  :use (logand-bits-9
			(:instance mod-logand (n j))))))

(local-defthm logand-bits-11
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp i) (>= i j)
		  (integerp j) (> j 0))
	     (= (bits (logand x y) i 0)
		(+ (* (expt 2 j) (logand (bits x i j) (bits y i j)))
		   (bits (logand x y) (1- j) 0))))
  :rule-classes ()
  :hints (("Goal" :use (logand-bits-10
                 (:instance logand-nat (i x) (j y))
                 (:instance logand-nat (i (bits x i j)) (j (bits y i j)))
                 (:instance bits< (x (logand x y)) (i (1- j)) (j 0))
                 (:instance or-dist-b
                            (x (logand (bits x i j) (bits y i j)))
                            (y (bits (logand x y) (1- j) 0))
                            (n j))))))

(local-defthm logand-bits-12
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp i) (>= i j)
		  (integerp j) (> j 0))
	     (= (bits (logand x y) i 0)
		(+ (* (expt 2 j) (bits (logand x y) i j))
		   (bits (logand x y) (1- j) 0))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance logand-nat (i x) (j y))
			(:instance bits-bits-thm (x (logand x y)) (m (1+ i)) (n j) (r 0))))))

(defthm bits-logand
    (implies (and (natp x)
		  (natp y)
		  (natp i)
		  (natp j)
		  (>= i j))
	     (equal (bits (logand x y) i j)
		    (logand (bits x i j) (bits y i j))))
  :hints (("Goal" :in-theory (enable bits-mod)
		  :use (logand-bits-11
			logand-bits-12
			(:instance cancel-equal-*
				   (a (expt 2 j))
				   (r (logand (bits x i j) (bits y i j)))
				   (s (bits (logand x y) i j)))
			(:instance logand-nat (i x) (j y))
			(:instance mod-logand (n (1+ i)))
;			(:instance expt-pos (x j))
                        ))))

(in-theory (disable bits-logand))

;why is comp1 enabled here?


(encapsulate
 ()

 (local (defthm bits-comp1-1
          (implies (and (integerp m) (> m i)
                        (integerp i) (>= i j) (>= i 0)
                        (integerp j) (>= j 0)
                        (integerp x) (>= x 0) (< x (expt 2 m)))
                   (= (bits (comp1 x m) i j)
                      (fl (/ (comp1 (mod x (expt 2 (1+ i))) (1+ i))
                             (expt 2 j)))))
          :rule-classes ()
          :hints (("Goal" :in-theory (set-difference-theories
                                      (enable bits)
                                      '(comp1))
                   :use ((:instance comp1-mod (n (1+ i))))))))

 (local (defthm bits-comp1-2
          (implies (and (integerp m) (> m i)
                        (integerp i) (>= i j) (>= i 0)
                        (integerp j) (>= j 0)
                        (integerp x) (>= x 0) (< x (expt 2 m)))
                   (= (bits (comp1 x m) i j)
                      (fl (+ (expt 2 (1+ (- i j))) 
                             (/ (- (1+ (mod x (expt 2 (1+ i)))))
                                (expt 2 j))))))
          :rule-classes ()
          :hints (("Goal" :in-theory (enable comp1)
                   :use (bits-comp1-1
                         (:instance expt- (a (1+ i)) (b j)))))))

 (local (defthm bits-comp1-3
          (implies (and (integerp m) (> m i)
                        (integerp i) (>= i j) (>= i 0)
                        (integerp j) (>= j 0)
                        (integerp x) (>= x 0) (< x (expt 2 m)))
                   (= (bits (comp1 x m) i j)
                      (+ (expt 2 (1+ (- i j))) 
                         (fl (/ (- (1+ (mod x (expt 2 (1+ i)))))
                                (expt 2 j))))))
          :rule-classes ()
          :hints (("Goal" :in-theory (disable a10)
                   :use (bits-comp1-2
;(:instance integerp-expt-type (n (- (1+ i) j)))
                         (:instance fl+int-rewrite 
                                    (x (/ (- (1+ (mod x (expt 2 (1+ i))))) (expt 2 j)))
                                    (n (expt 2 (1+ (- i j))))))))))

 (local (defthm bits-comp1-4
          (implies (and (integerp m) (> m i)
                        (integerp i) (>= i j) (>= i 0)
                        (integerp j) (>= j 0)
                        (integerp x) (>= x 0) (< x (expt 2 m)))
                   (= (bits (comp1 x m) i j)
                      (comp1 (fl (/ (mod x (expt 2 (1+ i)))
                                    (expt 2 j)))
                             (1+ (- i j)))))
          :rule-classes ()
          :hints (("Goal" :in-theory (enable natp comp1) ;added by eric
                   :use (bits-comp1-3
                         (:instance mod>=0 (m x) (n (expt 2 (1+ i))))
                         (:instance floor-m+1 (m (mod x (expt 2 (1+ i)))) (n (expt 2 j))))))))

 (local (defthm bits-comp1-5
          (implies (and (integerp m) (> m i)
                        (integerp i) (>= i j) (>= i 0)
                        (integerp j) (>= j 0)
                        (integerp x) (>= x 0) (< x (expt 2 m)))
                   (= (bits (comp1 x m) i j)
                      (comp1 (bits x i j) (1+ (- i j)))))
          :rule-classes ()
          :hints (("Goal" :in-theory (enable bits)
                   :use (bits-comp1-4)))))

 (defthm bits-comp1
   (implies (and (natp m) 
                 (natp i) 
                 (natp j)
                 (> m i)
                 (>= i j)
                 (bvecp x m))
            (equal (bits (comp1 x m) i j)
                   (comp1 (bits x i j) (1+ (- i j)))))
   :hints (("Goal" :use (bits-comp1-5))))
 )
;aa
;(in-theory (disable bits-comp1)) ;why
(in-theory (disable comp1))

(defthm bitn-comp1
    (implies (and (natp m) 
		  (natp n) 
		  (> m n)
		  (bvecp x m))
	     (equal (bitn (comp1 x m) n)
		    (comp1 (bitn x n) 1)))
  :hints (("Goal" :in-theory (set-difference-theories
                              (enable bitn)
                              '(bits-n-n-rewrite)))))

;(in-theory (disable bitn-comp1)) ;why?



(encapsulate
 ()

 (local-defthm logxor-bits-1
               (implies (and (integerp x) (>= x 0) (< x (expt 2 n))
                             (integerp y) (>= y 0) (< y (expt 2 n))
                             (integerp n) (>= n i)
                             (integerp i) (>= i j)
                             (integerp j) (>= j 0)
                             )
                        (= (bits (logxor x y) i j)
                           (logior (logand (bits x i j) (bits (comp1 y n) i j))
                                   (logand (bits y i j) (bits (comp1 x n) i j)))))
               :rule-classes ()
               :hints (("Goal" :in-theory (disable comp1-bvecp comp1)
                        :use (logxor-rewrite
                              comp1-bvecp
                              (:instance comp1-bvecp (x y))
                              (:instance bits-logior (x (logand x (comp1 y n))) (y (logand y (comp1 x n))))
                              (:instance logand-nat (i x) (j (comp1 y n)))
                              (:instance logand-nat (i y) (j (comp1 x n)))
                              (:instance bits-logand (y (comp1 y n)))
                              (:instance bits-logand (x y) (y (comp1 x n)))))))

 (local-defthm logxor-bits-2
               (implies (and (integerp x) (>= x 0) (< x (expt 2 n))
                             (integerp y) (>= y 0) (< y (expt 2 n))
                             (integerp n) (> n i)
                             (integerp i) (>= i j)
                             (integerp j) (>= j 0))
                        (= (bits (logxor x y) i j)
                           (logior (logand (bits x i j) (comp1 (bits y i j) (1+ (- i j))))
                                   (logand (bits y i j) (comp1 (bits x i j) (1+ (- i j)))))))
               :rule-classes ()
               :hints (("Goal" :in-theory (set-difference-theories
                                           (enable bvecp)
                                           '(comp1-bvecp comp1))
                        :use (logxor-bits-1
                              (:instance bits-comp1 (m n))
                              (:instance bits-comp1 (x y) (m n))))))


 (local (defthm bits-logxor-aux
          (implies (and (bvecp x n) ; Free variable n is bound here
                        (bvecp y n)
                        (natp n)
                        (natp i)
                        (natp j)
                        (> n i)
                        (>= i j))
                   (equal (bits (logxor x y) i j)
                          (logxor (bits x i j) (bits y i j))))
          :hints (("Goal" :in-theory (disable comp1-bvecp comp1)
                   :use (logxor-bits-2
                         (:instance logxor-rewrite (x (bits x i j)) (y (bits y i j)) (n (1+ (- i j))))
                         (:instance bits<)
                         (:instance bits< (x y)))))))

 (local (in-theory (disable bits-logxor-aux)))

 (local-defthm hack1
               (implies (natp x)
                        (> (expt 2 x) x))
               :hints (("Goal" :in-theory (enable expt)))
               :rule-classes ())

 (defthm bits-logxor
   (implies (and (case-split (natp x))
                 (case-split (natp y))
                 (case-split (natp i))
                 (case-split (natp j))
                 ;(>= i j)
                 )
            (equal (bits (logxor x y) i j)
                   (logxor (bits x i j) (bits y i j))))
   :hints (("Goal" :in-theory (enable bvecp natp)
            :use ((:instance hack1 (x (+ i x y)))
                  (:instance bits-logxor-aux (n (+ i x y)))))))
 )

;what is the purpose of this??
(defthm bits-logxor-upper-slice
    (implies (and (equal n (+ 1 i))
                  (bvecp x n)
		  (bvecp y n)
		  (natp n)
		  (natp i)
		  (natp j)
                  )
	     (equal (bits (logxor x y) i j)
		    (logxor (bits x i j) (bits y i j))))
    :hints (("Goal" :use ((:instance bits-logxor )))))

(in-theory (disable bits-logxor-upper-slice))

;;
;; cat
;;

(local (include-book "cat"))

#|old definition
(defun CAT (x y n)
  (+ (* (expt 2 n) x) y))
|#

;now always returns a nat
(defun cat (x y n)
  (+ (* (expt 2 (nfix n)) (nfix x)) (nfix y)))
(in-theory (disable cat))

(defthm cat-nonnegative-integer-type
  (and (integerp (CAT X Y N))
       (<= 0 (CAT X Y N)))
  :rule-classes (:type-prescription)
  )

;this rule is no better than cat-nonnegative-integer-type and might be worse
(in-theory (disable (:type-prescription cat)))

;just a rewrite rule
(defthm cat-natp
  (natp (cat x y n)))

;proved in cat.lisp
(defthm cat-bvecp
  (implies (and (>= p n) ;handle other case?
                (bvecp x (- p n))
                (case-split (natp n))
                (case-split (natp p))
                (case-split (bvecp y n))
                )
           (bvecp (cat x y n) p)))

(local (in-theory (disable cat-bvecp))) ;why?

;add cat-bvecp-rewrite!

;this has caused problems in the past (size information is lost)?
;proved in cat.lisp
(defthm cat-0-rewrite
    (implies (and (case-split (integerp x))
		  (case-split (<= 0 x)))
	     (equal (cat 0 x n) x)))

;proved in cat.lisp
(defthm cat-associative
  (implies (and (case-split (<= 0 m)) ;new now that cat fixes its args
                (case-split (<= 0 n)) ;new now that cat fixes its args
                (case-split (integerp m))
                (case-split (integerp n))
                )
           (equal (cat (cat x y m) z n)
                  (cat x (cat y z n) (+ m n)))))

;;bits-cat

(defthm bits-cat-1
  (implies (and (< i n)
                (case-split (natp y))
                (case-split (integerp i))
                (case-split (integerp j))
                (case-split (natp n))
                )
           (equal (bits (cat x y n) i j)
                  (bits y i j)))
  :hints (("Goal" :in-theory (set-difference-theories
                              (enable cat)
                              '(expt-2-integerp))
           :use ((:instance mod-bits (x (cat x y n)))
                 (:instance expt-2-integerp (i (- n (1+ i)))) ;elim?
                 (:instance mod+-thm (n (expt 2 (1+ i))) (m y) (a (* x (expt 2 (- n (1+ i))))))))))

(defthm bits-cat-2
  (implies (and (>= j n)
                (case-split (natp x))
                (case-split (bvecp y n))
                (case-split (natp n))
                (case-split (integerp i))
                (case-split (integerp j))
                )
           (equal (bits (cat x y n) i j)
                  (bits x (- i n) (- j n))))
  :hints (("Goal" :in-theory (enable cat)
           :use ((:instance fl-unique (x (/ (cat x y n) (expt 2 n))) (n x))
                 (:instance bit-bits-a (x (cat x y n)) (k n))))))

(defthm bits-cat-3
    (implies (and (>= i n)
		  (< j n)
                  (case-split (bvecp y n))
                  (case-split (natp x))
                  (case-split (natp n))
                  (case-split (natp i))
                  (case-split (natp j))
                  )
	     (equal (bits (cat x y n) i j)
		    (cat (bits x (- i n) 0)
			 (bits y (1- n) j)
			 (- n j))))
  :hints (("Goal" :use ((:instance bits-plus-bits (x (cat x y n)) (m (1+ i)) (r j))))
	  ("Goal'''" :in-theory (enable cat))))  ;; RBK:

;includes both bitn-cat-1, bitn-cat-2, and bitn-cat-3
;we expect the indices to be constants, so this won't cause case-splits
(defthm bits-cat
  (implies (and (case-split (bvecp y n))
                (case-split (natp x))
                (case-split (natp n))
                (case-split (natp i))
                (case-split (natp j))
                )
           (equal (bits (cat x y n) i j)
                  (if (< i n)
                      (bits y i j)
                    (if (>= j n)
                        (bits x (- i n) (- j n))
                      (cat (bits x (- i n) 0)
                           (bits y (1- n) j)
                           (- n j)))))))

;bits-cat should be all we need for simplifying (bits (cat...))
(in-theory (disable bits-cat-1 bits-cat-2 bits-cat-3))

;; bitn-cat

(defthm bitn-cat-1
  (implies (and (< i n)
                (case-split (natp n))
                (case-split (integerp i))
                (case-split (natp x))
                (case-split (natp y))
                )
           (equal (bitn (cat x y n) i)
                  (bitn y i)))
  :hints (("Goal" :in-theory (set-difference-theories
                              (enable bitn bits-cat-1)
                              '(bits-n-n-rewrite)))))

(defthm bitn-cat-2
  (implies (and (>= i n)
                (case-split (bvecp y n))
                (case-split (natp x))
                (case-split (natp n))
                (case-split (integerp i))
                )
           (equal (bitn (cat x y n) i)
                  (bitn x (- i n))))
  :hints (("Goal" :in-theory (set-difference-theories
                              (enable bitn)
                              '(bits-n-n-rewrite)))))

;includes both bitn-cat-1 and bitn-cat-2
(defthm bitn-cat
  (implies (and (case-split (bvecp y n))
                (case-split (natp x))
                (case-split (natp n))
                (case-split (integerp i))
                )
           (equal (bitn (cat x y n) i)
                  (if (< i n)
                      (bitn y i)
                    (bitn x (- i n))))))

;bitn-cat should be all we need for simplifying (bitn (cat...))
(in-theory (disable bitn-cat-1 bitn-cat-2))








;; shft

(local (include-book "shft"))

;from shft.lisp
(defun shft (x s l)
  (mod (fl (* (expt 2 s) x)) (expt 2 (nfix l))))

(in-theory (enable shft))

;proved in shft.lisp
(defthm shft-nonnegative-integer-type
  (and (integerp (shft x s l))
       (<= 0 (shft x s l)))
  :rule-classes (:type-prescription))

;this rule is no better than shft-nonnegative-integer-type and might be worse:
(in-theory (disable (:type-prescription shft)))

;proved in shft.lisp
(defthm shft-natp
  (natp (shft x s n)))

;proved in shft.lisp
(defthm shft-bvecp
  (implies (and (<= n k)
                (case-split (integerp k)))
           (bvecp (shft x s n) k)))

; The following rule allows us to relieve (integerp x) hypotheses when
; a rule applies to show (natp x).
;this rule can be very expensive.  we don't want to backchain to natp is all we need is integerp!
(defthm natp-integerp
  (implies (natp x)
           (integerp x)))

(in-theory (disable natp-integerp)) ;test to see if anything breaks

;move to bits?
(defthm bitn-bitn-0
    (equal (bitn (bitn x n) 0)
	   (bitn x n))
  :hints (("Goal" :use (bitn-0-1))))

(defthm bvecp+1
    (implies (and (natp n)
		  (bvecp x n))
	     (bvecp x (+ 1 n)))
  :hints (("Goal" :in-theory (enable bvecp expt))))

 ;begin stuff from merge4 (renamed stick2)


 ;; sumbits

(defun sumbits (x n)
  (if (zp n)
      0
    (+ (* (expt 2 (1- n)) (bitn x (1- n)))
       (sumbits x (1- n)))))

(defthm sumbits-bits
  (implies (and (natp x)
                (natp n)
                (> n 0))
           (equal (sumbits x n)
                  (bits x (1- n) 0)))
  :hints (("Goal" :in-theory (enable bits-n-n-rewrite)
           :induct (sumbits x n))
          ("Subgoal *1/2" :use ((:instance bits-plus-bitn (n (1- n)) (m 0))))))

(in-theory (disable sumbits-bits))

(defthm sumbits-thm
     (implies (and (bvecp x n)
                   (natp n)
                   (> n 0))
              (equal (sumbits x n)
                     x))
   :hints (("Goal" :in-theory (enable sumbits-bits))))

(in-theory (disable sumbits-thm))

 ;removed bitn-bvecp-0, et seq.







(defthm logior-0-x
     (implies (natp x)
              (equal (logior 0 x) x))
   :hints (("Goal" :in-theory (disable logior-commutative)
                   :use ((:instance logior-commutative (y 0))))))

(local (defun ls-induct (k x)
   (if (zp k)
       x
     (ls-induct (1- k) (fl (/ x 2))))))

(local-defthm logior-ones-3-1
     (implies (and (integerp k) (> k 0))
              (= (fl (/ (1- (expt 2 k)) 2))
                 (1- (expt 2 (1- k)))))
   :rule-classes ()
   :hints (("Goal" :in-theory (set-difference-theories
                               (enable expt)
                               '(a15))
           :use ((:instance fl-unique (x (/ (1- (expt 2 k)) 2)) (n (1- (expt 2 (1- k)))))))))

(local-defthm logior-ones-3-2
              (implies (and (integerp k) (> k 0))
                       (= (mod (1- (expt 2 k)) 2) 1))
              :rule-classes ()
              :hints (("Goal" :in-theory (enable expt)
                       :use ((:instance mod-mod-2 (x (1- (expt 2 k))))
                                    (:instance mod-mod-2-not-equal (x (1- (expt 2 k))))
                                    (:instance mod-2*i-rewrite (i (expt 2 (1- k))))))))

(local-defthm logior-ones-3
    (implies (and (integerp k) (>= k 0)
		  (integerp x) (>= x 0) (< x (expt 2 k)))
	     (= (logior (1- (expt 2 k)) x)
		(1- (expt 2 k))))
  :rule-classes ()
  :hints (("Goal" :in-theory (enable expt)
           :induct (ls-induct k x))
	  ("Subgoal *1/2" :use (logior-ones-3-1
				logior-ones-3-2
				mod-mod-2
				(:instance quot-mod (m x) (n 2))
				(:instance quot-mod (m (logior (1- (expt 2 k)) x)) (n 2))
				(:instance natp-logior (i (1- (expt 2 k))) (j x))
				(:instance fl-def-linear (x (/ x 2)))
				(:instance logior-fl-2 (i (1- (expt 2 k))) (j x))
				(:instance logior-mod-2 (i (1- (expt 2 k))) (j x))))))

(defthm logior-ones
    (implies (and (natp n)
		  (bvecp x n))
	     (equal (logior x (1- (expt 2 n)))
		    (1- (expt 2 n))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance logior-ones-3 (k n))))))

(defthm logxor-ones
    (implies (and (natp n)
		  (bvecp x n))
	     (equal (logxor x (1- (expt 2 n)))
		    (comp1 x n)))
  :rule-classes ()
  :hints (("Goal" :in-theory (set-difference-theories
                              (enable comp1 bvecp)
                              '(comp1-bvecp))
		  :use (comp1-bvecp
			(:instance logxor-rewrite-2 (y (1- (expt 2 n))))
			(:instance logand-ones (x (comp1 x n)))))))

(defthm bits-shift-5
    (implies (and (natp x)
		  (natp k)
		  (natp i))
	     (equal (* (expt 2 k) (bits x i 0))
		    (bits (* (expt 2 k) x) (+ i k) 0)))
  :rule-classes ()
  :hints (("Goal" :in-theory (enable natp bits-mod)
		  :use ((:instance expo+ (n k) (m (1+ i)))
			(:instance mod-prod (k (expt 2 k)) (m x) (n (expt 2 (1+ i))))))))

;proved in comp1.lisp
(defthm comp1-2+1
   (implies (and (case-split (natp x))
                 (case-split (natp n))
                 )
            (equal (+ 1 (* 2 (comp1 x n)))
                   (comp1 (* 2 x) (1+ n)))))

(in-theory (disable bitn-bvecp-0))

(in-theory (disable mod-equal))

(in-theory (enable bits-n-n-rewrite))





(defun logop-3-induct (x y z)
  (declare (xargs :measure (+ (nfix x) (nfix y) (nfix z))))
  (if (and (natp x) (natp y) (natp z))
      (if (and (zp x) (zp y) (zp z))
	  t
	(logop-3-induct (fl (/ x 2)) (fl (/ y 2)) (fl (/ z 2))))
    t))

(defun log3a (x y z)
  (logior (logand x y) (logior (logand x z) (logand y z))))

(defun log3b (x y z)
  (logior (logand x y) (logand (logxor x y) z)))

(defthm logand-fl-2-rewrite
    (implies (and (natp x)
		  (natp y))
	     (equal (fl (* 1/2 (logand x y)))
		    (logand (fl (* 1/2 x)) (fl (* 1/2 y)))))
  :hints (("Goal" :use (logand-fl-2))))

(defthm logior-fl-2-rewrite
    (implies (and (natp i)
		  (natp j))
	     (equal (fl (* 1/2 (logior i j)))
		    (logior (fl (* 1/2 i)) (fl (* 1/2 j)))))
  :hints (("Goal" :use (logior-fl-2))))

(defthm logxor-fl-2-rewrite
    (implies (and (natp i)
		  (natp j))
	     (equal (fl (* 1/2 (logxor i j)))
		    (logxor (fl (* 1/2 i)) (fl (* 1/2 j)))))
  :hints (("Goal" :use (logxor-fl-2))))

(in-theory (disable logand-fl-2-rewrite logior-fl-2-rewrite logxor-fl-2-rewrite))

(local-defthm log3-1
    (implies (and (natp x) (natp y) (natp z)
		  (equal (log3a (fl (/ x 2)) (fl (/ y 2)) (fl (/ z 2)))
			 (log3b (fl (/ x 2)) (fl (/ y 2)) (fl (/ z 2)))))
	     (equal (log3a x y z)
		    (log3b x y z)))
  :rule-classes ()
  :hints (("Goal" :in-theory (enable logior-mod-2 logand-mod-2 logxor-mod-2
				     logand-fl-2-rewrite logior-fl-2-rewrite logxor-fl-2-rewrite)
		  :use (mod-mod-2
			(:instance mod-mod-2 (x y))
			(:instance mod-mod-2 (x z))
			(:instance quot-mod (m (log3a x y z)) (n 2))
			(:instance quot-mod (m (log3b x y z)) (n 2))))))

(defun logop-induct (x y z)
  (declare (xargs :measure (+ (nfix x) (nfix y) (nfix z))))
  (if (and (natp x) (natp y) (natp z))
      (if (and (zp x) (zp y) (zp z))
	  t
	(logop-induct (fl (/ x 2)) (fl (/ y 2)) (fl (/ z 2))))
    t))

(defthm log3
    (implies (and (natp x) (natp y) (natp z))
	     (equal (logior (logand x y) (logior (logand x z) (logand y z)))
		    (logior (logand x y) (logand (logxor x y) z))))
  :rule-classes ()
  :hints (("Goal" :induct (logop-induct x y z))
	  ("Subgoal *1/2" :use (log3-1))))


(local-defthm hack-3
    (implies (and (natp k)
		  (natp n)
		  (> k n)
		  (natp y))
	     (NATP (* Y (EXPT 2 (+ -1 K (* -1 N))))))
  :hints (("Goal" :in-theory (union-theories (disable a14 expt-2-integerp) '(natp))
		  :use ((:instance expt-2-integerp (i (- k (1+ n))))))))

(defthm bits-plus-mult-2
    (implies (and (natp m)
		  (natp n)
		  (>= n m)
		  (natp k)
		  (> k n)
		  (natp y)
		  (natp x))
	     (= (bits (+ x (* y (expt 2 k))) n m)
		(bits x n m)))
    :rule-classes ()
    :hints (("Goal" :in-theory (disable a14)
		    :use ((:instance mod-bits-equal
				     (y (+ x (* y (expt 2 k))))
				     (i n)
				     (j m))
			  (:instance mod-mult
				     (m x)
				     (n (expt 2 (1+ n)))
				     (a (* (expt 2 (- k (1+ n))) y)))
                          ))))

(defthm mod-mod-sum
    (implies (and (natp a)
		  (natp b)
		  (natp n))
	     (equal (mod (+ (mod a n) (mod b n)) n)
		    (mod (+ a b) n)))
  :hints (("Goal" :use (mod-sum
			(:instance mod-sum (a (mod b n)) (b a))))))



(defthm exact-k+1
    (implies (and (natp n)
		  (natp x)
		  (= (expo x) (1- n))
		  (natp k)
		  (< k (1- n))
		  (exactp x (- n k)))
	     (iff (exactp x (1- (- n k)))
		  (= (bitn x k) 0)))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable exactp2 exactp2-lemma)
           :use (exact-bits-b-d
			(:instance exact-bits-b-d (k (1+ k)))
			(:instance bits-0-bitn-0 (n k))))))

;(in-theory (disable comp1))

(in-theory (disable bits-tail bits-reduce))

;move up?
(defthm bvecp<=
    (implies (and (natp n)
		  (bvecp x n))
	     (<= x (1- (expt 2 n))))
  :rule-classes ()
  :hints (("Goal" :in-theory (enable bvecp))))

;leave this disabled!
(defthm bits-plus-bitn-rewrite
    (implies (and (natp x)
		  (natp m)
		  (natp n)
		  (> n m))
	     (equal (bits x n m)
		    (+ (* (bitn x n) (expt 2 (- n m)))
		       (bits x (1- n) m))))
  :hints (("Goal" :use (bits-plus-bitn))))
(in-theory (disable bits-plus-bitn-rewrite))




;===


(defthm comp1-fl-rewrite
    (implies (and (not (zp n))
		  (bvecp x n))
	     (equal (fl (* 1/2 (comp1 x n)))
		    (comp1 (fl (* 1/2 x)) (1- n))))
  :hints (("Goal" :use ((:instance comp1-fl (k 1))))))

(defthm comp1-mod-2
    (implies (and (not (zp n))
		  (bvecp x n))
	     (equal (mod (comp1 x n) 2)
		    (comp1 (mod x 2) 1)))
  :hints (("Goal" :use (mod-comp1-2
			mod-mod-2
			(:instance mod-mod-2 (x (comp1 x n)))))))

(in-theory (disable comp1-fl-rewrite comp1-mod-2))

(in-theory (enable logxor-bvecp))

(local (in-theory (disable COMP1-2+1)))

(local-defthm comp1-logxor-1
    (implies (and (not (zp n))
		  (bvecp x n)
		  (bvecp y n)
		  (equal (comp1 (logxor (fl (/ x 2)) (fl (/ y 2))) (1- n))
			 (logxor (comp1 (fl (/ x 2)) (1- n)) (fl (/ y 2)))))
	     (equal (comp1 (logxor x y) n)
		    (logxor (comp1 x n) y)))
  :rule-classes ()
  :hints (("Goal" :in-theory (enable logand-mod-2 logxor-mod-2 comp1-mod-2 comp1-bvecp
				     comp1-fl-rewrite logand-fl-2-rewrite logxor-fl-2-rewrite)
		  :use (mod-mod-2
			(:instance mod-mod-2 (x y))
			(:instance quot-mod (m (comp1 (logxor x y) n)) (n 2))
			(:instance quot-mod (m (logxor (comp1 x n) y)) (n 2))))))

(defun logop2-induct (x y n)
  (if (zp n)
      (cons x y)
    (logop2-induct (fl (/ x 2)) (fl (/ y 2)) (1- n))))

;move up?
(defthm bvecp-fl
    (implies (and (not (zp n))
		  (bvecp x n))
	     (bvecp (fl (* 1/2 x)) (1- n)))
  :hints (("Goal" :in-theory (enable bvecp expt))))

(defthm bvecp-0-thm
  (equal (bvecp x 0)
         (equal x 0))
  :rule-classes ()
  :hints (("Goal" :in-theory (enable bvecp))))

(defthm comp1-logxor
  (implies (and (natp n)
                (bvecp x n)
                (bvecp y n))
           (equal (comp1 (logxor x y) n)
                  (logxor (comp1 x n) y)))
  :hints (("Goal" :induct (logop2-induct x y n))
	  ("Subgoal *1/2" :use (comp1-logxor-1))))


(in-theory (disable comp1-logxor))


(defthm bitn-rec-pos-def
    (implies (and (natp x)
		  (natp k)
		  (> k 0))
	     (equal (bitn x k)
		    (bitn (fl (/ x 2)) (1- k))))
  :rule-classes ((:definition :controller-alist ((bitn t t))))
  :hints (("Goal" :by bitn-rec-pos)))

(in-theory (disable bitn-rec-pos-def))

(defthm logior-x-1
    (implies (bvecp x 1)
	     (equal (logior x 1) 1))
  :hints (("Goal" :use ((:instance logior-ones (n 1))))))

(defthm logior-1-x
    (implies (bvecp x 1)
	     (equal (logior 1 x) 1))
    :hints (("Goal" :use ((:instance logior-commutative (y 1))))))


;move to float? ;rephrase?
(defthm exactp-shift-iff
    (implies (and (rationalp x)
		  (integerp m)
		  (integerp n))
	     (iff (exactp x m)
		  (exactp (* (expt 2 n) x) m)))
  :rule-classes ()
  :hints (("Goal" :use (exactp-shift
			(:instance exactp-shift (x (* (expt 2 n) x)) (n (- n)))))))



;move to basic


(defthm natp-mod
  (implies (and (natp m)
                (natp n))
           (natp (mod m n)))
  :rule-classes :type-prescription
  :hints (("Goal" :use mod>=0)))

;may be very expensive if we backchain from rationalp to integerp
(defthm integerp-rationalp
    (implies (integerp x)
	     (rationalp x)))

;move to wherever bvecp is defined
;may cause expensive backchaining
(defthm natp-bvecp
    (implies (bvecp x n)
	     (natp x)))




;moved bvecp-exactp to merge.lisp


(defthm exact-bits-1
    (implies (and (natp x)
		  (natp n)
		  (natp k)
		  (= (expo x) (1- n))
		  (< k n))
	     (iff (integerp (/ x (expt 2 k)))
		  (exactp x (- n k))))
  :rule-classes ()
  :hints (("Goal" :in-theory (enable natp)
		  :use exact-bits-a-b)))

(defthm exact-bits-2
    (implies (and (natp x)
		  (natp n)
		  (natp k)
		  (= (expo x) (1- n))
		  (< k n))
	     (iff (integerp (/ x (expt 2 k)))
		  (= (bits x (1- n) k)
		     (/ x (expt 2 k)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (enable natp)
		  :use exact-bits-a-c)))

(defthm exact-bits-3
    (implies (and (natp x)
		  (natp n)
		  (natp k)
		  (= (expo x) (1- n))
		  (< k n))
	     (iff (integerp (/ x (expt 2 k)))
		  (= (bits x (1- k) 0)
		     0)))
  :rule-classes ()
  :hints (("Goal" :in-theory (enable natp)
		  :use exact-bits-a-d)))

(defthm expt+
    (implies (and (integerp n)
		  (integerp m))
	     (= (* (expt 2 m) (expt 2 n))
		(expt 2 (+ m n))))		
  :rule-classes ()
  :hints (("Goal" :use expo+)))

;modoved expt-weak-monotone

(defthm logior-0-y
    (implies (natp y)
	     (equal (logior 0 y) y)))

(local
(defthm logior-x-x-1
    (implies (and (integerp x) (>= x 0))
	     (equal (logior (mod x 2) (mod x 2)) (mod x 2)))
  :rule-classes ()
  :hints (("Goal" :use ((:instance mod-mod-2))))))

(local
 (defthm fl-mod-equal
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (= (fl (/ x 2)) (fl (/ y 2)))
		  (= (mod x 2) (mod y 2)))
	     (= x y))
  :rule-classes ()
  :hints (("Goal" :use ((:instance mod-fl (m x) (n 2))
			(:instance mod-fl (m y) (n 2)))))))

(defun logop-2-induct (x y)
  (if (or (zp x) (zp y))
      ()
    (logop-2-induct (fl (/ x 2)) (fl (/ y 2)))))

(defthm logior-self
    (implies (natp x)
	     (equal (logior x x) x))
  :hints (("Goal" :induct (logop-2-induct x x))
	  ("Subgoal *1/2" :in-theory (enable logior-rewrite)
			  :use ((:instance logior-x-x-1)
				(:instance fl-mod-equal (y (logior x x)))))))

(defun logop-2-n-induct (x y n)
  (if (zp n)
      (cons x y)
    (logop-2-n-induct (fl (/ x 2)) (fl (/ y 2)) (1- n))))







(defthm roundup-1
    (implies (and (integerp x)
		  (integerp k))
	     (equal (bitn (* X (EXPT 2 K) (EXPT 2 (* -1 K))) n)
		    (bitn x n)))
  :rule-classes ())

(defthm roundup-2
    (implies (and (natp x)
		  (natp (* x (expt 2 k)))
		  (natp n)
		  (natp (+ n k))
		  (integerp k)
		  (<= k 0))
	     (= (bitn (* x (expt 2 k)) (+ n k))
		(bitn x n)))
  :rule-classes ()
  :hints (("Goal" :in-theory (enable natp)
		  :use ((:instance bitn-shift (x (* x (expt 2 k))) (n (+ n k)) (k (- k)))))
	  ("Goal'''" :use (roundup-1))))

(defthm bitn-shift-gen
    (implies (and (natp x)
		  (natp (* x (expt 2 k)))
		  (natp n)
		  (natp (+ n k))
		  (integerp k))
	     (= (bitn (* x (expt 2 k)) (+ n k))
		(bitn x n)))
  :rule-classes ()
  :hints (("Goal" :in-theory (enable natp)
		  :use (bitn-shift
			roundup-2))))

;; mulcat

(local (include-book "mulcat"))

(defun mulcat (l n x)
  (if (and (integerp n) (> n 0))
      (cat (mulcat l (1- n) x)
	   x
	   l)
    0))

;proved in mulcat.lisp
(defthm mulcat-1
  (implies (and (case-split (integerp x))
                (case-split (<= 0 x)))
           (equal (mulcat l 1 x) x)))



;proved in mulcat.lisp
(defthm mulcat-bvecp
   (implies (and (>= p (* l n))
                 (case-split (integerp p))
                 (case-split (natp l))
                 (case-split (bvecp x l))
                 )
            (bvecp (mulcat l n x) p)))

;proved in mulcat.lisp
(defthm mulcat-nonnegative-integer
  (and (integerp (mulcat l n x))
       (<= 0 (mulcat l n x))))

;proved in mulcat.lisp
(defthm mulcat-0
  (equal (mulcat l n 0) 0))

;proved in mulcat.lisp
(defthm mulcat-n-1
  (implies (case-split (<= 0 n))
           (equal (mulcat 1 n 1)
                  (1- (expt 2 n)))))

(defun mulcat-induct (n n2)
  (IF (AND (INTEGERP N) (> N 0) (INTEGERP N2) (> N2 0))
      (mulcat-induct (+ -1 n) (+ -1 n2))
      0))

;prove a bits-mulcat? could be used to prove-bitn-mulcat
;add to lib!
(defthm bitn-mulcat-1
  (implies (and (natp x)
                (integerp n) ;(natp n)
                (natp n2)
                (< n2 n)
                (case-split (bvecp x 1))
                )
           (equal (BITN (MULCAT 1 n x) n2)
                  x))
  :hints (("Goal" :induct (mulcat-induct  n n2)
           :do-not '(generalize)
           :in-theory (enable mulcat))))




(defthm logand-1-x
    (implies (bvecp x 1)
	     (equal (logand 1 x) x))
  :hints (("Goal" :in-theory (enable bvecp-1-rewrite))))

;;These go in the CAT section of "lib/bits.lisp":

(defthm cat-bits-bits
    (implies (and (>= i j)
		  (= j (1+ k))
		  (>= k l)
		  (= n (1+ (- k l)))
                  (natp x)
		  (natp i)
		  (natp j)
		  (natp k)
		  (natp l)
		  )
	     (equal (cat (bits x i j) (bits x k l) n)
		    (bits x i l)))
  :hints (("Goal" :in-theory (set-difference-theories
                              (enable cat)
                              ;; Some change during the development of ACL2
                              ;; Version 2.6, probably the change in
                              ;; assume-true-false, required the following to
                              ;; be disabled.
                              '(rearrange-negative-coefs-<))
		  :use ((:instance bits-plus-bits (m (1+ i)) (n j) (r l))))))

(defthm cat-bitn-bits
    (implies (and (= j (1+ k))
		  (>= k l)
		  (= n (1+ (- k l)))
                  (natp x)
		  (natp j)
		  (natp k)
		  (natp l)
		  )
	     (equal (cat (bitn x j) (bits x k l) n)
		    (bits x j l)))
  :hints (("Goal" :in-theory (enable cat)
		  :use ((:instance bits-plus-bitn (n j) (m l))
			(:instance expt+ (n (- k l)) (m 1))))))

(defthm cat-bits-bitn
    (implies (and (>= i j)
		  (= j (1+ k))
                  (natp x)
		  (natp i)
		  (natp j)
		  (natp k)
		  )
	     (equal (cat (bits x i j) (bitn x k) 1)
		    (bits x i k)))
  :hints (("Goal" :in-theory (enable cat)
		  :use ((:instance bitn-plus-bits (n i) (m k))))))



;perhaps use only expt-2-positive-integer-type;
(defthm natp-expt
  (implies (natp n)
           (and (integerp (expt 2 n))
                (< 0 (expt 2 n))))
  :rule-classes (:type-prescription :rewrite))

