
;   hons-help.lisp                                Boyer & Hunt

(in-package "ACL2")

; In this file one may find some helpful functions and lemmas in the
; "HONS School", but none of them require "under the hood"
; definitions.  That is, the "user" could do all this by himself.

(defmacro with-memoize (fn form)
  `(let ((fn ,fn))
     ((lambda (x y)
        (declare (ignore x))
        y)
      (memoize! fn)
      ((lambda (u v)
         (declare (ignore v))
         u)
       ,form (unmemoize! fn)))))

(defmacro with-fast-list (var term name form)
  `(let ((,var (build-fast-alist-from-list
                ,term 
                ,name)))
     (ansfl ,form ,var)))

(defmacro with-fast-alist (var l1 l2 name form)
  `(let ((,var (hons-put-list ,l1 ,l2 ,name)))
     (ansfl ,form ,var)))

; A new kind of association list.

(defmacro hons-list (&rest x)
  (if (atom x) nil
    (list 'hons (car x) (cons 'hons-list (cdr x)))))

(defmacro hons-list* (&rest x)
  (cond ((atom x) x)
        ((atom (cdr x)) (car x))
        (t (list 'hons (car x) (cons 'hons-list* (cdr x))))))

(defmacro defn (f a &rest r)
  `(defun ,f ,a (declare (xargs :guard t)) ,@r))

(defn defnp (x)
  (and (consp x)
       (symbolp (car x))
       (eq (car x) 'defn)
       (consp (cdr x))
       (symbolp (cadr x))
       (consp (cddr x))
       (symbol-listp (caddr x))
       (consp (cdddr x))
       (true-listp (cdddr x))))

(defn defn-listp (x)
  (if (atom x)
      (null x)
    (and (defnp (car x))
         (defn-listp (cdr x)))))

(defun mu-defn-fn (l)
  (declare (xargs :guard (defn-listp l)))
  (if (atom l) nil
    (cons `(defun
             ,(cadr (car l))
             ,(caddr (car l))
             (declare (xargs :guard t))
             ,@(cdddr (car l)))
          (mu-defn-fn (cdr l)))))

(defmacro mu-defn (&rest l)
  `(mutual-recursion ,@(mu-defn-fn l)))


; HONS-LEN1 and HONS-LEN defined to make a tail-recursive version of
; LEN.

(defun hons-len1 (x acc)
  (declare (xargs :guard (and (integerp acc)
                              (<= 0 acc))))
  (if (atom x)
      acc
    (hons-len1 (cdr x) (+ 1 acc))))

(defn hons-len (x)
  (hons-len1 x 0))

(defun cons-subtrees (x al)
  ;; (cons-subtrees x NIL) is an alist that associates each subtree
  ;; of x with t, without duplication.
  (declare (xargs :guard t
                  :guard-hints
                  (("Goal" :in-theory (e/d () (hons-acons))))
                  :verify-guards nil))
  (cond ((atom x) al)
        ((hons-get x al) al)
        (t (cons-subtrees
            (car x)
            (cons-subtrees (cdr x)
                           (hons-acons x t al))))))

(verify-guards cons-subtrees)

;  By implementing NUMBER-SUBTREES in "hons-raw.lisp" we could
;  eliminate the need to build the association list altogether and
;  just use the Common Lisp function HASH-TABLE-COUNT after entering
;  each unique tree into a hash table.

(defun prg1 (x y) ; sym-eval assumes this is the definition of prg1
  (declare (ignore y) (xargs :guard t))
  x)

(defabbrev ansfl (x y) (prg1 x (flush-hons-get-hash-table-link y)))

(defn number-subtrees (x)
  (hons-len (flush-hons-get-hash-table-link
             (cons-subtrees x 'number-subtrees))))

(defn hons-member-equal (x y)
  (cond ((atom y) nil)
        ((hons-equal x (car y)) y)
        (t (hons-member-equal x (cdr y)))))

(defn hons-binary-append (x y)
  (if (atom x)
      y
    (hons (car x) (hons-binary-append (cdr x) y))))

(defmacro hons-append (x y &rest rst)
  "APPEND using HONS instead of CONS"
  (xxxjoin 'hons-binary-append
           (cons x (cons y rst))))

(defn hons-revappend (x y)
  (if (atom x) y
    (hons-revappend (cdr x) (hons (car x) y))))

(defn hons-reverse (x)
  (if (stringp x) (reverse x)
    (hons-revappend x nil)))

(defthm alistp-hons-revappend
  (implies (and (alistp x)
                (alistp y))
           (alistp (hons-revappend x y))))

; The functions HONS-GETPROP and HONS-PUTPROP support fast property
; lists for any type of keys, not just symbols.  With HONS-PUTPROP you
; can cause X to have the value VAL under the key Y, and with
; HONS-GETPROP you can later ask for the value of X under the key Y and
; get back VAL.  As usual in Lisp, there is potential confusion over
; whether NIL is a value of an indication of no value.

(defn hons-getprop (x y al)
  (cdr (hons-get (hons x y) al)))

(defn hons-putprop (x y val al)
  (hons-acons (hons x y) val al))

(defthm key-theorem-about-hons-getprop-and-hons-putprop
  (equal (hons-getprop x1 y1 (hons-putprop x2 y2 val al))
	 (if (and (equal x1 x2)
		  (equal y1 y2))
	     val
	   (hons-getprop x1 y1 al))))
	   

; Some "fast" operations for "set" intersection, union, and set-diff
; intended for use on lists of ACL2 objects without duplications.

(defn hons-put-list (keys values l)
  ;; If there are not enough values, the last atom of values is used
  ;; for the remaining values.  If there are not as many keys as
  ;; values, the extra values are ignored.
  (if (atom keys)
      l
    (hons-put-list (cdr keys)
                   (if (consp values) (cdr values) values)
                   (hons-acons (car keys)
                               (if (consp values) (car values) values)
                               l))))

(defn build-fast-alist-from-list (l acc)
  (hons-put-list l t acc))

(defn build-fast-alist-from-alist (l acc)
  (cond ((atom l) acc)
        ((consp (car l))
         (hons-acons (caar l) (cdar l)
                     (build-fast-alist-from-alist (cdr l) acc)))
        (t (build-fast-alist-from-alist (cdr l) acc))))

(defn build-fast-alist-from-alist! (l acc)
  (cond ((atom l) acc)
        ((consp (car l))
         (hons-acons! (caar l) (cdar l)
                     (build-fast-alist-from-alist! (cdr l) acc)))
        (t (build-fast-alist-from-alist! (cdr l) acc))))

(defn rev-build-fast-alist-from-alist (l acc)
  (cond ((atom l) acc)
        ((consp (car l))
         (build-fast-alist-from-alist
          (cdr l)
          (hons-acons (caar l) (cdar l) acc)))
        (t (build-fast-alist-from-alist (cdr l) acc))))

(defn rev-build-fast-alist-from-alist! (l acc)
  (cond ((atom l) acc)
        ((consp (car l))
         (build-fast-alist-from-alist!
          (cdr l)
          (hons-acons! (caar l) (cdar l) acc)))
        (t (build-fast-alist-from-alist! (cdr l) acc))))

(defun worth-hashing1 (l n)
  (declare (type (integer 0 18) n)
           (xargs :guard t))
  (cond ((eql n 0) t)
        ((atom l) nil)
        (t (worth-hashing1 (cdr l) (the (integer 0 18) (1- n))))))

(defn worth-hashing (l)
  (worth-hashing1 l 18))

(defn hons-intersection1 (l1 al2)
  (cond ((atom l1) nil)
        ((hons-get (car l1) al2)
         (cons (car l1) (hons-intersection1 (cdr l1) al2)))
        (t (hons-intersection1 (cdr l1) al2))))

(defn hons-intersection2 (l1 l2)
  (cond ((atom l1) nil)
        ((hons-member-equal (car l1) l2)
         (cons (car l1) (hons-intersection2 (cdr l1) l2)))
        (t (hons-intersection2 (cdr l1) l2))))

(defn hons-intersection (l1 l2)  ; preserves order of members in l1
  (cond ((worth-hashing l2)
         (with-fast-list fl2 l2 '*hons-intersection-alist*
                         (hons-intersection1 l1 fl2)))
        (t (hons-intersection2 l1 l2))))

(defn hons-set-diff1 (l1 al2)
  (cond ((atom l1) nil)
        ((hons-get (car l1) al2)
         (hons-set-diff1 (cdr l1) al2))
        (t (cons (car l1) (hons-set-diff1 (cdr l1) al2)))))

(defn hons-set-diff2 (l1 l2)
  (cond ((atom l1) nil)
        ((hons-member-equal (car l1) l2)
         (hons-set-diff2 (cdr l1) l2))
        (t (cons (car l1) (hons-set-diff2 (cdr l1) l2)))))

(defn hons-set-diff (l1 l2) ; preserves order of members in l1
  (cond ((worth-hashing l2)
         (with-fast-list fl2 l2 '*hons-set-diff-alist* (hons-set-diff1 l1 fl2)))
        (t (hons-set-diff2 l1 l2))))

(defn safe-append (x y)
  (if (atom x)
      y
    (cons (car x)
          (safe-append (cdr x)
                       y))))

(defn hons-union (l1 l2)  ; l1 followed by relevant members of l2 in order.
  (safe-append l1 (hons-set-diff l2 l1)))

(defn hons-union-list (l)
  (if (atom l) nil (hons-union (car l) (hons-union-list (cdr l)))))

(defn hons-duplicates-1 (l tab)
  (cond ((atom l) (ansfl nil tab))
        ((hons-get (car l) tab) (ansfl t tab))
        (t (hons-duplicates-1 (cdr l) (hons-acons (car l) t tab)))))

(defn hons-duplicates (l)

; returns T or NIL

  (hons-duplicates-1 l '*hons-duplicates*))

(defn hons-remove-duplicates-1 (l tab)
  (cond ((atom l) (ansfl nil tab))
        ((hons-get (car l) tab)
         (hons-remove-duplicates-1 (cdr l) tab))
        (t (cons (car l)
                 (hons-remove-duplicates-1
                  (cdr l)
                  (hons-acons (car l) t tab))))))

(defn hons-remove-duplicates-2 (l seen)
  (cond ((atom l) nil)
        ((hons-member-equal (car l) seen)
         (hons-remove-duplicates-2 (cdr l) seen))
        (t (cons (car l)
                 (hons-remove-duplicates-2
                  (cdr l)
                  (cons (car l) seen))))))

(defn hons-remove-duplicates (l) 

; preserves order, deleting later occurrences

  (cond ((worth-hashing l)
         (hons-remove-duplicates-1 l '*hons-remove-duplicates*))
        (t (hons-remove-duplicates-2 l nil))))

(defn hons-subset1 (l al)
  (or (atom l)
      (and (hons-get (car l) al)
           (hons-subset1 (cdr l) al))))

(defn hons-subset2 (l1 l2)
  (cond ((atom l1) t)
        ((hons-member-equal (car l1) l2)
         (hons-subset2 (cdr l1) l2))))

(defn hons-subset (l1 l2)
  (cond ((worth-hashing l2)
         (with-fast-list fl2 l2 '*hons-subset-alist* (hons-subset1 l1 fl2)))
        (t (hons-subset2 l1 l2))))

(defn hons-set-equal (l1 l2)
  (and (hons-subset l1 l2)
       (hons-subset l2 l1)))

; A merge sort for association lists.

(defn odds1 (x ans)
  (cond ((atom x) ans)
        ((atom (cdr x)) (cons (car x) ans))
        (t (odds1 (cddr x) (cons (car x) ans)))))

(defn evens1 (x ans)
  (cond ((atom x) ans)
        ((atom (cdr x)) ans)
        (t (evens1 (cddr x) (cons (cadr x) ans)))))

(defthm odds1-length
  (implies (and (not (atom x))
                (not (atom (cdr x))))
           (< (len (odds1 x ans))
              (+ (len x)
                 (len ans))))
  :rule-classes :linear)

(defthm evens1-length
  (implies (and (not (atom x))
                (not (atom (cdr x))))
           (< (len (evens1 x ans))
              (+ (len x)
                 (len ans))))
  :rule-classes :linear)

(defun ms-merge (l1 l2 h)
  ;; We assume that L1 and L2 are both sorted with respect to the
  ;; hash table H.
  (declare (xargs :guard t
                  :measure (+ (len l1) (len l2))))
  (cond ((atom l1) l2)
        ((atom l2) l1)
        ((atom (car l1)) nil) ; to help with guards
        ((atom (car l2)) nil)
        ((< (nfix (cdr (hons-get (caar l1) h)))
            (nfix (cdr (hons-get (caar l2) h))))
         (cons (car l1) (ms-merge (cdr l1) l2 h)))
        (t (cons (car l2) (ms-merge l1 (cdr l2) h)))))

(defun merge-sort (a h)
  ;; The arguments are expected to be alists.  H better be backed by a
  ;; hash table, if one is looking for much efficiency.  H maps the car
  ;; of each member of A to a rational.  (MERGE-SORT A H) returns a
  ;; rearrangement of A that is sorted with respect to the values that H
  ;; assigns to the CARs.  This sort may be unstable (who knows) if
  ;; there are pairs whose CARs hash to the same rational.
  (declare (xargs :guard t
                  :verify-guards nil
                  :measure (len a)))
  (if (or (atom a) (atom (cdr a))) a
    (ms-merge (merge-sort (odds1 a nil) h)
              (merge-sort (evens1 a nil) h)
              h)))

(verify-guards merge-sort)

(defn hons-merge-sort (a h)
  (hons-copy (merge-sort a h)))

; This is an "under the hood" remark.  If the system is built with
; *break-honsp* non-NIL, then one will be rudely interrupted whenever
; HONSP returns NIL.  So if you wish to copy a CONS structure into a
; HONS structure, use HONS-COPY-R instead of HONS-COPY.

(defn hons-copy-r (x)
  ;; r stands for recursive
  (if (atom x) x
    (hons (hons-copy-r (car x))
          (hons-copy-r (cdr x)))))

(defn hons-copy-list-r (x)
  ;; r stands for recursive
  (if (atom x) x
    (hons (car x)
          (hons-copy-list-r (cdr x)))))

; There are probably many such lemmas that we could prove...

(defthm symbol-listp-hons-copy-list-r
  (implies (symbol-listp x)
           (symbol-listp (hons-copy-list-r x))))

;;; Defhonst

;; Defhonst is like defconst, but makes sure that a hons-copy of the
;; value is stored, and that the value remains a honsp, even after a
;; call of clear-hash-tables.  To this end, we keep a record of all
;; these values.  We also use that record to help with evisceration.

;; Maybe defhonst should be elevated to an event some day.  If so, then
;; the undoing of the event should probably flush the record for that
;; constant.

;; The record for all defhonst values is kept in the ACL2 global
;; 'defhonst.  To flush all defhonst records manually, one may:
;; (f-put-global 'defhonst nil state).

; !!! Eliminate this plev and evisc stuff when possible in favor of event-level
; macros.

(defabbrev plev1 (length level)
  (er-progn
   (pprogn (let* ((modules
                   (if (boundp-global 'defhonst state)
                       (get-global 'defhonst state)
                     nil))
                  (tuple (list modules level length nil))
                  (ign (plev0 length level tuple)))
             (declare (ignore ign))
             (pprogn
              (f-put-global 'evisc-print-length       length state)
              (f-put-global 'evisc-print-level        level state)
              (f-put-global 'brr-term-evisc-tuple     tuple state)
              (f-put-global 'default-term-evisc-tuple tuple state)
              (f-put-global 'user-default-evisc-tuple tuple state)
              (er-progn (set-ld-evisc-tuple           tuple state)
                        (value nil)))))))

(defmacro plev (&optional (length 'nil) (level 'nil))
  `(plev1 ,length ,level))

(defmacro update-evisc (f r)
  `(let ((f ,f) (r ,r))
     (pprogn
      (f-put-global
       'defhonst
       (hons (hons (cadr r)
                   (concatenate 'string "," (symbol-name f)))
             (if (boundp-global 'defhonst state)
                 (get-global 'defhonst state)
               nil))
       state)
      (er-progn (plev1 (if (boundp-global 'evisc-print-level state)
                           (get-global 'evisc-print-level state)
                         nil)
                       (if (boundp-global 'evisc-print-length state)
                           (get-global 'evisc-print-length state)
                         nil))
                (value f)))))

(defmacro defhonst (name form &key (evisc 'nil eviscp) check doc)
  `(progn (defconst ,name (hons-copy ,form) ,doc)
          (table evisc-table
                 ,name
                 ,(if eviscp
                      evisc
                    (concatenate 'string "#," (symbol-name name))))
          (table persistent-hons-table
                 (let ((x ,name))
                   (if (or (consp x) (stringp x)) ; honsp-check without check
                       x
                     nil))
                 t)
          ,@(and check
                 `((assert-event ,check)))
          (value-triple ',name)))

(defmacro all-memoized-fns (&optional show-conditions)
  (if show-conditions
      '(table-alist 'memoize-table (w state))
    '(strip-cars (table-alist 'memoize-table (w state)))))
