; ACL2 Version 3.1 -- A Computational Logic for Applicative Common
; Lisp Copyright (C) 2006 University of Texas at Austin

; This version of ACL2 is a descendent of ACL2 Version 1.9, Copyright
; (C) 1997 Computational Logic, Inc.  See the documentation topic
; NOTE-2-0.

; This program is free software; you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation; either version 2 of the License, or
; (at your option) any later version.

; This program is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.

; You should have received a copy of the GNU General Public License
; along with this program; if not, write to the Free Software
; Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

; Written by:  Matt Kaufmann               and J Strother Moore
; email:       Kaufmann@cs.utexas.edu      and Moore@cs.utexas.edu
; Department of Computer Sciences
; University of Texas at Austin
; Austin, TX 78712-1188 U.S.A.

; The original version of this file was contributed by Bob Boyer and
; Warren A. Hunt, Jr.  The design of this system of Hash CONS,
; function memoization, and fast association lists (applicative hash
; tables) was initially implemented by Boyer and Hunt.

(in-package "ACL2")

#+openmcl
(eval-when (load eval compile)
  (if (fboundp 'ccl::rdtsc) (pushnew :rdtsc *features*)))

(defconstant most-positive-fixnum/2 (floor most-positive-fixnum 2))

(defmacro defglobal (&rest r)

; In OpenMCL, DEFSTATIC behaves exactly as DEFPARAMETER; however, its
; use includes three promises: (1) never to locally bind the variable,
; e.g., with LET or LAMBDA, (2) never to reference the variable when
; it is not set, which would be an error anyway, of course, and (3)
; never to test whether the variable is BOUNDP.  OpenMCL uses about
; ten fewer machine instructions to reference such a variable.

  #+openmcl `(ccl::defstatic ,@r)
  #-openmcl `(defparameter ,@r))

(defmacro our-syntax (&rest args)
  `(let ((*package*                    *acl2-package*)
         (*print-base*                 10)
         (*print-circle*               nil)
         (*print-length*               nil)
         (*print-level*                nil)
         (*print-lines*                nil)
         (*read-suppress*              nil)
         (*readtable*                  *acl2-readtable*))
     ,@args))

(defglobal *count-hons-calls*                   t)
(defglobal *count-pons-calls*                   t)
(defglobal *break-honsp*                        nil)
(defglobal *hons-report-discipline-failure*     'break)

; Gary Byers recalls Lisp folklore that alists are faster than hash
; tables up to length 18.  If one changes these parameters at runtime,
; then it would be best to start afresh with (unmemoize-all) and
; (init-hash-tables).

(defglobal *start-car-ht-size*            18)
(defglobal *hons-acons-ht-threshold*      18)
(defglobal *small-ht-size*                60)
(defglobal *hons-cdr-ht-size*             (expt 2 20))
(defglobal *nil-ht-size*                  (expt 2 17))
(defglobal *max-memoize-fns*              10)
(defglobal *memoize-grow-size*            10)

(declaim (fixnum *start-car-ht-size* *hons-acons-ht-threshold*
                 *small-ht-size* *hons-cdr-ht-size* *nil-ht-size*
                 *max-memoize-fns* *memoize-grow-size*))

(defglobal *memoize-debug* nil)

(defglobal *memoize-hack-condition* nil)
(defglobal *memoize-hack-inline* nil)

#+cmu
(declaim (type (simple-array fixnum (2))
               *hons-call-counter* *hons-misses-counter*
               *pons-call-counter* *pons-hit-counter*))
#-cmu
(eval-when
 #-cltl2
 (load)
 #+cltl2
 (:load-toplevel)
 #+cmu
 (:execute)
 (progn
   (declaim (type (simple-array fixnum (2))
                  *hons-call-counter* *hons-misses-counter*
                  *pons-call-counter* *pons-hit-counter*))

; These correspond to the under-the-hood definitions of the alist
; shrinking functions.

  ))

;  The Hons Invariants

; If A and B are consp+honsp, then (eq A B) iff (equal A B).
; The car of a consp+honsp is an atom or a consp+honsp.
; The cdr of a consp+honsp is an atom or a consp+honsp.
; No consp+honsp is circular.  If a string occurs in any
; consp+honsp, then no other EQUAL string occurs in any
; consp+honsp.

; memoize-flush 'forgets' all that was remembered for certain
; functions that use certain stobjs.  We must keep memoize-flush very
; fast in execution so as not to slow down stojb update or resize
; operations in general.  We 'forget' the pons table later.

(defmacro memoize-flush (st)
  (let ((s (st-lst st)))
    `(loop for sym in ,s do
           (let ((old (symbol-value (the symbol sym))))
             (unless (or (null old) (empty-ht-p old))
               (setf (symbol-value (the symbol sym)) nil))))))

(defmacro mht (&key (test (quote (function eql)))
                    (size *small-ht-size*)
                    (shared nil)
                    (rehash-size 1.5)
                    (rehash-threshold 0.7)
                    (weak nil))
  #-openmcl (declare (ignore shared weak))
  `(make-hash-table :test             ,test
                    :size             (max *small-ht-size* ,size)
                    :rehash-size ,rehash-size
                    :rehash-threshold ,rehash-threshold
                    #+openmcl :weak   #+openmcl ,weak
                    #+openmcl :shared #+openmcl ,shared))

(declaim (hash-table
          *hons-cdr-ht*
          *hons-cdr-ht-eql*
          *nil-ht* *hons-acons-ht*
          *hons-str-ht*
          *memoize-info-ht*
          *compact-print-file-ht*
          *compact-read-file-ht*))

; Here are some basic data structures for honsing and memoizing.  Some
; of these are significantly expanded in size later by hons-init, but
; there is no reason to clog up saved images with large empty versions
; of them.

; The choice of :weak below deserves careful explanation. !!

(defglobal *hons-cdr-ht*        (mht :test #'eq :weak :key))
(defglobal *hons-cdr-ht-eql*    (mht))
(defglobal *nil-ht*             (mht :weak :value))
(defglobal *hons-acons-ht*      (mht :test #'eq :weak :key))
(defglobal *hons-str-ht*        (mht :test #'equal :weak :value))
(defglobal *hons-copy-aux-ht*   (mht :test #'eq))

(defglobal *memoize-info-ht*    (mht))
(defglobal *memoize-array*
  (make-array 1 :element-type 'fixnum :initial-element 0))

(defmacro str-hash-set (x)
  (check-type x symbol)
  `(setq ,x (maybe-str-hash ,x)))

(defun maybe-str-hash (x)
  (if (typep x '(and array string))
      (or (gethash x *hons-str-ht*)
          (setf (gethash x *hons-str-ht*) x))
    x))

#+cmu
(proclaim `(type (simple-array fixnum (*)) *memoize-array*))
#-cmu
(eval-when
 #-cltl2
 (load eval)
 #+cltl2
 (:load-toplevel :execute)
 (proclaim `(type (simple-array fixnum (*)) *memoize-array*)))

; *float-ticks-per-second* is set correctly by hons-init.
(defglobal *float-ticks-per-second* 1.0)
(declaim (float *float-ticks-per-second*))

(defun run-time-for-fn-in-secs (fn)
  (/ (aref *memoize-array*
           (the fixnum (* *max-memoize-fns*
                          (the fixnum (symbol-to-fixnum fn)))))
     *float-ticks-per-second*))

; internal-run-time uses the X86 RDTSC instruction when available.
(defmacro internal-run-time ()
  #+RDTSC '(the fixnum (ccl::rdtsc))
  #-RDTSC '(the fixnum
             (let ((n (get-internal-run-time)))
               (check-type n integer)
               n)))

(defmacro internal-run-time-in-seconds ()
  (/ (internal-run-time) *float-ticks-per-second*))
     
(defmacro maybe-break-honsp ()
  (cond (*break-honsp* `(progn (break "~&; HONSP returned nil.")))))

(defmacro honsp (x)

; Honsp checks a cons see if it's in our hons tables.
; Honsp assumes x is a consp.

  `(cond ((let* ((x ,x)
                 (ax (car x))
                 (dx (cdr x))
                 (v (cond ((null dx) *nil-ht*)
                          ((consp dx) (gethash dx *hons-cdr-ht*))
                          (t (gethash dx *hons-cdr-ht-eql*)))))
            (if (listp v)
                (let ((av (car v)))
                  (if (typep ax '(or cons symbol (and array string)))
                      (loop (if (eq ax (car av)) (return (eq x av)))
                            (setq v (cdr v))
                            (if (null v) (return nil))
                            (setq av (car v)))
                    (loop (if (eql ax (car av)) (return (eq x av)))
                          (setq v (cdr v))
                          (if (null v) (return nil))
                          (setq av (car v)))))
              (eq x (gethash (car x) v)))))
         (t (maybe-break-honsp) nil)))

; Counters

; When we universally have 64-bit computers, we may eliminate
; these two-word arrays for counting.

(defglobal *hons-call-counter*
  (make-array 2 :element-type 'fixnum :initial-element 0))
(defglobal *hons-misses-counter*
  (make-array 2 :element-type 'fixnum :initial-element 0))
(defglobal *pons-call-counter*
  (make-array 2 :element-type 'fixnum :initial-element 0))
(defglobal *pons-hit-counter*
  (make-array 2 :element-type 'fixnum :initial-element 0))

(defmacro clear-counter (c)
  `(progn (setf (aref ,c 0) 0) (setf (aref ,c 1) 0)))

(defun fincf1-error ()
  (ofe "~&; fincf1:  ** Error:  overflow."))

(defmacro fincf1 (x)
  `(progn (let ((cv (the fixnum (aref ,x 0)))) (declare (fixnum cv))
               (cond ((eql cv most-positive-fixnum)
                      (let ((d (the fixnum (aref ,x 1))))
                        (declare (fixnum d))
                        (if (eql d most-positive-fixnum)
                            (fincf1-error)
                          (setf (aref ,x 1) (the fixnum (+ 1 d))))
                        (setf (aref ,x 0) 0)))
                     (t (setf (aref ,x 0)
                              (the fixnum (+ 1 (the fixnum cv))))))
               nil)))

(defun safe-incf-error ()
  (ofe "~&; safe-incf:  ** Error:  overflow."))

(defmacro safe-incf (x)
  (let ((v (gensym "V")))
    `(let ((,v ,x))
       (declare (fixnum ,v))
       (if (eql ,v most-positive-fixnum)
           (safe-incf-error)
         (the fixnum (setf ,x (the fixnum (1+ ,v))))))))

(defmacro maybe-count-hons-calls ()
  (and *count-hons-calls* '(fincf1 *hons-call-counter*)))
(defmacro maybe-count-hons-misses ()
  (and *count-hons-calls* '(fincf1 *hons-misses-counter*)))
(defmacro maybe-count-pons-calls ()
  (and *count-pons-calls* '(fincf1 *pons-call-counter*)))
(defmacro maybe-count-pons-hits ()
  (and *count-pons-calls* '(fincf1 *pons-hit-counter*)))

(defun assoc-no-error-at-end (x l)
; Assumes that every element of l is consp.
  (if (typep x '(or cons symbol (and array string)))
      (loop (if (consp l)
                (let ((al (car l)))
                  (if (eq x (car al))
                      (return al)
                    (setq l (cdr l))))
              (return nil)))
    (loop (if (consp l)
              (let ((al (car l)))
                (if (eql x (car al))
                    (return al)
                  (setq l (cdr l))))
            (return nil)))))

; HONS

; Definition of normed.  Let us say that an ACL2 Common Lisp object x
; is 'normed' iff both (a) if x is a string, then it is in
; *hons-str-ht* and (b) if x is a cons, then it is also a honsp.
; HONS-COPY and HONS produce normed objects.  A nonhonsp cons whose
; car and cdr are normed is honsed simply by putting it in the right
; place in the hash tables, after first checking that it can be
; legitimately placed there.

(defun hons-normed-with-suggestion (x y nhons)
  (let* ((yt (if (consp y) *hons-cdr-ht* *hons-cdr-ht-eql*))
         (yval (if y (gethash y yt) *nil-ht*))
         (yp (listp yval)))
    (maybe-count-hons-calls)
    (cond ((if yp (assoc-no-error-at-end x yval) (gethash x yval)))
          (t (maybe-count-hons-misses)
             (cond (yp
                    (cond ((>= (length yval) *start-car-ht-size*)
                           (let ((tab (mht :weak :value)))
                             (loop for pair in yval do
                                   (setf (gethash (car pair) tab)
                                         pair))
                             (setf (gethash (car nhons) tab) nhons)
                             (setf (gethash y yt) tab)
                             nhons))
                          (t (setf (gethash y yt) (cons nhons yval))
                             nhons)))
                   (t (setf (gethash x yval) nhons)))))))

(defun hons-normed (x y)
  (let* ((yt (if (consp y) *hons-cdr-ht* *hons-cdr-ht-eql*))
         (yval (if y (gethash y yt) *nil-ht*))
         (yp (listp yval)))
    (maybe-count-hons-calls)
    (cond ((if yp (assoc-no-error-at-end x yval) (gethash x yval)))
          (t (maybe-count-hons-misses)
             (let ((nhons (cons x y)))
               (cond
                (yp
                 (cond ((>= (length yval) *start-car-ht-size*)
                        (let ((tab (mht :weak :value)))
                          (loop for pair in yval do
                                (setf (gethash (car pair) tab)
                                      pair))
                          (setf (gethash (car nhons) tab) nhons)
                          (setf (gethash y yt) tab)
                          nhons))
                       (t (setf (gethash y yt) (cons nhons yval))
                          nhons)))
                (t (setf (gethash x yval) nhons))))))))

; HONS-COPY

; HONS-COPY is partially and temporarily self-memoizing.

; In general, hons-copy has no justification for reusing, much less
; smashing, the conses it is passed.  However, during
; CLEAR-HONS-TABLES, in the rehons phase, such reuse is precisely what
; is needed and permitted.  If some day we learn that no cons in, say,
; (w state) will ever be RPLACAed or RPLACDed and that REPLACAing any
; cons in it with an EQUAL CAR value is ok, and same for RPLACD/CDR,
; then we could probably legitimately absorb the conses in (w state)
; as honses via HONS-COPY1-CONSUME.

(defun uncopy (x)
; For use in hons-copy.
  (unless (atom x)
    (when (remhash x *hons-copy-aux-ht*)
      (uncopy (car x))
      (uncopy (cdr x)))))

(defun hons-copy1-consume (x)
  (cond ((atom x) (maybe-str-hash x))
        ((honsp x) x)
        (t (unless (eql 0 (hash-table-count *hons-copy-aux-ht*))
             ; Only get here because of an error during a hons-copy.
             (setq *hons-copy-aux-ht* (mht :test #'eq)))
           (let ((ans (hons-copy2-consume x)))
             (unless (eql 0 (hash-table-count *hons-copy-aux-ht*))
               (uncopy (car ans))
               (uncopy (cdr ans)))
             ans))))

(defun hons-copy2-consume (x)
  (let ((a (hons-copy3-consume (car x)))
        (d (hons-copy3-consume (cdr x))))
    (or (eql a (car x)) (rplaca x a))
    (or (eql d (cdr x)) (rplacd x d))
    (hons-normed-with-suggestion a d x)))

(defun hons-copy3-consume (x)
  (cond ((atom x) (maybe-str-hash x))
        ((honsp x) x)
        ((gethash x *hons-copy-aux-ht*))
        (t (setf (gethash x *hons-copy-aux-ht*)
                 (hons-copy2-consume x)))))

(defun hons-copy1 (x)
  (cond ((atom x) (maybe-str-hash x))
        ((honsp x) x)
        (t (unless (eql 0 (hash-table-count *hons-copy-aux-ht*))
             ; Only get here because of an error during a hons-copy.
             (setq *hons-copy-aux-ht* (mht :test #'eq)))
           (let ((ans (hons-copy2 x)))
             (unless (eql 0 (hash-table-count *hons-copy-aux-ht*))
               (uncopy (car ans))
               (uncopy (cdr ans)))
             ans))))

(defun hons-copy2 (x)
  (let ((a (hons-copy3 (car x)))
        (d (hons-copy3 (cdr x))))
    (hons-normed a d)))

(defun hons-copy3 (x)
  (cond ((atom x) (maybe-str-hash x))
        ((honsp x) x)
        ((gethash x *hons-copy-aux-ht*))
        (t (setf (gethash x *hons-copy-aux-ht*)
                 (hons-copy2 x)))))

(defun hons-copy (x) ; user visible
  (hons-copy1 x))

(defun hons-when-x-is-honsp (x y)
  (hons-normed x (hons-copy1 y)))

(defun hons-when-y-is-honsp (x y)
  (hons-normed (hons-copy1 x) y))

; HONS-EQUAL is ACL2 user visible, identical in ACL2 logical meaning
; to EQUAL.
(defun hons-equal (x y)
  (cond ((eql x y))
        ((atom x) (equal x y))
        ((atom y) nil)
        ((honsp x)
         (cond ((honsp y) nil)
               (t (and (hons-equal-h1 (car x) (car y))
                       (hons-equal-h1 (cdr x) (cdr y))))))
        ((honsp y)
         (and (hons-equal-h1 (car y) (car x))
              (hons-equal-h1 (cdr y) (cdr x))))
        (t (and (hons-equal (car y) (car x))
                (hons-equal (cdr y) (cdr x))))))

; HONS-EQUAL-H1 is like HONS-EQUAL, but with the assumption that x has
; been normed.
(defun hons-equal-h1 (x y)
  (cond ((eql x y))
        ((atom x) (equal x y))
        ((atom y) nil)
        ((honsp y) nil)
        (t (and (hons-equal-h1 (car x) (car y))
                (hons-equal-h1 (cdr x) (cdr y))))))

; HONS is ACL2 user-visible, as a DEFUN, identical in ACL2 logical
; meaning to CONS.
(defun hons (x y)
  (hons-normed (hons-copy1 x) (hons-copy1 y)))

; HONS-GET, HONS-ACONS and HONS-ACONS!

; HONS-ACONS and HONS-GET provide fast lookup in alists, with
; ASSOC-EQUAL semantics but with the speed of hash tables in some
; cases.  These operations permit one to reasonably efficiently work
; with extremely long alists in some cases.  Informally speaking, each
; HONS-ACONS operation steals the hash table associated with the alist
; that is being extended.  The key is always honsed before the
; hashing operation in HONS-ACONS, and HONS-ACONS!.  In order to take
; advantage of the hash table lookup speed, when coding one must be
; very conscious of which object is the most recent extension of an
; alist and use that extension exclusively.  This may require careful
; passing of the alist up and down through function calls, as with any
; single threaded object in an applicative setting.  There is no
; syntactic enforcement to force one to only use the most recent
; extension of an alist, as there is for single threaded objects.  The
; only penalty for a failure to keep track of the most recent
; extension is a loss of execution speed, not of correctness.  And
; perhaps the annoyance of some warning messages about 'discipline'.

; If we limit ourselves to alists that are recognized by ALISTP, a
; possible gotcha when using HONS-ACONS! is that it might "steal" a
; hash table without one's expecting it.  For example, if you start
; two alists with (HONS-ACONS! '1 '2 nil) and (HONS-ACONS! '1 '2 nil),
; then adding something to the first with HONS-ACONS! will "steal" the
; table associated with what you thought was the second, with the
; result that adding things to the second will result in slow access.
; One can get around this annoyance to some extent by putting
; something 'unique' at the end of the alist, e.g., the final cdr.
; Our (partial) fix for this is to permit the final NIL of an
; HONS-GET/HONS-ACONS! association list to be any symbol, effectively
; naming each association list and maybe preventing the "gotcha" just
; mentioned.

(defmacro maybe-report-discipline-failure (fn)
  (cond (*hons-report-discipline-failure*
         `(cond ((eq *hons-report-discipline-failure* 't)
                 (format *debug-io*
                         "~&; Warning: ~a discipline failure.~%" ',fn)
                 nil)
                ((eq *hons-report-discipline-failure* 'break)
                 (break "~&; Break: ~a discipline failure.~%" ',fn)
                 nil)))))

(defun-one-output hons-get-fn-do-not-hopy (key l)
  (when (atom l) (return-from hons-get-fn-do-not-hopy nil))
  (let (h)
    (when (or (and (consp key) (not (honsp key)))
              (null (setq h (gethash l *hons-acons-ht*))))
      (return-from hons-get-fn-do-not-hopy (hons-assoc-equal key l)))
    (let ((key (hons-copy1 key)))
      (loop
       (cond ((typep h 'fixnum)
              (return (assoc-no-error-at-end key l)))
             (h (return (values (gethash key h))))
             (t (cond ((and (consp (car l))
                            (hons-equal-h1 key (caar l)))
                       (return (car l)))
                      (t (setq l (cdr l))
                         (when (atom l) (return nil))
                         (setq h (gethash (cdr l)
                                          *hons-acons-ht*))))))))))

(defun-one-output hons-get-fn-do-hopy (key l)
  (when (atom l) (return-from hons-get-fn-do-hopy nil))
  (let ((h (gethash l *hons-acons-ht*)))
    (when (null h)
      (maybe-report-discipline-failure 'hons-get-fn-do-hopy))
    (let ((key (hons-copy1 key)))
      (loop
       (cond ((typep h 'fixnum)
              (return (assoc-no-error-at-end key l)))
             (h (return (values (gethash key h))))
             (t (cond ((and (consp (car l))
                            (hons-equal-h1 key (caar l)))
                       (return (car l)))
                      (t (setq l (cdr l))
                         (when (atom l) (return nil))
                         (setq h (gethash (cdr l)
                                          *hons-acons-ht*))))))))))

; Why do we want both HONS-ACONS and HONS-ACONS!, which is the
; HONS'ing version of HONS-ACONS?  On the one hand, since it is quite
; possible that one will not want to look further into the alist that
; is formed, given the fast hashing lookup, one may not wish the
; overhead of HONS'ing it.  On the other hand, if the alist is going
; to be an argument to a function that is to be memoized, then the
; memoization process may hons it -- possibly over and over and over,
; which can be very time consuming if it is very long.

(defun hons-acons (key value l)
  (setq key (hons-copy1 key))
  (let ((ans (cons (cons key value) l)))
    (cond ((atom l)
           (setf (gethash ans *hons-acons-ht*) 0))
          (t (let ((tab (gethash l *hons-acons-ht*)))
               (remhash l *hons-acons-ht*)
               (cond
                ((typep tab 'fixnum)
                 (cond ((< (the fixnum tab) *hons-acons-ht-threshold*)
                        (setf (gethash ans *hons-acons-ht*)
                              (the fixnum (+ 1 tab))))
                       (t (let ((tab (mht :test #'eq :weak :key)))
                            (loop for tail on ans
                                  unless (gethash (caar tail) tab)
                                  do (setf (gethash (caar tail) tab)
                                           (car tail)))
                            (setf (gethash ans *hons-acons-ht*)
                                  tab)))))
                (tab
                 (setf (gethash key tab) (car ans))
                 (setf (gethash ans *hons-acons-ht*) tab))
                (t (maybe-report-discipline-failure 'hons-acons))))))
    ans))

(defun hons-acons! (key value l)
  (setq key (hons-copy1 key))
  (let ((ans (hons-when-x-is-honsp
              (hons-when-x-is-honsp key value)
              l)))
    (cond ((atom l)
           (setf (gethash ans *hons-acons-ht*) 0))
          (t (let ((tab (gethash l *hons-acons-ht*)))
               (remhash l *hons-acons-ht*)
               (cond
                ((typep tab 'fixnum)
                 (cond ((< (the fixnum tab) *hons-acons-ht-threshold*)
                        (setf (gethash ans *hons-acons-ht*)
                              (the fixnum (+ 1 tab))))
                       (t (let ((tab (mht :test #'eq :weak :key)))
                            (loop for tail on ans
                                  unless (gethash (caar tail) tab)
                                  do (setf (gethash (caar tail) tab)
                                           (car tail)))
                            (setf (gethash ans *hons-acons-ht*)
                                  tab)))))
                (tab
                 (setf (gethash key tab) (car ans))
                 (setf (gethash ans *hons-acons-ht*) tab))
                (t (maybe-report-discipline-failure 'hons-acons!))))))
    ans))

(defun hons-shrink-alist-orig (alcdr ans)
  (cond ((atom alcdr) ans)
        (t (let ((p (hons-get (car (car alcdr)) ans)))
             (cond (p (hons-shrink-alist-orig (cdr alcdr) ans))
                   (t (hons-shrink-alist-orig
                       (cdr alcdr)
                       (hons-acons (car (car alcdr))
                                   (cdr (car alcdr))
                                   ans))))))))

(defun hons-shrink-alist-help (alcdr ans tab)
  (if (atom alcdr)
      (progn (setf (gethash ans *hons-acons-ht*) tab)
             ans)
    (let* ((pair (car alcdr))
           ;; (key (hons-copy (car pair)))
           (key (car pair)) ; We know (car pair) is HONS-NORMEDP
           (val (gethash key tab))
           (ans (if val ans (cons (cons key (cdr pair)) ans)))
           (tab (if val tab
                  (progn
                    (setf (gethash key tab) (car ans))
                    tab))))
      (hons-shrink-alist-help (cdr alcdr) ans tab))))

(defun hons-shrink-alist (alcdr ans)
  (if (atom alcdr)
      ans
    (let* ((tab (gethash alcdr *hons-acons-ht*))
           (ans-size
            (if (hash-table-p tab)
                #-openmcl (1+ (ceiling
                               (hash-table-count tab)
                               .7))
                #+openmcl (1+ (hash-table-count tab))
                nil)))
      (if (or (not ans-size) (consp ans))
          (hons-shrink-alist-orig alcdr ans)
        (let ((ans-tab (mht :test #'eq :size ans-size :weak :key)))
          (hons-shrink-alist-help alcdr ans ans-tab))))))

(defun hons-shrink-alist!-orig (alcdr ans)
  (cond ((atom alcdr) ans)
        (t (let ((p (hons-get (car (car alcdr)) ans)))
             (cond (p (hons-shrink-alist!-orig (cdr alcdr) ans))
                   (t (hons-shrink-alist!-orig
                       (cdr alcdr)
                       (hons-acons! (car (car alcdr))
                                    (cdr (car alcdr))
                                    ans))))))))

(defun hons-shrink-alist!-help (alcdr ans tab)
  (if (atom alcdr)
      (progn (setf (gethash ans *hons-acons-ht*) tab)
             ans)
    (let* ((pair (car alcdr))
           ;; (key (hons-copy (car pair)))
           (key (car pair))
           (val (gethash key tab))
           (ans (if val ans (hons-normed
                             (hons-when-x-is-honsp key (cdr pair))
                             ans)))
           (tab (if val tab
                  (progn
                    (setf (gethash key tab) (car ans))
                    tab))))
      (hons-shrink-alist!-help (cdr alcdr) ans tab))))         

(defun hons-shrink-alist!-help-honsp-alcdr (alcdr ans tab)
  (if (atom alcdr)
      (progn (setf (gethash ans *hons-acons-ht*) tab)
             ans)
    (let* ((pair (car alcdr))
           ;; (key (hons-copy (car pair)))
           (key (car pair))
           (val (gethash key tab))
           (ans (if val ans (hons-normed pair ans)))
           (tab (if val tab
                  (progn
                    (setf (gethash key tab) (car ans))
                    tab))))
      (hons-shrink-alist!-help-honsp-alcdr (cdr alcdr) ans tab))))

(defun hons-shrink-alist! (alcdr ans)
  (if (atom alcdr)
      ans
    (let* ((ans (maybe-str-hash ans))
           (tab (gethash alcdr *hons-acons-ht*))
           (ans-size
            (if (hash-table-p tab)
                #-openmcl (1+ (ceiling
                               (hash-table-count tab)
                               .7))
                #+openmcl (1+ (hash-table-count tab))
                nil)))
      (if (or (not ans-size) (consp ans))
          (hons-shrink-alist!-orig alcdr ans)
        (let ((ans-tab (mht :test #'eq :size ans-size :weak :key)))
          (if (honsp alcdr)
              (hons-shrink-alist!-help-honsp-alcdr alcdr ans ans-tab)
            (hons-shrink-alist!-help alcdr ans ans-tab)))))))

; MEMOIZE

; *memoize-info-ht* maps each currently memoized function symbol to a
; list of eight elements:

; tablename      a symbol whose value is the memoize table or is nil;
; ponstablename  a symbol whose value is the the pons table or is nil;
; old-fn         a symbol whose value is the original definition; 
; memoized-fn    a symbol whose value is the new (memoized) function;
; inline         T or NIL;
; condition      the condition function (which can be T or NIL); and 
; num            a nonnegative integer, 'unique' to the function.
; sts            the stobj memotable lists for this function

; *memoize-info-ht* also maps num back to the corresponding symbol.

(defrec memoize-info-ht-entry
  (tablename ponstablename old-fn memoized-fn
             condition inline num sts)
  nil)

(defun memoize-array-grow (n)
  (check-type n fixnum)
  (let* ((nmax (+ *max-memoize-fns* n)))
    (unless (< (* nmax nmax) most-positive-fixnum)
      (ofe "~&; memoize-array-grow: ** Error: most-positive-fixnum ~
            exceeded."))
    (let ((nmax nmax)
          (narray
           (make-array (* nmax nmax)
                       :element-type 'fixnum
                       :initial-element 0)))
      (declare (fixnum nmax)
               (type (simple-array fixnum (*)) narray))
      (loop for i fixnum below *max-memoize-fns* do
            (let ((i1 (* i *max-memoize-fns*))
                  (i2 (* i nmax)))
              (declare (fixnum i1 i2))
              (loop for j fixnum below *max-memoize-fns* do
                    (let ((x (aref *memoize-array*
                                   (the fixnum (+ i1 j)))))
                      (unless (eql x 0)
                        (setf (aref narray
                                    (the fixnum (+ i2 j)))
                              x))))))
      (setq *max-memoize-fns* nmax
            *memoize-array* narray))))

(defglobal *max-symbol-to-fixnum* 2)
(declaim (fixnum *max-symbol-to-fixnum*))
(defun symbol-to-fixnum-create (s)
  (check-type s symbol)
  (let ((g (gethash s *memoize-info-ht*)))
    (if g (access memoize-info-ht-entry g :num)
      (let (new)
        ; We start at three because we use these slots:
        ;  0 for runtime and any outside-caller's id
        ;  1 for hits
        ;  2 for mht creation count
        (loop
         (loop for i from 3 below *max-memoize-fns* do
               (cond ((not (gethash i *memoize-info-ht*))
                      (setq new i)
                      (return nil))))
         (cond (new
                (setq *max-symbol-to-fixnum*
                      (max *max-symbol-to-fixnum* new))
                (return new))
               (t (let ((i *max-memoize-fns*))
                    (memoize-array-grow *memoize-grow-size*)
                    (setq *max-symbol-to-fixnum* i)
                    (return-from symbol-to-fixnum-create i)))))))))

(defun symbol-to-fixnum (s)
  (check-type s symbol)
  (let ((g (gethash s *memoize-info-ht*)))
    (if g (access memoize-info-ht-entry g :num)
      (ofe "~&; symbol-to-fixnum:  ** Error:  illegal symbol:  ~s."
           s))))

(defun fixnum-to-symbol (n) 
  (check-type n fixnum)
  (or (gethash n *memoize-info-ht*)
      (our-syntax
       (ofe "~&; fixnum-to-symbol:  ** Error:  illegal number:  ~d."
            n))))

; This code has the 'feature' that if the condition causes an error,
; so will the memoized function.

(defmacro pist (table &rest x)
  (cond ((atom x) nil)
        (t (list 'pons (car x)
                 (cons 'pist (cdr x)) table))))

(defmacro pist* (table &rest x)
  (cond ((atom x) x)
        ((atom (cdr x)) (car x))
        (t (list 'pons (car x)
                 (cons 'pist* (cons table (cdr x))) table))))

; PONS differs from HONS in that it does not honsify its arguments and
; in that it takes a hash table as a third argument.  We use PONS in
; memoization.

; We use PONS instead of HONS in memoization because we could not
; afford to honsify (using hons-shrink-alist!) certain alists in
; certain biology tests.  About the same time, we (gratuitously)
; decided to stop hons'ifying the output of memoized functions.

(defun pons (x y ht)
  (let ((xval nil)
        (yval nil)
        (ans nil))
    (maybe-count-pons-calls)

; We have taken string normalization out of pons because there might
; be a chance of confusing a 'normal' string with a stobj.

; If CONS exists, then return it.  Does CDR exist in hash table?

    (setq yval (gethash y ht))

; Does CAR exist in hash table?
    (cond (yval
           (cond ((not (consp yval))
                  (setq xval (gethash x yval))
                  (cond (xval
                         (maybe-count-pons-hits)
                         (setq ans xval))))
                 ((setq ans (assoc-no-error-at-end x yval))
                  (maybe-count-pons-hits)))))
    (cond
; If PONS found, then return previous CONS from hash table.
     (ans)
; Otherwise, maybe create new CONS and install in hash table.
     (t 
; Make sure that PONS arguments X and Y are both in the hash
; table.  If not, recursively copy X and Y into hash table.
; Find place for CDR.
      (setq yval (gethash y ht))
; Find place for CAR, and reuse CONS if possible.
      (cond
       ((hash-table-p yval)
        (setq xval (gethash x yval))
        (cond ((not xval)
               (setf (gethash x yval)
                     (setq ans (cons x y))))
              (t (maybe-count-pons-hits)
                 (setq ans xval)))
        ans)
       ((not yval)
        (setq ans (cons x y))
        (setf (gethash y ht)
              (list ans))
        ans)
; Create new CONS and put in hash table.  Remember ASSOC
; used for CDR table entries with less than 10 entries.
       (t (let ((ans (assoc-no-error-at-end x yval)))
            (cond
             (ans
              (maybe-count-pons-hits)
              ans)
             (t (let ((ans (cons (cons x y) yval)))
                  (cond
                   ((> (length ans) *start-car-ht-size*)
                    (let ((tab (mht)))
                      (loop for pair in ans do
                            (setf (gethash (car pair) tab) pair))
                      (setf (gethash y ht) tab)
                      (car ans)))
                   (t (setf (gethash y ht) ans)
                      (car ans)))))))))))))

;  Question:  Is it worth memoizing tail recursive calls?
;  Question:  Could we memoize something like + ?
;  Question:  Could we memoize a function with arrays?

;  Question: Can we make a weaker test than (get fn '*PREDEFINED*) so
;  we can memoize some of the ACL2 pre-defined functions.

;  Matt suggests dumping the test altogether!  But, perhaps later, we
;  should re-instate the check for functions that hereditarily call
;  ASET1 ASET2.  Matt believes that the worst that could happen is 
;  that we could get a slow AREF warning.

;  It would be crazy to memoize any function that we use in the
;  implementation of the memoization of a function.

(defun memoize-eval-compile (defun)
  (let* (#+GCL
         (system:*notify-gbc* nil)
         #+GCL
         (*load-verbose* nil))
    (multiple-value-bind (ans warn-p error-p)
        (compile (eval defun))
      (cond ((or warn-p error-p)
             (our-syntax
              (oft "~&; memoize-eval-compile: **  Problem in ~
                    compilation of~%~s,"
                    (list :defun defun
                          :values-returned-by-compile
                          (list ans warn-p error-p))))))
      ans)))

(defun memoizedp-raw (fn)
  (and (symbolp fn)
       (values (gethash fn *memoize-info-ht*))))

(defun maybe-unmemoize (fn)
  (when (memoizedp-raw fn)
    (unmemoize-fn fn)))

(defun ofe (&rest r)
  (our-syntax
   (apply #'format *error-output* r)
   (error "")))

(defmacro ofn (&rest r)
  `(format nil ,@r))

(defmacro oft (&rest r)
  `(format t ,@r))

(defun suffix (str sym)
  (check-type str string)
  (check-type sym symbol)
  (let ((spkn (package-name (symbol-package sym)))
        (sn (symbol-name sym)))
    (ofn "~s,~s,~s" str spkn sn)))

; HONS-GENTEMP
(defglobal *hons-gentemp-counter* 0)
(declaim (integer *hons-gentemp-counter*))
(defun-one-output hons-gentemp (root)
  (check-type root string)
  (loop
   (incf *hons-gentemp-counter*)
   (let ((name (ofn "HONS-G-~s,~s" root *hons-gentemp-counter*)))
     (multiple-value-bind (sym status)
         (intern name (find-package "ACL2_INVISIBLE"))
       (if (null status) (return sym))))))

; st-lst returns a symbol whose value is a list in which are saved the
; names of the memoize tables that will be set to nil whenever the
; stobj st is changed.
(defun st-lst (st)
  (check-type st symbol)
  (multiple-value-bind (symbol status)
      (intern (ofn "HONS-S-~s,~s"
                   (package-name (symbol-package st))
                   (symbol-name st))
              (find-package "ACL2_INVISIBLE"))
    (or status (eval `(defglobal ,symbol nil)))
    symbol))

(defun dcls (l)
     (loop for dec in l nconc
           (let ((temp
                  (if (consp dec)
                      (loop for d in (cdr dec) nconc
                            (if (and (consp d) (eq (car d) 'ignore))
                                nil
                              (cons d nil))))))
             (if temp (cons (cons 'declare temp) nil)))))

(defun timer-error ()
  (ofe "~&; timer-error:  ** Error."))

(defglobal *memoize-fn-reject-ht*
  (let ((h (mht :test #'eq)))

; We prohibit the memoization of some functions to avoid
; circularities in the memoization machinery.

    (loop for sym in 
          '(#+rdtsc ccl::rdtsc
            honsp-check hons-equal
            pons
            safe-incf-error
            timer-error
            1+ + * - <= < > >= =
            aref arrayp atom apply
            car cdr cons consp
            eq eql error
            format
            #-rdtsc get-internal-run-time
            gethash
            hash-table-p
            list
            make-hash-table
            max
            not null
            prin1 princ print print-object
            stringp
            svref symbolp
            typep
            write-char write-byte write)
          do (setf (gethash sym h) t))
    h))

; MEMOIZE-FN

(defglobal *memoize-fn-cl-error-msg*
 "
 ; Memoizing a function in the common-lisp package is a violation of
 ; the rules of Common Lisp, and consequently this ACL2 session is
 ; unsound, but we continue for the fun of it.  From an engineering
 ; standpoint, it is reasonable to do such memoizing for experimental
 ; reasons, just as one might trace such a function, which, strictly
 ; speaking, is also forbidden.")

; *caller* is bound by memoized functions, so we use defparameter
; rather than defglobal.
(defparameter *caller* 0)
(declaim (fixnum *caller*))

; The condition parameter of memoize-fn can either be t, or a function
; symbol defined by the user within the ACL2 loop, or a listp (consp
; or nil).  In the last case we think of the condition as an
; expression in the formals of fn.  If the inline parameter of
; memoize-fn is t, then the body of fn is placed inline in the
; memoized definition, whereas if the inline parameter is nil, a
; funcall of the original function is placed inline.

(defun memoize-fn
  (fn &key
   (condition  t)
   (inline     t)
   (cl-defun   nil cl-defun-supplied-p)
   (formals    t   formals-supplied-p)
   (stobjs-in  t   stobjs-in-supplied-p)
   (stobjs-out t   stobjs-out-supplied-p))
 (with-warnings-suppressed
  (unless (and (symbolp fn) (fboundp fn))
    (ofe "~%; Memoize-fn:  ** Error: ~a is not an fboundp symbol."
         fn))
  (when (memoizedp-raw fn)
    (ofe "~%; Memoize-fn:  ** Error:  ~a is already memoized." fn))
  (when (macro-function fn)
    (ofe "~%; Memoize-fn:  ** Error:  ~a is a macro." fn))
  (when (special-form-or-op-p fn)
    (ofe "~%; Memoize-fn:  ** Error:  ~a is special." fn))
  (when (gethash fn *memoize-fn-reject-ht*)
    (ofe "~%; Memoize-fn:  ** Error:  ~a is used in memoization."
         fn))
  (let*
   ((state *the-live-state*)
    (w (w state))
    (cl-defun (if cl-defun-supplied-p cl-defun
                (cltl-def-from-name fn nil w)))
    (formals (if formals-supplied-p formals
               (getprop fn 'formals t 'current-acl2-world w)))
    (stobjs-in (if stobjs-in-supplied-p stobjs-in                   
                 (getprop fn 'stobjs-in t 'current-acl2-world w)))
    (stobjs-out (if stobjs-out-supplied-p stobjs-out
                  (getprop fn 'stobjs-out t 'current-acl2-world w))))
   (when (and (null cl-defun) inline)
     (ofe "~%; Memoize-fn:  ** Error:  ~a lacks a cltl-def." fn))
   (when (eq t formals)
     (ofe "~%; Memoize-fn:  ** Error: ~a lacks a formals property."
          fn))
   (when (eq t stobjs-in)
     (ofe "~%; Memoize-fn:  ** Error: ~a lacks a stobjs-in property."
          fn))
   (when (eq t stobjs-out)
     (ofe "~%; Memoize-fn: ** Error: ~a lacks a stobjs-out property."
          fn))
   (when (or (member 'state stobjs-in) (member 'state stobjs-out))
     (ofe "~%; Memoize-fn: ** Error:  ~a uses STATE." fn))
   (when (eq (symbol-package fn) *main-lisp-package*)
     (oft "~&; Memoize-fn:  A warning about the memoization of ~a."
          fn)
     (oft *memoize-fn-cl-error-msg*)
     (f-put-global 'certify-book-disabledp t state))
   (when (member fn (trace))
     (oft "~%; Memoize-fn:  Untracing ~a before memoizing it." fn)
     (eval `(untrace ,fn)))
  (let*
   ((*acl2-unwind-protect-stack*
     (cons nil *acl2-unwind-protect-stack*))
    (old-fn (eval `(defglobal ,(hons-gentemp (suffix "ORIG-DEF-" fn))
                     ,(symbol-function fn))))
    (body
     (if inline (car (last cl-defun)) `(funcall ,old-fn ,@formals)))
    (condition-body
     (cond ((or (eq condition t) (eq condition nil)) condition)
           ((symbolp condition)
            (car (last (cltl-def-from-name condition nil w))))
           (t condition)))
    (dcls (dcls (cdddr (butlast cl-defun))))
    (fnn (symbol-to-fixnum-create fn))

; The variable outside is bound by any memoized function, so we use
; defparameter rather than defglobal.

    (outside
     (eval
      `(defparameter ,(hons-gentemp (suffix "OUTSIDE-~s-" fn)) t)))
    (tablename
     (eval 
      `(defglobal ,(hons-gentemp (suffix "MEMOIZE-HT-FOR-~s-" fn))
         nil)))
    (ponstablename
     (eval `(defglobal ,(hons-gentemp (suffix "PONS-HT-FOR-~s-" fn))
              nil)))
    (memoized-fn (eval `(defglobal ,(gensym "MEMOIZED-DEF") nil)))
    (start         (gensym "START"))
    (caller-callee (gensym "CALLER-CALLEE"))
    (ans           (gensym "ANS"))
    (ans-p         (gensym "ANS-P"))
    (ma            (gensym "MA"))
    (caller        (gensym "CALLER"))
    (sts (loop for x in (union stobjs-in stobjs-out) when x
               collect (st-lst x)))
    (args          (gensym "ARGS"))
    argq defun success)
   (setq argq (if (cdr formals) args (car formals)))
   (setq defun
    `(defun ,fn ,formals ,@dcls (declare (ignorable ,@formals))
       (let* ((,start (if ,outside (internal-run-time) -1))
              (,caller-callee (+ *caller* ,fnn))
              (,ma *memoize-array*)
              (,caller (the fixnum (* *max-memoize-fns* ,fnn)))
              (*caller* ,caller)
              ,outside ,ans ,ans-p ,args)
         (declare (ignorable ,ans ,ans-p ,args)
                  (fixnum ,start ,caller-callee ,caller)
                  (type (simple-array fixnum (*)) ,ma))
         (safe-incf (aref ,ma ,caller-callee))
         (multiple-value-prog1
          (cond
           ((not ,condition-body) ,body)
           (t
            (when (null ,tablename)
              (safe-incf (aref ,ma (the fixnum (+ ,caller 2))))
              (setq ,tablename (mht))
              ,@(if (cdr formals) `((setq ,ponstablename (mht)))))
            ,@(if (cdr formals)
                  `((setq ,args (pist* ,ponstablename ,@formals))))
            (multiple-value-setq (,ans ,ans-p)
              (gethash ,argq ,tablename))
            (cond
             (,ans-p
              (safe-incf (aref ,ma (the fixnum (1+ ,caller))))
              ,(cond ((eql 1 (length stobjs-out)) ans)
                     (t  (cons 'mv
                               (nconc
                                (loop for i below
                                      (1- (length stobjs-out))
                                      collect `(pop ,ans))
                                (list ans))))))
             (t ,(cond ((eql (length stobjs-out) 1)
                        `(setf (gethash ,argq ,tablename) ,body))
                       (t (let ((vars (loop for i
                                            below (length stobjs-out)
                                            collect (gensym "O"))))
                            `(mv-let ,vars ,body
                                     (setf (gethash ,argq ,tablename)
                                           (list* ,@vars))
                                     (mv ,@vars)))))))))
          (cond ((not (eql -1 ,start))
                 (let ((intr (the fixnum (internal-run-time))))
                   (declare (fixnum intr))
                   (cond
                    ((> intr ,start)
                     (let ((d (aref ,ma ,caller))
                           (diff (the fixnum (- intr ,start))))
                       (declare (fixnum diff d))
                       (if (and (<= d most-positive-fixnum/2)
                                (<= diff most-positive-fixnum/2))
                           (setf (aref ,ma ,caller)
                                 (the fixnum (+ diff d)))
                         (timer-error))))))))))))
   (when *memoize-debug*
     (our-syntax (print defun *debug-io*) (force-output *debug-io*)))
   (unwind-protect
       (progn
         (loop for i fixnum from (* fnn *max-memoize-fns*)
               to (+ (* fnn *max-memoize-fns*) *max-symbol-to-fixnum*)
               do (setf (aref *memoize-array* i) 0))
         (setf (gethash fn *memoize-info-ht*)
               (make memoize-info-ht-entry
                     :tablename     tablename
                     :ponstablename ponstablename
                     :old-fn        old-fn
                     :memoized-fn   memoized-fn
                     :condition     condition
                     :inline        inline
                     :num           fnn
                     :sts           sts))
         (setf (gethash fnn *memoize-info-ht*) fn)
         (memoize-eval-compile defun)
         (setf (symbol-value memoized-fn) (symbol-function fn))
         (and condition
              (loop for s in sts do
                    (push tablename (symbol-value s))))
         (setq success t)
         fn)
     (unless success
       (setf (symbol-function fn) (symbol-value old-fn))
       (remhash fn *memoize-info-ht*)
       (remhash fnn *memoize-info-ht*)
       (loop for s in sts do
             (setf (symbol-value s)
                   (remove tablename (symbol-value s))))
       (ofe "~&; Memoize-fn:  Failed to memoize ~a." fn)))))))

(defun unmemoize-fn (fn)
  (eval `(maybe-untrace ,fn))
  (let* ((l (or (memoizedp-raw fn)
                (ofe "~&; Unmemoize-fn: ~a is not memoized." fn)))
         (tablename (access memoize-info-ht-entry l :tablename))
         (ponstablename (access memoize-info-ht-entry l
                                :ponstablename))
         (old-fn (access memoize-info-ht-entry l :old-fn))
         (memoized-fn (access memoize-info-ht-entry l :memoized-fn))
         (num (access memoize-info-ht-entry l :num))
         (sts (access memoize-info-ht-entry l :sts)))

; Note: condition is a first-class ACL2 function, not to be messed
; with here.
           
    (remhash fn *memoize-info-ht*)
    (setf (symbol-function fn) (symbol-value old-fn))
    (setf (symbol-value tablename) nil)
    (setf (symbol-value ponstablename) nil)
    (setf (symbol-value old-fn) nil)
    (setf (symbol-value memoized-fn) nil)
    (remhash num *memoize-info-ht*)
    (loop for s in sts do
          (setf (symbol-value s)
                (remove tablename (symbol-value s))))
    fn))

(defun memoized-functions ()
  (let (l)
    (maphash (lambda (fn v) (declare (ignore v))
               (when (symbolp fn) (push fn l)))
             *memoize-info-ht*)
    l))

(defun unmemoize-all ()

; A warning to would-be code improvers.  It would be a bad idea to
; redefine UNMEMOIZE-ALL to maphash over *memoize-info-ht* because of
; the ANSI rules concerning which hash table entries may be modified
; during a maphash.

  (loop for x in (memoized-functions) do (unmemoize-fn x)))

(defun memoize-all-after (n &key (condition t) (inline t))
  (when (symbolp n) (setq n (event-number n)))
  (check-type n integer)
  (let* ((state *the-live-state*)
         (packs (get-global 'packages-created-by-defpkg state))
         (w (w state)))
    (labels
        ((no-use (pn fn)
           (let ((x (getprop fn pn t 'current-acl2-world w)))
             (or (eq x t) (member 'state x))))
         (unmemoizable (fn)
                       (or (memoizedp-raw fn)
                           (not (fboundp fn))
                           (eq (symbol-package fn)
                               *main-lisp-package*)
                           (gethash fn *memoize-fn-reject-ht*)
                           (macro-function fn)
                           (special-form-or-op-p fn)
                           (no-use 'stobjs-in fn)
                           (no-use 'stobjs-out fn)
                           (no-use 'formals fn)
                           (and inline
                                (null (cltl-def-from-name
                                       fn nil w))))))
      (loop for p in (cons "ACL2" packs) do
            (do-symbols (fn p)
              (unless (or (unmemoizable fn) (<= (event-number fn) n))
                (memoize-fn fn
                            :condition condition
                            :inline inline)))))))

(defun compliant-and-ideal ()
  (let* ((logic-fns
          (eval '(let ((world (w *the-live-state*)))
                   (strip-cadrs (set-difference-theories
                                 (function-theory :here)
                                 (universal-theory 'ground-zero))))))
         (ideal-fns (collect-non-common-lisp-compliants
                     logic-fns (w *the-live-state*))))
    (mv (set-difference-eq logic-fns ideal-fns) ideal-fns)))

(defun memoize-compliant-after (n &key (condition t) (inline t))
  (when (symbolp n) (setq n (event-number n)))
  (check-type n integer)
  (mv-let (c i)
          (compliant-and-ideal)
          (declare (ignore i))
          (loop for fn in c do
                (cond ((and (not (memoizedp-raw fn))
                            (> (event-number fn) n))
                       (memoize-fn fn
                                   :condition condition
                                   :inline inline))))))

; Statistics Gathering Routines

(defun hons-statistics ()
  (our-syntax
   (oft "~&; Examining *hons-cdr-ht*:")
   (maphash (lambda (key value)
              (oft "~&; To ~s has been honsed:~%" key)
              (cond ((hash-table-p value)
                     (maphash (lambda (key v2) (declare (ignore v2))
                                (oft "~s, " key))
                              value))
                    (t (loop for pair in value do
                             (oft "~s, " (car pair))))))
            *hons-cdr-ht*)
   (oft "~&; End of *hons-cdr-ht* examination. ~
         ~%~%; Examining *hons-cdr-ht-eql*:")
   (maphash (lambda (key value)
              (oft "~%; To ~s has been honsed:~%" key)
              (cond ((hash-table-p value)
                     (maphash (lambda (key v2) (declare (ignore v2))
                                (oft "~s, " key))
                              value))
                    (t (loop for pair in value do
                             (oft "~s, " (car pair))))))
            *hons-cdr-ht-eql*)
   (oft "~%; End of *hons-cdr-ht-eql* examination. ~
         ~%~%; Examining *nil-ht*:~%")
   (oft "; To NIL has been honsed:~%")
   (maphash (lambda (key v2) (declare (ignore v2))
              (oft "~s, " key))
            *nil-ht*)
   (oft "~% End of *nil-ht* examination.")))

(defun counter-value (ar)
  (declare (type (simple-array fixnum (2)) ar))
  (let ((n (aref ar 1)))
    (declare (fixnum n))
    (if (eql n 0) (aref ar 0)
      (+ (aref ar 0) (* n (+ 1 most-positive-fixnum))))))

; format a number with commas
(defun fwc (x) (ofn "~:d" x))

(defun print-alist (alist separation)
  (check-type separation (integer 0))
  (setq alist
        (loop for x in alist collect
              (progn
                (check-type x
                            (cons string
                                  (cons (or string (integer 0))
                                        null)))
                (list (car x)
                      (if (integerp (cadr x))
                          (fwc (cadr x))
                        (cadr x))))))
  (let* ((max1 (loop for x in alist maximize (length (car x))))
         (max2 (loop for x in alist maximize (length (cadr x))))
         (width (max (right-margin) (+ separation max1 max2))))
    (loop for x in alist do
          (fresh-line)
          (princ (car x))
          (loop for i below (- width (+ (length (car x))
                                        (length (cadr x))))
                do (write-char #\Space))
          (princ (cadr x))))
  nil)

(defun hons-count ()
  (let ((n 0))
    (declare (integer n))
    (loop for tab in '(*hons-cdr-ht* *hons-cdr-ht-eql*) do
          (maphash (lambda (k v) (declare (ignore k))
                     (cond ((hash-table-p v)
                            (incf n (hash-table-count v)))
                           (t (incf n (length v)))))
                   (symbol-value tab)))
    (+ n (hash-table-count *nil-ht*))))

(defun right-margin ()
  (or *print-right-margin* 70))

(defun hons-summary ()
  (our-syntax
   (let ((sssub 0) (nhonses 0) (nsubs 0))
     (declare (integer sssub nhonses nsubs))
     (loop for tab in '(*hons-cdr-ht* *hons-cdr-ht-eql*) do
           (maphash
            (lambda (k v) (declare (ignore k))
              (cond
               ((hash-table-p v)
                (incf sssub (hash-table-size v))
                (incf nhonses (hash-table-count v))
                (incf nsubs))
               (t (incf nhonses (length v)))))
            (symbol-value tab)))
     (incf nhonses (hash-table-count *nil-ht*))
     (print-alist
      `(("Hons hits/calls"
         ,(let* ((c (counter-value *hons-call-counter*))
                 (m (counter-value *hons-misses-counter*))
                 (d (- c m)))
            (ofn "~:d/~:d = ~,2f"
                 (- c m)
                 c
                 (/ d (+ .001 c)))))
        ,@(loop for tab in '(*hons-cdr-ht* *hons-cdr-ht-eql* *nil-ht*
                                           *hons-str-ht*)
                collect
                (let* ((tabv (symbol-value tab))
                       (c (hash-table-count tabv))
                       (s (hash-table-size tabv)))
                  (list (ofn "~a count/size"
                             (string-downcase (symbol-name tab)))
                        (ofn "~a/~a = ~,2f"
                             (fwc c)
                             (fwc s)
                             (float (/ c (+ .001 s)))))))
        ("Number of sub tables" ,nsubs)
        ("Sum of sub table sizes" ,sssub)
        ("Number of honses" ,nhonses))
      5)))
  nil)

(defun memoize-statistics (&optional (fn (memoized-functions)))
  (our-syntax
   (cond ((listp fn)
          (mapc #'memoize-statistics fn))
         ((not (memoizedp-raw fn))
          (oft "~%; Memoize-statistics:  ~a is not memoized." fn))
         (t (let ((tb (symbol-value
                       (access memoize-info-ht-entry
                               (gethash fn *memoize-info-ht*)
                               :tablename))))
              (cond ((and tb (not (eql 0 (hash-table-count tb))))
                     (oft "~%; Memoized values for ~a." fn)
                     (maphash (lambda (key v)
                                (format t "~%~s~%=>~%~s" key v))
                              tb)))))))
  nil)
  
(defun number-of-calls (fn)
  (let ((fnn (the fixnum (symbol-to-fixnum fn))))
    (declare (fixnum fnn))
    (loop for i fixnum
          to (* *max-symbol-to-fixnum* *max-memoize-fns*)
          by *max-memoize-fns*
          sum (aref *memoize-array* (the fixnum (+ i fnn))))))

(defun number-of-hits (fn)
  (aref *memoize-array*
        (the fixnum (+ 1 (the fixnum
                           (* *max-memoize-fns*
                              (the fixnum (symbol-to-fixnum fn))))))))

(defun number-of-mht-calls (fn)
  (aref *memoize-array*
        (the fixnum (+ 2 (the fixnum
                           (* *max-memoize-fns*
                              (the fixnum (symbol-to-fixnum fn))))))))

(defun run-time-for-non-hits-of-fn-in-secs-per-call (fn)
  (let ((n (- (number-of-calls fn) (number-of-hits fn))))
    (if (zerop n) 0 (/ (run-time-for-fn-in-secs fn) n))))

(defun run-time-for-fn-in-secs-per-call (fn)
  (let ((n (number-of-calls fn)))
    (if (zerop n) 0 (/ (run-time-for-fn-in-secs fn) n))))

(defun hits/calls (fn)
  (let ((n (number-of-calls fn)))
    (if (zerop n) 0 (/ (number-of-hits fn) (float n)))))
    
(defun event-number (fn)
  (fgetprop fn 'absolute-event-number t (w *the-live-state*)))

; MEMOIZE-SUMMARY

(defglobal *default-memoize-summary-fns* nil)

(defglobal *memoize-summary-order-fns*
  '(event-number hits/calls nil run-time-for-fn-in-secs
    run-time-for-fn-in-secs-per-call
    number-of-calls symbol-name))

(defglobal *default-memoize-summary-order* 'run-time-for-fn-in-secs)

(defglobal *default-memory-summary-hide-tables* t)
  
(defun memoize-summary (&key (fns *default-memoize-summary-fns*
                                  fns-supplied-p)
                    (order *default-memoize-summary-order*)
                    (hide-tables *default-memory-summary-hide-tables*)
                    (skip-not-called t))
  "

    Memoize-summary reports on the symbols in its :fns parameter.
  Once a function in memoized, each outermost call of the function is
  counted and timed.  Recursive calls are not timed.  Recursive calls
  are counted unless the function was memoized with :inline parameter
  nil.  A function is billed for the time used by its subroutines,
  including recursive calls of itself.
    The :fns parameter is first sorted by the :order parameter, which
  must be a member of *memoize-summary-order-fns*.  If the :fns
  parameter is not supplied, it defaults to (memoized-functions),
  unless *default-memoize-summary-fns*, which is initally nil, is not
  nil.
     If the :hide-tables parameter, which default to t, is nil, extra
  information about the sizes of memoize tables is printed.
     If the :skip-not-called parameter, which defaults to t, is t,
  only functions in the :fns parameter that have been called are
  included in the report.
    (clear-memoize-tables) clears away all memoized answers.
  (clear-counters) clears all counting and timing information for all
  memoized functions.

  "
  
  (oft "~&Below is a report")
  (cond
   ((null fns-supplied-p)
    (cond
     (*default-memoize-summary-fns*
      (oft " on *default-memoize-summary-fns*")
      (setq fns (copy-list *default-memoize-summary-fns*)))
     (t (oft " on all memoized functions")
        (setq fns (memoized-functions))))))
  (oft ", sorted by~%~a." order)
  (let ((num (/ (1- (hash-table-count *memoize-info-ht*)) 2)))
    (cond ((eql num 0)
           (oft "  No functions are currently memoized.~&"))
          ((eql num 1)
           (oft "  Only one function is currently memoized.~%" num))
          (t (oft "  ~@r functions are currently memoized.~%" num))))
  (our-syntax
   (when fns-supplied-p
     (setq fns
           (loop for fn in fns nconc
                 (cond ((not (memoizedp-raw fn))
                        (oft "~&Memoize-summary:  ~s is not memoized."
                             fn)
                        nil)
                       (t (list fn))))))
   (cond
    ((null order) nil)
    ((eq order 'symbol-name)
     (setq fns (sort fns #'string< :key #'symbol-name)))
    ((member order *memoize-summary-order-fns*)
     (setq fns (stable-sort fns #'> :key order)))
    (t (oft "~%~%The :order argument for memoize-summary should be a ~
             member of *memoize-summary-order-fns*,~%not:  ~a."
            order)))
   #+RDTSC
   (oft "~&The X86 instruction RDTSC is used for timing, rather ~
         than~%Lisp's GET-INTERNAL-RUN-TIME.")
   #-RDTSC
   (oft "~&Lisp's GET-INTERNAL-RUN-TIME is used for timing.")
   (if (not (eql 0 (counter-value *pons-call-counter*)))
       (print-alist
        `(("Pons hits/calls"
           ,(ofn "~a/~a = ~,2f"
                 (fwc (counter-value *pons-hit-counter*))
                 (fwc (counter-value *pons-call-counter*))
                 (/ (counter-value *pons-hit-counter*)
                    (+ .001 (counter-value *pons-call-counter*))))))
        5))
   (if (null fns) (oft "~&No memoized functions to report upon.")
     (loop for fn in fns do
      (let* ((l (gethash fn *memoize-info-ht*))
             (tab (symbol-value (access memoize-info-ht-entry l
                                        :tablename)))
             (ponstab (symbol-value (access memoize-info-ht-entry l
                                            :ponstablename)))
             (fnn (symbol-to-fixnum fn))
             (nhits (number-of-hits fn))
             (nmht (number-of-mht-calls fn))
             (ncalls (number-of-calls fn)))
        (cond
         ((and skip-not-called (eql 0 ncalls)) nil)
         (t (fresh-line)
            (print-alist
             `((,(if (eql 0 nhits)
                     (ofn "~a calls" fn)
                   (ofn "~a hits/calls" fn))
                ,(if (eql 0 nhits)
                     (ofn "~a" (fwc ncalls))
                   (ofn "~a/~a = ~,2f" (fwc nhits) (fwc ncalls)
                        (/ nhits (+ .0001 (float ncalls))))))
               ,@(if (>= nmht 2) `(("Calls to mht" ,(fwc nmht))))
               ("Run-time of all outermost calls in seconds"
                ,(ofn "~20f" (run-time-for-fn-in-secs fn)))
               ("Microseconds per outermost call"
                ,(ofn "~20f" (* 1000000
                                (run-time-for-fn-in-secs-per-call
                                 fn))))
               ,@(if (not (eql 0 nhits))
                   `(("Microseconds per missed outermost call"
                      ,(ofn
                        "~20f"
                        (* 1000000
                   (run-time-for-non-hits-of-fn-in-secs-per-call
                            fn))))))
               ,@(loop for i fixnum to *max-symbol-to-fixnum*
                       when (gethash i *memoize-info-ht*)
                       nconc
                       (let ((calls
                              (aref *memoize-array*
                                    (the fixnum
                                      (+ fnn (the fixnum
                                               (* i
                                                  *max-memoize-fns*
                                                  )))))))
                         `(,@(if (> calls 0)
                                 `((,(ofn "Calls from ~a"
                                          (if (eql i 0)
                                              "unmemoized functions"
                                            (fixnum-to-symbol i)))
                                    ,calls))))))
               .
               ,(if hide-tables nil
                  (let ((spsub 0) (nponses 0) (npsubs 0))
                    (and ponstab
                         (maphash
                          (lambda (key value) (declare (ignore key))
                            (cond
                             ((hash-table-p value)
                              (incf spsub (hash-table-size value))
                              (incf nponses (hash-table-count value))
                              (incf npsubs))
                             (t (incf nponses (length value)))))
                          ponstab))
                    `(,@(and tab
                             `((,(ofn "Memoize table count/size")
                                ,(ofn "~a/~a = ~,2f"
                                      (fwc (hash-table-count tab))
                                      (fwc (hash-table-size tab))
                                      (/ (hash-table-count tab)
                                         (+ .001 (hash-table-size
                                                  tab)))))))
                      ,@(and ponstab
                             `(("Pons table count/size"
                                ,(ofn "~a/~a = ~,2f"
                                      (fwc (hash-table-count
                                            ponstab))
                                      (fwc (hash-table-size
                                            ponstab))
                                      (/ (hash-table-count ponstab)
                                         (+ .001 (hash-table-size
                                                  ponstab)))))
                              ("Number of pons sub tables" ,npsubs)
                              ("Sum of sub pons table sizes" ,spsub)
                              ("Sum of sub pons table counts"
                               ,nponses)))))))
             5)))))) 
  (let ((count 0) (size 0) (number 0) last-key alist)
     (maphash
      (lambda (key v)
        (cond ((typep v 'fixnum)
               (push (list
                        (if (setq last-key (cdr (last key)))
                            last-key number)
                        (len key)
                        (len key))
                     alist)
               (incf number)
               (incf size (len key))
               (incf count (len key)))
              (t (push (list
                        (if (setq last-key (cdr (last key)))
                            last-key number)
                        (hash-table-size v)
                        (hash-table-count v))
                       alist)
                 (incf number)
                 (incf size (hash-table-size v))
                 (incf count (hash-table-count v)))))
      *hons-acons-ht*)
     (cond (alist
            (oft "~&Hons-acons statistics")
            (print-alist
             (list (list "Count/size"
                         (ofn "~a/~a = ~,2f"
                              (fwc (hash-table-count *hons-acons-ht*))
                              (fwc (hash-table-size *hons-acons-ht*))
                              (/ (hash-table-count *hons-acons-ht*)
                                 (+ .001 (hash-table-size
                                          *hons-acons-ht*)))))
                   (list "Total of counts" count)
                   (list "Total of sizes" size))
             5)
            (oft "~&For each HONS-ACONS entry~%(name size count)")
            (loop for x in alist do (print x)))))
   nil))

; State refreshment routines for hons and memoize

(defun init-hons-acons-table ()
  (setq *hons-acons-ht* (mht :test #'eq :weak :key)))

; The ACL2 persistent-hons-table, which is updated by defhonst, should
; be an alist whose keys are honses to be preserved as honses through
; any cleaning of hash tables, but not through an init-hash-tables.

(defglobal *hons-tables-initialized* nil)

(defun init-hash-tables ()
  (setq *hons-cdr-ht* (mht :size *hons-cdr-ht-size*
                           :weak :key :test #'eq))
  (setq *hons-cdr-ht-eql* (mht))
  (setq *nil-ht* (mht :size *nil-ht-size* :weak :value))
  (init-hons-acons-table)
  (setq *hons-str-ht* (mht :test #'equal :weak :value))
  (setq *memoize-array*
        (make-array (* *max-memoize-fns* *max-memoize-fns*)
                    :element-type 'fixnum
                    :initial-element 0))
  (setq *hons-copy-aux-ht* (mht :test #'eq))
  (unmemoize-all)
  (setq *memoize-info-ht* (mht))
  (setf (gethash 0 *memoize-info-ht*) "outside-caller")
  (setq *hons-tables-initialized* t))

(defglobal *rehons-culprit* nil)

(defun clear-hash-tables ()

; Should this function save any honses from *hons-acons-ht*?
; Currently it saves not only persistent honses, but also honses that
; are keys of, or keys of keys of, the *hons-acons-ht*.  An optional
; argument could presumably be added to make these additional saves
; optional (but then what should the default be for this optional
; argument?).

  (let (l)
    (declare (dynamic-extent l))
    (maphash (lambda (k v) (declare (ignore v))
               (if (honsp k) (push k l)))
             *hons-acons-ht*)
    ; hons-let assumes that clrhash is not used here.
    (setq *hons-cdr-ht* (mht :size *hons-cdr-ht-size* :test #'eq
                        :weak :key))
    (setq *hons-cdr-ht-eql* (mht))
    (setq *nil-ht* (mht :size *nil-ht-size* :weak :value))
    (flet ((rehons (x)
            (cond ((and (consp x)
                        (not (eq x (hons-copy1-consume x))))
                   (setq *rehons-culprit* x)
                   (ofe "~%; clear-hash-tables:  Error: ** failed to ~
                         rehons *rehons-culprit*."))
                  (t (maybe-str-hash x)))))
      (loop for x in (table-alist 'persistent-hons-table
                                  (w *the-live-state*))
            do (when (car x) (rehons (car x))))
      (mapc #'rehons l)
      (maphash (lambda (k v)
                 (cond ((and (consp k) (honsp k))
                        ;; all parts of k are already honsed.
                        nil)
                       ((hash-table-p v)
                        (maphash (lambda (k v)
                                   (declare (ignore v))
                                   (rehons k))
                                 v))
                       (t (loop for pair in k do
                                (rehons (car pair))))))
               *hons-acons-ht*))
    nil))

(defun empty-ht-p (x)
  (and (hash-table-p x)
       (eql 0 (hash-table-count x))))

(defun clear-memoize-tables ()
  "Clears away all memoized answers."
  (maphash
   (lambda (k l)
     (when (symbolp k)
       (unless (empty-ht-p
                (symbol-value
                 (the symbol (access memoize-info-ht-entry
                                     l :tablename))))
         (setf (symbol-value
                (the symbol
                  (access memoize-info-ht-entry l :tablename)))
               nil))
       (unless (empty-ht-p
                (symbol-value
                 (the symbol (access memoize-info-ht-entry
                                     l :ponstablename))))
         (setf (symbol-value
                (the symbol
                  (access memoize-info-ht-entry l :ponstablename)))
               nil))))
   *memoize-info-ht*))
    
(defun clear-counters ()
  "Clears counting and timing information for all memoized functions."
  (clear-counter *pons-call-counter*)
  (clear-counter *pons-hit-counter*)
  (clear-counter *hons-call-counter*)
  (clear-counter *hons-misses-counter*)
  (setq *memoize-array*
        (make-array (* *max-memoize-fns* *max-memoize-fns*)
                    :element-type 'fixnum
                    :initial-element 0)))

(defun clear-hash-and-memoize-tables ()
  (clear-hash-tables)
  (clear-memoize-tables))

(defun flush-hons-get-hash-table-link (x)
; breaks the link between x and a hash table in the *hons-acons-ht* if
; such a link exists, thus permitting the garbage collection of that
; table.
  (cond
   ((atom x) x)
   ((not (remhash x *hons-acons-ht*))
    (maybe-report-discipline-failure flush-hons-get-hash-table-link)
    x)
   (t x)))

; Hash-consing while reading: HONS-READ-OBJECT and HONS-READ.

; Hash consing when reading is implemented via a change to the
; readtable for the character including open parenthesis, close
; parenthesis, and the consing dot.

; *** NOTE:  The following collection of functions is just that: a 
;            collection.  Unless you understand everything about the
;            various read table macros, then don't touch this code!

; See matching comment below.

; Note: our implementation of the #=/## reader, which we built because
; some Lisps would not us get past #500 or so, does not comply with
; ANSI at least in this regard: it does not allow the entry of looping
; structures as in '(#0= (3 #0#)), which no problem for ACL2 uses.

(defglobal *hons-readtable* (copy-readtable *acl2-readtable*))
(declaim (readtable *acl2-readtable* *hons-readtable*))

; WARNING: Any call of read using *hons-readtable* as *readtable*
; needs to worry about the possible top-level return of
; *close-paren-obj* and *dot-obj*, which are simply part of the
; fiction whereby we read while honsing.  Those two objects should
; absolutely not be returned as the value of an ACL2 function.  See,
; for example, the definition of HONS-READ.

(defglobal *close-paren-obj* '(#\)))
(defglobal *dot-obj*         '(#\.))

(defun nonsense (x)
  (or (eq x *close-paren-obj*) (eq x *dot-obj*)))

(defun check-nonsense (x)
  (cond ((nonsense x)
         (hread-error "~&;  Illegal object: ~a." (car x)))))

(defun hread-error (string &rest r)
  (our-syntax
   (let* ((stream *standard-input*)
          (*standard-output* *error-output*)
          (*standard-input* *terminal-io*))
     (apply #'format *error-output* string r)
     (cond ((and (streamp stream) (file-position stream))
            (format *error-output*
                    "~&; near file-position ~a in stream ~a."
                    (file-position stream) stream)))
     (error "hread"))))

(defun illegal-error1 (x)
  (hread-error "~&; ** Error:  Illegal:  ~s." x))

(defun illegal-error2 (stream char)
  (declare (ignore stream))
  (illegal-error1 char))

(defun close-paren-read-macro (stream char)
  (declare (ignore char stream))
  (if *read-suppress* (illegal-error1 #\)))
  *close-paren-obj*)

(defun dot-read-macro (stream char)
  (declare (ignore char stream))
  (if *read-suppress* (illegal-error1 #\.))
  (let ((ch (peek-char nil nil t nil t)))
    (cond ((or (member ch '(#\( #\) #\' #\` #\, #\" #\;
                                #\Space #\Newline))
               (multiple-value-bind (fn nonterminating)
                   (get-macro-character ch)
                 (and fn (not nonterminating))))
           *dot-obj*)
          (t (let ((*readtable* *acl2-readtable*))
               (unread-char #\. nil)
               (read nil t nil t))))))

(defun hons-read-list ()
  (let ((o (read nil t nil t)))
    (cond
     ((eq o *close-paren-obj*) nil)
     ((eq o *dot-obj*)
      (let ((lo (read nil t nil t))
            (lp (read nil t nil t)))
        (check-nonsense lo)
        (cond
         ((eq lp *close-paren-obj*) lo)
         (t (illegal-error1 #\.)))))
     (t (hons-normed (maybe-str-hash o) (hons-read-list))))))

(defun hons-read-list-top ()
  (let ((o (read nil t nil t)))
    (cond
     ((eq o *close-paren-obj*) nil)
     (t (check-nonsense o)
        (hons-normed (maybe-str-hash o)
                     (hons-read-list))))))

(defun hons-read-reader (stream char)
  (declare (ignore char)) 
  (cond (*read-suppress*
         (unread-char #\( stream)
         (let ((*readtable* *acl2-readtable*))         
           (read nil t nil t)))
        (t (hons-read-list-top))))
  
(defun hons-read (&optional i (eep t) eofv rp)
  (let* ((*readtable* *hons-readtable*)
         (*standard-input* (or i *standard-input*)))
    (let ((x (read nil eep eofv rp)))
      (check-nonsense x)
      x)))

(defun hons-read-object (channel state-state)
  (let ((*acl2-readtable* *hons-readtable*))
    (mv-let
     (eofp obj state)
     (read-object channel state-state)
     (check-nonsense obj)
     (mv eofp obj state))))

(defun hons-read-file (file-name)
  (with-open-file (file file-name)
    (let (temp ans (eof (cons nil nil)))
      (declare (dynamic-extent eof))
      (loop (setq temp (hons-read file nil eof nil))
            (if (eq eof temp)
                (return (hons-copy1-consume (nreverse ans))))
            (setq ans (cons temp ans))))))

;  Compact printing and reading procedures.

(defglobal *compact-print-file-ht* (mht))
(defglobal *compact-read-file-ht* (mht))
(defglobal *compact-print-file-n* 0)
(declaim (fixnum *compact-print-file-n*))
(defglobal *space-owed* nil)
(declaim (type boolean *space-owed*))

(defun compact-print-file-scan (x)
  (unless (or (and (symbolp x) (<= (length (symbol-name x)) 4))
              (and (stringp x) (<= (length x) 2))
              (and (integerp x)
                   (cond ((< x 0) ;  -99 ... 999, leave it alone.
                          (> x -100))
                         (t (< x 1000))))
              (characterp x))
    (let ((g (gethash x *compact-print-file-ht*)))
      (unless (or (atom x) (eq g 'give-it-a-name))
        (compact-print-file-scan (car x))
        (compact-print-file-scan (cdr x)))
      (unless (eq g 'give-it-a-name)
        (setf (gethash x *compact-print-file-ht*)
              (if g 'give-it-a-name 'found-at-least-once))))))

(defmacro space-if-necessary ()
  '(when *space-owed* (write-char #\Space) (setq *space-owed* nil)))

(defun compact-print-file-help (x hash)
  (cond ((typep hash 'fixnum)
         (space-if-necessary)
         (write-char #\#)
         (princ hash)
         (write-char #\#))
        (t (cond ((eq hash 'give-it-a-name)
                  (let ((n *compact-print-file-n*))
                    (declare (fixnum n))
                    (when (eql n most-positive-fixnum)
                      (ofe "~&; *compact-print-file-n* overflow."))
                    (setq n (the fixnum (+ 1 n)))
                    (setq *compact-print-file-n* n)
                    (setf (gethash x *compact-print-file-ht*) n)
                    (space-if-necessary)
                    (write-char #\#)
                    (princ n)
                    (write-char #\=))))
           (cond
            ((atom x)
             (space-if-necessary)
             (prin1 x)
             (setq *space-owed* t))
            (t (write-char #\()
               (setq *space-owed* nil)
               (loop (compact-print-file-help
                      (car x)
                      (gethash (car x) *compact-print-file-ht*))
                     (cond
                      ((null (cdr x))
                       (write-char #\))
                       (setq *space-owed* nil)
                       (return))
                      ((or (progn
                             (setq hash
                                   (gethash (cdr x)
                                            *compact-print-file-ht*))
                             (or (eq hash 'give-it-a-name)
                                 (typep hash 'fixnum)))
                           (atom (cdr x)))
                       (space-if-necessary)
                       (write-char #\.)
                       (setq *space-owed* t)
                       (compact-print-file-help (cdr x) hash)
                       (write-char #\))
                       (setq *space-owed* nil)
                       (return))
                      (t (pop x)))))))))

(defun compact-print-file (data file-name)
  (setq *compact-print-file-ht* (mht))
  (setq *compact-print-file-n* 0)
  (setq *space-owed* nil)
  (with-open-file (*standard-output* file-name
                                     :direction :output
                                     :if-does-not-exist :create
                                     :if-exists :supersede)
    (with-standard-io-syntax
     (let ((*package* *acl2-package*)
           (*readtable* *acl2-readtable*))
       (compact-print-file-scan data)
       (compact-print-file-help data
                                (gethash data *compact-print-file-ht*))
       (setq *compact-print-file-ht* (mht)))))
  file-name)

(defun ns-=-reader (stream subchar arg)
  (declare (ignore stream subchar))
  (when (gethash arg *compact-read-file-ht*)
    (hread-error
     "~&; ns-=-reader:  ** Error:  #~a= is already defined."
     arg))
  (setf (gethash arg *compact-read-file-ht*) (read nil t nil t)))

(defun ns-ns-reader (stream subchar arg)
  (declare (ignore stream subchar))
  (or (gethash arg *compact-read-file-ht*)
      (hread-error "~&; ns-ns-reader:  ** Error:  meaningless #~a#."
                   arg)))

(defglobal *compact-read-file-readtable*
  (copy-readtable *hons-readtable*))
(declaim (readtable *compact-read-file-readtable*))

(defun compact-read-file (file-name)
  (setq *compact-read-file-ht* (mht))
  (let ((*readtable* *compact-read-file-readtable*))
    (with-open-file (file file-name)
      (let* ((*standard-input* file)
             (x (read)))
        (check-nonsense x)
        (setq *compact-read-file-ht* (mht))
        (hons-copy1-consume x)))))

; Newick Reader

(defun newick-read-error (str &rest r)
  (our-syntax
   (let* ((pos1 (file-position *standard-input*))
          (pos0 (max 0 (- pos1 30)))
          (d (- pos1 pos0))
          (ns *standard-input*)
          (*standard-output* *error-output*)
          (*standard-input* *terminal-io*))
     (apply #'format *error-output* str r)
     (if (eql (file-position ns) (file-length ns))
         (format *terminal-io* "~%; At end of file.")
       (format *terminal-io* "~%; File position is ~a."
               (file-position ns)))
     (format *terminal-io* "~%; The last ~a characters read were:" d)
     (file-position ns pos0)
     (loop for i below d do (write-char (read-char ns t nil t)))
     (break "newick-read-error. ~%"))))

(defun newick-illegal1 (x)
  (newick-read-error "~&; ** Error:  Illegal:  ~a" x))

(defglobal *newick-str*
  (make-array 100
              :element-type 'character
              :adjustable t
              :fill-pointer 0))
(defglobal *newick-pw*       (make-array 256 :initial-element nil))
(defglobal *newick-ordinary* (make-array 256 :initial-element nil))
(defglobal *newick-digit*    (make-array 256 :initial-element nil))
(defglobal *newick-white*    (make-array 256 :initial-element nil))
(defglobal *newick-punct*    (make-array 256 :initial-element nil))
(defglobal *newick-specials* (make-array 256 :initial-element nil))
(declaim (type (array character (*)) *newick-str*)
         (type (simple-array t (256))
               *newick-pw* *newick-ordinary* *newick-white*
               *newick-punct* *newick-specials* *newick-digit*))

(defun newick-consume-comment ()
  (let ((ch (read-char nil t nil t)))
    (cond ((eql ch #\[)
           (newick-consume-comment)
           (newick-consume-comment))
          ((eql ch #\]) nil)
          (t (newick-consume-comment)))))

(defun newick-read-token ()
  (let* ((p (read-char nil t nil t))
         (n (char-code (the character p)))
         (x nil))
    (declare (fixnum n))
    (when (> n 255) (newick-illegal1 p))
    (cond
     ((setq x (svref *newick-ordinary* n))
      (setf (fill-pointer *newick-str*) 0)
      (vector-push x *newick-str*)
      (loop
       (setq p (read-char nil t nil t))
       (setq n (char-code p))
       (when (> n 255) (newick-illegal1 p))
       (cond ((setq x (svref *newick-ordinary* n))
              (vector-push-extend x *newick-str*))
             ((eql p #\[) (newick-consume-comment))
             (t (unread-char p) (return))))
      (or (find-symbol *newick-str*)
          (values (intern (copy-seq *newick-str*)))))
     ((svref *newick-white* n) (newick-read-token))
     ((svref *newick-specials* n) p)
     ((eql p #\[) (newick-consume-comment) (newick-read-token))
     ((eq p #\')
      (let (ch1 ch2)
        (setf (fill-pointer *newick-str*) 0)
        (loop
         (setq ch1 (read-char nil t nil t))
         (cond ((eq ch1 #\')
                (setq ch2 (read-char nil t nil t))
                (cond ((not (eql ch2 #\'))
                       (unread-char ch2)
                       (return)))))
         (vector-push-extend (char-upcase (the character ch1))
                             *newick-str*))
        (let ((n 0) (flg t))
          (declare (fixnum n))
          (loop for i fixnum below (length *newick-str*) do
                (setq n (char-code (the character
                                     (aref *newick-str* i))))
                (when (> n 255) (newick-illegal1 n))
                (unless (svref *newick-pw* n) (setq flg nil)))
          (when flg
            (newick-read-error
             "~&; ** Error: a Newick name may not consist entirely ~
              of whitespace and punctuation,~%as does: ~s."
             *newick-str*)))
        (or (find-symbol *newick-str*)
            (values (intern (copy-seq *newick-str*))))))
     (t (newick-illegal1 p)))))

(defmacro newick-get-one ()
  '(loop (setq p (read-char nil t nil t))
         (setq n (char-code p))
         (when (> n 255) (newick-illegal1 p))
         (cond ((eql p #\[) (newick-consume-comment))
               ((or (svref *newick-specials* n)
                    (svref *newick-white* n))
                (unread-char p)
                (setq p nil)
                (return))
               (t (return)))))

(defglobal *newick-ignore-branch-lengths* nil)

(defun newick-read-number ()
  (let* ((p #\0) (n 0) (past-dot nil) (neg nil) (ans 0)
         (past-e nil) (eans 0) (eneg nil) (x nil))
    (declare (fixnum n))
    (loop (setq p (read-char nil t nil t))
          (setq n (char-code p))
          (when (> n 255) (newick-illegal1 p))
          (cond ((eql p #\[) (newick-consume-comment))
                ((svref *newick-white* n))
                (t (return))))
    (cond ((eql p #\+) (newick-get-one))
          ((eql p #\-) (setq neg t) (newick-get-one))
          ((eql p #\.) (newick-get-one) (setq past-dot 0)))
    (loop
     (cond ((null p)
            (cond (*newick-ignore-branch-lengths* (return 0))
                  (past-dot (setq ans (/ ans (expt 10 past-dot)))))
            (if neg (setq ans (- ans)))
            (cond (past-e
                   (cond (eneg (setq ans (/ ans (expt 10 eans))))
                         (t (setq ans (* ans (expt 10 eans)))))))
            (return ans))
           ((setq x (svref *newick-digit* n))
            (cond (*newick-ignore-branch-lengths*)
                  (past-e (setq eans (+ (* eans 10) x)))
                  (t (setq ans (+ (* ans 10) x)))))
           ((or (eql p #\e) (eql p #\E))
            (cond (past-e (newick-illegal1 p))
                  (t (setq past-e t)
                     (let ((nc (peek-char nil nil t nil t)))
                       (cond ((eql nc #\+) (read-char nil t nil t))
                             ((eql nc #\-)
                              (setq eneg t)
                              (read-char nil t nil t)))))))
           ((eql p #\.)
            (cond ((or past-dot past-e) (newick-illegal1 p))
                  (t (setq past-dot -1))))
           (t (newick-illegal1 p)))
     (cond (*newick-ignore-branch-lengths*)
           (past-e)
           (past-dot (incf past-dot)))
     (newick-get-one))))

(defglobal *newick-interior-node-alist* nil)

(defun newick-cleanup (x)
; We now consume the interior node name and/or branch length, if any,
; leaving a following comma, semicolon, or close unread.
  (let ((p (newick-read-token)))
    (cond
     ((eql p #\:)
      (let ((r (newick-read-number)))
        (cond (*newick-ignore-branch-lengths* x)
              (t (hons-normed x r)))))
     ((eql p #\() (newick-illegal1 p))
     ((characterp p) ; A comma, close paren, or semicolon.
      (unread-char p)
      x)
     ((and (consp x) (symbolp p))
      (when (assoc p *newick-interior-node-alist*)
        (newick-read-error "%&; newick-read-list: ** Error: Interior ~
                            node name ~s used twice."
                           p))
      (push (cons p x) *newick-interior-node-alist*)
      (newick-cleanup x))
     (t (newick-illegal1 p)))))

(defun newick-read-list (flg)
; The open paren has been read already.
  (let ((p (newick-read-token)))
    (cond
     ((null flg)
      (cond ((eql p #\)) (hons-normed nil nil))
            ((eql p #\,) (hons-normed nil (newick-read-list nil)))
            ((symbolp p) (hons-normed (newick-cleanup p)
                                      (newick-read-list t)))
            ((eql p #\()
             (hons-normed (newick-cleanup (newick-read-list nil))
                          (newick-read-list t)))
            (t (newick-illegal1 p))))
     ((eql p #\)) nil)
     ((eql p #\,)
      (setq p (newick-read-token))
      (cond ((symbolp p) (hons-normed (newick-cleanup p)
                                      (newick-read-list t)))
            ((eql p #\()
             (hons-normed (newick-cleanup (newick-read-list nil))
                          (newick-read-list t)))
            ((eql p #\,)
             (hons-normed nil (newick-read-list nil)))
            ((eql p #\))
             (hons-normed nil nil))
            (t (newick-illegal1 p))))
      (t (newick-illegal1 p)))))

(defun newick-first-report (report file-name)
  (when report
    (format *terminal-io*
            "~&; Parsing ~a.~
             ~&; Size:  ~:d bytes."
            file-name
            (file-length *standard-input*))))

(defglobal *newick-l* 0)
(defglobal *newick-d* 0)

(defun newick-final-report (report start)
  (let (d l)
    (when report
      (setq d (+ .000001 (/ (- (internal-run-time) start)
                            *float-ticks-per-second*)))
      (setq l (file-length *standard-input*))
      (incf *newick-l* l)
      (incf *newick-d* d)
      (format
       *terminal-io*
       "~&; Parsing time:   ~10,2f seconds.~
        ~&; Parsing speed:  ~10,2f megabytes/second."
       d
       (/ (/ l d) 1000000))
      (format
       *terminal-io*
       "~&; Cumulative parsing time:   ~10,2f seconds.~
        ~&; Cumulative parsing speed:  ~10,2f megabytes/second."
         *newick-d*
         (/ (/ *newick-l* *newick-d*) 1000000)))))
  
(defglobal *compact-read-file-output* nil)
(defglobal *compact-print-file-input* nil)

(defun compact-report (report bhz-file)
  (when report
    (format
     *terminal-io*
     "~&; Bytes written to file ~a:  ~:d."
     bhz-file (with-open-file (s bhz-file) (file-length s)))))

(defun newick-read-file1
  (file-name bhz-file ignore-branch-lengths report)
  (setq *newick-ignore-branch-lengths* ignore-branch-lengths)
  (let ((start (internal-run-time)))
    (setq file-name (maybe-decompress-to-tmp file-name report))
    (with-open-file (*standard-input* file-name)
      (newick-first-report report file-name)
      (let (temp (tempn 0) ans (x (cons nil nil)))
        (declare (fixnum tempn))
        (loop
         (loop
          (setq temp (read-char nil nil nil t))
          (cond ((null temp) ; eof
                 (newick-final-report report start)
                 (setq *newick-interior-node-alist* nil)
                 (setq ans (nreverse ans))
                 (cond
                  (bhz-file
                   (compact-print-file ans bhz-file)
                   (let ((x (compact-read-file bhz-file)))
                     (unless (hons-equal x ans)
                       (setq *compact-read-file-output* ans)
                       (setq *compact-print-file-input* x)
                       (ofe "~&; compact-print/read-file error. ~&; ~
                             Cf. *compact-read-file-output* and ~
                             *compact-print-file-input*."))
                     (compact-report report bhz-file))))
                 (return-from newick-read-file1 ans))
                ((eql temp #\[) (newick-consume-comment))
                ((> (setq tempn (char-code (the character temp))) 255)
                 (newick-illegal1 temp))
                ((svref *newick-white* tempn))
                (t (return))))
         (cond ((eql #\( temp)
                (setq *newick-interior-node-alist* nil)
                (setq temp x)
                (ignore-errors
                  (setq temp (newick-cleanup (newick-read-list nil))))
                (when (eq temp x)
                  (newick-read-error
                   "~%; newick-read-file: ** Error."))
                (unless (eql #\; (read-char nil nil nil t))
                  (newick-read-error "~&; Missing semicolon ?"))
                (push (cons temp *newick-interior-node-alist*) ans))
               (t (push (read-line nil t nil t) ans))))))))

(defun newick-read-file (file-name &key
  (bhz-file nil) (ignore-branch-lengths nil) (report t))
  (newick-read-file1 file-name bhz-file ignore-branch-lengths report))

;  hons-let and memoize-let

; It might be a good enhancement to ht-let to permit the carrying
; forward, with hopy-cons-consume, those honses that the user wishes.

(defun not-memoized-error (f)
  (ofe "~&; not-memoized-error:  ** Error:  ~a is not memoized." f))

(defmacro hons-let (form)
  (let ((old-cdr-ht     (gensym "OLD-CDR-HT"))
        (old-cdr-ht-eql (gensym "OLD-CDR-HT-EQL"))
        (old-nil-ht     (gensym "OLD-NIL-HT")))
    `(let ((,old-cdr-ht *hons-cdr-ht*)
           (,old-cdr-ht-eql *hons-cdr-ht-eql*)
           (,old-nil-ht *nil-ht*))
       (unwind-protect
           (progn (clear-hash-tables)
                  ,form)
         (setq *hons-cdr-ht* ,old-cdr-ht)
         (setq *hons-cdr-ht-eql* ,old-cdr-ht-eql)
         (setq *nil-ht* ,old-nil-ht)))))

(defmacro memoize-let (fn form)
  (let ((fn-name (gensym "FN-NAME"))
        (tablevalue (gensym "TABLEVALUE"))
        (ponstablevalue (gensym "PONSTABLEVALUE"))
        (h (gensym "H"))
        (ht1 (gensym "HT1")))
    `(let* ((,fn-name ,fn)
            (,h (memoizedp-raw ,fn-name)))
       (unless ,h (not-memoized-error ,fn-name))
       (let* ((,tablevalue
               (symbol-value
                (access memoize-info-ht-entry ,h :tablename)))
              (,ponstablevalue
               (symbol-value
                (access memoize-info-ht-entry ,h :ponstablename)))
              (,ht1 (mht)))
         (unwind-protect
             (progn (setf (symbol-value
                           (access memoize-info-ht-entry ,h
                                   :tablename))
                          ,ht1)
                    (setf (symbol-value
                           (access memoize-info-ht-entry ,h
                                   :ponstablename))
                           (mht))
                    ,form)
           ;; During the evaluation of form, a change to a stobj may
           ;; result in tablename getting a new value, in which case
           ;; we may not restore its old value.  And a change in the
           ;; memoization status of fn would make a restoration
           ;; pointless.
           (let ((test (and (eq (symbol-value
                                 (access memoize-info-ht-entry
                                         ,h :tablename))
                                ,ht1)
                            (eq ,h (memoizedp-raw ,fn-name)))))
             (setf (symbol-value (access memoize-info-ht-entry
                                         ,h :tablename))
                   (and test ,tablevalue))
             (setf (symbol-value (access memoize-info-ht-entry
                                         ,h :ponstablename))
                   (and test ,ponstablevalue))))))))

(defmacro memoize-on (fn x)
  (let ((f (gensym "F"))
        (h (gensym "H"))
        (o (gensym "O")))
    `(let* ((,f ,fn) (,h (memoizedp-raw ,f)))
       (unless ,h (not-memoized-error ,f))
       (let ((,o (symbol-function (the symbol ,f))))
         (unwind-protect
             (progn (setf (symbol-function (the symbol ,f))
                          (symbol-value
                           (the symbol (access memoize-info-ht-entry
                                               ,h :memoized-fn))))
                    ,x)
           (setf (symbol-function (the symbol ,f)) ,o))))))

(defmacro memoize-off (fn x)
  (let ((f (gensym "F"))
        (h (gensym "H"))
        (o (gensym "O")))
    `(let* ((,f ,fn) (,h (memoizedp-raw ,f)))
       (unless ,h (not-memoized-error ,f))
       (let ((,o (symbol-function (the symbol ,f))))
         (unwind-protect
             (progn (setf (symbol-function (the symbol ,f))
                          (symbol-value
                           (the symbol (access memoize-info-ht-entry
                                               ,h :old-fn))))
                    ,x)
           (setf (symbol-function (the symbol ,f)) ,o))))))

(defun modules ()
  (strip-cars (table-alist 'defm-table (w *the-live-state*))))

; Plev0 provides some support for printing control.

(defun plev0 (length level tuple)
  (our-syntax
   (cond ((not (standard-evisc-tuplep tuple))
          (oft "plev0: *** Illegal tuple: ~a." tuple)))
   (cond ((not (or (null length)
                   (and (integerp length) (>= length 0))))
          (oft "plev0: *** A length argument to plev1 must be an ~
                 integer or nil, not ~a." length)))
   (cond ((not (or (null level) (and (integerp level) (>= level 0))))
          (oft "plev0: *** A level argument to plev1 must be an ~
                 integer or nil, not ~a." level))))
  (setq *print-length* length)
  (setq *print-level* level)
  #+openmcl
  (progn
    (setq ccl:*backtrace-print-level* level)
    (setq ccl:*trace-print-level* level)
    (setq ccl:*backtrace-print-length* length)
    (setq ccl:*trace-print-length* length))
  (setf ; avoid warning that *trace-evisc-tuple* is undefined
   (symbol-value '*trace-evisc-tuple*) tuple)
  (setq *approved-user-default-evisc-tuple* tuple)
  (setq *approved-user-term-evisc-tuple*    tuple)
  nil)

(defun honsp-check (x)
  (cond
   ((consp x)
    #+hons
    (when (not (honsp x))
      (er hard 'honsp-check
          "The value ~X01 is a consp but, contrary to expectation, ~
           not a honsp."
          x
          (list nil 3 4 nil)))
    t)
   ((typep x '(and array string))
    #+hons
    (when (not (gethash x *hons-str-ht*))
      (er hard 'honsp-check
          "The value ~X01 is a string or bignum but, contrary to ~
           expectation, not a string or bignum hashed for the HONS ~
           implementation."
          x
          (list nil 3 4 nil)))
    t)
   (t nil)))

(defun needs-shrinking (h)
  (let ((size (hash-table-size h)))
    (declare (fixnum size))
    (and (> size (cond ((eq h *hons-cdr-ht*) *hons-cdr-ht-size*)
                       ((eq h *nil-ht*) *nil-ht-size*)
                       (t *small-ht-size*)))
         (< (* 3 (hash-table-count h)) size))))

(defun maybe-shrink-sub (h)
  (cond ((consp h)
         (let ((nh (mht :size (length h) :weak :value)))
           (loop for x in h do (setf (gethash (car x) nh) x))
           nh))
        ((needs-shrinking h)
         (let ((nh (mht :size (* 3 (hash-table-count h))
                        :weak :value)))
           (maphash (lambda (k v) (setf (gethash k nh) v)) h)
           nh))
        (t h)))

; We sometimes replace lists with subsidiary hash tables even though
; short lists are faster to search.  By converting to hash tables, we
; permit the garbage collection of honses that are referenced by no
; one else, thanks to idea of 'weak' in OpenMCL.  We sometimes convert
; subsidiary hash tables back to lists when their counts are low
; enough because short lists are faster to search.

; Possible improvement: In maybe-shrink-..., it might be a better idea
; to move all the honses on lists to a single weak value hash table,
; instead of to many small hash tables, then, do the gc, and then
; rehash all those values (back to lists).  One downside to putting
; all those honses into a hash table is that they would all have to be
; rehashed to be placed back into the correct sub lists; if they are
; kept in separate small hashtables, one merely needs to maphash and
; create a list.

(defun maybe-shrink-main (h)
  (maphash (lambda (k v)
             (cond ((and (not (listp v))
                         (eql 0 (hash-table-count v)))
                    (remhash k h))
                   (t (let ((nv (maybe-shrink-sub v)))
                        (or (eq nv v) (setf (gethash k h) nv))))))
           h)
  (cond ((needs-shrinking h)
         (let ((nh (mht :test (hash-table-test h)
                        :size (* 3 (hash-table-count h))
                        #+openmcl :weak
                        #+openmcl (ccl::hash-table-weak-p h))))
               (maphash (lambda (k v) (setf (gethash k nh) v)) h)
               nh))
        (t h)))

(defun ht-list (h)
  (maphash
   (lambda (k v)
     (if (not (listp v))
         (let ((c (hash-table-count v)))
           (declare (fixnum c))
           (cond ((eql 0 c) (remhash k h))
                 ((< c *start-car-ht-size*)
                  (setf (gethash k h)
                        (let (l)
                          (maphash (lambda (k v) (declare (ignore k))
                                     (push v l))
                                   v)
                          l)))))))
   h))

#+openmcl
; *gc-msg-for-our-gc* is set to t after a gc in OpenMCL
(defglobal *gc-msg-for-our-gc* nil)
(defun our-gc ()
  #+openmcl
  (progn (setq *gc-msg-for-our-gc* nil)
         (ccl::gc)
         (loop (sleep .01)
               (when *gc-msg-for-our-gc*
                 (setq *gc-msg-for-our-gc* nil)
                 (return nil))))
  #+GCL
  (si::gbc t))

(defun maybe-shrink-some-hash-tables ()
  (setq *hons-cdr-ht*     (maybe-shrink-main *hons-cdr-ht*))
  (setq *hons-cdr-ht-eql* (maybe-shrink-main *hons-cdr-ht-eql*))
  (setq *nil-ht*     (maybe-shrink-sub  *nil-ht*))
  (our-gc)
  (ht-list *hons-cdr-ht*)
  (ht-list *hons-cdr-ht-eql*)
  (our-gc))

(defglobal *hons-init-hook*
  #+openmcl
  '(progn
     ;; Entirely optional but subtle things that a user probably ought
     ;; to put into ~/openmcl-init.lisp but may not know to do so.
     (unless (equal '(t t) (multiple-value-list (ccl::gc-verbose-p)))
       (oft "~&; Setting OpenMCL's gc verbose.")
       (ccl::gc-verbose t t))
     (when (ccl::egc-active-p)
       (oft "~&; Turning off OpenMCL's ephemeral gc.")
       (ccl::egc nil))
     (when (< (ccl::lisp-heap-gc-threshold) (expt 10 9))
       (oft "~&; Grabbing a gb of heap.")
       (ccl::set-lisp-heap-gc-threshold (expt 10 9))
       (ccl::use-lisp-heap-gc-threshold))
     (when (and (boundp 'ccl::*quit-on-eof*)
                (not (eq t (symbol-value 'ccl::*quit-on-eof*))))
       (oft "~&; Control-d exits.")
       (setf (symbol-value 'ccl::*quit-on-eof*) t)))
  #-openmcl
  nil)

(defun hons-readtable-init ()
  (setq *hons-readtable* (copy-readtable *acl2-readtable*))
  (let ((*readtable* *hons-readtable*))
    (set-macro-character #\( #'hons-read-reader)
    (set-macro-character #\) #'close-paren-read-macro)
    (set-macro-character #\. #'dot-read-macro t)))

(defun compact-read-filetable-init ()
  (setq *compact-read-file-readtable*
        (copy-readtable *hons-readtable*))
  (let ((*readtable* *compact-read-file-readtable*))
    (set-dispatch-macro-character #\# #\# #'ns-ns-reader)
    (set-dispatch-macro-character #\# #\= #'ns-=-reader)))

(defun newick-init ()
  (loop for c in (coerce "0123456789" 'list)
        as i from 0 do (setf (aref *newick-digit* (char-code c)) i))
  (loop for c in (coerce "();:," 'list)
        do (setf (aref *newick-specials* (char-code c)) t))
  (loop for n from 0 to 6 do
        (setf (aref *newick-white* n) t))
  (loop for c in '(#\Tab #\Newline #\Space) do
        (setf (aref *newick-white* (char-code c)) t))
  (loop for c in (coerce "ABCDEFGHIJKLMNOPQRSTUVWXYZ" 'list) do
        (setf (aref *newick-ordinary* (char-code c)) c))
  (loop for c in (coerce "abcdefghijklmnopqrstuvwxyz" 'list) do
        (setf (aref *newick-ordinary* (char-code c)) (char-upcase c)))
  (loop for c in (coerce "0123456789" 'list) do
        (setf (aref *newick-ordinary* (char-code c)) c))
  (loop for c in (coerce ".<>~!@#$%^&|?" 'list) do
        (setf (aref *newick-ordinary* (char-code c)) c))
  (setf (aref *newick-ordinary* (char-code #\_)) #\Space)
  (loop for x in (coerce "()[]{}/\\,;:=*'\"`+-<>" 'list)
        do (setf (aref *newick-punct* (char-code x)) t))
  (loop for i below 256 when (or (aref *newick-white* i)
                                 (aref *newick-punct* i))
        do (setf (aref *newick-pw* i) t)))

(defun float-ticks-per-second-init ()
  (setq *float-ticks-per-second*
        #+RDTSC
        (let ((i1 (ccl::rdtsc))
              (i2 (progn (sleep .01) (ccl::rdtsc))))
          (if (>= i2 i1)
              (* 100 (float (- i2 i1)))
            (our-syntax
             (error "~&; hons-init:  ** Error:  RDTSC nonsense."))))
        #-RDTSC
        (float internal-time-units-per-second)))

(defun maybe-decompress-to-tmp (file report)
  #-(or gcl openmcl) (declare (ignore report))
  #+(or gcl openmcl)
  (let ((temp (ofn "/tmp/tmp-nwk-~a" (getpid$)))
        (name (namestring file)))
    (cond
     ((and (> (length name) 4)
           (equal ".bz2" (subseq name
                                 (- (length name) 4)
                                 (length name))))
      (when report
        (oft "~&; Bunzipping2 ~a ~&; to ~a." file temp))
      #+gcl (lisp:system (ofn "bunzip2 < ~a > ~a" name temp))
      #+openmcl (ccl::os-command (ofn "bunzip2 < ~a > ~a" name temp))
      temp)
     ((and (> (length name) 3)
           (equal ".gz" (subseq name
                                (- (length name) 3)
                                (length name))))
      (when report
        (oft "~&; Gunzipping ~a ~&; to ~a." file temp))
      #+gcl (lisp:system
             (ofn "gunzip --to-stdout < ~a > ~a" name temp))
      #+openmcl (ccl::os-command
                 (ofn "gunzip --to-stdout < ~a > ~a" name temp))
      temp)
     (t name)))
  #-(or gcl openmcl) 
  file)

(defun hons-init ()
  (hons-readtable-init)
  (compact-read-filetable-init)
  (newick-init)
  (float-ticks-per-second-init)
  (unless *hons-tables-initialized*
    (init-hash-tables))
  (when (>= 1 (length *memoize-array*))
    (setq *memoize-array*
          (make-array (* *max-memoize-fns* *max-memoize-fns*)
                      :element-type 'fixnum
                      :initial-element 0)))
  #+openmcl
  (ccl::add-gc-hook (lambda () (setq *gc-msg-for-our-gc* t))
                    :post-gc)
  (eval *hons-init-hook*))

; Watch

; These vars are bound, so we use defparameter, not defglobal.
(defvar *watch-start-run-time*)
(defvar *watch-start-real-time*)
(defvar *watch-start-universal-time*)

(defglobal *watch-processes* nil)

; *watch-remind-syms* may be freely reset by the user to
; whatever the user wishes to be reminded about.
(defglobal *watch-remind-syms*
  '(*default-memoize-summary-order*
    *default-memoize-summary-fns*
    *watch-remind-syms*))

(defglobal *watch-root1* "~a.watch.tmp.text")
(defglobal *watch-root2* "~a.watch.tmp.lisp")

; watch-forms may be freely redefined by the user to produce
; whatever forms one wants watched.
(defun watch-forms (x)
  (case x
    (1 '((princ (documentation 'watch 'function))
         (princ (documentation 'memoize-summary 'function))
         (time-of-last-watch-update)
         (memoize-summary)))
    (2 '((memsum)
         (hsum)))))

(defun watch-forms-silly (x)

 "This function, watch-forms-silly is mentioned in the documentation
  of the function watch.  Someone might give watch-forms this
  definition to see a lot of stuff in a watch buffer."

 (declare (ignore x)) '(
  (time-of-last-watch-update)
  (memoize-summary)
  (hons-summary)
  (run-time-since-watch-start)
  (real-time-since-watch-start)
  (elapsed-time-since-watch-start)
  (hons-calls-per-second-of-run-time)
  (pons-calls-per-second-of-run-time)
  (hons-hits/calls)
  (room t)
  (sh "pwd")
  (sh "ps uax | head")
  (number-of-cpus-on-this-machine)
  (number-of-gcs)
  (princ (documentation 'memoize-summary 'function))
  (princ (documentation 'watch 'function))
  (princ (get-doc-string 'ld state))
  (princ (get-doc-string 'memoize state))
  (watch-remind-syms)

))

#+openmcl
(defun watch-kill ()
  (mapc #'ccl::process-kill *watch-processes*)
  (setq *watch-processes* nil))

(defglobal *watch-list*
  '((*watch-start-run-time* (get-internal-run-time))
    (*watch-start-real-time* (get-internal-real-time))
    (*watch-start-universal-time* (get-universal-time))
    (*print-level* nil)
    (*print-length* nil)
    (*print-pretty* t)
    (*readtable* *acl2-readtable*)
    (*package* *acl2-package*)
    (*print-base* 10)
    (*print-circle* nil)
    (*read-suppress* nil)))

#+openmcl
(defun watch (&optional (n 1))
  "

    After invoking (watch n) in raw OpenMCL, the elements of
  (watch-forms n) are repeatedly evaluated in order every few seconds,
  and all the resulting *standard-output* is written to the file
  ~/n.watch.tmp.text.  n defaults to 1.
    If (load-file (symbol-name 'watch.el)) was in your .emacs file,
  then c-t c-w will start viewing ~/1.watch.tmp.text in auto-revert
  mode.  The file watch.el is in the emacs subdirectory of ACL2.
    c-t 2 sends the current form from Emacs for evaluation in ACL2;
  c-t 3 sends it for evaluation in raw OpenMCL.  Evaluation takes
  place as if it occurs during a control-c interrupt, so
  Kyrie Eleison.  For each such evaluation, for some n, a file named
  ~/n.watch.tmp.lisp is written, read, and deleted.
    After invoking (watch 2) in raw OpenMCL, use c-u 2 c-t c-w in
  Emacs, and so forth, for all 2, for as many channnels of
  communication between as many ACL2s and as many Emacs as you please.
  Equivalently, invoke (watch 2) in Emacs Lisp.
    (watch-kill) kills all the watch processes in an OpenMCL.  Killing
  the buffers in Emacs is accomplished, as usual, with c-x k.
    m-x auto-revert will toggle auto-revert mode thus stop
  the updating.  A subsequent m-x auto-revert will restart the
  updating.
    Many of the watching options are sensitive to *print-right-margin*.
  See the definition of the function watch-forms-silly for about 18
  typical watching options.
  "
  (let* ((uhp (user-homedir-pathname))
         (o (merge-pathnames (ofn *watch-root1* n) uhp))
         (e (merge-pathnames (ofn *watch-root2* n) uhp))
         (sem (ccl::make-semaphore))
         (il (slot-value ccl:*application*
                         'ccl::initial-listener-process))
         (p nil)
         (wvars (mapcar #'car *watch-list*))
         (wvals (mapcar #'eval (mapcar #'cadr *watch-list*)))
         (wstr (make-array 4000 :element-type 'character
                           :adjustable t :fill-pointer t))
         (z (cons 0 0)))
    (push
     (ccl::process-run-function
      (ofn "watch-process-~a" n)
      (lambda ()
        (loop for i from 1 do
         (ignore-errors
           (sleep 1)
           (when (or (setq p (probe-file e)) (eql 0 (mod i 5)))
             (ccl::process-interrupt il
              (lambda ()
               (ignore-errors
                (progv wvars wvals
                  (when p
                    (with-open-file (s e)
                      (let* ((*standard-output* *terminal-io*)
                             (*error-output* *terminal-io*)
                             (*print-level* 3)
                             (*print-length* 3)
                             (form (read s))
                             (v (multiple-value-list (eval form))))
                        (oft "~%~s~% =>" form)
                        (mapc #'print v)
                        (terpri)))
                    (delete-file e))
                  (setf (fill-pointer wstr) 0)
                  (with-output-to-string
                    (*standard-output* wstr)
                    (loop for form in (watch-forms n) do
                          (let ((*print-case* :downcase))
                            (oft "~&~s " form))
                          (let ((val z))
                            (ignore-errors (setq val (eval form)))
                            (when (eq val z)
                              (oft "~50t? error in eval ?")))))
                  (with-open-file (s o :direction :output
                                     :if-does-not-exist :create
                                     :if-exists :overwrite)
                    (ccl::stream-length s 0)
                    (write-string wstr s)
                    (ccl::stream-length s (length wstr)))
                  (ccl::signal-semaphore sem)))))
             (ccl::wait-on-semaphore sem))))))
     *watch-processes*)))

;  Miscellaneous functions for totally optional use via watch.

(defun time-of-last-watch-update ()
  (multiple-value-bind (sec mi h d mo y)
      (decode-universal-time (get-universal-time))
    (let (m)
      (cond ((> h 12)
             (setq m " p.m.")
             (setq h (- h 12)))
            (t (setq m " a.m.")))
      (let ((ans (ofn "~2,d:~2,'0d:~2,'0d~a ~4d/~d/~d"
                      h mi sec m y mo d)))
        (loop for i from 1 to
              (- (right-margin)
                 3
                 (length ans)
                 (length "time-of-last-watch-update"))
              do (write-char #\Space))
        (oft ans)))))

(defun hons-hits/calls ()
  (let ((c (counter-value *hons-call-counter*))
        (m (counter-value *hons-misses-counter*)))
    (let ((ans (ofn "~:d/~:d" (- c m) c)))
      (loop for i from 1 to
            (- (right-margin)
               3
               (length ans)
               (length "hons-hits/calls"))
            do (write-char #\Space))
      (oft ans))))

(defun hons-calls-per-second-of-run-time ()
  (let ((ans (ofn "~:d"
                  (floor (/ (counter-value *hons-call-counter*)
                            (/ (- (get-internal-run-time)
                                  *watch-start-run-time*)
                               (float internal-time-units-per-second
                                      )))))))
    (loop for i from 1 to
          (- (right-margin)
             3
             (length ans)
             (length "hons-calls-per-second-of-run-time"))
          do (write-char #\Space))
    (oft ans)))

(defun pons-calls-per-second-of-run-time ()
  (let ((ans (ofn "~:d"
                  (floor (/ (counter-value *pons-call-counter*)
                            (/ (- (get-internal-run-time)
                                  *watch-start-run-time*)
                               (float internal-time-units-per-second
                                      )))))))
    (loop for i from 1
          to (- (right-margin)
                3
                (length ans)
                (length "pons-calls-per-second-of-run-time"))
          do (write-char #\Space))
    (oft ans)))

(defun number-of-gcs ()
  #+openmcl
  (oft "~vt~7d"
       (- (right-margin)
          7)
       (cadr (multiple-value-list (ccl::gccounts))))
  #-openmcl
  "unknown")
 
(defun number-of-cpus-on-this-machine ()
  #-openmcl
  (oft "Unknown")
  #+openmcl
  (let ((ans (ofn "~s" (ccl::cpu-count))))
    (loop for i from 1 to
          (- (right-margin)
             3
             (length ans)
             (length "number-of-cpus-on-this-machine"))
          do (write-char #\Space))
    (oft ans)))

#+openmcl
(defun sh (str)
  (let ((file (ofn "/tmp/watch-~a.XXXX" (ccl::getpid))))
    (ccl::os-command (ofn "~a > ~a" str file))
    (with-open-file (stream file)
        (let (c)
          (terpri)
          (loop while (setq c (read-char stream nil nil))
                do (write-char c))))))

(defun current-form ()
  (let* ((*print-level* 3)
         (*print-length* 3)
         (*print-pretty* nil)
         (ans (ofn "~s" -)))
    (loop for i from 1 to (- (right-margin)
                             (length ans)
                             (length "current-form"))
          do (write-char #\Space))
    (oft ans)))

(defun run-time-since-watch-start ()
  (let (h mi (sec (floor (- (get-internal-run-time)
                            *watch-start-run-time*)
                         internal-time-units-per-second)))
    (multiple-value-setq (mi sec) (floor sec 60))
    (multiple-value-setq (h mi) (floor mi 60))
    (oft "~vt~2d:~2,'0d:~2,'0d" (- (right-margin) 8) h mi sec)))

(defun real-time-since-watch-start ()
  (let (h mi (sec (floor (- (get-internal-real-time)
                            *watch-start-real-time*)
                         internal-time-units-per-second)))
    (multiple-value-setq (mi sec) (floor sec 60))
    (multiple-value-setq (h mi) (floor mi 60))
    (oft "~vt~2d:~2,'0d:~2,'0d" (- (right-margin) 8) h mi sec)))

(defun elapsed-time-since-watch-start ()
  (let (h mi (sec (- (get-universal-time)
                     *watch-start-universal-time*)))
    (multiple-value-setq (mi sec) (floor sec 60))
    (multiple-value-setq (h mi) (floor mi 60))
    (oft "~vt~2d:~2,'0d:~2,'0d" (- (right-margin) 8) h mi sec)))

(defun watch-remind-syms ()
  (loop for v in *watch-remind-syms* do
        (oft "~%(setq ~s ~30t '~s)" v (symbol-value v))))

; Older shorter names for a few commands.

(defun memsum (&rest r)
  (apply #'memoize-summary r))
(defun hsum (&rest r)
  (apply #'hons-summary r))
(defun memstat (&rest r)
  (apply #'memoize-statistics r))
(defun hstat (&rest r)
  (apply #'hons-statistics r))
(defmacro memo-on (&rest r)
  `(memoize-on ,@r))
(defmacro memo-off (&rest r)
  `(memoize-off ,@r))
(defun clear-memo-tables (&rest r)
  (apply #'clear-memoize-tables r))
(defun clear-hash-and-memo-tables (&rest r)
  (apply #'clear-hash-and-memoize-tables r))

; Optional profiling functions.

(defun profile (fn n)
  (when (memoizedp-raw fn)
    (oft "~&; profile:  Unmemoizing ~a." fn)
    (unmemoize-fn fn))
  (memoize-fn fn
              :condition  nil
              :inline     nil
              :formals    (loop for i below n
                               collect (intern (ofn "x-~a" i)))
              :stobjs-in  (make-list n)
              :stobjs-out (list nil)))

(defun profile-hons ()
  (loop for fn in '(hons-copy1
                    hons-copy1-consume
                    compact-read-file)
        do (profile fn 1))
  (loop for fn in '(assoc-no-error-at-end
                    hons
                    hons-copy
                    hons-normed
                    hons-when-x-is-honsp
                    hons-when-y-is-honsp
                    hons-get-fn-do-not-hopy
                    hons-get-fn-do-hopy
                    compact-print-file)
        do (profile fn 2))
  (loop for fn in '(hons-normed-with-suggestion
                    hons-acons
                    hons-acons!)
        do (profile fn 3)))

(defun profile-newick ()
  (loop for fn in '(newick-consume-comment
                    newick-read-token
                    newick-read-number)
        do (profile fn 0))
  (loop for fn in '(newick-cleanup
                    newick-read-list
                    maybe-decompress-to-tmp)
        do (profile fn 1))
  (profile 'newick-read-file1 4)
  (profile-hons))


; Optional testing function.

(defun newick-test (str &key
                        (bhz-file              "/tmp/foo.bhz")
                        (ignore-branch-lengths nil)
                        (report                t))
  (with-open-file (*standard-output* "/tmp/foo.nwk"
                                     :direction :output
                                     :if-exists :supersede
                                     :if-does-not-exist :create)
    (princ str))
  (newick-read-file "/tmp/foo.nwk"
                    :bhz-file              bhz-file
                    :ignore-branch-lengths ignore-branch-lengths
                    :report                report))
