/*
 * Copyright (C) 2012-2015 Canonical, Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

import QtQuick 2.2
import Ubuntu.Components 1.0
import Ubuntu.Components.Popups 1.0 as Popups

MainView {
    id: mainWindow
    objectName: "addressBookAppMainWindow"

    property string modelErrorMessage: ""
    readonly property bool appActive: Qt.application.active

    signal applicationReady()

    function contact(contactId)
    {
        mainStack.resetStack()
        if (mainStack.contactListPage) {
            mainStack.contactListPage.showContact(contactId)
        } else {
            console.error("Contact preview requested but ContactListPage not loaded")
        }
        mainStack.quitOnDepth = 1
    }

    function create(phoneNumber)
    {
        mainStack.resetStack()
        if (mainStack.contactListPage) {
            mainStack.contactListPage.createContactWithPhoneNumber(phoneNumber)
        } else {
            console.error("Contact creation requested but ContactListPage not loaded")
        }
    }

    function addphone(contactId, phoneNumber)
    {
        mainStack.resetStack()
        if (mainStack.contactListPage) {
            mainStack.contactListPage.addPhoneToContact(contactId, phoneNumber)
        } else {
            console.error("Add phone to contact requested but ContactListPage not loaded")
        }
    }

    function pick(single)
    {
        console.debug("Pick mode:" + single)
        pickWithTransfer(single === "true", null)
    }

    function pickWithTransfer(single, activeTransfer)
    {
        mainStack.resetStack()
        if (mainStack.contactListPage) {
            mainStack.contactListPage.startPickMode(single, activeTransfer)
        } else {
            console.error("Pick mode requested but ContactListPage not loaded")
        }
    }

    function importvcard(_url)
    {
        importvcards([_url])
    }

    function importvcards(_urls)
    {
        mainStack.resetStack()
        if (mainStack.contactListPage) {
            mainStack.contactListPage.importContact(_urls)
        } else {
            console.error("Import vcard requested but ContactListPage not loaded")
        }
    }

    function addnewphone(phoneNumer)
    {
        mainStack.resetStack()
        if (mainStack.contactListPage) {
            mainStack.contactListPage.addNewPhone(phoneNumer)
        } else {
            console.error("Add new phone requested but ContactListPage not loaded")
        }
    }

    width: units.gu(40)
    height: units.gu(71)
    anchorToKeyboard: false
    useDeprecatedToolbar: false

    PageStack {
        id: mainStack

        property var contactListPage: null
        property int quitOnDepth: -1

        function resetStack()
        {
            while(depth > 1) {
                pop()
            }
        }

        onDepthChanged: {
            if (depth === quitOnDepth) {
                quitOnDepth = -1
                application.goBackToSourceApp()
            }
        }

        onContactListPageChanged: {
            if (contentHubLoader.status === Loader.Ready) {
                contentHubLoader.item.pageStack = mainStack
            } else {
                contentHubLoader.setSource(Qt.resolvedUrl("ContentHubProxy.qml"), {"pageStack": mainStack})
            }
        }

        anchors.fill: parent
    }

    Component.onCompleted: {
        application.elapsed()
        i18n.domain = "address-book-app"
        i18n.bindtextdomain("address-book-app", i18nDirectory)
        mainStack.push(Qt.resolvedUrl("ABContactListPage.qml"))
        mainWindow.applicationReady()
    }

    Component {
        id: errorDialog

        Popups.Dialog {
            id: dialogue

            title: i18n.tr("Error")
            text: mainWindow.modelErrorMessage

            Button {
                text: i18n.tr("Cancel")
                gradient: UbuntuColors.greyGradient
                onClicked: PopupUtils.close(dialogue)
            }
        }
    }

    Connections {
        target: UriHandler
        onOpened: {
            for (var i = 0; i < uris.length; ++i) {
                application.parseUrl(uris[i])
            }
        }
    }

    Loader {
        id: contentHubLoader

        //We can not use async load, the export requested signal can be received before the component get ready
        //asynchronous: true
        source: Qt.resolvedUrl("ContentHubProxy.qml")
        onStatusChanged: {
            if (status === Loader.Ready) {
                item.pageStack = mainStack
            }
        }
    }

    // If application was called from uri handler and lost the focus reset the app to normal state
    onAppActiveChanged: {
        if (!appActive && mainStack.contactListPage) {
            mainStack.quitOnDepth = -1
            mainStack.contactListPage.returnToNormalState()
        }
    }
}
