# C interpreter mediator.
#
# Author::    Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>
# Copyright:: Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
# License::   GPLv3+: GNU General Public License version 3 or later
#
# Owner::     Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>

#--
#     ___    ____  __    ___   _________
#    /   |  / _  |/ /   / / | / /__  __/           Source Code Static Analyzer
#   / /| | / / / / /   / /  |/ /  / /                   AdLint - Advanced Lint
#  / __  |/ /_/ / /___/ / /|  /  / /
# /_/  |_|_____/_____/_/_/ |_/  /_/   Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
#
# This file is part of AdLint.
#
# AdLint is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# AdLint is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# AdLint.  If not, see <http://www.gnu.org/licenses/>.
#
#++

require "adlint/c/type"
require "adlint/c/object"
require "adlint/c/enum"
require "adlint/c/syntax"

module AdLint #:nodoc:
module C #:nodoc:

  module TypeTableMediator
    # NOTE: This module works like a CRTP base entity in C++.

    include StandardTypeCatalogAccessor

    extend Forwardable

    def_delegator :type_table, :lookup, :type_of
    def_delegator :type_table, :undeclared_type
    def_delegator :type_table, :wchar_type
    def_delegator :type_table, :array_type
    def_delegator :type_table, :function_type
    def_delegator :type_table, :bitfield_type
    def_delegator :type_table, :pointer_type
    def_delegator :type_table, :qualified_type

    private
    def standard_type_catalog
      type_table.standard_type_catalog
    end

    def type_table
      # NOTE: Host class must respond to #type_table.
      subclass_responsibility
    end
  end

  module MemoryPoolMediator
    # NOTE: This module works like a CRTP base entity in C++.

    extend Forwardable

    def_delegator :memory_pool, :lookup, :memory_at

    def pointee_of(pointer)
      # FIXME: This method should return multiple objects, because domain of
      #        the pointer_value may have multiple address values.
      #
      # pointer.value.to_enum.map { |address|
      #   memory = memory_at(address) ? memory.binding.object : nil
      # }.compact

      pointer.value.to_enum.each do |address|
        if memory = memory_at(address)
          object = memory.binding.object
          if object.variable? or
              object.function? && pointer.type.unqualify.base_type.function?
            return object
          end
        end
      end
      nil
    end

    private
    def memory_pool
      # NOTE: Host class must respond to #memory_pool.
      subclass_responsibility
    end
  end

  module VariableTableMediator
    # NOTE: This module works like a CRTP base entity in C++.

    extend Forwardable

    def_delegator :variable_table, :lookup, :variable_named
    def_delegator :variable_table, :declare, :declare_variable
    def_delegator :variable_table, :define, :define_variable
    def_delegator :variable_table, :storage_duration_of

    def local_variables
      variable_table.all_named_variables.select { |variable|
        variable.scope.local?
      }
    end

    def temporary_variable(type = undeclared_type,
                           value = type.undefined_value)
      variable_table.define_temporary(type, value)
    end

    private
    def variable_table
      # NOTE: Host class must respond to #variable_table.
      subclass_responsibility
    end
  end

  module FunctionTableMediator
    # NOTE: The host class of this module must include TypeTableMediator.
    # NOTE: This module works like a CRTP base entity in C++.

    extend Forwardable

    def_delegator :function_table, :lookup, :function_named
    def_delegator :function_table, :declare, :declare_function
    def_delegator :function_table, :define, :define_function

    def define_explicit_function(function_declaration_or_definition)
      define_function(ExplicitFunction.new(function_declaration_or_definition))
    end

    def define_implicit_function(name)
      define_function(ImplicitFunction.new(default_function_type, name))
    end

    def define_anonymous_function(type)
      define_function(AnonymousFunction.new(type))
    end

    private
    def default_function_type
      function_type(int_type, [])
    end

    def function_table
      # NOTE: Host class must respond to #function_table.
      subclass_responsibility
    end
  end

  module EnumeratorTableMediator
    # NOTE: This module works like a CRTP base entity in C++.

    extend Forwardable

    def_delegator :enumerator_table, :lookup, :enumerator_named
    def_delegator :enumerator_table, :define, :define_enumerator

    private
    def enumerator_table
      # NOTE: Host class must respond to #enumerator_table.
      subclass_responsibility
    end
  end

  module NotifierMediator
    # NOTE: This module works like a CRTP base entity in C++.

    extend Forwardable

    def_delegator :interpreter, :notify_variable_declared
    def_delegator :interpreter, :notify_variable_defined
    def_delegator :interpreter, :notify_variable_initialized
    def_delegator :interpreter, :notify_function_declared
    def_delegator :interpreter, :notify_function_defined
    def_delegator :interpreter, :notify_struct_declared
    def_delegator :interpreter, :notify_union_declared
    def_delegator :interpreter, :notify_enum_declared
    def_delegator :interpreter, :notify_typedef_declared
    def_delegator :interpreter, :notify_function_started
    def_delegator :interpreter, :notify_function_ended
    def_delegator :interpreter, :notify_parameter_defined
    def_delegator :interpreter, :notify_variable_referred
    def_delegator :interpreter, :notify_constant_referred
    def_delegator :interpreter, :notify_variable_value_referred
    def_delegator :interpreter, :notify_variable_value_updated
    def_delegator :interpreter, :notify_function_referred
    def_delegator :interpreter, :notify_sizeof_expr_evaled
    def_delegator :interpreter, :notify_sizeof_type_expr_evaled
    def_delegator :interpreter, :notify_explicit_conv_performed
    def_delegator :interpreter, :notify_implicit_conv_performed
    def_delegator :interpreter, :notify_array_subscript_expr_evaled
    def_delegator :interpreter, :notify_function_call_expr_evaled
    def_delegator :interpreter, :notify_unary_arithmetic_expr_evaled
    def_delegator :interpreter, :notify_multiplicative_expr_evaled
    def_delegator :interpreter, :notify_additive_expr_evaled
    def_delegator :interpreter, :notify_shift_expr_evaled
    def_delegator :interpreter, :notify_relational_expr_evaled
    def_delegator :interpreter, :notify_equality_expr_evaled
    def_delegator :interpreter, :notify_and_expr_evaled
    def_delegator :interpreter, :notify_exclusive_or_expr_evaled
    def_delegator :interpreter, :notify_inclusive_or_expr_evaled
    def_delegator :interpreter, :notify_logical_and_expr_evaled
    def_delegator :interpreter, :notify_logical_or_expr_evaled
    def_delegator :interpreter, :notify_conditional_expr_evaled
    def_delegator :interpreter, :notify_address_expr_evaled
    def_delegator :interpreter, :notify_indirection_expr_evaled
    def_delegator :interpreter, :notify_member_access_expr_evaled
    def_delegator :interpreter, :notify_prefix_increment_expr_evaled
    def_delegator :interpreter, :notify_postfix_increment_expr_evaled
    def_delegator :interpreter, :notify_prefix_decrement_expr_evaled
    def_delegator :interpreter, :notify_postfix_decrement_expr_evaled
    def_delegator :interpreter, :notify_assignment_expr_evaled
    def_delegator :interpreter, :notify_switch_stmt_started
    def_delegator :interpreter, :notify_switch_stmt_ended
    def_delegator :interpreter, :notify_while_stmt_started
    def_delegator :interpreter, :notify_while_stmt_ended
    def_delegator :interpreter, :notify_do_stmt_started
    def_delegator :interpreter, :notify_do_stmt_ended
    def_delegator :interpreter, :notify_for_stmt_started
    def_delegator :interpreter, :notify_for_stmt_ended
    def_delegator :interpreter, :notify_c99_for_stmt_started
    def_delegator :interpreter, :notify_c99_for_stmt_ended
    def_delegator :interpreter, :notify_goto_stmt_evaled
    def_delegator :interpreter, :notify_return_stmt_evaled
    def_delegator :interpreter, :notify_if_ctrlexpr_evaled
    def_delegator :interpreter, :notify_if_else_ctrlexpr_evaled
    def_delegator :interpreter, :notify_switch_ctrlexpr_evaled
    def_delegator :interpreter, :notify_case_ctrlexpr_evaled
    def_delegator :interpreter, :notify_while_ctrlexpr_evaled
    def_delegator :interpreter, :notify_do_ctrlexpr_evaled
    def_delegator :interpreter, :notify_for_ctrlexpr_evaled
    def_delegator :interpreter, :notify_c99_for_ctrlexpr_evaled
    def_delegator :interpreter, :notify_label_defined
    def_delegator :interpreter, :notify_block_started
    def_delegator :interpreter, :notify_block_ended
    def_delegator :interpreter, :notify_branch_started
    def_delegator :interpreter, :notify_branch_ended
    def_delegator :interpreter, :notify_translation_unit_started
    def_delegator :interpreter, :notify_translation_unit_ended
    def_delegator :interpreter, :notify_sequence_point_reached

    private
    def interpreter
      # NOTE: Host class must respond to #interpreter.
      subclass_responsibility
    end
  end

  module InterpreterMediator
    # NOTE: This module works like a CRTP base entity in C++.

    include TypeTableMediator
    include MemoryPoolMediator
    include VariableTableMediator
    include FunctionTableMediator
    include EnumeratorTableMediator

    def interpret(node, *options)
      interpreter.execute(node, *options)
    end

    def reset_environment
      environment.reset
    end

    extend Forwardable

    def_delegator :interpreter, :object_to_variable
    def_delegator :interpreter, :value_of
    def_delegator :interpreter, :pointer_value_of

    private
    def scoped_eval(&block)
      environment.enter_scope
      yield
    ensure
      environment.leave_scope
    end

    def branched_eval(expression = nil, *options, &block)
      current_branch = environment.enter_branch(*options)
      interpreter.notify_branch_started(current_branch)
      current_branch.execute(interpreter, expression, &block)
    ensure
      interpreter.notify_branch_ended(current_branch)
      environment.leave_branch_group if current_branch.final?
      environment.leave_branch
    end

    def resolve_unresolved_type(node)
      interpreter.type_resolver.resolve(node) if node.type.unresolved?
    end

    extend Forwardable

    def_delegator :interpreter, :environment

    def_delegator :environment, :type_table
    def_delegator :environment, :memory_pool
    def_delegator :environment, :variable_table
    def_delegator :environment, :function_table
    def_delegator :environment, :enumerator_table

    def interpreter
      # NOTE: Host class must respond to #interpreter.
      subclass_responsibility
    end
  end

end
end
