# C type models.
#
# Author::    Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>
# Copyright:: Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
# License::   GPLv3+: GNU General Public License version 3 or later
#
# Owner::     Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>

#--
#     ___    ____  __    ___   _________
#    /   |  / _  |/ /   / / | / /__  __/           Source Code Static Analyzer
#   / /| | / / / / /   / /  |/ /  / /                   AdLint - Advanced Lint
#  / __  |/ /_/ / /___/ / /|  /  / /
# /_/  |_|_____/_____/_/_/ |_/  /_/   Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
#
# This file is part of AdLint.
#
# AdLint is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# AdLint is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# AdLint.  If not, see <http://www.gnu.org/licenses/>.
#
#++

require "adlint/traits"
require "adlint/token"
require "adlint/c/syntax"
require "adlint/c/scope"
require "adlint/c/object"
require "adlint/c/conv"
require "adlint/c/option"

module AdLint #:nodoc:
module C #:nodoc:

  module StandardTypeCatalogAccessor
    extend Forwardable

    def_delegator :standard_type_catalog, :void_type
    def_delegator :standard_type_catalog, :char_type
    def_delegator :standard_type_catalog, :signed_char_type
    def_delegator :standard_type_catalog, :unsigned_char_type
    def_delegator :standard_type_catalog, :short_type
    def_delegator :standard_type_catalog, :signed_short_type
    def_delegator :standard_type_catalog, :unsigned_short_type
    def_delegator :standard_type_catalog, :short_int_type
    def_delegator :standard_type_catalog, :signed_short_int_type
    def_delegator :standard_type_catalog, :unsigned_short_int_type
    def_delegator :standard_type_catalog, :int_type
    def_delegator :standard_type_catalog, :signed_type
    def_delegator :standard_type_catalog, :signed_int_type
    def_delegator :standard_type_catalog, :unsigned_type
    def_delegator :standard_type_catalog, :unsigned_int_type
    def_delegator :standard_type_catalog, :long_type
    def_delegator :standard_type_catalog, :signed_long_type
    def_delegator :standard_type_catalog, :unsigned_long_type
    def_delegator :standard_type_catalog, :long_int_type
    def_delegator :standard_type_catalog, :signed_long_int_type
    def_delegator :standard_type_catalog, :unsigned_long_int_type
    def_delegator :standard_type_catalog, :long_long_type
    def_delegator :standard_type_catalog, :signed_long_long_type
    def_delegator :standard_type_catalog, :unsigned_long_long_type
    def_delegator :standard_type_catalog, :long_long_int_type
    def_delegator :standard_type_catalog, :signed_long_long_int_type
    def_delegator :standard_type_catalog, :unsigned_long_long_int_type
    def_delegator :standard_type_catalog, :float_type
    def_delegator :standard_type_catalog, :double_type
    def_delegator :standard_type_catalog, :long_double_type

    private
    def standard_type_catalog
      # NOTE: Host class must respond to #standard_type_catalog.
      subclass_responsibility
    end
  end

  # == DESCRIPTION
  # === Type class hierarchy
  #  Type
  #    <--- UndeclaredType
  #    <--- UnresolvedType
  #    <--- QualifiedType
  #    <--- VoidType
  #    <--- FunctionType
  #    <--- ScalarDataType ------ UsualArithmeticTypeConversion <<module>>
  #           <--- IntegerType
  #                  <--- StandardIntegerType
  #                         <--- CharType
  #                         <--- SignedCharType
  #                         <--- UnsignedCharType
  #                         <--- ShortType
  #                         <--- SignedShortType
  #                         <--- UnsignedShortType
  #                         <--- ShortIntType
  #                         <--- SignedShortIntType
  #                         <--- UnsignedShortIntType
  #                         <--- IntType
  #                         <--- SignedType
  #                         <--- SignedIntType
  #                         <--- UnsignedType
  #                         <--- UnsignedIntType
  #                         <--- LongType
  #                         <--- SignedLongType
  #                         <--- UnsignedLongType
  #                         <--- LongIntType
  #                         <--- SignedLongIntType
  #                         <--- UnsignedLongIntType
  #                         <--- LongLongType
  #                         <--- SignedLongLongType
  #                         <--- UnsignedLongLongType
  #                         <--- LongLongIntType
  #                         <--- SignedLongLongIntType
  #                         <--- UnsignedLongLongIntType
  #                  <--- ExtendedBigIntType
  #                  <--- BitfieldType
  #                  <--- EnumType ------------------- Scopeable <<module>>
  #                  <--- PointerType                      |
  #           <--- FloatingType                            |
  #                  <--- FloatType                        |
  #                  <--- DoubleType                       |
  #                  <--- LongDoubleType                   |
  #    <--- ArrayType                                      |
  #    <--- CompositeDataType -----------------------------+
  #           <--- StructType                              |
  #           <--- UnionType                               |
  #    <--- UserType --------------------------------------+
  #    <--- ParameterType ---------------------------------+
  class Type
    include StandardTypeCatalogAccessor

    def initialize(type_table, name, type_declarations = [])
      @type_table = type_table
      @name = name
      @declarations = type_declarations
    end

    attr_reader :type_table
    attr_reader :name
    attr_reader :declarations

    def id
      subclass_responsibility
    end

    def image
      subclass_responsibility
    end

    def brief_image
      subclass_responsibility
    end

    def location
      subclass_responsibility
    end

    def bit_size
      subclass_responsibility
    end

    def byte_size
      (bit_size / 8.0).ceil
    end

    def bit_alignment
      subclass_responsibility
    end

    def byte_alignment
      (bit_alignment / 8.0).ceil
    end

    def aligned_bit_size
      bit_size + (bit_alignment - bit_size)
    end

    def aligned_byte_size
      (aligned_bit_size / 8.0).ceil
    end

    def real_type
      subclass_responsibility
    end

    def base_type
      subclass_responsibility
    end

    def unqualify
      subclass_responsibility
    end

    def incomplete?
      subclass_responsibility
    end

    def compatible?(to_type)
      subclass_responsibility
    end

    def coercible?(to_type)
      subclass_responsibility
    end

    def convertible?(to_type)
      self.same_as?(to_type)
    end

    def more_cv_qualified?(than_type)
      false
    end

    def same_as?(type)
      self.real_type.unqualify == type.real_type.unqualify
    end

    def parameter?
      false
    end

    def scalar?
      subclass_responsibility
    end

    def integer?
      subclass_responsibility
    end

    def floating?
      subclass_responsibility
    end

    def array?
      subclass_responsibility
    end

    def composite?
      struct? || union?
    end

    def struct?
      subclass_responsibility
    end

    def union?
      subclass_responsibility
    end

    def pointer?
      subclass_responsibility
    end

    def qualified?
      subclass_responsibility
    end

    def function?
      subclass_responsibility
    end

    def enum?
      subclass_responsibility
    end

    def user?
      subclass_responsibility
    end

    def void?
      subclass_responsibility
    end

    def undeclared?
      subclass_responsibility
    end

    def unresolved?
      subclass_responsibility
    end

    def const?
      subclass_responsibility
    end

    def volatile?
      subclass_responsibility
    end

    def restrict?
      subclass_responsibility
    end

    def bitfield?
      subclass_responsibility
    end

    def signed?
      subclass_responsibility
    end

    def unsigned?
      !signed?
    end

    def explicitly_signed?
      subclass_responsibility
    end

    def have_va_list?
      subclass_responsibility
    end

    def return_type
      subclass_responsibility
    end

    def parameter_types
      subclass_responsibility
    end

    def enumerators
      subclass_responsibility
    end

    def length
      subclass_responsibility
    end

    def impl_length
      subclass_responsibility
    end

    def members
      subclass_responsibility
    end

    def member_named(name)
      subclass_responsibility
    end

    def min_value
      subclass_responsibility
    end

    def max_value
      subclass_responsibility
    end

    def nil_value
      subclass_responsibility
    end

    def zero_value
      subclass_responsibility
    end

    def arbitrary_value
      subclass_responsibility
    end

    def undefined_value
      subclass_responsibility
    end

    def parameter_value
      subclass_responsibility
    end

    def return_value
      subclass_responsibility
    end

    def coerce_scalar_value(value)
      subclass_responsibility
    end

    def coerce_array_value(value)
      subclass_responsibility
    end

    def coerce_composite_value(value)
      subclass_responsibility
    end

    def integer_conversion_rank
      subclass_responsibility
    end

    def integer_promoted_type
      subclass_responsibility
    end

    def argument_promoted_type
      subclass_responsibility
    end

    def arithmetic_type_with(type)
      subclass_responsibility
    end

    def _arithmetic_type_with_undeclared(lhs, rhs = self)
      subclass_responsibility
    end

    def _arithmetic_type_with_unresolved(lhs, rhs = self)
      subclass_responsibility
    end

    def _arithmetic_type_with_void(lhs, rhs = self)
      subclass_responsibility
    end

    def _arithmetic_type_with_function(lhs, rhs = self)
      subclass_responsibility
    end

    def _arithmetic_type_with_char(lhs, rhs = self)
      subclass_responsibility
    end

    def _arithmetic_type_with_signed_char(lhs, rhs = self)
      subclass_responsibility
    end

    def _arithmetic_type_with_unsigned_char(lhs, rhs = self)
      subclass_responsibility
    end

    def _arithmetic_type_with_short(lhs, rhs = self)
      subclass_responsibility
    end

    def _arithmetic_type_with_signed_short(lhs, rhs = self)
      subclass_responsibility
    end

    def _arithmetic_type_with_unsigned_short(lhs, rhs = self)
      subclass_responsibility
    end

    def _arithmetic_type_with_short_int(lhs, rhs = self)
      subclass_responsibility
    end

    def _arithmetic_type_with_signed_short_int(lhs, rhs = self)
      subclass_responsibility
    end

    def _arithmetic_type_with_unsigned_short_int(lhs, rhs = self)
      subclass_responsibility
    end

    def _arithmetic_type_with_int(lhs, rhs = self)
      subclass_responsibility
    end

    def _arithmetic_type_with_signed(lhs, rhs = self)
      subclass_responsibility
    end

    def _arithmetic_type_with_signed_int(lhs, rhs = self)
      subclass_responsibility
    end

    def _arithmetic_type_with_unsigned(lhs, rhs = self)
      subclass_responsibility
    end

    def _arithmetic_type_with_unsigned_int(lhs, rhs = self)
      subclass_responsibility
    end

    def _arithmetic_type_with_long(lhs, rhs = self)
      subclass_responsibility
    end

    def _arithmetic_type_with_signed_long(lhs, rhs = self)
      subclass_responsibility
    end

    def _arithmetic_type_with_unsigned_long(lhs, rhs = self)
      subclass_responsibility
    end

    def _arithmetic_type_with_long_int(lhs, rhs = self)
      subclass_responsibility
    end

    def _arithmetic_type_with_signed_long_int(lhs, rhs = self)
      subclass_responsibility
    end

    def _arithmetic_type_with_unsigned_long_int(lhs, rhs = self)
      subclass_responsibility
    end

    def _arithmetic_type_with_long_long(lhs, rhs = self)
      subclass_responsibility
    end

    def _arithmetic_type_with_signed_long_long(lhs, rhs = self)
      subclass_responsibility
    end

    def _arithmetic_type_with_unsigned_long_long(lhs, rhs = self)
      subclass_responsibility
    end

    def _arithmetic_type_with_long_long_int(lhs, rhs = self)
      subclass_responsibility
    end

    def _arithmetic_type_with_signed_long_long_int(lhs, rhs = self)
      subclass_responsibility
    end

    def _arithmetic_type_with_unsigned_long_long_int(lhs, rhs = self)
      subclass_responsibility
    end

    def _arithmetic_type_with_float(lhs, rhs = self)
      subclass_responsibility
    end

    def _arithmetic_type_with_double(lhs, rhs = self)
      subclass_responsibility
    end

    def _arithmetic_type_with_long_double(lhs, rhs = self)
      subclass_responsibility
    end

    def _arithmetic_type_with_bitfield(lhs, rhs = self)
      subclass_responsibility
    end

    def _arithmetic_type_with_enum(lhs, rhs = self)
      subclass_responsibility
    end

    def _arithmetic_type_with_pointer(lhs, rhs = self)
      subclass_responsibility
    end

    def _arithmetic_type_with_array(lhs, rhs = self)
      subclass_responsibility
    end

    def _arithmetic_type_with_struct(lhs, rhs = self)
      subclass_responsibility
    end

    def _arithmetic_type_with_union(lhs, rhs = self)
      subclass_responsibility
    end

    def _arithmetic_type_with_extended_big_int(lhs, rhs = self)
      subclass_responsibility
    end

    def corresponding_signed_type
      subclass_responsibility
    end

    def corresponding_unsigned_type
      subclass_responsibility
    end

    def ==(rhs)
      case rhs
      when Type
        id == rhs.id
      else
        super
      end
    end

    def inspect
      if @name == image
        if location
          "#{@name} (#{location.inspect})"
        else
          @name
        end
      else
        if location
          "#{@name}=>#{image} (#{location.inspect})"
        else
          "#{@name}=>#{image}"
        end
      end
    end

    private
    def standard_type_catalog
      @type_table.standard_type_catalog
    end
  end

  class TypeId
    def initialize(value)
      @value = value
    end

    def ==(rhs)
      @value == rhs.value
    end

    def eql?(rhs)
      self == rhs
    end

    def hash
      @value.hash
    end

    protected
    attr_reader :value
  end

  class StandardTypeId < TypeId
    def initialize(name)
      super(name.split(" ").sort.join(" "))
    end
  end

  class UndeclaredType < Type
    def initialize(type_table)
      super(type_table, "__adlint__undeclared_type")
    end

    def id
      @id ||= TypeId.new(name)
    end

    def image
      name
    end

    def brief_image
      name
    end

    def location
      nil
    end

    def bit_size
      0
    end

    def bit_alignment
      0
    end

    def real_type
      self
    end

    def base_type
      self
    end

    def unqualify
      self
    end

    def incomplete?
      true
    end

    def compatible?(to_type)
      false
    end

    def coercible?(to_type)
      false
    end

    def convertible?(to_type)
      false
    end

    def same_as?(type)
      false
    end

    def scalar?
      false
    end

    def integer?
      false
    end

    def floating?
      false
    end

    def array?
      false
    end

    def struct?
      false
    end

    def union?
      false
    end

    def pointer?
      false
    end

    def qualified?
      false
    end

    def function?
      false
    end

    def enum?
      false
    end

    def user?
      false
    end

    def void?
      false
    end

    def undeclared?
      true
    end

    def unresolved?
      false
    end

    def const?
      false
    end

    def volatile?
      false
    end

    def restrict?
      false
    end

    def bitfield?
      false
    end

    def signed?
      false
    end

    def explicitly_signed?
      false
    end

    def have_va_list?
      false
    end

    def return_type
      self
    end

    def parameter_types
      []
    end

    def enumerators
      []
    end

    def length
      0
    end

    def impl_length
      0
    end

    def members
      []
    end

    def member_named(name)
      nil
    end

    def min_value
      0
    end

    def max_value
      0
    end

    def nil_value
      ScalarValue.of_nil # NOTREACHED
    end

    def zero_value
      ScalarValue.of_nil # NOTREACHED
    end

    def arbitrary_value
      ScalarValue.of_nil # NOTREACHED
    end

    def undefined_value
      ScalarValue.of_nil # NOTREACHED
    end

    def parameter_value
      ScalarValue.of_nil # NOTREACHED
    end

    def return_value
      ScalarValue.of_nil # NOTREACHED
    end

    def coerce_scalar_value(value)
      ScalarValue.of_nil # NOTREACHED
    end

    def coerce_array_value(value)
      ScalarValue.of_nil # NOTREACHED
    end

    def coerce_composite_value(value)
      ScalarValue.of_nil # NOTREACHED
    end

    def integer_conversion_rank
      0 # NOTREACHED
    end

    def integer_promoted_type
      self # NOTREACHED
    end

    def argument_promoted_type
      self # NOTREACHED
    end

    def arithmetic_type_with(type)
      # NOTE: An arithmetic operation with UndeclaredType must not be executed!
      type._arithmetic_type_with_undeclared(self)
    end

    def _arithmetic_type_with_undeclared(lhs, rhs = self)
      # NOTE: An arithmetic operation with UndeclaredType must not be executed!
      # NOTE: Binary operation with UndeclaredType and UndeclaredType
      #       raises a runtime error.
      raise TypeError,
        "arithmetic-type of UndeclaredType and UndeclaredType is not defined."
    end

    def _arithmetic_type_with_unresolved(lhs, rhs = self)
      # NOTE: An arithmetic operation with UndeclaredType must not be executed!
      # NOTE: Binary operation with UnresolvedType and UndeclaredType
      #       raises a runtime error.
      raise TypeError,
        "arithmetic-type of UnresolvedType and UndeclaredType is not defined."
    end

    def _arithmetic_type_with_void(lhs, rhs = self)
      # NOTE: An arithmetic operation with UndeclaredType must not be executed!
      # NOTE: Binary operation with `void' and UndeclaredType
      #       raises a runtime error.
      raise TypeError,
        "arithmetic-type of `void' and UndeclaredType is not defined."
    end

    def _arithmetic_type_with_function(lhs, rhs = self)
      # NOTE: An arithmetic operation with UndeclaredType must not be executed!
      # NOTE: Binary operation with FunctionType and UndeclaredType
      #       raises a runtime error.
      raise TypeError,
        "arithmetic-type of FunctionType and UndeclaredType is not defined."
    end

    def _arithmetic_type_with_char(lhs, rhs = self)
      # NOTE: An arithmetic operation with UndeclaredType must not be executed!
      # NOTE: Binary operation with `char' and UndeclaredType
      #       makes integer-promoted type of `char'.
      lhs.integer_promoted_type # NOTREACHED
    end

    def _arithmetic_type_with_signed_char(lhs, rhs = self)
      # NOTE: An arithmetic operation with UndeclaredType must not be executed!
      # NOTE: Binary operation with `signed char' and UndeclaredType
      #       makes integer-promoted type of `signed char'.
      lhs.integer_promoted_type # NOTREACHED
    end

    def _arithmetic_type_with_unsigned_char(lhs, rhs = self)
      # NOTE: An arithmetic operation with UndeclaredType must not be executed!
      # NOTE: Binary operation with `unsigned char' and UndeclaredType
      #       makes integer-promoted type of `unsigned char'.
      lhs.integer_promoted_type # NOTREACHED
    end

    def _arithmetic_type_with_short(lhs, rhs = self)
      # NOTE: An arithmetic operation with UndeclaredType must not be executed!
      # NOTE: Binary operation with `short' and UndeclaredType
      #       makes integer-promoted type of `short'.
      lhs.integer_promoted_type # NOTREACHED
    end

    def _arithmetic_type_with_signed_short(lhs, rhs = self)
      # NOTE: An arithmetic operation with UndeclaredType must not be executed!
      # NOTE: Binary operation with `signed short' and UndeclaredType
      #       makes integer-promoted type of `signed short'.
      lhs.integer_promoted_type # NOTREACHED
    end

    def _arithmetic_type_with_unsigned_short(lhs, rhs = self)
      # NOTE: An arithmetic operation with UndeclaredType must not be executed!
      # NOTE: Binary operation with `unsigned short' and UndeclaredType
      #       makes integer-promoted type of `unsigned short'.
      lhs.integer_promoted_type # NOTREACHED
    end

    def _arithmetic_type_with_short_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with UndeclaredType must not be executed!
      # NOTE: Binary operation with `short int' and UndeclaredType
      #       makes integer-promoted type of `short int'.
      lhs.integer_promoted_type # NOTREACHED
    end

    def _arithmetic_type_with_signed_short_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with UndeclaredType must not be executed!
      # NOTE: Binary operation with `signed short int' and UndeclaredType
      #       makes integer-promoted type of `signed short int'.
      lhs.integer_promoted_type # NOTREACHED
    end

    def _arithmetic_type_with_unsigned_short_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with UndeclaredType must not be executed!
      # NOTE: Binary operation with `unsigned short int' and UndeclaredType
      #       makes integer-promoted type of `unsigned short int'.
      lhs.integer_promoted_type # NOTREACHED
    end

    def _arithmetic_type_with_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with UndeclaredType must not be executed!
      # NOTE: Binary operation with `int' and UndeclaredType makes `int'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_signed(lhs, rhs = self)
      # NOTE: An arithmetic operation with UndeclaredType must not be executed!
      # NOTE: Binary operation with `signed' and UndeclaredType makes `signed'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_signed_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with UndeclaredType must not be executed!
      # NOTE: Binary operation with `signed int' and UndeclaredType
      #       makes `signed int'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_unsigned(lhs, rhs = self)
      # NOTE: An arithmetic operation with UndeclaredType must not be executed!
      # NOTE: Binary operation with `unsigned' and UndeclaredType
      #       makes `unsigned'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_unsigned_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with UndeclaredType must not be executed!
      # NOTE: Binary operation with `unsigned int' and UndeclaredType
      #       makes `unsigned int'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_long(lhs, rhs = self)
      # NOTE: An arithmetic operation with UndeclaredType must not be executed!
      # NOTE: Binary operation with `long' and UndeclaredType makes `long'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_signed_long(lhs, rhs = self)
      # NOTE: An arithmetic operation with UndeclaredType must not be executed!
      # NOTE: Binary operation with `signed long' and UndeclaredType
      #       makes `signed long'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_unsigned_long(lhs, rhs = self)
      # NOTE: An arithmetic operation with UndeclaredType must not be executed!
      # NOTE: Binary operation with `unsigned long' and UndeclaredType
      #       makes `unsigned long'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_long_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with UndeclaredType must not be executed!
      # NOTE: Binary operation with `long int' and UndeclaredType
      #       makes `long int'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_signed_long_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with UndeclaredType must not be executed!
      # NOTE: Binary operation with `signed long int' and UndeclaredType
      #       makes `signed long int'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_unsigned_long_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with UndeclaredType must not be executed!
      # NOTE: Binary operation with `unsigned long int' and UndeclaredType
      #       makes `unsigned long int'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_long_long(lhs, rhs = self)
      # NOTE: An arithmetic operation with UndeclaredType must not be executed!
      # NOTE: Binary operation with `long long' and UndeclaredType
      #       makes `long long'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_signed_long_long(lhs, rhs = self)
      # NOTE: An arithmetic operation with UndeclaredType must not be executed!
      # NOTE: Binary operation with `signed long long' and UndeclaredType
      #       makes `signed long long'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_unsigned_long_long(lhs, rhs = self)
      # NOTE: An arithmetic operation with UndeclaredType must not be executed!
      # NOTE: Binary operation with `unsigned long long' and UndeclaredType
      #       makes `unsigned long long'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_long_long_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with UndeclaredType must not be executed!
      # NOTE: Binary operation with `long long int' and UndeclaredType
      #       makes `long long int'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_signed_long_long_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with UndeclaredType must not be executed!
      # NOTE: Binary operation with `signed long long int' and UndeclaredType
      #       makes `signed long long int'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_unsigned_long_long_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with UndeclaredType must not be executed!
      # NOTE: Binary operation with `unsigned long long int' and UndeclaredType
      #       makes `unsigned long long int'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_float(lhs, rhs = self)
      # NOTE: An arithmetic operation with UndeclaredType must not be executed!
      # NOTE: Binary operation with `float' and UndeclaredType makes `float'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_double(lhs, rhs = self)
      # NOTE: An arithmetic operation with UndeclaredType must not be executed!
      # NOTE: Binary operation with `double' and UndeclaredType makes `double'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_long_double(lhs, rhs = self)
      # NOTE: An arithmetic operation with UndeclaredType must not be executed!
      # NOTE: Binary operation with `long double' and UndeclaredType
      #       makes `long double'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_bitfield(lhs, rhs = self)
      # NOTE: An arithmetic operation with UndeclaredType must not be executed!
      # NOTE: Binary operation with BitfieldType and UndeclaredType
      #       makes integer-promoted type of BitfieldType.
      lhs.integer_promoted_type # NOTREACHED
    end

    def _arithmetic_type_with_enum(lhs, rhs = self)
      # NOTE: An arithmetic operation with UndeclaredType must not be executed!
      # NOTE: Binary operation with EnumType and UndeclaredType makes EnumType.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_pointer(lhs, rhs = self)
      # NOTE: An arithmetic operation with UndeclaredType must not be executed!
      # NOTE: Binary operation with PointerType and UndeclaredType
      #       makes PointerType.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_array(lhs, rhs = self)
      # NOTE: An arithmetic operation with UndeclaredType must not be executed!
      # NOTE: Binary operation with ArrayType and UndeclaredType
      #       raises a runtime error.
      raise TypeError,
        "arithmetic-type of ArrayType and UndeclaredType is not defined."
    end

    def _arithmetic_type_with_struct(lhs, rhs = self)
      # NOTE: An arithmetic operation with UndeclaredType must not be executed!
      # NOTE: Binary operation with StructType and UndeclaredType
      #       raises a runtime error.
      raise TypeError,
        "arithmetic-type of StructType and UndeclaredType is not defined."
    end

    def _arithmetic_type_with_union(lhs, rhs = self)
      # NOTE: An arithmetic operation with UndeclaredType must not be executed!
      # NOTE: Binary operation with UnionType and UndeclaredType
      #       raises a runtime error.
      raise TypeError,
        "arithmetic-type of UnionType and UndeclaredType is not defined."
    end

    def _arithmetic_type_with_extended_big_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with UndeclaredType must not be executed!
      # NOTE: Binary operation with ExtendedBigIntType and UndeclaredType
      #       makes ExtendedBigIntType.
      lhs # NOTREACHED
    end

    def corresponding_signed_type
      self # NOTREACHED
    end

    def corresponding_unsigned_type
      self # NOTREACHED
    end
  end

  class UnresolvedType < Type
    def initialize(type_table)
      super(type_table, "__adlint__unresolved_type")
    end

    def id
      @id ||= TypeId.new(name)
    end

    def image
      name
    end

    def brief_image
      name
    end

    def location
      nil
    end

    def bit_size
      0
    end

    def bit_alignment
      0
    end

    def real_type
      self
    end

    def base_type
      self
    end

    def unqualify
      self
    end

    def incomplete?
      true
    end

    def compatible?(to_type)
      false
    end

    def coercible?(to_type)
      false
    end

    def convertible?(to_type)
      false
    end

    def same_as?(type)
      false
    end

    def scalar?
      false
    end

    def integer?
      false
    end

    def floating?
      false
    end

    def array?
      false
    end

    def struct?
      false
    end

    def union?
      false
    end

    def pointer?
      false
    end

    def qualified?
      false
    end

    def function?
      false
    end

    def enum?
      false
    end

    def user?
      false
    end

    def void?
      false
    end

    def undeclared?
      false
    end

    def unresolved?
      true
    end

    def const?
      false
    end

    def volatile?
      false
    end

    def restrict?
      false
    end

    def bitfield?
      false
    end

    def signed?
      false
    end

    def explicitly_signed?
      false
    end

    def have_va_list?
      false
    end

    def return_type
      self
    end

    def parameter_types
      []
    end

    def enumerators
      []
    end

    def length
      0
    end

    def impl_length
      0
    end

    def members
      []
    end

    def member_named(name)
      nil
    end

    def min_value
      0
    end

    def max_value
      0
    end

    def nil_value
      ScalarValue.of_nil
    end

    def zero_value
      ScalarValue.of_nil
    end

    def arbitrary_value
      ScalarValue.of_nil
    end

    def undefined_value
      ScalarValue.of_nil
    end

    def parameter_value
      ScalarValue.of_nil
    end

    def return_value
      ScalarValue.of_nil
    end

    def coerce_scalar_value(value)
      ScalarValue.of_nil
    end

    def coerce_array_value(value)
      ScalarValue.of_nil
    end

    def coerce_composite_value(value)
      ScalarValue.of_nil
    end

    def integer_conversion_rank
      0 # NOTREACHED
    end

    def integer_promoted_type
      self # NOTREACHED
    end

    def argument_promoted_type
      self # NOTREACHED
    end

    def arithmetic_type_with(type)
      # NOTE: An arithmetic operation with UnresolvedType must not be executed!
      type._arithmetic_type_with_unresolved(self)
    end

    def _arithmetic_type_with_undeclared(lhs, rhs = self)
      # NOTE: An arithmetic operation with UnresolvedType must not be executed!
      rhs.arithmetic_type_with(lhs)
    end

    def _arithmetic_type_with_unresolved(lhs, rhs = self)
      # NOTE: An arithmetic operation with UnresolvedType must not be executed!
      # NOTE: Binary operation with UnresolvedType and UnresolvedType
      #       raises a runtime error.
      raise TypeError,
        "arithmetic-type of UnresolvedType and UnresolvedType is not defined."
    end

    def _arithmetic_type_with_void(lhs, rhs = self)
      # NOTE: An arithmetic operation with UnresolvedType must not be executed!
      # NOTE: Binary operation with `void' and UnresolvedType
      #       raises a runtime error.
      raise TypeError,
        "arithmetic-type of `void' and UnresolvedType is not defined."
    end

    def _arithmetic_type_with_function(lhs, rhs = self)
      # NOTE: An arithmetic operation with UnresolvedType must not be executed!
      # NOTE: Binary operation with FunctionType and UnresolvedType
      #       raises a runtime error.
      raise TypeError,
        "arithmetic-type of FunctionType and UnresolvedType is not defined."
    end

    def _arithmetic_type_with_char(lhs, rhs = self)
      # NOTE: An arithmetic operation with UnresolvedType must not be executed!
      # NOTE: Binary operation with `char' and UnresolvedType
      #       makes integer-promoted type of `char'.
      lhs.integer_promoted_type # NOTREACHED
    end

    def _arithmetic_type_with_signed_char(lhs, rhs = self)
      # NOTE: An arithmetic operation with UnresolvedType must not be executed!
      # NOTE: Binary operation with `signed char' and UnresolvedType
      #       makes integer-promoted type of `signed char'.
      lhs.integer_promoted_type # NOTREACHED
    end

    def _arithmetic_type_with_unsigned_char(lhs, rhs = self)
      # NOTE: An arithmetic operation with UnresolvedType must not be executed!
      # NOTE: Binary operation with `unsigned char' and UnresolvedType
      #       makes integer-promoted type of `unsigned char'.
      lhs.integer_promoted_type # NOTREACHED
    end

    def _arithmetic_type_with_short(lhs, rhs = self)
      # NOTE: An arithmetic operation with UnresolvedType must not be executed!
      # NOTE: Binary operation with `short' and UnresolvedType
      #       makes integer-promoted type of `short'.
      lhs.integer_promoted_type # NOTREACHED
    end

    def _arithmetic_type_with_signed_short(lhs, rhs = self)
      # NOTE: An arithmetic operation with UnresolvedType must not be executed!
      # NOTE: Binary operation with `signed short' and UnresolvedType
      #       makes integer-promoted type of `signed short'.
      lhs.integer_promoted_type # NOTREACHED
    end

    def _arithmetic_type_with_unsigned_short(lhs, rhs = self)
      # NOTE: An arithmetic operation with UnresolvedType must not be executed!
      # NOTE: Binary operation with `unsigned short' and UnresolvedType
      #       makes integer-promoted type of `unsigned short'.
      lhs.integer_promoted_type # NOTREACHED
    end

    def _arithmetic_type_with_short_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with UnresolvedType must not be executed!
      # NOTE: Binary operation with `short int' and UnresolvedType
      #       makes integer-promoted type of `short int'.
      lhs.integer_promoted_type # NOTREACHED
    end

    def _arithmetic_type_with_signed_short_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with UnresolvedType must not be executed!
      # NOTE: Binary operation with `signed short int' and UnresolvedType
      #       makes integer-promoted type of `signed short int'.
      lhs.integer_promoted_type # NOTREACHED
    end

    def _arithmetic_type_with_unsigned_short_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with UnresolvedType must not be executed!
      # NOTE: Binary operation with `unsigned short int' and UnresolvedType
      #       makes integer-promoted type of `unsigned short int'.
      lhs.integer_promoted_type # NOTREACHED
    end

    def _arithmetic_type_with_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with UnresolvedType must not be executed!
      # NOTE: Binary operation with `int' and UnresolvedType makes `int'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_signed(lhs, rhs = self)
      # NOTE: An arithmetic operation with UnresolvedType must not be executed!
      # NOTE: Binary operation with `signed' and UnresolvedType makes `signed'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_signed_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with UnresolvedType must not be executed!
      # NOTE: Binary operation with `signed int' and UnresolvedType
      #       makes `signed int'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_unsigned(lhs, rhs = self)
      # NOTE: An arithmetic operation with UnresolvedType must not be executed!
      # NOTE: Binary operation with `unsigned' and UnresolvedType
      #       makes `unsigned'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_unsigned_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with UnresolvedType must not be executed!
      # NOTE: Binary operation with `unsigned int' and UnresolvedType
      #       makes `unsigned int'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_long(lhs, rhs = self)
      # NOTE: An arithmetic operation with UnresolvedType must not be executed!
      # NOTE: Binary operation with `long' and UnresolvedType makes `long'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_signed_long(lhs, rhs = self)
      # NOTE: An arithmetic operation with UnresolvedType must not be executed!
      # NOTE: Binary operation with `signed long' and UnresolvedType
      #       makes `signed long'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_unsigned_long(lhs, rhs = self)
      # NOTE: An arithmetic operation with UnresolvedType must not be executed!
      # NOTE: Binary operation with `unsigned long' and UnresolvedType
      #       makes `unsigned long'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_long_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with UnresolvedType must not be executed!
      # NOTE: Binary operation with `long int' and UnresolvedType
      #       makes `long int'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_signed_long_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with UnresolvedType must not be executed!
      # NOTE: Binary operation with `signed long int' and UnresolvedType
      #       makes `signed long int'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_unsigned_long_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with UnresolvedType must not be executed!
      # NOTE: Binary operation with `unsigned long int' and UnresolvedType
      #       makes `unsigned long int'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_long_long(lhs, rhs = self)
      # NOTE: An arithmetic operation with UnresolvedType must not be executed!
      # NOTE: Binary operation with `long long' and UnresolvedType
      #       makes `long long'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_signed_long_long(lhs, rhs = self)
      # NOTE: An arithmetic operation with UnresolvedType must not be executed!
      # NOTE: Binary operation with `signed long long' and UnresolvedType
      #       makes `signed long long'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_unsigned_long_long(lhs, rhs = self)
      # NOTE: An arithmetic operation with UnresolvedType must not be executed!
      # NOTE: Binary operation with `unsigned long long' and UnresolvedType
      #       makes `unsigned long long'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_long_long_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with UnresolvedType must not be executed!
      # NOTE: Binary operation with `long long int' and UnresolvedType
      #       makes `long long int'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_signed_long_long_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with UnresolvedType must not be executed!
      # NOTE: Binary operation with `signed long long int' and UnresolvedType
      #       makes `signed long long int'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_unsigned_long_long_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with UnresolvedType must not be executed!
      # NOTE: Binary operation with `unsigned long long int' and UnresolvedType
      #       makes `unsigned long long int'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_float(lhs, rhs = self)
      # NOTE: An arithmetic operation with UnresolvedType must not be executed!
      # NOTE: Binary operation with `float' and UnresolvedType makes `float'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_double(lhs, rhs = self)
      # NOTE: An arithmetic operation with UnresolvedType must not be executed!
      # NOTE: Binary operation with `double' and UnresolvedType makes `double'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_long_double(lhs, rhs = self)
      # NOTE: An arithmetic operation with UnresolvedType must not be executed!
      # NOTE: Binary operation with `long double' and UnresolvedType
      #       makes `long double'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_bitfield(lhs, rhs = self)
      # NOTE: An arithmetic operation with UnresolvedType must not be executed!
      # NOTE: Binary operation with BitfieldType and UnresolvedType
      #       makes integer-promoted type of BitfieldType.
      lhs.integer_promoted_type # NOTREACHED
    end

    def _arithmetic_type_with_enum(lhs, rhs = self)
      # NOTE: An arithmetic operation with UnresolvedType must not be executed!
      # NOTE: Binary operation with EnumType and UnresolvedType makes EnumType.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_pointer(lhs, rhs = self)
      # NOTE: An arithmetic operation with UnresolvedType must not be executed!
      # NOTE: Binary operation with PointerType and UnresolvedType
      #       makes PointerType.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_array(lhs, rhs = self)
      # NOTE: An arithmetic operation with UnresolvedType must not be executed!
      # NOTE: Binary operation with ArrayType and UnresolvedType
      #       raises a runtime error.
      raise TypeError,
        "arithmetic-type of ArrayType and UnresolvedType is not defined."
    end

    def _arithmetic_type_with_struct(lhs, rhs = self)
      # NOTE: An arithmetic operation with UnresolvedType must not be executed!
      # NOTE: Binary operation with StructType and UnresolvedType
      #       raises a runtime error.
      raise TypeError,
        "arithmetic-type of StructType and UnresolvedType is not defined."
    end

    def _arithmetic_type_with_union(lhs, rhs = self)
      # NOTE: An arithmetic operation with UnresolvedType must not be executed!
      # NOTE: Binary operation with UnionType and UnresolvedType
      #       raises a runtime error.
      raise TypeError,
        "arithmetic-type of UnionType and UnresolvedType is not defined."
    end

    def _arithmetic_type_with_extended_big_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with UnresolvedType must not be executed!
      # NOTE: Binary operation with ExtendedBigIntType and UnresolvedType
      #       makes ExtendedBigIntType.
      lhs # NOTREACHED
    end

    def corresponding_signed_type
      self # NOTREACHED
    end

    def corresponding_unsigned_type
      self # NOTREACHED
    end
  end

  class QualifiedType < Type
    def initialize(type_table, base_type, *cvr_qualifiers)
      super(type_table, create_name(base_type, cvr_qualifiers))

      @base_type = base_type
      @cvr_qualifiers = cvr_qualifiers
    end

    attr_reader :base_type

    def declarations
      @base_type.declarations
    end

    def id
      @id ||= QualifiedTypeId.new(@base_type, @cvr_qualifiers)
    end

    def image
      @image ||= create_image(@base_type, @cvr_qualifiers)
    end

    def brief_image
      @brief_image ||= create_brief_image(@base_type, @cvr_qualifiers)
    end

    def location
      @base_type.location
    end

    def bit_size
      @base_type.bit_size
    end

    def bit_alignment
      @base_type.bit_alignment
    end

    def real_type
      type_table.qualified_type(@base_type.real_type, *@cvr_qualifiers)
    end

    def unqualify
      @base_type.unqualify
    end

    def incomplete?
      @base_type.incomplete?
    end

    def compatible?(to_type)
      @base_type.compatible?(to_type)
    end

    def coercible?(to_type)
      @base_type.coercible?(to_type)
    end

    def more_cv_qualified?(than_type)
      # NOTE: The term `more cv-qualified' means:
      #         const          > none
      #         volatile       > none
      #         const volatile > none
      #         const volatile > const
      #         const volatile > volatile
      if than_type.qualified?
        if than_type.const? && than_type.volatile?
          false
        else
          if self.const? && self.volatile?
            true
          else
            false
          end
        end
      else
        true
      end
    end

    def scalar?
      @base_type.scalar?
    end

    def integer?
      @base_type.integer?
    end

    def floating?
      @base_type.floating?
    end

    def array?
      @base_type.array?
    end

    def struct?
      @base_type.struct?
    end

    def union?
      @base_type.union?
    end

    def pointer?
      @base_type.pointer?
    end

    def qualified?
      true
    end

    def function?
      @base_type.function?
    end

    def enum?
      @base_type.enum?
    end

    def user?
      @base_type.user?
    end

    def void?
      @base_type.void?
    end

    def undeclared?
      @base_type.undeclared?
    end

    def unresolved?
      @base_type.unresolved?
    end

    def const?
      @cvr_qualifiers.include?(:const)
    end

    def volatile?
      @cvr_qualifiers.include?(:volatile)
    end

    def restrict?
      @cvr_qualifiers.include?(:restrict)
    end

    def bitfield?
      @base_type.bitfield?
    end

    def signed?
      @base_type.signed?
    end

    def explicitly_signed?
      @base_type.explicitly_signed?
    end

    def have_va_list?
      @base_type.have_va_list?
    end

    def return_type
      @base_type.return_type
    end

    def parameter_types
      @base_type.parameter_types
    end

    def enumerators
      @base_type.enumerators
    end

    def length
      @base_type.length
    end

    def impl_length
      @base_type.impl_length
    end

    def members
      @base_type.members
    end

    def member_named(name)
      @base_type.member_named(name)
    end

    def min_value
      @base_type.min_value
    end

    def max_value
      @base_type.max_value
    end

    def nil_value
      @base_type.nil_value
    end

    def zero_value
      @base_type.zero_value
    end

    def arbitrary_value
      @base_type.arbitrary_value
    end

    def undefined_value
      @base_type.undefined_value
    end

    def parameter_value
      @base_type.parameter_value
    end

    def return_value
      @base_type.return_value
    end

    def coerce_scalar_value(value)
      @base_type.coerce_scalar_value(value)
    end

    def coerce_array_value(value)
      @base_type.coerce_array_value(value)
    end

    def coerce_composite_value(value)
      @base_type.coerce_composite_value(value)
    end

    def integer_conversion_rank
      @base_type.integer_conversion_rank
    end

    def integer_promoted_type
      @base_type.integer_promoted_type
    end

    def argument_promoted_type
      @base_type.argument_promoted_type
    end

    extend Forwardable

    def_delegator :@base_type, :arithmetic_type_with
    def_delegator :@base_type, :_arithmetic_type_with_undeclared
    def_delegator :@base_type, :_arithmetic_type_with_unresolved
    def_delegator :@base_type, :_arithmetic_type_with_void
    def_delegator :@base_type, :_arithmetic_type_with_function
    def_delegator :@base_type, :_arithmetic_type_with_char
    def_delegator :@base_type, :_arithmetic_type_with_signed_char
    def_delegator :@base_type, :_arithmetic_type_with_unsigned_char
    def_delegator :@base_type, :_arithmetic_type_with_short
    def_delegator :@base_type, :_arithmetic_type_with_signed_short
    def_delegator :@base_type, :_arithmetic_type_with_unsigned_short
    def_delegator :@base_type, :_arithmetic_type_with_short_int
    def_delegator :@base_type, :_arithmetic_type_with_signed_short_int
    def_delegator :@base_type, :_arithmetic_type_with_unsigned_short_int
    def_delegator :@base_type, :_arithmetic_type_with_int
    def_delegator :@base_type, :_arithmetic_type_with_signed
    def_delegator :@base_type, :_arithmetic_type_with_signed_int
    def_delegator :@base_type, :_arithmetic_type_with_unsigned
    def_delegator :@base_type, :_arithmetic_type_with_unsigned_int
    def_delegator :@base_type, :_arithmetic_type_with_long
    def_delegator :@base_type, :_arithmetic_type_with_signed_long
    def_delegator :@base_type, :_arithmetic_type_with_unsigned_long
    def_delegator :@base_type, :_arithmetic_type_with_long_int
    def_delegator :@base_type, :_arithmetic_type_with_signed_long_int
    def_delegator :@base_type, :_arithmetic_type_with_unsigned_long_int
    def_delegator :@base_type, :_arithmetic_type_with_long_long
    def_delegator :@base_type, :_arithmetic_type_with_signed_long_long
    def_delegator :@base_type, :_arithmetic_type_with_unsigned_long_long
    def_delegator :@base_type, :_arithmetic_type_with_long_long_int
    def_delegator :@base_type, :_arithmetic_type_with_signed_long_long_int
    def_delegator :@base_type, :_arithmetic_type_with_unsigned_long_long_int
    def_delegator :@base_type, :_arithmetic_type_with_float
    def_delegator :@base_type, :_arithmetic_type_with_double
    def_delegator :@base_type, :_arithmetic_type_with_long_double
    def_delegator :@base_type, :_arithmetic_type_with_bitfield
    def_delegator :@base_type, :_arithmetic_type_with_enum
    def_delegator :@base_type, :_arithmetic_type_with_pointer
    def_delegator :@base_type, :_arithmetic_type_with_array
    def_delegator :@base_type, :_arithmetic_type_with_struct
    def_delegator :@base_type, :_arithmetic_type_with_union
    def_delegator :@base_type, :_arithmetic_type_with_extended_big_int

    def_delegator :@base_type, :corresponding_signed_type
    def_delegator :@base_type, :corresponding_unsigned_type

    private
    def create_name(base_type, cvr_qualifiers)
      append_cvr_qualifiers(base_type.name, cvr_qualifiers)
    end

    def create_image(base_type, cvr_qualifiers)
      append_cvr_qualifiers(base_type.image, cvr_qualifiers)
    end

    def create_brief_image(base_type, cvr_qualifiers)
      append_cvr_qualifiers(base_type.brief_image, cvr_qualifiers)
    end

    def append_cvr_qualifiers(type_str, cvr_qualifiers)
      result = type_str
      result = "#{result} const" if cvr_qualifiers.include?(:const)
      result = "#{result} volatile" if cvr_qualifiers.include?(:volatile)
      result = "#{result} restrict" if cvr_qualifiers.include?(:restrict)
      result
    end
  end

  class QualifiedTypeId < TypeId
    def initialize(base_type, cvr_qualifiers)
      super(create_value(base_type, cvr_qualifiers))
    end

    private
    def create_value(base_type, cvr_qualifiers)
      value = base_type.real_type.brief_image
      value = "#{value} const" if cvr_qualifiers.include?(:const)
      value = "#{value} volatile" if cvr_qualifiers.include?(:volatile)
      value = "#{value} restrict" if cvr_qualifiers.include?(:restrict)
      value
    end
  end

  class VoidType < Type
    def initialize(type_table)
      super(type_table, "void")
    end

    def id
      @id ||= StandardTypeId.new("void")
    end

    def image
      name
    end

    def brief_image
      name
    end

    def location
      nil
    end

    def bit_size
      0
    end

    def bit_alignment
      0
    end

    def real_type
      self
    end

    def base_type
      nil
    end

    def unqualify
      self
    end

    def incomplete?
      true
    end

    def compatible?(to_type)
      false
    end

    def coercible?(to_type)
      false
    end

    def convertible?(to_type)
      to_type.void?
    end

    def same_as?(type)
      false
    end

    def scalar?
      false
    end

    def integer?
      false
    end

    def floating?
      false
    end

    def array?
      false
    end

    def struct?
      false
    end

    def union?
      false
    end

    def pointer?
      false
    end

    def qualified?
      false
    end

    def function?
      false
    end

    def enum?
      false
    end

    def user?
      false
    end

    def void?
      true
    end

    def undeclared?
      false
    end

    def unresolved?
      false
    end

    def const?
      false
    end

    def volatile?
      false
    end

    def restrict?
      false
    end

    def bitfield?
      false
    end

    def signed?
      false
    end

    def explicitly_signed?
      false
    end

    def have_va_list?
      false
    end

    def return_type
      self
    end

    def parameter_types
      []
    end

    def enumerators
      []
    end

    def length
      0
    end

    def impl_length
      0
    end

    def members
      []
    end

    def member_named(name)
      nil
    end

    def min_value
      0
    end

    def max_value
      0
    end

    def nil_value
      ScalarValue.of_nil # NOTREACHED
    end

    def zero_value
      ScalarValue.of_nil # NOTREACHED
    end

    def arbitrary_value
      ScalarValue.of_nil # NOTREACHED
    end

    def undefined_value
      ScalarValue.of_nil # NOTREACHED
    end

    def parameter_value
      ScalarValue.of_nil # NOTREACHED
    end

    def return_value
      ScalarValue.of_nil # NOTREACHED
    end

    def coerce_scalar_value(value)
      ScalarValue.of_nil # NOTREACHED
    end

    def coerce_array_value(value)
      ScalarValue.of_nil # NOTREACHED
    end

    def coerce_composite_value(value)
      ScalarValue.of_nil # NOTREACHED
    end

    def integer_conversion_rank
      0 # NOTREACHED
    end

    def integer_promoted_type
      self # NOTREACHED
    end

    def argument_promoted_type
      self # NOTREACHED
    end

    def arithmetic_type_with(type)
      # NOTE: An arithmetic operation with `void' must not be executed!
      type._arithmetic_type_with_void(self)
    end

    def _arithmetic_type_with_undeclared(lhs, rhs = self)
      rhs.arithmetic_type_with(lhs)
    end

    def _arithmetic_type_with_unresolved(lhs, rhs = self)
      rhs.arithmetic_type_with(lhs)
    end

    def _arithmetic_type_with_void(lhs, rhs = self)
      # NOTE: An arithmetic operation with `void' must not be executed!
      # NOTE: Binary operation with `void' and `void' raises a runtime error.
      raise TypeError,
        "arithmetic-type of `void' and `void' is not defined."
    end

    def _arithmetic_type_with_function(lhs, rhs = self)
      # NOTE: An arithmetic operation with `void' must not be executed!
      # NOTE: Binary operation with FunctionType and `void'
      #       raises a runtime error.
      raise TypeError,
        "arithmetic-type of FunctionType and `void' is not defined."
    end

    def _arithmetic_type_with_char(lhs, rhs = self)
      # NOTE: An arithmetic operation with `void' must not be executed!
      # NOTE: Binary operation with `char' and `void'
      #       makes integer-promoted type of `char'.
      lhs.integer_promoted_type # NOTREACHED
    end

    def _arithmetic_type_with_signed_char(lhs, rhs = self)
      # NOTE: An arithmetic operation with `void' must not be executed!
      # NOTE: Binary operation with `signed char' and `void'
      #       makes integer-promoted type of `signed char'.
      lhs.integer_promoted_type # NOTREACHED
    end

    def _arithmetic_type_with_unsigned_char(lhs, rhs = self)
      # NOTE: An arithmetic operation with `void' must not be executed!
      # NOTE: Binary operation with `unsigned char' and `void'
      #       makes integer-promoted type of `unsigned char'.
      lhs.integer_promoted_type # NOTREACHED
    end

    def _arithmetic_type_with_short(lhs, rhs = self)
      # NOTE: An arithmetic operation with `void' must not be executed!
      # NOTE: Binary operation with `short' and `void'
      #       makes integer-promoted type of `short'.
      lhs.integer_promoted_type # NOTREACHED
    end

    def _arithmetic_type_with_signed_short(lhs, rhs = self)
      # NOTE: An arithmetic operation with `void' must not be executed!
      # NOTE: Binary operation with `signed short' and `void'
      #       makes integer-promoted type of `signed short'.
      lhs.integer_promoted_type # NOTREACHED
    end

    def _arithmetic_type_with_unsigned_short(lhs, rhs = self)
      # NOTE: An arithmetic operation with `void' must not be executed!
      # NOTE: Binary operation with `unsigned short' and `void'
      #       makes integer-promoted type of `unsigned short'.
      lhs.integer_promoted_type # NOTREACHED
    end

    def _arithmetic_type_with_short_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with `void' must not be executed!
      # NOTE: Binary operation with `short int' and `void'
      #       makes integer-promoted type of `short int'.
      lhs.integer_promoted_type # NOTREACHED
    end

    def _arithmetic_type_with_signed_short_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with `void' must not be executed!
      # NOTE: Binary operation with `signed short int' and `void'
      #       makes integer-promoted type of `signed short int'.
      lhs.integer_promoted_type # NOTREACHED
    end

    def _arithmetic_type_with_unsigned_short_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with `void' must not be executed!
      # NOTE: Binary operation with `unsigned short int' and `void'
      #       makes integer-promoted type of `unsigned short int'.
      lhs.integer_promoted_type # NOTREACHED
    end

    def _arithmetic_type_with_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with `void' must not be executed!
      # NOTE: Binary operation with `int' and `void' makes `int'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_signed(lhs, rhs = self)
      # NOTE: An arithmetic operation with `void' must not be executed!
      # NOTE: Binary operation with `signed' and `void' makes `signed'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_signed_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with `void' must not be executed!
      # NOTE: Binary operation with `signed int' and `void' makes `signed int'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_unsigned(lhs, rhs = self)
      # NOTE: An arithmetic operation with `void' must not be executed!
      # NOTE: Binary operation with `unsigned' and `void' makes `unsigned'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_unsigned_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with `void' must not be executed!
      # NOTE: Binary operation with `unsigned int' and `void'
      #       makes `unsigned int'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_long(lhs, rhs = self)
      # NOTE: An arithmetic operation with `void' must not be executed!
      # NOTE: Binary operation with `long' and `void' makes `long'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_signed_long(lhs, rhs = self)
      # NOTE: An arithmetic operation with `void' must not be executed!
      # NOTE: Binary operation with `signed long' and `void'
      #       makes `signed long'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_unsigned_long(lhs, rhs = self)
      # NOTE: An arithmetic operation with `void' must not be executed!
      # NOTE: Binary operation with `unsigned long' and `void'
      #       makes `unsigned long'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_long_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with `void' must not be executed!
      # NOTE: Binary operation with `long int' and `void' makes `long int'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_signed_long_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with `void' must not be executed!
      # NOTE: Binary operation with `signed long int' and `void'
      #       makes `signed long int'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_unsigned_long_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with `void' must not be executed!
      # NOTE: Binary operation with `unsigned long int' and `void'
      #       makes `unsigned long int'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_long_long(lhs, rhs = self)
      # NOTE: An arithmetic operation with `void' must not be executed!
      # NOTE: Binary operation with `long long' and `void' makes `long long'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_signed_long_long(lhs, rhs = self)
      # NOTE: An arithmetic operation with `void' must not be executed!
      # NOTE: Binary operation with `signed long long' and `void'
      #       makes `signed long long'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_unsigned_long_long(lhs, rhs = self)
      # NOTE: An arithmetic operation with `void' must not be executed!
      # NOTE: Binary operation with `unsigned long long' and `void'
      #       makes `unsigned long long'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_long_long_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with `void' must not be executed!
      # NOTE: Binary operation with `long long int' and `void'
      #       makes `long long int'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_signed_long_long_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with `void' must not be executed!
      # NOTE: Binary operation with `signed long long int' and `void'
      #       makes `signed long long int'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_unsigned_long_long_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with `void' must not be executed!
      # NOTE: Binary operation with `unsigned long long int' and `void'
      #       makes `unsigned long long int'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_float(lhs, rhs = self)
      # NOTE: An arithmetic operation with `void' must not be executed!
      # NOTE: Binary operation with `float' and `void' makes `float'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_double(lhs, rhs = self)
      # NOTE: An arithmetic operation with `void' must not be executed!
      # NOTE: Binary operation with `double' and `void' makes `double'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_long_double(lhs, rhs = self)
      # NOTE: An arithmetic operation with `void' must not be executed!
      # NOTE: Binary operation with `long double' and `void'
      #       makes `long double'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_bitfield(lhs, rhs = self)
      # NOTE: An arithmetic operation with `void' must not be executed!
      # NOTE: Binary operation with BitfieldType and `void'
      #       makes integer-promoted type of BitfieldType.
      lhs.integer_promoted_type # NOTREACHED
    end

    def _arithmetic_type_with_enum(lhs, rhs = self)
      # NOTE: An arithmetic operation with `void' must not be executed!
      # NOTE: Binary operation with EnumType and `void' makes EnumType.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_pointer(lhs, rhs = self)
      # NOTE: An arithmetic operation with `void' must not be executed!
      # NOTE: Binary operation with PointerType and `void' makes PointerType.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_array(lhs, rhs = self)
      # NOTE: An arithmetic operation with `void' must not be executed!
      # NOTE: Binary operation with ArrayType and `void'
      #       raises a runtime error.
      raise TypeError,
        "arithmetic-type of ArrayType and `void' is not defined."
    end

    def _arithmetic_type_with_struct(lhs, rhs = self)
      # NOTE: An arithmetic operation with `void' must not be executed!
      # NOTE: Binary operation with StructType and `void'
      #       raises a runtime error.
      raise TypeError,
        "arithmetic-type of StructType and `void' is not defined."
    end

    def _arithmetic_type_with_union(lhs, rhs = self)
      # NOTE: An arithmetic operation with `void' must not be executed!
      # NOTE: Binary operation with UnionType and `void'
      #       raises a runtime error.
      raise TypeError,
        "arithmetic-type of UnionType and `void' is not defined."
    end

    def _arithmetic_type_with_extended_big_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with `void' must not be executed!
      # NOTE: Binary operation with ExtendedBigIntType and `void'
      #       makes ExtendedBigIntType.
      lhs # NOTREACHED
    end

    def corresponding_signed_type
      self # NOTREACHED
    end

    def corresponding_unsigned_type
      self # NOTREACHED
    end
  end

  class FunctionType < Type
    def initialize(type_table, return_type, parameter_types,
                   have_va_list = false)
      super(type_table,
            create_name(return_type, parameter_types, have_va_list))

      @return_type = return_type
      @parameter_types = parameter_types
      @have_va_list = have_va_list
    end

    attr_reader :return_type
    attr_reader :parameter_types

    def declarations
      @return_type.declarations +
        @parameter_types.reduce([]) { |decls, type| decls + type.declarations }
    end

    def id
      @id ||= FunctionTypeId.new(@return_type, @parameter_types, @have_va_list)
    end

    def image
      @image ||= create_image(@return_type, @parameter_types, @have_va_list)
    end

    def brief_image
      @brief_image ||=
        create_brief_image(@return_type, @parameter_types, @have_va_list)
    end

    def location
      nil
    end

    def bit_size
      0
    end

    def bit_alignment
      0
    end

    def real_type
      type_table.function_type(@return_type.real_type,
                               @parameter_types.map { |type| type.real_type },
                               @have_va_list)
    end

    def base_type
      nil
    end

    def unqualify
      self
    end

    def incomplete?
      @return_type.incomplete? || @parameter_types.empty? ||
        @parameter_types.any? { |type| type.incomplete? }
    end

    def compatible?(to_type)
      return false unless to_type.function?

      lhs_params = @parameter_types
      rhs_params = to_type.parameter_types

      @return_type.compatible?(to_type.return_type) &&
        lhs_params.size == rhs_params.size &&
        lhs_params.zip(rhs_params).all? { |lhs, rhs| lhs.compatible?(rhs) } &&
        @have_va_list == to_type.have_va_list?
    end

    def coercible?(to_type)
      false
    end

    def scalar?
      false
    end

    def integer?
      false
    end

    def floating?
      false
    end

    def array?
      false
    end

    def struct?
      false
    end

    def union?
      false
    end

    def pointer?
      false
    end

    def qualified?
      false
    end

    def function?
      true
    end

    def enum?
      false
    end

    def user?
      false
    end

    def void?
      false
    end

    def undeclared?
      false
    end

    def unresolved?
      @return_type.unresolved? ||
        @parameter_types.any? { |type| type.unresolved? }
    end

    def const?
      false
    end

    def volatile?
      false
    end

    def restrict?
      false
    end

    def bitfield?
      false
    end

    def signed?
      false
    end

    def explicitly_signed?
      false
    end

    def have_va_list?
      @have_va_list
    end

    def enumerators
      []
    end

    def length
      0
    end

    def impl_length
      0
    end

    def members
      []
    end

    def member_named(name)
      nil
    end

    def min_value
      0
    end

    def max_value
      0
    end

    def nil_value
      ScalarValue.of_nil # NOTREACHED
    end

    def zero_value
      ScalarValue.of_nil # NOTREACHED
    end

    def arbitrary_value
      ScalarValue.of_nil # NOTREACHED
    end

    def undefined_value
      ScalarValue.of_nil # NOTREACHED
    end

    def parameter_value
      ScalarValue.of_nil # NOTREACHED
    end

    def return_value
      ScalarValue.of_nil # NOTREACHED
    end

    def coerce_scalar_value(value)
      ScalarValue.of_nil # NOTREACHED
    end

    def coerce_array_value(value)
      ScalarValue.of_nil # NOTREACHED
    end

    def coerce_composite_value(value)
      ScalarValue.of_nil # NOTREACHED
    end

    def integer_conversion_rank
      0 # NOTREACHED
    end

    def integer_promoted_type
      self # NOTREACHED
    end

    def argument_promoted_type
      self # NOTREACHED
    end

    def arithmetic_type_with(type)
      # NOTE: An arithmetic operation with FunctionType must not be executed!
      type._arithmetic_type_with_function(self)
    end

    def _arithmetic_type_with_undeclared(lhs, rhs = self)
      rhs.arithmetic_type_with(lhs)
    end

    def _arithmetic_type_with_unresolved(lhs, rhs = self)
      rhs.arithmetic_type_with(lhs)
    end

    def _arithmetic_type_with_void(lhs, rhs = self)
      rhs.arithmetic_type_with(lhs)
    end

    def _arithmetic_type_with_function(lhs, rhs = self)
      # NOTE: An arithmetic operation with FunctionType must not be executed!
      # NOTE: Binary operation with FunctionType and FunctionType
      #       raises a runtime error.
      raise TypeError,
        "arithmetic-type of FunctionType and FunctionType is not defined."
    end

    def _arithmetic_type_with_char(lhs, rhs = self)
      # NOTE: An arithmetic operation with FunctionType must not be executed!
      # NOTE: Binary operation with `char' and FunctionType
      #       makes integer-promoted type of `char'.
      lhs.integer_promoted_type # NOTREACHED
    end

    def _arithmetic_type_with_signed_char(lhs, rhs = self)
      # NOTE: An arithmetic operation with FunctionType must not be executed!
      # NOTE: Binary operation with `signed char' and FunctionType
      #       makes integer-promoted type of `signed char'.
      lhs.integer_promoted_type # NOTREACHED
    end

    def _arithmetic_type_with_unsigned_char(lhs, rhs = self)
      # NOTE: An arithmetic operation with FunctionType must not be executed!
      # NOTE: Binary operation with `unsigned char' and FunctionType
      #       makes integer-promoted type of `unsigned char'.
      lhs.integer_promoted_type # NOTREACHED
    end

    def _arithmetic_type_with_short(lhs, rhs = self)
      # NOTE: An arithmetic operation with FunctionType must not be executed!
      # NOTE: Binary operation with `short' and FunctionType
      #       makes integer-promoted type of `short'.
      lhs.integer_promoted_type # NOTREACHED
    end

    def _arithmetic_type_with_signed_short(lhs, rhs = self)
      # NOTE: An arithmetic operation with FunctionType must not be executed!
      # NOTE: Binary operation with `signed short' and FunctionType
      #       makes integer-promoted type of `signed short'.
      lhs.integer_promoted_type # NOTREACHED
    end

    def _arithmetic_type_with_unsigned_short(lhs, rhs = self)
      # NOTE: An arithmetic operation with FunctionType must not be executed!
      # NOTE: Binary operation with `unsigned short' and FunctionType
      #       makes integer-promoted type of `unsigned short'.
      lhs.integer_promoted_type # NOTREACHED
    end

    def _arithmetic_type_with_short_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with FunctionType must not be executed!
      # NOTE: Binary operation with `short int' and FunctionType
      #       makes integer-promoted type of `short int'.
      lhs.integer_promoted_type # NOTREACHED
    end

    def _arithmetic_type_with_signed_short_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with FunctionType must not be executed!
      # NOTE: Binary operation with `signed short int' and FunctionType
      #       makes integer-promoted type of `signed short int'.
      lhs.integer_promoted_type # NOTREACHED
    end

    def _arithmetic_type_with_unsigned_short_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with FunctionType must not be executed!
      # NOTE: Binary operation with `unsigned short int' and FunctionType
      #       makes integer-promoted type of `unsigned short int'.
      lhs.integer_promoted_type # NOTREACHED
    end

    def _arithmetic_type_with_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with FunctionType must not be executed!
      # NOTE: Binary operation with `int' and FunctionType makes `int'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_signed(lhs, rhs = self)
      # NOTE: An arithmetic operation with FunctionType must not be executed!
      # NOTE: Binary operation with `signed' and FunctionType makes `signed'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_signed_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with FunctionType must not be executed!
      # NOTE: Binary operation with `signed int' and FunctionType
      #       makes `signed int'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_unsigned(lhs, rhs = self)
      # NOTE: An arithmetic operation with FunctionType must not be executed!
      # NOTE: Binary operation with `unsigned' and FunctionType
      #       makes `unsigned'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_unsigned_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with FunctionType must not be executed!
      # NOTE: Binary operation with `unsigned int' and FunctionType
      #       makes `unsigned int'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_long(lhs, rhs = self)
      # NOTE: An arithmetic operation with FunctionType must not be executed!
      # NOTE: Binary operation with `long' and FunctionType makes `long'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_signed_long(lhs, rhs = self)
      # NOTE: An arithmetic operation with FunctionType must not be executed!
      # NOTE: Binary operation with `signed long' and FunctionType
      #       makes `signed long'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_unsigned_long(lhs, rhs = self)
      # NOTE: An arithmetic operation with FunctionType must not be executed!
      # NOTE: Binary operation with `unsigned long' and FunctionType
      #       makes `unsigned long'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_long_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with FunctionType must not be executed!
      # NOTE: Binary operation with `long int' and FunctionType
      #       makes `long int'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_signed_long_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with FunctionType must not be executed!
      # NOTE: Binary operation with `signed long int' and FunctionType
      #       makes `signed long int'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_unsigned_long_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with FunctionType must not be executed!
      # NOTE: Binary operation with `unsigned long int' and FunctionType
      #       makes `unsigned long int'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_long_long(lhs, rhs = self)
      # NOTE: An arithmetic operation with FunctionType must not be executed!
      # NOTE: Binary operation with `long long' and FunctionType
      #       makes `long long'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_signed_long_long(lhs, rhs = self)
      # NOTE: An arithmetic operation with FunctionType must not be executed!
      # NOTE: Binary operation with `signed long long' and FunctionType
      #       makes `signed long long'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_unsigned_long_long(lhs, rhs = self)
      # NOTE: An arithmetic operation with FunctionType must not be executed!
      # NOTE: Binary operation with `unsigned long long' and FunctionType
      #       makes `unsigned long long'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_long_long_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with FunctionType must not be executed!
      # NOTE: Binary operation with `long long int' and FunctionType
      #       makes `long long int'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_signed_long_long_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with FunctionType must not be executed!
      # NOTE: Binary operation with `signed long long int' and FunctionType
      #       makes `signed long long int'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_unsigned_long_long_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with FunctionType must not be executed!
      # NOTE: Binary operation with `unsigned long long int' and FunctionType
      #       makes `unsigned long long int'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_float(lhs, rhs = self)
      # NOTE: An arithmetic operation with FunctionType must not be executed!
      # NOTE: Binary operation with `float' and FunctionType makes `float'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_double(lhs, rhs = self)
      # NOTE: An arithmetic operation with FunctionType must not be executed!
      # NOTE: Binary operation with `double' and FunctionType makes `double'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_long_double(lhs, rhs = self)
      # NOTE: An arithmetic operation with FunctionType must not be executed!
      # NOTE: Binary operation with `long double' and FunctionType
      #       makes `long double'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_bitfield(lhs, rhs = self)
      # NOTE: An arithmetic operation with FunctionType must not be executed!
      # NOTE: Binary operation with BitfieldType and FunctionType
      #       makes integer-promoted type of BitfieldType.
      lhs.integer_promoted_type # NOTREACHED
    end

    def _arithmetic_type_with_enum(lhs, rhs = self)
      # NOTE: An arithmetic operation with FunctionType must not be executed!
      # NOTE: Binary operation with EnumType and FunctionType makes EnumType.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_pointer(lhs, rhs = self)
      # NOTE: An arithmetic operation with FunctionType must not be executed!
      # NOTE: Binary operation with PointerType and FunctionType
      #       makes PointerType.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_array(lhs, rhs = self)
      # NOTE: An arithmetic operation with FunctionType must not be executed!
      # NOTE: Binary operation with ArrayType and FunctionType
      #       raises a runtime error.
      raise TypeError,
        "arithmetic-type of ArrayType and FunctionType is not defined."
    end

    def _arithmetic_type_with_struct(lhs, rhs = self)
      # NOTE: An arithmetic operation with FunctionType must not be executed!
      # NOTE: Binary operation with StructType and FunctionType
      #       raises a runtime error.
      raise TypeError,
        "arithmetic-type of StructType and FunctionType is not defined."
    end

    def _arithmetic_type_with_union(lhs, rhs = self)
      # NOTE: An arithmetic operation with FunctionType must not be executed!
      # NOTE: Binary operation with UnionType and FunctionType
      #       raises a runtime error.
      raise TypeError,
        "arithmetic-type of UnionType and FunctionType is not defined."
    end

    def _arithmetic_type_with_extended_big_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with FunctionType must not be executed!
      # NOTE: Binary operation with ExtendedBigIntType and FunctionType
      #       makes ExtendedBigIntType.
      lhs # NOTREACHED
    end

    def corresponding_signed_type
      self # NOTREACHED
    end

    def corresponding_unsigned_type
      self # NOTREACHED
    end

    def ==(rhs)
      case rhs
      when FunctionType
        if parameter_types.empty? || rhs.parameter_types.empty?
          return_type == rhs.return_type
        else
          return_type == rhs.return_type &&
            parameter_types == rhs.parameter_types &&
            have_va_list? == rhs.have_va_list?
        end
      else
        false
      end
    end

    private
    def create_name(return_type, parameter_types, have_va_list)
      "#{return_type.name}(" +
        parameter_types.map { |type| type.name }.join(", ") +
        (have_va_list ? ",...)" : ")")
    end

    def create_image(return_type, parameter_types, have_va_list)
      "#{return_type.image}(" +
        parameter_types.map { |type| type.image }.join(", ") +
        (have_va_list ? ",...)" : ")")
    end

    def create_brief_image(return_type, parameter_types, have_va_list)
      "#{return_type.brief_image}(" +
        parameter_types.map { |type| type.brief_image }.join(", ") +
        (have_va_list ? ",...)" : ")")
    end
  end

  class FunctionTypeId < TypeId
    def initialize(return_type, parameter_types, have_va_list)
      super(create_value(return_type, parameter_types, have_va_list))
    end

    private
    def create_value(return_type, parameter_types, have_va_list)
      "#{return_type.brief_image}(" +
        parameter_types.map { |type| type.brief_image }.join(",") +
        (have_va_list ? ",...)" : ")")
    end
  end

  class ScalarDataType < Type
    include UsualArithmeticTypeConversion

    def initialize(type_table, name, bit_size, bit_alignment,
                   type_declarations = [])
      super(type_table, name, type_declarations)

      @bit_size = bit_size
      @bit_alignment = bit_alignment
    end

    attr_reader :bit_size
    attr_reader :bit_alignment

    def id
      subclass_responsibility
    end

    def image
      subclass_responsibility
    end

    def brief_image
      subclass_responsibility
    end

    def location
      subclass_responsibility
    end

    def real_type
      self
    end

    def base_type
      nil
    end

    def unqualify
      self
    end

    def compatible?(to_type)
      subclass_responsibility
    end

    def coercible?(to_type)
      to_type.scalar?
    end

    def scalar?
      true
    end

    def integer?
      subclass_responsibility
    end

    def floating?
      subclass_responsibility
    end

    def array?
      false
    end

    def struct?
      false
    end

    def union?
      false
    end

    def pointer?
      subclass_responsibility
    end

    def qualified?
      false
    end

    def function?
      false
    end

    def enum?
      subclass_responsibility
    end

    def user?
      false
    end

    def void?
      false
    end

    def undeclared?
      false
    end

    def unresolved?
      false
    end

    def const?
      false
    end

    def volatile?
      false
    end

    def restrict?
      false
    end

    def bitfield?
      subclass_responsibility
    end

    def signed?
      subclass_responsibility
    end

    def explicitly_signed?
      subclass_responsibility
    end

    def have_va_list?
      false
    end

    def return_type
      nil
    end

    def parameter_types
      []
    end

    def enumerators
      subclass_responsibility
    end

    def length
      0
    end

    def impl_length
      0
    end

    def members
      []
    end

    def member_named(name)
      nil
    end

    def min_value
      subclass_responsibility
    end

    def max_value
      subclass_responsibility
    end

    def nil_value
      ScalarValue.of_nil
    end

    def zero_value
      ScalarValue.of(0)
    end

    def arbitrary_value
      ScalarValue.of_arbitrary
    end

    def undefined_value
      ScalarValue.of_undefined(min_value..max_value)
    end

    def parameter_value
      ScalarValue.of(min_value..max_value)
    end

    def return_value
      ScalarValue.of(min_value..max_value)
    end

    def coerce_scalar_value(value)
      value.dup.tap do |val|
        val.narrow_domain!(:==, ScalarValue.of(min_value..max_value))
      end
    end

    def coerce_array_value(value)
      first_value = value.values.first
      until first_value && first_value.scalar?
        first_value = first_value.values.first
      end

      if first_value && first_value.scalar?
        coerce_scalar_value(first_value)
      else
        undefined_value
      end
    end

    def coerce_composite_value(value)
      first_value = value.values.first
      until first_value && first_value.scalar?
        first_value = first_value.values.first
      end

      if first_value && first_value.scalar?
        coerce_scalar_value(first_value)
      else
        undefined_value
      end
    end

    def integer_conversion_rank
      subclass_responsibility
    end

    def integer_promoted_type
      subclass_responsibility
    end

    def argument_promoted_type
      subclass_responsibility
    end

    def arithmetic_type_with(type)
      subclass_responsibility
    end

    def _arithmetic_type_with_undeclared(lhs, rhs = self)
      rhs.arithmetic_type_with(lhs)
    end

    def _arithmetic_type_with_unresolved(lhs, rhs = self)
      rhs.arithmetic_type_with(lhs)
    end

    def _arithmetic_type_with_void(lhs, rhs = self)
      rhs.arithmetic_type_with(lhs)
    end

    def _arithmetic_type_with_function(lhs, rhs = self)
      rhs.arithmetic_type_with(lhs)
    end

    def _arithmetic_type_with_char(lhs, rhs = self)
      do_usual_arithmetic_type_conversion(lhs, rhs)
    end

    def _arithmetic_type_with_signed_char(lhs, rhs = self)
      do_usual_arithmetic_type_conversion(lhs, rhs)
    end

    def _arithmetic_type_with_unsigned_char(lhs, rhs = self)
      do_usual_arithmetic_type_conversion(lhs, rhs)
    end

    def _arithmetic_type_with_short(lhs, rhs = self)
      do_usual_arithmetic_type_conversion(lhs, rhs)
    end

    def _arithmetic_type_with_signed_short(lhs, rhs = self)
      do_usual_arithmetic_type_conversion(lhs, rhs)
    end

    def _arithmetic_type_with_unsigned_short(lhs, rhs = self)
      do_usual_arithmetic_type_conversion(lhs, rhs)
    end

    def _arithmetic_type_with_short_int(lhs, rhs = self)
      do_usual_arithmetic_type_conversion(lhs, rhs)
    end

    def _arithmetic_type_with_signed_short_int(lhs, rhs = self)
      do_usual_arithmetic_type_conversion(lhs, rhs)
    end

    def _arithmetic_type_with_unsigned_short_int(lhs, rhs = self)
      do_usual_arithmetic_type_conversion(lhs, rhs)
    end

    def _arithmetic_type_with_int(lhs, rhs = self)
      do_usual_arithmetic_type_conversion(lhs, rhs)
    end

    def _arithmetic_type_with_signed(lhs, rhs = self)
      do_usual_arithmetic_type_conversion(lhs, rhs)
    end

    def _arithmetic_type_with_signed_int(lhs, rhs = self)
      do_usual_arithmetic_type_conversion(lhs, rhs)
    end

    def _arithmetic_type_with_unsigned(lhs, rhs = self)
      do_usual_arithmetic_type_conversion(lhs, rhs)
    end

    def _arithmetic_type_with_unsigned_int(lhs, rhs = self)
      do_usual_arithmetic_type_conversion(lhs, rhs)
    end

    def _arithmetic_type_with_long(lhs, rhs = self)
      do_usual_arithmetic_type_conversion(lhs, rhs)
    end

    def _arithmetic_type_with_signed_long(lhs, rhs = self)
      do_usual_arithmetic_type_conversion(lhs, rhs)
    end

    def _arithmetic_type_with_unsigned_long(lhs, rhs = self)
      do_usual_arithmetic_type_conversion(lhs, rhs)
    end

    def _arithmetic_type_with_long_int(lhs, rhs = self)
      do_usual_arithmetic_type_conversion(lhs, rhs)
    end

    def _arithmetic_type_with_signed_long_int(lhs, rhs = self)
      do_usual_arithmetic_type_conversion(lhs, rhs)
    end

    def _arithmetic_type_with_unsigned_long_int(lhs, rhs = self)
      do_usual_arithmetic_type_conversion(lhs, rhs)
    end

    def _arithmetic_type_with_long_long(lhs, rhs = self)
      do_usual_arithmetic_type_conversion(lhs, rhs)
    end

    def _arithmetic_type_with_signed_long_long(lhs, rhs = self)
      do_usual_arithmetic_type_conversion(lhs, rhs)
    end

    def _arithmetic_type_with_unsigned_long_long(lhs, rhs = self)
      do_usual_arithmetic_type_conversion(lhs, rhs)
    end

    def _arithmetic_type_with_long_long_int(lhs, rhs = self)
      do_usual_arithmetic_type_conversion(lhs, rhs)
    end

    def _arithmetic_type_with_signed_long_long_int(lhs, rhs = self)
      do_usual_arithmetic_type_conversion(lhs, rhs)
    end

    def _arithmetic_type_with_unsigned_long_long_int(lhs, rhs = self)
      do_usual_arithmetic_type_conversion(lhs, rhs)
    end

    def _arithmetic_type_with_float(lhs, rhs = self)
      do_usual_arithmetic_type_conversion(lhs, rhs)
    end

    def _arithmetic_type_with_double(lhs, rhs = self)
      do_usual_arithmetic_type_conversion(lhs, rhs)
    end

    def _arithmetic_type_with_long_double(lhs, rhs = self)
      do_usual_arithmetic_type_conversion(lhs, rhs)
    end

    def _arithmetic_type_with_bitfield(lhs, rhs = self)
      do_usual_arithmetic_type_conversion(lhs, rhs)
    end

    def _arithmetic_type_with_enum(lhs, rhs = self)
      do_usual_arithmetic_type_conversion(lhs, rhs)
    end

    def _arithmetic_type_with_pointer(lhs, rhs = self)
      do_usual_arithmetic_type_conversion(lhs, rhs)
    end

    def _arithmetic_type_with_array(lhs, rhs = self)
      rhs.arithmetic_type_with(lhs)
    end

    def _arithmetic_type_with_struct(lhs, rhs = self)
      rhs.arithmetic_type_with(lhs)
    end

    def _arithmetic_type_with_union(lhs, rhs = self)
      rhs.arithmetic_type_with(lhs)
    end

    def _arithmetic_type_with_extended_big_int(lhs, rhs = self)
      # NOTE: Binary operation with ExtendedBigIntType and any scalar type
      #       makes ExtendedBigIntType.
      lhs
    end

    def corresponding_signed_type
      subclass_responsibility
    end

    def corresponding_unsigned_type
      subclass_responsibility
    end
  end

  class IntegerType < ScalarDataType
    def initialize(type_table, name, bit_size, bit_alignment,
                   signed, explicitly_signed, type_declarations = [])
      super(type_table, name, bit_size, bit_alignment, type_declarations)

      @signed = signed
      @explicitly_signed = explicitly_signed
    end

    def id
      subclass_responsibility
    end

    def image
      name
    end

    def brief_image
      name
    end

    def location
      nil
    end

    def compatible?(to_type)
      to_type.integer? &&
        to_type.min_value <= self.min_value &&
        self.max_value <= to_type.max_value
    end

    def integer?
      true
    end

    def floating?
      false
    end

    def pointer?
      subclass_responsibility
    end

    def enum?
      subclass_responsibility
    end

    def bitfield?
      subclass_responsibility
    end

    def signed?
      @signed
    end

    def explicitly_signed?
      @explicitly_signed
    end

    def enumerators
      subclass_responsibility
    end

    def min_value
      if @signed
        -2**(@bit_size - 1)
      else
        0
      end
    end

    def max_value
      if @signed
        2**(@bit_size - 1) - 1
      else
        2**@bit_size - 1
      end
    end

    def integer_conversion_rank
      subclass_responsibility
    end

    def integer_promoted_type
      # NOTE: The ISO C99 standard saids;
      #
      # 6.3.1 Arithmetic operands
      # 6.3.1.1 Boolean, characters, and integers
      #
      # 2 The following may be used in an expression wherever an int or
      #   unsigned int may be used:
      #
      #     -- An object or expression with an integer type whose integer
      #        conversion rank is less than or equal to the rank of int and
      #        unsigned int.
      #     -- A bit-field of type _Bool, int, signed int, or unsigned int.
      #
      #   If an int can represent all values of the original type, the value is
      #   converted to an int; otherwise, it is converted to an unsigned int.
      #   These are called the integer promotions.  All other types are
      #   unchanged by the integer promotions.
      if self.integer_conversion_rank <= int_type.integer_conversion_rank
        self.compatible?(int_type) ? int_type : unsigned_int_type
      else
        self
      end
    end

    def argument_promoted_type
      # NOTE: The ISO C99 standard saids;
      #
      # 6.5.2.2 Function calls
      #
      # 6 If the expression that denotes the called function has a type that
      #   does not include a prototype, the integer promotions are performed on
      #   each argument, and arguments that have type float are promoted to
      #   double.  These are called the default argument promotions.  If the
      #   number of arguments does not equal the number of parameters, the
      #   behavior is undefined.  If the function is defined with a type that
      #   includes a prototype, and either the prototype ends with an ellipsis
      #   (, ...) or the types of the arguments after promotion are not
      #   compatible with the types of the parameters, the behavior is
      #   undefined.  If the function is defined with a type that does not
      #   include a prototype, and the types of the arguments after promotion
      #   are not compatible with those of the parameters after promotion, the
      #   behavior is undefined, except for the following cases:
      #
      #     -- one promoted type is a signed integer type, the other promoted
      #        type is the corresponding unsigned integer type, and the value
      #        is representable in both types;
      #     -- both types are pointers to qualified or unqualified versions of
      #        a character type or void.
      self.integer_promoted_type
    end

    def arithmetic_type_with(type)
      subclass_responsibility
    end

    def corresponding_signed_type
      subclass_responsibility
    end

    def corresponding_unsigned_type
      subclass_responsibility
    end
  end

  class StandardIntegerType < IntegerType
    def id
      subclass_responsibility
    end

    def incomplete?
      false
    end

    def pointer?
      false
    end

    def enum?
      false
    end

    def bitfield?
      false
    end

    def enumerators
      []
    end

    def integer_conversion_rank
      subclass_responsibility
    end

    def arithmetic_type_with(type)
      subclass_responsibility
    end

    def corresponding_signed_type
      subclass_responsibility
    end

    def corresponding_unsigned_type
      subclass_responsibility
    end
  end

  class CharType < StandardIntegerType
    include StandardTypeAccessor

    def initialize(type_table)
      super(type_table,
            "char", char_size, char_alignment, !char_as_unsigned_char?, false)
    end

    def id
      # NOTE: `char' type may be treated as `unsigned char'.
      #       Specialized type comparison is implemented in CharTypeId,
      #       SignedCharTypeId and UnsignedCharTypeId.
      @id ||= CharTypeId.new
    end

    def integer_conversion_rank
      # NOTE: The ISO C99 standard saids;
      #
      # 6.3.1 Arithmetic operands
      # 6.3.1.1 Boolean, characters, and integers
      #
      # 1 Every integer type has an integer conversion rank defined as follows:
      #
      #     -- No two signed integer types shall have the same rank, even if
      #        they have the same representation.
      #     -- The rank of a signed integer type shall be greater than the rank
      #        of any signed integer type with less precision.
      #     -- The rank of long long int shall be greater than the rank of long
      #        int, which shall be greater than the rank of int, which shall be
      #        greater than the rank of short int, which shall be greater than
      #        the rank of signed char.
      #     -- The rank of any unsigned integer type shall equal the rank of
      #        the corresponding signed integer type, if any.
      #     -- The rank of any standard integer type shall be greater than the
      #        rank of any extended integer type with the same width.
      #     -- The rank of char shall equal the rank of signed char and
      #        unsigned char.
      #     -- The rank of _Bool shall be less than the rank of all other
      #        standard integer types.
      #     -- The rank of any enumerated type shall equal the rank of the
      #        compatible integer type.
      #     -- The rank of any extended signed integer type relative to another
      #        extended signed integer type with the same precision is
      #        implementation-defined, but still subject to the other rules for
      #        determining the integer conversion rank.
      #     -- For all integer types T1, T2, and T3, if T1 has greater rank
      #        than T2 and T2 has greater rank than T3, then T1 has greater
      #        rank than T3.
      #
      # NOTE: char = 1, short int = 2, int = 3, long int = 4, long long int = 5
      1
    end

    def arithmetic_type_with(type)
      type._arithmetic_type_with_char(self)
    end

    def corresponding_signed_type
      signed_char_type
    end

    def corresponding_unsigned_type
      unsigned_char_type
    end
  end

  class CharTypeId < StandardTypeId
    include StandardTypeAccessor

    def initialize
      super("char")
    end

    def ==(rhs)
      if char_as_unsigned_char?
        case rhs
        when CharTypeId, UnsignedCharTypeId
          return true
        end
      else
        case rhs
        when CharTypeId, SignedCharTypeId
          return true
        end
      end
      false
    end
  end

  class SignedCharType < StandardIntegerType
    include StandardTypeAccessor

    def initialize(type_table)
      super(type_table, "signed char", char_size, char_alignment, true, true)
    end

    def id
      # NOTE: `char' type may be treated as `unsigned char'.
      #       Specialized type comparison is implemented in CharTypeId,
      #       SignedCharTypeId and UnsignedCharTypeId.
      @id ||= SignedCharTypeId.new
    end

    def integer_conversion_rank
      1
    end

    def arithmetic_type_with(type)
      type._arithmetic_type_with_signed_char(self)
    end

    def corresponding_signed_type
      self
    end

    def corresponding_unsigned_type
      unsigned_char_type
    end
  end

  class SignedCharTypeId < StandardTypeId
    include StandardTypeAccessor

    def initialize
      super("signed char")
    end

    def ==(rhs)
      if char_as_unsigned_char?
        case rhs
        when SignedCharTypeId
          return true
        end
      else
        case rhs
        when SignedCharTypeId, CharTypeId
          return true
        end
      end
      false
    end
  end

  class UnsignedCharType < StandardIntegerType
    include StandardTypeAccessor

    def initialize(type_table)
      super(type_table,
            "unsigned char", char_size, char_alignment, false, true)
    end

    def id
      # NOTE: `char' type may be treated as `unsigned char'.
      #       Specialized type comparison is implemented in CharTypeId,
      #       SignedCharTypeId and UnsignedCharTypeId.
      @id ||= UnsignedCharTypeId.new
    end

    def integer_conversion_rank
      1
    end

    def arithmetic_type_with(type)
      type._arithmetic_type_with_unsigned_char(self)
    end

    def corresponding_signed_type
      signed_char
    end

    def corresponding_unsigned_type
      self
    end
  end

  class UnsignedCharTypeId < StandardTypeId
    include StandardTypeAccessor

    def initialize
      super("unsigned char")
    end

    def ==(rhs)
      if char_as_unsigned_char?
        case rhs
        when UnsignedCharTypeId, CharTypeId
          return true
        end
      else
        case rhs
        when UnsignedCharTypeId
          return true
        end
      end
      false
    end
  end

  class ShortType < StandardIntegerType
    include StandardTypeAccessor

    def initialize(type_table)
      super(type_table, "short", short_size, short_alignment, true, false)
    end

    def id
      @id ||= ShortTypeId.new
    end

    def integer_conversion_rank
      2
    end

    def arithmetic_type_with(type)
      type._arithmetic_type_with_short(self)
    end

    def corresponding_signed_type
      signed_short_type
    end

    def corresponding_unsigned_type
      unsigned_short_type
    end
  end

  class ShortTypeId < StandardTypeId
    def initialize
      super("short")
    end

    def ==(rhs)
      case rhs
      when ShortTypeId, SignedShortTypeId, ShortIntTypeId, SignedShortIntTypeId
        true
      else
        false
      end
    end
  end

  class SignedShortType < StandardIntegerType
    include StandardTypeAccessor

    def initialize(type_table)
      super(type_table,
            "signed short", short_size, short_alignment, true, true)
    end

    def id
      @id ||= SignedShortTypeId.new
    end

    def integer_conversion_rank
      2
    end

    def arithmetic_type_with(type)
      type._arithmetic_type_with_signed_short(self)
    end

    def corresponding_signed_type
      self
    end

    def corresponding_unsigned_type
      unsigned_short_type
    end
  end

  class SignedShortTypeId < StandardTypeId
    def initialize
      super("signed short")
    end

    def ==(rhs)
      case rhs
      when SignedShortTypeId, ShortTypeId, ShortIntTypeId, SignedShortIntTypeId
        true
      else
        false
      end
    end
  end

  class UnsignedShortType < StandardIntegerType
    include StandardTypeAccessor

    def initialize(type_table)
      super(type_table,
            "unsigned short", short_size, short_alignment, false, true)
    end

    def id
      @id ||= UnsignedShortTypeId.new
    end

    def integer_conversion_rank
      2
    end

    def arithmetic_type_with(type)
      type._arithmetic_type_with_unsigned_short(self)
    end

    def corresponding_signed_type
      signed_short_type
    end

    def corresponding_unsigned_type
      self
    end
  end

  class UnsignedShortTypeId < StandardTypeId
    def initialize
      super("unsigned short")
    end

    def ==(rhs)
      case rhs
      when UnsignedShortTypeId, UnsignedShortIntTypeId
        true
      else
        false
      end
    end
  end

  class ShortIntType < StandardIntegerType
    include StandardTypeAccessor

    def initialize(type_table)
      super(type_table, "short int", short_size, short_alignment, true, false)
    end

    def id
      @id ||= ShortIntTypeId.new
    end

    def integer_conversion_rank
      2
    end

    def arithmetic_type_with(type)
      type._arithmetic_type_with_short_int(self)
    end

    def corresponding_signed_type
      signed_short_int_type
    end

    def corresponding_unsigned_type
      unsigned_short_int_type
    end

  end

  class ShortIntTypeId < StandardTypeId
    def initialize
      super("short int")
    end

    def ==(rhs)
      case rhs
      when ShortIntTypeId, ShortTypeId, SignedShortTypeId, SignedShortIntTypeId
        true
      else
        false
      end
    end
  end

  class SignedShortIntType < StandardIntegerType
    include StandardTypeAccessor

    def initialize(type_table)
      super(type_table,
            "signed short int", short_size, short_alignment, true, true)
    end

    def id
      @id ||= SignedShortIntTypeId.new
    end

    def integer_conversion_rank
      2
    end

    def arithmetic_type_with(type)
      type._arithmetic_type_with_signed_short_int(self)
    end

    def corresponding_signed_type
      self
    end

    def corresponding_unsigned_type
      unsigned_short_int_type
    end
  end

  class SignedShortIntTypeId < StandardTypeId
    def initialize
      super("signed short int")
    end

    def ==(rhs)
      case rhs
      when SignedShortIntTypeId, ShortTypeId, ShortIntTypeId, SignedShortTypeId
        true
      else
        false
      end
    end
  end

  class UnsignedShortIntType < StandardIntegerType
    include StandardTypeAccessor

    def initialize(type_table)
      super(type_table,
            "unsigned short int", short_size, short_alignment, false, true)
    end

    def id
      @id ||= UnsignedShortIntTypeId.new
    end

    def integer_conversion_rank
      2
    end

    def arithmetic_type_with(type)
      type._arithmetic_type_with_unsigned_short_int(self)
    end

    def corresponding_signed_type
      signed_short_int_type
    end

    def corresponding_unsigned_type
      self
    end
  end

  class UnsignedShortIntTypeId < StandardTypeId
    def initialize
      super("unsigned short int")
    end

    def ==(rhs)
      case rhs
      when UnsignedShortIntTypeId, UnsignedShortTypeId
        true
      else
        false
      end
    end
  end

  class IntType < StandardIntegerType
    include StandardTypeAccessor

    def initialize(type_table)
      super(type_table, "int", int_size, int_alignment, true, false)
    end

    def id
      @id ||= IntTypeId.new
    end

    def integer_conversion_rank
      3
    end

    def arithmetic_type_with(type)
      type._arithmetic_type_with_int(self)
    end

    def corresponding_signed_type
      signed_int_type
    end

    def corresponding_unsigned_type
      unsigned_int_type
    end
  end

  class IntTypeId < StandardTypeId
    def initialize
      super("int")
    end

    def ==(rhs)
      case rhs
      when IntTypeId, SignedTypeId, SignedIntTypeId
        true
      else
        false
      end
    end
  end

  class SignedType < StandardIntegerType
    include StandardTypeAccessor

    def initialize(type_table)
      super(type_table, "signed", int_size, int_alignment, true, true)
    end

    def id
      @id ||= SignedTypeId.new
    end

    def integer_conversion_rank
      3
    end

    def arithmetic_type_with(type)
      type._arithmetic_type_with_signed(self)
    end

    def corresponding_signed_type
      self
    end

    def corresponding_unsigned_type
      unsigned_type
    end
  end

  class SignedTypeId < StandardTypeId
    def initialize
      super("signed")
    end

    def ==(rhs)
      case rhs
      when SignedTypeId, IntTypeId, SignedIntTypeId
        true
      else
        false
      end
    end
  end

  class SignedIntType < StandardIntegerType
    include StandardTypeAccessor

    def initialize(type_table)
      super(type_table, "signed int", int_size, int_alignment, true, true)
    end

    def id
      @id ||= SignedIntTypeId.new
    end

    def integer_conversion_rank
      3
    end

    def arithmetic_type_with(type)
      type._arithmetic_type_with_signed_int(self)
    end

    def corresponding_signed_type
      self
    end

    def corresponding_unsigned_type
      unsigned_int_type
    end
  end

  class SignedIntTypeId < StandardTypeId
    def initialize
      super("signed int")
    end

    def ==(rhs)
      case rhs
      when SignedIntTypeId, IntTypeId, SignedTypeId
        true
      else
        false
      end
    end
  end

  class UnsignedType < StandardIntegerType
    include StandardTypeAccessor

    def initialize(type_table)
      super(type_table, "unsigned", int_size, int_alignment, false, true)
    end

    def id
      @id ||= UnsignedTypeId.new
    end

    def integer_conversion_rank
      3
    end

    def arithmetic_type_with(type)
      type._arithmetic_type_with_unsigned(self)
    end

    def corresponding_signed_type
      signed_type
    end

    def corresponding_unsigned_type
      self
    end
  end

  class UnsignedTypeId < StandardTypeId
    def initialize
      super("unsigned")
    end

    def ==(rhs)
      case rhs
      when UnsignedTypeId, UnsignedIntTypeId
        true
      else
        false
      end
    end
  end

  class UnsignedIntType < StandardIntegerType
    include StandardTypeAccessor

    def initialize(type_table)
      super(type_table, "unsigned int", int_size, int_alignment, false, true)
    end

    def id
      @id ||= UnsignedIntTypeId.new
    end

    def integer_conversion_rank
      3
    end

    def arithmetic_type_with(type)
      type._arithmetic_type_with_unsigned_int(self)
    end

    def corresponding_signed_type
      signed_int_type
    end

    def corresponding_unsigned_type
      self
    end
  end

  class UnsignedIntTypeId < StandardTypeId
    def initialize
      super("unsigned int")
    end

    def ==(rhs)
      case rhs
      when UnsignedIntTypeId, UnsignedTypeId
        true
      else
        false
      end
    end
  end

  class LongType < StandardIntegerType
    include StandardTypeAccessor

    def initialize(type_table)
      super(type_table, "long", long_size, long_alignment, true, false)
    end

    def id
      @id ||= LongTypeId.new
    end

    def integer_conversion_rank
      4
    end

    def arithmetic_type_with(type)
      type._arithmetic_type_with_long(self)
    end

    def corresponding_signed_type
      signed_long_type
    end

    def corresponding_unsigned_type
      unsigned_long_type
    end
  end

  class LongTypeId < StandardTypeId
    def initialize
      super("long")
    end

    def ==(rhs)
      case rhs
      when LongTypeId, SignedLongTypeId, LongIntTypeId, SignedLongIntTypeId
        true
      else
        false
      end
    end
  end

  class SignedLongType < StandardIntegerType
    include StandardTypeAccessor

    def initialize(type_table)
      super(type_table, "signed long", long_size, long_alignment, true, true)
    end

    def id
      @id ||= SignedLongTypeId.new
    end

    def integer_conversion_rank
      4
    end

    def arithmetic_type_with(type)
      type._arithmetic_type_with_signed_long(self)
    end

    def corresponding_signed_type
      self
    end

    def corresponding_unsigned_type
      unsigned_long_type
    end
  end

  class SignedLongTypeId < StandardTypeId
    def initialize
      super("signed long")
    end

    def ==(rhs)
      case rhs
      when SignedLongTypeId, LongTypeId, LongIntTypeId, SignedLongIntTypeId
        true
      else
        false
      end
    end
  end

  class UnsignedLongType < StandardIntegerType
    include StandardTypeAccessor

    def initialize(type_table)
      super(type_table,
            "unsigned long", long_size, long_alignment, false, true)
    end

    def id
      @id ||= UnsignedLongTypeId.new
    end

    def integer_conversion_rank
      4
    end

    def arithmetic_type_with(type)
      type._arithmetic_type_with_unsigned_long(self)
    end

    def corresponding_signed_type
      signed_long_type
    end

    def corresponding_unsigned_type
      self
    end
  end

  class UnsignedLongTypeId < StandardTypeId
    def initialize
      super("unsigned long")
    end

    def ==(rhs)
      case rhs
      when UnsignedLongTypeId, UnsignedLongIntTypeId
        true
      else
        false
      end
    end
  end

  class LongIntType < StandardIntegerType
    include StandardTypeAccessor

    def initialize(type_table)
      super(type_table, "long int", long_size, long_alignment, true, false)
    end

    def id
      @id ||= LongIntTypeId.new
    end

    def integer_conversion_rank
      4
    end

    def arithmetic_type_with(type)
      type._arithmetic_type_with_long_int(self)
    end

    def corresponding_signed_type
      signed_long_int_type
    end

    def corresponding_unsigned_type
      unsigned_long_int_type
    end
  end

  class LongIntTypeId < StandardTypeId
    def initialize
      super("long int")
    end

    def ==(rhs)
      case rhs
      when LongIntTypeId, LongTypeId, SignedLongTypeId, SignedLongIntTypeId
        true
      else
        false
      end
    end
  end

  class SignedLongIntType < StandardIntegerType
    include StandardTypeAccessor

    def initialize(type_table)
      super(type_table,
            "signed long int", long_size, long_alignment, true, true)
    end

    def id
      @id ||= SignedLongIntTypeId.new
    end

    def integer_conversion_rank
      4
    end

    def arithmetic_type_with(type)
      type._arithmetic_type_with_signed_long_int(self)
    end

    def corresponding_signed_type
      self
    end

    def corresponding_unsigned_type
      unsigned_long_int_type
    end
  end

  class SignedLongIntTypeId < StandardTypeId
    def initialize
      super("signed long int")
    end

    def ==(rhs)
      case rhs
      when SignedLongIntTypeId, LongTypeId, LongIntTypeId, SignedLongTypeId
        true
      else
        false
      end
    end
  end

  class UnsignedLongIntType < StandardIntegerType
    include StandardTypeAccessor

    def initialize(type_table)
      super(type_table,
            "unsigned long int", long_size, long_alignment, false, true)
    end

    def id
      @id ||= UnsignedLongIntTypeId.new
    end

    def integer_conversion_rank
      4
    end

    def arithmetic_type_with(type)
      type._arithmetic_type_with_unsigned_long_int(self)
    end

    def corresponding_signed_type
      signed_long_int_type
    end

    def corresponding_unsigned_type
      self
    end
  end

  class UnsignedLongIntTypeId < StandardTypeId
    def initialize
      super("unsigned long int")
    end

    def ==(rhs)
      case rhs
      when UnsignedLongIntTypeId, UnsignedLongTypeId
        true
      else
        false
      end
    end
  end

  class LongLongType < StandardIntegerType
    include StandardTypeAccessor

    def initialize(type_table)
      super(type_table,
            "long long", long_long_size, long_long_alignment, true, false)
    end

    def id
      @id ||= LongLongTypeId.new
    end

    def integer_conversion_rank
      5
    end

    def arithmetic_type_with(type)
      type._arithmetic_type_with_long_long(self)
    end

    def corresponding_signed_type
      signed_long_long_type
    end

    def corresponding_unsigned_type
      unsigned_long_long_type
    end
  end

  class LongLongTypeId < StandardTypeId
    def initialize
      super("long long")
    end

    def ==(rhs)
      case rhs
      when LongLongTypeId, SignedLongLongTypeId, LongLongIntTypeId,
           SignedLongLongIntTypeId
        true
      else
        false
      end
    end
  end

  class SignedLongLongType < StandardIntegerType
    include StandardTypeAccessor

    def initialize(type_table)
      super(type_table, "signed long long",
            long_long_size, long_long_alignment, true, true)
    end

    def id
      @id ||= SignedLongLongTypeId.new
    end

    def integer_conversion_rank
      5
    end

    def arithmetic_type_with(type)
      type._arithmetic_type_with_signed_long_long(self)
    end

    def corresponding_signed_type
      self
    end

    def corresponding_unsigned_type
      unsigned_long_long_type
    end
  end

  class SignedLongLongTypeId < StandardTypeId
    def initialize
      super("signed long long")
    end

    def ==(rhs)
      case rhs
      when SignedLongLongTypeId, LongLongTypeId, LongLongIntTypeId,
           SignedLongLongIntTypeId
        true
      else
        false
      end
    end
  end

  class UnsignedLongLongType < StandardIntegerType
    include StandardTypeAccessor

    def initialize(type_table)
      super(type_table, "unsigned long long",
            long_long_size, long_long_alignment, false, true)
    end

    def id
      @id ||= UnsignedLongLongTypeId.new
    end

    def integer_conversion_rank
      5
    end

    def arithmetic_type_with(type)
      type._arithmetic_type_with_unsigned_long_long(self)
    end

    def corresponding_signed_type
      signed_long_long_type
    end

    def corresponding_unsigned_type
      self
    end
  end

  class UnsignedLongLongTypeId < StandardTypeId
    def initialize
      super("unsigned long long")
    end

    def ==(rhs)
      case rhs
      when UnsignedLongLongTypeId, UnsignedLongLongIntTypeId
        true
      else
        false
      end
    end
  end

  class LongLongIntType < StandardIntegerType
    include StandardTypeAccessor

    def initialize(type_table)
      super(type_table,
            "long long int", long_long_size, long_long_alignment, true, false)
    end

    def id
      @id ||= LongLongIntTypeId.new
    end

    def integer_conversion_rank
      5
    end

    def arithmetic_type_with(type)
      type._arithmetic_type_with_long_long_int(self)
    end

    def corresponding_signed_type
      signed_long_long_int_type
    end

    def corresponding_unsigned_type
      unsigned_long_long_int_type
    end
  end

  class LongLongIntTypeId < StandardTypeId
    def initialize
      super("long long int")
    end

    def ==(rhs)
      case rhs
      when LongLongIntTypeId, LongLongTypeId, SignedLongLongTypeId,
           SignedLongLongIntTypeId
        true
      else
        false
      end
    end
  end

  class SignedLongLongIntType < StandardIntegerType
    include StandardTypeAccessor

    def initialize(type_table)
      super(type_table, "signed long long int",
            long_long_size, long_long_alignment, true, true)
    end

    def id
      @id ||= SignedLongLongIntTypeId.new
    end

    def integer_conversion_rank
      5
    end

    def arithmetic_type_with(type)
      type._arithmetic_type_with_signed_long_long_int(self)
    end

    def corresponding_signed_type
      self
    end

    def corresponding_unsigned_type
      unsigned_long_long_int_type
    end
  end

  class SignedLongLongIntTypeId < StandardTypeId
    def initialize
      super("signed long long int")
    end

    def ==(rhs)
      case rhs
      when SignedLongLongIntTypeId, LongLongTypeId, LongLongIntTypeId,
           SignedLongLongType
        true
      else
        false
      end
    end
  end

  class UnsignedLongLongIntType < StandardIntegerType
    include StandardTypeAccessor

    def initialize(type_table)
      super(type_table, "unsigned long long int",
            long_long_size, long_long_alignment, false, true)
    end

    def id
      @id ||= UnsignedLongLongIntTypeId.new
    end

    def integer_conversion_rank
      5
    end

    def arithmetic_type_with(type)
      type._arithmetic_type_with_unsigned_long_long_int(self)
    end

    def corresponding_signed_type
      signed_long_long_int_type
    end

    def corresponding_unsigned_type
      self
    end
  end

  class UnsignedLongLongIntTypeId < StandardTypeId
    def initialize
      super("unsigned long long int")
    end

    def ==(rhs)
      case rhs
      when UnsignedLongLongIntTypeId, UnsignedLongLongTypeId
        true
      else
        false
      end
    end
  end

  class FloatingType < ScalarDataType
    def initialize(type_table, name, bit_size, bit_alignment)
      super(type_table, name, bit_size, bit_alignment)
    end

    def id
      subclass_responsibility
    end

    def image
      name
    end

    def brief_image
      name
    end

    def location
      nil
    end

    def incomplete?
      false
    end

    def compatible?(to_type)
      type.floating? &&
        to_type.min_value <= self.min_value &&
        self.max_value <= to_type.max_value
    end

    def integer?
      false
    end

    def floating?
      true
    end

    def pointer?
      false
    end

    def enum?
      false
    end

    def bitfield?
      false
    end

    def signed?
      true
    end

    def explicitly_signed?
      true
    end

    def enumerators
      []
    end

    def min_value
      (-2**fraction_bit_size * 10**(exponent_bit_size - 1)).to_f
    end

    def max_value
      (2**fraction_bit_size * 10**(exponent_bit_size - 1)).to_f
    end

    def integer_conversion_rank
      0 # NOTREACHED
    end

    def integer_promoted_type
      self # NOTREACHED
    end

    def argument_promoted_type
      subclass_responsibility
    end

    def arithmetic_type_with(type)
      subclass_responsibility
    end

    def corresponding_signed_type
      self # NOTREACHED
    end

    def corresponding_unsigned_type
      self # NOTREACHED
    end

    private
    def fraction_bit_size
      subclass_responsibility
    end

    def exponent_bit_size
      subclass_responsibility
    end
  end

  class FloatType < FloatingType
    include StandardTypeAccessor

    def initialize(type_table)
      super(type_table, "float", float_size, float_alignment)
    end

    def id
      @id ||= FloatTypeId.new
    end

    def argument_promoted_type
      # NOTE: The ISO C99 standard saids;
      #
      # 6.5.2.2 Function calls
      #
      # 6 If the expression that denotes the called function has a type that
      #   does not include a prototype, the integer promotions are performed on
      #   each argument, and arguments that have type float are promoted to
      #   double.  These are called the default argument promotions.  If the
      #   number of arguments does not equal the number of parameters, the
      #   behavior is undefined.  If the function is defined with a type that
      #   includes a prototype, and either the prototype ends with an ellipsis
      #   (, ...) or the types of the arguments after promotion are not
      #   compatible with the types of the parameters, the behavior is
      #   undefined.  If the function is defined with a type that does not
      #   include a prototype, and the types of the arguments after promotion
      #   are not compatible with those of the parameters after promotion, the
      #   behavior is undefined, except for the following cases:
      #
      #     -- one promoted type is a signed integer type, the other promoted
      #        type is the corresponding unsigned integer type, and the value
      #        is representable in both types;
      #     -- both types are pointers to qualified or unqualified versions of
      #        a character type or void.
      double_type
    end

    def arithmetic_type_with(type)
      type._arithmetic_type_with_float(self)
    end

    private
    def fraction_bit_size
      # TODO: Bit size of the fraction part of `float' should be configurable.
      23
    end

    def exponent_bit_size
      # TODO: Bit size of the exponent part of `float' should be configurable.
      8
    end
  end

  class FloatTypeId < StandardTypeId
    def initialize
      super("float")
    end
  end

  class DoubleType < FloatingType
    include StandardTypeAccessor

    def initialize(type_table)
      super(type_table, "double", double_size, double_alignment)
    end

    def id
      @id ||= DoubleTypeId.new
    end

    def argument_promoted_type
      self
    end

    def arithmetic_type_with(type)
      type._arithmetic_type_with_double(self)
    end

    private
    def fraction_bit_size
      # TODO: Bit size of the fraction part of `double' should be configurable.
      52
    end

    def exponent_bit_size
      # TODO: Bit size of the exponent part of `double' should be configurable.
      11
    end
  end

  class DoubleTypeId < StandardTypeId
    def initialize
      super("double")
    end
  end

  class LongDoubleType < FloatingType
    include StandardTypeAccessor

    def initialize(type_table)
      super(type_table, "long double", long_double_size, long_double_alignment)
    end

    def id
      @id ||= LongDoubleTypeId.new
    end

    def argument_promoted_type
      self
    end

    def arithmetic_type_with(type)
      type._arithmetic_type_with_long_double(self)
    end

    private
    def fraction_bit_size
      # TODO: Bit size of the fraction part of `long double' should be
      #       configurable.
      52
    end

    def exponent_bit_size
      # TODO: Bit size of the exponent part of `long double' should be
      #       configurable.
      11
    end
  end

  class LongDoubleTypeId < StandardTypeId
    def initialize
      super("long double")
    end
  end

  class BitfieldType < IntegerType
    def initialize(type_table, base_type, field_width)
      super(type_table,
            "#{base_type.real_type.name}:#{field_width}",
            field_width, base_type.bit_alignment,
            base_type.signed?, base_type.explicitly_signed?)

      @base_type = base_type
    end

    attr_reader :base_type

    def id
      @id ||= BitfieldTypeId.new(@base_type, bit_size)
    end

    def incomplete?
      @base_type.incomplete?
    end

    def pointer?
      false
    end

    def enum?
      false
    end

    def bitfield?
      true
    end

    def undeclared?
      @base_type.undeclared?
    end

    def unresolved?
      @base_type.unresolved?
    end

    def enumerators
      []
    end

    def integer_conversion_rank
      -1
    end

    def integer_promoted_type
      # TODO: Should support the C99 _Bool type.
      # NOTE: The ISO C99 standard saids;
      #
      # 6.3.1 Arithmetic operands
      # 6.3.1.1 Boolean, characters, and integers
      #
      # 2 The following may be used in an expression wherever an int or
      #   unsigned int may be used:
      #
      #     -- An object or expression with an integer type whose integer
      #        conversion rank is less than or equal to the rank of int and
      #        unsigned int.
      #     -- A bit-field of type _Bool, int, signed int, or unsigned int.
      #
      #   If an int can represent all values of the original type, the value is
      #   converted to an int; otherwise, it is converted to an unsigned int.
      #   These are called the integer promotions.  All other types are
      #   unchanged by the integer promotions.
      if self.undeclared? || self.unresolved?
        self
      else
        if @base_type.same_as?(int_type) ||
            @base_type.same_as?(unsigned_int_type)
          self.compatible?(int_type) ? int_type : unsigned_int_type
        else
          self
        end
      end
    end

    def arithmetic_type_with(type)
      type._arithmetic_type_with_bitfield(self)
    end

    def corresponding_signed_type
      self # NOTREACHED
    end

    def corresponding_unsigned_type
      self # NOTREACHED
    end
  end

  class BitfieldTypeId < TypeId
    def initialize(base_type, field_width)
      super(create_value(base_type, field_width))
    end

    private
    def create_value(base_type, field_width)
      "#{base_type.real_type.name.split(" ").sort.join(" ")}:#{field_width}"
    end
  end

  module Scopeable
    attr_accessor :scope
  end

  class EnumType < IntegerType
    include Scopeable
    include StandardTypeAccessor

    def initialize(type_table, enum_type_declaration)
      super(type_table, enum_type_declaration.identifier.value,
            int_size, int_alignment, true, true, [enum_type_declaration])

      @image = enum_type_declaration.enum_specifier.to_s
      @location = enum_type_declaration.location
    end

    attr_accessor :image
    attr_accessor :location

    def id
      @id ||= EnumTypeId.new(name)
    end

    def incomplete?
      declarations.all? { |decl| decl.enumerators.nil? }
    end

    def pointer?
      false
    end

    def enum?
      true
    end

    def bitfield?
      false
    end

    def brief_image
      "enum #{name}"
    end

    def enumerators
      declarations.map { |decl| decl.enumerators }.compact.flatten.uniq
    end

    def integer_conversion_rank
      # NOTE: The ISO C99 standard saids;
      #
      # 6.3.1 Arithmetic operands
      # 6.3.1.1 Boolean, characters, and integers
      #
      # 1 Every integer type has an integer conversion rank defined as follows:
      #
      #     -- No two signed integer types shall have the same rank, even if
      #        they have the same representation.
      #     -- The rank of a signed integer type shall be greater than the rank
      #        of any signed integer type with less precision.
      #     -- The rank of long long int shall be greater than the rank of long
      #        int, which shall be greater than the rank of int, which shall be
      #        greater than the rank of short int, which shall be greater than
      #        the rank of signed char.
      #     -- The rank of any unsigned integer type shall equal the rank of
      #        the corresponding signed integer type, if any.
      #     -- The rank of any standard integer type shall be greater than the
      #        rank of any extended integer type with the same width.
      #     -- The rank of char shall equal the rank of signed char and
      #        unsigned char.
      #     -- The rank of _Bool shall be less than the rank of all other
      #        standard integer types.
      #     -- The rank of any enumerated type shall equal the rank of the
      #        compatible integer type.
      #     -- The rank of any extended signed integer type relative to another
      #        extended signed integer type with the same precision is
      #        implementation-defined, but still subject to the other rules for
      #        determining the integer conversion rank.
      #     -- For all integer types T1, T2, and T3, if T1 has greater rank
      #        than T2 and T2 has greater rank than T3, then T1 has greater
      #        rank than T3.
      #
      # NOTE: The integer conversion rank of any enumerated type is equal to
      #       the rank of int.
      int_type.integer_conversion_rank
    end

    def integer_promoted_type
      int_type
    end

    def arithmetic_type_with(type)
      type._arithmetic_type_with_enum(self)
    end

    def corresponding_signed_type
      signed_int_type
    end

    def corresponding_unsigned_type
      unsigned_int_type
    end
  end

  class EnumTypeId < TypeId
    def initialize(name)
      super("enum #{name}")
    end
  end

  class PointerType < IntegerType
    include StandardTypeAccessor

    def initialize(type_table, base_type)
      super(type_table, create_name(base_type),
            base_type.function? ? code_ptr_size : data_ptr_size,
            base_type.function? ? code_ptr_alignment : data_ptr_alignment,
            false, true, base_type.declarations)

      @base_type = base_type
    end

    attr_reader :base_type

    def id
      @id ||= PointerTypeId.new(@base_type)
    end

    def image
      create_image(@base_type)
    end

    def brief_image
      create_brief_image(@base_type)
    end

    def real_type
      type_table.pointer_type(@base_type.real_type)
    end

    def incomplete?
      false
    end

    def convertible?(to_type)
      lhs_unqualified = self.real_type.unqualify
      rhs_unqualified = to_type.real_type.unqualify

      if rhs_unqualified.pointer? || rhs_unqualified.array?
        lhs_base = lhs_unqualified.base_type
        rhs_base = rhs_unqualified.base_type

        unless lhs_base.more_cv_qualified?(rhs_base)
          lhs_base.void? || lhs_base.convertible?(rhs_base)
        else
          false
        end
      else
        false
      end
    end

    def pointer?
      true
    end

    def enum?
      false
    end

    def undeclared?
      @base_type.undeclared?
    end

    def unresolved?
      # NOTE: To avoid the infinite recursive call of #unresolved? when the
      #       composite type contains the pointer to it's owner type.
      @base_type.kind_of?(UnresolvedType)
    end

    def bitfield?
      false
    end

    def enumerators
      []
    end

    def integer_conversion_rank
      # NOTE: Pointer variables must not be converted implicitly.
      100
    end

    def integer_promoted_type
      # NOTE: Pointer variables must not be converted implicitly.
      self
    end

    def arithmetic_type_with(type)
      type._arithmetic_type_with_pointer(self)
    end

    def corresponding_signed_type
      self # NOTREACHED
    end

    def corresponding_unsigned_type
      self # NOTREACHED
    end

    private
    def create_name(base_type)
      if base_type.function?
        "#{base_type.return_type.name}(*)(" +
          base_type.parameter_types.map { |type| type.name }.join(",") +
          (base_type.have_va_list? ? ",...)" : ")")
      else
        "#{base_type.name} *"
      end
    end

    def create_image(base_type)
      if base_type.function?
        "#{base_type.return_type.image}(*)(" +
          base_type.parameter_types.map { |type| type.image }.join(",") +
          (base_type.have_va_list? ? ",...)" : ")")
      else
        "#{base_type.image} *"
      end
    end

    def create_brief_image(base_type)
      if base_type.function?
        "#{base_type.return_type.brief_image}(*)(" +
          base_type.parameter_types.map { |type| type.brief_image }.join(",") +
          (base_type.have_va_list? ? ",...)" : ")")
      else
        "#{base_type.brief_image} *"
      end
    end
  end

  class PointerTypeId < TypeId
    def initialize(base_type)
      super(create_value(base_type))

      @base_type = base_type
    end

    def ==(rhs)
      case rhs
      when PointerTypeId
        @base_type == rhs.base_type
      else
        false
      end
    end

    def hash
      "#{@base_type.id.hash}*".hash
    end

    protected
    attr_reader :base_type

    private
    def create_value(base_type)
      real_type = base_type.real_type

      if real_type.function?
        "#{real_type.return_type.brief_image}(*)(" +
          real_type.parameter_types.map { |type| type.brief_image }.join(",") +
          (real_type.have_va_list? ? ",...)" : ")")
      else
        "#{real_type.brief_image} *"
      end
    end
  end

  class ArrayType < Type
    # NOTE: To avoid huge array allocation in interpreting phase.
    MAX_LENGTH = 256
    private_constant :MAX_LENGTH

    def initialize(type_table, base_type, length = nil)
      super(type_table, create_name(base_type, length))

      @base_type = base_type
      @length = length
    end

    attr_reader :base_type

    # NOTE: Length of the array type may be deducted by the size of the
    #       initializer in interpret phase.
    attr_accessor :length

    def impl_length
      # NOTE: Implementation defined length of this array.
      @length ? [[0, @length].max, MAX_LENGTH].min : 0
    end

    def id
      # NOTE: ID of the array type cannot be cached.
      #       Length of the variable length array will be deducted in the
      #       interpret phase.
      ArrayTypeId.new(@base_type, @length)
    end

    def image
      create_image(@base_type, @length)
    end

    def brief_image
      create_brief_image(@base_type, @length)
    end

    def location
      nil
    end

    def bit_size
      @length ? @base_type.bit_size * @length : 0
    end

    def bit_alignment
      aligned_bit_size
    end

    def aligned_bit_size
      @length ? @base_type.aligned_bit_size * @length : 0
    end

    def real_type
      type_table.array_type(@base_type.real_type, @length)
    end

    def unqualify
      self
    end

    def incomplete?
      @base_type.incomplete? || @length.nil?
    end

    def compatible?(to_type)
      to_type.array? &&
        @length == to_type.length && @base_type.compatible?(to_type.base_type)
    end

    def coercible?(to_type)
      to_type.array? && @base_type.coercible?(to_type.base_type)
    end

    def convertible?(to_type)
      lhs_unqualified = self.real_type.unqualify
      rhs_unqualified = to_type.real_type.unqualify

      if rhs_unqualified.pointer? || rhs_unqualified.array?
        lhs_base = lhs_unqualified.base_type
        rhs_base = rhs_unqualified.base_type

        unless lhs_base.more_cv_qualified?(rhs_base)
          lhs_base.convertible?(rhs_base)
        else
          false
        end
      else
        false
      end
    end

    def same_as?(type)
      lhs_unqualified = self.real_type.unqualify
      rhs_unqualified = type.real_type.unqualify

      case
      when rhs_unqualified.array?
        if lhs_unqualified.length
          lhs_unqualified.length == rhs_unqualified.length
        else
          lhs_unqualified.base_type.same_as?(rhs_unqualified.base_type)
        end
      when rhs_unqualified.pointer?
        lhs_unqualified.base_type.same_as?(rhs_unqualified.base_type)
      else
        false
      end
    end

    def scalar?
      false
    end

    def integer?
      false
    end

    def floating?
      false
    end

    def array?
      true
    end

    def struct?
      false
    end

    def union?
      false
    end

    def pointer?
      false
    end

    def qualified?
      false
    end

    def function?
      false
    end

    def enum?
      false
    end

    def user?
      false
    end

    def void?
      false
    end

    def undeclared?
      @base_type.undeclared?
    end

    def unresolved?
      @base_type.unresolved?
    end

    def const?
      false
    end

    def volatile?
      false
    end

    def restrict?
      false
    end

    def bitfield?
      false
    end

    def signed?
      false
    end

    def explicitly_signed?
      false
    end

    def have_va_list?
      false
    end

    def return_type
      nil
    end

    def parameter_types
      []
    end

    def enumerators
      []
    end

    def members
      []
    end

    def member_named(name)
      nil
    end

    def min_value
      0
    end

    def max_value
      0
    end

    def nil_value
      ArrayValue.new(impl_length.times.map { @base_type.nil_value })
    end

    def zero_value
      ArrayValue.new(impl_length.times.map { @base_type.zero_value })
    end

    def arbitrary_value
      ArrayValue.new(impl_length.times.map { @base_type.arbitrary_value })
    end

    def undefined_value
      ArrayValue.new(impl_length.times.map { @base_type.undefined_value })
    end

    def parameter_value
      ArrayValue.new(impl_length.times.map { @base_type.parameter_value })
    end

    def return_value
      ArrayValue.new(impl_length.times.map { @base_type.return_value })
    end

    def coerce_scalar_value(value)
      # NOTE: Cannot coerce scalar value into array in C language.
      undefined_value # NOTREACHED
    end

    def coerce_array_value(value)
      # NOTE: The ISO C99 standard saids;
      #
      # 6.7.8 Initialization
      #
      # Semantics
      #
      # 10 If an object that has automatic storage duration is not initialized
      #    explicitly, its value is indeterminate.  If an object that has
      #    static storage duration is not initialized explicitly, then:
      #
      #    -- if it has pointer type, it is initialized to a null pointer;
      #    -- if it has arithmetic type, it is initialized to (positive or
      #       unsigned) zero;
      #    -- if it is an aggregate, every member is initialized (recursively)
      #       according to these rules;
      #    -- if it is a union, the first named member is initialized
      #       (recursively) according to these rules.
      #
      # 21 If there are fewer initializers in a brace-enclosed list than there
      #    are elements or members of an aggregate, or fewer characters in a
      #    string literal used to initialize an array of known size that there
      #    are elements in the array, the remainder of the aggregate shall be
      #    initialized implicitly the same as objects that have static storage
      #    duration.
      values = ([@base_type] * impl_length).zip(value.values).map { |type, val|
        val ? val.coerce_to(type) : type.arbitrary_value
      }
      ArrayValue.new(values)
    end

    def coerce_composite_value(value)
      # NOTE: Cannot coerce composite value into array in C language.
      undefined_value # NOTREACHED
    end

    def integer_conversion_rank
      0 # NOTREACHED
    end

    def integer_promoted_type
      self # NOTREACHED
    end

    def argument_promoted_type
      self # NOTREACHED
    end

    def arithmetic_type_with(type)
      # NOTE: An arithmetic operation with ArrayType must not be executed!
      type._arithmetic_type_with_array(self) # NOTREACHED
    end

    def _arithmetic_type_with_undeclared(lhs, rhs = self)
      rhs.arithmetic_type_with(lhs)
    end

    def _arithmetic_type_with_unresolved(lhs, rhs = self)
      rhs.arithmetic_type_with(lhs)
    end

    def _arithmetic_type_with_void(lhs, rhs = self)
      rhs.arithmetic_type_with(lhs)
    end

    def _arithmetic_type_with_function(lhs, rhs = self)
      rhs.arithmetic_type_with(lhs)
    end

    def _arithmetic_type_with_char(lhs, rhs = self)
      # NOTE: An arithmetic operation with ArrayType must not be executed!
      # NOTE: Binary operation with `char' and ArrayType
      #       makes integer-promoted type of `char'.
      lhs.integer_promoted_type # NOTREACHED
    end

    def _arithmetic_type_with_signed_char(lhs, rhs = self)
      # NOTE: An arithmetic operation with ArrayType must not be executed!
      # NOTE: Binary operation with `signed char' and ArrayType
      #       makes integer-promoted type of `signed char'.
      lhs.integer_promoted_type # NOTREACHED
    end

    def _arithmetic_type_with_unsigned_char(lhs, rhs = self)
      # NOTE: An arithmetic operation with ArrayType must not be executed!
      # NOTE: Binary operation with `unsigned char' and ArrayType
      #       makes integer-promoted type of `unsigned char'.
      lhs.integer_promoted_type # NOTREACHED
    end

    def _arithmetic_type_with_short(lhs, rhs = self)
      # NOTE: An arithmetic operation with ArrayType must not be executed!
      # NOTE: Binary operation with `short' and ArrayType
      #       makes integer-promoted type of `short'.
      lhs.integer_promoted_type # NOTREACHED
    end

    def _arithmetic_type_with_signed_short(lhs, rhs = self)
      # NOTE: An arithmetic operation with ArrayType must not be executed!
      # NOTE: Binary operation with `signed short' and ArrayType
      #       makes integer-promoted type of `signed short'.
      lhs.integer_promoted_type # NOTREACHED
    end

    def _arithmetic_type_with_unsigned_short(lhs, rhs = self)
      # NOTE: An arithmetic operation with ArrayType must not be executed!
      # NOTE: Binary operation with `unsigned short' and ArrayType
      #       makes integer-promoted type of `unsigned short'.
      lhs.integer_promoted_type # NOTREACHED
    end

    def _arithmetic_type_with_short_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with ArrayType must not be executed!
      # NOTE: Binary operation with `short int' and ArrayType
      #       makes integer-promoted type of `short int'.
      lhs.integer_promoted_type # NOTREACHED
    end

    def _arithmetic_type_with_signed_short_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with ArrayType must not be executed!
      # NOTE: Binary operation with `signed short int' and ArrayType
      #       makes integer-promoted type of `signed short int'.
      lhs.integer_promoted_type # NOTREACHED
    end

    def _arithmetic_type_with_unsigned_short_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with ArrayType must not be executed!
      # NOTE: Binary operation with `unsigned short int' and ArrayType
      #       makes integer-promoted type of `unsigned short int'.
      lhs.integer_promoted_type # NOTREACHED
    end

    def _arithmetic_type_with_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with ArrayType must not be executed!
      # NOTE: Binary operation with `int' and ArrayType makes `int'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_signed(lhs, rhs = self)
      # NOTE: An arithmetic operation with ArrayType must not be executed!
      # NOTE: Binary operation with `signed' and ArrayType makes `signed'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_signed_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with ArrayType must not be executed!
      # NOTE: Binary operation with `signed int' and ArrayType
      #       makes `signed int'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_unsigned(lhs, rhs = self)
      # NOTE: An arithmetic operation with ArrayType must not be executed!
      # NOTE: Binary operation with `unsigned' and ArrayType makes `unsigned'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_unsigned_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with ArrayType must not be executed!
      # NOTE: Binary operation with `unsigned int' and ArrayType
      #       makes `unsigned int'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_long(lhs, rhs = self)
      # NOTE: An arithmetic operation with ArrayType must not be executed!
      # NOTE: Binary operation with `long' and ArrayType makes `long'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_signed_long(lhs, rhs = self)
      # NOTE: An arithmetic operation with ArrayType must not be executed!
      # NOTE: Binary operation with `signed long' and ArrayType
      #       makes `signed long'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_unsigned_long(lhs, rhs = self)
      # NOTE: An arithmetic operation with ArrayType must not be executed!
      # NOTE: Binary operation with `unsigned long' and ArrayType
      #       makes `unsigned long'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_long_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with ArrayType must not be executed!
      # NOTE: Binary operation with `long int' and ArrayType makes `long int'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_signed_long_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with ArrayType must not be executed!
      # NOTE: Binary operation with `signed long int' and ArrayType
      #       makes `signed long int'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_unsigned_long_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with ArrayType must not be executed!
      # NOTE: Binary operation with `unsigned long int' and ArrayType
      #       makes `unsigned long int'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_long_long(lhs, rhs = self)
      # NOTE: An arithmetic operation with ArrayType must not be executed!
      # NOTE: Binary operation with `long long' and ArrayType makes
      #       `long long'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_signed_long_long(lhs, rhs = self)
      # NOTE: An arithmetic operation with ArrayType must not be executed!
      # NOTE: Binary operation with `signed long long' and ArrayType
      #       makes `signed long long'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_unsigned_long_long(lhs, rhs = self)
      # NOTE: An arithmetic operation with ArrayType must not be executed!
      # NOTE: Binary operation with `unsigned long long' and ArrayType
      #       makes `unsigned long long'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_long_long_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with ArrayType must not be executed!
      # NOTE: Binary operation with `long long int' and ArrayType
      #       makes `long long int'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_signed_long_long_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with ArrayType must not be executed!
      # NOTE: Binary operation with `signed long long int' and ArrayType
      #       makes `signed long long int'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_unsigned_long_long_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with ArrayType must not be executed!
      # NOTE: Binary operation with `unsigned long long int' and ArrayType
      #       makes `unsigned long long int'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_float(lhs, rhs = self)
      # NOTE: An arithmetic operation with ArrayType must not be executed!
      # NOTE: Binary operation with `float' and ArrayType makes `float'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_double(lhs, rhs = self)
      # NOTE: An arithmetic operation with ArrayType must not be executed!
      # NOTE: Binary operation with `double' and ArrayType makes `double'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_long_double(lhs, rhs = self)
      # NOTE: An arithmetic operation with ArrayType must not be executed!
      # NOTE: Binary operation with `long double' and ArrayType
      #       makes `long double'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_bitfield(lhs, rhs = self)
      # NOTE: An arithmetic operation with ArrayType must not be executed!
      # NOTE: Binary operation with BitfieldType and ArrayType
      #       makes integer-promoted type of BitfieldType.
      lhs.integer_promoted_type # NOTREACHED
    end

    def _arithmetic_type_with_enum(lhs, rhs = self)
      # NOTE: An arithmetic operation with ArrayType must not be executed!
      # NOTE: Binary operation with EnumType and ArrayType makes EnumType.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_pointer(lhs, rhs = self)
      # NOTE: An arithmetic operation with ArrayType must not be executed!
      # NOTE: Binary operation with PointerType and ArrayType makes
      #       PointerType.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_array(lhs, rhs = self)
      # NOTE: An arithmetic operation with ArrayType must not be executed!
      # NOTE: Binary operation with ArrayType and ArrayType
      #       raises a runtime error.
      raise TypeError,
        "arithmetic-type of ArrayType and ArrayType is not defined."
    end

    def _arithmetic_type_with_struct(lhs, rhs = self)
      # NOTE: An arithmetic operation with ArrayType must not be executed!
      # NOTE: Binary operation with StructType and ArrayType
      #       raises a runtime error.
      raise TypeError,
        "arithmetic-type of StructType and ArrayType is not defined."
    end

    def _arithmetic_type_with_union(lhs, rhs = self)
      # NOTE: An arithmetic operation with ArrayType must not be executed!
      # NOTE: Binary operation with UnionType and ArrayType
      #       raises a runtime error.
      raise TypeError,
        "arithmetic-type of UnionType and ArrayType is not defined."
    end

    def _arithmetic_type_with_extended_big_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with ArrayType must not be executed!
      # NOTE: Binary operation with ExtendedBigIntType and ArrayType
      #       makes ExtendedBigIntType.
      lhs # NOTREACHED
    end

    def corresponding_signed_type
      self # NOTREACHED
    end

    def corresponding_unsigned_type
      self # NOTREACHED
    end

    private
    def create_name(base_type, length)
      "(#{base_type.name})[#{length ? length : ""}]"
    end

    def create_image(base_type, length)
      "(#{base_type.image})[#{length ? length : ""}]"
    end

    def create_brief_image(base_type, length)
      "(#{base_type.brief_image})[#{length ? length : ""}]"
    end
  end

  class ArrayTypeId < TypeId
    def initialize(base_type, length)
      super(create_value(base_type, length))
    end

    private
    def create_value(base_type, length)
      if length
        "#{base_type.brief_image}[#{length}]"
      else
        "#{base_type.brief_image}[]"
      end
    end
  end

  # == DESCRIPTION
  # Type of the `struct' or `union' data type.
  #
  # The ISO C99 standard specifies that the `struct' and array data type is an
  # aggregate type.
  # But the CompositeDataType is not an array type.
  class CompositeDataType < Type
    include Scopeable

    def initialize(type_table, name, type_declarations, members)
      super(type_table, name, type_declarations)

      @members = members
    end

    attr_reader :members

    def id
      subclass_responsibility
    end

    def image
      subclass_responsibility
    end

    def brief_image
      subclass_responsibility
    end

    def location
      subclass_responsibility
    end

    def bit_size
      @members.reduce(0) { |sum, memb| sum + memb.type.bit_size }
    end

    def bit_alignment
      bit_size
    end

    def aligned_bit_size
      @members.reduce(0) { |sum, memb| sum + memb.type.aligned_bit_size }
    end

    def real_type
      self
    end

    def base_type
      nil
    end

    def unqualify
      self
    end

    def incomplete?
      declarations.all? { |decl| decl.struct_declarations.nil? }
    end

    def compatible?(to_type)
      to_type.composite? &&
        @members.size == to_type.members.size &&
        @members.zip(to_type.members).all? { |lhs, rhs| lhs.compatible?(rhs) }
    end

    def coercible?(to_type)
      to_type.composite? &&
        @members.zip(to_type.members).all? { |lhs_member, rhs_member|
          rhs_member && lhs_member.type.coercible?(rhs_member.type)
        }
    end

    def scalar?
      false
    end

    def integer?
      false
    end

    def floating?
      false
    end

    def array?
      false
    end

    def pointer?
      false
    end

    def qualified?
      false
    end

    def function?
      false
    end

    def enum?
      false
    end

    def user?
      false
    end

    def void?
      false
    end

    def undeclared?
      @members.any? { |member| member.type.undeclared? }
    end

    def unresolved?
      @members.any? { |member| member.type.unresolved? }
    end

    def const?
      false
    end

    def volatile?
      false
    end

    def restrict?
      false
    end

    def bitfield?
      false
    end

    def signed?
      false
    end

    def explicitly_signed?
      false
    end

    def have_va_list?
      false
    end

    def return_type
      nil
    end

    def parameter_types
      []
    end

    def enumerators
      []
    end

    def length
      0
    end

    def impl_length
      0
    end

    def member_named(name)
      # FIXME: Should use the member name index.
      @members.find { |member| member.name == name }
    end

    def min_value
      0
    end

    def max_value
      0
    end

    def nil_value
      CompositeValue.new(@members.map { |member| member.type.nil_value })
    end

    def zero_value
      CompositeValue.new(@members.map { |member| member.type.zero_value })
    end

    def arbitrary_value
      CompositeValue.new(@members.map { |member| member.type.arbitrary_value })
    end

    def undefined_value
      CompositeValue.new(@members.map { |member| member.type.undefined_value })
    end

    def parameter_value
      CompositeValue.new(@members.map { |member| member.type.parameter_value })
    end

    def return_value
      CompositeValue.new(@members.map { |member| member.type.return_value })
    end

    def coerce_scalar_value(value)
      # NOTE: Cannot coerce scalar value into composite in C language.
      undefined_value # NOTREACHED
    end

    def coerce_array_value(value)
      # NOTE: Cannot coerce array value into composite in C language.
      undefined_value # NOTREACHED
    end

    def coerce_composite_value(value)
      values = @members.zip(value.values).map { |member, val|
        val ? val.coerce_to(member.type) : member.type.undefined_value
      }
      CompositeValue.new(values)
    end

    def integer_conversion_rank
      0 # NOTREACHED
    end

    def integer_promoted_type
      self # NOTREACHED
    end

    def argument_promoted_type
      self
    end

    def arithmetic_type_with(type)
      subclass_responsibility
    end

    def _arithmetic_type_with_undeclared(lhs, rhs = self)
      rhs.arithmetic_type_with(lhs)
    end

    def _arithmetic_type_with_unresolved(lhs, rhs = self)
      rhs.arithmetic_type_with(lhs)
    end

    def _arithmetic_type_with_void(lhs, rhs = self)
      rhs.arithmetic_type_with(lhs)
    end

    def _arithmetic_type_with_function(lhs, rhs = self)
      rhs.arithmetic_type_with(lhs)
    end

    def _arithmetic_type_with_char(lhs, rhs = self)
      # NOTE: An arithmetic operation with CompositeDataType must not be
      #       executed!
      # NOTE: Binary operation with `char' and CompositeDataType makes
      #       integer-promoted type of `char'.
      lhs.integer_promoted_type # NOTREACHED
    end

    def _arithmetic_type_with_signed_char(lhs, rhs = self)
      # NOTE: An arithmetic operation with CompositeDataType must not be
      #       executed!
      # NOTE: Binary operation with `signed char' and CompositeDataType makes
      #       integer-promoted type of `signed char'.
      lhs.integer_promoted_type # NOTREACHED
    end

    def _arithmetic_type_with_unsigned_char(lhs, rhs = self)
      # NOTE: An arithmetic operation with CompositeDataType must not be
      #       executed!
      # NOTE: Binary operation with `unsigned char' and CompositeDataType makes
      #       integer-promoted type of `unsigned char'.
      lhs.integer_promoted_type # NOTREACHED
    end

    def _arithmetic_type_with_short(lhs, rhs = self)
      # NOTE: An arithmetic operation with CompositeDataType must not be
      #       executed!
      # NOTE: Binary operation with `short' and CompositeDataType makes
      #       integer-promoted type of `short'.
      lhs.integer_promoted_type # NOTREACHED
    end

    def _arithmetic_type_with_signed_short(lhs, rhs = self)
      # NOTE: An arithmetic operation with CompositeDataType must not be
      #       executed!
      # NOTE: Binary operation with `signed short' and CompositeDataType makes
      #       integer-promoted type of `signed short'.
      lhs.integer_promoted_type # NOTREACHED
    end

    def _arithmetic_type_with_unsigned_short(lhs, rhs = self)
      # NOTE: An arithmetic operation with CompositeDataType must not be
      #       executed!
      # NOTE: Binary operation with `unsigned short' and CompositeDataType
      #       makes integer-promoted type of `unsigned short'.
      lhs.integer_promoted_type # NOTREACHED
    end

    def _arithmetic_type_with_short_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with CompositeDataType must not be
      #       executed!
      # NOTE: Binary operation with `short int' and CompositeDataType makes
      #       integer-promoted type of `short int'.
      lhs.integer_promoted_type # NOTREACHED
    end

    def _arithmetic_type_with_signed_short_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with CompositeDataType must not be
      #       executed!
      # NOTE: Binary operation with `signed short int' and CompositeDataType
      #       makes integer-promoted type of `signed short int'.
      lhs.integer_promoted_type # NOTREACHED
    end

    def _arithmetic_type_with_unsigned_short_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with CompositeDataType must not be
      #       executed!
      # NOTE: Binary operation with `unsigned short int' and CompositeDataType
      #       makes integer-promoted type of `unsigned short int'.
      lhs.integer_promoted_type # NOTREACHED
    end

    def _arithmetic_type_with_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with CompositeDataType must not be
      #       executed!
      # NOTE: Binary operation with `int' and CompositeDataType makes `int'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_signed(lhs, rhs = self)
      # NOTE: An arithmetic operation with CompositeDataType must not be
      #       executed!
      # NOTE: Binary operation with `signed' and CompositeDataType makes
      #       `signed'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_signed_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with CompositeDataType must not be
      #       executed!
      # NOTE: Binary operation with `signed int' and CompositeDataType makes
      #       `signed int'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_unsigned(lhs, rhs = self)
      # NOTE: An arithmetic operation with CompositeDataType must not be
      #       executed!
      # NOTE: Binary operation with `unsigned' and CompositeDataType makes
      #       `unsigned'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_unsigned_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with CompositeDataType must not be
      #       executed!
      # NOTE: Binary operation with `unsigned int' and CompositeDataType makes
      #       `unsigned int'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_long(lhs, rhs = self)
      # NOTE: An arithmetic operation with CompositeDataType must not be
      #       executed!
      # NOTE: Binary operation with `long' and CompositeDataType makes `long'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_signed_long(lhs, rhs = self)
      # NOTE: An arithmetic operation with CompositeDataType must not be
      #       executed!
      # NOTE: Binary operation with `signed long' and CompositeDataType makes
      #       `signed long'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_unsigned_long(lhs, rhs = self)
      # NOTE: An arithmetic operation with CompositeDataType must not be
      #       executed!
      # NOTE: Binary operation with `unsigned long' and CompositeDataType makes
      #       `unsigned long'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_long_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with CompositeDataType must not be
      #       executed!
      # NOTE: Binary operation with `long int' and CompositeDataType makes
      #       `long int'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_signed_long_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with CompositeDataType must not be
      #       executed!
      # NOTE: Binary operation with `signed long int' and CompositeDataType
      #       makes `signed long int'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_unsigned_long_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with CompositeDataType must not be
      #       executed!
      # NOTE: Binary operation with `unsigned long int' and CompositeDataType
      #       makes `unsigned long int'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_long_long(lhs, rhs = self)
      # NOTE: An arithmetic operation with CompositeDataType must not be
      #       executed!
      # NOTE: Binary operation with `long long' and CompositeDataType makes
      #       `long long'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_signed_long_long(lhs, rhs = self)
      # NOTE: An arithmetic operation with CompositeDataType must not be
      #       executed!
      # NOTE: Binary operation with `signed long long' and CompositeDataType
      #       makes `signed long long'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_unsigned_long_long(lhs, rhs = self)
      # NOTE: An arithmetic operation with CompositeDataType must not be
      #       executed!
      # NOTE: Binary operation with `unsigned long long' and CompositeDataType
      #       makes `unsigned long long'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_long_long_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with CompositeDataType must not be
      #       executed!
      # NOTE: Binary operation with `long long int' and CompositeDataType makes
      #       `long long int'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_signed_long_long_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with CompositeDataType must not be
      #       executed!
      # NOTE: Binary operation with `signed long long int' and
      #       CompositeDataType makes `signed long long int'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_unsigned_long_long_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with CompositeDataType must not be
      #       executed!
      # NOTE: Binary operation with `unsigned long long int' and
      #       CompositeDataType makes `unsigned long long int'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_float(lhs, rhs = self)
      # NOTE: An arithmetic operation with CompositeDataType must not be
      #       executed!
      # NOTE: Binary operation with `float' and CompositeDataType makes
      #       `float'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_double(lhs, rhs = self)
      # NOTE: An arithmetic operation with CompositeDataType must not be
      #       executed!
      # NOTE: Binary operation with `double' and CompositeDataType makes
      #       `double'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_long_double(lhs, rhs = self)
      # NOTE: An arithmetic operation with CompositeDataType must not be
      #       executed!
      # NOTE: Binary operation with `long double' and CompositeDataType makes
      #       `long double'.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_bitfield(lhs, rhs = self)
      # NOTE: An arithmetic operation with CompositeDataType must not be
      #       executed!
      # NOTE: Binary operation with BitfieldType and CompositeDataType makes
      #       integer-promoted type of BitfieldType.
      lhs.integer_promoted_type # NOTREACHED
    end

    def _arithmetic_type_with_enum(lhs, rhs = self)
      # NOTE: An arithmetic operation with CompositeDataType must not be
      #       executed!
      # NOTE: Binary operation with EnumType and CompositeDataType makes
      #       EnumType.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_pointer(lhs, rhs = self)
      # NOTE: An arithmetic operation with CompositeDataType must not be
      #       executed!
      # NOTE: Binary operation with PointerType and CompositeDataType makes
      #       PointerType.
      lhs # NOTREACHED
    end

    def _arithmetic_type_with_array(lhs, rhs = self)
      rhs.arithmetic_type_with(lhs)
    end

    def _arithmetic_type_with_struct(lhs, rhs = self)
      # NOTE: An arithmetic operation with CompositeDataType must not be
      #       executed!
      # NOTE: Binary operation with StructType and CompositeDataType raises a
      #       runtime error.
      raise TypeError,
        "arithmetic-type of StructType and CompositeDataType is not defined."
    end

    def _arithmetic_type_with_union(lhs, rhs = self)
      # NOTE: An arithmetic operation with CompositeDataType must not be
      #       executed!
      # NOTE: Binary operation with UnionType and CompositeDataType raises a
      #       runtime error.
      raise TypeError,
        "arithmetic-type of UnionType and CompositeDataType is not defined."
    end

    def _arithmetic_type_with_extended_big_int(lhs, rhs = self)
      # NOTE: An arithmetic operation with CompositeDataType must not be
      #       executed!
      # NOTE: Binary operation with ExtendedBigIntType and CompositeDataType
      #       makes ExtendedBigIntType.
      lhs # NOTREACHED
    end

    def corresponding_signed_type
      self # NOTREACHED
    end

    def corresponding_unsigned_type
      self # NOTREACHED
    end
  end

  class Member
    def initialize(name, type)
      @name = name
      @type = type
    end

    attr_reader :name
    attr_reader :type
  end

  class StructType < CompositeDataType
    def initialize(type_table, struct_type_declaration, members)
      super(type_table, struct_type_declaration.identifier.value,
            [struct_type_declaration], members)

      @image = struct_type_declaration.struct_specifier.to_s
      @location = struct_type_declaration.location
    end

    attr_accessor :image
    attr_accessor :location

    def id
      @id ||= StructTypeId.new(name)
    end

    def brief_image
      "struct #{name}"
    end

    def struct?
      true
    end

    def union?
      false
    end

    def arithmetic_type_with(type)
      type._arithmetic_type_with_struct(self)
    end
  end

  class StructTypeId < TypeId
    def initialize(name)
      super("struct #{name}")
    end
  end

  class UnionType < CompositeDataType
    # TODO: Must implement member overlapping semantics.

    def initialize(type_table, union_type_declaration, members)
      super(type_table, union_type_declaration.identifier.value,
            [union_type_declaration], members)

      @image = union_type_declaration.union_specifier.to_s
      @location = union_type_declaration.location
    end

    attr_accessor :image
    attr_accessor :location

    def id
      @id ||= UnionTypeId.new(name)
    end

    def brief_image
      "union #{name}"
    end

    def struct?
      false
    end

    def union?
      true
    end

    def arithmetic_type_with(type)
      type._arithmetic_type_with_union(self)
    end
  end

  class UnionTypeId < TypeId
    def initialize(name)
      super("union #{name}")
    end
  end

  class UserType < Type
    include Scopeable

    def initialize(type_table, typedef_declaration, base_type)
      super(type_table,
            typedef_declaration.identifier.value, [typedef_declaration])

      @location = typedef_declaration.location
      @base_type = base_type
    end

    attr_reader :location

    def id
      @id ||= UserTypeId.new(name)
    end

    extend Forwardable

    def_delegator :@base_type, :image
    def_delegator :@base_type, :brief_image
    def_delegator :@base_type, :bit_size
    def_delegator :@base_type, :bit_alignment
    def_delegator :@base_type, :real_type
    def_delegator :@base_type, :base_type
    def_delegator :@base_type, :unqualify
    def_delegator :@base_type, :incomplete?
    def_delegator :@base_type, :compatible?
    def_delegator :@base_type, :coercible?
    def_delegator :@base_type, :scalar?
    def_delegator :@base_type, :integer?
    def_delegator :@base_type, :floating?
    def_delegator :@base_type, :array?
    def_delegator :@base_type, :struct?
    def_delegator :@base_type, :union?
    def_delegator :@base_type, :pointer?
    def_delegator :@base_type, :qualified?
    def_delegator :@base_type, :function?
    def_delegator :@base_type, :enum?

    def user?
      true
    end

    def_delegator :@base_type, :void?
    def_delegator :@base_type, :undeclared?
    def_delegator :@base_type, :unresolved?
    def_delegator :@base_type, :const?
    def_delegator :@base_type, :volatile?
    def_delegator :@base_type, :restrict?
    def_delegator :@base_type, :bitfield?
    def_delegator :@base_type, :signed?
    def_delegator :@base_type, :explicitly_signed?
    def_delegator :@base_type, :have_va_list?
    def_delegator :@base_type, :return_type
    def_delegator :@base_type, :parameter_types
    def_delegator :@base_type, :enumerators
    def_delegator :@base_type, :length
    def_delegator :@base_type, :impl_length
    def_delegator :@base_type, :members
    def_delegator :@base_type, :member_named
    def_delegator :@base_type, :min_value
    def_delegator :@base_type, :max_value
    def_delegator :@base_type, :nil_value
    def_delegator :@base_type, :zero_value
    def_delegator :@base_type, :arbitrary_value
    def_delegator :@base_type, :undefined_value
    def_delegator :@base_type, :parameter_value
    def_delegator :@base_type, :return_value
    def_delegator :@base_type, :coerce_scalar_value
    def_delegator :@base_type, :coerce_array_value
    def_delegator :@base_type, :coerce_composite_value
    def_delegator :@base_type, :integer_conversion_rank
    def_delegator :@base_type, :integer_promoted_type
    def_delegator :@base_type, :argument_promoted_type

    def_delegator :@base_type, :arithmetic_type_with
    def_delegator :@base_type, :_arithmetic_type_with_undeclared
    def_delegator :@base_type, :_arithmetic_type_with_unresolved
    def_delegator :@base_type, :_arithmetic_type_with_void
    def_delegator :@base_type, :_arithmetic_type_with_function
    def_delegator :@base_type, :_arithmetic_type_with_char
    def_delegator :@base_type, :_arithmetic_type_with_signed_char
    def_delegator :@base_type, :_arithmetic_type_with_unsigned_char
    def_delegator :@base_type, :_arithmetic_type_with_short
    def_delegator :@base_type, :_arithmetic_type_with_signed_short
    def_delegator :@base_type, :_arithmetic_type_with_unsigned_short
    def_delegator :@base_type, :_arithmetic_type_with_short_int
    def_delegator :@base_type, :_arithmetic_type_with_signed_short_int
    def_delegator :@base_type, :_arithmetic_type_with_unsigned_short_int
    def_delegator :@base_type, :_arithmetic_type_with_int
    def_delegator :@base_type, :_arithmetic_type_with_signed
    def_delegator :@base_type, :_arithmetic_type_with_signed_int
    def_delegator :@base_type, :_arithmetic_type_with_unsigned
    def_delegator :@base_type, :_arithmetic_type_with_unsigned_int
    def_delegator :@base_type, :_arithmetic_type_with_long
    def_delegator :@base_type, :_arithmetic_type_with_signed_long
    def_delegator :@base_type, :_arithmetic_type_with_unsigned_long
    def_delegator :@base_type, :_arithmetic_type_with_long_int
    def_delegator :@base_type, :_arithmetic_type_with_signed_long_int
    def_delegator :@base_type, :_arithmetic_type_with_unsigned_long_int
    def_delegator :@base_type, :_arithmetic_type_with_long_long
    def_delegator :@base_type, :_arithmetic_type_with_signed_long_long
    def_delegator :@base_type, :_arithmetic_type_with_unsigned_long_long
    def_delegator :@base_type, :_arithmetic_type_with_long_long_int
    def_delegator :@base_type, :_arithmetic_type_with_signed_long_long_int
    def_delegator :@base_type, :_arithmetic_type_with_unsigned_long_long_int
    def_delegator :@base_type, :_arithmetic_type_with_float
    def_delegator :@base_type, :_arithmetic_type_with_double
    def_delegator :@base_type, :_arithmetic_type_with_long_double
    def_delegator :@base_type, :_arithmetic_type_with_bitfield
    def_delegator :@base_type, :_arithmetic_type_with_enum
    def_delegator :@base_type, :_arithmetic_type_with_pointer
    def_delegator :@base_type, :_arithmetic_type_with_array
    def_delegator :@base_type, :_arithmetic_type_with_struct
    def_delegator :@base_type, :_arithmetic_type_with_union
    def_delegator :@base_type, :_arithmetic_type_with_extended_big_int

    def_delegator :@base_type, :corresponding_signed_type
    def_delegator :@base_type, :corresponding_unsigned_type
  end

  class UserTypeId < TypeId
    def initialize(name)
      super("typedef #{name}")
    end
  end

  # NOTE: ParameterType is a decorator which attaches a parameter name to other
  #       types.
  class ParameterType < Type
    include Scopeable

    def initialize(type_table, type, declaration_or_definition = nil)
      super(type_table, type.name, type.declarations)

      @type = type

      if declaration_or_definition
        if declarator = declaration_or_definition.declarator
          identifier = declarator.identifier
        end
      end

      @param_name = identifier ? identifier.value : ""
      @declaration_or_definition = declaration_or_definition
    end

    attr_reader :param_name

    extend Forwardable

    def_delegator :@type, :id

    def location
      @declaration_or_definition ?
        @declaration_or_definition.location : @type.location
    end

    def_delegator :@type, :image
    def_delegator :@type, :brief_image
    def_delegator :@type, :bit_size
    def_delegator :@type, :bit_alignment

    def real_type
      ParameterType.new(type_table, @type.real_type,
                        @declaration_or_definition)
    end

    def_delegator :@type, :base_type
    def_delegator :@type, :unqualify

    def_delegator :@type, :incomplete?
    def_delegator :@type, :compatible?
    def_delegator :@type, :coercible?

    def parameter?
      true
    end

    def_delegator :@type, :scalar?
    def_delegator :@type, :integer?
    def_delegator :@type, :floating?
    def_delegator :@type, :array?
    def_delegator :@type, :struct?
    def_delegator :@type, :union?
    def_delegator :@type, :pointer?
    def_delegator :@type, :qualified?
    def_delegator :@type, :function?
    def_delegator :@type, :enum?
    def_delegator :@type, :user?
    def_delegator :@type, :void?
    def_delegator :@type, :undeclared?
    def_delegator :@type, :unresolved?
    def_delegator :@type, :const?
    def_delegator :@type, :volatile?
    def_delegator :@type, :restrict?
    def_delegator :@type, :bitfield?
    def_delegator :@type, :signed?
    def_delegator :@type, :explicitly_signed?
    def_delegator :@type, :have_va_list?
    def_delegator :@type, :return_type
    def_delegator :@type, :parameter_types
    def_delegator :@type, :enumerators
    def_delegator :@type, :length
    def_delegator :@type, :impl_length
    def_delegator :@type, :members
    def_delegator :@type, :member_named
    def_delegator :@type, :min_value
    def_delegator :@type, :max_value
    def_delegator :@type, :nil_value
    def_delegator :@type, :zero_value
    def_delegator :@type, :arbitrary_value
    def_delegator :@type, :undefined_value
    def_delegator :@type, :parameter_value
    def_delegator :@type, :return_value
    def_delegator :@type, :coerce_scalar_value
    def_delegator :@type, :coerce_array_value
    def_delegator :@type, :coerce_composite_value
    def_delegator :@type, :integer_conversion_rank
    def_delegator :@type, :integer_promoted_type
    def_delegator :@type, :argument_promoted_type

    def_delegator :@type, :arithmetic_type_with
    def_delegator :@type, :_arithmetic_type_with_undeclared
    def_delegator :@type, :_arithmetic_type_with_unresolved
    def_delegator :@type, :_arithmetic_type_with_void
    def_delegator :@type, :_arithmetic_type_with_function
    def_delegator :@type, :_arithmetic_type_with_char
    def_delegator :@type, :_arithmetic_type_with_signed_char
    def_delegator :@type, :_arithmetic_type_with_unsigned_char
    def_delegator :@type, :_arithmetic_type_with_short
    def_delegator :@type, :_arithmetic_type_with_signed_short
    def_delegator :@type, :_arithmetic_type_with_unsigned_short
    def_delegator :@type, :_arithmetic_type_with_short_int
    def_delegator :@type, :_arithmetic_type_with_signed_short_int
    def_delegator :@type, :_arithmetic_type_with_unsigned_short_int
    def_delegator :@type, :_arithmetic_type_with_int
    def_delegator :@type, :_arithmetic_type_with_signed
    def_delegator :@type, :_arithmetic_type_with_signed_int
    def_delegator :@type, :_arithmetic_type_with_unsigned
    def_delegator :@type, :_arithmetic_type_with_unsigned_int
    def_delegator :@type, :_arithmetic_type_with_long
    def_delegator :@type, :_arithmetic_type_with_signed_long
    def_delegator :@type, :_arithmetic_type_with_unsigned_long
    def_delegator :@type, :_arithmetic_type_with_long_int
    def_delegator :@type, :_arithmetic_type_with_signed_long_int
    def_delegator :@type, :_arithmetic_type_with_unsigned_long_int
    def_delegator :@type, :_arithmetic_type_with_long_long
    def_delegator :@type, :_arithmetic_type_with_signed_long_long
    def_delegator :@type, :_arithmetic_type_with_unsigned_long_long
    def_delegator :@type, :_arithmetic_type_with_long_long_int
    def_delegator :@type, :_arithmetic_type_with_signed_long_long_int
    def_delegator :@type, :_arithmetic_type_with_unsigned_long_long_int
    def_delegator :@type, :_arithmetic_type_with_float
    def_delegator :@type, :_arithmetic_type_with_double
    def_delegator :@type, :_arithmetic_type_with_long_double
    def_delegator :@type, :_arithmetic_type_with_bitfield
    def_delegator :@type, :_arithmetic_type_with_enum
    def_delegator :@type, :_arithmetic_type_with_pointer
    def_delegator :@type, :_arithmetic_type_with_array
    def_delegator :@type, :_arithmetic_type_with_struct
    def_delegator :@type, :_arithmetic_type_with_union
    def_delegator :@type, :_arithmetic_type_with_extended_big_int

    def_delegator :@type, :corresponding_signed_type
    def_delegator :@type, :corresponding_unsigned_type
  end

  class ExtendedBigIntType < IntegerType
    def initialize(type_table)
      super(type_table, "__adlint__extended_bigint_t", 256, 256, true, true)
    end

    def id
      @id ||= TypeId.new(name)
    end

    def incomplete?
      false
    end

    def pointer?
      false
    end

    def enum?
      false
    end

    def bitfield?
      false
    end

    def enumerators
      []
    end

    def integer_conversion_rank
      # NOTE: The ISO C99 standard saids;
      #
      # 6.3.1 Arithmetic operands
      # 6.3.1.1 Boolean, characters, and integers
      #
      # 1 Every integer type has an integer conversion rank defined as follows:
      #
      #     -- No two signed integer types shall have the same rank, even if
      #        they have the same representation.
      #     -- The rank of a signed integer type shall be greater than the rank
      #        of any signed integer type with less precision.
      #     -- The rank of long long int shall be greater than the rank of long
      #        int, which shall be greater than the rank of int, which shall be
      #        greater than the rank of short int, which shall be greater than
      #        the rank of signed char.
      #     -- The rank of any unsigned integer type shall equal the rank of
      #        the corresponding signed integer type, if any.
      #     -- The rank of any standard integer type shall be greater than the
      #        rank of any extended integer type with the same width.
      #     -- The rank of char shall equal the rank of signed char and
      #        unsigned char.
      #     -- The rank of _Bool shall be less than the rank of all other
      #        standard integer types.
      #     -- The rank of any enumerated type shall equal the rank of the
      #        compatible integer type.
      #     -- The rank of any extended signed integer type relative to another
      #        extended signed integer type with the same precision is
      #        implementation-defined, but still subject to the other rules for
      #        determining the integer conversion rank.
      #     -- For all integer types T1, T2, and T3, if T1 has greater rank
      #        than T2 and T2 has greater rank than T3, then T1 has greater
      #        rank than T3.
      -2
    end

    def integer_promoted_type
      # NOTE: ExtendedBigIntType is very big integer.
      #       So, it is not compatible with int or unsigned int.
      self
    end

    def arithmetic_type_with(type)
      type._arithmetic_type_with_extended_big_int(self)
    end

    def corresponding_signed_type
      self
    end

    def corresponding_unsigned_type
      self
    end
  end

  class StandardTypeCatalog
    def initialize(type_table)
      @hash = {}

      @char_type = install(CharType.new(type_table))
      @signed_char_type = install(SignedCharType.new(type_table))
      @unsigned_char_type = install(UnsignedCharType.new(type_table))

      @short_type = install(ShortType.new(type_table))
      @signed_short_type = install(SignedShortType.new(type_table))
      @unsigned_short_type = install(UnsignedShortType.new(type_table))
      @short_int_type = install(ShortIntType.new(type_table))
      @signed_short_int_type = install(SignedShortIntType.new(type_table))
      @unsigned_short_int_type = install(UnsignedShortIntType.new(type_table))

      @int_type = install(IntType.new(type_table))
      @signed_type = install(SignedType.new(type_table))
      @signed_int_type = install(SignedIntType.new(type_table))
      @unsigned_type = install(UnsignedType.new(type_table))
      @unsigned_int_type = install(UnsignedIntType.new(type_table))

      @long_type = install(LongType.new(type_table))
      @signed_long_type = install(SignedLongType.new(type_table))
      @unsigned_long_type = install(UnsignedLongType.new(type_table))
      @long_int_type = install(LongIntType.new(type_table))
      @signed_long_int_type = install(SignedLongIntType.new(type_table))
      @unsigned_long_int_type = install(UnsignedLongIntType.new(type_table))

      @long_long_type = install(LongLongType.new(type_table))
      @signed_long_long_type = install(SignedLongLongType.new(type_table))
      @unsigned_long_long_type = install(UnsignedLongLongType.new(type_table))
      @long_long_int_type = install(LongLongIntType.new(type_table))
      @signed_long_long_int_type =
        install(SignedLongLongIntType.new(type_table))
      @unsigned_long_long_int_type =
        install(UnsignedLongLongIntType.new(type_table))

      @float_type = install(FloatType.new(type_table))
      @double_type = install(DoubleType.new(type_table))
      @long_double_type = install(LongDoubleType.new(type_table))

      @void_type = install(VoidType.new(type_table))
    end

    attr_reader :char_type
    attr_reader :signed_char_type
    attr_reader :unsigned_char_type
    attr_reader :short_type
    attr_reader :signed_short_type
    attr_reader :unsigned_short_type
    attr_reader :short_int_type
    attr_reader :signed_short_int_type
    attr_reader :unsigned_short_int_type
    attr_reader :int_type
    attr_reader :signed_type
    attr_reader :signed_int_type
    attr_reader :unsigned_type
    attr_reader :unsigned_int_type
    attr_reader :long_type
    attr_reader :signed_long_type
    attr_reader :unsigned_long_type
    attr_reader :long_int_type
    attr_reader :signed_long_int_type
    attr_reader :unsigned_long_int_type
    attr_reader :long_long_type
    attr_reader :signed_long_long_type
    attr_reader :unsigned_long_long_type
    attr_reader :long_long_int_type
    attr_reader :signed_long_long_int_type
    attr_reader :unsigned_long_long_int_type
    attr_reader :float_type
    attr_reader :double_type
    attr_reader :long_double_type
    attr_reader :void_type

    def lookup_by_type_specifiers(type_specifiers)
      type_name = type_specifiers.map { |ts| ts.to_s }.sort.join(" ")
      @hash[type_name]
    end

    def all_types
      @hash.each_value
    end

    private
    def install(type)
      @hash[type.name.split(" ").sort.join(" ")] = type
    end
  end

  class TypeTable
    include StandardTypeCatalogAccessor

    def initialize
      @type_stack = [{}]
      @scope_stack = [GlobalScope.new]
      @standard_type_catalog = StandardTypeCatalog.new(self)
      @all_type_names = Set.new

      install_standard_types
    end

    attr_reader :standard_type_catalog
    attr_reader :all_type_names

    def undeclared_type
      @undeclared_type ||= UndeclaredType.new(self)
    end

    def unresolved_type
      @unresolved_type ||= UnresolvedType.new(self)
    end

    def wchar_type
      lookup(UserTypeId.new("wchar_t")) or int_type
    end

    def array_type(base_type, length = nil)
      ArrayType.new(self, base_type, length)
    end

    def function_type(return_type, parameter_types, have_va_list = false)
      FunctionType.new(self, return_type, parameter_types, have_va_list)
    end

    def builtin_function_type
      function_type(undeclared_type, [undeclared_type])
    end

    def bitfield_type(base_type, field_width)
      BitfieldType.new(self, base_type, field_width)
    end

    def pointer_type(base_type)
      PointerType.new(self, base_type)
    end

    def qualified_type(base_type, *cvr_qualifiers)
      QualifiedType.new(self, base_type, *cvr_qualifiers)
    end

    def lookup(type_id)
      @type_stack.reverse_each do |hash|
        if type = hash[type_id]
          return type
        end
      end
      nil
    end

    def lookup_standard_type(name_str)
      @standard_type_catalog.lookup_by_name(name_str)
    end

    def install(type)
      @type_stack.last[type.id] = type
      @all_type_names.add(type.name)
      type
    end

    def enter_scope
      @type_stack.push({})
      @scope_stack.push(Scope.new(@scope_stack.size))
    end

    def leave_scope
      @type_stack.pop
      @scope_stack.pop
    end

    def lookup_or_install_type(type_qualifiers, type_specifiers, declarator,
                               interpreter = nil)
      case first_type_specifier = type_specifiers.first
      when TypeofTypeSpecifier
        if type_name = first_type_specifier.type_name
          return type_name.type
        else
          if interpreter
            object = interpreter.execute(first_type_specifier.expression,
                                         InterpreterOptions::QUIET)
            return object.type
          else
            return nil
          end
        end
      else
        unless base_type = lookup_type(type_specifiers, interpreter)
          case first_type_specifier
          when StructSpecifier
            base_type = install_struct_type(
              PseudoStructTypeDeclaration.new(first_type_specifier))
          when UnionSpecifier
            base_type = install_union_type(
              PseudoUnionTypeDeclaration.new(first_type_specifier))
          when EnumSpecifier
            base_type = install_enum_type(
              PseudoEnumTypeDeclaration.new(first_type_specifier))
          else
            return nil
          end
        end
        qualify_type(base_type, type_qualifiers, declarator, interpreter)
      end
    end

    def lookup_type(type_specifiers, interpreter = nil)
      lookup(create_type_id(type_specifiers, interpreter))
    end

    def lookup_function_type(function_definition, interpreter = nil)
      if declaration_specifiers = function_definition.declaration_specifiers
        type = lookup_or_install_type(declaration_specifiers.type_qualifiers,
                                      declaration_specifiers.type_specifiers,
                                      function_definition.declarator,
                                      interpreter)
      else
        type = lookup_or_install_type([], [], function_definition.declarator,
                                      interpreter)
      end

      return nil unless type

      parameter_types =
        function_definition.parameter_definitions.map { |pdef| pdef.type }

      function_type(type.return_type, parameter_types, type.have_va_list?)
    end

    def lookup_parameter_type(declaration_or_definition, interpreter = nil)
      declarator = declaration_or_definition.declarator
      declaration_specifiers = declaration_or_definition.declaration_specifiers

      if declaration_specifiers
        type_qualifiers = declaration_specifiers.type_qualifiers
        type_specifiers = declaration_specifiers.type_specifiers
      else
        type_qualifiers = []
        type_specifiers = []
      end

      type = lookup_or_install_type(type_qualifiers, type_specifiers,
                                    declarator, interpreter)
      return nil unless type

      type = pointer_type(type) if type.function?

      ParameterType.new(self, type, declaration_or_definition)
    end

    def install_struct_type(struct_type_declaration)
      type_id = StructTypeId.new(struct_type_declaration.identifier.value)

      if type = lookup(type_id) and type.scope == current_scope
        if struct_type_declaration.struct_declarations
          rewrite_struct_type(type, struct_type_declaration)
          return type
        end
      end

      type = StructType.new(self, struct_type_declaration, [])
      type.scope = current_scope
      install(type)
      if struct_type_declaration.struct_declarations
        rewrite_struct_type(type, struct_type_declaration)
      end
      type
    end

    def install_union_type(union_type_declaration)
      type_id = UnionTypeId.new(union_type_declaration.identifier.value)

      if type = lookup(type_id) and type.scope == current_scope
        if union_type_declaration.struct_declarations
          rewrite_union_type(type, union_type_declaration)
          return type
        end
      end

      type = UnionType.new(self, union_type_declaration, [])
      type.scope = current_scope
      install(type)
      if union_type_declaration.struct_declarations
        rewrite_union_type(type, union_type_declaration)
      end
      type
    end

    def install_enum_type(enum_type_declaration)
      type_id = EnumTypeId.new(enum_type_declaration.identifier.value)

      if type = lookup(type_id) and type.scope == current_scope
        if enum_type_declaration.enumerators
          rewrite_enum_type(type, enum_type_declaration)
          return type
        end
      end

      type = EnumType.new(self, enum_type_declaration)
      type.scope = current_scope
      install(type)
      if enum_type_declaration.enumerators
        rewrite_enum_type(type, enum_type_declaration)
      end
      type
    end

    def install_user_type(typedef_declaration)
      base_type = lookup_or_install_type(typedef_declaration.type_qualifiers,
                                         typedef_declaration.type_specifiers,
                                         typedef_declaration.declarator)
      type = UserType.new(self, typedef_declaration, base_type)
      type.scope = current_scope
      install(type)
      type
    end

    private
    def create_type_id(type_specifiers, interpreter)
      case type_specifiers.first
      when StandardTypeSpecifier
        type_id = create_standard_type_id(type_specifiers)
      when TypedefTypeSpecifier
        type_id = create_user_type_id(type_specifiers)
      when StructSpecifier
        type_id = create_struct_type_id(type_specifiers)
      when UnionSpecifier
        type_id = create_union_type_id(type_specifiers)
      when EnumSpecifier
        type_id = create_enum_type_id(type_specifiers)
      when NilClass
        type_id = IntTypeId.new
      else
        raise TypeError
      end
      type_id
    end

    def create_standard_type_id(type_specifiers)
      type = @standard_type_catalog.lookup_by_type_specifiers(type_specifiers)
      return type.id if type

      IntTypeId.new
    end

    def create_user_type_id(type_specifiers)
      UserTypeId.new(type_specifiers.first.identifier.value)
    end

    def create_struct_type_id(type_specifiers)
      StructTypeId.new(type_specifiers.first.identifier.value)
    end

    def create_union_type_id(type_specifiers)
      UnionTypeId.new(type_specifiers.first.identifier.value)
    end

    def create_enum_type_id(type_specifiers)
      EnumTypeId.new(type_specifiers.first.identifier.value)
    end

    def qualify_type(type, type_qualifiers, declarator, interpreter = nil)
      type_qualifiers.each do |token|
        case token.type
        when :CONST
          type = qualified_type(type, :const)
        when :VOLATILE
          type = qualified_type(type, :volatile)
        when :RESTRICT
          # TODO: Should support C99 features.
        end
      end

      return type unless declarator

      decl_interp = DeclaratorInterpreter.new(self, interpreter, type)
      declarator.accept(decl_interp)
    end

    def create_members(struct_declarations)
      members = []
      struct_declarations.each do |struct_declaration|
        struct_declaration.items.each do |item|
          members.push(Member.new(item.identifier.value, item.type))
        end
      end
      members
    end

    def rewrite_struct_type(struct_type, struct_type_declaration)
      struct_type.declarations.push(struct_type_declaration)
      struct_type.image = struct_type_declaration.struct_specifier.to_s
      struct_type.location = struct_type_declaration.location
      struct_type.members.replace(
        create_members(struct_type_declaration.struct_declarations))
    end

    def rewrite_union_type(union_type, union_type_declaration)
      union_type.declarations.push(union_type_declaration)
      union_type.image = union_type_declaration.union_specifier.to_s
      union_type.location = union_type_declaration.location
      union_type.members.replace(
        create_members(union_type_declaration.struct_declarations))
    end

    def rewrite_enum_type(enum_type, enum_type_declaration)
      enum_type.declarations.push(enum_type_declaration)
      enum_type.image = enum_type_declaration.enum_specifier.to_s
      enum_type.location = enum_type_declaration.location
    end

    def current_scope
      @scope_stack.last
    end

    def install_standard_types
      @standard_type_catalog.all_types.each { |type| install(type) }
      install(ExtendedBigIntType.new(self))
    end
  end

  class DeclaratorInterpreter
    def initialize(type_table, interpreter, type)
      @type_table = type_table
      @interpreter = interpreter
      @type = type
    end

    def visit_identifier_declarator(node)
      @type = qualify_by_pointer(@type, node)
    end

    def visit_grouped_declarator(node)
      @type = qualify_by_pointer(@type, node)
      @type = node.base.accept(self)
    end

    def visit_array_declarator(node)
      @type = qualify_by_pointer(@type, node)
      if size_expression = node.size_expression
        if array_size = evaluate_size_expression(size_expression)
          @type = @type_table.array_type(@type, array_size)
        else
          return @type_table.unresolved_type
        end
      else
        @type = @type_table.array_type(@type)
      end
      @type = node.base.accept(self)
    end

    def visit_ansi_function_declarator(node)
      @type = qualify_by_pointer(@type, node)
      parameter_types = lookup_parameter_types(node)
      if parameter_types.include?(nil)
        return @type_table.unresolved_type
      else
        @type = @type_table.function_type(@type, parameter_types,
                                        node.parameter_type_list.have_va_list?)
        @type = node.base.accept(self)
      end
      @type
    end

    def visit_kandr_function_declarator(node)
      @type = qualify_by_pointer(@type, node)
      @type = @type_table.function_type(@type, [])
      @type = node.base.accept(self)
    end

    def visit_abbreviated_function_declarator(node)
      @type = qualify_by_pointer(@type, node)
      @type = @type_table.function_type(@type, [])
      @type = node.base.accept(self)
    end

    def visit_pointer_abstract_declarator(node)
      @type = qualify_by_pointer(@type, node)
      @type = node.base.accept(self) if node.base
      @type
    end

    def visit_grouped_abstract_declarator(node)
      @type = qualify_by_pointer(@type, node)
      @type = node.base.accept(self)
    end

    def visit_array_abstract_declarator(node)
      @type = qualify_by_pointer(@type, node)
      if size_expression = node.size_expression
        if array_size = evaluate_size_expression(size_expression)
          @type = @type_table.array_type(@type, array_size)
        else
          return @type_table.unresolved_type
        end
      else
        @type = @type_table.array_type(@type)
      end
      @type = node.base.accept(self) if node.base
      @type
    end

    def visit_function_abstract_declarator(node)
      @type = qualify_by_pointer(@type, node)
      parameter_types = lookup_parameter_types(node)
      if parameter_types.include?(nil)
        return @type_table.unresolved_type
      else
        @type = @type_table.function_type(@type, parameter_types)
        @type = node.base.accept(self) if node.base
      end
      @type
    end

    private
    def qualify_by_pointer(type, declarator)
      if declarator.pointer
        declarator.pointer.each do |token|
          case token.type
          when "*"
            type = @type_table.pointer_type(type)
          when :CONST
            type = @type_table.qualified_type(type, :const)
          when :VOLATILE
            type = @type_table.qualified_type(type, :volatile)
          when :RESTRICT
            # TODO: Should support C99 features.
          end
        end
      end
      type
    end

    def lookup_parameter_types(declarator, interpreter = nil)
      if parameter_type_list = declarator.parameter_type_list
        parameter_type_list.parameters.map do |parameter_declaration|
          @type_table.lookup_parameter_type(parameter_declaration,
                                            @interpreter)
        end
      else
        []
      end
    end

    def evaluate_size_expression(size_expression)
      if @interpreter
        object = @interpreter.execute(size_expression)
        if object.variable? && object.value.scalar?
          size = object.value.unique_sample
        end
        size ||= 0
      else
        if size_expression.object_specifiers.empty?
          object = Interpreter.new(@type_table).execute(size_expression)
          if object.variable? && object.value.scalar?
            size = object.value.unique_sample
          end
          size ||= 0
        else
          size = nil
        end
      end
      size ? size.to_i : nil
    end
  end

end
end
