# Message detection classes for cross module analysis.
#
# Author::    Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>
# Copyright:: Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
# License::   GPLv3+: GNU General Public License version 3 or later
#
# Owner::     Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>

#--
#     ___    ____  __    ___   _________
#    /   |  / _  |/ /   / / | / /__  __/           Source Code Static Analyzer
#   / /| | / / / / /   / /  |/ /  / /                   AdLint - Advanced Lint
#  / __  |/ /_/ / /___/ / /|  /  / /
# /_/  |_|_____/_____/_/_/ |_/  /_/   Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
#
# This file is part of AdLint.
#
# AdLint is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# AdLint is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# AdLint.  If not, see <http://www.gnu.org/licenses/>.
#
#++

require "adlint/message"

module AdLint #:nodoc:
module Ld #:nodoc:

  class W0555 < PassiveMessageDetection
    def initialize(context)
      super
      context[:ld_function_traversal].on_definition += method(:check)
    end

    private
    def check(function)
      call_graph = @context[:ld_function_call_graph]
      if call_graph.indirect_callers_of(function).include?(function)
        W(:W0555, function.location)
      end
    end
  end

  class W0586 < PassiveMessageDetection
    def initialize(context)
      super
      context[:ld_typedef_traversal].listeners += method(:check)
    end

    private
    def check(typedef)
      mapping = @context[:ld_typedef_mapping]
      if mapping.lookup(typedef.name).any? { |td| td != typedef }
        W(:W0586, typedef.location, typedef.name)
      end
    end
  end

  class W0589 < PassiveMessageDetection
    def initialize(context)
      super
      context[:ld_function_traversal].on_definition += method(:check_function)
      context[:ld_variable_traversal].on_definition += method(:check_variable)
    end

    private
    def check_function(function)
      return unless function.extern?
      call_graph = @context[:ld_function_call_graph]
      direct_callers = call_graph.direct_callers_of(function)
      if direct_callers.size == 1 &&
          direct_callers.first.location.fpath == function.location.fpath
        W(:W0589, function.location,
          function.signature, direct_callers.first.signature)
      end
    end

    def check_variable(variable)
      return unless variable.extern?
      reference_graph = @context[:ld_variable_reference_graph]
      direct_referrers = reference_graph.direct_referrers_of(variable)
      if direct_referrers.size == 1 &&
          direct_referrers.first.location.fpath == variable.location.fpath
        W(:W0589, variable.location,
          variable.name, direct_referrers.first.signature)
      end
    end
  end

  class W0591 < PassiveMessageDetection
    def initialize(context)
      super
      context[:ld_function_traversal].on_definition += method(:check)
    end

    private
    def check(function)
      return unless function.extern?
      call_graph = @context[:ld_function_call_graph]
      direct_callers = call_graph.direct_callers_of(function)
      if !direct_callers.empty? &&
          direct_callers.all? { |f|
            f.location.fpath == function.location.fpath }
        W(:W0591, function.location, function.signature)
      end
    end
  end

  class W0593 < PassiveMessageDetection
    def initialize(context)
      super
      context[:ld_variable_traversal].on_definition += method(:check)
    end

    private
    def check(variable)
      return unless variable.extern?
      reference_graph = @context[:ld_variable_reference_graph]
      direct_referrers = reference_graph.direct_referrers_of(variable)
      if !direct_referrers.empty? &&
          direct_referrers.all? { |f|
            f.location.fpath == variable.location.fpath }
        W(:W0593, variable.location, variable.name)
      end
    end
  end

  class W0628 < PassiveMessageDetection
    def initialize(context)
      super
      context[:ld_function_traversal].on_definition += method(:check)
    end

    private
    def check(function)
      call_graph = @context[:ld_function_call_graph]
      unless function.name == "main"
        if call_graph.direct_callers_of(function).empty?
          W(:W0628, function.location, function.signature)
        end
      end
    end
  end

  class W0770 < PassiveMessageDetection
    # NOTE: W0770 may be duplicative when function declarations of the same
    #       name appears twice or more in the project.
    ensure_uniqueness_of :W0770

    def initialize(context)
      super
      context[:ld_function_traversal].on_definition += method(:check_function)
      context[:ld_variable_traversal].on_definition += method(:check_variable)
    end

    private
    def check_function(function)
      return unless function.extern?
      mapping = @context[:ld_function_mapping]
      declarations = mapping.lookup_function_declarations(function.name)
      if declarations.size > 1
        declarations.each do |declaration|
          W(:W0770, declaration.location, declaration.signature)
        end
      end
    end

    def check_variable(variable)
      return unless variable.extern?
      mapping = @context[:ld_variable_mapping]
      declarations = mapping.lookup_variable_declarations(variable.name)
      if declarations.size > 1
        declarations.each do |declaration|
          W(:W0770, declaration.location, declaration.name)
        end
      end
    end
  end

  class W0791 < PassiveMessageDetection
    # NOTE: W0791 may be duplicative when definitions of the same name appears
    #       twice or more in the project.
    ensure_uniqueness_of :W0791

    def initialize(context)
      super
      context[:ld_function_traversal].on_definition += method(:check_function)
      context[:ld_variable_traversal].on_definition += method(:check_variable)
    end

    private
    def check_function(function)
      return unless function.extern?
      mapping = @context[:ld_function_mapping]
      functions = mapping.lookup_functions(function.name)
      if functions.size > 1
        functions.each { |fun| W(:W0791, fun.location, fun.signature) }
      end
    end

    def check_variable(variable)
      return unless variable.extern?
      mapping = @context[:ld_variable_mapping]
      variables = mapping.lookup_variables(variable.name)
      if variables.size > 1
        variables.each { |var| W(:W0791, var.location, var.name) }
      end
    end
  end

  class W1037 < PassiveMessageDetection
    # NOTE: W1037 may be duplicative when declarations or definitions of the
    #       same name appears twice or more in the project.
    ensure_uniqueness_of :W1037

    def initialize(context)
      super
      function_traversal = context[:ld_function_traversal]
      function_traversal.on_declaration += method(:check_function)
      function_traversal.on_definition += method(:check_function)
      variable_traversal = context[:ld_variable_traversal]
      variable_traversal.on_declaration += method(:check_variable)
      variable_traversal.on_definition += method(:check_variable)
    end

    private
    def check_function(function)
      return unless function.extern?
      mapping = @context[:ld_function_mapping]
      defs_and_decls =
        (mapping.lookup_function_declarations(function.name) +
         mapping.lookup_functions(function.name)).select { |fun| fun.extern? }
      if defs_and_decls.any? { |fun| fun.signature != function.signature }
        defs_and_decls.each { |fun| W(:W1037, fun.location, fun.name) }
      end
    end

    def check_variable(variable)
      return unless variable.extern?
      mapping = @context[:ld_variable_mapping]
      defs_and_decls =
        (mapping.lookup_variable_declarations(variable.name) +
         mapping.lookup_variables(variable.name)).select { |var| var.extern? }
      if defs_and_decls.any? { |var| var.type != variable.type }
        defs_and_decls.each { |var| W(:W1037, var.location, var.name) }
      end
    end
  end

end
end
