# Lexical token classes.
#
# Author::    Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>
# Copyright:: Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
# License::   GPLv3+: GNU General Public License version 3 or later
#
# Owner::     Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>

#--
#     ___    ____  __    ___   _________
#    /   |  / _  |/ /   / / | / /__  __/           Source Code Static Analyzer
#   / /| | / / / / /   / /  |/ /  / /                   AdLint - Advanced Lint
#  / __  |/ /_/ / /___/ / /|  /  / /
# /_/  |_|_____/_____/_/_/ |_/  /_/   Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
#
# This file is part of AdLint.
#
# AdLint is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# AdLint is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# AdLint.  If not, see <http://www.gnu.org/licenses/>.
#
#++

module AdLint #:nodoc:

  # == DESCRIPTION
  # Token.
  class Token
    include Comparable

    # === DESCRIPTION
    # Constructs a token.
    #
    # === PARAMETER
    # _type_:: Symbol | String -- Type of the token.
    # _value_:: String -- String value of the token.
    # _location_:: Location -- Location of the token.
    # _type_hint_:: Symbol | String -- Hint of the token type.
    def initialize(type, value, location, type_hint = nil)
      @type, @value, @location = type, value, location
      @type_hint = type_hint
    end

    # === VALUE
    # Symbol | String -- Type of this token.
    attr_reader :type

    # === VALUE
    # String -- Value of this token.
    attr_reader :value

    # === VALUE
    # Location -- Location of this token.
    attr_reader :location

    # === VALUE
    # Symbol | String -- Hint of the type of this token.
    attr_reader :type_hint

    def replaced?
      false
    end

    def need_no_further_replacement?
      false
    end

    # === DESCRIPTION
    # Compares tokens.
    #
    # === PARAMETER
    # _rhs_:: Token -- Right-hand-side token.
    #
    # === RETURN VALUE
    # Integer -- Comparision result.
    def <=>(rhs)
      case rhs
      when Symbol, String
        @type <=> rhs
      when Token
        if (type_diff = @type <=> rhs.type) == 0
          if (value_diff = @value <=> rhs.value) == 0
            @location <=> rhs.location
          else
            value_diff
          end
        else
          type_diff
        end
      else
        raise TypeError
      end
    end

    def eql?(rhs_token)
      @type == rhs_token.type && @value == rhs_token.value &&
        @location == rhs_token.location
    end

    def hash
      [@type, @value, @location].hash
    end
  end

  class ReplacedToken < Token
    def initialize(type, value, location, type_hint = nil,
                   need_no_further_replacement = true)
      super(type, value, location, type_hint)
      @need_no_further_replacement = need_no_further_replacement
    end

    def need_no_further_replacement?
      @need_no_further_replacement
    end

    def replaced?
      true
    end

    def eql?(rhs_token)
      equal?(rhs_token)
    end

    def hash
      object_id
    end
  end

  # == DESCRIPTION
  # Location identifier of tokens.
  class Location
    # === Constructs a location identifier.
    #
    # Param:: _fpath_ (Pathname) Path name of the file contains the token.
    # Param:: _line_no_ (Integer) Line-no where the token appears.
    # Param:: _column_no_ (Integer) Column-no where the token appears.
    def initialize(fpath = nil, line_no = nil, column_no = nil,
                   appearance_column_no = column_no)
      @fpath, @line_no, @column_no = fpath, line_no, column_no
      @appearance_column_no = appearance_column_no
    end

    # === VALUE
    # Pathname -- Path name of the file contains this token.
    attr_reader :fpath

    # === VALUE
    # Integer -- Line-no where this token appears.
    attr_reader :line_no

    # === VALUE
    # Integer -- Column-no where this token appears.
    attr_reader :column_no

    attr_reader :appearance_column_no

    def ==(rhs)
      @fpath == rhs.fpath &&
        @line_no == rhs.line_no && @column_no == rhs.column_no
    end

    def eql?(rhs)
      self == rhs
    end

    def hash
      to_a.hash
    end

    # === DESCRIPTION
    # Converts this location identifier to an array representation.
    #
    # === RETURN VALUE
    # Array< Object > -- Array representation of this location identifier.
    def to_a
      [@fpath, @line_no, @column_no]
    end

    # === DESCRIPTION
    # Converts this location to a human readable string representation.
    #
    # === RETURN VALUE
    # String -- String representation of this location identifier.
    def to_s
      result = ""
      result += "#{@fpath}:" if @fpath
      result += "#{@line_no}:" if @line_no
      result += "#{@column_no}" if @column_no
      result
    end

    # === DESCRIPTION
    # Converts this location to debugging dump representation.
    #
    # === RETURN VALUE
    # String -- String representation of this location identifier.
    def inspect
      "#{@fpath ? @fpath : 'nil'}:" +
        "#{@line_no ? @line_no : 'nil'}:#{@column_no ? @column_no : 'nil'}"
    end
  end

  # == DESCRIPTION
  # Array of tokens.
  class TokenArray < Array; end

end
