/* aewm - An Exiguous Window Manager - vim:sw=4:ts=4:et
 *
 * Copyright 1998-2006 Decklin Foster <decklin@red-bean.com>. This
 * program is free software; please see LICENSE for details. */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <locale.h>
#include <X11/Xatom.h>
#include <X11/cursorfont.h>
#ifdef SHAPE
#include <X11/extensions/shape.h>
#endif
#include "aewm.h"
#include "atom.h"
#include "parser.h"

client_t *head;
int screen;
unsigned long ndesks = 1;
unsigned long cur_desk = 0;
#ifdef SHAPE
Bool shape;
int shape_event;
#endif

XFontStruct *font;
#ifdef X_HAVE_UTF8_STRING
XFontSet font_set;
#endif
#ifdef XFT
XftFont *xftfont;
XftColor xft_fg;
#endif

Colormap def_cmap;
XColor fg;
XColor bg;
XColor bd;
GC invert_gc;
GC string_gc;
GC border_gc;
Cursor map_curs;
Cursor move_curs;
Cursor resize_curs;

char *opt_font = DEF_FONT;
#ifdef XFT
char *opt_xftfont = DEF_XFTFONT;
#endif
char *opt_fg = DEF_FG;
char *opt_bg = DEF_BG;
char *opt_bd = DEF_BD;
char *opt_new1 = DEF_NEW1;
char *opt_new2 = DEF_NEW2;
char *opt_new3 = DEF_NEW3;
int opt_bw = DEF_BW;
int opt_pad = DEF_PAD;
int opt_map = 0;

static void read_config(char *);
static void setup(void);

int main(int argc, char **argv)
{
    int i;
    struct sigaction act;

    setlocale(LC_ALL, "");
    read_config(NULL);

    for (i = 1; i < argc; i++) {
        if ARG("config", "rc", 1) {
            read_config(argv[++i]);
        } else if ARG("font", "fn", 1) {
            opt_font = argv[++i];
#ifdef XFT
        } else if ARG("xftfont", "fa", 1) {
            opt_xftfont = argv[++i];
#endif
        } else if ARG("fgcolor", "fg", 1) {
            opt_fg = argv[++i];
        } else if ARG("bgcolor", "bg", 1) {
            opt_bg = argv[++i];
        } else if ARG("bdcolor", "bd", 1) {
            opt_bd = argv[++i];
        } else if ARG("bdwidth", "bw", 1) {
            opt_bw = atoi(argv[++i]);
        } else if ARG("padding", "p", 1) {
            opt_pad = atoi(argv[++i]);
        } else if ARG("map", "m", 0) {
            opt_map = 1;
        } else if ARG("nomap", "M", 0) {
            opt_map = 0;
        } else if ARG("new1", "1", 1) {
            opt_new1 = argv[++i];
        } else if ARG("new2", "2", 1) {
            opt_new2 = argv[++i];
        } else if ARG("new3", "3", 1) {
            opt_new3 = argv[++i];
        } else if ARG("version", "v",0) {
            printf("aewm: version " VERSION "\n");
            exit(0);
        } else if ARG("help", "h",0) {
            printf(USAGE);
            exit(0);
        } else {
            fprintf(stderr, "aewm: unknown option: '%s'\n" USAGE, argv[i]);
            exit(2);
        }
    }

    act.sa_handler = sig_handler;
    act.sa_flags = 0;
    sigaction(SIGTERM, &act, NULL);
    sigaction(SIGINT, &act, NULL);
    sigaction(SIGHUP, &act, NULL);
    sigaction(SIGCHLD, &act, NULL);

    setup();
    event_loop();
    return 0;
}

static void read_config(char *rcfile)
{
    FILE *rc;
    char buf[BUF_SIZE], token[BUF_SIZE], *p;

    if ((rc = open_rc(rcfile, "aewmrc"))) {
        while (get_rc_line(buf, sizeof buf, rc)) {
            p = buf;
            while (get_token(&p, token)) {
                if (strcmp(token, "font") == 0) {
                    if (get_token(&p, token))
                        opt_font = strdup(token);
#ifdef XFT
                } else if (strcmp(token, "xftfont") == 0) {
                    if (get_token(&p, token))
                        opt_xftfont = strdup(token);
#endif
                } else if (strcmp(token, "fgcolor") == 0) {
                    if (get_token(&p, token))
                        opt_fg = strdup(token);
                } else if (strcmp(token, "bgcolor") == 0) {
                    if (get_token(&p, token))
                        opt_bg = strdup(token);
                } else if (strcmp(token, "bdcolor") == 0) {
                    if (get_token(&p, token))
                        opt_bd = strdup(token);
                } else if (strcmp(token, "bdwidth") == 0) {
                    if (get_token(&p, token))
                        opt_bw = atoi(token);
                } else if (strcmp(token, "padding") == 0) {
                    if (get_token(&p, token))
                        opt_pad = atoi(token);
                } else if (strcmp(token, "map") == 0) {
                    if (get_token(&p, token))
                        opt_map = atoi(token);
                } else if (strcmp(token, "button1") == 0) {
                    if (get_token(&p, token))
                        opt_new1 = strdup(token);
                } else if (strcmp(token, "button2") == 0) {
                    if (get_token(&p, token))
                        opt_new2 = strdup(token);
                } else if (strcmp(token, "button3") == 0) {
                    if (get_token(&p, token))
                        opt_new3 = strdup(token);
                }
            }
        }
        fclose(rc);
    } else {
        fprintf(stderr, "aewm: rc file '%s' not found\n", rcfile);
    }
}

static void setup(void)
{
#ifdef X_HAVE_UTF8_STRING
    char **missing;
    char *def_str;
    int nmissing;
#endif
    XGCValues gv;
    XColor exact;
    XSetWindowAttributes sattr;
    XWindowAttributes attr;
#ifdef SHAPE
    int shape_err;
#endif
    Window qroot, qparent, *wins;
    unsigned int nwins, i;

    dpy = XOpenDisplay(NULL);

    if (!dpy) {
        fprintf(stderr, "aewm: can't open $DISPLAY '%s'\n", getenv("DISPLAY"));
        exit(1);
    }

    XSetErrorHandler(handle_xerror);
    screen = DefaultScreen(dpy);
    root = RootWindow(dpy, screen);

    map_curs = XCreateFontCursor(dpy, XC_dotbox);
    move_curs = XCreateFontCursor(dpy, XC_fleur);
    resize_curs = XCreateFontCursor(dpy, XC_sizing);

    def_cmap = DefaultColormap(dpy, screen);
    XAllocNamedColor(dpy, def_cmap, opt_fg, &fg, &exact);
    XAllocNamedColor(dpy, def_cmap, opt_bg, &bg, &exact);
    XAllocNamedColor(dpy, def_cmap, opt_bd, &bd, &exact);

    font = XLoadQueryFont(dpy, opt_font);
    if (!font) {
        fprintf(stderr, "aewm: font '%s' not found\n", opt_font);
        exit(1);
    }

#ifdef X_HAVE_UTF8_STRING
    font_set = XCreateFontSet(dpy, opt_font, &missing, &nmissing, &def_str);
#endif

#ifdef XFT
    xft_fg.color.red = fg.red;
    xft_fg.color.green = fg.green;
    xft_fg.color.blue = fg.blue;
    xft_fg.color.alpha = 0xffff;
    xft_fg.pixel = fg.pixel;

    xftfont = XftFontOpenName(dpy, DefaultScreen(dpy), opt_xftfont);
    if (!xftfont) {
        fprintf(stderr, "aewm: Xft font '%s' not found\n", opt_font);
        exit(1);
    }
#endif

    gv.function = GXcopy;
    gv.foreground = fg.pixel;
    gv.font = font->fid;
    string_gc = XCreateGC(dpy, root, GCFunction|GCForeground|GCFont, &gv);

    gv.foreground = bd.pixel;
    gv.line_width = opt_bw;
    border_gc = XCreateGC(dpy, root,
        GCFunction|GCForeground|GCLineWidth, &gv);

    gv.function = GXinvert;
    gv.subwindow_mode = IncludeInferiors;
    invert_gc = XCreateGC(dpy, root,
        GCFunction|GCSubwindowMode|GCLineWidth|GCFont, &gv);

    utf8_string = XInternAtom(dpy, "UTF8_STRING", False);
    wm_protos = XInternAtom(dpy, "WM_PROTOCOLS", False);
    wm_delete = XInternAtom(dpy, "WM_DELETE_WINDOW", False);
    wm_state = XInternAtom(dpy, "WM_STATE", False);
    wm_change_state = XInternAtom(dpy, "WM_CHANGE_STATE", False);
    net_supported = XInternAtom(dpy, "_NET_SUPPORTED", False);
    net_cur_desk = XInternAtom(dpy, "_NET_CURRENT_DESKTOP", False);
    net_num_desktops = XInternAtom(dpy, "_NET_NUMBER_OF_DESKTOPS", False);
    net_client_list = XInternAtom(dpy, "_NET_CLIENT_LIST", False);
    net_client_stack = XInternAtom(dpy, "_NET_CLIENT_LIST_STACKING", False);
    net_active_window = XInternAtom(dpy, "_NET_ACTIVE_WINDOW", False);
    net_close_window = XInternAtom(dpy, "_NET_CLOSE_WINDOW", False);
    net_wm_name = XInternAtom(dpy, "_NET_WM_NAME", False);
    net_wm_desktop = XInternAtom(dpy, "_NET_WM_DESKTOP", False);
    net_wm_state = XInternAtom(dpy, "_NET_WM_STATE", False);
    net_wm_state_shaded = XInternAtom(dpy, "_NET_WM_STATE_SHADED", False);
    net_wm_state_mv = XInternAtom(dpy, "_NET_WM_STATE_MAXIMIZED_VERT", False);
    net_wm_state_mh = XInternAtom(dpy, "_NET_WM_STATE_MAXIMIZED_HORZ", False);
    net_wm_strut = XInternAtom(dpy, "_NET_WM_STRUT", False);
    net_wm_strut_partial = XInternAtom(dpy, "_NET_WM_STRUT_PARTIAL", False);
    net_wm_wintype = XInternAtom(dpy, "_NET_WM_WINDOW_TYPE", False);
    net_wm_type_dock = XInternAtom(dpy, "_NET_WM_WINDOW_TYPE_DOCK", False);
    net_wm_type_menu = XInternAtom(dpy, "_NET_WM_WINDOW_TYPE_MENU", False);
    net_wm_type_splash = XInternAtom(dpy, "_NET_WM_WINDOW_TYPE_SPLASH", False);
    net_wm_type_desk = XInternAtom(dpy, "_NET_WM_WINDOW_TYPE_DESKTOP", False);

    /* The bit about _NET_CLIENT_LIST_STACKING here is an evil lie. */
    append_atoms(root, net_supported, XA_ATOM, &net_cur_desk, 1);
    append_atoms(root, net_supported, XA_ATOM, &net_num_desktops, 1);
    append_atoms(root, net_supported, XA_ATOM, &net_active_window, 1);
    append_atoms(root, net_supported, XA_ATOM, &net_wm_desktop, 1);
    append_atoms(root, net_supported, XA_ATOM, &net_client_list, 1);
    append_atoms(root, net_supported, XA_ATOM, &net_client_stack, 1);
    append_atoms(root, net_supported, XA_ATOM, &net_wm_state, 1);
    append_atoms(root, net_supported, XA_ATOM, &net_wm_state_shaded, 1);
    append_atoms(root, net_supported, XA_ATOM, &net_wm_state_mv, 1);
    append_atoms(root, net_supported, XA_ATOM, &net_wm_state_mh, 1);
    append_atoms(root, net_supported, XA_ATOM, &net_wm_strut, 1);
    append_atoms(root, net_supported, XA_ATOM, &net_wm_strut_partial, 1);
    append_atoms(root, net_supported, XA_ATOM, &net_wm_wintype, 1);
    append_atoms(root, net_supported, XA_ATOM, &net_wm_type_dock, 1);

    get_atoms(root, net_num_desktops, XA_CARDINAL, 0, &ndesks, 1, NULL);
    get_atoms(root, net_cur_desk, XA_CARDINAL, 0, &cur_desk, 1, NULL);

#ifdef SHAPE
    shape = XShapeQueryExtension(dpy, &shape_event, &shape_err);
#endif

    XQueryTree(dpy, root, &qroot, &qparent, &wins, &nwins);
    for (i = 0; i < nwins; i++) {
        XGetWindowAttributes(dpy, wins[i], &attr);
        if (!attr.override_redirect && attr.map_state == IsViewable)
            map_new(wins[i]);
    }
    XFree(wins);

    sattr.event_mask = SubMask|ColormapChangeMask|ButtonMask;
    XChangeWindowAttributes(dpy, root, CWEventMask, &sattr);
}
