/* aewm - An Exiguous Window Manager - vim:sw=4:ts=4:et
 *
 * Copyright 1998-2006 Decklin Foster <decklin@red-bean.com>. This
 * program is free software; please see LICENSE for details. */

#include <stdlib.h>
#include <X11/Xatom.h>
#ifdef SHAPE
#include <X11/extensions/shape.h>
#endif
#include "aewm.h"
#include "atom.h"

static void do_map(client_t *, int);
static int init_geom(client_t *, strut_t *, int);
static void reparent(client_t *, strut_t *);

/* Set up a client structure for the new (not-yet-mapped) window. We have to
 * ignore two unmap events if the client was already mapped but has
 * IconicState set (for instance, when we are the second window manager in a
 * session). That's because there's one for the reparent (which happens on
 * all viewable windows) and then another for the unmapping itself. */

void map_new(Window w)
{
    client_t *c;
    XWindowAttributes attr;
    XColor exact;
    long supplied;
    Atom win_type, state;
    XWMHints *hints;
    int i, btn, want_raise = 1;
    strut_t s;
    unsigned long read, left;

    c = malloc(sizeof *c);
    c->next = head;
    head = c;

    XGrabServer(dpy);

    XGetTransientForHint(dpy, w, &c->trans);
    XGetWindowAttributes(dpy, w, &attr);

    c->name = get_wm_name(w);
    c->win = w;
    c->frame = None;
    c->geom.x = attr.x;
    c->geom.y = attr.y;
    c->geom.w = attr.width;
    c->geom.h = attr.height;
    c->cmap = attr.colormap;
    c->titled = 1;
    c->bordered = 1;
    c->shaded = 0;
    c->zoomed = 0;
#ifdef SHAPE
    c->shaped = 0;
#endif
    c->ignore_unmap = 0;

    XAllocNamedColor(dpy, c->cmap, opt_fg, &fg, &exact);
    XAllocNamedColor(dpy, c->cmap, opt_bg, &bg, &exact);
    XAllocNamedColor(dpy, c->cmap, opt_bd, &bd, &exact);

    c->size = XAllocSizeHints();
    XGetWMNormalHints(dpy, c->win, c->size, &supplied);

#ifdef DEBUG
    dump_title(c, "creating", 'w');
    dump_geom(c, "initial");
#endif

    if (get_atoms(c->win, net_wm_wintype, XA_ATOM, 0,
            &win_type, 1, NULL) && HAS_NO_BORDER(win_type)) {
        c->titled = 0;
        c->bordered = 0;
    }

    if (get_atoms(c->win, net_wm_desktop, XA_CARDINAL, 0,
            &c->desk, 1, NULL)) {
        if (c->desk >= ndesks && c->desk != DESK_ALL)
            c->desk = ndesks - 1;
    } else {
        set_atoms(c->win, net_wm_desktop, XA_CARDINAL, &cur_desk, 1);
        c->desk = cur_desk;
    }

    for (i = 0, left = 1; left; i += read) {
        read = get_atoms(c->win, net_wm_state, XA_ATOM, i, &state, 1, &left);
        if (read)
            handle_net_wm_state_item(c, state);
        else
            break;
    }

    collect_struts(c->desk, &s);

    if (attr.map_state == IsViewable) {
        c->ignore_unmap++;
    } else {
        if ((hints = XGetWMHints(dpy, w))) {
            if (hints->flags & StateHint)
                set_wm_state(c, hints->initial_state);
            XFree(hints);
        } else {
            set_wm_state(c, NormalState);
        }
        if (opt_map) {
            if (!init_geom(c, &s, 1)) {
                btn = sweep(c, map_curs, recalc_map, SWEEP_DOWN, &s);
                if (btn == Button2)
                    btn = sweep(c, resize_curs, recalc_resize, SWEEP_UP, &s);
                if (btn == Button3)
                    want_raise = 0;
            }
        } else {
            init_geom(c, &s, 0);
        }
#ifdef DEBUG
        dump_geom(c, "set to");
        dump_info(c);
#endif
    }

    reparent(c, &s);

#ifdef XFT
    c->xftdraw = XftDrawCreate(dpy, (Drawable)c->frame,
        DefaultVisual(dpy, DefaultScreen(dpy)),
        DefaultColormap(dpy, DefaultScreen(dpy)));
#endif

    if (attr.map_state == IsViewable) {
        if (get_wm_state(c->win) == IconicState) {
            c->ignore_unmap++;
            XUnmapWindow(dpy, c->win);
        } else {
            do_map(c, want_raise);
        }
    } else {
        if (get_wm_state(c->win) == NormalState)
            do_map(c, want_raise);
    }

    /* We are not actually keeping the stack one in order. However, every
     * fancy panel uses it and nothing else, no matter what the spec says.
     * (I'm not sure why, as rearranging the list every time the stacking
     * changes would be distracting. GNOME's window list applet doesn't.) */
    append_atoms(root, net_client_list, XA_WINDOW, &c->win, 1);
    append_atoms(root, net_client_stack, XA_WINDOW, &c->win, 1);

    XSync(dpy, False);
    XUngrabServer(dpy);
}

static void do_map(client_t *c, int do_raise)
{
    if (IS_ON_CUR_DESK(c)) {
        XMapWindow(dpy, c->win);
        if (do_raise) {
            XMapRaised(dpy, c->frame);
        } else {
            XLowerWindow(dpy, c->frame);
            XMapWindow(dpy, c->frame);
        }
    }
}

/* When we're ready to map, we have two things to consider: the literal
 * geometry of the window, and the size hints. We start with the geom as a
 * base. If the program-specified hints are there, and not insane, they have
 * the same precedence, and we use them. User-specified hints, however, take
 * precedence over both.
 *
 * If there are no user hints, and we end up at 0,0, then assume that the
 * client doesn't care and make up a location based on the cursor position.
 *
 * We also want to know if we are ``satisfied'' with the hints we have so far;
 * if interactive mapping is on, PPosition is not considered important enough
 * to override it, but if it's off, then we can stop placing when we get a
 * sane one. */

static int init_geom(client_t *c, strut_t *s, int weak_ppos)
{
    Atom win_type;
    int screen_x = DisplayWidth(dpy, screen);
    int screen_y = DisplayHeight(dpy, screen);
    int wmax = screen_x - s->left - s->right;
    int hmax = screen_y - s->top - s->bottom;
    int mouse_x, mouse_y;
    int placed = 0;
    geom_t f = frame_geom(c);

    if (c->size->flags & (USSize)) {
        if (c->size->width) c->geom.w = c->size->width;
        if (c->size->height) c->geom.h = c->size->height;
    } else {
        /* Make sure it's at least big enough to click at */
        if (c->geom.w < 2 * theight(c)) c->geom.w = 2 * theight(c);
        if (c->geom.h < theight(c)) c->geom.h = theight(c);
    }

    if (c->size->flags & USPosition) {
        /* Even if the user wants it at 0,0 or offscreen, obey. */
        c->geom.x = c->size->x;
        c->geom.y = c->size->y;
        placed = 1;
    } else if (c->size->flags & PPosition) {
        /* Only if it's actually set, and isn't offscreen. */
        if (c->size->x && c->size->y) {
            c->geom.x = c->size->x;
            c->geom.y = c->size->y;
        }
        if (c->geom.x < 0 || c->geom.x > screen_x || c->geom.y < 0 ||
                c->geom.y > screen_y)
            placed = 0;
    }

    /* Unless it is a dock etc, because 0,0 is fine there. */
    if (get_atoms(c->win, net_wm_wintype, XA_ATOM, 0, &win_type, 1, NULL))
        placed = CAN_PLACE_SELF(win_type);
    if (c->trans || c->zoomed || (!weak_ppos && c->geom.x && c->geom.y))
        placed = 1;

    if (placed) {
        /* OK, now we can make sure it's not overlapping any struts or
         * screen edges. If the window is bigger than we have available,
         * only move it down from 0,0 (we want the title visible at
         * least). */
        if (!c->trans) {
            if (f.x < s->left) f.x = s->left;
            if (f.y < s->top) f.y = s->top;
            if (f.w > wmax) f.x = s->left;
            else if (f.y + f.w > screen_x - s->right) f.x = s->right - f.w;
            if (f.h > hmax) f.x = s->top;
            else if (f.y + f.h > screen_y - s->bottom) f.y = s->bottom - f.h;
        }
    } else {
        /* Make a guess if we don't have anything else */
        get_pointer(&mouse_x, &mouse_y);
        recalc_map(c, c->geom, mouse_x, mouse_y, mouse_x, mouse_y, s);
    }

    return placed;
}

static void reparent(client_t *c, strut_t *s)
{
    XSetWindowAttributes pattr;
    geom_t f;

    if (c->zoomed) {
        c->save.x = c->geom.x;
        c->save.y = c->geom.y;
        c->save.w = c->geom.w;
        c->save.h = c->geom.h;
        c->geom.x = s->left;
        c->geom.y = s->top;
        c->geom.w = DisplayWidth(dpy, screen) - 2*BW(c) -
            s->left - s->right;
        c->geom.h = DisplayHeight(dpy, screen) - 2*BW(c) - theight(c) -
            s->top - s->bottom;
    }

    f = frame_geom(c);
    pattr.override_redirect = True;
    pattr.background_pixel = bg.pixel;
    pattr.border_pixel = bd.pixel;
    pattr.event_mask = SubMask|ButtonPressMask|ExposureMask|EnterWindowMask;
    c->frame = XCreateWindow(dpy, root, f.x, f.y, f.w, f.h, BW(c),
        DefaultDepth(dpy, screen), CopyFromParent, DefaultVisual(dpy, screen),
        CWOverrideRedirect|CWBackPixel|CWBorderPixel|CWEventMask, &pattr);

#ifdef SHAPE
    if (shape) {
        XShapeSelectInput(dpy, c->win, ShapeNotifyMask);
        set_shape(c);
    }
#endif

    XAddToSaveSet(dpy, c->win);
    XSelectInput(dpy, c->win, ColormapChangeMask|PropertyChangeMask);
    XSetWindowBorderWidth(dpy, c->win, 0);
    XResizeWindow(dpy, c->win, c->geom.w, c->geom.h);
    XReparentWindow(dpy, c->win, c->frame, 0, theight(c));

    send_config(c);
}
