/* aewm - An Exiguous Window Manager - vim:sw=4:ts=4:et
 *
 * Copyright 1998-2006 Decklin Foster <decklin@red-bean.com>. This
 * program is free software; please see LICENSE for details. */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <X11/Xatom.h>
#include "aewm.h"
#include "atom.h"

static void draw_outline(client_t *);
static void fix_size(client_t *, long *, long *);

void do_press(client_t *c, int x, int y, int button)
{
    int in_box, mouse_x, mouse_y, mask;

    in_box = (x >= c->geom.w - theight(c)) && (y <= theight(c));
    mask = get_pointer(&mouse_x, &mouse_y);

    /* For both of these, 1 and 3 operate immediately on press but 2 may be
     * either drag or click (maybe double- instead of shift-, later). */
    if (in_box) {
        switch (button) {
            case Button1:
                iconify_win(c);
                break;
            case Button3:
                send_wm_delete(c);
                break;
            case Button2:
                if (mask & ShiftMask)
                    zoom_win(c);
                else
                    resize_win(c);
                break;
        }
    } else {
        switch (button) {
            case Button1:
                XRaiseWindow(dpy, c->frame);
                break;
            case Button3:
                XLowerWindow(dpy, c->frame);
                break;
            case Button2:
                if (mask & ShiftMask)
                    shade_win(c);
                else
                    move_win(c);
                break;
        }
    }
}

void move_win(client_t *c)
{
    geom_t frame;

    c->zoomed = 0;
    sweep(c, move_curs, recalc_move, SWEEP_UP, NULL);
    frame = frame_geom(c);
    XMoveWindow(dpy, c->frame, frame.x, frame.y);
    send_config(c);
}

void resize_win(client_t *c)
{
    geom_t frame;

    if (c->zoomed) {
        c->save = c->geom; /* cheat */
        zoom_win(c);
    }
    sweep(c, resize_curs, recalc_resize, SWEEP_UP, NULL);
    frame = frame_geom(c);
    XMoveResizeWindow(dpy, c->frame, frame.x, frame.y, frame.w, frame.h);
    XMoveResizeWindow(dpy, c->win, 0, theight(c), c->geom.w, c->geom.h);
    send_config(c);
}

/* This can't do anything dangerous. See handle_enter_event. */

void set_active(client_t *c)
{
    set_atoms(root, net_active_window, XA_WINDOW, &c->win, 1);
    XSetInputFocus(dpy, c->win, RevertToPointerRoot, CurrentTime);
    XInstallColormap(dpy, c->cmap);
}

/* Transients will be iconified when their owner is iconified. */

void iconify_win(client_t *c)
{
    client_t *p;

    do_iconify(c);
    for (p = head; p; p = p->next)
        if (p->trans == c->win) do_iconify(p);
}

void do_iconify(client_t *c)
{
    if (!c->ignore_unmap)
        c->ignore_unmap++;

    XUnmapWindow(dpy, c->frame);
    XUnmapWindow(dpy, c->win);

    set_wm_state(c, IconicState);
}

void do_deiconify(client_t *c)
{
    XMapWindow(dpy, c->win);
    XMapRaised(dpy, c->frame);
    set_wm_state(c, NormalState);
}

void shade_win(client_t *c)
{
    geom_t f;

    if (!c->shaded) {
        c->shaded = 1;
        f = frame_geom(c);
        XMoveResizeWindow(dpy, c->frame, f.x, f.y, f.w, f.h);
        append_atoms(c->win, net_wm_state, XA_ATOM, &net_wm_state_shaded, 1);
    } else {
        c->shaded = 0;
        f = frame_geom(c);
        XMoveResizeWindow(dpy, c->frame, f.x, f.y, f.w, f.h);
        remove_atom(c->win, net_wm_state, XA_ATOM, net_wm_state_shaded);
    }

    send_config(c);
}

/* When zooming a window, the old geom gets stuffed into c->save. Once we
 * unzoom, this should be considered garbage. Despite the existence of
 * vertical and horizontal hints, we only provide both at once.
 *
 * Zooming implies unshading, but the inverse is not true. */

void zoom_win(client_t *c)
{
    strut_t s;
    geom_t f;

    if (c->zoomed) {
        c->zoomed = 0;
        remove_atom(c->win, net_wm_state, XA_ATOM, net_wm_state_mv);
        remove_atom(c->win, net_wm_state, XA_ATOM, net_wm_state_mh);
        c->geom.x = c->save.x;
        c->geom.y = c->save.y;
        c->geom.w = c->save.w;
        c->geom.h = c->save.h;
        f = frame_geom(c);
        XMoveResizeWindow(dpy, c->frame, f.x, f.y, f.w, f.h);
        XResizeWindow(dpy, c->win, c->geom.w, c->geom.h);
    } else {
        c->shaded = 0;
        c->zoomed = 1;
        remove_atom(c->win, net_wm_state, XA_ATOM, net_wm_state_shaded);
        append_atoms(c->win, net_wm_state, XA_ATOM, &net_wm_state_mv, 1);
        append_atoms(c->win, net_wm_state, XA_ATOM, &net_wm_state_mh, 1);
        c->save.x = c->geom.x;
        c->save.y = c->geom.y;
        c->save.w = c->geom.w;
        c->save.h = c->geom.h;
        collect_struts(c->desk, &s);
        c->geom.x = s.left;
        c->geom.y = s.top;
        c->geom.w = DisplayWidth(dpy, screen) - 2*BW(c) -
            s.left - s.right;
        c->geom.h = DisplayHeight(dpy, screen) - 2*BW(c) - theight(c) -
            s.top - s.bottom;
        XResizeWindow(dpy, c->win, c->geom.w, c->geom.h);
        XMoveResizeWindow(dpy, c->frame, c->geom.x, c->geom.y,
            c->geom.w, c->geom.h + theight(c));
    }

    redraw(c);
    send_config(c);
}

void map_desk(client_t *c)
{
    if (IS_ON_CUR_DESK(c) && get_wm_state(c->win) == NormalState)
        XMapWindow(dpy, c->frame);
    else
        XUnmapWindow(dpy, c->frame);
}

int sweep(client_t *c, Cursor curs, sweep_func cb, int mode, strut_t *s)
{
    int x0, y0, mask;
    geom_t orig = c->geom;
    XEvent ev;

    if (!GRAB(root, MouseMask, curs)) return 0;
    XGrabServer(dpy);
    draw_outline(c);

    mask = get_pointer(&x0, &y0);

    /* XXX: Ugh */
    if (!(mode == SWEEP_DOWN && (mask & ButtonPressMask))) {
        for (;;) {
            XMaskEvent(dpy, MouseMask, &ev);
            if (ev.type == MotionNotify) {
                draw_outline(c);
                cb(c, orig, x0, y0, ev.xmotion.x, ev.xmotion.y, s);
                draw_outline(c);
            } else if ((mode == SWEEP_UP && ev.type == ButtonRelease) ||
                    (mode == SWEEP_DOWN && ev.type == ButtonPress)) {
                draw_outline(c);
                break;
            }
        }
    }

    XUngrabServer(dpy);
    XUngrabPointer(dpy, CurrentTime);

    return ev.xbutton.button;
}

/* This is simple and dumb: if the cursor is in the center of the screen,
 * center the window on the available space. If it's at the top left, then
 * at the top left. As you go between, and to other edges, scale it. */

void recalc_map(client_t *c, geom_t orig, int x0, int y0, int x1, int y1,
    strut_t *s)
{
    int screen_x = DisplayWidth(dpy, screen);
    int screen_y = DisplayHeight(dpy, screen);
    int wmax = screen_x - s->left - s->right;
    int hmax = screen_y - s->top - s->bottom;

    if (c->geom.w < wmax)
        c->geom.x = s->left + ((float)x1 / (float)screen_x) *
            (wmax - c->geom.w - 2*BW(c));
    if (c->geom.h + theight(c) < hmax)
        c->geom.y = s->top + ((float)y1 / (float)screen_y) *
            (hmax - c->geom.h - theight(c) - 2*BW(c));
}

void recalc_move(client_t *c, geom_t orig, int x0, int y0, int x1, int y1,
    strut_t *s)
{
    c->geom.x = orig.x + x1 - x0;
    c->geom.y = orig.y + y1 - y0;
}

/* When considering the distance from the mouse to the center point of the
 * window (which remains fixed), we actually have to look at the top right
 * corner of the window (which remains a constant offset from wherever we
 * clicked in the box relative to the root, but not relative to the window,
 * because the distance can be negative). After that we just center the new
 * size. */

void recalc_resize(client_t *c, geom_t orig, int x0, int y0, int x1, int y1,
    strut_t *s)
{
    int mx = orig.x + orig.w / 2;
    int my = orig.y + theight(c) + orig.h / 2;
    int cx = x1 - (orig.x + orig.w - x0);
    int cy = y1 + (orig.y + theight(c) - y0);
    long lx, ly;

    c->geom.w = (x1 > mx) ? abs(orig.w + 2 * (x1 - x0)) : 2 * (mx - cx);
    c->geom.h = (cy < my) ? abs(orig.h - 2 * (y1 - y0)) : 2 * (cy - my);

    fix_size(c, &lx, &ly);
    c->geom.x = mx - c->geom.w/2;
    c->geom.y = my - c->geom.h/2 - theight(c);
}

/* If the window in question has a ResizeInc hint, then it wants to be resized
 * in multiples of some (x,y). We constrain the values in c->geom based on
 * that and any min/max size hints, and put the ``human readable'' values back
 * in lw_ret and lh_ret (80x25 for xterm, etc). */

static void fix_size(client_t *c, long *lw_ret, long *lh_ret)
{
    int width_inc, height_inc;
    int base_width, base_height;

    if (c->size->flags & PMinSize) {
        if (c->geom.w < c->size->min_width) c->geom.w = c->size->min_width;
        if (c->geom.h < c->size->min_height) c->geom.h = c->size->min_height;
    }
    if (c->size->flags & PMaxSize) {
        if (c->geom.w > c->size->max_width) c->geom.w = c->size->max_width;
        if (c->geom.h > c->size->max_height) c->geom.h = c->size->max_height;
    }
    if (c->size->flags & PResizeInc) {
        width_inc = c->size->width_inc ? c->size->width_inc : 1;
        height_inc = c->size->height_inc ? c->size->height_inc : 1;
        base_width = (c->size->flags & PBaseSize) ? c->size->base_width :
            (c->size->flags & PMinSize) ? c->size->min_width : 0;
        base_height = (c->size->flags & PBaseSize) ? c->size->base_height :
            (c->size->flags & PMinSize) ? c->size->min_height : 0;
        c->geom.w = c->geom.w - ((c->geom.w - base_width) % width_inc);
        c->geom.h = c->geom.h - ((c->geom.h - base_height) % height_inc);
        *lw_ret = (c->geom.w - base_width) / width_inc;
        *lh_ret = (c->geom.h - base_height) / height_inc;
    } else {
        *lw_ret = c->geom.w;
        *lh_ret = c->geom.h;
    }
}

static void draw_outline(client_t *c)
{
    geom_t f = frame_geom(c);
    char buf[32];
    long lw, lh;

    XDrawRectangle(dpy, root, invert_gc, f.x + BW(c)/2, f.y + BW(c)/2,
        f.w + BW(c), f.h + BW(c));
    if (c->titled && !c->shaded && c->geom.y)
        XDrawLine(dpy, root, invert_gc, f.x + BW(c), f.y + BW(c)/2 +
            theight(c), f.x + f.w + BW(c), f.y + theight(c) + BW(c)/2);

    fix_size(c, &lw, &lh);
    snprintf(buf, sizeof buf, "%ldx%ld%+ld%+ld",
        lw, lh, c->geom.x, c->geom.y);
    XDrawString(dpy, root, invert_gc,
        f.x + f.w - XTextWidth(font, buf, strlen(buf)) - opt_pad,
        f.y + f.h - opt_pad, buf, strlen(buf));
}
