// ---------------------------------------------------------------------------
// - Argument.cpp                                                            -
// - afnix engine - argument class implementation                            -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2007 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Stack.hpp"
#include "Vector.hpp"
#include "Boolean.hpp"
#include "Runnable.hpp"
#include "Argument.hpp"
#include "QuarkZone.hpp"
#include "Exception.hpp"

namespace afnix {

  // -------------------------------------------------------------------------
  // - class section                                                         -
  // -------------------------------------------------------------------------

  // create a argument by quark with an index

  Argument::Argument (const long quark, const long index) {
    d_quark = quark;
    d_index = index;
    d_const = false;
  }

  // copy construct this argument

  Argument::Argument (const Argument& that) {
    that.rdlock ();
    d_quark = that.d_quark;
    d_index = that.d_index;
    d_const = that.d_const;
    that.unlock ();
  }

  // return the class name

  String Argument::repr (void) const {
    return "Argument";
  }

  // return a clone of this argument

  Object* Argument::clone (void) const {
    return new Argument (*this);
  }

  // get a literal representation of this argument

  String Argument::toliteral (void) const {
    rdlock ();
    try {
      String result = String::qmap (d_quark);
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }
    
  // get a string representation of this argument

  String Argument::tostring (void) const {
    rdlock ();
    try {
      String result = String::qmap (d_quark);
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // set the const flag for this argument

  void Argument::setconst (const bool flag) {
    wrlock ();
    d_const = flag;
    unlock ();
  }

  // get the const flag for this argument

  bool Argument::getconst (void) const {
    rdlock ();
    bool result = d_const;
    unlock ();
    return result;
  }

  // -------------------------------------------------------------------------
  // - object section                                                        -
  // -------------------------------------------------------------------------

  // the quark zone
  static const long QUARK_ZONE_LENGTH = 4;
  static QuarkZone  zone (QUARK_ZONE_LENGTH);

  // the object supported quarks
  static const long QUARK_GETCONST  = zone.intern ("get-const");
  static const long QUARK_SETCONST  = zone.intern ("set-const");
  static const long QUARK_GETOBJECT = zone.intern ("get-object");
  static const long QUARK_SETOBJECT = zone.intern ("set-object");

  // set the object

  void Argument::setobj (Runnable* robj, Object* object) {
    wrlock ();
    // check for the const flag
    if (d_const == true) {
      unlock ();
      throw Exception ("const-error", "const violation for argument", 
		       String::qmap (d_quark));
    }
    // get the stack runnable and set the object
    try {
      Stack* stk = robj->getstk ();
      stk->set (d_index, object);
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // return true if the given quark is defined

  bool Argument::isquark (const long quark, const bool hflg) const {
    rdlock ();
    if (zone.exists (quark) == true) {
      unlock ();
      return true;
    }
    bool result = hflg ? Literal::isquark (quark, hflg) : false;
    unlock ();
    return result;
  }

  // set this object as a const object

  Object* Argument::cdef (Runnable* robj, Nameset* nset, Object* object) {
    wrlock ();
    try {
      setobj (robj, object);
      d_const = true;
      robj->post (object);
      unlock ();
      return object;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // set this object with an object

  Object* Argument::vdef (Runnable* robj, Nameset* nset, Object* object) {
    wrlock ();
    try {
      setobj (robj, object);
      robj->post (object);
      unlock ();
      return object;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // evaluate this object - this is like get object

  Object* Argument::eval (Runnable* robj, Nameset* nset) {
    rdlock ();
    try {
      Stack*     stk = robj->getstk ();
      Object* result = stk->get (d_index);
      robj->post (result);
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // apply this object with a set of arguments and a quark

  Object* Argument::apply (Runnable* robj, Nameset* nset, const long quark,
			   Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // dispatch 0 argument
    if (argc == 0) {
      if (quark == QUARK_GETCONST)  return new Boolean (getconst ());
      if (quark == QUARK_GETOBJECT) {
	rdlock ();
	try {
	  Object* result = eval (robj, nset);
	  robj->post (result);
	  unlock ();
	  return result;
	} catch (...) {
	  unlock ();
	  throw;
	}
      }
    }
    // dispatch 1 argument
    if (argc == 1) {
      if (quark == QUARK_SETCONST) {
	bool bval = argv->getbool (0);
	setconst (bval);
	return nilp;
      }
      if (quark == QUARK_SETOBJECT) {
	setobj (robj, argv->get (0));
	return nilp;
      }
    }
    // call the literal method
    return Literal::apply (robj, nset, quark, argv);
  }
}
