// ---------------------------------------------------------------------------
// - Buffer.hpp                                                              -
// - standard object library - character buffer class definition             -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2007 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  AFNIX_BUFFER_HPP
#define  AFNIX_BUFFER_HPP

#ifndef  AFNIX_OUTPUT_HPP
#include "Output.hpp"
#endif

namespace afnix {

  /// The Buffer class is a byte buffer which can be used to accumulate
  /// characters. By default, the class automatically resize itself when full.
  /// However, a flag controls whther or not this operation can proceed. If
  /// the buffer is full, an exception is raised. Standard methods to read 
  /// or write or extract data is provided. 
  /// @author amaury darsch

  class Buffer : public virtual Object {
  private:
    /// the byte buffer
    char* p_buffer;
    /// the buffer size
    long  d_size;
    /// the buffer length
    long  d_length;
    /// the resize flag
    bool  d_rflg;
    /// the buffer encoding
    System::t_emod d_emod;

  public:
    /// create a default buffer
    Buffer (void);

    /// create a new buffer with a default size.
    /// @param size the buffer default size
    Buffer (const long size);

    /// create a new buffer with a mode
    /// @param emdo the encoding mode
    Buffer (const System::t_emod emod);
    
    /// create a new buffer and initialize it with a c string
    /// @param value the c string to initialize
    Buffer (const char* value);

    /// create a new buffer and initialize it with a string
    /// @param value the string to initialize
    Buffer (const String& value);

    /// copy construct a buffer
    Buffer (const Buffer& that);

    /// destroy this buffer
    ~Buffer (void);

    /// @return the class name
    String repr (void) const;

    /// reset this buffer 
    void reset (void);
    
    /// set the buffer encoding mode
    /// param emod the encoding mode to set
    void setemod (const System::t_emod emod);

    /// @return the buffer encoding
    System::t_emod getemod (void) const;

    /// set the resize flag
    /// @param rflg the resize flag
    void setrflg (const bool rflg);

    /// @return the resize flag
    bool getrflg (void) const;

    /// @return true if the buffer is empty
    bool empty (void) const;

    /// @return true if the buffer is full
    bool full (void) const;

    /// add a character to this buffer
    /// @param value the character to add to this buffer
    void add (const char value);

    /// add a unicode character to this buffer
    /// @param value the unicode character to add to this buffer
    void add (const t_quad value);

    /// add a c-string to this buffer
    /// @param s the c-string to add
    void add (const char* s);

    /// add a c-string in this buffer
    /// @param s    the string to add
    /// @param size the string size
    void add (const char* s, const long size);

    /// add a string to this buffer
    /// @param value the string to add to this buffer
    void add (const String& value);

    /// add a buffer object to this buffer
    /// @param buffer the buffer to add
    void add (const Buffer& buffer);
    
    /// @return the next byte but do not remove it
    char get (void) const;

    /// @return the byte at the current index position
    char read (void);

    /// @return a line from this buffer
    String readln (void);

    /// pushback a character in this buffer
    /// @param value the character to push back
    void pushback (const char value);

    /// pushback a unicode character in this buffer
    /// @param value the character to push back
    void pushback (const t_quad value);

    /// pushback a c-string in this buffer
    /// @param s the string to pushback
    void pushback (const char* value);

    /// pushback a c-string in this buffer
    /// @param s    the string to pushback
    /// @param size the string size
    void pushback (const char* s, const long size);

    /// pushback a string in this buffer
    /// @param value the string to push back
    void pushback (const String& value);

    /// pushback a buffer in this buffer
    /// @param buffer the buffer to push back
    void pushback (const Buffer& value);

    /// @return the length of this buffer
    long length (void) const;

    /// @return the buffer contents as a string
    String tostring (void) const;

    /// map this buffer to an anonymous data structure
    /// @param data the data structure to map
    /// @param size the data size to map
    long tomap (void* data, const long size);

    /// @return a word (in big endian format) from this buffer 
    t_word getword (void);

    /// @return a quad (in big endian format) from this buffer 
    t_quad getquad (void);

    /// @return a octa (in big endian format) from this buffer 
    t_octa getocta (void);

    /// write the buffer content to an output stream
    /// @param os the output stream
    void write (Output& os) const;

  private:
    // make the assignment operator private
    Buffer& operator = (const Buffer&);

  public:
    /// create a new object in a generic way
    /// @param argv the argument vector
    static Object* mknew (Vector* argv);

    /// @return true if the given quark is defined
    bool isquark (const long quark, const bool hflg) const;

    /// apply this object with a set of arguments and a quark
    /// @param robj  the current runnable
    /// @param nset  the current nameset    
    /// @param quark the quark to apply these arguments
    /// @param argv  the arguments to apply
    Object* apply (Runnable* robj, Nameset* nset, const long quark,
		   Vector* argv);    
  };
}

#endif
