/*******************************************************************************
 *  PROJECT: GNOME Colorscheme
 *
 *  AUTHOR: Jonathon Jongsma
 *
 *  Copyright (c) 2005 Jonathon Jongsma
 *
 *  License:
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the 
 *    Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
 *    Boston, MA  02111-1307  USA
 *
 *******************************************************************************/

#include <iostream>
#include <sstream>
#include <fstream>

#include <glibmm/error.h>

#include "gcs-palette.h"
#include "gcs-debug.h"
#include "gcs-util.h"
#include "gcs-i18n.h"

namespace gcs
{

    Palette::Palette(Glib::ustring file_or_string) :
        Gtk::ListStore()
    {
        // NOTE: I tried to put m_modelColumns as an argument for the ListStore
        // constructor in the initializer list, but it gave me critical
        // warnings and eventually segfaulted.  Removing the constructor
        // argument and setting it manually here seems to fix things.
        set_column_types(m_modelColumns);

        if (!file_or_string.empty())
        {
            debug("Try Parsing palette from file: ", file_or_string);
            if(!set_from_file(file_or_string))
            {
                debug("Parsing palette from string");
                if (!set_from_string(file_or_string))
                {
                    debug("Error loading Palette definition from ", file_or_string);
                }
            }
            debug("Finished parsing palette.  Total colors: ", children().size());
        }
    }


    Glib::RefPtr<Palette> Palette::create(Glib::ustring fname)
    {
        return Glib::RefPtr<Palette>(new Palette(fname));
    }


    bool Palette::set_from_string(Glib::ustring& palette_def)
    {
        bool status = false;
        std::istringstream istrm(palette_def);
        if (!istrm.fail())
        {
            try
            {
                status = parse_palette(istrm);
            }
            catch (Glib::Error& e)
            { }
        }
        return status;
    }


    bool Palette::set_from_file(Glib::ustring filename)
    {
        bool status = false;
        std::ifstream gpl(filename.c_str());
        if(gpl.is_open())
        {
            try
            {
                status = parse_palette(gpl);
            }
            catch (Glib::Error& e)
            {
                std::cerr << "Problem parsing palette from " << gpl << std::endl;
            }
        }
        gpl.close();
        return status;
    }


    bool Palette::parse_palette(std::istream& istrm)
    {
        // first we need to free all of the existing colors
        reset();
        bool success = false;
        bool seen_header = false;
        bool seen_name = false;
        bool seen_colordef = false;
        bool seen_cols = false;
        const char LINE_END = '\n';
        Glib::ustring color_linestart_chars(" 012");
        const gint LINE_LENGTH = 80;
        char line_buf[LINE_LENGTH];

        // make sure the first line is "GIMP Palette"
        istrm.getline(line_buf, LINE_LENGTH);
        if (strcmp(line_buf,"GIMP Palette") == 0)
        {
            seen_header = true;
        }

        while (!istrm.fail())
        {
            // ignore any lines that are comments
            if (istrm.peek() == '#')
            {
                debug("* Parsing palette: Ignoring comment line");
                istrm.ignore(255, LINE_END);
                continue;
            }

            // process any lines that are color definitions
            if (color_linestart_chars.find(static_cast<char>(istrm.peek())) !=
                    Glib::ustring::npos)
            {
                istrm.getline(line_buf, LINE_LENGTH);
                try
                {
                    debug("* Parsing palette: parsing line -> ", line_buf);
                    color_def_t c = parse_line(line_buf);
                    append_color(c.first, c.second);
                    seen_colordef = true;
                }
                catch(const Glib::Error& ex)
                {
                    std::cerr << "** Bad color definition in palette file." << std::endl;
                }
                continue;
            }

            // we don't care about the "Columns:" field, just the "Name:" field
            if (istrm.peek() == 'N')
            {
                // get the next line
                istrm.getline(line_buf, LINE_LENGTH);
                // if it's a name field, we want to use it
                Glib::ustring s(line_buf);
                if (s.find("Name:") != Glib::ustring::npos)
                {
                    // extract the name and put it in m_name
                    Glib::ustring temp_name = s.substr(5);
                    m_name = trim(temp_name);
                    debug("* Parsing Palette: set name to ", m_name);
                    seen_name = true;
                    continue;
                }
            }
            else if (istrm.peek() == 'C')
            {
                istrm.getline(line_buf, LINE_LENGTH);
                // if it's a columns field, use it
                Glib::ustring c(line_buf);
                if (c.find("Columns:") != Glib::ustring::npos)
                {
                    Glib::ustring temp_columns = c.substr(8);
                    m_paletteColumns = atoi(trim(temp_columns).c_str());
                    debug("* Parsing Palette: set columns to ", m_paletteColumns);
                    seen_cols = true;
                    continue;
                }
            }
            else
            {
                istrm.ignore(255, LINE_END);
                continue;
            }
        }
        if (seen_header && seen_colordef)
        {
            success = true;
        }
        return success;
    }


    Palette::color_def_t Palette::parse_line(Glib::ustring palette_line)
    {
        color_def_t def;
        // Do a loose verification that the string is ok
        if (palette_line.size() < 11)
        {
            throw Glib::Error();
        }
        const gint WIDTH = 3;
        Glib::ustring red, green, blue;
        red = trim(palette_line.substr(0, WIDTH));
        green = trim(palette_line.substr(4, WIDTH));
        blue = trim(palette_line.substr(8, WIDTH));
        def.first = Color::create(atoi(red.c_str()), atoi(green.c_str()), atoi(blue.c_str()));
        def.second = trim(palette_line.substr(11, palette_line.size()));
        // ignore the rest of the line
        return def;
    }


    Gtk::TreeModel::Row Palette::add(ColorPtr clr, Glib::ustring name)
    {
        Gtk::TreeModel::Row row;
        g_return_val_if_fail(clr != NULL, row);

        // Assume the the color we want to add is not already in the list
        bool exists = false;
        // Loop through all the items in the list 
        for (Gtk::TreeModel::iterator iter = children().begin();
                iter != children().end(); ++iter)
        {
            g_assert(iter);
            // check if the current item in the list matches the 
            // color value to be added
            // NOTE: need to use get_value() here instead of operator[]
            if (*(*iter).get_value(m_modelColumns.m_colColor) == *clr)
            {
                exists = true;
                row = *iter;
            }
        }
        if (!exists)
        {
            row = append_color(clr, name);
        }
        return row;
    }


    Gtk::TreeModel::Row Palette::append_color(ColorPtr clr, Glib::ustring name)
    {
        // add a new row to the end
        Gtk::TreeModel::iterator iter = append();
        Gtk::TreeModel::Row row = *iter;
        // set the columns in the list store
        row[m_modelColumns.m_colColor] = clr;
        if (name.size())
        {
            row[m_modelColumns.m_colText] = name;
        }
        else
        {
            row[m_modelColumns.m_colText] = clr->get_hexstring();
        }
        return row;
    }


    ColorPtr Palette::get_color(int row_idx)
    {
        return children()[row_idx].get_value(m_modelColumns.m_colColor);
    }


    tHexString Palette::get_hexstring(int row_idx)
    {
        return children()[row_idx].get_value(m_modelColumns.m_colText);
    }


    void Palette::reset(void)
    {
        debug("\n*** RESETTING PALETTE ***\n");
        // clear the color definitions
        for (Gtk::TreeModel::iterator iter = children().begin();
                iter != children().end(); iter++)
        {
            debug("resetting shared_ptr<Color>");
            (*iter).get_value(m_modelColumns.m_colColor).reset();
        }
        clear();
        m_paletteColumns = 0;
    }

} // namespace gcs
