/*******************************************************************************
 *  PROJECT: GNOME Colorscheme
 *
 *  AUTHOR: Jonathon Jongsma
 *
 *  Copyright (c) 2005 Jonathon Jongsma
 *
 *  License:
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the 
 *    Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
 *    Boston, MA  02111-1307  USA
 *
 *******************************************************************************/

#include <iostream>
#include <sstream>
#include <fstream>

#include <glibmm/error.h>
#include <gdkmm/cursor.h>

#include "gcs-paletteview.h"
#include "gcs-colorswatch.h"
#include "gcs-debug.h"
#include "gcs-i18n.h"
#include "gcs-conf.h"

namespace gcs
{
    namespace Widgets
    {
        const gint PaletteView::m_minCellSize = 20;
        const gint PaletteView::m_minPaletteWidth = 200;
        const gint PaletteView::m_maxPaletteHeight = 100;

        PaletteView::PaletteView(Glib::ustring filename) :
            Gtk::Expander(), m_table(1, 1, true),
            m_refPalette(Palette::create(filename))
        {
            m_scrolledWindow.set_border_width(Conf::UI_SPACING_SMALL);
            m_scrolledWindow.set_policy(Gtk::POLICY_NEVER, Gtk::POLICY_AUTOMATIC);

            // we need to put the table into an event box so that we can
            // respond to GDK enter and leave signals which are used to change
            // the cursor to a hand and back to a regular pointer
            m_eventBox.add(m_table);
            m_scrolledWindow.add(m_eventBox);

            add(m_scrolledWindow);
            set_label(_("Palette"));

            // set spacing between table cells
            m_table.set_spacings(0);
            
            // connect signals so that we can change the X cursor
            m_eventBox.signal_enter_notify_event().connect(
                    sigc::mem_fun(*this, &PaletteView::on_enter_palette));
            m_eventBox.signal_leave_notify_event().connect(
                    sigc::mem_fun(*this, &PaletteView::on_leave_palette));

            populate();
        }


        void PaletteView::reset(void)
        {
            debug("\n*** RESETTING PALETTE VIEW ***\n");
            // delete the ColorSwatches from the widget
            std::list<Gtk::Widget*> swatches = m_table.get_children();
            for (std::list<Gtk::Widget*>::iterator sw = swatches.begin();
                    sw != swatches.end(); sw++)
            {
                debug("Removing swatch from table");
                m_table.remove(**sw);
                //delete (dynamic_cast<ColorSwatch*>(*sw));
                debug("Deleting swatch");
                delete (*sw);
            }
            m_rows = 0;
        }

        bool PaletteView::set_from_file(Glib::ustring fname)
        {
            if (m_refPalette->set_from_file(fname))
            {
                reset();
                populate();
                return true;
            }
            return false;
        }

        void PaletteView::populate(void)
        {
            hide_all();
            debug("Populating paletteView");
            gint cols = m_refPalette->get_columns();
            if (!cols) // columns wasn't defined by the palette file
            {
                cols = m_minPaletteWidth / m_minCellSize;
            }
            m_rows = m_refPalette->size() / cols;
            // round up -- ugly, i know. FIXME
            if (m_refPalette->size() % cols)
            {
                m_rows += 1;
            }
            m_table.resize(m_rows, cols);
            gint height = m_rows * 25;
            if (height > m_maxPaletteHeight)
            {
                height = m_maxPaletteHeight;
            }
            m_scrolledWindow.set_size_request(-1, height);

            for (gint r = 0; r < m_rows; r++)
            {
                for (gint c = 0; c < cols; c++)
                {
                    debug("Attaching column ", c);
                    gint index = c + r * cols;
                    if (index < m_refPalette->size())
                    {
                        debug("Attaching row ", r);
                        ColorSwatch *pSwatch = (new
                                ColorSwatch(m_refPalette->get_color(index)));
                        pSwatch->set_border_width(0);
                        pSwatch->signal_selected().connect(
                                sigc::bind(sigc::mem_fun(*this,
                                        &PaletteView::on_color_selected), index));
                        m_table.attach(*pSwatch, c, c + 1, r, r + 1);
                    }
                }
            }
            //set_label(m_refPalette->get_name());
            show_all();
        }


        bool PaletteView::on_color_selected(gint index)
        {
            signal_color_selected().emit(m_refPalette->get_color(index));
            debug("PaletteView::on_color_selected");
            return true;
        }


        bool PaletteView::on_enter_palette(GdkEventCrossing* event)
        {
            Glib::RefPtr<Gdk::Window> win = m_eventBox.get_window();
            if (win)
            {
                Glib::RefPtr<Gdk::Display> dpy = Gdk::Display::get_default();
                Gdk::Cursor cursor(dpy, Gdk::HAND2);
                win->set_cursor(cursor);
            }
            return true;
        }


        bool PaletteView::on_leave_palette(GdkEventCrossing* event)
        {
            Glib::RefPtr<Gdk::Window> win = m_eventBox.get_window();
            if (win)
            {
                win->set_cursor();
            }
            return true;
        }

    } // namespace Widgets
} // namespace gcs
