/**********************************************************************
 ** Level_List - maintains the chains of levels for players
 **
 ** Reviewed through:
 **
 **
 ** Copyright (C) 2000 George Noel (Slate)
 **
 **   This program is free software; you can redistribute it and/or modify
 **   it under the terms of the GNU General Public License as
 **   published by the Free Software Foundation; either version 2 of the 
 **   License, or any later version. 
 **
 **   This program is distributed in the hope that it will be useful, but 
 **   WITHOUT ANY WARRANTY; without even the implied warranty of 
 **   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 **   General Public License for more details. 
 ** 
 **   You should have received a copy of the GNU General Public License 
 **   along with this program (in the docs dir); if not, write to the Free
 **   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 **    
 **********************************************************************/

#ifndef LEVEL_LIST_C
#define LEVEL_LIST_C

#include "config.h"
#include "sysdep.h"
#include "strings.h"
#include "mudtypes.h"
#include "level_list.h"
#include "lexer.h"
#include "errlog.h"
#include "object_list.h"
#include "mudobject.h"
#include "specials.h"
#include "code.h"
#include "global.h"
#include "adminflags.h"
#include "newfuncts.h"
#include "memchk.h"
#include "utils.h"

/**************************************************************
 **         Code for the Level_List class 
 **************************************************************/

/***********************************************************************
 ** Level_List (constructor) - creates the levels list, reading in from
 **                            levels.dat and loading into the levels
 **                            chains in the appropriate order
 **
 ** Parameters: None
 **
 ** Returns: None
 **
 **
 ***********************************************************************/

Level_List::Level_List(ErrLog *error_log, Object_List *obj_dbase)
{
   status = load_levels(error_log, obj_dbase);
}


/***********************************************************************
 ** load_levels - loads all level information into memory from levels.dat
 **
 ** Parameters: error_log - the log to write errors to
 **             obj_dbase - the main object database
 **
 ** Returns: 1 for success, -1 for failure
 **
 **
 ***********************************************************************/

int Level_List::load_levels(ErrLog *error_log, Object_List *obj_dbase)
{
   Strings      holder;
   Strings      filename;
   token_record *the_token;
   Level        *new_level;
   FILE         *the_file = NULL;
   Specials     *the_special;

   the_list = NULL;

   filename.sprintf("%s/%s/levels.dat", 
                                     the_config.basedir.str_show(), DATA_DIR);

   /* open the area file for reading */
   if ((the_file = xfopen(filename.str_show(), "r", "level_file")) == NULL)
   {
#ifndef WIN32
      char         user_input[11];
      printf("The Levels data file is missing.  "
             "Would you like to create it? (y/n) ");
      fflush(stdout);

      fgets(user_input, 10, stdin);
      if (user_input[0] == 'n')
      {
         printf("Levels data file must exist.  Exiting.\n");
         exit(0);
      }      
#endif

      if ((the_file = xfopen(filename.str_show(), "w",
                                           "levels_create")) == NULL)
      {
         printf("Fatal Error, could not create levels file!\n");
         exit(0);
      }

      fprintf(the_file, "^%s^\n^^\n^^\n\n", FULLVERSION);
      xfclose(the_file, "levels_create");

      if ((the_file = xfopen(filename.str_show(), "r", 
                                            "level_file")) == NULL)
      {
         error_log->log_err("Error, could not open levels file!", 
                                            "load_levels");
         return -1;
      }
   }
  
   /* first read in the levels header */

   /* first read in the version number and compare, raise an error if
      the area version doesn't match the current version */
   the_token = get_token(the_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
      holder.sprintf("Invalid format in levels header");
      error_log->log_err(holder.str_show(), "load_levels");
      return -1;
   }
   the_token = get_token(the_file, '^');
   if (STRCASECMP(FULLVERSION, the_token->the_string))
   {
     Strings filename;
     Strings directory;

     xfclose(the_file, "load_area_file");

#ifdef WIN32
	 return -2;
#else

     directory.sprintf("%s/%s", the_config.basedir.str_show(), DATA_DIR);
     filename.sprintf("levels.dat");

     if (prompt_user_convert(filename.str_show(), directory.str_show()) != 1)
     {
       holder.sprintf(_("Invalid version '%s' for data file 'levels.dat'.  "
                     "Should be version '%s'.\n"), the_token->the_string,
                                                             FULLVERSION);
       error_log->log_err(holder.str_show(), "load_area");
       return -1;
     }
     else
     {
       return load_levels(error_log, obj_dbase);
     }
#endif
   }

   the_token = get_token(the_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
      error_log->log_err("Invalid format in levels file", "Level_List");
   }

   /* get the readdeny */
   the_token = get_token(the_file, '^');

   /* read in the writeallow list */
   the_token = get_token(the_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
      error_log->log_err("Invalid format in levels file", "Level_List");
   }

   /* get the writeallow */
   the_token = get_token(the_file, '^');

   the_token = get_token(the_file, '\0');
   while (the_token->token_type > 0)
   {
      the_token = get_token(the_file, '\0');
      new_level = new Level(the_token->the_string);
      if (new_level->load_level(the_file, error_log, 0) > 0)
         add_level(new_level);
	  else
	  {
	     delete new_level;
	     the_token = get_token(the_file, '\0');      
		 continue;
	  }

      /* load the special requirements special */
      if (new_level->get_special_req_str() != NULL)
      {
         if ((the_special = obj_dbase->get_special_obj("levels", 
                               new_level->get_special_req_str())) == NULL)
         {
            holder.sprintf(
                      "Level '%s' special '%s' doesn't seem to exist"
                      "or is not a special.", new_level->get_name(), 
                                new_level->get_special_req_str()); 
            error_log->log_err(holder.str_show(), "Level_List");
         }
         else
         {
            new_level->set_special_req(the_special);
         }
      } 

      /* load the special requirements special */
      if (new_level->get_when_awarded_str() != NULL)
      {
         if ((the_special = obj_dbase->get_special_obj("levels", 
                            new_level->get_when_awarded_str())) == NULL)
         {
            holder.sprintf(
                      "Level '%s' special '%s' doesn't seem to exist"
                      "or is not a special.", new_level->get_name(), 
                                new_level->get_when_awarded_str()); 
            error_log->log_err(holder.str_show(), "Level_List");
         }
         else
         {
            new_level->set_when_awarded(the_special);
         }
      } 
      the_token = get_token(the_file, '\0');      
   }

   if (the_file != NULL)
      xfclose(the_file, "level_file");

   return 1;
}


/***********************************************************************
 ** ~Level_List (destructor) - deletes all levels in the list
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 **
 ***********************************************************************/

Level_List::~Level_List()
{
   Level_Chain *tmp_list;

   while (the_list != NULL)
   {
      tmp_list = the_list;
      the_list = the_list->get_next_chain();
      delete_Level_Chain(tmp_list);
   }
}


/***********************************************************************
 ** add_level - adds a level to the level list
 **
 ** Parameters: the_level - the level to add to the list
 **
 ** Returns: 1 if successful, 
 **
 **
 ***********************************************************************/

int Level_List::add_level(Level *the_level)
{
   Level_Chain *tmp_chain;

   if (the_level == NULL)
      return -1;

   tmp_chain = the_list;
   while ((tmp_chain != NULL) && (STRCASECMP(tmp_chain->get_chain_name(), 
                                  the_level->get_chain_name())))
      tmp_chain = tmp_chain->get_next_chain();

   /* if we don't find the chain, we have to create a new one */
   if (tmp_chain == NULL)
   {
      Level_Chain *new_chain;

      new_chain = new Level_Chain(the_level->get_chain_name());
      new_chain->set_next_chain(the_list);
      the_list = new_chain;
      tmp_chain = new_chain;
   }

   return tmp_chain->add_level(the_level);
}


/***********************************************************************
 ** check_levels - checks the player's levels
 **
 ** Parameters: the_player - the player to check for
 **             the_chain - the chain name to check in
 **             the_lvl - the level that we are currently on
 **
 ** Returns: num upgraded for success
 **
 **
 ***********************************************************************/

int Level_List::check_levels(Player *the_player)
{
   Level_Chain *tmp_chain;
   int         cur_lvl;
   int         count = 0;

   tmp_chain = the_list;
   while (tmp_chain != NULL)
   {
      cur_lvl = the_player->check_levels(tmp_chain->get_chain_name());

      count += tmp_chain->check_chain(the_player, cur_lvl);
      tmp_chain = tmp_chain->get_next_chain();
   }

   return count;
}


/***********************************************************************
 ** get_level - gets a level based on the chain and level number
 **
 ** Parameters: the_chain - the chain name to check in
 **             the_lvl - the level that we are currently on
 **
 ** Returns: pointer to the level
 **
 **
 ***********************************************************************/

Level *Level_List::get_level(char *the_chain, int the_lvl)
{
   Level_Chain *tmp_chain;

   tmp_chain = the_list;
   while ((tmp_chain != NULL) && (STRCASECMP(tmp_chain->get_chain_name(), 
                                  the_chain)))
      tmp_chain = tmp_chain->get_next_chain();

   if (tmp_chain == NULL)
      return NULL;

   return tmp_chain->get_level(the_lvl);
}


/***********************************************************************
 ** show_chains - displays stats to the player on the level chains
 **
 ** Parameters: the_player - the player to send the info to
 **
 ** Returns: num of chains shown
 **
 **
 ***********************************************************************/

int Level_List::show_chains(Player *the_player)
{
   Level_Chain *tmp_chain;
   Strings     the_tabs;
   int         name_len;
   int         counter = 0;

   the_player->send_plr(
           "&+GChain Name          Number of Levels\n"
           "&+B----------------------------------------------&*\n");

   tmp_chain = the_list;
   while (tmp_chain != NULL)
   {
   
      name_len = strlen(tmp_chain->get_chain_name());
      if ((name_len > 0) && (name_len < 7))
         the_tabs = "\t\t\t";
      else if ((name_len > 6) && (name_len < 14))
         the_tabs = "\t\t";
      else
         the_tabs = "\t";

      the_player->send_plr("&+g%s%s&+W   %d&*\n", tmp_chain->get_chain_name(), 
                         the_tabs.str_show(), tmp_chain->get_num_levels()); 
      counter++;
      tmp_chain = tmp_chain->get_next_chain();
   }
   the_player->send_plr(
              "&+B----------------------------------------------&*\n");
   the_player->send_plr("&+GNum Chains: &+W%d&*\n\n", counter);
   return counter;
}


/***********************************************************************
 ** show_levels - shows the levels for a specific chain
 **
 ** Parameters: the_player - the player to send the info to
 **             the_chain - the chain to show the levels of
 **
 ** Returns: 
 **
 **
 ***********************************************************************/

int Level_List::show_levels(Player *the_player, char *the_chain)
{
   Level_Chain *tmp_chain;

   tmp_chain = the_list;
   while ((tmp_chain != NULL) && (STRCASECMP(tmp_chain->get_chain_name(), 
                                  the_chain)))
      tmp_chain = tmp_chain->get_next_chain();

   if (tmp_chain == NULL)
      return -1;

   return tmp_chain->show_levels(the_player);
}

/**************************************************************
 **         Code for the Level_Chain class 
 **************************************************************/ 


/***********************************************************************
 ** Level_Chain (constructor) - creates the chain of levels and manages
 **                             the list
 **
 ** Parameters: the_name - the name of the chain
 **
 ***********************************************************************/

Level_Chain::Level_Chain(char *the_name)
{
   set_chain_name(the_name);
   the_chain = NULL;
   next_chain = NULL;
}


/***********************************************************************
 ** ~Level_Chain (destructor) - deletes all levels in the chain
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 **
 ***********************************************************************/

Level_Chain::~Level_Chain()
{
   chain_link *tmp_link;

   while (the_chain != NULL)
   {
      tmp_link = the_chain;
      the_chain = the_chain->next_link;
      delete tmp_link->the_level;
      delete_chain_link(tmp_link);
   }
}


/***********************************************************************
 ** set_chain_name - sets the chain name of this chain
 **
 ** Parameters: the_name - the name to set to
 **
 ***********************************************************************/

void Level_Chain::set_chain_name(char *the_name)
{
   chain_name = the_name;
}


/***********************************************************************
 ** get_chain_name - gets the chain name of this chain
 **
 ***********************************************************************/

char *Level_Chain::get_chain_name(void)
{
   return chain_name.str_show();
}


/***********************************************************************
 ** get_next_chain - gets the next chain in the list
 **
 ***********************************************************************/

Level_Chain *Level_Chain::get_next_chain()
{
   return next_chain;
}

/***********************************************************************
 ** set_next_chain - sets the next chain in the list
 **
 ** Parameters: new_chain - the name of the chain to set next to
 **
 ***********************************************************************/

void Level_Chain::set_next_chain(Level_Chain *new_chain)
{
   next_chain = new_chain;
}


/***********************************************************************
 ** add_level - adds a level to the chain
 **
 ** Parameters: the_level - the level to add to the chain
 **
 ** Returns: 1 if successful, 
 **
 ***********************************************************************/

int Level_Chain::add_level(Level *the_level)
{
   int        this_lvl;
   chain_link *tmp_link;
   chain_link *new_link;

   if (the_level == NULL)
      return -1;

   this_lvl = the_level->get_lvl_num();

   new_link = new_chain_link();
   new_link->the_level = the_level;
   new_link->next_link = NULL;

   tmp_link = the_chain;
   if ((tmp_link == NULL) || ((tmp_link->the_level)->get_lvl_num() > this_lvl))
   {
      new_link->next_link = the_chain;
      the_chain = new_link;
      return 1;
   }
   else if ((tmp_link->the_level)->get_lvl_num() == this_lvl)
   {
      delete_chain_link(new_link);
      return -2;
   }

   while ((tmp_link != NULL) && (tmp_link->next_link != NULL))
   {
      if (((tmp_link->next_link)->the_level)->get_lvl_num() > this_lvl)
      {
         new_link->next_link = tmp_link->next_link;
         tmp_link->next_link = new_link;
         return 1;
      }
      else if (((tmp_link->next_link)->the_level)->get_lvl_num() == this_lvl)
      {
         delete_chain_link(new_link);
         return -2;
      }
      tmp_link = tmp_link->next_link;
   }
   
   tmp_link->next_link = new_link;

   return 1;
}


/***********************************************************************
 ** check_chain - checks the level in the chain to see if we can upgrade
 **               this player another level 
 **
 ** Parameters: the_player - the player to check this chain for
 **             the_lvl - the level we are currently at
 **
 ** Returns: 1 if successful, -1 if failed
 **
 **
 ***********************************************************************/

int Level_Chain::check_chain(Player *the_player, int the_lvl)
{
   chain_link     *tmp_link;
   special_holder *the_special;
   in_params      fill_param;
   Level          *the_level;
   Strings        holder;

   tmp_link = the_chain;
   while ((tmp_link != NULL) && 
                         (the_lvl >= (tmp_link->the_level)->get_lvl_num()))
   {
      tmp_link = tmp_link->next_link;
   }

   if (tmp_link == NULL)
      return 0;

   the_level = tmp_link->the_level;

   /* start performing checks */
   if ((the_level->get_min_str() > the_player->get_strength()) ||
       (the_level->get_min_dex() > the_player->get_dexterity()) ||
       (the_level->get_min_intel() > the_player->get_intel()) ||
       (the_level->get_min_con() > the_player->get_constitution()) ||
       (the_level->get_min_wisdom() > the_player->get_wisdom()) ||
       (the_level->get_min_cha() > the_player->get_charisma()) ||
       (the_level->get_min_exp() > the_player->get_exp()))
      return 0;

   /**** the check for required abilities goes here *****/

   /* now check the special requirements special for success */
   if ((the_special = the_level->get_special_req()) != NULL)
   {
      fill_param.primary_obj = NULL;
      fill_param.secondary_obj = NULL;
      fill_param.this_obj = the_player;

      if ((the_special->the_special)->run_special(the_player, &fill_param, 
                                            &the_special->environment) == 3)
         return 0;
   }

   /* if no problems, we upgrade the player's level */
   the_player->change_level(get_chain_name(), the_level->get_lvl_num());

   holder.sprintf("&+Y[&+M%s&+W has achieved %s level &+Y%d&+Y]&*\n",
          the_player->get_name(), get_chain_name(), the_level->get_lvl_num());

   mainstruct->send_all_players(holder.str_show(), NULL, 
                                                     ADMINFLAG_SEELEVEL);

   the_player->send_plr("%s\n", the_level->get_award_string());
   
   /* now execute the when awarded special to award attributes */
   if ((the_special = the_level->get_when_awarded()) != NULL)
   {
      fill_param.primary_obj = NULL;
      fill_param.secondary_obj = NULL;
      fill_param.this_obj = the_player;

      if ((the_special->the_special)->run_special(the_player, &fill_param, 
                                            &the_special->environment) == 3)
         return 1;
   }

   /* lets make sure the playerlist is in the right order now */
   mainstruct->sort_playerlist("exp");

   return 1;
}


/***********************************************************************
 ** get_level - gets a level based on the level number
 **
 ** Parameters: the_lvl - the level that we are currently on
 **
 ** Returns: pointer to the level
 **
 **
 ***********************************************************************/

Level *Level_Chain::get_level(int the_lvl)
{
   chain_link *tmp_link;
   
   tmp_link = the_chain;
   while ((tmp_link != NULL) && 
                         (the_lvl > (tmp_link->the_level)->get_lvl_num()))
   {
      tmp_link = tmp_link->next_link;
   }
   if (tmp_link == NULL)
      return NULL;

   return tmp_link->the_level;
}


/***********************************************************************
 ** get_num_levels - gets the number of levels in this chain
 **
 ***********************************************************************/

int Level_Chain::get_num_levels()
{
   chain_link *tmp_link;
   int counter = 0;
   
   tmp_link = the_chain;
   while (tmp_link != NULL)
   {
      tmp_link = tmp_link->next_link;
      counter++;
   }
   return counter;
}


/***********************************************************************
 ** show_levels - displays stats on the levels in the chain to the player
 **
 ** Parameters: the_player - the player to send the info to
 **
 **
 ***********************************************************************/

int Level_Chain::show_levels(Player *the_player)
{
   chain_link  *tmp_link;
   Strings     the_tabs;
   int         counter = 0;
   Level       *tmp_lvl;

   the_player->send_plr("&+gLevels for &+G%s&+g chain:\n\n", 
                                 get_chain_name());
   the_player->send_plr(
   "&+G LevelNum MinStr MinDex MinCon MinIntel MinWis MinCha MinExp Required Special\n"
   "&+G                                                             Abilities Criteria \n"
   "&+B-------------------------------------------------------------------------------&*");

   tmp_link = the_chain;
   while (tmp_link != NULL)
   {
      tmp_lvl = tmp_link->the_level;
      the_player->send_plr(
       "&+W     %d&*\t     &+Y%d\t     %d\t      %d\t      %d\t%s\t   %s\n",
         tmp_lvl->get_lvl_num(), 
         tmp_lvl->get_min_str(), tmp_lvl->get_min_dex(), tmp_lvl->get_min_con(),
       tmp_lvl->get_min_intel(), tmp_lvl->get_min_wisdom(), tmp_lvl->get_min_cha(),
         tmp_lvl->get_min_exp(),
        (tmp_lvl->get_req_abilities() == NULL) ? "&+gNone&*" : "&+GYes&*",
        (tmp_lvl->get_special_req_str() == NULL) ? "&+gNone&*" : "&+GYes&*"); 

      tmp_link = tmp_link->next_link;
      counter++;
   }
  
   the_player->send_plr(
    "&+B-------------------------------------------------------------------&*"
    "\n");
   the_player->send_plr("&+GNum Levels: &+W%d&*\n\n", counter);
   return counter;
}


/***********************************************************************
 ** get_status - gets whether or not this loaded successfully
 **
 ***********************************************************************/

int Level_List::get_status()
{
	return status;
}

#endif

