/**********************************************************************
 ** login functions: These functions are used to login a player, as well
 **                  as provide a non-class function so that we can use
 **                  it to create a pthread for the new player login
 **
 **
 **    
 **    Reviewed through: version 0.14
 **
 **
 ** Copyright (C) 2000 George Noel (Slate), Ed Boraas
 **
 **   This program is free software; you can redistribute it and/or modify
 **   it under the terms of the GNU General Public License as
 **   published by the Free Software Foundation; either version 2 of the 
 **   License, or any later version. 
 **
 **   This program is distributed in the hope that it will be useful, but 
 **   WITHOUT ANY WARRANTY; without even the implied warranty of 
 **   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 **   General Public License for more details. 
 ** 
 **   You should have received a copy of the GNU General Public License 
 **   along with this program (in the docs dir); if not, write to the Free
 **   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 **
 **********************************************************************/

#include "config.h"
#include "sysdep.h"
#include "strings.h"
#include "mudtypes.h"
#ifdef WIN32
#include "../win32/winconnection.h"
#else
#include "connection.h"
#endif
#include "mudobject.h"
#include "location.h"
#include "individual.h"
#include "player.h"
#include "builder.h"
#include "login.h"
#include "user_dbase.h"
#include "global.h"
#include "utils.h"
#include "flags.h"
#include "comflags.h"
#include "adminflags.h"
#include "gameflags.h"
#include "pager.h"
#include "newfuncts.h"
#include "inp_funct.h"
#include "access_list.h"

/***********************************************************************
 ** login_player - login the player and load the object with the info
 **
 ** Parameters: the_player - the player structure to be used by this player
 **
 ** Returns: Nothing
 **
 ***********************************************************************/


void *login_player(void *the_player)
{
   Player      *this_player; /* holds the player pointer */
   Access_List *tmp_access_list;
   Strings     the_file;
   Pager       *the_pager;
   Inp_Handler *tmp_handler;
   Strings     holder;
   Strings     host;
   int         found;

   /* make a new player object */
   this_player = (Player *) the_player;

#ifndef WIN32
   this_player->find_host_name();
#endif

   /* look to see if this player is denied */
   tmp_access_list = mainstruct->get_game_access_list();
   if ((this_player->get_host_addr() == NULL) || 
       (!(found = tmp_access_list->find_in_list(this_player->get_host_addr()))))
      found = tmp_access_list->find_in_list(this_player->get_ip_addr());

   if (((found) && (the_config.conf_flags->get_flag(CF_GAMEALLOW))) ||
       ((!found) && (!the_config.conf_flags->get_flag(CF_GAMEALLOW))))
   {
      this_player->
         send_plr("Error - could not connect your host successfully.\n");
      this_player->set_off();
      holder.sprintf(
          "&+WEntry &+Rattempt&* at game port by banned host: &+c%s&*", 
          (this_player->get_host_addr() == NULL) ? 
	     this_player->get_ip_addr() : this_player->get_host_addr());
      holder.sprintf("&+Y[&+WEntry &+Rattempt&* at game port by "
                     "banned host: &+c%s&+Y]&*\n", 
                  (this_player->get_host_addr() == NULL) ? 
	       this_player->get_ip_addr() : this_player->get_host_addr());
      mainstruct->send_all_players(holder.str_show(), NULL,ADMINFLAG_SEETCP);
      mainstruct->log_event(holder.str_show());
      return NULL;
   }

   holder.sprintf("Connection at game port by host: %s", 
          (this_player->get_host_addr() == NULL) ? 
	     this_player->get_ip_addr() : this_player->get_host_addr());
   mainstruct->log_event(holder.str_show());
   holder.sprintf("&+Y[&+WConnection at game port by host: &+c%s&+Y]&*\n", 
               (this_player->get_host_addr() == NULL) ? 
	         this_player->get_ip_addr() : this_player->get_host_addr());
   mainstruct->send_all_players(holder.str_show(), NULL, ADMINFLAG_SEETCP);

   /* Send connection confirmation */
   this_player->send_plr("Connected to %s\n", the_config.mudname.str_show());

   tmp_handler = this_player->get_input_handler();

   // Now check to see if listening ports are combined and the player has access to
   // the builder port, if so, give them a menu choice
   if (mainstruct->are_ports_combined())
   {
	   tmp_access_list = mainstruct->get_bldr_access_list();
	  if ((this_player->get_host_addr() == NULL) || 
		   (!(found = tmp_access_list->find_in_list(this_player->get_host_addr()))))
	      found = tmp_access_list->find_in_list(this_player->get_ip_addr());

      if (((found) && (!the_config.conf_flags->get_flag(CF_BLDRALLOW))) ||
          ((!found) && (the_config.conf_flags->get_flag(CF_BLDRALLOW))))
	  {
		 this_player->send_plr("\nAvailable Ports:\n"
			                  "   1. Game Port\n"
							  "   2. Builder Port\n\n");
         tmp_handler->push_input_handler(port_menu, "Which port would you like to enter? ", 0);
	     return NULL;
	  }
   }

   /* first we push this on, so that when we pop the login banner, this
      input handler is waiting to be executed */
   tmp_handler->push_input_handler(get_pname, "What shall I call you? ", 0);
   the_file.sprintf("%s/%s%s%s", the_config.basedir.str_show(), 
                                    BANNERDIR, "gamebanner", BANNEREXTENTION);

   /* now we push on the login banner */
   the_pager = new_Pager(the_file.str_show(), this_player->get_pager_lines());
   if (the_pager && the_pager->file_available())
   { 
       if (the_pager->paged_read(this_player->get_connection()))
          delete_Pager(the_pager);
       else
       {
          tmp_handler->push_input_handler(pager_funct, 
             "&+c<&+G******************&+Bpress return to continue, "
             "q to quit&+G**********************&+c>&*", HANDLER_DATA_PAGER);
          tmp_handler->set_data((void *) the_pager);
       }
   }    
   else
   {
      this_player->send_plr("Error displaying login banner.\n");
      mainstruct->log_error("Error displaying login banner.\n", 
                                                         "login_player");
   }

   return NULL; 
}


/***********************************************************************
 ** login_builder - login the builder and load the object with the info
 **
 ** Parameters: the_builder - the builder structure to be used by this player
 **
 ** Returns: Nothing
 **
 ***********************************************************************/


void *login_builder(void *the_builder)
{
   Builder *this_builder; /* holds the builder pointer */
   Access_List *tmp_access_list;
   Pager *the_pager;
   Strings the_file;
   Inp_Handler  *tmp_handler;
   Strings holder;      
   int     found;

   /* make a new builder object */
   this_builder = (Builder *) the_builder;

   /* mark us as successfully in the thread */
   this_builder->set_in_thread(1);

//   this_builder->find_host_name();

   tmp_access_list = mainstruct->get_bldr_access_list();
   if ((this_builder->get_host_addr() == NULL) || 
       (!(found = tmp_access_list->find_in_list(this_builder->get_host_addr()))))
      found = tmp_access_list->find_in_list(this_builder->get_ip_addr());

   if (((found) && (the_config.conf_flags->get_flag(CF_BLDRALLOW))) ||
       ((!found) && (!the_config.conf_flags->get_flag(CF_BLDRALLOW))))
   {
      this_builder->send_bldr("You do not have permission to connect"
                   " to the builder port.\n"
                   "Talk to an admin at port %hd for entry.\n", 
                                                          the_config.gameport);
      this_builder->set_off();
      holder.sprintf(
        "&+WEntry &+Rattempt&* at builder port by unauthorized host: &+c%s&*", 
          (this_builder->get_host_addr() == NULL) ? 
	     this_builder->get_ip_addr() : this_builder->get_host_addr());
      mainstruct->log_event(holder.str_show());
      holder.sprintf("&+Y[&+WEntry &+Rattempt&* at builder port by "
                     "unauthorized host: &+c%s&+Y]&*\n", 
                  (this_builder->get_host_addr() == NULL) ? 
	       this_builder->get_ip_addr() : this_builder->get_host_addr());
      mainstruct->send_all_players(holder.str_show(), NULL, 
                                                          ADMINFLAG_SEETCP);

      return NULL;
   }

   holder.sprintf("Connection at builder port by host: %s", 
          (this_builder->get_host_addr() == NULL) ? 
	     this_builder->get_ip_addr() : this_builder->get_host_addr());
   mainstruct->log_event(holder.str_show());

   holder.sprintf("&+Y[&+WConnection at bldr port by host: &+c%s&+Y]&*\n", 
               (this_builder->get_host_addr() == NULL) ? 
	    this_builder->get_ip_addr() : this_builder->get_host_addr());
   mainstruct->send_all_players(holder.str_show(), NULL, ADMINFLAG_SEETCP);


   /* Send connection confirmation */
   this_builder->send_bldr("Connected to the builder port of Newmud.\n");

   the_file.sprintf("%s/%s%s%s", the_config.basedir.str_show(), 
                                  BANNERDIR, "bldrbanner", BANNEREXTENTION);

   tmp_handler = this_builder->get_input_handler();

   /* first we push this on, so that when we pop the login banner, this
      input handler is waiting to be executed */   
   tmp_handler->push_input_handler(get_bname, "What shall I call you? ", 0);

   /* now we push on the login banner */
   the_pager = new_Pager(the_file.str_show(), this_builder->get_pager_lines());
   if (the_pager && the_pager->file_available())
   { 
       if (the_pager->paged_read(this_builder->get_connection()))
       {
          delete_Pager(the_pager);
       }
       else
       {
          tmp_handler->push_input_handler(pager_funct, 
             "&+c<&+G******************&+Bpress return to continue, "
             "q to quit&+G**********************&+c>&*", HANDLER_DATA_PAGER);
          tmp_handler->set_data((void *) the_pager);
       }
   }    
   else
   {
      this_builder->send_bldr("Error displaying login banner.\n");
      mainstruct->log_error("Error displaying login banner.\n", 
                                                         "login_builder");
   }

   return NULL; 
}


/***********************************************************************
 ** valid_name - checks to see if a string is a valid name
 **
 ** Parameters: the_name - the name to check
 **
 ** Returns: 1 if valid
 **          0 if invalid
 **
 ***********************************************************************/

int valid_name(char *the_name)
{
   char *tmp_str;

   if (the_name == NULL)
      return 0;

   tmp_str = the_name;

   while (*tmp_str)
   {
      if ((!isalpha(*tmp_str)) || (*tmp_str == ' '))
         return 0;
      tmp_str++;
   }

   return 1;
}


/***********************************************************************
 ** validate_passwd - ensures the password meets certain criteria for a
 **                   secure password
 **
 ** Parameters: the_player - who we send messages to
 **             the_passwd - the password they entered
 **
 ** Returns: 1 if valid
 **          0 if invalid
 **
 ***********************************************************************/

int validate_passwd(Player *the_player, char *the_passwd)
{
   if (strlen(the_passwd) < (unsigned int) the_config.minpasswdlen)
   {
      the_player->send_plr("\nYour password must be at least %d "
                            "characters in length.\n", the_config.minpasswdlen);
      return 0;
   }
   return 1;
}


