/*
    This file is part of AirSnort.

    AirSnort is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    AirSnort is distributed in the hope that it will be useful,
      but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with AirSnort; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


#include <string.h>
#include <stdlib.h>
#ifndef WIN32
#include <unistd.h>
#else
#include <windows.h>
#endif
#include <stdio.h>
#include "bssidlist.h"
#include "crc-32.h"
#include "capture.h"
#include "crack.h"

BssidList *head = 0;
BssidList *tail = 0;

extern int listCount;

int isResolved(const unsigned char *p) {
   unsigned char sum;
   if (p[1] == 255 && p[0] > 2 && p[0] < 16) return 1;
   sum = p[0] + p[1];
   if (sum == 1 && (p[2] <= 0x0A || p[2] == 0xFF)) return 1;
   if (sum <= 0x0C && (p[2] >= 0xF2 && p[2] <= 0xFE && p[2] != 0xFD)) return 1;
   return 0;
//   return p[1] == 255 && p[0] > 2 && p[0] < 16;
}

char *ivtostr(const unsigned char *iv) {
   static char ivstr[10];
   sprintf(ivstr, "%2.2X:%2.2X:%2.2X", iv[0], iv[1], iv[2]);
   return ivstr;
}

char *bssidtostr(const unsigned char *bssid) {
   static char bssidstr[20];
   sprintf(bssidstr, "%2.2X:%2.2X:%2.2X:%2.2X:%2.2X:%2.2X", 
                    bssid[0], bssid[1], bssid[2], bssid[3], bssid[4], bssid[5]);
   return bssidstr;
}

int bssidMatch(const unsigned char *bssid1, const unsigned char *bssid2) {
   int i = 5;
   for (; i >= 0; i--) {
      if (bssid1[i] != bssid2[i]) return 0;
   }
   return 1;
}

BssidList *bssidFind(const unsigned char *bssid) {
   BssidList *temp = head;
   for (; temp; temp = temp->next) {
      if (bssidMatch(temp->bssid, bssid)) {
         break;
      }
   }
   return temp;
}

BssidList *rowFind(int row) {
   BssidList *temp = head;
   for (; temp; temp = temp->next) {
      if (row == temp->rownum) {
         break;
      }
   }
   return temp;
}

//the capture thread runs through here to see if it should try a new crack.
//A terminating crack thread also runs through here to see if enough new packets
//have been added to warrent starting a new thread.
void checkThread(BssidList *apNode) {
#ifndef WIN32
   sem_wait(&apNode->crackSem);
   if ((apNode->crackerThread == 0) && ((apNode->queueLen > 9) || (apNode->channelChanged))) {
      pthread_create(&apNode->crackerThread, NULL, cracker, apNode->ap);
   }
   sem_post(&apNode->crackSem);
#else
   int threadId;
   WaitForSingleObject(apNode->crackSem, INFINITE);
   if ((apNode->crackerThread == 0) && ((apNode->queueLen > 9) || (apNode->channelChanged))) {
      apNode->crackerThread = CreateThread(NULL, 0, cracker, apNode->ap, 0, &threadId);
   }
   ReleaseSemaphore(apNode->crackSem, 1, NULL);
#endif
}

//ap should be a valid BssidList node
void addPacket(BssidList *apNode, CaptureRec *cap, int isData) {
   PacketInfo *pi = cap->pInfo;
   if (!apNode) {
      apNode = bssidFind(cap->bssid);
      if (!apNode) {   
         apNode = addBssid(cap);
      }
   }
   if (!apNode->usingWep && pi->wep) {
      addWep(apNode);
   }
   if (pi->channel > 0) {
      apNode->channel = pi->channel;
   }
   if (apNode->usingWep && isData) {
      memcpy(apNode->lastiv, cap->iv, 3);
      apNode->numEncrypted++;
      if (pi->pack) {
         if (!apNode->ap->cracked) {
            enqueuePacket(apNode->ap, pi->pack);
            checkThread(apNode);
         }
      }
   }
   apNode->lastSeen = pi->rxTime;
   apNode->numPackets++;
}

void addWep(BssidList *apNode) {
   apNode->usingWep = 1;
   apNode->ap = newCrackNode(apNode);
#ifndef WIN32
   sem_init(&(apNode->crackSem), 0, 1);
#else
   apNode->crackSem = CreateSemaphore(NULL, 1, 1, NULL);
#endif
}

//check for non-existence of bssid prior to calling this please
BssidList *addBssid(CaptureRec *cap) {
   BssidList *temp = (BssidList*) calloc(1, sizeof(BssidList));
   memcpy(temp->bssid, cap->bssid, 6);
   temp->channel = cap->pInfo->channel;
   temp->name = cap->pInfo->name;
   temp->rownum = tail ? tail->rownum + 1 : 0;
   if (tail) {
      tail = tail->next = temp;
   }
   else {
      head = tail = temp;
   }
   if (cap->pInfo->wep) {
      temp->usingWep = 1;
      temp->ap = newCrackNode(temp);
#ifndef WIN32
      sem_init(&(temp->crackSem), 0, 1);
#else
      temp->crackSem = CreateSemaphore(NULL, 1, 1, NULL);
#endif
   }
   return temp;
}

void clearList() {
   BssidList *temp = head, *next;
#ifndef WIN32
   void *result;
#endif
   for (; temp; temp = next) {
      next = temp->next;
      if (temp->crackerThread) {
         temp->ap->die = 1;
#ifndef WIN32
         pthread_join(temp->crackerThread, &result);
#else
         WaitForSingleObject(temp->crackerThread, INFINITE);
#endif
         destroyCrackNode(temp->ap);
      }
      free(temp->name);
      free(temp);
   }
   listCount = 0;
   head = tail = NULL;
}

