;; AIscm - Guile extension for numerical arrays and tensors.
;; Copyright (C) 2013, 2014, 2015, 2016, 2017 Jan Wedekind <jan@wedesoft.de>
;;
;; This program is free software: you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation, either version 3 of the License, or
;; (at your option) any later version.
;;
;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;;
;; You should have received a copy of the GNU General Public License
;; along with this program.  If not, see <http://www.gnu.org/licenses/>.
;;
(use-modules (oop goops)
             (rnrs bytevectors)
             (srfi srfi-1)
             (srfi srfi-26)
             (aiscm util)
             (aiscm asm)
             (aiscm mem)
             (aiscm jit)
             (aiscm element)
             (aiscm int)
             (aiscm float)
             (aiscm pointer)
             (aiscm sequence)
             (aiscm bool)
             (aiscm rgb)
             (aiscm obj)
             (aiscm complex)
             (guile-tap))
(define ctx (make <context>))
(ok (not (native-equivalent (rgb 1 2 3)))
    "RGB does not have a native equivalent")
(ok (eq? <int> (native-equivalent <int>))
    "integer is it's own native equivalent")
(ok (eq? <sint> (native-equivalent <sint>))
    "short integer is it's own native equivalent")
(ok (eq? <ubyte> (native-equivalent <bool>))
    "byte is native equivalent of boolean")
(ok (eq? <ulong> (native-equivalent (pointer <ubyte>)))
    "native equivalent of pointer is a 64 bit integer")
(ok (eq? <ulong> (native-equivalent <obj>))
    "native equivalent of Scheme reference is a 64 bit integer")
(ok (eq? <float> (native-equivalent <float>))
    "floating point number is it's own equivalent")
(ok (eq? <double> (native-equivalent <double>))
    "single-precision floating point number is it's own equivalent")
(let [(a (var <int>))
      (b (var <int>))
      (p (var <long>))]
  (ok (equal? (list b) (input (MOV a b)))
      "Get input variables of MOV")
  (ok (equal? (list a b) (input (ADD a b)))
      "Get input variables of ADD")
  (ok (equal? (list a) (input (ADD a a)))
      "Prevent duplication of input variables")
  (ok (equal? (list a) (output (MOV a b)))
      "Get output variables of MOV")
  (ok (equal? (list a) (output (ADD a b)))
      "Get output variables of ADD")
  (ok (equal? (list b a) (input (MOV (ptr <int> a) b)))
    "Get input variables of command writing to address")
  (ok (equal? (list a 0) (get-args (MOV a 0)))
      "Get arguments of command")
  (ok (equal?  (list a b) (variables (list (MOV a 0) (MOV b a))))
      "Get variables of a program")
  (ok (equal?  (list a p) (variables (list (MOV a 0) (MOV (ptr <int> p) a))))
      "Get variables of a program using a pointer")

  (ok (null? (get-ptr-args (MOV a 42)))
      "command without any pointer arguments")
  (ok (equal? (list p) (get-ptr-args (MOV (ptr <int> p) 42)))
      "get pointer argument of a command")
  (ok (null? (get-ptr-args (MOV EAX 42)))
      "compiled command without variables does not have pointer arguments")
  (ok (equal? (list p) (get-ptr-args (MOV (ptr <int> p 16) 42)))
      "only return variable arguments of pointer")

  (ok (equal? (list (MOV ECX 42)) (substitute-variables (list (MOV a 42)) (list (cons a RCX))))
      "Substitute integer variable with register")
  (ok (equal? (MOV b 0) (substitute-variables (MOV a 0) (list (cons a b))))
      "Substitute variable with another")
  (ok (equal? (MOV ECX EDX) (substitute-variables (MOV a b) (list (cons a RCX) (cons b RDX))))
      "Substitution works with 'MOV'")
  (let [(p (var <long>))]
    (ok (equal? (list (MOV RCX (ptr <long> RAX)))
                (substitute-variables (list (MOV p (ptr <long> RAX))) (list (cons p RCX))))
        "Substitute long integer variable with register")
    (ok (equal? (ptr <int> RCX) (substitute-variables (ptr <int> p) (list (cons p RCX))))
        "Substitute pointer variable with register")
    (ok (equal? (ptr <int> p 2) (substitute-variables (ptr <int> p 2) '()))
        "Pass through pointer with empty substitution")
    (ok (equal? (ptr <int> RCX 5) (substitute-variables (ptr <int> p 2) (list (cons p (cons RCX 3)))))
        "Substitute pointer variable with register and offset"))
  (let [(l (var <long>))
        (w (var <sint>))]
    (ok (equal? (MOVSX RCX EDX) (substitute-variables (MOVSX l a) (list (cons l RCX) (cons a RDX))))
        "Substitution works with 'MOVSX'")
    (ok (equal? (MOVZX ECX DX) (substitute-variables (MOVZX a w) (list (cons a RCX) (cons w RDX))))
        "Substitution works with 'MOVZX'"))
  (let [(p (var <long>))
        (q (var <long>))]
    (ok (equal? (LEA RCX (ptr <byte> RDX))
                (substitute-variables (LEA p (ptr <byte> q)) (list (cons p RCX) (cons q RDX))))
        "Substitution works with 'LEA"))
  (ok (equal? (SHL ECX) (substitute-variables (SHL a) (list (cons a RCX))))
      "Substitution works with 'SHL")
  (ok (equal? (SHR ECX) (substitute-variables (SHR a) (list (cons a RCX))))
      "Substitution works with 'SHR")
  (ok (equal? (SAL ECX) (substitute-variables (SAL a) (list (cons a RCX))))
      "Substitution works with 'SAL")
  (ok (equal? (SAR ECX) (substitute-variables (SAR a) (list (cons a RCX))))
      "Substitution works with 'SAR")
  (ok (equal? (ADD ECX EDX) (substitute-variables (ADD a b) (list (cons a RCX) (cons b RDX))))
      "Substitution works with 'ADD'")
  (ok (equal? (PUSH ECX) (substitute-variables (PUSH a) (list (cons a RCX))))
      "Substitution works with 'PUSH'")
  (ok (equal? (POP ECX) (substitute-variables (POP a) (list (cons a RCX))))
      "Substitution works with 'POP'")
  (ok (equal? (NEG ECX) (substitute-variables (NEG a) (list (cons a RCX))))
      "Substitution works with 'NEG'")
  (ok (equal? (SUB ECX EDX) (substitute-variables (SUB a b) (list (cons a RCX) (cons b RDX))))
      "Substitution works with 'SUB'")
  (ok (equal? (IMUL ECX EDX) (substitute-variables (IMUL a b) (list (cons a RCX) (cons b RDX))))
      "Substitution works with 'IMUL'")
  (ok (equal? (IMUL ECX EDX 2) (substitute-variables (IMUL a b 2) (list (cons a RCX) (cons b RDX))))
      "Substitution works with 'IMUL' and three arguments")
  (ok (equal? (CMP ECX EDX) (substitute-variables (CMP a b) (list (cons a RCX) (cons b RDX))))
      "Substitution works with 'CMP'")
  (let [(u (var <ubyte>))
        (x (var <sint>))]
    (ok (equal? (SETB CL) (substitute-variables (SETB u) (list (cons u RCX))))
        "Substitution works with 'SETB'")
    (ok (equal? (MOVZX DI (ptr <ubyte> RSP 24))
                (substitute-variables (mov-unsigned x u) (list (cons x RDI) (cons u (ptr <long> RSP 24)))))
        "Use correct type when substituting variable with pointer")))
(ok (equal? (MOV AX CX) (mov-signed AX CX))
    "copy signed 16-bit value")
(ok (equal? (MOVSX EAX CX) (mov-signed EAX CX))
    "copy with sign extension")
(ok (equal? (MOV CL SIL) (mov-signed CL ESI))
    "copy part of signed value")
(ok (equal? (MOV AX CX) (mov-unsigned AX CX))
    "copy unsigned 16-bit value")
(ok (equal? (MOVZX EAX CX) (mov-unsigned EAX CX))
    "copy with zero extension")
(ok (equal? (MOV CL SIL) (mov-unsigned CL ESI))
    "copy part of unsigned value")
(ok (equal? (MOV EAX ECX) (mov-unsigned RAX ECX)); TODO: map to MOV EAX ECX
    "zero-extending 32-bit value is done by default")
(ok (equal? '((a . 1) (b . 3)) (labels (list (JMP 'a) 'a (MOV AX 0) 'b (RET))))
    "'labels' should extract indices of labels")
(ok (equal? '((a . (0 . 3)) (b . (1 . 2))) (sort-live-intervals '((a . (0 . 3)) (b . (1 . 2))) '()))
    "pass through live intervals if they are already sorted")
(ok (equal? '((b . (1 . 3)) (a . (2 . 2))) (sort-live-intervals '((a . (2 . 2)) (b . (1 . 3))) '()))
    "sort live intervals by start point")
(ok (equal? '((a . (1 . 3)) (b . (0 . 2))) (sort-live-intervals '((b . (0 . 2)) (a . (1 . 3))) '(a)))
    "prioritise predefined variables when sorting live intervals")
(ok (equal? '((tmp . (0 . 0)) (a . (0 . 1))) (sort-live-intervals '((a . (0 . 1)) (tmp . (0 . 0))) '()))
    "sort live intervals by length if start point is the same")
(ok (equal? (list (cons RAX 0)) (initial-register-use (list RAX)))
    "initial availability points of registers are zero by default")

(ok (equal? RAX (find-available-register (list (cons RAX 0)) 0))
    "first register available")
(ok (not (find-available-register (list (cons RAX 1)) 0))
    "first register not available")
(ok (equal? RAX (find-available-register (list (cons RAX 1)) 1))
    "first register available at a later point in time")
(ok (equal? RAX (find-available-register (list (cons RAX 0)) 1))
    "first register already available")
(ok (equal? RCX (find-available-register (list (cons RAX 3) (cons RCX 2)) 2))
    "second register is available")

(ok (equal? (list (cons RAX 4)) (mark-used-till (list (cons RAX 1)) RAX 3))
    "mark first register as used")
(ok (equal? (list (cons RAX 4) (cons RCX 5)) (mark-used-till (list (cons RAX 1) (cons RCX 5)) RAX 3))
    "keep track of unaffected registers")
(ok (equal? (list (cons RAX 1) (cons RCX 9)) (mark-used-till (list (cons RAX 1) (cons RCX 5)) RCX 8))
    "mark second register as used")

(ok (eq? 'a (spill-candidate '((a . 0))))
    "spill the one variable if there is no other candidate")
(ok (eq? 'b (spill-candidate '((a . 0) (b . 1))))
    "spill second variable if it is allocated for a longer interval")
(ok (eq? 'a (spill-candidate '((a . 1) (b . 0))))
    "spill first variable if it is allocated for a longer interval")

(ok (equal? '((a . 2)) (ignore-spilled-variables '((a . 2)) (list (cons 'a RAX))))
    "do not ignore variables with allocated register")
(ok (null? (ignore-spilled-variables '((a . 2)) (list (cons 'a #f))))
    "ignore spilled variables")
(ok (equal? '((a . 2)) (ignore-spilled-variables '((a . 2)) '()))
    "do not ignore variable if it does not have a location assigned")

(ok (equal? (list (cons RAX 2)) (ignore-blocked-registers (list (cons RAX 2)) '(3 . 5) '()))
    "do not ignore register if it is not blocked")
(ok (null? (ignore-blocked-registers (list (cons RAX 2)) '(3 . 5) (list (cons RAX '(5 . 6)))))
    "ignore register for allocation if it is blocked")
(ok (equal? (list (cons RAX 2)) (ignore-blocked-registers (list (cons RAX 2)) '(3 . 5) (list (cons RAX '(6 . 8)))))
    "do not ignore register if it is blocked outside the specified interval")

(ok (equal? '(a b c) (register-parameters '(a b c)))
    "the first parameters are register parameters")
(ok (equal? '(a b c d e f) (register-parameters '(a b c d e f g)))
    "only the first six parameters are register parameters")
(ok (null? (stack-parameters '(a b c)))
    "the first few parameters are not stored on the stack")
(ok (equal? '(g h) (stack-parameters '(a b c d e f g h)))
    "appart from the first six parameters, all parameters are stored on the stack")

(let [(i (var <int>))
      (l (var <long>))]
  (ok (null? (register-parameter-locations '()))
      "initial parameter locations for no parameters")
  (ok (equal? (list (cons i RDI)) (register-parameter-locations (list i)))
      "initial parameter location for one parameter")
  (ok (equal? (list RDI RSI RDX RCX R8 R9) (map cdr (register-parameter-locations (make-list 6 l))))
      "initial parameter locations for first six parameters")

  (ok (null? (stack-parameter-locations '() 0))
      "initial stack parameter locations for no parameters")
  (ok (equal? (list (cons i (ptr <long> RSP 8))) (stack-parameter-locations (list i) 0))
      "initial parameter location of an integer stack parameter")
  (ok (equal? (list (ptr <long> RSP 8) (ptr <long> RSP 16)) (map cdr (stack-parameter-locations (list i i) 0)))
      "parameter locations of two stack parameters")
  (ok (equal? (list (ptr <long> RSP 24) (ptr <long> RSP 32)) (map cdr (stack-parameter-locations (list i i) 16)))
      "take stack offset into account when determining stack parameter locations")

  (ok (null? (parameter-locations '() 0))
      "parameter locations for empty set of parameters")
  (ok (equal? (list (cons 'a RDI) (cons 'b RSI)) (parameter-locations '(a b) 0))
      "parameter location for first parameter")
  (ok (equal? (list (cons 'a RDI) (cons 'b RSI) (cons 'c RDX) (cons 'd RCX) (cons 'e R8) (cons 'f R9)
                    (cons 'g (ptr <long> RSP 8)) (cons 'h (ptr <long> RSP 16)))
              (parameter-locations '(a b c d e f g h) 0))
      "parameter locations for register and stack parameters")
  (ok (equal? (list (cons 'a RDI) (cons 'b RSI) (cons 'c RDX) (cons 'd RCX) (cons 'e R8) (cons 'f R9)
                    (cons 'g (ptr <long> RSP 24)) (cons 'h (ptr <long> RSP 32)))
              (parameter-locations '(a b c d e f g h) 16))
      "parameter locations for register and stack parameters")

  (ok (null? (add-stack-parameter-information '() '()))
      "no stack location required")
  (ok (equal? (list (cons i (ptr <int> RSP 8)))
              (add-stack-parameter-information (list (cons i #f)) (list (cons i (ptr <int> RSP 8)))))
      "use stack location for register spilling")
  (ok (equal? (list (cons i RAX))
              (add-stack-parameter-information (list (cons i RAX)) (list (cons i (ptr <int> RSP 8)))))
      "do not use stack location if register already has a location allocated"))
(ok (equal? '(1) (next-indices '() (MOV CX 0) 0))
    "Get following indices for first statement in a program")
(ok (equal? '(2) (next-indices '() (MOV AX CX) 1))
    "Get following indices for second statement in a program")
(ok (null? (next-indices '() (RET) 2))
    "RET statement should not have any following indices")
(ok (equal? '(2) (next-indices '((a . 2)) (JMP 'a) 0))
    "Get following indices for a jump statement")
(ok (equal? '(1 2) (next-indices '((a . 2)) (JNE 'a) 0))
    "Get following indices for a conditional jump")
(let [(a (var <int>))
      (b (var <int>))
      (c (var <int>))
      (x (var <sint>))]
  (ok (equal? (list '() (list a) '()) (live-analysis (list 'x (MOV a 0) (RET)) '()))
      "Live-analysis for definition of unused variable")
  (ok (equal? (list (list a) (list a) (list b a) '()) (live-analysis (list (MOV a 0) (NOP) (MOV b a) (RET)) '()))
      "Live-analysis for definition and later use of a variable")
  (ok (equal? (list (list a) (list a) (list a) (list a) '())
              (live-analysis (list (MOV a 0) 'x (ADD a 1) (JE 'x) (RET)) '()))
      "Live-analysis with conditional jump statement")
  (ok (equal? (list (list a) (list a))
              (live-analysis (list (MOV a 0) (RET)) (list a)))
      "results should be propagated backwards from the return statement")
  (ok (null? (unallocated-variables '()))
      "no variables means no unallocated variables")
  (ok (equal? (list a) (unallocated-variables (list (cons a #f))))
      "return the unallocated variable")
  (ok (null? (unallocated-variables (list (cons a RAX))))
      "ignore the variable with register allocated")
  (ok (null? (register-allocations '()))
      "no variables means no variables with register allocated")
  (ok (equal? (list (cons a RAX)) (register-allocations (list (cons a RAX))))
      "return the variable with register allocation information")
  (ok (null? (register-allocations (list (cons a #f))))
      "filter out the variable which does not have a register allocated")
  (ok (null?  (assign-spill-locations '() 16 8))
      "assigning spill locations to an empty list of variables returns an empty list")
  (ok (equal? (list (cons a (ptr <long> RSP 16)))  (assign-spill-locations (list a) 16 8))
      "assign spill location to a variable")
  (ok (equal? (list (cons a (ptr <long> RSP 32)))  (assign-spill-locations (list a) 32 8))
      "assign spill location with a different offset")
  (ok (equal? (list (cons a (ptr <long> RSP 16)) (cons b (ptr <long> RSP 24)))
              (assign-spill-locations (list a b) 16 8))
      "use increasing offsets for spill locations")
  (ok (null? (add-spill-information '() 16 8))
      "do nothing if there are no variables")
  (ok (equal? (list (cons a RAX)) (add-spill-information (list (cons a RAX)) 16 8))
      "pass through variables with register allocation information")
  (ok (equal? (list (cons a (ptr <long> RSP 16))) (add-spill-information (list (cons a #f)) 16 8))
      "allocate spill location for a variable"))

(ok (null? (blocked-predefined '() '() '()))
    "no predefined variables")
(ok (equal? (list (cons 'a RDI))
            (blocked-predefined (list (cons 'a RDI)) '((a . (0 . 2))) (list (cons RDI '(1 . 3)))))
    "detect predefined variable with blocked register")
(ok (null? (blocked-predefined (list (cons 'a RDI)) '((a . (0 . 2))) '()))
    "ignore predefined variables if no registers are blocked")
(ok (null? (blocked-predefined (list (cons 'a RDI)) '((a . (0 . 2))) (list (cons RDI '(3 . 4)))))
    "ignore predefined variables if the associated register is not blocked")
(ok (null? (blocked-predefined (list (cons 'a RDI)) '() (list (cons RDI '(2 . 3)))))
    "ignore unused variable when checking for blocked registers")
(ok (null? (blocked-predefined (list (cons 'a RDI)) '((a . (0 . 2))) (list (cons RAX '(1 . 3)))))
    "only consider register associated with variable when blocking")

(ok (null? (move-blocked-predefined '()))
    "no predefined variables with blocked registers to move")
(let [(a (var <int>))]
  (ok (equal? (list (MOV a RAX)) (move-blocked-predefined (list (cons a RAX))))
      "copy variable from blocked register"))

(ok (equal? (list (cons 'a RDI)) (non-blocked-predefined (list (cons 'a RDI)) '()))
    "no predefinitions to discard")
(ok (equal? '() (non-blocked-predefined (list (cons 'a RDI)) (list (cons 'a RDI))))
    "discard predefined variables which are blocked")
(ok (equal? (list (cons 'b RSI)) (non-blocked-predefined (list (cons 'a RDI) (cons 'b RSI)) (list (cons 'a RDI))))
    "only discard predefined variables which are blocked")

(ok (null? (linear-scan-coloring '() '() '() '()))
    "linear scan with no variables returns empty mapping")
(ok (equal? (list (cons 'a RAX)) (linear-scan-coloring '((a . (0 . 0))) (list RAX RCX) '() '()))
    "allocate single variable")
(ok (equal? (list (cons 'a RAX) (cons 'b RAX)) (linear-scan-coloring '((a . (0 . 0)) (b . (1 . 1))) (list RAX RCX) '() '()))
    "reuse register with two variables")
(ok (equal? (list (cons 'a RAX) (cons 'b RCX)) (linear-scan-coloring '((a . (0 . 1)) (b . (1 . 1))) (list RAX RCX) '() '()))
    "allocate different registers for two variables conflicting at index 1")
(ok (equal? (list (cons 'a RAX) (cons 'b RCX)) (linear-scan-coloring '((b . (1 . 1)) (a . (0 . 1))) (list RAX RCX) '() '()))
    "sort live intervals by beginning of interval before performing linear-scan register allocation")
(ok (equal? (list (cons 'a RAX) (cons 'b RCX)) (linear-scan-coloring '((a . (0 . 0)) (b . (0 . 1))) (list RAX RCX) '() '()))
    "allocate different registers for two variables conflicting at index 0")
(ok (equal? (list (cons 'a RAX) (cons 'b #f)) (linear-scan-coloring '((a . (0 . 1)) (b . (1 . 3))) (list RAX) '() '()))
    "mark last variable for spilling if it has a longer live interval")
(ok (equal? (list (cons 'a #f) (cons 'b RAX)) (linear-scan-coloring '((a . (0 . 3)) (b . (1 . 1))) (list RAX) '() '()))
    "mark first variable for spilling if it has a longer live interval")
(ok (equal? (list (cons 'a #f) (cons 'b #f) (cons 'c RAX))
            (linear-scan-coloring '((a . (0 . 5)) (b . (1 . 4)) (c . (2 . 3))) (list RAX) '() '()))
    "do not spill same variable twice")
(ok (equal? (list (cons 'a RCX)) (linear-scan-coloring '((a . (0 . 0))) (list RAX RCX) (list (cons 'a RCX)) '()))
    "use predefined register for variable")
(ok (equal? (list (cons 'a RCX) (cons 'b RAX))
            (linear-scan-coloring '((a . (0 . 1)) (b . (1 . 1))) (list RAX RCX) (list (cons 'a RCX)) '()))
    "predefined registers take priority over normal register allocations")
(let [(a (var <int>))]
  (ok (equal? (list (cons a RCX))
              (linear-scan-coloring (list (cons a '(0 . 1))) (list RAX RCX) '() (list (cons RAX '(1 . 2)))))
      "do not allocate register if it is blocked while the variable is live"))

(ok (not (need-to-copy-first (list (cons 'a RSI) (cons 'b RDX)) (list (cons 'a RAX) (cons 'b RCX)) 'a 'b))
    "no need to copy RSI to RAX before RDX to RCX")
(ok (need-to-copy-first (list (cons 'a RSI) (cons 'b RDX)) (list (cons 'a RAX) (cons 'b RSI)) 'a 'b)
    "RSI needs to be copied to RAX before copying RDX to RSI")

(let [(a (var <int>))
      (b (var <int>))
      (c (var <int>))
      (d (var <int>))
      (e (var <int>))
      (f (var <int>))
      (g (var <int>))
      (r (var <int>))]
  (ok (null? (update-parameter-locations '() '() 0))
      "no parameters to move arround")
  (ok (equal? (list (MOV (ptr <int> RSP -8) EDI))
              (update-parameter-locations (list a) (list (cons a (ptr <long> RSP -8))) 0))
      "spill a register parameter")
  (ok (null? (update-parameter-locations (list a) '() 0))
      "ignore parameters which are not used")
  (ok (equal? (list (MOV EAX (ptr <int> RSP 8)))
              (update-parameter-locations (list a b c d e f g)
                                          (map cons (list a b c d e f g) (list RDI RSI RDX RCX R8 R9 RAX))
                                          0))
      "load a stack parameter")
  (ok (equal? (list (MOV EAX (ptr <int> RSP 24)))
              (update-parameter-locations (list a b c d e f g)
                                          (map cons (list a b c d e f g) (list RDI RSI RDX RCX R8 R9 RAX))
                                          16))
      "load a stack parameter taking into account the stack pointer offset")
  (ok (null? (update-parameter-locations (list a b c d e f g)
                                         (map cons (list a b c d e f g) (list RDI RSI RDX RCX R8 R9 (ptr <long> RSP 24)))
                                         16))
      "leave parameter on stack")
  (ok (equal? (list (MOV EAX ESI) (MOV ESI EDI))
              (update-parameter-locations (list a b) (map cons (list a b) (list RSI RAX)) 0))
      "adjust order of copy operations to avoid overwriting parameters"))

(let [(a (var <int>))]
  (ok (null? (move-variable-content a RCX RCX))
      "no need to move variable content if source and destination location are the same")
  (ok (equal? (MOV ECX EDX) (move-variable-content a RDX RCX))
      "move variable content from RDX to RCX")
  (ok (null? (move-variable-content a (ptr <long> RSP 24) (ptr <long> RSP 24)))
      "no need to move variable if stack locations are the same")
  (ok (null? (move-variable-content a RSI #f))
      "no need to move variable if destination is undefined"))

(let [(a (var <int>))
      (b (var <int>))
      (c (var <int>))
      (d (var <int>))
      (e (var <int>))
      (f (var <int>))
      (g (var <int>))
      (r (var <int>))
      (x (var <sint>))
      (p (var <long>))]
  (ok (eqv? 0 (number-spilled-variables '() '()))
      "count zero spilled variables")
  (ok (eqv? 1 (number-spilled-variables '((a . #f)) '()))
      "count one spilled variable")
  (ok (eqv? 0 (number-spilled-variables (list (cons a RAX)) '()))
      "ignore allocated variables when counting spilled variables")
  (ok (eqv? 0 (number-spilled-variables '((a . #f)) '(a)))
      "do not count stack parameters when allocating stack space")
  (ok (eqv? 1 (number-spilled-variables '((a . #f)) '(b)))
      "allocate stack space if spilled variable is not a stack parameter")

  (ok (null? (temporary-variables '()))
      "an empty program needs no temporary variables")
  (ok (equal? (list <var>) (map class-of (temporary-variables (list (MOV a 0)))))
      "create temporary variable for first argument of instruction")
  (ok (not (equal? (list a) (temporary-variables (list (MOV a 0)))))
      "temporary variable should be distinct from first argument of instruction")
  (ok (equal? (list <sint>) (map typecode (temporary-variables (list (MOV x 0)))))
      "temporary variable should have correct type")
  (ok (equal? (list #f) (temporary-variables (list (MOV EAX 0))))
      "it should not create a temporary variable if the statement does not contain variables")
  (ok (equal? (list #f) (temporary-variables (list (MOV EAX a))))
      "it should not create a temporary variable if the first argument is not a variable")
  (ok (equal? (list <var>) (map class-of (temporary-variables (list (MOV (ptr <int> p) a)))))
      "create temporary variable for pointer argument to instruction")
  (ok (equal? (list <long>) (map typecode (temporary-variables (list (MOV (ptr <int> p) a)))))
      "temporary variable for pointer argument needs to be long integer")

  (ok (null? (unit-intervals '()))
      "create empty list of unit intervals")
  (ok (equal? '((a . (0 . 0))) (unit-intervals '(a)))
      "generate unit interval for one temporary variable")
  (ok (equal? '((a . (0 . 0)) (b . (1 . 1))) (unit-intervals '(a b)))
      "generate unit interval for two temporary variables")
  (ok (equal? '((b . (1 . 1))) (unit-intervals '(#f b)))
      "filter out locations without temporary variable")
  (ok (null? (temporary-registers '() '()))
      "create empty list of temporary registers")
  (ok (equal? (list RCX) (temporary-registers (list (cons a RCX)) (list a)))
      "return a temporary register")
  (ok (equal? (list #f) (temporary-registers '() (list #f)))
      "return false if no temporary variable was required for a statement")

  (ok (equal? (list (SUB RSP 8) (MOV EAX 42) (ADD RSP 8) (RET))
              (linear-scan-allocate (list (MOV a 42) (RET))))
      "Allocate a single register")
  (ok (equal? (list (SUB RSP 8) (MOV ECX 42) (ADD RSP 8) (RET))
              (linear-scan-allocate (list (MOV a 42) (RET)) #:registers (list RCX RDX)))
      "Allocate a single register using custom list of registers")
  (ok (equal? (list (SUB RSP 8) (MOV ECX 1) (MOV EDX 2) (ADD ECX EDX) (MOV EAX ECX) (ADD RSP 8) (RET))
              (linear-scan-allocate (list (MOV a 1) (MOV b 2) (ADD a b) (MOV c a) (RET))))
      "Allocate multiple registers")
  (ok (equal? (list (SUB RSP 8) (MOV EAX 42) (ADD RSP 8) (RET))
              (linear-scan-allocate (list (MOV a 42) (RET))))
      "Allocate a single register")
  (ok (equal? (list (SUB RSP 8) (MOV ECX 42) (ADD RSP 8) (RET))
              (linear-scan-allocate (list (MOV a 42) (RET)) #:registers (list RCX RDX)))
      "Allocate a single register using custom list of registers")
  (ok (equal? (list (SUB RSP 8) (MOV ECX 1) (MOV EDX 2) (ADD ECX EDX) (MOV EAX ECX) (ADD RSP 8) (RET))
              (linear-scan-allocate (list (MOV a 1) (MOV b 2) (ADD a b) (MOV c a) (RET))))
      "Allocate multiple registers")
  (ok (equal? (list (SUB RSP 8) (MOV EDX 1) (ADD EDX EDI) (MOV EDI EDX) (ADD RSP 8) (RET))
              (linear-scan-allocate (list (MOV b 1) (ADD b a) (MOV c b) (RET))
                                    #:parameters (list a) #:registers (list RDI RSI RDX RCX)))
      "Register allocation with predefined parameter register")
  (ok (equal? (list (SUB RSP 16) (MOV (ptr <int> RSP 8) EDI) (MOV EDI 1) (ADD EDI (ptr <int> RSP 8)) (ADD RSP 16) (RET))
              (linear-scan-allocate (list (MOV b 1) (ADD b a) (RET)) #:parameters (list a) #:registers (list RDI RSI)))
      "Spill register parameter")
  (ok (equal? (list (SUB RSP 8) (MOV EDI (ptr <int> RSP 16)) (MOV EAX EDI) (ADD RSP 8) (RET))
              (linear-scan-allocate (list (MOV r g) (RET)) #:parameters (list a b c d e f g) #:registers (list RAX RDI RSI RAX)))
      "Fetch register parameter")
  (ok (equal? (list (SUB RSP 16) (MOV EAX 0) (MOV (ptr <int> RSP 8) EAX) (MOV EAX (ptr <int> RSP 8)) (ADD EAX (ptr <int> RSP 24)) (MOV (ptr <int> RSP 8) EAX) (ADD RSP 16) (RET))
              (linear-scan-allocate (list (MOV r 0) (ADD r g) (RET)) #:parameters (list a b c d e f g) #:registers (list RAX)))
      "Reuse stack location for spilled stack parameters")
  (ok (equal? (list (SUB RSP 8) (MOV ECX EDI) (MOV EAX ECX) (ADD RSP 8) (RET))
              (linear-scan-allocate (list (MOV r a) (RET)) #:parameters (list a) #:results (list r)))
      "Copy result to RAX register before restoring stack pointer and returning")

  (ok (equal? a (first-argument (ADD a b)))
      "get first argument of ADD statement")
  (ok (not (first-argument (ADD CX DX)))
      "return false if statement is compiled already")

  (ok (equal? (list (MOV EAX 0)) (replace-variables '() (MOV EAX 0) RAX))
      "only put instruction into a list if there are no variables to replace")
  (ok (equal? (list (MOV ESI ECX)) (replace-variables (list (cons a RCX)) (MOV ESI a) RAX))
      "replace input variable with allocated register")
  (ok (equal? (list (MOV ECX 0)) (replace-variables (list (cons a RCX)) (MOV a 0) RAX))
      "replace output variable with allocated register")
  (ok (equal? (list (MOV EDX (ptr <int> RSP 16))) (replace-variables (list (cons a (ptr <long> RSP 16))) (MOV EDX a) RAX))
      "read input variable from spill location")
  (ok (equal? (list (MOV AX (ptr <sint> RSP 16)) (CMP AX 0))
              (replace-variables (list (cons x (ptr <long> RSP 16))) (CMP x 0) RAX))
      "use temporary register for first argument and fetch value from spill location")
  (ok (equal? (list (MOV EAX (ptr <int> RSP 16)) (CMP EAX 0))
              (replace-variables (list (cons a (ptr <long> RSP 16))) (CMP a 0) RAX))
      "use correct type for temporary register")
  (ok (equal? (list (MOV EAX (ptr <int> RSP 16)) (ADD EAX 1) (MOV (ptr <int> RSP 16) EAX))
              (replace-variables (list (cons a (ptr <long> RSP 16))) (ADD a 1) RAX))
      "read and write back argument from stack into temporary register")
  (ok (equal? (list (MOV EAX 1) (MOV (ptr <int> RSP 16) EAX))
              (replace-variables (list (cons a (ptr <long> RSP 16))) (MOV a 1) RAX))
      "write output value in temporary register to the stack")
  (ok (equal? (list (MOV RAX (ptr <long> RSP 32)) (MOV EDX (ptr <int> RAX 8)))
              (replace-variables (list (cons a RDX) (cons p (ptr <long> RSP 32))) (MOV a (ptr <int> p 8)) RAX))
      "use temporary variable to implement reading from pointer to pointer")
  (ok (equal? (list (MOV EDX (ptr <int> RCX 8)))
              (replace-variables (list (cons a RDX) (cons p RCX)) (MOV a (ptr <int> p 8)) RAX))
      "do not use temporary variable when reading from register pointer")
  (ok (equal? (list (MOV RAX (ptr <long> RSP 32)) (MOV (ptr <int> RAX 8) EDX))
              (replace-variables (list (cons a RDX) (cons p (ptr <long> RSP 32))) (MOV (ptr <int> p 8) a) RAX))
      "use temporary variable to implement writing to pointer to pointer")

  (ok (equal? (list (SUB RSP 8) (MOV ECX EDI) (ADD ECX ESI) (MOV EAX ECX) (ADD RSP 8) (RET))
              (virtual-variables (list a) (list b c) (list (MOV a b) (ADD a c) (RET))))
      "'virtual-variables' uses the specified variables as parameters")
  (ok (equal? (list (SUB RSP 8) (MOV ECX EDI) (MOV EDX ECX) (MOV EAX EDX) (ADD RSP 8) (RET))
              (virtual-variables (list a) (list b) (list (MOV c b) (MOV a c) (RET))))
      "'virtual-variables' allocates local variables"))
(ok (eq? 'new (get-target (retarget (JMP 'old) 'new)))
    "'retarget' should update target of jump statement")
(ok (equal? (list (JMP 1) 'a (NOP) (RET))
            (flatten-code (list (list (JMP 1) 'a) (NOP) (RET))))
    "'flatten-code' should flatten nested environments")
(let [(a (var <int>))
      (b (var <int>))]
  (ok (equal? (list (SUB RSP 8) (MOV ECX EDI) (MOV EAX ECX) (ADD RSP 8) (RET))
              (virtual-variables (list a) (list b) (list (list (MOV a b)) (RET))))
      "'pass-parameter-variables' handles nested code blocks")
  (ok (equal? (list (SUB RSP 8) (MOV EAX (ptr <int> RSP 16)) (MOV EDX EAX) (MOV EAX EDX) (ADD RSP 8) (RET))
              (let [(args (map var (make-list 7 <int>)))]
                 (virtual-variables (list a) args (list (MOV a (last args)) (RET)))))
      "'virtual-variables' maps the 7th integer parameter correctly"))
(ok (equal? (resolve-jumps (list (JMP 'b) (JMP 'a) 'a (NOP) 'b))
            (resolve-jumps (flatten-code (relabel (list (JMP 'a) (list (JMP 'a) 'a) (NOP) 'a)))))
    "'relabel' should create separate namespaces for labels")

(let [(a (var <int>))]
  (ok (eqv? 3 ((asm ctx <int> '()
                    (virtual-variables (list a) '()
                                       (list (MOV a 0) (JMP 'a) (list 'a (MOV a 2)) 'a (ADD a 3) (RET)))) ))
      "'virtual-variables' creates separate namespaces for labels"))

(let [(r (var <int>))]
  (ok (equal? (list (NOP) (RET)) (place-result-variable '() '() (list (NOP) (RET))))
      "return unmodified code if there is no result variable")
  (ok (equal? (list (NOP) (MOV EAX EDI) (RET)) (place-result-variable (list r) (list (cons r RDI)) (list (NOP) (RET))))
      "copy result variable into RAX if it is in another location")
  (ok (equal? (list (NOP) (RET)) (place-result-variable (list r) (list (cons r RAX)) (list (NOP) (RET))))
      "return unmodified code if result already is residing in RAX"))

(ok (null? (used-callee-saved '()))
    "no registers in use")
(ok (equal? (list RBX) (used-callee-saved (list (cons 'a RBX))))
    "callee saved register in use")
(ok (equal? (list RBX) (used-callee-saved (list (cons 'a RBX) (cons 'b RBX))))
    "remove duplicate registers")
(ok (null? (used-callee-saved (list (cons 'a RAX))))
    "ignore caller saved register")
(ok (null?  (used-callee-saved (list (cons 'a #f))))
    "ignore variables without allocated register")

(ok (equal? (list (PUSH RBX) (NOP) (POP RBX) (RET)) (backup-registers (list RBX) (list (NOP) (RET))))
    "backup one register")
(ok (equal? (list (PUSH RBX) (PUSH RBP) (NOP) (POP RBP) (POP RBX) (RET)) (backup-registers (list RBX RBP) (list (NOP) (RET))))
    "backup two registers")
(let [(a (var <int>))
      (b (var <int>))]
  (ok (equal? (list (SUB RSP 8) (MOV EDI 1) (MOV EAX 2) (ADD EAX 3) (ADD EDI 4) (ADD RSP 8) (RET))
              (linear-scan-allocate (list (MOV a 1) (MOV b 2) (ADD b 3) (ADD a 4) (RET))
                                 #:registers (list RSI RDI RAX)))
      "'linear-scan-allocate' should use the specified set of registers")
  (ok (equal? (list (SUB RSP 16)
                    (MOV EAX 1)
                    (MOV (ptr <int> RSP 8) EAX)
                    (MOV ESI 2)
                    (ADD ESI 3)
                    (MOV EAX (ptr <int> RSP 8))
                    (ADD EAX 4)
                    (MOV (ptr <int> RSP 8) EAX)
                    (ADD RSP 16)
                    (RET))
              (linear-scan-allocate (list (MOV a 1) (MOV b 2) (ADD b 3) (ADD a 4) (RET))
                                    #:registers (list RAX RSI)))
      "'linear-scan-allocate' should spill variables")
  (let  [(c (var <int>))]
    (ok (equal? (list (SUB RSP 24)
                      (MOV ESI 1)
                      (MOV (ptr <int> RSP 8) ESI)
                      (MOV ESI 2)
                      (MOV (ptr <int> RSP 16) ESI)
                      (MOV EAX 3)
                      (ADD EAX 4)
                      (MOV ESI (ptr <int> RSP 16))
                      (ADD ESI 5)
                      (MOV (ptr <int> RSP 16) ESI)
                      (MOV ESI (ptr <int> RSP 8))
                      (ADD ESI 6)
                      (MOV (ptr <int> RSP 8) ESI)
                      (ADD RSP 24)
                      (RET))
                (linear-scan-allocate (list (MOV a 1) (MOV b 2) (MOV c 3) (ADD c 4) (ADD b 5) (ADD a 6) (RET))
                                      #:registers (list RSI RAX)))
        "'linear-scan-allocate' should assign separate stack locations"))
  (ok (equal? (list (PUSH RBX)
                    (SUB RSP 16)
                    (MOV EBX 1)
                    (MOV (ptr <int> RSP 8) EBX)
                    (MOV EAX 2)
                    (ADD EAX 3)
                    (MOV EBX (ptr <int> RSP 8))
                    (ADD EBX 4)
                    (MOV (ptr <int> RSP 8) EBX)
                    (ADD RSP 16)
                    (POP RBX)
                    (RET))
              (linear-scan-allocate (list (MOV a 1) (MOV b 2) (ADD b 3) (ADD a 4) (RET))
                                    #:registers (list RBX RAX)))
      "'linear-scan-allocate' should save callee-saved registers"))
(let [(a (var <int>))
      (b (var <int>))
      (c (var <int>))]
  (ok (equal? (list (SUB RSP 16)
                    (MOV (ptr <int> RSP 8) ESI)
                    (MOV ESI EDI)
                    (ADD ESI (ptr <int> RSP 8))
                    (ADD RSP 16)
                    (RET))
              (virtual-variables '() (list a b) (list (MOV c a) (ADD c b) (RET)) #:registers (list RSI RDI RAX)))
      "Spill register-parameter to the stack")
  (ok (equal? (list (SUB RSP 8) (MOV ECX 0) (MOV ESI 0) (CMP ESI EDX) (JE #x6) (INC ESI) (INC ECX) (JMP #x-a) (ADD RSP 8) (RET))
              (resolve-jumps (linear-scan-allocate (flatten-code (list (MOV a 0) (repeat b (INC a)) (RET))))))
      "'repeat' loop"))
(ok (equal? (list (MOV ECX 2) (RET)) (get-code (blocked AL (MOV ECX 2) (RET))))
    "'blocked' represents the specified code segment")
(ok (equal? RAX (get-reg (blocked RAX (MOV ECX 2) (RET))))
    "'blocked' stores the register to be blocked")
(ok (equal? (list (MOV ECX 2) (RET)) (blocked '() (MOV ECX 2) (RET)))
    "'blocked' with empty block list has no effect")
(ok (equal? (list (MOV ECX 2) (RET)) (filter-blocks (blocked RAX (MOV ECX 2) (RET))))
    "'filter-blocks' should remove blocked-register information")
(ok (equal? (list (MOV EDX 2) 'x (list (RET)))
            (filter-blocks (blocked RDX (MOV EDX 2) 'x (blocked RAX (RET)))))
    "'filter-blocks' should work recursively")
(ok (equal? (list (cons RAX '(0 . 1))) (blocked-intervals (blocked RAX (MOV EAX 0) (RET))))
    "'blocked-intervals' should extract the blocked intervals for each register")
(ok (equal? (list (cons RAX '(1 . 1))) (blocked-intervals (list (MOV EAX 0) (blocked RAX (RET)))))
    "Blocked intervals within a program should be offset correctly")
(ok (equal? (list (cons RAX '(2 . 2))) (blocked-intervals (list (list (MOV EAX 0) (NOP)) (blocked RAX (RET)))))
    "The offsets of 'blocked-intervals' should refer to the flattened code")
(ok (equal? (list (cons RAX '(1 . 4)) (cons RDX '(2 . 3)))
            (blocked-intervals (list 'x (blocked RAX (MOV AX 0) (blocked RDX (MOV DX 0) (IDIV CX)) (RET)))))
    "'blocked-intervals' should work recursively")
(ok (equal? (list (cons RCX '(0 . 1)) (cons RDX '(0 . 1))) (blocked-intervals (blocked (list RCX RDX) (MOV ECX 2) (RET))))
    "'blocked' with list of registers blocks all of them")
(let [(r (var <byte>))
      (a (var <byte>))
      (b (var <byte>))]
  (ok (equal? (list (SUB RSP 8) (MOV AL CL) (CBW) (IDIV DL) (MOV CL AL) (ADD RSP 8) (RET))
              (linear-scan-allocate (list (MOV AL a) (CBW) (IDIV b) (MOV r AL) (RET))
                                    #:registers (list RAX RCX RDX)
                                    #:blocked (list (cons RAX '(0 . 3)))))
      "'linear-scan-allocate' should block registers if specified"))
(let [(a (var <int>))
      (b (var <int>))
      (c (var <int>))
      (d (var <int>))
      (e (var <int>))
      (f (var <int>))
      (g (var <int>))
      (r (var <int>))]
  (ok (equal? (list (PUSH RBX) (SUB RSP 8) (MOV EBX 1) (ADD RSP 8) (POP RBX) (RET))
              (linear-scan-allocate (list (MOV a 1) (RET)) #:registers (list RBX RAX)))
      "save callee-saved registers")
  (ok (equal? (list (PUSH RBX) (SUB RSP 8) (MOV EBX (ptr <int> RSP 24)) (MOV EBX 42) (ADD RSP 8) (POP RBX) (RET))
              (linear-scan-allocate (list (MOV g 42) (RET)) #:parameters (list a b c d e f g) #:registers (list RBX RAX)))
      "add offset for callee-saved parameters when fetching stack parameters")
  (ok (equal? (list (PUSH RBX) (SUB RSP 8) (MOV EBX EAX) (MOV (ptr <int> RSP 24) EBX) (ADD RSP 8) (POP RBX) (RET))
              (linear-scan-allocate (list (MOV g r) (RET)) #:parameters (list a b c d e f g) #:registers (list RBX RAX)))
      "add offset for callee-saved parameters when using stack parameters")
  (ok (equal? (list (SUB RSP 8) (MOV EAX EDI) (MOV EDI EAX) (ADD RSP 8) (RET))
              (linear-scan-allocate (list (MOV EDI a) (RET))
                                    #:parameters (list a) #:registers (list RDI RAX RCX) #:blocked (list (cons RDI '(0 . 0)))))
      "move parameter variable into another location if the register is blocked")
  (ok (equal? (list (SUB RSP 8) (MOV ECX 42) (MOV EAX 0) (MOV EAX ECX) (ADD RSP 8) (RET))
              (linear-scan-allocate (list (MOV r 42) (MOV b 0) (RET)) #:results (list r)))
      "when allocating registers preserve result variables up to RET statement"))

(let  [(w (var <usint>))]
  (ok (equal? (list (SUB RSP 8) (MOV AX 0) (ADD RSP 8) (RET))
              (virtual-variables '() '() (list (blocked RCX (MOV w 0)) (RET))))
      "'virtual-variables' filters out the reserved-registers information")
  (ok (equal? (list (SUB RSP 8) (MOV CX 0) (ADD RSP 8) (RET))
              (virtual-variables '() '() (list (blocked RAX (MOV w 0)) (RET))))
      "'virtual-variables' avoids blocked registers when allocating variables"))

(ok (eq? <var> (class-of (var <int>)))
    "Shortcut for creating variables creates variables")
(ok (eq? <byte> (typecode (var <byte>)))
    "Shortcut for  creating variables uses specified type")
(ok (eq? <ubyte> (typecode (var <bool>)))
    "Boolean values are represented using unsigned byte")
(let  [(i (skeleton <int>))]
  (ok (is-a? i <int>)
      "skeleton of integer is of type integer")
  (ok (is-a? (value i) <var>)
      "value of integer skeleton is a variable")
  (ok (eq? <int> (typecode (value i)))
      "value of integer skeleton is of type integer"))
(let [(s (skeleton (sequence <byte>)))]
  (ok (is-a? s (sequence <byte>))
      "skeleton of a sequence is a sequence")
  (ok (equal? (list <long> <long> <ulong>) (map class-of (content (class-of s) s)))
      "skeleton of sequence consists of two long integer variables and an unsigned long integer")
  (ok (equal? (list <var> <var> <var>) (map class-of (map get (content (class-of s) s))))
      "sequence skeleton is based on three variables"))
(let [(m (skeleton (multiarray <int> 2)))]
  (ok (is-a? m (multiarray <int> 2))
      "skeleton of a 2D array is a 2D array")
  (ok (equal? (list <long> <long> <long> <long> <ulong>) (map class-of (content (class-of m) m)))
      "skeleton of 2D array consists of long integer variables and an unsigned long integer")
  (ok (equal? (make-list 5 <var>) (map class-of (map get (content (class-of m) m))))
      "2D array skeleton is based on five variables"))
(let [(a (skeleton <byte>))
      (b (skeleton (pointer <byte>)))
      (c (set-pointer-offset (skeleton (pointer <int>)) 3))]
  (ok (equal? (get a) (operand a))
      "element operand is value of element")
  (ok (equal? (ptr <byte> (get b)) (operand b))
      "pointer operand is pointer to element")
  (ok (equal? (ptr <int> (get c) 3) (operand c))
      "pointer operand can have offset"))
(let [(out (skeleton <int>))
      (in  (skeleton <int>))]
  (ok (equal? (list (list (mov-signed (get out) (get in)))) (code out in))
      "generate code for copying an integer")
  (ok (equal? (list (list (get out)) (list (get in)) (list (list (mov-signed (get out) (get in))) (RET)))
              (assemble (list out) (list in) (code out in)))
      "generate code for identity function"))
(ok (equal? (list (SUB RSP 8) (MOV EAX ECX) (ADD RSP 8) (RET))
            (linear-scan-allocate (flatten-code (attach (code (skeleton <ulong>) (skeleton <uint>)) (RET)))))
    "Use default zero-extension for 32-bit numbers")
(ok (eqv? 42 ((jit ctx (list <int>) identity) 42))
    "compile and run integer identity function")
(ok (eqv? #t ((jit ctx (list <bool>) identity) #t))
    "compile and run boolean identity function")
(let [(out (skeleton <int>))
      (in  (skeleton (pointer <int>)))]
  (ok (equal? (list (list (mov-signed (get out) (ptr <int> (get in))))) (code out in))
      "generate code for reading integer from memory"))
(let [(out (skeleton (pointer <int>)))
      (in  (skeleton <int>))]
  (ok (equal? (list (list (mov-signed (ptr <int> (get out)) (get in)))) (code out in))
      "generate code for writing integer to memory"))
(let [(out (skeleton <int>))]
  (ok (equal? (list (MOV (get out) 0)) (code out 0))
      "Generate code for setting variable to zero"))
(let [(out  (parameter (sequence <int>)))]
  (ok (equal? (list (IMUL (step out) (get (delegate (stride out))) (size-of (typecode out)))
                    (MOV (iterator out) (value out)))
              (setup out))
      "setup of array loop should define increment and initialise pointer")
  (ok (equal? (list (ADD (iterator out) (step out))) (increment out))
      "increment of array loop should increment the pointer")
  (ok (equal? (iterator out) (value (body out)))
      "body of loop should be rebased to the pointer")
  (ok (is-a? (delegate (body out)) (pointer <int>))
      "body of array loop should be a pointer object"))
(let [(in  (skeleton (pointer <byte>)))
      (out (skeleton (pointer <byte>)))]
  (ok (equal? (list (SUB RSP 8) (MOV DL (ptr <byte> RAX)) (MOV (ptr <byte> RSI) DL) (ADD RSP 8) (RET))
              (linear-scan-allocate (flatten-code (attach (code out in) (RET)))))
      "generate code for copying a byte from one memory location to another"))
(ok (equal? '(2 3 5) (to-list ((jit ctx (list (sequence <int>)) identity) (seq <int> 2 3 5))))
    "compile and run identity function for array")
(let [(out (skeleton (multiarray <int> 2)))
      (in  (skeleton (multiarray <int> 2)))]
  (ok (list? (code (parameter out) (parameter in)))
      "generating code for copying a 2D array should run without error"))
(ok (equal? '((2 3 5) (7 9 11))
            (to-list ((jit ctx (list (multiarray <int> 2)) identity) (arr <int> (2 3 5) (7 9 11)))))
    "compile and run identity function for 2D array")
(let [(out (skeleton <int>))
      (a   (skeleton <int>))
      (b   (skeleton <int>))]
  (ok (equal? (list (list (mov-signed (get out) (get a))) (ADD (get out) (get b)))
              (code (parameter out) (+ (parameter a) (parameter b))))
    "generate code for adding two numbers"))
(ok (equal? 42 ((jit ctx (list <int> <int>) +) 19 23))
    "compile and run function adding two numbers")
(let [(out (skeleton <byte>))
      (in  (skeleton <int>))]
  (ok (equal? (list (SUB RSP 8) (MOV AL CL) (ADD RSP 8) (RET))
              (linear-scan-allocate (flatten-code (list (code out in) (RET)))))
      "generate code for copying part of integer"))
(ok (eq? <int> (type (+ (parameter <usint>) (parameter <byte>))))
  "plus operation coerces return type correctly")
(let [(out (skeleton <int>))
      (a   (skeleton <byte>))
      (b   (skeleton <usint>))]
  (ok (equal? (list (SUB RSP 8) (MOVZX ESI AX) (MOVSX ECX DL) (ADD ESI ECX) (ADD RSP 8) (RET))
              (linear-scan-allocate (flatten-code (list (code (parameter out) (+ (parameter b) (parameter a))) (RET)))))
      "sign-extend second number when adding"))
(ok (+ (parameter (sequence <int>)) (parameter <int>))
    "create function from tensor and element")
(ok (+ (parameter <int>) (parameter (sequence <int>)))
    "create function from element and tensor")
(ok (+ (parameter (sequence <int>)) (parameter (sequence <int>)))
    "create function from two tensors")
(let* [(a    (parameter (sequence <int>)))
       (b    (parameter <int>))
       (f    (+ a b))
       (out  (parameter (sequence <int>)))]
  (ok (equal? (list (IMUL (step a) (get (delegate (stride a))) (size-of (typecode a)))
                    (MOV (iterator a) (value a)))
              (setup f))
      "setup of loop over array-scalar-function should setup looping over first argument")
  (ok (equal? (list (ADD (iterator a) (step a))) (increment f))
      "loop should increment input array iterator")
  (ok (equal? (iterator a) (value (car (arguments (body f)))))
      "body of loop should be function with element of first argument as argument")
  (ok (equal? b (cadr (arguments (body f))))
      "body of loop should maintain second argument")
  (ok (equal? (list (SUB RSP 8) (MOV ESI (ptr <int> RAX)) (ADD ESI EDX) (MOV (ptr <int> RDI) ESI) (ADD RSP 8) (RET))
              (linear-scan-allocate (flatten-code (attach (code (body out) (body f)) (RET)))))
      "instantiate loop body for array-scalar-function"))
(let [(out (skeleton (sequence <int>)))
      (a   (skeleton (sequence <int>)))
      (b   (skeleton <int>))]
  (ok (list? (code (parameter out) (+ (parameter a) (parameter b))))
      "generating code for array-scalar operation should run without error"))
(ok (equal? '(9 10 12) (to-list ((jit ctx (list (sequence <int>) <int>) +) (seq <int> 2 3 5) 7)))
    "compile and run array-scalar operation")
(let* [(a    (parameter <int>))
       (b    (parameter (sequence <int>)))
       (f    (+ a b))
       (out  (parameter (sequence <int>)))]
  (ok (equal? (list (IMUL (step b) (get (delegate (stride b))) (size-of (typecode b)))
                    (MOV (iterator b) (value b)))
              (setup f))
      "setup of loop over scalar-array-function should setup looping over second argument")
  (ok (equal? a (car (arguments (body f))))
      "body of loop should maintain first argument")
  (ok (equal? (iterator b) (value (cadr (arguments (body f)))))
      "body of loop should be function with element of second argument as argument")
  (ok (equal? (list (SUB RSP 8) (MOV ESI EAX) (ADD ESI (ptr <int> RDX)) (MOV (ptr <int> RDI) ESI) (ADD RSP 8) (RET))
              (linear-scan-allocate (flatten-code (attach (code (body out) (body f)) (RET)))))
      "instantiate loop body for scalar-array-function"))
(run-tests)
