# Binding for ICQ protocol library

package require icq 0.8.8

# Module metainformation '''1
namespace eval meta {
	set name "ICQ"
	set author "Ihar Viarheichyk <iverg@mail.ru>"
	set description "ICQ transport"
	set icon img:online
	array set uin {
		type digit description "ICQ UIN" weight .10
		property Global:ICQ:Info|Uin save change}
	array set alias {property Global:ICQ:Info|Alias save change
		weight .11}
	array set status {
		property Global:ICQ:Network|StartupStatus save change
		description "Status after startup" default offline
		type variant weight .12
		values {offline online occ dnd ffc away na invisible}
	}
	array set password { type password property Global:ICQ:Info|Password
		description "ICQ password" save change weight .101}
	array set encoding [list type variant save change\
		values [map x [encoding names] {set x [string index $x 0]:$x}]\
		default [encoding system] desciption "Default encoding"\
		property Global:ICQ:Info|encoding weight .13]
	unset x	
	array set ChangePassword {type action menu {Change "ICQ Password"}}
	array set add:contact {type action menu {Add Contact}
		script { Event AddItem Contact:ICQ }
	}
}

# Network metainformation '''2
namespace eval network::meta {
	array set server {
		type fqdn default login.icq.com save change
		property Global:ICQ:Network|server weight .21
	}
	array set port {
		type port default 5190 save change weight .22
		property Global:ICQ:Network|port
	}
	array set proxy {
		type variant save change default "" weight .23
		description "Proxy to use" empty "No proxy"
		valuescript {map x [select Proxy] { lindex [split $x :] end}}
	}
	array set reconnect {
		type boolean save change property Global:ICQ:Network|reconnect
		description "Automatically re-connect after disconnect"
		default 1 weight .24
	}
	array set ping {
		type integer save change property Global:ICQ:Network|ping
		description "Ping interval, seconds" default 0 weight .25
	}
	array set keepalive {
		type integer save change property Global:ICQ:Network|keepalive
		description "Keepalive interval, seconds" default 0 weight .26
	}
}

# Roster metainformation '''2
namespace eval roster::meta {
	array set use { 
		type boolean default 1 property Global:ICQ:Roster|use
		save change weight .31
		description "Use server-side roster"
	}
	array set modify {
		type boolean default 1 property Global:ICQ:Roster|modify
		save change weight .32
		description "Save Changes in contact list in server-side roster"
	}
	array set updated {
		type cache property Global:ICQ:Roster|updated
		save exit default 0
	}
	array set count {
		type cache property Global:ICQ:Roster|count
		save change default 0
	}
}
#```1

if {[catch { package require direct 0.3 } reason]} {
	Event Log warning "Direct connections are disabled: $reason"
	namespace eval direct { set enable 0 }
} else {
	namespace eval direct::meta {
		array set enable {
			type boolean default 0 save change
			description "Enable direct connections"
		}
	}
}

# ICQ-specific properties for ICQ contacts '''1
namespace eval [ref Contact:ICQ]::meta {
	set Identifier UIN
	set AddPrompt { You should specify numerical UIN of ICQ contact. 
	}
	array set Status  {
		type {cache variant} default offline 
		values {offline online occ dnd ffc away na invisible}
	}
	array set mobile 	{save change}
	array set auth		{type {cache boolean} default 0 save change}
	array set rid		{type integer save change}
	array set visible	{type boolean default 0 save change
		menu {Lists Visible} weight .31}
	array set invisible	{type boolean default 0 save change 
		menu {Lists Invisible} weight .32}
	array set Send:text	{type action menu {Send Message} weight .10}
	array set Send:contacts	{type action menu {Send Contacts} weight .11}
array set authorization:grant	{type action 
		menu {Authorization Grant} weight .20}
	array set authorization:deny	{type action
		menu {Authorization Deny} weight .21}
	array set encoding	{ menu Encoding }
}

set ref [ref Contact:ICQ]
array set ${ref}::meta::encoding [array get meta::encoding]

# Sync contact default encoding with global encoding
upvar #0  ${ref}::meta::encoding(default) [namespace current]::encoding

namespace eval [ref Group:common]::meta {
	array set rid	{type integer save change}
}

# Data validation functions '''1
namespace eval ::validate {
	proc port {val} {
		expr { [string is integer $val] && $val>=0 && $val<=65535 }
	}
	proc UIN {val} { string is digit $val }
}
# ```1

variable icq
variable recreate 0
variable itemCount 0

proc Uid2Uin {list} { map x $list { lindex [split $x :] end } }

# Process ICQ events '''1
proc ICQEvent {event args} {
	if {[llength [info commands icq:$event]]} {
		eval icq:$event $args
	} else { eval Event [list $event] $args}
}

proc icq:Log {tags str} { Event Log [concat $tags ICQ] $str }
# Process contact statuses
proc icq:Status {uin newstatus} {
	set [ref Contact:ICQ:$uin](Status) $newstatus
}

proc icq:IP {uin ip} {
	if {$ip!="0.0.0.0"} { set [ref Contact:ICQ:$uin](IP) $ip }
}

proc icq:LanDetails {uin type ip port cookie version} {
	set ref [ref Contact:ICQ:$uin]
	if {$ip!=[set ${ref}(IP)] && $ip!="0.0.0.0"} {set ${ref}(LocalIP) $ip}
	set ${ref}(DC:type) $type
	# If DC is disabled, delete previous instance, if present
	if {$type=="disabled" || $type=="web" || $ip=="0.0.0.0" || $port==0} {
		if {[info exists ${ref}(dc)]} { 
			[set ${ref}(dc)] delete
			unset ${ref}(dc) 
		}
		return
	}
	# If no previous instance found, create new DC handler
	if {![info exists ${ref}(dc)]} { 
		set ${ref}(dc) [::direct::peer $uin -event [nc DcEvent]]
	}
	variable encoding
	if {[info exists ${ref}(encoding)]} {
		set enc [set ${ref}(encoding)]
	} else { set enc $encoding }
	variable dcserver
	if {[info exists dcserver]} {
		set mp [lindex [$dcserver info] 0]
	} else { set mp 0 }
	[set ${ref}(dc)] configure -host $ip -port $port -cookie $cookie\
		-uin [set [namespace current]::uin] -encoding $enc -myport $mp
	Event Log info "DC connection to $uin [[set ${ref}(dc)] state]"
	Event Log warning "DC cookie for $uin $cookie"
}

# Process personal status
proc icq:MyStatus {newstatus} {
	if {$newstatus=="offline"} {
		foreach x [select Contact:ICQ] { set [ref $x](Status) offline }
	} 
	upvar 0 [ref Me](Status) status
	if {$status=="offline" && $newstatus!="offline"} onOnline
	set status $newstatus
}

# Translate ICQ message to alicq messages
proc icq:Incoming {type sender time message} {
	set id [clock seconds]:[clock clicks] 
	set message [string trimright $message "\0"]
	if {$type=="web"} {
		Event Incoming text Contact:ICQ:0 $time "$sender:\n$message" $id
	} else {
		if {$type=="auto"} { 
			set type "text"
			set [ref Contact:ICQ:$sender](description) $message
		} elseif {$type=="authorization" && $message=="granted"} {
			Event Log notice "autorization granted by $sender"
			set [ref Contact:ICQ:$sender](auth) 0
		}
		Event Incoming $type Contact:ICQ:$sender $time $message $id
	}
}

proc icq:MyIP {local remote} {
	Event Log info "Local IP $local, remote IP $remote"
}

proc icq:Registered {uin password} {
	variable recreate 1
	set [namespace current]::uin $uin
	set [namespace current]::password $password
	after idle [nc Event SetStatus online]
}

proc icq:RegistrationRefused {} {
	Event Error registration "ICQ server refused registration of new UIN"
}

proc icq:Outgoing {type sender time message msgid} {
	Event Outgoing $type Contact:ICQ:$sender $time $message $msgid
}

proc icq:Client {uin args} {
	set ref [ref Contact:ICQ:$uin]
	set ${ref}(Client) $args
	# If client supoprts unicode, use UTF-8 for direct connections
	if {[info exists ${ref}(dc)] && [lindex $args end]} {
		[set ${ref}(dc)] configure -encoding utf-8
	}
}

proc icq:ACK {type uin msgid} {
	Event Log info "Ack $type from $uin on $msgid"
	Event Contact:ICQ:$uin|Acknowledgement $type Contact:ICQ:$uin $msgid
}

proc icq:Capabilities {uin caps} {
	set [ref Contact:ICQ:$uin](Capabilities) $caps
	Event Log info "Capabilities $uin: $caps"
}

proc icq:Roster:Items {items time} {
	variable itemCount
	foreach x $items {
		foreach {type id name aux} $x break
		if {!$id && $type=="group"} {
			incr itemCount
			continue
		}
		if {[llength [info commands roster:$type]]} {
			Event Log debug "roster $type: $id $name $aux"
			if {[catch { roster:$type $id $name $aux } v]} {
				Event Error roster "In roster $type $name: $v"
			}
		} else { incr itemCount }
	}
	if {$time} { 
		set roster::updated $time
		set roster::count $itemCount
		set itemCount 0
		MonitorRoster 
	}
}

proc icq:Roster:OK {args} { MonitorRoster }

proc icq:Roster:Update {status} {
	Event Log debug "Contact updated with status $status"
	ProcessResult $status
}

proc icq:Missed {uin reason channel} {
	set uid Contact:ICQ:$uin
	Event Log {error missed} "Missed message from $uin: $reason"
	set auto [mc "Message from this contact was discarded by server."]
	if {[string is integer $reason]} {
		append auto "\n" [mc "The description of the reason is not available, please contact autor of Alicq and report error code."] \
			"\n" [mc "Error code"] " " $reason
	} else { append auto "\n" [mc "Reason:"] " " [mc $reason] }
	Event Incoming text $uid [clock seconds] $auto 0
}

# Direct connection events '''1
proc DcEvent {uin event args} {
	set uid Contact:ICQ:$uin
	if {[llength [info commands DC:$event]]} {
		eval [list DC:$event $uid] $args
	} else { Event Log warning "Unhandler DC event $event from $uid" }
}

proc DC:Log {uid tags str} { Event Log [concat $tags ICQ:DC] "Peer $uid: $str"}

proc DC:Disconnected {uid reason} {
	set ref [ref $uid]
	Event Log {info DC} "DC for $uid is closed: $reason"
	if {[info exists ${ref}(_is_open)]&&[set ${ref}(Status)]!="offline"} {
		Event Log {info DC} "reopening again"
		after 1000 [list [set ${ref}(dc)] connect]
	}
}

proc DC:Outgoing {uid type message time msgid} {
	Event Outgoing $type $uid $time $message $msgid
}

proc DC:Incoming {uid type message} {
	Event Log info "Peer incoming $type from $uid"
	set id [clock seconds]:[clock clicks] 
	Event Incoming $type $uid [clock seconds]\
		[string trimright $message "\0"] $id 
}

proc DC:ACK {uid type id} { Event $uid|Acknowledgement $type $uid $id }

proc DC:Timeout {uid type message id} {
	variable icq
	Event Log notice "DC timoeout, resending via server"
	$icq send $type [lindex [split $uid :] 2] $message $id
}
#}}}1
# Turn DC server on/off
proc MonitorDCServer {args} {
	variable dcserver
	variable uin
	variable icq
	set dcinfo [list]
	if {$direct::enable && ![info exists dcserver]} {
		if {[catch { 
			set dcserver [::direct::server 0 -uin $uin\
					-event [nc DcEvent]]
			set dcinfo [$dcserver info]  } r]
		} { Event Log error $r } else { Event Log info "DC server $dcserver"}
	} elseif {!$direct::enable && [info exists dcserver]} {
		$dcserver delete
		unset dcserver
	}
	if {[info exists icq]} { $icq configure -dcinfo $dcinfo }
}

# Install server-side roster related monitoring of objects
proc MonitorRoster {args} {
	variable groups
	foreach x [select Group:common] { CommonRosterMonitor $x }
	foreach x [select Contact:ICQ] {
		if {![info exists groups($x)]} { CalculateGroup $x }
		CommonRosterMonitor $x 
		ContactRosterMonitor $x
	}
	hook {New:Contact:ICQ:* New:Group:common:*} [nc CommonRosterMonitor]
	hook {New:Contact:ICQ:*} [nc ContactRosterMonitor]
	# Prevent re-entering
	proc MonitorRoster args return 
}

# Calculate group ID for contact
proc CalculateGroup {uid} {
	variable groups
	if {[info exists groups($uid)]} { unset groups($uid) }
	foreach x [get $uid Groups {}] {
		if {$x=="other"} continue
		set var [ref Group:common:$x](rid)
		if {[info exists $var]} { 
			set groups($uid) [set $var] 
			return
		}
	}
}

# Sent common handlers for contact and group: alias change and delete
proc CommonRosterMonitor {uid} {
	set ref [ref $uid]
	if {![info exists ${ref}(rid)]} { Enqueue add $uid }
	trace variable ${ref}(Alias) w [nc Enqueue update $uid]
	trace variable ${ref}(Groups) u [nc Enqueue delete $uid]
}

# Handle movement from one group to another
proc ContactRosterMonitor {uid} {
	trace variable [ref $uid](Groups) w [nc Move $uid]
}
proc Move {uid args} { foreach x {delete add} {Enqueue $x $uid} }

# Enqueue roster operation
proc Enqueue {action uid args} { 
	if {![string is true $roster::modify]} return
	lappend roster::queue [list $action $uid] 
	Event Log debug "enqueue roster item $action for $uid"
	ProcessQueue
}

# Process head of roster action queue if it exists and not locked by previous
# operation
proc ProcessQueue {} {
	variable rosterLock
	if {[info exists rosterLock] || [set [ref Me](Status)]=="offline" ||
	    ![llength $roster::queue]} return
	set rosterLock 1
	foreach {action uid} [lindex $roster::queue 0] break
	if {![RosterSync $action $uid]} { ProcessResult -1 }
}

# Handle result of SSI operation, and shedule processing of new item in
# roster actions queue if needed
proc ProcessResult {result} {
	variable icq
	variable rosterLock
	if {![info exists rosterLock] || ![llength $roster::queue]} return
	unset rosterLock
	foreach {action uid} [lindex $roster::queue 0] break
	# Contact is not authorized, try to add it with auth TLV
	if {$result==14} {
		Event Send authrequest $uid {I want to add you in contact list}
		Event Send futureauth $uid {}
		set [ref $uid](auth) 1
	} else {
		if {$result==0} {
			# Successfully added info to SSI, save rid,
			if {$action=="add"} { 
				variable last_rid
				set [ref $uid](rid) $last_rid 
			} elseif {$action=="delete"} {
				unset [ref $uid](rid)
			}
		} elseif {$result>0} {
			Event Error roster "Roster update error $result"
			if {$result==10} {
			}
		}
		set roster::queue [lrange $roster::queue 1 end]
	}
	after idle [nc ProcessQueue]
}

# Construct command to make proper sync of local and remote contact list item 
proc RosterSync {action uid} {
	variable icq
	set rid [Roster:$action $uid]
	foreach {type name aux} [RosterInfo $uid] break
	set list [list $action [list $type $rid $name $aux]]
	Event Log debug [list $icq roster $action [list $type $rid $name $aux]]
	$icq roster $action [list $type $rid $name $aux]
	return 1
}

proc Roster:add {uid} {
	variable groups
	variable last_rid
	# Whenever we add new contact, group id should be recalculated
	if {[string match Contact:* $uid]} { CalculateGroup $uid }

	# Disable adding contacts which have no group mapping, and
	# "other" group
	if {[string match Contact:* $uid] && ![info exists groups($uid)] ||
	    [string match Group:common:other $uid]} { 
	    	return -code return 0
	}
	# Calculate new unique roster id
	set ridlist [map x [select {Contact:ICQ Group:common}\
		{[info exists rid]}] { set [ref $x](rid)}]
	while 1 {
		set last_rid [expr {int(rand()*32000.0+500)}]
		if {[lsearch $ridlist $last_rid]==-1} { return $last_rid }
	}
}

proc Roster:delete {uid} { 
	set var [ref $uid](rid)
	if {[info exists $var]} { set $var } else { return -code return 0} 
}

proc Roster:update {uid} { 
	set var [ref $uid](rid)
	if {[info exists $var]} { set $var } else { return -code return 0} 
}

# gater information common for all roster update methods
proc RosterInfo {uid} {
	upvar #0 [ref $uid] info
	foreach {type _ name} [split $uid :] break
	set aux [list]
	if {[string match Contact:ICQ:* $uid]} {
		variable groups
		lappend aux group $groups($uid) Alias $info(Alias)
		if {[info exists info(auth)] && $info(auth)} {
			lappend aux auth {}
		}
	} else { set name $info(Alias) }
	list [string tolower $type] $name $aux
}

# Set variable only if it's value will change, to prevent unneeded traces 
# invokation
proc different {var val} {
	if {![info exists $var] || [set $var]!=$val} { set $var $val }
}

proc roster:contact {id name aux} {
	variable groups
	array set Aux $aux
	set uid Contact:ICQ:$name
	set ref [ref $uid]
	set groups($uid) $Aux(group)
	# If groups are not set for the contact yet, try to find proper mapping
	if {![info exists ${ref}(Groups)] || [set ${ref}(Groups)]=="other"} {
	     set g [select Group:common "\[info exists rid\] && \$rid=={$Aux(group)}"]
	     lappend list Groups [expr {($g=="")?"other":[lindex [split $g :] end]}]
	}
	if {[info exists Aux(Alias)]} { lappend list Alias $Aux(Alias) }
	lappend list auth [info exists Aux(auth)] rid $id
	unset Aux
	# If contact already exists, set only fields which differ
	if {[info exists $ref]} { 
		foreach {key val} $list { different ${ref}($key) $val }
	} else { new $uid $list }
}

proc roster:group {id name aux} {
	# Try to map by rid first
	set uid [select Group:common "\[info exists rid\] && \$rid=={$id}"]
	if {$uid!=""} { 
		different [ref $uid](Alias) $name 
		return
	}
	# If no group rid mapping found, try to map by alias
	set uid [select Group "\[info exists Alias\] && \$Alias=={$name}"]
	if {$uid!=""} {
		set [ref $uid](rid) $id
		return
	}
	# If no existsing group found, create new one
	new Group:common:$id [list rid $id Groups [list] Alias $name]
}

# Fake icq command is used when no UIN is specified.
# It always returns empty string as result and sends Error:authorization
# when trying to change status
proc fake_icq {sub args} {
	if {$sub=="status" && [lindex $args 0]!="offline"} {
		Event Error:auth "UIN is not specified or non-numeric"
	}
	list
}

proc EstimatedRoster {} {
	expr { [llength [select {Contact:ICQ Group} {[info exists rid]}]]+
	       ${roster::count} }
}

proc CreateConnection {{new 0} {local_password ""}} {
	foreach x {recreate alias uin password} { variable $x }
	set recreate 0
	if {!$new} {
		if {![info exists uin] || ![info exists password] ||
		    ![string is digit -strict $uin]} {
			return [namespace current]::fake_icq
		} elseif {![info exists alias]} {
			trace variable [ref Me](Status) w [nc GetAlias]
		}
		set local_uin $uin
		set local_password $password
	} else { set local_uin new }
	set icq [icq::icq $local_uin $local_password\
		-capabilities {ICQ RELAY UTF8}\
		-event [namespace current]::ICQEvent]
	foreach x [info vars network::*] {
		set val [set $x]
		if {[catch { $icq configure -[namespace tail $x] $val } r]} {
			Event Log error "Cant configure [namespace tail $x]: $r"
		}
	}
	if {$local_uin!="new" && [string is true $roster::use]} {
		if {$roster::updated>0} {
			Event Log info "Estimated roster: [EstimatedRoster], $roster::updated"
			set roster [list $roster::updated [EstimatedRoster]]
		} else {set roster 1}
	} else {set roster 0}
	$icq configure -roster $roster
	# Create contact lists
	$icq contacts all add [Uid2Uin [select Contact:ICQ]]
	foreach id {visible invisible} {
		set cmd "\[info exists $id\]&&\[string is true \$$id\]"
		$icq contacts $id add [Uid2Uin [select Contact:ICQ $cmd]]
	}
	# Set contact encodings
	variable encoding
	$icq encoding $encoding
	foreach x [select Contact:ICQ {[info exists encoding]}] {
		$icq encoding [set [ref $x](encoding)] [Uid2Uin $x]
	}
	variable dcserver
	if {[info exists dcserver]} { $icq configure -dcinfo [$dcserver info] }
	set icq
}

# Register new ICQ UIN. New icq command created for this purpose
handler ICQ:Register registrator {password} {
	variable reg
	Event ICQ:CancelRegistration
	set reg [CreateConnection 1 $password]
}
# Cancel ICQ UIN registration. Only one regustration command at the time
# allowed, thus no task identifiers required
handler ICQ:CancelRegistration registration_canceler {args} {
	variable reg
	if {[info exists reg]} { 
		$reg delete 
		unset reg
	}
}

handler search Search {info} {
	variable icq
	$icq search $info
}

handler UpdateInfo UserInfo {info} {
	variable icq
	$icq personal $info
}

# Add monitoring of contact properties
proc MonitorContact {uid} {
	set ref [ref $uid]
	foreach x {visible invisible} { 
		trace variable ${ref}($x) w [nc tracklist $x [Uid2Uin $uid]]
	}
	trace variable ${ref}(encoding) w [nc trackencoding [Uid2Uin $uid]]
	trace variable ${ref} u [nc trackall delete $uid]
}

# Track changes in contact list and sync it with icq command
proc trackall {action uid args} {
	if {[lindex $args 1]!=""} return
	variable icq
	$icq contacts all $action [Uid2Uin $uid]
}

# Track changes in visible/invisible lists and sync them with icq command
proc tracklist {list uin ref field args} {
	variable icq
	upvar ${ref}($field) val
	set action [expr {[string is true $val]?"add":"delete"}]
	$icq contacts $list $action $uin
}

# Track changes of contact's encoding
proc trackencoding {uin ref field args} {
	variable icq
	upvar 1 ${ref}($field) val
	$icq encoding $val $uin
	if {[info exists ${ref}(dc)]} { 
		[set ${ref}(dc)] configure -encoding $val
	}
}

# Track changes of global encoding
proc GlobalEncoding {ref args} {
	variable icq
	upvar 1 $ref val
	$icq encoding $val
}

proc Reconfigure {name args} {
	variable icq
	if {[catch { $icq configure -$name [set network::$name] } r]} {
		Event Log error "Cant configure $name: $r"
	}
}

proc CredChanged {what name args} {
	upvar 1 $name val
	Event Log debug "$what changed to $val"
	variable recreate 1
	# Cancel registration of new UIN
	Event ICQ:CancelRegistration
}

handler ConfigLoaded ConfigLoaded {args} {
	foreach x {icq alias status} { variable $x }

	if {[info exists alias]} { set [ref Me](Alias) $alias } else {
		trace variable [ref Me](Alias) w [nc SetAlias]
	}

	foreach x [select Contact:ICQ] { MonitorContact $x }
	hook New:Contact:ICQ:* [nc MonitorContact]
	hook New:Contact:ICQ:* [nc trackall add]

	trace variable [namespace current]::encoding w [nc GlobalEncoding]

	foreach x {uin password} {
		trace variable [namespace current]::$x w [nc CredChanged $x]
	}
	foreach x [info vars network::*] {
		trace variable $x w [nc Reconfigure [namespace tail $x]]
	}

	MonitorDCServer
	trace variable direct::enable w [nc MonitorDCServer]

	set icq [CreateConnection]
	Event SetStatus $status 
}

handler Send Send {type uid message} {
	if {![string match Contact:ICQ:* $uid]} { return -code continue }
	variable icq
	if {[string equal $type "SMS"]} {
		set var [ref $uid](mobile)
		if {[info exists $var]} {
			Event Send text Contact:SMS:[set $var] $message
		}
		return -code continue
	}
	set refdc [ref $uid](dc)
	if {[lsearch {furtureauth authorization authrequest} $type]==-1 &&
	    [info exists $refdc] && [[set $refdc] state]=="established"} {
		[set $refdc] send $type $message
	} else { $icq send $type [lindex [split $uid :] 2] $message }
} 0.90

handler Send SendSMS {type uid message} {
	if {![string match Contact:SMS:* $uid]} {return -code continue}
	variable icq
	$icq send SMS [lindex [split $uid :] 2] $message
} 0.95

handler MessageTypes messagetypes {uid} {
	if {![string match Contact:ICQ:* $uid]} { return -code continue }
	set lst {text URL authrequest}
	if {[info exists [ref $uid](mobile)]} { lappend lst SMS }
	set lst
}

handler SetStatus SetStatus {setstatus} {
	foreach x {icq recreate} { variable $x }
	if {[set [ref Me](Status)]=="offline" && $recreate} {
		if [info exists icq] { $icq delete }
		set icq [CreateConnection]
	}
	$icq status $setstatus
}

handler ChangeICQPassword ChangeICQPassword {password ref} {
	variable icq
	$icq password $password $ref
}

handler InfoRequest InfoRequest {uid ref} {
	variable icq
	if {![string match Contact:ICQ:* $uid]} return
	set uin [lindex [split $uid :] 2]
	if {[string is digit $uin]} { $icq info $uin $ref }
}

proc onOnline {} {
	RequestInfo
}

variable request_info [list]
handler New:Contact:ICQ:* RequestAlias {uid} {
	if {[info exists [ref $uid](Alias)]} return
	variable request_info
	lappend request_info $uid
	if {[set [ref Me](Status)]!="offline"} { RequestInfo }
} 0.06

array set need_info [list]
variable refcnt 100
# Send info request on each item in need_info list
proc RequestInfo {} {
	variable request_info
	if {![llength $request_info]} return
	variable refcnt
	variable need_info
	variable icq
	set uid [lindex $request_info 0]
	$icq info [lindex [split $uid :] end] $refcnt
	set need_info($refcnt) $uid
	if {[incr refcnt]>250} { set refcnt 100 }
	set request_info [lrange $request_info 1 end]
	after 300 [nc RequestInfo]
}

handler Info Nick2Alias {ref info} {
	variable need_info
	if {![info exists need_info($ref)]} return
	set uid $need_info($ref)
	unset need_info($ref)
	array set Info $info
	if {![info exists Info(Nick)]} return
	set [ref $uid](Alias) $Info(Nick)
}

# Sync icq alias when global alias changes
proc SetAlias {name idx args} {
	upvar 1 ${name}($idx) val
	variable alias $val
}

proc GetAlias {args} {
	foreach x {icq uin} { variable $x }
	if {[set [ref Me](Status)]=="offline"} return
	trace vdelete [ref Me](Status) w [nc GetAlias]
	set ref [expr {[clock seconds] & 0xFF}]
	hook Info [nc NewContactInfo Me $ref]
	$icq info $uin $ref
}

proc NewContactInfo {uid myref ref info} {
	if {$myref!=$ref} return
	array set Info $info
	if {[info exists Info(Nick)]} { set [ref $uid](Alias) $Info(Nick) }
	unhook Info [nc NewContactInfo $uid $ref]
}

handler ChatOpen PeerOpen {uid args} {
	if {!$direct::enable} return
	set ref [ref $uid]
	if {[info exists ${ref}(dc)]} { 
		set ${ref}(_is_open) 1
		[set ${ref}(dc)] connect 
	}
}

handler ChatClose PeerClose {uid args} {
	set ref [ref $uid]
	if {[info exists ${ref}(_is_open)]} { unset ${ref}(_is_open) }
	if {[info exists ${ref}(dc)]} { [set ${ref}(dc)] disconnect }
}

