(**************************************************************************)
(*                                                                        *)
(*     The Alt-ergo theorem prover                                        *)
(*     Copyright (C) 2006-2010                                            *)
(*                                                                        *)
(*     Sylvain Conchon                                                    *)
(*     Evelyne Contejean                                                  *)
(*     Stephane Lescuyer                                                  *)
(*     Mohamed Iguernelala                                                *)
(*     Alain Mebsout                                                      *)
(*                                                                        *)
(*     CNRS - INRIA - Universite Paris Sud                                *)
(*                                                                        *)
(*   This file is distributed under the terms of the CeCILL-C licence     *)
(*                                                                        *)
(**************************************************************************)



open Format
open Hashcons
open Options

module Sy = Symbols
module T = Term
module A = Literal
module L = List


type 'a affectation = {i : 'a ; v : 'a ; w : 'a ; p : int}

type 'a apply = Get of 'a | Set of 'a affectation list

type 'a abstract = {a : 'a ; op : 'a apply;  ty : Ty.t ; gp : int}

module type ALIEN = sig
  include Sig.X
  val embed : r abstract -> r
  val extract : r -> (r abstract) option
end

module Make(X : ALIEN) = struct

  type t = X.r abstract
  type r = X.r

  let name = "Arr"

  let existential _  = false

  let is_mine_a _ = false

  let is_mine_symb f = arrays && (Sy.is_get f || Sy.is_set f)

  let is_mine_type t = 
    fprintf fmt "revoir 1@."; assert false
    
  let type_info t = t.ty
    

  let print_l fmt = function
      [] -> assert false
    | af::l ->
	fprintf fmt "(%d) %a:=%a" af.p X.print af.i X.print af.v;
        L.iter (
          fun af -> fprintf fmt ", (%d) %a:=%a" 
            af.p X.print af.i X.print af.v) l
          
  let print fmt t = 
    match t with 
      | {a=a; op=Get i} -> 
          fprintf fmt "get(%a,%a)" X.print a X.print i
            
      | {a=a; op=Set [] } -> 
          fprintf fmt "%a" X.print a
            
      | {a=a; op=Set l} -> 
          fprintf fmt "set(%a,%a)" X.print a print_l l

            
  let embed r = 
    match X.extract r with
      | None   -> {a=r; op=Set[]; ty=X.type_info r; gp=0}
      | Some t -> t
        

  let rec flatten ({a=ae; op=ope;gp=gpe} as te) =
    match X.extract ae with
      | None -> te
      | Some ({a=ai; op=opi; gp=gpi} as ti) ->
          match ope, opi with
            | Set [] , _      -> flatten ti
            | Set le , Set [] -> flatten {te with a=ai}
            | Set le , Set li -> 
                let le = List.map (fun af -> {af with p = af.p + gpi}) le in
                flatten {te with gp = te.gp + ti.gp; a=ai; op=Set (li @ le)}
            | _ -> te


  let rec nz t = 
    let t' = flatten t in
    match t' with
      | { op=Set[] } -> t'
      | { op=Set l } -> nz_set t' l
      | { op=Get i } -> nz_get t' i

          
  and nz_x x = match nz (embed x) with
    | {op=Set []; a=a} -> a
    | t -> X.embed t
  
  (* Page 93 de la thse de Levitt *)
  and nz_set t l = 
    let l_rev = canon_set_5_11 l in
    let l_xxx = canon_set_5_12 l_rev in
    let l_ord = L.fast_sort (fun x y -> x.p - y.p) l_xxx in
    match l_ord with 
      | [] -> embed t.a
      | _ -> {t with op= Set l_ord}
          
  and canon_set_5_11 l =
    L.fold_left 
      (fun acc af -> 
         let v = nz_x af.v in
         let w = nz_x af.w in
         if X.equal v w then acc
         else {af with i=nz_x af.i; v=v; w=w} :: acc) [] l

  and canon_set_5_12 l_rev =
    let cmp af1 af2 = 
      let c = X.compare af2.i af1.i in
      if c <> 0 then c 
      else af2.p - af1.p in
    match List.fast_sort cmp l_rev with
      | []    -> []
      | af::l -> 
          let l_xxx,_ = 
            List.fold_left
              (fun (z,cr) af ->
                 if X.equal cr.i af.i then z,cr else af::z, af) ([af],af) l in
          l_xxx

  and canon_set_5_13 l_xxx = l_xxx
  
  (* XXX : a revoir *)
  and nz_get ({a=ae} as t) i = 
    let i = nz_x i in
    let res = match X.extract ae with
      | Some ({a=ai; op=Set l} as t') -> 
          begin match tighten_get i (L.rev l) with
            | Some v, [] -> embed v
            | None, []   -> {t' with op=Get i}
            | None, l    -> 
                let t'' = {t' with op=Set (L.rev l)} in
                {t with a=X.embed t''}
            | _ -> assert false
          end
      | _ -> {t with op=Get i}
    in 
    (*fprintf fmt "nz_get (%a) = %a @." print t print res;*)
    res
   
  and tighten_get i = function
      | [] -> None, []
      | af::r as l->
          try match X.solve (fun r -> r) i af.i with
            | [] -> Some af.v, []
            | _  -> None, l
          with Exception.Unsolvable -> tighten_get i r

(***
  let rec mem_assoc i = function
    | [] -> false
    | (j,_,_) ::l ->
        if X.equal i j then true else mem_assoc i l

  let rec normalize t = 
    (* Il faut normalize les indices et les valeurs aussi *)
    let art = match X.extract t.ar with
      | Some ({op = Get _ | Set (_::_)} as t') -> normalize t'
      | _  -> embed t.ar
    in
    match t.op, art with
      | Set l', {ar=a;op=Set l}  ->
          let l_ord = L.filter (fun (i,v,w) -> not (X.equal v w)) (l@l') in
          let l_rev = 
            L.fold_left
              (fun acc (i,v,w) ->
                 (* ce test fait chuter les performances *)
                 if mem_assoc i acc then acc 
                 else (i,v,w)::acc
              )[] l_ord
          in 
          {art with op = Set (L.rev l_rev)}
      | Get i, {ar=a;op=Set l}-> 
          let rec f = function
              [] -> None, []
            | ((j,v,_)::l) as l'->
                try match X.solve (fun r -> r) i j with
                  | [] -> Some v, []
                  | _  -> None, l'
                  
                with Exception.Unsolvable -> f l
          in 
          begin
            match f (L.rev l) with
            | Some v, [] -> embed v
            | None, []   -> {art with op=Get i}
            | None, l    -> 
                let art = {art with op=Set (L.rev l)} in
                {t with ar = X.embed art}
            | _ -> assert false
          end
      | _ ->  t
**)

  let normalize t =
    let res = nz t in
    if debug_arrays then
      fprintf fmt "normalize(%a) = %a@.@." print t print res;
    res


  let is_mine t = 
    match normalize t with 
        {a=r; op=Set[]} -> r | t -> X.embed t

  let rec fold_2 = function
      [],[] -> 0
    | af::l1, af'::l2 ->
        let c = X.compare af.i af'.i in
        if c <> 0 then c
        else 
          let c = X.compare af.v af'.v in
          if c <> 0 then c
          else fold_2 (l1,l2)
    | _ -> assert false

  let compare_operators = function
    | Get i, Get j -> X.compare i j
    | Get _, _     ->  1
    | _    , Get _ -> -1
        
    | Set [] , Set [] -> 0
    | Set [] , Set _ -> 1
    | Set _ , Set [] -> -1
    | Set l1 , Set l2 -> 
        let c = L.length l1 - L.length l2 in
        if c <> 0 then c else fold_2 (l1,l2)

  let compare t1 t2  = 
    let {a=a1; op=op1} = normalize t1 in
    let {a=a2; op=op2} = normalize t2 in

    let c = X.compare a1 a2 in
    if c <> 0 then c 
    else compare_operators (op1,op2)

  let leaves t = 
    let t = normalize t in
    let l = X.leaves t.a in
    match t.op with
      | Get i -> (is_mine t) :: (X.leaves i) @ l
      | Set la-> 
          L.fold_left 
            (fun z af -> (X.leaves af.i)@(X.leaves af.v)@(X.leaves af.w)@z)
            l la
            
  let color _ = assert false

  let get_cst l = l

  let set_cst l = l

  let make t = 
    let {T.f=f; xs=xs; ty=ty} = T.view t in
    match Sy.is_set f, Sy.is_get f, xs with
      | false, true, [a; i] ->
          let a, c1 = X.make a in
          let i, c2 = X.make i in 
          is_mine {a=a; op=Get i; ty=ty; gp=0} , get_cst (c1@c2)

      | true, false, [a; i; v] ->
          let a, c1 = X.make a in
          let i, c2 = X.make i in 
          let v, c3 = X.make v in 
          let w     = is_mine {a=a; op=Get i; ty=X.type_info v; gp=0} in
          let af = {i=i; v=v; w=w; p=1} in
          is_mine {a=a; op=Set [af]; ty=ty; gp=1} , set_cst (c1@c2@c3)

      | _ -> assert false

  let subst p _P t =
    let t = {t with a=X.subst p _P t.a} in
    match t with
      | {op=Get i} ->  
          if X.equal p (is_mine t) then _P
          else is_mine {t with op=Get (X.subst p _P i)}
      | {op=Set l} ->
          let l = 
            L.map (fun af -> 
                     {af with 
                        i = X.subst p _P af.i;
                        v = X.subst p _P af.v;
                        w = X.subst p _P af.w }) l 
          in
          is_mine {t with op=Set l}
    

  let subst p _P t =
    let res = subst p _P t in
    if debug_arrays then
      fprintf fmt "[%s] subst    %a@.\tby     %a@.\tin     %a@.\tyields %a@."
        name X.print p X.print _P X.print (is_mine t) X.print res;
    res

  let fresh_var = 
    let cpt = ref (-1) in
    fun () -> 
      incr cpt; 
      let sy = Sy.name (sprintf "arr_v_%d" !cpt) in
      let t = Term.make sy [] Ty.Tint in 
      fst (X.make t)


  let solve_set_empty r1 r2 = 
    let lvs = L.fast_sort X.compare (X.leaves r1) in
    let lvs = 
      L.fold_left 
        (fun acc e -> 
           if L.mem e acc then acc 
           else match embed e with {op=Get _} -> e::acc | _ -> acc
        ) [] lvs in
    if debug_arrays then 
      begin
	fprintf fmt "@. les feuilles de %a@." X.print r1;
	List.iter (fprintf fmt "> %a@." X.print) lvs;
	fprintf fmt "@.";
      end;
    let t2 = embed r2 in
    (* pas complet : il faut folder ces substs sur elles mmes *)
    let sbs = 
      L.fold_left
        (fun acc e ->
           if compare (embed e) t2 >= 0 then acc
           else (e,fresh_var ()) :: acc 
        ) [] lvs
    in 
    let r1 = L.fold_left (fun r (p,_P) -> X.subst p _P r) r1 sbs in
    (r2, r1)::sbs
    

  (* non compatible AC pour le moment *)
  let rec solve repr r1 r2 = 
    let t1 = normalize (embed r1) in
    let t2 = normalize (embed r2) in
    let r1 = is_mine t1 in
    let r2 = is_mine t2 in
    if compare t1 t2 = 0 then []
    else match t1.op, t2.op with
      | Get _ , Get _  -> let z = fresh_var () in [r1,z; r2,z]
      | Get _ , _      -> [r1,r2]
      | _     , Get _  -> [r2,r1]
      | Set [], Set [] -> 
          let c = X.compare t1.a t2.a in
          if c > 0 then [r1,r2]
          else if c < 0 then [r2,r1]
          else assert false

      | Set [], Set l  -> solve_set_empty r2 r1

      | Set l , Set [] -> solve_set_empty r1 r2

      | Set l , Set l' -> 
          (*failwith "Cannot solve set(...) = set(...)"*)
          let c = X.compare r1 r2 in
          if c > 0 then [r1,r2]
          else if c < 0 then [r2,r1]
          else assert false

  let solve repr r1 r2 = 
    if debug_arrays then 
      fprintf fmt "[%s] solving %a = %a yields:@." name X.print r1 X.print r2;
    let sbs = solve repr r1 r2 in
    let sbs = L.fast_sort (fun (a,_) (x,y) -> X.compare x a)sbs in
    let sbs = L.rev sbs in
    if debug_arrays then
      let cpt = ref 0 in
      L.iter 
        (fun (p,v) -> 
           incr cpt;
           fprintf fmt " %d) %a |-> %a@." !cpt X.print p X.print v) sbs
    else ();
    sbs
      

  module Rel = struct
    type r = X.r
    type t = unit 
    let assume env la  = env, []
    let case_split env = env, []
    let empty _ = ()
    let query _ _  = false
    let add env _ = env
  end

end


(*====================================================================

  let rec normalize t = 
    match t with
      |{ar=ar; op=Get i} ->
         begin
           match embed ar with
             | {op=Set(j,v)} when X.equal i j -> (normalize (embed v))
             | _ -> t
         end
           
      | {ar=ar; op=Set(i,v)} ->
          begin 
            match embed v with
              | {ar=ar'; op=Get j} when X.equal ar ar' && X.equal i j ->
                  normalize (embed ar)
              | _ -> t
          end

      | _ -> t

  let compare_operators = function
    | Get i, Get j -> X.compare i j
    | Nop  , Nop  -> 0
    | Set (i,v), Set (j,w) -> 
        let c = X.compare i j in
        if c <> 0 then c else X.compare v w
    
    | Get _, _     ->  1
    | _    , Get _ -> -1
    | Nop  , _     ->  1
    | _    , Nop   -> -1

  let compare t1 t2  = 
    match normalize t1, normalize t2 with
      | { ar=a1;op=op1 }, { ar=a2;op=op2 } ->
          let c = compare_operators (op1,op2) in
          if c <> 0 then c else X.compare a1 a2
        
  let leaves t = 
    let arl = X.leaves t.ar in
    match t.op with 
      | Nop -> arl
      | Get i -> (is_mine t) :: (X.leaves i) @ arl
      | Set(i,v) -> (X.leaves i) @ (X.leaves v) @arl


  let type_info t = t.ty

  let color _ = assert false

  let make t = 
    let {T.f=f; xs=xs; ty=ty} = T.view t in
    match is_set f, is_get f, xs with
      | true, false, [a; i; v] ->
          let a, c1 = X.make a in
          let i, c2 = X.make i in 
          let v, c3 = X.make v in 
          is_mine {ar=a; op=Set(i,v); ty=ty} , c1@c2@c3

      | false, true, [a; i] ->
          let a, c1 = X.make a in
          let i, c2 = X.make i in 
          is_mine {ar=a; op=Get i; ty=ty} , c1@c2
            
      | _ -> assert false
          

  let substit p _P t = 
    let t' = { t with ar=X.subst p _P t.ar} in
    match t with 
      | {ar=ar; op=Get i} when X.equal p (is_mine t) -> 
          _P
            
      | {ar=ar; op=Get i} -> 
          is_mine {t' with op = Get (X.subst p _P i)}

      | {ar=ar; op=Set (i,v)} -> 
          is_mine {t' with op = Set (X.subst p _P i,X.subst p _P v)}
    
      | {ar=ar; op=Nop} -> 
          t'.ar
          
  let subst p _P t = 
    let res = substit p _P t in
    if debug_arrays then
      fprintf fmt "[%s] subst    %a@.\tby     %a@.\tin     %a@.\tyields %a@."
        name X.print p X.print _P X.print (is_mine t) X.print res;
    res

  let fresh_var = 
    let cpt = ref (-1) in
    fun () -> 
      incr cpt; 
      let sy = Sy.name (sprintf "arr_v_%d" !cpt) in
      let t = Term.make sy [] Ty.Tint in 
      fst (X.make t)



  (* non compatible AC pour le moment *)
  let rec solve repr r1 r2 = 
    if X.equal r1 r2 then []
    else 
      let t1 = embed r1 in
      let t2 = embed r2 in
      match t1, t2 with

        (* Choose of a bad theory for sovling: OK *)
        | {op=Nop}, {op=Nop} -> 
            assert false 

              
        (* Solving get: OK *)
        | {op=Get _}, {op=Get _} -> let z = fresh_var () in [r1,z; r2,z]
                                                              
        | {op=Get _}, _  -> [r1,r2]
            
        | _ , {op=Get _} -> [r2,r1]
            


        (* Solving set : KO *)
        | {op=Set _}, {op=Set _} ->  assert false
            
        | {op=Set (i,v);ar=ar}, _  when X.equal ar r2 -> 
            let v' = repr (is_mine {ty= X.type_info v; ar=ar; op= Get i}) in
            solve repr v v'
                
        | {op=Set _}, _   -> [r2,r1]
            
        | _, {op=Set (i,v);ar=ar}  when X.equal ar r1 -> 
            let v' = repr (is_mine {ty= X.type_info v; ar=ar; op= Get i}) in
            solve repr v v'
              
        | _ , {op=Set _} -> [r1,r2]
            

  let solve repr r1 r2 = 
    let sbs = solve repr r1 r2 in
    let sbs = L.fast_sort (fun (a,_) (x,y) -> X.compare x a)sbs in

    let sbs = L.filter (fun (p,_P) -> not (L.mem p (X.leaves _P))) sbs in
    if debug_arrays then begin
      fprintf fmt "[%s] solving %a = %a yields:@." name X.print r1 X.print r2;
      let cpt = ref 0 in
      L.iter 
        (fun (p,v) -> 
           incr cpt;
           fprintf fmt " %d) %a |-> %a@." !cpt X.print p X.print v) sbs
    end;
    sbs

  (*********************************************************************************)
  module Rel = struct
    type r = X.r
        
    let values_of  = function
      | A.Eq (r1,r2) | A.Neq (r1,r2) -> [r1; r2]  | A.Builtin (_,_,l) -> l
          
    module Semantic_view = struct
    
      type t = r A.view
      exception Out of int 
      
      let compare a b = 
        match a,b with
          | A.Eq _, A.Eq _ | A.Neq _, A.Neq _ | A.Builtin _, A.Builtin _ ->
                let la = values_of a in
                let lb = values_of b in
                let c = L.length la - L.length lb in
                if c <> 0 then c 
                else
                  let la = L.fast_sort X.compare la in
                  let lb = L.fast_sort X.compare lb in
                  (try 
                     L.iter2
                       (fun r1 r2 ->
                          let c = X.compare r1 r2 in 
                          if c <> 0 then raise (Out c)
                       ) la lb;
                     0
                   with Out c -> c)
          | A.Eq _, _  ->  1
          | _, A.Eq _  -> -1
          | A.Neq _, _ ->  1
          | _, A.Neq _ -> -1
    end
      
    module Setsv = Set.Make(Semantic_view)

        
    type t = {
      eqs : Setsv.t;
      dis : Setsv.t}
        
        
        
    let assume_atom (env,eqs) (a,_) = 
      (*Print.consider a;*)
      let env, values = match a with
        | A.Eq(r1,r2)       -> {env with eqs = Setsv.add a env.eqs}, [r1;r2]
        | A.Neq(r1,r2)      -> {env with dis = Setsv.add a env.dis}, [r1;r2]
        | A.Builtin(_,_,rl) -> env, rl
      in 
      L.fold_left
        (fun ((env,eqs) as acc) r ->
           match embed r with
             | {ar=ar; op=Get i} as gt ->
                 begin
                   match embed ar with
                     | {ar=ar'; op=Set(j,v)} -> 
                         let i_eq_j    = A.Eq(i,j) in
                         let i_eq_j_c  = A.Eq(r,v) in
                         if Setsv.mem i_eq_j env.eqs then env, i_eq_j_c :: eqs
                         else  
                           let i_neq_j   = A.Neq(i,j) in
                           let i_neq_j_c = A.Eq(r,is_mine{gt with ar=ar'}) in
                           if Setsv.mem i_neq_j env.dis then env, i_neq_j_c :: eqs
                           else 
                             begin (*fprintf fmt "needs some additionnal treatment@.";*)
                               acc
                             end
                     | _ -> acc
                 end
             | _ -> acc
        ) (env,eqs) values
        
(*
                   let c1 = [i_neq_j; i_neq_j_c] in
                   let c2 = [i_eq_j; i_eq_j_c] in
                   let cs = M.add gs [c1; c2] env.cs in
                   let imp = Msa.accumulate i_neq_j i_neq_j_c env.imp in
                   let imp = Msa.accumulate i_eq_j i_eq_j_c imp in
                   { cs = cs; imp = imp}
               | _ -> assert false
*)



    let assume env la  = 
      let env,eqs = L.fold_left assume_atom (env,[]) la in
      (*Print.env env;*)
      env, L.map (fun a -> (a, None)) eqs
    
    
    let case_split env = env, []
    
    let empty _ = {
      eqs = Setsv.empty;
      dis = Setsv.empty;
    }
    
    let query _ _  = false
    
    let add env _ = env
  end

end
*)
