# $Id: EximPerl.pm,v 1.27 2005/01/13 00:14:33 braeucup Exp $

#
# MTA module for Exim+Embedded Perl setup
#

package AMAVIS::MTA::EximPerl;
use strict;
use vars qw($VERSION);
$VERSION='0.1';

use AMAVIS;
use AMAVIS::Logging;
use IO::File;
use File::Path;

use vars qw(
	    $cfg_exim_binary
	    $cfg_exim_args
	   );

sub init {
  my $self = shift;
  my $args = shift;
  $cfg_exim_binary = ($AMAVIS::cfg->val('Exim', 'exim') || '/usr/sbin/exim');
  $cfg_exim_args = ($AMAVIS::cfg->val('Exim', 'args') || '-oMr no-scan -i -f');

  if (! -x $cfg_exim_binary) {
    writelog($args,LOG_CRIT, "$cfg_exim_binary not executable");
    return 0;
  }

  writelog($args,LOG_DEBUG,__PACKAGE__." initialized.");
  # writelog($args,LOG_INFO,"Working on Message-ID ".$$args{'eximperl_mid'});
  # Return successfully
  return 1;
}

sub cleanup {
  my $self = shift;
  my $args = shift;
  return 1;
}

# Create directory for unpacking and inspecting message
sub create_unpack_dir() {
  my $directory = create_tmp_dir("$AMAVIS::cfg_unpack_dir/amavis-unpack-");
  return $directory;
}

# Create temp directory. Try creating $prefix[date]-[pid] 
# Return created dir or undef if unsuccessful after 10 tries
sub create_tmp_dir( $ ) {
  my $prefix = shift;
  my $i = 0;
  while (1) {
    my $tmpdir = sprintf("$prefix%.8x-%.4x",time,$$);
    unless (mkpath ($tmpdir, 0, 0770)) {
      if (++$i > 10) {
	return undef;
      }
      else {
	next;
      }
    }
    else {
      return $tmpdir;
    }
  }
}

# Create temp dir and write mail
sub get_directory($) {
  my $self = shift;
  my $args = shift;

  writelog($args,LOG_DEBUG, 
	   "Working with Exim Message ID $$args{eximperl_mid}");
  my $log_sender = ($$args{'sender'} or '<>');
  writelog($args,LOG_DEBUG, "Sender: $log_sender");
  writelog($args,LOG_DEBUG, "Recipient(s): "
	   .join (' ',@{$$args{'recipients'}}));

  # Create temp directory
  my $directory = create_unpack_dir();
  $$args{'directory'} = $directory;
  mkdir "$directory/parts", 0777;
  
  # Open message file that is later going to be disected and scanned
  my $output = IO::File->new("+>$directory/email.txt");

  # Get header file.
  my $exim = IO::File->new("$cfg_exim_binary -Mvh ".$$args{eximperl_mid}."|") 
    or return undef;

  # Skip over Exim's information
  while (<$exim>) {
    last if /^\s*$/;
  }

  # Read header
  $$args{'headers'}='';
  while (<$exim>) {
    s/^[0-9]{3,}. //;
    print $output $_;
    $$args{'headers'}.=$_;
  }
  $exim->close();

  # Insert newline to separate header lines and body
  print $output "\n";

  # Get body file
  $exim = IO::File->new("$cfg_exim_binary -Mvb ".$$args{eximperl_mid}."|") 
    or return undef;

  # Read body
  <$exim>; while (<$exim>) {
    print $output $_;
  }
  $exim->close();

  # Message file has been written, reset file pointer and put it into
  # the record.
  $output->seek(0,0);
  $$args{'filehandle'} = $output;

  # Return successfully
  return 1;
}

sub accept_message($) {
  my $self = shift;
  my $args = shift;
  writelog($args,LOG_INFO, __PACKAGE__.
	   ": Accepting message ".$$args{'eximperl_mid'});
  $$args{'eximperl_returncode'} = 'accept';
  $$args{'status'} = 'accept';

  # Return successfully
  return 1;
}

sub drop_message($) {
  my $self = shift;
  my $args = shift;
  writelog($args,LOG_WARNING, __PACKAGE__.
	   ": Dropping message ".$$args{'eximperl_mid'});
  $$args{'eximperl_returncode'} = 'drop';

  # Return successfully
  return 1;
}

sub freeze_message( $ ) {
  my $self = shift;
  my $args = shift;
  writelog($args,LOG_WARNING, __PACKAGE__.": Freezing message ".
	   $$args{'eximperl_mid'});

  if (AMAVIS->quarantine_problem_message($args)) {
    $$args{'eximperl_returncode'} = 'drop';
    # Return successfully
    return 1;
  }
  else {
    writelog($args,LOG_ERR, __PACKAGE__.
	     ': Unable to put message into problem dir. Freezing message.');
    $$args{'eximperl_returncode'} = 'freeze';
    # Return successfully
    return 1;
  }
}

sub send_message($$) {
  my $self = shift;
  my $args = shift;
  my $message = shift;
  my $sender = shift;
  my @recipients = @_;
  writelog($args,LOG_DEBUG, __PACKAGE__.": Sending mail from $sender to ".
	   join(', ',@recipients));

  my @exim_args;

  push @exim_args, split(/\s+/,$cfg_exim_args);
  push @exim_args, $sender;
  push @exim_args, @recipients;

  writelog($args,LOG_DEBUG, __PACKAGE__.": Running $cfg_exim_binary ".
	   join(' ',@exim_args));

  open(MAIL, "|-") || exec($cfg_exim_binary, @exim_args);
  print MAIL $message;
  close(MAIL);

  if ($? != 0) {
    writelog($args,LOG_ERR,
	     __PACKAGE__.": $cfg_exim_binary @exim_args exited with ".($?>>8));
  }

  # Return successfully
  return 1;
}

1;
