# $Id: PMilter.pm,v 1.4 2003/08/26 09:42:32 bengen Exp $

#
# MTA module for Milter setup, using Todd Vierling's PMilter from
# <http://www.duh.org/pmilter>
#

package AMAVIS::MTA::PMilter;
use strict;
use vars qw($VERSION);
$VERSION='0.1';

use AMAVIS;
use AMAVIS::Logging;
use Sendmail::Milter;

use IO::File;
use File::Path;

use File::Copy;
use Sys::Hostname;

# For receiving mail
use IO::Socket;

use POSIX qw(setsid);

use vars qw(
	    $cfg_x_header
	    $cfg_x_header_tag
	    $cfg_x_header_line

	    $cfg_daemon
	    $cfg_pidfile
	    $cfg_umask

	    $cfg_milter_socket

	    $cfg_sendmail_binary
	    $cfg_sendmail_args

	    %milter_callbacks

	    $hostname

	    $server
	    $conn

	    $saved_args
	    $mta_result

	    $server_pid
	    $running
	    $signame
	   );

sub init {
  my $self = shift;
  my $args = shift;

  $cfg_daemon = $AMAVIS::cfg->val('Milter', 'daemon');
  $cfg_pidfile = $AMAVIS::cfg->val('Milter', 'pidfile');
  $cfg_umask = ($AMAVIS::cfg->val('Milter', 'umask') || '000');

  $cfg_milter_socket=$AMAVIS::cfg->val('Milter', 'milter socket');
  if (! defined $cfg_milter_socket) {
    writelog($args,LOG_CRIT, __PACKAGE__.
	     ": Milter socket for client not specified");
    return 0;
  }

  $cfg_sendmail_binary=($AMAVIS::cfg->val('Milter', 'sendmail') ||
			'/usr/sbin/sendmail');
  if  (! -x $cfg_sendmail_binary) {
    writelog($args,LOG_CRIT,__PACKAGE__.
	     ": $cfg_sendmail_binary not executable");
    return 0;
  }
  $cfg_sendmail_args=$AMAVIS::cfg->val('Milter', 'args');

  $cfg_x_header_tag = $AMAVIS::cfg->val('global', 'x-header-tag');
  $cfg_x_header_line = $AMAVIS::cfg->val('global', 'x-header-line');

  $hostname=hostname();

  my $pid;
  if ((defined $cfg_daemon) && $cfg_daemon eq 'yes') {
    if (!defined ($pid = fork)) {
      writelog($args,LOG_ERR, __PACKAGE__.": fork() failed.");
      return 0;
    }
    # If all went well...
    if ($pid) {
      # We are the parent
      # The parent will (should) only have children that have been
      # forked for accepting socket connections.
      exit 0;
    }
    # We are the child.
    # So we become a daemon.
    setsid();
    chdir("/");
    (open(STDIN, "< /dev/null") &&
     open(STDOUT, "> /dev/null") &&
     open(STDERR, "> /dev/null")) || do {
       writelog($args,LOG_ERR,__PACKAGE__.
		": Error closing stdin, stdout, or stderr: $!");
       return 0;
     };
  }

  $server_pid=$$;

  # create PID file.
  writelog($args,LOG_DEBUG, __PACKAGE__.
	     ": Creating PID file: $cfg_pidfile");
  my $pidfile=IO::File->new("> $cfg_pidfile");
  unless (defined $pidfile) {
    writelog($args,LOG_ERR, __PACKAGE__.
	     ": Unable to create PID file.");
    return 0;
  }
  else {
    $pidfile->print("$$\n");
    $pidfile->close;
  }
  $0='amavisd';

  # Initialize Milter module
  my %milter_callbacks = 
    (
     'connect' =>    \&connect_callback,
     'helo' =>       \&helo_callback,
     'envfrom' =>    \&envfrom_callback,
     'envrcpt' =>    \&envrcpt_callback,
     'header' =>     \&header_callback,
     'eoh' =>        \&eoh_callback,
     'body' =>       \&body_callback,
     'eom' =>        \&eom_callback,
     'abort' =>      \&abort_callback,
     'close' =>      \&close_callback,
    );

  # Establish socket connection
  if (-S $cfg_milter_socket) {
    writelog($args, LOG_DEBUG, __PACKAGE__.
	     ": Removing socket $cfg_milter_socket.");
    unlink $cfg_milter_socket or do {
    writelog($args, LOG_ERR, __PACKAGE__.
	     ": Could not remove $cfg_milter_socket.");
      return undef;
    }
  }
  my $oldumask = umask $cfg_umask;
  Sendmail::Milter::setconn("local:$cfg_milter_socket") or do {
    writelog($args, LOG_ERR, __PACKAGE__.
	     ": Error setting up connection");
  };
  umask $oldumask;
  # Register callback functions
  Sendmail::Milter::register("AMaViS-ng",\%milter_callbacks, SMFI_CURR_ACTS) or do {
    writelog($args, LOG_ERR, __PACKAGE__.
	     ": Error registering callback functions");
  };

  writelog($args,LOG_DEBUG,__PACKAGE__." initialized.");

  # Safeguard for PMilter
  $running = 0;

  # Return successfully
  return 1;
}

sub cleanup {
  my $self = shift;
  my $args = shift;

  return 1;
}

# Create temp directory. Try creating $prefix[date]-[pid] 
# Return created dir or undef if unsuccessful after 10 tries
sub create_tmp_dir( $ ) {
  my $prefix = shift;
  my $i = 0;
  while (1) {
    my $tmpdir = sprintf("$prefix%.8x-%.4x",time,$$);
    unless (mkpath ($tmpdir, 0, 0770)) {
      if (++$i > 10) {
	return undef;
      }
      else {
	next;
      }
    }
    else {
      return $tmpdir;
    }
  }
}

########################################################################
# Begin of callback routines

sub connect_callback {
  my $ctx = shift;
  my $args = $ctx->getpriv();
  $args = $saved_args unless (keys %$args);

  # Create temp directory
  my $directory = create_tmp_dir("$AMAVIS::cfg_unpack_dir/amavis-unpack-");
  mkdir "$directory/parts", 0777;
  $$args{'directory'} = $directory;

  my $fh=IO::File->new("+>$$args{'directory'}/email.txt") or do {
    writelog($args,LOG_ERR,__PACKAGE__.
	     ": Could not open $$args{'directory'}/email.txt: $!");
    return SMFIS_TEMPFAIL;
  };

  $$args{'filehandle'}=$fh;

  $ctx->setpriv($args);
  return SMFIS_CONTINUE;
}

sub helo_callback {
  my $ctx = shift;
  my $args = $ctx->getpriv();
  $args = $saved_args unless (keys %$args);

  $ctx->setpriv($saved_args);
  return SMFIS_CONTINUE;
}

sub envfrom_callback {
  my $ctx = shift;
  my $args = $ctx->getpriv();
  $args = $saved_args unless (keys %$args);
  $$args{'sender'}=shift;
  $$args{'sender'} = "<>" if (!$$args{'sender'});
  $$args{'message_id'}=$ctx->getsymval('i');
  $ctx->setpriv($args);
  return SMFIS_CONTINUE;
}

sub envrcpt_callback {
  my $ctx = shift;
  my $args = $ctx->getpriv();
  $args = $saved_args unless (keys %$args);
  push @{$$args{'recipients'}}, shift;
  $ctx->setpriv($args);
  return SMFIS_CONTINUE;
}

sub header_callback {
  my $ctx = shift;
  my $args = $ctx->getpriv();
  $args = $saved_args unless (keys %$args);
  my $header= join(': ', @_);
  $$args{'headers'}.=$header;
  my $fh=$$args{'filehandle'};
  $fh->write("$header\n");
  $ctx->setpriv($args);
  return SMFIS_CONTINUE;
}

sub eoh_callback {
  my $ctx = shift;
  my $args = $ctx->getpriv();
  $args = $saved_args unless (keys %$args);
  my $fh=$$args{'filehandle'};
  $fh->write("\n");
  writelog($args,LOG_DEBUG,"Sender: ".$$args{'sender'});
  writelog($args,LOG_DEBUG,"Recipients: ".join (' ',@{$$args{'recipients'}}));
  writelog($args,LOG_DEBUG,"Message ID: ".$$args{'message_id'});
  $ctx->setpriv($args);
  return SMFIS_CONTINUE;
}

sub body_callback {
  my $ctx = shift;
  my $bodypart = shift;
  my $args = $ctx->getpriv();
  $args = $saved_args unless (keys %$args);
  my $fh=$$args{'filehandle'};

  $fh->write($bodypart);
  $ctx->setpriv($args);
  return SMFIS_CONTINUE;
}

sub eom_callback {
  my $ctx = shift;
  my $args = $ctx->getpriv();
  $args = $saved_args unless (keys %$args);
  my $fh=$$args{'filehandle'};
  my $oldsig = $SIG{CHLD};
  $fh->seek(0,0);

  local $SIG{CHLD} = 'DEFAULT';
  AMAVIS->process_message($args);
  $SIG{CHLD} = $oldsig;

  if ($cfg_x_header) {
    $ctx->addheader($cfg_x_header_tag, $cfg_x_header_line);
  }

  my $return = SMFIS_TEMPFAIL;
  foreach ($$args{'status'}) {
    /^accept$/ && ($return=SMFIS_ACCEPT);
    /^drop$/ && ($return=SMFIS_DISCARD);
    /^freeze$/ && ($return=SMFIS_TEMPFAIL);
  }

  $ctx->setpriv($args);
  return $return;
}

sub abort_callback {
  my $ctx = shift;
  my $args = $ctx->getpriv();
  $args = $saved_args unless (keys %$args);
  $ctx->setpriv($args);
  return SMFIS_CONTINUE;
}

sub close_callback {
  my $ctx = shift;
  my $args = $ctx->getpriv();
  $args = $saved_args unless (keys %$args);
  my $oldsig = $SIG{CHLD};
  local $SIG{CHLD} = 'DEFAULT';
  AMAVIS->cleanup($args);
  local $SIG{CHLD} = $oldsig;
  $ctx->setpriv($args);
  return SMFIS_CONTINUE;
}

# End of callback routines
########################################################################}

# Create temp dir and write mail
sub get_directory($) {
  my $self = shift;
  my $args = shift;

  if ($running) {
    return 1;
  }

  writelog($args,LOG_DEBUG, "Waiting for connection.");
  $running = 1;

  $saved_args = $args;
  # Hand off control to Sendmail::Milter
  Sendmail::Milter::main() or do {
    writelog($args, LOG_ERR, __PACKAGE__.": Error setting up Milter");
  };

  # We reach this point only if there was a problem in the accept loop. 
  $$args{'directory'}='END';
  return 0;
}

# Called from within AMAVIS.pm to continue message delivery
sub accept_message( $ ) {
  my $self = shift;
  my $args = shift;
  writelog($args,LOG_INFO, __PACKAGE__.": Accepting message");

  $$args{'status'} = 'accept';

  # Return successfully
  return 1;
}

# Called from within AMAVIS.pm to throw message away
sub drop_message( $ ) {
  my $self = shift;
  my $args = shift;
  writelog($args,LOG_WARNING, __PACKAGE__.": Dropping message");

  $$args{'status'} = 'drop';

  # Return successfully
  return 1;
}

# Called from within AMAVIS.pm to freeze message delivery
sub freeze_message( $ ) {
  my $self = shift;
  my $args = shift;
  writelog($args,LOG_WARNING, __PACKAGE__.": Freezing message");

  $$args{'status'} = 'freeze';

  # Return successfully
  return 1;
}

# Called from Notify::*.pm, i.e. for sending warining messages
sub send_message( $$$ ) {
  my $self = shift;
  my $args = shift;
  my $message = shift;
  my $sender = shift;
  my @recipients = @_;
  writelog($args,LOG_DEBUG, __PACKAGE__.": Sending mail from $sender to ".
	   join(', ',@recipients));

  my @sendmail_args;

  push @sendmail_args, split(/\s+/,$cfg_sendmail_args);
  push @sendmail_args, $sender;
  push @sendmail_args, @recipients;

  writelog($args,LOG_DEBUG, __PACKAGE__.": Running $cfg_sendmail_binary ".
	   join(' ',@sendmail_args));

  open(MAIL, "|-") || exec($cfg_sendmail_binary, @sendmail_args);
  print MAIL $message;
  close(MAIL);

  if ($? != 0) {
    writelog($args,LOG_ERR,__PACKAGE__.
	     ": $cfg_sendmail_binary @sendmail_args exited with ".($?>>8));
    return 0;
  }

  # Return successfully
  return 1;
}

1;
