// ========================================================================
// copyright (C) 1999-2003 by Tobias Erbsland <te@profzone.ch>
// ------------------------------------------------------------------------
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2 of the License, or (at your
// option) any later version.
// ========================================================================

#include "postfixfilter.h"
#include "plugin.h"
#include <ctime>
#include <list>
#include <algorithm>

postfixFilter::postfixFilter() :
		strPrefix( "postfix" ),
		strName( "Postfix Filter" ),
		strVersion( "V0.1" ),
		strAuthor( "Tobias Erbsland <te@profzone.ch>" ),
		strHelp( "Filter for default postfix maillog files." ),
		intern_mail_suffix( "@profzone.ch" )
{
	addParam( "year", "Set the start year of the postfix file.", "Year with 4 digits.", STRING, "now" );
	addParam( "counter", "Set this flag to display the linecounter.", "This is a Flag.", FLAG, "0" );
}

bool postfixFilter::run( void )
{
	long lc = 0;
	logline_t lgln;
	bool linecounter = getFlagParam( "counter" );

	if ( getStringParam( "year" ) == "now" )
	{
		// Set year to the current time.
		time_t now = time( 0 ); // now.
		struct tm tm_now = *localtime( &now ); // as tm struct.
		intYear = tm_now.tm_year; // Get year.
	}
	else
	{ // is the year valid?
		if( getIntParam( "year" ) >= 1900 && getIntParam( "year" ) <= 9999 )
		{
			intYear = getIntParam( "year" ) - 1900;
		}
		else
		{
			std::cerr << "Error: Parameter --postfix-year=" << getStringParam( "year" ) << " is not a valid year!" << std::endl;
			return false; // Abort.
		}
	}

	// read all loglines
	while ( ( *is ) )
	{
		try
		{
			lgln = getLogLine( ( *is ) );
			lc++;
		}
		catch( endOfLog ) { break; }

		if( lgln.command.substr( 0, 7 ) == "postfix" )
		{
			postfix( lgln.time, lgln.message );
		};
		if ( !( lc % 5000 ) && linecounter ) std::cerr << "  line " << lc << " ..." << std::endl;
	}

	// reduce multible delivery attepts with one message id to the last attept.
	std::map<std::string,unsigned long> msgid_to_id;
	std::map<unsigned long,std::string> id_to_lastmsg;
	std::list<std::string> delete_list;
	unsigned long id = 0;

	// build some maps and get the dups
	std::map<std::string, Analyse::unity_t>::iterator i;
	for( i = unity_map.begin(); i != unity_map.end(); i++ )
	{
		if( msgid_to_id.find( i->second.message_id ) == msgid_to_id.end() )
		{
#ifdef DEBUG
			//std::cerr << "DEBUG: assign id " << id << " to <" << i->second.message_id << ">" << std::endl;
#endif
			msgid_to_id[i->second.message_id] = id;
			id_to_lastmsg[id] = i->first;
			id++;
		}
		else
		{
			unsigned long mid = msgid_to_id[i->second.message_id];
			if( i->second.date > unity_map[id_to_lastmsg[mid]].date )
			{
				delete_list.push_back( id_to_lastmsg[mid] );
				id_to_lastmsg[mid] = i->first;
			}
			else
			{
				delete_list.push_back( i->first );
			}
		}
	}

	// cleanup the duplicates
#ifdef DEBUG
	std::cerr << "DEBUG: erasing " << delete_list.size() << " duplicated messages." << std::endl;
#endif
	std::list<std::string>::iterator di;
	for( di = delete_list.begin(); di != delete_list.end(); di++ )
	{
		unity_map.erase( *di );
	}
#ifdef DEBUG
	std::cerr << "DEBUG: keeping " << unity_map.size() << " unique messages." << std::endl;
#endif

	return true; // always ok.
}

int postfixFilter::readTwoChars( std::istream &streamin )
{
	int newInt;

	newInt = ( streamin.get() - '0' ) * 10;
	newInt += streamin.get() - '0';

	return newInt;
}

postfixFilter::logline_t postfixFilter::getLogLine( std::istream &streamin )
{
	logline_t logLine;
	struct tm tmTime;
	std::string strMonth;
	std::string strPid;
	std::string strProcess;
	static char buffer[ 1024 ];

	// Convert Time and Month to an universal format
	if ( !( streamin >> strMonth ) ) throw endOfLog();
	tmTime.tm_mon = strToMonth( strMonth ) - 1;
	if ( !( streamin >> tmTime.tm_mday ) ) throw endOfLog();
	if ( !( streamin.ignore() ) ) throw endOfLog();
	tmTime.tm_hour = readTwoChars( streamin );
	if ( !( streamin.ignore() ) ) throw endOfLog();
	tmTime.tm_min = readTwoChars( streamin );
	if ( !( streamin.ignore() ) ) throw endOfLog();
	tmTime.tm_sec = readTwoChars( streamin );
	// Convert the command and the pid
	if ( !( streamin >> logLine.host ) ) throw endOfLog();
	if ( !( streamin >> strProcess ) ) throw endOfLog();
	logLine.command = strProcess.substr( 0, strProcess.find( "[" ) );
	logLine.pid = atoi( strProcess.substr( strProcess.find( "[" ) + 1, strProcess.find( "[" ) - strProcess.find( "]" ) - 1 ).c_str() );
	tmTime.tm_year = intYear; // Set Year.
	logLine.time = mktime( &tmTime );
	streamin.ignore();
	// Read the until end of line
	if ( !( streamin.getline( buffer, 1024 ) ) ) throw endOfLog();
	logLine.message = buffer;
	return logLine;
}

std::string postfixFilter::translate( const std::string &email )
{
	std::string a = email;
	if( afp->addrFilter( a ) )  // Try change, changed!?
		return a;
	else
		return email;
}

void postfixFilter::postfix( const time_t time, const std::string &line )
{
	// first get the id
	if( line[10] == ':' ) // a line with an id?
	{
		// store the id
		std::string id = line.substr( 0, 10 );

		// Create a empty mailentry.
		Analyse::unity_t me;

		// flag to detect changes
		bool changes = false;

		// search a record with this id, or create a empty one
		if( unity_map.find( id ) == unity_map.end() ) // not found?
		{
			// Add an empty one with the current time
			me.size = 0;
			me.date = time;
			me.message_id = id;
			me.client = "unknown";
			unity_map[id] = me;
		}
		else
		{
			// get the current data instead
			me = unity_map[id];
		}

		// check for a "from" or a "to" line.
		if( line.substr( 12, 6 ) == "from=<" )
		{
			me.from = translate( line.substr( 18, line.find( '>', 18 ) - 18 ) );
			std::string::size_type spos = line.find( "size=" );
			if( spos != std::string::npos ) // not found?
			{
				std::string size = line.substr( spos+5, line.find_first_not_of( "0123456789", spos+5 ) - spos - 5 );
				me.size = atoi( size.c_str() );
			}
			// we made changes
			changes = true;
		}
		else if( line.substr( 12, 4 ) == "to=<" )
		{
			std::string to = translate( line.substr( 16, line.find( '>', 16 ) - 16 ) );
			if( to.find( '@' ) != std::string::npos ) // external?
			{
				if( std::find( me.to_extern.begin(), me.to_extern.end(), to ) == me.to_extern.end() )
				{
					me.to_extern.push_back( to );
				}
			}
			else
			{
				if( std::find( me.to_intern.begin(), me.to_intern.end(), to ) == me.to_intern.end() )
				{
					me.to_intern.push_back( to );
				}
			}

			// we made changes
			changes = true;
		}
		else if( line.substr( 12, 12 ) == "message-id=<" ) // a message id?
		{
			me.message_id = line.substr( 24, line.find( '>', 24 ) - 24 );
			if( me.message_id.size() == 0 )
			{
				me.message_id = id;
			}
			changes = true;
		}
		else if( line.substr( 12, 7 ) == "client=" ) // a client ip?
		{
			std::string::size_type pos = line.find( '[', 19 ) + 1;
			me.client = line.substr( pos, line.find( ']', pos ) - pos );
			changes = true;
		}

		// store changes
		if( changes )
		{
			unity_map[id] = me;
		}
	}
}

