//
// anyRemote
// a bluetooth remote for your PC.
//
// Copyright (C) 2006,2007,2008,2009 Mikhail Fedotov <anyremote@mail.ru>
// 
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
//
 
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <pwd.h>

#include "common.h"
#include "utils.h"
#include "conf.h"


extern char tmp[MAXMAXLEN];
extern CONF       conf;

// Some globals

char logfile [MAXLEN];
static char tofile[MAXLEN];

timerCmd* timers  = NULL;
varData*  vars    = NULL;

char * lastValues    [ID_SETMAX] = {NULL};
int    lastValuesSize[ID_SETMAX] = {-1};

// used by Flush() command to temporary store old configuration
static mode       *flushModes   = NULL;
static type_alias *flushAliases = NULL;
static type_key   *flushAlarms  = NULL;


//////////////////////////////////////////////////////////////////////////////////
//
// Functions related to logging
//
//////////////////////////////////////////////////////////////////////////////////

static void initFile(char* what, const char* name)
{
	// store data in first arg
        
        char *prefix = getenv("AR_TMPDIR");
        
        if (prefix) {
        	strcat(what, prefix);
        } else {
        	char *h = getenv("HOME");
		if (h) {
        		strcat(what, h);
        		strcat(what, "/.anyRemote");
		} else {			// could it ever happen ?
        		strcat(what, "/tmp");
		}
        }
        strcat(what, name);
        char *u = getenv("USER");
        if (prefix && u) {
        	strcat(what, ".");
        	strcat(what, u);
        }        
        return;
}

void initLog()
{
        if (getLog()) {
        
        	initFile(logfile, LOGFILE);
                
                printf("INFO: log file is %s\n",logfile);
                        
                // Just  truncate file
                FILE *fplog = fopen(logfile, "w");
                if (fplog) {
                	fclose(fplog);
                } 
		if(getuid()==0 && conf.uid) { // do not open file as superuser
			chown(logfile,conf.uid,conf.gid);
		}

                printConf();
        }
}

void logger(char *head, char *str)
{
	FILE *fplog;
        time_t logtime;

	if (logfile[0] == '\0') {
        	return;
        }
        
        if (getLog()) {
        	
                if (strcmp(head, "DEBUG") == 0 && getDebug() == 0) {
                    return;
                }

                fplog = fopen(logfile, "a");
                if (fplog!=NULL) {
                	if (strcmp(head, "CFG") != 0) {
                        	time(&logtime);
				struct tm* timeinfo = localtime (&logtime );
				
				char stime[32];
				stime[8] = '\0';
				
				sprintf(stime, "%2d:%2d:%2d", timeinfo->tm_hour, timeinfo->tm_min, timeinfo->tm_sec);
				fprintf(fplog, "[%s] - ",stime);
                        }   
                        if(head!=NULL) {
                        	fprintf(fplog, "%s - ",head);
                        }
                        fprintf(fplog, "%s\n",str);
                
                        fclose(fplog);
                } else {
                	printf("Can't open log file >%s<\n", logfile);
                }
        }
}

void printTime()
{
        time_t logtime;
	time(&logtime);
				
        char *timestr = (char *) ctime(&logtime);
        timestr[strlen(timestr)-1]=0;
  	
	logger("CFG", timestr);
}	

void setResFile()
{
	initFile(tofile, TOFILE);
}

char* getResFile()
{
	return tofile;
}

//////////////////////////////////////////////////////////////////////////////////
//
// Functions related to timer commands
//
//////////////////////////////////////////////////////////////////////////////////

timerCmd* findTimer(char *id, timerCmd **prev) 
{
	if (id == NULL) {
        	return NULL;
        }

	timerCmd* ptr = timers;

        while (ptr != NULL && strncmp(ptr->timer->descr,id,MTEXTLEN) != 0) {
        	if (prev != NULL) {
        		*prev = ptr;
                }
                ptr  = ptr->next;
        }

        return ptr;
}

timerCmd* createTimer(cmdItem *ci)
{
	if (ci->descr == NULL || ci->descr[0] == '\0' || ci->tdata == NULL) {
        	logger("DEBUG", "createTimer(): wrong input");                
                return NULL;
        }

        if (findTimer(ci->descr, NULL) != NULL) {
        	logger("DEBUG", "createTimer(): timer already exists");                
                return NULL;
        } 
        timerCmd *tm = calloc(sizeof(timerCmd),1);

        tm->ticks     = 0;
        tm->times     = 1;
        tm->timer     = ci;
        tm->status    = TIMER_RUN;

        // Insert in head
        if (timers == NULL) { // Insert first
        	tm->next = NULL;
        } else {
        	tm->next = timers;
        }
        timers = tm;

        return tm;
}

int cancelTimer(char *id) 
{
	DEBUG2("cancelTimer() >%s<", id);

        timerCmd *prev = NULL;
        timerCmd *tm = findTimer(id, (timerCmd **) &prev);
        if (tm == NULL) {
            logger("DEBUG", "Cant find timer to cancel!");
            return EXIT_NOK;
        }    
        if (tm == timers) { //remove in head
        	timers = tm->next;
        } else {
        	if (prev != NULL) {
                	prev->next = tm->next;
                } else {
                	logger("ERROR", "Previous item absent for non-first element");
                        return EXIT_NOK;
                }
        }       
         
	tm->timer = NULL;
        free(tm);

        return EXIT_OK;
}

void freeTimers()
{
	logger("DEBUG", "freeTimers()");
        timerCmd *tm;

        while (timers != NULL) {
        	tm     = timers;
                timers = timers->next;
                
        	tm->timer = NULL;
                free(tm);
        }
}

extern int macroCmd(char *macro, cmdParams* params);

int verifyTimers(int ticksInSec) 
{
	timerCmd *tm   = timers;

        while (tm != NULL) {
                if (tm->status == TIMER_RUN) {	// do not process paused timers
                	if (tm->ticks >= (tm->timer->tdata->timeout * ticksInSec)) {
                		DEBUG2("verifyTimers(): it is time to execute >%s<", (tm->timer->exec != NULL ? tm->timer->exec : "empty command"));

				int ret = macroCmd(tm->timer->descr,NULL);
               			if (ret == EXIT_ABORT || ret == EXIT_DISCON) {
                        		return ret;
                        	}
                
                        	tm->ticks = 0;
                
                        	if (tm->timer->tdata->times > 0) {
                        		tm->times++;
                                	if (tm->times >= tm->timer->tdata->times) {    // Cancel this timer 
                                		DEBUG2("verifyTimers(): timer %s self-canceled", tm->timer->descr);
                                
                                        	// Cancel this timer
                                        	timerCmd *tmc = tm;
                                        	tm = tm->next;

                                        	cancelTimer(tmc->timer->descr);
                                
                                        	continue;
                                	}
                        	}
                	} else {
                		tm->ticks++;
                	}
                }
                tm   = tm->next;
        }
	
        return EXIT_OK;
}

//////////////////////////////////////////////////////////////////////////////////
//
// To minimize data transfer cache results for some commands and send data only if data were changed
//
//////////////////////////////////////////////////////////////////////////////////

int isDataNew(int what, char *data, int size)
{
	if (what < 0 || what >= ID_SETMAX) {
 		return EXIT_OK;				// consider as new value
	}
        
        int isNew = EXIT_NOK;
        if (lastValues[what] == NULL) {
        	isNew = EXIT_OK;		     	// consider as new value
        } else if (lastValues[what] != NULL &&
                   (lastValuesSize[what] != size ||
                    memcmp(lastValues[what],data,lastValuesSize[what]) != 0)) {
             
        	free(lastValues[what]);
                lastValues[what]     = NULL;
                lastValuesSize[what] = -1;
                
                isNew = EXIT_OK;		     	// consider as new value
        }
        
        if (isNew == EXIT_OK) {
        	lastValues[what] = calloc(size + 1, 1);
        	memcpy(lastValues[what],data,size);
        	lastValuesSize[what] = size;
        }
        return isNew;
}

void freeCachedData()
{
	int i = 0;
        for (;i<ID_SETMAX;i++) {
        	if (lastValues[i] != NULL) {
                	free(lastValues[i]);lastValues[i] = NULL;
                }
        }
}

//////////////////////////////////////////////////////////////////////////////////
//
// Functions related to internal "variables" handling
//
//////////////////////////////////////////////////////////////////////////////////

varData* searchVar(const char *id, int len) 
{
	
	if (id == NULL || len <= 0) {
        	return NULL;
        }
	strcpy(tmp, "searchVar() >");
	strncat(tmp,id,len);
	strcat(tmp,"<");
	logger("DEBUG", tmp);

	varData* ptr = vars;

        while (ptr != NULL && (strlen(ptr->name) != len || strncmp(ptr->name,id,len) != 0)) {
                ptr  = ptr->next;
        }
        return ptr;
}

static int addVar(const char *name, const char *val, int sz)
{
	DEBUG2("addVar() >%s<", name);
                
        varData * v = (varData *) calloc(sizeof(varData),1);
        v->name  = strdup(name);

	if (sz > 0 && val != NULL) {
		v->value = (char*) calloc(sz,1);
		memcpy((void*)v->value, (const void *) val, sz); // can not user strdup() since val can contains binary data 
        } else {
		v->value = NULL;
	}
        v->size = sz;
	
        // Insert in head
        if (vars == NULL) { 		// Insert first
        	v->next = NULL;
        } else {
        	v->next = vars;
        }
        vars = v;
        
	return EXIT_OK;
}

void addInternalVars() 
{
        char *d = getenv("HOME");
        addVar("Home", d, d ? strlen(d) : 0);	

        d = getCfgDir();
        addVar("CfgDir", d, d ? strlen(d) : 0);	
	
	#ifdef USE_XTEST
        addVar("Xtest", "yes", 3);	
	#else
        addVar("Xtest", "no", 2);	
	#endif
	
	#ifdef USE_BLUEZ
        addVar("Bluez", "yes", 3);	
	#else
        addVar("Bluez", "no", 2);	
	#endif
        
	addVar("MixerCard",    "0",      1);	
        addVar("MixerChannel", "Master", 6);	
}

int setVar(const char *name, const char *val, int sz)
{
        if (name == NULL) {
              return EXIT_NOK;
        }
	DEBUG2("setVar() >%s->%s<", name, (val == NULL ? "NULL" : val));
        
	varData * v = searchVar(name, strlen(name));
        if (v == NULL) {		// Add variable if it not yet defined
        	return addVar(name, val, sz);
        }
	
        // Store new value in already defined variable
	if (v->value != NULL) {
        	free(v->value);v->value = NULL;
	}	 	
        if (sz > 0 && val != NULL) {
		v->value = (char*) calloc(sz,1);
		memcpy((void*)v->value, (const void *) val, sz);
	} else {
		v->value = NULL;
	}
        v->size = sz;

        return EXIT_OK;
}
 
void freeVars()
{
	//logger("DEBUG", "freeVars()");
        varData *vv;

        while (vars != NULL) {
        	vv   = vars;
                vars = vars->next;
                vv->next = NULL;
		 
        	free(vv->name); vv->name  = NULL;
		if (vv->value != NULL) {
        		free(vv->value);vv->value = NULL;
                }
                free(vv);
        }
}

//////////////////////////////////////////////////////////////////////////////////
//
// Functions related to Flush() command
//
//////////////////////////////////////////////////////////////////////////////////

extern mode       *currentMode;
extern mode       *modes;
extern type_alias *aliases;
extern type_key   *alarms;

extern int  flushConf;

void printHelp() 
{
	printf("Usage: anyremote [-f conf.file] [-s connect string] [-b baudrate] [-log] [-n] [-a] \n");
	printf("                 [-fe port] [-http] [-u username] [-name SDP service name]\n");
	//printf("   or: anyremote -h|-v|-getconfdir\n");
	printf("   or: anyremote -h|-v\n\n");
	printf(" -h|--help  print this help\n");
	printf(" -v|--version  print version of anyRemote\n");
	printf(" -f /path/to/file  specify configurational file\n");
	printf(" -s connect string,  possible values\n");
	printf("    bluetooth:<channel> (Server mode - bluetooth connection)\n");
	printf("    socket:<port>       (Server mode - TCP/IP connection)\n");
	printf("    local:/dev/ircommX  (Server mode - IR connection)\n");
	printf("    /dev/rfcomm#        (AT mode - bluetooth connection)\n");
	printf("    /dev/ttyACM#        (AT mode - cable connection)\n");
	printf("    /dev/ircomm#        (AT mode - IR connection)\n");
	printf("    ilirc:<AF_LOCAL socket file> (use with inputlircd)\n");
	printf("    stdin\n");
	printf(" -log	print verbose logging information to $HOME/.anyRemote/anyremote.log\n");
	printf(" -b	baudrate: specify baudrate, default is 19200\n");
	printf(" -n	do not send any AT+CKPD command to the phone\n");
	printf(" -a	reconnect automatically in case of connection failure\n");
	printf(" -fe <port> work as backend for GUI frontend. Use specified port to connect to frontend.\n");
	printf(" -http  tune work with anyremote2html\n");
	printf(" -name <SDP service name> if bluetooth connection is used, allows to specify SDP service name\n");
	printf(" -password ask password on connect\n"); 
	printf("           password should be stored in $HOME/.anyRemote/password file in plain text\n");
	printf(" -u|--user <username> if started from root, allows to set effective user ID to specified user\n\n");

}

//void printCfgDir() 
//{
//	printf("%s\n", getCfgDir()?getCfgDir():"directory with configuration files can not be determined");   
//}

int flushData() 
{
	//logger("DEBUG","flushData");
        
        flushConf = 1;
        
        flushModes = modes;
        modes = NULL;
        
        flushAliases = aliases;
        aliases = NULL;

        flushAlarms = alarms;
        alarms = NULL;
        
        currentMode = NULL;
        
        // Now we ready to load new cfg.
        printConf();
        
	return EXIT_OK;
}

void flushOldConf() {

	freeTimers();
        freeCfgEx(flushModes, flushAlarms, flushAliases);
        flushModes   = NULL;
        flushAlarms  = NULL;
        flushAliases = NULL;
}

int getUidGid(char *username, uid_t *uid, gid_t *gid) {
	/* Set uid and gid to the preferred user (found in setuid.h). Can either be
	* numeric or a string, found in /etc/passwd.	*/
	struct passwd *pw;
	
	if ((pw = getpwnam(username))) {
		// Name exists
		*uid = pw->pw_uid;
		*gid = pw->pw_gid;
		return EXIT_OK;
	}
	/* something Bad happened, so send back an error */
	return EXIT_NOK;
}

char* getStoredPass()
{
	FILE *fp;
	struct stat buf;
	
        char *h = getenv("HOME");
	if (!h) return NULL;
	
	char* resfile = calloc(strlen(h)+21,1);
	
        strcpy(resfile, h);
        strcat(resfile, "/.anyRemote/password");
	
        if(stat(resfile, &buf) == -1) {
	        free(resfile);
		return NULL;
	}
	
	long fLen = buf.st_size;
	DEBUG2("getStoredPass >%ld<", fLen);
	
        fp=fopen(resfile,"r");
	free(resfile);
	
	if (fp == NULL) {
		return NULL;
	}
    
    	char * fBuffer = calloc(fLen+1,1);
	fread(fBuffer, sizeof(char), fLen, fp);
    	fclose(fp);
	
	// strinp \n from the end
	int plen = strlen(fBuffer) - 1;
	while (plen > 0 && *(fBuffer+plen) == '\n') {
		*(fBuffer+plen) = '\0';
		plen--;
	}
	
	return fBuffer;
}


//////////////////////////////////////////////////////////////////////////////////
