//
// anyRemote
// a bluetooth remote for your PC.
//
// Copyright (C) 2006-2011 Mikhail Fedotov <anyremote@mail.ru>
// 
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
//

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <glib.h>

#include "ar_dbus.h"
#include "common.h"
#include "utils.h"
#include "cmds.h"
#include "conf.h"
#include "executor.h"
#include "dispatcher.h"
#include "pr_frontend.h"

extern int   getClip 	(char *string); // from atsend.h

extern char	tmp[MAXMAXLEN];
extern GAsyncQueue *q2exec;
extern GAsyncQueue *q2main;

extern int  gotSignal;
extern gboolean stillRun;

extern int  serverMode;

extern int  useCallId;
extern int  callOn;
extern int  callTimer;
extern char callerId[MAXLEN];


static int  handle_command(char* cmd);
static int  handle_alarm  (char* cmd) ;

static int  isBemusedCommand(char *cmd);
static int  isInpLircCommand(char *cmd);
static int  isLircCommand   (char *cmd);
static int  isIViewerCommand(char *cmd);

static void handleInit();
static void handleConnect();
static void handleDisconnect();
static void handleExit();
static void handleEndcall();

static void cleanAutoRepeatFlag();

int  commandTimer = 0;
int  repeatTimer  = 0;

char modifierString [MAXARGLEN];

int  flushConf    = 0;

void sendToExecutor(eMessage *buf)
{
	if (q2exec == NULL || buf == NULL) {
		return;
	}
	
	g_async_queue_push(q2exec,buf);

	return;	
} 			

void sendEventToExecutor(int event)
{
	eMessage* em = malloc(sizeof(eMessage));
	em->type  = EM_EVENT;
	em->value = malloc(sizeof(int));
	*((int *)em->value) = event;
	
	sendToExecutor(em);
	
	//free(em);
}

gpointer executorRoutine(gpointer thread)
{
	if (q2exec == NULL) {
	        logger("INF","[EX]: Do not start executor thread");
	        return NULL;
	}

	logger("INF","[EX]: Start executor thread");
	int exitFlag = 0;
	
	dbusInit();
	logger("INF","[EX]: DBUS initialized");
	handleInit();
 	logger("INF","[EX]: init hooks done");
               
	logger("INF", "[EX]: Send init OK event to dispatcher");
	dMessage* dm = malloc(sizeof(dMessage));
	dm->size  = 0;
	dm->value = (void*) NULL;
	dm->type  = DM_EVENT_INIT;
	sendToDispatcher(dm);
	
	while (stillRun) {
	  
		//logger("DBG", "[EX]: LOOP");
		
		eMessage *em = (eMessage *) g_async_queue_try_pop(q2exec);
		
		if (em != NULL) {
		        
		        logger("DBG", "[EX]: got event");
			
			if (em->type == EM_KEY) {
			
				char * cmd = (char *) em->value;
			
				sprintf(tmp, "[EX]: got key >%s<", (cmd ? cmd : "NULL"));
				logger("DBG", tmp);
				
				handle_command(cmd);
				free(em->value);

			} else if (em->type == EM_ITEM) {
			
				cmdItem *pg = (cmdItem*) em->value;
			
				logger("DBG", "[EX]: got cmdItem");
			
				processCommands(pg, NULL);

			} else if (em->type == EM_STRING) {
			        
				char * msgIn = (char*) em->value;
		
				sprintf(tmp, "[EX]: got string >%s<", msgIn);
				logger("DBG", tmp);
			
				execDynamically(msgIn);
				free(em->value);
				
			} else if (em->type == EM_EVENT) {
			
				int* evt = (int*) em->value;
				
				sprintf(tmp, "[EX]: got event >%d<", *evt);
				logger("DBG", tmp);
				
				if (*evt == ID_EVT_CONNECT) {
					handleConnect();
				} else if (*evt == ID_EVT_DISCONNECT) {
					handleDisconnect();
				//} else if (*evt == ID_EVT_INIT) {
				//	handleInit();
				} else if (*evt == ID_EVT_ENDCALL) {
					handleEndcall();
				} else if (*evt == ID_EVT_EXIT) {
					handleExit();
					exitFlag = 1;
				}
				free(em->value);
				
			} else if (em->type == EM_ALARM) {	
				
				char * cmd = (char *) em->value;
				
				sprintf(tmp, "[EX]: got alarm >%s<", cmd);
				logger("DBG", tmp);
				
				handle_alarm(cmd);
				 
				free(em->value);

			} else if (em->type == EM_AS_IS) {	
			
				logger("DBG", "[EX]: got as is");
				
 				dMessage* dm = malloc(sizeof(dMessage));
				dm->type  = DM_SET;
				dm->size  = strlen(em->value);
				dm->value = (void*) strdup(em->value);
				sendToDispatcher(dm);
				
				free(em->value);
			
			}
			free(em);
			
			if (exitFlag) {
				return NULL;
			}
		}
		//logger("DBG", "[EX]: command processed of empty input");
		
	    	// Should we auto-repeat command (timeout about 1/10 second) ?
	    	if (repeatNow() && getAutoRepeat()) {
			int isOdd = ((int)repeatTimer/2)*2;
			
			if (repeatTimer > 100) {  // Check for infinite loop. Only 50 autorepeats per one press
				cleanAutoRepeatFlag();
	    		} else if (repeatTimer > 0 && isOdd == repeatTimer) {
	    			DEBUG2("[EX]: Auto repeat command ... (%d)", repeatTimer);

	    			handleCmdByKey(repeatNow(),NULL);
	    	
				repeatTimer++;
	    		} else {
				//logger("DBG","[EX]: Auto repeat command ...Skip");
	    			repeatTimer++;
	    		}
	    	}
		
 		//DEBUG2("[EX]: check timers(%d)", commandTimer);
		// Verify commands executed by timer (timeout about 1 sec)
	    	if (commandTimer == 50) {
	    		//logger("DBG","[EX]: Verify timer commands ...");
	    		verifyTimers(1) ;
	    		commandTimer = 0;
			
	    	} else {
	    		commandTimer++;
	    	}
		
		if (!stillRun) {
			logger("DBG","[EX]: Break from loop ...");
			break;
		}
		
		//logger("DBG", "[EX]: wait a bit");
		usleep(20000); // loop timer (1/50 of second)
	}
	
	return NULL;
}

static void handleInit()
{
	int i = 0;
	type_key *k = findItem(EVT_INIT, &i, NULL);
	if (k && i == FLAG_EXACT) {
		logger("INF", "[EX]: Exec cmd on init");
		handleCmdByKey(k,NULL);
	}
	handleHook(ID_EVT_INIT);
}

static void handleConnect()
{
	int i = 0;
	int ok1 = EXIT_NOK;    
	type_key *k = findItem(EVT_CONNECT, &i, NULL);
	if (k && i == FLAG_EXACT) {
		logger("INF", "[EX]: Exec cmd on connect");
		ok1 = EXIT_OK;

		handleCmdByKey(k,NULL);
	}
	int ok2 = handleHook(ID_EVT_CONNECT);

	if (ok1 == EXIT_OK || ok2 == EXIT_OK) {
		if (getHttp()) {
			
 			dMessage* dm = malloc(sizeof(dMessage));
			dm->type  = DM_SET;
			dm->size  = 6;
			dm->value = (void*) strdup("End();");
			sendToDispatcher(dm);
		}
	}
}

static void handleDisconnect()
{
	int i = 0;
	
        cleanAutoRepeatFlag();
	
	type_key *k = findItem(EVT_DISCONNECT, &i, NULL);
	if (k && i == FLAG_EXACT) {
		logger("INF", "[EX]: Exec cmd on disconnect");
		handleCmdByKey(k,NULL);
	} 
	handleHook(ID_EVT_DISCONNECT);
	
	freeCachedData();
}

static void handleEndcall()
{
	int i = 0;
	type_key *k = findItem(EVT_ENDCALL, &i, NULL);
	if (k != NULL && i == FLAG_EXACT) {
		logger("INF","[EX]: EVT_ENDCALL found");
		handleCmdByKey(k,NULL);
	}
}

static void handleExit()
{
	printf("[EX]: handleExit\n");
	int i;
	type_key *k = findItem(EVT_EXIT, &i, NULL);
	if (k && i == FLAG_EXACT) {
		logger("INF", "[EX]: Exec cmd on exit");
		handleCmdByKey(k,NULL);   // Since we exiting now we did not interested in return code
	}
	printf("[EX]: handleExit handleHook\n");
	handleHook(ID_EVT_EXIT);
	
	printf("[EX]: handleExit free data\n");
	freeVars();
	freeCachedData();
	freeCfg();
	freeRegexps();
	dbusFinish();
	printf("[EX]: handleExit EXIT\n");
}

static void cleanAutoRepeatFlag() 
{
	// Clear auto repeat flag
	logger("DBG","[EX]: Clean auto repeat flag");
	setRepeatNow(NULL);
	repeatTimer  = 0;
}

static int handle_alarm(char* key) 
{
        DEBUG2("[EX]: handle_alarm() >%s<", (key ? key : "NULL"));
	
        if (!key) return EXIT_OK;
	
        int i;
        cmdParams params;
        type_key *k = findItem(key , &i, &params);

        if (k && i == FLAG_EXACT) {
		handleCmdByKeyEx(k, NULL,0);
	
         	if (flushConf == 1) {
                	logger("DBG", "Flush old configuration");
        		flushConf = 0;
                
                	flushOldConf();
        	}
		
        } 
	// else do nothing
	
        return EXIT_OK;
}

static int handle_key_press(char* key,int isRepeatable) 
{
        DEBUG2("[EX]: handle_key_press() >%s<", (key ? key :  "NULL"));
	if (!key) return EXIT_OK;
	
	sendToFrontEnd(key);

        // If modifier was set already it needs to verify which sequence we go
        if (modifierString[0] != '\0') {
        	strcat(modifierString," "); // Add one space first
                strcat(modifierString, key);
                DEBUG2("Modifier+sequence is: >%s<", modifierString);
        }

        // Execute appropriate command
        int i;
        cmdParams params;
        type_key *k = findItem((modifierString[0] == '\0' ? key : modifierString), &i, &params);

        if (k && i == FLAG_EXACT) {
        	if (isRepeatable && (!repeatNow()) && getAutoRepeat()) {
                	logger("DBG", "Set auto repeat flag");
                        setRepeatNow(k);
                 } 
                handleCmdByKey(k, NULL);
                    
                // Clean modifier string
                modifierString[0] = '\0';
                
         	if (flushConf == 1) {
                	logger("DBG", "Flush old configuration");
        		flushConf = 0;
                
                	flushOldConf();
        	}
                
        } else if (k && i == FLAG_MULTIKEY){
        	logger("DBG", "Got part of multi keys sequence. Nothing to do.");
                if (modifierString[0] == '\0') {
                	logger("DBG", "Start of multi key sequence");
                        strcpy(modifierString, key);
                } 

                if(serverMode == CLIENT_RFCOMM || serverMode == CLIENT_AT) {
			sendToMainMenu();
                }
        } else if (k && i == FLAG_PARAMETR){
                logger("DBG", "Got parametrized command");
        
                handleCmdByKey(k, &params);
                
                // Clean modifier string
                modifierString[0] = '\0';
            
        } else {
        	// User had pressed some key ... 
                // Send ToMainMenu sequence of CKPD's to show main menu again
                if (serverMode == CLIENT_RFCOMM || serverMode == CLIENT_AT) {
                	sendToMainMenu();
                }

                logger("DBG", "No approprite key definition was found by findItem()");
                if (modifierString[0] != '\0') {
                	// Clean modifier string
                        modifierString[0] = '\0';
                        logger("DBG", "Clean modifier string");
                }
        }

        return EXIT_OK;
}

static int handle_command(char* cmd) 
{
	logger("DBG","[EX]: handle_command");
	
	if (!cmd) return EXIT_OK;
	
	char *key = NULL;
	
	if (memcmp(cmd, DEF_CLDEBUG, 12) == 0) {	    // Got debug event from Java client

		return EXIT_OK;
                
	} else if (memcmp(cmd, DEF_MSG, 4) == 0) {	    // Got event from Java client
		
		key = cmd+4;
				
		logger("DBG","[EX]: Got event from client");
		return handle_key_press(key,0);

	} else if (memcmp(cmd, DEF_CKEV, 6) == 0) {
		    	
		// Handle different keypresses
		// Motorola: +CKEV: "1",1
		// SE	   : +CKEV: 1,1
		// Sagem: in rare ) cases it could be (between > and <) >,1:< ?

		int isSE = MODEL_SE;

		key = cmd+7;

		// Sagem test
		// key+=3;

		if (*key == '"') {	// Seems this is Motorola
			key++;
			//logger("DBG","[EX]: +CKEV is in Motorola format");
			isSE = MODEL_MOTOROLA;
		}

		if (*key == ',') {	// Seems this is Sagem
			logger("DBG","[EX]: +CKEV is in rare Sagem format (Empty code)");
			isSE = MODEL_SAGEM;
		}

		int i = 1;

		if (isSE == MODEL_SE) { 	// SonyEricsson & default
			while(key[i] != ',') i++;
			key[i] ='\0';
		
			i = i+1;
		} else if (isSE == MODEL_MOTOROLA) {	// Motorola
		    while(key[i] != '"') i++;
		    key[i] ='\0';

		    i = i+2;
		} else if (isSE == MODEL_SAGEM) {	// rare Sagem case
			*key ='\0';
		} else {
			logger("ERR","[EX]: Can't recognize +CKEV event !!!");
			return EXIT_NOK;
		}

		if (key[i] == '1') {
			//logger("DBG","[EX]: Button down event");
			return handle_key_press(key,1);
		} else if (key[i] == '0') {
			//logger("DBG","[EX]: Button up event");
			// In general we skip this
			cleanAutoRepeatFlag();

			return EXIT_OK;
		} else {
			DEBUG2("[EX]: something wrong: key should be pressed or released: cmd=>%s< char=>%c<",cmd, key[i]);
			return EXIT_NOK;
		}
	} else if (memcmp(cmd, DEF_CLCC,6) == 0) {
	
		//logger("INF","[EX]: Caller ID received. Skip it.");
		return EXIT_OK;
		
	} else if (strstr(cmd, DEF_RING) != NULL) {	// Incoming call
		logger("INF","[EX]: Incoming call"); 	// This event sent periodically until used answered a call

		if (callOn == 0) {
			if (useCallId) {
				//logger("DBG", "[EX]: Get CallerId on incoming call");
				int ret = getClip(callerId);

				INFO2("[EX]: CallerId is %s",callerId);
		
				if(ret == EXIT_EXACT) { // Active call exists
				//	logger("DBG","[EX]: Caller ID received");
				//} else { 
					logger("DBG","[EX]: Can't get Caller ID. Probably call was finished");
				}
			}
			//
			// May be we can get +CLIP: "+xxxxxxxxxxxx",145,,,,0< here ?
			// 

			int i = 0;
			type_key* k = findItem(EVT_INCALL, &i, NULL);

			if (k && i == FLAG_EXACT) {
				logger("DBG", "[EX]: Exec cmd on incoming call");
				handleCmdByKey(k,NULL);
			}

			// If there is (EndCall) command,then set timer to determine when call ends
			i = 0;
			k = findItem(EVT_ENDCALL, &i, NULL);
			if (k && i == FLAG_EXACT) {
				logger("DBG", "[EX]: Set call timer");
				callTimer = 0;
				callOn    = 1;
			}
		
			return EXIT_OK;
		} else {
			logger("DBG","[EX]: Skip it."); // This event sent periodically until used answered a call
		}
	} else if (isLircCommand(cmd)) {	    // LIRC tricks
		return EXIT_OK;
	} else if (isInpLircCommand(cmd)) {	    // inputLIRC tricks
		return EXIT_OK;
	} else if (isBemusedCommand(cmd)) {	    // Bemused server emulation tricks
		return EXIT_OK;
	} else if (isIViewerCommand(cmd)) {	    // CommandFusion iViewer protocol support
		return EXIT_OK;
	} else /*if (serverMode == SERVER_STDIN)*/ {    // Just direct input, try it
        	//printf("DEBUG: Is it SERVER_STDIN ? %s\n", cmd);
                
                char *p = cmd;
        	while (*p != '\0') {
                	if (*p == '\n') {
                        	*p = '\0';
                                break;
                        }
                        p++;
                }

		return handle_key_press(cmd,1);
	}

	DEBUG2("[EX]: Unhandled cmd %s", cmd);
	return EXIT_OK;
}

//
// Should be used in AT mode only
//
void sendToMainMenu()
{
	logger("INF","[EX]: sendToMainMenu");
	char *tmm = NULL;
	if ((tmm = getToMainMenu()) != NULL) {
	
		dMessage* dm = malloc(sizeof(dMessage));
		dm->type  = DM_CKPD;
		dm->value = (void*) tmm;
		dm->size  = strlen(dm->value);

		sendToDispatcher(dm);
	}
}

//
// Bemused server limited emulation
// we could get more than one command at once
//
      
static int isBemusedCommand(char *cmdIn)
{  
	DEBUG2("isBemusedCommand: >%s<",cmdIn);

        int isBemused = 0;

        char *cmd = cmdIn;
        char *last = cmdIn + strlen(cmdIn);

        while (cmd < last) {
        	int handle = 0;
                int shift  = 4;

                char oneCmd[16];
		memset(oneCmd,0,16);
		
                char paramValue[8];
                paramValue[0] = '\0';

                if (strncmp(cmd, "CHCK", 4) == 0 ||
                    strncmp(cmd, "DINF", 4) == 0 ||
                    strncmp(cmd, "EXIT", 4) == 0 ||
                    strncmp(cmd, "FADE", 4) == 0 ||
                    strncmp(cmd, "FFWD", 4) == 0 ||
                    strncmp(cmd, "GVOL", 4) == 0 ||
                    strncmp(cmd, "INF2", 4) == 0 ||
                    strncmp(cmd, "INF", 4) == 0 ||
                    strncmp(cmd, "LIST", 4) == 0 ||
                    strncmp(cmd, "NEXT", 4) == 0 ||
                    strncmp(cmd, "PAUS", 4) == 0 ||
                    strncmp(cmd, "PLEN", 4) == 0 ||
                    strncmp(cmd, "PLST", 4) == 0 ||
                    strncmp(cmd, "PREV", 4) == 0 ||
                    strncmp(cmd, "RMAL", 4) == 0 ||
                    strncmp(cmd, "RWND", 4) == 0 ||
                    strncmp(cmd, "SHUT", 4) == 0 ||
                    strncmp(cmd, "STEN", 4) == 0 ||
                    strncmp(cmd, "STOP", 4) == 0 ||
                    strncmp(cmd, "STRT", 4) == 0 ||
                    strncmp(cmd, "VERS", 4) == 0) {
                	handle = 1;
                }

                if (strncmp(cmd, "REPT", 4) == 0 ||
                    strncmp(cmd, "SHFL", 4) == 0) {
                
			// Just "toggle" functionality
                        //unsigned char bc = ((unsigned char) *(cmd + 4));
                        //sprintf(paramValue, "%d",bc);
                
                        handle = 1;
                        shift  = 5;
                }
                if (strncmp(cmd, "VOLM", 4) == 0) {
                	unsigned char bc;
                        
			// this must be handled in dispatcher thread
			if (strlen(cmd) == 4) {  	// read only VOLM without value to set
                        	logger("ERR","got VOLM without value");
				return 1;		// do nothing, but treat it as Bemused command
                        }
			bc = ((unsigned char) *(cmd + 4));

                        // VOLM 0-255 -> 0%-100%
                        sprintf(paramValue, "%d", ((int)bc * 100)/255);
        
                        DEBUG2("isBemusedCommand: VOLM parameter >%s<", paramValue);
                
                        handle = 1;
                        shift  = 5;
                }
                if (strncmp(cmd, "SLCT", 4) == 0) {
		
                	unsigned char bc1 = *((unsigned char*) (cmd + 4));
                	unsigned char bc2 = *((unsigned char*) (cmd + 5));
			
			unsigned int ix = bc1*256+bc2;
                        sprintf(paramValue, "%d",ix);
			
			DEBUG2("isBemusedCommand: SLCT parameter >%s<", paramValue);
                	
			handle = 1;
                        shift  = 6;
                }
                if (strncmp(cmd, "SEEK", 4) == 0)  {    

                	sprintf(paramValue, "%d", 
                                     (((unsigned char)*(cmd + 4)) << 24) + 
                                     (((unsigned char)*(cmd + 5)) << 16) + 
                                     (((unsigned char)*(cmd + 6)) << 8)  + 
                                       (unsigned char)*(cmd + 7));
                
                        handle = 1;
                        shift  = 8;
                }
                if (strncmp(cmd, "DLST", 4) == 0 || 
                    strncmp(cmd, "DOWN", 4) == 0 || 
                    strncmp(cmd, "FINF", 4) == 0 ||    
                    strncmp(cmd, "LADD", 4) == 0 ||
                    strncmp(cmd, "PLAY", 4) == 0) {             
                    
                	unsigned char bc = *((unsigned char*) (cmd + 4));
                        sprintf(paramValue, "%d",bc);
                
                        shift  = 5 + bc;
                        handle = 1;
                }
                if (handle) {
                	isBemused = 1;
                        strncpy(oneCmd,cmd,4); 
                        oneCmd[4] = '\0';
                        if (paramValue[0] != '\0') {
                        	strcat(oneCmd,"(-1,");
                                strcat(oneCmd,paramValue);
                                strcat(oneCmd,")");
                        }
                
                        DEBUG2("isBemusedCommand: one commmand >%s<", oneCmd);
                
                        handle_key_press(oneCmd,1);
                } else {
                	break;
                }

                cmd = cmd + shift;
        }
        return isBemused;
}

//
// LIRC dev/event handling
//

static int isLircCommand(char *cmdIn)
{
	//  should got replay in form "0000000000010184 00 TEXT linux-input-layer"
	//                            "0000000000010192 00 CHANNELUP linux-input-layer"
	DEBUG2("isLircCommand: >%s<",cmdIn);

        char *cmd = strdup(cmdIn);
        char *bufPtr = NULL;
	
	if (strncmp(cmdIn, "000000",6) != 0) {
		free(cmd);
		return 0;
	}
	
        char *token = strtok_r(cmd," ",&bufPtr);
	if (token == NULL) {
		free(cmd);
		return 0;
	}

	token = strtok_r(NULL," ",&bufPtr);
	if (token == NULL) {
		free(cmd);
		return 0;
	} 
	
	token = strtok_r(NULL," ",&bufPtr);
	if (token == NULL) {
		free(cmd);
		return 0;
	} 
	handle_key_press(token,1);

	free(cmd);
        return 1;
}  

//
// inputlirc dev/event handling
//
static int isInpLircCommand(char *cmdIn)
{  
	// should got replay in form "2 0 KEY_1 event6"
	
	DEBUG2("isInputLircCommand: >%s<",cmdIn);

	/*char *f = cmdIn;
	sprintf(tmp, "isLircCommand: DUMP >%c< >%d<",*f,*f);
        logger("DBG",tmp);
	*/
	
        int isLirc = 1;

	char *data = NULL;
        char *cmd = strdup(cmdIn);
        char *bufPtr = NULL;
        char *token = strtok_r(cmd," ",&bufPtr);
	if (token == NULL) {
		free(cmd);
		return 0;
	}
	
	token = strtok_r(NULL," ",&bufPtr);
	if (token == NULL) {
		free(cmd);
		return 0;
	} 

	token = strtok_r(NULL," ",&bufPtr);
	if (token && strncmp(token, "KEY_",4)==0) {
		data = token + 4;
		token = strtok_r(NULL," ",&bufPtr);
		if (token && strncmp(token, "event",5) != 0 ) {
			isLirc = 0;
		}
	} else {
		isLirc = 0;
	}

        if (isLirc) {
        	DEBUG2("isLircCommand: one commmand >%s<", data);
		handle_key_press(data,1);
	}
	 	
	free(cmd);
        return isLirc;
}

//
// Command Fusion iViewer protocol support
//
static int isIViewerCommand(char *cmdIn)
{  
	// should got replay in form
        // aXX=value\3
        // dXX=value\3
        // h=0\3
        // i=1\3
        // lXX=value\3	-- not yet supported
        // m=[portrait|landscape]\3
        // p=value\3
        // sXX=value\3
	
	DEBUG2("isIViewerCommand: >%s<",cmdIn);
        
        char* cmdTmp = calloc(strlen(cmdIn) + 1, sizeof(char));        
        strcpy(cmdTmp,cmdIn);
        
        if (!((cmdTmp[0] == 'a' && isdigit(cmdTmp[1])) ||
              (cmdTmp[0] == 'd' && isdigit(cmdTmp[1])) ||
              (cmdTmp[0] == 'l' && isdigit(cmdTmp[1])) ||
              (cmdTmp[0] == 's' && isdigit(cmdTmp[1])) ||
              (cmdTmp[0] == 'm' && cmdTmp[1] == '=')   ||
              (cmdTmp[0] == 'p' && cmdTmp[1] == '=')   ||
              strcmp(cmdTmp, "h=0") == 0 ||
              strcmp(cmdTmp, "i=1") == 0 
            )) {
		free(cmdTmp);
		return 0;
	} 
        
        char *bufPtr = NULL;
        char *token = strtok_r(cmdTmp,"=",&bufPtr);
	if (token == NULL) {
		free(cmdTmp);
		return 0;
	}
        token = strtok_r(NULL,"=",&bufPtr);

        DEBUG2("isIViewerCommand: OK >%d<", (int) strlen(cmdIn) + 6);

        char* cmd = calloc(strlen(cmdIn) + 6, sizeof(char));
        strcpy(cmd,cmdTmp);
        
        strcat(cmd,"(");
        if (cmdTmp[0] == 'm' || 
	    cmdTmp[0] == 'p' || 
	    cmdTmp[0] == 's') {
	    
        	strcat(cmd,"-1,");
        	if (token != NULL) {
                	strcat(cmd,token);      // token is string
        	}

                strcat(cmd,")");
        } else {
        	if (token != NULL) {
        		strcat(cmd,token);	// token is numeric
        	}
        	strcat(cmd,",)");
        }
	free(cmdTmp);

        DEBUG2("isIViewerCommand: one commmand >%s<", cmd);
	handle_key_press(cmd,1);
	
	logger("INF", "isIViewerCommand: EXIT");	
	free(cmd);
        return 1;
}
