/**
 *  @file    mgparse.h
 *  @ingroup NOsh
 *  @author  Nathan Baker
 *  @brief   Class NOsh parsing methods
 *  @version $Id: mgparse.c,v 2.40 2005/12/07 21:18:37 todd_dolinsky Exp $
 *  @attention
 *  @verbatim
 *
 * APBS -- Adaptive Poisson-Boltzmann Solver
 *
 * Nathan A. Baker (baker@biochem.wustl.edu)
 * Dept. of Biochemistry and Molecular Biophysics
 * Center for Computational Biology
 * Washington University in St. Louis
 *
 * Additional contributing authors listed in the code documentation.
 *
 * Copyright (c) 2002-2005.  Washington University in St. Louis.
 * All Rights Reserved.
 * Portions Copyright (c) 1999-2002.  The Regents of the University of
 * California.  
 * Portions Copyright (c) 1995.  Michael Holst.
 *
 * This file is part of APBS.
 *
 * APBS is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * APBS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with APBS; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 * @endverbatim
 */

#include "apbscfg.h"
#include "apbs/apbs.h"
#include "apbs/mgparm.h"
#include "apbs/nosh.h"
#include "apbs/vstring.h"

VEMBED(rcsid="$Id: mgparse.c,v 2.40 2005/12/07 21:18:37 todd_dolinsky Exp $")
/* File contains some extra stuff that was cluttering up nosh.c */

VPUBLIC int NOsh_parseMG(NOsh *thee, Vio *sock, MGparm_CalcType type) {

    char tok[VMAX_BUFSIZE];
    MGparm *mgparm = VNULL;    
    PBEparm *pbeparm = VNULL;    
    int rc;
    /* Check the arguments */
    if (thee == VNULL) {
        Vnm_print(2, "NOsh:  Got NULL thee!\n");
        return 0;
    }
    if (sock == VNULL) {
        Vnm_print(2, "NOsh:  Got pointer to NULL socket!\n");
        return 0;
    }

    Vnm_print(0, "NOsh_parseMG: Parsing parameters for MG calculation\n");

    /* Construct multigrid and PBE parameter objects to hold the parsed input */
    mgparm  = MGparm_ctor(type);
    if (type == MCT_PAR) {
        mgparm->proc_rank = thee->proc_rank;
        mgparm->proc_size = thee->proc_size;
        mgparm->setrank = 1;
        mgparm->setsize = 1;
    }
    pbeparm = PBEparm_ctor();

    /* Start snarfing tokens from the input stream */
    rc = 1;
    while (Vio_scanf(sock, "%s", tok) == 1) { 

        Vnm_print(0, "NOsh_parseMG:  Parsing %s...\n", tok);

        /* See if it's an END token */
        if (Vstring_strcasecmp(tok, "end") == 0) {
            mgparm->parsed = 1;
            pbeparm->parsed = 1;
            rc = 1;
            break;
        }

        /* Pass the token through a series of parsers */
        rc = PBEparm_parseToken(pbeparm, tok, sock);
        if (rc == -1) {
            Vnm_print(0, "NOsh_parseMG:  parsePBE error!\n");
            break;
        } else if (rc == 0) {
            /* Pass the token to the generic MG parser */
            rc = MGparm_parseToken(mgparm, tok, sock);
            if (rc == -1) { 
               Vnm_print(0, "NOsh_parseMG:  parseMG error!\n");
               break;
            } else if (rc == 0) {
                /* We ran out of parsers! */
                Vnm_print(2, "NOsh:  Unrecognized keyword: %s\n", tok);
                break;
            }
        }
    }

    /* Handle various errors arising in the token-snarfing loop -- these all
     * just result in simple returns right now */
    if (rc == -1) return 0;
    if (rc == 0) return 0;
   
    /* Check the status of the parameter objects */
    if ((!MGparm_check(mgparm)) || (!PBEparm_check(pbeparm))) {
        Vnm_print(2, "NOsh:  MG parameters not set correctly!\n");
        return 0;
    }

    /* Now we're ready to whatever sorts of post-processing operations that are
     * necessary for the various types of calculations */
    if ((type == MCT_MAN) || (type == MCT_DUM)) {
        return NOsh_setupMGMANUAL(thee, mgparm, pbeparm);
    } else if (type == MCT_AUT) { 
        return NOsh_setupMGAUTO(thee, mgparm, pbeparm);
    } else if (type == MCT_PAR) {
        return NOsh_setupMGPARA(thee, mgparm, pbeparm);
    } else {
        Vnm_print(2, "NOsh_parseMG:  Undefined calculation type (%d)!\n",
          type);
        return 0;
    }

    /* Should never get here */
    return 0;

}

VPUBLIC int NOsh_setupMGMANUAL(NOsh *thee, MGparm *mgparm, PBEparm *pbeparm) {

    if (thee == VNULL) {
        Vnm_print(2, "NOsh_setupMGMANUAL:  Got NULL thee!\n");
        return 0;
    }
    if (mgparm == VNULL) {
        Vnm_print(2, "NOsh_setupMGMANUAL:  Got NULL mgparm!\n");
        return 0;
    }
    if (pbeparm == VNULL) {
        Vnm_print(2, "NOsh_setupMGMANUAL:  Got NULL pbeparm!\n");
        return 0;
    }

    /* Set up missing MG parameters */
    if (mgparm->setgrid == 0) {
        mgparm->grid[0] = mgparm->glen[0]/((double)(mgparm->dime[0]-1));
        mgparm->grid[1] = mgparm->glen[1]/((double)(mgparm->dime[1]-1));
        mgparm->grid[2] = mgparm->glen[2]/((double)(mgparm->dime[2]-1));
    }
    if (mgparm->setglen == 0) {
        mgparm->glen[0] = mgparm->grid[0]*((double)(mgparm->dime[0]-1));
        mgparm->glen[1] = mgparm->grid[1]*((double)(mgparm->dime[1]-1));
        mgparm->glen[2] = mgparm->grid[2]*((double)(mgparm->dime[2]-1));
    }


    /* Check to see if he have any room left for this type of
     * calculation, if so: set the calculation type, update the number
     * of calculations of this type, and parse the rest of the section
     */
    if (thee->ncalc >= NOSH_MAXCALC) {
        Vnm_print(2, "NOsh:  Too many calculations in this run!\n");
        Vnm_print(2, "NOsh:  Current max is %d; ignoring this calculation\n",
          NOSH_MAXCALC);
        return 0;
    }

    /* Increment the number of calculations */
    (thee->ncalc)++;

    /* Set the calculation type to multigrid */
    thee->calc[thee->ncalc-1].calctype = NCT_MG;

    /* Associate ELEC statement with the calculation */
    thee->elec2calc[thee->nelec-1] = thee->ncalc-1;

    thee->calc[thee->ncalc-1].mgparm = mgparm;
    thee->calc[thee->ncalc-1].pbeparm = pbeparm;

    return 1;
}

VPUBLIC int NOsh_setupMGAUTO(NOsh *thee, MGparm *mgparm, PBEparm *pbeparm) {
 
    MGparm *tmgparm, *mgparms[NOSH_MAXCALC];
    PBEparm *tpbeparm, *pbeparms[NOSH_MAXCALC];
    int i, j, tnfocus[3], nfocus, nparm;
    double td, redrat[3], cgrid[3], fgrid[3], redfrac;


    if (thee == VNULL) {
        Vnm_print(2, "NOsh_setupMGAUTO:  Got NULL thee!\n");
        return 0;
    }
    if (mgparm == VNULL) {
        Vnm_print(2, "NOsh_setupMGAUTO:  Got NULL mgparm!\n");
        return 0;
    }
    if (pbeparm == VNULL) {
        Vnm_print(2, "NOsh_setupMGAUTO:  Got NULL pbeparm!\n");
        return 0;
    }

    /* Calculate the grid spacing on the coarse and fine levels */
    for (i=0; i<3; i++) {
        cgrid[i] = mgparm->cglen[i]/((double)(mgparm->dime[i]-1));
        fgrid[i] = mgparm->fglen[i]/((double)(mgparm->dime[i]-1));
    }
    Vnm_print(0, "NOsh:  Coarse grid spacing = %g, %g, %g\n",
      cgrid[0], cgrid[1], cgrid[2]);
    Vnm_print(0, "NOsh:  Fine grid spacing = %g, %g, %g\n",
      fgrid[0], fgrid[1], fgrid[2]);

    /* Now calculate the number of focusing levels, never reducing the grid
     * spacing by more than redfrac at each level */
    for (i=0; i<3; i++) {
        if (fgrid[i]/cgrid[i] < VREDFRAC) {
            redfrac = fgrid[i]/cgrid[i];
            td = log(redfrac)/log(VREDFRAC);
            tnfocus[i] = (int)ceil(td) + 1;
        } else tnfocus[i] = 2;
    }
    nfocus = VMAX2(tnfocus[0], tnfocus[1]);
    nfocus = VMAX2(nfocus, tnfocus[2]);

    /* Now set redrat to the actual value by which the grid spacing is
     * reduced at each level of focusing */
    for (i=0; i<3; i++) {
        redrat[i] = VPOW((fgrid[i]/cgrid[i]), 1.0/((double)nfocus-1.0));
    }
    Vnm_print(0, "NOsh:  %d levels of focusing with %g, %g, %g reductions\n",
      nfocus, redrat[0], redrat[1], redrat[2]);

    /* Now that we know how many focusing levels to use, we're ready to set up
     * the parameter objects */
    if (nfocus > NOSH_MAXCALC) {
        Vnm_print(2, "NOsh:  This calculation requires too many levels of \
focusing!\n");
        return 0;
    }
    nparm = nfocus;
    for (i=0; i<nfocus; i++) {
        mgparms[i] = MGparm_ctor(0);
        pbeparms[i] = PBEparm_ctor(0);
    }

    /* Set up the parameter objects */
    for (i=0; i<nfocus; i++) {
        tmgparm = mgparms[i];
        tpbeparm = pbeparms[i];

        /* Most of the parameters are derived from the coarsest level */
        MGparm_copy(tmgparm, mgparm);
        PBEparm_copy(tpbeparm, pbeparm);

        if (i == 0) {
            for (j=0; j<3; j++) {
                tmgparm->partOlapCenterShift[j] = 0;
                tmgparm->grid[j] = cgrid[j];
                tmgparm->glen[j] = mgparm->cglen[j];
            }
        } else {
            for (j=0; j<3; j++) {
                tmgparm->grid[j] = redrat[j]*(mgparms[i-1]->grid[j]);
                tmgparm->glen[j] = redrat[j]*(mgparms[i-1]->glen[j]);
            }
        }

        tmgparm->setgrid = 1;
        tmgparm->setglen = 1;

        /* Finer levels inherit centering method from the finest level */
        if (i == 0) {
            tmgparm->cmeth = mgparm->ccmeth;
            for (j=0; j<3; j++) tmgparm->center[j] = mgparm->ccenter[j];
            tmgparm->centmol = mgparm->ccentmol;
        } else {
            tmgparm->cmeth = mgparm->fcmeth;
            for (j=0; j<3; j++) tmgparm->center[j] = mgparm->fcenter[j];
            tmgparm->centmol = mgparm->fcentmol;
        }

        /* Finer levels have focusing boundary conditions */
        if (i != 0) tpbeparm->bcfl = BCFL_FOCUS;

        /* Only the finest level handles I/O and needs to worry about disjoint
         * partitioning */
        if (i != (nfocus-1)) tpbeparm->numwrite = 0;
        if (tmgparm->type != 2)  {
           Vnm_print(0, "NOsh_setupMGAUTO:  Resetting boundary flags\n");
           for (j=0; j<6; j++) tmgparm->partDisjOwnSide[j] = 0;
           for (j=0; j<3; j++) {
               tmgparm->partDisjCenterShift[j] = 0;
               tmgparm->partDisjLength[j] = tmgparm->glen[j];
           }
        }


        tmgparm->parsed = 1;
    }


    if ((thee->ncalc + nparm) >= NOSH_MAXCALC) {
        Vnm_print(2, "NOsh:  Focusing requires too many multigrid \
electrostatics calculations in this run!\n");
        Vnm_print(2, "NOsh:  Current max is %d; ignoring this \
calculation\n",
          NOSH_MAXCALC);
        return 0;
    }
    for (i=0; i<nparm; i++) {
        thee->calc[thee->ncalc+i].mgparm = mgparms[i];
        thee->calc[thee->ncalc+i].pbeparm = pbeparms[i];
        thee->calc[thee->ncalc+i].calctype = 0;
    }

    /* Setup the map from ELEC statement to actual calculation */
    thee->elec2calc[thee->nelec-1] = thee->ncalc + nparm - 1;
    thee->ncalc += nparm;

    /* Destroy the dummy parameter objects */
    MGparm_dtor(&mgparm);
    PBEparm_dtor(&pbeparm);

    return 1;
}

/* Author:   Nathan Baker and Todd Dolinsky */
VPUBLIC int NOsh_setupMGPARA(NOsh *thee, MGparm *mgparm, PBEparm *pbeparm) {

    int rank, size, npx, npy, npz, nproc, ip, jp, kp;
    double ofrac;
    double hx, hy, hzed;
    double xofrac, yofrac, zofrac;
    int xeffGlob, yeffGlob, zeffGlob, xDisj, yDisj, zDisj;
    int xigminDisj, xigmaxDisj, yigminDisj, yigmaxDisj, zigminDisj, zigmaxDisj;
    int xigminOlap, xigmaxOlap, yigminOlap, yigmaxOlap, zigminOlap, zigmaxOlap;
    int xOlapReg, yOlapReg, zOlapReg;
    double xlenDisj, ylenDisj, zlenDisj;
    double xcentDisj, ycentDisj, zcentDisj;
    double xcentOlap, ycentOlap, zcentOlap;
    double xlenOlap, ylenOlap, zlenOlap;
    double xminOlap, xmaxOlap, yminOlap, ymaxOlap, zminOlap, zmaxOlap;
    double xminDisj, xmaxDisj, yminDisj, ymaxDisj, zminDisj, zmaxDisj;
    double xcent, ycent, zcent;

    /* Grab some useful variables */
    ofrac = mgparm->ofrac;
    rank = thee->proc_rank;
    size = thee->proc_size;
    npx = mgparm->pdime[0];
    npy = mgparm->pdime[1];
    npz = mgparm->pdime[2];
    nproc = npx*npy*npz;

    /* We take a different code path if this is an asynchronous calculation */
    if (mgparm->setasync == 0) {

#ifndef HAVE_MPI_H

        Vnm_tprint(2, "NOsh:  Oops!  You're trying to perform an 'mg-para' \
(parallel) calculation\n");
        Vnm_tprint(2, "NOsh:  with a version of APBS that wasn't compiled with \
MPI -- not good!\n");
        Vnm_tprint(2, "NOsh:  Bailing out!\n");

        return 0;

#endif

        if (thee == VNULL) {
            Vnm_print(2, "NOsh_setupMGPARA:  Got NULL thee!\n");
            return 0;
        }
        if (mgparm == VNULL) {
            Vnm_print(2, "NOsh_setupMGPARA:  Got NULL mgparm!\n");
            return 0;
        }
        if (pbeparm == VNULL) {
            Vnm_print(2, "NOsh_setupMGPARA:  Got NULL pbeparm!\n");
            return 0;
        }
    
        rank = thee->proc_rank;
        size = thee->proc_size;

        Vnm_print(0, "NOsh_setupMGPARA:  Hello from processor %d of %d\n", rank,
                size);

    
        /* Check to see if we have too many processors.  If so, then simply set
         * this processor to duplicating the work of processor 0. */
        if (rank > (nproc-1)) {
            Vnm_print(2, "NOsh_setupMGPARA:  There are more processors available than\
     the %d you requested.\n", nproc);
            Vnm_print(2, "NOsh_setupMGPARA:  Eliminating processor %d\n", rank);
            thee->bogus = 1;
            rank = 0;
        }
    
        /* Check to see if we have too few processors.  If so, this is a fatal
         * error. */
        if (size < nproc) {
            Vnm_print(2, "NOsh_setupMGPARA:  There are too few processors (%d) to \
satisfy requirements (%d)\n", size, nproc);
            return 0;
        }

        Vnm_print(0, "NOsh_setupMGPARA:  Hello from processor %d of %d\n", rank, size);

    } else { /* if (mgparm->setasync == 0) */

        rank = mgparm->async;
        nproc = mgparm->pdime[0]*mgparm->pdime[1]*mgparm->pdime[2];
                
        /* Check to see if the async id is greater than the number of
         * processors.  If so, this is a fatal error. */
        if (rank > (nproc-1)) {
            Vnm_print(2, "NOsh_setupMGPARA:  The processor id you requested (%d) \
is not within the range of processors available (0-%d)\n", rank, (nproc-1));
            return 0;
        }
    }

    /* Grab some useful variables */
    ofrac = mgparm->ofrac;  
    npx = mgparm->pdime[0]; 
    npy = mgparm->pdime[1]; 
    npz = mgparm->pdime[2]; 
    nproc = npx*npy*npz;
 
    /* Calculate the processor's coordinates in the processor grid */
    kp = (int)floor(rank/(npx*npy));
    jp = (int)floor((rank-kp*npx*npy)/npx);
    ip = rank - kp*npx*npy - jp*npx;
    Vnm_print(0, "NOsh_setupMGPARA:  Hello world from PE (%d, %d, %d)\n", 
      ip, jp, kp);

    /* Calculate the global grid size and spacing */

    if (npx == 1) xofrac = 0.0;
    else xofrac = ofrac;
    if (npy == 1) yofrac = 0.0;
    else yofrac = ofrac;
    if (npz == 1) zofrac = 0.0;
    else zofrac = ofrac;

    xDisj = (int)VFLOOR(mgparm->dime[0]/(1 + 2*xofrac) + 0.5);
    xeffGlob = npx*xDisj;
    hx = mgparm->fglen[0]/(double)(xeffGlob-1);
    
    yDisj = (int)VFLOOR(mgparm->dime[1]/(1 + 2*yofrac) + 0.5);
    yeffGlob = npy*yDisj;
    hy = mgparm->fglen[1]/(double)(yeffGlob-1);

    zDisj = (int)VFLOOR(mgparm->dime[2]/(1 + 2*zofrac) + 0.5);
    zeffGlob = npz*zDisj;
    hzed = mgparm->fglen[2]/(double)(zeffGlob-1);

    Vnm_print(0, "NOsh_setupMGPARA:  Global Grid size = (%d, %d, %d)\n",
              xeffGlob, yeffGlob, zeffGlob);
    Vnm_print(0, "NOsh_setupMGPARA:  Global Grid Spacing = (%.3f, %.3f, %.3f)\n",
              hx, hy, hzed);
    Vnm_print(0, "NOsh_setupMGPARA:  Processor Grid Size = (%d, %d, %d)\n",
              xDisj, yDisj, zDisj);

    /* Calculate the maximum and minimum processor grid points */

    xigminDisj = ip*xDisj;
    xigmaxDisj = xigminDisj + xDisj - 1;
    yigminDisj = jp*yDisj;
    yigmaxDisj = yigminDisj + yDisj - 1;
    zigminDisj = kp*zDisj;
    zigmaxDisj = zigminDisj + zDisj - 1;
    
    Vnm_print(0, "NOsh_setupMGPARA:  Min Grid Points for this proc. (%d, %d, %d)\n",
              xigminDisj, yigminDisj, zigminDisj);
    Vnm_print(0, "NOsh_setupMGPARA:  Max Grid Points for this proc. (%d, %d, %d)\n",
              xigmaxDisj, yigmaxDisj, zigmaxDisj);

       
    /* Calculate the disjoint partition length and center displacement */
   
    xminDisj = VMAX2(hx*(xigminDisj-0.5), 0.0);
    xmaxDisj = VMIN2(hx*(xigmaxDisj+0.5), mgparm->fglen[0]);
    xlenDisj = xmaxDisj - xminDisj;
    yminDisj = VMAX2(hy*(yigminDisj-0.5), 0.0);
    ymaxDisj = VMIN2(hy*(yigmaxDisj+0.5), mgparm->fglen[1]);
    ylenDisj = ymaxDisj - yminDisj;
    zminDisj = VMAX2(hzed*(zigminDisj-0.5), 0.0);
    zmaxDisj = VMIN2(hzed*(zigmaxDisj+0.5), mgparm->fglen[2]);
    zlenDisj = zmaxDisj - zminDisj;       

    xcent = 0.5*mgparm->fglen[0];
    ycent = 0.5*mgparm->fglen[1];
    zcent = 0.5*mgparm->fglen[2];
    
    xcentDisj = xminDisj + 0.5*xlenDisj - xcent;
    ycentDisj = yminDisj + 0.5*ylenDisj - ycent;
    zcentDisj = zminDisj + 0.5*zlenDisj - zcent;
    if (VABS(xcentDisj) < VSMALL) xcentDisj = 0.0;
    if (VABS(ycentDisj) < VSMALL) ycentDisj = 0.0;
    if (VABS(zcentDisj) < VSMALL) zcentDisj = 0.0;
                              
    Vnm_print(0, "NOsh_setupMGPARA:  Disj part length = (%g, %g, %g)\n", 
      xlenDisj, ylenDisj, zlenDisj);
    Vnm_print(0, "NOsh_setupMGPARA:  Disj part center displacement = (%g, %g, %g)\n", 
      xcentDisj, ycentDisj, zcentDisj);

    /* Calculate the overlapping partition length and center displacement */
    xOlapReg = 0;
    yOlapReg = 0;
    zOlapReg = 0;
    if (npx != 1) xOlapReg = (int)VFLOOR(xofrac*mgparm->fglen[0]/npx/hx + 0.5) + 1;
    if (npy != 1) yOlapReg = (int)VFLOOR(yofrac*mgparm->fglen[1]/npy/hy + 0.5) + 1;
    if (npz != 1) zOlapReg = (int)VFLOOR(zofrac*mgparm->fglen[2]/npz/hzed + 0.5) + 1;

    Vnm_print(0, "NOsh_setupMGPARA:  No. of Grid Points in Overlap (%d, %d, %d)\n",
              xOlapReg, yOlapReg, zOlapReg);

    if (ip == 0) xigminOlap = 0;
    else if (ip == (npx - 1)) xigminOlap = xeffGlob - mgparm->dime[0];
    else xigminOlap = xigminDisj - xOlapReg;
    xigmaxOlap = xigminOlap + mgparm->dime[0] - 1;

    if (jp == 0) yigminOlap = 0;
    else if (jp == (npy - 1)) yigminOlap = yeffGlob - mgparm->dime[1];
    else yigminOlap = yigminDisj - yOlapReg;
    yigmaxOlap = yigminOlap + mgparm->dime[1] - 1;

    if (kp == 0) zigminOlap = 0;
    else if (kp == (npz - 1)) zigminOlap = zeffGlob - mgparm->dime[2];
    else zigminOlap = zigminDisj - zOlapReg;
    zigmaxOlap = zigminOlap + mgparm->dime[2] - 1;
    
    Vnm_print(0, "NOsh_setupMGPARA:  Min Grid Points with Overlap (%d, %d, %d)\n",
              xigminOlap, yigminOlap, zigminOlap);
    Vnm_print(0, "NOsh_setupMGPARA:  Max Grid Points with Overlap (%d, %d, %d)\n",
              xigmaxOlap, yigmaxOlap, zigmaxOlap);

    xminOlap = hx * xigminOlap;
    xmaxOlap = hx * xigmaxOlap;
    yminOlap = hy * yigminOlap;
    ymaxOlap = hy * yigmaxOlap;
    zminOlap = hzed * zigminOlap;
    zmaxOlap = hzed * zigmaxOlap;

    xlenOlap = xmaxOlap - xminOlap;
    ylenOlap = ymaxOlap - yminOlap;
    zlenOlap = zmaxOlap - zminOlap;

    xcentOlap = (xminOlap + 0.5*xlenOlap) - xcent;
    ycentOlap = (yminOlap + 0.5*ylenOlap) - ycent;
    zcentOlap = (zminOlap + 0.5*zlenOlap) - zcent;
    if (VABS(xcentOlap) < VSMALL) xcentOlap = 0.0;
    if (VABS(ycentOlap) < VSMALL) ycentOlap = 0.0;
    if (VABS(zcentOlap) < VSMALL) zcentOlap = 0.0;
    
    Vnm_print(0, "NOsh_setupMGPARA:  Olap part length = (%g, %g, %g)\n", 
      xlenOlap, ylenOlap, zlenOlap);
    Vnm_print(0, "NOsh_setupMGPARA:  Olap part center displacement = (%g, %g, %g)\n", 
      xcentOlap, ycentOlap, zcentOlap);

  
    /* Calculate the boundary flags:
       Flags are set to 1 when another processor is present along the boundary
       Flags are otherwise set to 0. */
 
    if (ip == 0) mgparm->partDisjOwnSide[VAPBS_LEFT] = 0;
    else mgparm->partDisjOwnSide[VAPBS_LEFT] = 1;
    if (ip == (npx-1)) mgparm->partDisjOwnSide[VAPBS_RIGHT] = 0;
    else mgparm->partDisjOwnSide[VAPBS_RIGHT] = 1;
    if (jp == 0) mgparm->partDisjOwnSide[VAPBS_BACK] = 0;
    else mgparm->partDisjOwnSide[VAPBS_BACK] = 1;
    if (jp == (npy-1)) mgparm->partDisjOwnSide[VAPBS_FRONT] = 0;
    else mgparm->partDisjOwnSide[VAPBS_FRONT] = 1;
    if (kp == 0) mgparm->partDisjOwnSide[VAPBS_DOWN] = 0;
    else mgparm->partDisjOwnSide[VAPBS_DOWN] = 1;
    if (kp == (npz-1)) mgparm->partDisjOwnSide[VAPBS_UP] = 0;
    else mgparm->partDisjOwnSide[VAPBS_UP] = 1;       

    Vnm_print(0, "NOsh_setupMGPARA:  partDisjOwnSide[LEFT] = %d\n", 
      mgparm->partDisjOwnSide[VAPBS_LEFT]);
    Vnm_print(0, "NOsh_setupMGPARA:  partDisjOwnSide[RIGHT] = %d\n", 
      mgparm->partDisjOwnSide[VAPBS_RIGHT]);
    Vnm_print(0, "NOsh_setupMGPARA:  partDisjOwnSide[FRONT] = %d\n", 
      mgparm->partDisjOwnSide[VAPBS_FRONT]);
    Vnm_print(0, "NOsh_setupMGPARA:  partDisjOwnSide[BACK] = %d\n", 
      mgparm->partDisjOwnSide[VAPBS_BACK]);
    Vnm_print(0, "NOsh_setupMGPARA:  partDisjOwnSide[UP] = %d\n", 
      mgparm->partDisjOwnSide[VAPBS_UP]);
    Vnm_print(0, "NOsh_setupMGPARA:  partDisjOwnSide[DOWN] = %d\n", 
      mgparm->partDisjOwnSide[VAPBS_DOWN]);


    /* Set the disjoint partition information */
    mgparm->partDisjCenterShift[0] = xcentDisj;
    mgparm->partDisjCenterShift[1] = ycentDisj;
    mgparm->partDisjCenterShift[2] = zcentDisj;
    mgparm->partDisjLength[0] = xlenDisj;
    mgparm->partDisjLength[1] = ylenDisj;
    mgparm->partDisjLength[2] = zlenDisj;
   
    /* Set the fine mesh parameters to the non-disjoint partition */ 
    mgparm->fglen[0] = xlenOlap;
    mgparm->fglen[1] = ylenOlap;
    mgparm->fglen[2] = zlenOlap;
    mgparm->partOlapLength[0] = xlenOlap;
    mgparm->partOlapLength[1] = ylenOlap;
    mgparm->partOlapLength[2] = zlenOlap;
    mgparm->partOlapCenterShift[0] = xcentOlap;
    mgparm->partOlapCenterShift[1] = ycentOlap;
    mgparm->partOlapCenterShift[2] = zcentOlap;

    return NOsh_setupMGAUTO(thee, mgparm, pbeparm);
}

