
#define LOCAL_DEBUG
#include "debug.h"

#include "config.h"
#include "acfg.h"
#include "dlcon.h"

#ifndef NO_TCP_TUNNING
#include <netinet/tcp.h>
#endif
#include <pthread.h>
#include <errno.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/socket.h>

using namespace MYSTD;


#ifndef AI_NUMERICSERV
#define AI_NUMERICSERV 0
#endif
#ifndef AI_ADDRCONFIG
#define AI_ADDRCONFIG 0
#endif

class CAddrInfo {
public:
	time_t m_nResTime;
	struct addrinfo * m_bestInfo;
	
	CAddrInfo() : m_nResTime(0), m_bestInfo(NULL), m_resolvedInfo(NULL) {};
	bool Resolve(const string & sHostname, const string &sPort, string & sErrorBuf)
	{
		static struct addrinfo hints = {
				 // we provide numbers, no resolution needed; only supported addresses
				AI_NUMERICSERV|AI_ADDRCONFIG,
				PF_UNSPEC,
				SOCK_STREAM, 
				IPPROTO_TCP, 
				0, NULL, NULL, NULL
				};

		ldbg("Resolving " << sHostname);
		if (m_resolvedInfo)
		{
			freeaddrinfo(m_resolvedInfo);
			m_resolvedInfo=NULL;
		}
		// TODO: allow arbitrary ports? Get them from hostname?
		int r=getaddrinfo(sHostname.c_str(), sPort.c_str(), &hints, &m_resolvedInfo);
		
		if (0!=r)
		{
			sErrorBuf="503 ";
			sErrorBuf+=gai_strerror(r);
			return false;
		}
		
		ldbg("got it, " << m_resolvedInfo->ai_canonname);

		for (struct addrinfo * pCur=m_resolvedInfo; pCur; pCur = pCur->ai_next)
		{
			if (pCur->ai_socktype == SOCK_STREAM&& pCur->ai_protocol == IPPROTO_TCP)
			{
				m_bestInfo=pCur;
				m_nResTime=time(NULL);
				return true;
			}
		}
		sErrorBuf="500 DNS resolution error";

		// TODO: remove me from the map
		return false;
	}
	~CAddrInfo() { if(m_resolvedInfo) freeaddrinfo(m_resolvedInfo); }
	
protected:
	struct addrinfo * m_resolvedInfo; // getaddrinfo excrements, to cleanup
private:
	// not to be copied ever
	CAddrInfo(const CAddrInfo&);
	CAddrInfo operator=(const CAddrInfo&);
	
};

typedef SHARED_PTR<CAddrInfo> CAddrInfoPtr;
static map<string, CAddrInfoPtr> mapDnsCache;
static pthread_mutex_t lockDnsCache = PTHREAD_MUTEX_INITIALIZER;


struct tDlJob
{
	bool m_bCheckUpdate;
	UINT m_nSourceId;
	MYSTD::string m_sHost, m_sPath;
	acfg::tHostiVec * m_pHostiVec;
	tDlStoragePtr m_pStorage;
	bool m_bHeadOnly;
	
	tDlJob(bool bCheckUpdate, tDlStoragePtr pFi, const string & sHost,
			const string & sPath, bool bHeadOnly) :
		m_bCheckUpdate(bCheckUpdate),
				m_nSourceId(0), // don't care
				m_sHost(sHost), m_sPath(sPath), m_pHostiVec(NULL),
				m_pStorage(pFi), m_bHeadOnly(bHeadOnly)
	{
	}

	tDlJob(bool bCheckUpdate, tDlStoragePtr pFi, acfg::tHostiVec * pBackends,
			const MYSTD::string & sPath, bool bHeadOnly) :
		m_bCheckUpdate(bCheckUpdate),
				m_nSourceId(-1), // becomes 0 RSN...
				m_sHost(""), m_sPath(sPath), m_pHostiVec(pBackends),
				m_pStorage(pFi), m_bHeadOnly(bHeadOnly)
	{
		ChangePeer();
	}
	
	bool ChangePeer()
	{
		if(!m_pHostiVec)
			return false;
		
		tStrPos nOldPrefixLen(0);
		if(m_nSourceId!=(UINT)-1)
			nOldPrefixLen=m_pHostiVec->at(m_nSourceId).sPath.length();
		
		m_nSourceId++;
		if(m_nSourceId >= m_pHostiVec->size())
			return false;
		m_sHost=m_pHostiVec->at(m_nSourceId).sHost;
		m_sPath.replace(0, nOldPrefixLen, m_pHostiVec->at(m_nSourceId).sPath);
		return true;
	}

};

dlcon::dlcon():
m_conFd(-1),
m_bOrphaned(false),
m_bIsAttached2Fitem(false),
m_bDropData(false),
m_bReconnectASAP(false),
m_nRest(0),
m_nOffset(0),
m_DlState(STATE_GETHEADER)
{
	ldbg("Creating dlcon");
	m_wakepipe[0]=m_wakepipe[1]=-1;
	m_proxy=acfg::proxy_info.sHost.empty() ? NULL : &acfg::proxy_info;
}

void dlcon::Reset()
{
	/*
	if(m_conFd>=0)
		close(m_conFd);
	
	m_conFd=-1;
	*/
	m_bOrphaned=false;
	m_bIsAttached2Fitem=false;
	m_bDropData=false;
	m_bReconnectASAP=false;
	m_nRest=0;
	m_nOffset=0;
	m_DlState=STATE_GETHEADER;
	//m_pCachedFitem.reset(NULL);
}

inline void dlcon::_AddJob(tDlJob* todo)
{
	setLockGuard;

	m_qToReceive.push_back(todo);
	m_qToRequest.push_back(todo);

	if (m_wakepipe[1]>=0)
		write(m_wakepipe[1], "", 1);

}
void dlcon::AddJob(bool bForceFreshnessChecks, tDlStoragePtr m_pItem, 
		acfg::tHostiVec *pBackends, const MYSTD::string & sPatSuffix, bool bHeadOnly)
{
	_AddJob(new tDlJob(bForceFreshnessChecks, m_pItem, pBackends, sPatSuffix, bHeadOnly));	
}
void dlcon::AddJob(bool bForceFreshnessChecks, tDlStoragePtr m_pItem, tHttpUrl hi, bool bHeadOnly)
{
	_AddJob(new tDlJob(bForceFreshnessChecks, m_pItem, hi.sHost, hi.sPath, bHeadOnly));
}

void dlcon::SignalStop()
{
	ldbg("dlcon::SignalStop");
	setLockGuard;
	m_bOrphaned=true;
	write(m_wakepipe[1], "", 1);
}

dlcon::~dlcon()
{
	ldbg("Destroying dlcon");
	if (m_conFd>=0)
	{
		shutdown(m_conFd, O_RDWR);
		close(m_conFd);		
	}
	
	if (m_wakepipe[0]>=0)
		close(m_wakepipe[0]);

	if (m_wakepipe[1]>=0)
		close(m_wakepipe[1]);
	
	if (m_pCachedFitem)
	{
		// shouldn't never happen
		m_pCachedFitem->SetFailureMode("500 Please report apt-cacher-ng bug #1");
		m_pCachedFitem.reset();
	}
	
	list<tDlJob*>::iterator it=m_qToReceive.begin();
	for(;it!=m_qToReceive.end();it++)
		delete *it;
	
	ldbg("destroying dlcon done. Any postponed tasks to do? ");
	/* TODO: periodical cleaning?
	{
		lockguard g(&lockDnsCache);
	}
	*/
}

// PREREQUISITES: current host is failed host and is first, refered by m_recvPos
void dlcon::_CleanupForFailedHost(const string & sErrorMsg)
{

	// kick all jobs that belong that hostname, adapt the position marks
	setLockGuard;
	if (m_bIsAttached2Fitem && !m_qToReceive.empty()) // on the first item, detach us if it's ours
	{
		m_qToReceive.front()->m_pStorage->ReleaseDownloaderUnlessBusy();
		// successfull or not, does not matter anymore?!
		m_bIsAttached2Fitem=false;
	}

	string sBadHost=m_qToReceive.front()->m_sHost;

	m_qToRequest.clear(); // filter the valid items and save in this list, then copy it back

	for (list<tDlJob*>::iterator it = m_qToReceive.begin(); 
	it!=m_qToReceive.end() ; it++)
	{
		if ((*it)->m_sHost == sBadHost)
		{
			(*it)->m_pStorage-> SetFailureMode(sErrorMsg);
			delete *it;
		}
		else
		{
			ldbg("host? " << sBadHost << " Kept job: " << (*it)->m_sHost << "/" << (*it)->m_sPath);
			m_qToRequest.push_back(*it);
		}

	}
	m_qToReceive = m_qToRequest;
	
	ldbg("cleanup one, force reconnection, m_sConnectedHost");
	m_sConnectedHost.clear(); // will trigger the _Connect call 

}

inline bool dlcon::_CleanOneJob()
{ // ONE job failed, EIO, ENOSPC? Whatever, ignore this job and reconnect
	setLockGuard;
	delete m_qToReceive.front();
	m_qToReceive.pop_front();
	//m_qToRequest=m_qToReceive; Connect will do that
	return false;
}

/*!
 * 
 * Process incoming traffic and write it down to disk/downloaders.
 * 
 * @return: 
 *  - true: ok
 *  - false: not ok, reconnect if sErrorOut is empty, threat as fatal error otherwise 
 */
bool dlcon::_HandleIncoming(string & sErrorOut)
{

	#define THROW_INC_ERROR(x) { sErrorOut=x; return false; }

	// closed connection (null-read) or some other failure
	int r = m_InBuf.sysread(m_conFd);
	ldbg("dlcon::_HandleIncoming()~sysread: " <<r);
	if (r <= 0)
		return false;

	Reswitch:
	ldbg("switch: " << m_DlState);
	
	
	switch (m_DlState)
	{

	case (STATE_GETHEADER):
	{
		ldbg("STATE_GETHEADER");

		m_bDropData=false;
		m_nOffset=0;

		dbgline;
		header h;
		if(m_InBuf.size()==0)
			return true;  // will come back
		dbgline;
		int l=h.LoadFromBuf(m_InBuf.rptr(), m_InBuf.size());
		if (0==l)
			return true; // will come back
		dbgline;
		if (0>l)
		{
			THROW_INC_ERROR("500 Invalid header");
		}
		
		ldbg("GOT: " << h.as_string(false));

		m_InBuf.drop(l);
		bool bHeadOnly(false);
		
		{
			setLockGuard;
			if(m_qToReceive.empty()) THROW_INC_ERROR("Peer sends unexpected data");
			
			if ( !m_qToReceive.front()->m_pStorage->AssignDownloaderUnlessConflicting())
			{
				ldbg("Item dl'ed by others or in error state --> drop it, reconnect");
				// cannot use _CleanOneJob, that would deadlock
				m_qToRequest.clear();
				delete m_qToReceive.front();
				m_qToReceive.pop_front();
				// _Reconnect will do: m_qToRequest=m_qToReceive;
				return false;
			}
			m_bIsAttached2Fitem=true;
			m_pCachedFitem=m_qToReceive.front()->m_pStorage;
			bHeadOnly=m_qToReceive.front()->m_bHeadOnly;
		}

		// DEBUG TEST h.frontLine="HTTP/1.1 200 OK";
		
		m_pCachedFitem->AddTransferCount(l);
		
		int code=h.getStatus();
		ldbg("http code: "<< code);
		
		if( 0==strcasecmp(h.get("Connection").c_str(), "close") ||
				0==strcasecmp(h.get("Proxy-Connection").c_str(), "close") )
		{
			ldbg("Peer wants to close connection after request");
			m_bReconnectASAP=true;
		}
		
		//cerr << "Incomming: " << m_pCachedFitem->ToString() << " with code=" << code<<endl << h.as_string(false);
		switch (code)
		{
		case(404):
		case(400):
		case(403):
		{
			m_bDropData=true;
			m_pCachedFitem->SetFailureMode(h.frontLine.substr(9), true);
			if ( 0==strcasecmp(h.get("Transfer-Encoding").c_str(), "chunked"))
			{
				dbgline;
				m_DlState=STATE_GETCHUNKHEAD;
			}
			else
			{
				dbgline;
				m_nRest=atol(h.get("Content-Length").c_str());
				ldbg("cl: " << m_nRest);
				ldbg("he? " << h.as_string(false));
				m_DlState= (m_nRest>0) ? STATE_GETDATA : STATE_FINISHJOB;
			}
			break;
		}
		case (200):
		{
			dbgline;
			if ( 0==strcasecmp(h.get("Transfer-Encoding").c_str(), "chunked"))
			{
				dbgline;
				m_DlState=STATE_GETCHUNKHEAD;
			}
			else
			{
				dbgline;
				string cl=h.get("Content-Length");
				if (cl.empty())
					THROW_INC_ERROR("500 Missing Content-Length"); // may support such endless stuff in the future but that's too unreliable for now
				m_nRest=atol(cl.c_str());
				m_DlState=STATE_GETDATA;
			}

			break;
		}
		case (206):
		{
			long nCachelen, contlen;
			bool dummy;
			m_pCachedFitem->GetStatusEx(nCachelen, contlen, dummy);
			if(nCachelen<=0)
				THROW_INC_ERROR("500 Unexpected partial content");
			/*
			 * Range: bytes=453291-
			 * ...
			 * Content-Length: 7271829
			 * Content-Range: bytes 453291-7725119/7725120
			 */
			string cr=h.get("Content-Range");
			
			if(cr.empty())
				THROW_INC_ERROR("500 Missing Content-Range in Partial Response");
			
			long myfrom, myto, mylen;
			int n=sscanf(cr.c_str(), "bytes %ld-%ld/%ld", &myfrom, &myto, &mylen);
			ldbg("resuming? n: "<< n << " und myfrom: " <<myfrom << "und nCacheLen: "<< nCachelen <<
								" und myto: " << myto << " und mylen: " << mylen);
						
			if(n!=3 || myfrom!=nCachelen-1 || myfrom>myto || myto!=mylen-1)
				THROW_INC_ERROR("500 Bad or unsupported content range");
			
			m_nOffset=myfrom; // FIXME: maybe allow to begin earlier in the check above?!
			m_nRest=mylen-myfrom;
			ldbg("toget, offset: " << m_nOffset << " und rest: " << m_nRest);
			h.frontLine="HTTP/1.1 200 OK";
			h.set("Content-Length", mylen);
			h.del("Content-Range");
			
			m_DlState=STATE_GETDATA;
			break;
		}
		/* TODO: rethink handling. Currently cannot receive a legal 304, at least one byte must be
		 * returned in a 2xx response... 
		case(304):
		{
			if ( !h.get("Content-Length").empty() )
				THROW_INC_ERROR("500 Data body seen, response was Not Modified");
			m_pCachedFitem->SetFailureMode("");
			goto Reswitch;
		}
		*/
		default: // everything else is critical for apt and should not happen when the server is OK
		{
			// TODO: delete the files then?
			THROW_INC_ERROR(h.frontLine.substr(9));
		}
		} // switch
		
		// save the header for all jobs that need to store data, or for explicite HEAD requests
		if ( bHeadOnly || !m_bDropData)
		{
			dbgline;
			h.set("X-Original-Source", 
					string("http://", 7)
					+m_qToReceive.front()->m_sHost
					+m_qToReceive.front()->m_sPath);
			
			if ( !m_pCachedFitem->StoreNewHead(h) )
				return _CleanOneJob();
		}
		
		// if the server is confused and still sends the contents, there will be a litte confusion first 
		// and then the reconnect will be triggered
		if(bHeadOnly)
		{
			dbgline;
			m_DlState=STATE_FINISHJOB;
		}
		
		dbgline;
		goto Reswitch;

	}
	case (STATE_GETDATA_CHUNKED): // fall through, just send it back to header parser hereafter
		ldbg("STATE_GETDATA_CHUNKED (to STATE_GETDATA)");
	case (STATE_GETDATA):
	{
		ldbg("STATE_GETDATA");
		
		while(true)
		{
			off_t nToStore = min((off_t)m_InBuf.size(), m_nRest);
			ldbg("To store: " <<nToStore);
			if(nToStore==0)
				break;
			if (m_bDropData)
			{
				ldbg("drop as much as it would do when storing for real");
				m_nRest-=nToStore;
				//ldbg("dropping data: " << m_InBuf.c_str());
				m_InBuf.drop(nToStore);
				m_pCachedFitem->AddTransferCount(nToStore);
				break;
			}
			long nStored=m_pCachedFitem->StoreData(m_InBuf.rptr(), nToStore, m_nOffset);
			if(nStored<0)
				return _CleanOneJob();
			dbgline;
			m_nOffset+=nStored;
			m_nRest-=nStored;
			m_InBuf.drop(nStored);
			m_pCachedFitem->AddTransferCount(nStored);
			dbgline;
		}
		
			
		ldbg("Rest: " << m_nRest );
		
		if (m_nRest==0)
		{
			m_DlState = (STATE_GETDATA==m_DlState) ? STATE_FINISHJOB : STATE_GETCHUNKHEAD;
		}
		else
			return true; // will come back

		goto Reswitch;
	}
	case (STATE_FINISHJOB):
	{
		ldbg("STATE_FINISHJOB");
		m_DlState=STATE_GETHEADER;
		m_pCachedFitem->Finalize(m_nOffset);
		m_pCachedFitem.reset();

		lock();
		m_bIsAttached2Fitem=false;
		delete m_qToReceive.front();
		m_qToReceive.pop_front();
		unlock();
		if(m_bReconnectASAP)
			return false; // reconnect, don't risk to hang here

		goto Reswitch;
	}
	case (STATE_GETCHUNKHEAD):
	{
		ldbg("STATE_GETCHUNKHEAD");
		char *p=m_InBuf.c_str();
		char *e=strstr(p, "\r\n");
		if(e==p)
		{ // came back from reading, drop remaining junk? 
			m_InBuf.drop(2);
			p+=2;
			e=strstr(p, "\r\n");
		}
		dbgline;
		if(!e)
		{
			m_InBuf.move();
			return true; // get more data
		}
		unsigned int len(0);
		int n = sscanf(p, "%x", &len); 
		ldbg("parsed " << n <<" numbers, value: "<<len);
		
		long nCheadSize=e-p+2;
		if(n==1 && len>0)
		{
			ldbg("ok, skip " << nCheadSize <<" bytes, " <<p);
			m_InBuf.drop(nCheadSize);
			m_nRest=len;
			m_DlState=STATE_GETDATA_CHUNKED;
		}
		else if(n==1)
		{
			// skip the additional \r\n of the null-sized part here as well
			ldbg("looks like the end, but needs to get everything into buffer to change the state reliably");
			if( m_InBuf.size() < nCheadSize+2 )
			{
				m_InBuf.move();
				return true;
			}
			ldbg("is the termination ok?");
			if( ! (e[2]=='\r' && e[3]=='\n'))
			{
				aclog::err(m_pCachedFitem->ToString()+" -- error in chunk format detected");
				return false;
			}
			
			m_InBuf.drop(nCheadSize+2);
			m_DlState=STATE_FINISHJOB;
		}
		else
			return false; // that's bad...
		goto Reswitch;
		break;
	}
	
	}
	return true;
}

void dlcon::_PrepareRequests()
{

	setLockGuard;
	while ( !m_qToRequest.empty())
	//if( !m_qToRequest.empty())
	{
		if (m_bOrphaned)
			return;
		
		/* Limit pipeline
		if(m_qToReceive.size()-m_qToRequest.size()>2)
			return;
*/
		tDlJob* j=m_qToRequest.front();
		m_qToRequest.pop_front();
		
		string head(j->m_bHeadOnly ? "HEAD " : "GET ");
		
		if(m_proxy)
		{
			head+="http://";
			head+=j->m_sHost;
		}
		head+= j->m_sPath +" HTTP/1.1\r\nHost: ";
		head+= j->m_sHost+"\r\n";//Accept: */*\r\n";
		head+= "Connection: keep-alive\r\n";
		if(m_proxy) // add auth (in sPath) if possible and other stuff
			head+=m_proxy->sPath+"Proxy-Connection: keep-alive\r\n";

		bool bDynType(false);
		long nCachedLen(-1), nContLen(-1);
		j->m_pStorage->GetStatusEx(nCachedLen, nContLen, bDynType);

		bool bSetRange(false), bSetIfRange(false);
#define IsIncomplete (nCachedLen>0 && nContLen != nCachedLen)

		ldbg("Is ifile? " << bDynType << ", is InIomplete? " << IsIncomplete);

		bool bSupportPartial=true;
		if(j->m_pStorage->ToString().find("Translation")!=stmiss)
			bSupportPartial=false;
		
		if (bSupportPartial && nCachedLen>0)
		{
			string mdate;
			if (bDynType )
			{
				header h;
				j->m_pStorage->GetHeader(h);
				mdate=h.get("Last-Modified");

				if ( !mdate.empty() )
				{
					bSetIfRange=true;

					//rtype= IsIncomplete ? RANGE_FILE : RANGE_PROBE;
					bSetRange=true;
				}
				// else no date available, cannot rely on anything
			}
			else
			{ // static file type, date does not matter
				//rtype= IsIncomplete ? RANGE_FILE : RANGE_NONE;
				if (IsIncomplete)
					bSetRange=true;
			}

			if (bSetRange)
			{
				/* use APT's trick - set the starting position one byte lower - this way the server has to
				 * send at least one byte if the assumed position is correct, and we never get a
				 * 416 error (one byte waste is acceptable). */
				nCachedLen--;
				char buf[50];
				sprintf(buf, "Range: bytes=%ld-\r\n", nCachedLen);
				head+=buf;
			}
			if (bSetIfRange)
			{
				head += "If-Range: ";
				head += mdate+"\r\n";
			}

		}
		// Debian Apt-Cacher/" ACVERSION 
		head+="User-Agent: " + acfg::agentname + "\r\n\r\n";
		m_sSendBuf+=head;
		ldbg("Request cooked: " << head);
		//cerr << "To request: " << j->m_sPath<<endl; 
	}
}

bool dlcon::_HotDetach()
{
	ldbg("Hot detach");
	if (!m_bIsAttached2Fitem)
		return true;
	ldbg("was attached, detaching...");
	if (m_qToReceive.front()->m_pStorage->ReleaseDownloaderUnlessBusy())
	{
		m_pCachedFitem.reset();
		m_bIsAttached2Fitem=false;
		return true;
		ldbg("...ok");
	}
	ldbg(" BUSY!");
	return false;
}

#define HANDLEERROR(x) { m_sErrorMsgBuf=x ; goto ERROR_SEE_BUF; } 
void dlcon::WorkLoop(bool bSingleRun)
{

	fd_set rfds, wfds;

	if (m_wakepipe[0]<0)
	{
		if (0==pipe(m_wakepipe))
		{
			set_nb(m_wakepipe[0]);
			set_nb(m_wakepipe[1]);
		}
		else
		{
			m_sErrorMsgBuf="500 Unable to create file descriptors";
			return;
		}
	}

	if (!m_InBuf.init(acfg::dlbufsize))
		m_sErrorMsgBuf="500 Out of memory";

	while (true)
	{
		// some flags temporarily stored to minimize locking
		
		string sReconnHost; // if non-empty -> reconnect RSN, otherwise ignore
		bool bExpectIncomming(false), bExpectOutgoing(false)/*, bGenRequests(false)*/;
		int maxfd, r;

		if (!m_sErrorMsgBuf.empty())
		{
			goto ERROR_SEE_BUF;
		}

		{ // critical decissions about (re)connection control, loop exit 
			
			setLockGuard;

			// if orphaned by our parent client connection, try to exit 
			// when not busy or can be detached from running job/fileitem
			if ( (m_bOrphaned && (m_qToReceive.empty() || _HotDetach()))
					// ... or the dler shall stop after one job and now it's time 
			|| (bSingleRun && m_qToReceive.empty())	)
			{
				ldbg("Stoping download loop");
				return;
			}

			bExpectIncomming = !m_qToReceive.empty();
			
			if (bExpectIncomming)
			{
				ldbg("has incoming jobs");
				sReconnHost=m_qToReceive.front()->m_sHost;
				if (m_qToReceive.front()->m_sHost == m_sConnectedHost)
				{
					ldbg("already connected, don't reconnect");
					sReconnHost.clear();
				}
			}
		}

		if ( ! sReconnHost.empty())
		{
			if( ! _Connect(sReconnHost) ) // sets error msg on failures
			{
				ldbg("Connection failure, m_sConnectedHost");
				m_sConnectedHost.clear(); // be sure about that
				
				lockguard g(&__mutex);	
				if(m_qToReceive.front()->ChangePeer())
					m_sErrorMsgBuf.clear(); // there is another chance, forget the failure
				
				continue;
			}
		}

		FD_ZERO(&rfds);
		FD_ZERO(&wfds);

		//if(bGenRequests) // call always since state might be changed in connect
		_PrepareRequests();

		if (m_conFd>=0)
		{
			if (bExpectIncomming)
			{
				FD_SET(m_conFd, &rfds);
			}

			if ( !m_sSendBuf.empty() )
			{
				ldbg("ExpectOutgoing: 1");
				FD_SET(m_conFd, &wfds);
				bExpectOutgoing=true;
			}
		}

		FD_SET(m_wakepipe[0], &rfds);
		maxfd=MYSTD::max(m_wakepipe[0], m_conFd);

		ldbg("select dlcon");
		struct timeval tv;
		tv.tv_sec = 60;
		tv.tv_usec = 0;
		r=select(maxfd+1, &rfds, &wfds, NULL, &tv);
		if (r<0)
		{
			if (EINTR==errno)
				continue;
			ldbg("FAILURE: select, errno: " << errno);
			HANDLEERROR("500 Internal malfunction, code 100");
		}
		else if(r==0)
		{
			ldbg("Select timeout for " << m_sConnectedHost << ". Trigger reconnection..., m_sConnectedHost.clear()");
			m_sConnectedHost.clear();
			continue;
		}

		if (FD_ISSET(m_wakepipe[0], &rfds))
		{
			int tmp;
			while (read(m_wakepipe[0], &tmp, 1) > 0)
				;
			setLockGuard;
			if(m_bOrphaned)
				continue; // recheck abort conditions ASAP
		}

		if (m_conFd<0)
		{
			ldbg("connection FD not open");
			continue;
		}

		// now do the real IO

		// do read first - might reconnect on reading

		if (FD_ISSET(m_conFd, &rfds))
		{
			ldbg("Receiving data...");
			if (_HandleIncoming(m_sErrorMsgBuf) )
			{
				ldbg("data received, OK");
			}
			else if(m_sErrorMsgBuf.empty())
			{
				ldbg("Non-fatal return (timeout?). Trigger reconnection..., m_sConnectedHost");
				m_sConnectedHost.clear();
				continue;
			}
			else
			{  // bad things, any backup servers?
				lockguard g(&__mutex);
				if( !m_qToReceive.empty() && m_qToReceive.front()->ChangePeer())
				{
					ldbg("Peer address changed, force reconnection, m_sConnectedHost");
					m_sConnectedHost.clear();
					continue;
				}
				goto ERROR_SEE_BUF;
			}
		}

		if (FD_ISSET(m_conFd, &wfds))
		{
			ldbg("Sending data...\n" << m_sSendBuf);
			//int s=send(m_conFd, m_sSendBuf.data(), m_sSendBuf.length(), 0);
			//int s=sendto(m_conFd, m_sSendBuf.data(), m_sSendBuf.length(), 0, NULL, 0);
			int s=write(m_conFd, m_sSendBuf.data(), m_sSendBuf.length());
			//cerr << " Sent bytes: " << s <<endl;
			if (s<0&& errno!=EAGAIN)HANDLEERROR("500 Peer communication, code 101");
			m_sSendBuf.erase(0, s);
		}

		continue;

		ERROR_SEE_BUF:
		// fatal error for that server
		ldbg("Got exception: "<<m_sErrorMsgBuf);
		_CleanupForFailedHost(m_sErrorMsgBuf.c_str()); // ... and drop the fitems for the dead server
		m_sErrorMsgBuf.clear();

	}
}

CAddrInfoPtr _Resolve(const string & sHostname, const string &sPort, string &sErrorMsgBuf)
{
	CAddrInfoPtr p;
	time_t timeExpired=time(NULL)-acfg::dnscachetime;

	map<string, CAddrInfoPtr>::iterator it;
	{
		lockguard g(&lockDnsCache);
		it=mapDnsCache.find(sHostname);
		if (it != mapDnsCache.end())
		{
			if (it->second->m_nResTime > timeExpired)
				return it->second; // fresh enough
			p=it->second;
		}
	}

	bool bAddToCache(false); // don't readd, optimization
	if (!p)
	{
		p.reset(new CAddrInfo);
		bAddToCache=true;
	}

	if (p->Resolve(sHostname, sPort, sErrorMsgBuf))
	{
		if (bAddToCache)
		{
			lockguard g(&lockDnsCache);
			if (mapDnsCache.size()>300) // DOS attack?
				mapDnsCache.clear();
			mapDnsCache[sHostname]=p;
		}
	}
	else // error, return empty ptr
	{
		// better log here, often APT does not display stupid user errors
		aclog::err(string("Error resolving ")+sHostname+": " +sErrorMsgBuf);
		p.reset();
	}

	return p;
}



bool dlcon::_Connect(const string & sHostname)
{
	CAddrInfoPtr dns = m_proxy ? 
			_Resolve(m_proxy->sHost, m_proxy->sPort, m_sErrorMsgBuf) 
			: _Resolve(sHostname, acfg::remoteport, m_sErrorMsgBuf);
	if(!dns)
		return false;
	
	struct addrinfo *pInfo(dns->m_bestInfo);
	
	signal(SIGPIPE, SIG_IGN);
	if (m_conFd>=0)
	{
		ldbg("Socket was connected before, shutting down, m_sConnectedHost");
		shutdown(m_conFd, O_RDWR);
		close(m_conFd);
		m_sConnectedHost.clear();
	}
	ldbg("Creating socket...");
	m_conFd = socket(pInfo->ai_family, pInfo->ai_socktype, pInfo->ai_protocol);
	
#ifndef NO_TCP_TUNNING
	int dummy(1);
    setsockopt(m_conFd,SOL_SOCKET,SO_REUSEADDR,&dummy,sizeof(dummy));
    setsockopt(m_conFd, IPPROTO_TCP, TCP_NODELAY, &dummy, sizeof(dummy));
#endif
    
	int r=connect(m_conFd, pInfo->ai_addr, pInfo->ai_addrlen);
	if (r<0)
	{
		m_sErrorMsgBuf="500 Connection failure";
		ldbg("m_sConnectedHost: Force reconnect, con. failure");
		m_sConnectedHost.clear();
		return false;
	}
	
	ldbg("connect() ok");
	// ready for polling with select now
	set_nb(m_conFd);
	m_sConnectedHost=sHostname;
	
	m_InBuf.clear();
	m_sSendBuf.clear();
	lock();
	m_qToRequest = m_qToReceive; // needs to resend the requests now
	ldbg("requests to resend: "<<m_qToRequest.size());
	unlock();
	m_DlState=STATE_GETHEADER;
	return true;
}

/*
void dlcon::lock(){
	ldbg("locking");
	lockable::lock();
	ldbg("locked");
}

void dlcon::unlock() {
	ldbg("unlocking");
	lockable::unlock();
	ldbg("unlocked");
}
*/

