#! /usr/bin/perl

# this are hook methods overload the hooks in apt-cacher-lib.pl and implement
# data checksumming methods

use strict;
no strict 'subs';
use warnings;
no warnings 'redefine';

use Digest::MD5;
use Fcntl qw(:DEFAULT :flock);

our ($cfg);

BEGIN {
    if (eval {require BerkeleyDB}) {
	import BerkeleyDB;
    }
    else {
	print "Checksum disabled as BerkeleyDB not found. Install libberkeleydb-perl\n";
	$cfg->{checksum}=0;
    }
}

sub sig_handler {
    warn "Got SIG@_. Exiting gracefully!\n" if $cfg->{debug};
    exit 1;
}

sub db {
    my $dbfile="$cfg->{cache_dir}/sums.db";
    debug_message('Init checksum database') if defined &debug_message;
    # Need to handle non-catastrophic signals so that END blocks get executed
    for ('INT', 'TERM', 'PIPE', 'QUIT', 'HUP', 'SEGV') {
	$SIG{$_} = \&sig_handler unless $SIG{$_};
    }

    my @envargs = (
		   -Home   => $cfg->{cache_dir},
		   -Flags => DB_CREATE | DB_INIT_MPOOL | DB_INIT_CDB,
		   -ThreadCount => 64
		  );

    my $logfile;
    open($logfile, ">>$cfg->{logdir}/db.log") && push (@envargs, (-ErrFile => $logfile,
								  -ErrPrefix => "[$$]"));
    debug_message('Create environment') if defined &debug_message;

    # Serialise enviroment handling
    sysopen(my $envlock, "$cfg->{cache_dir}/private/dbenvlock", O_RDONLY|O_CREAT) ||
      die "Unable to open DB environment lockfile: $!\n";
    flock($envlock, LOCK_EX)|| die "Unable to lock DB environment: $!\n";

    my $env = new BerkeleyDB::Env (@envargs);
    unless ($env) {
	warn "Failed to create DB environment: $BerkeleyDB::Error. Attempting recovery...\n";
	&db_recover;
	$env = new BerkeleyDB::Env (@envargs);
    }
    die "Unable to create DB environment: $BerkeleyDB::Error\n" unless $env;

    $env->set_isalive;
    &failchk($env);

    # Take shared lock. This protects verify which requests LOCK_EX
    &db_flock(LOCK_SH)|| die "Shared lock failed: $!\n";

    flock($envlock, LOCK_UN)||die "Unable to unlock DB environment: $!\n";
    close($envlock);

    debug_message('Tie database') if defined &debug_message;
    my $dbh = new BerkeleyDB::Btree
      -Filename => $dbfile,
	-Flags => DB_CREATE,
	  -Env => $env
	    or die "Unable to open DB file, $dbfile $BerkeleyDB::Error\n";

    return \$dbh;
}

# Arg is DB handle
# Arg is not undef for DB_WRITECURSOR
sub get_cursor {
    my $dbh=$_[0];
    my $write=$_[1];
    my $cursor = $dbh->db_cursor($write?DB_WRITECURSOR:undef) or die $!;
    return $cursor;
}

# Arg is cursor
# Arg is key reference
# Arg is data reference
sub cursor_next {
    my $cursor = $_[0];
    my $keyref = $_[1];
    my $dataref = $_[2];
    return $cursor->c_get($$keyref, $$dataref, DB_NEXT)
}

# Arg is the environment object
sub failchk {
    my $e = $_[0];
#    warn "$$ failchk on $e\n";
    if ($e->failchk == DB_RUNRECOVERY) {
	warn "Failed thread detected. Running database recovery\n";
	&db_recover;
    }
}

# Arg is flock flags
my $dblock;
sub db_flock {
    if (!$dblock){
	sysopen($dblock, "$cfg->{cache_dir}/private/dblock", O_RDONLY|O_CREAT) ||
	  die "Unable to open lockfile: $!\n";
    }
    return flock($dblock, $_[0]);
}

sub db_recover {
    &env_remove;
    my @envargs = (
		   -Home   => $cfg->{cache_dir},
		   -Flags  => DB_CREATE | DB_INIT_LOG |
		   DB_INIT_MPOOL | DB_INIT_TXN |
		   DB_RECOVER | DB_PRIVATE | DB_USE_ENVIRON
		  );
    # Cannot use for db4.7. Requires log_set_config() to be called before Env open
    push(@envargs, (-SetFlags => DB_LOG_INMEMORY)) if $BerkeleyDB::db_version <= 4.6;
    my $logfile;
    open($logfile, ">>$cfg->{logdir}/db.log") && push(@envargs, (-ErrFile => $logfile));
    my $renv = new BerkeleyDB::Env (@envargs)
      or die "Unable to create recovery environment: $BerkeleyDB::Error\n";
    close $logfile;
    unlink "$cfg->{cache_dir}/private/dblock";
    return defined $renv;
}

sub env_remove {
    return unlink <$$cfg{cache_dir}/__db.*>; # Remove environment
}

sub temp_env {
    # From db_verify.c
    # Return an unlocked environment
    # First try to attach to an existing MPOOL
    my $tempenv;
    $tempenv = new BerkeleyDB::Env (-Home   => $cfg->{cache_dir},
				    -Flags => DB_INIT_MPOOL | DB_USE_ENVIRON)
      or
	# Else create a private region
	$tempenv = new BerkeleyDB::Env (-Home   => $cfg->{cache_dir},
					-Flags => DB_CREATE | DB_INIT_MPOOL |
					DB_USE_ENVIRON | DB_PRIVATE)
	  or die "Unable to create temporary DB environment: $BerkeleyDB::Error\n";
    return \$tempenv;
}

sub db_verify {
    return BerkeleyDB::db_verify (-Filename=>shift, -Env=>$+{shift});
}

# Returns reference to status and hash of compaction data
# Arg: DB handle ref
sub _db_compact {
    my $dbh = ${$_[0]};
    my %hash;
    my $status;
    return (\'DB not initialised in _db_compact', undef) unless $dbh;
  SWITCH:
    for ($dbh->type) {
	/1/ && do { # Btree
	    $status = $dbh->compact(undef,undef,\%hash,DB_FREE_SPACE);
	    last SWITCH;
	};
	/2/ && do { # Hash
	    $status = $dbh->compact(undef,undef,\%hash,DB_FREELIST_ONLY);
	    last SWITCH;
	};
    }
    return (\$status,\%hash);
}

# arg: file or filehandle to be scanned and added to DB
# arg: ref to DB handle
sub import_sums {
    my $dbh=${$_[1]};
    my %temp;
    return unless $cfg->{checksum};
    extract_sums($_[0],\%temp);
    while (my ($filename,$data) = each %temp){
	$dbh->db_put($filename,$data) and warn "db_put $filename, $data failed with $!" ;
    }
}

{ # Scoping block
    my $ctx;
    # purpose: create hasher object
    sub data_init {
	return 1 unless $cfg->{checksum};
	$ctx = Digest::MD5->new;
	return 1;
    }

    # purpose: append data to be scanned
    sub data_feed {
	return unless $cfg->{checksum};
	my $ref=shift;
	$ctx->add($$ref);
    }

    # arg: filename
    # arg: DB handle ref
    sub check_sum {
	return 1 unless $cfg->{checksum};
	my $file = $_[0];
	my $dbh= ${$_[1]};
	my $digest = $ctx->hexdigest;
	my $data;
	if ($dbh->db_get($file, $data)) { # Returns 0 on success.
	    debug_message("db_get for $file failed: $!");
	    return 1;
	}

	my $href = hashify(\$data);
	if($href->{md5} && length($href->{md5}) == 32) {
	    # now find the faulty deb
	    debug_message("Verify $file: db $href->{md5}, file $digest");
	    return ($href->{md5} eq $digest);
	}
	debug_message("No stored md5sum found for $file. Ignoring");
	return 1;
    }
}

1;
