package Cache::Apt::Lookup;
use AptPkg::Config '$_config';
use AptPkg::System '$_system';
use AptPkg::Cache;
use AptPkg::Source;
use Cwd;
use Carp;
use File::HomeDir;
use Exporter;
use strict;
use warnings;
use vars qw(@ISA @EXPORT @depends $suite $arch $config $apt_cross_dir
$apt_cache $apt_source $home $host_config $mirror $verbose @dirs
@touch @source_list $self %deptype_dict $cachedir );
@ISA=qw(Exporter);
@EXPORT=qw( srclookup binlookup cache_update get_cache_iter get_cachedir
deplookup force_update check_update setup_config set_cachedir
update_sources prepare_sources_list get_aptcross_dir init_cache
get_verbose set_mirror set_verbose init_host_cache ver_compare 
clear_config update_multiarch );

=pod

=head1 Name

Cache::Apt::Lookup - Cache handler for apt-cross and related tools

=head1 Description

Extensively modified portion of NorthernCross using only the
Config and Cache modules and exposing the package details using a dedicated struct.

Includes code from apt-rdepends to get the Depends info efficiently.

=head1 Example

 use Cache::Apt::Lookup;
 use Cache::Apt::Config;
 use Cache::Apt::Package;
 use Debian::DpkgCross;   # for check_arch
 use Data::Dumper;

 my $verbose = 0;    # usually read a command line option to set an integer
 my $arch = "arm";   # usually read a command line string parameter for this.
 my $suite = "unstable";  # or use &get_suite();
 &set_verbose ($verbose);
 &check_arch($arch) if (defined ($arch));

 &setup_config;
 &update_sources;
 my $config = &init_cache($verbose);

=cut

=head1 Copyright and Licence

 Copyright (C) 2006, 2007  Neil Williams <codehelp@debian.org>

 Copyright (c) 2006, 2007  Alexander Shishkin <alexander.shishkin@siemens.com>

 Copyright (C) 2002-2005  Simon Law <sfllaw@debian.org>

 This package is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program.  If not, see <http://www.gnu.org/licenses/>.

=cut

# from apt-rdepends
#
# apt-rdepends has a wider support for dependencies than
# apt-cross currently uses.
#
# We will track all the packages we have ever seen in this hash.
# Therefore, we will never duplicate the display of an entry.
my %seen = ();
# Choose the direction of our recursive dependencies
my $reverse = 0;
# Are we following Build-Depends?
my $builddep = 0;
# choose the direction of our recursive dependencies
my $PkgReference;
if ($reverse) {
	$PkgReference = 'ParentPkg';
}
else {
	$PkgReference = 'TargetPkg';
}
# Which types of dependencies do we follow?
my @follow = ();
# Which types of dependencies do we show?
my @show = ();
# store of all ProvidesList because this is indexed under what is
# provided, not what provides.
my %provides=();

# We don't print package states by default.
my $printstate = 0;
# Which states should I follow?
my @statefollow = ();
# Which states should I show?
my @stateshow = ();

sub aptinit {
	my ($config, $verbose) = @_;
	$_config->init();
	$_config->set("Apt::Architecture", $config->{'main'}->{'arch'})
		if($config->{'main'}->{'arch'} ne '');
	$_config->set("APT::Get::List-Cleanup", "off");
	$_config->set("APT::Get::AllowUnauthenticated", "true");
	$_config->set("Dir", $config->{'main'}->{'maindir'});
	$_config->set("Dir::Etc", $config->{'main'}->{'maindir'});
	$_config->set("Dir::Etc::SourceList", &Cache::Apt::Config::get_sourceslist);
	$_config->set("Dir::State", $config->{'main'}->{'suite'});
	$_config->set("Dir::State::Status", $config->{'main'}->{'maindir'} . "/status");
	$_config->set("Dir::Cache", $config->{'main'}->{'suite'});

	use Data::Dumper; print Dumper($_config) if ($verbose >= 4);
	$_config->{quiet} = 2 if ($verbose <= 2);
	$_system = $_config->system;
	$apt_cache = AptPkg::Cache->new;
	$apt_source = AptPkg::Source->new;
}

=head1 compare (I<A>, I<B>)

Direct link to the apt perl binding for version comparison.

Compare package version A with B, returning a negative value if A
is an earlier version than B, zero if the same or a positive value
if A is later.

=cut

sub ver_compare
{
	my $vers = $_system->versioning;
	my $ver_a = shift;
	my $ver_b = shift;
	return $vers->compare ($ver_a, $ver_b);
}

=head1 init_host_cache($verbose)

Alternative function that initialises a copy of the host cache in the
apt-cross directory to provide data on cross packages available for
installation.

If $verbose >=4, the full Apt configuration is dumped to STDOUT before
the call returns.

The cache lists are not removed when switching from one architecture
to another.

Remember which cache has been initialised! Queries are made against
whichever cache was initialised most recently.

=cut

sub init_host_cache
{
	$verbose = shift;
	return ($host_config) if (defined($host_config));
	$apt_cross_dir = get_aptcross_dir();
	$suite = &Cache::Apt::Config::get_suite();
	$host_config = new Cache::Apt::Config(
		maindir => $apt_cross_dir,
		suite => $suite,
		arch => '',
		# set extra_conf if necessary - remember the '-o ' prefix.
	);
	print "Updating host cache:\n" . Dumper($host_config)
		if ($verbose >= 4);
	&aptinit($host_config, $verbose);
	return $host_config;
}

=head1 init_cache($verbose)

Loads the currently configured cache into memory.

Accepts a verbosity level to support debugging modes.

If $verbose >=4, the full Apt configuration is dumped to STDOUT before
the call returns.

=cut

sub init_cache
{
	$verbose = shift;
	return ($config) if (defined($config));
	$apt_cross_dir = get_aptcross_dir();
	$suite = &Cache::Apt::Config::get_suite();
	$arch = &Cache::Apt::Config::get_cache_architecture();
	$config = new Cache::Apt::Config(
		maindir => $apt_cross_dir,
		suite => $suite,
		arch => $arch,
		# set extra_conf if necessary - remember the '-o ' prefix.
	);
	print "Updating $arch cache:\n" . Dumper($config) if ($verbose >= 4);
	&aptinit($config, $verbose);
	return $config;
}

=head1 clear_config

Completely clears the current config in preparation for a new
set of &check_cache_arch, &set_suite and &init_cache;

=cut

sub clear_config
{
	undef $config;
}

=head1 srclookup($package_name)

Look up a package name in the source cache and return the
raw apt-pkg-perl data or undef on error.

=cut

sub srclookup {
	my $name = shift;
	my $res = $apt_source->{$name};

	return $res->[0] || undef;
}

=head1 binlookup($package_name)

Look up a package name in the Packages cache and return the
raw apt-pkg-perl data or undef on error.

=cut

sub binlookup {
	my $name = shift;
	my $res = $apt_cache->packages->lookup($name);

	return $res || undef;
}

=head1 deplookup($AptCrossPackage_struct)

Expects an AptCrossPackage and populates that variable with the
dependency information for that package using the current cache
configuration. Returns an array of AptCrossDependency instances
suitable to be dropped into the AptCrossPackage::Depends variable.

=cut

sub deplookup {
	@depends=();
	my $pkg = shift;
	&show_rdepends($pkg);
	%seen = ();
	return \@depends;
}

=head1 get_cache_iter

Provides an iterator over all package names.

Example:
(listing of all unique package names)

 my $iter = &lookup();
 my $pkg;
 my @package_names = ();
 my %hash = ();
 my $c = 0;

 do {
 	$pkg = $iter->next;
	$hash{$pkg}++ if ($pkg);
 } while ($pkg);

 @package_names = sort (keys %hash);
 foreach my $p (@package_names)
 {
	print "$p ";
	$c++;
 }

 print "\nTotal of $c package names\n";

=cut

sub get_cache_iter {
	my $iter = AptPkg::Cache::Iter->new($apt_cache);
	return $iter || undef;
}

=head1 set_cachedir($dir)

Override the default apt_cross_dir with a specified directory that
must already exist. This directory is also expected to contain any
sources.list files that your script will later specify to Config.

=cut

sub set_cachedir
{
	$cachedir = shift;
	if (not -d $cachedir)
	{
		undef $cachedir;
		carp ("Unable to use requested cache directory: does not exist.\n");
		return;
	}
	open (TEST, ">$cachedir/lock") or croak ("unable to use $cachedir: $!");
	print TEST "\n";
	close (TEST);
	unlink ("$cachedir/lock");
}

=head1 get_aptcross_dir

Sets and returns the location of ~/.apt-cross or uses .apt-cross if
no home could be found. Note that when running under sudo
(e.g. in a chroot) $home will be the home directory of the SUDO_USER
within the chroot.

If cachedir has been set and is usable, returns $cachedir.

=cut

sub get_aptcross_dir
{
	if (defined($cachedir))
	{
		$apt_cross_dir = $cachedir;
		return $apt_cross_dir;
	}
	$home = File::HomeDir->my_home;
	# safeguard, just in case.
	$home = "/tmp" if (!defined($home));
	$apt_cross_dir = "$home/.apt-cross";
	mkdir $apt_cross_dir if (! -d $apt_cross_dir);
	return $apt_cross_dir;
}

=head1 get_cachedir

Replacement for get_aptcross_dir to distance the modules from the
apt-cross package. get_aptcross_dir may be removed in future versions
so use this function to get the location of the cache.

=cut

sub get_cachedir
{
	return $cachedir if (defined($cachedir));
	return &get_aptcross_dir();
}

=head1 get_verbose

Return the current verbosity setting.

=cut

sub get_verbose {
	return $verbose;
}

=head1 set_verbose($verbose)

Used by scripts to pass a command-line verbose level to the module
to be used to create debugging output to the console.

 0 == quiet
 1 == normal
 2 == add some progress output or user-level messages.
 3 == add debug messages to level 2.
 4 == add detailed configuration dumps to level 3.

=cut

sub set_verbose {
	$verbose = shift;
}

=head1 set_mirror($mirror)

Supports the apt-cross -m option by adding a specific mirror to the
sources for the currently configured cache. An update will be needed
to load the cache data from the mirror.

=cut

sub set_mirror {
	$mirror = shift;
}

=head1 force_update

Removes the timestamp file for the current user-specific cache
and forces an update of the apt cache. Supports the apt-cross -u
command.

=cut

sub force_update
{
	$apt_cross_dir = &get_aptcross_dir();
	$arch = &Cache::Apt::Config::get_cache_architecture();
	$suite = &Cache::Apt::Config::get_suite();
	$verbose = &get_verbose;
	if (defined $mirror) {
		print "Updating $suite on $arch using $mirror\n" if $verbose >= 2;
	}
	else {
		my $srclist = &Cache::Apt::Config::get_sourceslist;
		print "Updating $arch cache using $srclist\n" if $verbose >= 2;
	}
	my $mysrc = &Cache::Apt::Config::check_mysources;
	unlink ("$apt_cross_dir/sources.$suite") unless defined($mysrc);
	unlink ("$apt_cross_dir/.aptupdate-stamp-$suite-$arch");
	if (not -e "$apt_cross_dir/preferences.d") {
		mkdir "$apt_cross_dir/preferences.d";
	}
	&setup_config;
	&update_sources;
	&update_multiarch;
}

=head1 check_update

Non-intrusive way to ensure the user-specific cache for the current
configuration is up to date without force. If the cache has been
updated within the last 24hrs, no update is performed.

=cut

sub check_update
{
	$verbose = shift;
	$arch = &Cache::Apt::Config::get_cache_architecture();
	$suite = &Cache::Apt::Config::get_suite();
	if (defined $mirror) {
		print "Checking $suite on $arch using $mirror\n" if $verbose >= 2;
	}
	else {
		print "Checking $suite on $arch using apt sources\n" if $verbose >= 2;
	}
	&setup_config;
	&update_sources;
}

=head1 setup_config

Init function for the ~/.apt-cross directory that prepares a
directory tree suitable for each suite and architecture combination.
Copies your apt sources for use with the cross caches and manages
the timestamp file to ensure the configured cache is up to date
using update_sources.

=cut

sub setup_config
{
	$suite = &Cache::Apt::Config::get_suite() if (not defined ($suite));
	$arch = &Cache::Apt::Config::get_cache_architecture();
	$apt_cross_dir = &get_aptcross_dir();
	@dirs = qw/ alternatives info parts updates/;
	@touch = qw/ diversion statoverride status lock/;
	#set up necessary dirs for cross-dpkg database
	if (not -e "$apt_cross_dir/$suite") {
		mkdir "$apt_cross_dir/$suite";
	}
	if (not -e "$apt_cross_dir/$suite/lists") {
		mkdir "$apt_cross_dir/$suite/lists";
	}
	if (not -e "$apt_cross_dir/preferences.d") {
		mkdir "$apt_cross_dir/preferences.d";
	}
	if (not -e "$apt_cross_dir/$suite/lists/partial") {
		mkdir "$apt_cross_dir/$suite/lists/partial";
	}
	if (not -e "$apt_cross_dir/$suite/archives") {
		mkdir "$apt_cross_dir/$suite/archives";
	}
	if (not -e "$apt_cross_dir/$suite/archives/partial") {
		mkdir "$apt_cross_dir/$suite/archives/partial";
	}
	foreach my $dir (@dirs) {
		if (not -d "$apt_cross_dir/$dir") {
			mkdir "$apt_cross_dir/$dir";
		}
	}
	foreach my $file (@touch) {
		utime(time, time, "$apt_cross_dir/$file") or (
			open(F, ">$apt_cross_dir/$file") && close F )
	}
	# only need to make sources file
	# read the sources from apt-cache policy so that all supported apt methods are available.
	# check if $mirror is missing from the file.
	my $mysrc = &Cache::Apt::Config::check_mysources;
	return if (defined $mysrc);
	&prepare_sources_list;
	if ((-f "$apt_cross_dir/sources.$suite") && (defined $mirror)) {
		open (SOURCES, "$apt_cross_dir/sources.$suite") || die "Cannot open apt-cross sources list for $suite $!";
		my @contents=<SOURCES>;
		my $check = join (@contents);
		close (SOURCES);
		if (!$check =~ /$mirror/) {
			# if it is missing, append it.
			open (SOURCES, ">>$apt_cross_dir/sources.$suite") ||
				die "Cannot open apt-cross sources list for $suite $!";
			print SOURCES<<END;
deb $mirror $suite main
deb-src $mirror $suite main
END
			close SOURCES;
			# update using the amended file.
			print "Updating cache of available $arch packages in $suite\n" if ($verbose >= 2);
			my $config = &init_cache($verbose);
			&cache_update($config);
			utime(time, time, "$apt_cross_dir/.aptupdate-stamp-$suite-$arch")
				or ( open(F, ">$apt_cross_dir/.aptupdate-stamp-$suite-$arch") && close F )
		}
	}
	# create sources file for this suite.
	if (! -f "$apt_cross_dir/sources.$suite")
	{ #file is missing, so create new.
		print "debug: recreating $apt_cross_dir/sources.$suite\n" if $verbose >= 2;
		if (defined $mirror) {
			print "Adding $mirror to $apt_cross_dir/sources.$suite\n" if $verbose >= 2;
			open (SOURCES, ">>$apt_cross_dir/sources.$suite") || die "Cannot open sources.$suite: $!";
			print SOURCES<<END;
deb $mirror $suite main
deb-src $mirror $suite main
END
			close SOURCES;
		}
		open (SOURCES, ">>$apt_cross_dir/sources.$suite") || die "Cannot open apt-cross sources list. $!";
		foreach my $source (@source_list)
		{
			next if $source =~ /^#/;
			next if $source =~ /^$/;
			next if ((defined $suite) and ($source !~ /$suite/));
			print SOURCES $source;
		}
		close SOURCES;
		# if an emdebian source has not been specified, the lack of this file produces an annoying warning.
		open (TOUCH, ">$apt_cross_dir/${suite}/lists/www.emdebian.org_debian_dists_${suite}_main_source_Sources");
		close (TOUCH);
		&update_sources;
		utime(time, time, "$apt_cross_dir/.aptupdate-stamp-$suite-$arch")
		or ( open(F, ">$apt_cross_dir/.aptupdate-stamp-$suite-$arch") && close F )
	}
}

=head1 update_sources

Actually performs the apt-get update for the currently configured
cache. Called by setup_config, force_update and check_update.

=cut

sub update_sources
{
	$apt_cross_dir = &get_aptcross_dir;
	$arch = &Cache::Apt::Config::get_cache_architecture;
	$suite = &Cache::Apt::Config::get_suite;
	carp ("No suite defined.\n") if (not defined ($suite));
	carp ("No architecture set.\n") if (not defined ($arch));
	carp ("No cache directory set.\n") if (not defined ($apt_cross_dir));
	#start by downloading sources, but only if not already done this 24hrs
	my $mtime = (stat ("$apt_cross_dir/.aptupdate-stamp-$suite-$arch"))[9];
	$mtime = 0 if (!defined($mtime));
	my $time_now = time();
	if (($time_now - $mtime) > 86400) {
		print "Updating apt-cache for $arch\n" if ($verbose >= 2);
		my $config = &init_cache($verbose);
		&cache_update($config);
		utime(time, time, "$apt_cross_dir/.aptupdate-stamp-$suite-$arch")
			or ( open(F, ">$apt_cross_dir/.aptupdate-stamp-$suite-$arch") && close F )
	}
}

=head1 cache_update

Update the user-specific apt-cross cache using the current
configuration. Errors are directed to /dev/null because switching
architectures and configurations can cause misleading and unnecessary
error messages from apt.

=cut

sub cache_update {
	my $self = shift;
	if (defined($self))
	{
		print "cache_update: " . Dumper($self) if ($verbose >=4);
		system(" apt-get ".$self->{main}->{'config_str'}." update 2>/dev/null")
			if ($verbose >= 2);
		my $str = $self->{main}->{'config_str'}; 
		`apt-get $str update 2>/dev/null`
			if ($verbose < 2);
	}
	&aptinit($self, $verbose);
	return unless defined($host_config);
	system(" apt-get ".$host_config->{main}->{'config_str'}." update 2>/dev/null")
		if ($verbose >= 2);
	my $str = $self->{main}->{'config_str'}; 
	`apt-get $str update 2>/dev/null`
		if ($verbose < 2);
	&aptinit($host_config, $verbose);
}

=head1 prepare_sources_list

Collates your various apt sources into one array that can be used to
provide sources for your cross caches.

=cut

sub prepare_sources_list
{
	# collate all available/configured sources into one list
	if (-e "/etc/apt/sources.list") {
		open (SOURCES, "/etc/apt/sources.list") or die "cannot open apt sources list. $!";
		@source_list = <SOURCES>;
		close (SOURCES);
	}
	if (-d "/etc/apt/sources.list.d/") {
		opendir (FILES, "/etc/apt/sources.list.d/")
			|| die "cannot open apt sources.list directory $!";
		my @files = grep(!/^\.\.?$/, readdir FILES);
		foreach my $f (@files) {
			next if ($f =~ /\.ucf-old$/);
			open (SOURCES, "/etc/apt/sources.list.d/$f") or
				die "cannot open /etc/apt/sources.list.d/$f $!";
			while(<SOURCES>) {
				push @source_list, $_;
			}
			close (SOURCES);
		}
		closedir (FILES);
	}
	return \@source_list;
}

=head1 get_provides

Retrieves the ProvidesList for the specified package.

The AptPkg bindings index the ProvidesList under what is
provided, not what the specified package provides, so this
routine iterates over the cache and creates a suitable
index. Subsequent queries are made against the new hash.

e.g. When querying apt-cache, you may expect to see:
 $ apt-cache show cdebconf | grep Provides
 Provides: debconf-2.0

However, in the AptPkg bindings, this relationship is
indexed under debconf-2.0, not under cdebconf. i.e.
 $cache->{"debconf-2.0"}{ProvidesList}[0]{Name} eq "cdebconf"
actually returns true when Cache::Apt::Lookup expects:
 $cache->{"cdebconf"}{ProvidesList}[0]{Name} eq "debconf-2.0"
to be true.

Returns a comma-separated list of packages that are
provided by the specified package or undef.

=cut

my $provide_done = 0;

sub get_provides {
	my $pkg = shift(@_);
	return $provides{$pkg} if ($provide_done > 0);
	my $ipkg;
	my $iter = &get_cache_iter();
	do {
		$ipkg = $iter->next;
		return undef unless (defined $ipkg);
		if ($apt_cache->{$ipkg}{ProvidesList})
		{
			my @list=();
			my $check;
			foreach(@{$apt_cache->{$ipkg}{ProvidesList}})
			{
				$check = $_->{OwnerPkg}{Name};
				push @list, $_->{Name};
			}
			$provides{$check} = join(', ', @list);
		}
		$provide_done++;
	} while ($ipkg);
	return $provides{$pkg};
}

=head1 update_multiarch

Calculates the Packages file for the requested suite and architecture
and uses that to update the multiarch lists. Warns if the calculated
file does not exist - in which case, use the C<--suite> or C<-S> options
to F<apt-cross> to explicitly state which suite to use. (This can happen
if you use codenames in your F</etc/apt/sources.list> rather than
suite names (sid instead of unstable). Apart from sid, the mapping from
a codename to a suite name changes with each release and this module
does not attempt to keep up with these changes.

Once found, the routine checks for 
F</usr/share/apt-cross/update-multiarch.pl> and passes the Packages
file to it.

=cut

sub update_multiarch
{
	my ($repo, $sname);
	$arch = &Cache::Apt::Config::get_cache_architecture();
	$suite = &Cache::Apt::Config::get_suite();
	$verbose = &get_verbose;
	$apt_cross_dir = get_aptcross_dir();
	my $mysrc = &prepare_sources_list;
	$sname = $suite;
	$sname = "sid" if ($suite eq "unstable");
	foreach my $src (@$mysrc)
	{
		chomp $src;
		next if ($src =~ /^#/);
		next if ($src eq "");
		next if ($src =~ /^deb-src/);
		if ($src =~ /$suite/ or $src =~ /$sname/)
		{
			$repo = $src;
			$repo =~ m#^deb (f|ht)tp://([^ ]*) ([^ ]*) *([a-z\-]+) *([a-z\-]*) *([a-z\-]*)$#;
			my $rep = $2;
			my $cmpnt = $4;
			$rep =~ s:/$::;
			$rep =~ s#/#_#g;
			$repo = $apt_cross_dir . "/" . $suite . "/lists/" . $rep .
				"_dists_${suite}_${cmpnt}_binary-${arch}_Packages";
			$repo = $apt_cross_dir . "/" . $suite . "/lists/" . $rep .
				"_dists_${sname}_${cmpnt}_binary-${arch}_Packages"
				if (not -f $repo);
			last if (-f $repo);
		}
	}
	if (not defined $repo)
	{
		warn "Unable to calculate the correct Packages file to update multiarch!\n";
		print "Try specifying a suite explicitly with --suite\n";
		return;
	}
	if (not -f $repo)
	{
		warn "Unable to calculate the correct Packages file to update multiarch!\n$repo\n";
		print "Try specifying a suite explicitly with --suite\n";
		return;
	}
	elsif ((-x "/usr/share/apt-cross/update-multiarch.pl") and
		(defined $repo))
	{
		print "Updating multiarch config using:\n  '$repo'\n"
			if ($verbose >= 2);
		system ("/usr/share/apt-cross/update-multiarch.pl -f $repo");
	}
}

### (End of public functions) ###

sub this_cache { return $apt_cache }

# Set defaults.
if ($builddep) {
  @follow = ("Build-Depends", "Build-Depends-Indep") unless (@follow);
  @show = ("Build-Depends", "Build-Depends-Indep") unless (@show);
}
else {
  @follow = (AptPkg::Dep::Depends, AptPkg::Dep::PreDepends) unless (@follow);
  @show = (AptPkg::Dep::Depends, AptPkg::Dep::PreDepends) unless (@show);
}

@statefollow = ("NotInstalled",
		"UnPacked",
		"HalfConfigured",
		"HalfInstalled",
		"ConfigFiles",
		"Installed")
	unless (@statefollow);
@stateshow = ("NotInstalled",
		"UnPacked",
		"HalfConfigured",
		"HalfInstalled",
		"ConfigFiles",
		"Installed")
	unless (@stateshow);

sub print_depcompareop {
	my $depcompareop = shift(@_);
	return "|"  if ($depcompareop == AptPkg::Dep::Or);
	return ""   if ($depcompareop == AptPkg::Dep::NoOp);
	return "<=" if ($depcompareop == AptPkg::Dep::LessEq);
	return ">=" if ($depcompareop == AptPkg::Dep::GreaterEq);
	return "<<" if ($depcompareop == AptPkg::Dep::Less);
	return ">>" if ($depcompareop == AptPkg::Dep::Greater);
	return "="  if ($depcompareop == AptPkg::Dep::Equals);
	return "!=" if ($depcompareop == AptPkg::Dep::NotEquals);
}

sub get_depends {
	my $pkg = shift(@_);
	my $reverse = shift(@_);

	# Resolve the package by name.
	my $p;
	if ($builddep) {
		unless ($p = $apt_source->get($pkg)) {
			warn "W: Unable to locate package $pkg\n";
			return;
		}
	}
	else {
		unless ($p = $apt_cache->get($pkg)) {
			warn "W: Unable to locate package $pkg\n";
			return;
		}
	}
	&get_provides($pkg);
	# Which way do our dependencies go?  Reverse, or forward.  Notice
	# how we get the last version for our forward dependencies.
	if ($reverse) {
		return $p->{RevDependsList};
	}
	elsif ($builddep) {
		if (my $i = pop(@$p)) {
		return $i->{BuildDepends};
		}
	}
	else {
		if (my $i = $p->{VersionList}) {
			if (my $j = pop(@$i)) {
				return $j->{DependsList};
			}
		}
	}
}

sub file_depends {
	my $results = shift(@_);
	my $deps = shift(@_);
	my $flag = 0;
	my @orlist = ();
	my $tag = "";
	my $or_ver = "";

	for my $dep (@$deps)
	{
		# get the comparison operator
		my $op = ($dep->{CompTypeDeb}) ? $dep->{CompTypeDeb} : 
			print_depcompareop ($dep->{CompType});
		$op = "" if (not defined $op);
		$tag = $op if ($flag == 0);
		# Figure out the version.
		my $version = ($reverse ? $dep->{ParentVer}->{VerStr} : $dep->{TargetVer});
		# Figure out the current state.
		my $state = ($reverse ? $dep->{ParentPkg}->{CurrentState} :
			$dep->{TargetPkg}->{CurrentState});
		# if $op eq "|", need to concatenate until $op != "|"
		# hack-alert for debconf
		if (($dep->{$PkgReference}->{Name} eq "debconf") and $op ne "|")
		{
			warn "W: debconf operator hack alert" if ($verbose >= 4);
			$tag = $op;
			$op = "|";
			$flag++;
		}
		if ($op eq "|")
		{
			$flag++;
			$or_ver = $dep->{$PkgReference}->{Name};
			$or_ver .= " ($tag $version)" if (defined $version);
			# hack alert
			if (($dep->{$PkgReference}->{Name} eq "debconf")
			 and ($or_ver !~ /\(.*\)/))
			{
				$or_ver .= " (>= 0.5)";
				warn "W: debconf: version hack alert" if ($verbose >= 4);
			}
			$tag = $op;
			push @orlist, $or_ver;
		}
		else
		{
			push @orlist, $dep->{$PkgReference}->{Name} if ($flag > 0);
			$flag = 0;
			# Push the name of this package into the right pigeonhole.
			$$results{0+$dep->{DepType}}{$dep->{$PkgReference}->{Name}} = 
				[ $version, $state, "$dep->{DepType}", $tag, join(" | ", @orlist) ];
			@orlist = ();
		}
		# Populate the dictionaries of names for this locale
		$deptype_dict{0+$dep->{DepType}} = lc("$dep->{DepType}");
		print Dumper ($$results{0+$dep->{DepType}}{$dep->{$PkgReference}->{Name}})
			if ($verbose >= 4);
	}
}

sub file_builddepends {
	my $results = shift(@_);
	my $deps = shift(@_);

	# Build-Depends are keyed by dependency type
	for my $deptype (keys %$deps) {
		# There is a list of packages within each
		for my $pkgs ($$deps{$deptype}) {
			# Each package may have a version
			for my $pkg (@$pkgs) {
				my $version = $pkg->[2];
				if ($version) {
					my $op = print_depcompareop($pkg->[1]);
					$version = ($pkg->[1] ? $op . " " : "") . $version;
				}
				my $p = $apt_cache->get($pkg->[0]);
				my $state = "Unknown";
				$state = $apt_cache->get($pkg->[0])->{CurrentState} if ($p);
				# Push the name of this package into the right pigeonhole.
				$$results{$deptype}{$pkg->[0]} = [ $version, $state, "$deptype" ];
				$deptype_dict{$deptype} = lc($deptype);
			}
		}
	}
}

sub show_rdepends {
	my $emp = shift;
	my $pkg = ($$emp->Package);
	# Only recurse if we have never seen this package before
	return @depends if ($seen{$pkg});
	$seen{$pkg} = 1;
	# Get the dependencies for this $pkg
	my $deps = get_depends($pkg, $reverse);
	return unless ($deps);
	# %results stores results for each category of dependency (Conflicts,
	# Depends, Replaces, Suggests)
	my %results;
	if ($builddep) {
		file_builddepends(\%results, $deps);
	}
	else {
		file_depends(\%results, $deps);
	}
	push (@depends, \%results);
}

1
