#
# Copyright (C) 2005 Chris Halls <halls@debian.org>
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of version 2.1 of the GNU Lesser General Public
# License as published by the Free Software Foundation.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

"""Unit test for apt_proxy.py"""

import copy, tempfile, os, shutil

from twisted.trial import unittest
from StringIO import StringIO

from apt_proxy.apt_proxy_conf import apConfig
from apt_proxy.apt_proxy import Factory

config1="""
[DEFAULT]
debug=all:9
port=9999
address=
cleanup_freq=off

[backend1]
backends = http://a.b.c/d

[backend2]
backends = http://d.e.f/g

[backend3]
backends = http://h.i.j/k
"""

config2="""
[DEFAULT]
debug=all:1
port=8888
address=1.2.3.4
cleanup_freq=off

# Backend 1 deleted

[backend2]
# no change
backends = http://d.e.f/g

[backend3]
# changed
backends = http://l.m.n/o

[backend4]
# new
backends = http://p.q.r/s

[backend5]
# another new
backends = http://t.u.v/w
"""

class apTestHelper(unittest.TestCase):
    default_config = "[DEFAULT]\ndebug=all:9 apt:0\n" # Config string to use
    def setUp(self):
        self.cache_dir = tempfile.mkdtemp('.aptproxy')
        self.config = self.default_config.replace('[DEFAULT]','[DEFAULT]\ncache_dir=' + self.cache_dir)
    def tearDown(self):
        shutil.rmtree(self.cache_dir)
    
class FactoryInitTest(apTestHelper):
    def setUp(self):
        self.default_config = config1
        apTestHelper.setUp(self)
        self.c = apConfig(StringIO(self.config))
    def testFactoryInit(self):
        factory = Factory(self.c)
        self.assertEquals(factory.config, self.c)
        del factory
    def testFactoryBackendInit(self):
        factory = Factory(self.c)
        factory.configurationChanged()
        self.assertEquals(len(factory.backends),3)
        self.assertEquals(factory.backends.keys(), ['backend1', 'backend2', 'backend3'])
        self.assertEquals(factory.backends['backend1'].uris[0].host, 'a.b.c')
        del factory

class StartFactoryTest(unittest.TestCase):
    def setUp(self):
        self.cache_dir = tempfile.mkdtemp('.aptproxy')
        config = config1.replace('[DEFAULT]','[DEFAULT]\ncache_dir=' + self.cache_dir)
        self.c = apConfig(StringIO(config))
    def tearDown(self):
        shutil.rmtree(self.cache_dir)
    def testFactoryStart(self):
        factory = Factory(self.c)
        factory.startFactory

class ConfigChangeTest(unittest.TestCase):
    def setUp(self):
        self.tempdir = tempfile.mkdtemp('.aptproxy')
        configOld = config1.replace('[DEFAULT]','[DEFAULT]\ncache_dir=%s/old'%(self.tempdir))
        self.cOld = apConfig(StringIO(configOld))
        self.factory = Factory(self.cOld)
        self.factory.configurationChanged()
    def tearDown(self):
        del(self.factory)
        shutil.rmtree(self.tempdir)
    def loadNewConfig(self):
        configNew = config2.replace('[DEFAULT]','[DEFAULT]\ncache_dir=%s/new'%(self.tempdir))
        self.cNew = apConfig(StringIO(configNew))
        self.factory.config = copy.copy(self.cNew)
        self.factory.configurationChanged(self.cOld)
    def testNotAllowedChanges(self):
        self.loadNewConfig()
        self.assertNotEquals(self.factory.config.port, self.cNew.port)
        self.assertEquals(self.factory.config.port, self.cOld.port)
        self.assertEquals(self.factory.config.address, self.cOld.address)
    def testGlobalChanges(self):
        self.loadNewConfig()
        self.assertEquals(self.factory.config.debug, 'all:1')
        self.assertEquals(self.factory.config.debug, self.cNew.debug)
    def testBackendCount(self):
        self.loadNewConfig()
        self.assertEquals(len(self.factory.backends),4)
    def testBackendChanges(self):
        self.assertEquals(self.factory.backends['backend3'].uris[0].host, 'h.i.j')
        self.loadNewConfig()
        self.assertEquals(self.factory.backends.keys(), ['backend2', 'backend3', 'backend4', 'backend5'])
        self.assertEquals(self.factory.backends['backend3'].uris[0].host, 'l.m.n')
