# $Progeny: main.py 4163 2004-03-19 16:35:36Z licquia $

# Copyright (C) 2002 Progeny Linux Systems, Inc.
# Authors: Jeff Licquia
#
# This is free software; you may redistribute it and/or modify
# it under the terms of the GNU General Public License as
# published by the Free Software Foundation; either version 2,
# or (at your option) any later version.
#
# This is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with the Debian GNU/Linux system; if not, write to the Free
# Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
# 02111-1307 USA

import sys
import os
import string
import re
import xreadlines
import configlet
import UserDict

try:
    import xml.dom.minidom
    do_xml = True
except:
    do_xml = False

# Utility functions

def split_sources_line(line):
    index = 0
    accumulator = ""
    is_bracketed = 0
    num_splits = 0
    result = []
    while index < len(line):
        if line[index:index+7] == "cdrom:[":
            is_bracketed = 1
        elif line[index] == "]" and is_bracketed:
            is_bracketed = 0

        if line[index] in string.whitespace \
             and not is_bracketed \
             and num_splits < 3 \
             and accumulator:
            if accumulator[:3] == "deb":
                num_splits = 0
            result.append(accumulator)
            accumulator = ""
            num_splits = num_splits + 1
        else:
            accumulator = accumulator + line[index]

        index = index + 1

    if accumulator:
        result.append(accumulator)
    return result

def get_list_path():
    conf_hash = {}
    conf = os.popen("apt-config dump")
    for line in conf:
        (name, value) = string.split(string.strip(line), maxsplit = 1)
        while value[-1] != value[0]:
            value = value[:-1]
        value = value[1:-1]
        conf_hash[name] = value
    conf.close()

    config_path = ["Dir", "Etc", "sourcelist"]
    config_str = conf_hash[string.join(config_path, "::")]
    while config_str[0] != "/" and len(config_path) > 0:
        config_path.pop()
        key = string.join(config_path, "::")
        if conf_hash.has_key(key):
            config_str = conf_hash[key] + config_str

    return config_str

# Classes

class SourceEntry(UserDict.UserDict):
    def __init__(self, sources_lines = None):
        UserDict.UserDict.__init__(self)

        if sources_lines is None:
            self["type"] = "unknown"
        else:
            if re.match(r'^#*\s*deb', sources_lines[-1]):
                for line in sources_lines:
                    if re.match(r'^#\s+X-AptConf', line):
                        (rawfield, value) = re.split(r':\s*', line, 1)
                        field = string.strip(rawfield[1:])[10:].lower()
                        self[field] = string.strip(value)

                real_line = sources_lines[-1]
                if real_line[0] == "#":
                    self["enabled"] = False
                else:
                    self["enabled"] = True
                # elements = re.split(r'\s+', real_line, 3)
                elements = split_sources_line(real_line)
                while not re.search(r'deb', elements[0]):
                    elements = elements[1:]
                while elements[0][0] == "#":
                    elements[0] = elements[0][1:]

                self["baseurl"] = elements[1]
                self["suite"] = elements[2]
                if self["suite"][-1] == "/":
                    self["suite"] = self["suite"][:-1]
                if len(elements) > 3:
                    component_str = string.strip(elements[3])
                    self["components"] = re.split(r'\s+', component_str)
                if elements[0] == "deb":
                    self["repotype"] = "binary"
                elif elements[0] == "deb-src":
                    self["repotype"] = "source"
                else:
                    self["repotype"] = "unknown"

                self["type"] = "repository"
            else:
                self["type"] = "comments"
                self["comments"] = sources_lines

    def getID(self):
        if not self.has_key("id"):
            if self["type"] == "comments":
                self["id"] = "comment"
            elif self.has_key("repo-desc"):
                self["id"] = self["repo-desc"].getID()
            elif self.has_key("Name"):
                self["id"] = self["Name"]
            else:
                id = self["baseurl"] + "-" + self["suite"]
                id = re.compile(r'[:/\[\]]+').sub("_", id)
                self["id"] = re.compile(r'\s+').sub("_", id)

        return self["id"]

    def getDescription(self):
        if not self.has_key("description"):
            if self.has_key("repo-desc"):
                self["description"] = self["repo-desc"].getDescription()
            else:
                self["description"] = self["baseurl"] + " " + self["suite"]

            self["description"] = string.strip(self["description"])

        if self["description"] == "\r":
            sys.stderr.write("Yup!\n")
        return self["description"]

    def isRepository(self):
        return self["type"] == "repository"

    def isEquivalent(self, other):
        """Determines if a source entry is equivalent to this one:
        whether one is the binary and the other is the source for
        the same repository.
        """

        if self["suite"] == other["suite"] and \
           self["baseurl"] == other["baseurl"] and \
           self["repotype"] != other["repotype"]:
            return True
        else:
            return False

    def isEnabled(self):
        return self.has_key("enabled") and self["enabled"]

    def mergeEquivalent(self, other):
        "Merge the information from other into self."

        if not self.isEquivalent(other):
            raise TypeError, "source entries are not equivalent"

        for key in other.keys():
            if not self.has_key(key):
                self[key] = other[key]

        self["repotype"] = "both"

    def isBinary(self):
        return self["repotype"] == "binary" or self["repotype"] == "both"

    def isSource(self):
        return self["repotype"] == "source" or self["repotype"] == "both"

class RepoDesc(UserDict.UserDict):
    def __init__(self, dom_node):
        UserDict.UserDict.__init__(self)

        if not do_xml:
            return

        for attr_key in dom_node.attributes.keys():
            self[attr_key] = dom_node.attributes[attr_key].nodeValue

        for node in dom_node.childNodes:
            if node.nodeType != node.ELEMENT_NODE:
                continue

            if node.tagName == "description":
                lang = node.attributes["lang"].nodeValue;
                self["description-%s" % (lang,)] = node.childNodes[0].nodeValue
            elif node.tagName == "source":
                self["source"] = "yes"
            elif node.tagName == "binary":
                self["binary"] = "yes"
            elif node.tagName == "mirror":
                if not self.has_key("mirrors"):
                    self["mirrors"] = {}
                location = node.attributes["location"].nodeValue
                if not self["mirrors"].has_key(location):
                    self["mirrors"][location] = []
                self["mirrors"][location].append(node.childNodes[0].nodeValue)
            elif node.tagName == "component":
                if not self.has_key("components"):
                    self["components"] = []
                self["components"].append(node.childNodes[0].nodeValue)
            else:
                self[node.tagName] = node.childNodes[0].nodeValue

    def getID(self):
        if not do_xml:
            return ""

        return self["name"]

    def getDescription(self):
        if not do_xml:
            return ""

        return self["description-en"]

    def matchSource(self, source):
        """Does the given source line match this repo?  Return the
        mirror location and URL if so; otherwise, return None."""

        if not do_xml:
            return None

        if source["type"] != "repository":
            return None

        if source.has_key("repo-desc"):
            return (source["repo-mirror-loc"], source["repo-mirror-url"])

        if self.has_key("suite"):
            repo_suite = self["suite"]
        else:
            repo_suite = self["dir"]
        if source["suite"] == repo_suite:
            for mirror_loc in self["mirrors"].keys():
                for mirror_url in self["mirrors"][mirror_loc]:
                    if source["baseurl"] == mirror_url:
                        return (mirror_loc, mirror_url)

        return None

    def blankSourceEntry(self):
        if not do_xml:
            return None

        source = SourceEntry()
        source["type"] = "repository"
        source["repo-desc"] = self
        source["enabled"] = False
        source["baseurl"] = self["mirrors"][self["mirrors"].keys()[0]][0]
        if self.has_key("suite"):
            source["suite"] = self["suite"]
            source["components"] = self["components"][:]
        else:
            source["suite"] = self["dir"]
            source["components"] = []

        if self.has_key("source") and self.has_key("binary"):
            source["repotype"] = "both"
        elif self.has_key("source"):
            source["repotype"] = "source"
        elif self.has_key("binary"):
            source["repotype"] = "binary"

        return source

# Functions that create, manipulate, etc. the above classes.

def parse_sources_list(fn):
    sources_file = open(fn)
    sources_list = []
    accumulator = []
    state = ""
    for line in xreadlines.xreadlines(sources_file):
        if re.match(r'^#*\s*deb', line):
            if state == "comment" and len(accumulator):
                sources_list.append(SourceEntry(accumulator))
                accumulator = []

            accumulator.append(line)
            sources_list.append(SourceEntry(accumulator))
            accumulator = []
            state = "comment"
        else:
            if re.match(r'^#\s+X-AptConf', line):
                if state != "entry":
                    if len(accumulator):
                        sources_list.append(SourceEntry(accumulator))
                    accumulator = []
                    state = "entry"
            else:
                if state != "comment":
                    if len(accumulator):
                        sources_list.append(SourceEntry(accumulator))
                    accumulator = []
                    state = "comment"

            accumulator.append(line)

    return sources_list

def parse_repo_description(fn):
    repo_list = []

    dom = xml.dom.minidom.parse(fn)
    top = dom.documentElement
    for node in top.getElementsByTagName("repository"):
        repo_list.append(RepoDesc(node))

    return repo_list

# The configlet itself.

class AptConf(configlet.Configlet):
    # Utility functions

    def redraw_repolist(self):
        list = self.wtree.get_widget("repolist")

        select_callback = getattr(self, "on_list_item_selected")
        list_selection = list.get_selection()
        list_selection.connect("changed", select_callback)

        toggled_callback = getattr(self, "on_list_item_checkbox_toggled")

        list_store = gtk.ListStore(gobject.TYPE_INT, gobject.TYPE_BOOLEAN,
                                   gobject.TYPE_STRING)
        self.list_sources = []
        source_index = 0
        for source in self.sources:
            if source.isRepository():
                self.list_sources.append(source)

                list_item = list_store.append()
                list_store.set(list_item,
                               0, source_index,
                               1, source.isEnabled(),
                               2, source.getDescription())

                source_index = source_index + 1

                #list_item_check.connect("toggled", toggled_callback, source)
                #list_item.connect("select", select_callback, source)
        
        list.set_model(list_store)

        for old_column in list.get_columns():
            list.remove_column(old_column)

        toggle_renderer = gtk.CellRendererToggle()
        toggle_renderer.connect("toggled", toggled_callback, list_store)
        toggle_col = gtk.TreeViewColumn("Enabled", toggle_renderer, active=1)
        text_col = gtk.TreeViewColumn("Repository", gtk.CellRendererText(),
                                      text=2)
        list.append_column(toggle_col)
        list.append_column(text_col)

    # Signal handlers

    def on_list_item_selected(self, selection):
        (model, iter) = selection.get_selected()
        index = model.get_value(iter, 0)
        self.selected_source = self.list_sources[index]
        remove_button = self.wtree.get_widget("remove_button")
        if self.selected_source.has_key("repo-desc"):
            remove_button.set_sensitive(False)
        else:
            remove_button.set_sensitive(True)

    def on_list_item_checkbox_toggled(self, item, argl, model):
        iter = model.get_iter((int(argl),))
        index = model.get_value(iter, 0)
        source = self.list_sources[index]

        if source.has_key("enabled"):
            source["enabled"] = not source["enabled"]

        model.set(iter, 1, source.isEnabled())

    def on_add_custom_button_clicked(self, button):
        component_list = self.wtree.get_widget("custom_component_list")
        component_list.set_headers_visible(False)

        model = gtk.ListStore(gobject.TYPE_STRING)
        component_list.set_model(model)

        for old_column in component_list.get_columns():
            component_list.remove_column(old_column)
        renderer = gtk.CellRendererText()
        col = gtk.TreeViewColumn(None, renderer, text=0)
        component_list.append_column(col)

        self.selected_source = None
        
        self.wtree.get_widget("custom_desc_entry").set_text("")
        self.wtree.get_widget("custom_baseurl_entry").set_text("")
        self.wtree.get_widget("custom_suite_entry").set_text("")
        self.wtree.get_widget("custom_type_combo_entry").set_text("Binary")
        self.wtree.get_widget("custom_component_check").set_active(False)
        self.wtree.get_widget("custom_component_frame").set_sensitive(False)

        self.wtree.get_widget("custom").show_all()

    def on_add_cd_button_clicked(self, button):
        mountpoint = "/cdrom"
        tag_line = "Source List entries for this Disc are"

        cd_dialog = gtk.Dialog("CD-ROM Mount Path", None, gtk.DIALOG_MODAL, (gtk.STOCK_OK, gtk.RESPONSE_OK, gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL))

        cd_label = gtk.Label("Enter the mount point for the CD drive:")
        cd_label.show()
        cd_dialog.vbox.pack_start(cd_label)

        cd_entry = gtk.Entry()
        cd_entry.set_text(mountpoint)
        cd_entry.show()
        cd_dialog.vbox.pack_start(cd_entry)

        response = cd_dialog.run()
        mountpoint = cd_entry.get_text()
        cd_dialog.destroy()

        if response == gtk.RESPONSE_OK:
            cd_source_entry = ""

            os.system("/bin/mount %s" % (mountpoint,))

            if os.environ.has_key("TMPDIR"):
                temp_path = "%s/aptconf.%d" \
                            % (os.environ["TMPDIR"], os.getpid())
            else:
                temp_path = "/tmp/aptconf.%d" % (os.getpid())

            try:
                os.mkdir(temp_path)
                configlet.privileged_run("/usr/bin/apt-cdrom -d %s -m -o Dir::Etc::sourcelist=%s/junk add"
                                         % (mountpoint, temp_path))

                apt_cdrom_out = open("%s/junk" % (temp_path,))
                for line in xreadlines.xreadlines(apt_cdrom_out):
                    if line[:3] == "deb":
                        cd_source_entry = line
                        break
                apt_cdrom_out.close()
                os.unlink("%s/junk" % (temp_path,))
                os.rmdir(temp_path)
            except:
                self.debug("Error detected adding CD.")

            os.system("/bin/umount %s" % (mountpoint,))

            if cd_source_entry:
                new_source = SourceEntry([cd_source_entry])
                self.sources.append(new_source)
                self.redraw_repolist()

    def on_remove_button_clicked(self, button):
        if self.selected_source:
            for index in range(0, len(self.sources)):
                if self.sources[index]["type"] == "repository":
                    if self.sources[index].getID() == self.selected_source.getID():
                        del self.sources[index]
                        break

            self.redraw_repolist()

    def on_prop_button_clicked(self, button):
        if self.selected_source:
            if self.selected_source.has_key("repo-desc"):
                repo = self.selected_source["repo-desc"]
                mirror_combo = self.wtree.get_widget("mirror_combo")

                mirror_item_list = []
                mirror_item_text = ""
                for mirror_loc in repo["mirrors"].keys():
                    for mirror_url in repo["mirrors"][mirror_loc]:
                        mirror_str = "%s (%s)" % (mirror_url, mirror_loc)
                        mirror_item_list.append(mirror_str)
                        if mirror_url == self.selected_source["baseurl"]:
                            mirror_item_text = mirror_str
                mirror_combo.set_popdown_strings(mirror_item_list)
                mirror_combo.entry.set_text(mirror_item_text)

                if self.selected_source.isSource():
                    self.wtree.get_widget("options_source_check").set_active(True)

                component_list = self.wtree.get_widget("complist")
                component_model = gtk.ListStore(gobject.TYPE_BOOLEAN,
                                                gobject.TYPE_STRING)
                component_list.set_model(component_model)

                for old_column in component_list.get_columns():
                    component_list.remove_column(old_column)
                col = gtk.TreeViewColumn(None, gtk.CellRendererToggle(),
                                         active=0)
                component_list.append_column(col)
                col = gtk.TreeViewColumn(None, gtk.CellRendererText(),
                                         text=1)
                component_list.append_column(col)

                if repo.has_key("components"):
                    for component in repo["components"]:
                        iter = component_model.append()
                        component_model.set_value(iter, 1, component)
                        if component in self.selected_source["components"]:
                            component_model.set_value(iter, 0, True)
                        else:
                            component_model.set_value(iter, 0, False)

                self.wtree.get_widget("options").show_all()
            else:
                component_list = self.wtree.get_widget("custom_component_list")
                component_model = gtk.ListStore(gobject.TYPE_STRING)
                component_list.set_model(component_model)

                for old_column in component_list.get_columns():
                    component_list.remove_column(old_column)
                renderer = gtk.CellRendererText()
                col = gtk.TreeViewColumn(None, renderer, text=0)
                component_list.append_column(col)

                source = self.selected_source
                self.wtree.get_widget("custom_desc_entry").set_text(source.getDescription())
                self.wtree.get_widget("custom_baseurl_entry").set_text(source["baseurl"])
                self.wtree.get_widget("custom_suite_entry").set_text(source["suite"])

                if source.isBinary() and source.isSource():
                    self.wtree.get_widget("custom_type_combo_entry").set_text("Both")
                elif source.isBinary():
                    self.wtree.get_widget("custom_type_combo_entry").set_text("Binary")
                elif source.isSource():
                    self.wtree.get_widget("custom_type_combo_entry").set_text("Source")

                if source.has_key("components") and len(source["components"]):
                    self.wtree.get_widget("custom_component_check").set_active(True)
                    self.wtree.get_widget("custom_component_frame").set_sensitive(True)
                    for component in source["components"]:
                        iter = component_model.append()
                        component_model.set_value(iter, 0, component)
                else:
                    self.wtree.get_widget("custom_component_check").set_active(False)
                    self.wtree.get_widget("custom_component_frame").set_sensitive(False)

                self.wtree.get_widget("custom").show_all()

    def on_help_button_clicked(self, button):
        self.debug("Help button clicked.")
        # XXX: need GTK/GNOME 2 help system.
        return False
        help_path = gnome.help.file_find_file("aptconf", "index.html")
        if help_path:
            self.debug("Starting help system for %s" % (help_path,))
            gnome.help.goto(help_path)

    def on_custom_component_check_toggled(self, button):
        frame = self.wtree.get_widget("custom_component_frame")
        frame.set_sensitive(button.get_active())

    def on_custom_add_button_clicked(self, button):
        self.debug("custom add button clicked")
        add_dialog = gtk.Dialog("Add Component", None, gtk.DIALOG_MODAL, (gtk.STOCK_OK, gtk.RESPONSE_OK, gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL))

        add_label = gtk.Label("Enter the repository name to add:")
        add_label.show()
        add_dialog.vbox.pack_start(add_label)

        add_entry = gtk.Entry()
        add_entry.show()
        add_dialog.vbox.pack_start(add_entry)

        response = add_dialog.run()
        add_dialog.hide()

        if response == gtk.RESPONSE_OK:
            component_list = self.wtree.get_widget("custom_component_list")
            component_model = component_list.get_model()
            iter = component_model.append()
            component_model.set(iter, 0, add_entry.get_text())

            component_list.unselect_all()

        add_dialog.destroy()

    def on_custom_remove_button_clicked(self, button):
        component_list = self.wtree.get_widget("custom_component_list")
        selected = component_list.get_selection()
        (model, iter) = selected.get_selected()
        if iter is not None:
            model.remove(iter)

    def on_custom_ok_button_clicked(self, button):
        source = None
        if self.selected_source is None:
            source = SourceEntry()
            source["type"] = "repository"
            self.sources.append(source)
        else:
            for index in range(0, len(self.sources)):
                if self.sources[index]["type"] == "repository":
                    if self.sources[index].getID() == self.selected_source.getID():
                        source = self.sources[index]
                        break

        source["description"] = \
            self.wtree.get_widget("custom_desc_entry").get_text()
        source["baseurl"] = \
            self.wtree.get_widget("custom_baseurl_entry").get_text()
        source["suite"] = \
            self.wtree.get_widget("custom_suite_entry").get_text()

        source["repotype"] = \
            string.lower(self.wtree.get_widget("custom_type_combo_entry").get_text())

        if self.wtree.get_widget("custom_component_check").get_active():
            new_components = []
            model = self.wtree.get_widget("custom_component_list").get_model()
            iter = model.get_iter_first()
            while iter:
                new_components.append(model.get_value(iter, 0))
                iter = model.iter_next(iter)

            source["components"] = new_components
        else:
            source["components"] = []

        self.wtree.get_widget("custom").hide()
        self.redraw_repolist()

    def on_custom_cancel_button_clicked(self, button):
        self.wtree.get_widget("custom").hide()

    def on_options_ok_button_clicked(self, button):
        source = None
        if self.selected_source is None:
            source = SourceEntry()
            source["type"] = "repository"
            self.sources.append(source)
        else:
            for index in range(0, len(self.sources)):
                if self.sources[index].getID() == self.selected_source.getID():
                    source = self.sources[index]
                    break

        mirror_str = self.wtree.get_widget("mirror_combo_entry").get_text()
        mirror_url = mirror_str[:-5]
        source["baseurl"] = mirror_url

        if self.wtree.get_widget("options_source_check").get_active():
            source["repotype"] = "both"
        else:
            source["repotype"] = "binary"

        complist = self.wtree.get_widget("complist")
        comp_model = complist.get_model()
        new_comp = []
        iter = comp_model.get_iter_first()
        while iter:
            if comp_model.get_value(iter, 0):
                new_comp.append(comp_model.get_value(iter, 1))
            iter = comp_model.iter_next(iter)
        source["components"] = new_comp

        self.wtree.get_widget("options").hide() 
        self.redraw_repolist()

    def on_options_cancel_button_clicked(self, button):
        self.wtree.get_widget("options").hide()

    # Configlet methods

    def gnome_setup(self):
        configlet.Configlet.gnome_setup(self)

        import pygtk
        pygtk.require("2.0")

        global gnome
        global gnome_ui
        import gnome.ui
        gnome_ui = gnome.ui
        global gtk
        import gtk
        global gobject
        import gobject

        self.sources = parse_sources_list(get_list_path())

        for index in range(0, len(self.sources) - 1):
            if not self.sources[index].isRepository():
                continue
            for index2 in range(index, len(self.sources)):
                if not self.sources[index2].isRepository():
                    continue
                if self.sources[index].isEquivalent(self.sources[index2]):
                    self.sources[index].mergeEquivalent(self.sources[index2])
                    self.sources[index2]["type"] = "disabled"
                    break

        self.repos = []
        if do_xml:
            for path in self.attributes["repo_desc_paths"]:
                for fn in os.listdir(path):
                    self.repos.extend(parse_repo_description("%s/%s"
                                                             % (path, fn)))

        seen_repos = []
        for source in self.sources:
            for repo in self.repos:
                match_info = repo.matchSource(source)
                if match_info:
                    source["repo-desc"] = repo
                    source["repo-mirror-loc"] = match_info[0]
                    source["repo-mirror-url"] = match_info[1]
                    seen_repos.append(repo)
                    break

        for repo in self.repos:
            if repo not in seen_repos:
                self.sources.append(repo.blankSourceEntry())

        self.selected_source = None
        self.selected_custom_component = None

        dict = {}
        for key in dir(AptConf):
            dict[key] = getattr(self, key)
        self.wtree.signal_autoconnect(dict)

        self.redraw_repolist()

    def report_debconf(self):
        id_str = ""
        dc = []
        for src in self.sources:
            if src["type"] != "repository":
                continue

            id = src.getID()
            id_str = id_str + id + "|"
            self.debug("id = %s" % (id,))

            for field in ("repotype", "description", "baseurl", "suite"):
                dc.append("aptconf/%s aptconf/%s-%s %s"
                          % (field, field, id, src[field]))

            if src.has_key("components"):
                component_str = string.join(src["components"])
            else:
                component_str = ""
            dc.append("aptconf/components aptconf/components-%s %s"
                      % (id, component_str))

            if src.isEnabled():
                dc.append("aptconf/disabled aptconf/disabled-%s false" % (id,))
            else:
                dc.append("aptconf/disabled aptconf/disabled-%s true" % (id,))

        # Remove trailing "|".
        id_str = id_str[:-1]
        dc.append("aptconf/idlist aptconf/idlist " + id_str)

        return dc

_attrs = { "name": "aptconf",
           "display_title": "Software Sources",
           "description": "Select this option to configure the locations from which software updates and new software may be retrieved.",
           "packages": ["aptconf"],
           "repo_desc_paths": ["/usr/share/aptconf", "/var/cache/aptconf"],
           "api_version": 2
}
configlet.register_configlet(AptConf, _attrs)

# vim:ai:et:sts=4:sw=4:tw=0:
